#ifndef _RHEO_DIS_ACCUMULATE_H
#define _RHEO_DIS_ACCUMULATE_H
//
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// =========================================================================

#include "rheolef/promote.h"
#include "rheolef/distributed.h"
#include <iterator>
namespace rheolef {

/*D:dis_accumulate
NAME:  dis_accumulate - distributed accumulate algorithm (@PACKAGE@-@VERSION@)
DESCRIPTION:       
 STL-like accumulate for distributed containers
 environment.
SYNOPSIS:
  @example
  template<class InputIterator, class Size>
  T dis_accumulate (InputIterator first, Size n);
  @end example
EXAMPLE:
   A sample usage writes:
   @example
     # include "rheolef/disarray.h"
     # include "rheolef/dis_accumulate.h"
     int main(int argc, char**argv) {
         environment distributed (argc, argv);
         unsigned int n = 100;
         disarray<double> x(n, 2.0);
         double sum = dis_accumulate(x.begin(),x.size(),x.comm());
         dout << "dot(x,1) = " << sum << endl;
     }
   @end example
SEE ALSO: "disarray"(1)
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   24 november 1998
End:
*/
template <class InputIterator, class Size>
typename std::iterator_traits<InputIterator>::value_type
dis_accumulate (
    	InputIterator first,
	Size n,
	const distributor::communicator_type& comm,
	sequential /* memory_model */)
{
    typedef typename std::iterator_traits<InputIterator>::value_type T;
    T sum = T(0);
    for (Size i = 0; i < n; ++i, ++first) {
        sum = sum + (*first);
    }
    return sum;
}
#ifdef _RHEOLEF_HAVE_MPI
template <class InputIterator,class Size>
inline
typename std::iterator_traits<InputIterator>::value_type
dis_accumulate (
    	InputIterator first,
	Size n,
	const distributor::communicator_type& comm,
	distributed /* memory_model */)
{
    typedef typename std::iterator_traits<InputIterator>::value_type T;
    T local_sum = dis_accumulate (first, n, comm, sequential());
    return mpi::all_reduce (comm, local_sum, std::plus<T>());    
}
#endif // _RHEOLEF_HAVE_MPI

template <class InputIterator, class Size>
inline
typename std::iterator_traits<InputIterator>::value_type
dis_accumulate (
    	InputIterator first,
	Size n,
	const distributor::communicator_type& comm)
{
    return dis_accumulate (first, n, comm, rheo_default_memory_model());
}

} // namespace rheolef
#endif // _RHEO_DIS_ACCUMULATE_H
