% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hashing.R
\name{Hash functions}
\alias{Hash functions}
\alias{hash}
\alias{hashing}
\alias{scrypt}
\alias{argon2}
\alias{shorthash}
\alias{sha512}
\alias{sha256}
\title{Hash Functions}
\usage{
hash(buf, key = NULL, size = 32)

scrypt(buf, salt = raw(32), size = 32)

argon2(buf, salt = raw(16), size = 32)

shorthash(buf, key)

sha512(buf, key = NULL)

sha256(buf, key = NULL)
}
\arguments{
\item{buf}{data to be hashed}

\item{key}{key for HMAC hashing. Optional, except for in \code{shorthash}.}

\item{size}{length of the output hash. Must be between 16 and 64 (recommended is 32)}

\item{salt}{non-confidential random data to seed the algorithm}
}
\description{
Functions to calculate cryptographic hash of a message, with optionally a key for
HMAC applications. For storing passwords, use \link{password_store} instead.
}
\details{
The generic \code{hash} function is recommended for most applications. It uses
dynamic length
\href{https://doc.libsodium.org/hashing/generic_hashing.html}{BLAKE2b}
where output size can be any value between 16 bytes (128bit) and 64 bytes (512bit).

The \link{scrypt} hash function is designed to be CPU and memory expensive to protect
against brute force attacks. This algorithm is also used by the \link{password_store}
function.

The \link{argon2} hash function is also designed to be CPU and memory expensive to protect
against brute force attacks. Argon2 is a password-hashing function that summarizes the
state of the art in the design of memory-hard functions

The \code{shorthash} function is a special 8 byte (64 bit) hash based on
\href{https://doc.libsodium.org/hashing/short-input_hashing.html}{SipHash-2-4}.
The output of this function is only 64 bits (8 bytes). It is useful for in e.g.
Hash tables, but it should not be considered collision-resistant.

Hash functions can be used for HMAC by specifying a secret \code{key}. They key size
for \code{shorthash} is 16 bytes, for \code{sha256} it is 32 bytes and for \code{sha512}
it is 64 bytes. For \code{hash} the key size can be any value between 16 and 62,
recommended is at least 32.
}
\examples{
# Basic hashing
msg <- serialize(iris, NULL)
hash(msg)
sha256(msg)
sha512(msg)
scrypt(msg)

# Generate keys from passphrase
passphrase <- charToRaw("This is super secret")
key <- hash(passphrase)
shortkey <- hash(passphrase, size = 16)
longkey <- hash(passphrase, size = 64)

# HMAC (hashing with key)
hash(msg, key = key)
shorthash(msg, shortkey)
sha256(msg, key = key)
sha512(msg, key = longkey)
}
\references{
\url{https://doc.libsodium.org/hashing/generic_hashing.html}
}
