% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot-methods.R
\name{plot_heatmap}
\alias{plot_heatmap}
\title{Create an ecologically-organized heatmap using ggplot2 graphics}
\usage{
plot_heatmap(physeq, method = "NMDS", distance = "bray",
  sample.label = NULL, taxa.label = NULL, low = "#000033",
  high = "#66CCFF", na.value = "black", trans = log_trans(4),
  max.label = 250, title = NULL, sample.order = NULL, taxa.order = NULL,
  first.sample = NULL, first.taxa = NULL, ...)
}
\arguments{
\item{physeq}{(Required). The data, in the form of an instance of the
\code{\link{phyloseq-class}}. This should be what you get as a result
from one of the
\code{\link{import}} functions, or any of the processing downstream.
No data components beyond the \code{\link{otu_table}} are strictly 
necessary, though they may be useful if you want to re-label the 
axis ticks according to some observable or taxonomic rank, for instance,
or if you want to use a \code{\link{UniFrac}}-based distance
(in which case your \code{physeq} data would need to have a tree included).}

\item{method}{(Optional).
The ordination method to use for organizing the 
heatmap. A great deal of the usefulness of a heatmap graphic depends upon 
the way in which the rows and columns are ordered.}

\item{distance}{(Optional). A character string. 
The ecological distance method to use in the ordination.
See \code{\link{distance}}.}

\item{sample.label}{(Optional). A character string.
The sample variable by which you want to re-label the sample (horizontal) axis.}

\item{taxa.label}{(Optional). A character string.
The name of the taxonomic rank by which you want to
re-label the taxa/species/OTU (vertical) axis.
You can see available options in your data using
\code{\link{rank_names}(physeq)}.}

\item{low}{(Optional). A character string. An R color.
See \code{?\link{colors}} for options support in R (there are lots).
The color that represents the lowest non-zero value
in the heatmap. Default is a dark blue color, \code{"#000033"}.}

\item{high}{(Optional). A character string. An R color.
See \code{\link{colors}} for options support in R (there are lots).
The color that will represent the highest 
value in the heatmap. The default is \code{"#66CCFF"}.
Zero-values are treated as \code{NA}, and set to \code{"black"}, to represent
a background color.}

\item{na.value}{(Optional). A character string. An R color.
See \code{\link{colors}} for options support in R (there are lots).
The color to represent what is essentially the background of the plot,
the non-observations that occur as \code{NA} or
\code{0} values in the abundance table. The default is \code{"black"}, which 
works well on computer-screen graphics devices, but may be a poor choice for
printers, in which case you might want this value to be \code{"white"}, and
reverse the values of \code{high} and \code{low}, above.}

\item{trans}{(Optional). \code{"trans"}-class transformer-definition object.
A numerical transformer to use in 
the continuous color scale. See \code{\link[scales]{trans_new}} for details.
The default is \code{\link{log_trans}(4)}.}

\item{max.label}{(Optional). Integer. Default is 250.
 The maximum number of labeles to fit on a given axis (either x or y). 
 If number of taxa or samples exceeds this value, 
 the corresponding axis will be stripped of any labels. 

 This supercedes any arguments provided to
 \code{sample.label} or \code{taxa.label}.
 Make sure to increase this value if, for example,
 you want a special label
 for an axis that has 300 indices.}

\item{title}{(Optional). Default \code{NULL}. Character string.
The main title for the graphic.}

\item{sample.order}{(Optional). Default \code{NULL}. 
Either a single character string matching 
one of the \code{\link{sample_variables}} in your data,
or a character vector of \code{\link{sample_names}}
in the precise order that you want them displayed in the heatmap.
This overrides any ordination ordering that might be done
with the \code{method}/\code{distance} arguments.}

\item{taxa.order}{(Optional). Default \code{NULL}. 
Either a single character string matching 
one of the \code{\link{rank_names}} in your data,
or a character vector of \code{\link{taxa_names}}
in the precise order that you want them displayed in the heatmap.
This overrides any ordination ordering that might be done
with the \code{method}/\code{distance} arguments.}

\item{first.sample}{(Optional). Default \code{NULL}.
A character string matching one of the \code{\link{sample_names}}
of your input data (\code{physeq}). 
It will become the left-most sample in the plot.
For the ordination-based ordering (recommended),
the left and right edges of the axes are adjaacent in a continuous ordering. 
Therefore, the choice of starting sample is meaningless and arbitrary,
but it is aesthetically poor to have the left and right edge split 
a natural cluster in the data.
This argument allows you to specify the left edge
and thereby avoid cluster-splitting, emphasize a gradient, etc.}

\item{first.taxa}{(Optional). Default \code{NULL}.
A character string matching one of the \code{\link{taxa_names}}
of your input data (\code{physeq}). 
This is equivalent to \code{first.sample} (above),
but for the taxa/OTU indices, usually the vertical axis.}

\item{...}{(Optional). Additional parameters passed to \code{\link{ordinate}}.}
}
\value{
A heatmap plot, in the form of a \code{\link{ggplot}2} plot object,
 which can be further saved and modified.
}
\description{
There are many useful examples of phyloseq heatmap graphics in the
\href{http://joey711.github.io/phyloseq/plot_heatmap-examples}{phyloseq online tutorials}.
In a 2010 article in BMC Genomics, Rajaram and Oono show describe an 
approach to creating a heatmap using ordination methods to organize the 
rows and columns instead of (hierarchical) cluster analysis. In many cases
the ordination-based ordering does a much better job than h-clustering. 
An immediately useful example of their approach is provided in the NeatMap
package for R. The NeatMap package can be used directly on the abundance 
table (\code{\link{otu_table-class}}) of phylogenetic-sequencing data, but 
the NMDS or PCA ordination options that it supports are not based on ecological
distances. To fill this void, phyloseq provides the \code{plot_heatmap()}
function as an ecology-oriented variant of the NeatMap approach to organizing
a heatmap and build it using ggplot2 graphics tools.
The \code{distance} and \code{method} arguments are the same as for the
\code{\link{plot_ordination}} function, and support large number of
distances and ordination methods, respectively, with a strong leaning toward
ecology.
This function also provides the options to re-label the OTU and sample 
axis-ticks with a taxonomic name and/or sample variable, respectively, 
in the hope that this might hasten your interpretation of the patterns
(See the \code{sample.label} and \code{taxa.label} documentation, below). 
Note that this function makes no attempt to overlay hierarchical 
clustering trees on the axes, as hierarchical clustering is not used to 
organize the plot. Also note that each re-ordered axis repeats at the edge,
and so apparent clusters at the far right/left or top/bottom of the 
heat-map may actually be the same. For now, the placement of this edge
can be considered arbitrary, so beware of this artifact of this graphical
representation. If you benefit from this phyloseq-specific implementation
of the NeatMap approach, please cite both our packages/articles.
}
\details{
This approach borrows heavily from the \code{heatmap1} function in the
\code{NeatMap} package. Highly recommended, and we are grateful for their
package and ideas, which we have adapted for our specific purposes here,
but did not use an explicit dependency. At the time of the first version
of this implementation, the NeatMap package depends on the rgl-package,
which is not needed in phyloseq, at present. Although likely a transient
issue, the rgl-package has some known installation issues that have further
influenced to avoid making NeatMap a formal dependency (Although we love
both NeatMap and rgl!).
}
\examples{
data("GlobalPatterns")
gpac <- subset_taxa(GlobalPatterns, Phylum=="Crenarchaeota")
# FYI, the base-R function uses a non-ecological ordering scheme,
# but does add potentially useful hclust dendrogram to the sides...
gpac <- subset_taxa(GlobalPatterns, Phylum=="Crenarchaeota")
# Remove the nearly-empty samples (e.g. 10 reads or less)
gpac = prune_samples(sample_sums(gpac) > 50, gpac)
# Arbitrary order if method set to NULL
plot_heatmap(gpac, method=NULL, sample.label="SampleType", taxa.label="Family")
# Use ordination
plot_heatmap(gpac, sample.label="SampleType", taxa.label="Family")
# Use ordination for OTUs, but not sample-order
plot_heatmap(gpac, sample.label="SampleType", taxa.label="Family", sample.order="SampleType")
# Specifying both orders omits any attempt to use ordination. The following should be the same.
p0 = plot_heatmap(gpac, sample.label="SampleType", taxa.label="Family", taxa.order="Phylum", sample.order="SampleType")
p1 = plot_heatmap(gpac, method=NULL, sample.label="SampleType", taxa.label="Family", taxa.order="Phylum", sample.order="SampleType")
#expect_equivalent(p0, p1)
# Example: Order matters. Random ordering of OTU indices is difficult to interpret, even with structured sample order
rando = sample(taxa_names(gpac), size=ntaxa(gpac), replace=FALSE)
plot_heatmap(gpac, method=NULL, sample.label="SampleType", taxa.label="Family", taxa.order=rando, sample.order="SampleType")
# # Select the edges of each axis. 
# First, arbitrary edge, ordering
plot_heatmap(gpac, method=NULL)
# Second, biological-ordering (instead of default ordination-ordering), but arbitrary edge
plot_heatmap(gpac, taxa.order="Family", sample.order="SampleType")
# Third, biological ordering, selected edges
plot_heatmap(gpac, taxa.order="Family", sample.order="SampleType", first.taxa="546313", first.sample="NP2")
# Fourth, add meaningful labels
plot_heatmap(gpac, sample.label="SampleType", taxa.label="Family", taxa.order="Family", sample.order="SampleType", first.taxa="546313", first.sample="NP2")
}
\references{
Because this function relies so heavily in principle, and in code, on some of the
 functionality in NeatMap, please site their article if you use this function
 in your work.

 Rajaram, S., & Oono, Y. (2010).
 NeatMap--non-clustering heat map alternatives in R. BMC Bioinformatics, 11, 45.

Please see further examples in the 
\href{http://joey711.github.io/phyloseq/plot_heatmap-examples}{phyloseq online tutorials}.
}
