///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/ColorPickerWidget.h>
#include <core/gui/properties/PropertiesPanel.h>
#include <core/gui/panels/CommandPanel.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/reference/RefTargetListener.h>

namespace Core {

class ModifierStack;		// defined in ModifierStack.h
class SceneObject;			// defined in SceneObject.h
class Modifier;				// defined in Modifier.h

/******************************************************************************
* The creation page lets the user modify the selected object.
******************************************************************************/
class ModifyCommandPage : public CommandPanelPage
{
	Q_OBJECT

public:

	/// Initializes the modify page.
    ModifyCommandPage();

	/// Resets the modify page to the initial state.
	virtual void reset();

	/// Is called when the user selects the page.
	virtual void onEnter();

	/// Is called when the user selects another page.
	virtual void onLeave();

	/// This is called after all changes to the selection set have been completed.
	virtual void onSelectionChangeComplete(SelectionSet* newSelection);

	/// Returns the object that is currently being edited in the properties panel.
	RefTarget* editObject() const { return propertiesPanel->editObject(); }

	/// Returns the object that encapsulates the modification stack of the selected node(s).
	ModifierStack* modifierStack() const { return stack; }

private:

	/// Finds all modifier classes provided by the installed plugins.
	void scanInstalledModifierClasses();

	/// Updates the editor controls for the node name, node color etc.
	void updateNodePropertiesEditor();

protected Q_SLOTS:

	/// Is called when the user has entered a new node name in the input text box.
	void onNodeNameEntered();

	/// Is called when the user has picked a new color for the selected node.
	void onNodeDisplayColorPicked();

	/// Handles the ACTION_MODIFIER_DELETE command, which deleted the selected modifier from the stack.
	void onDeleteModifier();

	/// Is called when the user has selected an item in the modifier class list.
	void onModifierAdd(int index);

	/// Is called when the user has selected another item in the modifier stack list box.
	void onModifierStackSelectionChanged();

	/// This called when the user double clicks on an item in the modifier stack.
	void onModifierStackDoubleClicked(const QModelIndex& index);

	/// This is called by the RefTargetListener that listens to notification messages sent by the
	/// current selection set.
	void onSelectionSetMessage(RefTargetMessage* msg);

	/// Handles the ACTION_MODIFIER_MOVE_UP command, which moves the selected modifier up one entry in the stack.
	void onModifierMoveUp();

	/// Handles the ACTION_MODIFIER_MOVE_DOWN command, which moves the selected modifier down one entry in the stack.
	void onModifierMoveDown();

	/// Handles the ACTION_MODIFIER_TOGGLE_STATE command, which toggles the enabled/disable state of the selected modifier.
	void onModifierToggleState(bool newState);

private:

	/// The entry box for the name of the selected scene node.
	QLineEdit* nodeNameBox;

	/// The color picker control.
	ColorPickerWidget* nodeDisplayColorPicker;

	/// This listens to notification message sent by the current selection set.
	RefTargetListener selectionSetListener;

	/// This list box shows the modifier stack of the selected scene node(s).
	QListView* stackBox;

	/// This control displays the list of available modifier classes and allows the user to apply a modifier.
	QComboBox* modifierSelector;

	/// This panel shows the properties of the selected modifier stack entry
	PropertiesPanel* propertiesPanel;

	/// The list of installed modifier plugin classes.
	QVector<PluginClassDescriptor*> modifierClasses;

	/// The object that holds the modification stack of the selected node(s).
	ModifierStack* stack;

	friend class ModifierStack;
	friend class ModifierStackEntry;
};


};
