/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/taskfacades.h"

// Boost C++
#include <boost/filesystem.hpp>
#include <boost/locale/format.hpp>

// Standard Library
#include <string>

// Media Scanner Library
#include "mediascanner/logging.h"
#include "mediascanner/writablemediaindex.h"

namespace mediascanner {

// Boost C++
using boost::locale::format;

// Context specific logging domains
static const logging::Domain kInfo("info/tasks", logging::info());

template<typename T>
class MediaIndexFacade<T>::Private {
public:
    Private(MediaRootManagerPtr root_manager)
        : media_index_(root_manager) {
        g_mutex_init(&mutex_);
    }

    FileSystemPath media_index_path_;
    T media_index_;

    mutable GMutex mutex_;
};

template<typename T>
MediaIndexFacade<T>::MediaIndexFacade(MediaRootManagerPtr root_manager,
                                      const FileSystemPath &path)
    : d(new Private(root_manager)) {
    set_media_index_path(path);
}

template<typename T>
MediaIndexFacade<T>::MediaIndexFacade(MediaRootManagerPtr root_manager) :
    d(new Private(root_manager)) {
    FileSystemPath f;
    set_media_index_path(f);
}

template<typename T>
MediaIndexFacade<T>::~MediaIndexFacade() {
    delete d;
}

template<typename T>
void MediaIndexFacade<T>::set_media_index_path(const FileSystemPath &path) {
    g_mutex_lock(&d->mutex_);
    d->media_index_path_ = path.empty() ? MediaIndex::default_path() : path;
    g_mutex_unlock(&d->mutex_);
}

template<typename T>
FileSystemPath MediaIndexFacade<T>::media_index_path() const {
    g_mutex_lock(&d->mutex_);
    const FileSystemPath path = d->media_index_path_;
    g_mutex_unlock(&d->mutex_);
    return path;
}

template<typename T>
MediaRootManagerPtr MediaIndexFacade<T>::root_manager() const {
    return d->media_index_.root_manager();
}

template<typename T>
void MediaIndexFacade<T>::Run(const TaskFunction &run_task,
                              const ErrorFunction &report_error) {
    std::string error_message;

    g_mutex_lock(&d->mutex_);

    if (d->media_index_.is_open()
            && d->media_index_path_ != d->media_index_.path())
        d->media_index_.Close();

    if (not d->media_index_.is_open()) {
        if (not d->media_index_.Open(d->media_index_path_)) {
            error_message = d->media_index_.error_message();

            if (error_message.empty()) {
                const FileSystemPath index_path = d->media_index_.path();
                error_message =
                        (format("Opening media index at \"{1}\" "
                                "failed without error message.")
                         % index_path).str();
            }
        }
    }

    g_mutex_unlock(&d->mutex_);

    if (error_message.empty()) {
        run_task(&d->media_index_);
    } else {
        report_error(error_message);
    }
}

template class MediaIndexFacade<MediaIndex>;
template class MediaIndexFacade<WritableMediaIndex>;

} // namespace mediascanner
