// license:BSD-3-Clause
// copyright-holders:David Haywood
/************************************************************************************************************

    Maygay EPOCH hardware
     'mechanical' fruit machine HW
     primary use 1997 - 2003

    HD6413002FN16 (H8-3002) CPU

    YMZ280B sound

    2x PICs for security
    the PIC marked 'Security' is apparently the RTC, and doesn't change
    the other is per game / manufacturer and provides data needed for booting?

    The majority of system tasks are handled by a custom ASIC which is capable of reading and writing
    directly from/to the main RAM.  This includes handling of the inputs, and lamps / LEDs

    -------------------------------------------------
    MOST GAMES DO NOT HAVE ANY SOUND ROMS DUMPED
    -------------------------------------------------
    SOME OF THESE GAMES USED HDDS, ONLY ONE IS DUMPED
    -------------------------------------------------

    Even where we have sound rom dumps they probably shouldn't be trusted, the source of them
    had many badly dumped Scorpion 4 sound roms (0x080000 bytes where they should be double that)
    and many of these sound roms are that size...

    Some games have EPOCH NEVADA in the header, others have EPOCH APOLLO, is there a difference?

************************************************************************************************************/


#include "emu.h"
#include "cpu/h8/h83002.h"
#include "sound/ymz280b.h"
#include "speaker.h"


class maygayep_state : public driver_device
{
public:
	maygayep_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag),
			m_maincpu(*this, "maincpu")
	{ }

	void maygayep(machine_config &config);

	void init_maygayep();

private:
	void maygayep_map(address_map &map);

	// devices
	required_device<cpu_device> m_maincpu;
};

// bp 29e58 in ep_simp reads the 'INITIALISE . . .' string
void maygayep_state::maygayep_map(address_map &map)
{
	map.global_mask(0xffffff);
	map(0x000000, 0x07ffff).rom().region("maincpu", 0);
	map(0xfe0000, 0xfffd0f).ram(); // merln at least?
}

static INPUT_PORTS_START( maygayep )
INPUT_PORTS_END


/* in many sets the 0x100 - 0x1ff area contains revision information
   and sometimes bugfix notes..

   in other sets 0x200 - 0x204 contains 'bank' or 'reels' maybe to
   indicate if the roms are for a topbox or the player unit?

   there doesn't seem to be a hard defined standard for this header
   information, it probably varies between manufacturer

   todo: use this info to better sort the sets (note, the time / build
   dates don't appear to be accurate at least)
*/

void maygayep_state::init_maygayep()
{
	uint8_t *src = memregion( "maincpu" )->base();
	for (int i = 0x100; i < 0x210; i++)
	{
		uint8_t val = src[i^1];

		if (i % 0x40 == 0) printf("\n");

		if ((val >= 0x20) && (val <= 0x7e))
		{
			printf("%c", val);
		}
		else
		{
			printf(" ");
		}
	}

	printf("\n");

}

void maygayep_state::maygayep(machine_config &config)
{
	H83002(config, m_maincpu, 16000000);
	m_maincpu->set_addrmap(AS_PROGRAM, &maygayep_state::maygayep_map);

	SPEAKER(config, "mono").front_center();

	YMZ280B(config, "ymz", 10000000).add_route(ALL_OUTPUTS, "mono", 1.0);
}

#define MISSING_SOUND \
	ROM_REGION( 0x100000, "ymz", ROMREGION_ERASE00 ) \
		ROM_LOAD16_BYTE( "sound_roms", 0x000000, 0x080000, NO_DUMP )

/* Only a handful of games have dumped sound roms! */

#define EP_SIMP_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "simpsnd0", 0x000000, 0x080000, CRC(d58d16cc) SHA1(75eb2ab106855156831399a08eed66295c0c288f) ) \
	ROM_LOAD( "simpsnd1", 0x080000, 0x080000, CRC(26d12470) SHA1(4ea92b9d80c73d784534185313a6d5534cb6f3b2) )

ROM_START( ep_simp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980256.g0", 0x000000, 0x020000, CRC(8b5b266a) SHA1(358a98f1cd4fc65a4dfe7e9436eeca5f0649de15) )
	ROM_LOAD16_BYTE( "sa980256.g1", 0x000001, 0x020000, CRC(a4ec81bb) SHA1(bac8475def260f53e0fd25037752295ba04c88de) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980188.g0", 0x000000, 0x020000, CRC(d5eb8db3) SHA1(8a8f5bfc7642899f3052ffc453198c1a57abf54d) )
	ROM_LOAD16_BYTE( "sa980188.g1", 0x000001, 0x020000, CRC(07f873ce) SHA1(87c7e6e2ce44ff6cdafb06fe9081763e6f3902fc) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980189.g0", 0x000000, 0x020000, CRC(7e00fedb) SHA1(1f5b3a812c49ece4315336c1db5c46a87bf3cfc2) )
	ROM_LOAD16_BYTE( "sa980189.g1", 0x000001, 0x020000, CRC(2d38481d) SHA1(2a69e9d50aaad05573de694b374faa06fa544789) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980190.g0", 0x000000, 0x020000, CRC(8ae70534) SHA1(13f386d83cb014d82245de23bd1a8ed23656ff3b) )
	ROM_LOAD16_BYTE( "sa980190.g1", 0x000001, 0x020000, CRC(a42db6ae) SHA1(2732ab70fb0f576b69c79b4fd32906a5e9b744cb) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980191.g0", 0x000000, 0x020000, CRC(210c765c) SHA1(416762bf46d3e67f908632551c6a3aab1c73f250) )
	ROM_LOAD16_BYTE( "sa980191.g1", 0x000001, 0x020000, CRC(01ec7d3a) SHA1(ae1040698a66930d3efb39dd6e1e88cd607ad499) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980192.g0", 0x000000, 0x020000, CRC(964712ef) SHA1(04d8b72a6d98a0114630d5d7e0a5fe8460b027b3) )
	ROM_LOAD16_BYTE( "sa980192.g1", 0x000001, 0x020000, CRC(a3130962) SHA1(02ebde05add1ea55d7cc5372f26fbfd57b7b6ceb) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980193.g0", 0x000000, 0x020000, CRC(3dac6187) SHA1(aee2444d8c605e582681f130c984d14a1c0cf2cb) )
	ROM_LOAD16_BYTE( "sa980193.g1", 0x000001, 0x020000, CRC(be767c57) SHA1(babae96459bd2aabbc9df8d14df0ed4ef3b5b035) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980194.g0", 0x000000, 0x020000, CRC(d8de351d) SHA1(33b728623af9bea4400581dba9f6940558b00e91) )
	ROM_LOAD16_BYTE( "sa980194.g1", 0x000001, 0x020000, CRC(114878a7) SHA1(7627d5d53dca49376b81dbb77433bfc7061c3af7) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simph )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980195.g0", 0x000000, 0x020000, CRC(73354675) SHA1(5804c9ee1ac93dfdc57793a831f6738581763752) )
	ROM_LOAD16_BYTE( "sa980195.g1", 0x000001, 0x020000, CRC(af6e6e64) SHA1(ae45bf1ec494bf62aaee9f4e8dd538e6534a39a1) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990077.g0", 0x000000, 0x020000, CRC(d8c221e5) SHA1(acce206e28451d26805abdfead4bdb6ed5605a65) )
	ROM_LOAD16_BYTE( "sa990077.g1", 0x000001, 0x020000, CRC(bbb3d924) SHA1(b5f500ec41c627eca14f211acf9bea8794caee99) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990078.g0", 0x000000, 0x020000, CRC(62bcfdf8) SHA1(48070d88bcd94f544614ff9659eaf7088716679e) )
	ROM_LOAD16_BYTE( "sa990078.g1", 0x000001, 0x020000, CRC(9173e2f7) SHA1(d232ef619c60dae2cd45b8bd0c8ec8c0bcd3d08a) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990079.g0", 0x000000, 0x020000, CRC(aa1cfbae) SHA1(8379d6eedee525af62370a02dda1c782fa4d7a42) )
	ROM_LOAD16_BYTE( "sa990079.g1", 0x000001, 0x020000, CRC(7277a8fd) SHA1(ee6f2cd01b1c5e2ff95b944a221eecce5e6327a9) )
	EP_SIMP_SOUND
ROM_END

ROM_START( ep_simpm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990080.g0", 0x000000, 0x020000, CRC(5efb0041) SHA1(6891faf1ace25fcb383b54c378f01605fa9cd771) )
	ROM_LOAD16_BYTE( "sa990080.g1", 0x000001, 0x020000, CRC(05e0ad1c) SHA1(b42cbcceb8f2aa2acf2209643d1bfc6346c0b0c1) )
	EP_SIMP_SOUND
ROM_END




#define EP_DBLIM_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "doubleimpact_snd0.bin", 0x000000, 0x080000, CRC(d83b2a30) SHA1(1bf94b246941dfd3b8c84249085e6f93072da432) ) \
	ROM_LOAD( "doubleimpact_snd1.bin", 0x080000, 0x080000, CRC(3f4bb32a) SHA1(af2f8e7f341dec0ffaae4c3b2efca847cddd8a11) )
ROM_START( ep_dblim )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ndi10.g0", 0x000000, 0x020000, CRC(caee435c) SHA1(3dc8e1c0cf2f8b7f177fd785260d0d14fa835743) )
	ROM_LOAD16_BYTE( "i99ndi10.g1", 0x000001, 0x020000, CRC(b4be42b1) SHA1(47c7d8af9eee7bd07fc724797e68fb3fa945fcb4) )
	EP_DBLIM_SOUND
ROM_END

ROM_START( ep_dblima )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ndi12.g0", 0x000000, 0x020000, CRC(caee435c) SHA1(3dc8e1c0cf2f8b7f177fd785260d0d14fa835743) )
	ROM_LOAD16_BYTE( "i99ndi12.g1", 0x000001, 0x020000, CRC(b8f3ec80) SHA1(839b5e1979d0006e6426cff867cc6bd4ba7d7133) )
	EP_DBLIM_SOUND
ROM_END

ROM_START( ep_dblimb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pdi11.g0", 0x000000, 0x020000, CRC(caee435c) SHA1(3dc8e1c0cf2f8b7f177fd785260d0d14fa835743) )
	ROM_LOAD16_BYTE( "i99pdi11.g1", 0x000001, 0x020000, CRC(5f201689) SHA1(906e17a2f48613d3250e00d69495039f47c6cf9c) )
	EP_DBLIM_SOUND
ROM_END

ROM_START( ep_dblimc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snd0.bin", 0x000000, 0x020000, CRC(ed015239) SHA1(3049eb654f786062ad0d9f2e675b73e5afbdd73c) )
	ROM_LOAD16_BYTE( "snd1.bin", 0x000001, 0x020000, CRC(8928fd0e) SHA1(2bd2e511b9d480f24f2f3c0b3a27d1afabda1c0d) )
	EP_DBLIM_SOUND
ROM_END

ROM_START( ep_dblimd )
	ROM_REGION( 0x100000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "d_imp 5pound.g0", 0x000000, 0x080000, CRC(82fa1341) SHA1(3c7eac065e97d8734e18e94b53b293a7ffbe05a3) )
	ROM_LOAD16_BYTE( "d_imp 5pound.g1", 0x000001, 0x080000, CRC(9c36fe52) SHA1(1f2aa87f420e89a1aed21245019f363feedbf3de) )
	EP_DBLIM_SOUND
ROM_END


#define EP_ITJB3_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "italianjobsound0.bin", 0x000000, 0x080000, CRC(680f5932) SHA1(08d212625c16ff63f57a8c0165f1b9a158513879) ) \
	ROM_LOAD( "italianjobsound1.bin", 0x080000, 0x080000, CRC(595514ab) SHA1(9586fb22dd748bafa10553090f48b42b750c7e97) )
ROM_START( ep_itjb3 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "italianjobgame0.bin", 0x000000, 0x020000, CRC(ceaefd73) SHA1(b4f756212773c88b0c09364c1981773e6a4c5527) )
	ROM_LOAD16_BYTE( "italianjobgame1.bin", 0x000001, 0x020000, CRC(2bfdea93) SHA1(7709c067042b186afd5deb5e9701ca46bbb90ebb) )
	ROM_LOAD16_BYTE( "italianjobgame2.bin", 0x040000, 0x020000, CRC(e38bd4c5) SHA1(30965cdfe4f5b251bed5ba175699ffb5bf9c06b1) )
	ROM_LOAD16_BYTE( "italianjobgame3.bin", 0x040001, 0x020000, CRC(3913d80e) SHA1(ccfa1b8e5e2e6468f5cb8144a2f329640c535f5c) )
	EP_ITJB3_SOUND
ROM_END

ROM_START( ep_itjb3a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "itj3_dr.g0", 0x000000, 0x020000, CRC(1e62e440) SHA1(f7300a786efdb33d08aaa4348d93b8dd9707cbd2) )
	ROM_LOAD16_BYTE( "itj3_dr.g1", 0x000001, 0x020000, CRC(93086180) SHA1(e17647ebbe613fd830ad17d455840882e797f54a) )
	ROM_LOAD16_BYTE( "itj3_dr.g2", 0x040000, 0x020000, CRC(5a736057) SHA1(92c04470b204d290e7450a6dabf68ec260cb3a29) )
	ROM_LOAD16_BYTE( "itj3_dr.g3", 0x040001, 0x020000, CRC(3c5a59a3) SHA1(a9507daa3af93f44ae829d4ce94581f0ba633c8b) )
	EP_ITJB3_SOUND
ROM_END

#define EP_MERLN_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "merlsnd0.bin", 0x000000, 0x080000, CRC(f0841966) SHA1(d67446723735a780675590ca05a19de58da1f6f2) ) \
	ROM_LOAD( "merlsnd1.bin", 0x080000, 0x080000, CRC(e7de17a6) SHA1(586d9843b4c235027f6f9a40f1155248ffc54ea9) )
ROM_START( ep_merln )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "merlg0.bin", 0x000000, 0x020000, CRC(1d64717f) SHA1(a259228d31968381da09786b3b2a9fabfb384b08) )
	ROM_LOAD16_BYTE( "merlg1.bin", 0x000001, 0x020000, CRC(38bcf0f2) SHA1(09ea6e4bc52274ce0de506e283b5a418ca7bd905) )
	EP_MERLN_SOUND
ROM_END




#define EP_WIPEO_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "wipeoutsnd0.bin", 0x000000, 0x080000, CRC(c8d75e9d) SHA1(fff911ad015f09cac5b92033a01de3775f47d87d) ) \
	ROM_LOAD( "wipeoutsnd1.bin", 0x080000, 0x080000, CRC(8e2332db) SHA1(c363bc254eb59f5fdee623bad49439a4cb19c746) )
ROM_START( ep_wipeo )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nwo18.g0", 0x000000, 0x020000, CRC(12c623e5) SHA1(3473adea1b4e93f1d848a8af996a7e18b3c0ee21) )
	ROM_LOAD16_BYTE( "i00nwo18.g1", 0x000001, 0x020000, CRC(1c768ee3) SHA1(19aed3ef546597b51e2c781791cf225e364da5c5) )
	ROM_LOAD16_BYTE( "i00nwo18.g2", 0x040000, 0x020000, CRC(7914c95c) SHA1(8c3a212cf4dd8bbbd58aad77052dda81719308f1) )
	ROM_LOAD16_BYTE( "i00nwo18.g3", 0x040001, 0x020000, CRC(0c45a675) SHA1(155c458620aa2ea8aaf4ff18a09b62df3f50b508) )
	EP_WIPEO_SOUND
ROM_END

ROM_START( ep_wipeoa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pwo19.g0", 0x000000, 0x020000, CRC(12c623e5) SHA1(3473adea1b4e93f1d848a8af996a7e18b3c0ee21) )
	ROM_LOAD16_BYTE( "i00pwo19.g1", 0x000001, 0x020000, CRC(f7e8dadb) SHA1(557492b26f25c98bbaf077a7e5e5e139567b5978) )
	ROM_LOAD16_BYTE( "i00pwo19.g2", 0x040000, 0x020000, CRC(7914c95c) SHA1(8c3a212cf4dd8bbbd58aad77052dda81719308f1) )
	ROM_LOAD16_BYTE( "i00pwo19.g3", 0x040001, 0x020000, CRC(0c45a675) SHA1(155c458620aa2ea8aaf4ff18a09b62df3f50b508) )
	EP_WIPEO_SOUND
ROM_END

ROM_START( ep_wipeob )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nwo14.g0", 0x000000, 0x020000, CRC(d1f446f4) SHA1(c0cf204c08f26a2231fc5a7f61d876d487fb41d7) )
	ROM_LOAD16_BYTE( "i00nwo14.g1", 0x000001, 0x020000, CRC(e5bf8da5) SHA1(6c6594a9258302449563de249caf34a10efdbd3c) )
	EP_WIPEO_SOUND
ROM_END

ROM_START( ep_wipeoc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nwo16.g0", 0x000000, 0x020000, CRC(d1f446f4) SHA1(c0cf204c08f26a2231fc5a7f61d876d487fb41d7) )
	ROM_LOAD16_BYTE( "i00nwo16.g1", 0x000001, 0x020000, CRC(e9f22394) SHA1(15123bc2cf06bd35a4eb60d31365b1f431ae4e02) )
	EP_WIPEO_SOUND
ROM_END

ROM_START( ep_wipeoe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pwo15.g0", 0x000000, 0x020000, CRC(d1f446f4) SHA1(c0cf204c08f26a2231fc5a7f61d876d487fb41d7) )
	ROM_LOAD16_BYTE( "i00pwo15.g1", 0x000001, 0x020000, CRC(0e21d99d) SHA1(6ee44e05ddd31747eec36604a34abc044dc12b00) )
	ROM_LOAD16_BYTE( "i00pwo17.g0", 0x000000, 0x020000, CRC(d1f446f4) SHA1(c0cf204c08f26a2231fc5a7f61d876d487fb41d7) )
	ROM_LOAD16_BYTE( "i00pwo17.g1", 0x000001, 0x020000, CRC(026c77ac) SHA1(deee91163e955dcc58d78fb3bad266dbf8036913) )
	EP_WIPEO_SOUND
ROM_END


#define EP_FUNNY_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "funnymoneysound0.bin", 0x000000, 0x080000, CRC(81354590) SHA1(f11e46ed0dadf98ffc2476ee993708cc15ffc7bd) ) \
	ROM_LOAD( "funnymoneysound1.bin", 0x080000, 0x080000, CRC(65b728e1) SHA1(aa5b8b20d33ffd834e58fa872e27594846a7567b) )

ROM_START( ep_funny )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gfm24_0.bin", 0x000000, 0x020000, CRC(682e19f1) SHA1(26f77ddb1e534c33475fef2504d6b751a51aa71c) )
	ROM_LOAD16_BYTE( "gfm24_1.bin", 0x000001, 0x020000, CRC(fdab3e70) SHA1(984374ca54ab68e1044b9102b71feadb8f80d24d) )
	ROM_LOAD16_BYTE( "gfm24_2.bin", 0x040000, 0x020000, CRC(1811820b) SHA1(9bb85243347d235cd05819f12fc98b4c5565b060) )
	ROM_LOAD16_BYTE( "gfm24_3.bin", 0x040001, 0x020000, CRC(f3fb667e) SHA1(a188f34953911f9ca06018ed4c90dba709392be2) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gfm25_0.bin", 0x000000, 0x020000, CRC(682e19f1) SHA1(26f77ddb1e534c33475fef2504d6b751a51aa71c) )
	ROM_LOAD16_BYTE( "gfm25_1.bin", 0x000001, 0x020000, CRC(16356a48) SHA1(f056634ca772cc81ebb40448a116481448a02e7f) )
	ROM_LOAD16_BYTE( "gfm25_2.bin", 0x040000, 0x020000, CRC(1811820b) SHA1(9bb85243347d235cd05819f12fc98b4c5565b060) )
	ROM_LOAD16_BYTE( "gfm25_3.bin", 0x040001, 0x020000, CRC(f3fb667e) SHA1(a188f34953911f9ca06018ed4c90dba709392be2) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnyb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfm23.g0", 0x000000, 0x020000, CRC(3a838ae7) SHA1(a0331d7bb8929305da4edafe320d5b40419bb3d3) )
	ROM_LOAD16_BYTE( "i99pfm23.g1", 0x000001, 0x020000, CRC(4b41e160) SHA1(9e2ab7d182a5cfc27a1cfdcc1a85395ef2a11105) )
	ROM_LOAD16_BYTE( "i99pfm23.g2", 0x040000, 0x020000, CRC(3f204dd8) SHA1(18e9c23bbb8935262a6ec7ff0dfd9897801858d5) )
	ROM_LOAD16_BYTE( "i99pfm23.g3", 0x040001, 0x020000, CRC(74a1c7fa) SHA1(aab57fdccf15ea0709655e245323ca2ecaf392bb) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnyc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfm18.g0", 0x000000, 0x020000, CRC(f71f8222) SHA1(51a7cdd47b674a639cb9de4e1252f17da9b583d9) )
	ROM_LOAD16_BYTE( "i99nfm18.g1", 0x000001, 0x020000, CRC(bc998d7b) SHA1(6e52ca91b1fc83345a76a2c68053ff0c9a996856) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnyd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfm20.g0", 0x000000, 0x020000, CRC(f71f8222) SHA1(51a7cdd47b674a639cb9de4e1252f17da9b583d9) )
	ROM_LOAD16_BYTE( "i99nfm20.g1", 0x000001, 0x020000, CRC(b0d4234a) SHA1(125cbaf66272da301ffb8984f4c0ea4f54a32aec) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnye )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfm22.g0", 0x000000, 0x020000, CRC(3a838ae7) SHA1(a0331d7bb8929305da4edafe320d5b40419bb3d3) )
	ROM_LOAD16_BYTE( "i99nfm22.g1", 0x000001, 0x020000, CRC(a0dfb558) SHA1(631850f51bee85eb4e5d5329a61ded048d2c3f82) )
	ROM_LOAD16_BYTE( "i99nfm22.g2", 0x040000, 0x020000, CRC(3f204dd8) SHA1(18e9c23bbb8935262a6ec7ff0dfd9897801858d5) )
	ROM_LOAD16_BYTE( "i99nfm22.g3", 0x040001, 0x020000, CRC(74a1c7fa) SHA1(aab57fdccf15ea0709655e245323ca2ecaf392bb) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnyf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfm19.g0", 0x000000, 0x020000, CRC(f71f8222) SHA1(51a7cdd47b674a639cb9de4e1252f17da9b583d9) )
	ROM_LOAD16_BYTE( "i99pfm19.g1", 0x000001, 0x020000, CRC(5707d943) SHA1(c64a29bfeb80ff4d8a7cef03261b40aea7e0b354) )
	EP_FUNNY_SOUND
ROM_END

ROM_START( ep_funnyg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfm21.g0", 0x000000, 0x020000, CRC(f71f8222) SHA1(51a7cdd47b674a639cb9de4e1252f17da9b583d9) )
	ROM_LOAD16_BYTE( "i99pfm21.g1", 0x000001, 0x020000, CRC(5b4a7772) SHA1(428a51a604c74baec068c50e9daf5355ab469483) )
	EP_FUNNY_SOUND
ROM_END

#define EP_BATHL_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "hellsnd.p0", 0x000000, 0x080000, CRC(c143174d) SHA1(7b86281a653d85e0020245e05db2489489c7fe55) ) \
	ROM_LOAD( "hellsnd.p1", 0x080000, 0x080000, CRC(44af956b) SHA1(49ed8fde3aec64304090be795eaae09a78b3619d) )

ROM_START( ep_bathl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell2-1n.p0", 0x000000, 0x020000, CRC(51101be4) SHA1(6263d14197818dedcbac2027416e8a8e41fc33d7) )
	ROM_LOAD16_BYTE( "hell2-1n.p1", 0x000001, 0x020000, CRC(fee90b27) SHA1(378e6f2d19dc37ec47dd6cbded084d1fd915af16) )
	ROM_LOAD16_BYTE( "hell2-1n.p2", 0x040000, 0x020000, CRC(bdb61bda) SHA1(c8c4b6cc1067c191bc936806adf59ad4c1c5dc99) )
	ROM_LOAD16_BYTE( "hell2-1n.p3", 0x040001, 0x020000, CRC(776f6d59) SHA1(f44734f7d110e8acc661daa4e6b33a5397bc13ce) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell2-1p.p0", 0x000000, 0x020000, CRC(e266b534) SHA1(a1b5ddee35a91d485cd8a053a68b5982cfd188a3) )
	ROM_LOAD16_BYTE( "hell2-1p.p1", 0x000001, 0x020000, CRC(64af55ba) SHA1(e9a3b15508bec74ad1e4f97e7b42afee9507492a) )
	ROM_LOAD16_BYTE( "hell2-1p.p2", 0x040000, 0x020000, CRC(bdb61bda) SHA1(c8c4b6cc1067c191bc936806adf59ad4c1c5dc99) )
	ROM_LOAD16_BYTE( "hell2-1p.p3", 0x040001, 0x020000, CRC(776f6d59) SHA1(f44734f7d110e8acc661daa4e6b33a5397bc13ce) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathlb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell2-2n.p0", 0x000000, 0x020000, CRC(cee93331) SHA1(9646202d6110136d9958479cd03a9a48a3efe77d) )
	ROM_LOAD16_BYTE( "hell2-2n.p1", 0x000001, 0x020000, CRC(9a324ce1) SHA1(80f54da671af500defd35f21d3ab68ff5eb8f25c) )
	ROM_LOAD16_BYTE( "hell2-2n.p2", 0x040000, 0x020000, CRC(62dc73fb) SHA1(25b69f83e9ca30b1517e6e243d2f99810ed6d55d) )
	ROM_LOAD16_BYTE( "hell2-2n.p3", 0x040001, 0x020000, CRC(429cf6c4) SHA1(63b237dbb13a3b9297d05de27da9e25c92442b25) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathlc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell2-2p.p0", 0x000000, 0x020000, CRC(027e41b0) SHA1(4a4dbc51e4f766ea0bd99dd6f0e76bb72803c14f) )
	ROM_LOAD16_BYTE( "hell2-2p.p1", 0x000001, 0x020000, CRC(ca437387) SHA1(8c22ef87722625c9ec12edeba9d3c24937a390e2) )
	ROM_LOAD16_BYTE( "hell2-2p.p2", 0x040000, 0x020000, CRC(62dc73fb) SHA1(25b69f83e9ca30b1517e6e243d2f99810ed6d55d) )
	ROM_LOAD16_BYTE( "hell2-2p.p3", 0x040001, 0x020000, CRC(429cf6c4) SHA1(63b237dbb13a3b9297d05de27da9e25c92442b25) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell3-1n.p0", 0x000000, 0x020000, CRC(f0f58f00) SHA1(a81fdda4a09522b6b6fc64db8d6ea1b7aa239503) )
	ROM_LOAD16_BYTE( "hell3-1n.p1", 0x000001, 0x020000, CRC(83ce311c) SHA1(e7292a03e20b85b3a87a0603c26f1869fedfd870) )
	ROM_LOAD16_BYTE( "hell3-1n.p2", 0x040000, 0x020000, CRC(3ea8dccb) SHA1(ffee9622684ae57d6a84d2a577c198dab65d3aa1) )
	ROM_LOAD16_BYTE( "hell3-1n.p3", 0x040001, 0x020000, CRC(e042d8e6) SHA1(e2d76c6803a1830c3e90587502cb65d5e1cd347a) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hell3-1p.p0", 0x000000, 0x020000, CRC(43c2783a) SHA1(2fa1235c99e74402ed1528fada4f9bb969a044e0) )
	ROM_LOAD16_BYTE( "hell3-1p.p1", 0x000001, 0x020000, CRC(5a4ef9e1) SHA1(8e1e91fdb0d704eaaebec61d35ea28718108a3ac) )
	ROM_LOAD16_BYTE( "hell3-1p.p2", 0x040000, 0x020000, CRC(3ea8dccb) SHA1(ffee9622684ae57d6a84d2a577c198dab65d3aa1) )
	ROM_LOAD16_BYTE( "hell3-1p.p3", 0x040001, 0x020000, CRC(e042d8e6) SHA1(e2d76c6803a1830c3e90587502cb65d5e1cd347a) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathlf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sbat4-1n.p0", 0x000000, 0x020000, CRC(ff6e1715) SHA1(2909f7e530e66fa689c70d774414a82cd1a1cc7c) )
	ROM_LOAD16_BYTE( "sbat4-1n.p1", 0x000001, 0x020000, CRC(951a7b3b) SHA1(db4a36ce22cbce2cc6407658e7b3062d37d559ad) )
	ROM_LOAD16_BYTE( "sbat4-1n.p2", 0x040000, 0x020000, CRC(f251e681) SHA1(2890e01269e49837302d69dfd88f2f780e0d9211) )
	ROM_LOAD16_BYTE( "sbat4-1n.p3", 0x040001, 0x020000, CRC(2b2fc591) SHA1(de194ed595a85c1bcf0c9cf469c84713aca279f8) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathlg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sbat4-1p.p0", 0x000000, 0x020000, CRC(a6bd02bf) SHA1(ed9b58bbcf85629c1c575ab2ab651b0096627148) )
	ROM_LOAD16_BYTE( "sbat4-1p.p1", 0x000001, 0x020000, CRC(6bbfca2f) SHA1(420c3a5632a0f69a2a92efd6fb9a1e408c390a80) )
	ROM_LOAD16_BYTE( "sbat4-1p.p2", 0x040000, 0x020000, CRC(f251e681) SHA1(2890e01269e49837302d69dfd88f2f780e0d9211) )
	ROM_LOAD16_BYTE( "sbat4-1p.p3", 0x040001, 0x020000, CRC(2b2fc591) SHA1(de194ed595a85c1bcf0c9cf469c84713aca279f8) )
	EP_BATHL_SOUND
ROM_END

ROM_START( ep_bathlh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bat_hell.g0", 0x000000, 0x020000, CRC(4c37ec64) SHA1(46d0b1b8d41c6c6b25db47733f6376ad5c5053e6) )
	ROM_LOAD16_BYTE( "bat_hell.g1", 0x000001, 0x020000, CRC(5d2c21a3) SHA1(bad44abe84e49255dde93247d5d350b5f766eff6) )
	ROM_LOAD16_BYTE( "bat_hell.g2", 0x040000, 0x020000, CRC(e9fb6023) SHA1(f7ca27633da6c0bd1586988570cd41aeff10cd99) )
	ROM_LOAD16_BYTE( "bat_hell.g3", 0x040001, 0x020000, CRC(ac20db53) SHA1(cfa700fa3bc718fe25736fb46ee2163b78d90cbb) )
	EP_BATHL_SOUND
ROM_END



#define EP_BEAVR_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "bblvsnd.p0", 0x000000, 0x100000, CRC(eeb9d956) SHA1(37961d32fe48d2815049ed12e10b61c533f5c610) )
// are these the same? check..
#define EP_BEAV3_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "blvesnd.p0", 0x000000, 0x080000, CRC(d7f5ad5a) SHA1(a73257a18ae3361b87fb62bf5dd9ceeb8c9ef8c3) ) \
	ROM_LOAD( "blvesnd.p1", 0x080000, 0x080000, CRC(bb892e3e) SHA1(536cc315e6b2d2f716becb938ef3533ffceaae4d) )

ROM_START( ep_beavr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-1g.p0", 0x000000, 0x020000, CRC(7365c84b) SHA1(f18f11537a110ed354bc34428b6491a4ce61bb8a) )
	ROM_LOAD16_BYTE( "bblv0-1g.p1", 0x000001, 0x020000, CRC(553863f2) SHA1(b3ad7295c3acf10dae459563f14f2f637a506dc0) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-1n.p0", 0x000000, 0x020000, CRC(ad45219e) SHA1(52e6b457edad98a5617fe08c9c8d1adbc5bad431) )
	ROM_LOAD16_BYTE( "bblv0-1n.p1", 0x000001, 0x020000, CRC(11d84dce) SHA1(b15f6853a98f7b280d3ac95b41f542ee8d549954) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-1p.p0", 0x000000, 0x020000, CRC(2e8e74eb) SHA1(d8dc3d79877478ed9b29e1b8662eb810f6b55cc5) )
	ROM_LOAD16_BYTE( "bblv0-1p.p1", 0x000001, 0x020000, CRC(ed3835a1) SHA1(1a5a6d4addb584938f31126e2cf335c6aa3c9529) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-2n.p0", 0x000000, 0x020000, CRC(1c5aa572) SHA1(25e51813240b3688b2749affd36fd9f6286e552a) )
	ROM_LOAD16_BYTE( "bblv0-2n.p1", 0x000001, 0x020000, CRC(3992c997) SHA1(a2668048db2d1b784c31d874f9ea0fcb3a813954) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-2p.p0", 0x000000, 0x020000, CRC(a039bbe3) SHA1(3674e674cb1398d470389a4783eec6aacb4988a8) )
	ROM_LOAD16_BYTE( "bblv0-2p.p1", 0x000001, 0x020000, CRC(e3461c69) SHA1(6495891056c079526176393a3f0af28741360921) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavre )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-3g.p0", 0x000000, 0x020000, CRC(a0ac030a) SHA1(2c16f3c95beea8df929c1550999a624620ef13c7) )
	ROM_LOAD16_BYTE( "bblv0-3g.p1", 0x000001, 0x020000, CRC(bc365c9a) SHA1(f08f89742cf6adf7257abaaccb2b7f55431dc42a) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-3n.p0", 0x000000, 0x020000, CRC(83a824fe) SHA1(132e460e1e93171d0ef6b65e0e6c716ca6cb365c) )
	ROM_LOAD16_BYTE( "bblv0-3n.p1", 0x000001, 0x020000, CRC(1ead18b9) SHA1(3fa595782ca04cddc631eca6efbf2fe90df20d07) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-3p.p0", 0x000000, 0x020000, CRC(3fcb3a6f) SHA1(4678eb1d5d85f2040b9905426138848ab405f819) )
	ROM_LOAD16_BYTE( "bblv0-3p.p1", 0x000001, 0x020000, CRC(7004b527) SHA1(bdee570b7da869d53a5803f361a56da94138dbd5) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-4g.p0", 0x000000, 0x020000, CRC(54b51ebb) SHA1(72a33b425f195427e67c8b531072734250331492) )
	ROM_LOAD16_BYTE( "bblv0-4g.p1", 0x000001, 0x020000, CRC(3826a9fa) SHA1(1d30631e9d37f6d2a4b5f3bddb95da333d0efe88) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavri )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-4n.p0", 0x000000, 0x020000, CRC(7e46701e) SHA1(507435eb2052837de157a56550aa82c8bbee3721) )
	ROM_LOAD16_BYTE( "bblv0-4n.p1", 0x000001, 0x020000, CRC(6fe70bc8) SHA1(c9bf28d1b968305996e0a63e7eebd80ddb2ff6e0) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv0-4p.p0", 0x000000, 0x020000, CRC(c2256e8f) SHA1(97229ee28d8e6048f4a8d74b64c5ffa955d75f6b) )
	ROM_LOAD16_BYTE( "bblv0-4p.p1", 0x000001, 0x020000, CRC(13613b74) SHA1(ee88c3100df2e38089d5d97cdf9016d4d9491faa) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv2-0g.p0", 0x000000, 0x020000, CRC(2c20bf5e) SHA1(7c77f0f954e1306c5ecadce38999a2a864f3819a) )
	ROM_LOAD16_BYTE( "bblv2-0g.p1", 0x000001, 0x020000, CRC(cdad5754) SHA1(b596078932dd87d7285ae7bb0164e5c5b4673721) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv2-0n.p0", 0x000000, 0x020000, CRC(cc05fe04) SHA1(751c740a70d66a917b89f5c9f7079e22e4b93e8f) )
	ROM_LOAD16_BYTE( "bblv2-0n.p1", 0x000001, 0x020000, CRC(9bb2b0a3) SHA1(0b4edc8ec9819069554ac3e085b4fa76b68fc77d) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bblv2-0p.p0", 0x000000, 0x020000, CRC(ac814c3e) SHA1(fc2db1d358470da43f8fa48be81a279a5c478404) )
	ROM_LOAD16_BYTE( "bblv2-0p.p1", 0x000001, 0x020000, CRC(c1136dae) SHA1(18fe9d134ef28f6fbc51e792e11dd4fab5373356) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavrn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15bl2-3n.p0", 0x000000, 0x020000, CRC(08ea1f13) SHA1(5dc6c4a6b8cb293fa27e4ca7c186f2401de0d919) )
	ROM_LOAD16_BYTE( "15bl2-3n.p1", 0x000001, 0x020000, CRC(f14226f5) SHA1(5a55d1d78edab771337221b60ebaf53bd58708f6) )
	ROM_LOAD16_BYTE( "15bl2-3n.p2", 0x040000, 0x020000, CRC(8ad7e47f) SHA1(308da5ec31cc1632ddf9a91743730ccb1ad2db6a) )
	ROM_LOAD16_BYTE( "15bl2-3n.p3", 0x040001, 0x020000, CRC(f6cb0366) SHA1(d224553fe374f2d1b474174ea13cef0d1562bc78) )
	EP_BEAVR_SOUND
ROM_END

ROM_START( ep_beavro )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15bl2-3p.p0", 0x000000, 0x020000, CRC(5beb6c71) SHA1(bfaba1fadb0e30c17c7cb4090ed35af322608866) )
	ROM_LOAD16_BYTE( "15bl2-3p.p1", 0x000001, 0x020000, CRC(c8bbdfd3) SHA1(8e854fe072a9bfe50d275722edbc4380ab10a792) )
	ROM_LOAD16_BYTE( "15bl2-3p.p2", 0x040000, 0x020000, CRC(8ad7e47f) SHA1(308da5ec31cc1632ddf9a91743730ccb1ad2db6a) )
	ROM_LOAD16_BYTE( "15bl2-3p.p3", 0x040001, 0x020000, CRC(f6cb0366) SHA1(d224553fe374f2d1b474174ea13cef0d1562bc78) )
	EP_BEAVR_SOUND
ROM_END


ROM_START( ep_beav3 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "blve3-3n.p0", 0x000000, 0x020000, CRC(3983f66e) SHA1(f9db844bb2d03e3036c4d9f04c6c1516a25b0167) )
	ROM_LOAD16_BYTE( "blve3-3n.p1", 0x000001, 0x020000, CRC(240a39a1) SHA1(bc8922f3696334d1166f181dcfa19b9e67ee1ff5) )
	ROM_LOAD16_BYTE( "blve3-3n.p2", 0x040000, 0x020000, CRC(a6264d4c) SHA1(7ae70513fe7aba975f04ae2f4d6b40a3bd0ed417) )
	ROM_LOAD16_BYTE( "blve3-3n.p3", 0x040001, 0x020000, CRC(3b5513b9) SHA1(3030fa6b1294d7eb9b89c39dd56fb652a8f6a5d0) )
	EP_BEAV3_SOUND
ROM_END

ROM_START( ep_beav3a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "blve3-3p.p0", 0x000000, 0x020000, CRC(666f7861) SHA1(7dff6b56ca175844c68a071a9ddb36264eed10a1) )
	ROM_LOAD16_BYTE( "blve3-3p.p1", 0x000001, 0x020000, CRC(af90dcc4) SHA1(86ebc929b25fdf7d8a775871a9c00708f0686126) )
	ROM_LOAD16_BYTE( "blve3-3p.p2", 0x040000, 0x020000, CRC(a6264d4c) SHA1(7ae70513fe7aba975f04ae2f4d6b40a3bd0ed417) )
	ROM_LOAD16_BYTE( "blve3-3p.p3", 0x040001, 0x020000, CRC(3b5513b9) SHA1(3030fa6b1294d7eb9b89c39dd56fb652a8f6a5d0) )
	EP_BEAV3_SOUND
ROM_END

ROM_START( ep_beav3b )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "blve4-2n.p0", 0x000000, 0x020000, CRC(6971889d) SHA1(b83f9ec2f3b5ebb537bbfda7d20c6ee3a351c72c) )
	ROM_LOAD16_BYTE( "blve4-2n.p1", 0x000001, 0x020000, CRC(daeb3d68) SHA1(9c1f51924c36f06f07dcebd3920ceb478628e436) )
	ROM_LOAD16_BYTE( "blve4-2n.p2", 0x040000, 0x020000, CRC(0f9a32c2) SHA1(8bfd1cc5affa16bcfeed70217c3e4fdcb4548877) )
	ROM_LOAD16_BYTE( "blve4-2n.p3", 0x040001, 0x020000, CRC(f845dbd0) SHA1(0b2fa1dad8d1963635905a22acbaf9ba75800def) )
	EP_BEAV3_SOUND
ROM_END

ROM_START( ep_beav3c )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "blve4-2p.p0", 0x000000, 0x020000, CRC(c35af6e5) SHA1(7ddca62b6eb08f44673d2fb6638da216e66173ef) )
	ROM_LOAD16_BYTE( "blve4-2p.p1", 0x000001, 0x020000, CRC(a122f2a8) SHA1(bc9947c108a2ca9f0427101f158be90556b2d84a) )
	ROM_LOAD16_BYTE( "blve4-2p.p2", 0x040000, 0x020000, CRC(0f9a32c2) SHA1(8bfd1cc5affa16bcfeed70217c3e4fdcb4548877) )
	ROM_LOAD16_BYTE( "blve4-2p.p3", 0x040001, 0x020000, CRC(f845dbd0) SHA1(0b2fa1dad8d1963635905a22acbaf9ba75800def) )
	EP_BEAV3_SOUND
ROM_END


#define EP_BVRUC_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "bvunsnd.p0", 0x000000, 0x080000, CRC(422ddcd3) SHA1(fc0f5b19e32a47ab04bc76f23c2839a823dc6836) ) \
	ROM_LOAD( "bvunsnd.p1", 0x080000, 0x080000, CRC(beb1de17) SHA1(084ff32a0b7a061f61d4dcc6d7cffe3d0af588ef) )

ROM_START( ep_bvruc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bvun1-4n.p0", 0x000000, 0x020000, CRC(90f8e413) SHA1(3279efa0f15c2d39f670f8e5a7c33a703ab87d01) )
	ROM_LOAD16_BYTE( "bvun1-4n.p1", 0x000001, 0x020000, CRC(efa6c3f2) SHA1(0ceb9eda1d4515db329db18cfb142f5bd8e29248) )
	ROM_LOAD16_BYTE( "bvun1-4n.p2", 0x040000, 0x020000, CRC(208ecce8) SHA1(ff5d029af3275773e09a55bbc1db66aa561d34e7) )
	ROM_LOAD16_BYTE( "bvun1-4n.p3", 0x040001, 0x020000, CRC(0ae67959) SHA1(3fd3f753c1cc29f6849a7141bb5c9fce75a018d5) )
	EP_BVRUC_SOUND
ROM_END


ROM_START( ep_bvruca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bvun1-4p.p0", 0x000000, 0x020000, CRC(522625f8) SHA1(b94154bf4265f34b3ac28dcf12b68c51b29a9d60) )
	ROM_LOAD16_BYTE( "bvun1-4p.p1", 0x000001, 0x020000, CRC(3e4820f9) SHA1(20146e436065e6fc87d27e829842fd6c35f4acd7) )
	ROM_LOAD16_BYTE( "bvun1-4p.p2", 0x040000, 0x020000, CRC(208ecce8) SHA1(ff5d029af3275773e09a55bbc1db66aa561d34e7) )
	ROM_LOAD16_BYTE( "bvun1-4p.p3", 0x040001, 0x020000, CRC(0ae67959) SHA1(3fd3f753c1cc29f6849a7141bb5c9fce75a018d5) )
	EP_BVRUC_SOUND
ROM_END

ROM_START( ep_bvrucb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "b_uncovr.g0", 0x000000, 0x020000, CRC(902b741f) SHA1(1f55f45a46444cfb0f91aa4f47b99d091844a19f) )
	ROM_LOAD16_BYTE( "b_uncove.g1", 0x000001, 0x020000, CRC(34635292) SHA1(edaa0b71a1a4d50e43e75638327e3422738d78ac) )
	ROM_LOAD16_BYTE( "b_uncove.g2", 0x040000, 0x020000, CRC(be9501f5) SHA1(939b7c4776061ed209846e59c3c7b25a590eb005) )
	ROM_LOAD16_BYTE( "b_uncove.g3", 0x040001, 0x020000, CRC(05250939) SHA1(f9a3056a4e7b9fcfc8d777166f6682dc703ae4aa) )
	EP_BVRUC_SOUND
ROM_END


ROM_START( ep_bvrucc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	/* this might be a mismatched set.. */
	ROM_LOAD16_BYTE( "b_uncovr.go", 0x000000, 0x020000, CRC(72ff0f24) SHA1(85d232c6aa47fb55b823ad69114512e23c2a8174) )
	ROM_LOAD16_BYTE( "g1",          0x000001, 0x020000, NO_DUMP )
	ROM_LOAD16_BYTE( "uncoverd.g2", 0x040000, 0x020000, CRC(31cfd4b2) SHA1(f215899f9de58a1d7200acfdc92e6c65a5f88c71) )
	ROM_LOAD16_BYTE( "uncoverd.g3", 0x040001, 0x020000, CRC(e7ef6191) SHA1(3e0f4149b36286a6f2efd546d1c44385f2233421) )
	EP_BVRUC_SOUND
ROM_END


#define EP_BJCLB_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "bkjksnd.p0", 0x000000, 0x0ab42c, BAD_DUMP CRC(8f574d3f) SHA1(3ce26f0bd7e88baddcf80abeebf190eb2a5f8333) )

ROM_START( ep_bjclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bkjk0-1n.p0", 0x000000, 0x020000, CRC(a031107f) SHA1(ac6437824193661bffcc0cc5a29cf1d2786615ca) )
	ROM_LOAD16_BYTE( "bkjk0-1n.p1", 0x000001, 0x020000, CRC(2db7c862) SHA1(3d729128fe93e0400fc0b683fd9189342db7cf89) )
	EP_BJCLB_SOUND
ROM_END


#define EP_CBRUN_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cannsnd.p0", 0x000000, 0x080000, CRC(435cc88e) SHA1(d1d4818ecc82580ee5d2c478afa595c38ebe3a18) ) \
	ROM_LOAD( "cannsnd.p1", 0x080000, 0x080000, CRC(86623111) SHA1(8515d5bb51266c472c4e87408bf112bec55015f7) )
ROM_START( ep_cbrun )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann2-2n.p0", 0x000000, 0x020000, CRC(7a896390) SHA1(18e70e8687bc0fc40d913619887a9f8baf233c61) )
	ROM_LOAD16_BYTE( "cann2-2n.p1", 0x000001, 0x020000, CRC(ba94a395) SHA1(82def4f1d779c6f433d96f6c604b3fd6846ddd3e) )
	ROM_LOAD16_BYTE( "cann2-2n.p2", 0x040000, 0x020000, CRC(c45a5577) SHA1(7aafc4d68e04b6c6d7ea09fbab084e3e2fc93d4a) )
	ROM_LOAD16_BYTE( "cann2-2n.p3", 0x040001, 0x020000, CRC(a02bf498) SHA1(15f3250afc5e5567d658baf43b279bf448374c63) )
	EP_CBRUN_SOUND
ROM_END

ROM_START( ep_cbruna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann2-2p.p0", 0x000000, 0x020000, CRC(b857a27b) SHA1(051cba9eef10269a70a88ea3be279d98fb24f9d1) )
	ROM_LOAD16_BYTE( "cann2-2p.p1", 0x000001, 0x020000, CRC(a6823856) SHA1(b2a90a1dd983b36f39a8ed2adb8408c0e2b9b7f8) )
	ROM_LOAD16_BYTE( "cann2-2p.p2", 0x040000, 0x020000, CRC(c45a5577) SHA1(7aafc4d68e04b6c6d7ea09fbab084e3e2fc93d4a) )
	ROM_LOAD16_BYTE( "cann2-2p.p3", 0x040001, 0x020000, CRC(a02bf498) SHA1(15f3250afc5e5567d658baf43b279bf448374c63) )
	EP_CBRUN_SOUND
ROM_END

ROM_START( ep_cbrunb ) // why are these larger?
	ROM_REGION( 0x400000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann2-4n.p0", 0x000000, 0x100000, CRC(17e1fa77) SHA1(cf06f0188036fc6501157f455d874ea061b195df) )
	ROM_LOAD16_BYTE( "cann2-4n.p1", 0x000001, 0x100000, CRC(df6c7a5a) SHA1(02a2d9dc2304555b1add0953ee94c72b3a6efa54) )
	ROM_LOAD16_BYTE( "cann2-4n.p2", 0x200000, 0x100000, CRC(19a1a7c2) SHA1(dab5b084d8b1ba21c31d7fb1ff052c81797c66c6) )
	ROM_LOAD16_BYTE( "cann2-4n.p3", 0x200001, 0x100000, CRC(0fcc8bfc) SHA1(311fe6283df3520bbe1695357df6ed4a3e0f0bd7) )
	EP_CBRUN_SOUND
ROM_END

ROM_START( ep_cbrunc ) // why are these larger?
	ROM_REGION( 0x400000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann2-4p.p0", 0x000000, 0x100000, CRC(80ac9c42) SHA1(a3fc5695a8bc32c07e257dfd3cf16286739f121f) )
	ROM_LOAD16_BYTE( "cann2-4p.p1", 0x000001, 0x100000, CRC(f7882ac2) SHA1(17c22fe7e5fd711aea306414703059d582dbd9b1) )
	ROM_LOAD16_BYTE( "cann2-4p.p2", 0x200000, 0x100000, CRC(19a1a7c2) SHA1(dab5b084d8b1ba21c31d7fb1ff052c81797c66c6) )
	ROM_LOAD16_BYTE( "cann2-4p.p3", 0x200001, 0x100000, CRC(0fcc8bfc) SHA1(311fe6283df3520bbe1695357df6ed4a3e0f0bd7) )
	EP_CBRUN_SOUND
ROM_END

ROM_START( ep_cbrund )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann3-1n.p0", 0x000000, 0x020000, CRC(1f220802) SHA1(220539f3dba2ed69ca389306f786166a9ef6a2cb) )
	ROM_LOAD16_BYTE( "cann3-1n.p1", 0x000001, 0x020000, CRC(c72d898e) SHA1(1fc826951be59f8a97c278254729d34567d57126) )
	ROM_LOAD16_BYTE( "cann3-1n.p2", 0x040000, 0x020000, CRC(d6436222) SHA1(053c8b6c0692d992f2e82e6f7ef238b7c8a941ee) )
	ROM_LOAD16_BYTE( "cann3-1n.p3", 0x040001, 0x020000, CRC(fdb37ccf) SHA1(e79ec72efbbb196a9193990a6c12a54107ee91bd) )
	EP_CBRUN_SOUND
ROM_END

ROM_START( ep_cbrune )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cann3-1p.p0", 0x000000, 0x020000, CRC(696db00a) SHA1(b616c5d0fa03b12e3f64984fe2e3bc8f2da7f819) )
	ROM_LOAD16_BYTE( "cann3-1p.p1", 0x000001, 0x020000, CRC(f43ff7b7) SHA1(67854994a0513953f2f40f7a65498187175f8520) )
	ROM_LOAD16_BYTE( "cann3-1p.p2", 0x040000, 0x020000, CRC(d6436222) SHA1(053c8b6c0692d992f2e82e6f7ef238b7c8a941ee) )
	ROM_LOAD16_BYTE( "cann3-1p.p3", 0x040001, 0x020000, CRC(fdb37ccf) SHA1(e79ec72efbbb196a9193990a6c12a54107ee91bd) )
	EP_CBRUN_SOUND
ROM_END


#define EP_CBRCL_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "ccansnd.p0", 0x000000, 0x100000, CRC(d0dc482a) SHA1(8ca7fdcf0fcd5d451c6d68269b1fef681095be20) ) \
	ROM_LOAD( "ccansnd.p1", 0x100000, 0x100000, CRC(9cf088a5) SHA1(94bef848066050b0f04556f59cc859cb646dafc5) )

ROM_START( ep_cbrcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-0n.p0", 0x000000, 0x020000, CRC(681fd2e2) SHA1(b57e2da852aacd80908dc185180df19efad6affe) )
	ROM_LOAD16_BYTE( "ccan1-0n.p1", 0x000001, 0x020000, CRC(badf9e15) SHA1(ad5b3a06959346cabc31ed0c2581c8771ba4cdb7) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-0p.p0", 0x000000, 0x020000, CRC(badf9e15) SHA1(ad5b3a06959346cabc31ed0c2581c8771ba4cdb7) )
	ROM_LOAD16_BYTE( "ccan1-0p.p1", 0x000001, 0x020000, CRC(254e6007) SHA1(67b1fc4e24b5dd80b0b896b8f8d1a6eb8f39cd96) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-3n.p0", 0x000000, 0x020000, CRC(320b7659) SHA1(0f1d5cce365671c25ebd507ddb75b9b8c14a2763) )
	ROM_LOAD16_BYTE( "ccan1-3n.p1", 0x000001, 0x020000, CRC(eea89853) SHA1(9501ebea211dc1b9048c110bc9ac6a06734eabd7) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-3p.p0", 0x000000, 0x020000, CRC(e9ede649) SHA1(566059c9f324aac4e05e4f7e352e2e048cd668b4) )
	ROM_LOAD16_BYTE( "ccan1-3p.p1", 0x000001, 0x020000, CRC(eea89853) SHA1(9501ebea211dc1b9048c110bc9ac6a06734eabd7) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrcld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-6n.p0", 0x000000, 0x020000, CRC(761b5923) SHA1(fbf9a5f6003ab9c1161a8f998d2463645e4b2eb2) )
	ROM_LOAD16_BYTE( "ccan1-6n.p1", 0x000001, 0x020000, CRC(f5212423) SHA1(f61ac3b163e17e0b98525d5556a47a6a685174fa) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrcle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-6p.p0", 0x000000, 0x020000, CRC(28fd8dc5) SHA1(14d7034e23d4d020e2332c86eb6cfdb470a3e1a7) )
	ROM_LOAD16_BYTE( "ccan1-6p.p1", 0x000001, 0x020000, CRC(f5212423) SHA1(f61ac3b163e17e0b98525d5556a47a6a685174fa) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-7n.p0", 0x000000, 0x020000, CRC(40500979) SHA1(23e71b7c44095d9926eb5d3c98fa2c1ba75c0e32) )
	ROM_LOAD16_BYTE( "ccan1-7n.p1", 0x000001, 0x020000, CRC(d01248ef) SHA1(402cd1230828497ba9b6cd7898beefe6cff8800e) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-7p.p0", 0x000000, 0x020000, CRC(c2517134) SHA1(128654f2ad7665bf24bedcf93da85002a18837c5) )
	ROM_LOAD16_BYTE( "ccan1-7p.p1", 0x000001, 0x020000, CRC(d01248ef) SHA1(402cd1230828497ba9b6cd7898beefe6cff8800e) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-8n.p0", 0x000000, 0x020000, CRC(da8b597d) SHA1(84500a782b9d9e73bcffe62b02a792f8adf04167) )
	ROM_LOAD16_BYTE( "ccan1-8n.p1", 0x000001, 0x020000, CRC(4b33152a) SHA1(55b9de7855ea9da8219d48fc03556aa49b842b0d) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrcli )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan1-8p.p0", 0x000000, 0x020000, CRC(003c7d1d) SHA1(06bba369458d3090fc8b9e0ee4d51abccdd78fad) )
	ROM_LOAD16_BYTE( "ccan1-8p.p1", 0x000001, 0x020000, CRC(4b33152a) SHA1(55b9de7855ea9da8219d48fc03556aa49b842b0d) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan2-0n.p0", 0x000000, 0x020000, CRC(320ccd6d) SHA1(ba36af242a117a4c485e351ef1abbe2806805bc9) )
	ROM_LOAD16_BYTE( "ccan2-0n.p1", 0x000001, 0x020000, CRC(2d6c60d6) SHA1(01966d522f3b522515eb1a7ab948c12a2c78cc5e) )
	EP_CBRCL_SOUND
ROM_END

ROM_START( ep_cbrclk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccan2-0p.p0", 0x000000, 0x020000, CRC(126f1cc7) SHA1(1c919a690e493b5381aa03cf68f9c7ccbb0f1132) )
	ROM_LOAD16_BYTE( "ccan2-0p.p1", 0x000001, 0x020000, CRC(2d6c60d6) SHA1(01966d522f3b522515eb1a7ab948c12a2c78cc5e) )
	EP_CBRCL_SOUND
ROM_END



#define EP_CCLAS_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "casisnd.p0", 0x000000, 0x100000, CRC(7050e393) SHA1(363e939675d47fa234a6ec29be9032dba3cc25dc) )
ROM_START( ep_cclas )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "casi1-5n.p0", 0x000000, 0x020000, CRC(b834ddca) SHA1(4547414d0de2fd4445a08b9787b8ebe96b6b6d10) )
	ROM_LOAD16_BYTE( "casi1-5n.p1", 0x000001, 0x020000, CRC(00b25d52) SHA1(015f044735dd3946d047d22e972e98c5e1039883) )
	EP_CCLAS_SOUND
ROM_END

ROM_START( ep_cclasa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "casi1-5p.p0", 0x000000, 0x020000, CRC(53d6f32a) SHA1(27f5098978e964fa11ce1051ae0d3da415abeeea) )
	ROM_LOAD16_BYTE( "casi1-5p.p1", 0x000001, 0x020000, CRC(00b25d52) SHA1(015f044735dd3946d047d22e972e98c5e1039883) )
	EP_CCLAS_SOUND
ROM_END

#define EP_CGRED_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cgresnd.p0", 0x000000, 0x100000, CRC(25e60f16) SHA1(06cbdd97faedcc6c9a14e02d49787ee94d7a1bb9) )

ROM_START( ep_cgred )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cgre0-1n.p0", 0x000000, 0x020000, CRC(ebdab218) SHA1(a875bb81ab402b21c143c150fc190049227c3939) )
	ROM_LOAD16_BYTE( "cgre0-1n.p1", 0x000001, 0x020000, CRC(89432c0c) SHA1(6ecddf98315b9661c2f146934bb2878de63b44a0) )
	EP_CGRED_SOUND
ROM_END


ROM_START( ep_cgreda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cgre0-1p.p0", 0x000000, 0x020000, CRC(e121873e) SHA1(aa5d1cfe4821c75aea66d4000b4413162214b7c0) )
	ROM_LOAD16_BYTE( "cgre0-1p.p1", 0x000001, 0x020000, CRC(89432c0c) SHA1(6ecddf98315b9661c2f146934bb2878de63b44a0) )
	EP_CGRED_SOUND
ROM_END


#define EP_CSTRK_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "clcssnd.p0", 0x000000, 0x100000, CRC(93baa48d) SHA1(95098b5be5f354811fd2f6c460fd8b3f8849f868) ) \
	ROM_LOAD( "clcssnd.p1", 0x100000, 0x100000, CRC(33208356) SHA1(189333ad3d9180dd3785fb6ce72d808d0b0ed083) )
ROM_START( ep_cstrk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-0n.p0", 0x000000, 0x020000, CRC(1a7dd627) SHA1(bd1ba7f3448f799bb05fb15e2dca98f843001e3f) )
	ROM_LOAD16_BYTE( "clcs1-0n.p1", 0x000001, 0x020000, CRC(809c97b8) SHA1(5e94d6969beddc5c9a1f38367c0ce0e96eb8e6b8) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-0p.p0", 0x000000, 0x020000, CRC(05a9f3d5) SHA1(98f8e884c6d9da3cdc8af526649c7b5c9f5d827a) )
	ROM_LOAD16_BYTE( "clcs1-0p.p1", 0x000001, 0x020000, CRC(10798c45) SHA1(0501f87c406becf8b74fdeecf61fff1325a8d35a) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrkb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-1n.p0", 0x000000, 0x020000, CRC(9be8f18c) SHA1(4a383a86e62cd29dd0a3f7819927e20e427cf3b7) )
	ROM_LOAD16_BYTE( "clcs1-1n.p1", 0x000001, 0x020000, CRC(199d01bb) SHA1(9f57603fca8437e4d82e9bf2e23ebcf04bf1a440) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrkc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-1p.p0", 0x000000, 0x020000, CRC(278bef1d) SHA1(3f23bd35a8f9e92d38b3ae5250e56133b1b812b2) )
	ROM_LOAD16_BYTE( "clcs1-1p.p1", 0x000001, 0x020000, CRC(a9fca927) SHA1(767d48ca7266abc94ace521e3ea48ec7ca9e2ec2) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrkd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-2n.p0", 0x000000, 0x020000, CRC(7383bf87) SHA1(0003692acb5b20dd3a40dcdc07e3906ccefd5468) )
	ROM_LOAD16_BYTE( "clcs1-2n.p1", 0x000001, 0x020000, CRC(c9afa6ec) SHA1(fb5146ac92829084210443919e9ecc18feb466dd) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrke )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-2p.p0", 0x000000, 0x020000, CRC(c0c17c8c) SHA1(26d507ac2fcd4080fd6774f53ef65004dc87cdde) )
	ROM_LOAD16_BYTE( "clcs1-2p.p1", 0x000001, 0x020000, CRC(99e9b26d) SHA1(d5a09700ecfc9d015f354cfeb634a20655ca6e23) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrkf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-3n.p0", 0x000000, 0x020000, CRC(ee417433) SHA1(2912a21f6ae7898f8325a691717cfade9033c9bd) )
	ROM_LOAD16_BYTE( "clcs1-3n.p1", 0x000001, 0x020000, CRC(5f074c24) SHA1(683dd880d6d6395c5e8f5b7acba74f98c00627fd) )
	EP_CSTRK_SOUND
ROM_END

ROM_START( ep_cstrkg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clcs1-3p.p0", 0x000000, 0x020000, CRC(f19551c1) SHA1(4771d6aea6239b47fa5cac8c1841c125a1da0acf) )
	ROM_LOAD16_BYTE( "clcs1-3p.p1", 0x000001, 0x020000, CRC(d83468e0) SHA1(49df49c60efd511712b7df94089bca14b39dafc8) )
	EP_CSTRK_SOUND
ROM_END


#define EP_DDQCL_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cddqsnd.p0", 0x000000, 0x100000, CRC(bc5eec74) SHA1(29b7676a263c043428f3028759c6609468162bee) )
ROM_START( ep_ddqcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-0n.p0", 0x000000, 0x020000, CRC(2371b69e) SHA1(9594b92dd13881b92ec91778531812eb6b450119) )
	ROM_LOAD16_BYTE( "cddq1-0n.p1", 0x000001, 0x020000, CRC(fc70b37a) SHA1(57c5d4c7896e59688883292f2a10d42ba4be6f85) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-0p.p0", 0x000000, 0x020000, CRC(9f12a80f) SHA1(d8cb59ce03e8d5c0c60d023cc6ea7066ba109d40) )
	ROM_LOAD16_BYTE( "cddq1-0p.p1", 0x000001, 0x020000, CRC(15735ae2) SHA1(095d592502107837dc4205d4a5684c224a499b90) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-1n.p0", 0x000000, 0x020000, CRC(ed889aad) SHA1(56f61be276836eaeb397f42498746eac34304194) )
	ROM_LOAD16_BYTE( "cddq1-1n.p1", 0x000001, 0x020000, CRC(abb87d7c) SHA1(beca0359dc6b2e2089a50a6aee485dee706bcfba) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqclc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-1p.p0", 0x000000, 0x020000, CRC(637fde84) SHA1(bd7d1c7edf9890be8c2fb521f64a5e89668e98ac) )
	ROM_LOAD16_BYTE( "cddq1-1p.p1", 0x000001, 0x020000, CRC(69e2160a) SHA1(b8fd9173d3f0a90ad678211a2110b9cc41f28be9) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqcld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-2n.p0", 0x000000, 0x020000, CRC(4521517f) SHA1(fd0fa7af048808fd7387ad46e3f225bb64987a4e) )
	ROM_LOAD16_BYTE( "cddq1-2n.p1", 0x000001, 0x020000, CRC(8b8e9daa) SHA1(c038b91df07e422252810d3834cf4b88050a4342) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqcle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-2p.p0", 0x000000, 0x020000, CRC(5af5748d) SHA1(92a927951373efe1762baa5ad411dd81e0e67ee1) )
	ROM_LOAD16_BYTE( "cddq1-2p.p1", 0x000001, 0x020000, CRC(77ad4e35) SHA1(d31ec2c0012e8c8aa26fce865a567332fa537168) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqclf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-3n.p0", 0x000000, 0x020000, CRC(588710f3) SHA1(bd56a59b393857aa9fff58759de49e5e93b958bc) )
	ROM_LOAD16_BYTE( "cddq1-3n.p1", 0x000001, 0x020000, CRC(b4f68717) SHA1(845b10621a1633eda81e0c32ff440fa063380fe8) )
	EP_DDQCL_SOUND
ROM_END

ROM_START( ep_ddqclg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cddq1-3p.p0", 0x000000, 0x020000, CRC(e4e40e62) SHA1(cdc3058125f763043e44266a9dbc16dfcccbb1b4) )
	ROM_LOAD16_BYTE( "cddq1-3p.p1", 0x000001, 0x020000, CRC(a2bf546d) SHA1(b67772b80f5a429c4cfd0fc42ba0241df36b953f) )
	EP_DDQCL_SOUND
ROM_END



#define EP_GRUN_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	/* there appears to be an older game (maygay m1ab? hardware) of the same name, these samples MIGHT come from that. */ \
	ROM_LOAD( "gridrunnersound0.bin", 0x000000, 0x080000, CRC(dc8f0bbb) SHA1(a375eff2fa5d109491479e050c5267ae73f63ebc) ) \
	ROM_LOAD( "gridrunnersound1.bin", 0x080000, 0x080000, CRC(d965860b) SHA1(ceaabf092777c50035baf13545a0280047d530ae) ) \
	/* this is verified EPOCH, but has no partner */ \
	ROM_LOAD( "grid run.s0", 0x0000, 0x080000, CRC(9c8938ef) SHA1(f6bc299097c64984cb8356d62bf935582b86d6bc) )
ROM_START( ep_grun )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grid031n.p0", 0x000000, 0x020000, CRC(02eae1b1) SHA1(68311fc3237da30cf96b42ae0cf16342ecf9cb45) )
	ROM_LOAD16_BYTE( "grid031n.p1", 0x000001, 0x020000, CRC(28a2ea37) SHA1(1c562c6b93f324752ade39aa40664ccc6c459bcd) )
	ROM_LOAD16_BYTE( "grid031n.p2", 0x040000, 0x020000, CRC(0f2e2461) SHA1(51c8ee2bb352c7fe5af928017a2698debfe3aa26) )
	ROM_LOAD16_BYTE( "grid031n.p3", 0x040001, 0x020000, CRC(78aa85ae) SHA1(65587da39d0d62d2dd34422df182dcad15fe734c) )
	EP_GRUN_SOUND
ROM_END


ROM_START( ep_gruna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grid031p.p0", 0x000000, 0x020000, CRC(02eae1b1) SHA1(68311fc3237da30cf96b42ae0cf16342ecf9cb45) )
	ROM_LOAD16_BYTE( "grid031p.p1", 0x000001, 0x020000, CRC(c33cbe0f) SHA1(4565288a67713f51313d4a3a5573d449470f20f8) )
	ROM_LOAD16_BYTE( "grid031p.p2", 0x040000, 0x020000, CRC(0f2e2461) SHA1(51c8ee2bb352c7fe5af928017a2698debfe3aa26) )
	ROM_LOAD16_BYTE( "grid031p.p3", 0x040001, 0x020000, CRC(78aa85ae) SHA1(65587da39d0d62d2dd34422df182dcad15fe734c) )
	EP_GRUN_SOUND
ROM_END

#define EP_HEYBG_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "heybsnd.p0", 0x000000, 0x100000, CRC(b81e94b2) SHA1(2fc0d84096ce00bd7ff5fd5518ab53f8cd94f8b0) )

ROM_START( ep_heybg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-4n.p0", 0x000000, 0x020000, CRC(c238a4a5) SHA1(5c1e06f469548809b65e5de66efc48fec2e50afd) )
	ROM_LOAD16_BYTE( "heyb1-4n.p1", 0x000001, 0x020000, CRC(1a21aaa4) SHA1(8236fca1349eea63300c64397bc61a406fd799dd) )
	EP_HEYBG_SOUND
ROM_END

ROM_START( ep_heybga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-4p.p0", 0x000000, 0x020000, CRC(395f5789) SHA1(9483d89dfc99aeee20ef3528aeefcebcb72ef84b) )
	ROM_LOAD16_BYTE( "heyb1-4p.p1", 0x000001, 0x020000, CRC(1a21aaa4) SHA1(8236fca1349eea63300c64397bc61a406fd799dd) )
	EP_HEYBG_SOUND
ROM_END

ROM_START( ep_heybgb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-6n.p0", 0x000000, 0x020000, CRC(9ba118bc) SHA1(9f330488729d775e06dd6f29893ea2e980725a1f) )
	ROM_LOAD16_BYTE( "heyb1-6n.p1", 0x000001, 0x020000, CRC(fa238271) SHA1(715f97e6e971883003e7633fc0251b9cca5bcb51) )
	EP_HEYBG_SOUND
ROM_END

ROM_START( ep_heybgc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-6p.p0", 0x000000, 0x020000, CRC(da9c894a) SHA1(ef0b57a06fd40327738b5d977368d7f671c8ba31) )
	ROM_LOAD16_BYTE( "heyb1-6p.p1", 0x000001, 0x020000, CRC(fa238271) SHA1(715f97e6e971883003e7633fc0251b9cca5bcb51) )
	EP_HEYBG_SOUND
ROM_END

#define EP_SPIRT_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "hispsnd.p1", 0x000000, 0x080000, CRC(3d4b265b) SHA1(6a5ca56395c4c57b302aa6f0a066ceb54deedacc) ) \
	ROM_LOAD( "hispsnd.p2", 0x080000, 0x080000, CRC(57a60f8d) SHA1(4de7fbf14da9ca214af96307dccfae719571eab8) )

ROM_START( ep_spirt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hisp2-3n.p0", 0x000000, 0x020000, CRC(a421f3b7) SHA1(92831776dd74d7b82d6cf8831df4d383adc20dc3) )
	ROM_LOAD16_BYTE( "hisp2-3n.p1", 0x000001, 0x020000, CRC(f3a21e6b) SHA1(6b828e22fcf52a13930e42a3a14a280fa16e5b45) )
	ROM_LOAD16_BYTE( "hisp2-3n.p2", 0x040000, 0x020000, CRC(352cd84f) SHA1(9ca7cac21ef4e5b5ba85039882175ab01c3ada24) )
	ROM_LOAD16_BYTE( "hisp2-3n.p3", 0x040001, 0x020000, CRC(6df30f74) SHA1(d12de46ed7132087fd816dde4e1e636ab41bd4ff) )
	EP_SPIRT_SOUND
ROM_END

ROM_START( ep_spirta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hisp2-3p.p0", 0x000000, 0x020000, CRC(00052eb9) SHA1(4017ef8fd6a16d3f4ed94c8779b77f4989932930) )
	ROM_LOAD16_BYTE( "hisp2-3p.p1", 0x000001, 0x020000, CRC(27a1b96e) SHA1(740d5f0b83b2b561e0e22c1fbc640d53cfc4b2db) )
	ROM_LOAD16_BYTE( "hisp2-3p.p2", 0x040000, 0x020000, CRC(352cd84f) SHA1(9ca7cac21ef4e5b5ba85039882175ab01c3ada24) )
	ROM_LOAD16_BYTE( "hisp2-3p.p3", 0x040001, 0x020000, CRC(6df30f74) SHA1(d12de46ed7132087fd816dde4e1e636ab41bd4ff) )
	EP_SPIRT_SOUND
ROM_END

ROM_START( ep_spirtb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hi_spirt.g0", 0x000000, 0x020000, CRC(851aba4c) SHA1(bf57a89b80c629d1961de420d8aaece17c3eaf44) )
	ROM_LOAD16_BYTE( "hi_spirt.g1", 0x000001, 0x020000, CRC(a517df08) SHA1(9aea6a3a307446fc3603d72dc845d2e8b8dfb165) )
	ROM_LOAD16_BYTE( "hi_spirt.g2", 0x040000, 0x020000, CRC(428b8a47) SHA1(f0fe9419b064cb86ff3bc22403580a4796a06da4) )
	ROM_LOAD16_BYTE( "hi_spirt.g3", 0x040001, 0x020000, CRC(6d8dbdd8) SHA1(6f387c7ffbc608a7e1d49c6fa0895d05e97dd0b9) )
	EP_SPIRT_SOUND
ROM_END


#define EP_MONRT_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "monrsnd.p0", 0x000000, 0x100000, CRC(01532b1e) SHA1(233973c57e314ef04a3910a7c8c9eafeb288363c) ) \
	ROM_LOAD( "monrsnd.p1", 0x100000, 0x100000, CRC(3cb3021d) SHA1(aa5858fc2c7030bcf37240a074739014d0e8caca) )

ROM_START( ep_monrt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-0n.p0", 0x000000, 0x020000, CRC(0c622e9c) SHA1(974b30322a65da9ce1061002cd0ed6b5fd481a93) )
	ROM_LOAD16_BYTE( "monr1-0n.p1", 0x000001, 0x020000, CRC(413275da) SHA1(e1019d4039b52f9fd9cd3797d760e18766359a47) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-1n.p0", 0x000000, 0x020000, CRC(d24e67ab) SHA1(c829d380686129fdfff3ee01fac01e33af2f9f88) )
	ROM_LOAD16_BYTE( "monr1-1n.p1", 0x000001, 0x020000, CRC(dfec5636) SHA1(e18e94061a73b06c02574f273239398b26c2cb5e) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrtb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-1p.p0", 0x000000, 0x020000, CRC(74b1f3e3) SHA1(36cad892cdac9451c0475f56e4910f92258228d7) )
	ROM_LOAD16_BYTE( "monr1-1p.p1", 0x000001, 0x020000, CRC(dfec5636) SHA1(e18e94061a73b06c02574f273239398b26c2cb5e) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrtc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-2n.p0", 0x000000, 0x020000, CRC(cbe00b34) SHA1(ac636310ebe17513da4bec0038025dd3c7d12262) )
	ROM_LOAD16_BYTE( "monr1-2n.p1", 0x000001, 0x020000, CRC(4018f41d) SHA1(1ca50591fec203ee435e56aa0139e71cf512a6ad) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrtd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-2p.p0", 0x000000, 0x020000, CRC(6d1f9f7c) SHA1(2f3732a423e18fc4d7d19a528e7e56ec5804120b) )
	ROM_LOAD16_BYTE( "monr1-2p.p1", 0x000001, 0x020000, CRC(4018f41d) SHA1(1ca50591fec203ee435e56aa0139e71cf512a6ad) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrte )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-3n.p0", 0x000000, 0x020000, CRC(963e1a68) SHA1(56d2d8ded2051da0aad5eca2e93df51d3c0b1976) )
	ROM_LOAD16_BYTE( "monr1-3n.p1", 0x000001, 0x020000, CRC(96433193) SHA1(6c83be149a4d32de451643e4a32b7a0a2c6e44b2) )
	EP_MONRT_SOUND
ROM_END

ROM_START( ep_monrtf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "monr1-3p.p0", 0x000000, 0x020000, CRC(774f8b46) SHA1(b269a3578a58ce2f3af3ac09df9b03c7daa1eb9b) )
	ROM_LOAD16_BYTE( "monr1-3p.p1", 0x000001, 0x020000, CRC(96433193) SHA1(6c83be149a4d32de451643e4a32b7a0a2c6e44b2) )
	EP_MONRT_SOUND
ROM_END


#define EP_PKNI_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "phoesnd.p0", 0x000000, 0x080000, CRC(4128577f) SHA1(016098257aec8e7b4e9c535479303aef4896b03c) ) \
	ROM_LOAD( "phoesnd.p1", 0x080000, 0x080000, CRC(62b78d0d) SHA1(92284b04257ff498ff10c4b026e33b06214c9bb8) )

ROM_START( ep_pkni )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-1n.p0", 0x000000, 0x020000, CRC(3383b8ff) SHA1(3f374cdbe9dd8bcef4e10bcee9a62403c94c0f6d) )
	ROM_LOAD16_BYTE( "phoe1-1n.p1", 0x000001, 0x020000, CRC(4de697d4) SHA1(a68dd1f6c725c6fa757b41a535873fa56a3dd797) )
	ROM_LOAD16_BYTE( "phoe1-1n.p2", 0x040000, 0x020000, CRC(446ef21c) SHA1(82a389aeb7262b365ddb32a4a90087be9b975cab) )
	ROM_LOAD16_BYTE( "phoe1-1n.p3", 0x040001, 0x020000, CRC(03149121) SHA1(71238d91175b68d157573da5f95208831938d741) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknia )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-1p.p0", 0x000000, 0x020000, CRC(b00bf080) SHA1(524776405e397fbfb0666bdd3046b6307c785476) )
	ROM_LOAD16_BYTE( "phoe1-1p.p1", 0x000001, 0x020000, CRC(00015dc1) SHA1(0dd01d3b52dc92d7b841fbd4b905940d52f3c2bd) )
	ROM_LOAD16_BYTE( "phoe1-1p.p2", 0x040000, 0x020000, CRC(446ef21c) SHA1(82a389aeb7262b365ddb32a4a90087be9b975cab) )
	ROM_LOAD16_BYTE( "phoe1-1p.p3", 0x040001, 0x020000, CRC(03149121) SHA1(71238d91175b68d157573da5f95208831938d741) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknib )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-3n.p0", 0x000000, 0x020000, CRC(eb2dc046) SHA1(fb32609286e04eebc1ec42eac7e031efd27c6c41) )
	ROM_LOAD16_BYTE( "phoe1-3n.p1", 0x000001, 0x020000, CRC(0b67f57d) SHA1(be765917d7a1234c67c4c7a126cd7945e52efab0) )
	ROM_LOAD16_BYTE( "phoe1-3n.p2", 0x040000, 0x020000, CRC(2c1ae8bb) SHA1(6872105851a5b663d3a65cd446848d15435b2503) )
	ROM_LOAD16_BYTE( "phoe1-3n.p3", 0x040001, 0x020000, CRC(9c32358b) SHA1(a35a458515e5d6c34abab78d28b55d52c13969d6) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknic )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-3p.p0", 0x000000, 0x020000, CRC(db332061) SHA1(3b2ddfd20c60976ab2449425ada6a919fdc33659) )
	ROM_LOAD16_BYTE( "phoe1-3p.p1", 0x000001, 0x020000, CRC(248dae90) SHA1(02562e17398c3c4756da003dabbb1cf3d4e4fd20) )
	ROM_LOAD16_BYTE( "phoe1-3p.p2", 0x040000, 0x020000, CRC(2c1ae8bb) SHA1(6872105851a5b663d3a65cd446848d15435b2503) )
	ROM_LOAD16_BYTE( "phoe1-3p.p3", 0x040001, 0x020000, CRC(9c32358b) SHA1(a35a458515e5d6c34abab78d28b55d52c13969d6) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknid )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-4p.p0", 0x000000, 0x020000, CRC(bb212178) SHA1(359cf68c18ba1214da7aecba59979c5e6521f955) )
	ROM_LOAD16_BYTE( "phoe1-4p.p1", 0x000001, 0x020000, CRC(c05c0898) SHA1(aaaffc2d61cf7eba217a99072f104ca9a29c04d8) )
	ROM_LOAD16_BYTE( "phoe1-4p.p2", 0x040000, 0x020000, CRC(40d233b0) SHA1(f0a9131b9abcc088fedc555db538da5b795f386a) )
	ROM_LOAD16_BYTE( "phoe1-4p.p3", 0x040001, 0x020000, CRC(9c32358b) SHA1(a35a458515e5d6c34abab78d28b55d52c13969d6) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknie )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-7n.p0", 0x000000, 0x020000, CRC(65768707) SHA1(5ee87d064fc34db512370bf747a1c716d9c3721b) )
	ROM_LOAD16_BYTE( "phoe1-7n.p1", 0x000001, 0x020000, CRC(9cb1cf4b) SHA1(b4416e967c7859a9a82ae79d7666dfa85f5c2f7b) )
	ROM_LOAD16_BYTE( "phoe1-7n.p2", 0x040000, 0x020000, CRC(5408e7c9) SHA1(10f97cea24f88efa9807c41d07b2d4aea1389708) )
	ROM_LOAD16_BYTE( "phoe1-7n.p3", 0x040001, 0x020000, CRC(5577daf4) SHA1(003ef65296d30c062c67ea81010436a4efff2017) )
	EP_PKNI_SOUND
ROM_END

ROM_START( ep_pknif )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "phoe1-7p.p0", 0x000000, 0x020000, CRC(423a14fe) SHA1(8bdcef29d29b76456bdd10f46b0a7fde3dbd9661) )
	ROM_LOAD16_BYTE( "phoe1-7p.p1", 0x000001, 0x020000, CRC(9cc6a4ad) SHA1(e08795f1415b458d9fea5d938f4c4cf013c358e9) )
	ROM_LOAD16_BYTE( "phoe1-7p.p2", 0x040000, 0x020000, CRC(5408e7c9) SHA1(10f97cea24f88efa9807c41d07b2d4aea1389708) )
	ROM_LOAD16_BYTE( "phoe1-7p.p3", 0x040001, 0x020000, CRC(5577daf4) SHA1(003ef65296d30c062c67ea81010436a4efff2017) )
	EP_PKNI_SOUND
ROM_END

#define EP_SNBEV_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "snbvsnd.p0", 0x000000, 0x100000, CRC(18e21765) SHA1(dd50499c779dde72964117178e004093777a6edf) ) \
	ROM_LOAD( "snbvsnd.p1", 0x100000, 0x100000, CRC(73d65f34) SHA1(13d11a60eb579d1a99d28ba42915fdc45b0c1681) )
ROM_START( ep_snbev )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv1-8n.p0", 0x000000, 0x020000, CRC(f018f20a) SHA1(d9e28fa94ed08c06de65018bbe7169233a5fad91) )
	ROM_LOAD16_BYTE( "snbv1-8n.p1", 0x000001, 0x020000, CRC(203e1c29) SHA1(b6154fa8513dbca903237b317ac87c84a3ff9407) )
	ROM_LOAD16_BYTE( "snbv1-8n.p2", 0x040000, 0x020000, CRC(b9623f1f) SHA1(da4472a3af73cfc4f6b504f505e8b8a553943139) )
	ROM_LOAD16_BYTE( "snbv1-8n.p3", 0x040001, 0x020000, CRC(d1dade50) SHA1(1c114b34afccf72649d1189c2fc5ec22bdb4ff62) )
	EP_SNBEV_SOUND
ROM_END

ROM_START( ep_snbeva )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv1-8p.p0", 0x000000, 0x020000, CRC(3a0e4c3c) SHA1(0955e138544b76d1c73a8e01ebe4afe80446b3f4) )
	ROM_LOAD16_BYTE( "snbv1-8p.p1", 0x000001, 0x020000, CRC(94758479) SHA1(b57f2a443c1c0bd7ba130cfe98e2159437d544a9) )
	ROM_LOAD16_BYTE( "snbv1-8p.p2", 0x040000, 0x020000, CRC(b9623f1f) SHA1(da4472a3af73cfc4f6b504f505e8b8a553943139) )
	ROM_LOAD16_BYTE( "snbv1-8p.p3", 0x040001, 0x020000, CRC(d1dade50) SHA1(1c114b34afccf72649d1189c2fc5ec22bdb4ff62) )
	EP_SNBEV_SOUND
ROM_END

ROM_START( ep_snbevb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv1-9n.p0", 0x000000, 0x020000, CRC(0b7f28ce) SHA1(0ca306de831499af1da2603274106c96aabe768e) )
	ROM_LOAD16_BYTE( "snbv1-9n.p1", 0x000001, 0x020000, CRC(66dd1d9b) SHA1(1d65d6983b990caa2eb22148d784ec5b6a888cf8) )
	ROM_LOAD16_BYTE( "snbv1-9n.p2", 0x040000, 0x020000, CRC(9ef514ca) SHA1(65ca93ff673feecd3e00beea1a89b326e58f8b3b) )
	ROM_LOAD16_BYTE( "snbv1-9n.p3", 0x040001, 0x020000, CRC(e1b85282) SHA1(94c042792993d850c20cee91a417c28841d41635) )
	EP_SNBEV_SOUND
ROM_END

ROM_START( ep_snbevc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv1-9p.p0", 0x000000, 0x020000, CRC(6a62e318) SHA1(bcdb17177287888b656ef0c38d63641c96349f14) )
	ROM_LOAD16_BYTE( "snbv1-9p.p1", 0x000001, 0x020000, CRC(f9573228) SHA1(f8328975e166004a144f39c45128b49401e29009) )
	ROM_LOAD16_BYTE( "snbv1-9p.p2", 0x040000, 0x020000, CRC(9ef514ca) SHA1(65ca93ff673feecd3e00beea1a89b326e58f8b3b) )
	ROM_LOAD16_BYTE( "snbv1-9p.p3", 0x040001, 0x020000, CRC(e1b85282) SHA1(94c042792993d850c20cee91a417c28841d41635) )
	EP_SNBEV_SOUND
ROM_END

ROM_START( ep_snbevd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv2-1n.p0", 0x000000, 0x020000, CRC(82a93993) SHA1(a637305df7c3207212f130ecfb017763f3dc6319) )
	ROM_LOAD16_BYTE( "snbv2-1n.p1", 0x000001, 0x020000, CRC(575fdc22) SHA1(90aa4b46f464e8636568d9081010d88760f1b106) )
	ROM_LOAD16_BYTE( "snbv2-1n.p2", 0x040000, 0x020000, CRC(0997cad7) SHA1(e7671cae103c5909a8191cec2b26d7a7603a5e91) )
	ROM_LOAD16_BYTE( "snbv2-1n.p3", 0x040001, 0x020000, CRC(c21170a8) SHA1(eab143f98187410e7ea247c18b6acaf23b9b59bb) )
	EP_SNBEV_SOUND
ROM_END

ROM_START( ep_snbeve )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snbv2-1p.p0", 0x000000, 0x020000, CRC(4e1646ff) SHA1(8f45521bb87cf09ee2e623d2e5b1f89100199f7c) )
	ROM_LOAD16_BYTE( "snbv2-1p.p1", 0x000001, 0x020000, CRC(d51b6ab6) SHA1(5d7b5ce63f605dc9dbf49f38c6c99f09b698a344) )
	ROM_LOAD16_BYTE( "snbv2-1p.p2", 0x040000, 0x020000, CRC(0997cad7) SHA1(e7671cae103c5909a8191cec2b26d7a7603a5e91) )
	ROM_LOAD16_BYTE( "snbv2-1p.p3", 0x040001, 0x020000, CRC(c21170a8) SHA1(eab143f98187410e7ea247c18b6acaf23b9b59bb) )
	EP_SNBEV_SOUND
ROM_END





#define EP_SPOOK_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "spkysnd.p0", 0x000000, 0x080000, CRC(005a1383) SHA1(92d3400a89906c990ba2ab46a70ed1e3453e50fc) ) \
	ROM_LOAD( "spkysnd.p1", 0x080000, 0x080000, CRC(6258dceb) SHA1(e7b00318ff57affe8ac54cca54d1e095ab8cd40e) )

ROM_START( ep_spook )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "spky1-3n.p0", 0x000000, 0x020000, CRC(77fefb03) SHA1(d3214ebc83dedbb62bfbb8463257ced55e01bf8d) )
	ROM_LOAD16_BYTE( "spky1-3n.p1", 0x000001, 0x020000, CRC(e0b681b8) SHA1(f9923af9b52dc35ad25297c01260729b3dcbbcc7) )
	ROM_LOAD16_BYTE( "spky1-3n.p2", 0x040000, 0x020000, CRC(2b1235e9) SHA1(7046ee9db5fbbffab345d79e2f816516337f26be) )
	ROM_LOAD16_BYTE( "spky1-3n.p3", 0x040001, 0x020000, CRC(4ad25dff) SHA1(6b2b96493ed7c1aff7463476a47094ed7e66c3fd) )
	EP_SPOOK_SOUND
ROM_END

ROM_START( ep_spooka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "spky1-3p.p0", 0x000000, 0x020000, CRC(8722dce9) SHA1(7e599094090fb569e2e0055ed188c4dd48ea63d6) )
	ROM_LOAD16_BYTE( "spky1-3p.p1", 0x000001, 0x020000, CRC(b6a21486) SHA1(15ddbef273d22fa899e95eba12dc9562f0e3cfa6) )
	ROM_LOAD16_BYTE( "spky1-3p.p2", 0x040000, 0x020000, CRC(2b1235e9) SHA1(7046ee9db5fbbffab345d79e2f816516337f26be) )
	ROM_LOAD16_BYTE( "spky1-3p.p3", 0x040001, 0x020000, CRC(4ad25dff) SHA1(6b2b96493ed7c1aff7463476a47094ed7e66c3fd) )
	EP_SPOOK_SOUND
ROM_END

ROM_START( ep_spookb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "s_hollow.g0", 0x000000, 0x020000, CRC(449b03e1) SHA1(611c5b1d6aa68ebd622c10ecfcb109745e851744) )
	ROM_LOAD16_BYTE( "s_hollow.g1", 0x000001, 0x020000, CRC(f09e8ad5) SHA1(7586a10429345a73d8853ee5302efadb27c6bb9c) )
	ROM_LOAD16_BYTE( "s_hollow.g2", 0x040000, 0x020000, CRC(3b6103bc) SHA1(5c1576362910e587a63571150252bdd35193b5f9) )
	ROM_LOAD16_BYTE( "s_hollow.g3", 0x040001, 0x020000, CRC(6ac5b510) SHA1(a6a20da671b52f834d8b31322e438121ec002b3e) )
	EP_SPOOK_SOUND
ROM_END


#define EP_SPDCLB_SND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	/* these sound roms might not belong to this as there was another HW game in here */ \
	ROM_LOAD( "spotteddisksnd.bin", 0x000000, 0x100000, CRC(f2c66aab) SHA1(6fe94a193779c91711588365591cf42d197cb7b9) ) \
	ROM_LOAD( "sdclsnd.p1", 0x000001, 0x100000, CRC(487ed1e5) SHA1(3f9b0a4d0655906a8863765c26099c8871f1f8cf) ) \
	ROM_LOAD( "sdclsnd.p2", 0x100000, 0x100000, CRC(91f0a42a) SHA1(d0953f89f44267be216495586c5e5d2cbbf1e349) )


ROM_START( ep_sdclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sdcl1-3n.p1", 0x000000, 0x020000, CRC(85cab681) SHA1(bf6892e937189d7bd0a4eff4276661412aed6189) )
	ROM_LOAD16_BYTE( "sdcl1-3n.p2", 0x000001, 0x020000, CRC(aca34a7e) SHA1(3bfe6a5e7f5c54dd57159d63d9a0ae9bc56ee4d1) )
	EP_SPDCLB_SND
ROM_END


ROM_START( ep_sdclba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sdcl1-3p.p1", 0x000000, 0x020000, CRC(f7334f80) SHA1(5dd11223df6dba586c15a2e9004b43279895a39b) )
	ROM_LOAD16_BYTE( "sdcl1-3p.p2", 0x000001, 0x020000, CRC(aca34a7e) SHA1(3bfe6a5e7f5c54dd57159d63d9a0ae9bc56ee4d1) )
	EP_SPDCLB_SND
ROM_END


#define EP_STMCL_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "clsfsnd.p0", 0x000000, 0x100000, CRC(09860f64) SHA1(0c32828b912bc1a94bd8ab2dc77604803835fc40) ) \
	ROM_LOAD( "clsfsnd.p1", 0x100000, 0x100000, CRC(eb751795) SHA1(db28fd0c00b08cad6db0c7d6d1768705610de119) )

ROM_START( ep_stmcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsf1-4n.p0", 0x000000, 0x020000, CRC(1c4c4166) SHA1(e7efe86a6f1cbc1042b9c6c9bf14dc3aa9402652) )
	ROM_LOAD16_BYTE( "clsf1-4n.p1", 0x000001, 0x020000, CRC(d341b77f) SHA1(2fee73a428f16636a433f6551defd9f03e5390e9) )
	EP_STMCL_SOUND
ROM_END

ROM_START( ep_stmcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsf1-4p.p0", 0x000000, 0x020000, CRC(a02f5ff7) SHA1(ebf179ffc5409aa06d8e59a77bd43d763d041309) )
	ROM_LOAD16_BYTE( "clsf1-4p.p1", 0x000001, 0x020000, CRC(aacc34f7) SHA1(88ef44e36394b2b4739447b253ef889b96f9aff5) )
	EP_STMCL_SOUND
ROM_END

#define EP_UTNUT_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "anutsnd0.p0", 0x000000, 0x100000, CRC(bb9b9a9e) SHA1(f6aee7e059fab9698e763a4c42b6b30cda2d6e77) )
ROM_START( ep_utnut )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "anut1-2n.p0", 0x000000, 0x020000, CRC(26d767a9) SHA1(a919d8b7f4733d21e1456f9ba21d3a55fef5bad1) )
	ROM_LOAD16_BYTE( "anut1-2n.p1", 0x000001, 0x020000, CRC(3a04317d) SHA1(7cf7424f7f2d528e96b704016737e3e18e115f3a) )
	EP_UTNUT_SOUND
ROM_END

ROM_START( ep_utnuta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "anut1-2p.p0", 0x040000, 0x020000, CRC(9ab47938) SHA1(7cfc006f087eef32a819e81e8105824c12503bfc) )
	ROM_LOAD16_BYTE( "anut1-2p.p1", 0x040001, 0x020000, CRC(df3049a5) SHA1(e05ff09bc7020aa6197c7e1af2b2726f62e8cbae) )
	EP_UTNUT_SOUND
ROM_END

ROM_START( ep_utnutb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "casu0-6p.p0", 0x000000, 0x020000, CRC(8e1e3fab) SHA1(9efa598dcbde3c045f49835ca4f63e7b763109a3) )
	ROM_LOAD16_BYTE( "casu0-6p.p1", 0x000001, 0x020000, CRC(79fda268) SHA1(0148de6859eee4bf9341e059f466a6b8d47f5bad) )
	EP_UTNUT_SOUND
ROM_END

ROM_START( ep_utnutc ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "casu0-8n.p0", 0x000000, 0x100000, CRC(b3e1b4f2) SHA1(9a0b6be63167ad867d32238cb2b300a5927fe4a6) )
	ROM_LOAD16_BYTE( "casu0-8n.p1", 0x000001, 0x100000, CRC(ac6b826b) SHA1(f86701ab151a69a94353e4ab76f9d94c66ff76ad) )
	EP_UTNUT_SOUND
ROM_END


#define EP_MILHR_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "haresnd.p0", 0x000000, 0x080000, CRC(093220bb) SHA1(064bf28aa031222f85c3c74a2693ad06df231a1c) ) \
	ROM_LOAD( "haresnd.p1", 0x080000, 0x080000, CRC(38ddf39f) SHA1(eec475b3329dc0f82a19a8f4134249d0803e7bc1) ) \
	ROM_REGION( 0x100000, "altymz", 0 ) \
	ROM_LOAD( "sound.0", 0x000000, 0x080000, CRC(093220bb) SHA1(064bf28aa031222f85c3c74a2693ad06df231a1c) ) \
	ROM_LOAD( "sound.1", 0x080000, 0x080000, CRC(2eb13cda) SHA1(9343fe11a43e5d0213533cb81ad4a3c3d5476b85) )

ROM_START( ep_milhr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-2n.p0", 0x000000, 0x020000, CRC(334b02a1) SHA1(497fe9c078993ee7d01dd3932944708bdbc1db35) )
	ROM_LOAD16_BYTE( "hare1-2n.p1", 0x000001, 0x020000, CRC(716d780f) SHA1(7dc36a948ac891cece8b7cd58acf861b2beeb622) )
	ROM_LOAD16_BYTE( "hare1-2n.p2", 0x040000, 0x020000, CRC(17816d55) SHA1(543fb2f09e5848396e2bcd8d26f64fecaccfc206) )
	ROM_LOAD16_BYTE( "hare1-2n.p3", 0x040001, 0x020000, CRC(8dff64a4) SHA1(1a54937833edf243fdf6fbd166ac46322fdeed0f) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-2p.p0", 0x000000, 0x020000, CRC(f69212f1) SHA1(5bd4515e17ddc8627beaba1e1ca4afd2cc5fb527) )
	ROM_LOAD16_BYTE( "hare1-2p.p1", 0x000001, 0x020000, CRC(6d288965) SHA1(3b636d8541d4b29786371d53494fce07fdd7bbf9) )
	ROM_LOAD16_BYTE( "hare1-2p.p2", 0x040000, 0x020000, CRC(17816d55) SHA1(543fb2f09e5848396e2bcd8d26f64fecaccfc206) )
	ROM_LOAD16_BYTE( "hare1-2p.p3", 0x040001, 0x020000, CRC(8dff64a4) SHA1(1a54937833edf243fdf6fbd166ac46322fdeed0f) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhrb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-3n.p0", 0x000000, 0x020000, CRC(52ea154a) SHA1(1705b26838cb7b0fefb2d696dab0de4416152f14) )
	ROM_LOAD16_BYTE( "hare1-3n.p1", 0x000001, 0x020000, CRC(35e09d9d) SHA1(2bd98ad0205b85d17f394c9c0cc9a9727eac9068) )
	ROM_LOAD16_BYTE( "hare1-3n.p2", 0x040000, 0x020000, CRC(e6f96a6b) SHA1(a4c084e9488c99aca1560999e6f0daf61c456fb3) )
	ROM_LOAD16_BYTE( "hare1-3n.p3", 0x040001, 0x020000, CRC(9c57b62c) SHA1(9a82ffb7b9084cc618fb53533a7e567fcebbc4ea) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-3p.p0", 0x000000, 0x020000, CRC(e69b6a21) SHA1(40fb03c449881eef61546c73d1e7feff3d6c3af5) )
	ROM_LOAD16_BYTE( "hare1-3p.p1", 0x000001, 0x020000, CRC(225f741b) SHA1(69ca6faeb6ad12c19f4670e034a57c12a62d598b) )
	ROM_LOAD16_BYTE( "hare1-3p.p2", 0x040000, 0x020000, CRC(e6f96a6b) SHA1(a4c084e9488c99aca1560999e6f0daf61c456fb3) )
	ROM_LOAD16_BYTE( "hare1-3p.p3", 0x040001, 0x020000, CRC(9c57b62c) SHA1(9a82ffb7b9084cc618fb53533a7e567fcebbc4ea) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhrd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-6n.p0", 0x000000, 0x020000, CRC(e2c65355) SHA1(2b29b3ebb49d8e978fd8307f2cf72ac060b4f3a8) )
	ROM_LOAD16_BYTE( "hare1-6n.p1", 0x000001, 0x020000, CRC(10e23614) SHA1(20449081088de469494a79fb2fdb737c0da6eb6e) )
	ROM_LOAD16_BYTE( "hare1-6n.p2", 0x040000, 0x020000, CRC(5865d27a) SHA1(4f73adf5a240f58120491fac7b61656d168dc8df) )
	ROM_LOAD16_BYTE( "hare1-6n.p3", 0x040001, 0x020000, CRC(59729140) SHA1(0c28ec88bc2b33fe514d630e2705604ae5d0cd7b) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhre )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1-6p.p0", 0x000000, 0x020000, CRC(c1d9978a) SHA1(3de3e8ac2b318c0586087649f35fde8b6cadb1ac) )
	ROM_LOAD16_BYTE( "hare1-6p.p1", 0x000001, 0x020000, CRC(98c935f5) SHA1(b4c68dff78d0d2d8e006d3490170f1cf11e10b22) )
	ROM_LOAD16_BYTE( "hare1-6p.p2", 0x040000, 0x020000, CRC(5865d27a) SHA1(4f73adf5a240f58120491fac7b61656d168dc8df) )
	ROM_LOAD16_BYTE( "hare1-6p.p3", 0x040001, 0x020000, CRC(59729140) SHA1(0c28ec88bc2b33fe514d630e2705604ae5d0cd7b) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhrf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hare1_9.p0", 0x000000, 0x020000, CRC(98673883) SHA1(31a0a7f7f1d9eab1e0d0136b1a67f30dde22c8d1) )
	ROM_LOAD16_BYTE( "hare1_9.p1", 0x000001, 0x020000, CRC(c767f8c4) SHA1(7e77b73d2a7322517e177bf0020f01034379da4f) )
	ROM_LOAD16_BYTE( "hare1_9.p2", 0x040000, 0x020000, CRC(95c502f1) SHA1(78f769acf3109e88580c90f8e86f42c2f141d812) )
	ROM_LOAD16_BYTE( "hare1_9.p3", 0x040001, 0x020000, CRC(220284be) SHA1(b470a2acd42b535fb62b1a99e228866ec19bef79) )
	EP_MILHR_SOUND
ROM_END

ROM_START( ep_milhrg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wwtbamh0.bin", 0x000000, 0x020000, CRC(98673883) SHA1(31a0a7f7f1d9eab1e0d0136b1a67f30dde22c8d1) )
	ROM_LOAD16_BYTE( "wwtbamh1.bin", 0x000001, 0x020000, CRC(c767f8c4) SHA1(7e77b73d2a7322517e177bf0020f01034379da4f) )
	ROM_LOAD16_BYTE( "wwtbamh2.bin", 0x040000, 0x020000, CRC(91a5db77) SHA1(b91d783e111116b871a61d899195a12eea3d2d0c) )
	ROM_LOAD16_BYTE( "wwtbamh3.bin", 0x040001, 0x020000, CRC(220284be) SHA1(b470a2acd42b535fb62b1a99e228866ec19bef79) )
	EP_MILHR_SOUND
ROM_END




#define EP_MLHRC_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cmlhsnd.p0", 0x000000, 0x100000, CRC(e6e60e63) SHA1(dba38ff4e1d0ed803083cce42c858a2110945a4e) )
ROM_START( ep_mlhrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-0n.p0", 0x000000, 0x020000, CRC(857b0ca9) SHA1(904190c2de27f69d29d6e472e78a422980f838d0) )
	ROM_LOAD16_BYTE( "cmlh1-0n.p1", 0x000001, 0x020000, CRC(d6a7f622) SHA1(027e75dda37861e8940b8b524620066b02542dff) )
	EP_MLHRC_SOUND
ROM_END

ROM_START( ep_mlhrca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-0p.p0", 0x000000, 0x020000, CRC(58cfd0bd) SHA1(5b8590cf2371636cc16ea9d7729eed8f5f2f86a5) )
	ROM_LOAD16_BYTE( "cmlh1-0p.p1", 0x000001, 0x020000, CRC(d6a7f622) SHA1(027e75dda37861e8940b8b524620066b02542dff) )
	EP_MLHRC_SOUND
ROM_END

ROM_START( ep_mlhrcb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-1n.p0", 0x000000, 0x020000, CRC(2f47dc0f) SHA1(12b9fe009263dad20d337a3006079edd8bf7d11e) )
	ROM_LOAD16_BYTE( "cmlh1-1n.p1", 0x000001, 0x020000, CRC(57d2f2d4) SHA1(70f3b4d28898a497c017b10ba25c5f94e5818bc4) )
	EP_MLHRC_SOUND
ROM_END

ROM_START( ep_mlhrcc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-1p.p0", 0x000000, 0x020000, CRC(6eec709c) SHA1(b1928976c6327d160c0a0718f922ee6af7c45e9d) )
	ROM_LOAD16_BYTE( "cmlh1-1p.p1", 0x000001, 0x020000, CRC(57d2f2d4) SHA1(70f3b4d28898a497c017b10ba25c5f94e5818bc4) )
	EP_MLHRC_SOUND
ROM_END

ROM_START( ep_mlhrcd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-6n.p0", 0x000000, 0x020000, CRC(7cf939bd) SHA1(b12b8276c9c9553bf6f93ba2ae0991927a8f6d74) )
	ROM_LOAD16_BYTE( "cmlh1-6n.p1", 0x000001, 0x020000, CRC(6660f7e5) SHA1(87a0125448e999b00f07db3a0371cf4641f7c842) )
	EP_MLHRC_SOUND
ROM_END

ROM_START( ep_mlhrce )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cmlh1-6p.p0", 0x000000, 0x020000, CRC(b55aa6bc) SHA1(253c1e632c512a776ecf651a27cf1c32804edee1) )
	ROM_LOAD16_BYTE( "cmlh1-6p.p1", 0x000001, 0x020000, CRC(6660f7e5) SHA1(87a0125448e999b00f07db3a0371cf4641f7c842) )
	EP_MLHRC_SOUND
ROM_END






/* Other (incomplete) dumps */

ROM_START( ep_bingb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980039.g0", 0x00000, 0x020000, CRC(fa294bfe) SHA1(0c98f9c1c9ef8989ede58eba57bd15ec651a9661) )
	ROM_LOAD16_BYTE( "sb980039.g1", 0x00001, 0x020000, CRC(bea98346) SHA1(5edf630a9099e47491737cd5e078c5899ea8ac0f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bingba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980040.g0", 0x00000, 0x020000, CRC(0eceb011) SHA1(33ac340c15c0cae298035b49510d9d3a14d1a4f7) )
	ROM_LOAD16_BYTE( "sb980040.g1", 0x00001, 0x020000, CRC(22ecd867) SHA1(902f7f0644673ac58f69b770e1c1d0be37c5e452) )
	MISSING_SOUND
ROM_END




ROM_START( ep_calyp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980306.g0", 0x00000, 0x020000, CRC(d1882325) SHA1(0b71a24486a3f9aae1fa7558953742fad32d84a3) )
	ROM_LOAD16_BYTE( "sa980306.g1", 0x00001, 0x020000, CRC(f24431e7) SHA1(694dc27c9ecf8dfc038951cbf57de08f7ec5c72f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_calypa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980307.g0", 0x00000, 0x020000, CRC(7a63504d) SHA1(d8df968a20f1e458563e75a40110938c9a7123ea) )
	ROM_LOAD16_BYTE( "sa980307.g1", 0x00001, 0x020000, CRC(fdbd8170) SHA1(1aaa4f3617d65b986a5f0d4a33a9440aedc5ccee) )
	MISSING_SOUND
ROM_END




ROM_START( ep_cow )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980037.g0", 0x00000, 0x020000, CRC(e58bbefa) SHA1(96c46753d6ec84bc4e4dbdc546ae40c1f865ff2e) )
	ROM_LOAD16_BYTE( "sb980037.g1", 0x00001, 0x020000, CRC(b7bb254a) SHA1(0dc23876e028322c25f34c1d28a25f2031950617) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cowa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980038.g0", 0x00000, 0x020000, CRC(5ff562e7) SHA1(91e117618fcacde55a04edb463e19dbfab2278b8) )
	ROM_LOAD16_BYTE( "sb980038.g1", 0x00001, 0x020000, CRC(0cf2d2fb) SHA1(5f4411cc10a3aedda98902e138c6473b2c544a91) )
	MISSING_SOUND
ROM_END


#define EP_CFALL_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cash falls sound 0 sy980007.bin", 0x000000, 0x080000, CRC(7dbe0b80) SHA1(d2454d923b8a479e14513f2201b9725f3b92ef4d) ) \
	ROM_LOAD( "cash falls sound 1 sy980007.bin", 0x080000, 0x080000, CRC(ff7f4c42) SHA1(491d38bdd73bcb9725bfb90942560379f6c8ee40) )

ROM_START( ep_cfall )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980008.g0", 0x00000, 0x020000, CRC(43e05a64) SHA1(33c994f693e6d904587e6b58959e018a7dfe9be2) )
	ROM_LOAD16_BYTE( "sa980008.g1", 0x00001, 0x020000, CRC(8366da15) SHA1(a7ca77ba8297df24e8f008844e25935848528613) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfalla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980170.g0", 0x00000, 0x020000, CRC(3625c374) SHA1(9559516ed6b3612749615ddc7638b0a26648d9e7) )
	ROM_LOAD16_BYTE( "sa980170.g1", 0x00001, 0x020000, CRC(02e4abbf) SHA1(6ca007184b33e6c8373ca5a44add880ec0a49057) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfallb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980171.g0", 0x00000, 0x020000, CRC(9dceb01c) SHA1(7ce11e59f345b3bd6c0dacf3f23473940aca8771) )
	ROM_LOAD16_BYTE( "sa980171.g1", 0x00001, 0x020000, CRC(a725602b) SHA1(07e82e995c2f8513d7859cc03e61edf04e8e3003) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfallc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980172.g0", 0x00000, 0x020000, BAD_DUMP CRC(8bbc3334) SHA1(30cada9c4d9e04ef9076b2b620d078c2739b0aef) )
	ROM_LOAD16_BYTE( "sa980172.g1", 0x00001, 0x020000, BAD_DUMP CRC(7ace66e4) SHA1(1a194385f93474348d62636c025869bb22b0d268) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfalld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980173.g0", 0x00000, 0x020000, CRC(2057405c) SHA1(21248e99d21b0817e2aae8b6df9e635493d59e49) )
	ROM_LOAD16_BYTE( "sa980173.g1", 0x00001, 0x020000, CRC(d6faa152) SHA1(a20a05fcaaedd5ce6f253e14ef3ed7469e448bdb) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfalle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980328.g0", 0x00000, 0x020000, CRC(1ac2d05a) SHA1(db718622150228b44b193fccd70d4e0e5538b99e) )
	ROM_LOAD16_BYTE( "sa980328.g1", 0x00001, 0x020000, CRC(a55730e8) SHA1(b0ce02e64d01263cbc734991be4d138531592d42) )
	EP_CFALL_SOUND
ROM_END

ROM_START( ep_cfallf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980329.g0", 0x00000, 0x020000, CRC(b129a332) SHA1(c58170067937cf96f45d6928ed790f4722e24b10) )
	ROM_LOAD16_BYTE( "sa980329.g1", 0x00001, 0x020000, CRC(8f970b3b) SHA1(e262de241e516722e10ffecec06ed238e45364d0) )
	EP_CFALL_SOUND
ROM_END



ROM_START( ep_cflow )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980260.g0", 0x00000, 0x020000, CRC(2e1c35f0) SHA1(03dddea4dad0d78d50d549865fcad428d80c25ad) ) // cfl260g0
	ROM_LOAD16_BYTE( "sa980260.g1", 0x00001, 0x020000, CRC(8b543c6f) SHA1(ac4dea7ab7092f0d02569d10c9369dbe1f05a53e) ) // cfl260g1
	MISSING_SOUND
ROM_END

ROM_START( ep_cflowa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980155.g0", 0x00000, 0x020000, CRC(eedfe7c5) SHA1(6d774d6f7febac7537542e90de5e71d6573cbbbe) )
	ROM_LOAD16_BYTE( "sa980155.g1", 0x00001, 0x020000, CRC(ef01f567) SHA1(eb6a16950e27875d4dcabf89d209c811b399d3a8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cflowc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990134.g0", 0x00000, 0x020000, CRC(b4a41f3c) SHA1(de8211ae63ed1b4b899ca11dd502e151eecd6a64) )
	ROM_LOAD16_BYTE( "sa990134.g1", 0x00001, 0x020000, CRC(72799360) SHA1(fef3d80b0c9c4a0e6d7a1313252d11d58cba57be) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cflowd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990135.g0", 0x00000, 0x020000, CRC(1f4f6c54) SHA1(3e781d5131f11eec25cfdf59bebf201890ac121a) )
	ROM_LOAD16_BYTE( "sa990135.g1", 0x00001, 0x020000, CRC(dc20baa8) SHA1(5b88b53183e3f87530e418654ae3dc4947c575ca) )
	MISSING_SOUND
ROM_END




ROM_START( ep_doubl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980236.g0", 0x00000, 0x020000, CRC(67be34e4) SHA1(e4a13c922a30636de37bbdf0f5e6eb6421c9615a) )
	ROM_LOAD16_BYTE( "sa980236.g1", 0x00001, 0x020000, CRC(59376f5c) SHA1(17415b717a3802872da3bf6ab7efb4d92244601e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_doubla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980237.g0", 0x00000, 0x020000, CRC(cc55478c) SHA1(5a954b210cc3bb0eec51d876aceb5f3f0e90706f) )
	ROM_LOAD16_BYTE( "sa980237.g1", 0x00001, 0x020000, CRC(cb53ea2e) SHA1(f8a9c83588087c335c8cd97240b970fac0b16706) )
	MISSING_SOUND
ROM_END

ROM_START( ep_doublb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980342.g0", 0x00000, 0x020000, CRC(aca5cd4c) SHA1(16167a7a62fd40e440caee0ee5a04a051fc2948c) )
	ROM_LOAD16_BYTE( "sa980342.g1", 0x00001, 0x020000, CRC(0c7a373d) SHA1(b31a6f8cfd4a520efe1303f8f7f9c113b71a5d5d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_doublc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980343.g0", 0x00000, 0x020000, CRC(074ebe24) SHA1(402a2ba811ded557f7da889e634a2e37bc697e30) )
	ROM_LOAD16_BYTE( "sa980343.g1", 0x00001, 0x020000, CRC(3426c94c) SHA1(73fbe71621fb759937f0179b8a774d32ab3f0913) )
	MISSING_SOUND
ROM_END

ROM_START( ep_doubld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "d_top.g0", 0x000000, 0x020000, CRC(05972fb5) SHA1(8c36a566f855ff86f4ec3986c7a8deb5d3d29b4f) )
	ROM_LOAD16_BYTE( "d_top.g1", 0x000001, 0x020000, CRC(7cfb82fc) SHA1(434890dd4d8550f9f6ca01e72b04d622ae831dcf) )
	MISSING_SOUND
ROM_END



ROM_START( ep_evil )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990114.g0", 0x00000, 0x020000, CRC(143511f6) SHA1(3249fbbadefa377674d0b7e1fb54d7691fb3b9d5) )
	ROM_LOAD16_BYTE( "sa990114.g1", 0x00001, 0x020000, CRC(9b7d395c) SHA1(6eb56d5d32931bf8503650037eaaddf57064f986) )
	MISSING_SOUND
ROM_END

ROM_START( ep_evila )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990115.g0", 0x00000, 0x020000, CRC(bfde629e) SHA1(725219d17672728ba8812569eda89f202034f069) )
	ROM_LOAD16_BYTE( "sa990115.g1", 0x00001, 0x020000, CRC(3ebcf2c8) SHA1(e3e6961c117d27b9b90de16dab1e0edd59309a91) )
	MISSING_SOUND
ROM_END

ROM_START( ep_evilb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "evil_str.g0", 0x00000, 0x020000, CRC(6205effe) SHA1(7f6f923c978edf54fffbfa8eb15baa2b9a5f2b66) )
	ROM_LOAD16_BYTE( "evil_str.g1", 0x00001, 0x020000, CRC(0bef0d51) SHA1(29cd74739f013573ef5c1dbd9f0800624c419a9a) )
	MISSING_SOUND
ROM_END


#define EP_NYNY_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "nyny.s0", 0x000000, 0x080000, CRC(ded53db1) SHA1(d1085d7ead67b2d125c74deccb1b9f154351c3ac) ) \
	ROM_LOAD( "nyny.s1", 0x080000, 0x080000, CRC(8a03bcb2) SHA1(0cb633fa9044be11ad2f60ab53225114d144df3d) )

ROM_START( ep_nyny )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980216.g0", 0x00000, 0x020000, CRC(1792f00c) SHA1(7214a23732f0b6cb1bbd1a2ea8807d34996abeac) )
	ROM_LOAD16_BYTE( "sa980216.g1", 0x00001, 0x020000, CRC(3bd161e3) SHA1(789ac40cba14d2a391aba13324d86b0dcd1f3d17) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980217.g0", 0x00000, 0x020000, CRC(bc798364) SHA1(0715dc87d4ecd962d42e203a1e5bc1de759e2f22) )
	ROM_LOAD16_BYTE( "sa980217.g1", 0x00001, 0x020000, CRC(392189b8) SHA1(93761fbee83bbcefdd53657c1bd181cbba4ab855) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynyb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980218.g0", 0x00000, 0x020000, CRC(06075f79) SHA1(449e2809fd3962e815fbf3b924b5fe447cdf7bbe) )
	ROM_LOAD16_BYTE( "sa980218.g1", 0x00001, 0x020000, CRC(b263a63a) SHA1(3ef038ff7b95a9f93368a71a626604ffc1cb4976) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynyc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980219.g0", 0x00000, 0x020000, CRC(adec2c11) SHA1(d769291e32d9494a945881a8f9353543474273c3) )
	ROM_LOAD16_BYTE( "sa980219.g1", 0x00001, 0x020000, CRC(edb0ec65) SHA1(f349110082c1200f2f568bed69e1ce1b153b70de) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynyd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980336.g0", 0x00000, 0x020000, CRC(aefde642) SHA1(2d9d55f38d8d28bc505bb8ea2769ef94cd98d822) )
	ROM_LOAD16_BYTE( "sa980336.g1", 0x00001, 0x020000, CRC(90bd04e0) SHA1(3fe78d888628185804ebb83a1357d9f39b77aaac) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynye )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980337.g0", 0x00000, 0x020000, CRC(0516952a) SHA1(a09a31a02a5f06e59513d89bf137f76e05f22a20) )
	ROM_LOAD16_BYTE( "sa980337.g1", 0x00001, 0x020000, CRC(8dd871d5) SHA1(5c3bb43563e11bbc1e861a6cf100d9ffef04f994) )
	EP_NYNY_SOUND
ROM_END

ROM_START( ep_nynyf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15nyny.g0", 0x000000, 0x020000, CRC(26c141fa) SHA1(0bf1870bca03fe200e8f1009dab4255e8c430e6f) )
	ROM_LOAD16_BYTE( "15nyny.g1", 0x000001, 0x020000, CRC(a626dce9) SHA1(72f5f3bdb4ad9bca44c7b59b64d7e4251d9cfec7) )
	EP_NYNY_SOUND
ROM_END




ROM_START( ep_pizza )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980292.g0", 0x00000, 0x020000, CRC(6568a589) SHA1(4747140728509737d1a9f51a23fd049fd317bf52) )
	ROM_LOAD16_BYTE( "sa980292.g1", 0x00001, 0x020000, CRC(62fb0e11) SHA1(f67d26090aa097ed4e7e450d3028cd9e117daee8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_pizzaa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980293.g0", 0x00000, 0x020000, CRC(ce83d6e1) SHA1(efaa7873c2570b058ff230c81673727758cb9b03) )
	ROM_LOAD16_BYTE( "sa980293.g1", 0x00001, 0x020000, CRC(c73ac585) SHA1(7c766fb506cca587382fea5dc31a2e90328597f5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_pizzab )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990029.g0", 0x00000, 0x020000, CRC(7480c061) SHA1(bac6fee079ac6314bd0b717741144e32219248e8) )
	ROM_LOAD16_BYTE( "sa990029.g1", 0x00001, 0x020000, CRC(636ef4b9) SHA1(4a2777d43740fcbf3caa58dee261b3fe8ddfddcd) )
	MISSING_SOUND
ROM_END

ROM_START( ep_pizzac )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990030.g0", 0x00000, 0x020000, CRC(80673b8e) SHA1(6b92973d1ab2d5b8bdf24a6c21274e6aac0195d1) )
	ROM_LOAD16_BYTE( "sa990030.g1", 0x00001, 0x020000, CRC(3487e144) SHA1(2d91c368de073e5f7096e9877f41cab3ad74c6e0) )
	MISSING_SOUND
ROM_END




ROM_START( ep_crazy )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980178.g0", 0x00000, 0x020000, CRC(18345f1a) SHA1(066451779c6852422223bcb5e8681f82484e3538) )
	ROM_LOAD16_BYTE( "sa980178.g1", 0x00001, 0x020000, CRC(89f0ea38) SHA1(987f23d15ad68234f30891b1ea7e71a781839cb3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crazya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980179.g0", 0x00000, 0x020000, CRC(b3df2c72) SHA1(63d1f111dbc6ca013a4713e698f1e3aa9125dc3e) )
	ROM_LOAD16_BYTE( "sa980179.g1", 0x00001, 0x020000, CRC(2c3121ac) SHA1(dc98950556b90e0e7c0f55ade852182eae99b2d6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crazyb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980180.g0", 0x00000, 0x020000, CRC(7606c74c) SHA1(d80d2ec4cdf900ff6a053db71f1e58e1a3b1c0c8) )
	ROM_LOAD16_BYTE( "sa980180.g1", 0x00001, 0x020000, CRC(8585bc3f) SHA1(9f487f4c1611ad54dee4fd97143c18cecf63aedb) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crazyc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980181.g0", 0x00000, 0x020000, CRC(ddedb424) SHA1(a86aa3be04f771d4ab9e79ac01a23e38e4f0b6f3) )
	ROM_LOAD16_BYTE( "sa980181.g1", 0x00001, 0x020000, CRC(204477ab) SHA1(f33fc101aa204abb88c3ec7732e9eb61e67692f3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crazyd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980326.g0", 0x00000, 0x020000, CRC(f3236e07) SHA1(19d929c9c56dcc2db4a8942344e0fd9bc0862cbd) )
	ROM_LOAD16_BYTE( "sa980326.g1", 0x00001, 0x020000, CRC(62e98d7f) SHA1(ccfb8e5a99dea6cbde6079409bab56677d22efae) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crazye )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980327.g0", 0x00000, 0x020000, CRC(a559b8f6) SHA1(89c2583ab67bf4779c74f2154da6a10b04123efa) )
	ROM_LOAD16_BYTE( "sa980327.g1", 0x00001, 0x020000, CRC(d0250297) SHA1(e25ce7545c27ab7e8fd36289c21dab0a528388b2) )
	MISSING_SOUND
ROM_END


#define EP_MARIO_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) /* contains 'Super Mario' string */ \
	ROM_LOAD( "sm64snd0", 0x000000, 0x080000, CRC(1e5c350d) SHA1(1f7e0ba84b122882b82ae4edd78628cfcae92977) ) \
	ROM_LOAD( "sm64snd1", 0x080000, 0x080000, CRC(5ed95303) SHA1(b58e557b56c7b943e19b9bf1229de85c3ca99fe8) )

ROM_START( ep_mario )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "970024g0", 0x00000, 0x020000, CRC(595c6d14) SHA1(aa7c71fa751bf9e0d93324f206a301fdb30dc2ba) )
	ROM_LOAD16_BYTE( "970024g1", 0x00001, 0x020000, CRC(968cac01) SHA1(de032081acdbd7242cde168a0c70d87233cb6b48) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_marioa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "970025g0", 0x00000, 0x020000, CRC(f2b71e7c) SHA1(b06f4fcc97b57b6ae4bbdaa7e99224c9af2d62df) )
	ROM_LOAD16_BYTE( "970025g1", 0x00001, 0x020000, CRC(99751c96) SHA1(1d9bb5b19dfda4ccaeb7774636c0ad42229bb4db) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_mariob )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970036.g0", 0x00000, 0x020000, CRC(97d95c0b) SHA1(44d5fa24fd0e62e8edf814279a530b74c7044a13) )
	ROM_LOAD16_BYTE( "sa970036.g1", 0x00001, 0x020000, CRC(9ba67bbd) SHA1(d3239941887b47a372124366ad1a3aba049bf1b8) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_marioc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970037.g0", 0x00000, 0x020000, CRC(3c322f63) SHA1(7a65bb4daf1f4336d0f15c45c75c31f697c7293b) )
	ROM_LOAD16_BYTE( "sa970037.g1", 0x00001, 0x020000, CRC(945fcb2a) SHA1(835fbe21a6a6f1988975424411733474d4e8a148) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_mariod )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970038.g0", 0x00000, 0x020000, CRC(864cf37e) SHA1(fb9103541ef30332ff2b6dcbbfda6bb205e78673) )
	ROM_LOAD16_BYTE( "sa970038.g1", 0x00001, 0x020000, CRC(29fd0a78) SHA1(f92d3c0c75f5dec7ed5ccd29c81396eb648f95e2) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_marioe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970039.g0", 0x00000, 0x020000, CRC(2da78016) SHA1(498c6f46c429ca187e067161a44e9017456358ff) )
	ROM_LOAD16_BYTE( "sa970039.g1", 0x00001, 0x020000, CRC(34987f4d) SHA1(6f4260165f9ae468ce434f522e48904b7e6fe5c6) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_mariof )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990033.g0", 0x00000, 0x020000, CRC(eec85e3d) SHA1(ac1d4eb061c994aef0eb3163d31f099ba67e30ce) )
	ROM_LOAD16_BYTE( "sa990033.g1", 0x00001, 0x020000, CRC(3e0ac4bb) SHA1(99ead4110f5a90a9a5f41481f528ccb234a1ebf2) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_mariog )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990034.g0", 0x00000, 0x020000, CRC(0bba0aa7) SHA1(21850b73fa8001a31ebea2d524396c0e62e1c0ce) )
	ROM_LOAD16_BYTE( "sa990034.g1", 0x00001, 0x020000, CRC(802cd278) SHA1(45d547790093cb98890979220b1d4efc37c200c4) )
	EP_MARIO_SOUND
ROM_END

ROM_START( ep_marioh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15smg0", 0x00000, 0x020000, CRC(e66591c1) SHA1(e8c9b8ca3634d1a0847f9b243344a3bc6f2c9719) )
	ROM_LOAD16_BYTE( "15smg1", 0x00001, 0x020000, CRC(8a8f8b5d) SHA1(850a72917046b94a800ade94d097e8262b3a49e7) )
	EP_MARIO_SOUND
ROM_END








ROM_START( ep_tincn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980334.g0", 0x00000, 0x020000, CRC(95216a97) SHA1(a89d9120de3978ca78982ae24bcebf071cf7dd79) )
	ROM_LOAD16_BYTE( "sa980334.g1", 0x00001, 0x020000, CRC(65ed2bdd) SHA1(4440679a5a093d138a0f4c33dc37bd7f36a9a470) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tincna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980335.g0", 0x00000, 0x020000, CRC(3eca19ff) SHA1(2f6adb1cbcc1bac152884bfcb8b1661ac2f0f959) )
	ROM_LOAD16_BYTE( "sa980335.g1", 0x00001, 0x020000, CRC(bbf2d746) SHA1(d309ece5a5808adf7cba880395b0aae69de07622) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tonfn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970034.g0", 0x00000, 0x020000, CRC(d9437166) SHA1(9ef4a8f7146c51b6d9f55e65c190c73ba93a899f) )
	ROM_LOAD16_BYTE( "sa970034.g1", 0x00001, 0x020000, CRC(456212ca) SHA1(b432dd2be88c30f345e4e3e20b9dd972c8f4a297) )
	MISSING_SOUND
ROM_END


ROM_START( ep_tonfna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa970035.g0", 0x00000, 0x020000, CRC(72a8020e) SHA1(5297769826a122c4184659984ba800915a5331eb) )
	ROM_LOAD16_BYTE( "sa970035.g1", 0x00001, 0x020000, CRC(ca261843) SHA1(0282296ce0627011f2269c135d685d3c739214b6) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tcrwn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980220.g0", 0x00000, 0x020000, CRC(6803e365) SHA1(482ba6e0886612968454893549d57ab40391ed98) )
	ROM_LOAD16_BYTE( "sa980220.g1", 0x00001, 0x020000, CRC(f4e04988) SHA1(cf9a0f7e0fd9aed7d8c3b0fbda617951d2b97c3b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tcrwna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980221.g0", 0x00000, 0x020000, CRC(c3e8900d) SHA1(c02297ef7df519c6f195a5c01a91f2379ed045de) )
	ROM_LOAD16_BYTE( "sa980221.g1", 0x00001, 0x020000, CRC(e9853cbd) SHA1(7074f9e5a5500affa7292a3e8d0d81e2d1dc3862) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tcrwnb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980222.g0", 0x00000, 0x020000, CRC(e4a403f4) SHA1(df8598038d6cce4ae3c201a593731ad05ca003c0) )
	ROM_LOAD16_BYTE( "sa980222.g1", 0x00001, 0x020000, CRC(42302dc9) SHA1(7167b2d717e178d687624d0781a2932151fbe36f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tcrwnc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980223.g0", 0x00000, 0x020000, CRC(4f4f709c) SHA1(ffb512cf26e3b005b405dcf4392a2c05952d9cd0) )
	ROM_LOAD16_BYTE( "sa980223.g1", 0x00001, 0x020000, CRC(7a6cd3b8) SHA1(b80b662a5600a868289153e7fe485e861e34db16) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tcrwnd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980300.g0", 0x00000, 0x020000, CRC(14fc108b) SHA1(8526a338586b7e83d6c274a454f1beae8510a170) )
	ROM_LOAD16_BYTE( "sa980300.g1", 0x00001, 0x020000, CRC(c1815654) SHA1(c78453469a4917eba2f531ca20716dd62d31d8c1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tcrwne )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980301.g0", 0x00000, 0x020000, CRC(bf1763e3) SHA1(6beae179c86815f87c16939662883cdabc76dfc8) )
	ROM_LOAD16_BYTE( "sa980301.g1", 0x00001, 0x020000, CRC(fc11d90b) SHA1(6249e898cb0da9460c680553662b9adb649e31ba) )
	MISSING_SOUND
ROM_END




ROM_START( ep_word )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980106.g0", 0x00000, 0x020000, CRC(981370cf) SHA1(9b647e1e198524dc48d7a545ea1e0a66c7af53a4) )
	ROM_LOAD16_BYTE( "sa980106.g1", 0x00001, 0x020000, CRC(c9f0340a) SHA1(e105ab3af725b126e4a73cb8d75628198aff645c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_worda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980107.g0", 0x00000, 0x020000, CRC(33f803a7) SHA1(eee1f540a1929355185093ea4af3cf803215751b) )
	ROM_LOAD16_BYTE( "sa980107.g1", 0x00001, 0x020000, CRC(d495413f) SHA1(79fc16c3fd2bb90607a0451b52f388e0164e95c5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wordb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980108.g0", 0x00000, 0x020000, CRC(b8b8cf6b) SHA1(bb782ed2d7317700f764421989c238f4bdecd3a0) )
	ROM_LOAD16_BYTE( "sa980108.g1", 0x00001, 0x020000, CRC(d7d9a33d) SHA1(fa4d27eb30748b797643e57eb741ce09867253c6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wordc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980109.g0", 0x00000, 0x020000, CRC(1353bc03) SHA1(8561923508dd55f1eaccc5032e2550345d488c05) )
	ROM_LOAD16_BYTE( "sa980109.g1", 0x00001, 0x020000, CRC(fd1998ee) SHA1(3831abaeff76bb32edc32b20e2cca4f589bf0eff) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wordd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980174.g0", 0x00000, 0x020000, CRC(5de39a09) SHA1(3bb043c3a7525d11917f5f834709052fe0a6b4cb) )
	ROM_LOAD16_BYTE( "sa980174.g1", 0x00001, 0x020000, CRC(3c07a01f) SHA1(916e1f2f162519cae2178267e492aba1a60743a6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_worde )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980175.g0", 0x00000, 0x020000, CRC(f608e961) SHA1(6bf870d6e13356f3030e66f855c6ec216b84cbd2) )
	ROM_LOAD16_BYTE( "sa980175.g1", 0x00001, 0x020000, CRC(c4e5c98f) SHA1(1d914a0c32cc44cf8e39577e3af1c07228cb05c1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wordf )
	ROM_REGION( 0x080000, "maincpu", 0 ) // positions were swapped on these at least
	ROM_LOAD16_BYTE( "sa980176_g0", 0x00000, 0x020000, CRC(1502a94a) SHA1(090c3f620eff3a411a9ec23e3100a1e53e1dd46b)  )
	ROM_LOAD16_BYTE( "sa980176_g1", 0x00001, 0x020000, CRC(86f647fd) SHA1(0ee46e40c35c6efeb8b09d95973b5e4428377c24)  )
	MISSING_SOUND
ROM_END

ROM_START( ep_wordg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa980177.g0", 0x00000, 0x020000, CRC(bee9da22) SHA1(e8cf0977bfa5034fbd5dbb5873331b4d06be5a23) )
	ROM_LOAD16_BYTE( "sa980177.g1", 0x00001, 0x020000, CRC(bb66c8a2) SHA1(ef698ce7d2d3c3a6eac64d3fc9c52c3e300eebcb) )
	MISSING_SOUND
ROM_END




ROM_START( ep_xtra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980035.g0", 0x00000, 0x020000, CRC(6655ccf6) SHA1(8a6c49dcf7c9ca635d4261d98a7e134b4f88df4b) )
	ROM_LOAD16_BYTE( "sb980035.g1", 0x00001, 0x020000, CRC(61a35fca) SHA1(f4900f35a8cc5a253a6b0cf6cf70a2b2deea1899) )
	MISSING_SOUND
ROM_END

ROM_START( ep_xtraa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980036.g0", 0x00000, 0x020000, CRC(e7a337f3) SHA1(9a461a528da0710eae8f67c521561634d2b35a92) )
	ROM_LOAD16_BYTE( "sb980036.g1", 0x00001, 0x020000, CRC(400c1c09) SHA1(1eb3ca199f317108a76f2b0732704a1d564b48aa) )
	MISSING_SOUND
ROM_END



#define EP_25CRT_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) /* not 100% these are right, '25.s0' isn't exactly a telling name */ \
	ROM_LOAD( "25.s0", 0x000000, 0x080000, CRC(fb4dd880) SHA1(4ccb92b4f14f49fef45d804671c429355cf75dcf) ) \
	ROM_LOAD( "25.s1", 0x080000, 0x080000, CRC(ba09b8c7) SHA1(91141f06425e4ac7fed371768062d171e300e8f1) )

ROM_START( ep_25crt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "2642.1g0", 0x000000, 0x020000, CRC(cbc001a7) SHA1(778e9a3714b58b7935fe0df5e64c77ce6ab52b3b) )
	ROM_LOAD16_BYTE( "1ba6.1g1", 0x000001, 0x020000, CRC(fcac8e99) SHA1(a5aeec7ba1415525ac57427c8eca2309377d1a42) )
	EP_25CRT_SOUND
ROM_END

ROM_START( ep_25crta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb02029.g0", 0x000000, 0x020000, CRC(0062a16b) SHA1(0025bca547826e47f27965284a6b9f5fe9394dc3) )
	ROM_LOAD16_BYTE( "msb02029.g1", 0x000001, 0x020000, CRC(90b0aa16) SHA1(5a27cd12f44e3c09d5ca2b990edcfec50696ba1a) )
	EP_25CRT_SOUND
ROM_END

ROM_START( ep_25crtb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02006.g0", 0x000000, 0x020000, CRC(ddd86b50) SHA1(95e6fd21748ba461b2317f1afcb50cb1e034977e) )
	ROM_LOAD16_BYTE( "mua02006.g1", 0x000001, 0x020000, CRC(fe2701d7) SHA1(a32774de3ddcb36e57bac5ad4eaaad83d4372c89) )
	EP_25CRT_SOUND
ROM_END

ROM_START( ep_25crtc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua03029.g0", 0x000000, 0x020000, CRC(fdc7f74c) SHA1(6803bfab44081b9c6354c025092cc10008dcfde2) )
	ROM_LOAD16_BYTE( "mua03029.g1", 0x000001, 0x020000, CRC(77762650) SHA1(27df8772be10b0520a6ead136dd400563b697654) )
	EP_25CRT_SOUND
ROM_END

ROM_START( ep_25crtd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua03030.g0", 0x000000, 0x020000, CRC(54d63436) SHA1(fb4cb6bc355fa53b11b34c19f5948f4a24e16ba4) )
	ROM_LOAD16_BYTE( "mua03030.g1", 0x000001, 0x020000, CRC(6559518f) SHA1(ad8df1856b6f778a3e6cf9e71ff83861436e7a31) )
	EP_25CRT_SOUND
ROM_END




ROM_START( ep_bartk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02253.g0", 0x000000, 0x020000, CRC(c928ac80) SHA1(5399fd8c6785493c2cb7c6f7e0a574691fec7724) )
	ROM_LOAD16_BYTE( "msa02253.g1", 0x000001, 0x020000, CRC(c18a8f70) SHA1(f345b68c53275adae3fb5091cd2013a7c4dedd6a) )
	ROM_LOAD16_BYTE( "msa02253.g2", 0x040000, 0x020000, CRC(243359c5) SHA1(dfe3e39f2eb8d097415f99e8d4bb69e65a3ff549) )
	ROM_LOAD16_BYTE( "msa02253.g3", 0x040001, 0x020000, CRC(abacfff3) SHA1(36459329859b5511d003bc4b545c533c301faa0d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02203.g0", 0x000000, 0x020000, CRC(cdc7084c) SHA1(e3e45fa17dd9547a99487c21136d14c15c9aacb4) )
	ROM_LOAD16_BYTE( "msa02203.g1", 0x000001, 0x020000, CRC(70837ae2) SHA1(2b6cd63858aaa2ecb36166ed5c1fcd15ee3a9fe0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartkb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02204.g0", 0x000000, 0x020000, CRC(184bba79) SHA1(6579545a17ddc3d0583bdd8df294121660a8a501) )
	ROM_LOAD16_BYTE( "msa02204.g1", 0x000001, 0x020000, CRC(6a82d988) SHA1(ef5ecb1f6df6237764ce16dddd9a3fc0ebbb5221) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartkc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02224.g0", 0x000000, 0x020000, CRC(d4b79750) SHA1(5678efc827ff6d47e8d45df7926ee5a9b98ee554) )
	ROM_LOAD16_BYTE( "msa02224.g1", 0x000001, 0x020000, CRC(1808c25e) SHA1(ee473f47266ee680ba165967b7d7ce8c22f31d95) )
	ROM_LOAD16_BYTE( "msa02224.g2", 0x040000, 0x020000, CRC(431a37d5) SHA1(a567027e613d91e09889bcdc0ab7bcdd21847289) )
	ROM_LOAD16_BYTE( "msa02224.g3", 0x040001, 0x020000, CRC(e008807b) SHA1(627110e445c65dff935c08506f729840780b4677) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartkd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02225.g0", 0x000000, 0x020000, CRC(fb337b74) SHA1(5b1a15aff7ffdd90cc8153a364e70feffead2902) )
	ROM_LOAD16_BYTE( "msa02225.g1", 0x000001, 0x020000, CRC(97113827) SHA1(7aa1011b51a17a73086b7de65bcc768f869f55e3) )
	ROM_LOAD16_BYTE( "msa02225.g2", 0x040000, 0x020000, CRC(431a37d5) SHA1(a567027e613d91e09889bcdc0ab7bcdd21847289) )
	ROM_LOAD16_BYTE( "msa02225.g3", 0x040001, 0x020000, CRC(e008807b) SHA1(627110e445c65dff935c08506f729840780b4677) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartke )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02252.g0", 0x000000, 0x020000, CRC(7524bfd9) SHA1(cf528eea960a2025a2fc074016f9d19afc34f7e8) )
	ROM_LOAD16_BYTE( "msa02252.g1", 0x000001, 0x020000, CRC(fecf0a26) SHA1(519787d87903567b51e2a9de5f3553d8c736e1ea) )
	ROM_LOAD16_BYTE( "msa02252.g2", 0x040000, 0x020000, CRC(243359c5) SHA1(dfe3e39f2eb8d097415f99e8d4bb69e65a3ff549) )
	ROM_LOAD16_BYTE( "msa02252.g3", 0x040001, 0x020000, CRC(abacfff3) SHA1(36459329859b5511d003bc4b545c533c301faa0d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bartkf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bar_trek.g0", 0x000000, 0x020000, CRC(661965f9) SHA1(f6fd83a9be14e4d2d5bbf1fbb33cc99f12d18ffa) )
	ROM_LOAD16_BYTE( "bar_trek.g1", 0x000001, 0x020000, CRC(24f86ffc) SHA1(bc839b08f5b6cc604710adb17497d5be3cd47d4c) )
	MISSING_SOUND
ROM_END



ROM_START( ep_batls )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02178.g0", 0x000000, 0x020000, CRC(debdf3f6) SHA1(35e10d2b3c5282e59d7dfed1575ccb2421404e8d) )
	ROM_LOAD16_BYTE( "msa02178.g1", 0x000001, 0x020000, CRC(198a113a) SHA1(ef6bc342fbb640269df4da04b51a2773ab1a1479) )
	MISSING_SOUND
ROM_END

ROM_START( ep_batlsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02179.g0", 0x000000, 0x020000, CRC(809170e9) SHA1(327df4fcc6e2a2f9e26e813f7cb4df7854498d67) )
	ROM_LOAD16_BYTE( "msa02179.g1", 0x000001, 0x020000, CRC(7c12bf42) SHA1(a9924b52e4b5230ff734a0fa00b8a1c54ed82c27) )
	MISSING_SOUND
ROM_END

ROM_START( ep_batlsb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02186.g0", 0x000000, 0x020000, CRC(43a11b5b) SHA1(6feec82d1aa1c8f589973df5b4df6c43844fc3c8) )
	ROM_LOAD16_BYTE( "msa02186.g1", 0x000001, 0x020000, CRC(ee1deb59) SHA1(3cc75cbc24ab9a59985e6c46dd0967cba4704faa) )
	MISSING_SOUND
ROM_END

ROM_START( ep_batlsc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02187.g0", 0x000000, 0x020000, CRC(d8b48e9c) SHA1(c6785f8c39f0e8949115cbede46bcf9fa1085590) )
	ROM_LOAD16_BYTE( "msa02187.g1", 0x000001, 0x020000, CRC(2671ad95) SHA1(b6dd67e8ae50892005041e51bf5519d2bf6e5b09) )
	MISSING_SOUND
ROM_END



ROM_START( ep_bbonz )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bib0003.g0", 0x000000, 0x020000, CRC(aff702f0) SHA1(d6e02437ab220bd5fd9ca2b25692e60163a1e0c8) )
	ROM_LOAD16_BYTE( "bib0003.g1", 0x000001, 0x020000, CRC(d07ee9d1) SHA1(7d9b9cb8fee36fea79e6965787eb0f02726dc305) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bbonza )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bib0004.g0", 0x000000, 0x020000, CRC(ae5ab14a) SHA1(dfed473431ab3195ce114624655e4fd2dcda2432) )
	ROM_LOAD16_BYTE( "bib0004.g1", 0x000001, 0x020000, CRC(5c2b5daa) SHA1(071c80b03a8710f5f86f9e8ebfdb4ddd754a3cb1) )
	MISSING_SOUND
ROM_END



ROM_START( ep_bubsq )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "basq02.p0", 0x000000, 0x020000, CRC(23b14f13) SHA1(603955176d2cd5a1f79e89e0a6a0d0dcfb66f553) )
	ROM_LOAD16_BYTE( "basq02.p1", 0x000001, 0x020000, CRC(ce8b94c3) SHA1(84cc9dc87ada2c30fa5b3f61d3dabf5c67304c93) )
	ROM_LOAD16_BYTE( "basq02.p2", 0x040000, 0x020000, CRC(b12130ab) SHA1(9e2e25b8bce753d3e6e0537715d91fb97c8b7774) )
	ROM_LOAD16_BYTE( "basq02.p3", 0x040001, 0x020000, CRC(e25f2b24) SHA1(f8172163fe8ffc7de22c0cd398500c201fc467ee) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bubsqa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "basq02p.p0", 0x000000, 0x020000, CRC(23b14f13) SHA1(603955176d2cd5a1f79e89e0a6a0d0dcfb66f553) )
	ROM_LOAD16_BYTE( "basq02p.p1", 0x000001, 0x020000, CRC(d11866b8) SHA1(89db4b1842782be1300eab33c0ad6b7debbbe3e8) )
	ROM_LOAD16_BYTE( "basq02p.p2", 0x040000, 0x020000, CRC(b12130ab) SHA1(9e2e25b8bce753d3e6e0537715d91fb97c8b7774) )
	ROM_LOAD16_BYTE( "basq02p.p3", 0x040001, 0x020000, CRC(e25f2b24) SHA1(f8172163fe8ffc7de22c0cd398500c201fc467ee) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bubsqb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "basq03.p0", 0x000000, 0x020000, CRC(109164cf) SHA1(3993b975be29f8cfbcd2bc10c16bcca487cae1e2) )
	ROM_LOAD16_BYTE( "basq03.p1", 0x000001, 0x020000, CRC(e02974ab) SHA1(9940c44cc90336d0dcca7c00f31e5419b889620d) )
	ROM_LOAD16_BYTE( "basq03.p2", 0x040000, 0x020000, CRC(9410298f) SHA1(ccd1982fd684bc9c5043cf47c0bf6bfc7e223b67) )
	ROM_LOAD16_BYTE( "basq03.p3", 0x040001, 0x020000, CRC(7e43ae60) SHA1(05a9db15a288e8c6445e1a80786e2cb055f7c6f0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bubsqc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "basq03p.p0", 0x000000, 0x020000, CRC(109164cf) SHA1(3993b975be29f8cfbcd2bc10c16bcca487cae1e2) )
	ROM_LOAD16_BYTE( "basq03p.p1", 0x000001, 0x020000, CRC(ffba86d0) SHA1(bb4c09f524b19876db354cafdcebf8523c0b0466) )
	ROM_LOAD16_BYTE( "basq03p.p2", 0x040000, 0x020000, CRC(9410298f) SHA1(ccd1982fd684bc9c5043cf47c0bf6bfc7e223b67) )
	ROM_LOAD16_BYTE( "basq03p.p3", 0x040001, 0x020000, CRC(7e43ae60) SHA1(05a9db15a288e8c6445e1a80786e2cb055f7c6f0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bubsqd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "basq04.p0", 0x000000, 0x020000, CRC(e9b6a02f) SHA1(f51c5219aa59c7c62a4b30bc68f04f245f7cde87) )
	ROM_LOAD16_BYTE( "basq04.p1", 0x000001, 0x020000, CRC(0b614382) SHA1(aea4d09ae1aa9538bf16b92caf50b9e54f844868) )
	ROM_LOAD16_BYTE( "basq04.p2", 0x040000, 0x020000, CRC(db0c7d1a) SHA1(e374a2850a42f1aa8e2c8be262678a66579a1143) )
	ROM_LOAD16_BYTE( "basq04.p3", 0x040001, 0x020000, CRC(61618588) SHA1(2cb0ea50f107cf2443bd1ff3dbf67959f07e1216) )
	MISSING_SOUND
ROM_END



ROM_START( ep_cdspn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02205.g0", 0x000000, 0x020000, CRC(55c5cadb) SHA1(e906760e27821e0c62b56c9cdde5dc80e8ef12c1) )
	ROM_LOAD16_BYTE( "msa02205.g1", 0x000001, 0x020000, CRC(57cb5538) SHA1(ea43eb4e0adda2a0ddbe04c473a37cbcff9e39d1) )
	ROM_LOAD16_BYTE( "msa02205.g2", 0x040000, 0x020000, CRC(d60ede09) SHA1(af00ad536788911fcf6374be09fecb1f9a04b040) )
	ROM_LOAD16_BYTE( "msa02205.g3", 0x040001, 0x020000, CRC(2335aa58) SHA1(1d69df6883223e12e6f053061d44b8e052409f3b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cdspna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02206.g0", 0x000000, 0x020000, CRC(66283d5c) SHA1(30b59409c84c37e7331c8f586a13437be27ec51a) )
	ROM_LOAD16_BYTE( "msa02206.g1", 0x000001, 0x020000, CRC(731a5ac8) SHA1(d2d3f4267479a10c93023ee3c397a2621b5bd0b2) )
	ROM_LOAD16_BYTE( "msa02206.g2", 0x040000, 0x020000, CRC(d60ede09) SHA1(af00ad536788911fcf6374be09fecb1f9a04b040) )
	ROM_LOAD16_BYTE( "msa02206.g3", 0x040001, 0x020000, CRC(2335aa58) SHA1(1d69df6883223e12e6f053061d44b8e052409f3b) )
	MISSING_SOUND
ROM_END


#define EP_COSC_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cosc.s0", 0x0000, 0x080000, CRC(4282b495) SHA1(7cdec8e8219a892c26311608ebd8953b381ecc70) ) \
	ROM_LOAD( "cosc.s1", 0x0000, 0x080000, CRC(ad191be1) SHA1(1d0934cf0d39ad842d68eeea5c91ddd8637644e6) )

ROM_START( ep_cosc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02142.g0", 0x000000, 0x020000, CRC(110c9b28) SHA1(5aca3ad3fc79ec288a47c294bf32c841a9c2a9c7) )
	ROM_LOAD16_BYTE( "msa02142.g1", 0x000001, 0x020000, CRC(3bea853f) SHA1(5752a11abe666da01d7f66b995d5274dc24c2403) )
	ROM_LOAD16_BYTE( "msa02142.g2", 0x040000, 0x020000, CRC(64584ab4) SHA1(08817c5d68099f316850947ada95fb864da9ae7c) )
	ROM_LOAD16_BYTE( "msa02142.g3", 0x040001, 0x020000, CRC(2a3ad9a8) SHA1(06c1df7e1c658c614098ee1939fb9af014927f28) )
	EP_COSC_SOUND
ROM_END

ROM_START( ep_cosca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02143.g0", 0x000000, 0x020000, CRC(43a48fa1) SHA1(1c225a47dfd553ce42eb7bc2dece1137f65c84e2) )
	ROM_LOAD16_BYTE( "msa02143.g1", 0x000001, 0x020000, CRC(a346b9be) SHA1(d43e328127a6c02739b47dc663a12f9b558b92dd) )
	ROM_LOAD16_BYTE( "msa02143.g2", 0x040000, 0x020000, CRC(64584ab4) SHA1(08817c5d68099f316850947ada95fb864da9ae7c) )
	ROM_LOAD16_BYTE( "msa02143.g3", 0x040001, 0x020000, CRC(2a3ad9a8) SHA1(06c1df7e1c658c614098ee1939fb9af014927f28) )
	EP_COSC_SOUND
ROM_END

#define EP_CAHOY_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "cahoy.s0", 0x0000, 0x080000, CRC(28512eb3) SHA1(60b34bcbd840740fab45d16f9ac9784a917a6c5d) ) \
	ROM_LOAD( "cahoy.s1", 0x0000, 0x080000, CRC(bc70d9c3) SHA1(b2b66956e5cbb53999386429bd32ddac32980289) )

//Sound ROMs taken from this set...
ROM_START( ep_cahoy )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cahoyg0.bin", 0x000000, 0x020000, CRC(b7aed7c8) SHA1(6aaf9fbb0b4d845d12d4a949aa794f958154e7d8) )
	ROM_LOAD16_BYTE( "cahoyg1.bin", 0x000001, 0x020000, CRC(35dc24f6) SHA1(32eb669ba61e683aea31231a3a2139abe5a4dba1) )
	EP_CAHOY_SOUND
ROM_END

ROM_START( ep_cahoya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "c_ahoy.g0", 0x000000, 0x020000, CRC(20be6dfc) SHA1(abda6674c7663f3136d270fbec5b214f3d7e33ee) )
	ROM_LOAD16_BYTE( "c_ahoy.g1", 0x000001, 0x020000, CRC(a363f246) SHA1(3c609d803e6f9b43717e38e7c9264d6a1a115837) )
	EP_CAHOY_SOUND
ROM_END

ROM_START( ep_cahoyb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "c_ahoy 5pound.g0", 0x000000, 0x020000, CRC(dc7a5c0d) SHA1(d6068e72ab3025c601752f18f144917a6da1d5df) )
	ROM_LOAD16_BYTE( "c_ahoy 5pound.g1", 0x000001, 0x020000, CRC(fb92b892) SHA1(3cfe1486ee6b312259efda4a1eeb4d415b0c9dc1) )
	EP_CAHOY_SOUND
ROM_END



#define EP_CCOCK_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "c_cockt.s0", 0x000000, 0x080000, CRC(ad3d60e4) SHA1(ac45ae46517e92fc41ebd249bd6a5a5a29c7bb38) ) \
	ROM_LOAD( "c_cockt.s1", 0x080000, 0x080000, CRC(25758aa4) SHA1(7a8ee383da39b2443affb807ff60a8a960042cef) )
ROM_START( ep_ccock )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01056.g0", 0x000000, 0x020000, CRC(8a7ccb9d) SHA1(a9dc43dd6427e3dda0f8c19c7b1eb5a09d80dc89) )
	ROM_LOAD16_BYTE( "msa01056.g1", 0x000001, 0x020000, CRC(3f838268) SHA1(57463ec685755b2345a611e46e431de242dfeaae) )
	EP_CCOCK_SOUND
ROM_END

ROM_START( ep_ccocka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01057.g0", 0x000000, 0x020000, CRC(2197b8f5) SHA1(7392fc2ec4a8b2d5c2fbdaf14569ca87ca785ff2) )
	ROM_LOAD16_BYTE( "msa01057.g1", 0x000001, 0x020000, CRC(5ab369f8) SHA1(a3d356fb09322d010fd8e801329a00764192175b) )
	EP_CCOCK_SOUND
ROM_END

ROM_START( ep_ccockb ) // why are these larger?
	ROM_REGION( 0x100000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "c_cockt.g0", 0x000000, 0x080000, CRC(55d023b7) SHA1(330d0289d2864ab7f00926f87231e8b32bda7e7d) )
	ROM_LOAD16_BYTE( "c_cockt.g1", 0x000001, 0x080000, CRC(8bc575f1) SHA1(93372f807172484c45e6165ebe5c18ff9799f924) )
	EP_CCOCK_SOUND
ROM_END

ROM_START( ep_ccockc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cocktail.g0", 0x000000, 0x020000, CRC(e4550c35) SHA1(2eaef1ee9df1e630a4bb4075c27683439d42f69b) )
	ROM_LOAD16_BYTE( "cocktail.g1", 0x000001, 0x020000, CRC(873a1b30) SHA1(e0b0e7415b6020487e284c066d6f2d324ec23df4) )
	EP_CCOCK_SOUND
ROM_END


ROM_START( ep_cgord )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cag205g0", 0x000000, 0x020000, CRC(c8da6357) SHA1(1ddea39c0ac2f3e97edcf01b6ce47cbc7660d822) )
	ROM_LOAD16_BYTE( "cag205g1", 0x000001, 0x020000, CRC(0975549d) SHA1(40bbf9f537db2149a5643d1f1dcc0c82f8ac8eac) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cgorda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02215.g0", 0x000000, 0x020000, CRC(f9ab0aa3) SHA1(5b01785f284705c5e74d457e46f42c8f6270e319) )
	ROM_LOAD16_BYTE( "msa02215.g1", 0x000001, 0x020000, CRC(5239ba25) SHA1(eac7681a01ed3e3be5d958517f782c66f6eeabfe) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cgordb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02216.g0", 0x000000, 0x020000, CRC(0ef19f6f) SHA1(006f76c0c5ea10beaeeee5ffbb4fe9a32e4f66b4) )
	ROM_LOAD16_BYTE( "msa02216.g1", 0x000001, 0x020000, CRC(661a054b) SHA1(433b5b73445b5f746f56caa8c5289d7b3991be4f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cgordc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cashgord.g0", 0x000000, 0x020000, CRC(8c252b86) SHA1(06f9d3fa0a04f1e2c8281c56e888f3830e79d229) )
	ROM_LOAD16_BYTE( "cashgord.g1", 0x000001, 0x020000, CRC(5740d927) SHA1(ca8aa9f7d650920a9a2d579e1344bdbcba343716) )
	MISSING_SOUND
ROM_END




ROM_START( ep_cshpn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02220.g0", 0x000000, 0x020000, CRC(063796bd) SHA1(9111e72d85b75a76c30d4d252c68c01a1b74e838) )
	ROM_LOAD16_BYTE( "msa02220.g1", 0x000001, 0x020000, CRC(075dd936) SHA1(8d51c6671cfd319ed02b126ba75d15c1287aba72) )
	ROM_LOAD16_BYTE( "msa02220.g2", 0x040000, 0x020000, CRC(e9b8ddd8) SHA1(04d2be1b4817556b306a99a28d28f8a0b465cf1e) )
	ROM_LOAD16_BYTE( "msa02220.g3", 0x040001, 0x020000, CRC(5350a912) SHA1(27f700d67a6b2e5e11d168e23ee704163b8b5402) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cshpna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02221.g0", 0x000000, 0x020000, CRC(c44d1516) SHA1(545595b1d3865761931a813a9633f4a186448b0f) )
	ROM_LOAD16_BYTE( "msa02221.g1", 0x000001, 0x020000, CRC(caafb0fc) SHA1(09df22160b40ddaafadc542480da5cb5dfb0e5bc) )
	ROM_LOAD16_BYTE( "msa02221.g2", 0x040000, 0x020000, CRC(e9b8ddd8) SHA1(04d2be1b4817556b306a99a28d28f8a0b465cf1e) )
	ROM_LOAD16_BYTE( "msa02221.g3", 0x040001, 0x020000, CRC(5350a912) SHA1(27f700d67a6b2e5e11d168e23ee704163b8b5402) )
	MISSING_SOUND
ROM_END



ROM_START( ep_ctit )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02170.g0", 0x000000, 0x020000, CRC(4cdde115) SHA1(c6269cebf6d0fea6f5f09bcff59371c950c1e513) )
	ROM_LOAD16_BYTE( "msa02170.g1", 0x000001, 0x020000, CRC(d3b89977) SHA1(6096085ea30f06688fa081f604ec2635ca897d3b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ctita )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02171.g0", 0x000000, 0x020000, CRC(96dfa4ac) SHA1(4e1705ad78cd9be6eed46ab59628383d04096d34) )
	ROM_LOAD16_BYTE( "msa02171.g1", 0x000001, 0x020000, CRC(cf90ad24) SHA1(0ff80c4b4300ce1453a2f015093455648af03eeb) )
	MISSING_SOUND
ROM_END



ROM_START( ep_cashn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cshi10.p0", 0x000000, 0x020000, CRC(c3362edb) SHA1(baa814f1bd7640ff989bd883c0f7afbccc89cba2) )
	ROM_LOAD16_BYTE( "cshi10.p1", 0x000001, 0x020000, CRC(45096db3) SHA1(7d7fd02759cf2b9c804ac80afaf5e8d5f648196c) )
	ROM_LOAD16_BYTE( "cshi10.p2", 0x040000, 0x020000, CRC(99fbd076) SHA1(fb2065d11860ed9bfed761f4fd433d7fd969ddfb) )
	ROM_LOAD16_BYTE( "cshi10.p3", 0x040001, 0x020000, CRC(08350ac4) SHA1(bfae50a9621159bccbb095f3c3f4b999fe4c1d53) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cashna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cshi10p.p0", 0x000000, 0x020000, CRC(c3362edb) SHA1(baa814f1bd7640ff989bd883c0f7afbccc89cba2) )
	ROM_LOAD16_BYTE( "cshi10p.p1", 0x000001, 0x020000, CRC(5a9a9fc8) SHA1(b50e7d819b262ca2d75ddae22bf2149809db1fea) )
	ROM_LOAD16_BYTE( "cshi10p.p2", 0x040000, 0x020000, CRC(99fbd076) SHA1(fb2065d11860ed9bfed761f4fd433d7fd969ddfb) )
	ROM_LOAD16_BYTE( "cshi10p.p3", 0x040001, 0x020000, CRC(08350ac4) SHA1(bfae50a9621159bccbb095f3c3f4b999fe4c1d53) )
	MISSING_SOUND
ROM_END


ROM_START( ep_casgc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02052.g0", 0x000000, 0x020000, CRC(21d8ea28) SHA1(64f18e3cf4ae35a4e5645f9c311a00c2e5033eda) )
	ROM_LOAD16_BYTE( "msc02052.g1", 0x000001, 0x020000, CRC(a304f488) SHA1(00827120dc53cf30a05d9ae3275e8d7671dee8a2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_casgca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02053.g0", 0x000000, 0x020000, CRC(5f5f2dfd) SHA1(b2dc34cffb8b0884289c710504cf379a83f304eb) )
	ROM_LOAD16_BYTE( "msc02053.g1", 0x000001, 0x020000, CRC(77b2a381) SHA1(9a813c03950620697555fc6b1c65dbaeab87799d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_casgcb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02068.g0", 0x000000, 0x020000, CRC(b60e8ad1) SHA1(773bd8d9e60c13cc209e0cacb07bf9a41638e6c4) )
	ROM_LOAD16_BYTE( "msc02068.g1", 0x000001, 0x020000, CRC(d9cee625) SHA1(c960fd4b7551ee7d493425fdb6cbf9c5a1b3d175) )
	MISSING_SOUND
ROM_END

ROM_START( ep_casgcc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02069.g0", 0x000000, 0x020000, CRC(e82209ce) SHA1(f73d03641ba434b072e764e9eb47b93d43f514c7) )
	ROM_LOAD16_BYTE( "msc02069.g1", 0x000001, 0x020000, CRC(4644c996) SHA1(6f299248a90f3d512b8f2bba6894e054f333c22d) )
	MISSING_SOUND
ROM_END




ROM_START( ep_casrd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02048.g0", 0x000000, 0x020000, CRC(d76a81dd) SHA1(4c1f1036a4e17f395ef22ffd741100afe0639204) )
	ROM_LOAD16_BYTE( "msc02048.g1", 0x000001, 0x020000, CRC(e949b126) SHA1(dc0ba42358d8885c81338959b674d27f33d33e13) )
	MISSING_SOUND
ROM_END

ROM_START( ep_casrda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02049.g0", 0x000000, 0x020000, CRC(8cbea418) SHA1(1edd7cb94d7bb1ad6518bf93bb2a447f03e6e5c2) )
	ROM_LOAD16_BYTE( "msc02049.g1", 0x000001, 0x020000, CRC(836fb72b) SHA1(42e9206502d45662d401cdc8fc3833ba672e7de9) )
	MISSING_SOUND
ROM_END



ROM_START( ep_chock )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01058.g0", 0x000000, 0x020000, CRC(db9485bc) SHA1(82c54d0f53ccb54c32773230fba9c0d24c63f687) )
	ROM_LOAD16_BYTE( "msa01058.g1", 0x000001, 0x020000, CRC(5256da67) SHA1(7ffac98b43f16a2129eff0d6e9e924324d86476f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_chocka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01059.g0", 0x000000, 0x020000, CRC(95eda4c6) SHA1(c6057cc47a487abcead05e184e8517b32ef618e9) )
	ROM_LOAD16_BYTE( "msa01059.g1", 0x000001, 0x020000, CRC(48ff3ce5) SHA1(c2ac0b94c8175b6b2fa8b00428f822060d322f4b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_chockb )
	ROM_REGION( 0x100000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "chocs_aw.g0", 0x000000, 0x080000, CRC(f84d5d17) SHA1(39cc1fc0d6a3ca6b7a7040b19b5a7da157f384ec) )
	ROM_LOAD16_BYTE( "chocs_aw.g1", 0x000001, 0x080000, CRC(73c38daf) SHA1(79da1d3a4b3841ba53d61b4afa3ae254ad4d067a) )
	MISSING_SOUND
ROM_END




ROM_START( ep_cock )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-4g.p0", 0x000000, 0x020000, CRC(c6af132b) SHA1(9ea388f5f64d8702d8c49d066eda871eecaffc67) )
	ROM_LOAD16_BYTE( "cock1-4g.p1", 0x000001, 0x020000, CRC(02f0c51d) SHA1(6eb95d9ec0f2c57bbe903e5fb1c5fb96400386a1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cocka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-4n.p0", 0x000000, 0x020000, CRC(c293f084) SHA1(e34216bec6eb1e2c47ef50ded1fd353ca4d4aa38) )
	ROM_LOAD16_BYTE( "cock1-4n.p1", 0x000001, 0x020000, CRC(ff20cb44) SHA1(f957305d0116b66b94d6a42be9a9f368ac88ed0a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cockb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-4p.p0", 0x000000, 0x020000, CRC(d6912420) SHA1(a434039de1bc7e6c7560a1becf2554a2f149fbc1) )
	ROM_LOAD16_BYTE( "cock1-4p.p1", 0x000001, 0x020000, CRC(ff20cb44) SHA1(f957305d0116b66b94d6a42be9a9f368ac88ed0a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cockc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-5g.p0", 0x000000, 0x020000, CRC(adc76a56) SHA1(24ab9e46ea0f4043a7d2a5133d17b585dc336216) )
	ROM_LOAD16_BYTE( "cock1-5g.p1", 0x000001, 0x020000, CRC(eed54aab) SHA1(8d5db7fe680cbdae0e2eb21fcc938cdabe6c6dc2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cockd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-5n.p0", 0x000000, 0x020000, CRC(e1308925) SHA1(7246f78813b0d4a40088e1526dfdd8798baf3d72) )
	ROM_LOAD16_BYTE( "cock1-5n.p1", 0x000001, 0x020000, CRC(1942b2cf) SHA1(4e7189db84a832b24416c849b43b22ba6ff0e940) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cocke )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cock1-5p.p0", 0x000000, 0x020000, CRC(bdf95d5d) SHA1(44bb0e65595c3e30faf78bc7032f0467d01d9068) )
	ROM_LOAD16_BYTE( "cock1-5p.p1", 0x000001, 0x020000, CRC(1942b2cf) SHA1(4e7189db84a832b24416c849b43b22ba6ff0e940) )
	MISSING_SOUND
ROM_END




ROM_START( ep_commd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02148.g0", 0x000000, 0x020000, CRC(9dfa1ac0) SHA1(c5fe139ee18359c786a27870d269ed83f2e038a2) )
	ROM_LOAD16_BYTE( "msa02148.g1", 0x000001, 0x020000, CRC(06faf6f7) SHA1(324b3b5b17ff4e6aafb786c87bc80621e80d67c5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_commda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02149.g0", 0x000000, 0x020000, CRC(361169a8) SHA1(ea5a4a426290f2f3ea0e9ac6c23b3e3b1ccf9f75) )
	ROM_LOAD16_BYTE( "msa02149.g1", 0x000001, 0x020000, CRC(95557da2) SHA1(b03af42a95953a43212e733357c7baf79ad58579) )
	MISSING_SOUND
ROM_END

ROM_START( ep_commdb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02150.g0", 0x000000, 0x020000, CRC(aeb070fd) SHA1(bf10561a1a796c07b270b74f6dd8417c98b419c5) )
	ROM_LOAD16_BYTE( "msa02150.g1", 0x000001, 0x020000, CRC(100f2d2e) SHA1(670717c82de1d0a76c3f13af8038663be2d55846) )
	MISSING_SOUND
ROM_END

ROM_START( ep_commdc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02151.g0", 0x000000, 0x020000, CRC(895ffc98) SHA1(7924d81a771002427bd87df2bed39892673cd88b) )
	ROM_LOAD16_BYTE( "msa02151.g1", 0x000001, 0x020000, CRC(0af4d5b3) SHA1(0b037b6eec86971f4f8b8b408124eba190d68ca1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_commdd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "comp_mad.g0", 0x000000, 0x020000, CRC(62635ff9) SHA1(6b3b8bbb2661e778b2f0b978dc36736576100ce7) )
	ROM_LOAD16_BYTE( "comp_mad.g1", 0x000001, 0x020000, CRC(7a4b3177) SHA1(c77ef01c9cfe9662f417d6ac7609c4864f2e8d43) )
	MISSING_SOUND
ROM_END




ROM_START( ep_cor2 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02160.g0", 0x000000, 0x020000, CRC(e231313e) SHA1(761ff14a5eb6621e1c9254aff07d793a4912c14d) )
	ROM_LOAD16_BYTE( "msa02160.g1", 0x000001, 0x020000, CRC(8ba2099f) SHA1(f7bc25df270ba5965dc5e4e6d23a330bf7a3e854) )
	ROM_LOAD16_BYTE( "msa02160.g2", 0x040000, 0x020000, CRC(74ed2202) SHA1(f8f046214f42269abe0d993af0a7e3d9843fed9f) )
	ROM_LOAD16_BYTE( "msa02160.g3", 0x040001, 0x020000, CRC(a08d2e8d) SHA1(8ec6c96b968044fc09622c40b35c1a4b035e0b15) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cor2a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02161.g0", 0x000000, 0x020000, CRC(fd4b3bb5) SHA1(84a6a030f3dcd4dfdc243720d4b35510dc211c4a) )
	ROM_LOAD16_BYTE( "msa02161.g1", 0x000001, 0x020000, CRC(56f4ed0e) SHA1(ea2cece8af9eb098a760488396317ebcce1d834c) )
	ROM_LOAD16_BYTE( "msa02161.g2", 0x040000, 0x020000, CRC(74ed2202) SHA1(f8f046214f42269abe0d993af0a7e3d9843fed9f) )
	ROM_LOAD16_BYTE( "msa02161.g3", 0x040001, 0x020000, CRC(a08d2e8d) SHA1(8ec6c96b968044fc09622c40b35c1a4b035e0b15) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cor2b )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02242.g0", 0x000000, 0x020000, CRC(de27a157) SHA1(34dec78663fe99bed2b5629f5ae2a99d49790f1a) )
	ROM_LOAD16_BYTE( "msa02242.g1", 0x000001, 0x020000, CRC(06c858fb) SHA1(d84a84bd521213e8bab78af3c2869942da6066d3) )
	ROM_LOAD16_BYTE( "msa02242.g2", 0x040000, 0x020000, CRC(5a83ac36) SHA1(00e1d508d1d29468f0886d848838f0f33c092803) )
	ROM_LOAD16_BYTE( "msa02242.g3", 0x040001, 0x020000, CRC(9cfb4eb2) SHA1(d4affe7f119eef3d72855ed56af928a74afb5c6d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cor2c )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02243.g0", 0x000000, 0x020000, CRC(8634b9ed) SHA1(a0fe0f0ca1728ad7d5b786b15b7e6b271fa71228) )
	ROM_LOAD16_BYTE( "msa02243.g1", 0x000001, 0x020000, CRC(809fc053) SHA1(49a03cc7987cfa7e27eb8b233e845efa4be64026) )
	ROM_LOAD16_BYTE( "msa02243.g2", 0x040000, 0x020000, CRC(5a83ac36) SHA1(00e1d508d1d29468f0886d848838f0f33c092803) )
	ROM_LOAD16_BYTE( "msa02243.g3", 0x040001, 0x020000, CRC(9cfb4eb2) SHA1(d4affe7f119eef3d72855ed56af928a74afb5c6d) )
	MISSING_SOUND
ROM_END


ROM_START( ep_cormn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02036.g0", 0x000000, 0x020000, CRC(544070c6) SHA1(5672e94dd190ff4962534388327e67b2102a80d0) )
	ROM_LOAD16_BYTE( "msc02036.g1", 0x000001, 0x020000, CRC(f4e0155e) SHA1(0281c81a1217af5a2f9dcfec67833d15650a7758) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cormna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02037.g0", 0x000000, 0x020000, CRC(323d4921) SHA1(62e1ee9b3cbdde3b526f60c490cb0987ed561abc) )
	ROM_LOAD16_BYTE( "msc02037.g1", 0x000001, 0x020000, CRC(a895304b) SHA1(635df164449570b9c541be93e3274d2bd71b490c) )
	MISSING_SOUND
ROM_END



ROM_START( ep_cycl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02038.g0", 0x000000, 0x020000, CRC(cb1eaa56) SHA1(b7ce3b9af70ca609e15a50cbb942f836def27be8) )
	ROM_LOAD16_BYTE( "msc02038.g1", 0x000001, 0x020000, CRC(5205278b) SHA1(bc18cd209076ac39cc2475f14fba4af11df95d52) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cycla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02039.g0", 0x000000, 0x020000, CRC(70a07b5b) SHA1(94b0c8271acbc57961e0e2c03c48faf919924919) )
	ROM_LOAD16_BYTE( "msc02039.g1", 0x000001, 0x020000, CRC(a2bbb208) SHA1(acec4ee8ce4786aa6a5554b81662f5c563c97879) )
	MISSING_SOUND
ROM_END


ROM_START( ep_cyclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cyclone.g0", 0x000000, 0x020000, CRC(3f98b4ad) SHA1(701865863e84aefade7de05f2fbf98c78db2cef7) )
	ROM_LOAD16_BYTE( "cyclone.g1", 0x000001, 0x020000, CRC(00b0e152) SHA1(9741b59b9756480f79e4ecdafbeb0a8d6e30ccba) )
	MISSING_SOUND
ROM_END


#define EP_DUFF_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "duff.s0", 0x000000, 0x080000, CRC(639b6528) SHA1(2ff6b84531f3a2a18e5c846b7a2b14102faf8864) ) \
	ROM_LOAD( "duff.s1", 0x080000, 0x080000, CRC(5c18f8cb) SHA1(c850c386b0b00ba982bd473a2ba4d464bbfd57d0) )

ROM_START( ep_duff )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01090.g0", 0x000000, 0x020000, CRC(97231280) SHA1(5850b89be58dd780e712784dc1bf313f561d60ba) )
	ROM_LOAD16_BYTE( "msa01090.g1", 0x000001, 0x020000, CRC(27361d06) SHA1(0c1360e67f3c45044b586644371f7e3d86d43b48) )
	ROM_LOAD16_BYTE( "msa01090.g2", 0x040000, 0x020000, CRC(b5b11374) SHA1(65a6d69431fd4342b9884ca07caa6c1cfa07c049) )
	ROM_LOAD16_BYTE( "msa01090.g3", 0x040001, 0x020000, CRC(386fce4c) SHA1(057fcba00569f9827fd7d68d39de3786d4fa47e4) )
	EP_DUFF_SOUND
ROM_END

ROM_START( ep_duffa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01091.g0", 0x000000, 0x020000, CRC(97231280) SHA1(5850b89be58dd780e712784dc1bf313f561d60ba) )
	ROM_LOAD16_BYTE( "msa01091.g1", 0x000001, 0x020000, CRC(cca8493e) SHA1(1907253363997ee5bddbeedb6350b53815564ff4) )
	ROM_LOAD16_BYTE( "msa01091.g2", 0x040000, 0x020000, CRC(b5b11374) SHA1(65a6d69431fd4342b9884ca07caa6c1cfa07c049) )
	ROM_LOAD16_BYTE( "msa01091.g3", 0x040001, 0x020000, CRC(386fce4c) SHA1(057fcba00569f9827fd7d68d39de3786d4fa47e4) )
	EP_DUFF_SOUND
ROM_END

ROM_START( ep_duffb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000031.g0", 0x000000, 0x020000, CRC(e42f3045) SHA1(016dc0b9ec507d7531b17338aef9d4738fa04a23) )
	ROM_LOAD16_BYTE( "sa000031.g1", 0x000001, 0x020000, CRC(26032e79) SHA1(363b32aab6fa3d0aeaba5cbdb8d3a82dcf70f64a) )
	EP_DUFF_SOUND
ROM_END

ROM_START( ep_duffc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000032.g0", 0x000000, 0x020000, CRC(e42f3045) SHA1(016dc0b9ec507d7531b17338aef9d4738fa04a23) )
	ROM_LOAD16_BYTE( "sa000032.g1", 0x000001, 0x020000, CRC(cd9d7a41) SHA1(9f0e177145f7c4b350b71dcdc9cc6479fc352ec2) )
	EP_DUFF_SOUND
ROM_END


#define EP_FLASH_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "fl_back.s0", 0x0000, 0x080000, CRC(acaa4b38) SHA1(6960e29181f67ca77fee9f6bce84dfa6bea4bdf7) ) \
	ROM_LOAD( "fl_back.s1", 0x0000, 0x080000, CRC(4ff8b006) SHA1(bfe7b83b1bcc81f5980263db26e0c3dd8074518b) )

ROM_START( ep_flash )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfl35.g0", 0x000000, 0x020000, CRC(dd32eeb0) SHA1(811a7f5b03cb24552282ba19962b99b794afcab5) )
	ROM_LOAD16_BYTE( "i99pfl35.g1", 0x000001, 0x020000, CRC(fa5a31d9) SHA1(79a6bc70882444094be384e0baab891c53de126f) )
	ROM_LOAD16_BYTE( "i99pfl35.g2", 0x040000, 0x020000, CRC(e05997ec) SHA1(b69cb3a62b6a28983ea24eb02464dd5f2f73d0a2) )
	ROM_LOAD16_BYTE( "i99pfl35.g3", 0x040001, 0x020000, CRC(b08851c5) SHA1(3d0734bf934728afd10cce9d19b6877384a15a17) )
	EP_FLASH_SOUND
ROM_END

ROM_START( ep_flasha )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfl30.g0", 0x000000, 0x020000, CRC(05c9fe87) SHA1(cfc507b1072de05ee084f0ef46fa1c8a66784ad0) )
	ROM_LOAD16_BYTE( "i99nfl30.g1", 0x000001, 0x020000, CRC(88f7bdd5) SHA1(07904e2c1d928310e89275d94a29ad1a01667bce) )
	EP_FLASH_SOUND
ROM_END

ROM_START( ep_flashb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfl32.g0", 0x000000, 0x020000, CRC(05c9fe87) SHA1(cfc507b1072de05ee084f0ef46fa1c8a66784ad0) )
	ROM_LOAD16_BYTE( "i99nfl32.g1", 0x000001, 0x020000, CRC(84ba13e4) SHA1(f44d72ede542d10e3bbe9a472e42b4c98f90af14) )
	EP_FLASH_SOUND
ROM_END

ROM_START( ep_flashc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nfl34.g0", 0x000000, 0x020000, CRC(dd32eeb0) SHA1(811a7f5b03cb24552282ba19962b99b794afcab5) )
	ROM_LOAD16_BYTE( "i99nfl34.g1", 0x000001, 0x020000, CRC(11c465e1) SHA1(76504ebc31b3824a2bfb80c9967fa5774604380a) )
	ROM_LOAD16_BYTE( "i99nfl34.g2", 0x040000, 0x020000, CRC(e05997ec) SHA1(b69cb3a62b6a28983ea24eb02464dd5f2f73d0a2) )
	ROM_LOAD16_BYTE( "i99nfl34.g3", 0x040001, 0x020000, CRC(b08851c5) SHA1(3d0734bf934728afd10cce9d19b6877384a15a17) )
	EP_FLASH_SOUND
ROM_END

ROM_START( ep_flashd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfl31.g0", 0x000000, 0x020000, CRC(05c9fe87) SHA1(cfc507b1072de05ee084f0ef46fa1c8a66784ad0) )
	ROM_LOAD16_BYTE( "i99pfl31.g1", 0x000001, 0x020000, CRC(6369e9ed) SHA1(f680b62729f6e2ed56a34e2bb4a45ead57bf5f6c) )
	EP_FLASH_SOUND
ROM_END


ROM_START( ep_flashe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pfl33.g0", 0x000000, 0x020000, CRC(05c9fe87) SHA1(cfc507b1072de05ee084f0ef46fa1c8a66784ad0) )
	ROM_LOAD16_BYTE( "i99pfl33.g1", 0x000001, 0x020000, CRC(6f2447dc) SHA1(d485ad084b22be3ef7f9496edee7c43cb817bc9a) )
	EP_FLASH_SOUND
ROM_END

ROM_START( ep_flashf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "fl_back.g0", 0x000000, 0x020000, CRC(dcf838f0) SHA1(1151d7c3ab2379b1ef7998e159e2d31fad840966) )
	ROM_LOAD16_BYTE( "fl_back.g1", 0x000001, 0x020000, CRC(6ddf2a50) SHA1(4460cfc5ed3e0b5d5a1932d33e3f6ca03daa1469) )
	EP_FLASH_SOUND
ROM_END


#define EP_FORTG_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "f_n_g.s0", 0x000000, 0x080000, CRC(e257502d) SHA1(5be9ea39299582c0c68feeb9537d887c7562b293) ) \
	ROM_LOAD( "f_n_g.s1", 0x080000, 0x080000, CRC(87e2f4bd) SHA1(b515b294384dbfa24bf3d7a561a572aa9c041831) )
ROM_START( ep_fortg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "f_n_gl.g0", 0x000000, 0x020000, CRC(ea112032) SHA1(2e06447058f5d697fc4263348106342faf9c41ab) )
	ROM_LOAD16_BYTE( "f_n_gl.g1", 0x000001, 0x020000, CRC(c35cbb80) SHA1(401e81a8ec2c23b3494f4603a4f9f6fb5f83c4dd) )
	EP_FORTG_SOUND
ROM_END

ROM_START( ep_fortga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nfg35.g0", 0x000000, 0x020000, CRC(99fd5271) SHA1(331ae4edafb13caae93a2a4e3f9bc53a56a081fa) )
	ROM_LOAD16_BYTE( "i01nfg35.g1", 0x000001, 0x020000, CRC(ce7012d0) SHA1(b2496ee23a135920bceab5a2b8fbbc5d06ffcaa2) )
	EP_FORTG_SOUND
ROM_END

ROM_START( ep_fortgb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01pfg36.g0", 0x000000, 0x020000, CRC(99fd5271) SHA1(331ae4edafb13caae93a2a4e3f9bc53a56a081fa) )
	ROM_LOAD16_BYTE( "i01pfg36.g1", 0x000001, 0x020000, CRC(25ee46e8) SHA1(44f86e9fad5ecf22a02d124b9792828f4b0e0791) )
	EP_FORTG_SOUND
ROM_END






ROM_START( ep_fran )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nfr06.g0", 0x000000, 0x020000, CRC(523edd45) SHA1(b519009982213aba84c11498782f8eda6c91a62e) )
	ROM_LOAD16_BYTE( "i01nfr06.g1", 0x000001, 0x020000, CRC(c43e874a) SHA1(2f984df79cdfd9da2001417a2bdf217b061ee092) )
	ROM_LOAD16_BYTE( "i01nfr06.g2", 0x040000, 0x020000, CRC(954920c3) SHA1(c58ec622bafd08a33b0d2000019b88ffb1859cd2) )
	ROM_LOAD16_BYTE( "i01nfr06.g3", 0x040001, 0x020000, CRC(f29fb08f) SHA1(522c7bf9684397b3c6ae2cbf61b2f3e3becea8c9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_frana )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01pfr07.g0", 0x000000, 0x020000, CRC(523edd45) SHA1(b519009982213aba84c11498782f8eda6c91a62e) )
	ROM_LOAD16_BYTE( "i01pfr07.g1", 0x000001, 0x020000, CRC(2fa0d372) SHA1(c97237787f9c82cc9dd517b060f84986c37771fa) )
	ROM_LOAD16_BYTE( "i01pfr07.g2", 0x040000, 0x020000, CRC(954920c3) SHA1(c58ec622bafd08a33b0d2000019b88ffb1859cd2) )
	ROM_LOAD16_BYTE( "i01pfr07.g3", 0x040001, 0x020000, CRC(f29fb08f) SHA1(522c7bf9684397b3c6ae2cbf61b2f3e3becea8c9) )
	MISSING_SOUND
ROM_END




ROM_START( ep_fnclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cfnn1-3n.p0", 0x000000, 0x020000, CRC(9694f4f5) SHA1(f2f71d76e1ceb88d67bbec7a357bcca0e50fe332) )
	ROM_LOAD16_BYTE( "cfnn1-3n.p1", 0x000001, 0x020000, CRC(3a42a565) SHA1(8abbee6439de2ec8786b8cd30ae8df4066cc22bb) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fnclba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cfnn1-3p.p0", 0x000000, 0x020000, CRC(21535490) SHA1(480b6c642bdd047ae19afcc5cdd5ee546df92ad7) )
	ROM_LOAD16_BYTE( "cfnn1-3p.p1", 0x000001, 0x020000, CRC(3a42a565) SHA1(8abbee6439de2ec8786b8cd30ae8df4066cc22bb) )
	MISSING_SOUND
ROM_END




ROM_START( ep_fgods )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01003.g0", 0x000000, 0x020000, CRC(5b42b792) SHA1(5e0c869ef3ab64d8b9e225b738f2ec3795f5126c) )
	ROM_LOAD16_BYTE( "msb01003.g1", 0x000001, 0x020000, CRC(aa17121a) SHA1(d7e7227ea828955a4978369a2f963d90ae6376d5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01004.g0", 0x000000, 0x020000, CRC(2a0ade21) SHA1(aa2aabf92e50ff45e889db7f3d537bf35a30c545) )
	ROM_LOAD16_BYTE( "msb01004.g1", 0x000001, 0x020000, CRC(8d6d15a8) SHA1(12c60e375fdc203a0b96fa3226f7d0c9d2287941) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01015.g0", 0x000000, 0x020000, CRC(9e05ab46) SHA1(514db454cdd4f09ad30135a7283be373cbd9f902) )
	ROM_LOAD16_BYTE( "msb01015.g1", 0x000001, 0x020000, CRC(57cc390b) SHA1(f91bcd186788c12a911fd0164ced69b42df322dc) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01016.g0", 0x000000, 0x020000, CRC(b94938bf) SHA1(266e72355f43cdd30c0a3dd63335f4cede4d3290) )
	ROM_LOAD16_BYTE( "msb01016.g1", 0x000001, 0x020000, CRC(90d7e26e) SHA1(27d49db46d30d918630e3e63964e0b490d9b3bb6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02007.g0", 0x000000, 0x020000, CRC(1ecc395d) SHA1(626638c3ea892f08a123026e63f3988c12b83c2e) )
	ROM_LOAD16_BYTE( "mua02007.g1", 0x000001, 0x020000, CRC(67bb8e13) SHA1(8b35c65ef102268863038a297f448e2fd9cc8a42) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodse )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02011.g0", 0x000000, 0x020000, CRC(2a4c401e) SHA1(18f5f0a4f7ccf518571fb7e40309dc8bd236465a) )
	ROM_LOAD16_BYTE( "mua02011.g1", 0x000001, 0x020000, CRC(cb003fc5) SHA1(7c40017fb90568fa8fd33611f96bb31f8543b97d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mub02001.g0", 0x000000, 0x020000, CRC(9fb3e5fd) SHA1(24dde2d3c1b1cd99a9b58ba74d573b43f4e2fc3b) )
	ROM_LOAD16_BYTE( "mub02001.g1", 0x000001, 0x020000, CRC(148523c2) SHA1(e0dfd527f4f38e1864a5add3e21e9c64e6cf0e6f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fgodsg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mub02002.g0", 0x000000, 0x020000, CRC(ab6334e4) SHA1(facc8c5b1d19179aa1eb1a1939eb31c94d2e0208) )
	ROM_LOAD16_BYTE( "mub02002.g1", 0x000001, 0x020000, CRC(042e909c) SHA1(5d3871c9997130526fa9cc1dfc7abc6f6f633b99) )
	MISSING_SOUND
ROM_END











ROM_START( ep_geron )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pge11.g0", 0x000000, 0x020000, CRC(4dcac10e) SHA1(65b83b8d0a46c9d1bc52b7eb4848627b65816b23) )
	ROM_LOAD16_BYTE( "i00pge11.g1", 0x000001, 0x020000, CRC(afdad382) SHA1(c615b929a38bd1fe4afe489ee0534dec53fca37a) )
	ROM_LOAD16_BYTE( "i00pge11.g2", 0x040000, 0x020000, CRC(309c31cc) SHA1(225ea8a01cb131c178a16bfd75aaf6ce8f7a2322) )
	ROM_LOAD16_BYTE( "i00pge11.g3", 0x040001, 0x020000, CRC(0aeb3fad) SHA1(9a971b2c20bec55362bfeec3222162d42a49dfa7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gerona )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nge06.g0", 0x000000, 0x020000, CRC(73c779dc) SHA1(32341abf00bd41ff1e0bb937a57059df6bdfc241) )
	ROM_LOAD16_BYTE( "i00nge06.g1", 0x000001, 0x020000, CRC(75d0974a) SHA1(421466443ddec9906083f37d5cae6dc054ede35a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_geronb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nge08.g0", 0x000000, 0x020000, CRC(73c779dc) SHA1(32341abf00bd41ff1e0bb937a57059df6bdfc241) )
	ROM_LOAD16_BYTE( "i00nge08.g1", 0x000001, 0x020000, CRC(799d397b) SHA1(c1c018e8c61a155f1fd4fb4878261771719b7060) )
	MISSING_SOUND
ROM_END

ROM_START( ep_geronc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nge10.g0", 0x000000, 0x020000, CRC(4dcac10e) SHA1(65b83b8d0a46c9d1bc52b7eb4848627b65816b23) )
	ROM_LOAD16_BYTE( "i00nge10.g1", 0x000001, 0x020000, CRC(444487ba) SHA1(a6d5d0ee05f286fd869eabed25bedbaf3c937669) )
	ROM_LOAD16_BYTE( "i00nge10.g2", 0x040000, 0x020000, CRC(309c31cc) SHA1(225ea8a01cb131c178a16bfd75aaf6ce8f7a2322) )
	ROM_LOAD16_BYTE( "i00nge10.g3", 0x040001, 0x020000, CRC(0aeb3fad) SHA1(9a971b2c20bec55362bfeec3222162d42a49dfa7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gerond )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pge07.g0", 0x000000, 0x020000, CRC(73c779dc) SHA1(32341abf00bd41ff1e0bb937a57059df6bdfc241) )
	ROM_LOAD16_BYTE( "i00pge07.g1", 0x000001, 0x020000, CRC(9e4ec372) SHA1(8061c2ae23766f2b97106b88580be9a5fc974693) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gerone )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pge09.g0", 0x000000, 0x020000, CRC(73c779dc) SHA1(32341abf00bd41ff1e0bb937a57059df6bdfc241) )
	ROM_LOAD16_BYTE( "i00pge09.g1", 0x000001, 0x020000, CRC(92036d43) SHA1(d776cd6d39116645ceeb6218d8ff10c3ac16e591) )
	MISSING_SOUND
ROM_END




ROM_START( ep_gldtp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02226.g0", 0x000000, 0x020000, CRC(e51f8068) SHA1(ecbed1b7c4de2e29ecae10f7d546ebc2df919fb4) )
	ROM_LOAD16_BYTE( "msa02226.g1", 0x000001, 0x020000, CRC(d48a07b1) SHA1(95f754e78b2d8195b161bbddc1a84df1cd97cefb) )
	ROM_LOAD16_BYTE( "msa02226.g2", 0x040000, 0x020000, CRC(b87811d9) SHA1(0e3c0829995a78ab033a5f1097a3923531395df4) )
	ROM_LOAD16_BYTE( "msa02226.g3", 0x040001, 0x020000, CRC(04a4db95) SHA1(ee67dbf616c7fdf00168d5b734d7690f1432d3a5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gldtpa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02227.g0", 0x000000, 0x020000, CRC(9b9847bd) SHA1(1acaf31bf59dc0763eed62368d29b6ab33be6b8e) )
	ROM_LOAD16_BYTE( "msa02227.g1", 0x000001, 0x020000, CRC(a11e3d53) SHA1(b24cf4b8f8ff707f1c2ff81ea709f8ae0fa4db8f) )
	ROM_LOAD16_BYTE( "msa02227.g2", 0x040000, 0x020000, CRC(b87811d9) SHA1(0e3c0829995a78ab033a5f1097a3923531395df4) )
	ROM_LOAD16_BYTE( "msa02227.g3", 0x040001, 0x020000, CRC(04a4db95) SHA1(ee67dbf616c7fdf00168d5b734d7690f1432d3a5) )
	MISSING_SOUND
ROM_END


ROM_START( ep_gesc2 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02250.g0", 0x000000, 0x020000, CRC(1f3b676e) SHA1(dcec262358bba165eda01170f7389d7ebcb5fe08) )
	ROM_LOAD16_BYTE( "msa02250.g1", 0x000001, 0x020000, CRC(ce49cbe5) SHA1(50019de60c9e0f4708aeb1e658021b7d7e82f5cd) )
	ROM_LOAD16_BYTE( "msa02250.g2", 0x040000, 0x020000, CRC(14dfb49c) SHA1(3b1b0f61a0592243899356a110b1c144105e65ff) )
	ROM_LOAD16_BYTE( "msa02250.g3", 0x040001, 0x020000, CRC(4ef89a71) SHA1(25f09afb0af5b2953ebfba45f0ee978d1d64e699) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gesc2a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02251.g0", 0x000000, 0x020000, CRC(71e902de) SHA1(4b8ff04beb36ba58a11f154c29e67b05de98e00b) )
	ROM_LOAD16_BYTE( "msa02251.g1", 0x000001, 0x020000, CRC(afbe9040) SHA1(35d0f1beb252c5d672015fe0ce8cf2c6fb6cc4d9) )
	ROM_LOAD16_BYTE( "msa02251.g2", 0x040000, 0x020000, CRC(14dfb49c) SHA1(3b1b0f61a0592243899356a110b1c144105e65ff) )
	ROM_LOAD16_BYTE( "msa02251.g3", 0x040001, 0x020000, CRC(4ef89a71) SHA1(25f09afb0af5b2953ebfba45f0ee978d1d64e699) )
	MISSING_SOUND
ROM_END




ROM_START( ep_geclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc990031.g0", 0x000000, 0x020000, CRC(373b4674) SHA1(b9bd98a2fc58c62026ef1d5ea05dfd82642f7d53) )
	ROM_LOAD16_BYTE( "sc990031.g1", 0x000001, 0x020000, CRC(6fa89ac3) SHA1(def22f8db35f9003a29364196edeade5bd58d0c5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_geclba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc990032.g0", 0x000000, 0x020000, CRC(1077d58d) SHA1(734ea53b89118c030e0af2dbaddd1dbc40dbbe24) )
	ROM_LOAD16_BYTE( "sc990032.g1", 0x000001, 0x020000, CRC(e4c8384f) SHA1(e62f876d8c1943fb5b6f8bd68ba52716a15383aa) )
	MISSING_SOUND
ROM_END

ROM_START( ep_geclbb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gesccl.g0", 0x00000, 0x020000, CRC(98b4a0c1) SHA1(41810bc3e9da98573e89b7e474623494bf91eb37) )
	ROM_LOAD16_BYTE( "gesccl.g1", 0x00001, 0x020000, CRC(8a38fa1b) SHA1(7cb29582a6ff72dd05dabbd8831bb5ddda3b7fbc) )
	MISSING_SOUND
ROM_END


ROM_START( ep_gresc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02228.g0", 0x000000, 0x020000, CRC(a7dabdeb) SHA1(a23971e65d1327fbed396b0e0fccfcc6f7268a4f) )
	ROM_LOAD16_BYTE( "msa02228.g1", 0x000001, 0x020000, CRC(3b6e9c54) SHA1(ddfe5b8d373e23aa45948a52dd1641be0ce42122) )
	ROM_LOAD16_BYTE( "msa02228.g2", 0x040000, 0x020000, CRC(044b956b) SHA1(402b9f1a9895c5c4a4c08a8611a4e1bb6fe2301c) )
	ROM_LOAD16_BYTE( "msa02228.g3", 0x040001, 0x020000, CRC(e0468eb6) SHA1(3c03ab66a2c76940f5be8eeb20ebcfa6b9c0da71) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gresca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02229.g0", 0x000000, 0x020000, CRC(6dcc03dd) SHA1(eb6cbdf7da075a43ff8f80ead41e8e4850ed8fcb) )
	ROM_LOAD16_BYTE( "msa02229.g1", 0x000001, 0x020000, CRC(36ed039b) SHA1(5da0af2e42bec0f00b5a8ab457ea521b06bd5377) )
	ROM_LOAD16_BYTE( "msa02229.g2", 0x040000, 0x020000, CRC(044b956b) SHA1(402b9f1a9895c5c4a4c08a8611a4e1bb6fe2301c) )
	ROM_LOAD16_BYTE( "msa02229.g3", 0x040001, 0x020000, CRC(e0468eb6) SHA1(3c03ab66a2c76940f5be8eeb20ebcfa6b9c0da71) )
	MISSING_SOUND
ROM_END




ROM_START( ep_gridr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pgr27.g0", 0x000000, 0x020000, CRC(b8af2359) SHA1(26bba05b5d51d959db39894416ce0bef4d9f5734) )
	ROM_LOAD16_BYTE( "i99pgr27.g1", 0x000001, 0x020000, CRC(4870971b) SHA1(5bd4f56bbff8c47b322f2805f784e2177368befc) )
	ROM_LOAD16_BYTE( "i99pgr27.g2", 0x040000, 0x020000, CRC(bf8a7a88) SHA1(d6ad9f4d9a986259ecc76ab64a66e260d3fe467b) )
	ROM_LOAD16_BYTE( "i99pgr27.g3", 0x040001, 0x020000, CRC(7f4e7c2b) SHA1(07de72c4bdcee62287eb6c0da4c1643a3f595578) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gridra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ngr22.g0", 0x000000, 0x020000, CRC(adb8cfd7) SHA1(dd31881cb1ef2d362b94809be696b91d03bd3112) )
	ROM_LOAD16_BYTE( "i99ngr22.g1", 0x000001, 0x020000, CRC(4d4f9b66) SHA1(1de662291e64ff8d71aaa8ce41ea49ea6303cb82) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gridrb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ngr24.g0", 0x000000, 0x020000, CRC(adb8cfd7) SHA1(dd31881cb1ef2d362b94809be696b91d03bd3112) )
	ROM_LOAD16_BYTE( "i99ngr24.g1", 0x000001, 0x020000, CRC(41023557) SHA1(3fb39a5e3c0dc016f9a8d9ace6b2f0bce82b41dd) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gridrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ngr26.g0", 0x000000, 0x020000, CRC(b8af2359) SHA1(26bba05b5d51d959db39894416ce0bef4d9f5734) )
	ROM_LOAD16_BYTE( "i99ngr26.g1", 0x000001, 0x020000, CRC(a3eec323) SHA1(62f5751b1fc47bd33c20afa014cb3f7f019e152d) )
	ROM_LOAD16_BYTE( "i99ngr26.g2", 0x040000, 0x020000, CRC(bf8a7a88) SHA1(d6ad9f4d9a986259ecc76ab64a66e260d3fe467b) )
	ROM_LOAD16_BYTE( "i99ngr26.g3", 0x040001, 0x020000, CRC(7f4e7c2b) SHA1(07de72c4bdcee62287eb6c0da4c1643a3f595578) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gridrd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99pgr23.g0", 0x000000, 0x020000, CRC(adb8cfd7) SHA1(dd31881cb1ef2d362b94809be696b91d03bd3112) )
	ROM_LOAD16_BYTE( "i99pgr23.g1", 0x000001, 0x020000, CRC(a6d1cf5e) SHA1(b1897a0718ca327a2cb5c15e0100118251e11217) )
	ROM_LOAD16_BYTE( "i99pgr25.g0", 0x000000, 0x020000, CRC(adb8cfd7) SHA1(dd31881cb1ef2d362b94809be696b91d03bd3112) )
	ROM_LOAD16_BYTE( "i99pgr25.g1", 0x000001, 0x020000, CRC(aa9c616f) SHA1(9e7eb848aaf15a4ed02df50de67cb30de6c096bb) )
	MISSING_SOUND
ROM_END


#define EP_HHCLB_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) /* guess based on name */ \
	ROM_LOAD( "h_house.s0", 0x000000, 0x080000, CRC(387fb6bb) SHA1(e532a74792792c17347560cdc5aa890223b99abc) ) \
	ROM_LOAD( "h_house.s1", 0x080000, 0x080000, CRC(a8df24cb) SHA1(b57590c128ce7cc40050ed1149741ca2c16e656b) )

ROM_START( ep_hhclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02044.g0", 0x000000, 0x020000, CRC(286f1d43) SHA1(44ac5d7aedcc5466debd3879ec97729813472faa) )
	ROM_LOAD16_BYTE( "msc02044.g1", 0x000001, 0x020000, CRC(e5d8437b) SHA1(10f863b522f75b9540b0a08bc33ee51ad4f4df79) )
	ROM_LOAD16_BYTE( "msc02044.g2", 0x040000, 0x020000, CRC(d5af8708) SHA1(e13e085186b90bea81a171e92f6056cb3c3245ab) )
	ROM_LOAD16_BYTE( "msc02044.g3", 0x040001, 0x020000, CRC(493e0979) SHA1(d2b20ee23dc2a2eaf6963a2f536d5303c2037570) )
	EP_HHCLB_SOUND
ROM_END

ROM_START( ep_hhclba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02045.g0", 0x000000, 0x020000, CRC(a32f2ae1) SHA1(44961cc4af575dbcb894372ebdf5b080b5964c14) )
	ROM_LOAD16_BYTE( "msc02045.g1", 0x000001, 0x020000, CRC(cd34bfcc) SHA1(d32a085220b5b8d0896d765852244cec0299c331) )
	ROM_LOAD16_BYTE( "msc02045.g2", 0x040000, 0x020000, CRC(d5af8708) SHA1(e13e085186b90bea81a171e92f6056cb3c3245ab) )
	ROM_LOAD16_BYTE( "msc02045.g3", 0x040001, 0x020000, CRC(493e0979) SHA1(d2b20ee23dc2a2eaf6963a2f536d5303c2037570) )
	EP_HHCLB_SOUND
ROM_END

ROM_START( ep_hhclbb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "h_h0use.0", 0x000000, 0x020000, CRC(71476052) SHA1(d97785b7e64c30d4bfdd12ffd68e3d1f1e9dd311) )
	ROM_LOAD16_BYTE( "h_house.1", 0x000001, 0x020000, CRC(5b2d7fd7) SHA1(885221d0ec354dc6bb8813fddd8db1729ec07d3b) )
	ROM_LOAD16_BYTE( "h_house.2", 0x040000, 0x020000, CRC(3277bba8) SHA1(b6b7005d3a056f9bcbc6960b50cb7acabe1781dd) )
	ROM_LOAD16_BYTE( "h_house.3", 0x040001, 0x020000, CRC(262e99ea) SHA1(a34a23bd9137e5cb5bddaad6f8f08f3753e1b0e0) )
	EP_HHCLB_SOUND
ROM_END

ROM_START( ep_hhclbc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "h_house.g0", 0x000000, 0x020000, CRC(71476052) SHA1(d97785b7e64c30d4bfdd12ffd68e3d1f1e9dd311) )
	ROM_LOAD16_BYTE( "h_house.g1", 0x000001, 0x020000, CRC(5b2d7fd7) SHA1(885221d0ec354dc6bb8813fddd8db1729ec07d3b) )
	ROM_LOAD16_BYTE( "h_house.g2", 0x040000, 0x020000, CRC(3277bba8) SHA1(b6b7005d3a056f9bcbc6960b50cb7acabe1781dd) )
	ROM_LOAD16_BYTE( "h_house.g3", 0x040001, 0x020000, CRC(9acfcbe8) SHA1(bada40fc9886aa59c6c9a793ef904d9d82a9efe5) )
	EP_HHCLB_SOUND
ROM_END



ROM_START( ep_hvns )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02nha05.g0", 0x000000, 0x020000, CRC(3dd8672f) SHA1(d88cdf2e056d145eba98d9bcc6c26f89dbc97369) )
	ROM_LOAD16_BYTE( "i02nha05.g1", 0x000001, 0x020000, CRC(9873ebd1) SHA1(7b8f890dc8f4309ee04ed73209054222938696d8) )
	ROM_LOAD16_BYTE( "i02nha05.g2", 0x040000, 0x020000, CRC(ad48d982) SHA1(8281abb862c2da5650fa0e126a61f690ef943a54) )
	ROM_LOAD16_BYTE( "i02nha05.g3", 0x040001, 0x020000, CRC(ee793d9f) SHA1(b53df0207645bf0652f247a9f1599766c1703167) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hvnsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02pha06.g0", 0x000000, 0x020000, CRC(3dd8672f) SHA1(d88cdf2e056d145eba98d9bcc6c26f89dbc97369) )
	ROM_LOAD16_BYTE( "i02pha06.g1", 0x000001, 0x020000, CRC(73edbfe9) SHA1(a564e8664c61c785fe5a6b0c89996bdb9b1bdfac) )
	ROM_LOAD16_BYTE( "i02pha06.g2", 0x040000, 0x020000, CRC(ad48d982) SHA1(8281abb862c2da5650fa0e126a61f690ef943a54) )
	ROM_LOAD16_BYTE( "i02pha06.g3", 0x040001, 0x020000, CRC(ee793d9f) SHA1(b53df0207645bf0652f247a9f1599766c1703167) )
	MISSING_SOUND
ROM_END




ROM_START( ep_highv )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phv27.g0", 0x000000, 0x020000, CRC(1a022252) SHA1(1f1c4263459688e4418f98358b05e0f917c2a356) )
	ROM_LOAD16_BYTE( "i00phv27.g1", 0x000001, 0x020000, CRC(745b7ff5) SHA1(ba95f005b8c6bcf7f895883aeb41bf0562c8dd25) )
	ROM_LOAD16_BYTE( "i00phv27.g2", 0x040000, 0x020000, CRC(8bbeba8d) SHA1(d5f39c5acd6ffe11c0b2427b950dd3c8320792ec) )
	ROM_LOAD16_BYTE( "i00phv27.g3", 0x040001, 0x020000, CRC(a6b747f6) SHA1(6e194a2c0cef6fb812fbd68df5779a36633b0290) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highva )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhv22.g0", 0x000000, 0x020000, CRC(4f97ea5c) SHA1(32744b8875362b5854338e9bbba0be7361c0661d) )
	ROM_LOAD16_BYTE( "i00nhv22.g1", 0x000001, 0x020000, CRC(2c9a11f7) SHA1(f632dbe8387885adac31a3c3c14b25708250e4ad) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highvb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhv24.g0", 0x000000, 0x020000, CRC(4f97ea5c) SHA1(32744b8875362b5854338e9bbba0be7361c0661d) )
	ROM_LOAD16_BYTE( "i00nhv24.g1", 0x000001, 0x020000, CRC(20d7bfc6) SHA1(39d573b5ddefe30b9805f4ecad1cc5a7e3505d63) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highvc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhv26.g0", 0x000000, 0x020000, CRC(1a022252) SHA1(1f1c4263459688e4418f98358b05e0f917c2a356) )
	ROM_LOAD16_BYTE( "i00nhv26.g1", 0x000001, 0x020000, CRC(9fc52bcd) SHA1(eefdb1f8f7af993363efa2b666b887bed89f038d) )
	ROM_LOAD16_BYTE( "i00nhv26.g2", 0x040000, 0x020000, CRC(8bbeba8d) SHA1(d5f39c5acd6ffe11c0b2427b950dd3c8320792ec) )
	ROM_LOAD16_BYTE( "i00nhv26.g3", 0x040001, 0x020000, CRC(a6b747f6) SHA1(6e194a2c0cef6fb812fbd68df5779a36633b0290) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highvd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phv23.g0", 0x000000, 0x020000, CRC(4f97ea5c) SHA1(32744b8875362b5854338e9bbba0be7361c0661d) )
	ROM_LOAD16_BYTE( "i00phv23.g1", 0x000001, 0x020000, CRC(c70445cf) SHA1(b038c596e3c62638f641bf6848446a0b0827dd3e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highve )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phv25.g0", 0x000000, 0x020000, CRC(4f97ea5c) SHA1(32744b8875362b5854338e9bbba0be7361c0661d) )
	ROM_LOAD16_BYTE( "i00phv25.g1", 0x000001, 0x020000, CRC(cb49ebfe) SHA1(1de1042dee5a3b36cd09bfc0f175c0a3ab1bb095) )
	MISSING_SOUND
ROM_END

ROM_START( ep_highvf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hi_volt.g0", 0x000000, 0x020000, CRC(863a7c78) SHA1(2de7fed4bd78508c1ae985ad97a4a265d5789c78) )
	ROM_LOAD16_BYTE( "hi_volt.g1", 0x000001, 0x020000, CRC(7c579950) SHA1(37e6edbfd0d969002fd98d2be824a9b85223d0ec) )
	MISSING_SOUND
ROM_END




ROM_START( ep_hogmn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nhm14.g0", 0x000000, 0x020000, CRC(51022b50) SHA1(47cb0d9b4a6012f87e12f07e2adcdd841d1e336a) )
	ROM_LOAD16_BYTE( "i99nhm14.g1", 0x000001, 0x020000, CRC(dcdbabc3) SHA1(7ee8529787f68de5973fad385b5821c523a87294) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hogmna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nhm16.g0", 0x000000, 0x020000, CRC(51022b50) SHA1(47cb0d9b4a6012f87e12f07e2adcdd841d1e336a) )
	ROM_LOAD16_BYTE( "i99nhm16.g1", 0x000001, 0x020000, CRC(d09605f2) SHA1(f4c9ac144c43fd653a73394b70ddde3cae5ed2db) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hogmnb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99phm15.g0", 0x000000, 0x020000, CRC(51022b50) SHA1(47cb0d9b4a6012f87e12f07e2adcdd841d1e336a) ) // was also in the set as i99phm17.g0 (& .g1!)
	ROM_LOAD16_BYTE( "i99phm15.g1", 0x000001, 0x020000, CRC(3745fffb) SHA1(f2e9839fa54a0d9a98158ea6352d4c10cf9fa78b) )
	MISSING_SOUND
ROM_END


#define EP_HOMER_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "homer.sn0", 0x000000, 0x080000, CRC(f276596d) SHA1(675679271da7dc71f0e93a8ce519c1a6214a467b) ) \
	ROM_LOAD( "homer.sn1", 0x080000, 0x080000, CRC(1820f976) SHA1(32c1af52906dd40494a002a66eb314bec74b3361) )
ROM_START( ep_homer )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000070.g0", 0x000000, 0x020000, CRC(cd838170) SHA1(2dcb8729dafa69c4afe335cc48822341bd58fc4f) )
	ROM_LOAD16_BYTE( "sa000070.g1", 0x000001, 0x020000, CRC(36dad5a7) SHA1(d7ebe242d2364270ba2ff203fb068d159c7f6f97) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homera )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02180.g0", 0x000000, 0x020000, CRC(5e43d5e7) SHA1(6141654f549e21ae5fc08cc8106f0f0a60a962a6) )
	ROM_LOAD16_BYTE( "msa02180.g1", 0x000001, 0x020000, CRC(78f84d26) SHA1(7f82bbf32d2f892e813296ab2c4be1e21c4b560b) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02181.g0", 0x000000, 0x020000, CRC(f5a8a68f) SHA1(4adb5cd05690a76e520d3f38b652f2565cc35db0) )
	ROM_LOAD16_BYTE( "msa02181.g1", 0x000001, 0x020000, CRC(eaceca55) SHA1(77fcad1150f2607f4b39f51757fff732651be508) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02182.g0", 0x000000, 0x020000, CRC(ec0b8c7f) SHA1(4967fb5b9ba315e6900cc06b31ad068dba4ba501) )
	ROM_LOAD16_BYTE( "msa02182.g1", 0x000001, 0x020000, CRC(cda805b0) SHA1(b5d027d597bf8d97df98f9d72c5d377e5eee9af5) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02183.g0", 0x000000, 0x020000, CRC(82d9e9cf) SHA1(1c32d3dd34f522847541986bacf5ef86f005dec1) )
	ROM_LOAD16_BYTE( "msa02183.g1", 0x000001, 0x020000, CRC(10941569) SHA1(1819bd08e6d2049597e01a76c3b5444f5cba2e2f) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homere )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03310.g0", 0x000000, 0x020000, CRC(6d0ef840) SHA1(9ae8c0d0cd834af0cb9b2e3d0e116c63a4b1c6e4) )
	ROM_LOAD16_BYTE( "msa03310.g1", 0x000001, 0x020000, CRC(477707ff) SHA1(f4a78ddc2811e837969dfdc8b58d530bcbc54d13) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03311.g0", 0x000000, 0x020000, CRC(d6b0294d) SHA1(7081262e9ec98c0fd3fef1152988fefce2855849) )
	ROM_LOAD16_BYTE( "msa03311.g1", 0x000001, 0x020000, CRC(24d5dc7b) SHA1(060bc49b55dc97ae25677c8299d6f5381ca551d6) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03312.g0", 0x000000, 0x020000, CRC(ef90bd0e) SHA1(6d3242348139a31bfdf3a686ccc7f04332fc829e) )
	ROM_LOAD16_BYTE( "msa03312.g1", 0x000001, 0x020000, CRC(18780d61) SHA1(3f3a643cc2cac1f01660f8c5d3ddeaa70d2a4b3a) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homerh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03313.g0", 0x000000, 0x020000, CRC(a1e99c74) SHA1(569dd73821807026ad257d29aa338271a1da8477) )
	ROM_LOAD16_BYTE( "msa03313.g1", 0x000001, 0x020000, CRC(7b70fa94) SHA1(c223d396e63310c5e9a167577996a81a12ddbc8c) )
	EP_HOMER_SOUND
ROM_END

ROM_START( ep_homeri )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "homer.g0", 0x000000, 0x020000, CRC(f3414194) SHA1(446daaa729377dce2cbb70b31c7c651f245d5455) )
	ROM_LOAD16_BYTE( "homer.g1", 0x000001, 0x020000, CRC(9a1c2fc2) SHA1(5166eb40797ceaa3a58f35b1a48f1b50c7684733) )
	EP_HOMER_SOUND
ROM_END




ROM_START( ep_htdgs )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02nht04.g0", 0x000000, 0x020000, CRC(80ef03f7) SHA1(00be10668f4d6e1ddad3997b3b45bfe814c59289) )
	ROM_LOAD16_BYTE( "i02nht04.g1", 0x000001, 0x020000, CRC(633e8071) SHA1(5c259e6241007cf0e7e7ee70b771834138e03c8f) )
	ROM_LOAD16_BYTE( "i02nht04.g2", 0x040000, 0x020000, CRC(7dd6ab73) SHA1(5263834b9f566979f68d1efb2eb8c8b53e62da19) )
	ROM_LOAD16_BYTE( "i02nht04.g3", 0x040001, 0x020000, CRC(1e7e309e) SHA1(1e34fc88e88f3a65d70d0cdb6cfdda7e7cd0f5cf) )
	MISSING_SOUND
ROM_END

ROM_START( ep_htdgsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02pht05.g0", 0x000000, 0x020000, CRC(80ef03f7) SHA1(00be10668f4d6e1ddad3997b3b45bfe814c59289) )
	ROM_LOAD16_BYTE( "i02pht05.g1", 0x000001, 0x020000, CRC(88a0d449) SHA1(9bbceaaeb42802cfe661877327c94004663c7f2b) )
	ROM_LOAD16_BYTE( "i02pht05.g2", 0x040000, 0x020000, CRC(7dd6ab73) SHA1(5263834b9f566979f68d1efb2eb8c8b53e62da19) )
	ROM_LOAD16_BYTE( "i02pht05.g3", 0x040001, 0x020000, CRC(1e7e309e) SHA1(1e34fc88e88f3a65d70d0cdb6cfdda7e7cd0f5cf) )
	MISSING_SOUND
ROM_END




ROM_START( ep_hubbl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghb01_0.bin", 0x000000, 0x020000, CRC(8dbe4c7a) SHA1(f84089a6d8fc048e0721eb1375050e8ff553f852) )
	ROM_LOAD16_BYTE( "ghb01_1.bin", 0x000001, 0x020000, CRC(ad78b0d2) SHA1(fe5f38d8cbb892f5e5149d340dee24a8b2ccec45) )
	ROM_LOAD16_BYTE( "ghb01_2.bin", 0x040000, 0x020000, CRC(c31cdd3d) SHA1(07793b5e90364ca2d655ecb7562c8b6948d0e83c) )
	ROM_LOAD16_BYTE( "ghb01_3.bin", 0x040001, 0x020000, CRC(052fbc2f) SHA1(e198639752ae790a8c9bd1b67b7805479d88bf4c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hubbla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghb02_0.bin", 0x000000, 0x020000, CRC(8dbe4c7a) SHA1(f84089a6d8fc048e0721eb1375050e8ff553f852) )
	ROM_LOAD16_BYTE( "ghb02_1.bin", 0x000001, 0x020000, CRC(46e6e4ea) SHA1(ddffa3aae9065c5af49f10d82d40187ae22bcefa) )
	ROM_LOAD16_BYTE( "ghb02_2.bin", 0x040000, 0x020000, CRC(c31cdd3d) SHA1(07793b5e90364ca2d655ecb7562c8b6948d0e83c) )
	ROM_LOAD16_BYTE( "ghb02_3.bin", 0x040001, 0x020000, CRC(052fbc2f) SHA1(e198639752ae790a8c9bd1b67b7805479d88bf4c) )
	MISSING_SOUND
ROM_END

#define EP_HURIC_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) /* one of these pairs is probably a different game */ \
	ROM_LOAD( "huricane.sn0", 0x000000, 0x080000, CRC(b34b4667) SHA1(8b0b7b3668506de8433aa662556118a8b0e203d6) ) \
	ROM_LOAD( "huricane.sn1", 0x080000, 0x080000, CRC(324e255d) SHA1(6a30a512f19774b99bdbc14026a3b1cbfb65fe6d) ) \
	ROM_LOAD( "hurrican.s0", 0x000000, 0x080000, CRC(a50f8fa5) SHA1(bde27c5b536cdc97384adb3c2eb842d1c31eddde) ) \
	ROM_LOAD( "hurrican.s1", 0x080000, 0x080000, CRC(22412ebf) SHA1(36fa6da835f2a1f005d08b8e1c6bde9b0fb3ef6b) )

ROM_START( ep_huric )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhu22.g0", 0x000000, 0x020000, CRC(f2d002f8) SHA1(f00b4f8fc0b69d81a2d5a1e02e1e428f1a446e82) )
	ROM_LOAD16_BYTE( "i00nhu22.g1", 0x000001, 0x020000, CRC(570bca8c) SHA1(a9068817df689ef98980a2d7ad50d31983beff7f) )
	EP_HURIC_SOUND
ROM_END

ROM_START( ep_hurica )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhu24.g0", 0x000000, 0x020000, CRC(f2d002f8) SHA1(f00b4f8fc0b69d81a2d5a1e02e1e428f1a446e82) )
	ROM_LOAD16_BYTE( "i00nhu24.g1", 0x000001, 0x020000, CRC(5b4664bd) SHA1(881b998949c21e033b3c0b2127b3552abe388e7e) )
	EP_HURIC_SOUND
ROM_END

ROM_START( ep_huricb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nhu26.g0", 0x000000, 0x020000, CRC(301fc753) SHA1(e829657a22270ee5202fbbe1b7e435fe7829f760) )
	ROM_LOAD16_BYTE( "i00nhu26.g1", 0x000001, 0x020000, CRC(11273c81) SHA1(e6ecf638c200a9e52416b7d4e4bd4c035418e4a4) )
	ROM_LOAD16_BYTE( "i00nhu26.g2", 0x040000, 0x020000, CRC(fde78ab3) SHA1(9eb3a38bafaf2a10657421da00629c89723db93a) )
	ROM_LOAD16_BYTE( "i00nhu26.g3", 0x040001, 0x020000, CRC(18e3ec99) SHA1(0f79fc837afe1953e0e49c8a191baf44b5ac11be) )
	EP_HURIC_SOUND
ROM_END

ROM_START( ep_huricc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phu23.g0", 0x000000, 0x020000, CRC(f2d002f8) SHA1(f00b4f8fc0b69d81a2d5a1e02e1e428f1a446e82) )
	ROM_LOAD16_BYTE( "i00phu23.g1", 0x000001, 0x020000, CRC(bc959eb4) SHA1(8324141a8cf6d9f18e732fd7e5666a3783c39230) )
	EP_HURIC_SOUND
ROM_END

ROM_START( ep_huricd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phu25.g0", 0x000000, 0x020000, CRC(f2d002f8) SHA1(f00b4f8fc0b69d81a2d5a1e02e1e428f1a446e82) )
	ROM_LOAD16_BYTE( "i00phu25.g1", 0x000001, 0x020000, CRC(b0d83085) SHA1(9c681228da7ae89df3a7de1a759aaf27eb001dd1) )
	EP_HURIC_SOUND
ROM_END

ROM_START( ep_hurice )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00phu27.g0", 0x000000, 0x020000, CRC(301fc753) SHA1(e829657a22270ee5202fbbe1b7e435fe7829f760) )
	ROM_LOAD16_BYTE( "i00phu27.g1", 0x000001, 0x020000, CRC(fab968b9) SHA1(e281309b414fdeaed2cc13eeef75f0a1ef733854) )
	ROM_LOAD16_BYTE( "i00phu27.g2", 0x040000, 0x020000, CRC(fde78ab3) SHA1(9eb3a38bafaf2a10657421da00629c89723db93a) )
	ROM_LOAD16_BYTE( "i00phu27.g3", 0x040001, 0x020000, CRC(18e3ec99) SHA1(0f79fc837afe1953e0e49c8a191baf44b5ac11be) )
	EP_HURIC_SOUND
ROM_END



ROM_START( ep_hur )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hurr051n.p0", 0x000000, 0x020000, CRC(b7b11af5) SHA1(0cb488717f3bb33eccb52e7c0bddd9499ffd52bf) )
	ROM_LOAD16_BYTE( "hurr051n.p1", 0x000001, 0x020000, CRC(f7a27cc0) SHA1(7ec133c114c371661d03c0b41c23deb845695ef1) )
	ROM_LOAD16_BYTE( "hurr051n.p2", 0x040000, 0x020000, CRC(79e9f184) SHA1(adf7343dc84cdf9d794d76f136ba9520e1bb9520) )
	ROM_LOAD16_BYTE( "hurr051n.p3", 0x040001, 0x020000, CRC(c90ef4bf) SHA1(ef60ca4df64fe5f517add57fac55a4bd5807a643) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hura )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hurr051p.p0", 0x000000, 0x020000, CRC(b7b11af5) SHA1(0cb488717f3bb33eccb52e7c0bddd9499ffd52bf) )
	ROM_LOAD16_BYTE( "hurr051p.p1", 0x000001, 0x020000, CRC(1c3c28f8) SHA1(744a57d0399efa6ab315ccea37513731bea2c9a2) )
	ROM_LOAD16_BYTE( "hurr051p.p2", 0x040000, 0x020000, CRC(79e9f184) SHA1(adf7343dc84cdf9d794d76f136ba9520e1bb9520) )
	ROM_LOAD16_BYTE( "hurr051p.p3", 0x040001, 0x020000, CRC(c90ef4bf) SHA1(ef60ca4df64fe5f517add57fac55a4bd5807a643) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hurb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hurr3-1n.p0", 0x000000, 0x020000, CRC(f520a788) SHA1(2a258cc124b32587f62464e723ed2d7534f32ca6) )
	ROM_LOAD16_BYTE( "hurr3-1n.p1", 0x000001, 0x020000, CRC(98e6d012) SHA1(7a4a798abd9f331e4be876891b185ce04e76fa24) )
	MISSING_SOUND
ROM_END




ROM_START( ep_hyst )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nhy13.g0", 0x000000, 0x020000, CRC(fb370893) SHA1(aea6c410f7426e2a3bdd54c784d0f48e22e212be) )
	ROM_LOAD16_BYTE( "i01nhy13.g1", 0x000001, 0x020000, CRC(73434105) SHA1(1b908589b12d6f8feeb8f1415e38589ae11b103f) )
	ROM_LOAD16_BYTE( "i01nhy13.g2", 0x040000, 0x020000, CRC(70f52648) SHA1(8233c2185034be43a0818f137fa20ddf09543b18) )
	ROM_LOAD16_BYTE( "i01nhy13.g3", 0x040001, 0x020000, CRC(6c469fa8) SHA1(fe3ae2a2e72fcf54982b30205526e6cf5a11d113) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hysta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01phy14.g0", 0x000000, 0x020000, CRC(fb370893) SHA1(aea6c410f7426e2a3bdd54c784d0f48e22e212be) )
	ROM_LOAD16_BYTE( "i01phy14.g1", 0x000001, 0x020000, CRC(98dd153d) SHA1(b303cb7a807eb928562d04a48b35a06b79d7a9b1) )
	ROM_LOAD16_BYTE( "i01phy14.g2", 0x040000, 0x020000, CRC(70f52648) SHA1(8233c2185034be43a0818f137fa20ddf09543b18) )
	ROM_LOAD16_BYTE( "i01phy14.g3", 0x040001, 0x020000, CRC(6c469fa8) SHA1(fe3ae2a2e72fcf54982b30205526e6cf5a11d113) )
	MISSING_SOUND
ROM_END




ROM_START( ep_icebg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02234.g0", 0x000000, 0x020000, CRC(7924e7f0) SHA1(fa7e35176c6cbf8cf8b7eab5ec86cc6a32ce4289) )
	ROM_LOAD16_BYTE( "msa02234.g1", 0x000001, 0x020000, CRC(6b94f722) SHA1(0af64c7e5285de158b153581baa3fc1831f89740) )
	ROM_LOAD16_BYTE( "msa02234.g2", 0x040000, 0x020000, CRC(a01880b5) SHA1(50e5e44088b171a98424cea86e8482dc5a5401ee) )
	ROM_LOAD16_BYTE( "msa02234.g3", 0x040001, 0x020000, CRC(a9369209) SHA1(e6c691266c5ca2db5193be0ef92720dae479eb8b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02235.g0", 0x000000, 0x020000, CRC(5358ad0e) SHA1(551a37c1bc45cd362237324986184683757ff725) )
	ROM_LOAD16_BYTE( "msa02235.g1", 0x000001, 0x020000, CRC(31b5aebb) SHA1(09a6b0650e3f144f4c56c340ddaa8486ed14450c) )
	ROM_LOAD16_BYTE( "msa02235.g2", 0x040000, 0x020000, CRC(a01880b5) SHA1(50e5e44088b171a98424cea86e8482dc5a5401ee) )
	ROM_LOAD16_BYTE( "msa02235.g3", 0x040001, 0x020000, CRC(a9369209) SHA1(e6c691266c5ca2db5193be0ef92720dae479eb8b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02240.g0", 0x000000, 0x020000, CRC(c46daa51) SHA1(fe2104761d60f991b133946d11ea8c197127e356) )
	ROM_LOAD16_BYTE( "msa02240.g1", 0x000001, 0x020000, CRC(4539d934) SHA1(43207468fc0ddb506266b6b610e067b847f806da) )
	ROM_LOAD16_BYTE( "msa02240.g2", 0x040000, 0x020000, CRC(18329ff8) SHA1(fd6e3f00561a812fd9fe21ccabd21a88dd8388a3) )
	ROM_LOAD16_BYTE( "msa02240.g3", 0x040001, 0x020000, CRC(c9a11d2e) SHA1(1129f3ad2e62f13eb5250a196d2a8ec574dad588) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02241.g0", 0x000000, 0x020000, CRC(b58e1659) SHA1(2a62e167b43411f687a827141493288a948bc8fd) )
	ROM_LOAD16_BYTE( "msa02241.g1", 0x000001, 0x020000, CRC(69740e8e) SHA1(921ae1add38e9ac6c9632c7fd71486480b6c92bf) )
	ROM_LOAD16_BYTE( "msa02241.g2", 0x040000, 0x020000, CRC(18329ff8) SHA1(fd6e3f00561a812fd9fe21ccabd21a88dd8388a3) )
	ROM_LOAD16_BYTE( "msa02241.g3", 0x040001, 0x020000, CRC(c9a11d2e) SHA1(1129f3ad2e62f13eb5250a196d2a8ec574dad588) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02256.g0", 0x000000, 0x020000, CRC(010a41c8) SHA1(750cad7cec4cf40a7eec6b1b30a53fd911690139) )
	ROM_LOAD16_BYTE( "msa02256.g1", 0x000001, 0x020000, CRC(cbc7d405) SHA1(f6cfc2bbcae5ccde77f5b16f15e1af942eb438a3) )
	ROM_LOAD16_BYTE( "msa02256.g2", 0x040000, 0x020000, CRC(2110809b) SHA1(9a23901c578ca237067ad912947df47d7112709f) )
	ROM_LOAD16_BYTE( "msa02256.g3", 0x040001, 0x020000, CRC(450b8bc6) SHA1(1c099b692fde54584183e9c306d21194f51823ad) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebge )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02257.g0", 0x000000, 0x020000, CRC(db495d9b) SHA1(295825d08b77ab8d87d936b522b2f36fbcc5749f) )
	ROM_LOAD16_BYTE( "msa02257.g1", 0x000001, 0x020000, CRC(9bfad602) SHA1(04971c5d57da21f1acada9903050da00a258923a) )
	ROM_LOAD16_BYTE( "msa02257.g2", 0x040000, 0x020000, CRC(2110809b) SHA1(9a23901c578ca237067ad912947df47d7112709f) )
	ROM_LOAD16_BYTE( "msa02257.g3", 0x040001, 0x020000, CRC(450b8bc6) SHA1(1c099b692fde54584183e9c306d21194f51823ad) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03304.g0", 0x000000, 0x020000, CRC(d38cbb80) SHA1(595cc4ea91d4552495adb51586ada82ba1c2be03) )
	ROM_LOAD16_BYTE( "msa03304.g1", 0x000001, 0x020000, CRC(0ab3bd1f) SHA1(4dbc01aee01c1a6af7a5deb804b45de5b65e334d) )
	ROM_LOAD16_BYTE( "msa03304.g2", 0x040000, 0x020000, CRC(433b7c7e) SHA1(2c2825892bf6c4545fbca5a4dc776fd3147373c2) )
	ROM_LOAD16_BYTE( "msa03304.g3", 0x040001, 0x020000, CRC(d59e6ab0) SHA1(6a7ea535b4ee4ec4de792f8992f6fd8f885a5055) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03305.g0", 0x000000, 0x020000, CRC(7867c8e8) SHA1(e76583b1367f6f85ebd04a5531240a745942b3a1) )
	ROM_LOAD16_BYTE( "msa03305.g1", 0x000001, 0x020000, CRC(e7e67097) SHA1(def18f6592e666754785e4770dd84c4ea8e42ecb) )
	ROM_LOAD16_BYTE( "msa03305.g2", 0x040000, 0x020000, CRC(433b7c7e) SHA1(2c2825892bf6c4545fbca5a4dc776fd3147373c2) )
	ROM_LOAD16_BYTE( "msa03305.g3", 0x040001, 0x020000, CRC(d59e6ab0) SHA1(6a7ea535b4ee4ec4de792f8992f6fd8f885a5055) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03323.g0", 0x000000, 0x020000, CRC(307777d4) SHA1(99af73b8ede92ad68af2e9fff7c1a8f8507adb50) )
	ROM_LOAD16_BYTE( "msa03323.g1", 0x000001, 0x020000, CRC(ece3a31b) SHA1(af566b42d7c9a49dd400caed391c06eda959ea4b) )
	ROM_LOAD16_BYTE( "msa03323.g2", 0x040000, 0x020000, CRC(8a4111ca) SHA1(f107d6491ee1d7bcf19c513d97971d82c16250b2) )
	ROM_LOAD16_BYTE( "msa03323.g3", 0x040001, 0x020000, CRC(ea4b3432) SHA1(a4c832adcfd7835493683387c3de7546a38bc036) )
	MISSING_SOUND
ROM_END

ROM_START( ep_icebgi )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03324.g0", 0x000000, 0x020000, CRC(d505234e) SHA1(edf1047ce38f68208f7eeb31b76a5989bc9e8325) )
	ROM_LOAD16_BYTE( "msa03324.g1", 0x000001, 0x020000, CRC(4c06195d) SHA1(c7d47fe63ab11a4a426c5d9d2031e8d8edaef9c0) )
	ROM_LOAD16_BYTE( "msa03324.g2", 0x040000, 0x020000, CRC(8a4111ca) SHA1(f107d6491ee1d7bcf19c513d97971d82c16250b2) )
	ROM_LOAD16_BYTE( "msa03324.g3", 0x040001, 0x020000, CRC(ea4b3432) SHA1(a4c832adcfd7835493683387c3de7546a38bc036) )
	MISSING_SOUND
ROM_END



ROM_START( ep_inca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "inca12.p0", 0x000000, 0x020000, CRC(122b3b0f) SHA1(b35e7afb7ea49ba4e5863c6e1f989cc94ef9f2c4) )
	ROM_LOAD16_BYTE( "inca12.p1", 0x000001, 0x020000, CRC(93d102b5) SHA1(08fce29b0f8b457f7be24c72511c61785f5a84bc) )
	ROM_LOAD16_BYTE( "inca12.p2", 0x040000, 0x020000, CRC(d6efe8ea) SHA1(2e39f8c9d00f2bea639dca8e2f2f2a7901cefce9) )
	ROM_LOAD16_BYTE( "inca12.p3", 0x040001, 0x020000, CRC(5cd5c64b) SHA1(68f259d3a323b2facbde196a56618f01535e7e0c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_incaa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "inca12p.p0", 0x000000, 0x020000, CRC(122b3b0f) SHA1(b35e7afb7ea49ba4e5863c6e1f989cc94ef9f2c4) )
	ROM_LOAD16_BYTE( "inca12p.p1", 0x000001, 0x020000, CRC(8c42f0ce) SHA1(92f4102b861b030279a1eb20a03e12533122c578) )
	ROM_LOAD16_BYTE( "inca12p.p2", 0x040000, 0x020000, CRC(d6efe8ea) SHA1(2e39f8c9d00f2bea639dca8e2f2f2a7901cefce9) )
	ROM_LOAD16_BYTE( "inca12p.p3", 0x040001, 0x020000, CRC(5cd5c64b) SHA1(68f259d3a323b2facbde196a56618f01535e7e0c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_incab )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "inca_din.g0", 0x000000, 0x020000, CRC(27dd25f6) SHA1(708491d2b816d08fec295d760c76d9071258f0a7) )
	ROM_LOAD16_BYTE( "inca_din.g1", 0x000001, 0x020000, CRC(39d2981c) SHA1(3e70b4f79c38b52f5fcf8e66aabcab3ba88e6fd3) )
	ROM_LOAD16_BYTE( "inca_din.g2", 0x040000, 0x020000, CRC(79546ee7) SHA1(95511b5df2f36fe5b8b8372a728a2bf68a42bb47) )
	ROM_LOAD16_BYTE( "inca_din.g3", 0x040001, 0x020000, CRC(31e42bb6) SHA1(cfb75c221f9c6c3eb05e4f0b2a4951195b0b8703) )
	MISSING_SOUND
ROM_END



ROM_START( ep_itjb2 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02156.g0", 0x000000, 0x020000, CRC(14389e19) SHA1(a30da07339e7f739bfd7c49bcae26103c795fc08) )
	ROM_LOAD16_BYTE( "msa02156.g1", 0x000001, 0x020000, CRC(c10bc359) SHA1(88de2344e01a5dd4c6b520348d99eb943588da05) )
	MISSING_SOUND
ROM_END

ROM_START( ep_itjb2a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02157.g0", 0x000000, 0x020000, CRC(ce7b824a) SHA1(df40918417fc25b4d8d72cf358af4ef12e7a5ce3) )
	ROM_LOAD16_BYTE( "msa02157.g1", 0x000001, 0x020000, CRC(d24bf013) SHA1(1bef11e7568222542a2e4a7227afd3d849d430da) )
	MISSING_SOUND
ROM_END

ROM_START( ep_itjb2b )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02158.g0", 0x000000, 0x020000, CRC(d2297a3d) SHA1(4cb5bcd63e28880b1f0220e5da1616d3a16206f7) )
	ROM_LOAD16_BYTE( "msa02158.g1", 0x000001, 0x020000, CRC(aff32681) SHA1(deeadeac4438567eab35cf4d3f8952f8a0ee7197) )
	MISSING_SOUND
ROM_END

ROM_START( ep_itjb2c )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02159.g0", 0x000000, 0x020000, CRC(6dd3831c) SHA1(7c124c008eb6a566fa62ac914ce71f46a361f954) )
	ROM_LOAD16_BYTE( "msa02159.g1", 0x000001, 0x020000, CRC(f085d809) SHA1(bdb78aa66e7c3b175aadd2f1a3896b461e9f2734) )
	MISSING_SOUND
ROM_END




ROM_START( ep_ijcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02046.g0", 0x000000, 0x020000, CRC(d965cff5) SHA1(ac9a6dd41ce83f4ba205419b2a25c60033258725) )
	ROM_LOAD16_BYTE( "msc02046.g1", 0x000001, 0x020000, CRC(4adec9e4) SHA1(67536015b1cf152e1bdeb0f51043a091c5a658e4) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ijcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02047.g0", 0x000000, 0x020000, CRC(f77c368e) SHA1(baa1eda6ff55fa4acfb2cba1822025f747603328) )
	ROM_LOAD16_BYTE( "msc02047.g1", 0x000001, 0x020000, CRC(0ea0e029) SHA1(610ff37e0e501bdf66c7d65dd10622acd7fd7126) )
	MISSING_SOUND
ROM_END


#define EP_IJOB_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "it_job.s1", 0x000000, 0x080000, CRC(77ea2225) SHA1(fb02bccf4555797b60871070821afe2162be26c2) ) \
	ROM_LOAD( "it_job.s2", 0x080000, 0x080000, CRC(b6f86703) SHA1(f09f9d88132919ca24fc1ed7094b5b22c1136eb8) )

ROM_START( ep_ijob )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "it_job.g0", 0x000000, 0x020000, CRC(92a63d81) SHA1(b21ebc02becf7db98887599c5632fade3b0dcbc5) )
	ROM_LOAD16_BYTE( "it_job.g1", 0x000001, 0x020000, CRC(8b814e44) SHA1(276a8e99e70279e32c285be3e12032ae4c36802a) )
	EP_IJOB_SOUND
ROM_END


ROM_START( ep_ijoba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i_job.g0", 0x000000, 0x020000, CRC(d3133b25) SHA1(b30268ab21a9e0c7193ae697710ff37d27af0a5b) )
	ROM_LOAD16_BYTE( "i_job.g1", 0x000001, 0x020000, CRC(0765b48b) SHA1(32eaa458e7f693c7bbbed94342f5f78906137c49) )
	EP_IJOB_SOUND
ROM_END



ROM_START( ep_jakbn ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jabs03.p0", 0x000000, 0x100000, CRC(620eb339) SHA1(51af7eb98200c18c52da601003f78c7cb99afc8c) )
	ROM_LOAD16_BYTE( "jabs03.p1", 0x000001, 0x100000, CRC(d984ef8b) SHA1(ae97b4f69d6f7be5562cc17a050e606eafb714d8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jakbna ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jabs03p.p0", 0x000000, 0x100000, CRC(620eb339) SHA1(51af7eb98200c18c52da601003f78c7cb99afc8c) )
	ROM_LOAD16_BYTE( "jabs03p.p1", 0x000001, 0x100000, CRC(8c6f8230) SHA1(a783392bdebe1103d2290b9dd49b589fb3967199) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jakbnb ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jabs05.p0", 0x000000, 0x100000, CRC(e1d730b9) SHA1(e23393c9c43d9079b9d7869c22a2599e777c2fc7) )
	ROM_LOAD16_BYTE( "jabs05.p1", 0x000001, 0x100000, CRC(4bd743d3) SHA1(f873028b83cd58fbe25ff427865d43e0517c5bf1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jakbnc ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jabs05p.p0", 0x000000, 0x100000, CRC(e1d730b9) SHA1(e23393c9c43d9079b9d7869c22a2599e777c2fc7) )
	ROM_LOAD16_BYTE( "jabs05p.p1", 0x000001, 0x100000, CRC(1e3c2e68) SHA1(f28f3a17a4bb8240fc2951027773569cfab9018b) )
	MISSING_SOUND
ROM_END




ROM_START( ep_jsttt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02174.g0", 0x000000, 0x020000, CRC(efc59fc8) SHA1(786a3090bcff15390172030d3680468ebe219aa0) )
	ROM_LOAD16_BYTE( "msa02174.g1", 0x000001, 0x020000, CRC(4366ce04) SHA1(08c0764a72b25104326c6f74d211863b2e3d5b73) )
	ROM_LOAD16_BYTE( "msa02174.g2", 0x040000, 0x020000, CRC(38492a88) SHA1(47da0f8dea19788d548d12e9e5314956231f9ff6) )
	ROM_LOAD16_BYTE( "msa02174.g3", 0x040001, 0x020000, CRC(243f1580) SHA1(a90711e4b4ea0a145b8dd7a95cc4fe20ad8a7c32) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jsttta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02175.g0", 0x000000, 0x020000, CRC(152dc751) SHA1(ec34f73bf6bb557920dc7329fb52ab894b23f05f) )
	ROM_LOAD16_BYTE( "msa02175.g1", 0x000001, 0x020000, CRC(25923bdb) SHA1(dc5f5d52fb0dff99f5ffbd19d8d41914737bdc17) )
	ROM_LOAD16_BYTE( "msa02175.g2", 0x040000, 0x020000, CRC(38492a88) SHA1(47da0f8dea19788d548d12e9e5314956231f9ff6) )
	ROM_LOAD16_BYTE( "msa02175.g3", 0x040001, 0x020000, CRC(243f1580) SHA1(a90711e4b4ea0a145b8dd7a95cc4fe20ad8a7c32) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jstttb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02218.g0", 0x000000, 0x020000, CRC(d3a0563f) SHA1(3020935da9378918da8fa7418bbcaf21ab58aa6d) )
	ROM_LOAD16_BYTE( "msa02218.g1", 0x000001, 0x020000, CRC(5a6eef17) SHA1(30dd0ac74168f58cbff51103aec37ae51b7a5d62) )
	ROM_LOAD16_BYTE( "msa02218.g2", 0x040000, 0x020000, CRC(5858eef0) SHA1(f9ddd309a77ba3b4bf9089ce547cce4e303d7d52) )
	ROM_LOAD16_BYTE( "msa02218.g3", 0x040001, 0x020000, CRC(201bb9d3) SHA1(e09b8cd4ebd9d1c08715780b6e778158f859e592) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jstttc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02219.g0", 0x000000, 0x020000, CRC(1b7c60e0) SHA1(f9f5ccd1f2bb80c1e24ea2c2c2a01cd7aa71c50e) )
	ROM_LOAD16_BYTE( "msa02219.g1", 0x000001, 0x020000, CRC(186d968b) SHA1(3628468c1332be149d7cf3083980502bca974a4a) )
	ROM_LOAD16_BYTE( "msa02219.g2", 0x040000, 0x020000, CRC(5858eef0) SHA1(f9ddd309a77ba3b4bf9089ce547cce4e303d7d52) )
	ROM_LOAD16_BYTE( "msa02219.g3", 0x040001, 0x020000, CRC(201bb9d3) SHA1(e09b8cd4ebd9d1c08715780b6e778158f859e592) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jstttd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03314.g0", 0x000000, 0x020000, CRC(5b3b43f1) SHA1(929e560d382d5849617185aaf2cf10775cf1c0f8) )
	ROM_LOAD16_BYTE( "msa03314.g1", 0x000001, 0x020000, CRC(719f81c2) SHA1(41334a3ece293876f5e692b7c3ecefd6932e1ab0) )
	ROM_LOAD16_BYTE( "msa03314.g2", 0x040000, 0x020000, CRC(4b628810) SHA1(591eb3113e0b1ed54e72f65a8caa8a539638f0d9) )
	ROM_LOAD16_BYTE( "msa03314.g3", 0x040001, 0x020000, CRC(e730e0ae) SHA1(69e59daf761a211e036707583ff731725a008385) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jsttte )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03316.g0", 0x000000, 0x020000, CRC(8feff4df) SHA1(a2bd74f93ded1bdf4d2718089a2e7c9a584fa494) )
	ROM_LOAD16_BYTE( "msa03316.g1", 0x000001, 0x020000, CRC(13aab934) SHA1(249b4086bb4e0708768b02bd6fe57a3f53afcce4) )
	ROM_LOAD16_BYTE( "msa03316.g2", 0x040000, 0x020000, CRC(4b628810) SHA1(591eb3113e0b1ed54e72f65a8caa8a539638f0d9) )
	ROM_LOAD16_BYTE( "msa03316.g3", 0x040001, 0x020000, CRC(e730e0ae) SHA1(69e59daf761a211e036707583ff731725a008385) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jstttf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03319.g0", 0x000000, 0x020000, CRC(ad83db20) SHA1(8c3c0d5b76f1bd7c9e91e8d304c8d2df61c8f376) )
	ROM_LOAD16_BYTE( "msa03319.g1", 0x000001, 0x020000, CRC(df95d1cd) SHA1(1641bd38301fc05aaf85fab776e45288ff886fc8) )
	ROM_LOAD16_BYTE( "msa03319.g2", 0x040000, 0x020000, CRC(5e4c7b53) SHA1(b4f4a021b2f2b273910a009ab7be69102a4b7503) )
	ROM_LOAD16_BYTE( "msa03319.g3", 0x040001, 0x020000, CRC(24f7b712) SHA1(f8cf7c744b90dcae2845421be7c5d0a750b76afe) )
	MISSING_SOUND
ROM_END

ROM_START( ep_jstttg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03320.g0", 0x000000, 0x020000, CRC(19d3188e) SHA1(8f0119f39d5f763f7d70e14b54f5fa336f6ad06e) )
	ROM_LOAD16_BYTE( "msa03320.g1", 0x000001, 0x020000, CRC(261f2d8e) SHA1(c6268c96bc22bdc92d2cf3f21c18db6cc8cac6c8) )
	ROM_LOAD16_BYTE( "msa03320.g2", 0x040000, 0x020000, CRC(5e4c7b53) SHA1(b4f4a021b2f2b273910a009ab7be69102a4b7503) )
	ROM_LOAD16_BYTE( "msa03320.g3", 0x040001, 0x020000, CRC(24f7b712) SHA1(f8cf7c744b90dcae2845421be7c5d0a750b76afe) )
	MISSING_SOUND
ROM_END




ROM_START( ep_loadd ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "load12d.g0", 0x000000, 0x100000, CRC(2a77e08f) SHA1(df30e26509efa27f6d7246dfe8025eee8b788a19) )
	ROM_LOAD16_BYTE( "load12d.g1", 0x000001, 0x100000, CRC(cb8a9216) SHA1(149062a5f3891fb101f8d9601f09b69d5701f838) )
	MISSING_SOUND
ROM_END

ROM_START( ep_loadda ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "load12nd.g0", 0x000000, 0x100000, CRC(2a77e08f) SHA1(df30e26509efa27f6d7246dfe8025eee8b788a19) )
	ROM_LOAD16_BYTE( "load12nd.g1", 0x000001, 0x100000, CRC(9e61ffad) SHA1(1ed6829a9fa86691e43164fc3b9e1a00d9dd191e) )
	MISSING_SOUND
ROM_END




ROM_START( ep_lukld ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "lula02.p0", 0x000000, 0x100000, CRC(c7ddb6bc) SHA1(17cbe6fc4fc4fdcae8ba6a7c8f2d56e30aff5c31) )
	ROM_LOAD16_BYTE( "lula02.p1", 0x000001, 0x100000, CRC(18ce1ecc) SHA1(eb2a24f95a803168d3319694ffc51e7a147d77bb) )
	MISSING_SOUND
ROM_END

ROM_START( ep_luklda ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "lula02p.p0", 0x000000, 0x100000, CRC(c7ddb6bc) SHA1(17cbe6fc4fc4fdcae8ba6a7c8f2d56e30aff5c31) )
	ROM_LOAD16_BYTE( "lula02p.p1", 0x000001, 0x100000, CRC(4d257377) SHA1(c49ae386b47814941e5b78ab09a56d5b5a6bdfe9) )
	MISSING_SOUND
ROM_END





ROM_START( ep_manic )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01pmm19.g0", 0x000000, 0x020000, CRC(13314c26) SHA1(206a6536975319a8e516e89c82f8f85409f40f74) )
	ROM_LOAD16_BYTE( "i01pmm19.g1", 0x000001, 0x020000, CRC(950bfd4f) SHA1(58e79d30f4ad88563c50895529b1573327622626) )
	ROM_LOAD16_BYTE( "i01pmm19.g2", 0x040000, 0x020000, CRC(3664d78a) SHA1(ab4ec3b931174b359723f23a8280517c1a6b55ba) )
	ROM_LOAD16_BYTE( "i01pmm19.g3", 0x040001, 0x020000, CRC(e666f3ce) SHA1(8daefb0fb621b9e850ec370a98fd35e0bf5e2bf3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manica )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nmm14.g0", 0x000000, 0x020000, CRC(15bd72e0) SHA1(02129b8f2a5b2c10faaea5723c22e9a9c6a2a496) )
	ROM_LOAD16_BYTE( "i01nmm14.g1", 0x000001, 0x020000, CRC(126aad2c) SHA1(6fe581b6624df574a1301f6635c1e80191f06b2f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manicb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nmm16.g0", 0x000000, 0x020000, CRC(15bd72e0) SHA1(02129b8f2a5b2c10faaea5723c22e9a9c6a2a496) )
	ROM_LOAD16_BYTE( "i01nmm16.g1", 0x000001, 0x020000, CRC(1e27031d) SHA1(50ccda20a2569d9be9cb236342c4c90a631fd219) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manicc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01nmm18.g0", 0x000000, 0x020000, CRC(13314c26) SHA1(206a6536975319a8e516e89c82f8f85409f40f74) )
	ROM_LOAD16_BYTE( "i01nmm18.g1", 0x000001, 0x020000, CRC(7e95a977) SHA1(972091501643321e83d2c88cb8cec447520af363) )
	ROM_LOAD16_BYTE( "i01nmm18.g2", 0x040000, 0x020000, CRC(3664d78a) SHA1(ab4ec3b931174b359723f23a8280517c1a6b55ba) )
	ROM_LOAD16_BYTE( "i01nmm18.g3", 0x040001, 0x020000, CRC(e666f3ce) SHA1(8daefb0fb621b9e850ec370a98fd35e0bf5e2bf3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manicd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01pmm15.g0", 0x000000, 0x020000, CRC(15bd72e0) SHA1(02129b8f2a5b2c10faaea5723c22e9a9c6a2a496) )
	ROM_LOAD16_BYTE( "i01pmm15.g1", 0x000001, 0x020000, CRC(f9f4f914) SHA1(4bd23ce3c73ef7f8e5e1ece785d44ddbbfdd8e96) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manice )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01pmm17.g0", 0x000000, 0x020000, CRC(15bd72e0) SHA1(02129b8f2a5b2c10faaea5723c22e9a9c6a2a496) )
	ROM_LOAD16_BYTE( "i01pmm17.g1", 0x000001, 0x020000, CRC(f5b95725) SHA1(128140dfcead9f8208034ed31e5d2804dc85eacf) )
	MISSING_SOUND
ROM_END

ROM_START( ep_manicf ) // why are these larger?
	ROM_REGION( 0x100000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "m_miner.g0", 0x000000, 0x080000, CRC(3ed1051f) SHA1(702edbdc1e3bccb3786d23aba752078ca9e71541) )
	ROM_LOAD16_BYTE( "m_miner.g1", 0x000001, 0x080000, CRC(04e0d2f2) SHA1(c674190622d58409e6b18af5ed5934c39a580870) )
	MISSING_SOUND
ROM_END


#define EP_MKART_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "m_kart.sn0", 0x000000, 0x080000, CRC(cb9f8fe6) SHA1(8e0d10c55ebf225c395094a720b797185df94365) ) \
	ROM_LOAD( "m_kart.sn1", 0x080000, 0x080000, CRC(f725261c) SHA1(1ad0cf2af55e7bf6c5711ffd03b12191f8489942) )

ROM_START( ep_mkart )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990031.g0", 0x000000, 0x020000, CRC(609a190b) SHA1(36cfbe917721e27b737f8a05754e262026eb9804) )
	ROM_LOAD16_BYTE( "sa990031.g1", 0x000001, 0x020000, CRC(4e1b5cf1) SHA1(c29071890055164624822b6bd4b2bc720f4d9041) )
	EP_MKART_SOUND
ROM_END

ROM_START( ep_mkarta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990032.g0", 0x000000, 0x020000, CRC(47d68af2) SHA1(cdfe544fff7cb6aa786bb93a9323e3b5f6514f4e) )
	ROM_LOAD16_BYTE( "sa990032.g1", 0x000001, 0x020000, CRC(2bc3b79d) SHA1(3ed8af47d4bfb6bf5da089488fb399ca27e0dccd) )
	EP_MKART_SOUND
ROM_END

ROM_START( ep_mkartb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990191.g0", 0x000000, 0x020000, CRC(591257bd) SHA1(bb3c6587b67968180f9a8049e93f4a842261217c) )
	ROM_LOAD16_BYTE( "sa990191.g1", 0x000001, 0x020000, CRC(ab8c0d89) SHA1(adf7589b92003471731749ab672cd1af72e77246) )
	EP_MKART_SOUND
ROM_END

ROM_START( ep_mkartc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990192.g0", 0x000000, 0x020000, CRC(b99bd232) SHA1(ce072315a6e0ea7bb5145f73471248ca4d219629) )
	ROM_LOAD16_BYTE( "sa990192.g1", 0x000001, 0x020000, CRC(c05058c2) SHA1(817cbd31055b5fe00cd731d5365692b577449c65) )
	EP_MKART_SOUND
ROM_END

ROM_START( ep_mkartd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mkart_el.g0", 0x000000, 0x020000, CRC(76a8f7de) SHA1(a455c022601eda3eb0abd889818d42d97977da83) )
	ROM_LOAD16_BYTE( "mkart_el.g1", 0x000001, 0x020000, CRC(fa78e0a4) SHA1(e3b7b39e9d077d04d3bd29c9963340a35ace5b68) )
	EP_MKART_SOUND
ROM_END

ROM_START( ep_mkarte )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mar_kart.g0", 0x000000, 0x020000, CRC(c0edd771) SHA1(d9a0dd4ba64877c41bf2e324106979d9b7720508) )
	ROM_LOAD16_BYTE( "mar_kart.g1", 0x000001, 0x020000, CRC(8017408c) SHA1(56a5374edf4d0b1200ced5867a83bf34d29f3ec5) )
	EP_MKART_SOUND
ROM_END





ROM_START( ep_matrx )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nmx02.g0", 0x000000, 0x020000, CRC(4b9eeaf3) SHA1(83a4a0808908ae1202744917cd101e0e9f81057e) )
	ROM_LOAD16_BYTE( "i00nmx02.g1", 0x000001, 0x020000, CRC(1bb35764) SHA1(3af0d6dd1ed04b6268bc83fc8a6670568d4517bc) )
	MISSING_SOUND
ROM_END




ROM_START( ep_midas )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc980025.g0", 0x000000, 0x020000, CRC(5bc3e21d) SHA1(8e2601996eeec6e1861e081f927deac1676c5ad5) )
	ROM_LOAD16_BYTE( "sc980025.g1", 0x000001, 0x020000, CRC(e8ef605a) SHA1(b5335a50459d64abf4fcbc2dc6a97d137aa1292b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_midasa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc980026.g0", 0x000000, 0x020000, CRC(7c8f71e4) SHA1(5162b75ca75aaeb33e3c3b944415bc75900b7e28) )
	ROM_LOAD16_BYTE( "sc980026.g1", 0x000001, 0x020000, CRC(2c51d544) SHA1(e2ea2a9b6b764a3e47c0bfb4fe2d7572b54ae938) )
	MISSING_SOUND
ROM_END




ROM_START( ep_monbs )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990098.g0", 0x000000, 0x020000, CRC(19182436) SHA1(4e9830541a805e1b07e90668e45fcba2c4043cf0) )
	ROM_LOAD16_BYTE( "sa990098.g1", 0x000001, 0x020000, CRC(e41c99eb) SHA1(46f0266db52d2af4a9a2c1e17f5a490d3c11380a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monbsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa990099.g0", 0x000000, 0x020000, CRC(b2f3575e) SHA1(c34cafdee4f1d89a278bcc73a6576c9000b5abf3) )
	ROM_LOAD16_BYTE( "sa990099.g1", 0x000001, 0x020000, CRC(d7d885c6) SHA1(27e30f5279795b5ce0a5a53f3c9251970aa9039e) )
	MISSING_SOUND
ROM_END


#define EP_MWOM_SOUND \
	ROM_REGION( 0x080000, "ymz", 0 ) \
	ROM_LOAD( "mortal.snd", 0x0000, 0x080000, CRC(34a24e83) SHA1(df464ffdaf3225beecd9facdb86a87ed501c261e) )

ROM_START( ep_mwom )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmw01_0.bin", 0x000000, 0x020000, CRC(716f2870) SHA1(205203591bb30e46946252f3a9cf21f5bed9515c) )
	ROM_LOAD16_BYTE( "gmw01_1.bin", 0x000001, 0x020000, CRC(01a823cd) SHA1(644bc17b5e78bcea5a3fe9460efd3581edf903c0) )
	ROM_LOAD16_BYTE( "gmw01_2.bin", 0x040000, 0x020000, CRC(5bff3688) SHA1(ffe895b9e186514e263d78bdea5b0fb0fb3979f7) )
	ROM_LOAD16_BYTE( "gmw01_3.bin", 0x040001, 0x020000, CRC(5af3b59b) SHA1(8e356b172fee4cba05057f7a8873ee71f7073daa) )
	EP_MWOM_SOUND
ROM_END

ROM_START( ep_mwoma )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmw02_0.bin", 0x000000, 0x020000, CRC(716f2870) SHA1(205203591bb30e46946252f3a9cf21f5bed9515c) )
	ROM_LOAD16_BYTE( "gmw02_1.bin", 0x000001, 0x020000, CRC(ea3677f5) SHA1(138c5b331de6fc0d0d266f21c48eed79e85b4422) )
	ROM_LOAD16_BYTE( "gmw02_2.bin", 0x040000, 0x020000, CRC(5bff3688) SHA1(ffe895b9e186514e263d78bdea5b0fb0fb3979f7) )
	ROM_LOAD16_BYTE( "gmw02_3.bin", 0x040001, 0x020000, CRC(5af3b59b) SHA1(8e356b172fee4cba05057f7a8873ee71f7073daa) )
	EP_MWOM_SOUND
ROM_END

ROM_START( ep_mwomb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmw03_0.bin", 0x000000, 0x020000, CRC(9040f24b) SHA1(cd0f4fc66ab2013f5a83141e5670146df0cd63ea) )
	ROM_LOAD16_BYTE( "gmw03_1.bin", 0x000001, 0x020000, CRC(189c0e1d) SHA1(f6b69c7263ffcf1bffcbadb17996f84927de560a) )
	ROM_LOAD16_BYTE( "gmw03_2.bin", 0x040000, 0x020000, CRC(ffdf1ef9) SHA1(d1a3f5231bbc3859b69e922e128e6033cd593db9) )
	ROM_LOAD16_BYTE( "gmw03_3.bin", 0x040001, 0x020000, CRC(9b371f55) SHA1(09ddfc076fc697f1751960b7d3f90fab6fefae1f) )
	EP_MWOM_SOUND
ROM_END

ROM_START( ep_mwomc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmw04_0.bin", 0x000000, 0x020000, CRC(9040f24b) SHA1(cd0f4fc66ab2013f5a83141e5670146df0cd63ea) )
	ROM_LOAD16_BYTE( "gmw04_1.bin", 0x000001, 0x020000, CRC(f3025a25) SHA1(1a1f409f9ddb7b076cc283f3285a745d9cbbb99a) )
	ROM_LOAD16_BYTE( "gmw04_2.bin", 0x040000, 0x020000, CRC(ffdf1ef9) SHA1(d1a3f5231bbc3859b69e922e128e6033cd593db9) )
	ROM_LOAD16_BYTE( "gmw04_3.bin", 0x040001, 0x020000, CRC(9b371f55) SHA1(09ddfc076fc697f1751960b7d3f90fab6fefae1f) )
	EP_MWOM_SOUND
ROM_END

ROM_START( ep_mwomd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mortal w.g0", 0x000000, 0x020000, CRC(81bfe12b) SHA1(40de7c9cf6f8a997a1edd81d321f2ef8694b1830) )
	ROM_LOAD16_BYTE( "mortal w.g1", 0x000001, 0x020000, CRC(38277aba) SHA1(223c89f0e897149852adf85b8eade89b360a01f1) )
	ROM_LOAD16_BYTE( "mortal w.g2", 0x040000, 0x020000, CRC(ec8d75db) SHA1(39234583136d7c6b8cebadefa5ba0e02ec0d4abe) )
	ROM_LOAD16_BYTE( "mortal w.g3", 0x040001, 0x020000, CRC(33a696a6) SHA1(2ab9f1e248b21ca262d8bf771cea0f177de03118) )
	EP_MWOM_SOUND
ROM_END






ROM_START( ep_mrmus )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02144.g0", 0x000000, 0x020000, CRC(ee6ed79b) SHA1(7959fd9c9ae315d6e2219a55bf641a952048c3ae) )
	ROM_LOAD16_BYTE( "msa02144.g1", 0x000001, 0x020000, CRC(bd59ce6e) SHA1(ee650411ff18a432379cdaaf39d37c4fc5c0fd5e) )
	ROM_LOAD16_BYTE( "msa02144.g2", 0x040000, 0x020000, CRC(0d10ea3b) SHA1(c11719ef2eeaaf3dbe629715850555f8928bb426) )
	ROM_LOAD16_BYTE( "msa02144.g3", 0x040001, 0x020000, CRC(3f3b76ed) SHA1(215912be57ce57e383ba3e209387b0f8e687f9ce) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mrmusa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02145.g0", 0x000000, 0x020000, CRC(f8499235) SHA1(2439b3aa1c1ad8e152356de83e8618bc71676ec1) )
	ROM_LOAD16_BYTE( "msa02145.g1", 0x000001, 0x020000, CRC(d9285a20) SHA1(904d8cc44ea984f11ba2b0bb0893d8a907a79bba) )
	ROM_LOAD16_BYTE( "msa02145.g2", 0x040000, 0x020000, CRC(0d10ea3b) SHA1(c11719ef2eeaaf3dbe629715850555f8928bb426) )
	ROM_LOAD16_BYTE( "msa02145.g3", 0x040001, 0x020000, CRC(3f3b76ed) SHA1(215912be57ce57e383ba3e209387b0f8e687f9ce) )
	MISSING_SOUND
ROM_END


ROM_START( ep_noter ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru01.p0", 0x000000, 0x100000, CRC(697a29a0) SHA1(cf032121371632b5642dc94568fff82e07e10c7d) )
	ROM_LOAD16_BYTE( "noru01.p1", 0x000001, 0x100000, CRC(a0b1cba8) SHA1(733f2591acc18a0d156f836aded9d428e3f53ff0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_notera ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru01p.p0", 0x000000, 0x100000, CRC(697a29a0) SHA1(cf032121371632b5642dc94568fff82e07e10c7d) )
	ROM_LOAD16_BYTE( "noru01p.p1", 0x000001, 0x100000, CRC(f55aa613) SHA1(00f44887bbaadead177f7066a774f1a055e23615) )
	MISSING_SOUND
ROM_END

ROM_START( ep_noterb ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru02.p0", 0x000000, 0x100000, CRC(cc567166) SHA1(7377b4c45b47357dfb2596efdc22860a1a1ed665) )
	ROM_LOAD16_BYTE( "noru02.p1", 0x000001, 0x100000, CRC(f5782e25) SHA1(0c6487f511d6c893fabc1c92687b21a29702dae1) )
	MISSING_SOUND
ROM_END

ROM_START( ep_noterc ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru02p.p0", 0x000000, 0x100000, CRC(cc567166) SHA1(7377b4c45b47357dfb2596efdc22860a1a1ed665) )
	ROM_LOAD16_BYTE( "noru02p.p1", 0x000001, 0x100000, CRC(a093439e) SHA1(aae510c5ec2dacc64939c0787b6721db71b2e048) )
	MISSING_SOUND
ROM_END

ROM_START( ep_noterd ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru10.p0", 0x000000, 0x100000, CRC(8d95f57d) SHA1(b990e38a0a8b11688fed06390b9cc8d06d160169) )
	ROM_LOAD16_BYTE( "noru10.p1", 0x000001, 0x100000, CRC(534e22c4) SHA1(7282adcbf7d41328224eb2a86034c30446e3eff6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_notere ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "noru10p.p0", 0x000000, 0x100000, CRC(8d95f57d) SHA1(b990e38a0a8b11688fed06390b9cc8d06d160169) )
	ROM_LOAD16_BYTE( "noru10p.p1", 0x000001, 0x100000, CRC(06a54f7f) SHA1(8545c53754f55fe1a5042b051c9f36cb9a6f7dde) )
	MISSING_SOUND
ROM_END


#define EP_NUNS_SOUND \
	ROM_REGION( 0x080000, "ymz", 0 ) \
	ROM_LOAD( "nuns s0", 0x0000, 0x080000, CRC(158dadca) SHA1(e65dd64517f5e9ccf93a993ee22ab1b96d0a2a22) )

ROM_START( ep_nuns )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02152.g0", 0x000000, 0x020000, CRC(621cab28) SHA1(19f037c5a819eace422dc29cd69b7bedb0aa6777) )
	ROM_LOAD16_BYTE( "msa02152.g1", 0x000001, 0x020000, CRC(df278346) SHA1(6f626549988bc970ae2de9897a49c721b7091c2d) )
	ROM_LOAD16_BYTE( "msa02152.g2", 0x040000, 0x020000, CRC(2b5e180c) SHA1(fad8f303d2d73eaf95d63955d7ed6712fee4dbce) )
	ROM_LOAD16_BYTE( "msa02152.g3", 0x040001, 0x020000, CRC(b616e33d) SHA1(8f2f03d8f6b5a8c7fa0134b2eb4b420609029a21) )
	EP_NUNS_SOUND
ROM_END

ROM_START( ep_nunsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02153.g0", 0x000000, 0x020000, CRC(1c9b6cfd) SHA1(0f1e6742fd0b319d077dcda2991d1a5b307fc066) )
	ROM_LOAD16_BYTE( "msa02153.g1", 0x000001, 0x020000, CRC(1391c05b) SHA1(75d72dbf4a81d03e184a75a39894158f85dc0115) )
	ROM_LOAD16_BYTE( "msa02153.g2", 0x040000, 0x020000, CRC(2b5e180c) SHA1(fad8f303d2d73eaf95d63955d7ed6712fee4dbce) )
	ROM_LOAD16_BYTE( "msa02153.g3", 0x040001, 0x020000, CRC(b616e33d) SHA1(8f2f03d8f6b5a8c7fa0134b2eb4b420609029a21) )
	EP_NUNS_SOUND
ROM_END




ROM_START( ep_otm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01021.g0", 0x000000, 0x020000, CRC(cd8a606f) SHA1(2c98ad6e517f7c6b190aaf6f2a1e663303fefd4c) )
	ROM_LOAD16_BYTE( "msb01021.g1", 0x000001, 0x020000, CRC(54982a59) SHA1(0a1aa72104e60179b3f05e69b000caf5b8ba0568) )
	MISSING_SOUND
ROM_END

ROM_START( ep_otma )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01022.g0", 0x000000, 0x020000, CRC(0f54a184) SHA1(4a8f5a449b601031ffbedba4ad90c677e7d0c23d) )
	ROM_LOAD16_BYTE( "msb01022.g1", 0x000001, 0x020000, CRC(f50e7ee9) SHA1(4a2e98a7b461d9ad8dbd0e2a02380364061d1bec) )
	MISSING_SOUND
ROM_END




ROM_START( ep_otmcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02030.g0", 0x000000, 0x020000, CRC(21088271) SHA1(add70898168d3e70be2c264964e3579abd900a1e) )
	ROM_LOAD16_BYTE( "msc02030.g1", 0x000001, 0x020000, CRC(c620bbbf) SHA1(f835e9125fda389836f94146e980da58e46d3f8e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_otmcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02031.g0", 0x000000, 0x020000, CRC(2da056e4) SHA1(ff5176810e7fe221191a4ffd3cf41b1e4bb591bc) )
	ROM_LOAD16_BYTE( "msc02031.g1", 0x000001, 0x020000, CRC(c620bbbf) SHA1(f835e9125fda389836f94146e980da58e46d3f8e) )
	MISSING_SOUND
ROM_END




ROM_START( ep_ozzie )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02008.g0", 0x000000, 0x020000, CRC(3a286f0b) SHA1(00bfb7a63d91fc807902edc823ff666f194ed7ad) )
	ROM_LOAD16_BYTE( "mua02008.g1", 0x000001, 0x020000, CRC(d3085168) SHA1(c41930ace5218f172c1bb8b93c9a0ba990b420e9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozziea )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02009.g0", 0x000000, 0x020000, CRC(64b19369) SHA1(04824ce1513768b70fcde9069df1b2d5eeb9a477) )
	ROM_LOAD16_BYTE( "mua02009.g1", 0x000001, 0x020000, CRC(298069d2) SHA1(caaaf85b1cc8816502aaa36c5ef58ddc0814e8e8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozzieb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua03027.g0", 0x000000, 0x020000, CRC(85a606a3) SHA1(b930775b3c9ab6d08732284e9bae8368e0116b7a) )
	ROM_LOAD16_BYTE( "mua03027.g1", 0x000001, 0x020000, CRC(ef6d2c53) SHA1(6000402551d672b66be2458c9eb293118b8fc978) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozziec )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua03028.g0", 0x000000, 0x020000, CRC(c46f92ca) SHA1(95f8827d57f4415216e5c39add6d61dcc7bfffb1) )
	ROM_LOAD16_BYTE( "mua03028.g1", 0x000001, 0x020000, CRC(5285e10a) SHA1(b12d61fa84818500cb0a030719e81f39bf5b6cae) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozzied )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01017.g0", 0x000000, 0x020000, CRC(bee0a5fb) SHA1(3e6227bf13199df068c620d1bf4b39ac34deace3) )
	ROM_LOAD16_BYTE( "msb01017.g1", 0x000001, 0x020000, CRC(bdff1f78) SHA1(23dd37cefec8fdfd39baaf284e2ae7e81b291393) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozziee )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01018.g0", 0x000000, 0x020000, CRC(264b8e56) SHA1(ec08004c384499463d781b9f53051765d7ba689a) )
	ROM_LOAD16_BYTE( "msb01018.g1", 0x000001, 0x020000, CRC(85dfe9a1) SHA1(19eb165f237189f2bffa18b95da64a5a5f1c3825) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozzief )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mua02014.g0", 0x000000, 0x020000, CRC(7a094a3a) SHA1(b7268c99767bda571ba500cac579ee4b901d295e) )
	ROM_LOAD16_BYTE( "mua02014.g1", 0x000001, 0x020000, CRC(1dd81c37) SHA1(ae7dbfc3e8b9fe80c911e76f5f690a115c5db2a6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozzieg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb010007.g0", 0x000000, 0x020000, CRC(94f571fb) SHA1(9a07ce7da646fb44567caa7ed43ead14e19c2d91) )
	ROM_LOAD16_BYTE( "sb010007.g1", 0x000001, 0x020000, CRC(6fc203e6) SHA1(ce3132b6fdc6a4ef6e8fa1e71a2da462dd971e8d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ozzieh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb010008.g0", 0x000000, 0x020000, CRC(2e8bade6) SHA1(39b0b4f344876d9583167ed8b1b339a814fe35ba) )
	ROM_LOAD16_BYTE( "sb010008.g1", 0x000001, 0x020000, CRC(dd537cfe) SHA1(804c52b74e307f5448eb61c5067fd6dd6794a722) )
	MISSING_SOUND
ROM_END




ROM_START( ep_pascl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02060.g0", 0x000000, 0x020000, CRC(fd1d8d72) SHA1(d273042c97c922ad177457e2f97a713a047159f6) )
	ROM_LOAD16_BYTE( "msc02060.g1", 0x000001, 0x020000, CRC(113677dd) SHA1(f11a7262b7b5b901690105c3a9dc0cd048960f6b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_pascla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02061.g0", 0x000000, 0x020000, CRC(a3310e6d) SHA1(d35b0327f0688c8de0a0761280b87e8e4dcf9bda) )
	ROM_LOAD16_BYTE( "msc02061.g1", 0x000001, 0x020000, CRC(76067ed4) SHA1(3647304b0bdd11e077fd5a9faaec94f97d5d1812) )
	MISSING_SOUND
ROM_END

ROM_START( ep_passpb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02066.g0", 0x000000, 0x020000, CRC(3276a6d8) SHA1(5707b68943a77de53a12b37d56b5820d86df66dd) )
	ROM_LOAD16_BYTE( "msc02066.g1", 0x000001, 0x020000, CRC(4b07a9ca) SHA1(ca6d225b336195b68d1d97a0aa9d542aa212c7a0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_passpc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02067.g0", 0x000000, 0x020000, CRC(2d0cac53) SHA1(9ed7b10269cc104743a77e8e2a1cb2c2397c2f70) )
	ROM_LOAD16_BYTE( "msc02067.g1", 0x000001, 0x020000, CRC(9efe90f1) SHA1(716db92b6a706b6b5afe77962e6350c4de0f7639) )
	MISSING_SOUND
ROM_END



ROM_START( ep_passp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02050.g0", 0x000000, 0x020000, CRC(b675f630) SHA1(74f52071ff20e05a626d7f42ae98d62b1cdae61f) )
	ROM_LOAD16_BYTE( "msc02050.g1", 0x000001, 0x020000, CRC(36b14ed2) SHA1(0ba24130c5204cf9876a81f2aacf1b1b5ad0f8d0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_passpa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02051.g0", 0x000000, 0x020000, CRC(2d6063f7) SHA1(b1f7b6020ce70240254d81e2228177abfa267758) )
	ROM_LOAD16_BYTE( "msc02051.g1", 0x000001, 0x020000, CRC(a7c77a83) SHA1(e59931205f5f8d6995b6595d156bd0a35f055218) )
	MISSING_SOUND
ROM_END




	/*
	Disk marked:

	PHARAOH'S TREASURE
	SS010003 DATA
	VIDEO EPOCH

	Fujistu MPF3102AT (10.2GB HDD)
	*/

#define EP_PHARO_HDD \
	DISK_REGION( "ata:0:hdd:image" ) \
	DISK_IMAGE( "ep_pharo", 0, SHA1(daf56705178bb2b6f3547418a98c361478702aed) )


ROM_START( ep_pharo )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01npt09.g0", 0x000000, 0x020000, CRC(db12dce7) SHA1(f6299b1acd4e0642de64cde13db901241052fe98) )
	ROM_LOAD16_BYTE( "i01npt09.g1", 0x000001, 0x020000, CRC(9ce8195a) SHA1(7127ceefbe545424cb4a7795527690dfd76b4b5c) )
	ROM_LOAD16_BYTE( "i01npt09.g2", 0x040000, 0x020000, CRC(3c725456) SHA1(b6c6bb799f462c4e11760c5b0661a03cd2151091) )
	ROM_LOAD16_BYTE( "i01npt09.g3", 0x040001, 0x020000, CRC(7823eb6f) SHA1(0adf369ca240aa15256d03a591c9e0b78d92c01e) )
	EP_PHARO_HDD
	MISSING_SOUND
ROM_END

ROM_START( ep_pharoa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i01ppt10.g0", 0x000000, 0x020000, CRC(db12dce7) SHA1(f6299b1acd4e0642de64cde13db901241052fe98) )
	ROM_LOAD16_BYTE( "i01ppt10.g1", 0x000001, 0x020000, CRC(77764d62) SHA1(81b5d9f6f8b6874a15e88bbb9a0c7479b9703538) )
	ROM_LOAD16_BYTE( "i01ppt10.g2", 0x040000, 0x020000, CRC(3c725456) SHA1(b6c6bb799f462c4e11760c5b0661a03cd2151091) )
	ROM_LOAD16_BYTE( "i01ppt10.g3", 0x040001, 0x020000, CRC(7823eb6f) SHA1(0adf369ca240aa15256d03a591c9e0b78d92c01e) )
	EP_PHARO_HDD
	MISSING_SOUND
ROM_END


#define EP_PESOS_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	/* guess based on filenames */ \
	ROM_LOAD( "p pesos.s0", 0x0000, 0x080000, CRC(970b2947) SHA1(14b0cc21a2fdb019143332869a656c4a4e3fb4a8) ) \
	ROM_LOAD( "p pesos.s1", 0x0000, 0x080000, CRC(d98d0a73) SHA1(db6fcf7d4f1e12d9484c308613660d6da9e423f6) )


ROM_START( ep_pesos )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gpk06_0.bin", 0x000000, 0x020000, CRC(fc29012c) SHA1(480a7f9771bfc91b80021911b88dfc164e401191) )
	ROM_LOAD16_BYTE( "gpk06_1.bin", 0x000001, 0x020000, CRC(9889ed1f) SHA1(6f17d2e7d8ea1f39158b41a8758c445583b99db8) )
	ROM_LOAD16_BYTE( "gpk06_2.bin", 0x040000, 0x020000, CRC(14ff63b1) SHA1(281d60527679f789ec5dc15b855a19c1e129f452) )
	ROM_LOAD16_BYTE( "gpk06_3.bin", 0x040001, 0x020000, CRC(d6a0db9f) SHA1(a66446714f3477142809a1208e807da56c764d5b) )
	EP_PESOS_SOUND
ROM_END

ROM_START( ep_pesosa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gpk07_0.bin", 0x000000, 0x020000, CRC(fc29012c) SHA1(480a7f9771bfc91b80021911b88dfc164e401191) )
	ROM_LOAD16_BYTE( "gpk07_1.bin", 0x000001, 0x020000, CRC(7317b927) SHA1(da5b88c1004930ebd2af848785e2e101fc4f6da3) )
	ROM_LOAD16_BYTE( "gpk07_2.bin", 0x040000, 0x020000, CRC(14ff63b1) SHA1(281d60527679f789ec5dc15b855a19c1e129f452) )
	ROM_LOAD16_BYTE( "gpk07_3.bin", 0x040001, 0x020000, CRC(d6a0db9f) SHA1(a66446714f3477142809a1208e807da56c764d5b) )
	EP_PESOS_SOUND
ROM_END

ROM_START( ep_pesosb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gpk08_0.bin", 0x000000, 0x020000, CRC(75a00628) SHA1(a29b200b5f06d221c4809c8e47b55130a2612481) )
	ROM_LOAD16_BYTE( "gpk08_1.bin", 0x000001, 0x020000, CRC(f818fa75) SHA1(f5424c36f39e9469f1eeb45bba3d229420299a61) )
	ROM_LOAD16_BYTE( "gpk08_2.bin", 0x040000, 0x020000, CRC(da400cf4) SHA1(763952a19736f34bcb2957e70a966cb6d5524313) )
	ROM_LOAD16_BYTE( "gpk08_3.bin", 0x040001, 0x020000, CRC(5ea4d325) SHA1(c47cfe04f443039c69da15f773c556564b4cd6b9) )
	EP_PESOS_SOUND
ROM_END

ROM_START( ep_pesosc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gpk09_0.bin", 0x000000, 0x020000, CRC(75a00628) SHA1(a29b200b5f06d221c4809c8e47b55130a2612481) )
	ROM_LOAD16_BYTE( "gpk09_1.bin", 0x000001, 0x020000, CRC(1386ae4d) SHA1(d9f270793fa2f787649e6b3ad8396f311e580446) )
	ROM_LOAD16_BYTE( "gpk09_2.bin", 0x040000, 0x020000, CRC(da400cf4) SHA1(763952a19736f34bcb2957e70a966cb6d5524313) )
	ROM_LOAD16_BYTE( "gpk09_3.bin", 0x040001, 0x020000, CRC(5ea4d325) SHA1(c47cfe04f443039c69da15f773c556564b4cd6b9) )
	EP_PESOS_SOUND
ROM_END




ROM_START( ep_baskr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02169.g0", 0x000000, 0x020000, CRC(b5d4854f) SHA1(9b9449fa1f8b4f51704e9bfa1a1f8c187e93641f) )
	ROM_LOAD16_BYTE( "msa02169.g1", 0x000001, 0x020000, CRC(f2fe8f9b) SHA1(d6720d22cc84c156e3c78c3329ef03ba92d24e34) )
	ROM_LOAD16_BYTE( "msa02169.g2", 0x040000, 0x020000, CRC(78dd9a42) SHA1(6e68ddca273b257f21f341a8ab3cb229bec08bd5) )
	ROM_LOAD16_BYTE( "msa02169.g3", 0x040001, 0x020000, CRC(752e7847) SHA1(05f4aa3d6a0a0f0b3b29d3fe7644a67b9c3942ab) )
	MISSING_SOUND
ROM_END

ROM_START( ep_baskra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02166.g0", 0x000000, 0x020000, CRC(e0440891) SHA1(8caa65f5b0351b7388f73da46518ccf6c0f6a687) )
	ROM_LOAD16_BYTE( "msa02166.g1", 0x000001, 0x020000, CRC(78a0b6f2) SHA1(f8225062151ab8b7e277e92ba0b27007a119399a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_baskrb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02167.g0", 0x000000, 0x020000, CRC(2f291334) SHA1(40554c002ccd83dea48982cb0e142b180f9128e9) )
	ROM_LOAD16_BYTE( "msa02167.g1", 0x000001, 0x020000, CRC(fbaef2f0) SHA1(e5d4f7b373f1a7421b19978949b114f816bbd7a2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_baskrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02168.g0", 0x000000, 0x020000, CRC(db06e0ff) SHA1(60cfed8272d74f91e59959f4a456a065e40d3450) )
	ROM_LOAD16_BYTE( "msa02168.g1", 0x000001, 0x020000, CRC(19e36866) SHA1(bd25abfecb6f3b02de3e96208453b92b252228d9) )
	ROM_LOAD16_BYTE( "msa02168.g2", 0x040000, 0x020000, CRC(78dd9a42) SHA1(6e68ddca273b257f21f341a8ab3cb229bec08bd5) )
	ROM_LOAD16_BYTE( "msa02168.g3", 0x040001, 0x020000, CRC(752e7847) SHA1(05f4aa3d6a0a0f0b3b29d3fe7644a67b9c3942ab) )
	MISSING_SOUND
ROM_END

ROM_START( ep_baskrd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pound_baskvi.g0", 0x000000, 0x020000, CRC(122edc2d) SHA1(051dce228e67335c72e75fcb92917a6c6e73b62f) )
	ROM_LOAD16_BYTE( "pound_baskvi.g1", 0x000001, 0x020000, CRC(35bc00f2) SHA1(bfd9db95b78a0edee0a77960c0ac227cffc717b8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_baskre )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "p_baskvi.g0", 0x000000, 0x020000, CRC(5fe739e1) SHA1(412fd46e48d0202b3976023ad5af797ff7ad9127) )
	ROM_LOAD16_BYTE( "p_baskvi.g1", 0x000001, 0x020000, CRC(05984c72) SHA1(9252ad7a8a54495b2d776548278712e0b4ee9bd4) )
	MISSING_SOUND
ROM_END




ROM_START( ep_pwrpl ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pply03.p0", 0x000000, 0x100000, CRC(0596031d) SHA1(3af59cc4b3271cabf538559345f7236d1af658a2) )
	ROM_LOAD16_BYTE( "pply03.p1", 0x000001, 0x100000, CRC(df93adf8) SHA1(89cd8d68d7b573238f0bc432dbef4c6acd645f49) )
	MISSING_SOUND
ROM_END

ROM_START( ep_pwrpla ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pply03p.p0", 0x000000, 0x100000, CRC(0596031d) SHA1(3af59cc4b3271cabf538559345f7236d1af658a2) )
	ROM_LOAD16_BYTE( "pply03p.p1", 0x000001, 0x100000, CRC(8a78c043) SHA1(14c239ac1ee63f22b30b6bb34b66bc3d7eb14041) )
	MISSING_SOUND
ROM_END




ROM_START( ep_rags )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc010001.g0", 0x000000, 0x020000, CRC(98546484) SHA1(b6d67508a6c57850cb2085183ee07788bb160b44) )
	ROM_LOAD16_BYTE( "sc010001.g1", 0x000001, 0x020000, CRC(1d81cc86) SHA1(66f57e985734a8346651d5b9d139aaec3a568f31) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ragsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sc010002.g0", 0x000000, 0x020000, CRC(bf18f77d) SHA1(01d0ada5f67295f26f1c852a52d89e7953da3705) )
	ROM_LOAD16_BYTE( "sc010002.g1", 0x000001, 0x020000, CRC(201143d9) SHA1(0026d30c80ba2102c32952ffd57cb8e2d35242d8) )
	MISSING_SOUND
ROM_END




ROM_START( ep_react )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pre23.g0", 0x000000, 0x020000, CRC(71ecf58b) SHA1(0ee06c3ad8f7f8b840cd49221d0a7da8da2d1b05) )
	ROM_LOAD16_BYTE( "i00pre23.g1", 0x000001, 0x020000, CRC(4bf53794) SHA1(6402d620ab81a5e4c32b1e294a941b6f55e4fc9d) )
	ROM_LOAD16_BYTE( "i00pre23.g2", 0x040000, 0x020000, CRC(97ec88f4) SHA1(69ba1c3807c6a441f937854c1ddf1f2a12d80120) )
	ROM_LOAD16_BYTE( "i00pre23.g3", 0x040001, 0x020000, CRC(5326102d) SHA1(06a39cdf078a8c3e32f48884b9943382f12b84d3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_reacta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nre18.g0", 0x000000, 0x020000, CRC(b322e0ac) SHA1(fed9906fead45a4b4759fc6740fbc4437914e79c) )
	ROM_LOAD16_BYTE( "i00nre18.g1", 0x000001, 0x020000, CRC(7355b9ef) SHA1(77dfa67435ae1b9de9764c977ec6714a76bf98f4) )
	MISSING_SOUND
ROM_END

ROM_START( ep_reactb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nre20.g0", 0x000000, 0x020000, CRC(b322e0ac) SHA1(fed9906fead45a4b4759fc6740fbc4437914e79c) )
	ROM_LOAD16_BYTE( "i00nre20.g1", 0x000001, 0x020000, CRC(7f1817de) SHA1(80e3a4aaed1557ccfd738d23404719f873729367) )
	MISSING_SOUND
ROM_END

ROM_START( ep_reactc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nre22.g0", 0x000000, 0x020000, CRC(71ecf58b) SHA1(0ee06c3ad8f7f8b840cd49221d0a7da8da2d1b05) )
	ROM_LOAD16_BYTE( "i00nre22.g1", 0x000001, 0x020000, CRC(a06b63ac) SHA1(2aab328c7b30df338be5a1fd9e67d9336a615c31) )
	ROM_LOAD16_BYTE( "i00nre22.g2", 0x040000, 0x020000, CRC(97ec88f4) SHA1(69ba1c3807c6a441f937854c1ddf1f2a12d80120) )
	ROM_LOAD16_BYTE( "i00nre22.g3", 0x040001, 0x020000, CRC(5326102d) SHA1(06a39cdf078a8c3e32f48884b9943382f12b84d3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_reactd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pre19.g0", 0x000000, 0x020000, CRC(b322e0ac) SHA1(fed9906fead45a4b4759fc6740fbc4437914e79c) )
	ROM_LOAD16_BYTE( "i00pre19.g1", 0x000001, 0x020000, CRC(98cbedd7) SHA1(7392970cadc7ee8a2b7a037f1ae9015e6ea85ed1) )
	ROM_LOAD16_BYTE( "i00pre21.g0", 0x000000, 0x020000, CRC(b322e0ac) SHA1(fed9906fead45a4b4759fc6740fbc4437914e79c) )
	ROM_LOAD16_BYTE( "i00pre21.g1", 0x000001, 0x020000, CRC(948643e6) SHA1(0aa32e6913e1382f051fa10506b1634695d55a7e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_reacte )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "reactor.g0", 0x000000, 0x020000, CRC(b3e1bb2c) SHA1(41a11dd0dc6fb2c2b2ef01fa7afc56333c2b4506) )
	ROM_LOAD16_BYTE( "reactor.g1", 0x000001, 0x020000, CRC(076973ad) SHA1(479ea988f82797635f5c3f51fdd15663f7e05be1) )
	MISSING_SOUND
ROM_END




ROM_START( ep_roost )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02222.g0", 0x000000, 0x020000, CRC(651bb473) SHA1(472d4a2fb6c2ce78e803332c08bf565308bb9949) )
	ROM_LOAD16_BYTE( "msa02222.g1", 0x000001, 0x020000, CRC(abb7bc38) SHA1(8934fa7761bff4c945717ecb305915f8f99b08fc) )
	ROM_LOAD16_BYTE( "msa02222.g2", 0x040000, 0x020000, CRC(dea80364) SHA1(b822f18b4d58f04e0bcb9dcd0d29d6f0cfb2f49f) )
	ROM_LOAD16_BYTE( "msa02222.g3", 0x040001, 0x020000, CRC(587a19a6) SHA1(6f30924206aad31355ffa84029cdb45f0d69ffb8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_roosta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02223.g0", 0x000000, 0x020000, CRC(9bae60d9) SHA1(b3886d1560df44a902bb9654f98dacf637822d8d) )
	ROM_LOAD16_BYTE( "msa02223.g1", 0x000001, 0x020000, CRC(76bf8dad) SHA1(9f85aa5bb7301197e9412b78d34e60e58dcd679c) )
	ROM_LOAD16_BYTE( "msa02223.g2", 0x040000, 0x020000, CRC(dea80364) SHA1(b822f18b4d58f04e0bcb9dcd0d29d6f0cfb2f49f) )
	ROM_LOAD16_BYTE( "msa02223.g3", 0x040001, 0x020000, CRC(587a19a6) SHA1(6f30924206aad31355ffa84029cdb45f0d69ffb8) )
	MISSING_SOUND
ROM_END




ROM_START( ep_rtt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grt02_0.bin", 0x000000, 0x020000, CRC(23f6f4dd) SHA1(6f1d243c61b851a1d3519c108dd14d7d7b199efe) )
	ROM_LOAD16_BYTE( "grt02_1.bin", 0x000001, 0x020000, CRC(7e7da84e) SHA1(8a967ea00f0ed1840dc0e455c6a326552fdf2a24) )
	ROM_LOAD16_BYTE( "grt02_2.bin", 0x040000, 0x020000, CRC(0a5249f2) SHA1(aba49163251eaff631190ac2e0ee0a23b90322a7) )
	ROM_LOAD16_BYTE( "grt02_3.bin", 0x040001, 0x020000, CRC(215ee1c8) SHA1(b3d9ffe3d18829bca84699666bbd2a8098696827) )
	MISSING_SOUND
ROM_END

ROM_START( ep_rtta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grt03_0.bin", 0x000000, 0x020000, CRC(23f6f4dd) SHA1(6f1d243c61b851a1d3519c108dd14d7d7b199efe) )
	ROM_LOAD16_BYTE( "grt03_1.bin", 0x000001, 0x020000, CRC(95e3fc76) SHA1(1e3e0371f2ab8268b4acb125043a25c2bfff5cf2) )
	ROM_LOAD16_BYTE( "grt03_2.bin", 0x040000, 0x020000, CRC(0a5249f2) SHA1(aba49163251eaff631190ac2e0ee0a23b90322a7) )
	ROM_LOAD16_BYTE( "grt03_3.bin", 0x040001, 0x020000, CRC(215ee1c8) SHA1(b3d9ffe3d18829bca84699666bbd2a8098696827) )
	MISSING_SOUND
ROM_END


#define EP_ROYRL_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	/* guess based on filename */ \
	ROM_LOAD( "royal r.s0", 0x000000, 0x080000, CRC(a1060eac) SHA1(47b0383e4c0eb714eaba55014fbc5597233a5d53) ) \
	ROM_LOAD( "royal r.s1", 0x080000, 0x080000, CRC(c0caaeb8) SHA1(79efa22844137a0326b5887df186a3220cd2577b) )

ROM_START( ep_royrl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr02_0.bin", 0x000000, 0x020000, CRC(e97cc6f9) SHA1(fbccd72f5372e06013a69d9c040b70b554a147f7) )
	ROM_LOAD16_BYTE( "grr02_1.bin", 0x000001, 0x020000, CRC(313e88a9) SHA1(27cd83f0f2fbcc6887e8eac8088d561c6b57c991) )
	ROM_LOAD16_BYTE( "grr02_2.bin", 0x040000, 0x020000, CRC(50b9e69b) SHA1(ea0142fec720a6957c92b2e3f7d9853b59504543) )
	ROM_LOAD16_BYTE( "grr02_3.bin", 0x040001, 0x020000, CRC(5ecfacf8) SHA1(c6cff9d0ab0543a64a5a8d34699b5719062210cb) )
	EP_ROYRL_SOUND
ROM_END


ROM_START( ep_royrla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr03_0.bin", 0x000000, 0x020000, CRC(e97cc6f9) SHA1(fbccd72f5372e06013a69d9c040b70b554a147f7) )
	ROM_LOAD16_BYTE( "grr03_1.bin", 0x000001, 0x020000, CRC(daa0dc91) SHA1(54d3a39948feea3cf48816228a2f6119e713ee96) )
	ROM_LOAD16_BYTE( "grr03_2.bin", 0x040000, 0x020000, CRC(50b9e69b) SHA1(ea0142fec720a6957c92b2e3f7d9853b59504543) )
	ROM_LOAD16_BYTE( "grr03_3.bin", 0x040001, 0x020000, CRC(5ecfacf8) SHA1(c6cff9d0ab0543a64a5a8d34699b5719062210cb) )
	EP_ROYRL_SOUND
ROM_END

ROM_START( ep_royrlb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr06_0.bin", 0x000000, 0x020000, CRC(421b28ef) SHA1(3f1a8d6eff9c669a179de3dea307f0bd05f3ab61) )
	ROM_LOAD16_BYTE( "grr06_1.bin", 0x000001, 0x020000, CRC(6c91cc3c) SHA1(8fe3715dba6cc115183ad58d0430b75ee9c76e98) )
	ROM_LOAD16_BYTE( "grr06_2.bin", 0x040000, 0x020000, CRC(9714afe2) SHA1(d681e0ba19df581e71ccd194171e6498c8f82bc5) )
	ROM_LOAD16_BYTE( "grr06_3.bin", 0x040001, 0x020000, CRC(b87c3c9f) SHA1(e251b6bf2bbca571a50917baf90537700dce52b9) )
	EP_ROYRL_SOUND
ROM_END

ROM_START( ep_royrlc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr07_0.bin", 0x000000, 0x020000, CRC(421b28ef) SHA1(3f1a8d6eff9c669a179de3dea307f0bd05f3ab61) )
	ROM_LOAD16_BYTE( "grr07_1.bin", 0x000001, 0x020000, CRC(870f9804) SHA1(5d935d2de4440054953c58a0669486dff36d3b20) )
	ROM_LOAD16_BYTE( "grr07_2.bin", 0x040000, 0x020000, CRC(9714afe2) SHA1(d681e0ba19df581e71ccd194171e6498c8f82bc5) )
	ROM_LOAD16_BYTE( "grr07_3.bin", 0x040001, 0x020000, CRC(b87c3c9f) SHA1(e251b6bf2bbca571a50917baf90537700dce52b9) )
	EP_ROYRL_SOUND
ROM_END

ROM_START( ep_royrld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr08_0.bin", 0x000000, 0x020000, CRC(271be228) SHA1(59ba40f2b27cb018904cd399fb478ee11fa3d204) )
	ROM_LOAD16_BYTE( "grr08_1.bin", 0x000001, 0x020000, CRC(d8ac2204) SHA1(fb5a984e1bd6790271fccabdcb078fe7f80efc6f) )
	ROM_LOAD16_BYTE( "grr08_2.bin", 0x040000, 0x020000, CRC(26cdfb40) SHA1(fef75914c3268e997e73ca353524126a379df8e1) )
	ROM_LOAD16_BYTE( "grr08_3.bin", 0x040001, 0x020000, CRC(6d9eae8d) SHA1(df0975f622772f35c0130245bfeb6037dfaf8c63) )
	EP_ROYRL_SOUND
ROM_END

ROM_START( ep_royrle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grr09_0.bin", 0x000000, 0x020000, CRC(271be228) SHA1(59ba40f2b27cb018904cd399fb478ee11fa3d204) )
	ROM_LOAD16_BYTE( "grr09_1.bin", 0x000001, 0x020000, CRC(3332763c) SHA1(3a127b6b2c1df261156577ad194339ad742bec4b) )
	ROM_LOAD16_BYTE( "grr09_2.bin", 0x040000, 0x020000, CRC(26cdfb40) SHA1(fef75914c3268e997e73ca353524126a379df8e1) )
	ROM_LOAD16_BYTE( "grr09_3.bin", 0x040001, 0x020000, CRC(6d9eae8d) SHA1(df0975f622772f35c0130245bfeb6037dfaf8c63) )
	EP_ROYRL_SOUND
ROM_END





ROM_START( ep_scrm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "scde20.p0", 0x000000, 0x020000, CRC(d8abf226) SHA1(717d4ddf65e3861c83bcaa6fe6d909335b21cffd) )
	ROM_LOAD16_BYTE( "scde20.p1", 0x000001, 0x020000, CRC(4dc08f93) SHA1(3838647f31b49631ffb65f4d7ef67a3dab49911d) )
	ROM_LOAD16_BYTE( "scde20.p2", 0x040000, 0x020000, CRC(441f4e38) SHA1(c2a349c8eacc3255c98b028001dfdb72152bf92d) )
	ROM_LOAD16_BYTE( "scde20.p3", 0x040001, 0x020000, CRC(ae12f3d9) SHA1(5172c7ff6926dba531f9ce4824f9ada12f1b0643) )
	MISSING_SOUND
ROM_END

ROM_START( ep_scrma )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "scde20p.p0", 0x000000, 0x020000, CRC(d8abf226) SHA1(717d4ddf65e3861c83bcaa6fe6d909335b21cffd) )
	ROM_LOAD16_BYTE( "scde20p.p1", 0x000001, 0x020000, CRC(52537de8) SHA1(48717daaccde8ed3d6301a12a256c3f0022b1a5a) )
	ROM_LOAD16_BYTE( "scde20p.p2", 0x040000, 0x020000, CRC(441f4e38) SHA1(c2a349c8eacc3255c98b028001dfdb72152bf92d) )
	ROM_LOAD16_BYTE( "scde20p.p3", 0x040001, 0x020000, CRC(ae12f3d9) SHA1(5172c7ff6926dba531f9ce4824f9ada12f1b0643) )
	MISSING_SOUND
ROM_END

ROM_START( ep_scrmb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "scde10.p0", 0x000000, 0x020000, CRC(8ba8757c) SHA1(730905662127262acd5029b77b6e62c99af83e4f) )
	ROM_LOAD16_BYTE( "scde10.p1", 0x000001, 0x020000, CRC(e039d7a0) SHA1(7ea6522c34b9eea9da4d96d96c2a036b20932277) )
	ROM_LOAD16_BYTE( "scde10.p2", 0x040000, 0x020000, CRC(fc1aab67) SHA1(46ddcd4a38390635da35384514323ba46b0594dc) )
	ROM_LOAD16_BYTE( "scde10.p3", 0x040001, 0x020000, CRC(fce7cd4a) SHA1(b7cfdbbf95c9f89bd87688db870547d197205af6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_scrmc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "scde10p.p0", 0x000000, 0x020000, CRC(8ba8757c) SHA1(730905662127262acd5029b77b6e62c99af83e4f) )
	ROM_LOAD16_BYTE( "scde10p.p1", 0x000001, 0x020000, CRC(ffaa25db) SHA1(863ce966d78ad8c49297a47afa6d80283044475a) )
	ROM_LOAD16_BYTE( "scde10p.p2", 0x040000, 0x020000, CRC(fc1aab67) SHA1(46ddcd4a38390635da35384514323ba46b0594dc) )
	ROM_LOAD16_BYTE( "scde10p.p3", 0x040001, 0x020000, CRC(fce7cd4a) SHA1(b7cfdbbf95c9f89bd87688db870547d197205af6) )
	MISSING_SOUND
ROM_END

#define EP_SECAG_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "s_agent.s0", 0x000000, 0x080000, CRC(58a509db) SHA1(339a69eab79c4e7886a18b92392a25ba2ccb7c9a) ) \
	ROM_LOAD( "s_agent.s1", 0x080000, 0x080000, CRC(8e0117f8) SHA1(e7d1c64d995a7f7e2d4e2ed0bccb303fe5bb267e) )

ROM_START( ep_secag )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02209.g0", 0x000000, 0x020000, CRC(2725ddaa) SHA1(f1902dde12aa771ff567b029e52936d548e7eef8) )
	ROM_LOAD16_BYTE( "msa02209.g1", 0x000001, 0x020000, CRC(a819d75b) SHA1(81207ae2a0c1f422f6cef0e0960425bcd9a5c21d) )
	EP_SECAG_SOUND
ROM_END

ROM_START( ep_secaga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02210.g0", 0x000000, 0x020000, CRC(7002fc31) SHA1(1c1228212be1cc6be81ebccb806d66c14af1619c) )
	ROM_LOAD16_BYTE( "msa02210.g1", 0x000001, 0x020000, CRC(b313226e) SHA1(fa13a75e2ac92db97f09612a2dfbb9cf3a8b7932) )
	EP_SECAG_SOUND
ROM_END

ROM_START( ep_secagb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "s_agent.g0", 0x000000, 0x020000, CRC(cd612622) SHA1(265623320c93dbfcc0798c969f6b954c9f03f4f7) )
	ROM_LOAD16_BYTE( "s_agent.g1", 0x000001, 0x020000, CRC(88cdbb76) SHA1(a75b8b6bf02fef213cabbc2b777ce10c678bf029) )
	EP_SECAG_SOUND
ROM_END




ROM_START( ep_simfr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01023.g0", 0x000000, 0x020000, CRC(d3d2e8a4) SHA1(75e586e5ba996155e3f131206556b014967641d2) )
	ROM_LOAD16_BYTE( "msb01023.g1", 0x000001, 0x020000, CRC(50411db6) SHA1(185043d76d8b1f5e96e1028a75013bc7f234bea8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_simfra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01024.g0", 0x000000, 0x020000, CRC(3faf8705) SHA1(532845b6f49f441ae48f8792c8688ec40baf8e32) )
	ROM_LOAD16_BYTE( "msb01024.g1", 0x000001, 0x020000, CRC(ad03232c) SHA1(838917a84216c9ba0d862b0f83463a40a3a9d7e3) )
	MISSING_SOUND
ROM_END





#define EP_SRWIN_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "s_winlot.s0", 0x000000, 0x080000, CRC(33ec2d5a) SHA1(57309e9053a685f54ebd8051c01862e9bd5cd72d) ) \
	ROM_LOAD( "s_winlot.s1", 0x080000, 0x080000, CRC(f0d3078e) SHA1(e89ccb9fa1d4e85a35497f677572a3723300fd79) )

ROM_START( ep_srwin )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02198.g0", 0x000000, 0x020000, CRC(901f1c40) SHA1(5f4514c6d3b5017ec95486d4397d0c51e3ed572b) )
	ROM_LOAD16_BYTE( "msa02198.g1", 0x000001, 0x020000, CRC(1badae1d) SHA1(a1bcc3e7558ba5725216cc1c1283697241943f1a) )
	ROM_LOAD16_BYTE( "msa02198.g2", 0x040000, 0x020000, CRC(ff4980ba) SHA1(4b239358eb81507da85b6aa8c17fe7b014cbe7b4) )
	ROM_LOAD16_BYTE( "msa02198.g3", 0x040001, 0x020000, CRC(6696b89c) SHA1(ead4da6f0ee305a869da6e07106517e480f2dc2a) )
	EP_SRWIN_SOUND
ROM_END

ROM_START( ep_srwina )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02146.g0", 0x000000, 0x020000, CRC(5687d0c9) SHA1(368e4fb8020a209b4ff3b7568c3dc363f10eb70a) )
	ROM_LOAD16_BYTE( "msa02146.g1", 0x000001, 0x020000, CRC(768748c6) SHA1(8213f14304ca426df5e66657dd2106d5b6bd0f87) )
	EP_SRWIN_SOUND
ROM_END

ROM_START( ep_srwinb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02147.g0", 0x000000, 0x020000, CRC(cd92450e) SHA1(b7ccf1c08784c20b3728111b757b4b8e4cb82eb7) )
	ROM_LOAD16_BYTE( "msa02147.g1", 0x000001, 0x020000, CRC(70cadd69) SHA1(cb3682474f1c15af975397645609e28b5e0b4efd) )
	EP_SRWIN_SOUND
ROM_END

ROM_START( ep_srwinc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02197.g0", 0x000000, 0x020000, CRC(e7a8d103) SHA1(49f9bb495d72adbb0d9576c818a9e99b8e772006) )
	ROM_LOAD16_BYTE( "msa02197.g1", 0x000001, 0x020000, CRC(87c349dc) SHA1(18eb21a263475dc564afc37d3bb464b5a32d9622) )
	ROM_LOAD16_BYTE( "msa02197.g2", 0x040000, 0x020000, CRC(ff4980ba) SHA1(4b239358eb81507da85b6aa8c17fe7b014cbe7b4) )
	ROM_LOAD16_BYTE( "msa02197.g3", 0x040001, 0x020000, CRC(6696b89c) SHA1(ead4da6f0ee305a869da6e07106517e480f2dc2a) )
	EP_SRWIN_SOUND
ROM_END

ROM_START( ep_srwind )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "s_winlot.g0", 0x000000, 0x020000, CRC(db63ffb0) SHA1(14abc6f5d9a93867075a64c6cc8f486011a94261) )
	ROM_LOAD16_BYTE( "s_winlot.g1", 0x000001, 0x020000, CRC(46d2f622) SHA1(b15bbcaffa5b75624517a4273c6a8b0335cfe2c9) )
	EP_SRWIN_SOUND
ROM_END





ROM_START( ep_spart )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nsp20.g0", 0x000000, 0x020000, CRC(8b20876d) SHA1(ce5ee4da44eb7d958afe859ac70b00b6b7c20b1b) )
	ROM_LOAD16_BYTE( "i99nsp20.g1", 0x000001, 0x020000, CRC(0e021d03) SHA1(edc404c3c2323311f1ea8ce18c30f2e149114e78) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sparta )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99nsp22.g0", 0x000000, 0x020000, CRC(8b20876d) SHA1(ce5ee4da44eb7d958afe859ac70b00b6b7c20b1b) )
	ROM_LOAD16_BYTE( "i99nsp22.g1", 0x000001, 0x020000, CRC(024fb332) SHA1(8a702541f3f71b080edc0bf2cd3be10a18dea7ab) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spartb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99psp21.g0", 0x000000, 0x020000, CRC(8b20876d) SHA1(ce5ee4da44eb7d958afe859ac70b00b6b7c20b1b) )
	ROM_LOAD16_BYTE( "i99psp21.g1", 0x000001, 0x020000, CRC(e59c493b) SHA1(56b630635b87c2d28354c77437e4f986b3a4c971) )
	MISSING_SOUND
ROM_END




ROM_START( ep_spcbw )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02207.g0", 0x000000, 0x020000, CRC(bd21f23e) SHA1(6ef98fc384bf3bd09cb7213b15a7de0a951d6625) )
	ROM_LOAD16_BYTE( "msa02207.g1", 0x000001, 0x020000, CRC(c77b101c) SHA1(11b6878c8706b11c3f768718d16b0cec63671f17) )
	ROM_LOAD16_BYTE( "msa02207.g2", 0x040000, 0x020000, CRC(bfae74f0) SHA1(74a8f51250a93da18f6f47e621d65eaf41b0f1fd) )
	ROM_LOAD16_BYTE( "msa02207.g3", 0x040001, 0x020000, CRC(fe7b4d33) SHA1(a1ce6b19e4bc6a28ab7a4ee90bc6cc9f91c1c6a9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02208.g0", 0x000000, 0x020000, CRC(3f0e94db) SHA1(22a985fcc0efc151ffb31c961dac10a014f3d99a) )
	ROM_LOAD16_BYTE( "msa02208.g1", 0x000001, 0x020000, CRC(7fc85672) SHA1(8c4a0d0dc4f7bdd1fe58528483854d91b0491a66) )
	ROM_LOAD16_BYTE( "msa02208.g2", 0x040000, 0x020000, CRC(bfae74f0) SHA1(74a8f51250a93da18f6f47e621d65eaf41b0f1fd) )
	ROM_LOAD16_BYTE( "msa02208.g3", 0x040001, 0x020000, CRC(fe7b4d33) SHA1(a1ce6b19e4bc6a28ab7a4ee90bc6cc9f91c1c6a9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02238.g0", 0x000000, 0x020000, CRC(6039cb85) SHA1(2c807de680a8ed0b86e43938ea3b8062037ce919) )
	ROM_LOAD16_BYTE( "msa02238.g1", 0x000001, 0x020000, CRC(a561239d) SHA1(bdd9ad089f597cd2ba5295375839fbfc24f3b082) )
	ROM_LOAD16_BYTE( "msa02238.g2", 0x040000, 0x020000, CRC(ee669522) SHA1(ac9807b2639c4862658cc80e5d8ca6cf4cfedf7d) )
	ROM_LOAD16_BYTE( "msa02238.g3", 0x040001, 0x020000, CRC(5209f92c) SHA1(0c3089a359f40ed6cc30275db10bf517f16ada13) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02239.g0", 0x000000, 0x020000, CRC(6f16636b) SHA1(a31eaf8621b282a7640a3a24a1cb4d03402d85d1) )
	ROM_LOAD16_BYTE( "msa02239.g1", 0x000001, 0x020000, CRC(54d09d38) SHA1(b369baffd78d4df1b770ba0ab7a3a76b68dd2f39) )
	ROM_LOAD16_BYTE( "msa02239.g2", 0x040000, 0x020000, CRC(ee669522) SHA1(ac9807b2639c4862658cc80e5d8ca6cf4cfedf7d) )
	ROM_LOAD16_BYTE( "msa02239.g3", 0x040001, 0x020000, CRC(5209f92c) SHA1(0c3089a359f40ed6cc30275db10bf517f16ada13) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02246.g0", 0x000000, 0x020000, CRC(06ddfa10) SHA1(850be19bf9e6cd2d77a3c6f720395b792f2605a6) )
	ROM_LOAD16_BYTE( "msa02246.g1", 0x000001, 0x020000, CRC(d96dd0f6) SHA1(170cd17d9c7be6c5b1409fecda39576f402a867a) )
	ROM_LOAD16_BYTE( "msa02246.g2", 0x040000, 0x020000, CRC(24d127bc) SHA1(be8a8d52b2c0f68ccea3355e1bd226cd76d38aea) )
	ROM_LOAD16_BYTE( "msa02246.g3", 0x040001, 0x020000, CRC(5ed0c3b4) SHA1(9e0d7583a86f2e2d7463e88343152e26415cacc4) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02247.g0", 0x000000, 0x020000, CRC(fc35a289) SHA1(8523e9e3ca6d767c528044eabefb631b90608193) )
	ROM_LOAD16_BYTE( "msa02247.g1", 0x000001, 0x020000, CRC(c3f14c50) SHA1(b3719560cbd22d565c250518c95d5cc8d9613c0c) )
	ROM_LOAD16_BYTE( "msa02247.g2", 0x040000, 0x020000, CRC(24d127bc) SHA1(be8a8d52b2c0f68ccea3355e1bd226cd76d38aea) )
	ROM_LOAD16_BYTE( "msa02247.g3", 0x040001, 0x020000, CRC(5ed0c3b4) SHA1(9e0d7583a86f2e2d7463e88343152e26415cacc4) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02258.g0", 0x000000, 0x020000, CRC(cb9e35d6) SHA1(6f753ea9ebed8f8e55012897c85c343e9fc9056b) )
	ROM_LOAD16_BYTE( "msa02258.g1", 0x000001, 0x020000, CRC(422365b6) SHA1(fd8a4b8a853eb2b8d9382a58205a2f78c71c9efb) )
	ROM_LOAD16_BYTE( "msa02258.g2", 0x040000, 0x020000, CRC(ba7716e1) SHA1(8b23a851f3c1301db2ecc23d7ff086bea481ba11) )
	ROM_LOAD16_BYTE( "msa02258.g3", 0x040001, 0x020000, CRC(179fd826) SHA1(8379f430d7684e2bfe63b32c05c2436474bc681f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02259.g0", 0x000000, 0x020000, CRC(9fc527fa) SHA1(646af511e89d877d1047fb5e4e2a109450105ab3) )
	ROM_LOAD16_BYTE( "msa02259.g1", 0x000001, 0x020000, CRC(9251080e) SHA1(46d527251b6a085313dc72f30c3058fe7e10970d) )
	ROM_LOAD16_BYTE( "msa02259.g2", 0x040000, 0x020000, CRC(ba7716e1) SHA1(8b23a851f3c1301db2ecc23d7ff086bea481ba11) )
	ROM_LOAD16_BYTE( "msa02259.g3", 0x040001, 0x020000, CRC(179fd826) SHA1(8379f430d7684e2bfe63b32c05c2436474bc681f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03302.g0", 0x000000, 0x020000, CRC(16bdb86d) SHA1(d38fe8339eebaa84cc6bcc2221ee63de081c8091) )
	ROM_LOAD16_BYTE( "msa03302.g1", 0x000001, 0x020000, CRC(846e8a7a) SHA1(28170b22fe547c8569602e6c948b54a002234958) )
	ROM_LOAD16_BYTE( "msa03302.g2", 0x040000, 0x020000, CRC(35220b0c) SHA1(cb2452e3bc787ef3674e344e4c45747274f18c77) )
	ROM_LOAD16_BYTE( "msa03302.g3", 0x040001, 0x020000, CRC(b9ffa995) SHA1(9254a181e3e1e653cd44bfbcf965b4a8e8eee4de) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwi )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03303.g0", 0x000000, 0x020000, CRC(296cf62c) SHA1(6d6586c5226cc3f8c5d286e5cd5faeda734fc463) )
	ROM_LOAD16_BYTE( "msa03303.g1", 0x000001, 0x020000, CRC(84d1375d) SHA1(fd38313851c1540f7f75eac2b3657d98f42c870d) )
	ROM_LOAD16_BYTE( "msa03303.g2", 0x040000, 0x020000, CRC(35220b0c) SHA1(cb2452e3bc787ef3674e344e4c45747274f18c77) )
	ROM_LOAD16_BYTE( "msa03303.g3", 0x040001, 0x020000, CRC(b9ffa995) SHA1(9254a181e3e1e653cd44bfbcf965b4a8e8eee4de) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03317.g0", 0x000000, 0x020000, CRC(d9b61f80) SHA1(7adec61b6d44cb6a1491b3a4c0c17e88d6d180ed) )
	ROM_LOAD16_BYTE( "msa03317.g1", 0x000001, 0x020000, CRC(f923b7a1) SHA1(baf195aaeeb01495907bc68dedf1c30f3cac8202) )
	ROM_LOAD16_BYTE( "msa03317.g2", 0x040000, 0x020000, CRC(8e65e6bf) SHA1(c59bcfc76548ce6cc6525e8863664266f25dad60) )
	ROM_LOAD16_BYTE( "msa03317.g3", 0x040001, 0x020000, CRC(ea95bb16) SHA1(13c0e332ae3596abcbbc6d092d55826112241dd6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03318.g0", 0x000000, 0x020000, CRC(739d61f8) SHA1(496435641190a8f0b96c44866ebd14d4fc584c3b) )
	ROM_LOAD16_BYTE( "msa03318.g1", 0x000001, 0x020000, CRC(aa951c0c) SHA1(1fa43c45358f9b28080d35e7c43c7db161287901) )
	ROM_LOAD16_BYTE( "msa03318.g2", 0x040000, 0x020000, CRC(8e65e6bf) SHA1(c59bcfc76548ce6cc6525e8863664266f25dad60) )
	ROM_LOAD16_BYTE( "msa03318.g3", 0x040001, 0x020000, CRC(ea95bb16) SHA1(13c0e332ae3596abcbbc6d092d55826112241dd6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03321.g0", 0x000000, 0x020000, CRC(33552877) SHA1(1af3dd31f9722ddbdb95106febdc6e9e02f6d9db) )
	ROM_LOAD16_BYTE( "msa03321.g1", 0x000001, 0x020000, CRC(4f30ae1a) SHA1(40f87565abf7f38f9280a349f1058812d13e39a0) )
	ROM_LOAD16_BYTE( "msa03321.g2", 0x040000, 0x020000, CRC(f0788216) SHA1(592d643273a8e1bdc0d5167205548bac9dffd114) )
	ROM_LOAD16_BYTE( "msa03321.g3", 0x040001, 0x020000, CRC(da750393) SHA1(68fa8d9f9e62c77d10607e3a648e14cafc52759d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spcbwm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03322.g0", 0x000000, 0x020000, CRC(24e75d21) SHA1(6e2937478ce78c71ef735b69973382b9273c6797) )
	ROM_LOAD16_BYTE( "msa03322.g1", 0x000001, 0x020000, CRC(5d228bfe) SHA1(a2a8473941b7684826d0cdbf97f0cd1693531979) )
	ROM_LOAD16_BYTE( "msa03322.g2", 0x040000, 0x020000, CRC(f0788216) SHA1(592d643273a8e1bdc0d5167205548bac9dffd114) )
	ROM_LOAD16_BYTE( "msa03322.g3", 0x040001, 0x020000, CRC(da750393) SHA1(68fa8d9f9e62c77d10607e3a648e14cafc52759d) )
	MISSING_SOUND
ROM_END




ROM_START( ep_spec )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02193.g0", 0x000000, 0x020000, CRC(3e3c0960) SHA1(07c73440019dfc3793eb9370c1fd171e3f1d1276) )
	ROM_LOAD16_BYTE( "msa02193.g1", 0x000001, 0x020000, CRC(2488953e) SHA1(597a05325a6d94d61d7a9e75e227b71e237f9fe2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_speca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02194.g0", 0x000000, 0x020000, CRC(e2edf470) SHA1(2a42017cc7838f8ec190bbbf01fe75e06c0cd53c) )
	ROM_LOAD16_BYTE( "msa02194.g1", 0x000001, 0x020000, CRC(f52b23fc) SHA1(eccd661510cd8ba058b35eb77b2ad5c8e7323097) )
	MISSING_SOUND
ROM_END

ROM_START( ep_specb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "spectre.g0", 0x000000, 0x020000, CRC(04735ad3) SHA1(ed708399a7a1b9eb1a7d5335901651e0916534fd) )
	ROM_LOAD16_BYTE( "spectre.g1", 0x000001, 0x020000, CRC(8e01ac8b) SHA1(51348d6dd5e0dd72661f701d9afd134b119b9054) )
	MISSING_SOUND
ROM_END




ROM_START( ep_spntn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gst12_0.bin", 0x000000, 0x020000, CRC(b5cfa6cb) SHA1(491ae3b406221c720f2007c191194751a7f47d77) )
	ROM_LOAD16_BYTE( "gst12_1.bin", 0x000001, 0x020000, CRC(3f7dd42d) SHA1(f481ac2a4927a79c9e12131a3965bcd5469fcc83) )
	ROM_LOAD16_BYTE( "gst12_2.bin", 0x040000, 0x020000, CRC(6c5ac607) SHA1(16388a55681de75b0ef5f86faa9888d8270f6c2b) )
	ROM_LOAD16_BYTE( "gst12_3.bin", 0x040001, 0x020000, CRC(2059763a) SHA1(9184963238f51edd2a7071438dbc5a0f3d7a8e8e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spntna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gst13_0.bin", 0x000000, 0x020000, CRC(b5cfa6cb) SHA1(491ae3b406221c720f2007c191194751a7f47d77) )
	ROM_LOAD16_BYTE( "gst13_1.bin", 0x000001, 0x020000, CRC(d4e38015) SHA1(83c4e46b798442c2da73ef44b015e21c6052a775) )
	ROM_LOAD16_BYTE( "gst13_2.bin", 0x040000, 0x020000, CRC(6c5ac607) SHA1(16388a55681de75b0ef5f86faa9888d8270f6c2b) )
	ROM_LOAD16_BYTE( "gst13_3.bin", 0x040001, 0x020000, CRC(2059763a) SHA1(9184963238f51edd2a7071438dbc5a0f3d7a8e8e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spntnb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gst14_0.bin", 0x000000, 0x020000, CRC(709e17a1) SHA1(a8e2cccf8f514b80afe8256fcb98b634e241da24) )
	ROM_LOAD16_BYTE( "gst14_1.bin", 0x000001, 0x020000, CRC(e85bb405) SHA1(bdf4faaf35f51d2a44dbf3172971d7cea990c0df) )
	ROM_LOAD16_BYTE( "gst14_2.bin", 0x040000, 0x020000, CRC(ef43aa53) SHA1(ab4ab8ce78d8c2f8ea1700bfdece22e8d4f70a43) )
	ROM_LOAD16_BYTE( "gst14_3.bin", 0x040001, 0x020000, CRC(b00806ab) SHA1(3a55f0f237eecda244b17d34d93ad2463ab05c4f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_spntnc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gst15_0.bin", 0x000000, 0x020000, CRC(709e17a1) SHA1(a8e2cccf8f514b80afe8256fcb98b634e241da24) )
	ROM_LOAD16_BYTE( "gst15_1.bin", 0x000001, 0x020000, CRC(03c5e03d) SHA1(0427b7a06c841b5f85a383a6585422f54678226a) )
	ROM_LOAD16_BYTE( "gst15_2.bin", 0x040000, 0x020000, CRC(ef43aa53) SHA1(ab4ab8ce78d8c2f8ea1700bfdece22e8d4f70a43) )
	ROM_LOAD16_BYTE( "gst15_3.bin", 0x040001, 0x020000, CRC(b00806ab) SHA1(3a55f0f237eecda244b17d34d93ad2463ab05c4f) )
	MISSING_SOUND
ROM_END

#define EP_SPIN_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "spinonit.sn0", 0x000000, 0x080000, CRC(ad92e501) SHA1(adea1d2751c0fbf4637d5eb2b1003995c829413d) ) \
	ROM_LOAD( "spinonit.sn1", 0x080000, 0x080000, CRC(e052ff0c) SHA1(afa2b2f4d51c95352d6fd1d69af87fef07c2f039) )

ROM_START( ep_spin )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nsi14.g0", 0x000000, 0x020000, CRC(395c722f) SHA1(7a1a12b1927fb87b101d0e3ad54e664f34a7f540) )
	ROM_LOAD16_BYTE( "i00nsi14.g1", 0x000001, 0x020000, CRC(ef4fe4b4) SHA1(3e3d73a0b2e736823575f2d3f6148a7458d14baa) )
	EP_SPIN_SOUND
ROM_END

ROM_START( ep_spina )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nsi16.g0", 0x000000, 0x020000, CRC(395c722f) SHA1(7a1a12b1927fb87b101d0e3ad54e664f34a7f540) )
	ROM_LOAD16_BYTE( "i00nsi16.g1", 0x000001, 0x020000, CRC(e3024a85) SHA1(6fc901b8e41712dcde95b8bbb9abafa2cef23fe8) )
	EP_SPIN_SOUND
ROM_END

ROM_START( ep_spinb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nsi18.g0", 0x000000, 0x020000, CRC(9faa2a48) SHA1(da903bbcbc3f62691b321853ac632b0b68be5a8c) )
	ROM_LOAD16_BYTE( "i00nsi18.g1", 0x000001, 0x020000, CRC(bff7dad4) SHA1(c1f7b3cc40bc95718341a3110ab7351aa0008ec4) )
	ROM_LOAD16_BYTE( "i00nsi18.g2", 0x040000, 0x020000, CRC(206f42f3) SHA1(4b227c5781b67de8a8e2dee52fcf91acda69d296) )
	ROM_LOAD16_BYTE( "i00nsi18.g3", 0x040001, 0x020000, CRC(1347d77b) SHA1(882a2bc8b092d7ea3a43abeb05706af7a6e7a620) )
	EP_SPIN_SOUND
ROM_END

ROM_START( ep_spinc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00psi15.g0", 0x000000, 0x020000, CRC(395c722f) SHA1(7a1a12b1927fb87b101d0e3ad54e664f34a7f540) )
	ROM_LOAD16_BYTE( "i00psi15.g1", 0x000001, 0x020000, CRC(04d1b08c) SHA1(63a2c10c35e5a7a148e92489df2bf316e1696ac2) )
	EP_SPIN_SOUND
ROM_END

ROM_START( ep_spind )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00psi17.g0", 0x000000, 0x020000, CRC(395c722f) SHA1(7a1a12b1927fb87b101d0e3ad54e664f34a7f540) )
	ROM_LOAD16_BYTE( "i00psi17.g1", 0x000001, 0x020000, CRC(089c1ebd) SHA1(fb71b27ea1e53880f8fe9ed78048158284cbc36b) )
	EP_SPIN_SOUND
ROM_END

ROM_START( ep_spine )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00psi19.g0", 0x000000, 0x020000, CRC(9faa2a48) SHA1(da903bbcbc3f62691b321853ac632b0b68be5a8c) )
	ROM_LOAD16_BYTE( "i00psi19.g1", 0x000001, 0x020000, CRC(54698eec) SHA1(79d222c4e0960c53ca8246dc0de2ad1158eb99a7) )
	ROM_LOAD16_BYTE( "i00psi19.g2", 0x040000, 0x020000, CRC(206f42f3) SHA1(4b227c5781b67de8a8e2dee52fcf91acda69d296) )
	ROM_LOAD16_BYTE( "i00psi19.g3", 0x040001, 0x020000, CRC(1347d77b) SHA1(882a2bc8b092d7ea3a43abeb05706af7a6e7a620) )
	EP_SPIN_SOUND
ROM_END




ROM_START( ep_step )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000076.g0", 0x000000, 0x020000, CRC(7c071d7e) SHA1(7c87a65e5493c040cea10e76ce6d59544bbf19ce) )
	ROM_LOAD16_BYTE( "sa000076.g1", 0x000001, 0x020000, CRC(28c47935) SHA1(8aae9db209b732a42e5882c2029a15492d2e4889) )
	MISSING_SOUND
ROM_END

ROM_START( ep_stepa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000078.g0", 0x000000, 0x020000, CRC(6d92b20b) SHA1(0041610bda6de9862d0b0278d7418dd095303b69) )
	ROM_LOAD16_BYTE( "sa000078.g1", 0x000001, 0x020000, CRC(ed0922dd) SHA1(711ea89a90e25f01fc8c29b4662afd88d053142e) )
	MISSING_SOUND
ROM_END




ROM_START( ep_strat )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gsg01_0.bin", 0x000000, 0x020000, CRC(35ded06f) SHA1(24cb059eb5342d155b5f2c614bccdf8889e098f6) )
	ROM_LOAD16_BYTE( "gsg01_1.bin", 0x000001, 0x020000, CRC(2eb921f1) SHA1(4d6d428a6ecf855b2600c63bc78ac0a7bbba6afb) )
	ROM_LOAD16_BYTE( "gsg01_2.bin", 0x040000, 0x020000, CRC(24425726) SHA1(5477e14dfbfb016aef00670e3daec2417d375496) )
	ROM_LOAD16_BYTE( "gsg01_3.bin", 0x040001, 0x020000, CRC(47cfe457) SHA1(31fe7781eec0594e4b488cb3c659cd4bd14089b9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_strata )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gsg02_0.bin", 0x000000, 0x020000, CRC(35ded06f) SHA1(24cb059eb5342d155b5f2c614bccdf8889e098f6) )
	ROM_LOAD16_BYTE( "gsg02_1.bin", 0x000001, 0x020000, CRC(c52775c9) SHA1(973224ee2ac5c01ee3aa5d353c2a83985b613d60) )
	ROM_LOAD16_BYTE( "gsg02_2.bin", 0x040000, 0x020000, CRC(24425726) SHA1(5477e14dfbfb016aef00670e3daec2417d375496) )
	ROM_LOAD16_BYTE( "gsg02_3.bin", 0x040001, 0x020000, CRC(47cfe457) SHA1(31fe7781eec0594e4b488cb3c659cd4bd14089b9) )
	MISSING_SOUND
ROM_END


#define EP_SUBB_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "subbuteo.s0", 0x000000, 0x080000, CRC(3140ed84) SHA1(418fd6a1fd06c59f524f03116f0d284ebf481f58) ) \
	ROM_LOAD( "subbuteo.s1", 0x080000, 0x080000, CRC(e87da3e7) SHA1(9443b8304ad75612186a97f0dafcdd698bf43c04) )

ROM_START( ep_subb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gsu54_0.bin", 0x000000, 0x020000, CRC(94131da2) SHA1(70c55abb5ffdc6a235d2775de1d5b0723bddb886) )
	ROM_LOAD16_BYTE( "gsu54_1.bin", 0x000001, 0x020000, CRC(8b6aa3b2) SHA1(d672f626b72d2d38d65983caec3ab9d31d66fcf0) )
	ROM_LOAD16_BYTE( "gsu54_2.bin", 0x040000, 0x020000, CRC(68331ae7) SHA1(59e563fd8be29bd1032d0fac6079590e941441fe) )
	ROM_LOAD16_BYTE( "gsu54_3.bin", 0x040001, 0x020000, CRC(ccf09a85) SHA1(bb6ef2cdacabf88b018f064cfa5a9dbcda4a556e) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gsu55_0.bin", 0x000000, 0x020000, CRC(94131da2) SHA1(70c55abb5ffdc6a235d2775de1d5b0723bddb886) )
	ROM_LOAD16_BYTE( "gsu55_1.bin", 0x000001, 0x020000, CRC(60f4f78a) SHA1(13b78140fbda4e92dda2916557efb23391a838a9) )
	ROM_LOAD16_BYTE( "gsu55_2.bin", 0x040000, 0x020000, CRC(68331ae7) SHA1(59e563fd8be29bd1032d0fac6079590e941441fe) )
	ROM_LOAD16_BYTE( "gsu55_3.bin", 0x040001, 0x020000, CRC(ccf09a85) SHA1(bb6ef2cdacabf88b018f064cfa5a9dbcda4a556e) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02248.g0", 0x000000, 0x020000, CRC(a1c29aa3) SHA1(e7d198b2a36261f2f5b53c3776a9c1f6ff275182) )
	ROM_LOAD16_BYTE( "msa02248.g1", 0x000001, 0x020000, CRC(1dbf0221) SHA1(121ddfab1530b49453a8856cd5c9b173e8c41b3a) )
	ROM_LOAD16_BYTE( "msa02248.g2", 0x040000, 0x020000, CRC(cd5954c0) SHA1(edc29e0207686c0f4cdcd8bc04182ef34e79ceb1) )
	ROM_LOAD16_BYTE( "msa02248.g3", 0x040001, 0x020000, CRC(af18ea8e) SHA1(7a542b811f52d5a8a6000805289d0e7b8b7bd8b7) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02249.g0", 0x000000, 0x020000, CRC(a1c29aa3) SHA1(e7d198b2a36261f2f5b53c3776a9c1f6ff275182) )
	ROM_LOAD16_BYTE( "msa02249.g1", 0x000001, 0x020000, CRC(f6215619) SHA1(c16650aff20bfc7c648922569f6d3fabb7912f50) )
	ROM_LOAD16_BYTE( "msa02249.g2", 0x040000, 0x020000, CRC(cd5954c0) SHA1(edc29e0207686c0f4cdcd8bc04182ef34e79ceb1) )
	ROM_LOAD16_BYTE( "msa02249.g3", 0x040001, 0x020000, CRC(af18ea8e) SHA1(7a542b811f52d5a8a6000805289d0e7b8b7bd8b7) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03300.g0", 0x000000, 0x020000, CRC(e6db7ccb) SHA1(6e9cbe740f19e4ee4674bbf889eb493025fbb366) )
	ROM_LOAD16_BYTE( "msa03300.g1", 0x000001, 0x020000, CRC(fe029b57) SHA1(2fae55c0e340ba0675b085ddf08a61f2a8c201dc) )
	ROM_LOAD16_BYTE( "msa03300.g2", 0x040000, 0x020000, CRC(d249735e) SHA1(3e6ffb812df6908692950e51a724f1e8265e0dd6) )
	ROM_LOAD16_BYTE( "msa03300.g3", 0x040001, 0x020000, CRC(f96306cb) SHA1(0a2fce12471fb2162dfac1f3c8f5c4bfdca365b4) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03301.g0", 0x000000, 0x020000, CRC(e6db7ccb) SHA1(6e9cbe740f19e4ee4674bbf889eb493025fbb366) )
	ROM_LOAD16_BYTE( "msa03301.g1", 0x000001, 0x020000, CRC(159ccf6f) SHA1(f41c13b2b25eac0b9cfa745595ab8b68a969c444) )
	ROM_LOAD16_BYTE( "msa03301.g2", 0x040000, 0x020000, CRC(d249735e) SHA1(3e6ffb812df6908692950e51a724f1e8265e0dd6) )
	ROM_LOAD16_BYTE( "msa03301.g3", 0x040001, 0x020000, CRC(f96306cb) SHA1(0a2fce12471fb2162dfac1f3c8f5c4bfdca365b4) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03308.g0.bin", 0x000000, 0x020000, CRC(a9386ca4) SHA1(f41a4c68925d2770a47213f0689f822012671353) )
	ROM_LOAD16_BYTE( "msa03308.g1.bin", 0x000001, 0x020000, CRC(c8464e40) SHA1(4cee501b6663c42fee4fde22f9b263b728ef73b2) )
	ROM_LOAD16_BYTE( "msa03308.g2.bin", 0x040000, 0x020000, CRC(41d9988b) SHA1(581c061a7420b902839567a27013880cdb3ea689) )
	ROM_LOAD16_BYTE( "msa03308.g3.bin", 0x040001, 0x020000, CRC(bcdc828a) SHA1(7e4c0be0fd05bc0257e672089ed52720108beab4) )
	EP_SUBB_SOUND
ROM_END

ROM_START( ep_subbg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa03309.g0.bin", 0x000000, 0x020000, CRC(a9386ca4) SHA1(f41a4c68925d2770a47213f0689f822012671353) )
	ROM_LOAD16_BYTE( "msa03309.g1.bin", 0x000001, 0x020000, CRC(23d81a78) SHA1(4be75a6989953851eeb5d23814b4711c67fad72c) )
	ROM_LOAD16_BYTE( "msa03309.g2.bin", 0x040000, 0x020000, CRC(41d9988b) SHA1(581c061a7420b902839567a27013880cdb3ea689) )
	ROM_LOAD16_BYTE( "msa03309.g3.bin", 0x040001, 0x020000, CRC(bcdc828a) SHA1(7e4c0be0fd05bc0257e672089ed52720108beab4) )
	EP_SUBB_SOUND
ROM_END




ROM_START( ep_snw )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bsnwb03.g0", 0x000000, 0x020000, CRC(5ab82280) SHA1(0b01fd83f09b06fc9d428fd90216e753874678c1) )
	ROM_LOAD16_BYTE( "bsnwb03.g1", 0x000001, 0x020000, CRC(bc34a937) SHA1(4b08de7a924395b2f7508ee9c1a1baa8f760f90a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_snwa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bsnwb04.g0", 0x000000, 0x020000, CRC(133724a1) SHA1(03ba0bf62ccbb692e22cd0d1e8b8b9d7322122c1) )
	ROM_LOAD16_BYTE( "bsnwb04.g1", 0x000001, 0x020000, CRC(cfc561fb) SHA1(07bfb449eca85e79deed8a326cdd3bdfaf19d3d8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_snwb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "csnw011.g0", 0x000000, 0x020000, CRC(92396091) SHA1(74911756429e519942aef872e29ce6b73acf4baf) )
	ROM_LOAD16_BYTE( "csnw011.g1", 0x000001, 0x020000, CRC(9be24de9) SHA1(d392430c195f4e2624a728c8ac958e47d75bc165) )
	MISSING_SOUND
ROM_END

ROM_START( ep_snwc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "csnw012.g0", 0x000000, 0x020000, CRC(c55eff74) SHA1(4a89d0471aebdd7a28428f43f37551d8cf05ab8a) )
	ROM_LOAD16_BYTE( "csnw012.g1", 0x000001, 0x020000, CRC(b00e9a6d) SHA1(6886e6295d0d2d78ccb4748aa9c63c23a9614f15) )
	MISSING_SOUND
ROM_END

ROM_START( ep_snwd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "snw.g0", 0x00000, 0x020000, CRC(29a7f942) SHA1(766aa01a28eda1f4201c1a891010684758401216) )
	ROM_LOAD16_BYTE( "snw.g1", 0x00001, 0x020000, CRC(549a0181) SHA1(41064c19fdf4ac270a4a904a32878ae541c9d593) )
	MISSING_SOUND
ROM_END




ROM_START( ep_survi )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02154.g0", 0x000000, 0x020000, CRC(bafed449) SHA1(c165e770f7de670fc6875bb8374df098fe9e3cab) )
	ROM_LOAD16_BYTE( "msa02154.g1", 0x000001, 0x020000, CRC(efa1e914) SHA1(bdd262557bb31976652085af7fa4eab3abe7976f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_survia )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02155.g0", 0x000000, 0x020000, CRC(1115a721) SHA1(468fdc58d187d33db7b6ea949daa5d70e2a54e38) )
	ROM_LOAD16_BYTE( "msa02155.g1", 0x000001, 0x020000, CRC(2900e95b) SHA1(fa06c1141a76b4c8b19842fad7659a59c5f32c06) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tak5 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tke5009.g0", 0x000000, 0x020000, CRC(baf2816b) SHA1(f8dc10ddf370a6452088c1b74a036a23149680f8) )
	ROM_LOAD16_BYTE( "tke5009.g1", 0x000001, 0x020000, CRC(6b5d2412) SHA1(c4c6d536403b87f7bf47503dde8796529f08c4d2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tak5a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tke5010.g0", 0x000000, 0x020000, CRC(bb6b5923) SHA1(36467cbe709e09cd14127fa0448417d551f85472) )
	ROM_LOAD16_BYTE( "tke5010.g1", 0x000001, 0x020000, CRC(8be758eb) SHA1(7018d6ebb9e29c789b64c21e0e487de4c26d6c46) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tits )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa01010.g0", 0x000000, 0x020000, CRC(632be638) SHA1(f8bbf427d0cd144200dd13a4ba36c71617a34239) )
	ROM_LOAD16_BYTE( "msa01010.g1", 0x000001, 0x020000, CRC(f9cc5242) SHA1(67987b48ebaa09732e49f5f807d16e9155744103) )
	MISSING_SOUND
ROM_END

ROM_START( ep_titsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc01009.g0", 0x000000, 0x020000, CRC(93eb5835) SHA1(0607d4c8658be3679c95459cbd3607f178e0613d) )
	ROM_LOAD16_BYTE( "msc01009.g1", 0x000001, 0x020000, CRC(7b150853) SHA1(c50455b092a566420685baaa64a935289734480f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_titsb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ttl_shot.g0", 0x000000, 0x020000, CRC(f2fcd2c7) SHA1(89fe906dae0a85ff548ed97c449a5f22df30c605) )
	ROM_LOAD16_BYTE( "ttl_shot.g1", 0x000001, 0x020000, CRC(3474b2be) SHA1(fa3173b45c967dd18712b0ea9bda264a47fa8e70) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tortr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000086.g0", 0x000000, 0x020000, CRC(fb809f20) SHA1(5d798e8724f11ebce7337e53521b6b0f9beef116) )
	ROM_LOAD16_BYTE( "sa000086.g1", 0x000001, 0x020000, CRC(46211b12) SHA1(1fbba9c6ada7db8c41de8c2e78f77ca3c6158f3f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tortra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sa000087.g0", 0x000000, 0x020000, CRC(506bec48) SHA1(d9f8308a9292bcabd2842ebcda6808e94a6b5ea7) )
	ROM_LOAD16_BYTE( "sa000087.g1", 0x000001, 0x020000, CRC(1254d1f9) SHA1(f997ea0d9e96aea810a4f203768ecb409bacab80) )
	MISSING_SOUND
ROM_END




ROM_START( ep_trail )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ntb32.g0", 0x000000, 0x020000, CRC(9bfadfd9) SHA1(e782768f87faca4fab80bde925bfe4345dcb0513) )
	ROM_LOAD16_BYTE( "i99ntb32.g1", 0x000001, 0x020000, CRC(1d1a3da6) SHA1(9f453b0a119d0eec733f5c5775f93d5306568bd2) )
	MISSING_SOUND
ROM_END

ROM_START( ep_traila )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i99ptb33.g0", 0x000000, 0x020000, CRC(9bfadfd9) SHA1(e782768f87faca4fab80bde925bfe4345dcb0513) )
	ROM_LOAD16_BYTE( "i99ptb33.g1", 0x000001, 0x020000, CRC(f684699e) SHA1(2d1552acf27d16930379a91842fe97d73c5f7bed) )
	MISSING_SOUND
ROM_END

ROM_START( ep_trailb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tblazeg0.bin", 0x000000, 0x020000, CRC(e99555e7) SHA1(c9da6a483ba1ee69f9f899d3809b1f3f9f00ab4a) )
	ROM_LOAD16_BYTE( "tblazeg1.bin", 0x000001, 0x020000, CRC(99620ac9) SHA1(cb8e7399393bcef134cd3f052048ead787bf0e9f) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tree )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tram03.p0", 0x000000, 0x020000, CRC(37126cd6) SHA1(4b745925ea9b051aacbb0670270df292d0ebc006) )
	ROM_LOAD16_BYTE( "tram03.p1", 0x000001, 0x020000, CRC(a351e80d) SHA1(ecad15c4faf47a47bd2ac8b2581f6ff7fe103ab1) )
	ROM_LOAD16_BYTE( "tram03.p2", 0x040000, 0x020000, CRC(a807a84e) SHA1(d39ea0e2b2d7105597fdb5c6e661afd062dbc6f6) )
	ROM_LOAD16_BYTE( "tram03.p3", 0x040001, 0x020000, CRC(fdffa62c) SHA1(f92107b09a1c52d92ba62193a443aa1456d85a92) )
	MISSING_SOUND
ROM_END

ROM_START( ep_treea )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tram03p.p0", 0x000000, 0x020000, CRC(37126cd6) SHA1(4b745925ea9b051aacbb0670270df292d0ebc006) )
	ROM_LOAD16_BYTE( "tram03p.p1", 0x000001, 0x020000, CRC(bcc21a76) SHA1(7f81645edb498f47f7636561b7db2b5fe6010cdd) )
	ROM_LOAD16_BYTE( "tram03p.p2", 0x040000, 0x020000, CRC(a807a84e) SHA1(d39ea0e2b2d7105597fdb5c6e661afd062dbc6f6) )
	ROM_LOAD16_BYTE( "tram03p.p3", 0x040001, 0x020000, CRC(fdffa62c) SHA1(f92107b09a1c52d92ba62193a443aa1456d85a92) )
	MISSING_SOUND
ROM_END



ROM_START( ep_trics )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tric003.g0", 0x000000, 0x020000, CRC(15633353) SHA1(e44749a244eb99ed11d919f2505698a25776d716) )
	ROM_LOAD16_BYTE( "tric003.g1", 0x000001, 0x020000, CRC(1b8448b8) SHA1(8285825429a51ba3315a8c6ff3f633e66f1f5572) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tricsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tric004.g0", 0x000000, 0x020000, CRC(283a288b) SHA1(c68942eea31ba1e3407aa9010895485bf4f843e3) )
	ROM_LOAD16_BYTE( "tric004.g1", 0x000001, 0x020000, CRC(5f0461a4) SHA1(59ea16ce89d09a9773e6d3fe558722c868238677) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02162.g0", 0x000000, 0x020000, CRC(249d6561) SHA1(7ea4303b0e2f6902844f7a1ad2d5e1ecbd1daa19) )
	ROM_LOAD16_BYTE( "msa02162.g1", 0x000001, 0x020000, CRC(bdead13e) SHA1(e00bb6c26b3920171b04b8f5c218e3ef5f9c6f10) )
	ROM_LOAD16_BYTE( "msa02162.g2", 0x040000, 0x020000, CRC(f000b66d) SHA1(f4d5662c1ad0f8d43f2836b0655a3d9c7b0d4e63) )
	ROM_LOAD16_BYTE( "msa02162.g3", 0x040001, 0x020000, CRC(61d070bf) SHA1(4a9e38acf5e16a5b225c53e201a6cb7ee61fbca6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tpa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02163.g0", 0x000000, 0x020000, CRC(8f761609) SHA1(fa0e986572042d036f688319755f72650d8937b5) )
	ROM_LOAD16_BYTE( "msa02163.g1", 0x000001, 0x020000, CRC(4e098f03) SHA1(f031560549346e8091ed42e756c1c5c5d3d787e0) )
	ROM_LOAD16_BYTE( "msa02163.g2", 0x040000, 0x020000, CRC(f000b66d) SHA1(f4d5662c1ad0f8d43f2836b0655a3d9c7b0d4e63) )
	ROM_LOAD16_BYTE( "msa02163.g3", 0x040001, 0x020000, CRC(61d070bf) SHA1(4a9e38acf5e16a5b225c53e201a6cb7ee61fbca6) )
	MISSING_SOUND
ROM_END


ROM_START( ep_tpb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "triv_pur.g0", 0x000000, 0x020000, CRC(3197e48e) SHA1(ae4328df20f27e0edf35ed5d872ad452c23e0094) )
	ROM_LOAD16_BYTE( "triv_pur.g1", 0x000001, 0x020000, CRC(7d63c3bd) SHA1(6625361a549baed733e7d0d2b1eb87e417a73fc6) )
	ROM_LOAD16_BYTE( "triv_pur.g2", 0x040000, 0x020000, CRC(aca47c0c) SHA1(c23926b98439cac5d7982a9ca7767b9e79ba096e) )
	ROM_LOAD16_BYTE( "triv_pur.g3", 0x040001, 0x020000, CRC(e2a446cf) SHA1(9327f1fe39b8d012928892d1ebec9a6a1405df2a) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tp2 )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02164.g0", 0x000000, 0x020000, CRC(59948da9) SHA1(84245e70bc26bec5c01a64737b5ab218a12dad68) )
	ROM_LOAD16_BYTE( "msa02164.g1", 0x000001, 0x020000, CRC(b0ca24f7) SHA1(0d2c09b44be2a5149aac9f2d4b64f8d7185ee336) )
	ROM_LOAD16_BYTE( "msa02164.g2", 0x040000, 0x020000, CRC(4b3e7d9b) SHA1(54600d0ce6ba549df6f90a00a0b1786fe83c3830) )
	ROM_LOAD16_BYTE( "msa02164.g3", 0x040001, 0x020000, CRC(6a23ec5a) SHA1(02249969d8eddcc4f4f7343cc95aa6c30a70855a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tp2a )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02165.g0", 0x000000, 0x020000, CRC(a37cd530) SHA1(d12576fa84a8d7326d99dbbfcad577995488a33d) )
	ROM_LOAD16_BYTE( "msa02165.g1", 0x000001, 0x020000, CRC(fbc2738e) SHA1(436e84745db085f6625d0702c8090778c6129273) )
	ROM_LOAD16_BYTE( "msa02165.g2", 0x040000, 0x020000, CRC(4b3e7d9b) SHA1(54600d0ce6ba549df6f90a00a0b1786fe83c3830) )
	ROM_LOAD16_BYTE( "msa02165.g3", 0x040001, 0x020000, CRC(6a23ec5a) SHA1(02249969d8eddcc4f4f7343cc95aa6c30a70855a) )
	MISSING_SOUND
ROM_END




ROM_START( ep_tutcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02040.g0", 0x000000, 0x020000, CRC(cbc34c7e) SHA1(a43ee8a288b3cd7c75f5732f4de46cef49d1f543) )
	ROM_LOAD16_BYTE( "msc02040.g1", 0x000001, 0x020000, CRC(fd1cd600) SHA1(7f4949fa5e0ac3424718f181370ef219aee9be35) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tutcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02041.g0", 0x000000, 0x020000, CRC(c4ece490) SHA1(2200192d649f9b8e16a0e9ade0c6c419dc0877cf) )
	ROM_LOAD16_BYTE( "msc02041.g1", 0x000001, 0x020000, CRC(c22b0cdf) SHA1(6689e495a04c49538ef979ae17465d2154020ea3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_tutclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "tut_karm.g0", 0x000000, 0x020000, CRC(34f9017d) SHA1(76187027a379dd35f6739b30cbf00916f4304b45) )
	ROM_LOAD16_BYTE( "tut_karm.g1", 0x000001, 0x020000, CRC(799dca4f) SHA1(92093e336c53489029d5b78c025dd8c248a1cf2b) )
	MISSING_SOUND
ROM_END




ROM_START( ep_21clb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02054.g0", 0x000000, 0x020000, CRC(f04d8bf5) SHA1(cf519ee3bfc5e59cd8464284ed98c9fff4381cc7) )
	ROM_LOAD16_BYTE( "msc02054.g1", 0x000001, 0x020000, CRC(f0baa69a) SHA1(a1f179b2f8a7ae33b7bf9bddc60220fba2bdcd90) )
	MISSING_SOUND
ROM_END

ROM_START( ep_21clba )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc02055.g0", 0x000000, 0x020000, CRC(76163c1e) SHA1(59ff48074d413b07237a72f2a1c5a1e0890bc1d5) )
	ROM_LOAD16_BYTE( "msc02055.g1", 0x000001, 0x020000, CRC(d6eac9a3) SHA1(2e58eeec566fdaa414a2232ac1fc2c026130a838) )
	MISSING_SOUND
ROM_END



ROM_START( ep_vipjv )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jive11.p0", 0x000000, 0x020000, CRC(8f0b3a2e) SHA1(ffd9fbd5e16d4738a6a49c5d8e49fd8f0644a6f1) )
	ROM_LOAD16_BYTE( "jive11.p1", 0x000001, 0x020000, CRC(9c6b64f2) SHA1(221b41f8301e76ed991883aa11b885c460c8dda3) )
	ROM_LOAD16_BYTE( "jive11.p2", 0x040000, 0x020000, CRC(c569d069) SHA1(6f011c07bd7242153c25fda5db8d2df3efb31bbe) )
	ROM_LOAD16_BYTE( "jive11.p3", 0x040001, 0x020000, CRC(29ad4a6b) SHA1(79bcb4856178b15900aa977b454fb7aa9b05c47c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_vipjva )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jive11p.p0", 0x000000, 0x020000, CRC(8f0b3a2e) SHA1(ffd9fbd5e16d4738a6a49c5d8e49fd8f0644a6f1) )
	ROM_LOAD16_BYTE( "jive11p.p1", 0x000001, 0x020000, CRC(83f89689) SHA1(d3ea0fb9b413ac4cdeb1621c97eb8b17e1b779b9) )
	ROM_LOAD16_BYTE( "jive11p.p2", 0x040000, 0x020000, CRC(c569d069) SHA1(6f011c07bd7242153c25fda5db8d2df3efb31bbe) )
	ROM_LOAD16_BYTE( "jive11p.p3", 0x040001, 0x020000, CRC(29ad4a6b) SHA1(79bcb4856178b15900aa977b454fb7aa9b05c47c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_vipjvb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jive22.p0", 0x000000, 0x020000, CRC(5b7340e2) SHA1(ab5b4b70e891ad244bf74b0dbc3a923a5b8b3ad6) )
	ROM_LOAD16_BYTE( "jive22.p1", 0x000001, 0x020000, CRC(654db698) SHA1(c88bbd91795d047bdad3490e682750302eec5c10) )
	ROM_LOAD16_BYTE( "jive22.p2", 0x040000, 0x020000, CRC(d069946b) SHA1(76053c5292c361753e9cd9fb8961e7825f79eea5) )
	ROM_LOAD16_BYTE( "jive22.p3", 0x040001, 0x020000, CRC(7ca29581) SHA1(ada2432e4a06799ed2a5c23d1d32a58fa69d8b4b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_vipjvc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jive22p.p0", 0x000000, 0x020000, CRC(5b7340e2) SHA1(ab5b4b70e891ad244bf74b0dbc3a923a5b8b3ad6) )
	ROM_LOAD16_BYTE( "jive22p.p1", 0x000001, 0x020000, CRC(7ade44e3) SHA1(254d7bad5aa19366f2111117cd2b6803d0a9be2e) )
	ROM_LOAD16_BYTE( "jive22p.p2", 0x040000, 0x020000, CRC(d069946b) SHA1(76053c5292c361753e9cd9fb8961e7825f79eea5) )
	ROM_LOAD16_BYTE( "jive22p.p3", 0x040001, 0x020000, CRC(7ca29581) SHA1(ada2432e4a06799ed2a5c23d1d32a58fa69d8b4b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_vipjvd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "viper_j.p0", 0x000000, 0x020000, CRC(fdad86da) SHA1(f0e0d9a780fbd4742810b5a3041b565a55811af0) )
	ROM_LOAD16_BYTE( "viper_j.p1", 0x000001, 0x020000, CRC(c1708c70) SHA1(ccde16e0be806d90f4a5a5f063cfd9f8b156bb3b) )
	ROM_LOAD16_BYTE( "viper_j.p2", 0x040000, 0x020000, CRC(20846c02) SHA1(ac11a4e4368f8cad5655b6903755616e5a46f8dd) )
	ROM_LOAD16_BYTE( "viper_j.p3", 0x040001, 0x020000, CRC(68b217e4) SHA1(29ec2cd517a354d25cd5ec306f597ce15ae64883) )
	MISSING_SOUND
ROM_END


ROM_START( ep_wildf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwf05_0.bin", 0x000000, 0x020000, CRC(40f34e50) SHA1(ff36e548e1a2c648792d67d1161d4998f50fc320) )
	ROM_LOAD16_BYTE( "gwf05_1.bin", 0x000001, 0x020000, CRC(0d3b44a0) SHA1(d4b9fbbbba7f93c8b19d6683369609781f7e34e5) )
	ROM_LOAD16_BYTE( "gwf05_2.bin", 0x040000, 0x020000, CRC(7392da1c) SHA1(d314c7787cad2c986af052f704fdbfe84a2354f5) )
	ROM_LOAD16_BYTE( "gwf05_3.bin", 0x040001, 0x020000, CRC(5353e8db) SHA1(c2c3899355b11a706aa8e2653b2df144768852af) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wildfa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwf06_0.bin", 0x000000, 0x020000, CRC(40f34e50) SHA1(ff36e548e1a2c648792d67d1161d4998f50fc320) )
	ROM_LOAD16_BYTE( "gwf06_1.bin", 0x000001, 0x020000, CRC(e6a51098) SHA1(b811ef4541811a8a5384df6179eab3aeaab73c99) )
	ROM_LOAD16_BYTE( "gwf06_2.bin", 0x040000, 0x020000, CRC(7392da1c) SHA1(d314c7787cad2c986af052f704fdbfe84a2354f5) )
	ROM_LOAD16_BYTE( "gwf06_3.bin", 0x040001, 0x020000, CRC(5353e8db) SHA1(c2c3899355b11a706aa8e2653b2df144768852af) )
	MISSING_SOUND
ROM_END

#define EP_XSPOT_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "x_marks.s0", 0x000000, 0x080000, CRC(2e6f4524) SHA1(8f3e0cf3bbdbb32c13b6bcaadee86ddb081e7e81) ) \
	ROM_LOAD( "x_marks.s1", 0x080000, 0x080000, CRC(5f0315d4) SHA1(d6ef3060f382fbc8012635196497b764ce635181) )
ROM_START( ep_xspot )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02195.g0", 0x000000, 0x020000, CRC(18cb2ae4) SHA1(610e7c6820c4b6ed45a50080e9c8624e653cd950) )
	ROM_LOAD16_BYTE( "msa02195.g1", 0x000001, 0x020000, CRC(bfc18c1d) SHA1(88ec0d320e46ee3d9c49fcd85d061b06a0d55d7b) )
	EP_XSPOT_SOUND
ROM_END

ROM_START( ep_xspota )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02196.g0", 0x000000, 0x020000, CRC(9b43629b) SHA1(ff42dce1fceb5fdcf850def3271f4f303c6bc20a) )
	ROM_LOAD16_BYTE( "msa02196.g1", 0x000001, 0x020000, CRC(a946d6b5) SHA1(27543882f416017b2a26eb103b5045a3c969b2d3) )
	EP_XSPOT_SOUND
ROM_END

ROM_START( ep_xspotb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "xmar15g0", 0x000000, 0x020000, CRC(991d02e1) SHA1(5685cf9a21a7e442a12f5b25492df96ae76cd150) )
	ROM_LOAD16_BYTE( "xmar15g1", 0x000001, 0x020000, CRC(48510488) SHA1(84c1189c92439e52f4afde21ea1ba0e0d62aebc7) )
	EP_XSPOT_SOUND
ROM_END






ROM_START( ep_braid ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bara01.p0", 0x000000, 0x100000, CRC(4af30c26) SHA1(26ef4dc53c2b909c87cf4edd45fe0c805564b5a4) )
	ROM_LOAD16_BYTE( "bara01.p1", 0x000001, 0x100000, CRC(6057140a) SHA1(6c0c426b9308f768e44fd346c23db051f7263e7a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_braida ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bara01p.p0", 0x000000, 0x100000, CRC(4af30c26) SHA1(26ef4dc53c2b909c87cf4edd45fe0c805564b5a4) )
	ROM_LOAD16_BYTE( "bara01p.p1", 0x000001, 0x100000, CRC(35bc79b1) SHA1(3d646a5a0f749f3d3075e24acf913d98e8160104) )
	MISSING_SOUND
ROM_END

ROM_START( ep_braidb ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bara05.p0", 0x000000, 0x100000, CRC(5b9d39d8) SHA1(0976afb8309b7f7908c65b29d2fa791fd1425c0a) )
	ROM_LOAD16_BYTE( "bara05.p1", 0x000001, 0x100000, CRC(66de37a3) SHA1(2ac5910d889f21fdb317c11e5390585691475dc3) )
	MISSING_SOUND
ROM_END

ROM_START( ep_braidc ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "bara05p.p0", 0x000000, 0x100000, CRC(5b9d39d8) SHA1(0976afb8309b7f7908c65b29d2fa791fd1425c0a) )
	ROM_LOAD16_BYTE( "bara05p.p1", 0x000001, 0x100000, CRC(33355a18) SHA1(27de5a4bb1b06ec75a72fb90f5d8afcbeceb4c48) )
	MISSING_SOUND
ROM_END

ROM_START( ep_braidd ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "barl05.p0", 0x000000, 0x100000, CRC(731d38c7) SHA1(6d66f415b37104d9d8d64ed081d9dcc5184362b5) )
	ROM_LOAD16_BYTE( "barl05.p1", 0x000001, 0x100000, CRC(61240a4e) SHA1(09de2eb21bc9810015c0b91105b0273e197a21ce) )
	MISSING_SOUND
ROM_END

ROM_START( ep_braide ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "barl05p.p0", 0x000000, 0x100000, CRC(731d38c7) SHA1(6d66f415b37104d9d8d64ed081d9dcc5184362b5) )
	ROM_LOAD16_BYTE( "barl05p.p1", 0x000001, 0x100000, CRC(34cf67f5) SHA1(72b5000e5821dc51cdce96ecefd4e35f2fdfc8c4) )
	MISSING_SOUND
ROM_END








ROM_START( ep_bvrcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv0-5t.p0", 0x000000, 0x020000, CRC(0ca82750) SHA1(a643e4f1d4833a63a7e8bbe55a1bba34f23543db) )
	ROM_LOAD16_BYTE( "cblv0-5t.p1", 0x000001, 0x020000, CRC(036c7f3c) SHA1(fcaf01c367a086c3a2d87d49e7317478981e1223) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv1-0n.p0", 0x000000, 0x020000, CRC(7d5868af) SHA1(b1fab72bee336271af17e6941bcb442275eb5bda) )
	ROM_LOAD16_BYTE( "cblv1-0n.p1", 0x000001, 0x020000, CRC(7e881d1c) SHA1(da1958f5dd7c0142c986791596a700be2928a410) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv1-0p.p0", 0x000000, 0x020000, CRC(ede4c950) SHA1(752aea190ebc2d5e89838e6e964da81fab6f5832) )
	ROM_LOAD16_BYTE( "cblv1-0p.p1", 0x000001, 0x020000, CRC(7e881d1c) SHA1(da1958f5dd7c0142c986791596a700be2928a410) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv1-3n.p0", 0x000000, 0x020000, CRC(7c72cec8) SHA1(38c6ba43b63c354cfd4d897ebbf67967fa12470a) )
	ROM_LOAD16_BYTE( "cblv1-3n.p1", 0x000001, 0x020000, CRC(dc236a6c) SHA1(009e394d8d8356d3b8ad0e814b11bf782f914f95) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrcld )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv1-3p.p0", 0x000000, 0x020000, CRC(50f7d21b) SHA1(5f0d3bc92b0b67cbd3857bdb820359038b787dbb) )
	ROM_LOAD16_BYTE( "cblv1-3p.p1", 0x000001, 0x020000, CRC(dc236a6c) SHA1(009e394d8d8356d3b8ad0e814b11bf782f914f95) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrcle )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-0n.p0", 0x000000, 0x020000, CRC(5f7a1601) SHA1(49b88c2a885841808dbd541bef4f7b0e037206f2) )
	ROM_LOAD16_BYTE( "cblv2-0n.p1", 0x000001, 0x020000, CRC(b3e57adf) SHA1(c068dc78dbb8c0276eb7f3a2971b10f485fb5a42) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-0p.p0", 0x000000, 0x020000, CRC(041450da) SHA1(6b3fa2c309c96d685f30c3c892d8f2ed152a9a4e) )
	ROM_LOAD16_BYTE( "cblv2-0p.p1", 0x000001, 0x020000, CRC(b3e57adf) SHA1(c068dc78dbb8c0276eb7f3a2971b10f485fb5a42) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-1n.p0", 0x000000, 0x020000, CRC(8bc132aa) SHA1(ee3beef42210ead1e35b3bf1e20ac1d7d9da510e) )
	ROM_LOAD16_BYTE( "cblv2-1n.p1", 0x000001, 0x020000, CRC(cce7c8a1) SHA1(bafb2f7906a9d01e20dae133c858a1cb4e021370) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-1p.p0", 0x000000, 0x020000, CRC(afffe3b9) SHA1(068bb6e960b5005787d8fcc566424eed12d2cd17) )
	ROM_LOAD16_BYTE( "cblv2-1p.p1", 0x000001, 0x020000, CRC(cce7c8a1) SHA1(bafb2f7906a9d01e20dae133c858a1cb4e021370) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrcli )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-2n.p0", 0x000000, 0x020000, CRC(c15ec0fb) SHA1(300bfdf72d818a641c5dd58fc9d83690dd22374e) )
	ROM_LOAD16_BYTE( "cblv2-2n.p1", 0x000001, 0x020000, CRC(fde881c1) SHA1(8b1a842664259722460e5bebf6c787a73647143e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cblv2-2p.p0", 0x000000, 0x020000, CRC(797f616f) SHA1(de881734992535afbeb6f2ca67047f6b58a9db7b) )
	ROM_LOAD16_BYTE( "cblv2-2p.p1", 0x000001, 0x020000, CRC(fde881c1) SHA1(8b1a842664259722460e5bebf6c787a73647143e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bvrclk )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "jcbl1-0n.p0", 0x000000, 0x020000, CRC(d43484eb) SHA1(a3bdb957dd9db48639fc7b3ef592523021b1184f) )
	ROM_LOAD16_BYTE( "jcbl1-0n.p1", 0x000001, 0x020000, CRC(7e375d11) SHA1(95ecf9727d45362419310b8b4b3d8718b005e255) )
	MISSING_SOUND
ROM_END






ROM_START( ep_cgrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc03070.g0", 0x000000, 0x020000, CRC(cea5f5e6) SHA1(4ef666b4cf57123b04ab8d01a16d35141bc341bc) )
	ROM_LOAD16_BYTE( "msc03070.g1", 0x000001, 0x020000, CRC(8ddeb47c) SHA1(818e68d59c5901f8afe978466272d76b3c0d18a5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cgrca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msc03071.g0", 0x000000, 0x020000, CRC(d1dfff6d) SHA1(a29b8422e1cb70a566a346bd34ec6e264792ba6b) )
	ROM_LOAD16_BYTE( "msc03071.g1", 0x000001, 0x020000, CRC(eb1a32a7) SHA1(33349697deb8f70387dc4cc8ebdc85e5a5d97ecc) )
	MISSING_SOUND
ROM_END







ROM_START( ep_ctc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cutt1-2n.p0", 0x000000, 0x020000, CRC(39ccf6c3) SHA1(750fe561d1b81e180492d2d0d10f86de1ae0fd03) )
	ROM_LOAD16_BYTE( "cutt1-2n.p1", 0x000001, 0x020000, CRC(7eb11d9b) SHA1(63ceaf76e78f44b3732a5d1f750bf8b3138a7169) )
	ROM_LOAD16_BYTE( "cutt1-2n.p2", 0x040000, 0x020000, CRC(164978c4) SHA1(9c2241a0dcf81ccffddc79edf623cb3a9d38da6f) )
	ROM_LOAD16_BYTE( "cutt1-2n.p3", 0x040001, 0x020000, CRC(ce715c82) SHA1(2f6dc4198ea5128fe393923afb91294e0628bc9c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ctca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cutt1-2p.p0", 0x000000, 0x020000, CRC(ecc1276b) SHA1(c6560713b8adb126c32caf4416cef3b704491fae) )
	ROM_LOAD16_BYTE( "cutt1-2p.p1", 0x000001, 0x020000, CRC(6d9f616c) SHA1(a3f8757869d342f2677c5ab661ef89bb59c07670) )
	ROM_LOAD16_BYTE( "cutt1-2p.p2", 0x040000, 0x020000, CRC(164978c4) SHA1(9c2241a0dcf81ccffddc79edf623cb3a9d38da6f) )
	ROM_LOAD16_BYTE( "cutt1-2p.p3", 0x040001, 0x020000, CRC(ce715c82) SHA1(2f6dc4198ea5128fe393923afb91294e0628bc9c) )
	MISSING_SOUND
ROM_END




ROM_START( ep_cyc )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycl02.p0", 0x000000, 0x100000, CRC(80c15dcf) SHA1(16be4fd743bb274194a9be602ed2608d82bcc05b) )
	ROM_LOAD16_BYTE( "cycl02.p1", 0x000001, 0x100000, CRC(f461c649) SHA1(220c5a3062a1dd5da09acf31c251cb1806317e29) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cyca )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycl02p.p0", 0x000000, 0x100000, CRC(80c15dcf) SHA1(16be4fd743bb274194a9be602ed2608d82bcc05b) )
	ROM_LOAD16_BYTE( "cycl02p.p1", 0x000001, 0x100000, CRC(a18aabf2) SHA1(6945977e78944c7fdc38c9bd22051e2359f5ac7d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cycb )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycl03.p0", 0x000000, 0x100000, CRC(26bcf9e4) SHA1(e7b34de400bfd9414166709b587d06995d8c20c6) )
	ROM_LOAD16_BYTE( "cycl03.p1", 0x000001, 0x100000, CRC(a2550508) SHA1(08f7cbc8fba08a13011eb62623d5cf108fb832a8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cycc )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycl03p.p0", 0x000000, 0x100000, CRC(26bcf9e4) SHA1(e7b34de400bfd9414166709b587d06995d8c20c6) )
	ROM_LOAD16_BYTE( "cycl03p.p1", 0x000001, 0x100000, CRC(f7be68b3) SHA1(87c24d8484148cae3b4edd962b335a5ef99cd65c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cycd )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycn01.p0", 0x000000, 0x100000, CRC(e5c2f005) SHA1(e217f2ef90016a987190f2ec338c3051d531e221) )
	ROM_LOAD16_BYTE( "cycn01.p1", 0x000001, 0x100000, CRC(cba5a3cb) SHA1(a50f273317a4e5cb37da3a7f888fe00142484382) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cyce )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cycn01p.p0", 0x000000, 0x100000, CRC(e5c2f005) SHA1(e217f2ef90016a987190f2ec338c3051d531e221) )
	ROM_LOAD16_BYTE( "cycn01p.p1", 0x000001, 0x100000, CRC(9e4ece70) SHA1(16509d931d431158a0d11abe8e94f707913761db) )
	MISSING_SOUND
ROM_END



#define EP_DMBUS_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "d bus.s0", 0x000000, 0x080000, CRC(dcf93e70) SHA1(c005daa041bcc2bb8c4aa422593599a530390b28) ) \
	ROM_LOAD( "d bus.s1", 0x080000, 0x080000, CRC(d6ebc176) SHA1(c554358839463d706f93e4eee4462d91a3961a59) )

ROM_START( ep_dmbus )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gdb03_0.bin", 0x000000, 0x020000, CRC(592ca38c) SHA1(56c6fcc31dacbdeb405dd0ff516f6d82d6156771) )
	ROM_LOAD16_BYTE( "gdb03_1.bin", 0x000001, 0x020000, CRC(d8fa7624) SHA1(e804e95560e28e8d8042e5f89718c918cb8eea7f) )
	ROM_LOAD16_BYTE( "gdb03_2.bin", 0x040000, 0x020000, CRC(b96b18af) SHA1(5da89f22a9c2d5243eca981533e74979d4626c63) )
	ROM_LOAD16_BYTE( "gdb03_3.bin", 0x040001, 0x020000, CRC(cf2e70aa) SHA1(fa06b4519a3ddf15cceeb0934f1f87d36d18b8a6) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbusa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gdb04_0.bin", 0x000000, 0x020000, CRC(592ca38c) SHA1(56c6fcc31dacbdeb405dd0ff516f6d82d6156771) )
	ROM_LOAD16_BYTE( "gdb04_1.bin", 0x000001, 0x020000, CRC(3364221c) SHA1(d778765e757bbca5e702910d8bcf6c5b760e616d) )
	ROM_LOAD16_BYTE( "gdb04_2.bin", 0x040000, 0x020000, CRC(b96b18af) SHA1(5da89f22a9c2d5243eca981533e74979d4626c63) )
	ROM_LOAD16_BYTE( "gdb04_3.bin", 0x040001, 0x020000, CRC(cf2e70aa) SHA1(fa06b4519a3ddf15cceeb0934f1f87d36d18b8a6) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbusb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gdb05_0.bin", 0x000000, 0x020000, CRC(7376398b) SHA1(1912ae95520d1f55d9dfcfdf1e8516ad31e281b8) )
	ROM_LOAD16_BYTE( "gdb05_1.bin", 0x000001, 0x020000, CRC(3fa379d0) SHA1(5d30b32ea48535b895356a7598689009eb1ab187) )
	ROM_LOAD16_BYTE( "gdb05_2.bin", 0x040000, 0x020000, CRC(28acae49) SHA1(74bb8eaa2c75689a1caff60ee996ec9b7774655d) )
	ROM_LOAD16_BYTE( "gdb05_3.bin", 0x040001, 0x020000, CRC(a3ada8e4) SHA1(6f3db1ba20c2daf16b60bceb6069b994be27c5ca) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbusc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gdb06_0.bin", 0x000000, 0x020000, CRC(7376398b) SHA1(1912ae95520d1f55d9dfcfdf1e8516ad31e281b8) )
	ROM_LOAD16_BYTE( "gdb06_1.bin", 0x000001, 0x020000, CRC(d43d2de8) SHA1(5928ecd25761f13a542dcea7fda5d4802c77a267) )
	ROM_LOAD16_BYTE( "gdb06_2.bin", 0x040000, 0x020000, CRC(28acae49) SHA1(74bb8eaa2c75689a1caff60ee996ec9b7774655d) )
	ROM_LOAD16_BYTE( "gdb06_3.bin", 0x040001, 0x020000, CRC(a3ada8e4) SHA1(6f3db1ba20c2daf16b60bceb6069b994be27c5ca) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbusd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03ndb01.g0.bin", 0x000000, 0x020000, CRC(98cff98c) SHA1(cf04296bb4500949a9e0c4062d4b18cf9cae778d) )
	ROM_LOAD16_BYTE( "i03ndb01.g1.bin", 0x000001, 0x020000, CRC(7c315d54) SHA1(d55dd55aa64375c3778184a397480f7674eae511) )
	ROM_LOAD16_BYTE( "i03ndb01.g2.bin", 0x040000, 0x020000, CRC(85fae0e7) SHA1(0cd8488e2cdd15fb9e0530142af66be320065bde) )
	ROM_LOAD16_BYTE( "i03ndb01.g3.bin", 0x040001, 0x020000, CRC(c2299aba) SHA1(7d5cc03142274e32c6b392ef31595fde99e11fa8) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbuse )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03pdb02.g0.bin", 0x000000, 0x020000, CRC(98cff98c) SHA1(cf04296bb4500949a9e0c4062d4b18cf9cae778d) )
	ROM_LOAD16_BYTE( "i03pdb02.g1.bin", 0x000001, 0x020000, CRC(97af096c) SHA1(47291335c2392a563165ab372e28518873d15a20) )
	ROM_LOAD16_BYTE( "i03pdb02.g2.bin", 0x040000, 0x020000, CRC(85fae0e7) SHA1(0cd8488e2cdd15fb9e0530142af66be320065bde) )
	ROM_LOAD16_BYTE( "i03pdb02.g3.bin", 0x040001, 0x020000, CRC(c2299aba) SHA1(7d5cc03142274e32c6b392ef31595fde99e11fa8) )
	EP_DMBUS_SOUND
ROM_END

ROM_START( ep_dmbusf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "d bus.g0", 0x000000, 0x020000, CRC(c466f4fb) SHA1(cf9041fd6b87498f6a6492995ae43f43b22dbf6a) )
	ROM_LOAD16_BYTE( "d bus.g1", 0x000001, 0x020000, CRC(f8cb1930) SHA1(f3854a13a60ad776b60c107f3bc200d5307b9ea2) )
	ROM_LOAD16_BYTE( "d bus.g2", 0x040000, 0x020000, CRC(895ef2cc) SHA1(094ae9365fda2da71e5ff5df3ee7ea548069647b) )
	ROM_LOAD16_BYTE( "d bus.g3", 0x040001, 0x020000, CRC(9ad29064) SHA1(4b2525e8058fcd4e520b381d8705bd642b793121) )
	EP_DMBUS_SOUND
ROM_END



ROM_START( ep_ddq )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung1-4n.p0", 0x000000, 0x020000, CRC(6824081c) SHA1(465fb4dac4de9044fae7183e7462588fe4a437f5) )
	ROM_LOAD16_BYTE( "dung1-4n.p1", 0x000001, 0x020000, CRC(32b84803) SHA1(1adf2555b51489e2152aa9b8f699ccfb83b47847) )
	ROM_LOAD16_BYTE( "dung1-4n.p2", 0x040000, 0x020000, CRC(3143f191) SHA1(de03b56c80a10d4368ff6e6a082dc8b852e15000) )
	ROM_LOAD16_BYTE( "dung1-4n.p3", 0x040001, 0x020000, CRC(ebe9bdf1) SHA1(68bc9fd9427f71025f76a6422c362bd98a34fac8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung1-4p.p0", 0x000000, 0x020000, CRC(86225a63) SHA1(2134a725373626fa673f27e147342eb74e4fc9a2) )
	ROM_LOAD16_BYTE( "dung1-4p.p1", 0x000001, 0x020000, CRC(028a50fc) SHA1(9f015496c777a8deb9967a524b279f899b65f6c9) )
	ROM_LOAD16_BYTE( "dung1-4p.p2", 0x040000, 0x020000, CRC(3143f191) SHA1(de03b56c80a10d4368ff6e6a082dc8b852e15000) )
	ROM_LOAD16_BYTE( "dung1-4p.p3", 0x040001, 0x020000, CRC(ebe9bdf1) SHA1(68bc9fd9427f71025f76a6422c362bd98a34fac8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-1n.p0", 0x000000, 0x020000, CRC(b624b18d) SHA1(754f1b9d3d3280bb7b4a2a3c8f0128fb524fa518) )
	ROM_LOAD16_BYTE( "dung2-1n.p1", 0x000001, 0x020000, CRC(6079e0ed) SHA1(59b0a0cb0b98ce4e0da7fbdf11c75f570e99af26) )
	ROM_LOAD16_BYTE( "dung2-1n.p2", 0x040000, 0x020000, CRC(b68feecb) SHA1(4b0832562679317615c917f8f209711598c20fb1) )
	ROM_LOAD16_BYTE( "dung2-1n.p3", 0x040001, 0x020000, CRC(b38ee9d0) SHA1(7d7ee2ed4e9b263727339edf8a4fa2a2e42ec124) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-1p.p0", 0x000000, 0x020000, CRC(42c34a62) SHA1(46378baacd3c34028740aa8ecfe9490794c2d683) )
	ROM_LOAD16_BYTE( "dung2-1p.p1", 0x000001, 0x020000, CRC(2bb06e44) SHA1(120bd6012aa72c245a9179508d13b077bfa949aa) )
	ROM_LOAD16_BYTE( "dung2-1p.p2", 0x040000, 0x020000, CRC(b68feecb) SHA1(4b0832562679317615c917f8f209711598c20fb1) )
	ROM_LOAD16_BYTE( "dung2-1p.p3", 0x040001, 0x020000, CRC(b38ee9d0) SHA1(7d7ee2ed4e9b263727339edf8a4fa2a2e42ec124) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-2n.p0", 0x000000, 0x020000, CRC(ffac1d06) SHA1(176c20455619a787913cbc3b06612b3499d7e5ad) )
	ROM_LOAD16_BYTE( "dung2-2n.p1", 0x000001, 0x020000, CRC(f564c0f3) SHA1(eb90ad4c23a2886088ac641834b8a1fb50cb69c2) )
	ROM_LOAD16_BYTE( "dung2-2n.p2", 0x040000, 0x020000, CRC(581c5775) SHA1(14e9673b2e14512522c0e7ce48ca56440ebd6a18) )
	ROM_LOAD16_BYTE( "dung2-2n.p3", 0x040001, 0x020000, CRC(2383111b) SHA1(821e096c3de5d0d47da2ee704986ac4e4940196e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-2p.p0", 0x000000, 0x020000, CRC(16af5990) SHA1(8276c4c8aa8340551f9b196a7a545eebf9da1e4c) )
	ROM_LOAD16_BYTE( "dung2-2p.p1", 0x000001, 0x020000, CRC(31e4da7d) SHA1(64add7ec48a36e014d5cc532d8fdc4d0b242963f) )
	ROM_LOAD16_BYTE( "dung2-2p.p2", 0x040000, 0x020000, CRC(581c5775) SHA1(14e9673b2e14512522c0e7ce48ca56440ebd6a18) )
	ROM_LOAD16_BYTE( "dung2-2p.p3", 0x040001, 0x020000, CRC(2383111b) SHA1(821e096c3de5d0d47da2ee704986ac4e4940196e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-4n.p0", 0x000000, 0x020000, CRC(8a7317e0) SHA1(0d67452df231d86aec7ca2b0dbf134e44b2bf074) )
	ROM_LOAD16_BYTE( "dung2-4n.p1", 0x000001, 0x020000, CRC(260391df) SHA1(089b6a924784e91ea08dbf6c6df02b7c27019f0d) )
	ROM_LOAD16_BYTE( "dung2-4n.p2", 0x040000, 0x020000, CRC(675d9549) SHA1(8d07588d1ef3e0e1486799ef06c2063c25b9bdd7) )
	ROM_LOAD16_BYTE( "dung2-4n.p3", 0x040001, 0x020000, CRC(faa41398) SHA1(550c852b491d401de7d812c3ad5ee4c6c25402cc) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-4p.p0", 0x000000, 0x020000, CRC(680692c1) SHA1(bcbc20ad1fd80449c71e467c31202cb1fd50b561) )
	ROM_LOAD16_BYTE( "dung2-4p.p1", 0x000001, 0x020000, CRC(a92f8d32) SHA1(d0731ab9bef3a6866105203a74759a57e315fbc0) )
	ROM_LOAD16_BYTE( "dung2-4p.p2", 0x040000, 0x020000, CRC(675d9549) SHA1(8d07588d1ef3e0e1486799ef06c2063c25b9bdd7) )
	ROM_LOAD16_BYTE( "dung2-4p.p3", 0x040001, 0x020000, CRC(faa41398) SHA1(550c852b491d401de7d812c3ad5ee4c6c25402cc) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-5n.p0", 0x000000, 0x020000, CRC(9a9f4cb1) SHA1(de1aab1d66d3195a5c5be6d3b09978f72b38d90b) )
	ROM_LOAD16_BYTE( "dung2-5n.p1", 0x000001, 0x020000, CRC(f5064e98) SHA1(fe4401d4c0eb0374cbd095a41dc5b9da8d81c1e8) )
	ROM_LOAD16_BYTE( "dung2-5n.p2", 0x040000, 0x020000, CRC(aec5f388) SHA1(56c5201d76c0398d675e58b3cbd6fa415fd04e23) )
	ROM_LOAD16_BYTE( "dung2-5n.p3", 0x040001, 0x020000, CRC(e623c817) SHA1(61e54d6f55b5d43731303498f9c811ccf14e4b1f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ddqi )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dung2-5p.p0", 0x000000, 0x020000, CRC(4ed3f394) SHA1(366c5bf47c3190d666ec7aba933608989980ab59) )
	ROM_LOAD16_BYTE( "dung2-5p.p1", 0x000001, 0x020000, CRC(2bd7ccd9) SHA1(0f5202bdbb58b6c9d1ab0779c8f14329fc37bfda) )
	ROM_LOAD16_BYTE( "dung2-5p.p2", 0x040000, 0x020000, CRC(aec5f388) SHA1(56c5201d76c0398d675e58b3cbd6fa415fd04e23) )
	ROM_LOAD16_BYTE( "dung2-5p.p3", 0x040001, 0x020000, CRC(e623c817) SHA1(61e54d6f55b5d43731303498f9c811ccf14e4b1f) )
	MISSING_SOUND
ROM_END







ROM_START( ep_fog )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "fog2-6n.p1", 0x000000, 0x020000, CRC(eb2092c1) SHA1(1cc9ed31468b8f9c3c656812a2f3e4030b1ad8f0) )
	ROM_LOAD16_BYTE( "fog2-6n.p2", 0x000001, 0x020000, CRC(42deb7e5) SHA1(5613f2bd160b5b74a25301caf67c683eb8c1db22) )
	MISSING_SOUND
ROM_END

ROM_START( ep_foga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "fog2-6p.p1", 0x000000, 0x020000, CRC(d60b3786) SHA1(b1d3f5a4bee1864dd6326b6cf720aaaed8b59f66) )
	ROM_LOAD16_BYTE( "fog2-6p.p2", 0x000001, 0x020000, CRC(42deb7e5) SHA1(5613f2bd160b5b74a25301caf67c683eb8c1db22) )
	MISSING_SOUND
ROM_END



#define EP_FULLM_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) /* guess based on name */ \
	ROM_LOAD( "fmoon.s0", 0x000000, 0x080000, CRC(e8b06465) SHA1(269c2f271f5338dea6aed0be7ef49ba2ae2025c3) ) \
	ROM_LOAD( "fmoon.s1", 0x080000, 0x080000, CRC(4781c1bd) SHA1(a9f66ad2b111c75edf29d4ea65c546a969504bb6) )

ROM_START( ep_fullm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pff19.g0", 0x000000, 0x020000, CRC(2296fec8) SHA1(b109f714d3ffbb4b4124cc1b48ed73076936f2c4) )
	ROM_LOAD16_BYTE( "i00pff19.g1", 0x000001, 0x020000, CRC(25855b60) SHA1(64682b7a4ba77c0b991ef53779865b6e2baa713c) )
	ROM_LOAD16_BYTE( "i00pff19.g2", 0x040000, 0x020000, CRC(7545e2ee) SHA1(ce64c0f17cab55f9c8ac3b7eff68f245a65a6808) )
	ROM_LOAD16_BYTE( "i00pff19.g3", 0x040001, 0x020000, CRC(d34de35f) SHA1(cc144e67153a329034cc54673c7c363c94be1b61) )
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullma )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nff14.g0", 0x000000, 0x020000, CRC(825e2004) SHA1(5462c5b813799409ab44161b40fbf8ae00736444) )
	ROM_LOAD16_BYTE( "i00nff14.g1", 0x000001, 0x020000, CRC(bd4cbc0b) SHA1(ec5d0a271d9d8e9789134ab808d4b54a2fd00cf5) )
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullmb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nff16.g0", 0x000000, 0x020000, CRC(825e2004) SHA1(5462c5b813799409ab44161b40fbf8ae00736444) )
	ROM_LOAD16_BYTE( "i00nff16.g1", 0x000001, 0x020000, CRC(b101123a) SHA1(22e083d3e14cfe7a4180a55549b5af41f4d5031e) )
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullmc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nff18.g0", 0x000000, 0x020000, CRC(2296fec8) SHA1(b109f714d3ffbb4b4124cc1b48ed73076936f2c4) )
	ROM_LOAD16_BYTE( "i00nff18.g1", 0x000001, 0x020000, CRC(ce1b0f58) SHA1(f37092f2aed762332d9d36c00e8090b48a1493d4) )
	ROM_LOAD16_BYTE( "i00nff18.g2", 0x040000, 0x020000, CRC(7545e2ee) SHA1(ce64c0f17cab55f9c8ac3b7eff68f245a65a6808) )
	ROM_LOAD16_BYTE( "i00nff18.g3", 0x040001, 0x020000, CRC(d34de35f) SHA1(cc144e67153a329034cc54673c7c363c94be1b61) )
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullmd )
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00nff18.g0", 0x000000, 0x020000, CRC(2296fec8) SHA1(b109f714d3ffbb4b4124cc1b48ed73076936f2c4) )
	ROM_LOAD16_BYTE( "i00nff18.g1", 0x000001, 0x020000, CRC(ce1b0f58) SHA1(f37092f2aed762332d9d36c00e8090b48a1493d4) )
	ROM_LOAD16_BYTE( "i00nff18.g2", 0x040000, 0x020000, CRC(7545e2ee) SHA1(ce64c0f17cab55f9c8ac3b7eff68f245a65a6808) )
	ROM_LOAD16_BYTE( "f_moon.g3",   0x040001, 0x080000, CRC(2ff659ab) SHA1(d7dd545abf6b3794b114690a5c53e488d8bb3f02) ) // paired with i00pff18.g0 i00nff18.g1 i00nff18.g2 (original names f_moon.gx) either i00nff18.g3 or this is probably a bad dump, proably this due size
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullme )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pff15.g0", 0x000000, 0x020000, CRC(825e2004) SHA1(5462c5b813799409ab44161b40fbf8ae00736444) )
	ROM_LOAD16_BYTE( "i00pff15.g1", 0x000001, 0x020000, CRC(56d2e833) SHA1(30d216800fe6f07874336365e740b9a61bccedfa) )
	EP_FULLM_SOUND
ROM_END

ROM_START( ep_fullmf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i00pff17.g0", 0x000000, 0x020000, CRC(825e2004) SHA1(5462c5b813799409ab44161b40fbf8ae00736444) )
	ROM_LOAD16_BYTE( "i00pff17.g1", 0x000001, 0x020000, CRC(5a9f4602) SHA1(9334841150114c16b8648ecc81d1de5cb6957367) )
	EP_FULLM_SOUND
ROM_END




ROM_START( ep_fmf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "fmfg051n.p0", 0x000000, 0x020000, CRC(9d567300) SHA1(f492144f6fe68d549fb60372c5fb7adb4c54eb20) )
	ROM_LOAD16_BYTE( "fmfg051n.p1", 0x000001, 0x020000, CRC(211396e2) SHA1(01447cddc981024d15c18c083ee37db688ef7d60) )
	ROM_LOAD16_BYTE( "fmfg051n.p2", 0x040000, 0x020000, CRC(0f8cc6ad) SHA1(7fe5b6a5fc40e448f4f67b36f9210d3cabeb3532) )
	ROM_LOAD16_BYTE( "fmfg051n.p3", 0x040001, 0x020000, CRC(495f776a) SHA1(a5c9071dc40c22570c5fbe81efcba8620fa6e25b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_fmfa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "fmfg051p.p0", 0x000000, 0x020000, CRC(9d567300) SHA1(f492144f6fe68d549fb60372c5fb7adb4c54eb20) )
	ROM_LOAD16_BYTE( "fmfg051p.p1", 0x000001, 0x020000, CRC(ca8dc2da) SHA1(e97c44f13dbac170263620209b92d8c8fa659be1) )
	ROM_LOAD16_BYTE( "fmfg051p.p2", 0x040000, 0x020000, CRC(0f8cc6ad) SHA1(7fe5b6a5fc40e448f4f67b36f9210d3cabeb3532) )
	ROM_LOAD16_BYTE( "fmfg051p.p3", 0x040001, 0x020000, CRC(495f776a) SHA1(a5c9071dc40c22570c5fbe81efcba8620fa6e25b) )
	MISSING_SOUND
ROM_END





ROM_START( ep_gtrot )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ggt01_0.bin", 0x000000, 0x020000, CRC(b6bd9dbd) SHA1(66495fd09540a72ea75bd8f812c811e3dc0af4a6) )
	ROM_LOAD16_BYTE( "ggt01_1.bin", 0x000001, 0x020000, CRC(acdc7470) SHA1(df37eeaa1ed582704acac8e41c4624286e9fa4a1) )
	ROM_LOAD16_BYTE( "ggt01_2.bin", 0x040000, 0x020000, CRC(753fbb4b) SHA1(abc29ad272198a92411c5a6c7f20a10a099a0be3) )
	ROM_LOAD16_BYTE( "ggt01_3.bin", 0x040001, 0x020000, CRC(90ca1e16) SHA1(502a6f37dddf92c9889afa85158ad170432f3cd8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_gtrota )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ggt02_0.bin", 0x000000, 0x020000, CRC(b6bd9dbd) SHA1(66495fd09540a72ea75bd8f812c811e3dc0af4a6) )
	ROM_LOAD16_BYTE( "ggt02_1.bin", 0x000001, 0x020000, CRC(47422048) SHA1(95b5dc29268d16cf54ee998c0e0c229876205db5) )
	ROM_LOAD16_BYTE( "ggt02_2.bin", 0x040000, 0x020000, CRC(753fbb4b) SHA1(abc29ad272198a92411c5a6c7f20a10a099a0be3) )
	ROM_LOAD16_BYTE( "ggt02_3.bin", 0x040001, 0x020000, CRC(90ca1e16) SHA1(502a6f37dddf92c9889afa85158ad170432f3cd8) )
	MISSING_SOUND
ROM_END




ROM_START( ep_greed )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gree1-3n.p0", 0x000000, 0x020000, CRC(758b5b6a) SHA1(cf26451f12c5e3e5f8d6d87487442795cc7aa3f2) )
	ROM_LOAD16_BYTE( "gree1-3n.p1", 0x000001, 0x020000, CRC(3ceb0a0f) SHA1(4d6119a3b800b34699bb76dd6e7fb0ea1aa3b226) )
	ROM_LOAD16_BYTE( "gree1-3n.p2", 0x040000, 0x020000, CRC(620db233) SHA1(95e82ba36a8f1856ec5f296c65f4b3bb5abc488f) )
	ROM_LOAD16_BYTE( "gree1-3n.p3", 0x040001, 0x020000, CRC(12fdb4a0) SHA1(b60df60cc84fc43f8e6439053278c503621e83aa) )
	MISSING_SOUND
ROM_END

ROM_START( ep_greeda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gree1-3p.p0", 0x000000, 0x020000, CRC(52c7c893) SHA1(18162a506167441fb0a22017c4d78005ea50d490) )
	ROM_LOAD16_BYTE( "gree1-3p.p1", 0x000001, 0x020000, CRC(739e730e) SHA1(e6a29ac2caaa18e314b6a267c810f1fdb996a573) )
	ROM_LOAD16_BYTE( "gree1-3p.p2", 0x040000, 0x020000, CRC(620db233) SHA1(95e82ba36a8f1856ec5f296c65f4b3bb5abc488f) )
	ROM_LOAD16_BYTE( "gree1-3p.p3", 0x040001, 0x020000, CRC(12fdb4a0) SHA1(b60df60cc84fc43f8e6439053278c503621e83aa) )
	MISSING_SOUND
ROM_END




ROM_START( ep_grncl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gr_data.p0", 0x000000, 0x020000, CRC(c2d4e843) SHA1(adc7b91e4cfea56cad4fbc554b6d90b7f05c0c0e) )
	ROM_LOAD16_BYTE( "gr_data.p1", 0x000001, 0x020000, CRC(d3881c7f) SHA1(ad7646979ad8a1b62cf8cf96566b0c49791c8803) )
	MISSING_SOUND
ROM_END

ROM_START( ep_grncla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gr_nondata.p0", 0x000000, 0x020000, CRC(0ee9aa47) SHA1(d85d04d3bb6bd2f379caf202592165beeb54e946) )
	ROM_LOAD16_BYTE( "gr_nondata.p1", 0x000001, 0x020000, CRC(d3881c7f) SHA1(ad7646979ad8a1b62cf8cf96566b0c49791c8803) )
	MISSING_SOUND
ROM_END

ROM_START( ep_grnclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grid1-1n.p0", 0x000000, 0x020000, CRC(3d8c469b) SHA1(2f46143c62157a72beefb015b4ab664d4d4b5895) )
	ROM_LOAD16_BYTE( "grid1-1n.p1", 0x000001, 0x020000, CRC(1dfc84ec) SHA1(7e9655df68011beac3cf5aac4d70614564b5a63c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_grnclc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "grid1-1p.p0", 0x000000, 0x020000, CRC(8ee19357) SHA1(5d47c2a32f78fe90e5d6c98b0f5709e88ee12f4f) )
	ROM_LOAD16_BYTE( "grid1-1p.p1", 0x000001, 0x020000, CRC(1dfc84ec) SHA1(7e9655df68011beac3cf5aac4d70614564b5a63c) )
	MISSING_SOUND
ROM_END








ROM_START( ep_heybc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-5n.p0", 0x000000, 0x020000, CRC(724e7f4b) SHA1(c7ac83fea12a0e91c84b68885338eb6d76727692) )
	ROM_LOAD16_BYTE( "heyb1-5n.p1", 0x000001, 0x020000, CRC(ff999846) SHA1(5cf52457e4e3b8151e8ae2e9e1da6f0a10c0f0fc) )
	MISSING_SOUND
ROM_END

ROM_START( ep_heybca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "heyb1-5p.p0", 0x000000, 0x020000, CRC(3373eebd) SHA1(fc2b736e02deb8234639ff18cbe3a4065f76a964) )
	ROM_LOAD16_BYTE( "heyb1-5p.p1", 0x000001, 0x020000, CRC(ff999846) SHA1(5cf52457e4e3b8151e8ae2e9e1da6f0a10c0f0fc) )
	MISSING_SOUND
ROM_END




ROM_START( ep_hispr )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15hs1-an.p0", 0x000000, 0x020000, CRC(b2af5867) SHA1(f62bae5aeddcf8ccff0ac2d049a4f1ab4ffe6b20) )
	ROM_LOAD16_BYTE( "15hs1-an.p1", 0x000001, 0x020000, CRC(35a6de1c) SHA1(684e1c17527b8fedbe637aba39253de4a5564b11) )
	ROM_LOAD16_BYTE( "15hs1-an.p2", 0x040000, 0x020000, CRC(ff29016e) SHA1(2f4d706f11b6612d41ca34e01f20facc0b259dfb) )
	ROM_LOAD16_BYTE( "15hs1-an.p3", 0x040001, 0x020000, CRC(c33ff217) SHA1(62aea359f9393fdf488a6705b045420bc700147f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hispra )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "15hs1-ap.p0", 0x000000, 0x020000, CRC(78b9e651) SHA1(c0117bcef4533712562dec57df42099ea482426b) )
	ROM_LOAD16_BYTE( "15hs1-ap.p1", 0x000001, 0x020000, CRC(d3f22439) SHA1(d7de5ff452ea70f167651fdf946d75068396d182) )
	ROM_LOAD16_BYTE( "15hs1-ap.p2", 0x040000, 0x020000, CRC(ff29016e) SHA1(2f4d706f11b6612d41ca34e01f20facc0b259dfb) )
	ROM_LOAD16_BYTE( "15hs1-ap.p3", 0x040001, 0x020000, CRC(c33ff217) SHA1(62aea359f9393fdf488a6705b045420bc700147f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hisprb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hisp4-2n.p0", 0x000000, 0x020000, CRC(54b2f68d) SHA1(b0c86157ed2f5eb65d5b94a2ae3e285d06f45dfe) )
	ROM_LOAD16_BYTE( "hisp4-2n.p1", 0x000001, 0x020000, CRC(956d19fc) SHA1(898ab9b3deb54e8c4818d3a9b0b7c7dfcf431ed0) )
	ROM_LOAD16_BYTE( "hisp4-2n.p2", 0x040000, 0x020000, CRC(519a1e21) SHA1(ab746a62c467ff884509955258646b4848cc7050) )
	ROM_LOAD16_BYTE( "hisp4-2n.p3", 0x040001, 0x020000, CRC(8f2bec00) SHA1(403517bed0ad328f80d80b13aa7f65c6665a7797) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hisprc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "hisp4-2p.p0", 0x000000, 0x020000, CRC(535521be) SHA1(22a39b1e70ab2380c7467b3cfc268fec949c28a7) )
	ROM_LOAD16_BYTE( "hisp4-2p.p1", 0x000001, 0x020000, CRC(8aa0cbb8) SHA1(6afff85cfc79bc6867dfdf0797243372b3497c6b) )
	ROM_LOAD16_BYTE( "hisp4-2p.p2", 0x040000, 0x020000, CRC(519a1e21) SHA1(ab746a62c467ff884509955258646b4848cc7050) )
	ROM_LOAD16_BYTE( "hisp4-2p.p3", 0x040001, 0x020000, CRC(8f2bec00) SHA1(403517bed0ad328f80d80b13aa7f65c6665a7797) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hisprd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "rhis3-2n.p0", 0x000000, 0x020000, CRC(16d0c419) SHA1(864b9186e3be5a8152d4f7458275198761ed570c) )
	ROM_LOAD16_BYTE( "rhis3-2n.p1", 0x000001, 0x020000, CRC(c0ce3587) SHA1(adf53422fecaa78499759aee174d0008f616e916) )
	ROM_LOAD16_BYTE( "rhis3-2n.p2", 0x040000, 0x020000, CRC(9755355f) SHA1(07b7a4228162642c0402432b4d52e2982dd472d8) )
	ROM_LOAD16_BYTE( "rhis3-2n.p3", 0x040001, 0x020000, CRC(fa8696a3) SHA1(5593534b271e00ce57072637bce8b240df4a8491) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hispre )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "rhis3-2p.p0", 0x000000, 0x020000, CRC(19ff6cf7) SHA1(4429caeb81874a95fe12921a7d9f5c651bb92463) )
	ROM_LOAD16_BYTE( "rhis3-2p.p1", 0x000001, 0x020000, CRC(1625bd9f) SHA1(14b953fdc3b0485763dbff3212ae68aefcf4e820) )
	ROM_LOAD16_BYTE( "rhis3-2p.p2", 0x040000, 0x020000, CRC(9755355f) SHA1(07b7a4228162642c0402432b4d52e2982dd472d8) )
	ROM_LOAD16_BYTE( "rhis3-2p.p3", 0x040001, 0x020000, CRC(fa8696a3) SHA1(5593534b271e00ce57072637bce8b240df4a8491) )
	MISSING_SOUND
ROM_END



ROM_START( ep_hiscl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "chis1-2n.p0", 0x000000, 0x020000, CRC(ab038703) SHA1(1d3dcaeb7513fc11fc79d451a17a57f83bf5c006) )
	ROM_LOAD16_BYTE( "chis1-2n.p1", 0x000001, 0x020000, CRC(de4d79be) SHA1(91f6e2ae3c363b371504f16d4b1be7593e306269) )
	MISSING_SOUND
ROM_END

ROM_START( ep_hiscla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "chis1-2p.p0", 0x000000, 0x020000, CRC(af1884eb) SHA1(7f1107f993aee42fe212ebe4fa14dada4721f01c) )
	ROM_LOAD16_BYTE( "chis1-2p.p1", 0x000001, 0x020000, CRC(de4d79be) SHA1(91f6e2ae3c363b371504f16d4b1be7593e306269) )
	MISSING_SOUND
ROM_END








ROM_START( ep_smoke )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs01_0.bin", 0x000000, 0x020000, CRC(24fc4a8e) SHA1(8aa72da2ad7ad3a1e495543d236da1edb579e78e) )
	ROM_LOAD16_BYTE( "ghs01_1.bin", 0x000001, 0x020000, CRC(cbb41228) SHA1(b8f84a560b516e7fd27303939172a53b29cc48da) )
	ROM_LOAD16_BYTE( "ghs01_2.bin", 0x040000, 0x020000, CRC(f4d8bab6) SHA1(05e0bcc8c4d176688238214c65ecbd28459ed874) )
	ROM_LOAD16_BYTE( "ghs01_3.bin", 0x040001, 0x020000, CRC(2257930b) SHA1(fea484dbb7c0595898390174c4d3824024ac7980) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokea )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs02_0.bin", 0x000000, 0x020000, CRC(24fc4a8e) SHA1(8aa72da2ad7ad3a1e495543d236da1edb579e78e) )
	ROM_LOAD16_BYTE( "ghs02_1.bin", 0x000001, 0x020000, CRC(202a4610) SHA1(1242de214337158afa3169254bdadda1e18adeed) )
	ROM_LOAD16_BYTE( "ghs02_2.bin", 0x040000, 0x020000, CRC(f4d8bab6) SHA1(05e0bcc8c4d176688238214c65ecbd28459ed874) )
	ROM_LOAD16_BYTE( "ghs02_3.bin", 0x040001, 0x020000, CRC(2257930b) SHA1(fea484dbb7c0595898390174c4d3824024ac7980) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokeb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs03_0.bin", 0x000000, 0x020000, CRC(56ba8e31) SHA1(d04111d8befb34ecef97f8d76db0b47b6c80de73) )
	ROM_LOAD16_BYTE( "ghs03_1.bin", 0x000001, 0x020000, CRC(e17a51dd) SHA1(2b61c23ee492f902ee10a4679141f4c112cc3e05) )
	ROM_LOAD16_BYTE( "ghs03_2.bin", 0x040000, 0x020000, CRC(8c18d2df) SHA1(c1c6168f939da795fc50d8226444f8b5a0e225a2) )
	ROM_LOAD16_BYTE( "ghs03_3.bin", 0x040001, 0x020000, CRC(73af813e) SHA1(c4d6b05b9caae9fed06a232902265cb21a59d481) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokec )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs04_0.bin", 0x000000, 0x020000, CRC(56ba8e31) SHA1(d04111d8befb34ecef97f8d76db0b47b6c80de73) )
	ROM_LOAD16_BYTE( "ghs04_1.bin", 0x000001, 0x020000, CRC(0ae405e5) SHA1(b091034ff39a3dfae398cd40617d2d684712aca6) )
	ROM_LOAD16_BYTE( "ghs04_2.bin", 0x040000, 0x020000, CRC(8c18d2df) SHA1(c1c6168f939da795fc50d8226444f8b5a0e225a2) )
	ROM_LOAD16_BYTE( "ghs04_3.bin", 0x040001, 0x020000, CRC(73af813e) SHA1(c4d6b05b9caae9fed06a232902265cb21a59d481) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smoked )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs05_0.bin", 0x000000, 0x020000, CRC(fedfef71) SHA1(1892eb516dd40f8997b22ba24d25e7bf90b50b06) )
	ROM_LOAD16_BYTE( "ghs05_1.bin", 0x000001, 0x020000, CRC(496629ae) SHA1(03450593593d6550323857b07cc6d2e7aaa5bd7b) )
	ROM_LOAD16_BYTE( "ghs05_2.bin", 0x040000, 0x020000, CRC(f75125ad) SHA1(6e54270fcbdc1d586a91aff50116260a6d8cea3c) )
	ROM_LOAD16_BYTE( "ghs05_3.bin", 0x040001, 0x020000, CRC(4f76ce51) SHA1(453a8e10fe1661bd2b40cc399db8cbc6a45f16e9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokee )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs06_0.bin", 0x000000, 0x020000, CRC(fedfef71) SHA1(1892eb516dd40f8997b22ba24d25e7bf90b50b06) )
	ROM_LOAD16_BYTE( "ghs06_1.bin", 0x000001, 0x020000, CRC(a2f87d96) SHA1(d561c84b0d9ff58cb0aeca158c816931a2abcfad) )
	ROM_LOAD16_BYTE( "ghs06_2.bin", 0x040000, 0x020000, CRC(f75125ad) SHA1(6e54270fcbdc1d586a91aff50116260a6d8cea3c) )
	ROM_LOAD16_BYTE( "ghs06_3.bin", 0x040001, 0x020000, CRC(4f76ce51) SHA1(453a8e10fe1661bd2b40cc399db8cbc6a45f16e9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokef )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs07_0.bin", 0x000000, 0x020000, CRC(e121f95a) SHA1(497bc333fd36416a7583170ec60bf8e0dd46bbbf) )
	ROM_LOAD16_BYTE( "ghs07_1.bin", 0x000001, 0x020000, CRC(62e04e1c) SHA1(c52ed2a116eb2d9057449cc42e7c1d97ad92521d) )
	ROM_LOAD16_BYTE( "ghs07_2.bin", 0x040000, 0x020000, CRC(b6983794) SHA1(29e7c59c6a004d6f05cacf52315c8595748113a3) )
	ROM_LOAD16_BYTE( "ghs07_3.bin", 0x040001, 0x020000, CRC(80888695) SHA1(56896a99a5b7db590e30c7275180d9e5dc892fe6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokeg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs08_0.bin", 0x000000, 0x020000, CRC(e121f95a) SHA1(497bc333fd36416a7583170ec60bf8e0dd46bbbf) )
	ROM_LOAD16_BYTE( "ghs08_1.bin", 0x000001, 0x020000, CRC(897e1a24) SHA1(d9704372e737418aeb6c36b5441f620f3d61c50e) )
	ROM_LOAD16_BYTE( "ghs08_2.bin", 0x040000, 0x020000, CRC(b6983794) SHA1(29e7c59c6a004d6f05cacf52315c8595748113a3) )
	ROM_LOAD16_BYTE( "ghs08_3.bin", 0x040001, 0x020000, CRC(80888695) SHA1(56896a99a5b7db590e30c7275180d9e5dc892fe6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokeh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs09_0.bin", 0x000000, 0x020000, CRC(6e56b7a5) SHA1(d1cb2b4b40e950ca3be6a44f1a03d8e0d357c35f) )
	ROM_LOAD16_BYTE( "ghs09_1.bin", 0x000001, 0x020000, CRC(5e4c2a31) SHA1(740e3625e6c035f50da912957d7a25e1c66eb617) )
	ROM_LOAD16_BYTE( "ghs09_2.bin", 0x040000, 0x020000, CRC(b5e00b7a) SHA1(6d0d5fdd4378ec7a323fa41b65fd7dc645ec948c) )
	ROM_LOAD16_BYTE( "ghs09_3.bin", 0x040001, 0x020000, CRC(2193d414) SHA1(4db02bf3a8c76f0fbe55fc6531c9fa9027cd1f06) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokei )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ghs10_0.bin", 0x000000, 0x020000, CRC(6e56b7a5) SHA1(d1cb2b4b40e950ca3be6a44f1a03d8e0d357c35f) )
	ROM_LOAD16_BYTE( "ghs10_1.bin", 0x000001, 0x020000, CRC(b5d27e09) SHA1(02778f7ee009b1c3affb1a9abcca39f04c9d8925) )
	ROM_LOAD16_BYTE( "ghs10_2.bin", 0x040000, 0x020000, CRC(b5e00b7a) SHA1(6d0d5fdd4378ec7a323fa41b65fd7dc645ec948c) )
	ROM_LOAD16_BYTE( "ghs10_3.bin", 0x040001, 0x020000, CRC(2193d414) SHA1(4db02bf3a8c76f0fbe55fc6531c9fa9027cd1f06) )
	MISSING_SOUND
ROM_END

ROM_START( ep_smokej )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "h smoke.g0", 0x000000, 0x020000, CRC(dc3d37f5) SHA1(f9bf1ff7f97044f66ded4886ad8e81ee47376c7e) )
	ROM_LOAD16_BYTE( "h smoke.g1", 0x000001, 0x020000, CRC(c6eba9e8) SHA1(4c79b9e193d3d5f0ceaaad0ba28ae35ef671d033) )
	ROM_LOAD16_BYTE( "h smoke.g2", 0x040000, 0x020000, CRC(28faa741) SHA1(97009cd81086a910a0888cb7ed0e61c55f8d4571) )
	ROM_LOAD16_BYTE( "h smoke.g3", 0x040001, 0x020000, CRC(bf4d199a) SHA1(c43597ff62f0609ad9faf576cc75f0aa7b583de7) )
	MISSING_SOUND
ROM_END






ROM_START( ep_ifern )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gin01_0.bin", 0x000000, 0x020000, CRC(e3ee387d) SHA1(56ddb892eb1b0164af7de012e302e97a1af3665d) )
	ROM_LOAD16_BYTE( "gin01_1.bin", 0x000001, 0x020000, CRC(12620b00) SHA1(89b4c1af5bc910a3de22e2693364bed12ae172b3) )
	ROM_LOAD16_BYTE( "gin01_2.bin", 0x040000, 0x020000, CRC(f18801c6) SHA1(9eb5f20690ace1dcef6ca069e47efed656c51b64) )
	ROM_LOAD16_BYTE( "gin01_3.bin", 0x040001, 0x020000, CRC(6d7f6355) SHA1(699060d4fa91dcf051cca49d39d18521a10b7eff) )
	MISSING_SOUND
ROM_END

ROM_START( ep_iferna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gin02_0.bin", 0x000000, 0x020000, CRC(e3ee387d) SHA1(56ddb892eb1b0164af7de012e302e97a1af3665d) )
	ROM_LOAD16_BYTE( "gin02_1.bin", 0x000001, 0x020000, CRC(f9fc5f38) SHA1(9b5767704d91c3b1f477ea0fa67dca15d92b31f7) )
	ROM_LOAD16_BYTE( "gin02_2.bin", 0x040000, 0x020000, CRC(f18801c6) SHA1(9eb5f20690ace1dcef6ca069e47efed656c51b64) )
	ROM_LOAD16_BYTE( "gin02_3.bin", 0x040001, 0x020000, CRC(6d7f6355) SHA1(699060d4fa91dcf051cca49d39d18521a10b7eff) )
	MISSING_SOUND
ROM_END



ROM_START( ep_kopcl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "knoc2-0n.p1", 0x000000, 0x020000, CRC(570cc4f3) SHA1(d7e06185b64100a72b582d613cfc8d0d9d8b8e52) )
	ROM_LOAD16_BYTE( "knoc2-0n.p2", 0x000001, 0x020000, CRC(3a801dbb) SHA1(916e29f9c73a97f56bfaf2e02e1559dc88ae04b0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_kopcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "knoc2-0p.p1", 0x000000, 0x020000, CRC(b26ec5d0) SHA1(daece1facd10346cd128bf8e604ed75449b14444) )
	ROM_LOAD16_BYTE( "knoc2-0p.p2", 0x000001, 0x020000, CRC(3a801dbb) SHA1(916e29f9c73a97f56bfaf2e02e1559dc88ae04b0) )
	MISSING_SOUND
ROM_END

ROM_START( ep_kopclb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ko_punch.g0", 0x000000, 0x020000, CRC(03e371c7) SHA1(d34091d27800fd3d681f74f85141bc313afe3186) )
	ROM_LOAD16_BYTE( "ko_punch.g1", 0x000001, 0x020000, CRC(e495b309) SHA1(5daed67ed107df904aa64bad8ff5d73b23a8d904) )
	MISSING_SOUND
ROM_END





ROM_START( ep_ltt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "thrl1-2n.p1", 0x000000, 0x020000, CRC(b44687ba) SHA1(817d15307772f73e6b66ab708ecf353d8248a030) )
	ROM_LOAD16_BYTE( "thrl1-2n.p2", 0x000001, 0x020000, CRC(98843c71) SHA1(af4dd8000e74ada26e9e92d82b884446476f98d8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_ltta )
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "thrl1-2p.p1", 0x000000, 0x020000, CRC(b44687ba) SHA1(817d15307772f73e6b66ab708ecf353d8248a030) )
	ROM_LOAD16_BYTE( "thrl1-2p.p2", 0x000001, 0x100000, CRC(a93c8b13) SHA1(a27aeac98295bb615cb3662a7a3cde84f107b6d6) ) // overdump?
	MISSING_SOUND
ROM_END





ROM_START( ep_ll )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "lula03.p0", 0x000000, 0x020000, CRC(dfacff4d) SHA1(39a9dd338e88bd7d92c2aa3837769486bd2e2ec1) )
	ROM_LOAD16_BYTE( "lula03.p1", 0x000001, 0x020000, CRC(4436fe62) SHA1(106a198c4e49cfcddfc7ee193e521d77f3dc667c) )
	ROM_LOAD16_BYTE( "lula03.p2", 0x040000, 0x020000, CRC(20a03718) SHA1(e08e31703ff3558f34c266b865e78e7815d25af8) )
	ROM_LOAD16_BYTE( "lula03.p3", 0x040001, 0x020000, CRC(2891b68f) SHA1(f4cc6836cbac8602cd32f3ccd1904f22a04d2899) )
	MISSING_SOUND
ROM_END

ROM_START( ep_lla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "lula03p.p0", 0x000000, 0x020000, CRC(dfacff4d) SHA1(39a9dd338e88bd7d92c2aa3837769486bd2e2ec1) )
	ROM_LOAD16_BYTE( "lula03p.p1", 0x000001, 0x020000, CRC(5ba50c19) SHA1(fe0326f9b0d4243492cbe66ff08e6e8fb7292897) )
	ROM_LOAD16_BYTE( "lula03p.p2", 0x040000, 0x020000, CRC(20a03718) SHA1(e08e31703ff3558f34c266b865e78e7815d25af8) )
	ROM_LOAD16_BYTE( "lula03p.p3", 0x040001, 0x020000, CRC(2891b68f) SHA1(f4cc6836cbac8602cd32f3ccd1904f22a04d2899) )
	MISSING_SOUND
ROM_END


#define EP_RLGDT_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "rlgdt1_0.s0", 0x000000, 0x100000, CRC(f20867b8) SHA1(13bd978dca616599b3768da482a75fc524c147ce) ) \
	ROM_LOAD( "rlgdt1_0.s1", 0x080000, 0x100000, CRC(54969491) SHA1(b689da0ec62a7eefa2d9758cddae67a5bdf0c4c7) )

ROM_START( ep_rlgdt )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "rlgdt1_0.g0", 0x000000, 0x020000, CRC(3133ed91) SHA1(62d665a7ad58686c2b9d6486118dce92de7cbd0d) )
	ROM_LOAD16_BYTE( "rlgdt1_0.g1", 0x000001, 0x020000, CRC(7f08a205) SHA1(5166d3ff781b645fa02fb7be96a926c65e5b2a93) )
	EP_RLGDT_SOUND
ROM_END






ROM_START( ep_monsh )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg02430.g0", 0x000000, 0x020000, CRC(25a036e6) SHA1(12445a8d5b9799beac1681ee957fd08f73cc4eb8) )
	ROM_LOAD16_BYTE( "gg02430.g1", 0x000001, 0x020000, CRC(41d76ef1) SHA1(abdbeb838120c19c090491213122852fed5a7537) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monsha )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg02431.g0", 0x000000, 0x020000, CRC(3963ec9c) SHA1(2e1c6c200c03a881bda892ca8dbf8edba51d99aa) )
	ROM_LOAD16_BYTE( "gg02431.g1", 0x000001, 0x020000, CRC(41d76ef1) SHA1(abdbeb838120c19c090491213122852fed5a7537) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monshb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg03452.g0", 0x000000, 0x020000, CRC(8b3cf60b) SHA1(a7499a769537f434e04c28cf09d0fa571713e462) )
	ROM_LOAD16_BYTE( "gg03452.g1", 0x000001, 0x020000, CRC(3bbb3d24) SHA1(2543375d3a446e62e3974f0ab1950d9ff49cbaf5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monshc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg03453.g0", 0x000000, 0x020000, CRC(cde52d81) SHA1(6511bf1ec530fd6f168569710ac75de7170422e2) )
	ROM_LOAD16_BYTE( "gg03453.g1", 0x000001, 0x020000, CRC(3bbb3d24) SHA1(2543375d3a446e62e3974f0ab1950d9ff49cbaf5) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monshd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg03476.g0", 0x000000, 0x020000, CRC(88736638) SHA1(0dc7057c5add2309115d7623237dffd2661abba6) )
	ROM_LOAD16_BYTE( "gg03476.g1", 0x000001, 0x020000, CRC(fadae0b6) SHA1(0289a1f6820be5f8f80f31b438e95d125870ebf6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monshe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gg03477.g0", 0x000000, 0x020000, CRC(6d1d86d1) SHA1(ee495fade242a3ee1e2122cc4c51f81dbce1f219) )
	ROM_LOAD16_BYTE( "gg03477.g1", 0x000001, 0x020000, CRC(fadae0b6) SHA1(0289a1f6820be5f8f80f31b438e95d125870ebf6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monshf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "moonshin.g0", 0x000000, 0x020000, CRC(cfbd4301) SHA1(98a9d44d1a6821453479f5dc45ed37a12349a040) )
	ROM_LOAD16_BYTE( "moonshin.g1", 0x000001, 0x020000, CRC(1bdb323e) SHA1(d44c0bd0526ef45182171df3230d387d3c58e968) )
	MISSING_SOUND
ROM_END




ROM_START( ep_mummy )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmt02_0.bin", 0x000000, 0x020000, CRC(55310f3f) SHA1(1d7816724cb21580b4dac9c01cbb6c7d80c6d4c3) )
	ROM_LOAD16_BYTE( "gmt02_1.bin", 0x000001, 0x020000, CRC(9cb83e4e) SHA1(0f9cce89ea5ef8deb57d7afb6a5cd8a5628fbb93) )
	ROM_LOAD16_BYTE( "gmt02_2.bin", 0x040000, 0x020000, CRC(d9c969b6) SHA1(be151f43a783b46dffdbe3afaa396e61b95c0f4f) )
	ROM_LOAD16_BYTE( "gmt02_3.bin", 0x040001, 0x020000, CRC(6520bdb9) SHA1(90120121b90877423b6c48cf65ebd691b6a61403) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmt03_0.bin", 0x000000, 0x020000, CRC(55310f3f) SHA1(1d7816724cb21580b4dac9c01cbb6c7d80c6d4c3) )
	ROM_LOAD16_BYTE( "gmt03_1.bin", 0x000001, 0x020000, CRC(77266a76) SHA1(c4f2d8814021fe3556466234825fe9e16f3b9b14) )
	ROM_LOAD16_BYTE( "gmt03_2.bin", 0x040000, 0x020000, CRC(d9c969b6) SHA1(be151f43a783b46dffdbe3afaa396e61b95c0f4f) )
	ROM_LOAD16_BYTE( "gmt03_3.bin", 0x040001, 0x020000, CRC(6520bdb9) SHA1(90120121b90877423b6c48cf65ebd691b6a61403) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummyb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmt06_0.bin", 0x000000, 0x020000, CRC(4bc654a3) SHA1(94eb79244d44d643be825dcb9ef7848d110815ac) )
	ROM_LOAD16_BYTE( "gmt06_1.bin", 0x000001, 0x020000, CRC(a9fc1e2b) SHA1(dc787d6a2845c0bb48239a12ab9a631c68939683) )
	ROM_LOAD16_BYTE( "gmt06_2.bin", 0x040000, 0x020000, CRC(c67de26e) SHA1(25ceb20f132b0baa1f8b9ffbfa483a9d350ddb87) )
	ROM_LOAD16_BYTE( "gmt06_3.bin", 0x040001, 0x020000, CRC(305cfb61) SHA1(40f54f2cd54d42ce5d74fa1689f14f3a6f97466f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummyc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gmt07_0.bin", 0x000000, 0x020000, CRC(4bc654a3) SHA1(94eb79244d44d643be825dcb9ef7848d110815ac) )
	ROM_LOAD16_BYTE( "gmt07_1.bin", 0x000001, 0x020000, CRC(42624a13) SHA1(c3c20a8c6fffd120abcdfece9676d0db08609666) )
	ROM_LOAD16_BYTE( "gmt07_2.bin", 0x040000, 0x020000, CRC(c67de26e) SHA1(25ceb20f132b0baa1f8b9ffbfa483a9d350ddb87) )
	ROM_LOAD16_BYTE( "gmt07_3.bin", 0x040001, 0x020000, CRC(305cfb61) SHA1(40f54f2cd54d42ce5d74fa1689f14f3a6f97466f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummyd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02nmt04.g0.bin", 0x000000, 0x020000, CRC(57d828d5) SHA1(169958de3c7b26891c4e1e66f58b49e61bd0e521) )
	ROM_LOAD16_BYTE( "i02nmt04.g1.bin", 0x000001, 0x020000, CRC(3649a285) SHA1(a69a4299fe23e3b6b5aeb4409071ce6156eeca96) )
	ROM_LOAD16_BYTE( "i02nmt04.g2.bin", 0x040000, 0x020000, CRC(d1b74cc3) SHA1(58f54cf9ab15d3f0012adaf11d3211440534d612) )
	ROM_LOAD16_BYTE( "i02nmt04.g3.bin", 0x040001, 0x020000, CRC(29a59239) SHA1(e526b89e6cf4be62c0e2761eff48b8c2455e03f7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummye )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i02pmt05.g0.bin", 0x000000, 0x020000, CRC(57d828d5) SHA1(169958de3c7b26891c4e1e66f58b49e61bd0e521) )
	ROM_LOAD16_BYTE( "i02pmt05.g1.bin", 0x000001, 0x020000, CRC(ddd7f6bd) SHA1(39ebd89ccc7ce1d7dc27db4843b7f5ce4bc8a004) )
	ROM_LOAD16_BYTE( "i02pmt05.g2.bin", 0x040000, 0x020000, CRC(d1b74cc3) SHA1(58f54cf9ab15d3f0012adaf11d3211440534d612) )
	ROM_LOAD16_BYTE( "i02pmt05.g3.bin", 0x040001, 0x020000, CRC(29a59239) SHA1(e526b89e6cf4be62c0e2761eff48b8c2455e03f7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_mummyf ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "mummytal.g0", 0x000000, 0x080000, CRC(443c5607) SHA1(5c83124ebdde55608ffd94707692e005e12d5bde) )
	ROM_LOAD16_BYTE( "mummytal.g1", 0x000001, 0x080000, CRC(d5d9b807) SHA1(da931bc562a86d6957729d81cf49b4d651d91024) )
	ROM_LOAD16_BYTE( "mummytal.g2", 0x100000, 0x080000, CRC(7237b6f9) SHA1(f89e894868f3ac0c84c333b853bd0a8cf6f728f2) )
	ROM_LOAD16_BYTE( "mummytal.g3", 0x100001, 0x080000, CRC(8569eb03) SHA1(8e13ad5fda3c95a99e684ce931d05d5b1ad48d95) )
	MISSING_SOUND
ROM_END









ROM_START( ep_redl ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "reli01.p0", 0x000000, 0x100000, CRC(457e3ba5) SHA1(184a3a3879f18ab58811ab1f6e0c4f59ddd4dec3) )
	ROM_LOAD16_BYTE( "reli01.p1", 0x000001, 0x100000, CRC(1548b52a) SHA1(7d430f3b8c82f8e5ff6b43b33bcdb0eaeac6854a) )
	MISSING_SOUND
ROM_END

ROM_START( ep_redla ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "reli01p.p0", 0x000000, 0x100000, CRC(457e3ba5) SHA1(184a3a3879f18ab58811ab1f6e0c4f59ddd4dec3) )
	ROM_LOAD16_BYTE( "reli01p.p1", 0x000001, 0x100000, CRC(40a3d891) SHA1(22fe515fc735daa5e49f40f2d746359f0d2d3b7d) )
	MISSING_SOUND
ROM_END





ROM_START( ep_rchik )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "crcs1-1n.p0", 0x000000, 0x020000, CRC(80125d92) SHA1(8e976b2f8f5a33c4f7a70e82ab7b04ec1da221ac) )
	ROM_LOAD16_BYTE( "crcs1-1n.p1", 0x000001, 0x020000, CRC(4a0d6fae) SHA1(e032756c50dbe0c2317d39c60ad3f01f376756c9) )
	MISSING_SOUND
ROM_END

ROM_START( ep_rchika )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "crcs1-1p.p0", 0x000000, 0x020000, CRC(34b0b3f6) SHA1(fea45a97ec4004de4e836591d011b437d961e1b2) )
	ROM_LOAD16_BYTE( "crcs1-1p.p1", 0x000001, 0x020000, CRC(4a0d6fae) SHA1(e032756c50dbe0c2317d39c60ad3f01f376756c9) )
	MISSING_SOUND
ROM_END





ROM_START( ep_royrc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03ncr01.g0", 0x000000, 0x020000, CRC(ebd47fe5) SHA1(041e6d2341a9dd4b60f46b58361d305484d1b9d0) )
	ROM_LOAD16_BYTE( "i03ncr01.g1", 0x000001, 0x020000, CRC(cf1bcdea) SHA1(9fc6124d26ccfaaf82d7846f4c3935a46ca41015) )
	MISSING_SOUND
ROM_END

ROM_START( ep_royrca )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03pcr02.g0", 0x000000, 0x020000, CRC(4bd62a35) SHA1(335ba89fc918a454275fe49aa9a638d344011666) )
	ROM_LOAD16_BYTE( "i03pcr02.g1", 0x000001, 0x020000, CRC(cf1bcdea) SHA1(9fc6124d26ccfaaf82d7846f4c3935a46ca41015) )
	MISSING_SOUND
ROM_END







ROM_START( ep_sdcla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsp1-2n.p0", 0x000000, 0x020000, CRC(4647b782) SHA1(d3eda59820ec7133c7ee818a22a469bdab6984be) )
	ROM_LOAD16_BYTE( "clsp1-2n.p1", 0x000001, 0x020000, CRC(3fc72f6b) SHA1(3791c000e03692628c47fae67f0f161c673223b6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclaa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsp1-2p.p0", 0x000000, 0x020000, CRC(fa24a913) SHA1(fbdd7b949da8572beb7572d977b892291833f295) )
	ROM_LOAD16_BYTE( "clsp1-2p.p1", 0x000001, 0x020000, CRC(5312296b) SHA1(4cf44fddb759d3ec54cba1f57ecfd145319f887f) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclab )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsp1-5n.p0", 0x000000, 0x020000, CRC(eb0e5e3e) SHA1(4fdadf738bdd069667fb1110542881ceeedd023e) )
	ROM_LOAD16_BYTE( "clsp1-5n.p1", 0x000001, 0x020000, CRC(631936bb) SHA1(9937bedf1bf83cce9b189c449f65a87929ae1452) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclac )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "clsp1-5p.p0", 0x000000, 0x020000, CRC(576d40af) SHA1(7bd524d109093026aaceca09b7308f394315387c) )
	ROM_LOAD16_BYTE( "clsp1-5p.p1", 0x000001, 0x020000, CRC(99a39e99) SHA1(72c6c41e7c166936ac8d5c4438d68cbe6ade212b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclad )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cs251-1n.p0", 0x000000, 0x100000, CRC(0e76093c) SHA1(78a14ed3a16c376a3ebb1f7a14f769ce49a7c407) )
	ROM_LOAD16_BYTE( "cs251-1n.p1", 0x000001, 0x100000, CRC(cad2bff2) SHA1(47f599d3874b3f9ed64755816ad470aa18764555) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclae )
	ROM_REGION( 0x200000, "maincpu", 0 ) // why are these larger?
	ROM_LOAD16_BYTE( "cs251-1p.p0", 0x000000, 0x100000, CRC(9a925aa6) SHA1(e606a737508f24e913ed1d7ae4e0967fff8a0619) )
	ROM_LOAD16_BYTE( "cs251-1p.p1", 0x000001, 0x100000, CRC(133688dc) SHA1(78d367bf35cd047e7400cdbf984d07129c5a59fb) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclaf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cs251-2n.p0", 0x000000, 0x020000, CRC(59a12087) SHA1(21254f2508a7171e4724d0408063d37d78e297a1) )
	ROM_LOAD16_BYTE( "cs251-2n.p1", 0x000001, 0x020000, CRC(713b61bd) SHA1(e86aef1b65e6dba11e1d040455e355f069733bf8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_sdclag )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cs251-2p.p0", 0x000000, 0x020000, CRC(e5c23e16) SHA1(b5984891261fb80d5d82e1c42bfb3b565b4cb9fe) )
	ROM_LOAD16_BYTE( "cs251-2p.p1", 0x000001, 0x020000, CRC(247367f6) SHA1(01a6f9854a7adb7bbbb06c043bee7a0882d47f51) )
	MISSING_SOUND
ROM_END




ROM_START( ep_stm )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "stor0-7n.p0", 0x000000, 0x020000, CRC(afca552e) SHA1(a3685d9f3a730c897858ae233496d6de00bb64ba) )
	ROM_LOAD16_BYTE( "stor0-7n.p1", 0x000001, 0x020000, CRC(9673858a) SHA1(760c6fc5881253d95123c101ec6e0e6cad0af71e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_stma )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "stor0-7p.p0", 0x000000, 0x020000, CRC(13a94bbf) SHA1(73a7fdac7737ef020b2e4d3bd881ea76bc2ebac3) )
	ROM_LOAD16_BYTE( "stor0-7p.p1", 0x000001, 0x020000, CRC(1e258852) SHA1(569390c676777d6b3c05d83041f49b1255031e2b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_stmb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "stor1-0n.p0", 0x000000, 0x020000, CRC(cadbf7ac) SHA1(a1fe7d545882ca6f6e24962c6cadf284fc7c7ed6) )
	ROM_LOAD16_BYTE( "stor1-0n.p1", 0x000001, 0x020000, CRC(fe3c319d) SHA1(427aa98d5263bebe3b6cae9c704813c996a19b26) )
	MISSING_SOUND
ROM_END

ROM_START( ep_stmc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "stor1-0p.p0", 0x000000, 0x020000, CRC(d50fd25e) SHA1(ea3071d32ee4363a733c43d553c89d07fc3b266d) )
	ROM_LOAD16_BYTE( "stor1-0p.p1", 0x000001, 0x020000, CRC(0157e2b4) SHA1(ad98bcefb23683b180df9693a4d267ddf85c31bb) )
	MISSING_SOUND
ROM_END










ROM_START( ep_snset ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sust01.p0", 0x000000, 0x100000, CRC(2c400a0a) SHA1(d40f4ad0c13aa78399120c802836d2ccc2bd4096) )
	ROM_LOAD16_BYTE( "sust01.p1", 0x000001, 0x100000, CRC(f9e8c839) SHA1(9a88ce7a5f81d39bb2077ec4c454e6d7e62960ed) )
	MISSING_SOUND
ROM_END

ROM_START( ep_snseta ) // why are these larger?
	ROM_REGION( 0x200000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sust01p.p0", 0x000000, 0x100000, CRC(2c400a0a) SHA1(d40f4ad0c13aa78399120c802836d2ccc2bd4096) )
	ROM_LOAD16_BYTE( "sust01p.p1", 0x000001, 0x100000, CRC(ac03a582) SHA1(07709fce3e65a43a938dbede38fe97ee4dc856f7) )
	MISSING_SOUND
ROM_END


#define EP_TREAS_SOUND \
	ROM_REGION( 0x100000, "ymz", 0 ) \
	ROM_LOAD( "treas1_6.s0", 0x000000, 0x080000, CRC(df792dc0) SHA1(96647eb579a3a60f1a4a2bea53d7a13030838437) )

ROM_START( ep_treas )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "treas1_6.g0", 0x000000, 0x020000, CRC(5b8bab5d) SHA1(a44254d468309f536a1e1e35536328bf3bc02704) )
	ROM_LOAD16_BYTE( "treas1_6.g1", 0x000001, 0x020000, CRC(752117ed) SHA1(3c8ddf8acd1307749a03e90bd2d76f0b81d6e388) )
	ROM_LOAD16_BYTE( "treas1_6.g2", 0x040000, 0x020000, CRC(d4f61c7e) SHA1(4f62fca291fcc4b24c3e3bbbbd6b0f5216f3e0ef) )
	ROM_LOAD16_BYTE( "treas1_6.g3", 0x040001, 0x020000, CRC(8c287556) SHA1(e5c9dc34381c94d3ef4474f47fef119066ee380b) )
	EP_TREAS_SOUND
ROM_END

ROM_START( ep_twarp )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "twrp01.p0", 0x000000, 0x020000, CRC(eacf4d3e) SHA1(7a5ab3042aec0457139e2ac485b455a2ba0d9c24) )
	ROM_LOAD16_BYTE( "twrp01.p1", 0x000001, 0x020000, CRC(3ba9f21e) SHA1(b6c7aa1e926f856f665ed0811e81cab6668b9ab3) )
	ROM_LOAD16_BYTE( "twrp01.p2", 0x040000, 0x020000, CRC(a3d3e515) SHA1(cccab33833954ab7b218c6cb6963a34adc8d309d) )
	ROM_LOAD16_BYTE( "twrp01.p3", 0x040001, 0x020000, CRC(bdc69161) SHA1(facf7d24a29508c2945216f07ad117481b8a05dd) )
	MISSING_SOUND
ROM_END

ROM_START( ep_twarpa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "twrp01p.p0", 0x000000, 0x020000, CRC(eacf4d3e) SHA1(7a5ab3042aec0457139e2ac485b455a2ba0d9c24) )
	ROM_LOAD16_BYTE( "twrp01p.p1", 0x000001, 0x020000, CRC(243a0065) SHA1(e77957d6ba255c4b20252590cbcb903f07ac4d70) )
	ROM_LOAD16_BYTE( "twrp01p.p2", 0x040000, 0x020000, CRC(a3d3e515) SHA1(cccab33833954ab7b218c6cb6963a34adc8d309d) )
	ROM_LOAD16_BYTE( "twrp01p.p3", 0x040001, 0x020000, CRC(bdc69161) SHA1(facf7d24a29508c2945216f07ad117481b8a05dd) )
	MISSING_SOUND
ROM_END

ROM_START( ep_twarpb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "twrp04.p0", 0x000000, 0x020000, CRC(d2acb2e4) SHA1(4685396059ec2de8f2c077e68431be9d3d59dbf6) )
	ROM_LOAD16_BYTE( "twrp04.p1", 0x000001, 0x020000, CRC(182e60c4) SHA1(e82036152eb7eb41d9f6f532b2f80c918716628a) )
	ROM_LOAD16_BYTE( "twrp04.p2", 0x040000, 0x020000, CRC(3d6d1d59) SHA1(4f4d8a4ad314819a66ad7c753934e3e887458ba1) )
	ROM_LOAD16_BYTE( "twrp04.p3", 0x040001, 0x020000, CRC(1dfb0bce) SHA1(14b680249418a10dc828bc02832275eaacd1b9f6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_twarpc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "twrp04p.p0", 0x000000, 0x020000, CRC(d2acb2e4) SHA1(4685396059ec2de8f2c077e68431be9d3d59dbf6) )
	ROM_LOAD16_BYTE( "twrp04p.p1", 0x000001, 0x020000, CRC(07bd92bf) SHA1(529890044d9592abffc965c76a8fb22bbca2deac) )
	ROM_LOAD16_BYTE( "twrp04p.p2", 0x040000, 0x020000, CRC(3d6d1d59) SHA1(4f4d8a4ad314819a66ad7c753934e3e887458ba1) )
	ROM_LOAD16_BYTE( "twrp04p.p3", 0x040001, 0x020000, CRC(1dfb0bce) SHA1(14b680249418a10dc828bc02832275eaacd1b9f6) )
	MISSING_SOUND
ROM_END





ROM_START( ep_tod )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dare1-3n.p0", 0x000000, 0x020000, CRC(bca2fc2b) SHA1(dbd34991116254ec6ea7c0bbe66a1857da5ee4fe) )
	ROM_LOAD16_BYTE( "dare1-3n.p1", 0x000001, 0x020000, CRC(bb771f06) SHA1(1bbf75898f326bf71f6d1810c4d4821bd33fe904) )
	MISSING_SOUND
ROM_END

ROM_START( ep_toda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "dare1-3p.p0", 0x000000, 0x020000, CRC(3f69a95e) SHA1(faa727fcedf4b53a8109127c8dca18d0623a595c) )
	ROM_LOAD16_BYTE( "dare1-3p.p1", 0x000001, 0x020000, CRC(731c117d) SHA1(2e13213f00bf469ef48ead88aecd3d1613df7409) )
	MISSING_SOUND
ROM_END









ROM_START( ep_utncl )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "nutt1-9n.p0", 0x000000, 0x020000, CRC(18962918) SHA1(bdd3017e89240d8ef252b16168b807ac4c74754d) )
	ROM_LOAD16_BYTE( "nutt1-9n.p1", 0x000001, 0x020000, CRC(bb4135c5) SHA1(6fac94181eade7d8904cef6210e3db0c6c399c65) )
	MISSING_SOUND
ROM_END

ROM_START( ep_utncla )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "nutt1-9p.p0", 0x000000, 0x020000, CRC(a4f53789) SHA1(b6dc7cc25eabbcd8b1d40d7131b39c5ecdcbf609) )
	ROM_LOAD16_BYTE( "nutt1-9p.p1", 0x000001, 0x020000, CRC(93924e6a) SHA1(2e4fe44a519f0a18f66423c3c8f0ddf14d82b0b8) )
	MISSING_SOUND
ROM_END






ROM_START( ep_wleek )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "leek0-3n.p0", 0x000000, 0x020000, CRC(a59144ee) SHA1(771973aa6bac68a03884137e6869bdd16f5d4816) )
	ROM_LOAD16_BYTE( "leek0-3n.p1", 0x000001, 0x020000, CRC(45f62998) SHA1(5b9b475d56cdd39e13da034aebb6a863056a28ef) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wleeka )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "leek0-3p.p0", 0x000000, 0x020000, CRC(5b254963) SHA1(73680418b17a3d1419a107cf9d4ec9380bfcf7ca) )
	ROM_LOAD16_BYTE( "leek0-3p.p1", 0x000001, 0x020000, CRC(45f62998) SHA1(5b9b475d56cdd39e13da034aebb6a863056a28ef) )
	MISSING_SOUND
ROM_END





ROM_START( ep_wud )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd01_0.bin", 0x000000, 0x020000, CRC(f15700e0) SHA1(398abf6c40db8a6b30c65cf3be626ab4eb963101) )
	ROM_LOAD16_BYTE( "gwd01_1.bin", 0x000001, 0x020000, CRC(5f91ad6e) SHA1(de930e174259b00a83aa7e6795e4dbee9154a6a3) )
	ROM_LOAD16_BYTE( "gwd01_2.bin", 0x040000, 0x020000, CRC(29cd8ebc) SHA1(9e50523b209c6f575d9593fbf86046168d63be79) )
	ROM_LOAD16_BYTE( "gwd01_3.bin", 0x040001, 0x020000, CRC(dbbddf5a) SHA1(dbd610ec89216882ec8f1c2a32d035454a61ecb8) )
ROM_END


ROM_START( ep_wuda )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd02_0.bin", 0x000000, 0x020000, CRC(f15700e0) SHA1(398abf6c40db8a6b30c65cf3be626ab4eb963101) )
	ROM_LOAD16_BYTE( "gwd02_1.bin", 0x000001, 0x020000, CRC(b40ff956) SHA1(e83fed7bea228119742c526190124893f1f328cf) )
	ROM_LOAD16_BYTE( "gwd02_2.bin", 0x040000, 0x020000, CRC(29cd8ebc) SHA1(9e50523b209c6f575d9593fbf86046168d63be79) )
	ROM_LOAD16_BYTE( "gwd02_3.bin", 0x040001, 0x020000, CRC(dbbddf5a) SHA1(dbd610ec89216882ec8f1c2a32d035454a61ecb8) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wudb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd03_0.bin", 0x000000, 0x020000, CRC(84ec3bb4) SHA1(c291aea754659b686d599be96cf62bb5ee6f3c18) )
	ROM_LOAD16_BYTE( "gwd03_1.bin", 0x000001, 0x020000, CRC(c531f2d7) SHA1(9fbb9ed0102e0e71ba8fbc8c1045909da66a637c) )
	ROM_LOAD16_BYTE( "gwd03_2.bin", 0x040000, 0x020000, CRC(8f60c1d0) SHA1(3f8ce3001b79fe718b3ed53ab6949bcd315630c4) )
	ROM_LOAD16_BYTE( "gwd03_3.bin", 0x040001, 0x020000, CRC(edbec84f) SHA1(29f5a54b7c204a3830704671d4a350fcc96abab7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wudc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd04_0.bin", 0x000000, 0x020000, CRC(84ec3bb4) SHA1(c291aea754659b686d599be96cf62bb5ee6f3c18) )
	ROM_LOAD16_BYTE( "gwd04_1.bin", 0x000001, 0x020000, CRC(2eafa6ef) SHA1(24b8424304df778309f1c75aa760c883d80a0570) )
	ROM_LOAD16_BYTE( "gwd04_2.bin", 0x040000, 0x020000, CRC(8f60c1d0) SHA1(3f8ce3001b79fe718b3ed53ab6949bcd315630c4) )
	ROM_LOAD16_BYTE( "gwd04_3.bin", 0x040001, 0x020000, CRC(edbec84f) SHA1(29f5a54b7c204a3830704671d4a350fcc96abab7) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wudd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd05_0.bin", 0x000000, 0x020000, CRC(4dac7bb4) SHA1(5d135072b6916531a8dc46acfb5f6d37bbaa051d) )
	ROM_LOAD16_BYTE( "gwd05_1.bin", 0x000001, 0x020000, CRC(7696dfcf) SHA1(88b007e6a9a035176f6a81afa965cd3fe2411887) )
	ROM_LOAD16_BYTE( "gwd05_2.bin", 0x040000, 0x020000, CRC(4ebc551e) SHA1(91aaacfd2fb5434337f62f33f108ea8477d74fe2) )
	ROM_LOAD16_BYTE( "gwd05_3.bin", 0x040001, 0x020000, CRC(2d27ec13) SHA1(4757fc1666a64e570d3c977c40c3606e4389069e) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wude )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwd06_0.bin", 0x000000, 0x020000, CRC(4dac7bb4) SHA1(5d135072b6916531a8dc46acfb5f6d37bbaa051d) )
	ROM_LOAD16_BYTE( "gwd06_1.bin", 0x000001, 0x020000, CRC(9d088bf7) SHA1(99fd731530a9ad06ad95e268ca8f07c457950e43) )
	ROM_LOAD16_BYTE( "gwd06_2.bin", 0x040000, 0x020000, CRC(4ebc551e) SHA1(91aaacfd2fb5434337f62f33f108ea8477d74fe2) )
	ROM_LOAD16_BYTE( "gwd06_3.bin", 0x040001, 0x020000, CRC(2d27ec13) SHA1(4757fc1666a64e570d3c977c40c3606e4389069e) )
	MISSING_SOUND
ROM_END







ROM_START( ep_wf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwf07_0.bin", 0x000000, 0x020000, CRC(c375f87a) SHA1(1775a7da4ff10ab568252165356b38e18e805516) )
	ROM_LOAD16_BYTE( "gwf07_1.bin", 0x000001, 0x020000, CRC(2173c52b) SHA1(b820e8acc7c454aa119888f9adbe0f689bddf908) )
	ROM_LOAD16_BYTE( "gwf07_2.bin", 0x040000, 0x020000, CRC(d071a22c) SHA1(87c82c991266d7b1024c0f5bd4e5991b1106afed) )
	ROM_LOAD16_BYTE( "gwf07_3.bin", 0x040001, 0x020000, CRC(7899a20e) SHA1(beee53847e1d48a22d21b39b3e53a1c9277f90b6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gwf08_0.bin", 0x000000, 0x020000, CRC(c375f87a) SHA1(1775a7da4ff10ab568252165356b38e18e805516) )
	ROM_LOAD16_BYTE( "gwf08_1.bin", 0x000001, 0x020000, CRC(caed9113) SHA1(dfa0294af0cb220a1acf4859dd6484ad0e328a33) )
	ROM_LOAD16_BYTE( "gwf08_2.bin", 0x040000, 0x020000, CRC(d071a22c) SHA1(87c82c991266d7b1024c0f5bd4e5991b1106afed) )
	ROM_LOAD16_BYTE( "gwf08_3.bin", 0x040001, 0x020000, CRC(7899a20e) SHA1(beee53847e1d48a22d21b39b3e53a1c9277f90b6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-1n.p0", 0x000000, 0x020000, CRC(aeac646d) SHA1(e2671dda3702146ad6911295fdb81cf73cff4ad8) )
	ROM_LOAD16_BYTE( "wild4-1n.p1", 0x000001, 0x020000, CRC(797272b0) SHA1(855374e9ae9ca23bfca460cda3554dec06c11d18) )
	ROM_LOAD16_BYTE( "wild4-1n.p2", 0x040000, 0x020000, CRC(c26ffc24) SHA1(8a7c165e7d9e37dfd96a8923a1d0b9b985bc3489) )
	ROM_LOAD16_BYTE( "wild4-1n.p3", 0x040001, 0x020000, CRC(e0646d97) SHA1(462db6cba653cfdfc18db4796bd55de0982f55ba) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-1p.p0", 0x000000, 0x020000, CRC(aeac646d) SHA1(e2671dda3702146ad6911295fdb81cf73cff4ad8) )
	ROM_LOAD16_BYTE( "wild4-1p.p1", 0x000001, 0x020000, CRC(92ec2688) SHA1(27741963e5d7573c5bbcfa5305489eb9a2269d84) )
	ROM_LOAD16_BYTE( "wild4-1p.p2", 0x040000, 0x020000, CRC(c26ffc24) SHA1(8a7c165e7d9e37dfd96a8923a1d0b9b985bc3489) )
	ROM_LOAD16_BYTE( "wild4-1p.p3", 0x040001, 0x020000, CRC(e0646d97) SHA1(462db6cba653cfdfc18db4796bd55de0982f55ba) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfd )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-2n.p0", 0x000000, 0x020000, CRC(76e492bd) SHA1(67cb66ad4f8a44a5aa98149fe878ac3b205eb411) )
	ROM_LOAD16_BYTE( "wild4-2n.p1", 0x000001, 0x020000, CRC(0a94d80f) SHA1(498c14db4d3b81b8f71338c84553e91317afed8d) )
	ROM_LOAD16_BYTE( "wild4-2n.p2", 0x040000, 0x020000, CRC(be0ca653) SHA1(eb8c2492347f446dcebaf9fe1af41ee7100d3d96) )
	ROM_LOAD16_BYTE( "wild4-2n.p3", 0x040001, 0x020000, CRC(3085630a) SHA1(ae7ff460313e1afc6befa8583d980818a29435a6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfe )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-2p.p0", 0x000000, 0x020000, CRC(76e492bd) SHA1(67cb66ad4f8a44a5aa98149fe878ac3b205eb411) )
	ROM_LOAD16_BYTE( "wild4-2p.p1", 0x000001, 0x020000, CRC(e10a8c37) SHA1(6afade398c3e63c7a5bdfbdc4dd7f520158f93fe) )
	ROM_LOAD16_BYTE( "wild4-2p.p2", 0x040000, 0x020000, CRC(be0ca653) SHA1(eb8c2492347f446dcebaf9fe1af41ee7100d3d96) )
	ROM_LOAD16_BYTE( "wild4-2p.p3", 0x040001, 0x020000, CRC(3085630a) SHA1(ae7ff460313e1afc6befa8583d980818a29435a6) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wff )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-3n.p0", 0x000000, 0x020000, CRC(dcea0746) SHA1(b8e672090cc6510efd2d7b86c8df9466c9cdd9ce) )
	ROM_LOAD16_BYTE( "wild4-3n.p1", 0x000001, 0x020000, CRC(d261ded7) SHA1(47bee1ac559c9c2742bf3f42a1cbbc036dc47d6c) )
	ROM_LOAD16_BYTE( "wild4-3n.p2", 0x040000, 0x020000, CRC(04028028) SHA1(e09b47f4b918a15319ae1edd5077ec36c3bad02a) )
	ROM_LOAD16_BYTE( "wild4-3n.p3", 0x040001, 0x020000, CRC(01773b64) SHA1(5bf746eff179b00b22bab629279309ea7487ed0d) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wfg )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wild4-3p.p0", 0x000000, 0x020000, CRC(dcea0746) SHA1(b8e672090cc6510efd2d7b86c8df9466c9cdd9ce) )
	ROM_LOAD16_BYTE( "wild4-3p.p1", 0x000001, 0x020000, CRC(39ff8aef) SHA1(df3c07891d053f8ab08a5826bb294375bf7b3e3a) )
	ROM_LOAD16_BYTE( "wild4-3p.p2", 0x040000, 0x020000, CRC(04028028) SHA1(e09b47f4b918a15319ae1edd5077ec36c3bad02a) )
	ROM_LOAD16_BYTE( "wild4-3p.p3", 0x040001, 0x020000, CRC(01773b64) SHA1(5bf746eff179b00b22bab629279309ea7487ed0d) )
	MISSING_SOUND
ROM_END





ROM_START( ep_bbars )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980023.g0", 0x000000, 0x020000, CRC(1e586e19) SHA1(41626ca7de77fa4a4c3a7ccc91ccc3d566d1a755) )
	ROM_LOAD16_BYTE( "sb980023.g1", 0x000001, 0x020000, CRC(8cafe00c) SHA1(2ca6f78b6752b9e4d4916063c98967342f54480c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bbarsa )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980024.g0", 0x000000, 0x020000, CRC(fb2a3a83) SHA1(7d7603e134520b1fb55a09810780f0c4a934ea5c) )
	ROM_LOAD16_BYTE( "sb980024.g1", 0x000001, 0x020000, CRC(cf59f55f) SHA1(14b725e41b90ab33250b710141791a4f5f547a35) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bbarsb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980043.g0", 0x000000, 0x020000, CRC(04a742fd) SHA1(abc497adc230569194f8315558d39aaf80a0533d) )
	ROM_LOAD16_BYTE( "sb980043.g1", 0x000001, 0x020000, CRC(a763298d) SHA1(dd651331acaf0e9e4a045624a14d7dcec5ec762c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_bbarsc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sb980044.g0", 0x000000, 0x020000, CRC(e1d51667) SHA1(99416b97511d04498d5a05b469b130e5819f520e) )
	ROM_LOAD16_BYTE( "sb980044.g1", 0x000001, 0x020000, CRC(5eaf5e1b) SHA1(e4c74f7604f1e9a468da62affa762e2916cdb664) )
	MISSING_SOUND
ROM_END


ROM_START( ep_cslay )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "c_slayer.p0", 0x000000, 0x020000, CRC(599c720c) SHA1(3b7461e9591a30401374d0aa1dfcd1e6c33cebb3) )
	ROM_LOAD16_BYTE( "c_slayer.p1", 0x000001, 0x020000, CRC(bc8942eb) SHA1(3c3d933fbf14aea81f3ec94f62c0933235170e58) )
	ROM_LOAD16_BYTE( "c_slayer.p2", 0x040000, 0x020000, CRC(cec502db) SHA1(283c53418bafa6e20a7553278fb16a7dba34af69) )
	ROM_LOAD16_BYTE( "c_slayer.p3", 0x040001, 0x020000, CRC(0d63b8fe) SHA1(f2b3ac2d05e7c11797b8d29062787c1d75344b87) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cslaya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "cash.sla.p0", 0x000000, 0x020000, NO_DUMP ) // file was 0 bytes in size
	ROM_LOAD16_BYTE( "cash.sla.p1", 0x000001, 0x020000, CRC(c7806f74) SHA1(6d98f6f8475bd8ef0b9e47c891c8ef1460b7c029) )
	MISSING_SOUND
ROM_END




ROM_START( ep_monky )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03nmb03.g0", 0x000000, 0x020000, CRC(64465543) SHA1(b951e4a9052fbc5e8927e6ddddcc68f7dc77600f) )
	ROM_LOAD16_BYTE( "i03nmb03.g1", 0x000001, 0x020000, CRC(8e2e241b) SHA1(6481e69ee8106ecbb4bcd87d65c6e9f28e23e688) )
	MISSING_SOUND
ROM_END

ROM_START( ep_monkya )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "i03pmb04.g0", 0x000000, 0x020000, CRC(c5afd11f) SHA1(eee7c00aac92868bc6a24be02e99132b33278c07) )
	ROM_LOAD16_BYTE( "i03pmb04.g1", 0x000001, 0x020000, CRC(22a18d5e) SHA1(535466d05c17f1cd38921c5ba7f762b30577af96) )
	MISSING_SOUND
ROM_END


ROM_START( ep_makmv )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "make1-1p.p1", 0x00000, 0x020000, CRC(13389695) SHA1(032aaa5053f57f06a240f8d9bdccda656e52ade2) )
	ROM_LOAD16_BYTE( "make1-1p.p2", 0x00001, 0x020000, CRC(c0f6e421) SHA1(b254546f5062ce812736e48b04dbaceb0da85a05) )
	/* code doesn't end.. missing other roms? */
	MISSING_SOUND
ROM_END

ROM_START( ep_makmva )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "make1-1n.p1", 0x00000, 0x020000, NO_DUMP ) // was the same as p2..
	ROM_LOAD16_BYTE( "make1-1n.p2", 0x00001, 0x020000, CRC(09323ee2) SHA1(f853d6d27b94cb7c2a19eae1868eed0ea1541b69) )
	MISSING_SOUND
ROM_END


#define EP_LUG_SOUND \
	ROM_REGION( 0x200000, "ymz", 0 ) \
	ROM_LOAD( "udrgnd.sn0", 0x000000, 0x080000, CRC(743e3936) SHA1(89d34a36c0506ee690011951713c8bc1844955cf) ) \
	ROM_LOAD( "udrgnd.s1" , 0x080000, 0x080000, CRC(db9aa3db) SHA1(5688e21db228d0d7d195fd70fb1b32d75d350a54) )
ROM_START( ep_lug )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "undergr.g0", 0x00000, 0x020000, CRC(ad34ecc5) SHA1(b0122ed340e76b52529ec12b237e93773674d314) )
	ROM_LOAD16_BYTE( "undergr.g1", 0x00001, 0x020000, CRC(e93d5f7e) SHA1(52fa3b57fc8b1143a43fa3d61dbfccd68fd7f200) )
	EP_LUG_SOUND
ROM_END

ROM_START( ep_luga )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "lond09g0", 0x00000, 0x020000, CRC(2d3343e0) SHA1(7618ab08c23f8f0e748a7b85ec3c7a52f1f64602) )
	ROM_LOAD16_BYTE( "lond09g1", 0x00001, 0x020000, CRC(0f36cfbb) SHA1(61c9df9247ede89734f24a2838c49f5e44884993) )
	EP_LUG_SOUND
ROM_END

ROM_START( ep_lugb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02176.g0", 0x00000, 0x020000, CRC(12c5414a) SHA1(a570093adb67da2c7a3bdab38a419848ee188cdc) )
	ROM_LOAD16_BYTE( "msa02176.g1", 0x00001, 0x020000, CRC(c2aae470) SHA1(51e0523b68b0c45e80a7553ae5cadc74b7497b19) )
	EP_LUG_SOUND
ROM_END

ROM_START( ep_lugc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msa02177.g0", 0x00000, 0x020000, CRC(a97b9047) SHA1(362ad27401fc2ae889761349af7dcf403318a6b6) )
	ROM_LOAD16_BYTE( "msa02177.g1", 0x00001, 0x020000, CRC(8f0d6b71) SHA1(83d81c528e72d5c0a754e47b0f0cbde7156ad908) )
	EP_LUG_SOUND
ROM_END



ROM_START( ep_crzbn )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01007.g0", 0x00000, 0x020000, CRC(3b3e2c4f) SHA1(6669e661fad6994a9b866aa2eb62a2eeff445831) )
	ROM_LOAD16_BYTE( "msb01007.g1", 0x00001, 0x020000, CRC(fcf6baa3) SHA1(53d07d6146046bcef2d43806c8601671483e453c) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crzbna )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01008.g0", 0x00000, 0x020000, CRC(17480519) SHA1(084a74f94e3ed424715e6ed2ecee57e9d8c014ee) )
	ROM_LOAD16_BYTE( "msb01008.g1", 0x00001, 0x020000, CRC(e9bee688) SHA1(2f5a014cbe0d80e42cc94c21a3fe8095a89feb19) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crzbnb )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01009.g0", 0x00000, 0x020000, CRC(075fd0a3) SHA1(e8e665410d334a9a68fb65f9d5e2b948101810a4) )
	ROM_LOAD16_BYTE( "msb01009.g1", 0x00001, 0x020000, CRC(162d469b) SHA1(8feb622b835c5cb7dcd7e528b6ae05d9c00e704b) )
	MISSING_SOUND
ROM_END

ROM_START( ep_crzbnc )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "msb01010.g0", 0x00000, 0x020000, CRC(162a795e) SHA1(b74d3b75b49791b548c2b6bcf09bb0a0ddada7b2) )
	ROM_LOAD16_BYTE( "msb01010.g1", 0x00001, 0x020000, CRC(cba57dda) SHA1(8b2511333e0e6dd93920c9a7370a239175d528f6) )
	MISSING_SOUND
ROM_END


ROM_START( ep_cascz )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "ccst0-2p.p0", 0x00000, 0x020000, CRC(b95f9d85) SHA1(00f9f568f28ffaff6bea0b8c06afb69459fff1de) )
	ROM_LOAD16_BYTE( "ccst0-2p.p1", 0x00001, 0x020000, CRC(d348b0be) SHA1(ea41d845d92886634eba43e304acd457cba89213) )
	MISSING_SOUND
ROM_END

ROM_START( ep_cascza ) // not sure about this set
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "casin.cra",  0x000000, 0x020000, CRC(b4f8c41d) SHA1(bb4d9b8ce807220a6bdd484e65276e4b08eb9640) )
	ROM_LOAD16_BYTE( "c_crazy.p1", 0x000001, 0x020000, CRC(dc523e26) SHA1(77170d5b4b693c52bd7602e332ff15bf6a78678d) )
	ROM_LOAD16_BYTE( "c_crazy.p2", 0x040000, 0x020000, CRC(3a6b3615) SHA1(b39d85090c816343cf1220dd0a54e3f7f4c6ad7e) )
	ROM_LOAD16_BYTE( "c_crazy.p3", 0x040001, 0x020000, CRC(99f2f8be) SHA1(bcd8740c626bfb08094f31ffb9a9f0f3f03f1025) )
	MISSING_SOUND
ROM_END


ROM_START( ep_wside )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "w_side.g0", 0x000000, 0x020000, CRC(30cff210) SHA1(e22578d0c86096b6b021e1b8a00b05cd3e734a22) )
	ROM_LOAD16_BYTE( "w_side.g1", 0x000001, 0x020000, CRC(c8908cd4) SHA1(67e87f45d6df61161307ca3e8f378f6b8b00a604) )
	ROM_LOAD16_BYTE( "w_side.g2", 0x040000, 0x020000, CRC(d23e430e) SHA1(76885d331fad3a916a74ef770dcc604e8532b0fb) )
	ROM_LOAD16_BYTE( "w_side.g3", 0x040001, 0x020000, CRC(fbb09673) SHA1(5d0febf5ce46cb791cd4246a14f8c47badedb598) )
	MISSING_SOUND
ROM_END

ROM_START( ep_wsidea )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "wildside.g0", 0x000000, 0x018000, BAD_DUMP CRC(040e5f1d) SHA1(e5bade050871841ad13c19ff39a84918549cd699) ) // wrong size, but a different revision
	ROM_LOAD16_BYTE( "wildside.g1", 0x000001, 0x020000, NO_DUMP )
	MISSING_SOUND
ROM_END

ROM_START( ep_goldf )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "gfev_arc.g0", 0x00000, 0x020000, CRC(4a9e8304) SHA1(a28938702fdb760185b42a0d6e24bfdb3df4d757) )
	ROM_LOAD16_BYTE( "gfev_arc.g1", 0x00001, 0x020000, CRC(96415731) SHA1(3e40358914af0efe1237bd83a48f9e3323e61dea) )
	MISSING_SOUND
ROM_END

ROM_START( ep_imj )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "imajacpo.g0", 0x000000, 0x020000, CRC(58284716) SHA1(526332ac2fc949c0cb7a811b0ef09d9885f1ba65) )
	ROM_LOAD16_BYTE( "imajacpo.g1", 0x000001, 0x020000, CRC(fdd35c2d) SHA1(ac2d118262b3b0146bedc9de578900937d566cc9) )
	ROM_LOAD16_BYTE( "imajacpo.g2", 0x040000, 0x020000, CRC(431474b7) SHA1(f2d5b88ba5c2e5e55f795e1750c99f203d4d761c) )
	ROM_LOAD16_BYTE( "imajacpo.g3", 0x040001, 0x020000, CRC(1396cb7e) SHA1(c7efbdca676265412dbd6403c6f5b2f7880453a4) )
	MISSING_SOUND
ROM_END

ROM_START( ep_party )
	ROM_REGION( 0x080000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "p_party.g0", 0x00000, 0x020000, CRC(913ae15a) SHA1(7d4c24da5d271a02e05a226e76645058074fb707) )
	ROM_LOAD16_BYTE( "p_party.g1", 0x00001, 0x020000, CRC(d5cf55db) SHA1(f8aac12d1fd83072644311f5e4402f02db37a00c) )
	MISSING_SOUND
ROM_END


/* header info */

GAME( 1998, ep_simp,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (3.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpa,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (3.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpb,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (3.5, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpc,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (4.5, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpd,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (4.5, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpe,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (1.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpf,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (1.5, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simpg,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (2.5, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_simph,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (2.5, set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_simpj,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (1.8, set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_simpk,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (1.8, set 11)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_simpl,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (3.7, set 12)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_simpm,   ep_simp,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons (Maygay) (EPOCH) (3.7, set 13)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1999, ep_merln,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Merlin's Magic (Maygay) (EPOCH) (1.91)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_bingb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bingo Belle (Maygay) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_bingba,  ep_bingb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bingo Belle (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_calyp,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Calypso (Maygay) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_calypa,  ep_calyp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Calypso (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_cow,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Carry On Winning (Maygay) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cowa,    ep_cow,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Carry On Winning (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_cfall,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfalla,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfallb,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfallc,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (2.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfalld,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (2.3, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfalle,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (3.2, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cfallf,  ep_cfall,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Falls (Maygay) (EPOCH) (3.2, set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_cflow,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Flow (Maygay) (EPOCH) (3.7, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_cflowa,  ep_cflow,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Flow (Maygay) (EPOCH) (3.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_cflowc,  ep_cflow,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Flow (Maygay) (EPOCH) (3.A, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_cflowd,  ep_cflow,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Flow (Maygay) (EPOCH) (3.A, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_doubl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Top (Maygay) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_doubla,  ep_doubl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Top (Maygay) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_doublb,  ep_doubl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Top (Maygay) (EPOCH) (1.6, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_doublc,  ep_doubl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Top (Maygay) (EPOCH) (1.6, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_doubld,  ep_doubl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Top (Maygay) (EPOCH) (1.4, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_evil,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Evil Streak (Maygay) (EPOCH) (1.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_evila,   ep_evil,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Evil Streak (Maygay) (EPOCH) (1.6, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_evilb,   ep_evil,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Evil Streak (Maygay) (EPOCH) (1.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_nyny,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (3.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynya,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (3.6, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynyb,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (4.6, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynyc,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (4.6, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynyd,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (3.A, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynye,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (3.A, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_nynyf,   ep_nyny,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "New York New York (Maygay) (EPOCH) (3.9, set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_pizza,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pizza The Action (Maygay) (EPOCH) (2.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pizzaa,  ep_pizza,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pizza The Action (Maygay) (EPOCH) (2.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pizzab,  ep_pizza,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pizza The Action (Maygay) (EPOCH) (2.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pizzac,  ep_pizza,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pizza The Action (Maygay) (EPOCH) (2.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_crazy,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (1.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crazya,  ep_crazy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (1.6, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crazyb,  ep_crazy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (2.6, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crazyc,  ep_crazy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (2.6, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crazyd,  ep_crazy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (1.9, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crazye,  ep_crazy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reel Crazy (Maygay) (EPOCH) (1.9, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000527 SUPER MARIO  - version number seems to jump backwards from 2.A after 1.A, with 2.A having the exact same build date as 1.A..  is the KJ / SM / CW bit significant? or is the 2.A revision a mistake, it comes after the 1.A in label tho, so 1.A isn't a correction.
GAME( 1997, ep_mario,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970024 SA970018 11:40:00 DEC 02 1997 MAB, V1.5 EPOCH APOLLO AllCash
GAME( 1997, ep_marioa,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970025 SA970019 11:40:00 DEC 02 1997 MAB, V1.5 EPOCH APOLLO AllCash
GAME( 1997, ep_mariob,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.A, set 3)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970036 SA970024 11:15:00 DEC 23 1997 KJGio.x, 1.A  EPOCH APOLLO
GAME( 1997, ep_marioc,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.A, set 4)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970037 SA970025 11:15:00 DEC 23 1997 KJGio.x, 1.A  EPOCH APOLLO
GAME( 1997, ep_mariod,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (2.A, set 5)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970038 SA970026 11:15:00 DEC 23 1997 KJGio.x, 2.A  EPOCH APOLLO
GAME( 1997, ep_marioe,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (2.A, set 6)",MACHINE_IS_SKELETON_MECHANICAL ) // SA970039 SA970027 11:15:00 DEC 23 1997 KJGio.x, 2.A  EPOCH APOLLO
GAME( 1998, ep_marioh,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.B, set 9)",MACHINE_IS_SKELETON_MECHANICAL ) // SA980286 SA970037 10:05:00 AUG 26 1998 SJMio.x, 1.B  EPOCH APOLLO
GAME( 1999, ep_mariof,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.C, set 7)",MACHINE_IS_SKELETON_MECHANICAL ) // SA990033 SA980286 14:05:00 FEB 26 1999 CWGio.x, 1.C  EPOCH APOLLO
GAME( 1999, ep_mariog,  ep_mario,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Mario (Maygay) (EPOCH) (1.C, set 8)",MACHINE_IS_SKELETON_MECHANICAL ) // SA990034 SA980287 14:05:00 FEB 26 1999 CWGio.x, 1.C  EPOCH APOLLO

GAME( 199?, ep_tincn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tin Can Alley (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tincna,  ep_tincn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tin Can Alley (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tonfn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tons Of Fun (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tonfna,  ep_tonfn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tons Of Fun (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tcrwn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tcrwna,  ep_tcrwn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tcrwnb,  ep_tcrwn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (2.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tcrwnc,  ep_tcrwn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (2.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tcrwnd,  ep_tcrwn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (3.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tcrwne,  ep_tcrwn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Crown (Maygay) (EPOCH) (3.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_word,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_worda,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wordb,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (2.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wordc,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (2.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wordd,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (3.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_worde,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (3.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wordf,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (4.1, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wordg,   ep_word,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Word Up (Maygay) (EPOCH) (4.1, set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_xtra,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "X-tra X-tra (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_xtraa,   ep_xtra,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "X-tra X-tra (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 2002, ep_25crt,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "25 Carrot Gold (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_25crta,  ep_25crt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "25 Carrot Gold (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_25crtb,  ep_25crt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "25 Carrot Gold (Maygay) (EPOCH) (3.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_25crtc,  ep_25crt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "25 Carrot Gold (Maygay) (EPOCH) (4.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_25crtd,  ep_25crt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "25 Carrot Gold (Maygay) (EPOCH) (5.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_bartk,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (4.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartka,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (3.9, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartkb,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (3.9, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartkc,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (4.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartkd,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (4.4, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartke,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (4.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bartkf,  ep_bartk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bar Trekkin (Maygay) (EPOCH) (4.2, set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_batls,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Battleships (Maygay) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_batlsa,  ep_batls,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Battleships (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_batlsb,  ep_batls,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Battleships (Maygay) (EPOCH) (1.9, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_batlsc,  ep_batls,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Battleships (Maygay) (EPOCH) (1.9, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cdspn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cardinal Spin (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cdspna,  ep_cdspn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cardinal Spin (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cosc,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Carry On Screaming (Maygay) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cosca,   ep_cosc,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Carry On Screaming (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// these 3 are 'fruit cocktail' (CA000627) in the header
GAME( 199?, ep_ccock,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Cocktail (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ccocka,  ep_ccock,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Cocktail (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ccockb,  ep_ccock,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Cocktail (Maygay) (EPOCH) (1.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
// this one is 'cash cocktail' (CA000710)
GAME( 199?, ep_ccockc,  ep_ccock,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Cocktail (Maygay) (EPOCH) (1.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cgord,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Gordon (Maygay) (EPOCH) (2.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cgorda,  ep_cgord,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Gordon (Maygay) (EPOCH) (2.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cgordb,  ep_cgord,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Gordon (Maygay) (EPOCH) (2.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cgordc,  ep_cgord,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Gordon (Maygay) (EPOCH) (1.9, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cshpn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash In The Pan (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cshpna,  ep_cshpn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash In The Pan (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ctit,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Of The Titans (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ctita,   ep_ctit,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Of The Titans (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CASINO GRAND
GAME( 199?, ep_casgc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Grand Club (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_casgca,  ep_casgc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Grand Club (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
// CLASSIC CASINO GRAND
GAME( 199?, ep_casgcb,  ep_casgc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Grand Club (Maygay) (EPOCH) (1.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_casgcc,  ep_casgc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Grand Club (Maygay) (EPOCH) (1.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_casrd,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Royale Deluxe Club (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_casrda,  ep_casrd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Casino Royale Deluxe Club (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_chock,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Chocks Away (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_chocka,  ep_chock,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Chocks Away (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_chockb,  ep_chock,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Chocks Away (Maygay) (EPOCH) (1.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_commd,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Complete Madness (Maygay) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_commda,  ep_commd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Complete Madness (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_commdb,  ep_commd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Complete Madness (Maygay) (EPOCH) (1.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_commdc,  ep_commd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Complete Madness (Maygay) (EPOCH) (1.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_commdd,  ep_commd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Complete Madness (Maygay) (EPOCH) (2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cor2,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street 2 (Maygay) (EPOCH) (3.7, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cor2a,   ep_cor2,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street 2 (Maygay) (EPOCH) (3.7, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cor2b,   ep_cor2,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street 2 (Maygay) (EPOCH) (3.8, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cor2c,   ep_cor2,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street 2 (Maygay) (EPOCH) (3.8, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cormn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street Monopoly Club (Maygay) (EPOCH) (1.7, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cormna,  ep_cormn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Coronation Street Monopoly Club (Maygay) (EPOCH) (1.7, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cycl,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cyclone Club (Maygay) (EPOCH) (3.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cycla,   ep_cycl,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cyclone Club (Maygay) (EPOCH) (3.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cyclb,   ep_cycl,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cyclone Club (Maygay) (EPOCH) (2.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

// these are dated 2001 with a MMLNFOTG string (game CA000727)
GAME( 2001, ep_fgods,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2001, ep_fgodsa,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2001, ep_fgodsb,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (2.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2001, ep_fgodsc,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (2.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
// below are dated 2002 with a UGLNFOTG string (still CA000727) versioning seems independent of above
GAME( 2002, ep_fgodsd,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_fgodse,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (3.2, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_fgodsf,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (1.1, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_fgodsg,  ep_fgods,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit Of The Gods (Maygay) (EPOCH) (1.1, set 8)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 2002, ep_gldtp,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gold Top (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_gldtpa,  ep_gldtp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gold Top (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000782 GREATESCAPE
GAME( 2002, ep_gesc2,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape 2 (Maygay) (EPOCH) (2.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_gesc2a,  ep_gesc2,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape 2 (Maygay) (EPOCH) (2.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_gresc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_gresca,  ep_gresc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CC000043 CLUB ESCAPE
GAME( 1999, ep_geclb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape Club (Maygay) (EPOCH) (1.C, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_geclba,  ep_geclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape Club (Maygay) (EPOCH) (1.C, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_geclbb,  ep_geclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Great Escape Club (Maygay) (EPOCH) (1.9, set 3)",MACHINE_IS_SKELETON_MECHANICAL )



GAME( 2002, ep_hhclb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Haunted House Club (Maygay) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_hhclba,  ep_hhclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Haunted House Club (Maygay) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2000, ep_hhclbb,  ep_hhclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Haunted House Club (Maygay) (EPOCH) (1.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL ) // only one of the last roms differs from below
GAME( 2000, ep_hhclbc,  ep_hhclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Haunted House Club (Maygay) (EPOCH) (1.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000623 HOMERS MELT
GAME( 2000, ep_homer,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.A, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2000, ep_homeri,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.9, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
// CA000623 MELTDOWN lt
GAME( 2002, ep_homera,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (3.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_homerb,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (3.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_homerc,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.B, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_homerd,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.B, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
// CA000555 HOMERS MELT
GAME( 2003, ep_homere,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (3.1, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_homerf,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (3.1, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_homerg,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.1, set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_homerh,  ep_homer,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Homer's Meltdown (Maygay) (EPOCH) (2.1, set 10)",MACHINE_IS_SKELETON_MECHANICAL )

// versioning doesn't make much sense here
GAME( 2002, ep_icebg,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_icebga,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_icebgb,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_icebgc,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_icebgd,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_icebge,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
// these have 2003 dates, despite being versioned lower?
GAME( 2003, ep_icebgf,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.3, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_icebgg,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.3, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_icebgh,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.4, set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_icebgi,  ep_icebg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ice Burger (Maygay) (EPOCH) (1.4, set 10)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000719 ITALIAN JOB
GAME( 2002, ep_itjb2,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job 2 (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_itjb2a,  ep_itjb2,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job 2 (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_itjb2b,  ep_itjb2,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job 2 (Maygay) (EPOCH) (2.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_itjb2c,  ep_itjb2,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job 2 (Maygay) (EPOCH) (2.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
// same game code, but 2.1 is (c)2001 wheras 1.5 above is 2002...
GAME( 2001, ep_ijob,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job (Maygay) (EPOCH, v2.1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2001, ep_ijoba,   ep_ijob,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job (Maygay) (EPOCH, v1.1)",MACHINE_IS_SKELETON_MECHANICAL )

// CC000059 CLUBITALIAN
GAME( 2002, ep_ijcl,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job Club (Maygay) (EPOCH) (2.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_ijcla,   ep_ijcl,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Italian Job Club (Maygay) (EPOCH) (2.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// again build date vs build number doesn't make sense
GAME( 2002, ep_jsttt,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (4.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_jsttta,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (4.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_jstttb,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.5, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_jstttc,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_jstttd,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.5, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_jsttte,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_jstttf,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.6, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2003, ep_jstttg,  ep_jsttt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Just The Ticket (Maygay) (EPOCH) (3.6, set 8)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000593 MARIO KART    'ELEGANCE'
GAME( 1999, ep_mkart,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_mkarta,  ep_mkart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_mkartd,  ep_mkart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL ) // First Master For Elegance cab
// CA000580 MARIOKART     'EPOCH APOLLO'
GAME( 1999, ep_mkartb,  ep_mkart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.6, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_mkartc,  ep_mkart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.6, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_mkarte,  ep_mkart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mario Kart (Maygay) (EPOCH) (1.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_midas,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Midas Touch Club (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1998, ep_midasa,  ep_midas,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Midas Touch Club (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000612 MONTE CARLO     'ELEGANCE'
GAME( 199?, ep_monbs,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Monte Carlo Or Bust (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monbsa,  ep_monbs,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Monte Carlo Or Bust (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_mrmus,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mr Muscle (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mrmusa,  ep_mrmus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mr Muscle (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_nuns,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Nuns Of Navarone (Maygay) (EPOCH) (2.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_nunsa,   ep_nuns,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Nuns Of Navarone (Maygay) (EPOCH) (2.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_otm,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Over The Moon (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_otma,    ep_otm,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Over The Moon (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// messy headers..
GAME( 199?, ep_ozzie,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (2.Z, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozziea,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) ( .2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozzieb,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (5.J, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozziec,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (6.J, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozzied,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (1.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozziee,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (1.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozzief,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (2.Z, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozzieg,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (0.2, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ozzieh,  ep_ozzie,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Ozzie Ozzie Ozzie (Maygay) (EPOCH) (0.2, set 9)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_pascl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Classic Club (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pascla,  ep_pascl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Classic Club (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_passpb,  ep_pascl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Classic Club (Maygay) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_passpc,  ep_pascl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Classic Club (Maygay) (EPOCH) (1.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_passp,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Club (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_passpa,  ep_passp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Passport To Riches Club (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_baskr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (1.7, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_baskra,  ep_baskr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_baskrb,  ep_baskr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (2.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_baskrc,  ep_baskr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (1.7, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_baskrd,  ep_baskr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_baskre,  ep_baskr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pounds Of The Baskervilles (Maygay) (EPOCH) (1.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_rags,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Rags To Riches Club (Maygay) (EPOCH) (1.10, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ragsa,   ep_rags,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Rags To Riches Club (Maygay) (EPOCH) (1.10, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_roost,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Roosters Millions (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_roosta,  ep_roost,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Roosters Millions (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 2002, ep_secag,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Secret Agent (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_secaga,  ep_secag,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Secret Agent (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2000, ep_secagb,  ep_secag,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Secret Agent (Maygay) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_simfr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Simply Fruits (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_simfra,  ep_simfr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Simply Fruits (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 2002, ep_srwin,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Sir Winalot (Maygay) (EPOCH) (2.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_srwina,  ep_srwin,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Sir Winalot (Maygay) (EPOCH) (3.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_srwinb,  ep_srwin,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Sir Winalot (Maygay) (EPOCH) (3.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_srwinc,  ep_srwin,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Sir Winalot (Maygay) (EPOCH) (2.6, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2000, ep_srwind,  ep_srwin,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Sir Winalot (Maygay) (EPOCH) (2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_spcbw,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwa,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwb,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwc,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwd,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.5, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwe,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwf,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.6, set 7)",MACHINE_IS_SKELETON_MECHANICAL ) // version seems typoed as " .6" in ROM
GAME( 199?, ep_spcbwg,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.6, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwh,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.4, set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwi,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.4, set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwj,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.8, set 11)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwk,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.8, set 12)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwl,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.9, set 13)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spcbwm,  ep_spcbw,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Special Brew (Maygay) (EPOCH) (1.9, set 14)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 2002, ep_spec,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spectre (Maygay) (EPOCH) (1.6, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_speca,   ep_spec,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spectre (Maygay) (EPOCH) (1.6, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_specb,   ep_spec,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spectre (Maygay) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_step,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Stepping Stones (Maygay) (EPOCH) (1.0, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_stepa,   ep_step,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Stepping Stones (Maygay) (EPOCH) (1.0, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_survi,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Survival (Maygay) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_survia,  ep_survi,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Survival (Maygay) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tits,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Title Shot Club (Maygay) (EPOCH) (1.7, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_titsa,   ep_tits,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Title Shot Club (Maygay) (EPOCH) (1.7, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_titsb,   ep_tits,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Title Shot Club (Maygay) (EPOCH) (1.5, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 2000, ep_tortr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Torture TV (Maygay) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2000, ep_tortra,  ep_tortr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Torture TV (Maygay) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000715 TRIVIAL PUR
GAME( 199?, ep_tp,      0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trivial Pursuit (Maygay) (EPOCH) (3.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tpa,     ep_tp,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trivial Pursuit (Maygay) (EPOCH) (3.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tpb,     ep_tp,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trivial Pursuit (Maygay) (EPOCH) (2.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
// same code as above?
GAME( 199?, ep_tp2,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trivial Pursuit 2 (Maygay) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tp2a,    ep_tp2,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trivial Pursuit 2 (Maygay) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tutcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tutankhamun Club (Maygay) (EPOCH) (2.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tutcla,  ep_tutcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tutankhamun Club (Maygay) (EPOCH) (2.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tutclb,  ep_tutcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tutankhamun Club (Maygay) (EPOCH) (1.8, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_21clb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Twenty One Club (Maygay) (EPOCH) (3.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_21clba,  ep_21clb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Twenty One Club (Maygay) (EPOCH) (3.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 2002, ep_xspot,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "X Marks The Spot (Maygay) (EPOCH) (1.5, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_xspota,  ep_xspot,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "X Marks The Spot (Maygay) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_xspotb,  ep_xspot,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "X Marks The Spot (Maygay) (EPOCH) (1.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_bathl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (2.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathla,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (2.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathlb,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (2.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathlc,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (2.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathld,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (3.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathle,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (3.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathlf,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (4.1, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathlg,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (4.1, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bathlh,  ep_bathl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Bat Outa Hell (Global) (EPOCH) (3.3, set 9)",MACHINE_IS_SKELETON_MECHANICAL )


// CA000720 BEAVER LAS  (2.3 has a newer date than 3.3...)
GAME( 2001, ep_beav3,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 3.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2001, ep_beav3a,  ep_beav3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 3.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_beav3b,  ep_beav3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 4.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_beav3c,  ep_beav3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 4.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_beavrn,  ep_beav3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 2.3, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_beavro,  ep_beav3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (CA000720, 2.3, set 6)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_bvruc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Uncovered (Global) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvruca,  ep_bvruc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Uncovered (Global) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrucb,  ep_bvruc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Uncovered (Global) (EPOCH) (1.6, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrucc,  ep_bvruc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Uncovered (Global) (EPOCH) (2.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_cbrun,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (2.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbruna,  ep_cbrun,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (2.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrunb,  ep_cbrun,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (2.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrunc,  ep_cbrun,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (2.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrund,  ep_cbrun,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (3.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrune,  ep_cbrun,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run (Global) (EPOCH) (3.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ctc,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cut Throat Cash (Global) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ctca,    ep_ctc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cut Throat Cash (Global) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )



GAME( 199?, ep_ddq,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqa,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (1.4, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqb,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqc,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqd,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.2, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqe,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.2, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqf,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.4, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqg,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.4, set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqh,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.5, set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqi,    ep_ddq,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens (Global) (EPOCH) (2.5, set 10)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_greed,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Greed (Global) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_greeda,  ep_greed,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Greed (Global) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// CA000702 HI SPIRITS
GAME( 199?, ep_hispr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (1.A, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hispra,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (1.A, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hisprb,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (4.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hisprc,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (4.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hisprd,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (3.2, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hispre,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (3.2, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spirt,   ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (2.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL ) // same game code
GAME( 199?, ep_spirta,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (2.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spirtb,  ep_hispr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits (Global) (EPOCH) (4.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_pkni,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknia,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknib,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknic,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknid,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.4, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknie,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.7, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pknif,   ep_pkni,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Phoenix Knights (Global) (EPOCH) (1.7, set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_snbev,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (1.8, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snbeva,  ep_snbev,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (1.8, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snbevb,  ep_snbev,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (1.9, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snbevc,  ep_snbev,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (1.9, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snbevd,  ep_snbev,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snbeve,  ep_snbev,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Saturday Night Beaver (Global) (EPOCH) (2.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_spook,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spooky Hollow (Global) (EPOCH) (1.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spooka,  ep_spook,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spooky Hollow (Global) (EPOCH) (1.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spookb,  ep_spook,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spooky Hollow (Global) (EPOCH) (2.7, set 3)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_milhr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhra,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhrb,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhrc,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhrd,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.6, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhre,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.6, set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhrf,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.9, set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_milhrg,  ep_milhr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Wants To Be A Millionhare? (Global) (EPOCH) (1.9, set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_bbars,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Balloon Bars (Maygay) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bbarsa,  ep_bbars,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Balloon Bars (Maygay) (EPOCH) (1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bbarsb,  ep_bbars,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Balloon Bars (Maygay) (EPOCH) (2.0, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bbarsc,  ep_bbars,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Balloon Bars (Maygay) (EPOCH) (2.0, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cslay,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cash Slayer (Global) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cslaya,  ep_cslay,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cash Slayer (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL ) // incomplete pair

GAME( 199?, ep_monky,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Monkey Business (Global) (EPOCH) (1.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monkya,  ep_monky,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Monkey Business (Global) (EPOCH) (1.5, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1999, ep_lug,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "London Underground (Maygay) (EPOCH) (2.4, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 1999, ep_luga,    ep_lug,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "London Underground (Maygay) (EPOCH) (2.9, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_lugb,    ep_lug,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "London Underground (Maygay) (EPOCH) (3.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 2002, ep_lugc,    ep_lug,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "London Underground (Maygay) (EPOCH) (3.1, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_crzbn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Crazy Bingo (Maygay) (EPOCH) (1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crzbna,  ep_crzbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Crazy Bingo (Maygay) (EPOCH) (1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crzbnb,  ep_crzbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Crazy Bingo (Maygay) (EPOCH) (1.1 Gala, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_crzbnc,  ep_crzbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Crazy Bingo (Maygay) (EPOCH) (1.1 Gala, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

// Z0000024 CASINOCR AZ
GAME( 199?, ep_cascza,  ep_cascz,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Crazy (Global) (EPOCH) (2.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wside,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildside (Global) (EPOCH) (1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wsidea,  ep_wside,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildside (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL ) // incomplete pair

GAME( 199?, ep_imj,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "I'm A Jackpot (Global) (EPOCH) (1.5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 1998, ep_party,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Party Party (Global) (EPOCH) (1.1)",MACHINE_IS_SKELETON_MECHANICAL )


/* no header info */

GAME( 199?, ep_dblim,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Impact (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dblima,  ep_dblim,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Impact (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dblimb,  ep_dblim,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Impact (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dblimc,  ep_dblim,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Impact (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dblimd,  ep_dblim,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Double Impact (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_itjb3,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Italian Job 3 (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_itjb3a,  ep_itjb3,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Italian Job 3 (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wipeo,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wipeout (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wipeoa,  ep_wipeo,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wipeout (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wipeob,  ep_wipeo,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wipeout (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wipeoc,  ep_wipeo,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wipeout (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wipeoe,  ep_wipeo,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wipeout (Maygay) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_bbonz,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bingo Bonanza (Maygay - Union) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bbonza,  ep_bbonz,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bingo Bonanza (Maygay - Union) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cahoy,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay - Eclipse?",        "Cash Ahoy (Maygay - Eclipse?) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cahoya,  ep_cahoy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Ahoy (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cahoyb,  ep_cahoy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cash Ahoy (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cock,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cocka,   ep_cock,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cockb,   ep_cock,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cockc,   ep_cock,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cockd,   ep_cock,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cocke,   ep_cock,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cock A Doodle Dosh (Maygay - Union) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_duff,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Duff Beer Guide (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_duffa,   ep_duff,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Duff Beer Guide (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_duffb,   ep_duff,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Duff Beer Guide (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_duffc,   ep_duff,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "The Simpsons - Duff Beer Guide (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_flash,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flasha,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flashb,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flashc,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flashd,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flashe,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_flashf,  ep_flash,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Flashback (Maygay - Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fortg,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fortune & Glory (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fortga,  ep_fortg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fortune & Glory (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fortgb,  ep_fortg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fortune & Glory (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fran,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Frantic (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_frana,   ep_fran,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Frantic (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fnclb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit & Nudge Club (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fnclba,  ep_fnclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Fruit & Nudge Club (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fullm,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullma,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullmb,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullmc,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullmd,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullme,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fullmf,  ep_fullm,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Full Moon Fever (Maygay - Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_funny,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnya,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnyb,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnyc,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnyd,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnye,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnyf,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_funnyg,  ep_funny,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Funny Money (Maygay) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_geron,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gerona,  ep_geron,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_geronb,  ep_geron,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_geronc,  ep_geron,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gerond,  ep_geron,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gerone,  ep_geron,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Geronimo (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_gridr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gridrunner (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gridra,  ep_gridr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gridrunner (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gridrb,  ep_gridr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gridrunner (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gridrc,  ep_gridr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gridrunner (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gridrd,  ep_gridr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Gridrunner (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hvns,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Heavens Above (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hvnsa,   ep_hvns,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Heavens Above (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_highv,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highva,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highvb,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highvc,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highvd,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highve,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_highvf,  ep_highv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "High Voltage (Maygay - Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hogmn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hog Money (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hogmna,  ep_hogmn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hog Money (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hogmnb,  ep_hogmn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hog Money (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_htdgs,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hot Dogs (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_htdgsa,  ep_htdgs,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hot Dogs (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hubbl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hubble Bubble (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hubbla,  ep_hubbl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hubble Bubble (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_huric,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hurica,  ep_huric,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_huricb,  ep_huric,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_huricc,  ep_huric,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_huricd,  ep_huric,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hurice,  ep_huric,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hurricane (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hur,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hurricane (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hura,    ep_hur,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hurricane (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hurb,    ep_hur,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hurricane (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hyst,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hysteria (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hysta,   ep_hyst,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Hysteria (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_manic,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manica,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manicb,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manicc,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manicd,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manice,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_manicf,  ep_manic,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Manic Miner (Maygay - Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_matrx,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Matrix (Maygay - Impulse) (EPOCH)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_mwom,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mortal Wombat (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mwoma,   ep_mwom,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mortal Wombat (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mwomb,   ep_mwom,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mortal Wombat (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mwomc,   ep_mwom,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mortal Wombat (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mwomd,   ep_mwom,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Mortal Wombat (Maygay) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_otmcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Over The Moon Club (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_otmcla,  ep_otmcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Over The Moon Club (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

// 'awards'
GAME( 199?, ep_pharo,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pharaoh's Treasure (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL ) // has hdd dump
GAME( 199?, ep_pharoa,  ep_pharo,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pharaoh's Treasure (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_pesos,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pick Yer Pesos (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pesosa,  ep_pesos,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pick Yer Pesos (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pesosb,  ep_pesos,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pick Yer Pesos (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pesosc,  ep_pesos,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Pick Yer Pesos (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_react,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_reacta,  ep_react,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_reactb,  ep_react,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_reactc,  ep_react,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_reactd,  ep_react,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_reacte,  ep_react,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Reactor (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_rtt,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Round The Twist (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_rtta,    ep_rtt,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Round The Twist (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_royrl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrla,  ep_royrl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrlb,  ep_royrl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrlc,  ep_royrl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrld,  ep_royrl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrle,  ep_royrl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Royal Roulette (Maygay) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_spart,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spartacash (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sparta,  ep_spart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spartacash (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spartb,  ep_spart,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spartacash (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_spntn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin & Tonic (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spntna,  ep_spntn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin & Tonic (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spntnb,  ep_spntn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin & Tonic (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spntnc,  ep_spntn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin & Tonic (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_spin,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spina,   ep_spin,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spinb,   ep_spin,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spinc,   ep_spin,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spind,   ep_spin,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_spine,   ep_spin,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Spin On It (Maygay - Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_strat,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Stratagem (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_strata,  ep_strat,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Stratagem (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_subb,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subba,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbb,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbc,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbd,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbe,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbf,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_subbg,   ep_subb,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Subbuteo (Maygay) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_snw,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Nudge Wink (Maygay - Union) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snwa,    ep_snw,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Nudge Wink (Maygay - Union) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snwb,    ep_snw,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Nudge Wink (Maygay - Union) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snwc,    ep_snw,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Nudge Wink (Maygay - Union) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snwd,    ep_snw,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Super Nudge Wink (Maygay - Union) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tak5,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Take Five (Maygay - Union) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tak5a,   ep_tak5,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Take Five (Maygay - Union) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_trail,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trailblazer (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_traila,  ep_trail,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trailblazer (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_trailb,  ep_trail,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Trailblazer (Maygay - Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_trics,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Cash (Maygay - Union) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_tricsa,  ep_trics,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Triple Cash (Maygay - Union) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wildf,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wildfire (Maygay - Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wildfa,  ep_wildf,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Wildfire (Maygay - Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_beavr,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavra,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrb,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrc,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrd,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavre,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrf,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrg,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrh,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavri,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrj,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 11)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrk,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 12)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrl,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 13)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_beavrm,  ep_beavr,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Beaver Las Vegas (Global) (EPOCH) (set 14)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_bvrcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrcla,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclb,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclc,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrcld,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrcle,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclf,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclg,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclh,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrcli,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclj,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 11)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bvrclk,  ep_bvrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Beaver Las Vegas Club (Global) (EPOCH) (set 12)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_bjclb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Blackjack Club (Global) (EPOCH)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_cbrcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrcla,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclb,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclc,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrcld,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrcle,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclf,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclg,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclh,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrcli,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclj,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 11)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cbrclk,  ep_cbrcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Cannonball Run Club (Global) (EPOCH) (set 12)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cclas,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Classic (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cclasa,  ep_cclas,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Classic (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cgrc,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Grand Classic (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cgrca,   ep_cgrc,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Grand Classic (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cgred,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Club Greed (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cgreda,  ep_cgred,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Club Greed (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cstrk,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrka,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrkb,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrkc,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrkd,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrke,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL ) // bad? (or at least has garbage fill, not 0xff like the other sets)
GAME( 199?, ep_cstrkf,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cstrkg,  ep_cstrk,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Crazy Streak Club (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_dmbus,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbusa,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbusb,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbusc,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbusd,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbuse,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_dmbusf,  ep_dmbus,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Dambusters (Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ddqcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqcla,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqclb,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqclc,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqcld,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqcle,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqclf,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ddqclg,  ep_ddqcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Dungeons & Drag Queens Club (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fog,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Fields of Gold (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_foga,    ep_fog,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Fields of Gold (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_fmf,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Full Moon Fever (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_fmfa,    ep_fmf,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Full Moon Fever (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_gtrot,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Globe Trotter (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL ) // contains maygay, global and impulse strings
GAME( 199?, ep_gtrota,  ep_gtrot,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Globe Trotter (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL ) // contains maygay, global and impulse strings

GAME( 199?, ep_grun,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_gruna,   ep_grun,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_grncl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_grncla,  ep_grncl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_grnclb,  ep_grncl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_grnclc,  ep_grncl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Grid Runner Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_heybg,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_heybga,  ep_heybg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_heybgb,  ep_heybg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_heybgc,  ep_heybg,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_heybc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_heybca,  ep_heybc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hey Big Spender Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_hiscl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_hiscla,  ep_hiscl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Hi Spirits Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_smoke,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokea,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokeb,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokec,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smoked,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokee,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokef,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokeg,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokeh,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 9)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokei,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 10)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_smokej,  ep_smoke,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Holy Smoke! (Impulse) (EPOCH) (set 11)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ifern,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Inferno (Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_iferna,  ep_ifern,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Inferno (Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_kopcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Knockout Punch Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_kopcla,  ep_kopcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Knockout Punch Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_kopclb,  ep_kopcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Knockout Punch Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ltt,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Licence To Thrill (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_ltta,    ep_ltt,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Licence To Thrill (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_monrt,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrta,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrtb,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrtc,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrtd,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrte,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monrtf,  ep_monrt,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Money Returns Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_monsh,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monsha,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monshb,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monshc,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monshd,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monshe,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_monshf,  ep_monsh,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "The Moonshine Club (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_mummy,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummya,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummyb,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummyc,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummyd,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummye,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mummyf,  ep_mummy,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Mummy Talks (Impulse) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_rlgdt,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Reel Good Time (Rebuild) (Global) (Version 1.0) (EPOCH)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_rchik,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Rich Chics Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_rchika,  ep_rchik,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Rich Chics Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_royrc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Royal Roulette Club (Impulse) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_royrca,  ep_royrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Royal Roulette Club (Impulse) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_sdcla,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclaa,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclab,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclac,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclaf,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclag,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclad,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL ) // larger roms, header not cleared
GAME( 199?, ep_sdclae,  ep_sdcla,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Classic (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL ) // larger roms, header not cleared

GAME( 199?, ep_sdclb,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_sdclba,  ep_sdclb,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Spotted Dick Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_stm,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_stma,    ep_stm,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_stmb,    ep_stm,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_stmc,    ep_stm,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_stmcl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_stmcla,  ep_stmcl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Storm Force Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tod,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Truth Or Dare (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_toda,    ep_tod,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Truth Or Dare (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_utnut,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_utnuta,  ep_utnut,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_utnutb,  ep_utnut,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_utnutc,  ep_utnut,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_utncl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_utncla,  ep_utncl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Utter Nutter Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wleek,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Weakest Leek Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wleeka,  ep_wleek,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Weakest Leek Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wud,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wuda,    ep_wud,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wudb,    ep_wud,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wudc,    ep_wud,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wudd,    ep_wud,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wude,    ep_wud,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "What's Up Doc (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_mlhrc,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mlhrca,  ep_mlhrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mlhrcb,  ep_mlhrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mlhrcc,  ep_mlhrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mlhrcd,  ep_mlhrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_mlhrce,  ep_mlhrc,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Who Whats To Be A Millionhare Club (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_wf,      0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfa,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfb,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfc,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfd,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfe,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 6)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wff,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 7)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_wfg,     ep_wf,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Wildfire (Global) (EPOCH) (set 8)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_makmv,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Make Your Move (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_makmva,  ep_makmv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Make Your Move (Global) (EPOCH) (set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cascz,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Casino Crazy (Global) (EPOCH) (set 1)",MACHINE_IS_SKELETON_MECHANICAL ) // check identity, other set has header

GAME( 199?, ep_goldf,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Impulse",       "Gold Fever (Impulse) (EPOCH)",MACHINE_IS_SKELETON_MECHANICAL )

/* minimal header info */

GAME( 199?, ep_bubsq,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bubble & Squeak (Extreme) (EPOCH) (BASQ 0.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bubsqa,  ep_bubsq,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bubble & Squeak (Extreme) (EPOCH) (BASQ 0.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bubsqb,  ep_bubsq,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bubble & Squeak (Extreme) (EPOCH) (BASQ 0.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bubsqc,  ep_bubsq,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bubble & Squeak (Extreme) (EPOCH) (BASQ 0.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_bubsqd,  ep_bubsq,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Bubble & Squeak (Extreme) (EPOCH) (BASQ 0.4, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cashn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cashino (Maygay - Extreme) (EPOCH) (CSHI 1.0, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cashna,  ep_cashn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Cashino (Maygay - Extreme) (EPOCH) (CSHI 1.0, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_inca,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Inca Dinka Do (Maygay - Extreme) (EPOCH) (INCA 1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_incaa,   ep_inca,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Inca Dinka Do (Maygay - Extreme) (EPOCH) (INCA 1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_incab,   ep_inca,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Inca Dinka Do (Maygay - Extreme) (EPOCH) (INCA 1.1, set 3)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_jakbn,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Jackpot & The Beanstreak (Extreme) (EPOCH) (JABS 0.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_jakbna,  ep_jakbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Jackpot & The Beanstreak (Extreme) (EPOCH) (JABS 0.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_jakbnb,  ep_jakbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Jackpot & The Beanstreak (Extreme) (EPOCH) (JABS 0.5, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_jakbnc,  ep_jakbn,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Jackpot & The Beanstreak (Extreme) (EPOCH) (JABS 0.5, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_loadd,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Loaded (Maygay) (EPOCH) (LOAD 1.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_loadda,  ep_loadd,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Loaded (Maygay) (EPOCH) (LOAD 1.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_lukld,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Lucky Ladders (Maygay) (EPOCH) (LULA 0.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_luklda,  ep_lukld,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Lucky Ladders (Maygay) (EPOCH) (LULA 0.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_noter,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 0.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_notera,  ep_noter,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 0.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_noterb,  ep_noter,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 0.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_noterc,  ep_noter,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 0.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_noterd,  ep_noter,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 1.0, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_notere,  ep_noter,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Note Runner (Maygay) (EPOCH) (NORU 1.0, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_pwrpl,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Power Play (Maygay) (EPOCH) (PPLY 0.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_pwrpla,  ep_pwrpl,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Power Play (Maygay) (EPOCH) (PPLY 0.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_scrm,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Screamin Demon (Maygay) (EPOCH) (SCDE 2.0, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_scrma,   ep_scrm,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Screamin Demon (Maygay) (EPOCH) (SCDE 2.0, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_scrmb,   ep_scrm,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Screamin Demon (Maygay) (EPOCH) (SCDE 1.0, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_scrmc,   ep_scrm,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Screamin Demon (Maygay) (EPOCH) (SCDE 1.0, set 4)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_tree,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tree Amigos (Maygay) (EPOCH) (TRAM 0.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_treea,   ep_tree,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Tree Amigos (Maygay) (EPOCH) (TRAM 0.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_vipjv,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Viper Jive (Maygay - Extreme) (EPOCH) (JIVE 1.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_vipjva,  ep_vipjv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Viper Jive (Maygay - Extreme) (EPOCH) (JIVE 1.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_vipjvb,  ep_vipjv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Viper Jive (Maygay - Extreme) (EPOCH) (JIVE 2.2, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_vipjvc,  ep_vipjv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Viper Jive (Maygay - Extreme) (EPOCH) (JIVE 2.2, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_vipjvd,  ep_vipjv,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Maygay",        "Viper Jive (Maygay - Extreme) (EPOCH) (JIVE 2.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_braid,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_braida,  ep_braid,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_braidb,  ep_braid,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.5, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_braidc,  ep_braid,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.5, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_braidd,  ep_braid,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.5, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_braide,  ep_braid,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Bank Raid (Extreme) (EPOCH) (BARA 0.5, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_cyc,     0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.2, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cyca,    ep_cyc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.2, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cycb,    ep_cyc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.3, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cycc,    ep_cyc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.3, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cycd,    ep_cyc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.1, set 5)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_cyce,    ep_cyc,     maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Cyclone (Extreme) (EPOCH) (CYCL 0.1, set 6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_ll,      0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Lucky Ladders (Extreme) (EPOCH) (LULA 0.3, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_lla,     ep_ll,      maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Lucky Ladders (Extreme) (EPOCH) (LULA 0.3, set 2)",MACHINE_IS_SKELETON_MECHANICAL )


GAME( 199?, ep_redl,    0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Red Line (Extreme) (EPOCH) (RELI 0.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_redla,   ep_redl,    maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Red Line (Extreme) (EPOCH) (RELI 0.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_snset,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Sunset Strip (Extreme) (EPOCH) (SUST 0.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_snseta,  ep_snset,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Sunset Strip (Extreme) (EPOCH) (SUST 0.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_treas,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Global",        "Treasure Hunt (Global) (EPOCH) (Version 1.6)",MACHINE_IS_SKELETON_MECHANICAL )

GAME( 199?, ep_twarp,   0,          maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Time Warp (Extreme) (EPOCH) (TWRP 0.1, set 1)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_twarpa,  ep_twarp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Time Warp (Extreme) (EPOCH) (TWRP 0.1, set 2)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_twarpb,  ep_twarp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Time Warp (Extreme) (EPOCH) (TWRP 0.4, set 3)",MACHINE_IS_SKELETON_MECHANICAL )
GAME( 199?, ep_twarpc,  ep_twarp,   maygayep,  maygayep, maygayep_state, init_maygayep, ROT0, "Extreme",       "Time Warp (Extreme) (EPOCH) (TWRP 0.4, set 4)",MACHINE_IS_SKELETON_MECHANICAL )
