/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */



/*!
 *  \file pdl_rule.c
 *
 *  \brief Implementation of the pdl rules.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 15701 $
 *  \date    $Date: 2011-12-13 12:06:42 +0100 (Tue, 13 Dec 2011) $
 *
 */


#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lcmaps_log.h"
#include "pdl_rule.h"
#include "pdl_policy.h"
#include "pdl_variable.h"

rule_t *top_rule  = NULL;
rule_t *last_rule = NULL;
static BOOL add_new_rules = TRUE;

rule_t* _lcmaps_add_rule(record_t* state, record_t* true_branch, record_t* false_branch);
/* const rule_t* lcmaps_find_state(const rule_t* rule, const char* state); */
recursion_t lcmaps_has_recursion(rule_t* rule, unsigned int* list, unsigned int depth, unsigned int* seen_rules);
int lcmaps_find_insert_position(unsigned int* list, unsigned int my_rule_number, unsigned int high);
unsigned int lcmaps_rule_number(rule_t* rule);
BOOL lcmaps_make_list(unsigned int* new_list, unsigned int* list, unsigned int my_rule_number, unsigned int depth);
unsigned int lcmaps_count_rules(rule_t* rule);
void lcmaps_update_list(unsigned int* rules, unsigned int rule);
rule_t* lcmaps_get_rule_number(unsigned int rule_num);


/*!
 *  Start a new list of rules.
 */
void lcmaps_start_new_rules(void)
{
  /* lcmaps_log_debug(1, "start_new_rule: starting new rules.\n"); */
  top_rule = last_rule = 0;
}


/*!
 *  Is it allowed to add new rules?
 *
 *  \param allow  TRUE if adding new rules is allowed, FALSE otherwise.
 *
 */
void lcmaps_allow_new_rules(BOOL allow)
{
  add_new_rules = allow;
}


/*!
 *  Add a new rule to the list of rules. This function acts as a
 *  wrapper function for _lcmaps_add_rule().
 *
 *  \param state        Starting state
 *  \param true_branch  True transit state
 *  \param false_branch False transit state
 *
 */
rule_t* lcmaps_add_rule(record_t* state, record_t* true_branch,
                 record_t* false_branch)
{
  /*
   *  This is for a somewhat dirty hack.
   *
   *  The yacc parser cannot determine the end of expression of the 
   *  form 'a -> b'. This is because 'a -> b' might be followed
   *  by '| c'. Therefore, yacc continues to scan on the next line.
   *  Lex on the other hand simply looks at new lines and increases
   *  the line counter each time it sees one. When an error needs
   *  to be reported, the global line number might be advanced too
   *  far. To counter act this problem, the state variable contains
   *  the right line number, i.e. the line where the current rule
   *  was started. By setting the the global lineno variable to the
   *  correct line number; doing out stuff; and finally setting back
   *  the lineno variable to its original value, we circumvent the
   *  problem of misreported line nubmers.
   */
  rule_t* rule = NULL;

  if (!(rule = _lcmaps_add_rule(state, true_branch, false_branch))) {
    free(state->string);
    if (true_branch)  free(true_branch->string);
    if (false_branch) free(false_branch->string);
  }

  free(state);
  if (true_branch)  free(true_branch);
  if (false_branch) free(false_branch);

  return rule;
}


/*!
 *  Rules come in three different forms:
 *    -#  a -> b
 *    -#  a -> b | c
 *    -# ~a ->b 
 *
 *  They share a common structure. First the left hand side gives
 *  the starting state and right hand side the states to transit
 *  to. This means that each rule has a starting state and depending
 *  on the form one or two transit states:
 *    - The first form has only the true transit state;
 *    - The second form had both true and false transit states;
 *    - The thrird for has only the false transit state.
 *  When either the true or false transit state for a rule does not
 *  exists, 0 should be supplied.
 *
 *  \param state        Starting state
 *  \param true_branch  True transit state
 *  \param false_branch False transit state
 *
 *  \return TRUE if the rule has been added successfully, FALSE otherwise.
 *
 */
rule_t* _lcmaps_add_rule(record_t* state, record_t* true_branch,
                  record_t* false_branch)
{
  rule_t* rule     = NULL;
  policy_t* policy = NULL;

  if ((policy = lcmaps_find_policy(state->string))) {
    lcmaps_warning(PDL_ERROR, "Left hand side of a rule cannot be a policy; see also line %d.", policy->lineno);
    return 0;
  }

  /* cast away constness. */
  if ((rule = lcmaps_find_state(top_rule, state->string))) {
    lcmaps_warning(PDL_ERROR, "State '%s' is already in use. See line %d.\n", state->string, rule->lineno);
    return 0;
  }

  if ((true_branch && (policy = lcmaps_find_policy(true_branch->string)))
      || (false_branch && (policy = lcmaps_find_policy(false_branch->string))))
    lcmaps_warning(PDL_ERROR, "Rule contians reference to a policy. This is currently not supported.");
  /*  Do not return 0 if a warning has been issued. Ignore the fact that  *
   *  a policy rule has been used. Use its name as a module instead.      */

  /*  Check if we just needed to check the rule for errors, or
      add the rule after checking as well. */
  if (!add_new_rules)
    return 0;

  if (!(rule = (rule_t *)malloc(sizeof(rule_t)))) {
    lcmaps_warning(PDL_ERROR, "out of memory.");
    return 0;
  }

  rule->state        = state->string;
  rule->true_branch  = true_branch  ? true_branch->string  : 0;
  rule->false_branch = false_branch ? false_branch->string : 0;
  rule->lineno       = state->lineno;
  rule->next         = 0;

  if (top_rule)
    last_rule->next = rule;
  else
    top_rule = rule;
  
  last_rule = rule;

  return rule;
}


/*!
 *  Find a state with name state.
 *
 *  \param state Name of the state to be found.
 *  \param rule  list of rules to be searched
 *  \return Rule which contains the state or 0 when no such rule could
 *          be found.
 *
 */
rule_t* lcmaps_find_state(rule_t* rule, char* state)
{
  if (!rule || !state)
    return NULL;

  while (rule && strcmp(state, rule->state))
    rule = rule->next;

  return rule;
}


/*!
 *  Check the rule for occurances of recursion.
 *
 *  \return TRUE if a recursion have been found, FALSE otherwise.
 *
 */
BOOL lcmaps_check_rule_for_recursion(rule_t* rule)
{
  unsigned int num_rules = lcmaps_count_rules(rule);
  unsigned int *rules = NULL;
  recursion_t rc;
  int i = 0;

  rules = (unsigned int*)calloc((num_rules+1), sizeof(*rules));

  top_rule = rule;

  rc = lcmaps_has_recursion(rule, 0, 0, rules);

  if (rules[0] != num_rules) {
    int j=1;
    for (i=1; i<=num_rules; i++) {
      if (rules[j] != i) {
        lineno = lcmaps_get_rule_number(i-1)->lineno;
        lcmaps_warning(PDL_WARNING, "rule is not part of the chain.");
      } else
        j++;
    }
  }

  free(rules);

  return (rc & RECURSION) ? TRUE : FALSE;
}


/*!
 *  Count the number of rules that follow 'rule' inclusive.
 *
 *  \param rule  The rule to start count from.
 *  \return Number of counted rules.
 *
 */
unsigned int lcmaps_count_rules(rule_t* rule)
{
  unsigned int c=0;

  while (rule) {
    c++;
    rule = rule->next;
  }

  return c;
}


/*!
 *  Give the position of the rule in the policy, return that rule.
 *
 *  \param rule_num  Position of the rule in the current policy.
 *  \return Rule that is associated with the rule_num, NULL if the
 *          rule cannot be found.
 *
 */
rule_t* lcmaps_get_rule_number(unsigned int rule_num)
{
  unsigned int i = 0;
  rule_t* r = NULL;

  for (i=0, r=top_rule; r && i<rule_num; i++)
    r = r->next;

  return r;
}


/*!
 *  Check the a rule for recursion. This is done in a recursive
 *  manner.  From the top rule, all possible paths are considered.
 *  Each path becomes a top of its own and from their all possible
 *  paths are traveled. Each time the tree is searched at a greater
 *  depth, a list is kept to tell which states have been seen for
 *  the current path. In this list of states no duplicates should
 *  be present. If a seen state state already appears in the list,
 *  the path taken is recursive. This information is propagated back
 *  up the traveled tree.
 *
 *  At the same time another list is maintained. In this list all
 *  visited states are remembered. Duplicates are not added. When all
 *  possible paths have been traveled, the list tells all visited
 *  rules. When a particular rule is not part of the tree, it is also
 *  not listed in the list. This way one can check for disconnected
 *  rules.
 *
 *  \param rule  Rule to check for recursion.
 *  \param list  List to keep track of each traveled path.
 *  \param depth Current depth of the tree.
 *  \param seen_rules  Rules that have been visited and hence are part
 *                     of the path.
 *
 *  \return Whether or not recursion has been detected and also if it
 *          has been reported.
 *
 */
recursion_t lcmaps_has_recursion(rule_t* rule, unsigned int* list, unsigned int depth,
                          unsigned int* seen_rules)
{
  unsigned int rule_num = 0;
  recursion_t rc;
  unsigned int* new_list = NULL;

  ++depth;

  /*
   *  If there is no more rule we have reached the end and hance did
   *  no encounter any recursion.
   */
  if (!rule)
    return NO_RECURSION;

  /*
   *  Allocate memory to hold a new list. The number of elements in
   *  the equals the depth of the tree.
   */
  new_list = (unsigned int*)malloc(depth*sizeof(unsigned int));
  rc       = NO_RECURSION;
  rule_num = lcmaps_rule_number(rule);

  /* Update the list of visited rules. */
  lcmaps_update_list(seen_rules, rule_num);

  /*
   *  First make a new list, this list is based upon the current list.
   *  The list is extended with the current state. If the list cannot
   *  be created, because the current state is already in the list, we
   *  have found recursion and the current path is abandoned.
   */
  if (lcmaps_make_list(new_list, list, rule_num, depth)) {
    /* No recursion yet, keep on looking... */
    recursion_t rcl=rc, rcr=rc;

    /* If present, check if the true branch has any recursion. */
    if (rule->true_branch) {
      rcl = lcmaps_has_recursion(lcmaps_find_state(top_rule, rule->true_branch), new_list, depth, seen_rules);

      /*
       *  Check the return condition. It tells whether recursion has
       *  been found. If so, it also tells if the recursion has been
       *  reported. This is to prevent multiple error messages on the
       *  same rule for the same path.
       *
       *  NOTE: When the same rule causes a recursion in a different
       *        path, the error is displayed for each of these paths.
       *
       */
      if ((rcl&RECURSION) && !(rcl&RECURSION_HANDLED)) {
  lineno = rule->lineno;
  if (rule->false_branch)
    lcmaps_warning(PDL_ERROR, "rule  %s -> %s | %s causes infinite loop on true transition %s.", rule->state, rule->true_branch, rule->false_branch, rule->true_branch);
  else
    lcmaps_warning(PDL_ERROR, "rule  %s -> %s causes infinite loop on transition %s.", rule->state, rule->true_branch, rule->true_branch);

  /*
   *  Set the handled bit to indicate that the current recursion
   *  has been reported.
   */
  rcl |= RECURSION_HANDLED;
      }
    }

    /* If present, check if the false branch has any recursion. */
    if (rule->false_branch) {
      rcr = lcmaps_has_recursion(lcmaps_find_state(top_rule, rule->false_branch), new_list, depth, seen_rules);

      if ((rcr&RECURSION) && !(rcr&RECURSION_HANDLED)) {
  lineno = rule->lineno;
  if (rule->true_branch)
    lcmaps_warning(PDL_ERROR, "rule  %s -> %s | %s causes infinite loop on false transition %s.", rule->state, rule->true_branch, rule->false_branch, rule->false_branch);
  else
    lcmaps_warning(PDL_ERROR, "rule ~%s -> %s causes infinite loop on transition %s.", rule->state, rule->false_branch, rule->false_branch);

  rcr |= RECURSION_HANDLED;
      }
    }

    /*
     *  Join the results of both the true branch and false branch
     *  together. This is the result for the current node in the tree.
     */
    rc = rcl|rcr;
  } else
    /* Well, there it is, recursion! */
    rc = RECURSION;

  /*
   *  Let's be nice and free the allocated memory to hold the traveled
   *  path.
   */
  free(new_list);

  return rc;
}


/*!
 *  Given a rule, find the corresponding position in the policy.
 *
 *  \param rule  Rule of which the position must be found.
 *  \return Position of the rule in the current policy.
 *
 */
unsigned int lcmaps_rule_number(rule_t* rule)
{
  int n = 0;
  rule_t* t = NULL;

  for (n=0, t=top_rule; t; ++n, t=t->next) {
    if (t==rule) break;
  }

  return n;
}


/*!
 *  Update the list that hold the visited rules. This is a sorted
 *  list for easy insertion and look-up. Duplicate rules are not
 *  inserted. The first element of the list tells the total number
 *  of elements that follow.
 *
 *  \note  The list expects rules to be numbered starting from 1.
 *         This is because 0 denotes empty cells. The
 *         lcmaps_find_insert_position() returns numbers starting from 0.
 *         This is corrected for in this function.
 *
 *  \param rules  List of visited rules.
 *  \param rule   rule to insert.
 *
 */
void lcmaps_update_list(unsigned int* rules, unsigned int rule)
{
  /* Rules in the list start at 1. */
  int p = 1 + lcmaps_find_insert_position(rules+1, rule, rules[0]);

  /* Same here, rules in the list start at 1. */
  rule++;

  /* Find if the rule number needs to be inserted. */
  if (rules[p] != rule) {
    if (p <= rules[0])
      memmove(rules+p+1, rules+p, (1+rules[0]-p)*sizeof(unsigned int));
    rules[p] = rule;

    /*
     *  Do not forget to reflect the fact that a new rule has been
     *  added to the list.
     */
    rules[0]++;
  }
}


/*!
 *  Based on a sorted list, find the position where to insert an new
 *  element without disturbing the ordering in the list. The search
 *  is a binary search.
 *
 *  \param list         List of sorted numbers.
 *  \param lcmaps_rule_number  Number to be inserted.
 *  \param high         Element number of last element in the list.
 *
 *  \return Position of insertion.
 *
 */
int lcmaps_find_insert_position(unsigned int* list, unsigned int my_rule_number, unsigned int high)
{
  int low=0, mid;

  /*
   *  low and high specify the current part of the list to be
   *  examined. Once the search space has become zero, the insertion
   *  position has been found.
   */
  while (low<high) {
    /* Determine the middle of the list. */
    mid = (low+high)/2;

    /* Determine on which side of the middle the insertion point lays. */
    if (my_rule_number < list[mid])
      high = mid;
    else
      low = mid + 1;
  }

  /* High points to the correct position now. */
  return high;
}


/*!
 *  Make a new sorted list based on the current list and the element
 *  to be inserted. The element will only be added to the list if it
 *  is not already present.
 *
 *  \param new_list     New list after sorted insertion of new element.
 *  \param list         Old list.
 *  \param lcmaps_rule_number  Number to be inserted into the list.
 *  \param depth        Current depth of the tree. It is used to detmine
 *                      the number of elements of the list.
 *
 *  \return TRUE if element has been added, FALSE otherwise
 *
 */
BOOL lcmaps_make_list(unsigned int* new_list, unsigned int* list, unsigned int my_rule_number,
                      unsigned int depth)
{
  int insert = 0;

  if (!list) {
    *new_list = my_rule_number;
    return TRUE;
  }

  insert = lcmaps_find_insert_position(list, my_rule_number, depth-1);

  if ((insert>0) && (list[insert-1] == my_rule_number))
    return FALSE;

  memcpy(new_list, list, insert*sizeof(int));
  if ((depth-insert-1)>0)
    memcpy(new_list+insert+1, list+insert, (depth-insert-1)*sizeof(int));

  new_list[insert] = my_rule_number;

  return TRUE;
}


/*!
 *  Reduce a rule to its elementry form, i.e. all variables in the
 *  rule are substituted by their respective values.
 *
 *  \param rule Rule to reduce.
 *
 */
void lcmaps_reduce_rule(rule_t* rule)
{
  /*
   *  The state part of the rule can be a variable. It cannot be
   *  another policy rule. Therefore, it can be either reduced
   *  to a variable, or the state part is as it is.
   */
  lcmaps_reduce_to_var(&rule->state, STATE);

  /*
   *  In case of the true branch of a rule, it can be one of three
   *  things:
   *    1 - a variable;
   *    2 - a policy rule;
   *    3 - a state.
   *  Therefore, the value of the true branch is first reduced to
   *  a variable. If not successful, it is reduced to a policy rule;
   *  which is reduced to a set of rules. If it is not a variable or
   *  policy rule, the value is taken as it is.
   */
  lcmaps_reduce_to_var(&rule->true_branch, TRUE_BRANCH);

  /* See comment on the previous statements. */
  lcmaps_reduce_to_var(&rule->false_branch, FALSE_BRANCH);
}


rule_t* lcmaps_get_rule(char* rule, side_t side)
{
  rule_t* r = top_rule;

  switch (side) {
  case left_side:
    while (r && (strcmp(r->state, rule)!=0)) {
      r = r->next;
    }
    break;
  case right_side:
    while (r && ((r->true_branch && (strcmp(r->true_branch, rule)!=0)) ||
          ((r->false_branch && strcmp(r->false_branch, rule)!=0)))) {
      r = r->next;
    }
    break;
  default:
    r = 0;
  }

  return r;
}


/*!
 *  Show a rule and its descendants.
 *
 *  \param rule Rule to display.
 *
 */
void lcmaps_show_rules(rule_t* rule)
{
  while (rule) {
    if (rule->true_branch) {
      if (!rule->false_branch) 
  lcmaps_log_debug(1, " %s -> %s\n", rule->state, rule->true_branch);
      else
  lcmaps_log_debug(1, " %s -> %s | %s\n", rule->state, rule->true_branch, rule->false_branch);
    } else
      lcmaps_log_debug(1, "~%s -> %s\n", rule->state, rule->false_branch);

    rule = rule->next;
  }
}


/*!
 *  Free all resources associated with the rules.
 *
 *  \param rule Rule for which the resources must be freed.
 *
 */
void lcmaps_free_rules(rule_t* rule)
{
  rule_t* tmp = NULL;

  while (rule) {
    tmp = rule->next;
    free((char *)rule->state);        rule->state        = NULL;  /*  Does not hurt to clear it... */
    free((char *)rule->true_branch);  rule->true_branch  = NULL;  /*  Does not hurt to clear it... */
    free((char *)rule->false_branch); rule->false_branch = NULL;  /*  Does not hurt to clear it... */
    free((char *)rule);
    rule = tmp;
  }
}


/*!
 *  Get the top rule.
 *
 *  \return Top rule.
 *
 */
rule_t* lcmaps_get_top_rule(void)
{
  return top_rule;
}


/*!
 *  Set the top rule to a new value.
 *
 *  \param rule  New value of top rule.
 *
 */
void lcmaps_set_top_rule(rule_t* rule)
{
  top_rule = rule;
}
