/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifndef _DISKSTORAGE_H
#define _DISKSTORAGE_H

#include <libxml++/libxml++.h>
#include <iostream>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <gtkmm.h>
#include "Book.h"

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif

using namespace std;

enum STORAGE_ERRORS {
  STORAGE_ERROR_MAKEDIR_FAILED,
  STORAGE_ERROR_DIR_OPEN_FAILED,
  STORAGE_ERROR_FILE_DELETE_FAILED,
  STORAGE_ERROR_PERMISSION_DENIED,
  STORAGE_ERROR_NO_SUCH_FILE,
  STORAGE_ERROR_XML_EXCEPTION,
  STORAGE_ERROR_XML,
  STORAGE_ERROR_EXCEPTION
};

class DiskStorage : public sigc::trackable {
public:
  DiskStorage();
  ~DiskStorage();
  
  // Emitted whenever load_all() loaded a book.
  SigC::Signal1<void, Book*> signal_book_loaded;
  
  /* Initialize the DiskStorage using the given directory.
   * If the directory does not yet exist, it will be created.
   * Returns an error code, or 0 on success. */
  int init(string directory);
  
  /* Loads all files from the directory structure into newly created Book
   * objects and emits them in a signal_book_loaded().
   * Returns an error code, or 0 on success. */
  int load_all(void);
  
  /* Saves a book to the disk. Returns an error code, or 0 on success. */
  int save_book(Book* book);
  
  /* Deletes a book from the disk. Returns an error code, or 0 on success. */
  int delete_book(Book* book);
  
protected:
  /* Does the work for load_all. */
  int load_all_recursive(string directory);
  
  /* Load data from a file into the given Book object. */
  int load_book(string filename, Book* book);
  
  /* Copy data from an xml-structure into the given Book object. */
  int xml2book(const xmlpp::Node* rootnode, Book* book);
  
  string directory;
};

#endif /* _STORAGE_H */
