/*
 * Copyright (c) 2019-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include <algorithm>

#include "arm_gemm.hpp"

#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void a64_smallK_hybrid_s8s32_dot_8x4(const int8_t *A, int lda, const int8_t *B, int32_t *C, int ldc, int M, int N, int K, const int32_t *, Activation, bool) {
    const long loops_count = iceildiv(N, (int)4) - 1;
    const long ldab = lda * sizeof(int8_t);
    const long ldcb = ldc * sizeof(int32_t);
    const long odds_count = K % 4;
    K = (K + 3) / 4;

    for (int y0=0; y0<M; y0+=8) {
        long loops = loops_count;
        long oob_rows = std::max(8 - (M-y0), 0);
        long odds = odds_count;
        const int8_t *b_ptr0 = B;
        const int8_t *a_ptr0 = A + (y0 * lda);

        int32_t *c_ptr0 = C + (y0 * ldc);

        switch(K) {
            case 1:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr s0, [%[a_ptr0]]\n"
                    "ldr s1, [a_ptr1]\n"
                    "ldr s2, [a_ptr2]\n"
                    "ldr s3, [a_ptr3]\n"
                    "ldr s4, [a_ptr4]\n"
                    "ldr s5, [a_ptr5]\n"
                    "ldr s6, [a_ptr6]\n"
                    "ldr s7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ldr b0, [%[a_ptr0]]\n"
                    "ldr b1, [a_ptr1]\n"
                    "ldr b2, [a_ptr2]\n"
                    "ldr b3, [a_ptr3]\n"
                    "ldr b4, [a_ptr4]\n"
                    "ldr b5, [a_ptr5]\n"
                    "ldr b6, [a_ptr6]\n"
                    "ldr b7, [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h0, [%[a_ptr0]], #0x2\n"
                    "ldr h1, [a_ptr1], #0x2\n"
                    "ldr h2, [a_ptr2], #0x2\n"
                    "ldr h3, [a_ptr3], #0x2\n"
                    "ldr h4, [a_ptr4], #0x2\n"
                    "ldr h5, [a_ptr5], #0x2\n"
                    "ldr h6, [a_ptr6], #0x2\n"
                    "ldr h7, [a_ptr7], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[2], [a_ptr1]\n"
                    "ld1 {v2.b}[2], [a_ptr2]\n"
                    "ld1 {v3.b}[2], [a_ptr3]\n"
                    "ld1 {v4.b}[2], [a_ptr4]\n"
                    "ld1 {v5.b}[2], [a_ptr5]\n"
                    "ld1 {v6.b}[2], [a_ptr6]\n"
                    "ld1 {v7.b}[2], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 2:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr d0, [%[a_ptr0]]\n"
                    "ldr d1, [a_ptr1]\n"
                    "ldr d2, [a_ptr2]\n"
                    "ldr d3, [a_ptr3]\n"
                    "ldr d4, [a_ptr4]\n"
                    "ldr d5, [a_ptr5]\n"
                    "ldr d6, [a_ptr6]\n"
                    "ldr d7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr s0, [%[a_ptr0]], #0x4\n"
                    "ldr s1, [a_ptr1], #0x4\n"
                    "ldr s2, [a_ptr2], #0x4\n"
                    "ldr s3, [a_ptr3], #0x4\n"
                    "ldr s4, [a_ptr4], #0x4\n"
                    "ldr s5, [a_ptr5], #0x4\n"
                    "ldr s6, [a_ptr6], #0x4\n"
                    "ldr s7, [a_ptr7], #0x4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[4], [a_ptr1]\n"
                    "ld1 {v2.b}[4], [a_ptr2]\n"
                    "ld1 {v3.b}[4], [a_ptr3]\n"
                    "ld1 {v4.b}[4], [a_ptr4]\n"
                    "ld1 {v5.b}[4], [a_ptr5]\n"
                    "ld1 {v6.b}[4], [a_ptr6]\n"
                    "ld1 {v7.b}[4], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[2], [a_ptr1], #2\n"
                    "ld1 {v2.h}[2], [a_ptr2], #2\n"
                    "ld1 {v3.h}[2], [a_ptr3], #2\n"
                    "ld1 {v4.h}[2], [a_ptr4], #2\n"
                    "ld1 {v5.h}[2], [a_ptr5], #2\n"
                    "ld1 {v6.h}[2], [a_ptr6], #2\n"
                    "ld1 {v7.h}[2], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[6], [a_ptr1]\n"
                    "ld1 {v2.b}[6], [a_ptr2]\n"
                    "ld1 {v3.b}[6], [a_ptr3]\n"
                    "ld1 {v4.b}[6], [a_ptr4]\n"
                    "ld1 {v5.b}[6], [a_ptr5]\n"
                    "ld1 {v6.b}[6], [a_ptr6]\n"
                    "ld1 {v7.b}[6], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 3:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr d0, [%[a_ptr0]], #0x8\n"
                    "ldr d1, [a_ptr1], #0x8\n"
                    "ldr d2, [a_ptr2], #0x8\n"
                    "ldr d3, [a_ptr3], #0x8\n"
                    "ldr d4, [a_ptr4], #0x8\n"
                    "ldr d5, [a_ptr5], #0x8\n"
                    "ldr d6, [a_ptr6], #0x8\n"
                    "ldr d7, [a_ptr7], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v0.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v1.s}[2], [a_ptr1]\n"
                    "ld1 {v2.s}[2], [a_ptr2]\n"
                    "ld1 {v3.s}[2], [a_ptr3]\n"
                    "ld1 {v4.s}[2], [a_ptr4]\n"
                    "ld1 {v5.s}[2], [a_ptr5]\n"
                    "ld1 {v6.s}[2], [a_ptr6]\n"
                    "ld1 {v7.s}[2], [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[8], [a_ptr1]\n"
                    "ld1 {v2.b}[8], [a_ptr2]\n"
                    "ld1 {v3.b}[8], [a_ptr3]\n"
                    "ld1 {v4.b}[8], [a_ptr4]\n"
                    "ld1 {v5.b}[8], [a_ptr5]\n"
                    "ld1 {v6.b}[8], [a_ptr6]\n"
                    "ld1 {v7.b}[8], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[4], [a_ptr1], #2\n"
                    "ld1 {v2.h}[4], [a_ptr2], #2\n"
                    "ld1 {v3.h}[4], [a_ptr3], #2\n"
                    "ld1 {v4.h}[4], [a_ptr4], #2\n"
                    "ld1 {v5.h}[4], [a_ptr5], #2\n"
                    "ld1 {v6.h}[4], [a_ptr6], #2\n"
                    "ld1 {v7.h}[4], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[10], [a_ptr1]\n"
                    "ld1 {v2.b}[10], [a_ptr2]\n"
                    "ld1 {v3.b}[10], [a_ptr3]\n"
                    "ld1 {v4.b}[10], [a_ptr4]\n"
                    "ld1 {v5.b}[10], [a_ptr5]\n"
                    "ld1 {v6.b}[10], [a_ptr6]\n"
                    "ld1 {v7.b}[10], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 4:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q0, [%[a_ptr0]]\n"
                    "ldr q1, [a_ptr1]\n"
                    "ldr q2, [a_ptr2]\n"
                    "ldr q3, [a_ptr3]\n"
                    "ldr q4, [a_ptr4]\n"
                    "ldr q5, [a_ptr5]\n"
                    "ldr q6, [a_ptr6]\n"
                    "ldr q7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr d0, [%[a_ptr0]], #0x8\n"
                    "ldr d1, [a_ptr1], #0x8\n"
                    "ldr d2, [a_ptr2], #0x8\n"
                    "ldr d3, [a_ptr3], #0x8\n"
                    "ldr d4, [a_ptr4], #0x8\n"
                    "ldr d5, [a_ptr5], #0x8\n"
                    "ldr d6, [a_ptr6], #0x8\n"
                    "ldr d7, [a_ptr7], #0x8\n"
                    "ld1 {v0.s}[2], [%[a_ptr0]], #4\n"
                    "ld1 {v1.s}[2], [a_ptr1], #4\n"
                    "ld1 {v2.s}[2], [a_ptr2], #4\n"
                    "ld1 {v3.s}[2], [a_ptr3], #4\n"
                    "ld1 {v4.s}[2], [a_ptr4], #4\n"
                    "ld1 {v5.s}[2], [a_ptr5], #4\n"
                    "ld1 {v6.s}[2], [a_ptr6], #4\n"
                    "ld1 {v7.s}[2], [a_ptr7], #4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[12], [a_ptr1]\n"
                    "ld1 {v2.b}[12], [a_ptr2]\n"
                    "ld1 {v3.b}[12], [a_ptr3]\n"
                    "ld1 {v4.b}[12], [a_ptr4]\n"
                    "ld1 {v5.b}[12], [a_ptr5]\n"
                    "ld1 {v6.b}[12], [a_ptr6]\n"
                    "ld1 {v7.b}[12], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[6], [a_ptr1], #2\n"
                    "ld1 {v2.h}[6], [a_ptr2], #2\n"
                    "ld1 {v3.h}[6], [a_ptr3], #2\n"
                    "ld1 {v4.h}[6], [a_ptr4], #2\n"
                    "ld1 {v5.h}[6], [a_ptr5], #2\n"
                    "ld1 {v6.h}[6], [a_ptr6], #2\n"
                    "ld1 {v7.h}[6], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[14], [a_ptr1]\n"
                    "ld1 {v2.b}[14], [a_ptr2]\n"
                    "ld1 {v3.b}[14], [a_ptr3]\n"
                    "ld1 {v4.b}[14], [a_ptr4]\n"
                    "ld1 {v5.b}[14], [a_ptr5]\n"
                    "ld1 {v6.b}[14], [a_ptr6]\n"
                    "ld1 {v7.b}[14], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa1ea79 // sdot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x4fa2ea7a // sdot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa3ea7b // sdot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x4fa4ea7c // sdot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa5ea7d // sdot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x4fa6ea7e // sdot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa7ea7f // sdot v31.4s, v19.16b, v7.4b[3]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa1ea79 // sdot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x4fa2ea7a // sdot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa3ea7b // sdot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x4fa4ea7c // sdot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa5ea7d // sdot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x4fa6ea7e // sdot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa7ea7f // sdot v31.4s, v19.16b, v7.4b[3]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa1ea79 // sdot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x4fa2ea7a // sdot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa3ea7b // sdot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x4fa4ea7c // sdot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa5ea7d // sdot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x4fa6ea7e // sdot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa7ea7f // sdot v31.4s, v19.16b, v7.4b[3]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f81e219 // sdot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x4f82e21a // sdot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f83e21b // sdot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x4f84e21c // sdot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f85e21d // sdot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x4f86e21e // sdot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f87e21f // sdot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa1e239 // sdot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x4fa2e23a // sdot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa3e23b // sdot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x4fa4e23c // sdot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa5e23d // sdot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x4fa6e23e // sdot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa7e23f // sdot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f81ea59 // sdot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x4f82ea5a // sdot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f83ea5b // sdot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x4f84ea5c // sdot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f85ea5d // sdot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x4f86ea5e // sdot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f87ea5f // sdot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa1ea79 // sdot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x4fa2ea7a // sdot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa3ea7b // sdot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x4fa4ea7c // sdot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa5ea7d // sdot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x4fa6ea7e // sdot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa7ea7f // sdot v31.4s, v19.16b, v7.4b[3]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 5:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr s1, [%[a_ptr0]]\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr s3, [a_ptr1]\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr s5, [a_ptr2]\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr s7, [a_ptr3]\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr s9, [a_ptr4]\n"
                    "ldr s11, [a_ptr5]\n"
                    "ldr s13, [a_ptr6]\n"
                    "ldr s15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "b.ne 4f\n"
                    "ldr b1, [%[a_ptr0]]\n"
                    "ldr b3, [a_ptr1]\n"
                    "ldr b5, [a_ptr2]\n"
                    "ldr b7, [a_ptr3]\n"
                    "ldr b9, [a_ptr4]\n"
                    "ldr b11, [a_ptr5]\n"
                    "ldr b13, [a_ptr6]\n"
                    "ldr b15, [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h1, [%[a_ptr0]], #0x2\n"
                    "ldr h3, [a_ptr1], #0x2\n"
                    "ldr h5, [a_ptr2], #0x2\n"
                    "ldr h7, [a_ptr3], #0x2\n"
                    "ldr h9, [a_ptr4], #0x2\n"
                    "ldr h11, [a_ptr5], #0x2\n"
                    "ldr h13, [a_ptr6], #0x2\n"
                    "ldr h15, [a_ptr7], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[2], [a_ptr1]\n"
                    "ld1 {v5.b}[2], [a_ptr2]\n"
                    "ld1 {v7.b}[2], [a_ptr3]\n"
                    "ld1 {v9.b}[2], [a_ptr4]\n"
                    "ld1 {v11.b}[2], [a_ptr5]\n"
                    "ld1 {v13.b}[2], [a_ptr6]\n"
                    "ld1 {v15.b}[2], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 6:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr d1, [%[a_ptr0]]\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr d3, [a_ptr1]\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr d5, [a_ptr2]\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr d7, [a_ptr3]\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr d9, [a_ptr4]\n"
                    "ldr d11, [a_ptr5]\n"
                    "ldr d13, [a_ptr6]\n"
                    "ldr d15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr s1, [%[a_ptr0]], #0x4\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr s3, [a_ptr1], #0x4\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr s5, [a_ptr2], #0x4\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr s7, [a_ptr3], #0x4\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr s9, [a_ptr4], #0x4\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr s11, [a_ptr5], #0x4\n"
                    "ldr s13, [a_ptr6], #0x4\n"
                    "ldr s15, [a_ptr7], #0x4\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[4], [a_ptr1]\n"
                    "ld1 {v5.b}[4], [a_ptr2]\n"
                    "ld1 {v7.b}[4], [a_ptr3]\n"
                    "ld1 {v9.b}[4], [a_ptr4]\n"
                    "ld1 {v11.b}[4], [a_ptr5]\n"
                    "ld1 {v13.b}[4], [a_ptr6]\n"
                    "ld1 {v15.b}[4], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[2], [a_ptr1], #2\n"
                    "ld1 {v5.h}[2], [a_ptr2], #2\n"
                    "ld1 {v7.h}[2], [a_ptr3], #2\n"
                    "ld1 {v9.h}[2], [a_ptr4], #2\n"
                    "ld1 {v11.h}[2], [a_ptr5], #2\n"
                    "ld1 {v13.h}[2], [a_ptr6], #2\n"
                    "ld1 {v15.h}[2], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[6], [a_ptr1]\n"
                    "ld1 {v5.b}[6], [a_ptr2]\n"
                    "ld1 {v7.b}[6], [a_ptr3]\n"
                    "ld1 {v9.b}[6], [a_ptr4]\n"
                    "ld1 {v11.b}[6], [a_ptr5]\n"
                    "ld1 {v13.b}[6], [a_ptr6]\n"
                    "ld1 {v15.b}[6], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 7:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr d1, [%[a_ptr0]], #0x8\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr d3, [a_ptr1], #0x8\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr d5, [a_ptr2], #0x8\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr d7, [a_ptr3], #0x8\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr d9, [a_ptr4], #0x8\n"
                    "ldr d11, [a_ptr5], #0x8\n"
                    "ldr d13, [a_ptr6], #0x8\n"
                    "ldr d15, [a_ptr7], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v1.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v3.s}[2], [a_ptr1]\n"
                    "ld1 {v5.s}[2], [a_ptr2]\n"
                    "ld1 {v7.s}[2], [a_ptr3]\n"
                    "ld1 {v9.s}[2], [a_ptr4]\n"
                    "ld1 {v11.s}[2], [a_ptr5]\n"
                    "ld1 {v13.s}[2], [a_ptr6]\n"
                    "ld1 {v15.s}[2], [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[8], [a_ptr1]\n"
                    "ld1 {v5.b}[8], [a_ptr2]\n"
                    "ld1 {v7.b}[8], [a_ptr3]\n"
                    "ld1 {v9.b}[8], [a_ptr4]\n"
                    "ld1 {v11.b}[8], [a_ptr5]\n"
                    "ld1 {v13.b}[8], [a_ptr6]\n"
                    "ld1 {v15.b}[8], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[4], [a_ptr1], #2\n"
                    "ld1 {v5.h}[4], [a_ptr2], #2\n"
                    "ld1 {v7.h}[4], [a_ptr3], #2\n"
                    "ld1 {v9.h}[4], [a_ptr4], #2\n"
                    "ld1 {v11.h}[4], [a_ptr5], #2\n"
                    "ld1 {v13.h}[4], [a_ptr6], #2\n"
                    "ld1 {v15.h}[4], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[10], [a_ptr1]\n"
                    "ld1 {v5.b}[10], [a_ptr2]\n"
                    "ld1 {v7.b}[10], [a_ptr3]\n"
                    "ld1 {v9.b}[10], [a_ptr4]\n"
                    "ld1 {v11.b}[10], [a_ptr5]\n"
                    "ld1 {v13.b}[10], [a_ptr6]\n"
                    "ld1 {v15.b}[10], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            default:
            case 8:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr q1, [%[a_ptr0]]\n"
                    "ldr q3, [a_ptr1]\n"
                    "ldr q5, [a_ptr2]\n"
                    "ldr q7, [a_ptr3]\n"
                    "ldr q9, [a_ptr4]\n"
                    "ldr q11, [a_ptr5]\n"
                    "ldr q13, [a_ptr6]\n"
                    "ldr q15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr d1, [%[a_ptr0]], #0x8\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr d3, [a_ptr1], #0x8\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr d5, [a_ptr2], #0x8\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr d7, [a_ptr3], #0x8\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr d9, [a_ptr4], #0x8\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr d11, [a_ptr5], #0x8\n"
                    "ldr d13, [a_ptr6], #0x8\n"
                    "ld1 {v1.s}[2], [%[a_ptr0]], #4\n"
                    "ldr d15, [a_ptr7], #0x8\n"
                    "ld1 {v3.s}[2], [a_ptr1], #4\n"
                    "ld1 {v5.s}[2], [a_ptr2], #4\n"
                    "ld1 {v7.s}[2], [a_ptr3], #4\n"
                    "ld1 {v9.s}[2], [a_ptr4], #4\n"
                    "ld1 {v11.s}[2], [a_ptr5], #4\n"
                    "ld1 {v13.s}[2], [a_ptr6], #4\n"
                    "ld1 {v15.s}[2], [a_ptr7], #4\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[12], [a_ptr1]\n"
                    "ld1 {v5.b}[12], [a_ptr2]\n"
                    "ld1 {v7.b}[12], [a_ptr3]\n"
                    "ld1 {v9.b}[12], [a_ptr4]\n"
                    "ld1 {v11.b}[12], [a_ptr5]\n"
                    "ld1 {v13.b}[12], [a_ptr6]\n"
                    "ld1 {v15.b}[12], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[6], [a_ptr1], #2\n"
                    "ld1 {v5.h}[6], [a_ptr2], #2\n"
                    "ld1 {v7.h}[6], [a_ptr3], #2\n"
                    "ld1 {v9.h}[6], [a_ptr4], #2\n"
                    "ld1 {v11.h}[6], [a_ptr5], #2\n"
                    "ld1 {v13.h}[6], [a_ptr6], #2\n"
                    "ld1 {v15.h}[6], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[14], [a_ptr1]\n"
                    "ld1 {v5.b}[14], [a_ptr2]\n"
                    "ld1 {v7.b}[14], [a_ptr3]\n"
                    "ld1 {v9.b}[14], [a_ptr4]\n"
                    "ld1 {v11.b}[14], [a_ptr5]\n"
                    "ld1 {v13.b}[14], [a_ptr6]\n"
                    "ld1 {v15.b}[14], [a_ptr7]\n"
                    "3:\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "cbz %[loops], 6f\n"
                    "movi v24.4s, #0\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x4fa1eaf8 // sdot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x4fa3eaf9 // sdot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x4fa5eafa // sdot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x4fa7eafb // sdot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x4fa9eafc // sdot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x4fabeafd // sdot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x4fadeafe // sdot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x4fafeaff // sdot v31.4s, v23.16b, v15.4b[3]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x4fa1eaf8 // sdot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x4fa3eaf9 // sdot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x4fa5eafa // sdot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x4fa7eafb // sdot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x4fa9eafc // sdot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x4fabeafd // sdot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x4fadeafe // sdot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x4fafeaff // sdot v31.4s, v23.16b, v15.4b[3]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x4fa1eaf8 // sdot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x4fa3eaf9 // sdot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x4fa5eafa // sdot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x4fa7eafb // sdot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x4fa9eafc // sdot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x4fabeafd // sdot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x4fadeafe // sdot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x4fafeaff // sdot v31.4s, v23.16b, v15.4b[3]\n"
                    "b 9f\n"
                    "6:\n"
                    "movi v24.4s, #0\n"
                    "movi v25.4s, #0\n"
                    "movi v26.4s, #0\n"
                    "movi v27.4s, #0\n"
                    "movi v28.4s, #0\n"
                    "movi v29.4s, #0\n"
                    "movi v30.4s, #0\n"
                    "movi v31.4s, #0\n"
                    ".inst 0x4f80e218 // sdot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x4f82e219 // sdot v25.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x4f84e21a // sdot v26.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x4f86e21b // sdot v27.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x4f88e21c // sdot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x4f8ae21d // sdot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x4f8ce21e // sdot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x4f8ee21f // sdot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x4fa0e238 // sdot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x4fa2e239 // sdot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x4fa4e23a // sdot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x4fa6e23b // sdot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x4fa8e23c // sdot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x4faae23d // sdot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x4face23e // sdot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x4faee23f // sdot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x4f80ea58 // sdot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x4f82ea59 // sdot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x4f84ea5a // sdot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x4f86ea5b // sdot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x4f88ea5c // sdot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x4f8aea5d // sdot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x4f8cea5e // sdot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x4f8eea5f // sdot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x4fa0ea78 // sdot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x4fa2ea79 // sdot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x4fa4ea7a // sdot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x4fa6ea7b // sdot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x4fa8ea7c // sdot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x4faaea7d // sdot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x4facea7e // sdot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x4faeea7f // sdot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x4f81e298 // sdot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x4f83e299 // sdot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x4f85e29a // sdot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x4f87e29b // sdot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x4f89e29c // sdot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x4f8be29d // sdot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x4f8de29e // sdot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x4f8fe29f // sdot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x4fa1e2b8 // sdot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x4fa3e2b9 // sdot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x4fa5e2ba // sdot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x4fa7e2bb // sdot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x4fa9e2bc // sdot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x4fabe2bd // sdot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x4fade2be // sdot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x4fafe2bf // sdot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x4f81ead8 // sdot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x4f83ead9 // sdot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x4f85eada // sdot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x4f87eadb // sdot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x4f89eadc // sdot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x4f8beadd // sdot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x4f8deade // sdot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x4f8feadf // sdot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x4fa1eaf8 // sdot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x4fa3eaf9 // sdot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x4fa5eafa // sdot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x4fa7eafb // sdot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x4fa9eafc // sdot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x4fabeafd // sdot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x4fadeafe // sdot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x4fafeaff // sdot v31.4s, v23.16b, v15.4b[3]\n"
                    "9:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "str q26, [c_ptr2]\n"
                    "str q27, [c_ptr3]\n"
                    "str q28, [c_ptr4]\n"
                    "str q29, [c_ptr5]\n"
                    "str q30, [c_ptr6]\n"
                    "str q31, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
        }
    }
}

} // namespace arm_gemm

#endif // __aarch64__
