---
id: tutorial-todo-gql-tx-mutation
title: Transactional Mutations
sidebar_label: Transactional Mutations
---

In this section, we continue the [GraphQL example](tutorial-todo-gql.md) by explaining how to set our GraphQL mutations
to be transactional. That means, to automatically wrap our GraphQL mutations with a database transaction and either
commit at the end, or rollback the transaction in case of a GraphQL error.

#### Clone the code (optional)

The code for this tutorial is available under [github.com/a8m/ent-graphql-example](https://github.com/a8m/ent-graphql-example),
and tagged (using Git) in each step. If you want to skip the basic setup and start with the initial version of the GraphQL
server, you can clone the repository and checkout `v0.1.0` as follows:

```console
git clone git@github.com:a8m/ent-graphql-example.git
cd ent-graphql-example 
go run ./cmd/todo/
```

## Usage

The GraphQL extensions provides a handler named `entgql.Transactioner` that executes each GraphQL mutation in a
transaction. The injected client for the resolver is a [transactional `ent.Client`](transactions.md#transactional-client).
Hence, GraphQL resolvers that uses `ent.Client` won't need to be changed. In order to add it to our todo list application
we follow these steps:

1\. Edit the `cmd/todo/main.go` and add to the GraphQL server initialization the `entgql.Transactioner` handler as
follows:

```diff title="cmd/todo/main.go"
srv := handler.NewDefaultServer(todo.NewSchema(client))
+srv.Use(entgql.Transactioner{TxOpener: client})
```

2\. Then, in the GraphQL mutations, use the client from context as follows:
```diff title="todo.resolvers.go"
func (mutationResolver) CreateTodo(ctx context.Context, todo TodoInput) (*ent.Todo, error) {
+	client := ent.FromContext(ctx)
+	return client.Todo.
-	return r.client.Todo.
		Create().
		SetText(todo.Text).
		SetStatus(todo.Status).
		SetNillablePriority(todo.Priority). // Set the "priority" field if provided.
		SetNillableParentID(todo.Parent).   // Set the "parent_id" field if provided.
		Save(ctx)
}
```

## Isolation Levels

If you'd like to tweak the transaction's isolation level, you can do so by implementing your own `TxOpener`. For example:

```go
srv.Use(entgql.Transactioner{
	TxOpener: entgql.TxOpenerFunc(func(ctx context.Context) (context.Context, driver.Tx, error) {
		tx, err := client.BeginTx(ctx, &sql.TxOptions{Isolation: sql.LevelRepeatableRead})
		if err != nil {
			return nil, nil, err
		}
		ctx = NewTxContext(ctx, tx)
		ctx = NewContext(ctx, tx.Client())
		return ctx, tx, nil
	})
})
```

---

Great! With a few lines of code, our application now supports automatic transactional mutations. Please continue to the
next section where we explain how to extend the Ent code generator and generate [GraphQL input types](https://graphql.org/graphql-js/mutations-and-input-types/)
for our GraphQL mutations.