package Pista::Object::Pic;

use strict;
use warnings;
use vars qw(@ISA);
@ISA = qw(Pista::Object);
use Pista::Section;
use Pista::Programmer;
use Pista::Util qw(min max dup);
use Pista::Device;

use Dumpvalue;
my $dumper = new Dumpvalue('quoteHighBit'=>1, 'arrayDepth'=>16);

sub new {
	my $pkg = shift;
	my $self = { programmer => shift, device => shift };
	if (!$self->{programmer}) {
		print "420 No programmer selected\n";
		return undef;
	}
	if (!$self->{device}) {
		print "440 No device selected\n";
		return undef;
	}
#$dumper->dumpValue($self);
	bless $self, $pkg;
}

sub read_bits_to_preserve {
	my $self = shift;
	my $saved;
	for my $sect (qw(cal conf)) {
		next unless exists $self->{device}->{$sect};
		my $section = $self->{device}->{$sect};
		next unless exists $section->{preserve};
		my $preserve = $section->{preserve};
		my $start = $section->{addr};
		my $end = $section->{addr} + $section->{len} - 1;
		$saved->{$sect} = $self->_do_read("read_$sect",
						$start, $end, -1, 0) or return;
#print "raw $sect:\n";
#$dumper->dumpValue($saved->{$sect});
		my $content = $saved->{$sect}->{content};
		if ($#{$preserve} > 0) {
			# we have a list of preserve masks
			for (my $i=0; $i<=$#$content; $i++) {
				$content->[$i] &= $preserve->[$i];
			}
		}
		else {
			# preserve mask is same for all element
			for (my $i=0; $i<=$#$content; $i++) {
				$content->[$i] &= $preserve->[0];
			}
		}
	}
	print "121 Saving factory preset settings\n" if $saved;
#$dumper->dumpValue($saved);
	return $saved;
}

sub read {
#print "->read(@_)\n";
	my $self = shift;
	my %arg = (@_);
	my $noblank = $arg{noblank};
	my $range = $arg{range};
	undef %arg;
	my $has_data = 0;

	if (!$range) {
		# default range
		$range = Pista::Device::setup_range($self->{device}) or return;
	}

# TODO: Rethink 2/8 (r/w) byte blocksize concept
#	if ($self->{device}->{addressing} eq 'byte') {
#	}

#$dumper->dumpValue($range);
	for my $sect (qw(prog cal userid devid conf eeprom)) {
		next unless exists $range->{$sect};
		my $section = $self->{device}->{$sect};
		my $start = max($section->{addr}, $range->{$sect}->{start});
		my $end = min($section->{addr} + $section->{len} - 1,
				$range->{$sect}->{end});
		my $blank = $section->{blank} if ($noblank);
		my $mask = $section->{mask} if ($noblank);

		if ($start <= $end) {
			printf "121 Reading $sect section 0x%x-0x%x\n",
					$start,$end;
			$self->{$sect} = $self->_do_read("read_$sect",
					$start, $end, $blank, $mask);
			return unless $self->{$sect}->{content};
			$has_data++;
#			print "121 Done\n";
		}
		else {
			print "130 Address is out of range\n";
		}
	}

	if (!$has_data) {
		print "531 No data has been read\n";
		return;
	}

	return $self;
}

sub _do_read {
	my ($self, $method, $start, $end, $blank, $mask) = @_;
#print "$method: start=$start, end=$end\n";
	no strict qw(refs);
	my $retval = $self->{programmer}->$method($self->{device},$start,$end);
#print "$self:\n";
#$dumper->dumpValue($retval);
	$retval or return;

	# drop blank values
	if ($#{$blank} > 0) {
		# we have a list of mask/blank values
		for (my $i=0; $i<=$#{$retval}; $i++) {
			delete $retval->[$i]
				if $blank->[$i] == $retval->[$i] & $mask->[$i];
		}
	}
	elsif ($blank->[0]) {
		# $mask/$blank are the same for each element
		for (my $i=0; $i<=$#{$retval}; $i++) {
			delete $retval->[$i]
				if $blank->[0] == $retval->[$i] & $mask->[0];
		}
	}
	return Pista::Section->new($start, $end, $retval);
}

sub write {
#print "->write(@_)\n";
	my $self = shift;
	my $src = shift;
	my %arg = (@_);
	my $force = $arg{force};
	undef %arg;
	my $device = $self->{device};
	my $saved;
#$dumper->dumpValue($src);

	my $empty = 1;
	for my $sect (qw(prog cal userid eeprom conf)) {
		next unless exists $src->{$sect};
		$empty = 0;
	}
	return $self if $empty;		# nothing to write

	if (!$force) {
		$saved = $self->read_bits_to_preserve() or return;
	}
#print "saved:\n";
#$dumper->dumpValue($saved);

	if ($device->{addressing} eq 'byte') {
		# Program and userid sections of 18Fxxx devices can be written
		# in 8 byte chunks only. We have to read a few bytes and to
		# prepend/append to content to write.
		# (18Cxxx devices don't require this preparation but
		# it does not hurt them.)
		$src = dup($src);		# keep orginal object intact
		for my $sect (qw(prog userid)) {
			next unless exists $src->{$sect};
			my $s = $src->{$sect};
#print "elotte:\n";
#$dumper->dumpValue($s);
			my $start = $s->{start};
			if ($start % 8) {
				my $head = $self->_do_read("read_$sect",
					$start&~7, ($start&~1)+1, undef, undef);
				return undef unless $head;
#print "head\n";
#$dumper->dumpValue($head);
				$s = $head->merge($s);
			}
			my $end = $s->{end};
			if (($end+1) % 8) {
				my $tail = $self->_do_read("read_$sect",
					($end+1)&~1, ($end&~7)+7, undef, undef);
				return undef unless $tail;
#print "tail\n";
#$dumper->dumpValue($tail);
				$s = $tail->merge($s);
			}
#print "utana\n";
#$dumper->dumpValue($s);
			$src->{$sect} = $s;
		}
	}

	for my $sect (qw(prog cal userid eeprom conf)) {
		next unless exists $src->{$sect};
		next unless exists $device->{$sect};
		my $start = $src->{$sect}->{start};
		my $end = $src->{$sect}->{end};
		my $content = $src->{$sect}->{content};
		if ($saved and exists $saved->{$sect}) {
			$content = dup($content);
			my $mask = $device->{$sect}->{preserve};
			if ($#{$mask} > 0) {
				for my $i (0..$#$content) {
					$content->[$i] &= ~$mask->[$i];
					$content->[$i] |= $saved->{$sect}->{content}->[$i];
				}
			}
			else {
				for my $i (0..$#$content) {
					$content->[$i] &= ~$mask->[0];
					$content->[$i] |= $saved->{$sect}->{content}->[$i];
				}
			}
		}
#$dumper->dumpValue($content);
		printf "121 Writing $sect section 0x%x-0x%x\n", $start, $end;
		$self->_do_write("write_$sect", $start, $end, $content) or
			return undef;
	}
	return $self;
}

sub _do_write {
	my ($self, $method, $start, $end, $content) = @_;

# TODO: do not write undef values
	no strict qw(refs);
	if (!$self->{programmer}->can($method)) {
		print "523 Unimplemented programmer method \"$method\"\n";
		return undef;
	}
	return $self->{programmer}->$method($self->{device},
					$start, $end, $content);
}

sub checksum {
	my $self = shift;
	my %arg = (@_);
	my $range = $arg{range};
	undef %arg;
	my %chksum;

#print "Object::Pic::checksum\n";
	my $device = dup($self->{device});
#$dumper->dumpValue($device);
	for my $sect (qw(prog cal userid conf eeprom)) {
#print "sect=$sect\n";
		next unless exists $device->{$sect};
		if (!exists $range->{$sect}) {
			$device->{$sect}->{len} = 0;
		}
		my $section = $self->{device}->{$sect};
		my $start = max($section->{addr}, $range->{$sect}->{start});
		my $end = min($section->{addr} + $section->{len} - 1,
				$range->{$sect}->{end});
		if ($start <= $end) {
			printf "121 Computing checksum on $sect section 0x%x-0x%x\n",
					$start,$end;
			$chksum{$sect} = $self->_do_checksum("chksum_$sect",
					$start, $end);
		}
		else {
			print "130 Address is out of range\n";
		}
	}

	my $section = $self->{device}->{prog};
	$chksum{official} = $self->_do_checksum("chksum_prog", $section->{addr},
				$section->{addr} + $section->{len} - 1);
	$section = $self->{device}->{conf};
	my @conf = @{$self->_do_read("read_conf", $section->{addr},
				$section->{addr} + $section->{len} - 1) ->
								{content} };
	for my $i (0..$#conf) {
		$chksum{official} += $conf[$i] & $section->{mask}->[$i];
	}
	$chksum{official} &= 0xffff;

	for my $sect (qw(prog cal userid conf eeprom official)) {
		next unless exists $chksum{$sect};
		printf "$sect:\t0x%.4x\n", $chksum{$sect};
	}

	return $self;
}

sub _do_checksum {
	my ($self, $method, $start, $end) = @_;
#print "$method: start=$start, end=$end\n";
	no strict qw(refs);
	return $self->{programmer}->$method($self->{device}, $start, $end)->[0];
}

sub blankcheck {
	my $self = shift;
	my %arg = (@_);
	my $range = $arg{range};
	undef %arg;

	my $device = dup($self->{device});
#$dumper->dumpValue($range);
#print "=====================\n";
	for (qw(prog cal userid conf eeprom)) {
		next unless exists $device->{$_};
		if (!exists $range->{$_}) {
			$device->{$_}->{len} = 0;
		}
		else {
			print "100 Arbitrary range is unsupported yet\n"
				if $range->{$_}->{start} != $device->{$_}->{addr};
			$device->{$_}->{len} = $range->{$_}->{end} -
						$device->{$_}->{addr} + 1;
		}
	}
	my $status = $self->{programmer}->blankcheck($device) or return;
	print "133 Blank status follows\n";
	for (qw(prog cal userid conf eeprom)) {
		next unless exists $range->{$_};
		print "$_:\t" . ($status->{$_} ? "not_blank" : "blank") . "\n";
	}
	print ".\n" . "233 Done\n";
}

# This routine must save and restore words/bits to be preserved
sub erase {
	my $self = shift;
	my %arg = (@_);
	my $noblank = $arg{noblank};
	my $eraserange = $arg{range};
	my $restore = ! $arg{norestore};
	undef %arg;
	my ($tmpbuf, $saved);

	if (! $self->{device}->{features}->{rewritable}) {
		print "540 Cannot erase OTP chip\n";
		return;
	}
	if ($eraserange) {				# Partial erasure
		# TODO: check possible range conflicts

		# Save old content into tmp buffer
		$tmpbuf = Pista::Object::Buffer->new();
		$self->copy($tmpbuf);
#print "tmpbuf1\n";
#$dumper->dumpValue($tmpbuf);
		# Then erase selected range
		$tmpbuf->erase('range' => $eraserange);
		delete $tmpbuf->{devid};
	}
#print "tmpbuf2\n";
#$dumper->dumpValue($tmpbuf);

	if ($restore) {
		# Save calibration words and bandgap bits
		$saved = $self->read_bits_to_preserve();
#print "saved:\n";
#$dumper->dumpValue($saved);
	}

	$tmpbuf = $tmpbuf || Pista::Object::Buffer->new() if $saved;
	for my $sect (qw(cal conf)) {
		last unless $saved;
		next unless exists $saved->{$sect};
		# Create a blank destination section if necessary
		$tmpbuf->{$sect} = Pista::Section->new(
				$saved->{$sect}->{start},
				$saved->{$sect}->{end},
				$self->{device}->{$sect}->{blank})
			unless exists $tmpbuf->{$sect};
#print "tmpbuf $sect:\n";
#$dumper->dumpValue($tmpbuf);
		my $dst = $tmpbuf->{$sect}->{content};
		my $src = $saved->{$sect}->{content};
		if ($tmpbuf->{$sect}->{start} != $saved->{$sect}->{start} or
		    $tmpbuf->{$sect}->{end} != $saved->{$sect}->{end}) {
			print "570 Internal program error\n";
			return undef;
		}
		my $mask = $self->{device}->{$sect}->{preserve};
		if ($#{$mask} > 0) {
			for (my $i=0; $i<=$#$dst; $i++) {
				$dst->[$i] &= ~($mask->[$i]);
				$dst->[$i] |=   $src->[$i];
			}
		}
		else {
			for (my $i=0; $i<=$#$dst; $i++) {
				$dst->[$i] &= ~$mask->[0];
				$dst->[$i] |=  $src->[$i];
			}
		}
	}

	for my $sect (qw(prog userid eeprom)) {
		next unless exists $tmpbuf->{prog};
		my $section = $self->{device}->{$sect};
		$tmpbuf->{$sect}->cleanup($section->{mask}->[0],
					$section->{blank}->[0]) or
			delete $tmpbuf->{$sect};
	}

#print "tmpbuf3\n";
#$dumper->dumpValue($tmpbuf);
#for my $sect (qw(prog userid devid eeprom cal conf)) {
# next unless exists $tmpbuf->{$sect};
# print "$sect length = ",$#{$tmpbuf->{$sect}->{content}}+1,"\n";
#}

	# Do a full chip erase
	$self->{programmer}->bulk_erase($self->{device});

#print "Restore content of tmp buffer\n";
	# Restore content of tmp buffer
	$self->write($tmpbuf, 'force' => 1) if ($tmpbuf);
	print $eraserange ? "132 Device reprogrammed\n" : "132 Device erased\n";
	return $self;
}
