package Pista::Device;

use strict;
use warnings;
use Pista::Util qw(parse_ints CCtov vtoCC);
use Pista::Section;
use Pista::Grupid;

use Dumpvalue;
my $dumper = new Dumpvalue('quoteHighBit'=>1, 'arrayDepth'=>16);

# Read a device file and fill a device structure
sub new {
	my $class = shift;
	my $GRUPID = shift;
	my $device = { dev => lc(shift) };
	my %arg = (@_);
	my $single_column = !$arg{interactive};
	undef %arg;
	if (!$device->{dev}) {
		print "500 Syntax error. Try 'help device'.\n";
		return;
	}
	if ($device->{dev} eq '--help') {	# query available devices
		if (!opendir(DIR, $GRUPID)) {
			print "550 Cannot open dir $GRUPID $!\n";
			return;
		}
		my @d = sort _picsort grep {s/\.xml$//} readdir(DIR);
		closedir(DIR);

# TODO: delete unsupported devices
# TODO: mark devices requiring an adapter

		print "111-'device pictype' - select your microcontroller\n".
			"111 Available devices:\n";
		_columnar_print($single_column, @d);
		print ".\n";
		return print "240 Device list finished\n";
	}

	# Find device file
	$device->{dev} =~ s/(p(ic)?)?(\d+)/$3/;
	my $xmlfile = "$GRUPID/$device->{dev}.xml";
	if ($device->{dev} !~ /^\w+$/ or ! -f $xmlfile) {
		print "560 Unknown device: $device->{dev}. Try 'help device'.\n";
		return;
	}

	my $grupid = Pista::Grupid::read_xml($xmlfile) or
		die "Cannot open file $xmlfile: $!\n";

	if (int($grupid->{attrib}->{format}) != 1) {
		print "561 Unsupported file format: $grupid->{attrib}->{format}\n";
		return;
	}
	$grupid = $grupid->{content};

	for my $element (@$grupid) {
		if ($element->{name} eq 'pic') {
			$device = process_element_pic($device,
						$element->{attrib},
						$element->{content});
			next;
		}
		if ($element->{name} eq 'programmer') {
			delete $element->{name};
			$device->{deferred}->{programmer} = $element;
			next;
		}
		warn "Unknown GRUPID element '$element->{name}' ignored\n";
	}
#$dumper->dumpValue($device);
	print "242 Device $device->{dev} selected\n";
	bless  $device, $class;
}

sub process_element_pic {
	my ($device, $attrib, $content) = @_;
	warn "Unknown pic attributes ignored\n" if (ref $attrib);
	for my $element (@$content) {
		if ($element->{name} eq 'addressing') {
			$device->{addressing} = $element->{attrib}->{mode};
			next;
		}
		if ($element->{name} eq 'features') {
			no warnings qw(uninitialized);
			my $f = $element->{attrib};
			$device->{features}->{parity} =
						$f->{parity} eq 'yes';
			$device->{features}->{rewritable} =
						$f->{rewritable} eq 'yes';
			next;
		}
		if ($element->{name} eq 'section') {
			$device = process_section($device,
						$element->{attrib},
						$element->{content});
			next;
		}
	}
	return $device;
}

sub process_section {
	my ($device, $attrib, $content) = @_;
	my $type = $attrib->{type};
	$device->{features}->{$type} = 1;
	my $section = $device->{$type} = {
		addr	=> parse_ints($attrib->{addr}),
		len	=> parse_ints($attrib->{len}),
	};
	$section->{end} = $section->{addr} + $section->{len} - 1;
	for my $element (@$content) {
		my $name = $element->{name};
		$section->{$name} = [ parse_ints($element->{content}->[0]) ];
	}
	$section->{blank} = $section->{mask} unless exists $section->{blank};
	$section->{width} = $device->{addressing} eq 'byte' ? 1 : 2;

	if ($type eq 'devid') {
		$section->{blank} = [ ];
		my $mask = $section->{width} == 2 ?
			$section->{mask}->[0] : CCtov($section->{mask})->[0];
		my $devidrevmask = unpack('b*', pack('S',~$mask));
		$devidrevmask =~ s/^(1+)(0.*)$/$1/;
		$devidrevmask = substr($devidrevmask . '0' x 16, 0, 16);
		$devidrevmask = unpack('S',pack('b*', $devidrevmask));
		$section->{revmask} = $section->{width} == 2 ?
				[ $devidrevmask ] : vtoCC([ $devidrevmask ]);
	}
	return $device;
}

# Find a section which an address belongs to
sub qualify {
	my $self = shift;
	my $address = shift;

	for (qw(prog cal userid devid conf eeprom)) {
		next unless exists $self->{$_};
		return $_ if ($self->{$_}->{addr} <= $address and
				$address <= $self->{$_}->{end});
	}
	return undef;
}


# Class method
sub setup_range {
	my $range = { };
	my $device = shift;
	my $rangespec = shift;
	my $at_most_possible = undef;
	if (!$rangespec) {
		$rangespec = "max_of:,prog,cal,userid,devid,eeprom,conf";
	}
	my @requests = split(/,/, $rangespec);
	undef $rangespec;
	if ($requests[0] eq 'max_of:') {
		shift @requests;
		$at_most_possible = 1;
	}
	my $NUMBER = '\dx?[\da-f]*';
	my ($start, $end, $section);

	if (!$device) {
		print "440 No device selected\n";
		return;
	}

	foreach (@requests) {
		if ($_ !~ m|^(\w+)(/(\+?)($NUMBER)([+-])($NUMBER)?)?$|i or
				!exists $device->{$1} or
				!exists $device->{$1}->{addr}) {
			next if $at_most_possible;
			print "562 Unknown section: $_\n";
			return;
		}
		$section = $1;
		my $s = $device->{$section};
		if (!$2) {
			$start = $s->{addr};
			$end = $s->{addr} + $s->{len} - 1;
			next;
		}
		$start = parse_ints($4);
		$start += $s->{addr} if ($3 eq '+');
		if (!length $6) {
			$end = $s->{addr} + $s->{len} - 1;
			next;
		}
		$end = parse_ints($6);			# end or len ?
		$end += $start-1 if ($5 eq '+');	# len
	}
	continue {
		$range->{$section} = Pista::Section->new($start, $end);
		delete $range->{$section}->{content};
	}
#$dumper->dumpValue($range);
	return $range;
}

sub _picsort {
	my ($pa, $fa, $ta, $na, $sa);	# prefix, family, type, number, suffix
	my ($pb, $fb, $tb, $nb, $sb);
	if ($a =~ /^([a-z]*)(\d\d)([a-z]?)(?:[a-z]*?)(\d+)([a-z]*)$/) {
		($pa, $fa, $ta, $na, $sa) = ($1,$2,$3,$4,$5);
	}
	else {
		return $a cmp $b;
	}
	if ($b =~ /^([a-z]*)(\d\d)([a-z]?)(?:[a-z]*?)(\d+)([a-z]*)$/) {
		($pb, $fb, $tb, $nb, $sb) = ($1,$2,$3,$4,$5);
	}
	else {
		return $a cmp $b;
	}
	return $fa <=> $fb unless $fa == $fb;
	return $ta cmp $tb unless $ta eq $tb;
	return $na <=> $nb unless $na == $nb;
	return $pa cmp $pb unless $pa eq $pb;
	return $sa cmp $sb;
}
			
sub _columnar_print {
	my $single = shift;
	my @d = @_;
	my $gap = 1;
	my $linelen = $single ? 1 : 80+$gap;	# TODO: get actual window size
	my (@l, $collen, @width, $rowlen, $row, $col, $item);

	for (@d) {
		push(@l, length($_)+$gap);
	}

	COLLEN:
	for ($collen=1; $collen<=$#l+1; $collen++) {
		# compute column widths
		@width = ();
		my $sum = 0;
		$rowlen = 0;
		for ($col=0; ; $col++) {
			$width[$col] = 0;
			for ($row=0,$item=$col*$collen;
					$row<$collen; $row++,$item++) {
				last if $item>$#l;
				$width[$col] = $l[$item]
					if $width[$col] < $l[$item];
			}
			last unless $width[$col];
			$sum += $width[$col];
			next COLLEN if $sum > $linelen;
			$rowlen = $col if $rowlen < $col;
		}
		last COLLEN;
	}
	$rowlen++;

	for ($row=0; $row<$collen; $row++) {
		my $line = '';
		for ($col=0; $col<$rowlen; $col++) {
			$item = $col*$collen + $row;
			last if $item > $#d;
			$line .= sprintf "%-*s", $width[$col], $d[$item];
		}
		$line =~ s/ *$//;
		print "$line\n" if $line;
	}
}

1;
