/*************************************************/
/* methods for class klogicIO                    */
/*                                               */
/* load/save digital circuits                    */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qstring.h>
#include <qtextstream.h>
#include <qregexp.h>

#include <klogic.h>
#include <circuit.h>
#include <klogicIO.h>
#include <xmlExport.h>
#include <xmlImport.h>
#include <xmlSymbolMap.h>
#include <version.h>

int klogicIO::file_format = klogicIO::SAVE_XML;

// static
void klogicIO::setSaveFileFormat(int new_format)
{
	file_format = new_format;
}

// static
int klogicIO::getSaveFileFormat()
{
	return file_format;
}

klogicIO::klogicIO(QString filename, Circuit *_net)
{
	net = _net;
	fname = filename;
	if (-1 == fname.find(QRegExp("\\.circuit$")))
		fname.append(".circuit");
	net->initImport();
}

klogicIO::klogicIO(QString filename)
{
	net = (Circuit *)NULL;
	fname = filename;
	if (-1 == fname.find(QRegExp("\\.circuit$")))
		fname.append(".circuit");
}

klogicIO::klogicIO(Circuit *_net)
{
	net = _net;
	fname = "";
	net->initImport();
}

// read first line of file
int klogicIO::checkFileFormat()
{	QString line1;
	QString line2;

	QFile f(fname);
	if (!f.open(IO_ReadOnly)) return -1;

	QTextStream ts(&f);
	line1 = ts.readLine();
	line2 = ts.readLine();
	f.close();

	// XML file format types
	if (line1.contains("xml")) {
		if (line2.contains(XmlSymbolMap::m_oMap[XmlSymbol::KLOGIC_MAIN])) {
			if (line2.contains(XmlObject::ATT_SUBCIRCUIT))
				return XML_SUB;
			return XML_MAIN;
		}
	}

	if (line1.contains(XmlSymbolMap::m_oMap[XmlSymbol::KLOGIC_MAIN])) {
		if (line1.contains(XmlObject::ATT_SUBCIRCUIT))
			return XML_SUB;
		return XML_MAIN;
	}

	// klogic file format types
	if (7 == type(line1))
		return KLOGIC_SUB;
	return KLOGIC_MAIN;

	// unknown file format
	return -1;
}

void klogicIO::setNet(Circuit *_net)
{
	net = _net;
	net->initImport();
}

void klogicIO::setSubFilename()
{
	if (-1 == fname.find(QRegExp("\\.sub\\.circuit$")))
		fname.replace( QRegExp("\\.circuit$"), ".sub.circuit" );
}

bool klogicIO::write(bool subnet, bool selected, int dx, int dy)
{
	return writeXML(subnet, selected, dx, dy);
}

bool klogicIO::writeXML(bool subnet, bool selected, int dx, int dy)
{
	if (!net) return false;
	if (fname.isEmpty()) return false;

	QFile f(fname);
	if (!f.open(IO_WriteOnly|IO_Truncate))
		return false;

	QTextStream ts(&f);
	writeXML(ts, subnet, selected, dx, dy);

	f.flush();
	f.close();
	return true;
}

bool klogicIO::writeXML(QTextStream &ts, bool subnet, bool selected, int dx, int dy)
{
	// write global circuit information
	XmlObject header(XmlSymbolMap::m_oMap[XmlSymbol::KLOGIC_MAIN]);

	// type
	if (subnet) header.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::FILE_TYPE], XmlObject::ATT_SUBCIRCUIT);
	else header.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::FILE_TYPE], XmlObject::ATT_CIRCUIT);

	// meta tags
	header.setVersion();
	if (Global::CurrentCircuit::creation.isEmpty()) header.setCreation();
	else header.setCreation(Global::CurrentCircuit::creation);
	header.setUpdate();
	header.setAuthor(Global::CurrentCircuit::author);
	header.setComment(Global::CurrentCircuit::comment);

	ts << XmlObject::XML_ROOT;
	ts << header.getFieldStart();
	ts << header.getFieldContent();

	// export the circuit
	bool ret = net->exportXML(ts, selected, 0, dx, dy);

	ts << header.getFieldEnd();

	return ret;
}

bool klogicIO::readXML()
{
	if (!net) return false;
	if (fname.isEmpty()) return false;
	
	QFile f((const char *)fname);
	if (!f.open(IO_ReadOnly)) return false;

	QTextStream t(&f);

	bool ret = readXML(t, false, 0, 0);

	f.close();
	return ret;
}

bool klogicIO::readXML(QTextStream& t, bool selected, int dx = 0, int dy = 0)
{
	return XMLImportHandler::import(net, selected, dx, dy, t);
}

bool klogicIO::readNet(bool create_sub)
{
	if (!net) return false;
	if (fname.isEmpty()) return false;

	QFile f(fname);
	if (!f.open(IO_ReadOnly)) return false;

	QTextStream t(&f);

	bool ret = readNet(create_sub, &t, NO_SELECTION);

	f.close();
	return ret;
}

bool klogicIO::readNet(bool create_sub, QTextStream *t, int select, Circuit *_net, int dx, int dy)
{	QString s;
	int told=0, tnew=0;
	int sub = 0;
	int sub_is_main = 0;
	int must_change = 0;
	Circuit *__net;
	bool ret = true;
	bool retmsg = false;
	int event_cnt = 0;	// for better debugging

	if (!_net) _net = net;
	else {
		sub = 1;
		told = 1;
	}

	Circuit::magic_version = 0;
	while(!t->eof()) {
		s = t->readLine();
		if (s.isEmpty()) continue;

		// change status
		if (0 != (tnew = type(s))) {
			must_change = 0;
			// avoid impossible status
			if ((tnew != told + 1) &&
			    (!(told == 1 && tnew == 3)) &&
			    (!(told == 6 && tnew == 9)) &&
			    (!(told == 8 && tnew == 7)) &&
			    (!(told == 0 && tnew == 7)) &&
			    (tnew != 8) && (tnew != 2)) {
				QString errmsg = "corrupt file content\nimpossible status change (told=";
				errmsg += QString::number(told);
				errmsg += " tnew=";
				errmsg += QString::number(tnew);
				errmsg += ")\n";
				warning(errmsg);
				return false;
			}
		}
		else tnew = told;

		// special case: subcircuit is the main circuit
		if (told == 0 && tnew == 7 && sub == 0) {
			tnew = 1;
			sub_is_main = 1;
		}

		// process current status
		if (told == tnew) {
			switch (tnew) {
			case 1:
				event_cnt++;
				if (create_sub) {
					// create sub circuit
					ret = ret && _net->importInstance(s, dx, dy);
					__net = _net->subCircuit(s);
					if (!__net) {
						warning("corrupt content\ncannot find sub-circuit");
						return false;
					}
					// (NO_SELECTION) do never select the content of a sub circuit
					ret = ret && readNet(0, t, NO_SELECTION, __net);
					if (!ret && !retmsg) {
						warning("error during sub circuit creation");
						retmsg = true;
					}
					told = 8;
				} else {
					if (!Circuit::magic_version) {
						int posa = s.findRev(1);
						if (posa != -1) {       // of course not!
							Circuit::magic_version = s.right(s.length() - posa - 1).toInt();
						}
					}
				}
				break;
			case 3:
				event_cnt++;
				ret = ret && _net->importWire(s, select, dx, dy);
				if (!ret && !retmsg) {
					warning("error during wire import creation");
					retmsg = true;
				}
				break;
			case 5:
				event_cnt++;
				ret = ret && _net->importDevice(s, select, dx, dy);
				if (!ret && !retmsg) {
					warning("error during device import creation");
					retmsg = true;
				}
				break;
			case 7:
				event_cnt++;
				__net = _net->subCircuit(s);
				if (!__net) {
					warning("corrupt content\ncannot find sub-circuit");
					return false;
				}
				// (NO_SELECTION) do never select the content of a sub circuit
				ret = ret && readNet(0, t, NO_SELECTION, __net);
				if (!ret && !retmsg) {
					warning("error during sub circuit creation");
					retmsg = true;
				}
				told = 8;
				break;
			case 9:
				event_cnt++;
				ret = ret && _net->importWireConn(s, select, dx, dy);
				if (!ret && !retmsg) {
					warning("error during wire-wire connection");
					retmsg = true;
				}
				break;
			case 11:
				event_cnt++;
				ret = ret && _net->importDevConn(s, select, dx, dy);
				if (!ret && !retmsg) {
					warning("error during wire-dev connection");
					retmsg = true;
				}
				break;
			default:
				event_cnt++;
				warning("corrupt content\nunknown status while reading file");
				return false;
				break;
			}
		}
		// current status actually changed
		else {
			told = tnew;
			if (tnew == 2 || (tnew == 8 && sub_is_main)) {
				return ret;
			}
			if (tnew == 8) return ret;
		}
	}
	return ret;
}

int klogicIO::type(QString s)
{
	if (s.contains(IO_B_NET)) return 1;
	if (s.contains(IO_E_NET)) return 2;
	if (s.contains(IO_B_WIRE)) return 3;
	if (s.contains(IO_E_WIRE)) return 4;
	if (s.contains(IO_B_DEVICE)) return 5;
	if (s.contains(IO_E_DEVICE)) return 6;
	if (s.contains(IO_B_SUBNET)) return 7;
	if (s.contains(IO_E_SUBNET)) return 8;
	if (s.contains(IO_B_CONNWIRE)) return 9;
	if (s.contains(IO_E_CONNWIRE)) return 10;
	if (s.contains(IO_B_CONNDEVICE)) return 11;
	if (s.contains(IO_E_CONNDEVICE)) return 12;
	return 0;
}

