// Command interfaces
package main

import (
	"bytes"
	"errors"
	"flag"
	"fmt"
	"go/format"
	"os"
	"strings"
	"text/template"

	"github.com/rjeczalik/interfaces"
)

var (
	query  = flag.String("for", "", "Type to generate an interface for.")
	as     = flag.String("as", "main.Interface", `Generated interface name.`)
	output = flag.String("o", "-", "Output file.")
	all    = flag.Bool("all", false, "Include also unexported methods.")
)

var tmpl = template.Must(template.New("").Parse(`// Code generated by interfacer; DO NOT EDIT

package {{.PackageName}}

import (
{{range .Deps}}	"{{.}}"
{{end}})

// {{.InterfaceName}} is an interface generated for {{.Type}}.
type {{.InterfaceName}} interface {
{{range .Interface}}	{{.}}
{{end}}}
`))

type vars struct {
	PackageName   string
	InterfaceName string
	Type          string
	Deps          []string
	Interface     interfaces.Interface
}

func main() {
	if err := run(); err != nil {
		fmt.Fprintln(os.Stderr, err)
		os.Exit(1)
	}
}

func run() error {
	flag.Parse()
	if *query == "" {
		return errors.New("empty -for flag value; see -help for details")
	}
	if *output == "" {
		return errors.New("empty -o flag value; see -help for details")
	}
	q, err := interfaces.ParseQuery(*query)
	if err != nil {
		return err
	}
	opts := &interfaces.Options{
		Query:      q,
		Unexported: *all,
	}
	i, err := interfaces.NewWithOptions(opts)
	if err != nil {
		return err
	}
	v := &vars{
		Type:      fmt.Sprintf(`"%s"`, *query),
		Deps:      i.Deps(),
		Interface: i,
	}
	if i := strings.IndexRune(*as, '.'); i != -1 {
		v.PackageName = (*as)[:i]
		v.InterfaceName = (*as)[i+1:]
	} else {
		v.InterfaceName = *as
	}
	var buf bytes.Buffer
	if err := tmpl.Execute(&buf, v); err != nil {
		return err
	}
	formatted, err := format.Source(buf.Bytes())
	if err != nil {
		return err
	}
	f := os.Stdout
	if *output != "-" {
		f, err = os.OpenFile(*output, os.O_TRUNC|os.O_CREATE|os.O_WRONLY, 0644)
		if err != nil {
			return err
		}
		defer f.Close()
	}
	if _, err := f.Write(formatted); err != nil {
		return err
	}
	return nil
}
