;;;
;;; filesystem.cl -- various lisp utilities that make my life easier
;;;
;;; Author: Cyrus Harmon <ch-lisp@bobobeach.com>
;;; Time-stamp: <2005-07-01 08:18:31 sly>
;;;

(in-package :ch-util)

#+sbcl
(eval-when (:compile-toplevel :load-toplevel :execute)
  (require :sb-posix))

#+openmcl
(defun pwd ()
  (ccl::current-directory-name))

(defun component-present-p (value)
  (and value (not (eql value :unspecific))))

;;; I don't remember where I got this from. Probably from KMR somewhere
;;; along the line...
(defun pathname-as-directory (pathname)
  "Return a pathname reperesenting the given pathname in `directory
  form', i.e. with all the name elements in the directory component and
  NIL in the name and type components. Can not be used on wild
  pathnames. Returns its argument if name and type are both nil or
  :unspecific."
  (setf pathname (pathname pathname))
  (when (wild-pathname-p pathname)
    (error "Can't reliably convert wild pathnames to directory names."))
  (cond 
   ((or (component-present-p (pathname-name pathname))
	(component-present-p (pathname-type pathname)))
    (make-pathname 
     :directory (append (pathname-directory pathname) (list
						       (file-namestring pathname)))
     :name      nil
     :type      nil
     :defaults pathname))
   (t pathname)))

;;; I don't remember where I got this from. Probably from KMR somewhere
;;; along the line...
(defun list-directory (dirname)
  "Return a list of the contents of the directory named by dirname.
  Names of subdirectories will be returned in `directory normal form'.
  Unlike CL:DIRECTORY, LIST-DIRECTORY does not accept wildcard
  pathnames; `dirname' should simply be a pathname that names a
  directory. It can be in either file or directory form."
  (let ((wildcard (make-pathname :name :wild 
				 :type :wild
				 :defaults (pathname-as-directory
					    dirname))))
    
    (declare (ignorable wildcard))
    #+openmcl
    ;; OpenMCl by default doesn't return subdirectories at all. But
    ;; when prodded to do so with the special argument :directories,
    ;; it returns them in directory form.
    (directory wildcard :directories t)))

(defun ls (&optional (dirname ""))
  (list-directory dirname))

(defmacro with-open-file-preserving-case (&rest args)
  `(let ((*readtable* (copy-readtable)))
    (setf (readtable-case *readtable*) :preserve)
    (with-open-file ,@args)))

(defparameter *tmp-file-directory* (make-pathname :directory '(:absolute "tmp")))

(defun tmp-file-name (&key (prefix "tmp."))
  (concatenate 'string prefix (format nil "~8,'0',X" (random #xffffffff))))
  
(defun tmp-file (&key (name (tmp-file-name)))
  (merge-pathnames name *tmp-file-directory*))

(defun remove-keyword-args (list &rest remove)
  (loop for x on list by #'cddr when (not (member (car x) remove)) append (list (car x) (cadr x))))
  
(defmacro with-temporary-file ((path stream &rest options &key (delete t) &allow-other-keys) &body body)
  `(let ((,path (tmp-file)))
     (prog1
	 (with-open-file (,stream ,path ,@(remove-keyword-args options :delete))
	   ,@body)
       ,(when delete `(delete-file ,path)))))

;;; from antifuchs on #lisp via paste.lisp.org
;;; http://paste.lisp.org/display/9527
(defmacro with-current-directory (dir &body body)
  `(unwind-protect (progn
		     #+sbcl
                     (sb-posix:chdir ,dir)
                     (let ((*default-pathname-defaults* ,dir))
                       ,@body))
     #+sbcl (sb-posix:chdir *default-pathname-defaults*)))

(defmacro run-program (&rest args)
  #+sbcl `(sb-ext::run-program ,@args))

(defmacro process-output-stream (&rest args)
  #+sbcl `(sb-ext::process-output ,@args))
