;;;
;;; ch-util.cl -- various lisp utilities that make my life easier
;;;
;;; Author: Cyrus Harmon <ch-lisp@bobobeach.com>
;;; Time-stamp: <2005-07-01 08:12:37 sly>
;;;

(in-package :ch-util)

;;; Miscellaneous list utilities

(defun insert-before (new old list)
  (labels ((build-list (old c &optional newlist)
	     (if c
		 (if (eq old (car c))
		     (append (reverse (cdr c)) (cons (car c) (cons new newlist)))
		     (build-list old (cdr c) (cons (car c) newlist)))
		 (cons new newlist))))
    (reverse (build-list old list))))

(defun insert-before-all (new old list)
  (labels ((build-list (old c &optional newlist)
	     (if c
		 (if (eq old (car c))
		     (build-list old (cdr c) (cons (car c) (cons new newlist)))
		     (build-list old (cdr c) (cons (car c) newlist)))
		 newlist)))
    (reverse (build-list old list))))


(flet ((cca (l1 l2)
	 (dolist (x l1)
	   (let ((y (member x l2)))
	     (if y (return y))))))
  (defun closest-common-ancestor (itm &rest lis)
    (if (null lis)
	itm
	(cca itm (apply #'closest-common-ancestor lis)))))

;;; Miscellaneous class utilities
    
(defun subclassp (c1 c2)
  (subtypep (class-name c1) (class-name c2)))

;;; Miscellaneous string utilities
 
(defun strcat (&rest strs)
  (apply #'concatenate 'string strs))

(defun trim (seq suffix)
  (subseq seq 0 (search suffix seq)))

;;; This is a hack so that one day we might run under a case-sensitive lisp
;;; like Allegro mlisp (modern lisp). For now we encapsulate the uppercasing
;;; here so we can do the right thing later.
(defun interncase (x)
  (string-upcase x))

;;; simple wrapper for intern to allow us 
(defun make-intern (x &optional (package *package*))
  (intern (interncase x) package))

(defun make-keyword (x)
  (make-intern x 'keyword))

(defun keyword-list-names (k)
  (mapcar #'(lambda (x)
	    (symbol-name x))
	k))

(defun double-float-divide (&rest args)
  (apply #'/ (mapcar #'(lambda (x) (coerce x 'double-float)) args)))

(defun single-float-divide (&rest args)
  (apply #'/ (mapcar #'(lambda (x) (coerce x 'single-float)) args)))

(defmacro defun-export (func args body)
  `(progn
    (defun ,func ,args ,body)
    (export ',func)))

(defmacro defparameter-export (param value)
  `(progn
    (defparameter ,param ,value)
    (export ',param)))

(defmacro defclass-export (class &rest args)
  `(progn
    (defclass ,class ,@args)
    (export ',class)))

(defmacro defmethod-export (func args body)
  `(progn
    (defmethod ,func ,args ,body)
    (export ',func)))
  

;;
;; Reference implementation of with-unique-names from cliki
;;
(defmacro with-unique-names ((&rest bindings) &body body)
  `(let ,(mapcar #'(lambda (binding)
                     (destructuring-bind (var prefix)
			 (if (consp binding) binding (list binding binding))
                       `(,var (gensym ,(string prefix)))))
                 bindings)
    ,@body))

(defmacro mapv (function &rest vals)
  `(values-list (mapcar ,function (multiple-value-list ,@vals))))

;;
;; Silly little macro to do a postincrement, that is
;; return the value of the place prior to incrementing
;; it. Like incf, this only works on valid places.
;;
(defmacro postincf (x &optional (step 1))
  (let ((pre (gensym)))
    `(let ((,pre ,x))
       (incf ,x ,step)
       ,pre)))

;; another silly little function.
;; this one to sum a 2d array.
;; undoubtedly a better way to do this.
(defun array-sum (a)
  (destructuring-bind (height width) (array-dimensions a)
    (let ((acc 0))
      (dotimes (h height)
	(dotimes (w width)
	  (incf acc (aref a h w))))
      acc)))

(defun array-from-string (str)
  (let ((a (make-array (length str) :element-type '(unsigned-byte 8))))
    (dotimes (i (length str))
      (setf (aref a i) (char-code (elt str i))))
    a))

