/*!
 * \brief Configuration options for TLS channels
 *
 * \copyright Copyright (c) 2016-2021 Governikus GmbH & Co. KG, Germany
 */

#pragma once

#include <QByteArrayList>
#include <QJsonObject>
#include <QList>
#include <QSslCipher>
#include <QSslConfiguration>
#include <QSslEllipticCurve>
#include <QStringList>
#include <QVector>


class test_TlsConfiguration;


namespace governikus
{

class SecureStorage;


class SslCipherList
	: public QList<QSslCipher>
{
	public:
		SslCipherList& operator +=(const QString& pCipherName);
};


class SslEllipticCurveVector
	: public QVector<QSslEllipticCurve>
{
	public:
		SslEllipticCurveVector& operator +=(const QString& pEllipticCurveName);
};


class TlsConfiguration final
{
	friend class SecureStorage;
	friend class ::test_TlsConfiguration;
	friend bool operator==(const TlsConfiguration& pLeft, const TlsConfiguration& pRight);

	private:
		QSslConfiguration mConfiguration;

		TlsConfiguration();
		~TlsConfiguration();

		[[nodiscard]] QJsonArray readJsonArray(const QJsonObject& pConfig, const QLatin1String pName) const;
		[[nodiscard]] QSsl::SslProtocol readSslProtocol(const QJsonObject& pConfig, const QLatin1String pName) const;
		[[nodiscard]] bool readOcspStapling(const QJsonObject& pConfig, const QLatin1String pName) const;
		[[nodiscard]] QByteArrayList readSignatureAlgorithms(const QJsonObject& pConfig, const QLatin1String pKey) const;

	public:
		void load(const QJsonObject& pConfig);

		[[nodiscard]] QSsl::SslProtocol getProtocolVersion() const;
		[[nodiscard]] bool getOcspStapling() const;
		[[nodiscard]] QList<QSslCipher> getCiphers() const;
		[[nodiscard]] QVector<QSslEllipticCurve> getEllipticCurves() const;
		[[nodiscard]] QByteArrayList getSignatureAlgorithms() const;
		[[nodiscard]] const QSslConfiguration& getConfiguration() const;
};


inline bool operator==(const TlsConfiguration& pLeft, const TlsConfiguration& pRight)
{
	return &pLeft == &pRight || (
		pLeft.mConfiguration == pRight.mConfiguration);
}


inline bool operator!=(const TlsConfiguration& pLeft, const TlsConfiguration& pRight)
{
	return !(pLeft == pRight);
}


} // namespace governikus
