/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store.cache;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Locale;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.IntStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.io.Channels;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.store.BaseSearchableSnapshotIndexInput;
import org.elasticsearch.index.store.IndexInputStats;
import org.elasticsearch.index.store.SearchableSnapshotDirectory;
import org.elasticsearch.index.store.cache.CacheFile;
import org.elasticsearch.index.store.cache.CacheKey;

public class CachedBlobContainerIndexInput
extends BaseSearchableSnapshotIndexInput {
    public static final IOContext CACHE_WARMING_CONTEXT = new IOContext();
    private static final Logger logger = LogManager.getLogger(CachedBlobContainerIndexInput.class);
    private static final int COPY_BUFFER_SIZE = 8192;
    private final SearchableSnapshotDirectory directory;
    private final CacheFileReference cacheFileReference;
    private final int defaultRangeSize;
    private long lastReadPosition;
    private long lastSeekPosition;

    public CachedBlobContainerIndexInput(SearchableSnapshotDirectory directory, BlobStoreIndexShardSnapshot.FileInfo fileInfo, IOContext context, IndexInputStats stats, int rangeSize) {
        this("CachedBlobContainerIndexInput(" + fileInfo.physicalName() + ")", directory, fileInfo, context, stats, 0L, fileInfo.length(), new CacheFileReference(directory, fileInfo.physicalName(), fileInfo.length()), rangeSize);
        stats.incrementOpenCount();
    }

    private CachedBlobContainerIndexInput(String resourceDesc, SearchableSnapshotDirectory directory, BlobStoreIndexShardSnapshot.FileInfo fileInfo, IOContext context, IndexInputStats stats, long offset, long length, CacheFileReference cacheFileReference, int rangeSize) {
        super(resourceDesc, directory.blobContainer(), fileInfo, context, stats, offset, length);
        this.directory = directory;
        this.cacheFileReference = cacheFileReference;
        this.lastReadPosition = this.offset;
        this.lastSeekPosition = this.offset;
        this.defaultRangeSize = rangeSize;
    }

    @Override
    public void innerClose() {
        if (!this.isClone) {
            this.cacheFileReference.releaseOnClose();
        }
    }

    private void ensureContext(Predicate<IOContext> predicate) throws IOException {
        if (!predicate.test(this.context)) {
            assert (false) : "this method should not be used with this context " + this.context;
            throw new IOException("Cannot read the index input using context [context=" + this.context + ", input=" + (Object)((Object)this) + ']');
        }
    }

    private long getDefaultRangeSize() {
        return this.context != CACHE_WARMING_CONTEXT ? (long)this.defaultRangeSize : this.fileInfo.partSize().getBytes();
    }

    private Tuple<Long, Long> computeRange(long position) {
        long rangeSize = this.getDefaultRangeSize();
        long start = position / rangeSize * rangeSize;
        long end = Math.min(start + rangeSize, this.fileInfo.length());
        return Tuple.tuple((Object)start, (Object)end);
    }

    private CacheFile getCacheFileSafe() throws Exception {
        CacheFile cacheFile = this.cacheFileReference.get();
        if (cacheFile == null) {
            throw new AlreadyClosedException("Failed to acquire a non-evicted cache file");
        }
        return cacheFile;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected void readInternal(ByteBuffer b) throws IOException {
        this.ensureContext(ctx -> ctx != CACHE_WARMING_CONTEXT);
        assert (CachedBlobContainerIndexInput.assertCurrentThreadIsNotCacheFetchAsync());
        long position = this.getFilePointer() + this.offset;
        int length = b.remaining();
        int totalBytesRead = 0;
        while (totalBytesRead < length) {
            long pos = position + (long)totalBytesRead;
            int len = length - totalBytesRead;
            int bytesRead = 0;
            try {
                CacheFile cacheFile = this.getCacheFileSafe();
                Releasable ignored = cacheFile.fileLock();
                try {
                    Tuple<Long, Long> rangeToWrite = this.computeRange(pos);
                    Tuple rangeToRead = Tuple.tuple((Object)pos, (Object)Math.min(pos + (long)len, (Long)rangeToWrite.v2()));
                    bytesRead = cacheFile.fetchAsync(rangeToWrite, (Tuple<Long, Long>)rangeToRead, channel -> {
                        int read;
                        if ((Long)rangeToRead.v2() - (Long)rangeToRead.v1() < (long)b.remaining()) {
                            ByteBuffer duplicate = b.duplicate();
                            duplicate.limit(duplicate.position() + Math.toIntExact((Long)rangeToRead.v2() - (Long)rangeToRead.v1()));
                            read = this.readCacheFile(channel, pos, duplicate);
                            assert (duplicate.position() <= b.limit());
                            b.position(duplicate.position());
                        } else {
                            read = this.readCacheFile(channel, pos, b);
                        }
                        return read;
                    }, this::writeCacheFile, this.directory.cacheFetchAsyncExecutor()).get();
                }
                finally {
                    if (ignored == null) continue;
                    ignored.close();
                }
            }
            catch (Exception e) {
                if (!(e instanceof AlreadyClosedException) && (e.getCause() == null || !(e.getCause() instanceof AlreadyClosedException))) throw new IOException("Fail to read data from cache", e);
                try {
                    bytesRead = this.readDirectly(pos, pos + (long)len, b);
                }
                catch (Exception inner) {
                    e.addSuppressed(inner);
                    throw new IOException("Fail to read data from cache", e);
                }
            }
            finally {
                totalBytesRead += bytesRead;
            }
        }
        assert (totalBytesRead == length) : "partial read operation, read [" + totalBytesRead + "] bytes of [" + length + "]";
        this.stats.incrementBytesRead(this.lastReadPosition, position, totalBytesRead);
        this.lastSeekPosition = this.lastReadPosition = position + (long)totalBytesRead;
    }

    public void prefetchPart(int part) throws IOException {
        this.ensureContext(ctx -> ctx == CACHE_WARMING_CONTEXT);
        if ((long)part >= this.fileInfo.numberOfParts()) {
            throw new IllegalArgumentException("Unexpected part number [" + part + "]");
        }
        Tuple<Long, Long> partRange = this.computeRange(IntStream.range(0, part).mapToLong(arg_0 -> ((BlobStoreIndexShardSnapshot.FileInfo)this.fileInfo).partBytes(arg_0)).sum());
        assert (this.assertRangeIsAlignedWithPart(partRange));
        try {
            CacheFile cacheFile = this.getCacheFileSafe();
            try (Releasable ignored = cacheFile.fileLock();){
                Tuple<Long, Long> range = cacheFile.getAbsentRangeWithin((Long)partRange.v1(), (Long)partRange.v2());
                if (range == null) {
                    logger.trace("prefetchPart: part [{}] bytes [{}-{}] is already fully available for cache file [{}]", (Object)part, partRange.v1(), partRange.v2(), (Object)this.cacheFileReference);
                    return;
                }
                long rangeStart = (Long)range.v1();
                long rangeEnd = (Long)range.v2();
                long rangeLength = rangeEnd - rangeStart;
                logger.trace("prefetchPart: prewarming part [{}] bytes [{}-{}] by fetching bytes [{}-{}] for cache file [{}]", (Object)part, partRange.v1(), partRange.v2(), (Object)rangeStart, (Object)rangeEnd, (Object)this.cacheFileReference);
                FileChannel fc = cacheFile.getChannel();
                assert (CachedBlobContainerIndexInput.assertFileChannelOpen(fc));
                byte[] copyBuffer = new byte[Math.toIntExact(Math.min(8192L, rangeLength))];
                long totalBytesRead = 0L;
                AtomicLong totalBytesWritten = new AtomicLong();
                long startTimeNanos = this.stats.currentTimeNanos();
                try (InputStream input = this.openInputStream(rangeStart, rangeLength);){
                    int bytesRead;
                    for (long remainingBytes = rangeEnd - rangeStart; remainingBytes > 0L; remainingBytes -= (long)bytesRead) {
                        assert (totalBytesRead + remainingBytes == rangeLength);
                        bytesRead = CachedBlobContainerIndexInput.readSafe(input, copyBuffer, rangeStart, rangeEnd, remainingBytes, this.cacheFileReference);
                        long readStart = rangeStart + totalBytesRead;
                        Tuple rangeToWrite = Tuple.tuple((Object)readStart, (Object)(readStart + (long)bytesRead));
                        cacheFile.fetchAsync((Tuple<Long, Long>)rangeToWrite, (Tuple<Long, Long>)rangeToWrite, channel -> bytesRead, (channel, start, end, progressUpdater) -> {
                            ByteBuffer byteBuffer = ByteBuffer.wrap(copyBuffer, Math.toIntExact(start - readStart), Math.toIntExact(end - start));
                            int writtenBytes = CachedBlobContainerIndexInput.positionalWrite(channel, start, byteBuffer);
                            logger.trace("prefetchPart: writing range [{}-{}] of file [{}], [{}] bytes written", (Object)start, (Object)end, (Object)this.fileInfo.physicalName(), (Object)writtenBytes);
                            totalBytesWritten.addAndGet(writtenBytes);
                            progressUpdater.accept(start + (long)writtenBytes);
                        }, this.directory.cacheFetchAsyncExecutor()).get();
                        totalBytesRead += (long)bytesRead;
                    }
                    long endTimeNanos = this.stats.currentTimeNanos();
                    this.stats.addCachedBytesWritten(totalBytesWritten.get(), endTimeNanos - startTimeNanos);
                }
                assert (totalBytesRead == rangeLength);
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to prefetch file part in cache", e);
        }
    }

    @SuppressForbidden(reason="Use positional writes on purpose")
    private static int positionalWrite(FileChannel fc, long start, ByteBuffer byteBuffer) throws IOException {
        assert (CachedBlobContainerIndexInput.assertCurrentThreadMayWriteCacheFile());
        return fc.write(byteBuffer, start);
    }

    private static int readSafe(InputStream inputStream, byte[] copyBuffer, long rangeStart, long rangeEnd, long remaining, CacheFileReference cacheFileReference) throws IOException {
        int len = remaining < (long)copyBuffer.length ? Math.toIntExact(remaining) : copyBuffer.length;
        int bytesRead = inputStream.read(copyBuffer, 0, len);
        if (bytesRead == -1) {
            throw new EOFException(String.format(Locale.ROOT, "unexpected EOF reading [%d-%d] ([%d] bytes remaining) from %s", rangeStart, rangeEnd, remaining, cacheFileReference));
        }
        assert (bytesRead > 0) : bytesRead;
        return bytesRead;
    }

    private boolean assertRangeIsAlignedWithPart(Tuple<Long, Long> range) {
        if (this.fileInfo.numberOfParts() == 1L) {
            long length = this.fileInfo.length();
            assert ((Long)range.v1() == 0L) : "start of range [" + range.v1() + "] is not aligned with zero";
            assert ((Long)range.v2() == length) : "end of range [" + range.v2() + "] is not aligned with file length [" + length + ']';
        } else {
            long length = this.fileInfo.partSize().getBytes();
            assert ((Long)range.v1() % length == 0L) : "start of range [" + range.v1() + "] is not aligned with part start";
            assert ((Long)range.v2() % length == 0L || ((Long)range.v2()).longValue() == this.fileInfo.length()) : "end of range [" + range.v2() + "] is not aligned with part end or with file length";
        }
        return true;
    }

    private int readCacheFile(FileChannel fc, long position, ByteBuffer buffer) throws IOException {
        assert (CachedBlobContainerIndexInput.assertFileChannelOpen(fc));
        int bytesRead = Channels.readFromFileChannel((FileChannel)fc, (long)position, (ByteBuffer)buffer);
        if (bytesRead == -1) {
            throw new EOFException(String.format(Locale.ROOT, "unexpected EOF reading [%d-%d] from %s", position, position + (long)buffer.remaining(), this.cacheFileReference));
        }
        this.stats.addCachedBytesRead(bytesRead);
        return bytesRead;
    }

    private void writeCacheFile(FileChannel fc, long start, long end, Consumer<Long> progressUpdater) throws IOException {
        assert (CachedBlobContainerIndexInput.assertFileChannelOpen(fc));
        assert (CachedBlobContainerIndexInput.assertCurrentThreadMayWriteCacheFile());
        long length = end - start;
        byte[] copyBuffer = new byte[Math.toIntExact(Math.min(8192L, length))];
        logger.trace(() -> new ParameterizedMessage("writing range [{}-{}] to cache file [{}]", new Object[]{start, end, this.cacheFileReference}));
        long bytesCopied = 0L;
        long startTimeNanos = this.stats.currentTimeNanos();
        try (InputStream input = this.openInputStream(start, length);){
            int bytesRead;
            for (long remaining = end - start; remaining > 0L; remaining -= (long)bytesRead) {
                bytesRead = CachedBlobContainerIndexInput.readSafe(input, copyBuffer, start, end, remaining, this.cacheFileReference);
                CachedBlobContainerIndexInput.positionalWrite(fc, start + bytesCopied, ByteBuffer.wrap(copyBuffer, 0, bytesRead));
                progressUpdater.accept(start + (bytesCopied += (long)bytesRead));
            }
            long endTimeNanos = this.stats.currentTimeNanos();
            this.stats.addCachedBytesWritten(bytesCopied, endTimeNanos - startTimeNanos);
        }
    }

    protected void seekInternal(long pos) throws IOException {
        if (pos > this.length()) {
            throw new EOFException("Reading past end of file [position=" + pos + ", length=" + this.length() + "] for " + this.toString());
        }
        if (pos < 0L) {
            throw new IOException("Seeking to negative position [" + pos + "] for " + this.toString());
        }
        long position = pos + this.offset;
        this.stats.incrementSeeks(this.lastSeekPosition, position);
        this.lastSeekPosition = position;
    }

    @Override
    public CachedBlobContainerIndexInput clone() {
        return (CachedBlobContainerIndexInput)super.clone();
    }

    public IndexInput slice(String sliceDescription, long offset, long length) {
        if (offset < 0L || length < 0L || offset + length > this.length()) {
            throw new IllegalArgumentException("slice() " + sliceDescription + " out of bounds: offset=" + offset + ",length=" + length + ",fileLength=" + this.length() + ": " + (Object)((Object)this));
        }
        CachedBlobContainerIndexInput slice = new CachedBlobContainerIndexInput(this.getFullSliceDescription(sliceDescription), this.directory, this.fileInfo, this.context, this.stats, this.offset + offset, length, this.cacheFileReference, this.defaultRangeSize);
        slice.isClone = true;
        return slice;
    }

    public String toString() {
        return "CachedBlobContainerIndexInput{cacheFileReference=" + this.cacheFileReference + ", offset=" + this.offset + ", length=" + this.length() + ", position=" + this.getFilePointer() + ", rangeSize=" + this.getDefaultRangeSize() + '}';
    }

    private int readDirectly(long start, long end, ByteBuffer b) throws IOException {
        long length = end - start;
        byte[] copyBuffer = new byte[Math.toIntExact(Math.min(8192L, length))];
        logger.trace(() -> new ParameterizedMessage("direct reading of range [{}-{}] for cache file [{}]", new Object[]{start, end, this.cacheFileReference}));
        int bytesCopied = 0;
        long startTimeNanos = this.stats.currentTimeNanos();
        try (InputStream input = this.openInputStream(start, length);){
            int bytesRead;
            for (long remaining = end - start; remaining > 0L; remaining -= (long)bytesRead) {
                int len = remaining < (long)copyBuffer.length ? (int)remaining : copyBuffer.length;
                bytesRead = input.read(copyBuffer, 0, len);
                if (bytesRead == -1) {
                    throw new EOFException(String.format(Locale.ROOT, "unexpected EOF reading [%d-%d] ([%d] bytes remaining) from %s", start, end, remaining, this.cacheFileReference));
                }
                b.put(copyBuffer, 0, bytesRead);
                bytesCopied += bytesRead;
            }
            long endTimeNanos = this.stats.currentTimeNanos();
            this.stats.addDirectBytesRead(bytesCopied, endTimeNanos - startTimeNanos);
        }
        return bytesCopied;
    }

    private static boolean assertFileChannelOpen(FileChannel fileChannel) {
        assert (fileChannel != null);
        assert (fileChannel.isOpen());
        return true;
    }

    private static boolean isCacheFetchAsyncThread(String threadName) {
        return threadName.contains("[searchable_snapshots_cache_fetch_async]");
    }

    private static boolean assertCurrentThreadMayWriteCacheFile() {
        String threadName = Thread.currentThread().getName();
        assert (CachedBlobContainerIndexInput.isCacheFetchAsyncThread(threadName)) : "expected the current thread [" + threadName + "] to belong to the cache fetch async thread pool";
        return true;
    }

    private static boolean assertCurrentThreadIsNotCacheFetchAsync() {
        String threadName = Thread.currentThread().getName();
        assert (!CachedBlobContainerIndexInput.isCacheFetchAsyncThread(threadName)) : "expected the current thread [" + threadName + "] to belong to the cache fetch async thread pool";
        return true;
    }

    private static class CacheFileReference
    implements CacheFile.EvictionListener {
        private final long fileLength;
        private final CacheKey cacheKey;
        private final SearchableSnapshotDirectory directory;
        private final AtomicReference<CacheFile> cacheFile = new AtomicReference();

        private CacheFileReference(SearchableSnapshotDirectory directory, String fileName, long fileLength) {
            this.cacheKey = directory.createCacheKey(fileName);
            this.fileLength = fileLength;
            this.directory = directory;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        CacheFile get() throws Exception {
            CacheFile currentCacheFile = this.cacheFile.get();
            if (currentCacheFile != null) {
                return currentCacheFile;
            }
            CacheFile newCacheFile = this.directory.getCacheFile(this.cacheKey, this.fileLength);
            CacheFileReference cacheFileReference = this;
            synchronized (cacheFileReference) {
                currentCacheFile = this.cacheFile.get();
                if (currentCacheFile != null) {
                    return currentCacheFile;
                }
                if (newCacheFile.acquire(this)) {
                    CacheFile previousCacheFile = this.cacheFile.getAndSet(newCacheFile);
                    assert (previousCacheFile == null);
                    return newCacheFile;
                }
            }
            return null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onEviction(CacheFile evictedCacheFile) {
            CacheFileReference cacheFileReference = this;
            synchronized (cacheFileReference) {
                if (this.cacheFile.compareAndSet(evictedCacheFile, null)) {
                    evictedCacheFile.release(this);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void releaseOnClose() {
            CacheFileReference cacheFileReference = this;
            synchronized (cacheFileReference) {
                CacheFile currentCacheFile = this.cacheFile.getAndSet(null);
                if (currentCacheFile != null) {
                    currentCacheFile.release(this);
                }
            }
        }

        public String toString() {
            return "CacheFileReference{cacheKey='" + this.cacheKey + '\'' + ", fileLength=" + this.fileLength + ", acquired=" + (this.cacheFile.get() != null) + '}';
        }
    }
}

