/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store.cache;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.util.concurrent.AbstractRefCounted;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.store.cache.SparseFileTracker;

public class CacheFile {
    private static final StandardOpenOption[] OPEN_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.SPARSE};
    private final AbstractRefCounted refCounter = new AbstractRefCounted("CacheFile"){

        protected void closeInternal() {
            CacheFile.this.finishEviction();
        }
    };
    private final ReentrantReadWriteLock.WriteLock evictionLock;
    private final ReentrantReadWriteLock.ReadLock readLock;
    private final SparseFileTracker tracker;
    private final String description;
    private final Path file;
    private volatile Set<EvictionListener> listeners;
    private volatile boolean evicted;
    @Nullable
    private volatile FileChannel channel;

    public CacheFile(String description, long length, Path file) {
        this.tracker = new SparseFileTracker(file.toString(), length);
        this.description = Objects.requireNonNull(description);
        this.file = Objects.requireNonNull(file);
        this.listeners = new HashSet<EvictionListener>();
        this.evicted = false;
        ReentrantReadWriteLock cacheLock = new ReentrantReadWriteLock();
        this.evictionLock = cacheLock.writeLock();
        this.readLock = cacheLock.readLock();
        assert (this.invariant());
    }

    public long getLength() {
        return this.tracker.getLength();
    }

    public Path getFile() {
        return this.file;
    }

    Releasable fileLock() {
        boolean success = false;
        this.readLock.lock();
        try {
            this.ensureOpen();
            if (this.channel == null) {
                throw new AlreadyClosedException("Cache file channel has been released and closed");
            }
            success = true;
            Releasable releasable = this.readLock::unlock;
            return releasable;
        }
        finally {
            if (!success) {
                this.readLock.unlock();
            }
        }
    }

    @Nullable
    public FileChannel getChannel() {
        return this.channel;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean acquire(EvictionListener listener) throws IOException {
        assert (listener != null);
        this.ensureOpen();
        boolean success = false;
        if (this.refCounter.tryIncRef()) {
            this.evictionLock.lock();
            try {
                this.ensureOpen();
                HashSet<EvictionListener> newListeners = new HashSet<EvictionListener>(this.listeners);
                boolean added = newListeners.add(listener);
                assert (added) : "listener already exists " + listener;
                this.maybeOpenFileChannel(newListeners);
                this.listeners = Collections.unmodifiableSet(newListeners);
                success = true;
            }
            finally {
                try {
                    if (!success) {
                        this.refCounter.decRef();
                    }
                }
                finally {
                    this.evictionLock.unlock();
                }
            }
        }
        assert (this.invariant());
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean release(EvictionListener listener) {
        assert (listener != null);
        boolean success = false;
        this.evictionLock.lock();
        try {
            try {
                HashSet<EvictionListener> newListeners = new HashSet<EvictionListener>(this.listeners);
                boolean removed = newListeners.remove(Objects.requireNonNull(listener));
                assert (removed) : "listener does not exist " + listener;
                if (!removed) {
                    throw new IllegalStateException("Cannot remove an unknown listener");
                }
                this.maybeCloseFileChannel(newListeners);
                this.listeners = Collections.unmodifiableSet(newListeners);
                success = true;
            }
            finally {
                if (success) {
                    this.refCounter.decRef();
                }
            }
        }
        finally {
            this.evictionLock.unlock();
        }
        assert (this.invariant());
        return success;
    }

    private void finishEviction() {
        assert (this.evictionLock.isHeldByCurrentThread());
        assert (this.listeners.isEmpty());
        assert (this.channel == null);
        try {
            Files.deleteIfExists(this.file);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public void startEviction() {
        if (!this.evicted) {
            HashSet<EvictionListener> evictionListeners = new HashSet<EvictionListener>();
            this.evictionLock.lock();
            try {
                if (!this.evicted) {
                    this.evicted = true;
                    evictionListeners.addAll(this.listeners);
                    this.refCounter.decRef();
                }
            }
            finally {
                this.evictionLock.unlock();
            }
            evictionListeners.forEach(listener -> listener.onEviction(this));
        }
        assert (this.invariant());
    }

    private void maybeOpenFileChannel(Set<EvictionListener> listeners) throws IOException {
        assert (this.evictionLock.isHeldByCurrentThread());
        if (listeners.size() == 1) {
            assert (this.channel == null);
            this.channel = FileChannel.open(this.file, OPEN_OPTIONS);
        }
    }

    private void maybeCloseFileChannel(Set<EvictionListener> listeners) {
        assert (this.evictionLock.isHeldByCurrentThread());
        if (listeners.size() == 0) {
            assert (this.channel != null);
            try {
                this.channel.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException("Exception when closing channel", e);
            }
            finally {
                this.channel = null;
            }
        }
    }

    private boolean invariant() {
        this.readLock.lock();
        try {
            assert (this.listeners != null);
            if (this.listeners.isEmpty()) {
                assert (this.channel == null);
                assert (!this.evicted || this.refCounter.refCount() != 0 || Files.notExists(this.file, new LinkOption[0]));
            } else {
                assert (this.channel != null);
                assert (this.refCounter.refCount() > 0);
                assert (this.channel.isOpen());
                assert (Files.exists(this.file, new LinkOption[0]));
            }
        }
        finally {
            this.readLock.unlock();
        }
        return true;
    }

    public String toString() {
        return "CacheFile{desc='" + this.description + "', file=" + this.file + ", length=" + this.tracker.getLength() + ", channel=" + (this.channel != null ? "yes" : "no") + ", listeners=" + this.listeners.size() + ", evicted=" + this.evicted + ", tracker=" + this.tracker + '}';
    }

    private void ensureOpen() {
        if (this.evicted) {
            throw new AlreadyClosedException("Cache file is evicted");
        }
    }

    CompletableFuture<Integer> fetchAsync(Tuple<Long, Long> rangeToWrite, Tuple<Long, Long> rangeToRead, RangeAvailableHandler reader, final RangeMissingHandler writer, Executor executor) {
        CompletableFuture<Integer> future = new CompletableFuture<Integer>();
        try {
            this.ensureOpen();
            final List<SparseFileTracker.Gap> gaps = this.tracker.waitForRange(rangeToWrite, rangeToRead, (ActionListener<Void>)ActionListener.wrap(success -> {
                int read = reader.onRangeAvailable(this.channel);
                assert ((long)read == (Long)rangeToRead.v2() - (Long)rangeToRead.v1()) : "partial read [" + read + "] does not match the range to read [" + rangeToRead.v2() + '-' + rangeToRead.v1() + ']';
                future.complete(read);
            }, future::completeExceptionally));
            if (!gaps.isEmpty()) {
                executor.execute((Runnable)new AbstractRunnable(){

                    protected void doRun() {
                        for (SparseFileTracker.Gap gap : gaps) {
                            try {
                                CacheFile.this.ensureOpen();
                                if (!CacheFile.this.readLock.tryLock()) {
                                    throw new AlreadyClosedException("Cache file channel is being evicted, writing attempt cancelled");
                                }
                                try {
                                    CacheFile.this.ensureOpen();
                                    if (CacheFile.this.channel == null) {
                                        throw new AlreadyClosedException("Cache file channel has been released and closed");
                                    }
                                    writer.fillCacheRange(CacheFile.this.channel, gap.start(), gap.end(), gap::onProgress);
                                    gap.onCompletion();
                                }
                                finally {
                                    CacheFile.this.readLock.unlock();
                                }
                            }
                            catch (Exception e) {
                                gap.onFailure(e);
                            }
                        }
                    }

                    public void onFailure(Exception e) {
                        gaps.forEach(gap -> gap.onFailure(e));
                    }
                });
            }
        }
        catch (Exception e) {
            future.completeExceptionally(e);
        }
        return future;
    }

    public Tuple<Long, Long> getAbsentRangeWithin(long start, long end) {
        this.ensureOpen();
        return this.tracker.getAbsentRangeWithin(start, end);
    }

    @FunctionalInterface
    static interface RangeAvailableHandler {
        public int onRangeAvailable(FileChannel var1) throws IOException;
    }

    @FunctionalInterface
    static interface RangeMissingHandler {
        public void fillCacheRange(FileChannel var1, long var2, long var4, Consumer<Long> var6) throws IOException;
    }

    @FunctionalInterface
    public static interface EvictionListener {
        public void onEviction(CacheFile var1);
    }
}

