/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.dataframe;

import java.time.Clock;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexAction;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.admin.indices.get.GetIndexAction;
import org.elasticsearch.action.admin.indices.get.GetIndexRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshAction;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.ParentTaskAssigningClient;
import org.elasticsearch.client.node.NodeClient;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.ReindexAction;
import org.elasticsearch.index.reindex.ReindexRequest;
import org.elasticsearch.persistent.PersistentTaskState;
import org.elasticsearch.script.Script;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.tasks.TaskCancelledException;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.MlStatsIndex;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsConfig;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsState;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsTaskState;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.job.persistence.AnomalyDetectorsIndex;
import org.elasticsearch.xpack.core.ml.job.persistence.ElasticsearchMappings;
import org.elasticsearch.xpack.core.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.ml.dataframe.DataFrameAnalyticsTask;
import org.elasticsearch.xpack.ml.dataframe.DestinationIndex;
import org.elasticsearch.xpack.ml.dataframe.extractor.DataFrameDataExtractorFactory;
import org.elasticsearch.xpack.ml.dataframe.persistence.DataFrameAnalyticsConfigProvider;
import org.elasticsearch.xpack.ml.dataframe.process.AnalyticsProcessManager;
import org.elasticsearch.xpack.ml.notifications.DataFrameAnalyticsAuditor;

public class DataFrameAnalyticsManager {
    private static final Logger LOGGER = LogManager.getLogger(DataFrameAnalyticsManager.class);
    private final NodeClient client;
    private final DataFrameAnalyticsConfigProvider configProvider;
    private final AnalyticsProcessManager processManager;
    private final DataFrameAnalyticsAuditor auditor;
    private final IndexNameExpressionResolver expressionResolver;

    public DataFrameAnalyticsManager(NodeClient client, DataFrameAnalyticsConfigProvider configProvider, AnalyticsProcessManager processManager, DataFrameAnalyticsAuditor auditor, IndexNameExpressionResolver expressionResolver) {
        this.client = Objects.requireNonNull(client);
        this.configProvider = Objects.requireNonNull(configProvider);
        this.processManager = Objects.requireNonNull(processManager);
        this.auditor = Objects.requireNonNull(auditor);
        this.expressionResolver = Objects.requireNonNull(expressionResolver);
    }

    public void execute(DataFrameAnalyticsTask task, DataFrameAnalyticsState currentState, ClusterState clusterState) {
        ActionListener configListener = ActionListener.wrap(config -> {
            task.getStatsHolder().adjustProgressTracker(config.getAnalysis().getProgressPhases(), config.getAnalysis().supportsInference());
            switch (currentState) {
                case STARTED: {
                    this.executeStartingJob(task, (DataFrameAnalyticsConfig)config);
                    break;
                }
                case ANALYZING: {
                    LOGGER.debug("[{}] Reassigning job that was analyzing", (Object)config.getId());
                    this.startAnalytics(task, (DataFrameAnalyticsConfig)config);
                    break;
                }
                case REINDEXING: {
                    LOGGER.debug("[{}] Reassigning job that was reindexing", (Object)config.getId());
                    this.executeJobInMiddleOfReindexing(task, (DataFrameAnalyticsConfig)config);
                    break;
                }
                default: {
                    task.setFailed((Exception)((Object)ExceptionsHelper.serverError((String)("Cannot execute analytics task [" + config.getId() + "] as it is in unknown state [" + currentState + "]. Must be one of [STARTED, REINDEXING, ANALYZING]"))));
                }
            }
        }, task::setFailed);
        ActionListener statsIndexListener = ActionListener.wrap(aBoolean -> this.configProvider.get(task.getParams().getId(), (ActionListener<DataFrameAnalyticsConfig>)configListener), arg_0 -> ((ActionListener)configListener).onFailure(arg_0));
        ActionListener stateAliasListener = ActionListener.wrap(aBoolean -> this.createStatsIndexAndUpdateMappingsIfNecessary((Client)new ParentTaskAssigningClient((Client)this.client, task.getParentTaskId()), clusterState, (ActionListener<Boolean>)statsIndexListener), arg_0 -> ((ActionListener)configListener).onFailure(arg_0));
        AnomalyDetectorsIndex.createStateIndexAndAliasIfNecessary((Client)new ParentTaskAssigningClient((Client)this.client, task.getParentTaskId()), (ClusterState)clusterState, (IndexNameExpressionResolver)this.expressionResolver, (ActionListener)stateAliasListener);
    }

    private void createStatsIndexAndUpdateMappingsIfNecessary(Client client, ClusterState clusterState, ActionListener<Boolean> listener) {
        ActionListener createIndexListener = ActionListener.wrap(aBoolean -> ElasticsearchMappings.addDocMappingIfMissing((String)MlStatsIndex.writeAlias(), MlStatsIndex::mapping, (Client)client, (ClusterState)clusterState, (ActionListener)listener), arg_0 -> listener.onFailure(arg_0));
        MlStatsIndex.createStatsIndexAndAliasIfNecessary((Client)client, (ClusterState)clusterState, (IndexNameExpressionResolver)this.expressionResolver, (ActionListener)createIndexListener);
    }

    private void executeStartingJob(DataFrameAnalyticsTask task, DataFrameAnalyticsConfig config) {
        if (task.isStopping()) {
            LOGGER.debug("[{}] task is stopping. Marking as complete before starting job.", (Object)task.getParams().getId());
            task.markAsCompleted();
            return;
        }
        DataFrameAnalyticsTaskState reindexingState = new DataFrameAnalyticsTaskState(DataFrameAnalyticsState.REINDEXING, task.getAllocationId(), null);
        DataFrameAnalyticsTask.StartingState startingState = DataFrameAnalyticsTask.determineStartingState(config.getId(), task.getParams().getProgressOnStart());
        LOGGER.debug("[{}] Starting job from state [{}]", (Object)config.getId(), (Object)startingState);
        switch (startingState) {
            case FIRST_TIME: {
                task.updatePersistentTaskState((PersistentTaskState)reindexingState, ActionListener.wrap(updatedTask -> this.reindexDataframeAndStartAnalysis(task, config), task::setFailed));
                break;
            }
            case RESUMING_REINDEXING: {
                task.updatePersistentTaskState((PersistentTaskState)reindexingState, ActionListener.wrap(updatedTask -> this.executeJobInMiddleOfReindexing(task, config), task::setFailed));
                break;
            }
            case RESUMING_ANALYZING: {
                this.startAnalytics(task, config);
                break;
            }
            default: {
                task.setFailed((Exception)((Object)ExceptionsHelper.serverError((String)("Unexpected starting state [" + (Object)((Object)startingState) + "]"))));
            }
        }
    }

    private void executeJobInMiddleOfReindexing(DataFrameAnalyticsTask task, DataFrameAnalyticsConfig config) {
        if (task.isStopping()) {
            LOGGER.debug("[{}] task is stopping. Marking as complete before restarting reindexing.", (Object)task.getParams().getId());
            task.markAsCompleted();
            return;
        }
        ClientHelper.executeAsyncWithOrigin((Client)new ParentTaskAssigningClient((Client)this.client, task.getParentTaskId()), (String)"ml", (ActionType)DeleteIndexAction.INSTANCE, (ActionRequest)new DeleteIndexRequest(config.getDest().getIndex()), (ActionListener)ActionListener.wrap(r -> this.reindexDataframeAndStartAnalysis(task, config), e -> {
            Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
            if (cause instanceof IndexNotFoundException) {
                this.reindexDataframeAndStartAnalysis(task, config);
            } else {
                task.setFailed((Exception)e);
            }
        }));
    }

    private void reindexDataframeAndStartAnalysis(DataFrameAnalyticsTask task, DataFrameAnalyticsConfig config) {
        if (task.isStopping()) {
            LOGGER.debug("[{}] task is stopping. Marking as complete before starting reindexing and analysis.", (Object)task.getParams().getId());
            task.markAsCompleted();
            return;
        }
        ParentTaskAssigningClient parentTaskClient = new ParentTaskAssigningClient((Client)this.client, task.getParentTaskId());
        ActionListener reindexCompletedListener = ActionListener.wrap(reindexResponse -> {
            if (task.isStopping()) {
                LOGGER.debug("[{}] task is stopping. Marking as complete before marking reindex as finished.", (Object)task.getParams().getId());
                task.markAsCompleted();
                return;
            }
            task.setReindexingTaskId(null);
            Exception reindexError = DataFrameAnalyticsManager.getReindexError(task.getParams().getId(), reindexResponse);
            if (reindexError != null) {
                task.setFailed(reindexError);
                return;
            }
            LOGGER.debug("[{}] Reindex completed; created [{}]; retries [{}]", (Object)task.getParams().getId(), (Object)reindexResponse.getCreated(), (Object)reindexResponse.getBulkRetries());
            this.auditor.info(config.getId(), Messages.getMessage((String)"Finished reindexing to destination index [{0}], took [{1}]", (Object[])new Object[]{config.getDest().getIndex(), reindexResponse.getTook()}));
            this.startAnalytics(task, config);
        }, error -> {
            if (task.isStopping() && DataFrameAnalyticsManager.isTaskCancelledException(error)) {
                LOGGER.debug((Message)new ParameterizedMessage("[{}] Caught task cancelled exception while task is stopping", (Object)config.getId()), (Throwable)error);
                task.markAsCompleted();
            } else {
                task.setFailed((Exception)error);
            }
        });
        ActionListener copyIndexCreatedListener = ActionListener.wrap(createIndexResponse -> {
            ReindexRequest reindexRequest = new ReindexRequest();
            reindexRequest.setRefresh(true);
            reindexRequest.setSourceIndices(config.getSource().getIndex());
            reindexRequest.setSourceQuery(config.getSource().getParsedQuery());
            reindexRequest.getSearchRequest().source().fetchSource(config.getSource().getSourceFiltering());
            reindexRequest.setDestIndex(config.getDest().getIndex());
            reindexRequest.setScript(new Script("ctx._source.ml__id_copy = ctx._id"));
            reindexRequest.setParentTask(task.getParentTaskId());
            reindexRequest.getSearchRequest().allowPartialSearchResults(false);
            ThreadContext threadContext = parentTaskClient.threadPool().getThreadContext();
            Supplier supplier = threadContext.newRestorableContext(false);
            try (ThreadContext.StoredContext ignore = threadContext.stashWithOrigin("ml");){
                LOGGER.info("[{}] Started reindexing", (Object)config.getId());
                Task reindexTask = this.client.executeLocally((ActionType)ReindexAction.INSTANCE, (ActionRequest)reindexRequest, (ActionListener)new ContextPreservingActionListener(supplier, reindexCompletedListener));
                task.setReindexingTaskId(reindexTask.getId());
                this.auditor.info(config.getId(), Messages.getMessage((String)"Started reindexing to destination index [{0}]", (Object[])new Object[]{config.getDest().getIndex()}));
            }
        }, arg_0 -> ((ActionListener)reindexCompletedListener).onFailure(arg_0));
        ActionListener destIndexListener = ActionListener.wrap(indexResponse -> {
            this.auditor.info(config.getId(), Messages.getMessage((String)"Using existing destination index [{0}]", (Object[])new Object[]{indexResponse.indices()[0]}));
            LOGGER.info("[{}] Using existing destination index [{}]", (Object)config.getId(), (Object)indexResponse.indices()[0]);
            DestinationIndex.updateMappingsToDestIndex((Client)parentTaskClient, config, indexResponse, (ActionListener<AcknowledgedResponse>)ActionListener.wrap(acknowledgedResponse -> copyIndexCreatedListener.onResponse(null), arg_0 -> ((ActionListener)copyIndexCreatedListener).onFailure(arg_0)));
        }, e -> {
            if (ExceptionsHelper.unwrapCause((Throwable)e) instanceof IndexNotFoundException) {
                this.auditor.info(config.getId(), Messages.getMessage((String)"Creating destination index [{0}]", (Object[])new Object[]{config.getDest().getIndex()}));
                LOGGER.info("[{}] Creating destination index [{}]", (Object)config.getId(), (Object)config.getDest().getIndex());
                DestinationIndex.createDestinationIndex((Client)parentTaskClient, Clock.systemUTC(), config, (ActionListener<CreateIndexResponse>)copyIndexCreatedListener);
            } else {
                copyIndexCreatedListener.onFailure(e);
            }
        });
        ClientHelper.executeWithHeadersAsync((Map)config.getHeaders(), (String)"ml", (Client)parentTaskClient, (ActionType)GetIndexAction.INSTANCE, (ActionRequest)((GetIndexRequest)new GetIndexRequest().indices(new String[]{config.getDest().getIndex()})), (ActionListener)destIndexListener);
    }

    private static Exception getReindexError(String jobId, BulkByScrollResponse reindexResponse) {
        if (!reindexResponse.getBulkFailures().isEmpty()) {
            LOGGER.error("[{}] reindexing encountered {} failures", (Object)jobId, (Object)reindexResponse.getBulkFailures().size());
            for (BulkItemResponse.Failure failure : reindexResponse.getBulkFailures()) {
                LOGGER.error("[{}] reindexing failure: {}", (Object)jobId, (Object)failure);
            }
            return ExceptionsHelper.serverError((String)("reindexing encountered " + reindexResponse.getBulkFailures().size() + " failures"));
        }
        if (reindexResponse.getReasonCancelled() != null) {
            LOGGER.error("[{}] reindex task got cancelled with reason [{}]", (Object)jobId, (Object)reindexResponse.getReasonCancelled());
            return ExceptionsHelper.serverError((String)("reindex task got cancelled with reason [" + reindexResponse.getReasonCancelled() + "]"));
        }
        if (reindexResponse.isTimedOut()) {
            LOGGER.error("[{}] reindex task timed out after [{}]", (Object)jobId, (Object)reindexResponse.getTook().getStringRep());
            return ExceptionsHelper.serverError((String)("reindex task timed out after [" + reindexResponse.getTook().getStringRep() + "]"));
        }
        return null;
    }

    private static boolean isTaskCancelledException(Exception error) {
        return ExceptionsHelper.unwrapCause((Throwable)error) instanceof TaskCancelledException || ExceptionsHelper.unwrapCause((Throwable)error.getCause()) instanceof TaskCancelledException;
    }

    private void startAnalytics(DataFrameAnalyticsTask task, DataFrameAnalyticsConfig config) {
        if (task.isStopping()) {
            LOGGER.debug("[{}] task is stopping. Marking as complete before starting analysis.", (Object)task.getParams().getId());
            task.markAsCompleted();
            return;
        }
        ParentTaskAssigningClient parentTaskClient = new ParentTaskAssigningClient((Client)this.client, task.getParentTaskId());
        ActionListener dataExtractorFactoryListener = ActionListener.wrap(dataExtractorFactory -> {
            DataFrameAnalyticsTaskState analyzingState = new DataFrameAnalyticsTaskState(DataFrameAnalyticsState.ANALYZING, task.getAllocationId(), null);
            task.updatePersistentTaskState((PersistentTaskState)analyzingState, ActionListener.wrap(updatedTask -> {
                if (task.isStopping()) {
                    LOGGER.debug("[{}] task is stopping. Marking as complete before starting native process.", (Object)task.getParams().getId());
                    task.markAsCompleted();
                    return;
                }
                this.processManager.runJob(task, config, (DataFrameDataExtractorFactory)dataExtractorFactory);
            }, error -> {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)error);
                if (!(cause instanceof ResourceNotFoundException)) {
                    task.setFailed((Exception)error);
                }
            }));
        }, task::setFailed);
        ActionListener refreshListener = ActionListener.wrap(refreshResponse -> {
            task.setReindexingFinished();
            DataFrameDataExtractorFactory.createForDestinationIndex((Client)parentTaskClient, config, (ActionListener<DataFrameDataExtractorFactory>)dataExtractorFactoryListener);
        }, arg_0 -> ((ActionListener)dataExtractorFactoryListener).onFailure(arg_0));
        ClientHelper.executeWithHeadersAsync((Map)config.getHeaders(), (String)"ml", (Client)parentTaskClient, (ActionType)RefreshAction.INSTANCE, (ActionRequest)new RefreshRequest(new String[]{config.getDest().getIndex()}), (ActionListener)refreshListener);
    }

    public void stop(DataFrameAnalyticsTask task) {
        this.processManager.stop(task);
    }
}

