/*************************************************************************
* watchpid                                                               *
* ---------------------------------------------------------------------- *
* Copyright (C) 2000 Klaus Reimer, k@ailis.de                            *
*                                                                        *
* This library is free software; you can redistribute it and/or          *
* modify it under the terms of the GNU Library General Public            *
* License as published by the Free Software Foundation; either           *
* version 2 of the License, or (at your option) any later version.       *
*                                                                        *
* This library is distributed in the hope that it will be useful,        *
* but WITHOUT ANY WARRANTY; without even the implied warranty of         *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
* Library General Public License for more details.                       *
*                                                                        *
* You should have received a copy of the GNU Library General Public      *
* License along with this progam; if not, write to the                   *
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,           *
* Boston, MA 02111-1307, USA.                                            *
*************************************************************************/


#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <stdlib.h>
#include "config.h"

int iPID;
int iVerbose=0;
int iDelay=DELAY;
int iSignal=SIGNAL;
char sPath[256];

void displayUsage(void) {
  printf("Usage: %s [OPTIONS] PID\n",PACKAGE);
  printf("Waits for PID to terminate and terminates PID if watchpid is terminated itself.\n");
  printf("\nOptions\n");
  printf("  -v, --verbose    Print status messages during watching\n");
  printf("  -d, --delay=n    Specify watch interval in seconds. Default: %i\n",DELAY);
  printf("  -s, --signal=n   Specify signal to terminate PID. Default: %i\n",SIGNAL);	
  printf("  -h, --help       Display help and exit\n");
  printf("  -V, --version    Display version and exit\n");
  printf("\nThis utility was written to get background-forking daemons like apache running\n");
  printf("under Bernstein's daemontools.\n");
  printf("\nReport bugs to %s <%s>\n",AUTHOR,EMAIL);
}

void displayVersion(void) {
  printf("%s %s\n",PACKAGE,VERSION);
  printf("Written by %s\n",AUTHOR);
  printf("\n%s\n",COPYRIGHT);
  printf("This is free software; you can redistribute it and/or modify it under\n");
  printf("the terms of the GNU General Public License as published by the Free\n");
  printf("Software Foundation; either version 2 of the License, or (at your\n");
  printf("option) any later version.\n");
}

void checkOptions(int argc, char *argv[]) {
  char cOpt;
  int iOptIndex;
  static struct option SOpts[]={
    {"verbose",0,NULL,'v'},
    {"delay",0,NULL,'d'},
    {"signal",0,NULL,'s'},
    {"help",0,NULL,'h'},
    {"version",0,NULL,'V'}
  };
  
  opterr=0;
  while((cOpt=getopt_long(argc,argv,"vd:s:hV",SOpts,&iOptIndex))!=-1) {
    switch(cOpt) {
      case 'd':
        if (!optarg) {
          displayUsage();
          exit(1);
        }
        iDelay=atoi(optarg);
        break;
      case 's':
        if (!optarg) {
          displayUsage();
          exit(1);
        }
        iSignal=atoi(optarg);
        break;
      case 'v':
        iVerbose=1;
        break;
      case 'h':
        displayUsage();
        exit(0);
        break;
      case 'V':
        displayVersion();
        exit(0);
        break;
      default:
        displayUsage();
        exit(1);
    }
  }
}

static void shutdown(int iSig) {
  if (iVerbose) {
    fprintf(stderr,"Killing %i (%s) with Signal %i\n",iPID,sPath,iSignal);
    fflush(stderr);
  }
  kill(iPID,iSignal);
  exit(2);
}

int main(int argc, char *argv[]) {
  char sLink[256];
  
  checkOptions(argc,argv);
  argc-=optind;
  argv+=optind;
  if (!argc) {
    displayUsage();
    exit(1);
  }
  iPID=atoi(argv[0]);
  snprintf(sLink,255,"/proc/%i/exe",iPID);
  signal(SIGTERM,&shutdown);
  signal(SIGINT,&shutdown);
  while (1) {
    if (readlink(sLink,sPath,255)<0) {
      if (iVerbose) {
        fprintf(stderr,"PID %i was terminated. Terminating me, too\n",iPID);
        fflush(stderr);
      }
      exit(1);
    }
    if (iVerbose) {
      fprintf(stderr,"PID %i (%s) still running\n",iPID,sPath);
      fflush(stderr);
    }
    sleep(iDelay);
  }
  return 0;
}
