#lang typed/racket/base

#|
Compute f(x) = log(abs(gamma(x))) near the roots of `f', accurately. This is done more or less by
brute force: a huge, nested `cond' determines whether `x' is near a root (or a tricky spot, as around
-2.625), and applies one of these many Taylor series expansions.

The Taylor series coefficients were computed by Maxima using
  
  fpprec:45;
  terms(x0) := cons(log(abs(gamma(x0))),map(lambda([n],psi[n](x0)/(n+1)!),[0,1,2,3,4,5,6,7,8,9,...]));
  terms(bfloat(x0))

where x0 = -16.000000000000046, etc.

The last few digits in each are always wrong, but are fine after rounding to 53 bits, and are
probably fine for larger flonum precisions as well (though we'd need more terms to be more correct).
|#

(require "../../flonum.rkt")

(provide fllog-gamma-special-negative)

(: fllog-gamma-taylor-neg-16-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-16-eps x)
  ((make-flpolyfun
    (3.42552033350358311733334120197391285997329364e-2
     2.16519212854378804364052503881457935471686475e13 
     2.34402847675338276903976186040675803105186722e26 
     3.38351467129883525133298714499292391230168593e39 
     5.49446949983078390448145147509511783209827688e52 
     9.51726568924479766428829792505066627072141892e65 
     1.71722572963416977329253566951727611382059435e79 
     3.18696311377395951980006026131098541415051276e92 
     6.03783901691414892971142619565256231858278562e105
     1.16205169002981209467014755360005441193795642e119 
     2.26445865499190633858041606972575513957782768e132 
     4.45726186836426745991677397375017774085015343e145 
     8.84659261955282023171874967614556357794891805e158 
     1.76811440347263289710978356338608287606666092e172 
     3.55485686099453407775024429998387503068981764e185 
     7.18381822062263098616415005664454304133158426e198 
     1.45821999976683265621312998703694106904655702e212 
     2.97160137899281100204818274837937667034221446e225 
     6.07663858635514120494896403934157760213721947e238 
     1.24646116122756945135868661223357906051452277e252 
     2.56388650008387675084781633626915713290845764e265 
     5.28695892234340315776889816230237875306469639e278 
     1.09269508497457661879972605257434704162622542e292 
     2.26302980571621375226727382723173851714553023e305))
   (fl+ x 16.000000000000046)))

(: fllog-gamma-taylor-neg-16+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-16+eps x)
  ((make-flpolyfun
    (+3.42552033352947932572398727165416329469955525e-2
     -2.16519212854322734097485954674089084927480270e13
     +2.34402847675338276903976186040675803105186553e26
     -3.38351467129883525133298714499292391230168716e39
     +5.49446949983078390448145147509511783209827688e52
     -9.51726568924479766428829792505066627072141892e65
     +1.71722572963416977329253566951727611382059435e79
     -3.18696311377395951980006026131098541415051276e92
     +6.03783901691414892971142619565256231858278562e105
     -1.16205169002981209467014755360005441193795642e119
     +2.26445865499190633858041606972575513957782768e132
     -4.45726186836426745991677397375017774085015343e145
     +8.84659261955282023171874967614556357794891805e158
     -1.76811440347263289710978356338608287606666092e172
     +3.55485686099453407775024429998387503068981764e185
     -7.18381822062263098616415005664454304133158426e198
     +1.45821999976683265621312998703694106904655702e212
     -2.97160137899281100204818274837937667034221446e225
     +6.07663858635514120494896403934157760213721947e238
     -1.24646116122756945135868661223357906051452277e252
     +2.56388650008387675084781633626915713290845764e265
     -5.28695892234340315776889816230237875306469639e278
     +1.09269508497457661879972605257434704162622542e292
     -2.26302980571621375226727382723173851714553023e305))
   (fl+ x 15.999999999999954)))

(: fllog-gamma-taylor-neg-15-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-15-eps x)
  ((make-flpolyfun
    (1.15525490672681022185711739708783012180346524e-3
     1.30918593819183868774692964788238733025798397e12
     8.56983910376033938275220112806158380788981778e23
     7.47967523145739725777963978774141027614667188e35
     7.34421422643398169572992335407374271401272603e47
     7.69195359383655244163976623081075588379362466e59
     8.39183123521159038610388997065510730021596055e71
     9.41697209896719459325296135717142749205948794e83
     1.07874965207510576326789906427969304722014057e96
     1.25536344473165336499588602372903828497531979e108
     1.47915375224337636920269163322857941330755097e120
     1.76044299350606257350766868342715948489863137e132
     2.11268494440818660541928314338955356827359956e144
     2.55313608096278164214923128293506360284438946e156
     3.10377772294543137103271301928750979038863189e168
     3.79252734013508678551325226736139002262904078e180
     4.65480324740864694393008748960697082816914839e192
     5.73553219439468045888102489477691980653676664e204
     7.09171820265835310726774563051367548937070073e216
     8.79572628807155765623735540489237622206442855e228
     1.09394791138782982168391460633220354706621609e241
     1.36398211685740940272776847469329107382146839e253
     1.70453774336152266979708967262322846846670140e265
     2.13453263408141799832562100864983139919557550e277
     2.67806260459741351643411574808075407119390293e289
     3.36583862738784320386893917339053511137912179e301))
   (fl+ x 15.000000000000764)))

(: fllog-gamma-taylor-neg-15+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-15+eps x)
  ((make-flpolyfun
    (+1.15525491091416550634620392019108538425965918e-3
     -1.30918593818635666109027472714561810093410983e12
     +8.56983910376033938275220112806158380785805751e23
     -7.47967523145739725777963978774141027616053193e35
     +7.34421422643398169572992335407374271401272603e47
     -7.69195359383655244163976623081075588379362466e59
     +8.39183123521159038610388997065510730021596055e71
     -9.41697209896719459325296135717142749205948794e83
     +1.07874965207510576326789906427969304722014057e96
     -1.25536344473165336499588602372903828497531979e108
     +1.47915375224337636920269163322857941330755097e120
     -1.76044299350606257350766868342715948489863137e132
     +2.11268494440818660541928314338955356827359956e144
     -2.55313608096278164214923128293506360284438946e156
     +3.10377772294543137103271301928750979038863189e168
     -3.79252734013508678551325226736139002262904078e180
     +4.65480324740864694393008748960697082816914839e192
     -5.73553219439468045888102489477691980653676664e204
     +7.09171820265835310726774563051367548937070073e216
     -8.79572628807155765623735540489237622206442855e228
     +1.09394791138782982168391460633220354706621609e241
     -1.36398211685740940272776847469329107382146839e253
     +1.70453774336152266979708967262322846846670140e265
     -2.13453263408141799832562100864983139919557550e277
     +2.67806260459741351643411574808075407119390293e289
     -3.36583862738784320386893917339053511137912179e301))
   (fl+ x 14.999999999999236)))

(: fllog-gamma-taylor-neg-14-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-14-eps x)
  ((make-flpolyfun
    (7.05720414970836382974343795932367836228204258e-5
     8.71844437724299607238818643993229044314817323e10
     3.80056361772083925440439547664641650199223408e21
     2.20900016681707654152682369867627068687209827e32
     1.44442838123433135774968727885183741983569213e43
     1.00745347986530949975061019345278118790493350e54
     7.31952260549680798260379944355600071336462531e64
     5.46984434588907051495462936676216116146823788e75
     4.17274669703046177687747276824264410147395985e86
     3.23376533131276231243373110151708319509140853e97
     2.53740628523180393404941068473096530601602486e108
     2.01111232359833350081224410957694968261608159e119
     1.60726233517136341629818195230703984100819403e130
     1.29349174785225078248774509860609135667443638e141
     1.04717190088891217889982283945450481764946424e152
     8.52106263962962986047317245298885003374543513e162
     6.96472599903145134668055154185376490901542828e173
     5.71497187985235353695785444163409490053443581e184
     4.70575719810015273300645000279164929924814153e195
     3.88675727840733627688644182717269723570884573e206
     3.21921532816693137167156670390289667070550200e217
     2.67300474058385899419220218361791442066160170e228
     2.22451502797099844556885973986663205320032769e239
     1.85510796441934987891353225082737600968833191e250
     1.54997532843630249255321082937107619733383802e261
     1.29728387391900772334015429402303245587856606e272
     1.08752858614521596590096294535581746089179482e283))
   (fl+ x 14.00000000001147)))

(: fllog-gamma-taylor-neg-14+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-14+eps x)
  ((make-flpolyfun
    (+7.05721028462543395166741252987437638446269345e-5
     -8.71844437670812674005602769958828165617820307e10
     +3.80056361772083925440439547664641644750312443e21
     -2.20900016681707654152682369867627070270744066e32
     +1.44442838123433135774968727885183741983569213e43
     -1.00745347986530949975061019345278118790493350e54
     +7.31952260549680798260379944355600071336462531e64
     -5.46984434588907051495462936676216116146823788e75
     +4.17274669703046177687747276824264410147395985e86
     -3.23376533131276231243373110151708319509140853e97
     +2.53740628523180393404941068473096530601602486e108
     -2.01111232359833350081224410957694968261608159e119
     +1.60726233517136341629818195230703984100819403e130
     -1.29349174785225078248774509860609135667443638e141
     +1.04717190088891217889982283945450481764946424e152
     -8.52106263962962986047317245298885003374543513e162
     +6.96472599903145134668055154185376490901542828e173
     -5.71497187985235353695785444163409490053443581e184
     +4.70575719810015273300645000279164929924814153e195
     -3.88675727840733627688644182717269723570884573e206
     +3.21921532816693137167156670390289667070550200e217
     -2.67300474058385899419220218361791442066160170e228
     +2.22451502797099844556885973986663205320032769e239
     -1.85510796441934987891353225082737600968833191e250
     +1.54997532843630249255321082937107619733383802e261
     -1.29728387391900772334015429402303245587856606e272
     +1.08752858614521596590096294535581746089179482e283))
   (fl+ x 13.99999999998853)))

(: fllog-gamma-taylor-neg-13-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-13-eps x)
  ((make-flpolyfun
    (4.20070692126859694083520385628430681145952866e-6
     6.22704696315015051305989305476581988406584914e9
     1.93880569244302627999559554043966627136075792e19
     8.04868939614601273767623947637595966084723446e28
     3.75896751304948261029487848517086715088665702e38
     1.87258137815444559355579202413716800486894348e48
     9.71721014934499007442790146433391077060891507e57
     5.18653062218378047648854673573809085536221324e67
     2.82596735283228244362324473616029571252961981e77
     1.56421612578310868828508606287502751413734642e87
     8.76640254452700692209329834562293888490992251e96
     4.96261821089367480005194792833420777195398124e106
     2.83272519259599852918567471322345942565749047e116
     1.62826272005727565118236773772165623741858943e126
     9.41503496319752555773534041626705004304107636e135
     5.47193405276234040824249545461113755887770115e145
     3.19443659170955917253473379654127143885350501e155
     1.87217945120619961652450055643490432162160360e165
     1.10104743967196314558478384285610834792427926e175
     6.49541758038405192149141424551847761237213585e184
     3.84249067863447907906831685703870486213896574e194
     2.27879713345922908440402121716008913743249147e204
     1.35451687289014457756507563651717649104370105e214
     8.06791669041266820186883279414411645394575841e223
     4.81459921006082217623617815487345538030488875e233
     2.87815059621718549831528798396536979929342541e243
     1.72330566559340670986129762640372053529825002e253))
   (fl+ x 13.00000000016059)))

(: fllog-gamma-taylor-neg-13+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-13+eps x)
  ((make-flpolyfun
    (+4.20154292527072621914889857511622485654985997e-6
     -6.22704695794431433259544850846871775302229610e9
     +1.93880569244302627999559554043957827675071196e19
     -8.04868939614601273767623947637614230970357822e28
     +3.75896751304948261029487848517086715088665702e38
     -1.87258137815444559355579202413716800486894348e48
     +9.71721014934499007442790146433391077060891507e57
     -5.18653062218378047648854673573809085536221324e67
     +2.82596735283228244362324473616029571252961981e77
     -1.56421612578310868828508606287502751413734642e87
     +8.76640254452700692209329834562293888490992251e96
     -4.96261821089367480005194792833420777195398124e106
     +2.83272519259599852918567471322345942565749047e116
     -1.62826272005727565118236773772165623741858943e126
     +9.41503496319752555773534041626705004304107636e135
     -5.47193405276234040824249545461113755887770115e145
     +3.19443659170955917253473379654127143885350501e155
     -1.87217945120619961652450055643490432162160360e165
     +1.10104743967196314558478384285610834792427926e175
     -6.49541758038405192149141424551847761237213585e184
     +3.84249067863447907906831685703870486213896574e194
     -2.27879713345922908440402121716008913743249147e204
     +1.35451687289014457756507563651717649104370105e214
     -8.06791669041266820186883279414411645394575841e223
     +4.81459921006082217623617815487345538030488875e233
     -2.87815059621718549831528798396536979929342541e243
     +1.72330566559340670986129762640372053529825002e253))
   (fl+ x 12.99999999983941)))

(: fllog-gamma-taylor-neg-12-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-12-eps x)
  ((make-flpolyfun
    (7.92331889477662838176704627169679553918772480e-7
     4.79001984580383156934295092578649219589398137e8
     1.14721449406016195410092484708147166014958633e17
     3.66345344330893113284755742851916848671782064e25
     1.31610109538171333064294605008518749624155058e34
     5.04332026617433628923319069504402651198760215e42
     2.01313366969380958404839816353804766505096440e51
     8.26538586791793206180756921553598759148456961e59
     3.46424418652989138003252353575322225544849778e68
     1.47500429481618622832411424312125875871473290e77
     6.35876982680119455930209240151498201553943053e85
     2.76896668224128240282212673231364014487122016e94
     1.21581215161645932855804092279695053285167943e103
     5.37578243473933196559200253766562886942293722e111
     2.39108112409521134748162268919338033426509823e120
     1.06897709118147796032098934601791965876435654e129
     4.80039511356245954450366286956611957635723875e137
     2.16414002262646382463492539807218520945450358e146
     9.79036951376787668833331620803049384838472992e154
     4.44278501255337092574540257286295318903639058e163
     2.02169768551186469267664691423449813562994708e172
     9.22283046166621209468650365753657254785284645e180
     4.21694706804926224603612832242519303537955950e189
     1.93210313408299233443396459331600576138416068e198
     8.86919512810946571537814217489880825607931245e206
     4.07842756376789482418778552477199675182815872e215
     1.87843739106737609633367311897011914701215515e224))
   (fl+ x 12.000000002087674)))

(: fllog-gamma-taylor-neg-12+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-12+eps x)
  ((make-flpolyfun
    (+8.02878797874473321229715558287816009952298183e-7
     -4.79001979528393130316004392463101229275295420e8
     +1.14721449406016195410092484694807350730298753e17
     -3.66345344330893113284755742873216175210420413e25
     +1.31610109538171333064294605008518749624155058e34
     -5.04332026617433628923319069504402651198760215e42
     +2.01313366969380958404839816353804766505096440e51
     -8.26538586791793206180756921553598759148456961e59
     +3.46424418652989138003252353575322225544849778e68
     -1.47500429481618622832411424312125875871473290e77
     +6.35876982680119455930209240151498201553943053e85
     -2.76896668224128240282212673231364014487122016e94
     +1.21581215161645932855804092279695053285167943e103
     -5.37578243473933196559200253766562886942293722e111
     +2.39108112409521134748162268919338033426509823e120
     -1.06897709118147796032098934601791965876435654e129
     +4.80039511356245954450366286956611957635723875e137
     -2.16414002262646382463492539807218520945450358e146
     +9.79036951376787668833331620803049384838472992e154
     -4.44278501255337092574540257286295318903639058e163
     +2.02169768551186469267664691423449813562994708e172
     -9.22283046166621209468650365753657254785284645e180
     +4.21694706804926224603612832242519303537955950e189
     -1.93210313408299233443396459331600576138416068e198
     +8.86919512810946571537814217489880825607931245e206
     -4.07842756376789482418778552477199675182815872e215
     +1.87843739106737609633367311897011914701215515e224))
   (fl+ x 11.999999997912326)))

(: fllog-gamma-taylor-neg-11-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-11-eps x)
  ((make-flpolyfun
    (2.73464038994808618853990996713900546500363245e-8
     3.99168059769041149523568978415501579131928981e7
     7.96675602195659478355932730588437298457319315e14
     2.12004956622456136373539147270355509119210030e22
     6.34692015133814117725645182781078049260408379e29
     2.02679011782791482635892447400196186445606399e37
     6.74191524487342619925839700081682702106088296e44
     2.30670605377823204002172584808694044342738686e52
     8.05667908149243458094760269141474835194991556e59
     2.85863889808333292404959945241460300717711546e67
     1.02696954543122321631238034925453188849766691e75
     3.72666741642077092523775278109077602761444219e82
     1.36360263507170131054023629385517986334747809e90
     5.02436847534761479437446945468643428751502305e97
     1.86231248648800037351163307354282477952748617e105
     6.93817241996743010308160148208151319569151205e112
     2.59640311288631117341622381354216175006694215e120
     9.75436357202621873441244105109792101963099867e127
     3.67731735733579134615816317769649110144961188e135
     1.39061135828683574763248386608711524913705288e143
     5.27333223621606622842738556284264689587972908e150
     2.00471016041737579551608260847917280484435997e158
     7.63842751793674500724945237092880255345519039e165
     2.91645018781248713953023068931212782470625220e173
     1.11564728782469245018582802734818152422134002e181
     4.27517506575444098042953315198093408442090338e188
     1.64087810744434949149168426042942529202007384e196))
   (fl+ x 11.000000025052106)))

(: fllog-gamma-taylor-neg-11+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-11+eps x)
  ((make-flpolyfun
    (+7.92108817813104976338185676938186326377669157e-9
     -3.99167954308609387899529490984926221844918726e7
     +7.96675376237834734151704486675254042966165671e14
     -2.12004866427321558401046925783755161819242013e22
     +6.34691655103692976844553017728587709050044410e29
     -2.02678868070285552882543026813904999725868034e37
     +6.74190950832996834570809466930634504334668633e44
     -2.30670376393361267698492163809552552947300333e52
     +8.05666994116530318362888597122002967632381485e59
     -2.85863524955716820122896866897527331390101936e67
     +1.02696808905630541487289935539691440983932245e75
     -3.72666160303803652380271345450756622764891911e82
     +1.36360031455530106555937243988871390508116396e90
     -5.02435921258854692720513068976400735845784268e97
     +1.86230878909092357208500698651798409488295265e105
     -6.93815766114120632269202387172502572591998986e112
     +2.59639722163506119449664891799231414302804036e120
     -9.75434005603601454303869335757998663598800980e127
     +3.67730797050484799427924584335172213978792387e135
     -1.39060761136423567937255934204138572903642527e143
     +5.27331727969979824813958778773305685936184817e150
     -2.00470419025458486185381910054896628118088328e158
     +7.63840368695691546299468028336596467987131921e165
     -2.91644067524776279755223104858712947907833526e173
     +1.11564349071318960661856937765706035633976974e181
     -4.27515990890013003716033146535924707095784523e188
     +1.64087205731302121418417611988033067328540294e196))
   (fl+ x 10.99999997494789)))

(: fllog-gamma-taylor-neg-10-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-10-eps x)
  ((make-flpolyfun
    (1.26680513875652366050062659798555748836557448e-9
     3.62880470810007800254459685298884367724025843e6
     6.58410327072133608976010077379791026736696158e12
     1.59282729754858621739197785943978568482028327e19
     4.33504158795023612686636032483709817064018640e25
     1.25848073033775948098011226489583728293213049e32
     3.80564819972389861941189673309097808504256595e38
     1.18370958467922967670391102325402180458143347e45
     3.75851711385162096495478164533275796515982118e51
     1.21234806747737194697431625227112203768112890e58
     3.95943437157865124122656766052022043628615746e64
     1.30618225249012131647321416064683238106758288e71
     4.34488746601852516679129150823293370559068272e77
     1.45539118843906333947353570381417641608669330e84
     4.90408933301516044397622239756677554793394680e90
     1.66095728655624292159781305892609910043079695e97
     5.65058035804620748434779937530937928416307129e103
     1.92986722993071906383516065435167362039046618e110
     6.61404526522253281591452715469943479530100677e116
     2.27378491990430232583217799728811483480130071e123
     7.83856027141921443400170282290734423438418616e129
     2.70900818888616748861088994297863316679286423e136
     9.38361642196091868726726724708956328592144715e142
     3.25707985403383821127106141913575458300804628e149
     1.13268282554189284115966562035709984608248417e156
     3.94587082206887966633395263207824801882064758e162
     1.37680628240094496354203214974016406530882430e169))
   (fl+ x 10.000000275573013)))

(: fllog-gamma-taylor-neg-10+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-10+eps x)
  ((make-flpolyfun
    (+1.77195439588259352534680065934485472482130717e-9
     -3.62879530292280735910715329248732327546191066e6
     +6.58408620929362087648818675232137910051457097e12
     -1.59282110629768751512482236172364227133645553e19
     +4.33501912113894078161722440069397118596511042e25
     -1.25847257758126423824236671781135806425817561e32
     +3.80561861500108372538274017835347112468902220e38
     -1.18369884897499829239437269713787403220880276e45
     +3.75847815612804690620280204631632142745125439e51
     -1.21233393049861990393492066947282644615217999e58
     +3.95938307131007784243873625636415505263472321e64
     -1.30616363664863309971765989200245939579148067e71
     +4.34481991285283001479406276580844541578499284e77
     -1.45536667474625860750715747873145948010476603e84
     +4.90400037772657535589147870859508009105310077e90
     -1.66092500646107207109975037317568159706562566e97
     +5.65046322003667333433184832173283915514569311e103
     -1.92982472288975465202547012222555503529161663e110
     +6.61389101567204650054160169384231724920491877e116
     -2.27372894582733669125740551080176860211820673e123
     +7.83835715268841248059221182226788240029783775e129
     -2.70893448116102191423806141451948887115494664e136
     +9.38334895136750337784284833671366500472886100e142
     -3.25698279430475194021685214932308042953682239e149
     +1.13264760450734840426305896015949764939171829e156
     -3.94574301197853025276755197047628891407891290e162
     +1.37675990267528835957901841535135801525088333e169))
   (fl+ x 9.99999972442663)))

(: fllog-gamma-taylor-neg-9-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-9-eps x)
  ((make-flpolyfun
    (3.44426332839150908016964631288898671467297262e-11
     3.62884503497526248849171273798030585155825256e5
     6.58417643205150350091693244126939847445604555e10
     1.59285384636586398381399114273935377829076317e16
     4.33513792862856030255974363976361743274012555e21
     1.25851569056292837773342167745950796476172579e27
     3.80577506382379113627666588338929419238118679e32
     1.18375562128157942007882679381420739030312789e38
     3.75868417204678487994283942978690056504502681e43
     1.21240868976300344095453444730782134597876236e49
     3.95965435848271122941100998848857882203084585e54
     1.30626208161146490626829498676570842825079611e60
     4.34517715092689444904089136353389740181780492e65
     1.45549630965889482118320478663843966519680255e71
     4.90447079820508329271112475618258161571787672e76
     1.66109571311879282678623022864996620010812047e82
     5.65108268207800990378493219357852914908486471e87
     1.93004951390012408175087355952942867414993847e93
     6.61470673955779226107136828988040612386835018e98
     2.27402495653376361378728330312251897555961576e104
     7.83943131932556570530934853960007532314000750e109
     2.70932427583375698078031816909880748505620829e115
     9.38476344220754571768181381486618051607588406e120
     3.25749608616749922641758390851141820070524336e126
     1.13283386837622981970366646739494916230346492e132
     3.94641892818466520527929989856415386232041462e137
     1.37700517982992993725897300416472670257692228e143))
   (fl+ x 9.000002755714823)))

(: fllog-gamma-taylor-neg-9+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-9+eps x)
  ((make-flpolyfun
    (+5.64090189601985554680682097784018867907670734e-10
     -3.62875496679368153944184754133409853882919443e5
     +6.58401301583004682272280455950037839439354423e10
     -1.59279454588740513697244106046241980273670396e16
     +4.33492273905226566045020044289934117963772572e21
     -1.25843760256945865003678562494637299374750375e27
     +3.80549169811299772192812168424181981155527489e32
     -1.18365279353241255772678524980019211222218158e38
     +3.75831103071047946418137890075594568351805075e43
     -1.21227328423483648919948531417540545823921189e49
     +3.95916299890191690076295645348221776277255504e54
     -1.30608377704671789364044842432919687821102207e60
     +4.34453011932208409681805831881788942103628916e65
     -1.45526151483004518742812595490882771735564618e71
     +4.90361877472979424936320260354852162497927007e76
     -1.66078653083994551814322808521680173590850409e82
     +5.64996072142402660565900542226566050612257751e87
     -1.92964237681781916467130434877438291278101621e93
     +6.61322932051282496993218213129464466379177035e98
     -2.27348883071046527400646257811471807542243270e104
     +7.83748582593701002317826220765303313699816913e109
     -2.70861829519255659885436495915569346459484308e115
     +9.38220157965541382228541682696143903308915602e120
     -3.25656643748408198086308251966072278642266846e126
     +1.13249651746180573680346026142972526977559758e132
     -3.94519474918571375789557929523084018794181615e137
     +1.37656094975453458883642260488249654746293086e143))
   (fl+ x 8.999997244250979)))

(: fllog-gamma-taylor-neg-8-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-8-eps x)
  ((make-flpolyfun
    (1.75595561986039004670754217836682458096818294e-11
     4.03242811088323551704009365369178172715764118e4
     8.12937510693064158715017081704767855649826588e8
     2.18529203315647635471555257729626032343249241e13
     6.6086739371290823870140997909983100301785661e17
     2.13180703452641813904222756674375930517880698e22
     7.16325190526575076978227354054622864556151838e26
     2.47575128649206217994857552477475876552461931e31
     8.73491424145784134221578357402078942389932035e35
     3.13075946401403942724197387949046574992807783e40
     1.13615030811408023488698439557270185507766119e45
     4.16472840047189860030185080087660141591495787e49
     1.53936533574880225322600394516463438385409716e54
     5.72958511566917473460716808668509800351815546e58
     2.1452705513984853310444983046579742329344217e63
     8.07351073038984550014037705417280543114023001e67
     3.05194907222492063206752094264072370160708407e72
     1.15822230051367698912467390023051867513568234e77
     4.41074466968106394593634292461279123515132827e81
     1.68490105089661117260686046397644316674926057e86
     6.45418761313859726010066816217053814833309179e90
     2.47853961949922330841354774036414813781846341e95
     9.53972945733347045297958182220264476100227504e99
     3.67937863604060660505188687357526850154343885e104
     1.42178738214301361299962576117341916756694372e109
     5.50363302233685355532541957273539197222287724e113
     2.13382946385035377749021639708399303746666443e118))
   (fl+ x 8.000024800270682)))

(: fllog-gamma-taylor-neg-8+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-8+eps x)
  ((make-flpolyfun
    (+3.36862440532578048192178695865761670703028502e-11
     -4.03157185435460182531674643103516701459589637e4
     +8.12764889390244843573678049475628009879133800e8
     -2.18459602406449894115960848402883573355750660e13
     +6.60586762847157542794164811192607499775649392e17
     -2.13067553087519165884280930132032897449909007e22
     +7.15868968253762378377140863376159796660903207e26
     -2.47391179828377991703980507205741954451491512e31
     +8.72749742497773520398866157186857418285340923e35
     -3.12776900359664828915446087881115386441329558e40
     +1.13494455446756995096973106168430195047603979e45
     -4.15986680173952967240784319130458379603509249e49
     +1.53740513912611874374960210051662718040768558e54
     -5.72168160282347819780640318082936113875273834e58
     +2.14208385499069002078835036682845477702444095e63
     -8.06066197034699160137863809357603591304015435e67
     +3.04676845211971994901145131666001174825250184e72
     -1.15613347446274974536712901225073621871500266e77
     +4.40232252293698652626946856575742521461284998e81
     -1.68150524128317151014581150044800262715047306e86
     +6.44049570857795427576454116828154920479724211e90
     -2.47301904349484581391700690657951303035314196e95
     +9.51747049451806490671496916087842154040751852e99
     -3.67040382207753008155004667954943602632015024e104
     +1.41816873708684031832737844076233529786494910e109
     -5.48904264518949055831628569939857570065655799e113
     +2.12794662366576202332888936112799382426183472e118))
   (fl+ x 7.9999751970958215)))

(: fllog-gamma-taylor-neg-7-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-7-eps x)
  ((make-flpolyfun
    (3.21679479852632759745367246513718980001635171e-12
     5.04402994112704184663363113670311009599091759e3
     1.27109588340331385101120105639416555466909227e7
     4.27258908022913507559604232033349315815168982e10
     1.61568434355360945943281198598054725594658597e14
     6.51704366037850112328249122862144378132933165e17
     2.73825261727442009104395249038917105694434166e21
     1.18339804857683193718808233344172104509872992e25
     5.22087179600848088036826889784761504704247785e28
     2.33988564629529232925154915680180765765285024e32
     1.06179645178199030729449917518216123215693221e36
     4.86690323452448117072549165516644604170335840e39
     2.24940821880306352619712347055263942397326750e43
     1.04691229045618429129108351747714342578718893e47
     4.90150829481408858461161948929810753980258873e50
     2.30659127823273559467878821641159615034308609e54
     1.09030009241427283176959849157211798338464498e58
     5.17393820186604758100969262963284482134212880e61
     2.46377917698243204656988411760922010406851181e65
     1.17686002585829238416238718500232471902547531e69
     5.63705852508412233807453056382118699556756662e72
     2.70686983054420215675911784186745327670050266e76
     1.30277109018236713325248958831916182321469469e80
     6.28300035623188169471632557335605421793373425e83
     3.03590240089126254927158800341104055505805124e87
     1.46947826096328253735160422033263210654528334e91
     7.12416473540339088510115562375712909673904392e94))
   (fl+ x 7.000198333407324)))

(: fllog-gamma-taylor-neg-7+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-7+eps x)
  ((make-flpolyfun
    (+5.31301106573590200897337106433467557164151869e-14
     -5.03596737376839319033525480958087182844986601e3
     +1.26906411660485282940047764431761330962540069e7
     -4.26234897647649129521641689329891672052035096e10
     +1.61052333144508032407479397436457390656912554e14
     -6.49103215888755446261074330125753000135307780e17
     +2.72514281959020665462626702046286032474881585e21
     -1.17679070972160310124859616823702810887510836e25
     +5.18757080225788525722758914222250520804782104e28
     -2.32310194156637801114545923844840993838349165e32
     +1.05333746219281065206009120880147302639211980e36
     -4.82426991259678266759054292661418820803932093e39
     +2.22792101614921848435241950907062703630531062e43
     -1.03608273550652442552036018281974271437287285e47
     +4.84692731070476016126702851590916393164116867e50
     -2.27908244708484583471179841061427804773346514e54
     +1.07643563313754077061546826446905160613202497e58
     -5.10406128115977026425966817266068851772181710e61
     +2.42856118390926383792291521433738334610659660e65
     -1.15911014378434091950170457728800186180055890e69
     +5.54759904629395404906129519141374436420742647e72
     -2.66178221396886203009894511168298544991620888e76
     +1.28004691042726129933820514080552438749211119e80
     -6.16847037831432940796431424756445185876829920e83
     +2.97817923251961975033066678329168308250404735e87
     -1.44038575256431930975064080992353469234753888e91
     +6.97753832629543722195438695192947848095269032e94))
   (fl+ x 6.999801507890638)))

(: fllog-gamma-taylor-neg-6-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-6-eps x)
  ((make-flpolyfun
    (5.96373419303467500413792265481786840068918471e-13
     7.23736629925709518153113039087848917130246729e2
     2.60548410303403933758252826421735218541543110e5
     1.25386984941127419905133271364532710453052978e8
     6.78846569389324815646189948077493720661377202e10
     3.92030195653034258327699392283536055154819091e13
     2.35828439915423448508952653715714843386001618e16
     1.45917470704069516669512400406851985503810018e19
     9.21665329528624029220037775084067740031616086e21
     5.91396340919928318917705315549706443020703352e24
     3.84219185759390857523956199815669468967426722e27
     2.52141462110708982986693669907718860598317088e30
     1.66845159218827458030659413493976770775046283e33
     1.11175581372838399194969164249804721425253752e36
     7.45216788983554497707935428135827871164530098e38
     5.02085075223011742997849667660296336748325148e41
     3.39786791862042848358472219461791491184338177e44
     2.30852966161124419756843153557542240458089730e47
     1.57387334424066037164075966856835758499992029e50
     1.07633283427581150041381904802190294929609939e53
     7.38121913528046491626428063066656081632724397e55
     5.07454028548546652282923260996205238775728209e58
     3.49664243034293515162189387270168949733615312e61
     2.41437081551384922759964267834853643796077387e64
     1.67023842928535310631607098959951607194524987e67
     1.15746431270476848578187718611073050993194347e70
     8.03400643083259808484932459059707879728773282e72))
   (fl+ x 6.001385294453154)))

(: fllog-gamma-taylor-neg-6+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-6+eps x)
  ((make-flpolyfun
    (+3.98943097843441932465652026666091000412175563e-13
     -7.16245430427834534714673409720501524934493419e2
     +2.57851589639768891326867318498133764598929190e5
     -1.23445282157985016580945759579913384375292671e8
     +6.64866335820997155004760461801734186213432062e10
     -3.81964423882096250884715503265367483472767970e13
     +2.28581065302601343849063472135311113229589014e16
     -1.40699335579708310084132549548594488557761312e19
     +8.84094489001693642835942147140025472914301438e21
     -5.64345082040892431617497604635292443260529529e24
     +3.64742052778954305530816853872536028608224385e27
     -2.38117731596356223311008014087648517813605274e30
     +1.56747910244616295308347662889941379734448921e33
     -1.03905428353497913012761193552317328229344863e36
     +6.92870606139998047359222823627264086936374153e38
     -4.64394960184288117953293901427853908990631646e41
     +3.12649226193266320638056018879596813548202725e44
     -2.11313383169320236892244849825165874991284698e47
     +1.43318417230683724207672956565653826555288418e50
     -9.75033396470380191538409915247952968713605747e52
     +6.65183825327027441169299935628486491306778882e55
     -4.54936691657501621666687834880905567118335622e58
     +3.11850297553268885844035989960765352437005977e61
     -2.14209926096916557565934538015232402947359683e64
     +1.47419444196325716857338412118878929175682948e67
     -1.01630622646690800002319571389560738233771459e70
     +7.01761972686052721453154108904317582064920521e72))
   (fl+ x 5.998607480080876)))

(: fllog-gamma-taylor-neg-5-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-5-eps x)
  ((make-flpolyfun
    (5.41885092655381025719595516331601539815956969e-15
     1.23362184563354043865731028461414057744606020e2
     7.40476243222876130900574826280080918143125990e3
     6.00556168452820552068729377561100555529370641e5
     5.48074863566570265324417981653566617471824549e7
     5.33525074338452723153487397127655583797976263e9
     5.41001609074791125558485600315244634161655653e11
     5.64256713562444548684611921421374417220809909e13
     6.00772100286369220122905461643344677542613570e15
     6.49803732093028846484598058316207969435715638e17
     7.11622512374105590783954671230253655863195756e19
     7.87194337362483304708962106716775948612245357e21
     8.78048223063437069160511062733533974970161719e23
     9.86236859897728368260206634391870540840187331e25
     1.11434973580101773101371794731069019794053142e28
     1.26556151272168476205402111916756928204819325e30
     1.44370846592998098137993770587584944505535249e32
     1.65339085935578868895584777638475475524988199e34
     1.90010200608912963279940507532952645389745432e36
     2.19038667338894880563303719195863734661098943e38
     2.53203300058817032335469415689374134458741212e40
     2.93430349185055042449367970919755908893213651e42
     3.40821208626557548218602650412971833405190776e44
     3.96685590281426661708411918644940947372395895e46
     4.62581209214915604420872188953110757048353555e48
     5.40361237948187342779585787258489671029005435e50
     6.32231043483605471166429476809359358293138224e52))
   (fl+ x 5.0082181683225935)))

(: fllog-gamma-taylor-neg-5+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-5+eps x)
  ((make-flpolyfun
    (+1.76836193508496134514687044485233022507412166e-14
     -1.16535781616245753768503617985375944578579193e2
     +6.99523595489431486546890813466674970256552674e3
     -5.51419997851318044632166207860447589187985901e5
     +4.89115827090625556320989173735148990068192623e7
     -4.62774717173966484754236449207884450962286344e9
     +4.56096042262246697956773952789936663634595578e11
     -4.62357002765258798875348281864575148589999927e13
     +4.78468574036034408626909392503392949786418539e15
     -5.03000968987250701086043900444438734899839871e17
     +5.35401097142582048519214039246283264279679805e19
     -5.75644682625551268823564997647690621683299800e21
     +6.24070799300926007561084330972156813164729147e23
     -6.81302027775316489238836963643326917419501011e25
     +7.48208993021257914055045142198078739956243209e27
     -8.25900304352361522965771494862295861316385777e29
     +9.15728705360304464559850646551668374267131592e31
     -1.01930889756736968113544397731888110415266705e34
     +1.13854488660950913720734782365262882725974297e36
     -1.27566601891368590344217106037433472891809611e38
     +1.43327167410740289698783159819255907081451586e40
     -1.61438511479149974470063382576325819998775478e42
     +1.82251740349289405753525333412414171829955385e44
     -2.06174264827513679839882304004797713214906927e46
     +2.33678617524058001428564879357591130591224694e48
     -2.65312757448388074721204114962473788958612848e50
     +3.01712094482992778631151481754365566073619428e52))
   (fl+ x 4.991544640560048)))

(: fllog-gamma-taylor-neg-4-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-4-eps x)
  ((make-flpolyfun
    (1.81301489023985618481731357730667339196237550e-14
     2.67904808861410321327214397331379787908541540e1
     3.24255322937858253736006013570778920615832578e2
     5.46569182077741616725319810542178832035822966e3
     1.04145555201462573673149727720949548371162345e5
     2.11666941757771701328409351874486396871812932e6
     4.48122159501603318236465855420526550067324334e7
     9.75830163538331950363973953929618675079580747e8
     2.16923649607162932956165726741988370482884479e10
     4.89867910498275971952758694178019464785226189e11
     1.12007244162039286479579040474081852339397694e13
     2.58689045256111345612494171926626369243546123e14
     6.02440400096176701173680787572491817237764037e15
     1.41278668508301263198302773720333978605091028e17
     3.33285577879941677496169133681524910584979240e18
     7.90274391149113501309340654162886498293963454e19
     1.88223479030962949513233495444370666816866525e21
     4.50059018694097294452319570025381466521375657e22
     1.07986756381211853649283994199009213197263987e24
     2.59904627328961985271105601748597429270721239e25
     6.27281137238709819146504016485952916588925394e26
     1.51774055997381403059397905691577190791272758e28
     3.68060121745351564615021740987651034473640274e29
     8.94413244088949726258433997879455026504805354e30
     2.17760661400824862848456267793034288891939313e32
     5.31098740106663851246556066853754013568800558e33
     1.29737804140610088127865888231535370941492990e35))
   (fl+ x 4.039361839740536)))

(: fllog-gamma-taylor-neg-4+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-4+eps x)
  ((make-flpolyfun
    (+8.78939314434977824362286615570184975741305041e-15
     -2.07250608458039191801398354912074701126554797e1
     +2.51714682586894146537732187837038486323386563e2
     -3.73060471568071868389371231096324364995674596e3
     +6.25880819187684354729315259842094937424442594e4
     -1.11999150166557525866438224883417960801815493e6
     +2.08771147265045175361955519775825227765181884e7
     -4.00276998187490238282584216189014567996179702e8
     +7.83439818373891296219323306813420731726214027e9
     -1.55772242416585467557455053761730286461842180e11
     +3.13595292529358977965704401186423364249660428e12
     -6.37696161822735582434913575368842994406753210e13
     +1.30756171365397339075006701778228546488926118e15
     -2.69983387930444407360331692215175599784681891e16
     +5.60775863730578162721388295019809496365914994e17
     -1.17074697159654892886605119268916507675086169e19
     +2.45511179588604297371791674939611648546057739e20
     -5.16867574457309810939753320374439139711959337e21
     +1.09192461783841750528498906123439438876072262e23
     -2.31392093346042541326971161202647385124381260e24
     +4.91710037482801295013628404667352873797929314e25
     -1.04750650406095202238738008578691313752209369e27
     +2.23661017141897372233705223490312596918865879e28
     -4.78544240743225380874419144451265275847103823e29
     +1.02583058100822899833668001151690176951567802e31
     -2.20284443421392884416725580252156642000839635e32
     +4.73791693429605430177700526875053064928423719e33))
   (fl+ x 3.9552942848585984)))

(: fllog-gamma-taylor-neg-3-eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-3-eps x)
  ((make-flpolyfun
    (1.69786559061210843022272847964297037675859473e-15
     7.78188465813136214399528129997442792941073176e0
     2.58313383723880324208953213398090230714514980e1
     1.12268986297176516519245402454125678034401274e2
     5.88890742238004921853852404638860526426305979e2
     3.27719378549536664789687328052979137659712232e3
     1.90230619843565318084065380848899905923057322e4
     1.13559664237136370180852663772361533676044070e5
     6.92049602289783024945417378060367661802816637e5
     4.28437744364603821541980538763698670942244785e6
     2.68555250555762961389414458921497932348535306e7
     1.70037345821745965876919924552878737697863407e8
     1.08557322157002487766499595339125057568363179e9
     6.97911539667148383495178196114992142539256805e9
     4.51355833575162388564796789592490869277550350e10
     2.93399385882337138148673434081036371240625868e11
     1.91572797345398693597903943399980424905205048e12
     1.25576468663909262494300353917876036735521610e13
     8.26015213762303044035070558862351046563406452e13
     5.45017333309697979666936223400225945240521114e14
     3.60609599644037084765420160822868349591590907e15
     2.39194587729231083416912378422842318565585423e16
     1.59019845254247279515221145045196949680406309e17
     1.05937454968300158726334055013820581387376515e18
     7.07081530914871512696425122433809109739803684e18
     4.72763664773893258007091476870784841957917544e19
     3.16602336231497561963030757035805419348902704e20))
   (fl+ x 3.14358088834998)))

(: fllog-gamma-taylor-neg-3+eps (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-3+eps x)
  ((make-flpolyfun
    (+1.73350924402450086109664880252280930803796840e-16
     -1.91435018561160055062671998360109773008757615e0
     +9.57518947570967212969871531897887298425867401e0
     -2.00951349168426252669134955591785433284090043e1
     +6.26272827135138013766227814092855203458008016e1
     -1.94766155303446568125118695588276590924110440e2
     +6.46905999712929665878051101089935247949317056e2
     -2.19311277648062925608544541675088274405618955e3
     +7.61039798665835057465104404450865651687943724e3
     -2.68045784206181952947864995136383095044210192e4
     +9.56174516433500463141401730092716465102711170e4
     -3.44497593434834251677126191694547877536433278e5
     +1.25156744652510559191680924102071419509151639e6
     -4.57871659458173884379082476598177277475632514e6
     +1.68504842706974508404657029185505612957061835e7
     -6.23306848324077503176646009404048417739453040e7
     +2.31593359803530705843535585519480903266158016e8
     -8.63873252929326222756433185277849724061407159e8
     +3.23354812607607746932238396579196197173278373e9
     -1.21409064008644206975493902628524125704746969e10
     +4.57117235417958792595283855057138618743523254e10
     -1.72540549493287770025997384282337316967575029e11
     +6.52740666207530857195855094792769006968850608e11
     -2.47450533666850553797231795437602547002229320e12
     +9.39848534763072962083260627797430059451185761e12
     -3.57587213746677906598735906878666200860291196e13
     +1.36270397145886520070135314755598407978744695e14))
   (fl+ x 2.7476826467274127)))

(: fllog-gamma-taylor-neg-2.5 (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-2.5 x)
  ((make-flpolyfun
    (5.61919235895009645091256851695098111749959486e-17
     1.51560344802165768188749164398339627893778105e0
     4.85832095163399627746734377392871693605230490e0
     1.41129114307798124401510783653592177347873738e0
     8.72178258381534729322013445636550188290755750e0
     5.80041456659987799413315121154253330844366082e0
     2.48249421218940773331265335049735451396738220e1
     2.40388230622929786781070091918937520366699213e1
     8.22291568270425660106801850355547569796638822e1
     1.00648123545511701102453344855177421851423816e2
     2.96436413507613369083757276470488051933710099e2
     4.25179845684382619948469561167761367915934946e2
     1.13045785817751386159338509375334334152608605e3
     1.81120541880187359111104929471617512361784951e3
     4.48721845318900263604051887722090475536449478e3
     7.77615810234425735666130539513148262342615631e3
     1.83471706999770464528633009895266966505869873e4
     3.36304570557943903790064207274422028127017659e4
     7.67210360239843918816147110553574167448750938e4
     1.46431998292134176729885025076314444052562342e5
     3.26433702221563796981145197985984661873939623e5
     6.41568327741749042873610986617199799626504025e5
     1.40797959337829879108442601929312734834777980e6
     2.82702369687106235551822865228378508072204174e6
     6.13938581529136603068173225483911055935198571e6
     1.25222596563403475911690432815529823250889260e7
     2.70076153104356837368592373996661893611674396e7))
   (fl+ x 2.4570247382208006)))

(: fllog-gamma-taylor-neg-2.625 (Flonum -> Flonum))
(define (fllog-gamma-taylor-neg-2.625 x)
  ((make-flpolyfun
    (-1.17511100308904859994666567899607070603472748e-1
     -1.57662052808143238767682222467571768255158395e-1
     +5.62280028708343342127100665301732994258009008e0
     -5.03224286025454307606105505974515135386153910e0
     +1.44026056764382911633999836745275241447715481e1
     -2.48971464377258885456665463953421569146427274e1
     +6.27638301132273807125391353825831515701418911e1
     -1.33164187794833814803033486866170142306775977e2
     +3.25020330318424079519181342335613727175845720e2
     -7.50032453555325748153334099067598984903044042e2
     +1.82939126670926077976630411313170194810520239e3
     -4.39223051811670135063825899001404441576172064e3
     +1.07991099207708003425000890908369348850196235e4
     -2.64900388706763510403708190506232237093229157e4
     +6.57316327437471829125602547425833118008518082e4
     -1.63393764818489768773926609932691194453735854e5
     +4.08791856900606820819408084967296134162542699e5
     -1.02552442766763569800382316377975918319805950e6
     +2.58350187169077781879301640113211587992619404e6))
   (fl+ x 2.625)))

(: fllog-gamma-special-negative (Flonum -> Flonum))
;; For x < 0, either computes log(abs(gamma(x))) to good precision (so far, largest observed error is
;; 11 ulps; usually <= 5) or returns 0.0. Having 0.0 as the "I won't compute this" value works fine,
;; since log(abs(gamma(x))) has no zeros at any negative, rational x.
(define (fllog-gamma-special-negative x)
  (cond
    [(x . fl>= . -10.0)
     (cond
       [(x . fl>= . -6.0)
        (cond
          [(x . fl>= . -4.0)
           (cond
             [(and (x . fl>= . -2.5775) (x . fl<= . -2.3359))
              (fllog-gamma-taylor-neg-2.5 x)]
             [(and (x . fl>= . (fl- -2.7476826467274127 6.9e-2))
                   (x . fl<= . (fl+ -2.7476826467274127 6.9e-2)))
              (fllog-gamma-taylor-neg-3+eps x)]
             [(and (x . fl>= . -2.7) (x . fl<= . -2.575))
              (fllog-gamma-taylor-neg-2.625 x)]
             [(and (x . fl>= . (fl- -3.14358088834998 4.1e-2))
                   (x . fl<= . (fl+ -3.14358088834998 4.1e-2)))
              (fllog-gamma-taylor-neg-3-eps x)]
             [(and (x . fl>= . (fl- -3.9552942848585984 1.25e-2))
                   (x . fl<= . (fl+ -3.9552942848585984 1.25e-2)))
              (fllog-gamma-taylor-neg-4+eps x)]
             [else
              0.0])]
          [(and (x . fl>= . (fl- -4.039361839740536 1.1e-2))
                (x . fl<= . (fl+ -4.039361839740536 1.1e-2)))
           (fllog-gamma-taylor-neg-4-eps x)]
          [(and (x . fl>= . (fl- -4.991544640560048 2.3e-3))
                (x . fl<= . (fl+ -4.991544640560048 2.4e-3)))
           (fllog-gamma-taylor-neg-5+eps x)]
          [(and (x . fl>= . (fl- -5.0082181683225935 2.3e-3))
                (x . fl<= . (fl+ -5.0082181683225935 2.3e-3)))
           (fllog-gamma-taylor-neg-5-eps x)]
          [(and (x . fl>= . (fl- -5.998607480080876 4.0e-4))
                (x . fl<= . (fl+ -5.998607480080876 4.0e-4)))
           (fllog-gamma-taylor-neg-6+eps x)]
          [else
           0.0])]
       [(x . fl>= . -8.0)
        (cond
          [(and (x . fl>= . (fl- -6.001385294453154 4.0e-4))
                (x . fl<= . (fl+ -6.001385294453154 4.0e-4)))
           (fllog-gamma-taylor-neg-6-eps x)]
          [(and (x . fl>= . (fl- -6.999801507890638 5.5e-5))
                (x . fl<= . (fl+ -6.999801507890638 5.5e-5)))
           (fllog-gamma-taylor-neg-7+eps x)]
          [(and (x . fl>= . (fl- -7.000198333407324 5.5e-5))
                (x . fl<= . (fl+ -7.000198333407324 5.5e-5)))
           (fllog-gamma-taylor-neg-7-eps x)]
          [(and (x . fl>= . (fl- -7.9999751970958215 7.1e-6))
                (x . fl<= . (fl+ -7.9999751970958215 7.1e-6)))
           (fllog-gamma-taylor-neg-8+eps x)]
          [else
           0.0])]
       [(and (x . fl>= . (fl- -8.000024800270682 7.1e-6))
             (x . fl<= . (fl+ -8.000024800270682 7.1e-6)))
        (fllog-gamma-taylor-neg-8-eps x)]
       [(and (x . fl>= . (fl- -8.999997244250979 7.8e-7))
             (x . fl<= . (fl+ -8.999997244250979 7.8e-7)))
        (fllog-gamma-taylor-neg-9+eps x)]
       [(and (x . fl>= . (fl- -9.000002755714823 7.8e-7))
             (x . fl<= . (fl+ -9.000002755714823 7.8e-7)))
        (fllog-gamma-taylor-neg-9-eps x)]
       [(and (x . fl>= . (fl- -9.99999972442663 8.0e-8))
             (x . fl<= . (fl+ -9.99999972442663 8.0e-8)))
        (fllog-gamma-taylor-neg-10+eps x)]
       [else
        0.0])]
    [(x . fl>= . -13.0)
     (cond
       [(x . fl>= . -11.5)
        (cond
          [(and (x . fl>= . (fl- -10.000000275573013 8.0e-8))
                (x . fl<= . (fl+ -10.000000275573013 8.0e-8)))
           (fllog-gamma-taylor-neg-10-eps x)]
          [(and (x . fl>= . (fl- -10.99999997494789 7.2e-9))
                (x . fl<= . (fl+ -10.99999997494789 7.2e-9)))
           (fllog-gamma-taylor-neg-11+eps x)]
          [(and (x . fl>= . (fl- -11.000000025052106 7.2e-9))
                (x . fl<= . (fl+ -11.000000025052106 7.2e-9)))
           (fllog-gamma-taylor-neg-11-eps x)]
          [else
           0.0])]
       [(and (x . fl>= . (fl- -11.999999997912326 6.0e-10))
             (x . fl<= . (fl+ -11.999999997912326 6.0e-10)))
        (fllog-gamma-taylor-neg-12+eps x)]
       [(and (x . fl>= . (fl- -12.000000002087674 6.0e-10))
             (x . fl<= . (fl+ -12.000000002087674 6.0e-10)))
        (fllog-gamma-taylor-neg-12-eps x)]
       [(and (x . fl>= . (fl- -12.99999999983941 4.5e-11))
             (x . fl<= . (fl+ -12.99999999983941 4.5e-11)))
        (fllog-gamma-taylor-neg-13+eps x)]
       [else
        0.0])]
    [(x . fl>= . -15.0)
     (cond
       [(and (x . fl>= . (fl- -13.00000000016059 4.5e-11))
             (x . fl<= . (fl+ -13.00000000016059 4.5e-11)))
        (fllog-gamma-taylor-neg-13-eps x)]
       [(and (x . fl>= . (fl- -13.99999999998853 3.4e-12))
             (x . fl<= . (fl+ -13.99999999998853 3.4e-12)))
        (fllog-gamma-taylor-neg-14+eps x)]
       [(and (x . fl>= . (fl- -14.00000000001147 3.4e-12))
             (x . fl<= . (fl+ -14.00000000001147 3.4e-12)))
        (fllog-gamma-taylor-neg-14-eps x)]
       [(and (x . fl>= . (fl- -14.999999999999236 2.1e-13))
             (x . fl<= . (fl+ -14.999999999999236 2.1e-13)))
        (fllog-gamma-taylor-neg-15+eps x)]
       [else
        0.0])]
    [(and (x . fl>= . (fl- -15.000000000000764 2.1e-13))
          (x . fl<= . (fl+ -15.000000000000764 2.1e-13)))
     (fllog-gamma-taylor-neg-15-eps x)]
    [(and (x . fl>= . (fl- -15.999999999999954 1e-14))
          (x . fl<= . (fl+ -15.999999999999954 1e-14)))
     (fllog-gamma-taylor-neg-16+eps x)]
    [(and (x . fl>= . (fl- -16.000000000000046 1e-14))
          (x . fl<= . (fl+ -16.000000000000046 1e-14)))
     (fllog-gamma-taylor-neg-16-eps x)]
    [else
     0.0]))
