// Emacs style mode select   -*- C++ -*-
//-----------------------------------------------------------------------------
//
// $Id: 5920918e6a03e48750784981c35e11f6a1565a47 $
//
// Copyright (C) 2021 by Alex Mayfield.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// DESCRIPTION:
//  A handle that wraps a resolved file on disk.
//
//-----------------------------------------------------------------------------

#ifndef __M_RESFILE_H__
#define __M_RESFILE_H__

#include <string>
#include <vector>

enum ofile_t
{
	/**
	 * @brief Unknown file type.
	 */
	OFILE_UNKNOWN,

	/**
	 * @brief Anything that would be passed as a -file or -iwad parameter.
	 */
	OFILE_WAD,

	/**
	 * @brief Anything that would be passed as a -deh parameter.
	 */
	OFILE_DEH,
};

/**
 * @brief A handle that wraps a resolved file on disk.
 */
class OResFile
{
	std::string m_fullpath;
	std::string m_hash;
	std::string m_basename;

  public:
	bool operator==(const OResFile& other) const
	{
		return m_hash == other.m_hash;
	}

	bool operator!=(const OResFile& other) const
	{
		return !(operator==(other));
	}

	/**
	 * @brief Get the full absolute path to the file.
	 */
	const std::string& getFullpath() const
	{
		return m_fullpath;
	}

	/**
	 * @brief Get a unique hash of the file.
	 */
	const std::string& getHash() const
	{
		return m_hash;
	}

	/**
	 * @brief Get the base filename, with no path.
	 */
	const std::string& getBasename() const
	{
		return m_basename;
	}

	static bool make(OResFile& out, const std::string& file);
	static bool makeWithHash(OResFile& out, const std::string& file,
	                         const std::string& hash);
};
typedef std::vector<OResFile> OResFiles;

/**
 * @brief A handle that represents a "wanted" file that may or may not exist.
 */
class OWantFile
{
	std::string m_wantedpath;
	ofile_t m_wantedtype;
	std::string m_wantedhash;
	std::string m_basename;
	std::string m_extension;

  public:
	/**
	 * @brief Get the original "wanted" path.
	 */
	const std::string& getWantedPath() const
	{
		return m_wantedpath;
	}

	/**
	 * @brief Get the original "wanted" path.
	 */
	ofile_t getWantedType() const
	{
		return m_wantedtype;
	}

	/**
	 * @brief Get the assumed hash of the file, or an empty string if there
	 *        is no hash.
	 */
	const std::string& getWantedHash() const
	{
		return m_wantedhash;
	}

	/**
	 * @brief Get the base filename of the resource, with no directory.
	 */
	const std::string& getBasename() const
	{
		return m_basename;
	}

	/**
	 * @brief Get the extension of the resource.
	 */
	const std::string& getExt() const
	{
		return m_extension;
	}

	static bool make(OWantFile& out, const std::string& file, ofile_t type);
	static bool makeWithHash(OWantFile& out, const std::string& file, ofile_t type,
	                         const std::string& hash);
};
typedef std::vector<OWantFile> OWantFiles;

std::string M_ResFilesToString(const OResFiles& files);
const std::vector<std::string>& M_FileTypeExts(ofile_t type);
std::vector<std::string> M_FileSearchDirs();
bool M_ResolveWantedFile(OResFile& out, const OWantFile& wanted);

#endif // __ORESFILE_H__
