"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});

const synthetics_1 = require("@elastic/synthetics");

const make_checks_1 = require("../../helpers/make_checks");

const monitor_details_1 = require("../../page_objects/monitor_details");

const utils_1 = require("../utils");

synthetics_1.journey('MonitorPingRedirects', async ({
  page,
  params
}) => {
  const monitorDetails = monitor_details_1.monitorDetailsPageProvider({
    page,
    kibanaUrl: params.kibanaUrl
  });
  const testMonitor = {
    id: '0000-intermittent',
    start: 'now-15m',
    end: 'now',
    redirects: ['http://localhost:3000/first', 'https://www.washingtonpost.com/']
  };
  synthetics_1.before(async () => {
    await monitorDetails.waitForLoadingToFinish();
    await make_checks_1.makeChecksWithStatus(params.getService('es'), testMonitor.id, 5, 2, 10000, {
      http: {
        rtt: {
          total: {
            us: 157784
          }
        },
        response: {
          status_code: 200,
          redirects: testMonitor.redirects,
          body: {
            bytes: 642102,
            hash: '597a8cfb33ff8e09bff16283306553c3895282aaf5386e1843d466d44979e28a'
          }
        }
      }
    }, 'up');
    await utils_1.delay(5000);
  });
  synthetics_1.step('go to monitor-management', async () => {
    await monitorDetails.navigateToOverviewPage({
      dateRangeEnd: testMonitor.end,
      dateRangeStart: testMonitor.start
    });
  });
  synthetics_1.step('login to Kibana', async () => {
    await monitorDetails.loginToKibana();
  });
  synthetics_1.step('go to monitor details', async () => {
    await monitorDetails.navigateToMonitorDetails(testMonitor.id);
  });
  synthetics_1.step('displays redirect info in detail panel', async () => {
    await monitorDetails.waitForLoadingToFinish();
    synthetics_1.expect(await monitorDetails.getMonitorRedirects()).toEqual(`${testMonitor.redirects.length}`);
  });
  synthetics_1.step('displays redirects in ping list expand row', async () => {
    await monitorDetails.expandPingDetails();
    await monitorDetails.waitForLoadingToFinish();
    await page.waitForSelector(utils_1.byTestId('uptimeMonitorPingListRedirectInfo'));
  });
});