"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});

const synthetics_1 = require("@elastic/synthetics");

const lodash_1 = require("lodash");

const monitor_details_1 = require("../../page_objects/monitor_details");

const utils_1 = require("../utils");

const dateRangeStart = '2019-09-10T12:40:08.078Z';
const dateRangeEnd = '2019-09-11T19:40:08.078Z';
const monitorId = '0000-intermittent';
const alertId = 'uptime-anomaly-alert';
const alertThreshold = 'major';
synthetics_1.journey('MonitorAlerts', async ({
  page,
  params
}) => {
  const monitorDetails = monitor_details_1.monitorDetailsPageProvider({
    page,
    kibanaUrl: params.kibanaUrl
  });
  synthetics_1.before(async () => {
    await monitorDetails.waitForLoadingToFinish();
  });
  synthetics_1.step('go to overview', async () => {
    await monitorDetails.navigateToOverviewPage({
      dateRangeEnd,
      dateRangeStart
    });
  });
  synthetics_1.step('login to Kibana', async () => {
    await monitorDetails.loginToKibana();
  });
  synthetics_1.step('go to monitor details', async () => {
    await monitorDetails.navigateToMonitorDetails(monitorId);
    await monitorDetails.waitForLoadingToFinish();
  });
  synthetics_1.step('clean previous data if available', async () => {
    // Should only happen locally
    await monitorDetails.disableAnomalyDetectionAlert().catch(lodash_1.noop);
    await monitorDetails.disableAnomalyDetection().catch(lodash_1.noop);
  });
  synthetics_1.step('open anomaly detection flyout', async () => {
    await monitorDetails.waitAndRefresh(5000);
    await monitorDetails.enableAnomalyDetection();
    await monitorDetails.ensureAnomalyDetectionFlyoutIsOpen();
  });
  synthetics_1.step('can create job', async () => {
    const canCreateJob = await monitorDetails.canCreateJob();
    const missingLicense = await page.waitForSelector('uptimeMLLicenseInfo', {
      timeout: 10000
    }).then(() => true).catch(() => false);
    synthetics_1.expect(canCreateJob).toBeTruthy();
    synthetics_1.expect(missingLicense).toBeFalsy();
  });
  synthetics_1.step('creates ML job', async () => {
    await page.click(utils_1.byTestId('uptimeMLCreateJobBtn'));
    await page.waitForSelector(utils_1.byTestId('uptimeMLJobSuccessfullyCreated'), {
      timeout: 30000
    });
    await page.click(utils_1.byTestId('toastCloseButton'));
  });
  synthetics_1.step('close anomaly detection flyout', async () => {
    await page.click(utils_1.byTestId('cancelSaveAlertButton'));
  });
  synthetics_1.step('open anomaly detection alert', async () => {
    await monitorDetails.waitAndRefresh(3000);
    await monitorDetails.openAnomalyDetectionMenu();
    await page.click(utils_1.byTestId('uptimeEnableAnomalyAlertBtn'));
  });
  synthetics_1.step('update anomaly detection alert', async () => {
    await monitorDetails.updateAlert({
      id: alertId,
      threshold: alertThreshold
    });
  });
  synthetics_1.step('save anomaly detection alert', async () => {
    await page.click(utils_1.byTestId('saveAlertButton'));
    await page.click(utils_1.byTestId('confirmModalConfirmButton'));
    await page.waitForSelector(`text=Created rule "${alertId}"`);
  });
  synthetics_1.step('disable anomaly detection alert', async () => {
    await monitorDetails.waitAndRefresh(5000);
    await monitorDetails.disableAnomalyDetectionAlert();
    await utils_1.delay(1000); // Menu has delay when closing

    await monitorDetails.disableAnomalyDetection();
  });
});