"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getChecksDateRange = exports.makeChecksWithStatus = exports.makeChecks = exports.makeCheck = void 0;

const tslib_1 = require("tslib");

const uuid_1 = tslib_1.__importDefault(require("uuid"));

const lodash_1 = require("lodash");

const make_ping_1 = require("./make_ping");

const getRandomMonitorId = () => {
  return 'monitor-' + Math.random().toString(36).substring(7);
};

const makeCheck = async ({
  es,
  monitorId = getRandomMonitorId(),
  numIps = 1,
  fields = {},
  mogrify = d => d,
  refresh = true,
  tls = false,
  isFleetManaged = false
}) => {
  const cgFields = {
    monitor: {
      check_group: uuid_1.default.v4()
    }
  };
  const docs = [];
  const summary = {
    up: 0,
    down: 0
  };

  for (let i = 0; i < numIps; i++) {
    const pingFields = lodash_1.merge(fields, cgFields, {
      monitor: {
        ip: `127.0.0.${i}`
      }
    });

    if (i === numIps - 1) {
      pingFields.summary = summary;
    }

    const doc = await make_ping_1.makePing(es, monitorId, pingFields, mogrify, false, tls, isFleetManaged);
    docs.push(doc); // @ts-ignore

    summary[doc.monitor.status]++;
  }

  if (refresh) {
    await es.indices.refresh();
  }

  return {
    monitorId,
    docs
  };
};

exports.makeCheck = makeCheck;

const makeChecks = async (es, monitorId, numChecks = 1, numIps = 1, every = 10000, // number of millis between checks
fields = {}, mogrify = d => d, refresh = true, isFleetManaged = false) => {
  const checks = [];
  const oldestTime = new Date().getTime() - numChecks * every;
  let newestTime = oldestTime;

  for (let li = 0; li < numChecks; li++) {
    const checkDate = new Date(newestTime + every);
    newestTime = checkDate.getTime() + every;
    fields = lodash_1.merge(fields, {
      '@timestamp': checkDate.toISOString(),
      monitor: {
        timespan: {
          gte: checkDate.toISOString(),
          lt: new Date(newestTime).toISOString()
        }
      }
    });
    const {
      docs
    } = await exports.makeCheck({
      es,
      monitorId,
      numIps,
      fields,
      mogrify,
      refresh: false,
      isFleetManaged
    });
    checks.push(docs);
  }

  if (refresh) {
    await es.indices.refresh();
  }

  return checks;
};

exports.makeChecks = makeChecks;

const makeChecksWithStatus = async (es, monitorId, numChecks, numIps, every, fields = {}, status, mogrify = d => d, refresh = true, isFleetManaged = false) => {
  const oppositeStatus = status === 'up' ? 'down' : 'up';
  return await exports.makeChecks(es, monitorId, numChecks, numIps, every, fields, d => {
    d.monitor.status = status;

    if (d.summary) {
      d.summary[status] += d.summary[oppositeStatus];
      d.summary[oppositeStatus] = 0;
    }

    return mogrify(d);
  }, refresh, isFleetManaged);
};

exports.makeChecksWithStatus = makeChecksWithStatus; // Helper for processing a list of checks to find the time picker bounds.

const getChecksDateRange = checks => {
  // Flatten 2d arrays
  const flattened = lodash_1.flattenDeep(checks);
  let startTime = 1 / 0;
  let endTime = -1 / 0;
  flattened.forEach(c => {
    const ts = Date.parse(c['@timestamp']);

    if (ts < startTime) {
      startTime = ts;
    }

    if (ts > endTime) {
      endTime = ts;
    }
  });
  return {
    start: new Date(startTime).toISOString(),
    end: new Date(endTime).toISOString()
  };
};

exports.getChecksDateRange = getChecksDateRange;