"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getResponsePayloadBytes = getResponsePayloadBytes;

var _lodash = require("lodash");

var _fs = require("fs");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const isBuffer = obj => Buffer.isBuffer(obj);

const isFsReadStream = obj => typeof obj === 'object' && obj !== null && 'bytesRead' in obj && obj instanceof _fs.ReadStream;

const isZlibStream = obj => {
  return typeof obj === 'object' && obj !== null && 'bytesWritten' in obj;
};

const isString = obj => typeof obj === 'string';
/**
 * Attempts to determine the size (in bytes) of a hapi/good
 * responsePayload based on the payload type. Falls back to
 * `undefined` if the size cannot be determined.
 *
 * This is similar to the implementation in `core/server/http/logging`,
 * however it uses more duck typing as we do not have access to the
 * entire hapi request object like we do in the HttpServer.
 *
 * @param headers responseHeaders from hapi/good event
 * @param payload responsePayload from hapi/good event
 *
 * @internal
 */


function getResponsePayloadBytes(payload, headers = {}) {
  const contentLength = headers['content-length'];

  if (contentLength) {
    const val = parseInt( // hapi response headers can be `string | string[]`, so we need to handle both cases
    Array.isArray(contentLength) ? String(contentLength) : contentLength, 10);
    return !isNaN(val) ? val : undefined;
  }

  if (isBuffer(payload)) {
    return payload.byteLength;
  }

  if (isFsReadStream(payload)) {
    return payload.bytesRead;
  }

  if (isZlibStream(payload)) {
    return payload.bytesWritten;
  }

  if (isString(payload)) {
    return Buffer.byteLength(payload);
  }

  if ((0, _lodash.isPlainObject)(payload) || Array.isArray(payload)) {
    return Buffer.byteLength(JSON.stringify(payload));
  }

  return undefined;
}