import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _toArray from "@babel/runtime/helpers/toArray";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { pick, map, mapValues } from 'lodash';
import { nodeTypes } from '../node_types';
import * as ast from '../ast';
import { getRangeScript } from '../../filters';
import { getFields } from './utils/get_fields';
import { getTimeZoneFromSettings, getDataViewFieldSubtypeNested } from '../../utils';
import { getFullFieldNameNode } from './utils/get_full_field_name_node';
export function buildNodeParams(fieldName, params) {
  var paramsToMap = pick(params, 'gt', 'lt', 'gte', 'lte', 'format');
  var fieldNameArg = typeof fieldName === 'string' ? ast.fromLiteralExpression(fieldName) : nodeTypes.literal.buildNode(fieldName);
  var args = map(paramsToMap, function (value, key) {
    return nodeTypes.namedArg.buildNode(key, value);
  });
  return {
    arguments: [fieldNameArg].concat(_toConsumableArray(args))
  };
}
export function toElasticsearchQuery(node, indexPattern) {
  var config = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var context = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};

  var _node$arguments = _toArray(node.arguments),
      fieldNameArg = _node$arguments[0],
      args = _node$arguments.slice(1);

  var fullFieldNameArg = getFullFieldNameNode(fieldNameArg, indexPattern, context !== null && context !== void 0 && context.nested ? context.nested.path : undefined);
  var fields = indexPattern ? getFields(fullFieldNameArg, indexPattern) : [];
  var namedArgs = extractArguments(args);
  var queryParams = mapValues(namedArgs, function (arg) {
    return ast.toElasticsearchQuery(arg);
  }); // If no fields are found in the index pattern we send through the given field name as-is. We do this to preserve
  // the behaviour of lucene on dashboards where there are panels based on different index patterns that have different
  // fields. If a user queries on a field that exists in one pattern but not the other, the index pattern without the
  // field should return no results. It's debatable whether this is desirable, but it's been that way forever, so we'll
  // keep things familiar for now.

  if (fields && fields.length === 0) {
    fields.push({
      name: ast.toElasticsearchQuery(fullFieldNameArg),
      scripted: false,
      type: ''
    });
  }

  var queries = fields.map(function (field) {
    var wrapWithNestedQuery = function wrapWithNestedQuery(query) {
      // Wildcards can easily include nested and non-nested fields. There isn't a good way to let
      // users handle this themselves so we automatically add nested queries in this scenario.
      var subTypeNested = getDataViewFieldSubtypeNested(field);

      if (!(fullFieldNameArg.type === 'wildcard') || !(subTypeNested !== null && subTypeNested !== void 0 && subTypeNested.nested) || context.nested) {
        return query;
      } else {
        return {
          nested: {
            path: subTypeNested.nested.path,
            query: query,
            score_mode: 'none'
          }
        };
      }
    };

    if (field.scripted) {
      return {
        script: getRangeScript(field, queryParams)
      };
    } else if (field.type === 'date') {
      var timeZoneParam = config.dateFormatTZ ? {
        time_zone: getTimeZoneFromSettings(config.dateFormatTZ)
      } : {};
      return wrapWithNestedQuery({
        range: _defineProperty({}, field.name, _objectSpread(_objectSpread({}, queryParams), timeZoneParam))
      });
    }

    return wrapWithNestedQuery({
      range: _defineProperty({}, field.name, queryParams)
    });
  });
  return {
    bool: {
      should: queries,
      minimum_should_match: 1
    }
  };
}

function extractArguments(args) {
  if (args.gt && args.gte || args.lt && args.lte) {
    throw new Error('range ends cannot be both inclusive and exclusive');
  }

  var unnamedArgOrder = ['gte', 'lte', 'format'];
  return args.reduce(function (acc, arg, index) {
    if (arg.type === 'namedArg') {
      acc[arg.name] = arg.value;
    } else {
      acc[unnamedArgOrder[index]] = arg;
    }

    return acc;
  }, {});
}