"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConfigFromFiles = void 0;

var _fs = require("fs");

var _jsYaml = require("js-yaml");

var _saferLodashSet = require("@elastic/safer-lodash-set");

var _lodash = require("lodash");

var _ensure_deep_object = require("./ensure_deep_object");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const readYaml = path => (0, _jsYaml.safeLoad)((0, _fs.readFileSync)(path, 'utf8'));

function replaceEnvVarRefs(val) {
  return val.replace(/\$\{(\w+)\}/g, (match, envVarName) => {
    const envVarValue = process.env[envVarName];

    if (envVarValue !== undefined) {
      return envVarValue;
    }

    throw new Error(`Unknown environment variable referenced in config : ${envVarName}`);
  });
}

function merge(target, value, key) {
  if (((0, _lodash.isPlainObject)(value) || Array.isArray(value)) && Object.keys(value).length > 0) {
    for (const [subKey, subVal] of Object.entries(value)) {
      merge(target, subVal, key ? `${key}.${subKey}` : subKey);
    }
  } else if (key !== undefined) {
    (0, _saferLodashSet.set)(target, key, typeof value === 'string' ? replaceEnvVarRefs(value) : value);
  }

  return target;
}
/** @internal */


const getConfigFromFiles = configFiles => {
  let mergedYaml = {};

  for (const configFile of configFiles) {
    const yaml = readYaml(configFile);

    if (yaml !== null) {
      mergedYaml = merge(mergedYaml, yaml);
    }
  }

  return (0, _ensure_deep_object.ensureDeepObject)(mergedYaml);
};

exports.getConfigFromFiles = getConfigFromFiles;