"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileLayer = exports.EMSFormatType = void 0;

var _url = _interopRequireDefault(require("url"));

var _ems_service = require("./ems_service");

var topojson = _interopRequireWildcard(require("topojson-client"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

let EMSFormatType;
exports.EMSFormatType = EMSFormatType;

(function (EMSFormatType) {
  EMSFormatType["geojson"] = "geojson";
  EMSFormatType["topojson"] = "topojson";
})(EMSFormatType || (exports.EMSFormatType = EMSFormatType = {}));

class FileLayer extends _ems_service.AbstractEmsService {
  constructor(config, emsClient, proxyPath) {
    super(config, emsClient, proxyPath);

    _defineProperty(this, "_config", void 0);

    this._config = config;
  }

  async getGeoJson() {
    const cachedGeoJson = this._emsClient.getCachedGeoJson(this.getId());

    if (cachedGeoJson) {
      return cachedGeoJson;
    }

    const format = this.getDefaultFormatType();
    const fetchUrl = this.getDefaultFormatUrl();
    let geojson;
    const fetchedJson = await this._emsClient.getManifest(fetchUrl);

    if (fetchedJson) {
      if (format === 'geojson') {
        geojson = fetchedJson;
      } else if (format === 'topojson') {
        const meta = this.getDefaultFormatMeta();
        const featureCollectionPath = (meta === null || meta === void 0 ? void 0 : meta.feature_collection_path) ?? 'data'; // @ts-expect-error see https://github.com/DefinitelyTyped/DefinitelyTyped/pull/52156

        geojson = topojson.feature(fetchedJson, featureCollectionPath);
      } else {
        return;
      }

      this._emsClient.cacheGeoJson(this.getId(), geojson);

      return geojson;
    }

    return;
  }

  getFields() {
    return this._config.fields;
  }

  getFieldsInLanguage() {
    return this.getFields().map(field => {
      return {
        type: field.type,
        name: field.id,
        description: this._emsClient.getValueInLanguage(field.label)
      };
    });
  }

  getDisplayName() {
    const layerName = this._emsClient.getValueInLanguage(this._config.layer_name);

    return layerName ? layerName : '';
  }

  getId() {
    return this._config.layer_id;
  }

  hasId(id) {
    const matchesLegacyId = this._config.legacy_ids.indexOf(id) >= 0;
    return this._config.layer_id === id || matchesLegacyId;
  }

  getEMSHotLink() {
    const landingPageString = this._emsClient.getLandingPageUrl();

    const urlObject = _url.default.parse(landingPageString, true);

    urlObject.hash = `file/${this.getId()}`;
    urlObject.query = { ...urlObject.query,
      locale: this._emsClient.getLocale()
    };
    return _url.default.format(urlObject);
  }

  getDefaultFormatType() {
    const format = this._getDefaultFormat();

    return format.type;
  }

  getFormatOfType(type) {
    const format = this._getFormatOfType(type);

    return format.type;
  }

  getDefaultFormatMeta() {
    const format = this._getDefaultFormat();

    return this._getFormatMeta(format);
  }

  getFormatOfTypeMeta(type) {
    const format = this._getFormatOfType(type);

    return this._getFormatMeta(format);
  }

  getDefaultFormatUrl() {
    const format = this._getDefaultFormat();

    return this._getFormatUrl(format);
  }

  getFormatOfTypeUrl(type) {
    const format = this._getFormatOfType(type);

    return this._getFormatUrl(format);
  }

  getCreatedAt() {
    return this._config.created_at;
  }

  getApiUrl() {
    return this._emsClient.getFileApiUrl();
  }

  _getFormatUrl(format) {
    const url = this._proxyPath + this._getAbsoluteUrl(format.url);

    return this._emsClient.extendUrlWithParams(url);
  }

  _getFormatMeta(format) {
    if ('meta' in format) {
      return format.meta;
    } else {
      return;
    }
  }

  _getDefaultFormat() {
    const defaultFormat = this._config.formats.find(format => {
      return format.legacy_default;
    });

    if (defaultFormat) {
      return defaultFormat;
    }

    return this._config.formats[0];
  }

  _getFormatOfType(type) {
    const requestedFormat = this._config.formats.find(format => {
      return format.type === type;
    });

    return requestedFormat || this._getDefaultFormat();
  }

}

exports.FileLayer = FileLayer;