/***************************************************************************
                          kdrawview.cpp  -  description
                             -------------------
    begin                : Thu Nov 11 1999
    copyright            : (C) 1999 by Franois Dupoux
    email                : dupoux@dupoux.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kgraphspace.h"
#include "kgraphspacedoc.h"
#include "kdrawview.h"
#include "kgraphspaceview.h"
#include "ktreeitem.h"
#include "kgraphpart.h"
#include "ktreeview.h"
#include "filelib.h"

#include <qpainter.h>
#include <qdir.h>
#include <qfileinfo.h>

#include <kpopupmenu.h>

#include <math.h>
#include <stdio.h>

// ===========================================================================================================================
KDrawView::KDrawView(QWidget * parent, const char *name): QWidget(parent, name)
{	setMouseTracking(true); // To receive move event
	m_app = (KGraphSpaceApp *) parentWidget() -> parentWidget();
}

// ===========================================================================================================================
KDrawView::~KDrawView()
{
}

// ===========================================================================================================================
void KDrawView::drawWindow()
{
   convertTreeToClass();
   QPainter qp(this);
   drawGraph(&qp,false);
}

// ===========================================================================================================================
void KDrawView::paintEvent(QPaintEvent * /*e*/)
{	
   setUpdatesEnabled(false);
   drawWindow();
   setUpdatesEnabled(true);
}

// ===========================================================================================================================
void KDrawView::mouseMoveEvent (QMouseEvent *me)
{
   // Don't look events while working
   //if (g_bThreadRunning)
   //	return;

   int nRegion;
   QString strMsg;
   QString strType;
   int nType;
   QString strUsedSize, strDataSize;

   nRegion = getRegionNumber(me -> pos());
   if (nRegion == -1) // Mouse is not on a region
   {	//QToolTip::remove(this);
      return;
   }
	
   nType = m_kgpPart[nRegion].m_nType;

   if (nType == KGraphPart::Directory) // DIRECTORY
   {	strUsedSize = formatSize(m_kgpPart[nRegion].m_llUsedSize);
   strDataSize = formatSize(m_kgpPart[nRegion].m_llDataSize);
   strMsg.sprintf(i18n("Directory %s: %s = %ld %% (data size = %s)"), m_kgpPart[nRegion].m_strTitle.data(), strUsedSize.data(), (int) (m_kgpPart[nRegion].m_dPourcent * (double)100), strDataSize.data());
   }

   if (nType == KGraphPart::File) // FILE
   {	strUsedSize = formatSize(m_kgpPart[nRegion].m_llUsedSize);
   strDataSize = formatSize(m_kgpPart[nRegion].m_llDataSize);
   strMsg.sprintf(i18n("File %s: %s = %ld %% (data size = %s)"), m_kgpPart[nRegion].m_strTitle.data(), strUsedSize.data(), (int) (m_kgpPart[nRegion].m_dPourcent * (double)100), strDataSize.data());
   }

   if (nType == KGraphPart::Other) // OTHER
   {	strUsedSize = formatSize(m_kgpPart[nRegion].m_llUsedSize);
   strDataSize = formatSize(m_kgpPart[nRegion].m_llDataSize);
   strMsg.sprintf(i18n("%s = %ld %%: all files and directories which are too small to be drawn (data size = %s)"), strUsedSize.data(), (int) (m_kgpPart[nRegion].m_dPourcent * (double)100), strDataSize.data());
   }

   m_app -> slotStatusMsg(strMsg);
}

// ===========================================================================================================================
void KDrawView::mouseDoubleClickEvent ( QMouseEvent *me )
{
   printf ("Doubleclick (threadrun=%d):\n", g_bThreadRunning);

   // Don't look events while working
   if (g_bThreadRunning)
      return;

   int nRegion;

   nRegion = getRegionNumber(me -> pos());
   if (nRegion == -1) // Mouse is not on a region
      return;
	
   if (m_kgpPart[nRegion].m_nType != KGraphPart::Directory) // if not a directory
      return;

   getTreeView()->selectChild(m_kgpPart[nRegion].m_strTitle);
   printf("----> %s\n", m_kgpPart[nRegion].m_strTitle.data());
   drawWindow();

   m_app->updateCommands();
}

// ===========================================================================================================================
void KDrawView::mouseReleaseEvent (QMouseEvent *event)
{
   // Don't look events while working
   if (g_bThreadRunning)
      return;

   if (event->button() != QMouseEvent::RightButton)
      return;

   KPopupMenu *popup;
   QPoint qpPos;
   int nRegion;
   QString strPath;
    
   qpPos = event->pos();
   nRegion = getRegionNumber(qpPos);
   if (nRegion == -1) // Mouse is not on a region
      return;
	
   if (m_kgpPart[nRegion].m_nType == KGraphPart::Directory) // if a directory
      popup = getMainView()->getFolderPopup();
   else if (m_kgpPart[nRegion].m_nType == KGraphPart::File) // if a file
      popup = getMainView()->getFilePopup();
   else
      return; // Not a File and not a Folder
    
   strPath = getTreeView()->getPathFromLvi((QListViewItem *) m_kgpPart[nRegion].m_kti);
   getMainView()->getDocument()->setCurrentItem(strPath, m_kgpPart[nRegion].m_strTitle, m_kgpPart[nRegion].m_nType);

   popup->popup(event->globalPos());
}

// ===========================================================================================================================
/** Create KGraphPart in the memory from the tree */
int KDrawView::convertTreeToClass()
{
   // List all the dirs of the tree
   KTreeItem *kdCurrent, *kdFirst, *kdCurSel;
   long long llTotalSize; // Total size of the parent directory
   long long llCurrentDataSize, llCurrentUsedSize;
   long long llLimitSize;
   int nType;
	
   // initialization
   kdCurSel = getTreeView() -> currentDirectory();
	
   // List of all colors to be used
   const uint MAX_COLORS = 10;
   QColor qcColors[MAX_COLORS] = {QColor(255,255,0), QColor(255,0,0), QColor(255,0,255), QColor(0,255,0), QColor(0,0,255), QColor(255,192,0), QColor(0,255,192), QColor(192,192,255), QColor(255,255,192), QColor(192,0,255)};
	
   // Total Space
   kdCurSel -> getUsedSize(&llTotalSize);

   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* INITIALIZE *-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   m_nCurPartNum = 0;
   LLSIZE llOtherDataSize = 0, llOtherUsedSize = 0;
   int nUsedColorNb = 0;

   // Caclulate the Limit Size (If size is most, go to Divers part)
   llLimitSize = (LLSIZE) ( ((float)llTotalSize) * ((float)m_app->getSettings().nOtherPct / 100.0)); // 5 % of total space is the limit

   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* FILES and DIRECTORIES *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   kdCurrent = kdFirst = (KTreeItem *) (kdCurSel -> firstChild());
   if (kdCurrent == NULL)
   {	return -1;
   }

   do
   {	kdCurrent -> getUsedSize(&llCurrentUsedSize);
   kdCurrent -> getDataSize(&llCurrentDataSize);

   // If the directory or file is small (size < limit)
   if (llCurrentUsedSize < llLimitSize)
   {	llOtherDataSize += llCurrentDataSize;
   llOtherUsedSize += llCurrentUsedSize;
   }
   else // if item is big -> new part
   {	
      // KTreeItem pointer
      m_kgpPart[m_nCurPartNum].m_kti = kdCurrent;	
		
      // Type
      kdCurrent -> getType(&nType); // If it is a Directory
      m_kgpPart[m_nCurPartNum].m_nType = nType;

      // Size
      m_kgpPart[m_nCurPartNum].m_llUsedSize = llCurrentUsedSize;
      m_kgpPart[m_nCurPartNum].m_llDataSize = llCurrentDataSize;

      // Title
      m_kgpPart[m_nCurPartNum].m_strTitle = kdCurrent -> text(0);

      // Brush
      if (nType == KTreeItem::Directory) // Directory
      {	m_kgpPart[m_nCurPartNum].m_qbBrush = QBrush(qcColors[nUsedColorNb % MAX_COLORS]);
      nUsedColorNb++;
      }
      else // File
      {	m_kgpPart[m_nCurPartNum].m_qbBrush = QBrush(QColor(0,255,255), BDiagPattern);
      }

      // Pourcent %
      m_kgpPart[m_nCurPartNum].m_dPourcent = ((double)llCurrentUsedSize) / ((double)llTotalSize);
	
      m_nCurPartNum++;
   }

   kdCurrent = (KTreeItem *) (kdCurrent -> nextSibling());
	
   } while(kdCurrent && kdCurrent != kdFirst);

   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* OTHERS *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
		
   // KTreeItem pointer
   m_kgpPart[m_nCurPartNum].m_kti = 0; // No real file or directory	
	
   // Type
   m_kgpPart[m_nCurPartNum].m_nType = KGraphPart::Other;

   // Size
   m_kgpPart[m_nCurPartNum].m_llDataSize = llOtherDataSize;
   m_kgpPart[m_nCurPartNum].m_llUsedSize = llOtherUsedSize;

   // Title
   m_kgpPart[m_nCurPartNum].m_strTitle.sprintf(i18n("OTHER"));

   // Brush
   m_kgpPart[m_nCurPartNum].m_qbBrush = QBrush(QColor(255,0,0), FDiagPattern);

   // Pourcent %
   m_kgpPart[m_nCurPartNum].m_dPourcent = ((double)llOtherUsedSize) / ((double)llTotalSize);

   m_nCurPartNum++;
   return 0;
}


// ===========================================================================================================================
void KDrawView::drawGraph(QPainter *painter, bool bPrinting)
{
   double dCurrentAngle = 0.0;
   double dOldAngle;
   uint nRayonX, nRayonY;
   QSize qsWinSize;
   QRect qrDrawRect;
   QPoint ptCenter;
   double ptTextX, ptTextY;
   QPointArray ptaPolygon(PART_REGIONPTSNB+2);
   double a, b;
   //float fSize;
   QString strText, strSize;
   QSize sizeText;
   int i, j;
   int nHeaderHeight = 0; // not null if we print a title (to the printer)
	
   painter -> setPen( QColor(0,0,0) ); // Must be black, because can be white with some themes	
   const int nFrameSize = 15;

   // Erase Background
   setBackgroundColor( white );
   painter -> eraseRect(0,0,size().width(),size().height());

   printf ("printing = %d\n", bPrinting);
	
   // size of the view
   qsWinSize = size();
	
   // print header
   if (bPrinting)
   {
      strText = getTreeView() -> currentPath(); // directory path
      QFontMetrics fm = painter -> fontMetrics();
      sizeText = fm.size(SingleLine, strText.data());
		
      int ptTextX = (qsWinSize.width() -  (2 * nFrameSize)) / 2;
      int ptTextY = nFrameSize + sizeText.height();
			
      painter -> drawText( ((int)ptTextX) - (sizeText.width() / 2), ((int)ptTextY) - (sizeText.height()/2), strText);
      nHeaderHeight = nFrameSize + sizeText.height(); // nFrameSize will make space under the title
   }
	
   // Get center of the window
   ptCenter.setX(qsWinSize.width() / 2);
   ptCenter.setY(((qsWinSize.height()-nHeaderHeight) / 2)+nHeaderHeight);
   qrDrawRect = QRect(nFrameSize, nHeaderHeight+nFrameSize, qsWinSize.width() -  (2 * nFrameSize), qsWinSize.height() - nHeaderHeight -  (2 * nFrameSize));

   // Get rayon of circle
   nRayonX = (qsWinSize.width() / 2) - (2 * nFrameSize);
   nRayonY = ((qsWinSize.height()-nHeaderHeight) / 2) - (2 * nFrameSize);
	
   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* DRAW PART *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   dOldAngle = 0.0;

   for (i=0; i < m_nCurPartNum; i++)
   {
      // Set drawing properties
      painter -> setBrush( m_kgpPart[i].m_qbBrush );
	
      // Calculate the angle		
      if (m_kgpPart[i].m_dPourcent == 1.0) // Draw ellipse
      {   painter -> drawEllipse(qrDrawRect);		
      }
      else if(m_kgpPart[i].m_dPourcent > 0.0)
      {   dCurrentAngle = m_kgpPart[i].m_dPourcent * ((double) (360 * 16) ); // Draw pie
      painter -> drawPie(qrDrawRect, (int)dOldAngle, (int)dCurrentAngle);
      }

      dOldAngle += dCurrentAngle; // Calculate total angle for next dir
   }

   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* DRAW TITLE *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   dOldAngle = 0.0;

   for (i=0; i < m_nCurPartNum; i++)
   {
      // Draw angle only if more than 1 % of the size
      if (m_kgpPart[i].m_dPourcent > PARTS_LIMITTITLEDRAW)
      {
	 // Calculate the angle		
	 dCurrentAngle = dOldAngle + (m_kgpPart[i].m_dPourcent / 2);
	 dCurrentAngle *= (double) (2.0 * M_PI);

	 ptTextX = cos(dCurrentAngle) * nRayonX;
	 ptTextX /= 2.0;
	 ptTextX += (double) ptCenter.x();
	 ptTextY = -sin(dCurrentAngle) * nRayonY;
	 ptTextY /= 2.0;
	 ptTextY += (double) ptCenter.y();

	 strSize = formatSize(m_kgpPart[i].m_llUsedSize);
	 strText.sprintf(i18n("%s = %s"), m_kgpPart[i].m_strTitle.data(), strSize.data());

	 // Get size of text to center it
	 QFontMetrics fm = painter -> fontMetrics();
	 sizeText = fm.size(SingleLine, strText.data());

	 // Draw the text		
	 if (m_kgpPart[i].m_dPourcent == 1.0) // If 100 % --> Draw in center
	 {   painter -> drawText( ptCenter.x() - (sizeText.width() / 2), ptCenter.y() - (sizeText.height()/2), strText);
	 }
	 else if(m_kgpPart[i].m_dPourcent > 0.0) // Draw text in the pie: calculate the angle
	 {
	    painter -> drawText( ((int)ptTextX) - (sizeText.width() / 2), ((int)ptTextY) - (sizeText.height()/2), strText);
	 }
		
      }
      dOldAngle += m_kgpPart[i].m_dPourcent; // Calculate total angle for next dir
   }

   // *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-* CREATE REGIONS *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   dOldAngle = 0.0;

   for (i=0; i < m_nCurPartNum; i++)
   {
      // Draw angle only if more than 1 % of the size
      if (m_kgpPart[i].m_dPourcent > PARTS_LIMITTITLEDRAW)
      {
	 // Center of the circle
	 ptaPolygon.setPoint(0, ptCenter);
	 dCurrentAngle = dOldAngle;

	 // Set 10 points between the arc extremums
	 for (j=0; j < PART_REGIONPTSNB; j++)
	 {	//fprintf(stderr, "%ld: %f\n", i, dCurrentAngle);
	    a = (double) cos(dCurrentAngle * 2.0 * M_PI) * ((double) nRayonX);
	    a += (double) ptCenter.x();
	    b = (double) -sin(dCurrentAngle * 2.0 * M_PI) * ((double) nRayonY);
	    b += (double) ptCenter.y();
	    ptaPolygon.setPoint(1+j, (int)a, (int)b);
	    dCurrentAngle += (double) m_kgpPart[i].m_dPourcent / (double) (PART_REGIONPTSNB-1);
	 }

	 // Center of the circle
	 ptaPolygon.setPoint(PART_REGIONPTSNB+1, ptCenter);
      }

      // Create the region
      m_qrRegion[i] = QRegion(ptaPolygon);
		
      dOldAngle += m_kgpPart[i].m_dPourcent; // Calculate total angle for next dir
   }
}


// ===========================================================================================================================
/** Return le region on the graph on the point (x,y) */
int KDrawView::getRegionNumber(QPoint pt)
{
   for (int i=0; i < m_nCurPartNum; i++)
   { 
      if (m_qrRegion[i].contains(pt)) // The mouse is in the current part
      {
	 return i;
      }
   }

   return -1;
}

// ===========================================================================================================================
KTreeView *KDrawView::getTreeView()
{
   return m_app->getTreeView();
}

// ===========================================================================================================================
KGraphSpaceView *KDrawView::getMainView()
{
   return m_app->getMainView();
}

// ===========================================================================================================================
int KDrawView::printGraph(QPrinter *printer)
{
   QPainter paint(printer);
	
   drawGraph(&paint, true);
   return 0;
}
