var documenterSearchIndex = {"docs":
[{"title":"Command-line Options","location":"manual/command-line-options.html#command-line-options","category":"section","text":"","page":"Command-line Options"},{"title":"Command-line Options","location":"manual/command-line-options.html","category":"page","text":"The following is a complete list of command-line switches available when launching julia:","page":"Command-line Options"},{"title":"Command-line Options","location":"manual/command-line-options.html","category":"page","text":"Switch Description\n-v, --version Display version information\n-h, --help Print command-line options (this message).\n--project[={<dir>|@.}] Set <dir> as the home project/environment. The default @. option will search through parent directories until a Project.toml or JuliaProject.toml file is found.\n-J, --sysimage <file> Start up with the given system image file\n-H, --home <dir> Set location of julia executable\n--startup-file={yes|no} Load ~/.julia/config/startup.jl\n--handle-signals={yes|no} Enable or disable Julia's default signal handlers\n--sysimage-native-code={yes|no} Use native code from system image if available\n--compiled-modules={yes|no} Enable or disable incremental precompilation of modules\n-e, --eval <expr> Evaluate <expr>\n-E, --print <expr> Evaluate <expr> and display the result\n-L, --load <file> Load <file> immediately on all processors\n-t, --threads {N|auto} Enable N threads; auto currently sets N to the number of local CPU threads but this might change in the future\n-p, --procs {N|auto} Integer value N launches N additional local worker processes; auto launches as many workers as the number of local CPU threads (logical cores)\n--machine-file <file> Run processes on hosts listed in <file>\n-i Interactive mode; REPL runs and isinteractive() is true\n-q, --quiet Quiet startup: no banner, suppress REPL warnings\n--banner={yes|no|auto} Enable or disable startup banner\n--color={yes|no|auto} Enable or disable color text\n--history-file={yes|no} Load or save history\n--depwarn={yes|no|error} Enable or disable syntax and method deprecation warnings (error turns warnings into errors)\n--warn-overwrite={yes|no} Enable or disable method overwrite warnings\n-C, --cpu-target <target> Limit usage of CPU features up to <target>; set to help to see the available options\n-O, --optimize={0,1,2,3} Set the optimization level (default level is 2 if unspecified or 3 if used without a level)\n-g, -g <level> Enable / Set the level of debug info generation (default level is 1 if unspecified or 2 if used without a level)\n--inline={yes|no} Control whether inlining is permitted, including overriding @inline declarations\n--check-bounds={yes|no|auto} Emit bounds checks always, never, or respect @inbounds declarations\n--math-mode={ieee,fast} Disallow or enable unsafe floating point optimizations (overrides @fastmath declaration)\n--code-coverage={none|user|all} Count executions of source lines\n--code-coverage equivalent to --code-coverage=user\n--track-allocation={none|user|all} Count bytes allocated by each source line\n--track-allocation equivalent to --track-allocation=user","page":"Command-line Options"},{"title":"Command-line Options","location":"manual/command-line-options.html","category":"page","text":"compat: Julia 1.1\nIn Julia 1.0, the default --project=@. option did not search up from the root directory of a Git repository for the Project.toml file. From Julia 1.1 forward, it does.","page":"Command-line Options"},{"title":"Embedding Julia","location":"manual/embedding.html#Embedding-Julia","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"As we have seen in Calling C and Fortran Code, Julia has a simple and efficient way to call functions written in C. But there are situations where the opposite is needed: calling Julia function from C code. This can be used to integrate Julia code into a larger C/C++ project, without the need to rewrite everything in C/C++. Julia has a C API to make this possible. As almost all programming languages have some way to call C functions, the Julia C API can also be used to build further language bridges (e.g. calling Julia from Python or C#).","page":"Embedding Julia"},{"title":"High-Level Embedding","location":"manual/embedding.html#High-Level-Embedding","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Note: This section covers embedding Julia code in C on Unix-like operating systems. For doing this on Windows, please see the section following this.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"We start with a simple C program that initializes Julia and calls some Julia code:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"#include <julia.h>\nJULIA_DEFINE_FAST_TLS() // only define this once, in an executable (not in a shared library) if you want fast code.\n\nint main(int argc, char *argv[])\n{\n    /* required: setup the Julia context */\n    jl_init();\n\n    /* run Julia commands */\n    jl_eval_string(\"print(sqrt(2.0))\");\n\n    /* strongly recommended: notify Julia that the\n         program is about to terminate. this allows\n         Julia time to cleanup pending write requests\n         and run all finalizers\n    */\n    jl_atexit_hook(0);\n    return 0;\n}","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"In order to build this program you have to put the path to the Julia header into the include path and link against libjulia. For instance, when Julia is installed to $JULIA_DIR, one can compile the above test program test.c with gcc using:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"gcc -o test -fPIC -I$JULIA_DIR/include/julia -L$JULIA_DIR/lib -Wl,-rpath,$JULIA_DIR/lib test.c -ljulia","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Alternatively, look at the embedding.c program in the Julia source tree in the test/embedding/ folder. The file cli/loader_exe.c program is another simple example of how to set jl_options options while linking against libjulia.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The first thing that has to be done before calling any other Julia C function is to initialize Julia. This is done by calling jl_init, which tries to automatically determine Julia's install location. If you need to specify a custom location, or specify which system image to load, use jl_init_with_image instead.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The second statement in the test program evaluates a Julia statement using a call to jl_eval_string.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Before the program terminates, it is strongly recommended to call jl_atexit_hook.  The above example program calls this before returning from main.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"note: Note\nCurrently, dynamically linking with the libjulia shared library requires passing the RTLD_GLOBAL option. In Python, this looks like:>>> julia=CDLL('./libjulia.dylib',RTLD_GLOBAL)\n>>> julia.jl_init.argtypes = []\n>>> julia.jl_init()\n250593296","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"note: Note\nIf the julia program needs to access symbols from the main executable, it may be necessary to add -Wl,--export-dynamic linker flag at compile time on Linux in addition to the ones generated by julia-config.jl described below. This is not necessary when compiling a shared library.","page":"Embedding Julia"},{"title":"Using julia-config to automatically determine build parameters","location":"manual/embedding.html#Using-julia-config-to-automatically-determine-build-parameters","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The script julia-config.jl was created to aid in determining what build parameters are required by a program that uses embedded Julia.  This script uses the build parameters and system configuration of the particular Julia distribution it is invoked by to export the necessary compiler flags for an embedding program to interact with that distribution.  This script is located in the Julia shared data directory.","page":"Embedding Julia"},{"title":"Example","location":"manual/embedding.html#Example","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"#include <julia.h>\n\nint main(int argc, char *argv[])\n{\n    jl_init();\n    (void)jl_eval_string(\"println(sqrt(2.0))\");\n    jl_atexit_hook(0);\n    return 0;\n}","page":"Embedding Julia"},{"title":"On the command line","location":"manual/embedding.html#On-the-command-line","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"A simple use of this script is from the command line.  Assuming that julia-config.jl is located in /usr/local/julia/share/julia, it can be invoked on the command line directly and takes any combination of 3 flags:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"/usr/local/julia/share/julia/julia-config.jl\nUsage: julia-config [--cflags|--ldflags|--ldlibs]","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If the above example source is saved in the file embed_example.c, then the following command will compile it into a running program on Linux and Windows (MSYS2 environment), or if on OS/X, then substitute clang for gcc.:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"/usr/local/julia/share/julia/julia-config.jl --cflags --ldflags --ldlibs | xargs gcc embed_example.c","page":"Embedding Julia"},{"title":"Use in Makefiles","location":"manual/embedding.html#Use-in-Makefiles","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"But in general, embedding projects will be more complicated than the above, and so the following allows general makefile support as well – assuming GNU make because of the use of the shell macro expansions.  Additionally, though many times julia-config.jl may be found in the directory /usr/local, this is not necessarily the case, but Julia can be used to locate julia-config.jl too, and the makefile can be used to take advantage of that.  The above example is extended to use a Makefile:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"JL_SHARE = $(shell julia -e 'print(joinpath(Sys.BINDIR, Base.DATAROOTDIR, \"julia\"))')\nCFLAGS   += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nCXXFLAGS += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nLDFLAGS  += $(shell $(JL_SHARE)/julia-config.jl --ldflags)\nLDLIBS   += $(shell $(JL_SHARE)/julia-config.jl --ldlibs)\n\nall: embed_example","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Now the build command is simply make.","page":"Embedding Julia"},{"title":"High-Level Embedding on Windows with Visual Studio","location":"manual/embedding.html#High-Level-Embedding-on-Windows-with-Visual-Studio","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If the JULIA_DIR environment variable hasn't been setup, add it using the System panel before starting Visual Studio. The bin folder under JULIA_DIR should be on the system PATH.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"We start by opening Visual Studio and creating a new Console Application project. To the 'stdafx.h' header file, add the following lines at the end:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"#include <julia.h>","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Then, replace the main() function in the project with this code:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"int main(int argc, char *argv[])\n{\n    /* required: setup the Julia context */\n    jl_init();\n\n    /* run Julia commands */\n    jl_eval_string(\"print(sqrt(2.0))\");\n\n    /* strongly recommended: notify Julia that the\n         program is about to terminate. this allows\n         Julia time to cleanup pending write requests\n         and run all finalizers\n    */\n    jl_atexit_hook(0);\n    return 0;\n}","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The next step is to set up the project to find the Julia include files and the libraries. It's important to know whether the Julia installation is 32- or 64-bits. Remove any platform configuration that doesn't correspond to the Julia installation before proceeding.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Using the project Properties dialog, go to C/C++ | General and add $(JULIA_DIR)\\include\\julia\\ to the Additional Include Directories property. Then, go to the Linker | General section and add $(JULIA_DIR)\\lib to the Additional Library Directories property. Finally, under Linker | Input, add libjulia.dll.a;libopenlibm.dll.a; to the list of libraries.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"At this point, the project should build and run.","page":"Embedding Julia"},{"title":"Converting Types","location":"manual/embedding.html#Converting-Types","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Real applications will not just need to execute expressions, but also return their values to the host program. jl_eval_string returns a jl_value_t*, which is a pointer to a heap-allocated Julia object. Storing simple data types like Float64 in this way is called boxing, and extracting the stored primitive data is called unboxing. Our improved sample program that calculates the square root of 2 in Julia and reads back the result in C looks as follows:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\n\nif (jl_typeis(ret, jl_float64_type)) {\n    double ret_unboxed = jl_unbox_float64(ret);\n    printf(\"sqrt(2.0) in C: %e \\n\", ret_unboxed);\n}\nelse {\n    printf(\"ERROR: unexpected return type from sqrt(::Float64)\\n\");\n}","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"In order to check whether ret is of a specific Julia type, we can use the jl_isa, jl_typeis, or jl_is_... functions. By typing typeof(sqrt(2.0)) into the Julia shell we can see that the return type is Float64 (double in C). To convert the boxed Julia value into a C double the jl_unbox_float64 function is used in the above code snippet.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Corresponding jl_box_... functions are used to convert the other way:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *a = jl_box_float64(3.0);\njl_value_t *b = jl_box_float32(3.0f);\njl_value_t *c = jl_box_int32(3);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"As we will see next, boxing is required to call Julia functions with specific arguments.","page":"Embedding Julia"},{"title":"Calling Julia Functions","location":"manual/embedding.html#Calling-Julia-Functions","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"While jl_eval_string allows C to obtain the result of a Julia expression, it does not allow passing arguments computed in C to Julia. For this you will need to invoke Julia functions directly, using jl_call:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_function_t *func = jl_get_function(jl_base_module, \"sqrt\");\njl_value_t *argument = jl_box_float64(2.0);\njl_value_t *ret = jl_call1(func, argument);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"In the first step, a handle to the Julia function sqrt is retrieved by calling jl_get_function. The first argument passed to jl_get_function is a pointer to the Base module in which sqrt is defined. Then, the double value is boxed using jl_box_float64. Finally, in the last step, the function is called using jl_call1. jl_call0, jl_call2, and jl_call3 functions also exist, to conveniently handle different numbers of arguments. To pass more arguments, use jl_call:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *jl_call(jl_function_t *f, jl_value_t **args, int32_t nargs)","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Its second argument args is an array of jl_value_t* arguments and nargs is the number of arguments.","page":"Embedding Julia"},{"title":"Memory Management","location":"manual/embedding.html#Memory-Management","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"As we have seen, Julia objects are represented in C as pointers. This raises the question of who is responsible for freeing these objects.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Typically, Julia objects are freed by a garbage collector (GC), but the GC does not automatically know that we are holding a reference to a Julia value from C. This means the GC can free objects out from under you, rendering pointers invalid.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The GC can only run when Julia objects are allocated. Calls like jl_box_float64 perform allocation, and allocation might also happen at any point in running Julia code. However, it is generally safe to use pointers in between jl_... calls. But in order to make sure that values can survive jl_... calls, we have to tell Julia that we still hold a reference to Julia root values, a process called \"GC rooting\". Rooting a value will ensure that the garbage collector does not accidentally identify this value as unused and free the memory backing that value. This can be done using the JL_GC_PUSH macros:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\nJL_GC_PUSH1(&ret);\n// Do something with ret\nJL_GC_POP();","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The JL_GC_POP call releases the references established by the previous JL_GC_PUSH. Note that JL_GC_PUSH stores references on the C stack, so it must be exactly paired with a JL_GC_POP before the scope is exited. That is, before the function returns, or control flow otherwise leaves the block in which the JL_GC_PUSH was invoked.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Several Julia values can be pushed at once using the JL_GC_PUSH2 , JL_GC_PUSH3 , JL_GC_PUSH4 , JL_GC_PUSH5 , and JL_GC_PUSH6 macros. To push an array of Julia values one can use the JL_GC_PUSHARGS macro, which can be used as follows:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t **args;\nJL_GC_PUSHARGS(args, 2); // args can now hold 2 `jl_value_t*` objects\nargs[0] = some_value;\nargs[1] = some_other_value;\n// Do something with args (e.g. call jl_... functions)\nJL_GC_POP();","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Each scope must have only one call to JL_GC_PUSH*. Hence, if all variables cannot be pushed once by a single call to JL_GC_PUSH*, or if there are more than 6 variables to be pushed and using an array of arguments is not an option, then one can use inner blocks:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *ret1 = jl_eval_string(\"sqrt(2.0)\");\nJL_GC_PUSH1(&ret1);\njl_value_t *ret2 = 0;\n{\n    jl_function_t *func = jl_get_function(jl_base_module, \"exp\");\n    ret2 = jl_call1(func, ret1);\n    JL_GC_PUSH1(&ret2);\n    // Do something with ret2.\n    JL_GC_POP();    // This pops ret2.\n}\nJL_GC_POP();    // This pops ret1.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If it is required to hold the pointer to a variable between functions (or block scopes), then it is not possible to use JL_GC_PUSH*. In this case, it is necessary to create and keep a reference to the variable in the Julia global scope. One simple way to accomplish this is to use a global IdDict that will hold the references, avoiding deallocation by the GC. However, this method will only work properly with mutable types.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"// This functions shall be executed only once, during the initialization.\njl_value_t* refs = jl_eval_string(\"refs = IdDict()\");\njl_function_t* setindex = jl_get_function(jl_base_module, \"setindex!\");\n\n...\n\n// `var` is the variable we want to protect between function calls.\njl_value_t* var = 0;\n\n...\n\n// `var` is a `Vector{Float64}`, which is mutable.\nvar = jl_eval_string(\"[sqrt(2.0); sqrt(4.0); sqrt(6.0)]\");\n\n// To protect `var`, add its reference to `refs`.\njl_call3(setindex, refs, var, var);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If the variable is immutable, then it needs to be wrapped in an equivalent mutable container or, preferably, in a RefValue{Any} before it is pushed to IdDict. In this approach, the container has to be created or filled in via C code using, for example, the function jl_new_struct. If the container is created by jl_call*, then you will need to reload the pointer to be used in C code.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"// This functions shall be executed only once, during the initialization.\njl_value_t* refs = jl_eval_string(\"refs = IdDict()\");\njl_function_t* setindex = jl_get_function(jl_base_module, \"setindex!\");\njl_datatype_t* reft = (jl_datatype_t*)jl_eval_string(\"Base.RefValue{Any}\");\n\n...\n\n// `var` is the variable we want to protect between function calls.\njl_value_t* var = 0;\n\n...\n\n// `var` is a `Float64`, which is immutable.\nvar = jl_eval_string(\"sqrt(2.0)\");\n\n// Protect `var` until we add its reference to `refs`.\nJL_GC_PUSH1(&var);\n\n// Wrap `var` in `RefValue{Any}` and push to `refs` to protect it.\njl_value_t* rvar = jl_new_struct(reft, var);\nJL_GC_POP();\n\njl_call3(setindex, refs, rvar, rvar);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The GC can be allowed to deallocate a variable by removing the reference to it from refs using the function delete!, provided that no other reference to the variable is kept anywhere:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_function_t* delete = jl_get_function(jl_base_module, \"delete!\");\njl_call2(delete, refs, rvar);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"As an alternative for very simple cases, it is possible to just create a global container of type Vector{Any} and fetch the elements from that when necessary, or even to create one global variable per pointer using","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_set_global(jl_main_module, jl_symbol(\"var\"), var);","page":"Embedding Julia"},{"title":"Updating fields of GC-managed objects","location":"manual/embedding.html#Updating-fields-of-GC-managed-objects","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The garbage collector operates under the assumption that it is aware of every old-generation object pointing to a young-generation one. Any time a pointer is updated breaking that assumption, it must be signaled to the collector with the jl_gc_wb (write barrier) function like so:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t *parent = some_old_value, *child = some_young_value;\n((some_specific_type*)parent)->field = child;\njl_gc_wb(parent, child);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"It is in general impossible to predict which values will be old at runtime, so the write barrier must be inserted after all explicit stores. One notable exception is if the parent object was just allocated and garbage collection was not run since then. Remember that most jl_... functions can sometimes invoke garbage collection.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The write barrier is also necessary for arrays of pointers when updating their data directly. For example:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_array_t *some_array = ...; // e.g. a Vector{Any}\nvoid **data = (void**)jl_array_data(some_array);\njl_value_t *some_value = ...;\ndata[0] = some_value;\njl_gc_wb(some_array, some_value);","page":"Embedding Julia"},{"title":"Manipulating the Garbage Collector","location":"manual/embedding.html#Manipulating-the-Garbage-Collector","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"There are some functions to control the GC. In normal use cases, these should not be necessary.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Function Description\njl_gc_collect() Force a GC run\njl_gc_enable(0) Disable the GC, return previous state as int\njl_gc_enable(1) Enable the GC,  return previous state as int\njl_gc_is_enabled() Return current state as int","page":"Embedding Julia"},{"title":"Working with Arrays","location":"manual/embedding.html#Working-with-Arrays","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Julia and C can share array data without copying. The next example will show how this works.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Julia arrays are represented in C by the datatype jl_array_t*. Basically, jl_array_t is a struct that contains:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Information about the datatype\nA pointer to the data block\nInformation about the sizes of the array","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"To keep things simple, we start with a 1D array. Creating an array containing Float64 elements of length 10 is done by:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_value_t* array_type = jl_apply_array_type((jl_value_t*)jl_float64_type, 1);\njl_array_t* x          = jl_alloc_array_1d(array_type, 10);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Alternatively, if you have already allocated the array you can generate a thin wrapper around its data:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"double *existingArray = (double*)malloc(sizeof(double)*10);\njl_array_t *x = jl_ptr_to_array_1d(array_type, existingArray, 10, 0);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"The last argument is a boolean indicating whether Julia should take ownership of the data. If this argument is non-zero, the GC will call free on the data pointer when the array is no longer referenced.","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"In order to access the data of x, we can use jl_array_data:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"double *xData = (double*)jl_array_data(x);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Now we can fill the array:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"for(size_t i=0; i<jl_array_len(x); i++)\n    xData[i] = i;","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Now let us call a Julia function that performs an in-place operation on x:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_function_t *func = jl_get_function(jl_base_module, \"reverse!\");\njl_call1(func, (jl_value_t*)x);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"By printing the array, one can verify that the elements of x are now reversed.","page":"Embedding Julia"},{"title":"Accessing Returned Arrays","location":"manual/embedding.html#Accessing-Returned-Arrays","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If a Julia function returns an array, the return value of jl_eval_string and jl_call can be cast to a jl_array_t*:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_function_t *func  = jl_get_function(jl_base_module, \"reverse\");\njl_array_t *y = (jl_array_t*)jl_call1(func, (jl_value_t*)x);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Now the content of y can be accessed as before using jl_array_data. As always, be sure to keep a reference to the array while it is in use.","page":"Embedding Julia"},{"title":"Multidimensional Arrays","location":"manual/embedding.html#Multidimensional-Arrays","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Julia's multidimensional arrays are stored in memory in column-major order. Here is some code that creates a 2D array and accesses its properties:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"// Create 2D array of float64 type\njl_value_t *array_type = jl_apply_array_type(jl_float64_type, 2);\njl_array_t *x  = jl_alloc_array_2d(array_type, 10, 5);\n\n// Get array pointer\ndouble *p = (double*)jl_array_data(x);\n// Get number of dimensions\nint ndims = jl_array_ndims(x);\n// Get the size of the i-th dim\nsize_t size0 = jl_array_dim(x,0);\nsize_t size1 = jl_array_dim(x,1);\n\n// Fill array with data\nfor(size_t i=0; i<size1; i++)\n    for(size_t j=0; j<size0; j++)\n        p[j + size0*i] = i + j;","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Notice that while Julia arrays use 1-based indexing, the C API uses 0-based indexing (for example in calling jl_array_dim) in order to read as idiomatic C code.","page":"Embedding Julia"},{"title":"Exceptions","location":"manual/embedding.html#Exceptions","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"Julia code can throw exceptions. For example, consider:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_eval_string(\"this_function_does_not_exist()\");","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"This call will appear to do nothing. However, it is possible to check whether an exception was thrown:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"if (jl_exception_occurred())\n    printf(\"%s \\n\", jl_typeof_str(jl_exception_occurred()));","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"If you are using the Julia C API from a language that supports exceptions (e.g. Python, C#, C++), it makes sense to wrap each call into libjulia with a function that checks whether an exception was thrown, and then rethrows the exception in the host language.","page":"Embedding Julia"},{"title":"Throwing Julia Exceptions","location":"manual/embedding.html#Throwing-Julia-Exceptions","category":"section","text":"","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"When writing Julia callable functions, it might be necessary to validate arguments and throw exceptions to indicate errors. A typical type check looks like:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"if (!jl_typeis(val, jl_float64_type)) {\n    jl_type_error(function_name, (jl_value_t*)jl_float64_type, val);\n}","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"General exceptions can be raised using the functions:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"void jl_error(const char *str);\nvoid jl_errorf(const char *fmt, ...);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_error takes a C string, and jl_errorf is called like printf:","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"jl_errorf(\"argument x = %d is too large\", x);","page":"Embedding Julia"},{"title":"Embedding Julia","location":"manual/embedding.html","category":"page","text":"where in this example x is assumed to be an integer.","page":"Embedding Julia"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions.","page":"Mathematical Operations and Elementary Functions"},{"title":"Arithmetic Operators","location":"manual/mathematical-operations.html#Arithmetic-Operators","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following arithmetic operators are supported on all primitive numeric types:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x,y)","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"A numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x+y), is treated as a multiplication, except with higher precedence than other binary operations.  See Numeric Literal Coefficients for details.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some simple examples using arithmetic operators:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"When used in multiplication, false acts as a strong zero:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> NaN * false\n0.0\n\njulia> false * Inf\n0.0","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"This is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation.","page":"Mathematical Operations and Elementary Functions"},{"title":"Boolean Operators","location":"manual/mathematical-operations.html#Boolean-Operators","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following Boolean operators are supported on Bool types:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Negation changes true to false and vice versa. The short-circuiting opeations are explained on the linked page.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note that Bool is an integer type and all the usual promotion rules and numeric operators are also defined on it.","page":"Mathematical Operations and Elementary Functions"},{"title":"Bitwise Operators","location":"manual/mathematical-operations.html#Bitwise-Operators","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following bitwise operators are supported on all primitive integer types:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Expression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some examples with bitwise operators:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84","page":"Mathematical Operations and Elementary Functions"},{"title":"Updating operators","location":"manual/mathematical-operations.html#Updating-operators","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The updating versions of all the binary arithmetic and bitwise operators are:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"note: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64","page":"Mathematical Operations and Elementary Functions"},{"title":"Vectorized \"dot\" operators","location":"manual/mathematical-operations.html#man-dot-operators","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1,2,3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" a (non-square) array, but [1,2,3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3].  Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> [1,2,3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"More specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Furthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation).","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A,B) = kron(A,B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A,B] .⊗ [C,D] will compute [A⊗C, B⊗D] with no additional coding.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Combining dot operators with numeric literals can be ambiguous. For example, it is not clear whether 1.+x means 1. + x or 1 .+ x. Therefore this syntax is disallowed, and spaces must be used around the operator in such cases.","page":"Mathematical Operations and Elementary Functions"},{"title":"Numeric Comparisons","location":"manual/mathematical-operations.html#Numeric-Comparisons","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Standard comparison operations are defined for all the primitive numeric types:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Operator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Here are some simple examples:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Integers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Finite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The last point is potentially surprising and thus worth noting:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"and can cause headaches when working with arrays:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> [1 NaN] == [1 NaN]\nfalse","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"isequal considers NaNs equal to each other:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"isequal can also be used to distinguish signed zeros:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Mixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For other types, isequal defaults to calling ==, so if you want to define equality for your own types then you only need to add a == method.  If you define your own equality function, you should probably define a corresponding hash method to ensure that isequal(x,y) implies hash(x) == hash(y).","page":"Mathematical Operations and Elementary Functions"},{"title":"Chaining comparisons","location":"manual/mathematical-operations.html#Chaining-comparisons","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Chaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note the evaluation behavior of chained comparisons:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation).","page":"Mathematical Operations and Elementary Functions"},{"title":"Elementary Functions","location":"manual/mathematical-operations.html#Elementary-Functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Moreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A.","page":"Mathematical Operations and Elementary Functions"},{"title":"Operator Precedence and Associativity","location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Category Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - √ Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a)).","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scm. Note that some of the operators there are not defined in the Base module but may be given definitions by standard libraries, packages or user code.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"You can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"A symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Note that symbols such as :sin return precedence 0. This value represents invalid operators and not operators of lowest precedence. Similarly, such operators are assigned associativity :none.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Numeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Juxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y).","page":"Mathematical Operations and Elementary Functions"},{"title":"Numerical Conversions","location":"manual/mathematical-operations.html#Numerical-Conversions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The notation T(x) or convert(T,x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x)).","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"The following examples show the different forms.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"julia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: trunc(Int8, 128.0)\nStacktrace:\n[...]","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"See Conversion and Promotion for how to define your own conversions and promotions.","page":"Mathematical Operations and Elementary Functions"},{"title":"Rounding functions","location":"manual/mathematical-operations.html#Rounding-functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T","page":"Mathematical Operations and Elementary Functions"},{"title":"Division functions","location":"manual/mathematical-operations.html#Division-functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\ndiv(x,y), x÷y truncated division; quotient rounded towards zero\nfld(x,y) floored division; quotient rounded towards -Inf\ncld(x,y) ceiling division; quotient rounded towards +Inf\nrem(x,y) remainder; satisfies x == div(x,y)*y + rem(x,y); sign matches x\nmod(x,y) modulus; satisfies x == fld(x,y)*y + mod(x,y); sign matches y\nmod1(x,y) mod with offset 1; returns r∈(0,y] for y>0 or r∈[y,0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x,y) returns (div(x,y),rem(x,y))\nfldmod(x,y) returns (fld(x,y),mod(x,y))\ngcd(x,y...) greatest positive common divisor of x, y,...\nlcm(x,y...) least positive common multiple of x, y,...","page":"Mathematical Operations and Elementary Functions"},{"title":"Sign and absolute value functions","location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x,y) a value with the magnitude of x and the sign of y\nflipsign(x,y) a value with the magnitude of x and the sign of x*y","page":"Mathematical Operations and Elementary Functions"},{"title":"Powers, logs and roots","location":"manual/mathematical-operations.html#Powers,-logs-and-roots","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nhypot(x,y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x)-1 for x near zero\nldexp(x,n) x*2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b,x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1+x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"For an overview of why functions like hypot, expm1, and log1p are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot.","page":"Mathematical Operations and Elementary Functions"},{"title":"Trigonometric and hyperbolic functions","location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"All the standard trigonometric and hyperbolic functions are also defined:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"sin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"These are all single-argument functions, with atan also accepting two arguments corresponding to a traditional atan2 function.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Additionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi*x) and cos(pi*x) respectively.","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"In order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"sind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd","page":"Mathematical Operations and Elementary Functions"},{"title":"Special functions","location":"manual/mathematical-operations.html#Special-functions","category":"section","text":"","page":"Mathematical Operations and Elementary Functions"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html","category":"page","text":"Many other special mathematical functions are provided by the package SpecialFunctions.jl.","page":"Mathematical Operations and Elementary Functions"},{"title":"Parallel Computing","location":"manual/parallel-computing.html#Parallel-Computing","category":"section","text":"","page":"Parallel Computing"},{"title":"Parallel Computing","location":"manual/parallel-computing.html","category":"page","text":"Julia supports these four categories of concurrent and parallel programming:","page":"Parallel Computing"},{"title":"Parallel Computing","location":"manual/parallel-computing.html","category":"page","text":"Asynchronous \"tasks\", or coroutines:\nJulia Tasks allow suspending and resuming computations  for I/O, event handling, producer-consumer processes, and similar patterns.  Tasks can synchronize through operations like wait and fetch, and  communicate via Channels. While strictly not parallel computing by themselves,  Julia lets you schedule Tasks on several threads.\nMulti-threading:\nJulia's multi-threading provides the ability to schedule Tasks  simultaneously on more than one thread or CPU core, sharing memory. This is usually the easiest way  to get parallelism on one's PC or on a single large multi-core server. Julia's multi-threading  is composable. When one multi-threaded function calls another multi-threaded function, Julia  will schedule all the threads globally on available resources, without oversubscribing.\nDistributed computing:\nDistributed computing runs multiple Julia processes with separate memory spaces. These can be on the same  computer or multiple computers. The Distributed standard library provides the capability for remote execution  of a Julia function. With this basic building block, it is possible to build many different kinds of  distributed computing abstractions. Packages like DistributedArrays.jl  are an example of such an abstraction. On the other hand, packages like MPI.jl and  Elemental.jl provide access to the existing MPI ecosystem of libraries.\nGPU computing:\nThe Julia GPU compiler provides the ability to run Julia code natively on GPUs. There  is a rich ecosystem of Julia packages that target GPUs. The JuliaGPU.org  website provides a list of capabilities, supported GPUs, related packages and documentation.","page":"Parallel Computing"},{"title":"Multi-Threading","location":"manual/multi-threading.html#man-multithreading","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Visit this blog post for a presentation of Julia multi-threading features.","page":"Multi-Threading"},{"title":"Starting Julia with multiple threads","location":"manual/multi-threading.html#Starting-Julia-with-multiple-threads","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"By default, Julia starts up with a single thread of execution. This can be verified by using the command Threads.nthreads():","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> Threads.nthreads()\n1","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"The number of execution threads is controlled either by using the -t/--threads command line argument or by using the JULIA_NUM_THREADS environment variable. When both are specified, then -t/--threads takes precedence.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"compat: Julia 1.5\nThe -t/--threads command line argument requires at least Julia 1.5. In older versions you must use the environment variable instead.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Lets start Julia with 4 threads:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"$ julia --threads 4","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Let's verify there are 4 threads at our disposal.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> Threads.nthreads()\n4","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"But we are currently on the master thread. To check, we use the function Threads.threadid","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> Threads.threadid()\n1","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"note: Note\nIf you prefer to use the environment variable you can set it as follows in Bash (Linux/macOS):export JULIA_NUM_THREADS=4C shell on Linux/macOS, CMD on Windows:set JULIA_NUM_THREADS=4Powershell on Windows:$env:JULIA_NUM_THREADS=4Note that this must be done before starting Julia.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"note: Note\nThe number of threads specified with -t/--threads is propagated to worker processes that are spawned using the -p/--procs or --machine-file command line options. For example, julia -p2 -t2 spawns 1 main process with 2 worker processes, and all three processes have 2 threads enabled. For more fine grained control over worker threads use addprocs and pass -t/--threads as exeflags.","page":"Multi-Threading"},{"title":"Data-race freedom","location":"manual/multi-threading.html#Data-race-freedom","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"You are entirely responsible for ensuring that your program is data-race free, and nothing promised here can be assumed if you do not observe that requirement. The observed results may be highly unintuitive.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"The best way to ensure this is to acquire a lock around any access to data that can be observed from multiple threads. For example, in most cases you should use the following code pattern:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> lock(lk) do\n           use(a)\n       end\n\njulia> begin\n           lock(lk)\n           try\n               use(a)\n           finally\n               unlock(lk)\n           end\n       end","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"where lk is a lock (e.g. ReentrantLock()) and a data.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Additionally, Julia is not memory safe in the presence of a data race. Be very careful about reading any data if another thread might write to it! Instead, always use the lock pattern above when changing data (such as assigning to a global or closure variable) accessed by other threads.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Thread 1:\nglobal b = false\nglobal a = rand()\nglobal b = true\n\nThread 2:\nwhile !b; end\nbad_read1(a) # it is NOT safe to access `a` here!\n\nThread 3:\nwhile !@isdefined(a); end\nbad_read2(a) # it is NOT safe to access `a` here","page":"Multi-Threading"},{"title":"The @threads Macro","location":"manual/multi-threading.html#The-@threads-Macro","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Let's work a simple example using our native threads. Let us create an array of zeros:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> a = zeros(10)\n10-element Vector{Float64}:\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Let us operate on this array simultaneously using 4 threads. We'll have each thread write its thread ID into each location.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Julia supports parallel loops using the Threads.@threads macro. This macro is affixed in front of a for loop to indicate to Julia that the loop is a multi-threaded region:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> Threads.@threads for i = 1:10\n           a[i] = Threads.threadid()\n       end","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"The iteration space is split among the threads, after which each thread writes its thread ID to its assigned locations:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> a\n10-element Array{Float64,1}:\n 1.0\n 1.0\n 1.0\n 2.0\n 2.0\n 2.0\n 3.0\n 3.0\n 4.0\n 4.0","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Note that Threads.@threads does not have an optional reduction parameter like @distributed.","page":"Multi-Threading"},{"title":"Atomic Operations","location":"manual/multi-threading.html#Atomic-Operations","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Julia supports accessing and modifying values atomically, that is, in a thread-safe way to avoid race conditions. A value (which must be of a primitive type) can be wrapped as Threads.Atomic to indicate it must be accessed in this way. Here we can see an example:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> i = Threads.Atomic{Int}(0);\n\njulia> ids = zeros(4);\n\njulia> old_is = zeros(4);\n\njulia> Threads.@threads for id in 1:4\n           old_is[id] = Threads.atomic_add!(i, id)\n           ids[id] = id\n       end\n\njulia> old_is\n4-element Array{Float64,1}:\n 0.0\n 1.0\n 7.0\n 3.0\n\njulia> ids\n4-element Array{Float64,1}:\n 1.0\n 2.0\n 3.0\n 4.0","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Had we tried to do the addition without the atomic tag, we might have gotten the wrong answer due to a race condition. An example of what would happen if we didn't avoid the race:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"julia> using Base.Threads\n\njulia> nthreads()\n4\n\njulia> acc = Ref(0)\nBase.RefValue{Int64}(0)\n\njulia> @threads for i in 1:1000\n          acc[] += 1\n       end\n\njulia> acc[]\n926\n\njulia> acc = Atomic{Int64}(0)\nAtomic{Int64}(0)\n\njulia> @threads for i in 1:1000\n          atomic_add!(acc, 1)\n       end\n\njulia> acc[]\n1000","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"note: Note\nNot all primitive types can be wrapped in an Atomic tag. Supported types are Int8, Int16, Int32, Int64, Int128, UInt8, UInt16, UInt32, UInt64, UInt128, Float16, Float32, and Float64. Additionally, Int128 and UInt128 are not supported on AAarch32 and ppc64le.","page":"Multi-Threading"},{"title":"Side effects and mutable function arguments","location":"manual/multi-threading.html#Side-effects-and-mutable-function-arguments","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"When using multi-threading we have to be careful when using functions that are not pure as we might get a wrong answer. For instance functions that have a name ending with ! by convention modify their arguments and thus are not pure.","page":"Multi-Threading"},{"title":"@threadcall","location":"manual/multi-threading.html#@threadcall","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"External libraries, such as those called via ccall, pose a problem for Julia's task-based I/O mechanism. If a C library performs a blocking operation, that prevents the Julia scheduler from executing any other tasks until the call returns. (Exceptions are calls into custom C code that call back into Julia, which may then yield, or C code that calls jl_yield(), the C equivalent of yield.)","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"The @threadcall macro provides a way to avoid stalling execution in such a scenario. It schedules a C function for execution in a separate thread. A threadpool with a default size of 4 is used for this. The size of the threadpool is controlled via environment variable UV_THREADPOOL_SIZE. While waiting for a free thread, and during function execution once a thread is available, the requesting task (on the main Julia event loop) yields to other tasks. Note that @threadcall does not return until the execution is complete. From a user point of view, it is therefore a blocking call like other Julia APIs.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"It is very important that the called function does not call back into Julia, as it will segfault.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"@threadcall may be removed/changed in future versions of Julia.","page":"Multi-Threading"},{"title":"Caveats","location":"manual/multi-threading.html#Caveats","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"At this time, most operations in the Julia runtime and standard libraries can be used in a thread-safe manner, if the user code is data-race free. However, in some areas work on stabilizing thread support is ongoing. Multi-threaded programming has many inherent difficulties, and if a program using threads exhibits unusual or undesirable behavior (e.g. crashes or mysterious results), thread interactions should typically be suspected first.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"There are a few specific limitations and warnings to be aware of when using threads in Julia:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Base collection types require manual locking if used simultaneously by multiple threads where at least one thread modifies the collection (common examples include push! on arrays, or inserting items into a Dict).\nAfter a task starts running on a certain thread (e.g. via @spawn), it will always be restarted on the same thread after blocking. In the future this limitation will be removed, and tasks will migrate between threads.\n@threads currently uses a static schedule, using all threads and assigning equal iteration counts to each. In the future the default schedule is likely to change to be dynamic.\nThe schedule used by @spawn is nondeterministic and should not be relied on.\nCompute-bound, non-memory-allocating tasks can prevent garbage collection from running in other threads that are allocating memory. In these cases it may be necessary to insert a manual call to GC.safepoint() to allow GC to run. This limitation will be removed in the future.\nAvoid running top-level operations, e.g. include, or eval of type, method, and module definitions in parallel.\nBe aware that finalizers registered by a library may break if threads are enabled. This may require some transitional work across the ecosystem before threading can be widely adopted with confidence. See the next section for further details.","page":"Multi-Threading"},{"title":"Safe use of Finalizers","location":"manual/multi-threading.html#Safe-use-of-Finalizers","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"Because finalizers can interrupt any code, they must be very careful in how they interact with any global state. Unfortunately, the main reason that finalizers are used is to update global state (a pure function is generally rather pointless as a finalizer). This leads us to a bit of a conundrum. There are a few approaches to dealing with this problem:","page":"Multi-Threading"},{"title":"Multi-Threading","location":"manual/multi-threading.html","category":"page","text":"When single-threaded, code could call the internal jl_gc_enable_finalizers C function to prevent finalizers from being scheduled inside a critical region. Internally, this is used inside some functions (such as our C locks) to prevent recursion when doing certain operations (incremental package loading, codegen, etc.). The combination of a lock and this flag can be used to make finalizers safe.\nA second strategy, employed by Base in a couple places, is to explicitly delay a finalizer until it may be able to acquire its lock non-recursively. The following example demonstrates how this strategy could be applied to Distributed.finalize_ref:\nfunction finalize_ref(r::AbstractRemoteRef)\n    if r.where > 0 # Check if the finalizer is already run\n        if islocked(client_refs) || !trylock(client_refs)\n            # delay finalizer for later if we aren't free to acquire the lock\n            finalizer(finalize_ref, r)\n            return nothing\n        end\n        try # `lock` should always be followed by `try`\n            if r.where > 0 # Must check again here\n                # Do actual cleanup here\n                r.where = 0\n            end\n        finally\n            unlock(client_refs)\n        end\n    end\n    nothing\nend\nA related third strategy is to use a yield-free queue. We don't currently have a lock-free queue implemented in Base, but Base.InvasiveLinkedListSynchronized{T} is suitable. This can frequently be a good strategy to use for code with event loops. For example, this strategy is employed by Gtk.jl to manage lifetime ref-counting. In this approach, we don't do any explicit work inside the finalizer, and instead add it to a queue to run at a safer time. In fact, Julia's task scheduler already uses this, so defining the finalizer as x -> @spawn do_cleanup(x) is one example of this approach. Note however that this doesn't control which thread do_cleanup runs on, so do_cleanup would still need to acquire a lock. That doesn't need to be true if you implement your own queue, as you can explicitly only drain that queue from your thread.","page":"Multi-Threading"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html#Static-analyzer-annotations-for-GC-correctness-in-C-code","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Running the analysis","location":"devdocs/gc-sa.html#Running-the-analysis","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer plugin that drives the anlysis ships with julia. Its source code can be found in src/clangsa. Running it requires the clang dependency to be build. Set the BUILD_LLVM_CLANG variable in your Make.user in order to build an appropriate version of clang. You may also want to use the prebuilt binaries using the USE_BINARYBUILDER_LLVM options. Afterwards, running the analysis over the source tree is as simple as running make -C src analyzegc.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"General Overview","location":"devdocs/gc-sa.html#General-Overview","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Since Julia's GC is precise, it needs to maintain correct rooting information for any value that may be referenced at any time GC may occur. These places are known as safepoints and in the function local context, we extend this designation to any function call that may recursively end up at a safepoint.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"In generated code, this is taken care of automatically by the GC root placement pass (see the chapter on GC rooting in the LLVM codegen devdocs). However, in C code, we need to inform the runtime of any GC roots manually. This is done using the following macros:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"// The value assigned to any slot passed as an argument to these\n// is rooted for the duration of this GC frame.\nJL_GC_PUSH{1,...,6}(args...)\n// The values assigned into the size `n` array `rts` are rooted\n// for the duration of this GC frame.\nJL_GC_PUSHARGS(rts, n)\n// Pop a GC frame\nJL_GC_POP","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"If these macros are not used where they need to be, or they are used incorrectly, the result is silent memory corruption. As such it is very important that they are placed correctly in all applicable code.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"As such, we employ static analysis (and in particular the clang static analyzer) to help ensure that these macros are used correctly. The remainder of this document gives an overview of this static analysis and describes the support needed in the julia code base to make things work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"GC Invariants","location":"devdocs/gc-sa.html#GC-Invariants","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"There is two simple invariants correctness:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"All GC_PUSH calls need to be followed by an appropriate GC_POP (in practice we enforce this at the function level)\nIf a value was previously not rooted at any safepoint, it may no longer be referenced afterwards","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Of course the devil is in the details here. In particular to satisfy the second of the above conditions, we need to know:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Which calls are safepoints and which are not\nWhich values are rooted at any given safepoint and which are not\nWhen is a value referenced","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"For the second point in particular, we need to know which memory locations will be considered rooting at runtime (i.e. values assigned to such locations are rooted). This includes locations explicitly designated as such by passing them to one of the GC_PUSH macros, globally rooted locations and values, as well as any location recursively reachable from one of those locations.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static Analysis Algorithm","location":"devdocs/gc-sa.html#Static-Analysis-Algorithm","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The idea itself is very simple, although the implementation is quite a bit more complicated (mainly due to a large number of special cases and intricacies of C and C++). In essence, we keep track of all locations that are rooting, all values that are rootable and any expression (assignments, allocations, etc) affect the rootedness of any rootable values. Then, at any safepoint, we perform a \"symbolic GC\" and poison any values that are not rooted at said location. If these values are later referenced, we emit an error.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The clang static analyzer works by constructing a graph of states and exploring this graph for sources of errors. Several nodes in this graph are generated by the analyzer itself (e.g. for control flow), but the definitions above augment this graph with our own state.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The static analyzer is interprocedural and can analyze control flow across function boundaries. However, the static analyzer is not fully recursive and makes heuristic decisions about which calls to explore (additionally some calls are cross-translation unit and invisible to the analyzer). In our case, our definition of correctness requires total information. As such, we need to annotate the prototypes of all function calls with whatever information the analysis required, even if that information would otherwise be available by interprocedural static analysis.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Luckily however, we can still use this interprocedural analysis to ensure that the annotations we place on a given function are indeed correct given the implementation of said function.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"The analyzer annotations","location":"devdocs/gc-sa.html#The-analyzer-annotations","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"These annotations are found in src/support/analyzer_annotations.h. The are only active when the analyzer is being used and expand either to nothing (for prototype annotations) or to no-ops (for function like annotations).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_NOTSAFEPOINT","location":"devdocs/gc-sa.html#JL_NOTSAFEPOINT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is perhaps the most common annotation, and should be placed on any function that is known not to possibly lead to reaching a GC safepoint. In general, it is only safe for such a function to perform arithmetic, memory accesses and calls to functions either annotated JL_NOTSAFEPOINT or otherwise known not to be safepoints (e.g. function in the C standard library, which are hardcoded as such in the analyzer)","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"It is valid to keep values unrooted across calls to any function annotated with this attribute:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_get_one() JL_NOTSAFEPOINT {\n  return 1;\n}\n\njl_value_t *example() {\n  jl_value_t *val = jl_alloc_whatever();\n  // This is valid, even though `val` is unrooted, because\n  // jl_get_one is not a safepoint\n  jl_get_one();\n  return val;\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","location":"devdocs/gc-sa.html#JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"When JL_MAYBE_UNROOTED is annotated as an argument on a function, indicates that said argument may be passed, even if it is not rooted. In the ordinary course of events, the julia ABI guarantees that callers root values before passing them to callees. However, some functions do not follow this ABI and allow values to be passed to them even though they are not rooted. Note however, that this does not automatically imply that said argument will be preserved. The ROOTS_TEMPORARILY annotation provides the stronger guarantee that, not only may the value be unrooted when passed, it will also be preserved across any internal safepoints by the callee.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Note that JL_NOTSAFEPOINT essentially implies JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY, because the rootedness of an argument is irrelevant if the function contains no safepoints.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"One additional point to note is that these annotations apply on both the caller and the callee side. On the caller side, they lift rootedness restrictions that are normally required for julia ABI functions. On the callee side, they have the reverse effect of preventing these arguments from being considered implicitly rooted.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"If either of these annotations is applied to the function as a whole, it applies to all arguments of the function. This should generally only be necessary for varargs functions.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"JL_DLLEXPORT void JL_NORETURN jl_throw(jl_value_t *e JL_MAYBE_UNROOTED);\njl_value_t *jl_alloc_error();\n\nvoid example() {\n  // The return value of the allocation is unrooted. This would normally\n  // be an error, but is allowed because of the above annotation.\n  jl_throw(jl_alloc_error());\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_PROPAGATES_ROOT","location":"devdocs/gc-sa.html#JL_PROPAGATES_ROOT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation is commonly found on accessor functions that return one rootable object stored within another. When annotated on a function argument, it tells the analyzer that the root for that argument also applies to the value returned by the function.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"jl_value_t *jl_svecref(jl_svec_t *t JL_PROPAGATES_ROOT, size_t i) JL_NOTSAFEPOINT;\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_svecref(svec, 1)\n  // This is valid, because, as annotated by the PROPAGATES_ROOT annotation,\n  // jl_svecref propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","location":"devdocs/gc-sa.html#JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is essentially the assignment counterpart to JL_PROPAGATES_ROOT. When assigning a value to a field of another value that is already rooted, the assigned value will inherit the root of the value it is assigned into.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage Example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_svecset(void *t JL_ROOTING_ARGUMENT, size_t i, void *x JL_ROOTED_ARGUMENT) JL_NOTSAFEPOINT\n\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_box_long(10000);\n  jl_svecset(svec, val);\n  // This is valid, because the annotations imply that the\n  // jl_svecset propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_DISABLED","location":"devdocs/gc-sa.html#JL_GC_DISABLED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation implies that this function is only called with the GC runtime-disabled. Functions of this kind are most often encountered during startup and in the GC code itself. Note that this annotation is checked against the runtime enable/disable calls, so clang will know if you lie. This is not a good way to disable processing of a given function if the GC is not actually disabled (use ifdef __clang_analyzer__ for that if you must).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_do_magic() JL_GC_DISABLED {\n  // Wildly allocate here with no regard for roots\n}\n\nvoid example() {\n  int en = jl_gc_enable(0);\n  jl_do_magic();\n  jl_gc_enable(en);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_REQUIRE_ROOTED_SLOT","location":"devdocs/gc-sa.html#JL_REQUIRE_ROOTED_SLOT","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation requires the caller to pass in a slot that is rooted (i.e. values assigned to this slot will be rooted).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void jl_do_processing(jl_value_t **slot JL_REQUIRE_ROOTED_SLOT) {\n  *slot = jl_box_long(1);\n  // Ok, only, because the slot was annotated as rooting\n  jl_gc_safepoint();\n}\n\nvoid example() {\n  jl_value_t *slot = NULL;\n  JL_GC_PUSH1(&slot);\n  jl_do_processing(&slot);\n  JL_GC_POP();\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GLOBALLY_ROOTED","location":"devdocs/gc-sa.html#JL_GLOBALLY_ROOTED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotation implies that a given value is always globally rooted. It can be applied to global variable declarations, in which case it will apply to the value of those variables (or values if the declaration if for an array), or to functions, in which case it will apply to the return value of such functions (e.g. for functions that always return some private, globally rooted value).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"Usage example:","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"extern JL_DLLEXPORT jl_datatype_t *jl_any_type JL_GLOBALLY_ROOTED;\njl_ast_context_t *jl_ast_ctx(fl_context_t *fl) JL_GLOBALLY_ROOTED;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ALWAYS_LEAFTYPE","location":"devdocs/gc-sa.html#JL_ALWAYS_LEAFTYPE","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This annotations is essentially equivalent to JL_GLOBALLY_ROOTED, except that is should only be used if those values are globally rooted by virtue of being a leaftype. The rooting of leaftypes is a bit complicated. They are generally rooted through cache field of the corresponding TypeName, which itself is rooted by the containing module (so they're rooted as long as the containing module is ok) and we can generally assume that leaftypes are rooted where they are used, but we may refine this property in the future, so the separate annotation helps split out the reason for being globally rooted.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer also automatically detects checks for leaftype-ness and will not complain about missing GC roots on these paths.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"JL_DLLEXPORT jl_value_t *jl_apply_array_type(jl_value_t *type, size_t dim) JL_ALWAYS_LEAFTYPE;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_PROMISE_ROOTED","location":"devdocs/gc-sa.html#JL_GC_PROMISE_ROOTED","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"This is a function-like annotation. Any value passed to this annotation will be considered rooted for the scope of the current function. It is designed as an escape hatch for analyzer inadequacy or complicated situations. However, it should be used sparingly, in favor of improving the analyzer itself.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"void example() {\n  jl_value_t *val = jl_alloc_something();\n  if (some_condition) {\n    // We happen to know for complicated external reasons\n    // that val is rooted under these conditions\n    JL_GC_PROMISE_ROOTED(val);\n  }\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Completeness of analysis","location":"devdocs/gc-sa.html#Completeness-of-analysis","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html","category":"page","text":"The analyzer only looks at local information. In particular, e.g. in the PROPAGATES_ROOT case above, it assumes that such memory is only modified in ways it can see, not in any called functions (unless it happens to decide to consider them in its analysis) and not in any concurrently running threads. As such, it may miss a few problematic cases, though in practice such concurrent modification is fairly rare. Improving the analyzer to handle more such cases may be an interesting topic for future work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Iteration utilities","location":"base/iterators.html#Iteration-utilities","category":"section","text":"","page":"Iteration utilities"},{"title":"Iteration utilities","location":"base/iterators.html","category":"page","text":"Base.Iterators.Stateful\nBase.Iterators.zip\nBase.Iterators.enumerate\nBase.Iterators.rest\nBase.Iterators.countfrom\nBase.Iterators.take\nBase.Iterators.takewhile\nBase.Iterators.drop\nBase.Iterators.dropwhile\nBase.Iterators.cycle\nBase.Iterators.repeated\nBase.Iterators.product\nBase.Iterators.flatten\nBase.Iterators.partition\nBase.Iterators.map\nBase.Iterators.filter\nBase.Iterators.accumulate\nBase.Iterators.reverse\nBase.Iterators.only\nBase.Iterators.peel","page":"Iteration utilities"},{"title":"Base.Iterators.Stateful","location":"base/iterators.html#Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.zip","location":"base/iterators.html#Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.enumerate","location":"base/iterators.html#Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far. Note that i may not be valid for indexing iter; it's also possible that x != iter[i], if iter has indices that do not start at 1. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.rest","location":"base/iterators.html#Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state.\n\nExamples\n\njulia> collect(Iterators.rest([1,2,3,4], 2))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.countfrom","location":"base/iterators.html#Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.take","location":"base/iterators.html#Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.takewhile","location":"base/iterators.html#Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.drop","location":"base/iterators.html#Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.dropwhile","location":"base/iterators.html#Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.cycle","location":"base/iterators.html#Base.Iterators.cycle","category":"function","text":"cycle(iter)\n\nAn iterator that cycles through iter forever. If iter is empty, so is cycle(iter).\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.repeated","location":"base/iterators.html#Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.product","location":"base/iterators.html#Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.flatten","location":"base/iterators.html#Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.partition","location":"base/iterators.html#Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.map","location":"base/iterators.html#Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.filter","location":"base/iterators.html#Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.accumulate","location":"base/iterators.html#Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> f = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, f)\n1\n3\n6\n10\n\njulia> f = Iterators.accumulate(+, [1,2,3]; init = 100);\n\njulia> foreach(println, f)\n101\n103\n106\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.reverse","location":"base/iterators.html#Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order.\n\nThis iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from r = Iterators.reverse(itr) by r.itr.)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.only","location":"base/iterators.html#Base.Iterators.only","category":"function","text":"only(x)\n\nReturns the one and only element of collection x, and throws an ArgumentError if the collection has zero or multiple elements.\n\nSee also: first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.peel","location":"base/iterators.html#Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"C Standard Library","location":"base/libc.html#C-Standard-Library","category":"section","text":"","page":"C Standard Library"},{"title":"C Standard Library","location":"base/libc.html","category":"page","text":"Base.Libc.malloc\nBase.Libc.calloc\nBase.Libc.realloc\nBase.Libc.free\nBase.Libc.errno\nBase.Libc.strerror\nBase.Libc.GetLastError\nBase.Libc.FormatMessage\nBase.Libc.time(::Base.Libc.TmStruct)\nBase.Libc.strftime\nBase.Libc.strptime\nBase.Libc.TmStruct\nBase.Libc.flush_cstdio\nBase.Libc.systemsleep","page":"C Standard Library"},{"title":"Base.Libc.malloc","location":"base/libc.html#Base.Libc.malloc","category":"function","text":"malloc(size::Integer) -> Ptr{Cvoid}\n\nCall malloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.calloc","location":"base/libc.html#Base.Libc.calloc","category":"function","text":"calloc(num::Integer, size::Integer) -> Ptr{Cvoid}\n\nCall calloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.realloc","location":"base/libc.html#Base.Libc.realloc","category":"function","text":"realloc(addr::Ptr, size::Integer) -> Ptr{Cvoid}\n\nCall realloc from the C standard library.\n\nSee warning in the documentation for free regarding only using this on memory originally obtained from malloc.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.free","location":"base/libc.html#Base.Libc.free","category":"function","text":"free(addr::Ptr)\n\nCall free from the C standard library. Only use this on memory obtained from malloc, not on pointers retrieved from other C libraries. Ptr objects obtained from C libraries should be freed by the free functions defined in that library, to avoid assertion failures if multiple libc libraries exist on the system.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.errno","location":"base/libc.html#Base.Libc.errno","category":"function","text":"errno([code])\n\nGet the value of the C library's errno. If an argument is specified, it is used to set the value of errno.\n\nThe value of errno is only valid immediately after a ccall to a C library routine that sets it. Specifically, you cannot call errno at the next prompt in a REPL, because lots of code is executed between prompts.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strerror","location":"base/libc.html#Base.Libc.strerror","category":"function","text":"strerror(n=errno())\n\nConvert a system call error code to a descriptive string\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.GetLastError","location":"base/libc.html#Base.Libc.GetLastError","category":"function","text":"GetLastError()\n\nCall the Win32 GetLastError function [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.FormatMessage","location":"base/libc.html#Base.Libc.FormatMessage","category":"function","text":"FormatMessage(n=GetLastError())\n\nConvert a Win32 system call error code to a descriptive string [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.time","location":"base/libc.html#Base.Libc.time-Tuple{Base.Libc.TmStruct}","category":"method","text":"time(t::TmStruct)\n\nConverts a TmStruct struct to a number of seconds since the epoch.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strftime","location":"base/libc.html#Base.Libc.strftime","category":"function","text":"strftime([format], time)\n\nConvert time, given as a number of seconds since the epoch or a TmStruct, to a formatted string using the given format. Supported formats are the same as those in the standard C library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strptime","location":"base/libc.html#Base.Libc.strptime","category":"function","text":"strptime([format], timestr)\n\nParse a formatted time string into a TmStruct giving the seconds, minute, hour, date, etc. Supported formats are the same as those in the standard C library. On some platforms, timezones will not be parsed correctly. If the result of this function will be passed to time to convert it to seconds since the epoch, the isdst field should be filled in manually. Setting it to -1 will tell the C library to use the current system settings to determine the timezone.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.TmStruct","location":"base/libc.html#Base.Libc.TmStruct","category":"type","text":"TmStruct([seconds])\n\nConvert a number of seconds since the epoch to broken-down format, with fields sec, min, hour, mday, month, year, wday, yday, and isdst.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.flush_cstdio","location":"base/libc.html#Base.Libc.flush_cstdio","category":"function","text":"flush_cstdio()\n\nFlushes the C stdout and stderr streams (which may have been written to by external C code).\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.systemsleep","location":"base/libc.html#Base.Libc.systemsleep","category":"function","text":"systemsleep(s::Real)\n\nSuspends execution for s seconds. This function does not yield to Julia's scheduler and therefore blocks the Julia thread that it is running on for the duration of the sleep time.\n\nSee also: sleep\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"EditURL = \"https://github.com/JuliaLang/Pkg.jl/blob/master/docs/src/basedocs.md\"","page":"Pkg"},{"title":"Pkg","location":"stdlib/Pkg.html#Pkg","category":"section","text":"","page":"Pkg"},{"title":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"Pkg is Julia's builtin package manager, and handles operations such as installing, updating and removing packages.","page":"Pkg"},{"title":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"note: Note\nWhat follows is a very brief introduction to Pkg. For more information on Project.toml files, Manifest.toml files, package version compatibility ([compat]), environments, registries, etc., it is highly recommended to read the full manual, which is available here: https://julialang.github.io/Pkg.jl/v1/.","page":"Pkg"},{"title":"Pkg","location":"stdlib/Pkg.html","category":"page","text":"import Markdown\nfile = joinpath(Sys.STDLIB, \"Pkg\", \"docs\", \"src\", \"getting-started.md\")\nstr = read(file, String)\nstr = replace(str, r\"^#.*$\"m => \"\")\nstr = replace(str, \"[API Reference](@ref)\" =>\n          \"[API Reference](https://julialang.github.io/Pkg.jl/v1/api/)\")\nMarkdown.parse(str)","page":"Pkg"},{"title":"Random Numbers","location":"stdlib/Random.html#Random-Numbers","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"DocTestSetup = :(using Random)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random number generation in Julia uses the Mersenne Twister library via MersenneTwister objects. Julia has a global RNG, which is used by default. Other RNG types can be plugged in by inheriting the AbstractRNG type; they can then be used to have multiple streams of random numbers. Besides MersenneTwister, Julia also provides the RandomDevice RNG type, which is a wrapper over the OS provided entropy.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Most functions related to random generation accept an optional AbstractRNG object as first argument, which defaults to the global one if not provided. Moreover, some of them accept optionally dimension specifications dims... (which can be given as a tuple) to generate arrays of random values.  In a multi-threaded program, you should generally use different RNG objects from different threads in order to be thread-safe. However, the default global RNG is thread-safe as of Julia 1.3 (because it internally corresponds to a per-thread RNG).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"A MersenneTwister or RandomDevice RNG can generate uniformly random numbers of the following types: Float16, Float32, Float64, BigFloat, Bool, Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, BigInt (or complex numbers of those types). Random floating point numbers are generated uniformly in 0 1). As BigInt represents unbounded integers, the interval must be specified (e.g. rand(big.(1:6))).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Additionally, normal and exponential distributions are implemented for some AbstractFloat and Complex types, see randn and randexp for details.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"warning: Warning\nBecause the precise way in which random numbers are generated is considered an implementation detail, bug fixes and speed improvements may change the stream of numbers that are generated after a version change. Relying on a specific seed or generated stream of numbers during unit testing is thus discouraged - consider testing properties of the methods in question instead.","page":"Random Numbers"},{"title":"Random numbers module","location":"stdlib/Random.html#Random-numbers-module","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.Random","page":"Random Numbers"},{"title":"Random.Random","location":"stdlib/Random.html#Random.Random","category":"module","text":"Random\n\nSupport for generating random numbers. Provides rand, randn, AbstractRNG, MersenneTwister, and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random generation functions","location":"stdlib/Random.html#Random-generation-functions","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.rand\nRandom.rand!\nRandom.bitrand\nRandom.randn\nRandom.randn!\nRandom.randexp\nRandom.randexp!\nRandom.randstring","page":"Random Numbers"},{"title":"Base.rand","location":"stdlib/Random.html#Base.rand","category":"function","text":"rand([rng=GLOBAL_RNG], [S], [dims...])\n\nPick a random element or array of random elements from the set of values specified by S; S can be\n\nan indexable collection (for example 1:9 or ('x', \"y\", :z)),\nan AbstractDict or AbstractSet object,\na string (considered as a collection of characters), or\na type: the set of values to pick from is then equivalent to typemin(S):typemax(S) for integers (this is not applicable to BigInt), to 0 1) for floating point numbers and to 0 1)+i0 1) for complex floating point numbers;\n\nS defaults to Float64. When only one argument is passed besides the optional rng and is a Tuple, it is interpreted as a collection of values (S) and not as dims.\n\ncompat: Julia 1.1\nSupport for S as a tuple requires at least Julia 1.1.\n\nExamples\n\njulia> rand(Int, 2)\n2-element Array{Int64,1}:\n 1339893410598768192\n 1575814717733606317\n\njulia> using Random\n\njulia> rand(MersenneTwister(0), Dict(1=>2, 3=>4))\n1=>2\n\njulia> rand((2, 3))\n3\n\njulia> rand(Float64, (2, 3))\n2×3 Array{Float64,2}:\n 0.999717  0.0143835  0.540787\n 0.696556  0.783855   0.938235\n\nnote: Note\nThe complexity of rand(rng, s::Union{AbstractDict,AbstractSet}) is linear in the length of s, unless an optimized method with constant complexity is available, which is the case for Dict, Set and BitSet. For more than a few calls, use rand(rng, collect(s)) instead, or either rand(rng, Dict(s)) or rand(rng, Set(s)) as appropriate.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.rand!","location":"stdlib/Random.html#Random.rand!","category":"function","text":"rand!([rng=GLOBAL_RNG], A, [S=eltype(A)])\n\nPopulate the array A with random values. If S is specified (S can be a type or a collection, cf. rand for details), the values are picked randomly from S. This is equivalent to copyto!(A, rand(rng, S, size(A))) but without allocating a new array.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> rand!(rng, zeros(5))\n5-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n 0.5662374165061859\n 0.4600853424625171\n 0.7940257103317943\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.bitrand","location":"stdlib/Random.html#Random.bitrand","category":"function","text":"bitrand([rng=GLOBAL_RNG], [dims...])\n\nGenerate a BitArray of random boolean values.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> bitrand(rng, 10)\n10-element BitVector:\n 0\n 0\n 0\n 0\n 1\n 0\n 0\n 0\n 1\n 1\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Base.randn","location":"stdlib/Random.html#Base.randn","category":"function","text":"randn([rng=GLOBAL_RNG], [T=Float64], [dims...])\n\nGenerate a normally-distributed random number of type T with mean 0 and standard deviation 1. Optionally generate an array of normally-distributed random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default), and their Complex counterparts. When the type argument is complex, the values are drawn from the circularly symmetric complex normal distribution of variance 1 (corresponding to real and imaginary part having independent normal distribution with mean zero and variance 1/2).\n\nExamples\n\njulia> using Random\n\njulia> rng = MersenneTwister(1234);\n\njulia> randn(rng, ComplexF64)\n0.6133070881429037 - 0.6376291670853887im\n\njulia> randn(rng, ComplexF32, (2, 3))\n2×3 Matrix{ComplexF32}:\n -0.349649-0.638457im  0.376756-0.192146im  -0.396334-0.0136413im\n  0.611224+1.56403im   0.355204-0.365563im  0.0905552+1.31012im\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randn!","location":"stdlib/Random.html#Random.randn!","category":"function","text":"randn!([rng=GLOBAL_RNG], A::AbstractArray) -> A\n\nFill the array A with normally-distributed (mean 0, standard deviation 1) random numbers. Also see the rand function.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randn!(rng, zeros(5))\n5-element Vector{Float64}:\n  0.8673472019512456\n -0.9017438158568171\n -0.4944787535042339\n -0.9029142938652416\n  0.8644013132535154\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp","location":"stdlib/Random.html#Random.randexp","category":"function","text":"randexp([rng=GLOBAL_RNG], [T=Float64], [dims...])\n\nGenerate a random number of type T according to the exponential distribution with scale 1. Optionally generate an array of such random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randexp(rng, Float32)\n2.4835055f0\n\njulia> randexp(rng, 3, 3)\n3×3 Matrix{Float64}:\n 1.5167    1.30652   0.344435\n 0.604436  2.78029   0.418516\n 0.695867  0.693292  0.643644\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp!","location":"stdlib/Random.html#Random.randexp!","category":"function","text":"randexp!([rng=GLOBAL_RNG], A::AbstractArray) -> A\n\nFill the array A with random numbers following the exponential distribution (with scale 1).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randexp!(rng, zeros(5))\n5-element Vector{Float64}:\n 2.4835053723904896\n 1.516703605376473\n 0.6044364871025417\n 0.6958665886385867\n 1.3065196315496677\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randstring","location":"stdlib/Random.html#Random.randstring","category":"function","text":"randstring([rng=GLOBAL_RNG], [chars], [len=8])\n\nCreate a random string of length len, consisting of characters from chars, which defaults to the set of upper- and lower-case letters and the digits 0-9. The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> Random.seed!(3); randstring()\n\"Y7m62wOj\"\n\njulia> randstring(MersenneTwister(3), 'a':'z', 6)\n\"ocucay\"\n\njulia> randstring(\"ACGT\")\n\"ATTTGCGT\"\n\nnote: Note\nchars can be any collection of characters, of type Char or UInt8 (more efficient), provided rand can randomly pick characters from it.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Subsequences, permutations and shuffling","location":"stdlib/Random.html#Subsequences,-permutations-and-shuffling","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.randsubseq\nRandom.randsubseq!\nRandom.randperm\nRandom.randperm!\nRandom.randcycle\nRandom.randcycle!\nRandom.shuffle\nRandom.shuffle!","page":"Random Numbers"},{"title":"Random.randsubseq","location":"stdlib/Random.html#Random.randsubseq","category":"function","text":"randsubseq([rng=GLOBAL_RNG,] A, p) -> Vector\n\nReturn a vector consisting of a random subsequence of the given array A, where each element of A is included (in order) with independent probability p. (Complexity is linear in p*length(A), so this function is efficient even if p is small and A is large.) Technically, this process is known as \"Bernoulli sampling\" of A.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> randsubseq(rng, 1:8, 0.3)\n2-element Vector{Int64}:\n 7\n 8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randsubseq!","location":"stdlib/Random.html#Random.randsubseq!","category":"function","text":"randsubseq!([rng=GLOBAL_RNG,] S, A, p)\n\nLike randsubseq, but the results are stored in S (which is resized as needed).\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> S = Int64[];\n\njulia> randsubseq!(rng, S, 1:8, 0.3)\n2-element Vector{Int64}:\n 7\n 8\n\njulia> S\n2-element Vector{Int64}:\n 7\n 8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm","location":"stdlib/Random.html#Random.randperm","category":"function","text":"randperm([rng=GLOBAL_RNG,] n::Integer)\n\nConstruct a random permutation of length n. The optional rng argument specifies a random number generator (see Random Numbers). The element type of the result is the same as the type of n.\n\nTo randomly permute an arbitrary vector, see shuffle or shuffle!.\n\ncompat: Julia 1.1\nIn Julia 1.1 randperm returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randperm(MersenneTwister(1234), 4)\n4-element Vector{Int64}:\n 2\n 1\n 4\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm!","location":"stdlib/Random.html#Random.randperm!","category":"function","text":"randperm!([rng=GLOBAL_RNG,] A::Array{<:Integer})\n\nConstruct in A a random permutation of length length(A). The optional rng argument specifies a random number generator (see Random Numbers). To randomly permute an arbitrary vector, see shuffle or shuffle!.\n\nExamples\n\njulia> randperm!(MersenneTwister(1234), Vector{Int}(undef, 4))\n4-element Vector{Int64}:\n 2\n 1\n 4\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle","location":"stdlib/Random.html#Random.randcycle","category":"function","text":"randcycle([rng=GLOBAL_RNG,] n::Integer)\n\nConstruct a random cyclic permutation of length n. The optional rng argument specifies a random number generator, see Random Numbers. The element type of the result is the same as the type of n.\n\ncompat: Julia 1.1\nIn Julia 1.1 randcycle returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randcycle(MersenneTwister(1234), 6)\n6-element Vector{Int64}:\n 3\n 5\n 4\n 6\n 1\n 2\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle!","location":"stdlib/Random.html#Random.randcycle!","category":"function","text":"randcycle!([rng=GLOBAL_RNG,] A::Array{<:Integer})\n\nConstruct in A a random cyclic permutation of length length(A). The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> randcycle!(MersenneTwister(1234), Vector{Int}(undef, 6))\n6-element Vector{Int64}:\n 3\n 5\n 4\n 6\n 1\n 2\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle","location":"stdlib/Random.html#Random.shuffle","category":"function","text":"shuffle([rng=GLOBAL_RNG,] v::AbstractArray)\n\nReturn a randomly permuted copy of v. The optional rng argument specifies a random number generator (see Random Numbers). To permute v in-place, see shuffle!. To obtain randomly permuted indices, see randperm.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> shuffle(rng, Vector(1:10))\n10-element Vector{Int64}:\n  6\n  1\n 10\n  2\n  3\n  9\n  5\n  7\n  4\n  8\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle!","location":"stdlib/Random.html#Random.shuffle!","category":"function","text":"shuffle!([rng=GLOBAL_RNG,] v::AbstractArray)\n\nIn-place version of shuffle: randomly permute v in-place, optionally supplying the random-number generator rng.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> shuffle!(rng, Vector(1:16))\n16-element Vector{Int64}:\n  2\n 15\n  5\n 14\n  1\n  9\n 10\n  6\n 11\n  3\n 16\n  7\n  4\n 12\n  8\n 13\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Generators (creation and seeding)","location":"stdlib/Random.html#Generators-(creation-and-seeding)","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.seed!\nRandom.AbstractRNG\nRandom.MersenneTwister\nRandom.RandomDevice","page":"Random Numbers"},{"title":"Random.seed!","location":"stdlib/Random.html#Random.seed!","category":"function","text":"seed!([rng=GLOBAL_RNG], seed) -> rng\nseed!([rng=GLOBAL_RNG]) -> rng\n\nReseed the random number generator: rng will give a reproducible sequence of numbers if and only if a seed is provided. Some RNGs don't accept a seed, like RandomDevice. After the call to seed!, rng is equivalent to a newly created object initialized with the same seed.\n\nIf rng is not specified, it defaults to seeding the state of the shared thread-local generator.\n\nExamples\n\njulia> Random.seed!(1234);\n\njulia> x1 = rand(2)\n2-element Array{Float64,1}:\n 0.590845\n 0.766797\n\njulia> Random.seed!(1234);\n\njulia> x2 = rand(2)\n2-element Array{Float64,1}:\n 0.590845\n 0.766797\n\njulia> x1 == x2\ntrue\n\njulia> rng = MersenneTwister(1234); rand(rng, 2) == x1\ntrue\n\njulia> MersenneTwister(1) == Random.seed!(rng, 1)\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible\ntrue\n\njulia> rand(Random.seed!(rng), Bool)\nfalse\n\njulia> rand(MersenneTwister(), Bool) # not reproducible either\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.AbstractRNG","location":"stdlib/Random.html#Random.AbstractRNG","category":"type","text":"AbstractRNG\n\nSupertype for random number generators such as MersenneTwister and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.MersenneTwister","location":"stdlib/Random.html#Random.MersenneTwister","category":"type","text":"MersenneTwister(seed)\nMersenneTwister()\n\nCreate a MersenneTwister RNG object. Different RNG objects can have their own seeds, which may be useful for generating different streams of random numbers. The seed may be a non-negative integer or a vector of UInt32 integers. If no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing MersenneTwister object.\n\nExamples\n\njulia> rng = MersenneTwister(1234);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n\njulia> rng = MersenneTwister(1234);\n\njulia> x2 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.5908446386657102\n 0.7667970365022592\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.RandomDevice","location":"stdlib/Random.html#Random.RandomDevice","category":"type","text":"RandomDevice()\n\nCreate a RandomDevice RNG object. Two such objects will always generate different streams of random numbers. The entropy is obtained from the operating system.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Hooking into the Random API","location":"stdlib/Random.html#Hooking-into-the-Random-API","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"There are two mostly orthogonal ways to extend Random functionalities:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"generating random values of custom types\ncreating new generators","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API for 1) is quite functional, but is relatively recent so it may still have to evolve in subsequent releases of the Random module. For example, it's typically sufficient to implement one rand method in order to have all other usual methods work automatically.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API for 2) is still rudimentary, and may require more work than strictly necessary from the implementor, in order to support usual types of generated values.","page":"Random Numbers"},{"title":"Generating random values of custom types","location":"stdlib/Random.html#Generating-random-values-of-custom-types","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Generating random values for some distributions may involve various trade-offs. Pre-computed values, such as an alias table for discrete distributions, or “squeezing” functions for univariate distributions, can speed up sampling considerably. How much information should be pre-computed can depend on the number of values we plan to draw from a distribution. Also, some random number generators can have certain properties that various algorithms may want to exploit.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The Random module defines a customizable framework for obtaining random values that can address these issues. Each invocation of rand generates a sampler which can be customized with the above trade-offs in mind, by adding methods to Sampler, which in turn can dispatch on the random number generator, the object that characterizes the distribution, and a suggestion for the number of repetitions. Currently, for the latter, Val{1} (for a single sample) and Val{Inf} (for an arbitrary number) are used, with Random.Repetition an alias for both.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The object returned by Sampler is then used to generate the random values. When implementing the random generation interface for a value X that can be sampled from, the implementor should define the method","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"rand(rng, sampler)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"for the particular sampler returned by Sampler(rng, X, repetition).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Samplers can be arbitrary values that implement rand(rng, sampler), but for most applications the following predefined samplers may be sufficient:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"SamplerType{T}() can be used for implementing samplers that draw from type T (e.g. rand(Int)). This is the default returned by Sampler for types.\nSamplerTrivial(self) is a simple wrapper for self, which can be accessed with []. This is the recommended sampler when no pre-computed information is needed (e.g. rand(1:3)), and is the default returned by Sampler for values.\nSamplerSimple(self, data) also contains the additional data field, which can be used to store arbitrary pre-computed values, which should be computed in a custom method of Sampler.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"We provide examples for each of these. We assume here that the choice of algorithm is independent of the RNG, so we use AbstractRNG in our signatures.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.Sampler\nRandom.SamplerType\nRandom.SamplerTrivial\nRandom.SamplerSimple","page":"Random Numbers"},{"title":"Random.Sampler","location":"stdlib/Random.html#Random.Sampler","category":"type","text":"Sampler(rng, x, repetition = Val(Inf))\n\nReturn a sampler object that can be used to generate random values from rng for x.\n\nWhen sp = Sampler(rng, x, repetition), rand(rng, sp) will be used to draw random values, and should be defined accordingly.\n\nrepetition can be Val(1) or Val(Inf), and should be used as a suggestion for deciding the amount of precomputation, if applicable.\n\nRandom.SamplerType and Random.SamplerTrivial are default fallbacks for types and values, respectively. Random.SamplerSimple can be used to store pre-computed values without defining extra types for only this purpose.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerType","location":"stdlib/Random.html#Random.SamplerType","category":"type","text":"SamplerType{T}()\n\nA sampler for types, containing no other information. The default fallback for Sampler when called with types.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerTrivial","location":"stdlib/Random.html#Random.SamplerTrivial","category":"type","text":"SamplerTrivial(x)\n\nCreate a sampler that just wraps the given value x. This is the default fall-back for values. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values without precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerSimple","location":"stdlib/Random.html#Random.SamplerSimple","category":"type","text":"SamplerSimple(x, data)\n\nCreate a sampler that wraps the given value x and the data. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values with precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Decoupling pre-computation from actually generating the values is part of the API, and is also available to the user. As an example, assume that rand(rng, 1:20) has to be called repeatedly in a loop: the way to take advantage of this decoupling is as follows:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"rng = MersenneTwister()\nsp = Random.Sampler(rng, 1:20) # or Random.Sampler(MersenneTwister, 1:20)\nfor x in X\n    n = rand(rng, sp) # similar to n = rand(rng, 1:20)\n    # use n\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"This is the mechanism that is also used in the standard library, e.g. by the default implementation of random array generation (like in rand(1:20, 10)).","page":"Random Numbers"},{"title":"Generating values from a type","location":"stdlib/Random.html#Generating-values-from-a-type","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Given a type T, it's currently assumed that if rand(T) is defined, an object of type T will be produced. SamplerType is the default sampler for types. In order to define random generation of values of type T, the rand(rng::AbstractRNG, ::Random.SamplerType{T}) method should be defined, and should return values what rand(rng, T) is expected to return.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Let's take the following example: we implement a Die type, with a variable number n of sides, numbered from 1 to n. We want rand(Die) to produce a Die with a random number of up to 20 sides (and at least 4):","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"struct Die\n    nsides::Int # number of sides\nend\n\nRandom.rand(rng::AbstractRNG, ::Random.SamplerType{Die}) = Die(rand(rng, 4:20))\n\n# output\n","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Scalar and array methods for Die now work as expected:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"julia> rand(Die)\nDie(15)\n\njulia> rand(MersenneTwister(0), Die)\nDie(11)\n\njulia> rand(Die, 3)\n3-element Vector{Die}:\n Die(18)\n Die(5)\n Die(4)\n\njulia> a = Vector{Die}(undef, 3); rand!(a)\n3-element Vector{Die}:\n Die(5)\n Die(20)\n Die(15)","page":"Random Numbers"},{"title":"A simple sampler without pre-computed data","location":"stdlib/Random.html#A-simple-sampler-without-pre-computed-data","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Here we define a sampler for a collection. If no pre-computed data is required, it can be implemented with a SamplerTrivial sampler, which is in fact the default fallback for values.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In order to define random generation out of objects of type S, the following method should be defined: rand(rng::AbstractRNG, sp::Random.SamplerTrivial{S}). Here, sp simply wraps an object of type S, which can be accessed via sp[]. Continuing the Die example, we want now to define rand(d::Die) to produce an Int corresponding to one of d's sides:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"julia> Random.rand(rng::AbstractRNG, d::Random.SamplerTrivial{Die}) = rand(rng, 1:d[].nsides);\n\njulia> rand(Die(4))\n3\n\njulia> rand(Die(4), 3)\n3-element Vector{Any}:\n 4\n 1\n 1","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Given a collection type S, it's currently assumed that if rand(::S) is defined, an object of type eltype(S) will be produced. In the last example, a Vector{Any} is produced; the reason is that eltype(Die) == Any. The remedy is to define Base.eltype(::Type{Die}) = Int.","page":"Random Numbers"},{"title":"Generating values for an AbstractFloat type","location":"stdlib/Random.html#Generating-values-for-an-AbstractFloat-type","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"AbstractFloat types are special-cased, because by default random values are not produced in the whole type domain, but rather in [0,1). The following method should be implemented for T <: AbstractFloat: Random.rand(::AbstractRNG, ::Random.SamplerTrivial{Random.CloseOpen01{T}})","page":"Random Numbers"},{"title":"An optimized sampler with pre-computed data","location":"stdlib/Random.html#An-optimized-sampler-with-pre-computed-data","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Consider a discrete distribution, where numbers 1:n are drawn with given probabilities that sum to one. When many values are needed from this distribution, the fastest method is using an alias table. We don't provide the algorithm for building such a table here, but suppose it is available in make_alias_table(probabilities) instead, and draw_number(rng, alias_table) can be used to draw a random number from it.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Suppose that the distribution is described by","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"struct DiscreteDistribution{V <: AbstractVector}\n    probabilities::V\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"and that we always want to build an alias table, regardless of the number of values needed (we learn how to customize this below). The methods","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Random.eltype(::Type{<:DiscreteDistribution}) = Int\n\nfunction Random.Sampler(::Type{<:AbstractRNG}, distribution::DiscreteDistribution, ::Repetition)\n    SamplerSimple(disribution, make_alias_table(distribution.probabilities))\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"should be defined to return a sampler with pre-computed data, then","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"function rand(rng::AbstractRNG, sp::SamplerSimple{<:DiscreteDistribution})\n    draw_number(rng, sp.data)\nend","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"will be used to draw the values.","page":"Random Numbers"},{"title":"Custom sampler types","location":"stdlib/Random.html#Custom-sampler-types","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The SamplerSimple type is sufficient for most use cases with precomputed data. However, in order to demonstrate how to use custom sampler types, here we implement something similar to SamplerSimple.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Going back to our Die example: rand(::Die) uses random generation from a range, so there is an opportunity for this optimization. We call our custom sampler SamplerDie.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"import Random: Sampler, rand\n\nstruct SamplerDie <: Sampler{Int} # generates values of type Int\n    die::Die\n    sp::Sampler{Int} # this is an abstract type, so this could be improved\nend\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerDie(die, Sampler(RNG, 1:die.nsides, r))\n# the `r` parameter will be explained later on\n\nrand(rng::AbstractRNG, sp::SamplerDie) = rand(rng, sp.sp)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"It's now possible to get a sampler with sp = Sampler(rng, die), and use sp instead of die in any rand call involving rng. In the simplistic example above, die doesn't need to be stored in SamplerDie but this is often the case in practice.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Of course, this pattern is so frequent that the helper type used above, namely Random.SamplerSimple, is available, saving us the definition of SamplerDie: we could have implemented our decoupling with:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerSimple(die, Sampler(RNG, 1:die.nsides, r))\n\nrand(rng::AbstractRNG, sp::SamplerSimple{Die}) = rand(rng, sp.data)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Here, sp.data refers to the second parameter in the call to the SamplerSimple constructor (in this case equal to Sampler(rng, 1:die.nsides, r)), while the Die object can be accessed via sp[].","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Like SamplerDie, any custom sampler must be a subtype of Sampler{T} where T is the type of the generated values. Note that SamplerSimple(x, data) isa Sampler{eltype(x)}, so this constrains what the first argument to SamplerSimple can be (it's recommended to use SamplerSimple like in the Die example, where x is simply forwarded while defining a Sampler method). Similarly, SamplerTrivial(x) isa Sampler{eltype(x)}.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Another helper type is currently available for other cases, Random.SamplerTag, but is considered as internal API, and can break at any time without proper deprecations.","page":"Random Numbers"},{"title":"Using distinct algorithms for scalar or array generation","location":"stdlib/Random.html#Using-distinct-algorithms-for-scalar-or-array-generation","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In some cases, whether one wants to generate only a handful of values or a large number of values will have an impact on the choice of algorithm. This is handled with the third parameter of the Sampler constructor. Let's assume we defined two helper types for Die, say SamplerDie1 which should be used to generate only few random values, and SamplerDieMany for many values. We can use those types as follows:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{1}) = SamplerDie1(...)\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{Inf}) = SamplerDieMany(...)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Of course, rand must also be defined on those types (i.e. rand(::AbstractRNG, ::SamplerDie1) and rand(::AbstractRNG, ::SamplerDieMany)). Note that, as usual, SamplerTrivial and SamplerSimple can be used if custom types are not necessary.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Note: Sampler(rng, x) is simply a shorthand for Sampler(rng, x, Val(Inf)), and Random.Repetition is an alias for Union{Val{1}, Val{Inf}}.","page":"Random Numbers"},{"title":"Creating new generators","location":"stdlib/Random.html#Creating-new-generators","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The API is not clearly defined yet, but as a rule of thumb:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"any rand method producing \"basic\" types (isbitstype integer and floating types in Base) should be defined for this specific RNG, if they are needed;\nother documented rand methods accepting an AbstractRNG should work out of the box, (provided the methods from 1) what are relied on are implemented), but can of course be specialized for this RNG if there is room for optimization;\ncopy for pseudo-RNGs should return an independent copy that generates the exact same random sequence as the original from that point when called in the same way. When this is not feasible (e.g. hardware-based RNGs), copy must not be implemented.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Concerning 1), a rand method may happen to work automatically, but it's not officially supported and may break without warnings in a subsequent release.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"To define a new rand method for an hypothetical MyRNG generator, and a value specification s (e.g. s == Int, or s == 1:10) of type S==typeof(s) or S==Type{s} if s is a type, the same two methods as we saw before must be defined:","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Sampler(::Type{MyRNG}, ::S, ::Repetition), which returns an object of type say SamplerS\nrand(rng::MyRNG, sp::SamplerS)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"It can happen that Sampler(rng::AbstractRNG, ::S, ::Repetition) is already defined in the Random module. It would then be possible to skip step 1) in practice (if one wants to specialize generation for this particular RNG type), but the corresponding SamplerS type is considered as internal detail, and may be changed without warning.","page":"Random Numbers"},{"title":"Specializing array generation","location":"stdlib/Random.html#Specializing-array-generation","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"In some cases, for a given RNG type, generating an array of random values can be more efficient with a specialized method than by merely using the decoupling technique explained before. This is for example the case for MersenneTwister, which natively writes random values in an array.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"To implement this specialization for MyRNG and for a specification s, producing elements of type S, the following method can be defined: rand!(rng::MyRNG, a::AbstractArray{S}, ::SamplerS), where SamplerS is the type of the sampler returned by Sampler(MyRNG, s, Val(Inf)). Instead of AbstractArray, it's possible to implement the functionality only for a subtype, e.g. Array{S}. The non-mutating array method of rand will automatically call this specialization internally.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"DocTestSetup = nothing","page":"Random Numbers"},{"title":"Reproducibility","location":"stdlib/Random.html#Reproducibility","category":"section","text":"","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"By using an RNG parameter initialized with a given seed, you can reproduce the same pseudorandom number sequence when running your program multiple times. However, a minor release of Julia (e.g. 1.3 to 1.4) may change the sequence of pseudorandom numbers generated from a specific seed, in particular if MersenneTwister is used. (Even if the sequence produced by a low-level function like rand does not change, the output of higher-level functions like randsubseq may change due to algorithm updates.) Rationale: guaranteeing that pseudorandom streams never change prohibits many algorithmic improvements.","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"If you need to guarantee exact reproducibility of random data, it is advisable to simply save the data (e.g. as a supplementary attachment in a scientific publication). (You can also, of course, specify a particular Julia version and package manifest, especially if you require bit reproducibility.)","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"Software tests that rely on specific \"random\" data should also generally either save the data, embed it into the test code, or use third-party packages like StableRNGs.jl. On the other hand, tests that should pass for most random data (e.g. testing A \\ (A*x) ≈ x for a random matrix A = randn(n,n)) can use an RNG with a fixed seed to ensure that simply running the test many times does not encounter a failure due to very improbable data (e.g. an extremely ill-conditioned matrix).","page":"Random Numbers"},{"title":"Random Numbers","location":"stdlib/Random.html","category":"page","text":"The statistical distribution from which random samples are drawn is guaranteed to be the same across any minor Julia releases.","page":"Random Numbers"},{"title":"Mathematics","location":"base/math.html#Mathematics","category":"section","text":"","page":"Mathematics"},{"title":"Mathematical Operators","location":"base/math.html#math-ops","category":"section","text":"","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"Base.:-(::Any)\nBase.:(+)\nBase.:-(::Any, ::Any)\nBase.:*(::Any, ::Any...)\nBase.:(/)\nBase.:\\(::Any, ::Any)\nBase.:^(::Number, ::Number)\nBase.fma\nBase.muladd\nBase.inv(::Number)\nBase.div\nBase.fld\nBase.cld\nBase.mod\nBase.rem\nBase.rem2pi\nBase.Math.mod2pi\nBase.divrem\nBase.fldmod\nBase.fld1\nBase.mod1\nBase.fldmod1\nBase.:(//)\nBase.rationalize\nBase.numerator\nBase.denominator\nBase.:(<<)\nBase.:(>>)\nBase.:(>>>)\nBase.bitrotate\nBase.:(:)\nBase.range\nBase.OneTo\nBase.StepRangeLen\nBase.:(==)\nBase.:(!=)\nBase.:(!==)\nBase.:(<)\nBase.:(<=)\nBase.:(>)\nBase.:(>=)\nBase.cmp\nBase.:(~)\nBase.:(&)\nBase.:(|)\nBase.xor\nBase.:(!)\n&&\n||","page":"Mathematics"},{"title":"Base.:-","location":"base/math.html#Base.:--Tuple{Any}","category":"method","text":"-(x)\n\nUnary minus operator.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:+","location":"base/math.html#Base.:+","category":"function","text":"dt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of the Time are used along with the year, month, and day of the Date to create the new DateTime. Non-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n+(x, y...)\n\nAddition operator. x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...).\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:-","location":"base/math.html#Base.:--Tuple{Any, Any}","category":"method","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:*","location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any, N} where N}","category":"method","text":"*(x, y...)\n\nMultiplication operator. x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...).\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:/","location":"base/math.html#Base.:/","category":"function","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:\\","location":"base/math.html#Base.:\\-Tuple{Any, Any}","category":"method","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:^","location":"base/math.html#Base.:^-Tuple{Number, Number}","category":"method","text":"^(x, y)\n\nExponentiation operator. If x is a matrix, computes matrix exponentiation.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code x^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to enable compile-time specialization on the value of the exponent. (As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y), where usually ^ == Base.^ unless ^ has been defined in the calling namespace.) If y is a negative integer literal, then Base.literal_pow transforms the operation to inv(x)^-y by default, where -y is positive.\n\njulia> 3^5\n243\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> A^3\n2×2 Matrix{Int64}:\n 37   54\n 81  118\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fma","location":"base/math.html#Base.fma","category":"function","text":"fma(x, y, z)\n\nComputes x*y+z without rounding the intermediate result x*y. On some systems this is significantly more expensive than x*y+z. fma is used to improve accuracy in certain algorithms. See muladd.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.muladd","location":"base/math.html#Base.muladd","category":"function","text":"muladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication. The result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\nmuladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged with each other or with surrounding operations for performance. For example, this may be implemented as an fma if the hardware supports it efficiently. The result can be different on different machines and can also be different on the same machine due to constant propagation or other optimizations. See fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.inv","location":"base/math.html#Base.inv-Tuple{Number}","category":"method","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x yields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for some types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.div","location":"base/math.html#Base.div","category":"function","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean division. Computes x/y, truncated to an integer.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fld","location":"base/math.html#Base.fld","category":"function","text":"fld(x, y)\n\nLargest integer less than or equal to x/y. Equivalent to div(x, y, RoundDown).\n\nSee also: div\n\nExamples\n\njulia> fld(7.3,5.5)\n1.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cld","location":"base/math.html#Base.cld","category":"function","text":"cld(x, y)\n\nSmallest integer larger than or equal to x/y. Equivalent to div(x, y, RoundUp).\n\nSee also: div\n\nExamples\n\njulia> cld(5.5,2.2)\n3.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.mod","location":"base/math.html#Base.mod","category":"function","text":"mod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x  y (mod n), where n = length(r), i.e. y = mod(x - first(r), n) + first(r).\n\nSee also: mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))\n3\n\njulia> mod(3, 0:2)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored division by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y, and magnitude less than abs(y) (with some exceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the type, and so rounding error may occur. In particular, if the exact result is very close to y, then it may be rounded to y.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\n\n\n\n\nrem(x::Integer, T::Type{<:Integer}) -> T\nmod(x::Integer, T::Type{<:Integer}) -> T\n%(x::Integer, T::Type{<:Integer}) -> T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable in T, and y is an integer in [typemin(T),typemax(T)]. If T can represent any integer (e.g. T == BigInt), then this operation corresponds to a conversion to T.\n\nExamples\n\njulia> 129 % Int8\n-127\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.rem","location":"base/math.html#Base.rem","category":"function","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in magnitude than y. This value is always exact.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.rem2pi","location":"base/math.html#Base.Math.rem2pi","category":"function","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of 2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally be the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,. or -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π. See also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0. See also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.mod2pi","location":"base/math.html#Base.Math.mod2pi","category":"function","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by numerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would compute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may be less than 2π. For example, the expression mod2pi(2π) will not return 0, because the intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See rem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.divrem","location":"base/math.html#Base.divrem","category":"function","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division. Equivalent to (div(x,y,r), rem(x,y,r)). Equivalently, with the default value of r, this call is equivalent to (x÷y, x%y).\n\nExamples\n\njulia> divrem(3,7)\n(0, 3)\n\njulia> divrem(7,3)\n(2, 1)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fldmod","location":"base/math.html#Base.fldmod","category":"function","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for divrem(x, y, RoundDown). Equivalent to (fld(x,y), mod(x,y)).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fld1","location":"base/math.html#Base.fld1","category":"function","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also: mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.mod1","location":"base/math.html#Base.mod1","category":"function","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y) in the range (0 y for positive y and in the range y0) for negative y.\n\nSee also: fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1(4, 3)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fldmod1","location":"base/math.html#Base.fldmod1","category":"function","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also: fld1, mod1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.://","location":"base/math.html#Base.://","category":"function","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.rationalize","location":"base/math.html#Base.rationalize","category":"function","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components of the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.numerator","location":"base/math.html#Base.numerator","category":"function","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.denominator","location":"base/math.html#Base.denominator","category":"function","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<<","location":"base/math.html#Base.:<<","category":"function","text":"<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left by n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0, this is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>.\n\n\n\n\n\n<<(B::BitVector, n) -> BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B with elements shifted n positions backwards, filling with false values. If n < 0, elements are shifted forwards. Equivalent to B >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>>","location":"base/math.html#Base.:>>","category":"function","text":">>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted right by n bits, where n >= 0, filling with 0s if x >= 0, 1s if x < 0, preserving the sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is equivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n>>(B::BitVector, n) -> BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B with elements shifted n positions forward, filling with false values. If n < 0, elements are shifted backwards. Equivalent to B << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>>>","location":"base/math.html#Base.:>>>","category":"function","text":">>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x shifted right by n bits, where n >= 0, filling with 0s. For n < 0, this is equivalent to x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For Signed integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this is equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n>>>(B::BitVector, n) -> BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for details and examples.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.bitrotate","location":"base/math.html#Base.bitrotate","category":"function","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation. It returns the value of x with its bits rotated left k times. A negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.::","location":"base/math.html#Base.::","category":"function","text":"(:)(start::CartesianIndex, [step::CartesianIndex], stop::CartesianIndex)\n\nConstruct CartesianIndices from two CartesianIndex and an optional step.\n\ncompat: Julia 1.1\nThis method requires at least Julia 1.1.\n\ncompat: Julia 1.6\nThe step range method start:step:stop requires at least Julia 1.6.\n\nExamples\n\njulia> I = CartesianIndex(2,1);\n\njulia> J = CartesianIndex(3,3);\n\njulia> I:J\n2×3 CartesianIndices{2, Tuple{UnitRange{Int64}, UnitRange{Int64}}}:\n CartesianIndex(2, 1)  CartesianIndex(2, 2)  CartesianIndex(2, 3)\n CartesianIndex(3, 1)  CartesianIndex(3, 2)  CartesianIndex(3, 3)\n\njulia> I:CartesianIndex(1, 2):J\n2×2 CartesianIndices{2, Tuple{StepRange{Int64, Int64}, StepRange{Int64, Int64}}}:\n CartesianIndex(2, 1)  CartesianIndex(2, 3)\n CartesianIndex(3, 1)  CartesianIndex(3, 3)\n\n\n\n\n\n(:)(start, [step], stop)\n\nRange operator. a:b constructs a range from a to b with a step size of 1 (a UnitRange) , and a:s:b is similar but uses a step size of s (a StepRange).\n\n: is also used in indexing to select whole dimensions  and for Symbol literals, as in e.g. :hello.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.range","location":"base/math.html#Base.range","category":"function","text":"range(start[, stop]; length, stop, step=1)\n\nGiven a starting value, construct a range either by length or from start to stop, optionally with a given step (defaults to 1, a UnitRange). One of length or stop is required.  If length, stop, and step are all specified, they must agree.\n\nIf length and stop are provided and step is not, the step size will be computed automatically such that there are length linearly spaced elements in the range.\n\nIf step and stop are provided and length is not, the overall range length will be computed automatically such that the elements are step spaced.\n\nSpecial care is taken to ensure intermediate values are computed rationally. To avoid this induced overhead, see the LinRange constructor.\n\nstop may be specified as either a positional or keyword argument.\n\ncompat: Julia 1.1\nstop as a positional argument requires at least Julia 1.1.\n\nExamples\n\njulia> range(1, length=100)\n1:100\n\njulia> range(1, stop=100)\n1:100\n\njulia> range(1, step=5, length=100)\n1:5:496\n\njulia> range(1, step=5, stop=100)\n1:5:96\n\njulia> range(1, 10, length=101)\n1.0:0.09:10.0\n\njulia> range(1, 100, step=5)\n1:5:96\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.OneTo","location":"base/math.html#Base.OneTo","category":"type","text":"Base.OneTo(n)\n\nDefine an AbstractUnitRange that behaves like 1:n, with the added distinction that the lower limit is guaranteed (by the type system) to be 1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.StepRangeLen","location":"base/math.html#Base.StepRangeLen","category":"type","text":"StepRangeLen{T,R,S}(ref::R, step::S, len, [offset=1]) where {T,R,S}\nStepRangeLen(       ref::R, step::S, len, [offset=1]) where {  R,S}\n\nA range r where r[i] produces values of type T (in the second form, T is deduced automatically), parameterized by a reference value, a step, and the length. By default ref is the starting value r[1], but alternatively you can supply it as the value of r[offset] for some other index 1 <= offset <= len. In conjunction with TwicePrecision this can be used to implement ranges that are free of roundoff error.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:==","location":"base/math.html#Base.:==","category":"function","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===. Should be implemented for all types with a notion of equality, based on the abstract value that an instance represents. For example, all numeric types are compared by numeric value, ignoring type. Strings are compared as sequences of characters, ignoring encoding. For collections, == is generally called recursively on all contents, though other properties (like the shape for arrays) may also be taken into account.\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and NaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing, in which case missing is returned (three-valued logic). For collections, missing is returned if at least one of the operands contains a missing value and all non-missing values are equal. Use isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and handle comparison to other types via promotion rules where possible.\n\nisequal falls back to ==, so new methods of == will be used by the Dict type to compare keys. If your type will be used as a dictionary key, it should therefore also implement hash.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:!=","location":"base/math.html#Base.:!=","category":"function","text":"!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition !=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n!=(x)\n\nCreate a function that compares its argument to x using !=, i.e. a function equivalent to y -> y != x. The returned function is of type Base.Fix2{typeof(!=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:!==","location":"base/math.html#Base.:!==","category":"function","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<","location":"base/math.html#Base.:<","category":"function","text":"<(x, y)\n\nLess-than comparison operator. Falls back to isless. Because of the behavior of floating-point NaN values, this operator implements a partial order.\n\nImplementation\n\nNew numeric types with a canonical partial order should implement this function for two arguments of the new type. Types with a canonical total order should implement isless instead. (x < y) | (x == y)\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n<(x)\n\nCreate a function that compares its argument to x using <, i.e. a function equivalent to y -> y < x. The returned function is of type Base.Fix2{typeof(<)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<=","location":"base/math.html#Base.:<=","category":"function","text":"<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n<=(x)\n\nCreate a function that compares its argument to x using <=, i.e. a function equivalent to y -> y <= x. The returned function is of type Base.Fix2{typeof(<=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>","location":"base/math.html#Base.:>","category":"function","text":">(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function, and rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n>(x)\n\nCreate a function that compares its argument to x using >, i.e. a function equivalent to y -> y > x. The returned function is of type Base.Fix2{typeof(>)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>=","location":"base/math.html#Base.:>=","category":"function","text":">=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n>=(x)\n\nCreate a function that compares its argument to x using >=, i.e. a function equivalent to y -> y >= x. The returned function is of type Base.Fix2{typeof(>=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cmp","location":"base/math.html#Base.cmp","category":"function","text":"cmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(a::AbstractString, b::AbstractString) -> Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:~","location":"base/math.html#Base.:~","category":"function","text":"~(x)\n\nBitwise not.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:&","location":"base/math.html#Base.:&","category":"function","text":"x & y\n\nBitwise and. Implements three-valued logic, returning missing if one operand is missing and the other is true. Add parentheses for function application form: (&)(x, y).\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:|","location":"base/math.html#Base.:|","category":"function","text":"x | y\n\nBitwise or. Implements three-valued logic, returning missing if one operand is missing and the other is false.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.xor","location":"base/math.html#Base.xor","category":"function","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and ⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:!","location":"base/math.html#Base.:!","category":"function","text":"!(x)\n\nBoolean not. Implements three-valued logic, returning missing if x is missing.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a function which computes the boolean negation of f.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\n\n\n\n\n","page":"Mathematics"},{"title":"&&","location":"base/math.html#&&","category":"keyword","text":"x && y\n\nShort-circuiting boolean AND.\n\n\n\n\n\n","page":"Mathematics"},{"title":"||","location":"base/math.html#||","category":"keyword","text":"x || y\n\nShort-circuiting boolean OR.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Mathematical Functions","location":"base/math.html#Mathematical-Functions","category":"section","text":"","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"Base.isapprox\nBase.sin(::Number)\nBase.cos(::Number)\nBase.sincos(::Float64)\nBase.tan(::Number)\nBase.Math.sind\nBase.Math.cosd\nBase.Math.tand\nBase.Math.sinpi\nBase.Math.cospi\nBase.Math.sincospi\nBase.sinh(::Number)\nBase.cosh(::Number)\nBase.tanh(::Number)\nBase.asin(::Number)\nBase.acos(::Number)\nBase.atan(::Number)\nBase.Math.asind\nBase.Math.acosd\nBase.Math.atand\nBase.Math.sec(::Number)\nBase.Math.csc(::Number)\nBase.Math.cot(::Number)\nBase.Math.secd\nBase.Math.cscd\nBase.Math.cotd\nBase.Math.asec(::Number)\nBase.Math.acsc(::Number)\nBase.Math.acot(::Number)\nBase.Math.asecd\nBase.Math.acscd\nBase.Math.acotd\nBase.Math.sech(::Number)\nBase.Math.csch(::Number)\nBase.Math.coth(::Number)\nBase.asinh(::Number)\nBase.acosh(::Number)\nBase.atanh(::Number)\nBase.Math.asech(::Number)\nBase.Math.acsch(::Number)\nBase.Math.acoth(::Number)\nBase.Math.sinc\nBase.Math.cosc\nBase.Math.deg2rad\nBase.Math.rad2deg\nBase.Math.hypot\nBase.log(::Number)\nBase.log(::Number, ::Number)\nBase.log2\nBase.log10\nBase.log1p\nBase.Math.frexp\nBase.exp(::Float64)\nBase.exp2\nBase.exp10\nBase.Math.ldexp\nBase.Math.modf\nBase.expm1\nBase.round(::Type, ::Any)\nBase.Rounding.RoundingMode\nBase.Rounding.RoundNearest\nBase.Rounding.RoundNearestTiesAway\nBase.Rounding.RoundNearestTiesUp\nBase.Rounding.RoundToZero\nBase.Rounding.RoundFromZero\nBase.Rounding.RoundUp\nBase.Rounding.RoundDown\nBase.round(::Complex{<: AbstractFloat}, ::RoundingMode, ::RoundingMode)\nBase.ceil\nBase.floor\nBase.trunc\nBase.unsafe_trunc\nBase.min\nBase.max\nBase.minmax\nBase.Math.clamp\nBase.Math.clamp!\nBase.abs\nBase.Checked.checked_abs\nBase.Checked.checked_neg\nBase.Checked.checked_add\nBase.Checked.checked_sub\nBase.Checked.checked_mul\nBase.Checked.checked_div\nBase.Checked.checked_rem\nBase.Checked.checked_fld\nBase.Checked.checked_mod\nBase.Checked.checked_cld\nBase.Checked.add_with_overflow\nBase.Checked.sub_with_overflow\nBase.Checked.mul_with_overflow\nBase.abs2\nBase.copysign\nBase.sign\nBase.signbit\nBase.flipsign\nBase.sqrt(::Real)\nBase.isqrt\nBase.Math.cbrt\nBase.real(::Complex)\nBase.imag\nBase.reim\nBase.conj\nBase.angle\nBase.cis\nBase.cispi\nBase.binomial\nBase.factorial\nBase.gcd\nBase.lcm\nBase.gcdx\nBase.ispow2\nBase.nextpow\nBase.prevpow\nBase.nextprod\nBase.invmod\nBase.powermod\nBase.ndigits\nBase.widemul\nBase.Math.evalpoly\nBase.Math.@evalpoly\nBase.FastMath.@fastmath","page":"Mathematics"},{"title":"Base.isapprox","location":"base/math.html#Base.isapprox","category":"function","text":"isapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison: true if norm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol is zero and the default rtol depends on the types of x and y. The keyword argument nans determines whether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to the square root of eps of the type of x or y, whichever is bigger (least precise). This corresponds to requiring equality of about half of the significand digits. Otherwise, e.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for arrays (where an alternative norm choice is sometimes useful). When x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf or NaN), the comparison falls back to checking whether all elements of x and y are approximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y is equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is equivalent to x == 0 since the default atol is 0.  In such cases, you should either supply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g. use x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol automatically because it depends on the overall scaling (the \"units\") of your problem: for example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the radius of the Earth in meters, but an absurdly large tolerance if x is the radius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments requires Julia 1.6 or later.\n\nExamples\n\njulia> 0.1 ≈ (0.1 - 1e-10)\ntrue\n\njulia> isapprox(10, 11; atol = 2)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0])\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\n\n\n\n\nisapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sin","location":"base/math.html#Base.sin-Tuple{Number}","category":"method","text":"sin(x)\n\nCompute sine of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cos","location":"base/math.html#Base.cos-Tuple{Number}","category":"method","text":"cos(x)\n\nCompute cosine of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sincos","location":"base/math.html#Base.Math.sincos-Tuple{Float64}","category":"method","text":"sincos(x)\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning a tuple (sine, cosine).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.tan","location":"base/math.html#Base.tan-Tuple{Number}","category":"method","text":"tan(x)\n\nCompute tangent of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sind","location":"base/math.html#Base.Math.sind","category":"function","text":"sind(x)\n\nCompute sine of x, where x is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cosd","location":"base/math.html#Base.Math.cosd","category":"function","text":"cosd(x)\n\nCompute cosine of x, where x is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.tand","location":"base/math.html#Base.Math.tand","category":"function","text":"tand(x)\n\nCompute tangent of x, where x is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sinpi","location":"base/math.html#Base.Math.sinpi","category":"function","text":"sinpi(x)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cospi","location":"base/math.html#Base.Math.cospi","category":"function","text":"cospi(x)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sincospi","location":"base/math.html#Base.Math.sincospi","category":"function","text":"sincospi(x)\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x, where x is in radians), returning a tuple (sine, cosine).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sinh","location":"base/math.html#Base.sinh-Tuple{Number}","category":"method","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cosh","location":"base/math.html#Base.cosh-Tuple{Number}","category":"method","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.tanh","location":"base/math.html#Base.tanh-Tuple{Number}","category":"method","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.asin","location":"base/math.html#Base.asin-Tuple{Number}","category":"method","text":"asin(x)\n\nCompute the inverse sine of x, where the output is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.acos","location":"base/math.html#Base.acos-Tuple{Number}","category":"method","text":"acos(x)\n\nCompute the inverse cosine of x, where the output is in radians\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.atan","location":"base/math.html#Base.atan-Tuple{Number}","category":"method","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one argument, this is the angle in radians between the positive x-axis and the point (1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the point (x, y), returning a value in the interval -pi pi. This corresponds to a standard atan2 function. Note that by convention atan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asind","location":"base/math.html#Base.Math.asind","category":"function","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acosd","location":"base/math.html#Base.Math.acosd","category":"function","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.atand","location":"base/math.html#Base.Math.atand","category":"function","text":"atand(y)\natand(y,x)\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sec","location":"base/math.html#Base.Math.sec-Tuple{Number}","category":"method","text":"sec(x)\n\nCompute the secant of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.csc","location":"base/math.html#Base.Math.csc-Tuple{Number}","category":"method","text":"csc(x)\n\nCompute the cosecant of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cot","location":"base/math.html#Base.Math.cot-Tuple{Number}","category":"method","text":"cot(x)\n\nCompute the cotangent of x, where x is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.secd","location":"base/math.html#Base.Math.secd","category":"function","text":"secd(x)\n\nCompute the secant of x, where x is in degrees.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cscd","location":"base/math.html#Base.Math.cscd","category":"function","text":"cscd(x)\n\nCompute the cosecant of x, where x is in degrees.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cotd","location":"base/math.html#Base.Math.cotd","category":"function","text":"cotd(x)\n\nCompute the cotangent of x, where x is in degrees.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asec","location":"base/math.html#Base.Math.asec-Tuple{Number}","category":"method","text":"asec(x)\n\nCompute the inverse secant of x, where the output is in radians. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acsc","location":"base/math.html#Base.Math.acsc-Tuple{Number}","category":"method","text":"acsc(x)\n\nCompute the inverse cosecant of x, where the output is in radians. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acot","location":"base/math.html#Base.Math.acot-Tuple{Number}","category":"method","text":"acot(x)\n\nCompute the inverse cotangent of x, where the output is in radians. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asecd","location":"base/math.html#Base.Math.asecd","category":"function","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acscd","location":"base/math.html#Base.Math.acscd","category":"function","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acotd","location":"base/math.html#Base.Math.acotd","category":"function","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sech","location":"base/math.html#Base.Math.sech-Tuple{Number}","category":"method","text":"sech(x)\n\nCompute the hyperbolic secant of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.csch","location":"base/math.html#Base.Math.csch-Tuple{Number}","category":"method","text":"csch(x)\n\nCompute the hyperbolic cosecant of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.coth","location":"base/math.html#Base.Math.coth-Tuple{Number}","category":"method","text":"coth(x)\n\nCompute the hyperbolic cotangent of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.asinh","location":"base/math.html#Base.asinh-Tuple{Number}","category":"method","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.acosh","location":"base/math.html#Base.acosh-Tuple{Number}","category":"method","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.atanh","location":"base/math.html#Base.atanh-Tuple{Number}","category":"method","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asech","location":"base/math.html#Base.Math.asech-Tuple{Number}","category":"method","text":"asech(x)\n\nCompute the inverse hyperbolic secant of x. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acsch","location":"base/math.html#Base.Math.acsch-Tuple{Number}","category":"method","text":"acsch(x)\n\nCompute the inverse hyperbolic cosecant of x. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acoth","location":"base/math.html#Base.Math.acoth-Tuple{Number}","category":"method","text":"acoth(x)\n\nCompute the inverse hyperbolic cotangent of x. \n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sinc","location":"base/math.html#Base.Math.sinc","category":"function","text":"sinc(x)\n\nCompute sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cosc","location":"base/math.html#Base.Math.cosc","category":"function","text":"cosc(x)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if x = 0. This is the derivative of sinc(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.deg2rad","location":"base/math.html#Base.Math.deg2rad","category":"function","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.rad2deg","location":"base/math.html#Base.Math.rad2deg","category":"function","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.hypot","location":"base/math.html#Base.Math.hypot","category":"function","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in: An Improved Algorithm for hypot(a,b) by Carlos F. Borges The article is available online at ArXiv at the link   https://arxiv.org/abs/1904.09481\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log","location":"base/math.html#Base.log-Tuple{Number}","category":"method","text":"log(x)\n\nCompute the natural logarithm of x. Throws DomainError for negative Real arguments. Use complex negative arguments to obtain complex results.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log","location":"base/math.html#Base.log-Tuple{Number, Number}","category":"method","text":"log(b,x)\n\nCompute the base b logarithm of x. Throws DomainError for negative Real arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will typically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log2","location":"base/math.html#Base.log2","category":"function","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throws DomainError for negative Real arguments.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] nan_dom_err at ./math.jl:325 [inlined]\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log10","location":"base/math.html#Base.log10","category":"function","text":"log10(x)\n\nCompute the logarithm of x to base 10. Throws DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] nan_dom_err at ./math.jl:325 [inlined]\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log1p","location":"base/math.html#Base.log1p","category":"function","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throws DomainError for Real arguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.frexp","location":"base/math.html#Base.Math.frexp","category":"function","text":"frexp(val)\n\nReturn (x,exp) such that x has a magnitude in the interval 12 1) or 0, and val is equal to x times 2^exp.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp","location":"base/math.html#Base.exp-Tuple{Float64}","category":"method","text":"exp(x)\n\nCompute the natural base exponential of x, in other words e^x.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp2","location":"base/math.html#Base.exp2","category":"function","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nExamples\n\njulia> exp2(5)\n32.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp10","location":"base/math.html#Base.exp10","category":"function","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.ldexp","location":"base/math.html#Base.Math.ldexp","category":"function","text":"ldexp(x, n)\n\nCompute x times 2^n.\n\nExamples\n\njulia> ldexp(5., 2)\n20.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.modf","location":"base/math.html#Base.Math.modf","category":"function","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts have the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.expm1","location":"base/math.html#Base.expm1","category":"function","text":"expm1(x)\n\nAccurately compute e^x-1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.round","location":"base/math.html#Base.round-Tuple{Type, Any}","category":"method","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type T, or of the same type of x if no T is provided. An InexactError will be thrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits after the decimal place (or before if negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of significant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer. Note that round may give incorrect results if the global rounding mode is changed (see rounding).\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when operating on binary floating point numbers. For example, the Float64 value represented by 1.15 is actually less than 1.15, yet will be rounded to 1.2.Examplesjulia> x = 1.15\n1.15\n\njulia> @sprintf \"%.20f\" x\n\"1.14999999999999991118\"\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundingMode","location":"base/math.html#Base.Rounding.RoundingMode","category":"type","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via rounding/setrounding functions), or as optional arguments for rounding to the nearest integer (via the round function).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero (BigFloat only)\nRoundUp\nRoundDown\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearest","location":"base/math.html#Base.Rounding.RoundNearest","category":"constant","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearestTiesAway","location":"base/math.html#Base.Rounding.RoundNearestTiesAway","category":"constant","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++ round behaviour).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearestTiesUp","location":"base/math.html#Base.Rounding.RoundNearestTiesUp","category":"constant","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript round behaviour).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundToZero","location":"base/math.html#Base.Rounding.RoundToZero","category":"constant","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundFromZero","location":"base/math.html#Base.Rounding.RoundFromZero","category":"constant","text":"RoundFromZero\n\nRounds away from zero. This rounding mode may only be used with T == BigFloat inputs to round.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundUp","location":"base/math.html#Base.Rounding.RoundUp","category":"constant","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundDown","location":"base/math.html#Base.Rounding.RoundDown","category":"constant","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.round","location":"base/math.html#Base.round-Tuple{Complex{var\"#s3\"} where var\"#s3\"<:AbstractFloat, RoundingMode, RoundingMode}","category":"method","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits=, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z, breaking ties using the specified RoundingModes. The first RoundingMode is used for rounding the real components while the second is used for rounding the imaginary components.\n\nExample\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ceil","location":"base/math.html#Base.ceil","category":"function","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or equal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the value is not representable.\n\ndigits, sigdigits and base work as for round.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.floor","location":"base/math.html#Base.floor","category":"function","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or equal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the value is not representable.\n\ndigits, sigdigits and base work as for round.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.trunc","location":"base/math.html#Base.trunc","category":"function","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value is less than or equal to x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the value is not representable.\n\ndigits, sigdigits and base work as for round.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.unsafe_trunc","location":"base/math.html#Base.unsafe_trunc","category":"function","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is less than or equal to x. If the value is not representable by T, an arbitrary value will be returned.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.min","location":"base/math.html#Base.min","category":"function","text":"min(x, y, ...)\n\nReturn the minimum of the arguments. See also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.max","location":"base/math.html#Base.max","category":"function","text":"max(x, y, ...)\n\nReturn the maximum of the arguments. See also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.minmax","location":"base/math.html#Base.minmax","category":"function","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)). See also: extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.clamp","location":"base/math.html#Base.Math.clamp","category":"function","text":"clamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments are promoted to a common type.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10.)], 2., 9.)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11,8,5],10,6) # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nExamples\n\njulia> clamp(200, Int8)\n127\njulia> clamp(-200, Int8)\n-128\n\n\n\n\n\nclamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.clamp!","location":"base/math.html#Base.Math.clamp!","category":"function","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place. See also clamp.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.abs","location":"base/math.html#Base.abs","category":"function","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur, resulting in the return of a negative value. This overflow occurs only when abs is applied to the minimum representable value of a signed integer. That is, when x == typemin(typeof(x)), abs(x) == x < 0, not -x as might be expected.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs(typemin(Int64))\n-9223372036854775808\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_abs","location":"base/math.html#Base.Checked.checked_abs","category":"function","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_neg","location":"base/math.html#Base.Checked.checked_neg","category":"function","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_add","location":"base/math.html#Base.Checked.checked_add","category":"function","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_sub","location":"base/math.html#Base.Checked.checked_sub","category":"function","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_mul","location":"base/math.html#Base.Checked.checked_mul","category":"function","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_div","location":"base/math.html#Base.Checked.checked_div","category":"function","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_rem","location":"base/math.html#Base.Checked.checked_rem","category":"function","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_fld","location":"base/math.html#Base.Checked.checked_fld","category":"function","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_mod","location":"base/math.html#Base.Checked.checked_mod","category":"function","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_cld","location":"base/math.html#Base.Checked.checked_cld","category":"function","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.add_with_overflow","location":"base/math.html#Base.Checked.add_with_overflow","category":"function","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.sub_with_overflow","location":"base/math.html#Base.Checked.sub_with_overflow","category":"function","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.mul_with_overflow","location":"base/math.html#Base.Checked.mul_with_overflow","category":"function","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.abs2","location":"base/math.html#Base.abs2","category":"function","text":"abs2(x)\n\nSquared absolute value of x.\n\nExamples\n\njulia> abs2(-3)\n9\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.copysign","location":"base/math.html#Base.copysign","category":"function","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sign","location":"base/math.html#Base.sign","category":"function","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.signbit","location":"base/math.html#Base.signbit","category":"function","text":"signbit(x)\n\nReturns true if the value of the sign of x is negative, otherwise false.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.flipsign","location":"base/math.html#Base.flipsign","category":"function","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sqrt","location":"base/math.html#Base.sqrt-Tuple{Real}","category":"method","text":"sqrt(x)\n\nReturn sqrtx. Throws DomainError for negative Real arguments. Use complex negative arguments instead. The prefix operator √ is equivalent to sqrt.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.isqrt","location":"base/math.html#Base.isqrt","category":"function","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cbrt","location":"base/math.html#Base.Math.cbrt","category":"function","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted (returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.real","location":"base/math.html#Base.real-Tuple{Complex}","category":"method","text":"real(z)\n\nReturn the real part of the complex number z.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.imag","location":"base/math.html#Base.imag","category":"function","text":"imag(z)\n\nReturn the imaginary part of the complex number z.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.reim","location":"base/math.html#Base.reim","category":"function","text":"reim(z)\n\nReturn both the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.conj","location":"base/math.html#Base.conj","category":"function","text":"conj(z)\n\nCompute the complex conjugate of a complex number z.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.angle","location":"base/math.html#Base.angle","category":"function","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 - im))\n-135.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cis","location":"base/math.html#Base.cis","category":"function","text":"cis(z)\n\nReturn exp(iz).\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cispi","location":"base/math.html#Base.cispi","category":"function","text":"cispi(z)\n\nCompute exp(ipi x) more accurately than cis(pi*x), especially for large x.\n\nExamples\n\njulia> cispi(1)\n-1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645952924 + 0.030556854645952924im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.binomial","location":"base/math.html#Base.binomial","category":"function","text":"binomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in the polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nSee also\n\nfactorial\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.factorial","location":"base/math.html#Base.factorial","category":"function","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an integer (promoted to at least 64 bits). Note that this may overflow if n is not small, but you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nSee also\n\nbinomial\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.gcd","location":"base/math.html#Base.gcd","category":"function","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero). The arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6,9)\n3\n\njulia> gcd(6,-9)\n3\n\njulia> gcd(6,0)\n6\n\njulia> gcd(0,0)\n0\n\njulia> gcd(1//3,2//3)\n1//3\n\njulia> gcd(1//3,-2//3)\n1//3\n\njulia> gcd(1//3,2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.lcm","location":"base/math.html#Base.lcm","category":"function","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero). The arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2,3)\n6\n\njulia> lcm(-2,3)\n6\n\njulia> lcm(0,3)\n0\n\njulia> lcm(0,0)\n0\n\njulia> lcm(1//3,2//3)\n2//3\n\njulia> lcm(1//3,-2//3)\n2//3\n\njulia> lcm(1//3,2)\n2//1\n\njulia> lcm(1,3,5,7)\n105\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.gcdx","location":"base/math.html#Base.gcdx","category":"function","text":"gcdx(a, b)\n\nComputes the greatest common (positive) divisor of a and b and their Bézout coefficients, i.e. the integer coefficients u and v that satisfy ua+vb = d = gcd(a b). gcdx(a b) returns (d u v).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal Bézout coefficients that are computed by the extended Euclidean algorithm. (Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.) For signed integers, these coefficients u and v are minimal in the sense that u  yd and v  xd. Furthermore, the signs of u and v are chosen so that d is positive. For unsigned integers, the coefficients u and v might be near their typemax, and the identity then holds only via the unsigned integers' modulo arithmetic.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ispow2","location":"base/math.html#Base.ispow2","category":"function","text":"ispow2(n::Number) -> Bool\n\nTest whether n is an integer power of two.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nextpow","location":"base/math.html#Base.nextpow","category":"function","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be greater than 1, and x must be greater than 0.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\nSee also prevpow.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.prevpow","location":"base/math.html#Base.prevpow","category":"function","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer. a must be greater than 1, and x must not be less than 1.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\nSee also nextpow.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nextprod","location":"base/math.html#Base.nextprod","category":"function","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers p_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.invmod","location":"base/math.html#Base.invmod","category":"function","text":"invmod(n, m)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m, and div(ym) = 0. This will throw an error if m = 0, or if gcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2,5)\n3\n\njulia> invmod(2,3)\n2\n\njulia> invmod(5,6)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.powermod","location":"base/math.html#Base.powermod","category":"function","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ndigits","location":"base/math.html#Base.ndigits","category":"function","text":"ndigits(n::Integer; base::Integer=10, pad::Integer=1)\n\nCompute the number of digits in integer n written in base base (base must not be in [-1, 0, 1]), optionally padded with zeros to a specified size (the result will never be less than pad).\n\nExamples\n\njulia> ndigits(12345)\n5\n\njulia> ndigits(1022, base=16)\n3\n\njulia> string(1022, base=16)\n\"3fe\"\n\njulia> ndigits(123, pad=5)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.widemul","location":"base/math.html#Base.widemul","category":"function","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nExamples\n\njulia> widemul(Float32(3.), 4.)\n12.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.evalpoly","location":"base/math.html#Base.Math.evalpoly","category":"function","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...; that is, the coefficients are given in ascending order by power of x. Loops are unrolled at compile time if the number of coefficients is statically known, i.e. when p is a Tuple. This function generates efficient code using Horner's method if x is real, or using a Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExample\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.@evalpoly","location":"base/math.html#Base.Math.@evalpoly","category":"macro","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...; that is, the coefficients are given in ascending order by power of z.  This macro expands to efficient inline code that uses either Horner's method or, for complex z, a more efficient Goertzel-like algorithm.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.FastMath.@fastmath","location":"base/math.html#Base.FastMath.@fastmath","category":"macro","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that may violate strict IEEE semantics. This allows the fastest possible operation, but results are undefined – be careful when doing this, as it may change numerical results.\n\nThis sets the LLVM Fast-Math flags, and corresponds to the -ffast-math option in clang. See the notes on performance annotations for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n","page":"Mathematics"},{"title":"Customizable binary operators","location":"base/math.html#Customizable-binary-operators","category":"section","text":"","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"Some unicode characters can be used to define new binary operators that support infix notation. For example ⊗(x,y) = kron(x,y) defines the ⊗ (otimes) function to be the Kronecker product, and one can call it as binary operator using infix syntax: C = A ⊗ B as well as with the usual prefix syntax C = ⊗(A,B).","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"Other characters that support such extensions include \\odot ⊙ and \\oplus ⊕","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"The complete list is in the parser code: https://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm","page":"Mathematics"},{"title":"Mathematics","location":"base/math.html","category":"page","text":"Those that are parsed like * (in terms of precedence) include * / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗ and those that are parsed like + include + - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣ There are many others that are related to arrows, comparisons, and powers.","page":"Mathematics"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html#isbits-Union-Optimizations","category":"section","text":"","page":"isbits Union Optimizations"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html","category":"page","text":"In Julia, the Array type holds both \"bits\" values as well as heap-allocated \"boxed\" values. The distinction is whether the value itself is stored inline (in the direct allocated memory of the array), or if the memory of the array is simply a collection of pointers to objects allocated elsewhere. In terms of performance, accessing values inline is clearly an advantage over having to follow a pointer to the actual value. The definition of \"isbits\" generally means any Julia type with a fixed, determinate size, meaning no \"pointer\" fields, see ?isbitstype.","page":"isbits Union Optimizations"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html","category":"page","text":"Julia also supports Union types, quite literally the union of a set of types. Custom Union type definitions can be extremely handy for applications wishing to \"cut across\" the nominal type system (i.e. explicit subtype relationships) and define methods or functionality on these, otherwise unrelated, set of types. A compiler challenge, however, is in determining how to treat these Union types. The naive approach (and indeed, what Julia itself did pre-0.7), is to simply make a \"box\" and then a pointer in the box to the actual value, similar to the previously mentioned \"boxed\" values. This is unfortunate, however, because of the number of small, primitive \"bits\" types (think UInt8, Int32, Float64, etc.) that would easily fit themselves inline in this \"box\" without needing any indirection for value access. There are two main ways Julia can take advantage of this optimization as of 0.7: isbits Union fields in types, and isbits Union Arrays.","page":"isbits Union Optimizations"},{"title":"isbits Union Structs","location":"devdocs/isbitsunionarrays.html#isbits-Union-Structs","category":"section","text":"","page":"isbits Union Optimizations"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html","category":"page","text":"Julia now includes an optimization wherein \"isbits Union\" fields in types (mutable struct, struct, etc.) will be stored inline. This is accomplished by determining the \"inline size\" of the Union type (e.g. Union{UInt8, Int16} will have a size of two bytes, which represents the size needed of the largest Union type Int16), and in addition, allocating an extra \"type tag byte\" (UInt8), whose value signals the type of the actual value stored inline of the \"Union bytes\". The type tag byte value is the index of the actual value's type in the Union type's order of types. For example, a type tag value of 0x02 for a field with type Union{Nothing, UInt8, Int16} would indicate that an Int16 value is stored in the 16 bits of the field in the structure's memory; a 0x01 value would indicate that a UInt8 value was stored in the first 8 bits of the 16 bits of the field's memory. Lastly, a value of 0x00 signals that the nothing value will be returned for this field, even though, as a singleton type with a single type instance, it technically has a size of 0. The type tag byte for a type's Union field is stored directly after the field's computed Union memory.","page":"isbits Union Optimizations"},{"title":"isbits Union Arrays","location":"devdocs/isbitsunionarrays.html#isbits-Union-Arrays","category":"section","text":"","page":"isbits Union Optimizations"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html","category":"page","text":"Julia can now also store \"isbits Union\" values inline in an Array, as opposed to requiring an indirection box. The optimization is accomplished by storing an extra \"type tag array\" of bytes, one byte per array element, alongside the bytes of the actual array data. This type tag array serves the same function as the type field case: its value signals the type of the actual stored Union value in the array. In terms of layout, a Julia Array can include extra \"buffer\" space before and after its actual data values, which are tracked in the a->offset and a->maxsize fields of the jl_array_t* type. The \"type tag array\" is treated exactly as another jl_array_t*, but which shares the same a->offset, a->maxsize, and a->len fields. So the formula to access an isbits Union Array's type tag bytes is a->data + (a->maxsize - a->offset) * a->elsize + a->offset; i.e. the Array's a->data pointer is already shifted by a->offset, so correcting for that, we follow the data all the way to the max of what it can hold a->maxsize, then adjust by a->offset more bytes to account for any present \"front buffering\" the array might be doing. This layout in particular allows for very efficient resizing operations as the type tag data only ever has to move when the actual array's data has to move.","page":"isbits Union Optimizations"},{"title":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray.  This page documents some of the design principles and implementation of SubArrays.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"One of the major design goals is to ensure high performance for views of both IndexLinear and IndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be IndexLinear to the extent that it is possible.","page":"SubArrays"},{"title":"Index replacement","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Consider making 2d slices of a 3d array:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"DocTestSetup = :(import Random; Random.seed!(1234))","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.200586  0.066423\n 0.298614  0.956753\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.200586  0.066423\n 0.246837  0.646691\n 0.648882  0.276021","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"DocTestSetup = nothing","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"view drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2 are two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j]. To extract the value from the parent array A, the natural approach is to replace S1[i,j] with A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]].","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The key feature of the design of SubArrays is that this index replacement can be performed without any runtime overhead.","page":"SubArrays"},{"title":"SubArray design","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":"","page":"SubArrays"},{"title":"Type parameters and fields","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The strategy adopted is first and foremost expressed in the definition of the type:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"struct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"SubArray has 5 type parameters.  The first two are the standard element type and dimensionality.  The next is the type of the parent AbstractArray.  The most heavily-used is the fourth parameter, a Tuple of the types of the indices for each dimension. The final one, L, is only provided as a convenience for dispatch; it's a boolean that represents whether the index types support fast linear indexing. More on that later.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"If in our example above A is a Array{Float64, 3}, our S1 case above would be a SubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}. Note in particular the tuple parameter, which stores the types of the indices used to create S1. Likewise,","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Storing these values allows index replacement, and having the types encoded as parameters allows one to dispatch to efficient algorithms.","page":"SubArrays"},{"title":"Index translation","location":"devdocs/subarrays.html#Index-translation","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Performing index translation requires that you do different things for different concrete SubArray types.  For example, for S1, one needs to apply the i,j indices to the first and third dimensions of the parent array, whereas for S2 one needs to apply them to the second and third.  The simplest approach to indexing would be to do the type-analysis at runtime:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"parentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Unfortunately, this would be disastrous in terms of performance: each element access would allocate memory, and involves the running of a lot of poorly-typed code.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"The better approach is to dispatch to specific methods to handle each type of stored index. That's what reindex does: it dispatches on the type of the first stored index and consumes the appropriate number of input indices, and then it recurses on the remaining indices. In the case of S1, this expands to","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Base.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"for any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below).","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is the core of a SubArray; indexing methods depend upon reindex to do this index translation. Sometimes, though, we can avoid the indirection and make it even faster.","page":"SubArrays"},{"title":"Linear indexing","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that separates successive elements, starting from some offset. This means that we can pre-compute these values and represent linear indexing simply as an addition and multiplication, avoiding the indirection of reindex and (more importantly) the slow computation of the cartesian coordinates entirely.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"For SubArray types, the availability of efficient linear indexing is based purely on the types of the indices, and does not depend on values like the size of the parent array. You can ask whether a given set of indices supports fast linear indexing with the internal Base.viewindexing function:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"This is computed during construction of the SubArray and stored in the L type parameter as a boolean that encodes fast linear indexing support. While not strictly necessary, it means that we can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries.","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Since this computation doesn't depend on runtime values, it can miss some cases in which the stride happens to be uniform:","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"A view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear indexing indeed could be performed efficiently.  However, success in this case depends on the size of the array: if the first dimension instead were odd,","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"julia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"then A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.  Since we have to base this decision based purely on types encoded in the parameters of the SubArray, S = view(A, 2:2:4, :) cannot implement efficient linear indexing.","page":"SubArrays"},{"title":"A few details","location":"devdocs/subarrays.html#A-few-details","category":"section","text":"","page":"SubArrays"},{"title":"SubArrays","location":"devdocs/subarrays.html","category":"page","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply determines how and where the stored indices should be reindexed. It not only supports integer indices, but it supports non-scalar indexing, too. This means that views of views don't need two levels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given by the parameter N, is not necessarily equal to the dimensionality of the parent array or the length of the indices tuple.  Neither do user-supplied indices necessarily line up with entries in the indices tuple (e.g., the second user-supplied index might correspond to the third dimension of the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal to the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting this dramatically complicates the reindexing process, especially for views of views. Not only do you need to dispatch on the types of the stored indices, but you need to examine whether a given index is the final one and \"merge\" any remaining stored indices together. This is not an easy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly if possible. Consequently, view ensures that the parent array is the appropriate dimensionality for the given indices by reshaping it if needed. The inner SubArray constructor ensures that this invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that reindex simply dispatches on the type of the stored indices in order to determine how many passed indices should be used and where they should go. But with CartesianIndex, there's no longer a one-to-one correspondence between the number of passed arguments and the number of dimensions that they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)), you can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1). It should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across all combinations of dimensionalities in an intractable manner. As such, reindex must never be called with CartesianIndex indices. Fortunately, the scalar case is easily handled by first flattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however, cannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view must ensure that there are no arrays of CartesianIndex in the argument list. If there are, it can simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray with two levels of indirection instead.","page":"SubArrays"},{"title":"Essentials","location":"base/base.html#Essentials","category":"section","text":"","page":"Essentials"},{"title":"Introduction","location":"base/base.html#Introduction","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Julia Base contains a range of functions and macros appropriate for performing scientific and numerical computing, but is also as broad as those of many general purpose programming languages.  Additional functionality is available from a growing collection of available packages. Functions are grouped by topic below.","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Some general notes:","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"To use module functions, use import Module to import the module, and Module.fn(x) to use the functions.\nAlternatively, using Module will import all exported Module functions into the current namespace.\nBy convention, function names ending with an exclamation point (!) modify their arguments. Some functions have both modifying (e.g., sort!) and non-modifying (sort) versions.","page":"Essentials"},{"title":"Getting Around","location":"base/base.html#Getting-Around","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.exit\nBase.atexit\nBase.isinteractive\nBase.summarysize\nBase.require\nBase.compilecache\nBase.__precompile__\nBase.include\nBase.MainInclude.include\nBase.include_string\nBase.include_dependency\nBase.which(::Any, ::Any)\nBase.methods\nBase.@show\nans\nBase.active_project","page":"Essentials"},{"title":"Base.exit","location":"base/base.html#Base.exit","category":"function","text":"exit(code=0)\n\nStop the program with an exit code. The default exit code is zero, indicating that the program completed successfully. In an interactive session, exit() can be called with the keyboard shortcut ^D.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.atexit","location":"base/base.html#Base.atexit","category":"function","text":"atexit(f)\n\nRegister a zero-argument function f() to be called at process exit. atexit() hooks are called in last in first out (LIFO) order and run before object finalizers.\n\nExit hooks are allowed to call exit(n), in which case Julia will exit with exit code n (instead of the original exit code). If more than one exit hook calls exit(n), then Julia will exit with the exit code corresponding to the last called exit hook that calls exit(n). (Because exit hooks are called in LIFO order, \"last called\" is equivalent to \"first registered\".)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isinteractive","location":"base/base.html#Base.isinteractive","category":"function","text":"isinteractive() -> Bool\n\nDetermine whether Julia is running an interactive session.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.summarysize","location":"base/base.html#Base.summarysize","category":"function","text":"Base.summarysize(obj; exclude=Union{...}, chargeall=Union{...}) -> Int\n\nCompute the amount of memory, in bytes, used by all unique objects reachable from the argument.\n\nKeyword Arguments\n\nexclude: specifies the types of objects to exclude from the traversal.\nchargeall: specifies the types of objects to always charge the size of all of their fields, even if those fields would normally be excluded.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.require","location":"base/base.html#Base.require","category":"function","text":"require(into::Module, module::Symbol)\n\nThis function is part of the implementation of using / import, if a module is not already defined in Main. It can also be called directly to force reloading a module, regardless of whether it has been loaded before (for example, when interactively developing libraries).\n\nLoads a source file, in the context of the Main module, on every active node, searching standard locations for files. require is considered a top-level operation, so it sets the current include path but does not use it to search for files (see help for include). This function is typically used to load library code, and is implicitly called by using to load packages.\n\nWhen searching for files, require first looks for package code in the global array LOAD_PATH. require is case-sensitive on all platforms, including those with case-insensitive filesystems like macOS and Windows.\n\nFor more details regarding code loading, see the manual sections on modules and parallel computing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.compilecache","location":"base/base.html#Base.compilecache","category":"function","text":"Base.compilecache(module::PkgId)\n\nCreates a precompiled cache file for a module and all of its dependencies. This can be used to reduce package load times. Cache files are stored in DEPOT_PATH[1]/compiled. See Module initialization and precompilation for important notes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.__precompile__","location":"base/base.html#Base.__precompile__","category":"function","text":"__precompile__(isprecompilable::Bool)\n\nSpecify whether the file calling this function is precompilable, defaulting to true. If a module or file is not safely precompilable, it should call __precompile__(false) in order to throw an error if Julia attempts to precompile it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include","location":"base/base.html#Base.include","category":"function","text":"Base.include([mapexpr::Function,] [m::Module,] path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of module m. Every module (except those defined with baremodule) has its own definition of include omitting the m argument, which evaluates the file in that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MainInclude.include","location":"base/base.html#Base.MainInclude.include","category":"function","text":"include([mapexpr::Function,] path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of the containing module. Every module (except those defined with baremodule) has its own definition of include, which evaluates the file in that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\nUse Base.include to evaluate a file into another module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include_string","location":"base/base.html#Base.include_string","category":"function","text":"include_string([mapexpr::Function,] m::Module, code::AbstractString, filename::AbstractString=\"string\")\n\nLike include, except reads code from the given string rather than from a file.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in code, the include_string function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include_dependency","location":"base/base.html#Base.include_dependency","category":"function","text":"include_dependency(path::AbstractString)\n\nIn a module, declare that the file specified by path (relative or absolute) is a dependency for precompilation; that is, the module will need to be recompiled if this file changes.\n\nThis is only needed if your module depends on a file that is not used via include. It has no effect outside of compilation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.which","location":"base/base.html#Base.which-Tuple{Any, Any}","category":"method","text":"which(f, types)\n\nReturns the method of f (a Method object) that would be called for arguments of the given types.\n\nIf types is an abstract type, then the method that would be called by invoke is returned.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.methods","location":"base/base.html#Base.methods","category":"function","text":"methods(f, [types], [module])\n\nReturn the method table for f.\n\nIf types is specified, return an array of methods whose types match. If module is specified, return an array of methods defined in that module. A list of modules can also be specified as an array.\n\ncompat: Julia 1.4\nAt least Julia 1.4 is required for specifying a module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@show","location":"base/base.html#Base.@show","category":"macro","text":"@show\n\nShow an expression and result, returning the result. See also show.\n\n\n\n\n\n","page":"Essentials"},{"title":"ans","location":"base/base.html#ans","category":"keyword","text":"ans\n\nA variable referring to the last computed value, automatically set at the interactive prompt.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.active_project","location":"base/base.html#Base.active_project","category":"function","text":"active_project()\n\nReturn the path of the active Project.toml file.\n\n\n\n\n\n","page":"Essentials"},{"title":"Keywords","location":"base/base.html#Keywords","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"This is the list of reserved keywords in Julia: baremodule, begin, break, catch, const, continue, do, else, elseif, end, export, false, finally, for, function, global, if, import, let, local, macro, module, quote, return, struct, true, try, using, while. Those keywords are not allowed to be used as variable names.","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"The following two-word sequences are reserved: abstract type, mutable struct, primitive type. However, you can create variables with names: abstract, mutable, primitive and type.","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Finally, where is parsed as an infix operator for writing parametric method and type definitions. Also in and isa are parsed as infix operators. Creation of a variable named where, in or isa is allowed though.","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"module\nexport\nimport\nusing\nbaremodule\nfunction\nmacro\nreturn\ndo\nbegin\nend\nlet\nif\nfor\nwhile\nbreak\ncontinue\ntry\nfinally\nquote\nlocal\nglobal\nconst\nstruct\nmutable struct\nabstract type\nprimitive type\nwhere\n...\n;\n=\n?:","page":"Essentials"},{"title":"module","location":"base/base.html#module","category":"keyword","text":"module\n\nmodule declares a Module, which is a separate global variable workspace. Within a module, you can control which names from other modules are visible (via importing), and specify which of your names are intended to be public (via exporting). Modules allow you to create top-level definitions without worrying about name conflicts when your code is used together with somebody else’s. See the manual section about modules for more details.\n\nExamples\n\nmodule Foo\nimport Base.show\nexport MyType, foo\n\nstruct MyType\n    x\nend\n\nbar(x) = 2x\nfoo(a::MyType) = bar(a.x) + 1\nshow(io::IO, a::MyType) = print(io, \"MyType $(a.x)\")\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"export","location":"base/base.html#export","category":"keyword","text":"export\n\nexport is used within modules to tell Julia which functions should be made available to the user. For example: export foo makes the name foo available when using the module. See the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"import","location":"base/base.html#import","category":"keyword","text":"import\n\nimport Foo will load the module or package Foo. Names from the imported Foo module can be accessed with dot syntax (e.g. Foo.foo to access the name foo). See the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"using","location":"base/base.html#using","category":"keyword","text":"using\n\nusing Foo will load the module or package Foo and make its exported names available for direct use. Names can also be used via dot syntax (e.g. Foo.foo to access the name foo), whether they are exported or not. See the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"baremodule","location":"base/base.html#baremodule","category":"keyword","text":"baremodule\n\nbaremodule declares a module that does not contain using Base or local definitions of eval and include. It does still import Core. In other words,\n\nmodule Mod\n\n...\n\nend\n\nis equivalent to\n\nbaremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"function","location":"base/base.html#function","category":"keyword","text":"function\n\nFunctions are defined with the function keyword:\n\nfunction add(a, b)\n    return a + b\nend\n\nOr the short form notation:\n\nadd(a, b) = a + b\n\nThe use of the return keyword is exactly the same as in other languages, but is often optional. A function without an explicit return statement will return the last expression in the function body.\n\n\n\n\n\n","page":"Essentials"},{"title":"macro","location":"base/base.html#macro","category":"keyword","text":"macro\n\nmacro defines a method for inserting generated code into a program. A macro maps a sequence of argument expressions to a returned expression, and the resulting expression is substituted directly into the program at the point where the macro is invoked. Macros are a way to run generated code without calling eval, since the generated code instead simply becomes part of the surrounding program. Macro arguments may include expressions, literal values, and symbols. Macros can be defined for variable number of arguments (varargs), but do not accept keyword arguments. Every macro also implicitly gets passed the arguments __source__, which contains the line number and file name the macro is called from, and __module__, which is the module the macro is expanded in.\n\nExamples\n\njulia> macro sayhello(name)\n           return :( println(\"Hello, \", $name, \"!\") )\n       end\n@sayhello (macro with 1 method)\n\njulia> @sayhello \"Charlie\"\nHello, Charlie!\n\njulia> macro saylots(x...)\n           return :( println(\"Say: \", $(x...)) )\n       end\n@saylots (macro with 1 method)\n\njulia> @saylots \"hey \" \"there \" \"friend\"\nSay: hey there friend\n\n\n\n\n\n","page":"Essentials"},{"title":"return","location":"base/base.html#return","category":"keyword","text":"return\n\nreturn x causes the enclosing function to exit early, passing the given value x back to its caller. return by itself with no value is equivalent to return nothing (see nothing).\n\nfunction compare(a, b)\n    a == b && return \"equal to\"\n    a < b ? \"less than\" : \"greater than\"\nend\n\nIn general you can place a return statement anywhere within a function body, including within deeply nested loops or conditionals, but be careful with do blocks. For example:\n\nfunction test1(xs)\n    for x in xs\n        iseven(x) && return 2x\n    end\nend\n\nfunction test2(xs)\n    map(xs) do x\n        iseven(x) && return 2x\n        x\n    end\nend\n\nIn the first example, the return breaks out of test1 as soon as it hits an even number, so test1([5,6,7]) returns 12.\n\nYou might expect the second example to behave the same way, but in fact the return there only breaks out of the inner function (inside the do block) and gives a value back to map. test2([5,6,7]) then returns [5,12,7].\n\nWhen used in a top-level expression (i.e. outside any function), return causes the entire current top-level expression to terminate early.\n\n\n\n\n\n","page":"Essentials"},{"title":"do","location":"base/base.html#do","category":"keyword","text":"do\n\nCreate an anonymous function and pass it as the first argument to a function call. For example:\n\nmap(1:10) do x\n    2x\nend\n\nis equivalent to map(x->2x, 1:10).\n\nUse multiple arguments like so:\n\nmap(1:10, 11:20) do x, y\n    x + y\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"begin","location":"base/base.html#begin","category":"keyword","text":"begin\n\nbegin...end denotes a block of code.\n\nbegin\n    println(\"Hello, \")\n    println(\"World!\")\nend\n\nUsually begin will not be necessary, since keywords such as function and let implicitly begin blocks of code. See also ;.\n\n\n\n\n\n","page":"Essentials"},{"title":"end","location":"base/base.html#end","category":"keyword","text":"end\n\nend marks the conclusion of a block of expressions, for example module, struct, mutable struct, begin, let, for etc. end may also be used when indexing into an array to represent the last index of a dimension.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Array{Int64, 2}:\n 1  2\n 3  4\n\njulia> A[end, :]\n2-element Array{Int64, 1}:\n 3\n 4\n\n\n\n\n\n","page":"Essentials"},{"title":"let","location":"base/base.html#let","category":"keyword","text":"let\n\nlet statements allocate new variable bindings each time they run. Whereas an assignment modifies an existing value location, let creates new locations. This difference is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:\n\nlet var1 = value1, var2, var3 = value3\n    code\nend\n\nThe assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x, since the two x variables are distinct and have separate storage.\n\n\n\n\n\n","page":"Essentials"},{"title":"if","location":"base/base.html#if","category":"keyword","text":"if/elseif/else\n\nif/elseif/else performs conditional evaluation, which allows portions of code to be evaluated or not evaluated depending on the value of a boolean expression. Here is the anatomy of the if/elseif/else conditional syntax:\n\nif x < y\n    println(\"x is less than y\")\nelseif x > y\n    println(\"x is greater than y\")\nelse\n    println(\"x is equal to y\")\nend\n\nIf the condition expression x < y is true, then the corresponding block is evaluated; otherwise the condition expression x > y is evaluated, and if it is true, the corresponding block is evaluated; if neither expression is true, the else block is evaluated. The elseif and else blocks are optional, and as many elseif blocks as desired can be used.\n\nIn contrast to some other languages conditions must be of type Bool. It does not suffice for conditions to be convertible to Bool.\n\njulia> if 1 end\nERROR: TypeError: non-boolean (Int64) used in boolean context\n\n\n\n\n\n","page":"Essentials"},{"title":"for","location":"base/base.html#for","category":"keyword","text":"for\n\nfor loops repeatedly evaluate a block of statements while iterating over a sequence of values.\n\nExamples\n\njulia> for i in [1, 4, 0]\n           println(i)\n       end\n1\n4\n0\n\n\n\n\n\n","page":"Essentials"},{"title":"while","location":"base/base.html#while","category":"keyword","text":"while\n\nwhile loops repeatedly evaluate a conditional expression, and continue evaluating the body of the while loop as long as the expression remains true. If the condition expression is false when the while loop is first reached, the body is never evaluated.\n\nExamples\n\njulia> i = 1\n1\n\njulia> while i < 5\n           println(i)\n           global i += 1\n       end\n1\n2\n3\n4\n\n\n\n\n\n","page":"Essentials"},{"title":"break","location":"base/base.html#break","category":"keyword","text":"break\n\nBreak out of a loop immediately.\n\nExamples\n\njulia> i = 0\n0\n\njulia> while true\n           global i += 1\n           i > 5 && break\n           println(i)\n       end\n1\n2\n3\n4\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"continue","location":"base/base.html#continue","category":"keyword","text":"continue\n\nSkip the rest of the current loop iteration.\n\nExamples\n\njulia> for i = 1:6\n           iseven(i) && continue\n           println(i)\n       end\n1\n3\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"try","location":"base/base.html#try","category":"keyword","text":"try/catch\n\nA try/catch statement allows intercepting errors (exceptions) thrown by throw so that program execution can continue. For example, the following code attempts to write a file, but warns the user and proceeds instead of terminating execution if the file cannot be written:\n\ntry\n    open(\"/danger\", \"w\") do f\n        println(f, \"Hello\")\n    end\ncatch\n    @warn \"Could not write file.\"\nend\n\nor, when the file cannot be read into a variable:\n\nlines = try\n    open(\"/danger\", \"r\") do f\n        readlines(f)\n    end\ncatch\n    @warn \"File not found.\"\nend\n\nThe syntax catch e (where e is any variable) assigns the thrown exception object to the given variable within the catch block.\n\nThe power of the try/catch construct lies in the ability to unwind a deeply nested computation immediately to a much higher level in the stack of calling functions.\n\n\n\n\n\n","page":"Essentials"},{"title":"finally","location":"base/base.html#finally","category":"keyword","text":"finally\n\nRun some code when a given block of code exits, regardless of how it exits. For example, here is how we can guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    operate_on_file(f)\nfinally\n    close(f)\nend\n\nWhen control leaves the try block (for example, due to a return, or just finishing normally), close(f) will be executed. If the try block exits due to an exception, the exception will continue propagating. A catch block may be combined with try and finally as well. In this case the finally block will run after catch has handled the error.\n\n\n\n\n\n","page":"Essentials"},{"title":"quote","location":"base/base.html#quote","category":"keyword","text":"quote\n\nquote creates multiple expression objects in a block without using the explicit Expr constructor. For example:\n\nex = quote\n    x = 1\n    y = 2\n    x + y\nend\n\nUnlike the other means of quoting, :( ... ), this form introduces QuoteNode elements to the expression tree, which must be considered when directly manipulating the tree. For other purposes, :( ... ) and quote .. end blocks are treated identically.\n\n\n\n\n\n","page":"Essentials"},{"title":"local","location":"base/base.html#local","category":"keyword","text":"local\n\nlocal introduces a new local variable. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> function foo(n)\n           x = 0\n           for i = 1:n\n               local x # introduce a loop-local x\n               x = i\n           end\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> foo(10)\n0\n\n\n\n\n\n","page":"Essentials"},{"title":"global","location":"base/base.html#global","category":"keyword","text":"global\n\nglobal x makes x in the current scope and its inner scopes refer to the global variable of that name. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> z = 3\n3\n\njulia> function foo()\n           global z = 6 # use the z variable defined outside foo\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n6\n\njulia> z\n6\n\n\n\n\n\n","page":"Essentials"},{"title":"const","location":"base/base.html#const","category":"keyword","text":"const\n\nconst is used to declare global variables whose values will not change. In almost all code (and particularly performance sensitive code) global variables should be declared constant in this way.\n\nconst x = 5\n\nMultiple variables can be declared within a single const:\n\nconst y, z = 7, 11\n\nNote that const only applies to one = operation, therefore const x = y = 1 declares x to be constant but not y. On the other hand, const x = const y = 1 declares both x and y constant.\n\nNote that \"constant-ness\" does not extend into mutable containers; only the association between a variable and its value is constant. If x is an array or dictionary (for example) you can still modify, add, or remove elements.\n\nIn some cases changing the value of a const variable gives a warning instead of an error. However, this can produce unpredictable behavior or corrupt the state of your program, and so should be avoided. This feature is intended only for convenience during interactive use.\n\n\n\n\n\n","page":"Essentials"},{"title":"struct","location":"base/base.html#struct","category":"keyword","text":"struct\n\nThe most commonly used kind of type in Julia is a struct, specified as a name and a set of fields.\n\nstruct Point\n    x\n    y\nend\n\nFields can have type restrictions, which may be parameterized:\n\nstruct Point{X}\n    x::X\n    y::Float64\nend\n\nA struct can also declare an abstract super type via <: syntax:\n\nstruct Point <: AbstractPoint\n    x\n    y\nend\n\nstructs are immutable by default; an instance of one of these types cannot be modified after construction. Use mutable struct instead to declare a type whose instances can be modified.\n\nSee the manual section on Composite Types for more details, such as how to define constructors.\n\n\n\n\n\n","page":"Essentials"},{"title":"mutable struct","location":"base/base.html#mutable struct","category":"keyword","text":"mutable struct\n\nmutable struct is similar to struct, but additionally allows the fields of the type to be set after construction. See the manual section on Composite Types for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"abstract type","location":"base/base.html#abstract type","category":"keyword","text":"abstract type\n\nabstract type declares a type that cannot be instantiated, and serves only as a node in the type graph, thereby describing sets of related concrete types: those concrete types which are their descendants. Abstract types form the conceptual hierarchy which makes Julia’s type system more than just a collection of object implementations. For example:\n\nabstract type Number end\nabstract type Real <: Number end\n\nNumber has no supertype, whereas Real is an abstract subtype of Number.\n\n\n\n\n\n","page":"Essentials"},{"title":"primitive type","location":"base/base.html#primitive type","category":"keyword","text":"primitive type\n\nprimitive type declares a concrete type whose data consists only of a series of bits. Classic examples of primitive types are integers and floating-point values. Some example built-in primitive type declarations:\n\nprimitive type Char 32 end\nprimitive type Bool <: Integer 8 end\n\nThe number after the name indicates how many bits of storage the type requires. Currently, only sizes that are multiples of 8 bits are supported. The Bool declaration shows how a primitive type can be optionally declared to be a subtype of some supertype.\n\n\n\n\n\n","page":"Essentials"},{"title":"where","location":"base/base.html#where","category":"keyword","text":"where\n\nThe where keyword creates a type that is an iterated union of other types, over all values of some variable. For example Vector{T} where T<:Real includes all Vectors where the element type is some kind of Real number.\n\nThe variable bound defaults to Any if it is omitted:\n\nVector{T} where T    # short for `where T<:Any`\n\nVariables can also have lower bounds:\n\nVector{T} where T>:Int\nVector{T} where Int<:T<:Real\n\nThere is also a concise syntax for nested where expressions. For example, this:\n\nPair{T, S} where S<:Array{T} where T<:Number\n\ncan be shortened to:\n\nPair{T, S} where {T<:Number, S<:Array{T}}\n\nThis form is often found on method signatures.\n\nNote that in this form, the variables are listed outermost-first. This matches the order in which variables are substituted when a type is \"applied\" to parameter values using the syntax T{p1, p2, ...}.\n\n\n\n\n\n","page":"Essentials"},{"title":"...","location":"base/base.html#...","category":"keyword","text":"...\n\nThe \"splat\" operator, ..., represents a sequence of arguments. ... can be used in function definitions, to indicate that the function accepts an arbitrary number of arguments. ... can also be used to apply a function to a sequence of arguments.\n\nExamples\n\njulia> add(xs...) = reduce(+, xs)\nadd (generic function with 1 method)\n\njulia> add(1, 2, 3, 4, 5)\n15\n\njulia> add([1, 2, 3]...)\n6\n\njulia> add(7, 1:100..., 1000:1100...)\n111107\n\n\n\n\n\n","page":"Essentials"},{"title":";","location":"base/base.html#;","category":"keyword","text":";\n\n; has a similar role in Julia as in many C-like languages, and is used to delimit the end of the previous statement. ; is not necessary after new lines, but can be used to separate statements on a single line or to join statements into a single expression. ; is also used to suppress output printing in the REPL and similar interfaces.\n\nExamples\n\njulia> function foo()\n           x = \"Hello, \"; x *= \"World!\"\n           return x\n       end\nfoo (generic function with 1 method)\n\njulia> bar() = (x = \"Hello, Mars!\"; return x)\nbar (generic function with 1 method)\n\njulia> foo();\n\njulia> bar()\n\"Hello, Mars!\"\n\n\n\n\n\n","page":"Essentials"},{"title":"=","location":"base/base.html#=","category":"keyword","text":"=\n\n= is the assignment operator.\n\nFor variable a and expression b, a = b makes a refer to the value of b.\nFor functions f(x), f(x) = x defines a new function constant f, or adds a new method to f if f is already defined; this usage is equivalent to function f(x); x; end.\na[i] = v calls setindex!(a,v,i).\na.b = c calls setproperty!(a,:b,c).\nInside a function call, f(a=b) passes b as the value of keyword argument a.\nInside parentheses with commas, (a=1,) constructs a NamedTuple.\n\nExamples\n\nAssigning a to b does not create a copy of b; instead use copy or deepcopy.\n\njulia> b = [1]; a = b; b[1] = 2; a\n1-element Array{Int64, 1}:\n 2\n\njulia> b = [1]; a = copy(b); b[1] = 2; a\n1-element Array{Int64, 1}:\n 1\n\n\nCollections passed to functions are also not copied. Functions can modify (mutate) the contents of the objects their arguments refer to. (The names of functions which do this are conventionally suffixed with '!'.)\n\njulia> function f!(x); x[:] .+= 1; end\nf! (generic function with 1 method)\n\njulia> a = [1]; f!(a); a\n1-element Array{Int64, 1}:\n 2\n\n\nAssignment can operate on multiple variables in parallel, taking values from an iterable:\n\njulia> a, b = 4, 5\n(4, 5)\n\njulia> a, b = 1:3\n1:3\n\njulia> a, b\n(1, 2)\n\n\nAssignment can operate on multiple variables in series, and will return the value of the right-hand-most expression:\n\njulia> a = [1]; b = [2]; c = [3]; a = b = c\n1-element Array{Int64, 1}:\n 3\n\njulia> b[1] = 2; a, b, c\n([2], [2], [2])\n\n\nAssignment at out-of-bounds indices does not grow a collection. If the collection is a Vector it can instead be grown with push! or append!.\n\njulia> a = [1, 1]; a[3] = 2\nERROR: BoundsError: attempt to access 2-element Array{Int64, 1} at index [3]\n[...]\n\njulia> push!(a, 2, 3)\n4-element Array{Int64, 1}:\n 1\n 1\n 2\n 3\n\n\nAssigning [] does not eliminate elements from a collection; instead use filter!.\n\njulia> a = collect(1:3); a[a .<= 1] = []\nERROR: DimensionMismatch(\"tried to assign 0 elements to 1 destinations\")\n[...]\n\njulia> filter!(x -> x > 1, a) # in-place & thus more efficient than a = a[a .> 1]\n2-element Array{Int64, 1}:\n 2\n 3\n\n\n\n\n\n\n","page":"Essentials"},{"title":"?:","location":"base/base.html#?:","category":"keyword","text":"a ? b : c\n\nShort form for conditionals; read \"if a, evaluate b otherwise evaluate c\". Also known as the ternary operator.\n\nThis syntax is equivalent to if a; b else c end, but is often used to emphasize the value b-or-c which is being used as part of a larger expression, rather than the side effects that evaluating b or c may have.\n\nSee the manual section on control flow for more details.\n\nExamples\n\njulia> x = 1; y = 2;\n\njulia> println(x > y ? \"x is larger\" : \"y is larger\")\ny is larger\n\n\n\n\n\n","page":"Essentials"},{"title":"Standard Modules","location":"base/base.html#Standard-Modules","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Main\nCore\nBase","page":"Essentials"},{"title":"Main","location":"base/base.html#Main","category":"module","text":"Main\n\nMain is the top-level module, and Julia starts with Main set as the current module.  Variables defined at the prompt go in Main, and varinfo lists variables in Main.\n\njulia> @__MODULE__\nMain\n\n\n\n\n\n","page":"Essentials"},{"title":"Core","location":"base/base.html#Core","category":"module","text":"Core\n\nCore is the module that contains all identifiers considered \"built in\" to the language, i.e. part of the core language and not libraries. Every module implicitly specifies using Core, since you can't do anything without those definitions.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base","location":"base/base.html#Base","category":"module","text":"Base\n\nThe base library of Julia. Base is a module that contains basic functionality (the contents of base/). All modules implicitly contain using Base, since this is needed in the vast majority of cases.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base Submodules","location":"base/base.html#Base-Submodules","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.Broadcast\nBase.Docs\nBase.Iterators\nBase.Libc\nBase.Meta\nBase.StackTraces\nBase.Sys\nBase.Threads\nBase.GC","page":"Essentials"},{"title":"Base.Broadcast","location":"base/base.html#Base.Broadcast","category":"module","text":"Base.Broadcast\n\nModule containing the broadcasting implementation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs","location":"base/base.html#Base.Docs","category":"module","text":"Docs\n\nThe Docs module provides the @doc macro which can be used to set and retrieve documentation metadata for Julia objects.\n\nPlease see the manual section on documentation for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Iterators","location":"base/base.html#Base.Iterators","category":"module","text":"Methods for working with Iterators.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc","location":"base/base.html#Base.Libc","category":"module","text":"Interface to libc, the C standard library.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta","location":"base/base.html#Base.Meta","category":"module","text":"Convenience functions for metaprogramming.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.StackTraces","location":"base/base.html#Base.StackTraces","category":"module","text":"Tools for collecting and manipulating stack traces. Mainly used for building errors.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys","location":"base/base.html#Base.Sys","category":"module","text":"Provide methods for retrieving information about hardware and the operating system.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Threads","location":"base/base.html#Base.Threads","category":"module","text":"Multithreading support.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC","location":"base/base.html#Base.GC","category":"module","text":"Base.GC\n\nModule with garbage collection utilities.\n\n\n\n\n\n","page":"Essentials"},{"title":"All Objects","location":"base/base.html#All-Objects","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Core.:(===)\nCore.isa\nBase.isequal\nBase.isless\nBase.ifelse\nCore.typeassert\nCore.typeof\nCore.tuple\nBase.ntuple\nBase.objectid\nBase.hash\nBase.finalizer\nBase.finalize\nBase.copy\nBase.deepcopy\nBase.getproperty\nBase.setproperty!\nBase.propertynames\nBase.hasproperty\nCore.getfield\nCore.setfield!\nCore.isdefined\nBase.@isdefined\nBase.convert\nBase.promote\nBase.oftype\nBase.widen\nBase.identity","page":"Essentials"},{"title":"Core.:===","location":"base/base.html#Core.:===","category":"function","text":"===(x,y) -> Bool\n≡(x,y) -> Bool\n\nDetermine whether x and y are identical, in the sense that no program could distinguish them. First the types of x and y are compared. If those are identical, mutable objects are compared by address in memory and immutable objects (such as numbers) are compared by contents at the bit level. This function is sometimes called \"egal\". It always returns a Bool value.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a == b\ntrue\n\njulia> a === b\nfalse\n\njulia> a === a\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.isa","location":"base/base.html#Core.isa","category":"function","text":"isa(x, type) -> Bool\n\nDetermine whether x is of the given type. Can also be used as an infix operator, e.g. x isa type.\n\nExamples\n\njulia> isa(1, Int)\ntrue\n\njulia> isa(1, Matrix)\nfalse\n\njulia> isa(1, Char)\nfalse\n\njulia> isa(1, Number)\ntrue\n\njulia> 1 isa Number\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isequal","location":"base/base.html#Base.isequal","category":"function","text":"isequal(x, y)\n\nSimilar to ==, except for the treatment of floating point numbers and of missing values. isequal treats all floating-point NaN values as equal to each other, treats -0.0 as unequal to 0.0, and missing as equal to missing. Always returns a Bool value.\n\nImplementation\n\nThe default implementation of isequal calls ==, so a type that does not involve floating-point values generally only needs to define ==.\n\nisequal is the comparison function used by hash tables (Dict). isequal(x,y) must imply that hash(x) == hash(y).\n\nThis typically means that types for which a custom == or isequal method exists must implement a corresponding hash method (and vice versa). Collections typically implement isequal by calling isequal recursively on all contents.\n\nScalar types generally do not need to implement isequal separate from ==, unless they represent floating-point numbers amenable to a more efficient implementation than that provided as a generic fallback (based on isnan, signbit, and ==).\n\nExamples\n\njulia> isequal([1., NaN], [1., NaN])\ntrue\n\njulia> [1., NaN] == [1., NaN]\nfalse\n\njulia> 0.0 == -0.0\ntrue\n\njulia> isequal(0.0, -0.0)\nfalse\n\n\n\n\n\nisequal(x)\n\nCreate a function that compares its argument to x using isequal, i.e. a function equivalent to y -> isequal(y, x).\n\nThe returned function is of type Base.Fix2{typeof(isequal)}, which can be used to implement specialized methods.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isless","location":"base/base.html#Base.isless","category":"function","text":"isless(x, y)\n\nTest whether x is less than y, according to a fixed total order. isless is not defined on all pairs of values (x, y). However, if it is defined, it is expected to satisfy the following:\n\nIf isless(x, y) is defined, then so is isless(y, x) and isequal(x, y), and exactly one of those three yields true.\nThe relation defined by isless is transitive, i.e., isless(x, y) && isless(y, z) implies isless(x, z).\n\nValues that are normally unordered, such as NaN, are ordered in an arbitrary but consistent fashion. missing values are ordered last.\n\nThis is the default comparison used by sort.\n\nImplementation\n\nNon-numeric types with a total order should implement this function. Numeric types only need to implement it if they have special values such as NaN. Types with a partial order should implement <. See the documentation on Alternate orderings for how to define alternate ordering methods that can be used in sorting and related functions.\n\nExamples\n\njulia> isless(1, 3)\ntrue\n\njulia> isless(\"Red\", \"Blue\")\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ifelse","location":"base/base.html#Core.ifelse","category":"function","text":"ifelse(condition::Bool, x, y)\n\nReturn x if condition is true, otherwise return y. This differs from ? or if in that it is an ordinary function, so all the arguments are evaluated first. In some cases, using ifelse instead of an if statement can eliminate the branch in generated code and provide higher performance in tight loops.\n\nExamples\n\njulia> ifelse(1 > 2, 1, 2)\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.typeassert","location":"base/base.html#Core.typeassert","category":"function","text":"typeassert(x, type)\n\nThrow a TypeError unless x isa type. The syntax x::type calls this function.\n\nExamples\n\njulia> typeassert(2.5, Int)\nERROR: TypeError: in typeassert, expected Int64, got a value of type Float64\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.typeof","location":"base/base.html#Core.typeof","category":"function","text":"typeof(x)\n\nGet the concrete type of x.\n\nExamples\n\njulia> a = 1//2;\n\njulia> typeof(a)\nRational{Int64}\n\njulia> M = [1 2; 3.5 4];\n\njulia> typeof(M)\nMatrix{Float64} (alias for Array{Float64, 2})\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.tuple","location":"base/base.html#Core.tuple","category":"function","text":"tuple(xs...)\n\nConstruct a tuple of the given objects.\n\nExamples\n\njulia> tuple(1, 'a', pi)\n(1, 'a', π)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ntuple","location":"base/base.html#Base.ntuple","category":"function","text":"ntuple(f::Function, n::Integer)\n\nCreate a tuple of length n, computing each element as f(i), where i is the index of the element.\n\nExamples\n\njulia> ntuple(i -> 2*i, 4)\n(2, 4, 6, 8)\n\n\n\n\n\nntuple(f, ::Val{N})\n\nCreate a tuple of length N, computing each element as f(i), where i is the index of the element. By taking a Val(N) argument, it is possible that this version of ntuple may generate more efficient code than the version taking the length as an integer. But ntuple(f, N) is preferable to ntuple(f, Val(N)) in cases where N cannot be determined at compile time.\n\nExamples\n\njulia> ntuple(i -> 2*i, Val(4))\n(2, 4, 6, 8)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.objectid","location":"base/base.html#Base.objectid","category":"function","text":"objectid(x) -> UInt\n\nGet a hash value for x based on object identity. objectid(x)==objectid(y) if x === y.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hash","location":"base/base.html#Base.hash","category":"function","text":"hash(x[, h::UInt])\n\nCompute an integer hash code such that isequal(x,y) implies hash(x)==hash(y). The optional second argument h is a hash code to be mixed with the result.\n\nNew types should implement the 2-argument form, typically by calling the 2-argument hash method recursively in order to mix hashes of the contents with each other (and with h). Typically, any type that implements hash should also implement its own == (hence isequal) to guarantee the property mentioned above. Types supporting subtraction (operator -) should also implement widen, which is required to hash values inside heterogeneous arrays.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.finalizer","location":"base/base.html#Base.finalizer","category":"function","text":"finalizer(f, x)\n\nRegister a function f(x) to be called when there are no program-accessible references to x, and return x. The type of x must be a mutable struct, otherwise the behavior of this function is unpredictable.\n\nf must not cause a task switch, which excludes most I/O operations such as println. Using the @async macro (to defer context switching to outside of the finalizer) or ccall to directly invoke IO functions in C may be helpful for debugging purposes.\n\nExamples\n\nfinalizer(my_mutable_struct) do x\n    @async println(\"Finalizing $x.\")\nend\n\nfinalizer(my_mutable_struct) do x\n    ccall(:jl_safe_printf, Cvoid, (Cstring, Cstring), \"Finalizing %s.\", repr(x))\nend\n\nA finalizer may be registered at object construction. In the following example note that we implicitly rely on the finalizer returning the newly created mutable struct x.\n\nExample\n\nmutable struct MyMutableStruct\n    bar\n    function MyMutableStruct(bar)\n        x = new(bar)\n        f(t) = @async println(\"Finalizing $t.\")\n        finalizer(f, x)\n    end\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.finalize","location":"base/base.html#Base.finalize","category":"function","text":"finalize(x)\n\nImmediately run finalizers registered for object x.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.copy","location":"base/base.html#Base.copy","category":"function","text":"copy(x)\n\nCreate a shallow copy of x: the outer structure is copied, but not all internal values. For example, copying an array produces a new array with identically-same elements as the original.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.deepcopy","location":"base/base.html#Base.deepcopy","category":"function","text":"deepcopy(x)\n\nCreate a deep copy of x: everything is copied recursively, resulting in a fully independent object. For example, deep-copying an array produces a new array whose elements are deep copies of the original elements. Calling deepcopy on an object should generally have the same effect as serializing and then deserializing it.\n\nWhile it isn't normally necessary, user-defined types can override the default deepcopy behavior by defining a specialized version of the function deepcopy_internal(x::T, dict::IdDict) (which shouldn't otherwise be used), where T is the type to be specialized for, and dict keeps track of objects copied so far within the recursion. Within the definition, deepcopy_internal should be used in place of deepcopy, and the dict variable should be updated as appropriate before returning.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.getproperty","location":"base/base.html#Base.getproperty","category":"function","text":"getproperty(value, name::Symbol)\n\nThe syntax a.b calls getproperty(a, :b).\n\nExamples\n\njulia> struct MyType\n           x\n       end\n\njulia> function Base.getproperty(obj::MyType, sym::Symbol)\n           if sym === :special\n               return obj.x + 1\n           else # fallback to getfield\n               return getfield(obj, sym)\n           end\n       end\n\njulia> obj = MyType(1);\n\njulia> obj.special\n2\n\njulia> obj.x\n1\n\nSee also propertynames and setproperty!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setproperty!","location":"base/base.html#Base.setproperty!","category":"function","text":"setproperty!(value, name::Symbol, x)\n\nThe syntax a.b = c calls setproperty!(a, :b, c).\n\nSee also propertynames and getproperty.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.propertynames","location":"base/base.html#Base.propertynames","category":"function","text":"propertynames(x, private=false)\n\nGet a tuple or a vector of the properties (x.property) of an object x. This is typically the same as fieldnames(typeof(x)), but types that overload getproperty should generally overload propertynames as well to get the properties of an instance of the type.\n\npropertynames(x) may return only \"public\" property names that are part of the documented interface of x.   If you want it to also return \"private\" fieldnames intended for internal use, pass true for the optional second argument. REPL tab completion on x. shows only the private=false properties.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasproperty","location":"base/base.html#Base.hasproperty","category":"function","text":"hasproperty(x, s::Symbol)\n\nReturn a boolean indicating whether the object x has s as one of its own properties.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.getfield","location":"base/base.html#Core.getfield","category":"function","text":"getfield(value, name::Symbol)\ngetfield(value, i::Int)\n\nExtract a field from a composite value by name or position. See also getproperty and fieldnames.\n\nExamples\n\njulia> a = 1//2\n1//2\n\njulia> getfield(a, :num)\n1\n\njulia> a.num\n1\n\njulia> getfield(a, 1)\n1\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.setfield!","location":"base/base.html#Core.setfield!","category":"function","text":"setfield!(value, name::Symbol, x)\n\nAssign x to a named field in value of composite type. The value must be mutable and x must be a subtype of fieldtype(typeof(value), name). See also setproperty!.\n\nExamples\n\njulia> mutable struct MyMutableStruct\n           field::Int\n       end\n\njulia> a = MyMutableStruct(1);\n\njulia> setfield!(a, :field, 2);\n\njulia> getfield(a, :field)\n2\n\njulia> a = 1//2\n1//2\n\njulia> setfield!(a, :num, 3);\nERROR: setfield! immutable struct of type Rational cannot be changed\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.isdefined","location":"base/base.html#Core.isdefined","category":"function","text":"isdefined(m::Module, s::Symbol)\nisdefined(object, s::Symbol)\nisdefined(object, index::Int)\n\nTests whether a global variable or object field is defined. The arguments can be a module and a symbol or a composite object and field name (as a symbol) or index.\n\nTo test whether an array element is defined, use isassigned instead.\n\nSee also @isdefined.\n\nExamples\n\njulia> isdefined(Base, :sum)\ntrue\n\njulia> isdefined(Base, :NonExistentMethod)\nfalse\n\njulia> a = 1//2;\n\njulia> isdefined(a, 2)\ntrue\n\njulia> isdefined(a, 3)\nfalse\n\njulia> isdefined(a, :num)\ntrue\n\njulia> isdefined(a, :numerator)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@isdefined","location":"base/base.html#Base.@isdefined","category":"macro","text":"@isdefined s -> Bool\n\nTests whether variable s is defined in the current scope.\n\nSee also isdefined.\n\nExamples\n\njulia> @isdefined newvar\nfalse\n\njulia> newvar = 1\n1\n\njulia> @isdefined newvar\ntrue\n\njulia> function f()\n           println(@isdefined x)\n           x = 3\n           println(@isdefined x)\n       end\nf (generic function with 1 method)\n\njulia> f()\nfalse\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.convert","location":"base/base.html#Base.convert","category":"function","text":"convert(T, x)\n\nConvert x to a value of type T.\n\nIf T is an Integer type, an InexactError will be raised if x is not representable by T, for example if x is not integer-valued, or is outside the range supported by T.\n\nExamples\n\njulia> convert(Int, 3.0)\n3\n\njulia> convert(Int, 3.5)\nERROR: InexactError: Int64(3.5)\nStacktrace:\n[...]\n\nIf T is a AbstractFloat or Rational type, then it will return the closest value to x representable by T.\n\njulia> x = 1/3\n0.3333333333333333\n\njulia> convert(Float32, x)\n0.33333334f0\n\njulia> convert(Rational{Int32}, x)\n1//3\n\njulia> convert(Rational{Int64}, x)\n6004799503160661//18014398509481984\n\nIf T is a collection type and x a collection, the result of convert(T, x) may alias all or part of x.\n\njulia> x = Int[1, 2, 3];\n\njulia> y = convert(Vector{Int}, x);\n\njulia> y === x\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote","location":"base/base.html#Base.promote","category":"function","text":"promote(xs...)\n\nConvert all arguments to a common type, and return them all (as a tuple). If no arguments can be converted, an error is raised.\n\nExamples\n\njulia> promote(Int8(1), Float16(4.5), Float32(4.1))\n(1.0f0, 4.5f0, 4.1f0)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.oftype","location":"base/base.html#Base.oftype","category":"function","text":"oftype(x, y)\n\nConvert y to the type of x (convert(typeof(x), y)).\n\nExamples\n\njulia> x = 4;\n\njulia> y = 3.;\n\njulia> oftype(x, y)\n3\n\njulia> oftype(y, x)\n4.0\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.widen","location":"base/base.html#Base.widen","category":"function","text":"widen(x)\n\nIf x is a type, return a \"larger\" type, defined so that arithmetic operations + and - are guaranteed not to overflow nor lose precision for any combination of values that type x can hold.\n\nFor fixed-size integer types less than 128 bits, widen will return a type with twice the number of bits.\n\nIf x is a value, it is converted to widen(typeof(x)).\n\nExamples\n\njulia> widen(Int32)\nInt64\n\njulia> widen(1.5f0)\n1.5\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.identity","location":"base/base.html#Base.identity","category":"function","text":"identity(x)\n\nThe identity function. Returns its argument.\n\nExamples\n\njulia> identity(\"Well, what did you expect?\")\n\"Well, what did you expect?\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Properties of Types","location":"base/base.html#Properties-of-Types","category":"section","text":"","page":"Essentials"},{"title":"Type relations","location":"base/base.html#Type-relations","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.supertype\nCore.Type\nCore.DataType\nCore.:(<:)\nBase.:(>:)\nBase.typejoin\nBase.typeintersect\nBase.promote_type\nBase.promote_rule\nBase.isdispatchtuple","page":"Essentials"},{"title":"Base.supertype","location":"base/base.html#Base.supertype","category":"function","text":"supertype(T::DataType)\n\nReturn the supertype of DataType T.\n\nExamples\n\njulia> supertype(Int32)\nSigned\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Type","location":"base/base.html#Core.Type","category":"type","text":"Core.Type{T}\n\nCore.Type is an abstract type which has all type objects as its instances. The only instance of the singleton type Core.Type{T} is the object T.\n\nExamples\n\njulia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DataType","location":"base/base.html#Core.DataType","category":"type","text":"DataType <: Type{T}\n\nDataType represents explicitly declared types that have names, explicitly declared supertypes, and, optionally, parameters.  Every concrete value in the system is an instance of some DataType.\n\nExamples\n\njulia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataType\n\njulia> struct Point\n           x::Int\n           y\n       end\n\njulia> typeof(Point)\nDataType\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.:<:","location":"base/base.html#Core.:<:","category":"function","text":"<:(T1, T2)\n\nSubtype operator: returns true if and only if all values of type T1 are also of type T2.\n\nExamples\n\njulia> Float64 <: AbstractFloat\ntrue\n\njulia> Vector{Int} <: AbstractArray\ntrue\n\njulia> Matrix{Float64} <: Matrix{AbstractFloat}\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.:>:","location":"base/base.html#Base.:>:","category":"function","text":">:(T1, T2)\n\nSupertype operator, equivalent to T2 <: T1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typejoin","location":"base/base.html#Base.typejoin","category":"function","text":"typejoin(T, S)\n\nReturn the closest common ancestor of T and S, i.e. the narrowest type from which they both inherit.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typeintersect","location":"base/base.html#Base.typeintersect","category":"function","text":"typeintersect(T, S)\n\nCompute a type that contains the intersection of T and S. Usually this will be the smallest such type or one close to it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote_type","location":"base/base.html#Base.promote_type","category":"function","text":"promote_type(type1, type2)\n\nPromotion refers to converting values of mixed types to a single common type. promote_type represents the default promotion behavior in Julia when operators (usually mathematical) are given arguments of differing types. promote_type generally tries to return a type which can at least approximate most values of either input type without excessively widening.  Some loss is tolerated; for example, promote_type(Int64, Float64) returns Float64 even though strictly, not all Int64 values can be represented exactly as Float64 values.\n\njulia> promote_type(Int64, Float64)\nFloat64\n\njulia> promote_type(Int32, Int64)\nInt64\n\njulia> promote_type(Float32, BigInt)\nBigFloat\n\njulia> promote_type(Int16, Float16)\nFloat16\n\njulia> promote_type(Int64, Float16)\nFloat16\n\njulia> promote_type(Int8, UInt16)\nUInt16\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote_rule","location":"base/base.html#Base.promote_rule","category":"function","text":"promote_rule(type1, type2)\n\nSpecifies what type should be used by promote when given values of types type1 and type2. This function should not be called directly, but should have definitions added to it for new types as appropriate.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isdispatchtuple","location":"base/base.html#Base.isdispatchtuple","category":"function","text":"isdispatchtuple(T)\n\nDetermine whether type T is a tuple \"leaf type\", meaning it could appear as a type signature in dispatch and has no subtypes (or supertypes) which could appear in a call.\n\n\n\n\n\n","page":"Essentials"},{"title":"Declared structure","location":"base/base.html#Declared-structure","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.ismutable\nBase.isimmutable\nBase.isabstracttype\nBase.isprimitivetype\nBase.issingletontype\nBase.isstructtype\nBase.nameof(::DataType)\nBase.fieldnames\nBase.fieldname\nBase.hasfield","page":"Essentials"},{"title":"Base.ismutable","location":"base/base.html#Base.ismutable","category":"function","text":"ismutable(v) -> Bool\n\nReturn true iff value v is mutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a type, it will tell you that a value of DataType is mutable.\n\nExamples\n\njulia> ismutable(1)\nfalse\n\njulia> ismutable([1,2])\ntrue\n\ncompat: Julia 1.5\nThis function requires at least Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isimmutable","location":"base/base.html#Base.isimmutable","category":"function","text":"isimmutable(v) -> Bool\n\nwarning: Warning\nConsider using !ismutable(v) instead, as isimmutable(v) will be replaced by !ismutable(v) in a future release. (Since Julia 1.5)\n\nReturn true iff value v is immutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a type, it will tell you that a value of DataType is mutable.\n\nExamples\n\njulia> isimmutable(1)\ntrue\n\njulia> isimmutable([1,2])\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isabstracttype","location":"base/base.html#Base.isabstracttype","category":"function","text":"isabstracttype(T)\n\nDetermine whether type T was declared as an abstract type (i.e. using the abstract keyword).\n\nExamples\n\njulia> isabstracttype(AbstractArray)\ntrue\n\njulia> isabstracttype(Vector)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isprimitivetype","location":"base/base.html#Base.isprimitivetype","category":"function","text":"isprimitivetype(T) -> Bool\n\nDetermine whether type T was declared as a primitive type (i.e. using the primitive keyword).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.issingletontype","location":"base/base.html#Base.issingletontype","category":"function","text":"Base.issingletontype(T)\n\nDetermine whether type T has exactly one possible instance; for example, a struct type with no fields.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isstructtype","location":"base/base.html#Base.isstructtype","category":"function","text":"isstructtype(T) -> Bool\n\nDetermine whether type T was declared as a struct type (i.e. using the struct or mutable struct keyword).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{DataType}","category":"method","text":"nameof(t::DataType) -> Symbol\n\nGet the name of a (potentially UnionAll-wrapped) DataType (without its parent module) as a symbol.\n\nExamples\n\njulia> module Foo\n           struct S{T}\n           end\n       end\nFoo\n\njulia> nameof(Foo.S{T} where T)\n:S\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldnames","location":"base/base.html#Base.fieldnames","category":"function","text":"fieldnames(x::DataType)\n\nGet a tuple with the names of the fields of a DataType.\n\nExamples\n\njulia> fieldnames(Rational)\n(:num, :den)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldname","location":"base/base.html#Base.fieldname","category":"function","text":"fieldname(x::DataType, i::Integer)\n\nGet the name of field i of a DataType.\n\nExamples\n\njulia> fieldname(Rational, 1)\n:num\n\njulia> fieldname(Rational, 2)\n:den\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasfield","location":"base/base.html#Base.hasfield","category":"function","text":"hasfield(T::Type, name::Symbol)\n\nReturn a boolean indicating whether T has name as one of its own fields.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\n\n\n\n\n","page":"Essentials"},{"title":"Memory layout","location":"base/base.html#Memory-layout","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.sizeof(::Type)\nBase.isconcretetype\nBase.isbits\nBase.isbitstype\nCore.fieldtype\nBase.fieldtypes\nBase.fieldcount\nBase.fieldoffset\nBase.datatype_alignment\nBase.datatype_haspadding\nBase.datatype_pointerfree","page":"Essentials"},{"title":"Base.sizeof","location":"base/base.html#Base.sizeof-Tuple{Type}","category":"method","text":"sizeof(T::DataType)\nsizeof(obj)\n\nSize, in bytes, of the canonical binary representation of the given DataType T, if any. Size, in bytes, of object obj if it is not DataType.\n\nExamples\n\njulia> sizeof(Float32)\n4\n\njulia> sizeof(ComplexF64)\n16\n\njulia> sizeof(1.0)\n8\n\njulia> sizeof([1.0:10.0;])\n80\n\nIf DataType T does not have a specific size, an error is thrown.\n\njulia> sizeof(AbstractArray)\nERROR: Abstract type AbstractArray does not have a definite size.\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isconcretetype","location":"base/base.html#Base.isconcretetype","category":"function","text":"isconcretetype(T)\n\nDetermine whether type T is a concrete type, meaning it could have direct instances (values x such that typeof(x) === T).\n\nExamples\n\njulia> isconcretetype(Complex)\nfalse\n\njulia> isconcretetype(Complex{Float32})\ntrue\n\njulia> isconcretetype(Vector{Complex})\ntrue\n\njulia> isconcretetype(Vector{Complex{Float32}})\ntrue\n\njulia> isconcretetype(Union{})\nfalse\n\njulia> isconcretetype(Union{Int,String})\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbits","location":"base/base.html#Base.isbits","category":"function","text":"isbits(x)\n\nReturn true if x is an instance of an isbitstype type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbitstype","location":"base/base.html#Base.isbitstype","category":"function","text":"isbitstype(T)\n\nReturn true if type T is a \"plain data\" type, meaning it is immutable and contains no references to other values, only primitive types and other isbitstype types. Typical examples are numeric types such as UInt8, Float64, and Complex{Float64}. This category of types is significant since they are valid as type parameters, may not track isdefined / isassigned status, and have a defined layout that is compatible with C.\n\nExamples\n\njulia> isbitstype(Complex{Float64})\ntrue\n\njulia> isbitstype(Complex)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.fieldtype","location":"base/base.html#Core.fieldtype","category":"function","text":"fieldtype(T, name::Symbol | index::Int)\n\nDetermine the declared type of a field (specified by name or index) in a composite DataType T.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtype(Foo, :x)\nInt64\n\njulia> fieldtype(Foo, 2)\nString\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldtypes","location":"base/base.html#Base.fieldtypes","category":"function","text":"fieldtypes(T::Type)\n\nThe declared types of all fields in a composite DataType T as a tuple.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtypes(Foo)\n(Int64, String)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldcount","location":"base/base.html#Base.fieldcount","category":"function","text":"fieldcount(t::Type)\n\nGet the number of fields that an instance of the given type would have. An error is thrown if the type is too abstract to determine this.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldoffset","location":"base/base.html#Base.fieldoffset","category":"function","text":"fieldoffset(type, i)\n\nThe byte offset of field i of a type relative to the data start. For example, we could use it in the following manner to summarize information about a struct:\n\njulia> structinfo(T) = [(fieldoffset(T,i), fieldname(T,i), fieldtype(T,i)) for i = 1:fieldcount(T)];\n\njulia> structinfo(Base.Filesystem.StatStruct)\n12-element Vector{Tuple{UInt64, Symbol, DataType}}:\n (0x0000000000000000, :device, UInt64)\n (0x0000000000000008, :inode, UInt64)\n (0x0000000000000010, :mode, UInt64)\n (0x0000000000000018, :nlink, Int64)\n (0x0000000000000020, :uid, UInt64)\n (0x0000000000000028, :gid, UInt64)\n (0x0000000000000030, :rdev, UInt64)\n (0x0000000000000038, :size, Int64)\n (0x0000000000000040, :blksize, Int64)\n (0x0000000000000048, :blocks, Int64)\n (0x0000000000000050, :mtime, Float64)\n (0x0000000000000058, :ctime, Float64)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_alignment","location":"base/base.html#Base.datatype_alignment","category":"function","text":"Base.datatype_alignment(dt::DataType) -> Int\n\nMemory allocation minimum alignment for instances of this type. Can be called on any isconcretetype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_haspadding","location":"base/base.html#Base.datatype_haspadding","category":"function","text":"Base.datatype_haspadding(dt::DataType) -> Bool\n\nReturn whether the fields of instances of this type are packed in memory, with no intervening padding bytes. Can be called on any isconcretetype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_pointerfree","location":"base/base.html#Base.datatype_pointerfree","category":"function","text":"Base.datatype_pointerfree(dt::DataType) -> Bool\n\nReturn whether instances of this type can contain references to gc-managed memory. Can be called on any isconcretetype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Special values","location":"base/base.html#Special-values","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.typemin\nBase.typemax\nBase.floatmin\nBase.floatmax\nBase.maxintfloat\nBase.eps(::Type{<:AbstractFloat})\nBase.eps(::AbstractFloat)\nBase.instances","page":"Essentials"},{"title":"Base.typemin","location":"base/base.html#Base.typemin","category":"function","text":"typemin(T)\n\nThe lowest value representable by the given (real) numeric DataType T.\n\nExamples\n\njulia> typemin(Float16)\n-Inf16\n\njulia> typemin(Float32)\n-Inf32\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typemax","location":"base/base.html#Base.typemax","category":"function","text":"typemax(T)\n\nThe highest value representable by the given (real) numeric DataType.\n\nExamples\n\njulia> typemax(Int8)\n127\n\njulia> typemax(UInt32)\n0xffffffff\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.floatmin","location":"base/base.html#Base.floatmin","category":"function","text":"floatmin(T = Float64)\n\nReturn the smallest positive normal number representable by the floating-point type T.\n\nExamples\n\njulia> floatmin(Float16)\nFloat16(6.104e-5)\n\njulia> floatmin(Float32)\n1.1754944f-38\n\njulia> floatmin()\n2.2250738585072014e-308\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.floatmax","location":"base/base.html#Base.floatmax","category":"function","text":"floatmax(T = Float64)\n\nReturn the largest finite number representable by the floating-point type T.\n\nExamples\n\njulia> floatmax(Float16)\nFloat16(6.55e4)\n\njulia> floatmax(Float32)\n3.4028235f38\n\njulia> floatmax()\n1.7976931348623157e308\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.maxintfloat","location":"base/base.html#Base.maxintfloat","category":"function","text":"maxintfloat(T=Float64)\n\nThe largest consecutive integer-valued floating-point number that is exactly represented in the given floating-point type T (which defaults to Float64).\n\nThat is, maxintfloat returns the smallest positive integer-valued floating-point number n such that n+1 is not exactly representable in the type T.\n\nWhen an Integer-type value is needed, use Integer(maxintfloat(T)).\n\n\n\n\n\nmaxintfloat(T, S)\n\nThe largest consecutive integer representable in the given floating-point type T that also does not exceed the maximum integer representable by the integer type S.  Equivalently, it is the minimum of maxintfloat(T) and typemax(S).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.eps","location":"base/base.html#Base.eps-Tuple{Type{var\"#s3\"} where var\"#s3\"<:AbstractFloat}","category":"method","text":"eps(::Type{T}) where T<:AbstractFloat\neps()\n\nReturn the machine epsilon of the floating point type T (T = Float64 by default). This is defined as the gap between 1 and the next largest value representable by typeof(one(T)), and is equivalent to eps(one(T)).  (Since eps(T) is a bound on the relative error of T, it is a \"dimensionless\" quantity like one.)\n\nExamples\n\njulia> eps()\n2.220446049250313e-16\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> 1.0 + eps()\n1.0000000000000002\n\njulia> 1.0 + eps()/2\n1.0\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.eps","location":"base/base.html#Base.eps-Tuple{AbstractFloat}","category":"method","text":"eps(x::AbstractFloat)\n\nReturn the unit in last place (ulp) of x. This is the distance between consecutive representable floating point values at x. In most cases, if the distance on either side of x is different, then the larger of the two is taken, that is\n\neps(x) == max(x-prevfloat(x), nextfloat(x)-x)\n\nThe exceptions to this rule are the smallest and largest finite values (e.g. nextfloat(-Inf) and prevfloat(Inf) for Float64), which round to the smaller of the values.\n\nThe rationale for this behavior is that eps bounds the floating point rounding error. Under the default RoundNearest rounding mode, if y is a real number and x is the nearest floating point number to y, then\n\ny-x leq operatornameeps(x)2\n\nExamples\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(prevfloat(2.0))\n2.220446049250313e-16\n\njulia> eps(2.0)\n4.440892098500626e-16\n\njulia> x = prevfloat(Inf)      # largest finite Float64\n1.7976931348623157e308\n\njulia> x + eps(x)/2            # rounds up\nInf\n\njulia> x + prevfloat(eps(x)/2) # rounds down\n1.7976931348623157e308\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.instances","location":"base/base.html#Base.instances","category":"function","text":"instances(T::Type)\n\nReturn a collection of all instances of the given type, if applicable. Mostly used for enumerated types (see @enum).\n\nExample\n\njulia> @enum Color red blue green\n\njulia> instances(Color)\n(red, blue, green)\n\n\n\n\n\n","page":"Essentials"},{"title":"Special Types","location":"base/base.html#Special-Types","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Core.Any\nCore.Union\nUnion{}\nCore.UnionAll\nCore.Tuple\nCore.NamedTuple\nBase.@NamedTuple\nBase.Val\nCore.Vararg\nCore.Nothing\nBase.isnothing\nBase.Some\nBase.something\nBase.Enums.Enum\nBase.Enums.@enum\nCore.Expr\nCore.Symbol\nCore.Symbol(x...)\nCore.Module","page":"Essentials"},{"title":"Core.Any","location":"base/base.html#Core.Any","category":"type","text":"Any::DataType\n\nAny is the union of all types. It has the defining property isa(x, Any) == true for any x. Any therefore describes the entire universe of possible values. For example Integer is a subset of Any that includes Int, Int8, and other integer types.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Union","location":"base/base.html#Core.Union","category":"type","text":"Union{Types...}\n\nA type union is an abstract type which includes all instances of any of its argument types. The empty union Union{} is the bottom type of Julia.\n\nExamples\n\njulia> IntOrString = Union{Int,AbstractString}\nUnion{Int64, AbstractString}\n\njulia> 1 :: IntOrString\n1\n\njulia> \"Hello!\" :: IntOrString\n\"Hello!\"\n\njulia> 1.0 :: IntOrString\nERROR: TypeError: in typeassert, expected Union{Int64, AbstractString}, got a value of type Float64\n\n\n\n\n\n","page":"Essentials"},{"title":"Union{}","location":"base/base.html#Union{}","category":"keyword","text":"Union{}\n\nUnion{}, the empty Union of types, is the type that has no values. That is, it has the defining property isa(x, Union{}) == false for any x. Base.Bottom is defined as its alias and the type of Union{} is Core.TypeofBottom.\n\nExamples\n\njulia> isa(nothing, Union{})\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UnionAll","location":"base/base.html#Core.UnionAll","category":"type","text":"UnionAll\n\nA union of types over all values of a type parameter. UnionAll is used to describe parametric types where the values of some parameters are not known.\n\nExamples\n\njulia> typeof(Vector)\nUnionAll\n\njulia> typeof(Vector{Int})\nDataType\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Tuple","location":"base/base.html#Core.Tuple","category":"type","text":"Tuple{Types...}\n\nTuples are an abstraction of the arguments of a function – without the function itself. The salient aspects of a function's arguments are their order and their types. Therefore a tuple type is similar to a parameterized immutable type where each parameter is the type of one field. Tuple types may have any number of parameters.\n\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore Tuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters are. Tuples do not have field names; fields are only accessed by index.\n\nSee the manual section on Tuple Types.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.NamedTuple","location":"base/base.html#Core.NamedTuple","category":"type","text":"NamedTuple\n\nNamedTuples are, as their name suggests, named Tuples. That is, they're a tuple-like collection of values, where each entry has a unique name, represented as a Symbol. Like Tuples, NamedTuples are immutable; neither the names nor the values can be modified in place after construction.\n\nAccessing the value associated with a name in a named tuple can be done using field access syntax, e.g. x.a, or using getindex, e.g. x[:a]. A tuple of the names can be obtained using keys, and a tuple of the values can be obtained using values.\n\nnote: Note\nIteration over NamedTuples produces the values without the names. (See example below.) To iterate over the name-value pairs, use the pairs function.\n\nThe @NamedTuple macro can be used for conveniently declaring NamedTuple types.\n\nExamples\n\njulia> x = (a=1, b=2)\n(a = 1, b = 2)\n\njulia> x.a\n1\n\njulia> x[:a]\n1\n\njulia> keys(x)\n(:a, :b)\n\njulia> values(x)\n(1, 2)\n\njulia> collect(x)\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(pairs(x))\n2-element Vector{Pair{Symbol, Int64}}:\n :a => 1\n :b => 2\n\nIn a similar fashion as to how one can define keyword arguments programmatically, a named tuple can be created by giving a pair name::Symbol => value or splatting an iterator yielding such pairs after a semicolon inside a tuple literal:\n\njulia> (; :a => 1)\n(a = 1,)\n\njulia> keys = (:a, :b, :c); values = (1, 2, 3);\n\njulia> (; zip(keys, values)...)\n(a = 1, b = 2, c = 3)\n\nAs in keyword arguments, identifiers and dot expressions imply names:\n\njulia> x = 0\n0\n\njulia> t = (; x)\n(x = 0,)\n\njulia> (; t.x)\n(x = 0,)\n\ncompat: Julia 1.5\nImplicit names from identifiers and dot expressions are available as of Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@NamedTuple","location":"base/base.html#Base.@NamedTuple","category":"macro","text":"@NamedTuple{key1::Type1, key2::Type2, ...}\n@NamedTuple begin key1::Type1; key2::Type2; ...; end\n\nThis macro gives a more convenient syntax for declaring NamedTuple types. It returns a NamedTuple type with the given keys and types, equivalent to NamedTuple{(:key1, :key2, ...), Tuple{Type1,Type2,...}}. If the ::Type declaration is omitted, it is taken to be Any.   The begin ... end form allows the declarations to be split across multiple lines (similar to a struct declaration), but is otherwise equivalent.\n\nFor example, the tuple (a=3.1, b=\"hello\") has a type NamedTuple{(:a, :b),Tuple{Float64,String}}, which can also be declared via @NamedTuple as:\n\njulia> @NamedTuple{a::Float64, b::String}\nNamedTuple{(:a, :b), Tuple{Float64, String}}\n\njulia> @NamedTuple begin\n           a::Float64\n           b::String\n       end\nNamedTuple{(:a, :b), Tuple{Float64, String}}\n\ncompat: Julia 1.5\nThis macro is available as of Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Val","location":"base/base.html#Base.Val","category":"type","text":"Val(c)\n\nReturn Val{c}(), which contains no run-time data. Types like this can be used to pass the information between functions through the value c, which must be an isbits value or a Symbol. The intent of this construct is to be able to dispatch on constants directly (at compile time) without having to test the value of the constant at run time.\n\nExamples\n\njulia> f(::Val{true}) = \"Good\"\nf (generic function with 1 method)\n\njulia> f(::Val{false}) = \"Bad\"\nf (generic function with 2 methods)\n\njulia> f(Val(true))\n\"Good\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Vararg","location":"base/base.html#Core.Vararg","category":"type","text":"Vararg{T,N}\n\nThe last parameter of a tuple type Tuple can be the special type Vararg, which denotes any number of trailing elements. The type Vararg{T,N} corresponds to exactly N elements of type T. Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are used to represent the arguments accepted by varargs methods (see the section on Varargs Functions in the manual.)\n\nExamples\n\njulia> mytupletype = Tuple{AbstractString, Vararg{Int}}\nTuple{AbstractString, Vararg{Int64, N} where N}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Nothing","location":"base/base.html#Core.Nothing","category":"type","text":"Nothing\n\nA type with no fields that is the type of nothing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isnothing","location":"base/base.html#Base.isnothing","category":"function","text":"isnothing(x)\n\nReturn true if x === nothing, and return false if not.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Some","location":"base/base.html#Base.Some","category":"type","text":"Some{T}\n\nA wrapper type used in Union{Some{T}, Nothing} to distinguish between the absence of a value (nothing) and the presence of a nothing value (i.e. Some(nothing)).\n\nUse something to access the value wrapped by a Some object.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.something","location":"base/base.html#Base.something","category":"function","text":"something(x, y...)\n\nReturn the first value in the arguments which is not equal to nothing, if any. Otherwise throw an error. Arguments of type Some are unwrapped.\n\nSee also coalesce.\n\nExamples\n\njulia> something(nothing, 1)\n1\n\njulia> something(Some(1), nothing)\n1\n\njulia> something(missing, nothing)\nmissing\n\njulia> something(nothing, nothing)\nERROR: ArgumentError: No value arguments present\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Enums.Enum","location":"base/base.html#Base.Enums.Enum","category":"type","text":"Enum{T<:Integer}\n\nThe abstract supertype of all enumerated types defined with @enum.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Enums.@enum","location":"base/base.html#Base.Enums.@enum","category":"macro","text":"@enum EnumName[::BaseType] value1[=x] value2[=y]\n\nCreate an Enum{BaseType} subtype with name EnumName and enum member values of value1 and value2 with optional assigned values of x and y, respectively. EnumName can be used just like other types and enum member values as regular values, such as\n\nExamples\n\njulia> @enum Fruit apple=1 orange=2 kiwi=3\n\njulia> f(x::Fruit) = \"I'm a Fruit with value: $(Int(x))\"\nf (generic function with 1 method)\n\njulia> f(apple)\n\"I'm a Fruit with value: 1\"\n\njulia> Fruit(1)\napple::Fruit = 1\n\nValues can also be specified inside a begin block, e.g.\n\n@enum EnumName begin\n    value1\n    value2\nend\n\nBaseType, which defaults to Int32, must be a primitive subtype of Integer. Member values can be converted between the enum type and BaseType. read and write perform these conversions automatically. In case the enum is created with a non-default BaseType, Integer(value1) will return the integer value1 with the type BaseType.\n\nTo list all the instances of an enum use instances, e.g.\n\njulia> instances(Fruit)\n(apple, orange, kiwi)\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Expr","location":"base/base.html#Core.Expr","category":"type","text":"Expr(head::Symbol, args...)\n\nA type representing compound expressions in parsed julia code (ASTs). Each expression consists of a head Symbol identifying which kind of expression it is (e.g. a call, for loop, conditional statement, etc.), and subexpressions (e.g. the arguments of a call). The subexpressions are stored in a Vector{Any} field called args.\n\nSee the manual chapter on Metaprogramming and the developer documentation Julia ASTs.\n\nExamples\n\njulia> Expr(:call, :+, 1, 2)\n:(1 + 2)\n\njulia> dump(:(a ? b : c))\nExpr\n  head: Symbol if\n  args: Array{Any}((3,))\n    1: Symbol a\n    2: Symbol b\n    3: Symbol c\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Symbol","location":"base/base.html#Core.Symbol","category":"type","text":"Symbol\n\nThe type of object used to represent identifiers in parsed julia code (ASTs). Also often used as a name or label to identify an entity (e.g. as a dictionary key). Symbols can be entered using the : quote operator:\n\njulia> :name\n:name\n\njulia> typeof(:name)\nSymbol\n\njulia> x = 42\n42\n\njulia> eval(:x)\n42\n\nSymbols can also be constructed from strings or other values by calling the constructor Symbol(x...).\n\nSymbols are immutable and should be compared using ===. The implementation re-uses the same object for all Symbols with the same name, so comparison tends to be efficient (it can just compare pointers).\n\nUnlike strings, Symbols are \"atomic\" or \"scalar\" entities that do not support iteration over characters.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Symbol","location":"base/base.html#Core.Symbol-Tuple","category":"method","text":"Symbol(x...) -> Symbol\n\nCreate a Symbol by concatenating the string representations of the arguments together.\n\nExamples\n\njulia> Symbol(\"my\", \"name\")\n:myname\n\njulia> Symbol(\"day\", 4)\n:day4\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Module","location":"base/base.html#Core.Module","category":"type","text":"Module\n\nA Module is a separate global variable workspace. See module and the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"Generic Functions","location":"base/base.html#Generic-Functions","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Core.Function\nBase.hasmethod\nCore.applicable\nCore.invoke\nBase.invokelatest\nBase.@invokelatest\nnew\nBase.:(|>)\nBase.:(∘)\nBase.ComposedFunction","page":"Essentials"},{"title":"Core.Function","location":"base/base.html#Core.Function","category":"type","text":"Function\n\nAbstract type of all functions.\n\nExamples\n\njulia> isa(+, Function)\ntrue\n\njulia> typeof(sin)\ntypeof(sin)\n\njulia> ans <: Function\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasmethod","location":"base/base.html#Base.hasmethod","category":"function","text":"hasmethod(f, t::Type{<:Tuple}[, kwnames]; world=typemax(UInt)) -> Bool\n\nDetermine whether the given generic function has a method matching the given Tuple of argument types with the upper bound of world age given by world.\n\nIf a tuple of keyword argument names kwnames is provided, this also checks whether the method of f matching t has the given keyword argument names. If the matching method accepts a variable number of keyword arguments, e.g. with kwargs..., any names given in kwnames are considered valid. Otherwise the provided names must be a subset of the method's keyword arguments.\n\nSee also applicable.\n\ncompat: Julia 1.2\nProviding keyword argument names requires Julia 1.2 or later.\n\nExamples\n\njulia> hasmethod(length, Tuple{Array})\ntrue\n\njulia> f(; oranges=0) = oranges;\n\njulia> hasmethod(f, Tuple{}, (:oranges,))\ntrue\n\njulia> hasmethod(f, Tuple{}, (:apples, :bananas))\nfalse\n\njulia> g(; xs...) = 4;\n\njulia> hasmethod(g, Tuple{}, (:a, :b, :c, :d))  # g accepts arbitrary kwargs\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.applicable","location":"base/base.html#Core.applicable","category":"function","text":"applicable(f, args...) -> Bool\n\nDetermine whether the given generic function has a method applicable to the given arguments.\n\nSee also hasmethod.\n\nExamples\n\njulia> function f(x, y)\n           x + y\n       end;\n\njulia> applicable(f, 1)\nfalse\n\njulia> applicable(f, 1, 2)\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.invoke","location":"base/base.html#Core.invoke","category":"function","text":"invoke(f, argtypes::Type, args...; kwargs...)\n\nInvoke a method for the given generic function f matching the specified types argtypes on the specified arguments args and passing the keyword arguments kwargs. The arguments args must conform with the specified types in argtypes, i.e. conversion is not automatically performed. This method allows invoking a method other than the most specific matching method, which is useful when the behavior of a more general definition is explicitly needed (often as part of the implementation of a more specific method of the same function).\n\nBe careful when using invoke for functions that you don't write.  What definition is used for given argtypes is an implementation detail unless the function is explicitly states that calling with certain argtypes is a part of public API.  For example, the change between f1 and f2 in the example below is usually considered compatible because the change is invisible by the caller with a normal (non-invoke) call.  However, the change is visible if you use invoke.\n\nExamples\n\njulia> f(x::Real) = x^2;\n\njulia> f(x::Integer) = 1 + invoke(f, Tuple{Real}, x);\n\njulia> f(2)\n5\n\njulia> f1(::Integer) = Integer\n       f1(::Real) = Real;\n\njulia> f2(x::Real) = _f2(x)\n       _f2(::Integer) = Integer\n       _f2(_) = Real;\n\njulia> f1(1)\nInteger\n\njulia> f2(1)\nInteger\n\njulia> invoke(f1, Tuple{Real}, 1)\nReal\n\njulia> invoke(f2, Tuple{Real}, 1)\nInteger\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.invokelatest","location":"base/base.html#Base.invokelatest","category":"function","text":"invokelatest(f, args...; kwargs...)\n\nCalls f(args...; kwargs...), but guarantees that the most recent method of f will be executed.   This is useful in specialized circumstances, e.g. long-running event loops or callback functions that may call obsolete versions of a function f. (The drawback is that invokelatest is somewhat slower than calling f directly, and the type of the result cannot be inferred by the compiler.)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@invokelatest","location":"base/base.html#Base.@invokelatest","category":"macro","text":"@invokelatest f(args...; kwargs...)\n\nProvides a convenient way to call Base.invokelatest. @invokelatest f(args...; kwargs...) will simply be expanded into Base.invokelatest(f, args...; kwargs...).\n\n\n\n\n\n","page":"Essentials"},{"title":"new","location":"base/base.html#new","category":"keyword","text":"new\n\nSpecial function available to inner constructors which created a new object of the type. See the manual section on Inner Constructor Methods for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.:|>","location":"base/base.html#Base.:|>","category":"function","text":"|>(x, f)\n\nApplies a function to the preceding argument. This allows for easy function chaining.\n\nExamples\n\njulia> [1:5;] |> x->x.^2 |> sum |> inv\n0.01818181818181818\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.:∘","location":"base/base.html#Base.:∘","category":"function","text":"f ∘ g\n\nCompose functions: i.e. (f ∘ g)(args...) means f(g(args...)). The ∘ symbol can be entered in the Julia REPL (and most editors, appropriately configured) by typing \\circ<tab>.\n\nFunction composition also works in prefix form: ∘(f, g) is the same as f ∘ g. The prefix form supports composition of multiple functions: ∘(f, g, h) = f ∘ g ∘ h and splatting ∘(fs...) for composing an iterable collection of functions.\n\ncompat: Julia 1.4\nMultiple function composition requires at least Julia 1.4.\n\ncompat: Julia 1.5\nComposition of one function ∘(f)  requires at least Julia 1.5.\n\nExamples\n\njulia> map(uppercase∘first, [\"apple\", \"banana\", \"carrot\"])\n3-element Vector{Char}:\n 'A': ASCII/Unicode U+0041 (category Lu: Letter, uppercase)\n 'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)\n 'C': ASCII/Unicode U+0043 (category Lu: Letter, uppercase)\n\njulia> fs = [\n           x -> 2x\n           x -> x/2\n           x -> x-1\n           x -> x+1\n       ];\n\njulia> ∘(fs...)(3)\n3.0\n\nSee also ComposedFunction.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ComposedFunction","location":"base/base.html#Base.ComposedFunction","category":"type","text":"ComposedFunction{Outer,Inner} <: Function\n\nRepresents the composition of two callable objects outer::Outer and inner::Inner. That is\n\nComposedFunction(outer, inner)(args...; kw...) === outer(inner(args...; kw...))\n\nThe preferred way to construct instance of ComposedFunction is to use the composition operator ∘:\n\njulia> sin ∘ cos === ComposedFunction(sin, cos)\ntrue\n\njulia> typeof(sin∘cos)\nComposedFunction{typeof(sin), typeof(cos)}\n\nThe composed pieces are stored in the fields of ComposedFunction and can be retrieved as follows:\n\njulia> composition = sin ∘ cos\nsin ∘ cos\n\njulia> composition.outer === sin\ntrue\n\njulia> composition.inner === cos\ntrue\n\ncompat: Julia 1.6\nComposedFunction requires at least Julia 1.6. In earlier versions ∘ returns an anonymous function instead.\n\nSee also ∘.\n\n\n\n\n\n","page":"Essentials"},{"title":"Syntax","location":"base/base.html#Syntax","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Core.eval\nBase.MainInclude.eval\nBase.@eval\nBase.evalfile\nBase.esc\nBase.@inbounds\nBase.@boundscheck\nBase.@propagate_inbounds\nBase.@inline\nBase.@noinline\nBase.@nospecialize\nBase.@specialize\nBase.gensym\nBase.@gensym\nvar\"name\"\nBase.@goto\nBase.@label\nBase.@simd\nBase.@polly\nBase.@generated\nBase.@pure\nBase.@deprecate","page":"Essentials"},{"title":"Core.eval","location":"base/base.html#Core.eval","category":"function","text":"Core.eval(m::Module, expr)\n\nEvaluate an expression in the given module and return the result.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MainInclude.eval","location":"base/base.html#Base.MainInclude.eval","category":"function","text":"eval(expr)\n\nEvaluate an expression in the global scope of the containing module. Every Module (except those defined with baremodule) has its own 1-argument definition of eval, which evaluates expressions in that module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@eval","location":"base/base.html#Base.@eval","category":"macro","text":"@eval [mod,] ex\n\nEvaluate an expression with values interpolated into it using eval. If two arguments are provided, the first is the module to evaluate in.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.evalfile","location":"base/base.html#Base.evalfile","category":"function","text":"evalfile(path::AbstractString, args::Vector{String}=String[])\n\nLoad the file using include, evaluate all expressions, and return the value of the last one.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.esc","location":"base/base.html#Base.esc","category":"function","text":"esc(e)\n\nOnly valid in the context of an Expr returned from a macro. Prevents the macro hygiene pass from turning embedded variables into gensym variables. See the Macros section of the Metaprogramming chapter of the manual for more details and examples.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@inbounds","location":"base/base.html#Base.@inbounds","category":"macro","text":"@inbounds(blk)\n\nEliminates array bounds checking within expressions.\n\nIn the example below the in-range check for referencing element i of array A is skipped to improve performance.\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i = 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nwarning: Warning\nUsing @inbounds may return incorrect results/crashes/corruption for out-of-bounds indices. The user is responsible for checking it manually. Only use @inbounds when it is certain from the information locally available that all accesses are in bounds.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@boundscheck","location":"base/base.html#Base.@boundscheck","category":"macro","text":"@boundscheck(blk)\n\nAnnotates the expression blk as a bounds checking block, allowing it to be elided by @inbounds.\n\nnote: Note\nThe function in which @boundscheck is written must be inlined into its caller in order for @inbounds to have effect.\n\nExamples\n\njulia> @inline function g(A, i)\n           @boundscheck checkbounds(A, i)\n           return \"accessing ($A)[$i]\"\n       end;\n\njulia> f1() = return g(1:2, -1);\n\njulia> f2() = @inbounds return g(1:2, -1);\n\njulia> f1()\nERROR: BoundsError: attempt to access 2-element UnitRange{Int64} at index [-1]\nStacktrace:\n [1] throw_boundserror(::UnitRange{Int64}, ::Tuple{Int64}) at ./abstractarray.jl:455\n [2] checkbounds at ./abstractarray.jl:420 [inlined]\n [3] g at ./none:2 [inlined]\n [4] f1() at ./none:1\n [5] top-level scope\n\njulia> f2()\n\"accessing (1:2)[-1]\"\n\nwarning: Warning\nThe @boundscheck annotation allows you, as a library writer, to opt-in to allowing other code to remove your bounds checks with @inbounds. As noted there, the caller must verify—using information they can access—that their accesses are valid before using @inbounds. For indexing into your AbstractArray subclasses, for example, this involves checking the indices against its size. Therefore, @boundscheck annotations should only be added to a getindex or setindex! implementation after you are certain its behavior is correct.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@propagate_inbounds","location":"base/base.html#Base.@propagate_inbounds","category":"macro","text":"@propagate_inbounds\n\nTells the compiler to inline a function while retaining the caller's inbounds context.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@inline","location":"base/base.html#Base.@inline","category":"macro","text":"@inline\n\nGive a hint to the compiler that this function is worth inlining.\n\nSmall functions typically do not need the @inline annotation, as the compiler does it automatically. By using @inline on bigger functions, an extra nudge can be given to the compiler to inline it. This is shown in the following example:\n\n@inline function bigfunction(x)\n    #=\n        Function Definition\n    =#\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@noinline","location":"base/base.html#Base.@noinline","category":"macro","text":"@noinline\n\nGive a hint to the compiler that it should not inline a function.\n\nSmall functions are typically inlined automatically. By using @noinline on small functions, auto-inlining can be prevented. This is shown in the following example:\n\n@noinline function smallfunction(x)\n    #=\n        Function Definition\n    =#\nend\n\nnote: Note\nIf the function is trivial (for example returning a constant) it might get inlined anyway.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@nospecialize","location":"base/base.html#Base.@nospecialize","category":"macro","text":"@nospecialize\n\nApplied to a function argument name, hints to the compiler that the method should not be specialized for different types of that argument, but instead to use precisely the declared type for each argument. This is only a hint for avoiding excess code generation. Can be applied to an argument within a formal argument list, or in the function body. When applied to an argument, the macro must wrap the entire argument expression. When used in a function body, the macro must occur in statement position and before any code.\n\nWhen used without arguments, it applies to all arguments of the parent scope. In local scope, this means all arguments of the containing function. In global (top-level) scope, this means all methods subsequently defined in the current module.\n\nSpecialization can reset back to the default by using @specialize.\n\nfunction example_function(@nospecialize x)\n    ...\nend\n\nfunction example_function(x, @nospecialize(y = 1))\n    ...\nend\n\nfunction example_function(x, y, z)\n    @nospecialize x y\n    ...\nend\n\n@nospecialize\nf(y) = [x for x in y]\n@specialize\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@specialize","location":"base/base.html#Base.@specialize","category":"macro","text":"@specialize\n\nReset the specialization hint for an argument back to the default. For details, see @nospecialize.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.gensym","location":"base/base.html#Base.gensym","category":"function","text":"gensym([tag])\n\nGenerates a symbol which will not conflict with other variable names.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@gensym","location":"base/base.html#Base.@gensym","category":"macro","text":"@gensym\n\nGenerates a gensym symbol for a variable. For example, @gensym x y is transformed into x = gensym(\"x\"); y = gensym(\"y\").\n\n\n\n\n\n","page":"Essentials"},{"title":"var\"name\"","location":"base/base.html#var\"name\"","category":"keyword","text":"var\n\nThe syntax var\"#example#\" refers to a variable named Symbol(\"#example#\"), even though #example# is not a valid Julia identifier name.\n\nThis can be useful for interoperability with programming languages which have different rules for the construction of valid identifiers. For example, to refer to the R variable draw.segments, you can use var\"draw.segments\" in your Julia code.\n\nIt is also used to show julia source code which has gone through macro hygiene or otherwise contains variable names which can't be parsed normally.\n\nNote that this syntax requires parser support so it is expanded directly by the parser rather than being implemented as a normal string macro @var_str.\n\ncompat: Julia 1.3\nThis syntax requires at least Julia 1.3.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@goto","location":"base/base.html#Base.@goto","category":"macro","text":"@goto name\n\n@goto name unconditionally jumps to the statement at the location @label name.\n\n@label and @goto cannot create jumps to different top-level statements. Attempts cause an error. To still use @goto, enclose the @label and @goto in a block.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@label","location":"base/base.html#Base.@label","category":"macro","text":"@label name\n\nLabels a statement with the symbolic label name. The label marks the end-point of an unconditional jump with @goto name.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.SimdLoop.@simd","location":"base/base.html#Base.SimdLoop.@simd","category":"macro","text":"@simd\n\nAnnotate a for loop to allow the compiler to take extra liberties to allow loop re-ordering\n\nwarning: Warning\nThis feature is experimental and could change or disappear in future versions of Julia. Incorrect use of the @simd macro may cause unexpected results.\n\nThe object iterated over in a @simd for loop should be a one-dimensional range. By using @simd, you are asserting several properties of the loop:\n\nIt is safe to execute iterations in arbitrary or overlapping order, with special consideration for reduction variables.\nFloating-point operations on reduction variables can be reordered, possibly causing different results than without @simd.\n\nIn many cases, Julia is able to automatically vectorize inner for loops without the use of @simd. Using @simd gives the compiler a little extra leeway to make it possible in more situations. In either case, your inner loop should have the following properties to allow vectorization:\n\nThe loop must be an innermost loop\nThe loop body must be straight-line code. Therefore, @inbounds is   currently needed for all array accesses. The compiler can sometimes turn   short &&, ||, and ?: expressions into straight-line code if it is safe   to evaluate all operands unconditionally. Consider using the ifelse   function instead of ?: in the loop if it is safe to do so.\nAccesses must have a stride pattern and cannot be \"gathers\" (random-index   reads) or \"scatters\" (random-index writes).\nThe stride should be unit stride.\n\nnote: Note\nThe @simd does not assert by default that the loop is completely free of loop-carried memory dependencies, which is an assumption that can easily be violated in generic code. If you are writing non-generic code, you can use @simd ivdep for ... end to also assert that:\n\nThere exists no loop-carried memory dependencies\nNo iteration ever waits on a previous iteration to make forward progress.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@polly","location":"base/base.html#Base.@polly","category":"macro","text":"@polly\n\nTells the compiler to apply the polyhedral optimizer Polly to a function.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@generated","location":"base/base.html#Base.@generated","category":"macro","text":"@generated f\n@generated(f)\n\n@generated is used to annotate a function which will be generated. In the body of the generated function, only types of arguments can be read (not the values). The function returns a quoted expression evaluated when the function is called. The @generated macro should not be used on functions mutating the global scope or depending on mutable elements.\n\nSee Metaprogramming for further details.\n\nExample:\n\njulia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@pure","location":"base/base.html#Base.@pure","category":"macro","text":"@pure ex\n@pure(ex)\n\n@pure gives the compiler a hint for the definition of a pure function, helping for type inference.\n\nA pure function can only depend on immutable information. This also means a @pure function cannot use any global mutable state, including generic functions. Calls to generic functions depend on method tables which are mutable global state. Use with caution, incorrect @pure annotation of a function may introduce hard to identify bugs. Double check for calls to generic functions. This macro is intended for internal compiler use and may be subject to changes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@deprecate","location":"base/base.html#Base.@deprecate","category":"macro","text":"@deprecate old new [ex=true]\n\nDeprecate method old and specify the replacement call new. Prevent @deprecate from exporting old by setting ex to false. @deprecate defines a new method with the same signature as old.\n\ncompat: Julia 1.5\nAs of Julia 1.5, functions defined by @deprecate do not print warning when julia is run without the --depwarn=yes flag set, as the default value of --depwarn option is no.  The warnings are printed from tests run by Pkg.test().\n\nExamples\n\njulia> @deprecate old(x) new(x)\nold (generic function with 1 method)\n\njulia> @deprecate old(x) new(x) false\nold (generic function with 1 method)\n\n\n\n\n\n","page":"Essentials"},{"title":"Missing Values","location":"base/base.html#Missing-Values","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.Missing\nBase.missing\nBase.coalesce\nBase.ismissing\nBase.skipmissing\nBase.nonmissingtype","page":"Essentials"},{"title":"Base.Missing","location":"base/base.html#Base.Missing","category":"type","text":"Missing\n\nA type with no fields whose singleton instance missing is used to represent missing values.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.missing","location":"base/base.html#Base.missing","category":"constant","text":"missing\n\nThe singleton instance of type Missing representing a missing value.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.coalesce","location":"base/base.html#Base.coalesce","category":"function","text":"coalesce(x, y...)\n\nReturn the first value in the arguments which is not equal to missing, if any. Otherwise return missing.\n\nSee also something.\n\nExamples\n\njulia> coalesce(missing, 1)\n1\n\njulia> coalesce(1, missing)\n1\n\njulia> coalesce(nothing, 1)  # returns `nothing`\n\njulia> coalesce(missing, missing)\nmissing\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ismissing","location":"base/base.html#Base.ismissing","category":"function","text":"ismissing(x)\n\nIndicate whether x is missing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.skipmissing","location":"base/base.html#Base.skipmissing","category":"function","text":"skipmissing(itr)\n\nReturn an iterator over the elements in itr skipping missing values. The returned object can be indexed using indices of itr if the latter is indexable. Indices corresponding to missing values are not valid: they are skipped by keys and eachindex, and a MissingException is thrown when trying to use them.\n\nUse collect to obtain an Array containing the non-missing values in itr. Note that even if itr is a multidimensional array, the result will always be a Vector since it is not possible to remove missings while preserving dimensions of the input.\n\nExamples\n\njulia> x = skipmissing([1, missing, 2])\nskipmissing(Union{Missing, Int64}[1, missing, 2])\n\njulia> sum(x)\n3\n\njulia> x[1]\n1\n\njulia> x[2]\nERROR: MissingException: the value at index (2,) is missing\n[...]\n\njulia> argmax(x)\n3\n\njulia> collect(keys(x))\n2-element Vector{Int64}:\n 1\n 3\n\njulia> collect(skipmissing([1, missing, 2]))\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(skipmissing([1 missing; 2 missing]))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nonmissingtype","location":"base/base.html#Base.nonmissingtype","category":"function","text":"nonmissingtype(T::Type)\n\nIf T is a union of types containing Missing, return a new type with Missing removed.\n\nExamples\n\njulia> nonmissingtype(Union{Int64,Missing})\nInt64\n\njulia> nonmissingtype(Any)\nAny\n\ncompat: Julia 1.3\nThis function is exported as of Julia 1.3.\n\n\n\n\n\n","page":"Essentials"},{"title":"System","location":"base/base.html#System","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.run\nBase.devnull\nBase.success\nBase.process_running\nBase.process_exited\nBase.kill(::Base.Process, ::Integer)\nBase.Sys.set_process_title\nBase.Sys.get_process_title\nBase.ignorestatus\nBase.detach\nBase.Cmd\nBase.setenv\nBase.withenv\nBase.pipeline(::Any, ::Any, ::Any, ::Any...)\nBase.pipeline(::Base.AbstractCmd)\nBase.Libc.gethostname\nBase.Libc.getpid\nBase.Libc.time()\nBase.time_ns\nBase.@time\nBase.@timev\nBase.@timed\nBase.@elapsed\nBase.@allocated\nBase.EnvDict\nBase.ENV\nBase.Sys.isunix\nBase.Sys.isapple\nBase.Sys.islinux\nBase.Sys.isbsd\nBase.Sys.isfreebsd\nBase.Sys.isopenbsd\nBase.Sys.isnetbsd\nBase.Sys.isdragonfly\nBase.Sys.iswindows\nBase.Sys.windows_version\nBase.Sys.free_memory\nBase.Sys.total_memory\nBase.@static","page":"Essentials"},{"title":"Base.run","location":"base/base.html#Base.run","category":"function","text":"run(command, args...; wait::Bool = true)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual). Throws an error if anything goes wrong, including the process exiting with a non-zero status (when wait is true).\n\nIf wait is false, the process runs asynchronously. You can later wait for it and check its exit status by calling success on the returned process object.\n\nWhen wait is false, the process' I/O streams are directed to devnull. When wait is true, I/O streams are shared with the parent process. Use pipeline to control I/O redirection.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.devnull","location":"base/base.html#Base.devnull","category":"constant","text":"devnull\n\nUsed in a stream redirect to discard all data written to it. Essentially equivalent to /dev/null on Unix or NUL on Windows. Usage:\n\nrun(pipeline(`cat test.txt`, devnull))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.success","location":"base/base.html#Base.success","category":"function","text":"success(command)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual), and tell whether it was successful (exited with a code of 0). An exception is raised if the process cannot be started.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.process_running","location":"base/base.html#Base.process_running","category":"function","text":"process_running(p::Process)\n\nDetermine whether a process is currently running.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.process_exited","location":"base/base.html#Base.process_exited","category":"function","text":"process_exited(p::Process)\n\nDetermine whether a process has exited.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.kill","location":"base/base.html#Base.kill-Tuple{Base.Process, Integer}","category":"method","text":"kill(p::Process, signum=Base.SIGTERM)\n\nSend a signal to a process. The default is to terminate the process. Returns successfully if the process has already exited, but throws an error if killing the process failed for other reasons (e.g. insufficient permissions).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.set_process_title","location":"base/base.html#Base.Sys.set_process_title","category":"function","text":"Sys.set_process_title(title::AbstractString)\n\nSet the process title. No-op on some operating systems.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.get_process_title","location":"base/base.html#Base.Sys.get_process_title","category":"function","text":"Sys.get_process_title()\n\nGet the process title. On some systems, will always return an empty string.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ignorestatus","location":"base/base.html#Base.ignorestatus","category":"function","text":"ignorestatus(command)\n\nMark a command object so that running it will not throw an error if the result code is non-zero.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.detach","location":"base/base.html#Base.detach","category":"function","text":"detach(command)\n\nMark a command object so that it will be run in a new process group, allowing it to outlive the julia process, and not have Ctrl-C interrupts passed to it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Cmd","location":"base/base.html#Base.Cmd","category":"type","text":"Cmd(cmd::Cmd; ignorestatus, detach, windows_verbatim, windows_hide, env, dir)\n\nConstruct a new Cmd object, representing an external program and arguments, from cmd, while changing the settings of the optional keyword arguments:\n\nignorestatus::Bool: If true (defaults to false), then the Cmd will not throw an error if the return code is nonzero.\ndetach::Bool: If true (defaults to false), then the Cmd will be run in a new process group, allowing it to outlive the julia process and not have Ctrl-C passed to it.\nwindows_verbatim::Bool: If true (defaults to false), then on Windows the Cmd will send a command-line string to the process with no quoting or escaping of arguments, even arguments containing spaces. (On Windows, arguments are sent to a program as a single \"command-line\" string, and programs are responsible for parsing it into arguments. By default, empty arguments and arguments with spaces or tabs are quoted with double quotes \" in the command line, and \\ or \" are preceded by backslashes. windows_verbatim=true is useful for launching programs that parse their command line in nonstandard ways.) Has no effect on non-Windows systems.\nwindows_hide::Bool: If true (defaults to false), then on Windows no new console window is displayed when the Cmd is executed. This has no effect if a console is already open or on non-Windows systems.\nenv: Set environment variables to use when running the Cmd. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", an array or tuple of \"var\"=>val pairs. In order to modify (rather than replace) the existing environment, initialize env with copy(ENV) and then set env[\"var\"]=val as desired.  To add to an environment block within a Cmd object without replacing all elements, use addenv() which will return a Cmd object with the updated environment.\ndir::AbstractString: Specify a working directory for the command (instead of the current directory).\n\nFor any keywords that are not specified, the current settings from cmd are used. Normally, to create a Cmd object in the first place, one uses backticks, e.g.\n\nCmd(`echo \"Hello world\"`, ignorestatus=true, detach=false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setenv","location":"base/base.html#Base.setenv","category":"function","text":"setenv(command::Cmd, env; dir=\"\")\n\nSet environment variables to use when running the given command. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", or zero or more \"var\"=>val pair arguments. In order to modify (rather than replace) the existing environment, create env through copy(ENV) and then setting env[\"var\"]=val as desired, or use addenv.\n\nThe dir keyword argument can be used to specify a working directory for the command.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.withenv","location":"base/base.html#Base.withenv","category":"function","text":"withenv(f::Function, kv::Pair...)\n\nExecute f in an environment that is temporarily modified (not replaced as in setenv) by zero or more \"var\"=>val arguments kv. withenv is generally used via the withenv(kv...) do ... end syntax. A value of nothing can be used to temporarily unset an environment variable (if it is set). When withenv returns, the original environment has been restored.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pipeline","location":"base/base.html#Base.pipeline-Tuple{Any, Any, Any, Vararg{Any, N} where N}","category":"method","text":"pipeline(from, to, ...)\n\nCreate a pipeline from a data source to a destination. The source and destination can be commands, I/O streams, strings, or results of other pipeline calls. At least one argument must be a command. Strings refer to filenames. When called with more than two arguments, they are chained together from left to right. For example, pipeline(a,b,c) is equivalent to pipeline(pipeline(a,b),c). This provides a more concise way to specify multi-stage pipelines.\n\nExamples:\n\nrun(pipeline(`ls`, `grep xyz`))\nrun(pipeline(`ls`, \"out.txt\"))\nrun(pipeline(\"out.txt\", `grep xyz`))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pipeline","location":"base/base.html#Base.pipeline-Tuple{Base.AbstractCmd}","category":"method","text":"pipeline(command; stdin, stdout, stderr, append=false)\n\nRedirect I/O to or from the given command. Keyword arguments specify which of the command's streams should be redirected. append controls whether file output appends to the file. This is a more general version of the 2-argument pipeline function. pipeline(from, to) is equivalent to pipeline(from, stdout=to) when from is a command, and to pipeline(to, stdin=from) when from is another kind of data source.\n\nExamples:\n\nrun(pipeline(`dothings`, stdout=\"out.txt\", stderr=\"errs.txt\"))\nrun(pipeline(`update`, stdout=\"log.txt\", append=true))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.gethostname","location":"base/base.html#Base.Libc.gethostname","category":"function","text":"gethostname() -> AbstractString\n\nGet the local machine's host name.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.getpid","location":"base/base.html#Base.Libc.getpid","category":"function","text":"getpid() -> Int32\n\nGet Julia's process ID.\n\n\n\n\n\ngetpid(process) -> Int32\n\nGet the child process ID, if it still exists.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.time","location":"base/base.html#Base.Libc.time-Tuple{}","category":"method","text":"time()\n\nGet the system time in seconds since the epoch, with fairly high (typically, microsecond) resolution.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.time_ns","location":"base/base.html#Base.time_ns","category":"function","text":"time_ns()\n\nGet the time in nanoseconds. The time corresponding to 0 is undefined, and wraps every 5.8 years.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@time","location":"base/base.html#Base.@time","category":"macro","text":"@time\n\nA macro to execute an expression, printing the time it took to execute, the number of allocations, and the total number of bytes its execution caused to be allocated, before returning the value of the expression. Any time spent garbage collecting (gc) or compiling is shown as a percentage.\n\nIn some cases the system will look inside the @time expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @time @eval ....\n\nSee also @timev, @timed, @elapsed, and @allocated.\n\nnote: Note\nFor more serious benchmarking, consider the @btime macro from the BenchmarkTools.jl package which among other things evaluates the function multiple times in order to reduce noise.\n\njulia> x = rand(10,10);\n\njulia> @time x * x;\n  0.606588 seconds (2.19 M allocations: 116.555 MiB, 3.75% gc time, 99.94% compilation time)\n\njulia> @time x * x;\n  0.000009 seconds (1 allocation: 896 bytes)\n\njulia> @time begin\n           sleep(0.3)\n           1+1\n       end\n  0.301395 seconds (8 allocations: 336 bytes)\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@timev","location":"base/base.html#Base.@timev","category":"macro","text":"@timev\n\nThis is a verbose version of the @time macro. It first prints the same information as @time, then any non-zero memory allocation counters, and then returns the value of the expression.\n\nSee also @time, @timed, @elapsed, and @allocated.\n\njulia> x = rand(10,10);\n\njulia> @timev x * x;\n  0.546770 seconds (2.20 M allocations: 116.632 MiB, 4.23% gc time, 99.94% compilation time)\nelapsed time (ns): 546769547\ngc time (ns):      23115606\nbytes allocated:   122297811\npool allocs:       2197930\nnon-pool GC allocs:1327\nmalloc() calls:    36\nrealloc() calls:   5\nGC pauses:         3\n\njulia> @timev x * x;\n  0.000010 seconds (1 allocation: 896 bytes)\nelapsed time (ns): 9848\nbytes allocated:   896\npool allocs:       1\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@timed","location":"base/base.html#Base.@timed","category":"macro","text":"@timed\n\nA macro to execute an expression, and return the value of the expression, elapsed time, total bytes allocated, garbage collection time, and an object with various memory allocation counters.\n\nIn some cases the system will look inside the @timed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @timed @eval ....\n\nSee also @time, @timev, @elapsed, and @allocated.\n\njulia> stats = @timed rand(10^6);\n\njulia> stats.time\n0.006634834\n\njulia> stats.bytes\n8000256\n\njulia> stats.gctime\n0.0055765\n\njulia> propertynames(stats.gcstats)\n(:allocd, :malloc, :realloc, :poolalloc, :bigalloc, :freecall, :total_time, :pause, :full_sweep)\n\njulia> stats.gcstats.total_time\n5576500\n\ncompat: Julia 1.5\nThe return type of this macro was changed from Tuple to NamedTuple in Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@elapsed","location":"base/base.html#Base.@elapsed","category":"macro","text":"@elapsed\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the number of seconds it took to execute as a floating-point number.\n\nIn some cases the system will look inside the @elapsed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @elapsed @eval ....\n\nSee also @time, @timev, @timed, and @allocated.\n\njulia> @elapsed sleep(0.3)\n0.301391426\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@allocated","location":"base/base.html#Base.@allocated","category":"macro","text":"@allocated\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the total number of bytes allocated during evaluation of the expression.\n\nSee also @time, @timev, @timed, and @elapsed.\n\njulia> @allocated rand(10^6)\n8000080\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.EnvDict","location":"base/base.html#Base.EnvDict","category":"type","text":"EnvDict() -> EnvDict\n\nA singleton of this type provides a hash table interface to environment variables.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ENV","location":"base/base.html#Base.ENV","category":"constant","text":"ENV\n\nReference to the singleton EnvDict, providing a dictionary interface to system environment variables.\n\n(On Windows, system environment variables are case-insensitive, and ENV correspondingly converts all keys to uppercase for display, iteration, and copying. Portable code should not rely on the ability to distinguish variables by case, and should beware that setting an ostensibly lowercase variable may result in an uppercase ENV key.)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isunix","location":"base/base.html#Base.Sys.isunix","category":"function","text":"Sys.isunix([os])\n\nPredicate for testing if the OS provides a Unix-like interface. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isapple","location":"base/base.html#Base.Sys.isapple","category":"function","text":"Sys.isapple([os])\n\nPredicate for testing if the OS is a derivative of Apple Macintosh OS X or Darwin. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.islinux","location":"base/base.html#Base.Sys.islinux","category":"function","text":"Sys.islinux([os])\n\nPredicate for testing if the OS is a derivative of Linux. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isbsd","location":"base/base.html#Base.Sys.isbsd","category":"function","text":"Sys.isbsd([os])\n\nPredicate for testing if the OS is a derivative of BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nThe Darwin kernel descends from BSD, which means that Sys.isbsd() is true on macOS systems. To exclude macOS from a predicate, use Sys.isbsd() && !Sys.isapple().\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isfreebsd","location":"base/base.html#Base.Sys.isfreebsd","category":"function","text":"Sys.isfreebsd([os])\n\nPredicate for testing if the OS is a derivative of FreeBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on FreeBSD but also on other BSD-based systems. Sys.isfreebsd() refers only to FreeBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isopenbsd","location":"base/base.html#Base.Sys.isopenbsd","category":"function","text":"Sys.isopenbsd([os])\n\nPredicate for testing if the OS is a derivative of OpenBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on OpenBSD but also on other BSD-based systems. Sys.isopenbsd() refers only to OpenBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isnetbsd","location":"base/base.html#Base.Sys.isnetbsd","category":"function","text":"Sys.isnetbsd([os])\n\nPredicate for testing if the OS is a derivative of NetBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on NetBSD but also on other BSD-based systems. Sys.isnetbsd() refers only to NetBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isdragonfly","location":"base/base.html#Base.Sys.isdragonfly","category":"function","text":"Sys.isdragonfly([os])\n\nPredicate for testing if the OS is a derivative of DragonFly BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on DragonFly but also on other BSD-based systems. Sys.isdragonfly() refers only to DragonFly.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.iswindows","location":"base/base.html#Base.Sys.iswindows","category":"function","text":"Sys.iswindows([os])\n\nPredicate for testing if the OS is a derivative of Microsoft Windows NT. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.windows_version","location":"base/base.html#Base.Sys.windows_version","category":"function","text":"Sys.windows_version()\n\nReturn the version number for the Windows NT Kernel as a VersionNumber, i.e. v\"major.minor.build\", or v\"0.0.0\" if this is not running on Windows.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.free_memory","location":"base/base.html#Base.Sys.free_memory","category":"function","text":"Sys.free_memory()\n\nGet the total free memory in RAM in bytes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.total_memory","location":"base/base.html#Base.Sys.total_memory","category":"function","text":"Sys.total_memory()\n\nGet the total memory in RAM (including that which is currently used) in bytes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@static","location":"base/base.html#Base.@static","category":"macro","text":"@static\n\nPartially evaluate an expression at parse time.\n\nFor example, @static Sys.iswindows() ? foo : bar will evaluate Sys.iswindows() and insert either foo or bar into the expression. This is useful in cases where a construct would be invalid on other platforms, such as a ccall to a non-existent function. @static if Sys.isapple() foo end and @static foo <&&,||> bar are also valid syntax.\n\n\n\n\n\n","page":"Essentials"},{"title":"Versioning","location":"base/base.html#Versioning","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.VersionNumber\nBase.@v_str","page":"Essentials"},{"title":"Base.VersionNumber","location":"base/base.html#Base.VersionNumber","category":"type","text":"VersionNumber\n\nVersion number type which follow the specifications of semantic versioning, composed of major, minor and patch numeric values, followed by pre-release and build alpha-numeric annotations. See also @v_str.\n\nExamples\n\njulia> VersionNumber(\"1.2.3\")\nv\"1.2.3\"\n\njulia> VersionNumber(\"2.0.1-rc1\")\nv\"2.0.1-rc1\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@v_str","location":"base/base.html#Base.@v_str","category":"macro","text":"@v_str\n\nString macro used to parse a string to a VersionNumber.\n\nExamples\n\njulia> v\"1.2.3\"\nv\"1.2.3\"\n\njulia> v\"2.0.1-rc1\"\nv\"2.0.1-rc1\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Errors","location":"base/base.html#Errors","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.error\nCore.throw\nBase.rethrow\nBase.backtrace\nBase.catch_backtrace\nBase.catch_stack\nBase.@assert\nBase.Experimental.register_error_hint\nBase.Experimental.show_error_hints\nBase.ArgumentError\nBase.AssertionError\nCore.BoundsError\nBase.CompositeException\nBase.DimensionMismatch\nCore.DivideError\nCore.DomainError\nBase.EOFError\nCore.ErrorException\nCore.InexactError\nCore.InterruptException\nBase.KeyError\nBase.LoadError\nBase.MethodError\nBase.MissingException\nCore.OutOfMemoryError\nCore.ReadOnlyMemoryError\nCore.OverflowError\nBase.ProcessFailedException\nCore.StackOverflowError\nBase.SystemError\nCore.TypeError\nCore.UndefKeywordError\nCore.UndefRefError\nCore.UndefVarError\nBase.StringIndexError\nBase.InitError\nBase.retry\nBase.ExponentialBackOff","page":"Essentials"},{"title":"Base.error","location":"base/base.html#Base.error","category":"function","text":"error(message::AbstractString)\n\nRaise an ErrorException with the given message.\n\n\n\n\n\nerror(msg...)\n\nRaise an ErrorException with the given message.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.throw","location":"base/base.html#Core.throw","category":"function","text":"throw(e)\n\nThrow an object as an exception.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.rethrow","location":"base/base.html#Base.rethrow","category":"function","text":"rethrow()\n\nRethrow the current exception from within a catch block. The rethrown exception will continue propagation as if it had not been caught.\n\nnote: Note\nThe alternative form rethrow(e) allows you to associate an alternative exception object e with the current backtrace. However this misrepresents the program state at the time of the error so you're encouraged to instead throw a new exception using throw(e). In Julia 1.1 and above, using throw(e) will preserve the root cause exception on the stack, as described in catch_stack.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.backtrace","location":"base/base.html#Base.backtrace","category":"function","text":"backtrace()\n\nGet a backtrace object for the current program point.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.catch_backtrace","location":"base/base.html#Base.catch_backtrace","category":"function","text":"catch_backtrace()\n\nGet the backtrace of the current exception, for use within catch blocks.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.catch_stack","location":"base/base.html#Base.catch_stack","category":"function","text":"catch_stack(task=current_task(); [inclue_bt=true])\n\nGet the stack of exceptions currently being handled. For nested catch blocks there may be more than one current exception in which case the most recently thrown exception is last in the stack. The stack is returned as a Vector of (exception,backtrace) pairs, or a Vector of exceptions if include_bt is false.\n\nExplicitly passing task will return the current exception stack on an arbitrary task. This is useful for inspecting tasks which have failed due to uncaught exceptions.\n\ncompat: Julia 1.1\nThis function is experimental in Julia 1.1 and will likely be renamed in a future release (see https://github.com/JuliaLang/julia/pull/29901).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@assert","location":"base/base.html#Base.@assert","category":"macro","text":"@assert cond [text]\n\nThrow an AssertionError if cond is false. Preferred syntax for writing assertions. Message text is optionally displayed upon assertion failure.\n\nwarning: Warning\nAn assert might be disabled at various optimization levels. Assert should therefore only be used as a debugging tool and not used for authentication verification (e.g., verifying passwords), nor should side effects needed for the function to work correctly be used inside of asserts.\n\nExamples\n\njulia> @assert iseven(3) \"3 is an odd number!\"\nERROR: AssertionError: 3 is an odd number!\n\njulia> @assert isodd(3) \"What even are numbers?\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Experimental.register_error_hint","location":"base/base.html#Base.Experimental.register_error_hint","category":"function","text":"Experimental.register_error_hint(handler, exceptiontype)\n\nRegister a \"hinting\" function handler(io, exception) that can suggest potential ways for users to circumvent errors.  handler should examine exception to see whether the conditions appropriate for a hint are met, and if so generate output to io. Packages should call register_error_hint from within their __init__ function.\n\nFor specific exception types, handler is required to accept additional arguments:\n\nMethodError: provide handler(io, exc::MethodError, argtypes, kwargs), which splits the combined arguments into positional and keyword arguments.\n\nWhen issuing a hint, the output should typically start with \\n.\n\nIf you define custom exception types, your showerror method can support hints by calling Experimental.show_error_hints.\n\nExample\n\njulia> module Hinter\n\n       only_int(x::Int)      = 1\n       any_number(x::Number) = 2\n\n       function __init__()\n           Base.Experimental.register_error_hint(MethodError) do io, exc, argtypes, kwargs\n               if exc.f == only_int\n                    # Color is not necessary, this is just to show it's possible.\n                    print(io, \"\\nDid you mean to call \")\n                    printstyled(io, \"`any_number`?\", color=:cyan)\n               end\n           end\n       end\n\n       end\n\nThen if you call Hinter.only_int on something that isn't an Int (thereby triggering a MethodError), it issues the hint:\n\njulia> Hinter.only_int(1.0)\nERROR: MethodError: no method matching only_int(::Float64)\nDid you mean to call `any_number`?\nClosest candidates are:\n    ...\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice. To insulate yourself against changes, consider putting any registrations inside an if isdefined(Base.Experimental, :register_error_hint) ... end block.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Experimental.show_error_hints","location":"base/base.html#Base.Experimental.show_error_hints","category":"function","text":"Experimental.show_error_hints(io, ex, args...)\n\nInvoke all handlers from Experimental.register_error_hint for the particular exception type typeof(ex). args must contain any other arguments expected by the handler for that type.\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ArgumentError","location":"base/base.html#Core.ArgumentError","category":"type","text":"ArgumentError(msg)\n\nThe parameters to a function call do not match a valid signature. Argument msg is a descriptive error string.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.AssertionError","location":"base/base.html#Core.AssertionError","category":"type","text":"AssertionError([msg])\n\nThe asserted condition did not evaluate to true. Optional argument msg is a descriptive error string.\n\nExamples\n\njulia> @assert false \"this is not true\"\nERROR: AssertionError: this is not true\n\nAssertionError is usually thrown from @assert.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.BoundsError","location":"base/base.html#Core.BoundsError","category":"type","text":"BoundsError([a],[i])\n\nAn indexing operation into an array, a, tried to access an out-of-bounds element at index i.\n\nExamples\n\njulia> A = fill(1.0, 7);\n\njulia> A[8]\nERROR: BoundsError: attempt to access 7-element Vector{Float64} at index [8]\n\n\njulia> B = fill(1.0, (2,3));\n\njulia> B[2, 4]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [2, 4]\n\n\njulia> B[9]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [9]\n\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.CompositeException","location":"base/base.html#Base.CompositeException","category":"type","text":"CompositeException\n\nWrap a Vector of exceptions thrown by a Task (e.g. generated from a remote worker over a channel or an asynchronously executing local I/O write or a remote worker under pmap) with information about the series of exceptions. For example, if a group of workers are executing several tasks, and multiple workers fail, the resulting CompositeException will contain a \"bundle\" of information from each worker indicating where and why the exception(s) occurred.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.DimensionMismatch","location":"base/base.html#Base.DimensionMismatch","category":"type","text":"DimensionMismatch([msg])\n\nThe objects called do not have matching dimensionality. Optional argument msg is a descriptive error string.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DivideError","location":"base/base.html#Core.DivideError","category":"type","text":"DivideError()\n\nInteger division was attempted with a denominator value of 0.\n\nExamples\n\njulia> 2/0\nInf\n\njulia> div(2, 0)\nERROR: DivideError: integer division error\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DomainError","location":"base/base.html#Core.DomainError","category":"type","text":"DomainError(val)\nDomainError(val, msg)\n\nThe argument val to a function or constructor is outside the valid domain.\n\nExamples\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.EOFError","location":"base/base.html#Base.EOFError","category":"type","text":"EOFError()\n\nNo more data was available to read from a file or stream.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ErrorException","location":"base/base.html#Core.ErrorException","category":"type","text":"ErrorException(msg)\n\nGeneric error type. The error message, in the .msg field, may provide more specific details.\n\nExamples\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> ex.msg\n\"I've done a bad thing\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InexactError","location":"base/base.html#Core.InexactError","category":"type","text":"InexactError(name::Symbol, T, val)\n\nCannot exactly convert val to type T in a method of function name.\n\nExamples\n\njulia> convert(Float64, 1+2im)\nERROR: InexactError: Float64(1 + 2im)\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InterruptException","location":"base/base.html#Core.InterruptException","category":"type","text":"InterruptException()\n\nThe process was stopped by a terminal interrupt (CTRL+C).\n\nNote that, in Julia script started without -i (interactive) option, InterruptException is not thrown by default.  Calling Base.exit_on_sigint(false) in the script can recover the behavior of the REPL.  Alternatively, a Julia script can be started with\n\njulia -e \"include(popfirst!(ARGS))\" script.jl\n\nto let InterruptException be thrown by CTRL+C during the execution.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.KeyError","location":"base/base.html#Base.KeyError","category":"type","text":"KeyError(key)\n\nAn indexing operation into an AbstractDict (Dict) or Set like object tried to access or delete a non-existent element.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.LoadError","location":"base/base.html#Core.LoadError","category":"type","text":"LoadError(file::AbstractString, line::Int, error)\n\nAn error occurred while includeing, requireing, or using a file. The error specifics should be available in the .error field.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.MethodError","location":"base/base.html#Core.MethodError","category":"type","text":"MethodError(f, args)\n\nA method with the required type signature does not exist in the given generic function. Alternatively, there is no unique most-specific method.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MissingException","location":"base/base.html#Base.MissingException","category":"type","text":"MissingException(msg)\n\nException thrown when a missing value is encountered in a situation where it is not supported. The error message, in the msg field may provide more specific details.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.OutOfMemoryError","location":"base/base.html#Core.OutOfMemoryError","category":"type","text":"OutOfMemoryError()\n\nAn operation allocated too much memory for either the system or the garbage collector to handle properly.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ReadOnlyMemoryError","location":"base/base.html#Core.ReadOnlyMemoryError","category":"type","text":"ReadOnlyMemoryError()\n\nAn operation tried to write to memory that is read-only.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.OverflowError","location":"base/base.html#Core.OverflowError","category":"type","text":"OverflowError(msg)\n\nThe result of an expression is too large for the specified type and will cause a wraparound.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ProcessFailedException","location":"base/base.html#Base.ProcessFailedException","category":"type","text":"ProcessFailedException\n\nIndicates problematic exit status of a process. When running commands or pipelines, this is thrown to indicate a nonzero exit code was returned (i.e. that the invoked process failed).\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.StackOverflowError","location":"base/base.html#Core.StackOverflowError","category":"type","text":"StackOverflowError()\n\nThe function call grew beyond the size of the call stack. This usually happens when a call recurses infinitely.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.SystemError","location":"base/base.html#Base.SystemError","category":"type","text":"SystemError(prefix::AbstractString, [errno::Int32])\n\nA system call failed with an error code (in the errno global variable).\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.TypeError","location":"base/base.html#Core.TypeError","category":"type","text":"TypeError(func::Symbol, context::AbstractString, expected::Type, got)\n\nA type assertion failure, or calling an intrinsic function with an incorrect argument type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefKeywordError","location":"base/base.html#Core.UndefKeywordError","category":"type","text":"UndefKeywordError(var::Symbol)\n\nThe required keyword argument var was not assigned in a function call.\n\nExamples\n\njulia> function my_func(;my_arg)\n           return my_arg + 1\n       end\nmy_func (generic function with 1 method)\n\njulia> my_func()\nERROR: UndefKeywordError: keyword argument my_arg not assigned\nStacktrace:\n [1] my_func() at ./REPL[1]:2\n [2] top-level scope at REPL[2]:1\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefRefError","location":"base/base.html#Core.UndefRefError","category":"type","text":"UndefRefError()\n\nThe item or field is not defined for the given object.\n\nExamples\n\njulia> struct MyType\n           a::Vector{Int}\n           MyType() = new()\n       end\n\njulia> A = MyType()\nMyType(#undef)\n\njulia> A.a\nERROR: UndefRefError: access to undefined reference\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefVarError","location":"base/base.html#Core.UndefVarError","category":"type","text":"UndefVarError(var::Symbol)\n\nA symbol in the current scope is not defined.\n\nExamples\n\njulia> a\nERROR: UndefVarError: a not defined\n\njulia> a = 1;\n\njulia> a\n1\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.StringIndexError","location":"base/base.html#Base.StringIndexError","category":"type","text":"StringIndexError(str, i)\n\nAn error occurred when trying to access str at index i that is not valid.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InitError","location":"base/base.html#Core.InitError","category":"type","text":"InitError(mod::Symbol, error)\n\nAn error occurred when running a module's __init__ function. The actual error thrown is available in the .error field.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.retry","location":"base/base.html#Base.retry","category":"function","text":"retry(f;  delays=ExponentialBackOff(), check=nothing) -> Function\n\nReturn an anonymous function that calls function f.  If an exception arises, f is repeatedly called again, each time check returns true, after waiting the number of seconds specified in delays.  check should input delays's current state and the Exception.\n\ncompat: Julia 1.2\nBefore Julia 1.2 this signature was restricted to f::Function.\n\nExamples\n\nretry(f, delays=fill(5.0, 3))\nretry(f, delays=rand(5:10, 2))\nretry(f, delays=Base.ExponentialBackOff(n=3, first_delay=5, max_delay=1000))\nretry(http_get, check=(s,e)->e.status == \"503\")(url)\nretry(read, check=(s,e)->isa(e, IOError))(io, 128; all=false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ExponentialBackOff","location":"base/base.html#Base.ExponentialBackOff","category":"type","text":"ExponentialBackOff(; n=1, first_delay=0.05, max_delay=10.0, factor=5.0, jitter=0.1)\n\nA Float64 iterator of length n whose elements exponentially increase at a rate in the interval factor * (1 ± jitter).  The first element is first_delay and all elements are clamped to max_delay.\n\n\n\n\n\n","page":"Essentials"},{"title":"Events","location":"base/base.html#Events","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.Timer(::Function, ::Real)\nBase.Timer\nBase.AsyncCondition\nBase.AsyncCondition(::Function)","page":"Essentials"},{"title":"Base.Timer","location":"base/base.html#Base.Timer-Tuple{Function, Real}","category":"method","text":"Timer(callback::Function, delay; interval = 0)\n\nCreate a timer that runs the function callback at each timer expiration.\n\nWaiting tasks are woken and the function callback is called after an initial delay of delay seconds, and then repeating with the given interval in seconds. If interval is equal to 0, the callback is only run once. The function callback is called with a single argument, the timer itself. Stop a timer by calling close. The cb may still be run one final time, if the timer has already expired.\n\nExamples\n\nHere the first number is printed after a delay of two seconds, then the following numbers are printed quickly.\n\njulia> begin\n           i = 0\n           cb(timer) = (global i += 1; println(i))\n           t = Timer(cb, 2, interval=0.2)\n           wait(t)\n           sleep(0.5)\n           close(t)\n       end\n1\n2\n3\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Timer","location":"base/base.html#Base.Timer","category":"type","text":"Timer(delay; interval = 0)\n\nCreate a timer that wakes up tasks waiting for it (by calling wait on the timer object).\n\nWaiting tasks are woken after an initial delay of at least delay seconds, and then repeating after at least interval seconds again elapse. If interval is equal to 0, the timer is only triggered once. When the timer is closed (by close) waiting tasks are woken with an error. Use isopen to check whether a timer is still active.\n\nNote: interval is subject to accumulating time skew. If you need precise events at a particular absolute time, create a new timer at each expiration with the difference to the next time computed.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.AsyncCondition","location":"base/base.html#Base.AsyncCondition","category":"type","text":"AsyncCondition()\n\nCreate a async condition that wakes up tasks waiting for it (by calling wait on the object) when notified from C by a call to uv_async_send. Waiting tasks are woken with an error when the object is closed (by close). Use isopen to check whether it is still active.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.AsyncCondition","location":"base/base.html#Base.AsyncCondition-Tuple{Function}","category":"method","text":"AsyncCondition(callback::Function)\n\nCreate a async condition that calls the given callback function. The callback is passed one argument, the async condition object itself.\n\n\n\n\n\n","page":"Essentials"},{"title":"Reflection","location":"base/base.html#Reflection","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.nameof(::Module)\nBase.parentmodule\nBase.pathof(::Module)\nBase.moduleroot\nBase.@__MODULE__\nBase.@__FILE__\nBase.@__DIR__\nBase.@__LINE__\nBase.fullname\nBase.names\nCore.nfields\nBase.isconst\nBase.nameof(::Function)\nBase.functionloc(::Any, ::Any)\nBase.functionloc(::Method)","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{Module}","category":"method","text":"nameof(m::Module) -> Symbol\n\nGet the name of a Module as a Symbol.\n\nExamples\n\njulia> nameof(Base.Broadcast)\n:Broadcast\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.parentmodule","location":"base/base.html#Base.parentmodule","category":"function","text":"parentmodule(m::Module) -> Module\n\nGet a module's enclosing Module. Main is its own parent.\n\nExamples\n\njulia> parentmodule(Main)\nMain\n\njulia> parentmodule(Base.Broadcast)\nBase\n\n\n\n\n\nparentmodule(t::DataType) -> Module\n\nDetermine the module containing the definition of a (potentially UnionAll-wrapped) DataType.\n\nExamples\n\njulia> module Foo\n           struct Int end\n       end\nFoo\n\njulia> parentmodule(Int)\nCore\n\njulia> parentmodule(Foo.Int)\nFoo\n\n\n\n\n\nparentmodule(f::Function) -> Module\n\nDetermine the module containing the (first) definition of a generic function.\n\n\n\n\n\nparentmodule(f::Function, types) -> Module\n\nDetermine the module containing a given definition of a generic function.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pathof","location":"base/base.html#Base.pathof-Tuple{Module}","category":"method","text":"pathof(m::Module)\n\nReturn the path of the m.jl file that was used to import module m, or nothing if m was not imported from a package.\n\nUse dirname to get the directory part and basename to get the file name part of the path.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.moduleroot","location":"base/base.html#Base.moduleroot","category":"function","text":"moduleroot(m::Module) -> Module\n\nFind the root module of a given module. This is the first module in the chain of parent modules of m which is either a registered root module or which is its own parent module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__MODULE__","location":"base/base.html#Base.@__MODULE__","category":"macro","text":"@__MODULE__ -> Module\n\nGet the Module of the toplevel eval, which is the Module code is currently being read from.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__FILE__","location":"base/base.html#Base.@__FILE__","category":"macro","text":"@__FILE__ -> AbstractString\n\nExpand to a string with the path to the file containing the macrocall, or an empty string if evaluated by julia -e <expr>. Return nothing if the macro was missing parser source information. Alternatively see PROGRAM_FILE.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__DIR__","location":"base/base.html#Base.@__DIR__","category":"macro","text":"@__DIR__ -> AbstractString\n\nExpand to a string with the absolute path to the directory of the file containing the macrocall. Return the current working directory if run from a REPL or if evaluated by julia -e <expr>.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__LINE__","location":"base/base.html#Base.@__LINE__","category":"macro","text":"@__LINE__ -> Int\n\nExpand to the line number of the location of the macrocall. Return 0 if the line number could not be determined.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fullname","location":"base/base.html#Base.fullname","category":"function","text":"fullname(m::Module)\n\nGet the fully-qualified name of a module as a tuple of symbols. For example,\n\nExamples\n\njulia> fullname(Base.Iterators)\n(:Base, :Iterators)\n\njulia> fullname(Main)\n(:Main,)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.names","location":"base/base.html#Base.names","category":"function","text":"names(x::Module; all::Bool = false, imported::Bool = false)\n\nGet an array of the names exported by a Module, excluding deprecated names. If all is true, then the list also includes non-exported names defined in the module, deprecated names, and compiler-generated names. If imported is true, then names explicitly imported from other modules are also included.\n\nAs a special case, all names defined in Main are considered \"exported\", since it is not idiomatic to explicitly export names from Main.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.nfields","location":"base/base.html#Core.nfields","category":"function","text":"nfields(x) -> Int\n\nGet the number of fields in the given object.\n\nExamples\n\njulia> a = 1//2;\n\njulia> nfields(a)\n2\n\njulia> b = 1\n1\n\njulia> nfields(b)\n0\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> nfields(ex)\n1\n\nIn these examples, a is a Rational, which has two fields. b is an Int, which is a primitive bitstype with no fields at all. ex is an ErrorException, which has one field.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isconst","location":"base/base.html#Base.isconst","category":"function","text":"isconst(m::Module, s::Symbol) -> Bool\n\nDetermine whether a global is declared const in a given Module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{Function}","category":"method","text":"nameof(f::Function) -> Symbol\n\nGet the name of a generic Function as a symbol. For anonymous functions, this is a compiler-generated name. For explicitly-declared subtypes of Function, it is the name of the function's type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.functionloc","location":"base/base.html#Base.functionloc-Tuple{Any, Any}","category":"method","text":"functionloc(f::Function, types)\n\nReturns a tuple (filename,line) giving the location of a generic Function definition.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.functionloc","location":"base/base.html#Base.functionloc-Tuple{Method}","category":"method","text":"functionloc(m::Method)\n\nReturns a tuple (filename,line) giving the location of a Method definition.\n\n\n\n\n\n","page":"Essentials"},{"title":"Internals","location":"base/base.html#Internals","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Base.GC.gc\nBase.GC.enable\nBase.GC.@preserve\nBase.GC.safepoint\nMeta.lower\nMeta.@lower\nMeta.parse(::AbstractString, ::Int)\nMeta.parse(::AbstractString)\nMeta.ParseError\nCore.QuoteNode\nBase.macroexpand\nBase.@macroexpand\nBase.@macroexpand1\nBase.code_lowered\nBase.code_typed\nBase.precompile","page":"Essentials"},{"title":"Base.GC.gc","location":"base/base.html#Base.GC.gc","category":"function","text":"GC.gc([full=true])\n\nPerform garbage collection. The argument full determines the kind of collection: A full collection (default) sweeps all objects, which makes the next GC scan much slower, while an incremental collection may only sweep so-called young objects.\n\nwarning: Warning\nExcessive use will likely lead to poor performance.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.enable","location":"base/base.html#Base.GC.enable","category":"function","text":"GC.enable(on::Bool)\n\nControl whether garbage collection is enabled using a boolean argument (true for enabled, false for disabled). Return previous GC state.\n\nwarning: Warning\nDisabling garbage collection should be used only with caution, as it can cause memory use to grow without bound.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.@preserve","location":"base/base.html#Base.GC.@preserve","category":"macro","text":"GC.@preserve x1 x2 ... xn expr\n\nMark the objects x1, x2, ... as being in use during the evaluation of the expression expr. This is only required in unsafe code where expr implicitly uses memory or other resources owned by one of the xs.\n\nImplicit use of x covers any indirect use of resources logically owned by x which the compiler cannot see. Some examples:\n\nAccessing memory of an object directly via a Ptr\nPassing a pointer to x to ccall\nUsing resources of x which would be cleaned up in the finalizer.\n\n@preserve should generally not have any performance impact in typical use cases where it briefly extends object lifetime. In implementation, @preserve has effects such as protecting dynamically allocated objects from garbage collection.\n\nExamples\n\nWhen loading from a pointer with unsafe_load, the underlying object is implicitly used, for example x is implicitly used by unsafe_load(p) in the following:\n\njulia> let\n           x = Ref{Int}(101)\n           p = Base.unsafe_convert(Ptr{Int}, x)\n           GC.@preserve x unsafe_load(p)\n       end\n101\n\nWhen passing pointers to ccall, the pointed-to object is implicitly used and should be preserved. (Note however that you should normally just pass x directly to ccall which counts as an explicit use.)\n\njulia> let\n           x = \"Hello\"\n           p = pointer(x)\n           Int(GC.@preserve x @ccall strlen(p::Cstring)::Csize_t)\n           # Preferred alternative\n           Int(@ccall strlen(x::Cstring)::Csize_t)\n       end\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.safepoint","location":"base/base.html#Base.GC.safepoint","category":"function","text":"GC.safepoint()\n\nInserts a point in the program where garbage collection may run. This can be useful in rare cases in multi-threaded programs where some threads are allocating memory (and hence may need to run GC) but other threads are doing only simple operations (no allocation, task switches, or I/O). Calling this function periodically in non-allocating threads allows garbage collection to run.\n\ncompat: Julia 1.4\nThis function is available as of Julia 1.4.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.lower","location":"base/base.html#Base.Meta.lower","category":"function","text":"lower(m, x)\n\nTakes the expression x and returns an equivalent expression in lowered form for executing in module m. See also code_lowered.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.@lower","location":"base/base.html#Base.Meta.@lower","category":"macro","text":"@lower [m] x\n\nReturn lowered form of the expression x in module m. By default m is the module in which the macro is called. See also lower.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.parse","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString, Int32}","category":"method","text":"parse(str, start; greedy=true, raise=true, depwarn=true)\n\nParse the expression string and return an expression (which could later be passed to eval for execution). start is the index of the first character to start parsing. If greedy is true (default), parse will try to consume as much input as it can; otherwise, it will stop as soon as it has parsed a valid expression. Incomplete but otherwise syntactically valid expressions will return Expr(:incomplete, \"(error message)\"). If raise is true (default), syntax errors other than incomplete expressions will raise an error. If raise is false, parse will return an expression that will raise an error upon evaluation. If depwarn is false, deprecation warnings will be suppressed.\n\njulia> Meta.parse(\"x = 3, y = 5\", 7)\n(:(y = 5), 13)\n\njulia> Meta.parse(\"x = 3, y = 5\", 5)\n(:((3, y) = 5), 13)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.parse","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString}","category":"method","text":"parse(str; raise=true, depwarn=true)\n\nParse the expression string greedily, returning a single expression. An error is thrown if there are additional characters after the first expression. If raise is true (default), syntax errors will raise an error; otherwise, parse will return an expression that will raise an error upon evaluation.  If depwarn is false, deprecation warnings will be suppressed.\n\njulia> Meta.parse(\"x = 3\")\n:(x = 3)\n\njulia> Meta.parse(\"x = \")\n:($(Expr(:incomplete, \"incomplete: premature end of input\")))\n\njulia> Meta.parse(\"1.0.2\")\nERROR: Base.Meta.ParseError(\"invalid numeric constant \\\"1.0.\\\"\")\nStacktrace:\n[...]\n\njulia> Meta.parse(\"1.0.2\"; raise = false)\n:($(Expr(:error, \"invalid numeric constant \\\"1.0.\\\"\")))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.ParseError","location":"base/base.html#Base.Meta.ParseError","category":"type","text":"ParseError(msg)\n\nThe expression passed to the parse function could not be interpreted as a valid Julia expression.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.QuoteNode","location":"base/base.html#Core.QuoteNode","category":"type","text":"QuoteNode\n\nA quoted piece of code, that does not support interpolation. See the manual section about QuoteNodes for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.macroexpand","location":"base/base.html#Base.macroexpand","category":"function","text":"macroexpand(m::Module, x; recursive=true)\n\nTake the expression x and return an equivalent expression with all macros removed (expanded) for executing in module m. The recursive keyword controls whether deeper levels of nested macros are also expanded. This is demonstrated in the example below:\n\njulia> module M\n           macro m1()\n               42\n           end\n           macro m2()\n               :(@m1())\n           end\n       end\nM\n\njulia> macroexpand(M, :(@m2()), recursive=true)\n42\n\njulia> macroexpand(M, :(@m2()), recursive=false)\n:(#= REPL[16]:6 =# M.@m1)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@macroexpand","location":"base/base.html#Base.@macroexpand","category":"macro","text":"@macroexpand\n\nReturn equivalent expression with all macros removed (expanded).\n\nThere are differences between @macroexpand and macroexpand.\n\nWhile macroexpand takes a keyword argument recursive, @macroexpand is always recursive. For a non recursive macro version, see @macroexpand1.\nWhile macroexpand has an explicit module argument, @macroexpand always expands with respect to the module in which it is called.\n\nThis is best seen in the following example:\n\njulia> module M\n           macro m()\n               1\n           end\n           function f()\n               (@macroexpand(@m),\n                macroexpand(M, :(@m)),\n                macroexpand(Main, :(@m))\n               )\n           end\n       end\nM\n\njulia> macro m()\n           2\n       end\n@m (macro with 1 method)\n\njulia> M.f()\n(1, 1, 2)\n\nWith @macroexpand the expression expands where @macroexpand appears in the code (module M in the example). With macroexpand the expression expands in the module given as the first argument.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@macroexpand1","location":"base/base.html#Base.@macroexpand1","category":"macro","text":"@macroexpand1\n\nNon recursive version of @macroexpand.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.code_lowered","location":"base/base.html#Base.code_lowered","category":"function","text":"code_lowered(f, types; generated=true, debuginfo=:default)\n\nReturn an array of the lowered forms (IR) for the methods matching the given generic function and type signature.\n\nIf generated is false, the returned CodeInfo instances will correspond to fallback implementations. An error is thrown if no fallback implementation exists. If generated is true, these CodeInfo instances will correspond to the method bodies yielded by expanding the generators.\n\nThe keyword debuginfo controls the amount of code metadata present in the output.\n\nNote that an error will be thrown if types are not leaf types when generated is true and any of the corresponding methods are an @generated method.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.code_typed","location":"base/base.html#Base.code_typed","category":"function","text":"code_typed(f, types; optimize=true, debuginfo=:default)\n\nReturns an array of type-inferred lowered form (IR) for the methods matching the given generic function and type signature. The keyword argument optimize controls whether additional optimizations, such as inlining, are also applied. The keyword debuginfo controls the amount of code metadata present in the output, possible options are :source or :none.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.precompile","location":"base/base.html#Base.precompile","category":"function","text":"precompile(f, args::Tuple{Vararg{Any}})\n\nCompile the given function f for the argument tuple (of types) args, but do not execute it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Meta","location":"base/base.html#Meta","category":"section","text":"","page":"Essentials"},{"title":"Essentials","location":"base/base.html","category":"page","text":"Meta.quot\nMeta.isexpr\nMeta.isidentifier\nMeta.isoperator\nMeta.isunaryoperator\nMeta.isbinaryoperator\nMeta.show_sexpr","page":"Essentials"},{"title":"Base.Meta.quot","location":"base/base.html#Base.Meta.quot","category":"function","text":"Meta.quot(ex)::Expr\n\nQuote expression ex to produce an expression with head quote. This can for instance be used to represent objects of type Expr in the AST. See also the manual section about QuoteNode.\n\nExamples\n\njulia> eval(Meta.quot(:x))\n:x\n\njulia> dump(Meta.quot(:x))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Symbol x\n\njulia> eval(Meta.quot(:(1+2)))\n:(1 + 2)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.isexpr","location":"base/base.html#Base.Meta.isexpr","category":"function","text":"Meta.isexpr(ex, head[, n])::Bool\n\nReturn true if ex is an Expr with the given type head and optionally that the argument list is of length n. head may be a Symbol or collection of Symbols. For example, to check that a macro was passed a function call expression, you might use isexpr(ex, :call).\n\nExamples\n\njulia> ex = :(f(x))\n:(f(x))\n\njulia> Meta.isexpr(ex, :block)\nfalse\n\njulia> Meta.isexpr(ex, :call)\ntrue\n\njulia> Meta.isexpr(ex, [:block, :call]) # multiple possible heads\ntrue\n\njulia> Meta.isexpr(ex, :call, 1)\nfalse\n\njulia> Meta.isexpr(ex, :call, 2)\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isidentifier","location":"base/base.html#Base.isidentifier","category":"function","text":" isidentifier(s) -> Bool\n\nReturn whether the symbol or string s contains characters that are parsed as a valid identifier in Julia code.\n\nInternally Julia allows any sequence of characters in a Symbol (except \\0s), and macros automatically use variable names containing # in order to avoid naming collision with the surrounding code. In order for the parser to recognize a variable, it uses a limited set of characters (greatly extended by Unicode). isidentifier() makes it possible to query the parser directly whether a symbol contains valid characters.\n\nExamples\n\njulia> Meta.isidentifier(:x), Meta.isidentifier(\"1x\")\n(true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isoperator","location":"base/base.html#Base.isoperator","category":"function","text":"isoperator(s::Symbol)\n\nReturn true if the symbol can be used as an operator, false otherwise.\n\nExamples\n\njulia> Meta.isoperator(:+), Meta.isoperator(:f)\n(true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isunaryoperator","location":"base/base.html#Base.isunaryoperator","category":"function","text":"isunaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a unary (prefix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isunaryoperator(:-), Meta.isunaryoperator(:√), Meta.isunaryoperator(:f)\n(true, true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbinaryoperator","location":"base/base.html#Base.isbinaryoperator","category":"function","text":"isbinaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a binary (infix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isbinaryoperator(:-), Meta.isbinaryoperator(:√), Meta.isbinaryoperator(:f)\n(true, false, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.show_sexpr","location":"base/base.html#Base.Meta.show_sexpr","category":"function","text":"Meta.show_sexpr([io::IO,], ex)\n\nShow expression ex as a lisp style S-expression.\n\nExamples\n\njulia> Meta.show_sexpr(:(f(x, g(y,z))))\n(:call, :f, :x, (:call, :g, :y, :z))\n\n\n\n\n\n","page":"Essentials"},{"title":"Bounds checking","location":"devdocs/boundscheck.html#Bounds-checking","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection.","page":"Bounds checking"},{"title":"Eliding bounds checks","location":"devdocs/boundscheck.html#Eliding-bounds-checks","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"function sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i = 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"With a custom array-like type MyArray having:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A,i); A.data[to_index(i)])","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Then when getindex is inlined into sum, the call to checkbounds(A,i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack.","page":"Bounds checking"},{"title":"Propagating inbounds","location":"devdocs/boundscheck.html#Propagating-inbounds","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"To override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining.","page":"Bounds checking"},{"title":"The bounds checking call hierarchy","location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"The overall hierarchy is:","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Here A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"checkbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"so checkindex checks a single dimension.  All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"If you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"If you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array.  If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Note this hierarchy has been designed to reduce the likelihood of method ambiguities.  We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument).","page":"Bounds checking"},{"title":"Emit bounds checks","location":"devdocs/boundscheck.html#Emit-bounds-checks","category":"section","text":"","page":"Bounds checking"},{"title":"Bounds checking","location":"devdocs/boundscheck.html","category":"page","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations.","page":"Bounds checking"},{"title":"Multi-Threading","location":"base/multi-threading.html#lib-multithreading","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.@threads\nBase.Threads.foreach\nBase.Threads.@spawn\nBase.Threads.threadid\nBase.Threads.nthreads","page":"Multi-Threading"},{"title":"Base.Threads.@threads","location":"base/multi-threading.html#Base.Threads.@threads","category":"macro","text":"Threads.@threads [schedule] for ... end\n\nA macro to parallelize a for loop to run with multiple threads. Splits the iteration space among multiple tasks and runs those tasks on threads according to a scheduling policy. A barrier is placed at the end of the loop which waits for all tasks to finish execution.\n\nThe schedule argument can be used to request a particular scheduling policy. The only currently supported value is :static, which creates one task per thread and divides the iterations equally among them. Specifying :static is an error if used from inside another @threads loop or from a thread other than 1.\n\nThe default schedule (used when no schedule argument is present) is subject to change.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.foreach","location":"base/multi-threading.html#Base.Threads.foreach","category":"function","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.nthreads())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to f are split across ntasks tasks spawned by Threads.@spawn. This function will wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a manner that enables Julia's scheduler to more freely load-balance work items across threads. This approach generally has higher per-item overhead, but may perform better than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner that incurs lower per-item overhead than FairSchedule, but is less amenable to load-balancing. This approach thus may be more suitable for fine-grained, uniform workloads, but may perform worse than FairSchedule in concurrence with other multithreaded workloads.\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.@spawn","location":"base/multi-threading.html#Base.Threads.@spawn","category":"macro","text":"Threads.@spawn expr\n\nCreate and run a Task on any available thread. To wait for the task to finish, call wait on the result of this macro, or call fetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nnote: Note\nSee the manual chapter on threading for important caveats.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadid","location":"base/multi-threading.html#Base.Threads.threadid","category":"function","text":"Threads.threadid()\n\nGet the ID number of the current thread of execution. The master thread has ID 1.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.nthreads","location":"base/multi-threading.html#Base.Threads.nthreads","category":"function","text":"Threads.nthreads()\n\nGet the number of threads available to the Julia process. This is the inclusive upper bound on threadid().\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Synchronization","location":"base/multi-threading.html#Synchronization","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.Condition\nBase.Threads.Event","page":"Multi-Threading"},{"title":"Base.Threads.Condition","location":"base/multi-threading.html#Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call lock on it. When wait is called, the lock is atomically released during blocking, and will be reacquired before wait returns. Therefore idiomatic use of a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Event","location":"base/multi-threading.html#Base.Event","category":"type","text":"Event()\n\nCreate a level-triggered event source. Tasks that call wait on an Event are suspended and queued until notify is called on the Event. After notify is called, the Event remains in a signaled state and tasks will no longer block when waiting for it.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"See also Synchronization.","page":"Multi-Threading"},{"title":"Atomic operations","location":"base/multi-threading.html#Atomic-operations","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"warning: Warning\nThe API for atomic operations has not yet been finalized and is likely to change.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.Atomic\nBase.Threads.atomic_cas!\nBase.Threads.atomic_xchg!\nBase.Threads.atomic_add!\nBase.Threads.atomic_sub!\nBase.Threads.atomic_and!\nBase.Threads.atomic_nand!\nBase.Threads.atomic_or!\nBase.Threads.atomic_xor!\nBase.Threads.atomic_max!\nBase.Threads.atomic_min!\nBase.Threads.atomic_fence","page":"Multi-Threading"},{"title":"Base.Threads.Atomic","location":"base/multi-threading.html#Base.Threads.Atomic","category":"type","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only accessed atomically, i.e. in a thread-safe manner.\n\nOnly certain \"simple\" types can be used atomically, namely the primitive boolean, integer, and float-point types. These are Bool, Int8...Int128, UInt8...UInt128, and Float16...Float64.\n\nNew atomic objects can be created from a non-atomic values; if none is specified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!, atomic_xchg!, etc.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_cas!","location":"base/multi-threading.html#Base.Threads.atomic_cas!","category":"function","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write newval to x. Otherwise, leaves x unmodified. Returns the old value in x. By comparing the returned value to cmp (via ===) one knows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before the transaction, one records the value in x. After the transaction, the new value is stored only if x has not been modified in the mean time.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xchg!","location":"base/multi-threading.html#Base.Threads.atomic_xchg!","category":"function","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_add!","location":"base/multi-threading.html#Base.Threads.atomic_add!","category":"function","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_sub!","location":"base/multi-threading.html#Base.Threads.atomic_sub!","category":"function","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_and!","location":"base/multi-threading.html#Base.Threads.atomic_and!","category":"function","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_nand!","location":"base/multi-threading.html#Base.Threads.atomic_nand!","category":"function","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_or!","location":"base/multi-threading.html#Base.Threads.atomic_or!","category":"function","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xor!","location":"base/multi-threading.html#Base.Threads.atomic_xor!","category":"function","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_max!","location":"base/multi-threading.html#Base.Threads.atomic_max!","category":"function","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_min!","location":"base/multi-threading.html#Base.Threads.atomic_min!","category":"function","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_fence","location":"base/multi-threading.html#Base.Threads.atomic_fence","category":"function","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering semantics. There are algorithms where this is needed, i.e. where an acquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic operations in Julia already have acquire/release semantics, explicit fences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"ccall using a threadpool (Experimental)","location":"base/multi-threading.html#ccall-using-a-threadpool-(Experimental)","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.@threadcall","page":"Multi-Threading"},{"title":"Base.@threadcall","location":"base/multi-threading.html#Base.@threadcall","category":"macro","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work in a different thread. This is useful when you want to call a blocking C function without causing the main julia thread to become blocked. Concurrency is limited by size of the libuv thread pool, which defaults to 4 threads but can be increased by setting the UV_THREADPOOL_SIZE environment variable and restarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Low-level synchronization primitives","location":"base/multi-threading.html#Low-level-synchronization-primitives","category":"section","text":"","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"These building blocks are used to create the regular synchronization objects.","page":"Multi-Threading"},{"title":"Multi-Threading","location":"base/multi-threading.html","category":"page","text":"Base.Threads.SpinLock","page":"Multi-Threading"},{"title":"Base.Threads.SpinLock","location":"base/multi-threading.html#Base.Threads.SpinLock","category":"type","text":"SpinLock()\n\nCreate a non-reentrant, test-and-test-and-set spin lock. Recursive use will result in a deadlock. This kind of lock should only be used around code that takes little time to execute and does not block (e.g. perform I/O). In general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock.\n\nTest-and-test-and-set spin locks are quickest up to about 30ish contending threads. If you have more contention than that, different synchronization approaches should be considered.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Documentation","location":"manual/documentation.html#man-documentation","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Julia enables package developers and users to document functions, types and other objects easily via a built-in documentation system.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"The basic syntax is simple: any string appearing at the toplevel right before an object (function, macro, type or instance) will be interpreted as documenting it (these are called docstrings). Note that no blank lines or comments may intervene between a docstring and the documented object. Here is a basic example:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"Tell whether there are too foo items in the array.\"\nfoo(xs::Array) = ...","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Documentation is interpreted as Markdown, so you can use indentation and code fences to delimit code examples from text. Technically, any object can be associated with any other as metadata; Markdown happens to be the default, but one can construct other string macros and pass them to the @doc macro just as well.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"note: Note\nMarkdown support is implemented in the Markdown standard library and for a full list of supported syntax see the documentation.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Here is a more complex example, still using Markdown:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"\"\"\n    bar(x[, y])\n\nCompute the Bar index between `x` and `y`.\n\nIf `y` is unspecified, compute the Bar index between all pairs of columns of `x`.\n\n# Examples\n```julia-repl\njulia> bar([1, 2], [1, 2])\n1\n```\n\"\"\"\nfunction bar(x, y) ...","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"As in the example above, we recommend following some simple conventions when writing documentation:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Always show the signature of a function at the top of the documentation, with a four-space indent so that it is printed as Julia code.\nThis can be identical to the signature present in the Julia code (like mean(x::AbstractArray)), or a simplified form. Optional arguments should be represented with their default values (i.e. f(x, y=1)) when possible, following the actual Julia syntax. Optional arguments which do not have a default value should be put in brackets (i.e. f(x[, y]) and f(x[, y[, z]])). An alternative solution is to use several lines: one without optional arguments, the other(s) with them. This solution can also be used to document several related methods of a given function. When a function accepts many keyword arguments, only include a <keyword arguments> placeholder in the signature (i.e. f(x; <keyword arguments>)), and give the complete list under an # Arguments section (see point 4 below).\nInclude a single one-line sentence describing what the function does or what the object represents after the simplified signature block. If needed, provide more details in a second paragraph, after a blank line.\nThe one-line sentence should use the imperative form (\"Do this\", \"Return that\") instead of the third person (do not write \"Returns the length...\") when documenting functions. It should end with a period. If the meaning of a function cannot be summarized easily, splitting it into separate composable parts could be beneficial (this should not be taken as an absolute requirement for every single case though).\nDo not repeat yourself.\nSince the function name is given by the signature, there is no need to start the documentation with \"The function bar...\": go straight to the point. Similarly, if the signature specifies the types of the arguments, mentioning them in the description is redundant.\nOnly provide an argument list when really necessary.\nFor simple functions, it is often clearer to mention the role of the arguments directly in the description of the function's purpose. An argument list would only repeat information already provided elsewhere. However, providing an argument list can be a good idea for complex functions with many arguments (in particular keyword arguments). In that case, insert it after the general description of the function, under an # Arguments header, with one - bullet for each argument. The list should mention the types and default values (if any) of the arguments:\n\"\"\"\n...\n# Arguments\n- `n::Integer`: the number of elements to compute.\n- `dim::Integer=1`: the dimensions along which to perform the computation.\n...\n\"\"\"\nProvide hints to related functions.\nSometimes there are functions of related functionality. To increase discoverability please provide a short list of these in a See also: paragraph.\nSee also: [`bar!`](@ref), [`baz`](@ref), [`baaz`](@ref)\nInclude any code examples in an # Examples section.\nExamples should, whenever possible, be written as doctests. A doctest is a fenced code block (see Code blocks) starting with ```jldoctest and contains any number of julia> prompts together with inputs and expected outputs that mimic the Julia REPL.\nnote: Note\nDoctests are enabled by Documenter.jl. For more detailed documentation see Documenter's manual.\nFor example in the following docstring a variable a is defined and the expected result, as printed in a Julia REPL, appears afterwards:\n\"\"\"\nSome nice documentation here.\n\n# Examples\n```jldoctest\njulia> a = [1 2; 3 4]\n2×2 Array{Int64,2}:\n 1  2\n 3  4\n```\n\"\"\"\nwarning: Warning\nCalling rand and other RNG-related functions should be avoided in doctests since they will not produce consistent outputs during different Julia sessions. If you would like to show some random number generation related functionality, one option is to explicitly construct and seed your own MersenneTwister (or other pseudorandom number generator) and pass it to the functions you are doctesting.Operating system word size (Int32 or Int64) as well as path separator differences (/ or \\) will also affect the reproducibility of some doctests.Note that whitespace in your doctest is significant! The doctest will fail if you misalign the output of pretty-printing an array, for example.\nYou can then run make -C doc doctest=true to run all the doctests in the Julia Manual and API documentation, which will ensure that your example works.\nTo indicate that the output result is truncated, you may write [...] at the line where checking should stop. This is useful to hide a stacktrace (which contains non-permanent references to lines of julia code) when the doctest shows that an exception is thrown, for example:\n```jldoctest\njulia> div(1, 0)\nERROR: DivideError: integer division error\n[...]\n```\nExamples that are untestable should be written within fenced code blocks starting with ```julia so that they are highlighted correctly in the generated documentation.\ntip: Tip\nWherever possible examples should be self-contained and runnable so that readers are able to try them out without having to include any dependencies.\nUse backticks to identify code and equations.\nJulia identifiers and code excerpts should always appear between backticks ` to enable highlighting. Equations in the LaTeX syntax can be inserted between double backticks ``. Use Unicode characters rather than their LaTeX escape sequence, i.e. ``α = 1`` rather than ``\\\\alpha = 1``.\nPlace the starting and ending \"\"\" characters on lines by themselves.\nThat is, write:\n\"\"\"\n...\n\n...\n\"\"\"\nf(x, y) = ...\nrather than:\n\"\"\"...\n\n...\"\"\"\nf(x, y) = ...\nThis makes it clearer where docstrings start and end.\nRespect the line length limit used in the surrounding code.\nDocstrings are edited using the same tools as code. Therefore, the same conventions should apply. It is recommended that lines are at most 92 characters wide.\nProvide information allowing custom types to implement the function in an # Implementation section. These implementation details are intended for developers rather than users, explaining e.g. which functions should be overridden and which functions automatically use appropriate fallbacks. Such details are best kept separate from the main description of the function's behavior.\nFor long docstrings, consider splitting the documentation with an # Extended help header. The typical help-mode will show only the material above the header; you can access the full help by adding a '?' at the beginning of the expression (i.e., \"??foo\" rather than \"?foo\").","page":"Documentation"},{"title":"Accessing Documentation","location":"manual/documentation.html#Accessing-Documentation","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Documentation can be accessed at the REPL or in IJulia by typing ? followed by the name of a function or macro, and pressing Enter. For example,","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"?cos\n?@time\n?r\"\"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"will show documentation for the relevant function, macro or string macro respectively. In Juno using Ctrl-J, Ctrl-D will show the documentation for the object under the cursor.","page":"Documentation"},{"title":"Functions & Methods","location":"manual/documentation.html#Functions-and-Methods","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Functions in Julia may have multiple implementations, known as methods. While it's good practice for generic functions to have a single purpose, Julia allows methods to be documented individually if necessary. In general, only the most generic method should be documented, or even the function itself (i.e. the object created without any methods by function bar end). Specific methods should only be documented if their behaviour differs from the more generic ones. In any case, they should not repeat the information provided elsewhere. For example:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"\"\"\n    *(x, y, z...)\n\nMultiplication operator. `x * y * z *...` calls this function with multiple\narguments, i.e. `*(x, y, z...)`.\n\"\"\"\nfunction *(x, y, z...)\n    # ... [implementation sold separately] ...\nend\n\n\"\"\"\n    *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\nWhen applied to strings, concatenates them.\n\"\"\"\nfunction *(x::AbstractString, y::AbstractString, z::AbstractString...)\n    # ... [insert secret sauce here] ...\nend\n\nhelp?> *\nsearch: * .*\n\n  *(x, y, z...)\n\n  Multiplication operator. x * y * z *... calls this function with multiple\n  arguments, i.e. *(x,y,z...).\n\n  *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\n  When applied to strings, concatenates them.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"When retrieving documentation for a generic function, the metadata for each method is concatenated with the catdoc function, which can of course be overridden for custom types.","page":"Documentation"},{"title":"Advanced Usage","location":"manual/documentation.html#Advanced-Usage","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"The @doc macro associates its first argument with its second in a per-module dictionary called META.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"To make it easier to write documentation, the parser treats the macro name @doc specially: if a call to @doc has one argument, but another expression appears after a single line break, then that additional expression is added as an argument to the macro. Therefore the following syntax is parsed as a 2-argument call to @doc:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"@doc raw\"\"\"\n...\n\"\"\"\nf(x) = x","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"This makes it possible to use expressions other than normal string literals (such as the raw\"\" string macro) as a docstring.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"When used for retrieving documentation, the @doc macro (or equally, the doc function) will search all META dictionaries for metadata relevant to the given object and return it. The returned object (some Markdown content, for example) will by default display itself intelligently. This design also makes it easy to use the doc system in a programmatic way; for example, to re-use documentation between different versions of a function:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"@doc \"...\" foo!\n@doc (@doc foo!) foo","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Or for use with Julia's metaprogramming functionality:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"for (f, op) in ((:add, :+), (:subtract, :-), (:multiply, :*), (:divide, :/))\n    @eval begin\n        $f(a,b) = $op(a,b)\n    end\nend\n@doc \"`add(a,b)` adds `a` and `b` together\" add\n@doc \"`subtract(a,b)` subtracts `b` from `a`\" subtract","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Documentation written in non-toplevel blocks, such as begin, if, for, and let, is added to the documentation system as blocks are evaluated. For example:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"if condition()\n    \"...\"\n    f(x) = x\nend","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"will add documentation to f(x) when condition() is true. Note that even if f(x) goes out of scope at the end of the block, its documentation will remain.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"It is possible to make use of metaprogramming to assist in the creation of documentation. When using string-interpolation within the docstring you will need to use an extra $ as shown with $($name):","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"for func in (:day, :dayofmonth)\n    name = string(func)\n    @eval begin\n        @doc \"\"\"\n            $($name)(dt::TimeType) -> Int64\n\n        The day of month of a `Date` or `DateTime` as an `Int64`.\n        \"\"\" $func(dt::Dates.TimeType)\n    end\nend","page":"Documentation"},{"title":"Dynamic documentation","location":"manual/documentation.html#Dynamic-documentation","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Sometimes the appropriate documentation for an instance of a type depends on the field values of that instance, rather than just on the type itself. In these cases, you can add a method to Docs.getdoc for your custom type that returns the documentation on a per-instance basis. For instance,","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"struct MyType\n    value::String\nend\n\nDocs.getdoc(t::MyType) = \"Documentation for MyType with value $(t.value)\"\n\nx = MyType(\"x\")\ny = MyType(\"y\")","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"?x will display \"Documentation for MyType with value x\" while ?y will display \"Documentation for MyType with value y\".","page":"Documentation"},{"title":"Syntax Guide","location":"manual/documentation.html#Syntax-Guide","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"This guide provides a comprehensive overview of how to attach documentation to all Julia syntax constructs for which providing documentation is possible.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"In the following examples \"...\" is used to illustrate an arbitrary docstring.","page":"Documentation"},{"title":"$ and \\ characters","location":"manual/documentation.html#and-\\-characters","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"The $ and \\ characters are still parsed as string interpolation or start of an escape sequence in docstrings too. The raw\"\" string macro together with the @doc macro can be used to avoid having to escape them. This is handy when the docstrings include LaTeX or Julia source code examples containing interpolation:","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"@doc raw\"\"\"\n```math\n\\LaTeX\n```\n\"\"\"\nfunction f end","page":"Documentation"},{"title":"Functions and Methods","location":"manual/documentation.html#Functions-and-Methods-2","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nfunction f end\n\n\"...\"\nf","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the function f. The first version is the preferred syntax, however both are equivalent.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nf(x) = x\n\n\"...\"\nfunction f(x)\n    x\nend\n\n\"...\"\nf(x)","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the method f(::Any).","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nf(x, y = 1) = x + y","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to two Methods, namely f(::Any) and f(::Any, ::Any).","page":"Documentation"},{"title":"Macros","location":"manual/documentation.html#Macros","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nmacro m(x) end","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the @m(::Any) macro definition.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\n:(@m)","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the macro named @m.","page":"Documentation"},{"title":"Types","location":"manual/documentation.html#Types","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nabstract type T1 end\n\n\"...\"\nmutable struct T2\n    ...\nend\n\n\"...\"\nstruct T3\n    ...\nend","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds the docstring \"...\" to types T1, T2, and T3.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nstruct T\n    \"x\"\n    x\n    \"y\"\n    y\nend","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to type T, \"x\" to field T.x and \"y\" to field T.y. Also applicable to mutable struct types.","page":"Documentation"},{"title":"Modules","location":"manual/documentation.html#Modules","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nmodule M end\n\nmodule M\n\n\"...\"\nM\n\nend","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the Module M. Adding the docstring above the Module is the preferred syntax, however both are equivalent.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nbaremodule M\n# ...\nend\n\nbaremodule M\n\nimport Base: @doc\n\n\"...\"\nf(x) = x\n\nend","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Documenting a baremodule by placing a docstring above the expression automatically imports @doc into the module. These imports must be done manually when the module expression is not documented. Empty baremodules cannot be documented.","page":"Documentation"},{"title":"Global Variables","location":"manual/documentation.html#Global-Variables","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nconst a = 1\n\n\"...\"\nb = 2\n\n\"...\"\nglobal c = 3","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the Bindings a, b, and c.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Bindings are used to store a reference to a particular Symbol in a Module without storing the referenced value itself.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"note: Note\nWhen a const definition is only used to define an alias of another definition, such as is the case with the function div and its alias ÷ in Base, do not document the alias and instead document the actual function.If the alias is documented and not the real definition then the docsystem (? mode) will not return the docstring attached to the alias when the real definition is searched for.For example you should write\"...\"\nf(x) = x + 1\nconst alias = frather thanf(x) = x + 1\n\"...\"\nconst alias = f","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\nsym","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the value associated with sym. However, it is preferred that sym is documented where it is defined.","page":"Documentation"},{"title":"Multiple Objects","location":"manual/documentation.html#Multiple-Objects","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\na, b","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to a and b each of which should be a documentable expression. This syntax is equivalent to","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\na\n\n\"...\"\nb","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Any number of expressions many be documented together in this way. This syntax can be useful when two functions are related, such as non-mutating and mutating versions f and f!.","page":"Documentation"},{"title":"Macro-generated code","location":"manual/documentation.html#Macro-generated-code","category":"section","text":"","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"\"...\"\n@m expression","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Adds docstring \"...\" to the expression generated by expanding @m expression. This allows for expressions decorated with @inline, @noinline, @generated, or any other macro to be documented in the same way as undecorated expressions.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Macro authors should take note that only macros that generate a single expression will automatically support docstrings. If a macro returns a block containing multiple subexpressions then the subexpression that should be documented must be marked using the @__doc__ macro.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"The @enum macro makes use of @__doc__ to allow for documenting Enums. Examining its definition should serve as an example of how to use @__doc__ correctly.","page":"Documentation"},{"title":"Documentation","location":"manual/documentation.html","category":"page","text":"Core.@__doc__","page":"Documentation"},{"title":"Core.@__doc__","location":"manual/documentation.html#Core.@__doc__","category":"macro","text":"@__doc__(ex)\n\nLow-level macro used to mark expressions returned by a macro that should be documented. If more than one expression is marked then the same docstring is applied to each expression.\n\nmacro example(f)\n    quote\n        $(f)() = 0\n        @__doc__ $(f)(x) = 1\n        $(f)(x, y) = 2\n    end |> esc\nend\n\n@__doc__ has no effect when a macro that uses it is not documented.\n\n\n\n\n\n","page":"Documentation"},{"title":"Tasks","location":"base/parallel.html#Tasks","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Core.Task\nBase.@task\nBase.@async\nBase.asyncmap\nBase.asyncmap!\nBase.current_task\nBase.istaskdone\nBase.istaskstarted\nBase.istaskfailed\nBase.task_local_storage(::Any)\nBase.task_local_storage(::Any, ::Any)\nBase.task_local_storage(::Function, ::Any, ::Any)","page":"Tasks"},{"title":"Core.Task","location":"base/parallel.html#Core.Task","category":"type","text":"Task(func)\n\nCreate a Task (i.e. coroutine) to execute the given function func (which must be callable with no arguments). The task exits when this function returns.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@task","location":"base/parallel.html#Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only creates a task, and does not run it.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@async","location":"base/parallel.html#Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap","location":"base/parallel.html#Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple equal length collections). For multiple collection arguments, f is applied elementwise.\n\nntasks specifies the number of tasks to run concurrently. Depending on the length of the collections, if ntasks is unspecified, up to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the number of tasks to run in parallel is checked before processing every element and a new task started if the value of ntasks_func is less than the current number of tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must then be a function that must accept a Vector of argument tuples and must return a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning the objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Array{UInt64,1}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Array{UInt64,1}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array of argument tuples and return an array of results. map is used in the modified mapping function to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Array{String,1}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\nnote: Note\nCurrently, all tasks in Julia are executed in a single OS thread co-operatively. Consequently, asyncmap is beneficial only when the mapping function involves any I/O - disk, network, remote worker invocation, etc.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap!","location":"base/parallel.html#Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than returning a collection.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.current_task","location":"base/parallel.html#Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskdone","location":"base/parallel.html#Base.istaskdone","category":"function","text":"istaskdone(t::Task) -> Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskstarted","location":"base/parallel.html#Base.istaskstarted","category":"function","text":"istaskstarted(t::Task) -> Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskfailed","location":"base/parallel.html#Base.istaskfailed","category":"function","text":"istaskfailed(t::Task) -> Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to key; the previous value of key, or lack thereof, is restored afterwards. Useful for emulating dynamic scoping.\n\n\n\n\n\n","page":"Tasks"},{"title":"Scheduling","location":"base/parallel.html#Scheduling","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.yield\nBase.yieldto\nBase.sleep\nBase.schedule","page":"Tasks"},{"title":"Base.yield","location":"base/parallel.html#Base.yield","category":"function","text":"yield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this function is still runnable, and will be restarted immediately if there are no other runnable tasks.\n\n\n\n\n\nyield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which immediately yields to t before calling the scheduler.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.yieldto","location":"base/parallel.html#Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is called with no arguments. On subsequent switches, arg is returned from the task's last call to yieldto. This is a low-level call that only switches tasks, not considering states or scheduling in any way. Its use is discouraged.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.sleep","location":"base/parallel.html#Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1 millisecond or input of 0.001.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.schedule","location":"base/parallel.html#Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system is otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of yieldto) when it runs again. If error is true, the value is raised as an exception in the woken task.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Synchronization","location":"base/parallel.html#lib-task-sync","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.@sync\nBase.wait\nBase.fetch(t::Task)\nBase.timedwait\n\nBase.Condition\nBase.notify\n\nBase.Semaphore\nBase.acquire\nBase.release\n\nBase.AbstractLock\nBase.lock\nBase.unlock\nBase.trylock\nBase.islocked\nBase.ReentrantLock","page":"Tasks"},{"title":"Base.@sync","location":"base/parallel.html#Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn, @spawnat and @distributed are complete. All exceptions thrown by enclosed async operations are collected and thrown as a CompositeException.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.wait","location":"base/parallel.html#Base.wait","category":"function","text":"Special note for Threads.Condition:\n\nThe caller must be holding the lock that owns a Threads.Condition before calling this method. The calling task will be blocked until some other task wakes it, usually by calling notify on the same Threads.Condition object. The lock will be atomically released when blocking (even if it was locked recursively), and will be reacquired before returning.\n\n\n\n\n\nwait(r::Future)\n\nWait for a value to become available for the specified Future.\n\n\n\n\n\nwait(r::RemoteChannel, args...)\n\nWait for a value to become available on the specified RemoteChannel.\n\n\n\n\n\nwait([x])\n\nBlock the current task until some event occurs, depending on the type of the argument:\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process can be used to determine success or failure.\nTask: Wait for a Task to finish. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be restarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before proceeding.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Task}","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.timedwait","location":"base/parallel.html#Base.timedwait","category":"function","text":"timedwait(callback::Function, timeout::Real; pollint::Real=0.1)\n\nWaits until callback returns true or timeout seconds have passed, whichever is earlier. callback is polled every pollint seconds. The minimum value for timeout and pollint is 0.001, that is, 1 millisecond.\n\nReturns :ok or :timed_out\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Condition","location":"base/parallel.html#Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a Condition are suspended and queued. Tasks are woken up when notify is later called on the Condition. Edge triggering means that only tasks waiting at the time notify is called can be woken up. For level-triggered notifications, you must keep extra state to keep track of whether a notification has happened. The Channel and Threads.Event types do this, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.notify","location":"base/parallel.html#Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default), all waiting tasks are woken, otherwise only one is. If error is true, the passed value is raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Semaphore","location":"base/parallel.html#Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size acquires to be in use at any time. Each acquire must be matched with a release.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.acquire","location":"base/parallel.html#Base.acquire","category":"function","text":"acquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available, blocking until one can be acquired.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.release","location":"base/parallel.html#Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool, possibly allowing another task to acquire it and resume execution.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.AbstractLock","location":"base/parallel.html#Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that implement the synchronization primitives: lock, trylock, unlock, and islocked.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.lock","location":"base/parallel.html#Base.lock","category":"function","text":"lock(lock)\n\nAcquire the lock when it becomes available. If the lock is already locked by a different task/thread, wait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f returns. If the lock is already locked by a different task/thread, wait for it to become available.\n\nWhen this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.unlock","location":"base/parallel.html#Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an internal counter and return immediately.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.trylock","location":"base/parallel.html#Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available, and return true if successful. If the lock is already locked by a different task/thread, return false.\n\nEach successful trylock must be matched by an unlock.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.islocked","location":"base/parallel.html#Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread. This should not be used for synchronization (see instead trylock).\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.ReentrantLock","location":"base/parallel.html#Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreates a re-entrant lock for synchronizing Tasks. The same task can acquire the lock as many times as required. Each lock must be matched with an unlock.\n\nCalling 'lock' will also inhibit running of finalizers on that thread until the corresponding 'unlock'. Use of the standard lock pattern illustrated below should naturally be supported, but beware of inverting the try/lock order or missing the try block entirely (e.g. attempting to return with the lock still held):\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\n\n\n\n\n","page":"Tasks"},{"title":"Channels","location":"base/parallel.html#Channels","category":"section","text":"","page":"Tasks"},{"title":"Tasks","location":"base/parallel.html","category":"page","text":"Base.Channel\nBase.Channel(::Function)\nBase.put!(::Channel, ::Any)\nBase.take!(::Channel)\nBase.isready(::Channel)\nBase.fetch(::Channel)\nBase.close(::Channel)\nBase.bind(c::Channel, task::Task)","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstructs a Channel with an internal buffer that can hold a maximum of size objects of type T. put! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called. And vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel-Tuple{Function}","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false)\n\nCreate a new task from func, bind it to a new channel of type T and size size, and schedule the task, all in a single call.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via the keyword argument taskref.\n\nIf spawn = true, the Task created for func may be scheduled on another thread in parallel, equivalent to creating a task via Threads.@spawn.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3. In earlier versions of Julia, Channel used keyword arguments to set size and T, but those constructors are deprecated.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end\nChannel{Char}(1) (1 item available)\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.put!","location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different task.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.take!","location":"base/parallel.html#Base.take!-Tuple{Channel}","category":"method","text":"take!(c::Channel)\n\nRemove and return a value from a Channel. Blocks until data is available.\n\nFor unbuffered channels, blocks until a put! is performed by a different task.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.isready","location":"base/parallel.html#Base.isready-Tuple{Channel}","category":"method","text":"isready(c::Channel)\n\nDetermine whether a Channel has a value stored to it. Returns immediately, does not block.\n\nFor unbuffered channels returns true if there are tasks waiting on a put!.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Channel}","category":"method","text":"fetch(c::Channel)\n\nWait for and get the first available item from the channel. Does not remove the item. fetch is unsupported on an unbuffered (0-size) channel.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.close","location":"base/parallel.html#Base.close-Tuple{Channel}","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.bind","location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task. Channel chnl is automatically closed when the task terminates. Any uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination. Terminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will close the channel. When multiple channels are bound to the same task, termination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n","page":"Tasks"},{"title":"Frequently Asked Questions","location":"manual/faq.html#Frequently-Asked-Questions","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"General","location":"manual/faq.html#General","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Is Julia named after someone or something?","location":"manual/faq.html#Is-Julia-named-after-someone-or-something?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"No.","page":"Frequently Asked Questions"},{"title":"Why don't you compile Matlab/Python/R/… code to Julia?","location":"manual/faq.html#Why-don't-you-compile-Matlab/Python/R/…-code-to-Julia?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Since many people are familiar with the syntax of other dynamic languages, and lots of code has already been written in those languages, it is natural to wonder why we didn't just plug a Matlab or Python front-end into a Julia back-end (or “transpile” code to Julia) in order to get all the performance benefits of Julia without requiring programmers to learn a new language.  Simple, right?","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The basic issue is that there is nothing special about Julia's compiler: we use a commonplace compiler (LLVM) with no “secret sauce” that other language developers don't know about.  Indeed, Julia's compiler is in many ways much simpler than those of other dynamic languages (e.g. PyPy or LuaJIT).   Julia's performance advantage derives almost entirely from its front-end: its language semantics allow a well-written Julia program to give more opportunities to the compiler to generate efficient code and memory layouts.  If you tried to compile Matlab or Python code to Julia, our compiler would be limited by the semantics of Matlab or Python to producing code no better than that of existing compilers for those languages (and probably worse).  The key role of semantics is also why several existing Python compilers (like Numba and Pythran) only attempt to optimize a small subset of the language (e.g. operations on Numpy arrays and scalars), and for this subset they are already doing at least as well as we could for the same semantics.  The people working on those projects are incredibly smart and have accomplished amazing things, but retrofitting a compiler onto a language that was designed to be interpreted is a very difficult problem.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Julia's advantage is that good performance is not limited to a small subset of “built-in” types and operations, and one can write high-level type-generic code that works on arbitrary user-defined types while remaining fast and memory-efficient.  Types in languages like Python simply don't provide enough information to the compiler for similar capabilities, so as soon as you used those languages as a Julia front-end you would be stuck.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"For similar reasons, automated translation to Julia would also typically generate unreadable, slow, non-idiomatic code that would not be a good starting point for a native Julia port from another language.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"On the other hand, language interoperability is extremely useful: we want to exploit existing high-quality code in other languages from Julia (and vice versa)!  The best way to enable this is not a transpiler, but rather via easy inter-language calling facilities.  We have worked hard on this, from the built-in ccall intrinsic (to call C and Fortran libraries) to JuliaInterop packages that connect Julia to Python, Matlab, C++, and more.","page":"Frequently Asked Questions"},{"title":"Sessions and the REPL","location":"manual/faq.html#Sessions-and-the-REPL","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I delete an object in memory?","location":"manual/faq.html#How-do-I-delete-an-object-in-memory?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Julia does not have an analog of MATLAB's clear function; once a name is defined in a Julia session (technically, in module Main), it is always present.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"If memory usage is your concern, you can always replace objects with ones that consume less memory.  For example, if A is a gigabyte-sized array that you no longer need, you can free the memory with A = nothing.  The memory will be released the next time the garbage collector runs; you can force this to happen with GC.gc(). Moreover, an attempt to use A will likely result in an error, because most methods are not defined on type Nothing.","page":"Frequently Asked Questions"},{"title":"How can I modify the declaration of a type in my session?","location":"manual/faq.html#How-can-I-modify-the-declaration-of-a-type-in-my-session?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Perhaps you've defined a type and then realize you need to add a new field.  If you try this at the REPL, you get the error:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"ERROR: invalid redefinition of constant MyType","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Types in module Main cannot be redefined.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"While this can be inconvenient when you are developing new code, there's an excellent workaround.  Modules can be replaced by redefining them, and so if you wrap all your new code inside a module you can redefine types and constants.  You can't import the type names into Main and then expect to be able to redefine them there, but you can use the module name to resolve the scope.  In other words, while developing you might use a workflow something like this:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"include(\"mynewcode.jl\")              # this defines a module MyModule\nobj1 = MyModule.ObjConstructor(a, b)\nobj2 = MyModule.somefunction(obj1)\n# Got an error. Change something in \"mynewcode.jl\"\ninclude(\"mynewcode.jl\")              # reload the module\nobj1 = MyModule.ObjConstructor(a, b) # old objects are no longer valid, must reconstruct\nobj2 = MyModule.somefunction(obj1)   # this time it worked!\nobj3 = MyModule.someotherfunction(obj2, c)\n...","page":"Frequently Asked Questions"},{"title":"Scripting","location":"manual/faq.html#man-scripting","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I check if the current file is being run as the main script?","location":"manual/faq.html#How-do-I-check-if-the-current-file-is-being-run-as-the-main-script?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"When a file is run as the main script using julia file.jl one might want to activate extra functionality like command line argument handling. A way to determine that a file is run in this fashion is to check if abspath(PROGRAM_FILE) == @__FILE__ is true.","page":"Frequently Asked Questions"},{"title":"How do I catch CTRL-C in a script?","location":"manual/faq.html#catch-ctrl-c","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Running a Julia script using julia file.jl does not throw InterruptException when you try to terminate it with CTRL-C (SIGINT).  To run a certain code before terminating a Julia script, which may or may not be caused by CTRL-C, use atexit. Alternatively, you can use julia -e 'include(popfirst!(ARGS))' file.jl to execute a script while being able to catch InterruptException in the try block.","page":"Frequently Asked Questions"},{"title":"How do I pass options to julia using #!/usr/bin/env?","location":"manual/faq.html#How-do-I-pass-options-to-julia-using-#!/usr/bin/env?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Passing options to julia in so-called shebang by, e.g., #!/usr/bin/env julia --startup-file=no may not work in some platforms such as Linux.  This is because argument parsing in shebang is platform-dependent and not well-specified.  In a Unix-like environment, a reliable way to pass options to julia in an executable script would be to start the script as a bash script and use exec to replace the process to julia:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"#!/bin/bash\n#=\nexec julia --color=yes --startup-file=no \"${BASH_SOURCE[0]}\" \"$@\"\n=#\n\n@show ARGS  # put any Julia code here","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In the example above, the code between #= and =# is run as a bash script.  Julia ignores this part since it is a multi-line comment for Julia.  The Julia code after =# is ignored by bash since it stops parsing the file once it reaches to the exec statement.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"note: Note\nIn order to catch CTRL-C in the script you can use#!/bin/bash\n#=\nexec julia --color=yes --startup-file=no -e 'include(popfirst!(ARGS))' \\\n    \"${BASH_SOURCE[0]}\" \"$@\"\n=#\n\n@show ARGS  # put any Julia code hereinstead. Note that with this strategy PROGRAM_FILE will not be set.","page":"Frequently Asked Questions"},{"title":"Functions","location":"manual/faq.html#Functions","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"I passed an argument x to a function, modified it inside that function, but on the outside, the variable x is still unchanged. Why?","location":"manual/faq.html#I-passed-an-argument-x-to-a-function,-modified-it-inside-that-function,-but-on-the-outside,-the-variable-x-is-still-unchanged.-Why?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Suppose you call a function like this:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> x = 10\n10\n\njulia> function change_value!(y)\n           y = 17\n       end\nchange_value! (generic function with 1 method)\n\njulia> change_value!(x)\n17\n\njulia> x # x is unchanged!\n10","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In Julia, the binding of a variable x cannot be changed by passing x as an argument to a function. When calling change_value!(x) in the above example, y is a newly created variable, bound initially to the value of x, i.e. 10; then y is rebound to the constant 17, while the variable x of the outer scope is left untouched.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"However, if x is bound to an object of type Array (or any other mutable type). From within the function, you cannot \"unbind\" x from this Array, but you can change its content. For example:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> x = [1,2,3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> function change_array!(A)\n           A[1] = 5\n       end\nchange_array! (generic function with 1 method)\n\njulia> change_array!(x)\n5\n\njulia> x\n3-element Vector{Int64}:\n 5\n 2\n 3","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Here we created a function change_array!, that assigns 5 to the first element of the passed array (bound to x at the call site, and bound to A within the function). Notice that, after the function call, x is still bound to the same array, but the content of that array changed: the variables A and x were distinct bindings referring to the same mutable Array object.","page":"Frequently Asked Questions"},{"title":"Can I use using or import inside a function?","location":"manual/faq.html#Can-I-use-using-or-import-inside-a-function?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"No, you are not allowed to have a using or import statement inside a function.  If you want to import a module but only use its symbols inside a specific function or set of functions, you have two options:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Use import:\nimport Foo\nfunction bar(...)\n    # ... refer to Foo symbols via Foo.baz ...\nend\nThis loads the module Foo and defines a variable Foo that refers to the module, but does not import any of the other symbols from the module into the current namespace.  You refer to the Foo symbols by their qualified names Foo.bar etc.\nWrap your function in a module:\nmodule Bar\nexport bar\nusing Foo\nfunction bar(...)\n    # ... refer to Foo.baz as simply baz ....\nend\nend\nusing Bar\nThis imports all the symbols from Foo, but only inside the module Bar.","page":"Frequently Asked Questions"},{"title":"What does the ... operator do?","location":"manual/faq.html#What-does-the-...-operator-do?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"The two uses of the ... operator: slurping and splatting","location":"manual/faq.html#The-two-uses-of-the-...-operator:-slurping-and-splatting","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Many newcomers to Julia find the use of ... operator confusing. Part of what makes the ... operator confusing is that it means two different things depending on context.","page":"Frequently Asked Questions"},{"title":"... combines many arguments into one argument in function definitions","location":"manual/faq.html#...-combines-many-arguments-into-one-argument-in-function-definitions","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In the context of function definitions, the ... operator is used to combine many different arguments into a single argument. This use of ... for combining many different arguments into a single argument is called slurping:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function printargs(args...)\n           println(typeof(args))\n           for (i, arg) in enumerate(args)\n               println(\"Arg #$i = $arg\")\n           end\n       end\nprintargs (generic function with 1 method)\n\njulia> printargs(1, 2, 3)\nTuple{Int64, Int64, Int64}\nArg #1 = 1\nArg #2 = 2\nArg #3 = 3","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"If Julia were a language that made more liberal use of ASCII characters, the slurping operator might have been written as <-... instead of ....","page":"Frequently Asked Questions"},{"title":"... splits one argument into many different arguments in function calls","location":"manual/faq.html#...-splits-one-argument-into-many-different-arguments-in-function-calls","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In contrast to the use of the ... operator to denote slurping many different arguments into one argument when defining a function, the ... operator is also used to cause a single function argument to be split apart into many different arguments when used in the context of a function call. This use of ... is called splatting:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function threeargs(a, b, c)\n           println(\"a = $a::$(typeof(a))\")\n           println(\"b = $b::$(typeof(b))\")\n           println(\"c = $c::$(typeof(c))\")\n       end\nthreeargs (generic function with 1 method)\n\njulia> x = [1, 2, 3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> threeargs(x...)\na = 1::Int64\nb = 2::Int64\nc = 3::Int64","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"If Julia were a language that made more liberal use of ASCII characters, the splatting operator might have been written as ...-> instead of ....","page":"Frequently Asked Questions"},{"title":"What is the return value of an assignment?","location":"manual/faq.html#What-is-the-return-value-of-an-assignment?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The operator = always returns the right-hand side, therefore:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function threeint()\n           x::Int = 3.0\n           x # returns variable x\n       end\nthreeint (generic function with 1 method)\n\njulia> function threefloat()\n           x::Int = 3.0 # returns 3.0\n       end\nthreefloat (generic function with 1 method)\n\njulia> threeint()\n3\n\njulia> threefloat()\n3.0","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"and similarly:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function threetup()\n           x, y = [3, 3]\n           x, y # returns a tuple\n       end\nthreetup (generic function with 1 method)\n\njulia> function threearr()\n           x, y = [3, 3] # returns an array\n       end\nthreearr (generic function with 1 method)\n\njulia> threetup()\n(3, 3)\n\njulia> threearr()\n2-element Vector{Int64}:\n 3\n 3","page":"Frequently Asked Questions"},{"title":"Types, type declarations, and constructors","location":"manual/faq.html#Types,-type-declarations,-and-constructors","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What does \"type-stable\" mean?","location":"manual/faq.html#man-type-stability","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"It means that the type of the output is predictable from the types of the inputs.  In particular, it means that the type of the output cannot vary depending on the values of the inputs. The following code is not type-stable:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function unstable(flag::Bool)\n           if flag\n               return 1\n           else\n               return 1.0\n           end\n       end\nunstable (generic function with 1 method)","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"It returns either an Int or a Float64 depending on the value of its argument. Since Julia can't predict the return type of this function at compile-time, any computation that uses it must be able to cope with values of both types, which makes it hard to produce fast machine code.","page":"Frequently Asked Questions"},{"title":"Why does Julia give a DomainError for certain seemingly-sensible operations?","location":"manual/faq.html#faq-domain-errors","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Certain operations make mathematical sense but result in errors:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> sqrt(-2.0)\nERROR: DomainError with -2.0:\nsqrt will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This behavior is an inconvenient consequence of the requirement for type-stability.  In the case of sqrt, most users want sqrt(2.0) to give a real number, and would be unhappy if it produced the complex number 1.4142135623730951 + 0.0im.  One could write the sqrt function to switch to a complex-valued output only when passed a negative number (which is what sqrt does in some other languages), but then the result would not be type-stable and the sqrt function would have poor performance.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In these and other cases, you can get the result you want by choosing an input type that conveys your willingness to accept an output type in which the result can be represented:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> sqrt(-2.0+0im)\n0.0 + 1.4142135623730951im","page":"Frequently Asked Questions"},{"title":"How can I constrain or compute type parameters?","location":"manual/faq.html#How-can-I-constrain-or-compute-type-parameters?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The parameters of a parametric type can hold either types or bits values, and the type itself chooses how it makes use of these parameters. For example, Array{Float64, 2} is parameterized by the type Float64 to express its element type and the integer value 2 to express its number of dimensions.  When defining your own parametric type, you can use subtype constraints to declare that a certain parameter must be a subtype (<:) of some abstract type or a previous type parameter.  There is not, however, a dedicated syntax to declare that a parameter must be a value of a given type — that is, you cannot directly declare that a dimensionality-like parameter isa Int within the struct definition, for example.  Similarly, you cannot do computations (including simple things like addition or subtraction) on type parameters.  Instead, these sorts of constraints and relationships may be expressed through additional type parameters that are computed and enforced within the type's constructors.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"As an example, consider","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"struct ConstrainedType{T,N,N+1} # NOTE: INVALID SYNTAX\n    A::Array{T,N}\n    B::Array{T,N+1}\nend","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"where the user would like to enforce that the third type parameter is always the second plus one. This can be implemented with an explicit type parameter that is checked by an inner constructor method (where it can be combined with other checks):","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"struct ConstrainedType{T,N,M}\n    A::Array{T,N}\n    B::Array{T,M}\n    function ConstrainedType(A::Array{T,N}, B::Array{T,M}) where {T,N,M}\n        N + 1 == M || throw(ArgumentError(\"second argument should have one more axis\" ))\n        new{T,N,M}(A, B)\n    end\nend","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This check is usually costless, as the compiler can elide the check for valid concrete types. If the second argument is also computed, it may be advantageous to provide an outer constructor method that performs this calculation:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"ConstrainedType(A) = ConstrainedType(A, compute_B(A))","page":"Frequently Asked Questions"},{"title":"Why does Julia use native machine integer arithmetic?","location":"manual/faq.html#faq-integer-arithmetic","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Julia uses machine arithmetic for integer computations. This means that the range of Int values is bounded and wraps around at either end so that adding, subtracting and multiplying integers can overflow or underflow, leading to some results that can be unsettling at first:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> x = typemax(Int)\n9223372036854775807\n\njulia> y = x+1\n-9223372036854775808\n\njulia> z = -y\n-9223372036854775808\n\njulia> 2*z\n0","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Clearly, this is far from the way mathematical integers behave, and you might think it less than ideal for a high-level programming language to expose this to the user. For numerical work where efficiency and transparency are at a premium, however, the alternatives are worse.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"One alternative to consider would be to check each integer operation for overflow and promote results to bigger integer types such as Int128 or BigInt in the case of overflow. Unfortunately, this introduces major overhead on every integer operation (think incrementing a loop counter) – it requires emitting code to perform run-time overflow checks after arithmetic instructions and branches to handle potential overflows. Worse still, this would cause every computation involving integers to be type-unstable. As we mentioned above, type-stability is crucial for effective generation of efficient code. If you can't count on the results of integer operations being integers, it's impossible to generate fast, simple code the way C and Fortran compilers do.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"A variation on this approach, which avoids the appearance of type instability is to merge the Int and BigInt types into a single hybrid integer type, that internally changes representation when a result no longer fits into the size of a machine integer. While this superficially avoids type-instability at the level of Julia code, it just sweeps the problem under the rug by foisting all of the same difficulties onto the C code implementing this hybrid integer type. This approach can be made to work and can even be made quite fast in many cases, but has several drawbacks. One problem is that the in-memory representation of integers and arrays of integers no longer match the natural representation used by C, Fortran and other languages with native machine integers. Thus, to interoperate with those languages, we would ultimately need to introduce native integer types anyway. Any unbounded representation of integers cannot have a fixed number of bits, and thus cannot be stored inline in an array with fixed-size slots – large integer values will always require separate heap-allocated storage. And of course, no matter how clever a hybrid integer implementation one uses, there are always performance traps – situations where performance degrades unexpectedly. Complex representation, lack of interoperability with C and Fortran, the inability to represent integer arrays without additional heap storage, and unpredictable performance characteristics make even the cleverest hybrid integer implementations a poor choice for high-performance numerical work.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"An alternative to using hybrid integers or promoting to BigInts is to use saturating integer arithmetic, where adding to the largest integer value leaves it unchanged and likewise for subtracting from the smallest integer value. This is precisely what Matlab™ does:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":">> int64(9223372036854775807)\n\nans =\n\n  9223372036854775807\n\n>> int64(9223372036854775807) + 1\n\nans =\n\n  9223372036854775807\n\n>> int64(-9223372036854775808)\n\nans =\n\n -9223372036854775808\n\n>> int64(-9223372036854775808) - 1\n\nans =\n\n -9223372036854775808","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"At first blush, this seems reasonable enough since 9223372036854775807 is much closer to 9223372036854775808 than -9223372036854775808 is and integers are still represented with a fixed size in a natural way that is compatible with C and Fortran. Saturated integer arithmetic, however, is deeply problematic. The first and most obvious issue is that this is not the way machine integer arithmetic works, so implementing saturated operations requires emitting instructions after each machine integer operation to check for underflow or overflow and replace the result with typemin(Int) or typemax(Int) as appropriate. This alone expands each integer operation from a single, fast instruction into half a dozen instructions, probably including branches. Ouch. But it gets worse – saturating integer arithmetic isn't associative. Consider this Matlab computation:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":">> n = int64(2)^62\n4611686018427387904\n\n>> n + (n - 1)\n9223372036854775807\n\n>> (n + n) - 1\n9223372036854775806","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This makes it hard to write many basic integer algorithms since a lot of common techniques depend on the fact that machine addition with overflow is associative. Consider finding the midpoint between integer values lo and hi in Julia using the expression (lo + hi) >>> 1:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> n = 2^62\n4611686018427387904\n\njulia> (n + 2n) >>> 1\n6917529027641081856","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"See? No problem. That's the correct midpoint between 2^62 and 2^63, despite the fact that n + 2n is -4611686018427387904. Now try it in Matlab:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":">> (n + 2*n)/2\n\nans =\n\n  4611686018427387904","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Oops. Adding a >>> operator to Matlab wouldn't help, because saturation that occurs when adding n and 2n has already destroyed the information necessary to compute the correct midpoint.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Not only is lack of associativity unfortunate for programmers who cannot rely it for techniques like this, but it also defeats almost anything compilers might want to do to optimize integer arithmetic. For example, since Julia integers use normal machine integer arithmetic, LLVM is free to aggressively optimize simple little functions like f(k) = 5k-1. The machine code for this function is just this:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> code_native(f, Tuple{Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 1\n  leaq  -1(%rdi,%rdi,4), %rax\n  popq  %rbp\n  retq\n  nopl  (%rax,%rax)","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The actual body of the function is a single leaq instruction, which computes the integer multiply and add at once. This is even more beneficial when f gets inlined into another function:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function g(k, n)\n           for i = 1:n\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 1 methods)\n\njulia> code_native(g, Tuple{Int,Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 2\n  testq %rsi, %rsi\n  jle L26\n  nopl  (%rax)\nSource line: 3\nL16:\n  leaq  -1(%rdi,%rdi,4), %rdi\nSource line: 2\n  decq  %rsi\n  jne L16\nSource line: 5\nL26:\n  movq  %rdi, %rax\n  popq  %rbp\n  retq\n  nop","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Since the call to f gets inlined, the loop body ends up being just a single leaq instruction. Next, consider what happens if we make the number of loop iterations fixed:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> function g(k)\n           for i = 1:10\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 2 methods)\n\njulia> code_native(g,(Int,))\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 3\n  imulq $9765625, %rdi, %rax    # imm = 0x9502F9\n  addq  $-2441406, %rax         # imm = 0xFFDABF42\nSource line: 5\n  popq  %rbp\n  retq\n  nopw  %cs:(%rax,%rax)","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Because the compiler knows that integer addition and multiplication are associative and that multiplication distributes over addition – neither of which is true of saturating arithmetic – it can optimize the entire loop down to just a multiply and an add. Saturated arithmetic completely defeats this kind of optimization since associativity and distributivity can fail at each loop iteration, causing different outcomes depending on which iteration the failure occurs in. The compiler can unroll the loop, but it cannot algebraically reduce multiple operations into fewer equivalent operations.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The most reasonable alternative to having integer arithmetic silently overflow is to do checked arithmetic everywhere, raising errors when adds, subtracts, and multiplies overflow, producing values that are not value-correct. In this blog post, Dan Luu analyzes this and finds that rather than the trivial cost that this approach should in theory have, it ends up having a substantial cost due to compilers (LLVM and GCC) not gracefully optimizing around the added overflow checks. If this improves in the future, we could consider defaulting to checked integer arithmetic in Julia, but for now, we have to live with the possibility of overflow.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In the meantime, overflow-safe integer operations can be achieved through the use of external libraries such as SaferIntegers.jl. Note that, as stated previously, the use of these libraries significantly increases the execution time of code using the checked integer types. However, for limited usage, this is far less of an issue than if it were used for all integer operations. You can follow the status of the discussion here.","page":"Frequently Asked Questions"},{"title":"What are the possible causes of an UndefVarError during remote execution?","location":"manual/faq.html#What-are-the-possible-causes-of-an-UndefVarError-during-remote-execution?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"As the error states, an immediate cause of an UndefVarError on a remote node is that a binding by that name does not exist. Let us explore some of the possible causes.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> module Foo\n           foo() = remotecall_fetch(x->x, 2, \"Hello\")\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: Foo not defined\nStacktrace:\n[...]","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The closure x->x carries a reference to Foo, and since Foo is unavailable on node 2, an UndefVarError is thrown.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Globals under modules other than Main are not serialized by value to the remote node. Only a reference is sent. Functions which create global bindings (except under Main) may cause an UndefVarError to be thrown later.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> @everywhere module Foo\n           function foo()\n               global gvar = \"Hello\"\n               remotecall_fetch(()->gvar, 2)\n           end\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: gvar not defined\nStacktrace:\n[...]","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In the above example, @everywhere module Foo defined Foo on all nodes. However the call to Foo.foo() created a new global binding gvar on the local node, but this was not found on node 2 resulting in an UndefVarError error.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Note that this does not apply to globals created under module Main. Globals under module Main are serialized and new bindings created under Main on the remote node.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> gvar_self = \"Node1\"\n\"Node1\"\n\njulia> remotecall_fetch(()->gvar_self, 2)\n\"Node1\"\n\njulia> remotecall_fetch(varinfo, 2)\nname          size summary\n––––––––– –––––––– –––––––\nBase               Module\nCore               Module\nMain               Module\ngvar_self 13 bytes String","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This does not apply to function or struct declarations. However, anonymous functions bound to global variables are serialized as can be seen below.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> bar() = 1\nbar (generic function with 1 method)\n\njulia> remotecall_fetch(bar, 2)\nERROR: On worker 2:\nUndefVarError: #bar not defined\n[...]\n\njulia> anon_bar  = ()->1\n(::#21) (generic function with 1 method)\n\njulia> remotecall_fetch(anon_bar, 2)\n1","page":"Frequently Asked Questions"},{"title":"Why does Julia use * for string concatenation? Why not + or something else?","location":"manual/faq.html#Why-does-Julia-use-*-for-string-concatenation?-Why-not-or-something-else?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The main argument against + is that string concatenation is not commutative, while + is generally used as a commutative operator. While the Julia community recognizes that other languages use different operators and * may be unfamiliar for some users, it communicates certain algebraic properties.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Note that you can also use string(...) to concatenate strings (and other values converted to strings); similarly, repeat can be used instead of ^ to repeat strings. The interpolation syntax is also useful for constructing strings.","page":"Frequently Asked Questions"},{"title":"Packages and Modules","location":"manual/faq.html#Packages-and-Modules","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What is the difference between \"using\" and \"import\"?","location":"manual/faq.html#What-is-the-difference-between-\"using\"-and-\"import\"?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"There is only one difference, and on the surface (syntax-wise) it may seem very minor. The difference between using and import is that with using you need to say function Foo.bar(.. to extend module Foo's function bar with a new method, but with import Foo.bar, you only need to say function bar(... and it automatically extends module Foo's function bar.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The reason this is important enough to have been given separate syntax is that you don't want to accidentally extend a function that you didn't know existed, because that could easily cause a bug. This is most likely to happen with a method that takes a common type like a string or integer, because both you and the other module could define a method to handle such a common type. If you use import, then you'll replace the other module's implementation of bar(s::AbstractString) with your new implementation, which could easily do something completely different (and break all/many future usages of the other functions in module Foo that depend on calling bar).","page":"Frequently Asked Questions"},{"title":"Nothingness and missing values","location":"manual/faq.html#Nothingness-and-missing-values","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How does \"null\", \"nothingness\" or \"missingness\" work in Julia?","location":"manual/faq.html#faq-nothing","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Unlike many languages (for example, C and Java), Julia objects cannot be \"null\" by default. When a reference (variable, object field, or array element) is uninitialized, accessing it will immediately throw an error. This situation can be detected using the isdefined or isassigned functions.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Some functions are used only for their side effects, and do not need to return a value. In these cases, the convention is to return the value nothing, which is just a singleton object of type Nothing. This is an ordinary type with no fields; there is nothing special about it except for this convention, and that the REPL does not print anything for it. Some language constructs that would not otherwise have a value also yield nothing, for example if false; end.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"For situations where a value x of type T exists only sometimes, the Union{T, Nothing} type can be used for function arguments, object fields and array element types as the equivalent of Nullable, Option or Maybe in other languages. If the value itself can be nothing (notably, when T is Any), the Union{Some{T}, Nothing} type is more appropriate since x == nothing then indicates the absence of a value, and x == Some(nothing) indicates the presence of a value equal to nothing. The something function allows unwrapping Some objects and using a default value instead of nothing arguments. Note that the compiler is able to generate efficient code when working with Union{T, Nothing} arguments or fields.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"To represent missing data in the statistical sense (NA in R or NULL in SQL), use the missing object. See the Missing Values section for more details.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In some languages, the empty tuple (()) is considered the canonical form of nothingness. However, in julia it is best thought of as just a regular tuple that happens to contain zero values.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The empty (or \"bottom\") type, written as Union{} (an empty union type), is a type with no values and no subtypes (except itself). You will generally not need to use this type.","page":"Frequently Asked Questions"},{"title":"Memory","location":"manual/faq.html#Memory","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why does x += y allocate memory when x and y are arrays?","location":"manual/faq.html#Why-does-x-y-allocate-memory-when-x-and-y-are-arrays?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In Julia, x += y gets replaced during parsing by x = x + y. For arrays, this has the consequence that, rather than storing the result in the same location in memory as x, it allocates a new array to store the result.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"While this behavior might surprise some, the choice is deliberate. The main reason is the presence of immutable objects within Julia, which cannot change their value once created.  Indeed, a number is an immutable object; the statements x = 5; x += 1 do not modify the meaning of 5, they modify the value bound to x. For an immutable, the only way to change the value is to reassign it.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"To amplify a bit further, consider the following function:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"function power_by_squaring(x, n::Int)\n    ispow2(n) || error(\"This implementation only works for powers of 2\")\n    while n >= 2\n        x *= x\n        n >>= 1\n    end\n    x\nend","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"After a call like x = 5; y = power_by_squaring(x, 4), you would get the expected result: x == 5 && y == 625.  However, now suppose that *=, when used with matrices, instead mutated the left hand side.  There would be two problems:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"For general square matrices, A = A*B cannot be implemented without temporary storage: A[1,1] gets computed and stored on the left hand side before you're done using it on the right hand side.\nSuppose you were willing to allocate a temporary for the computation (which would eliminate most of the point of making *= work in-place); if you took advantage of the mutability of x, then this function would behave differently for mutable vs. immutable inputs. In particular, for immutable x, after the call you'd have (in general) y != x, but for mutable x you'd have y == x.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Because supporting generic programming is deemed more important than potential performance optimizations that can be achieved by other means (e.g., using explicit loops), operators like += and *= work by rebinding new values.","page":"Frequently Asked Questions"},{"title":"Asynchronous IO and concurrent synchronous writes","location":"manual/faq.html#faq-async-io","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why do concurrent writes to the same stream result in inter-mixed output?","location":"manual/faq.html#Why-do-concurrent-writes-to-the-same-stream-result-in-inter-mixed-output?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"While the streaming I/O API is synchronous, the underlying implementation is fully asynchronous.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Consider the printed output from the following:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> @sync for i in 1:3\n           @async write(stdout, string(i), \" Foo \", \" Bar \")\n       end\n123 Foo  Foo  Foo  Bar  Bar  Bar","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This is happening because, while the write call is synchronous, the writing of each argument yields to other tasks while waiting for that part of the I/O to complete.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"print and println \"lock\" the stream during a call. Consequently changing write to println in the above example results in:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> @sync for i in 1:3\n           @async println(stdout, string(i), \" Foo \", \" Bar \")\n       end\n1 Foo  Bar\n2 Foo  Bar\n3 Foo  Bar","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"You can lock your writes with a ReentrantLock like this:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> l = ReentrantLock();\n\njulia> @sync for i in 1:3\n           @async begin\n               lock(l)\n               try\n                   write(stdout, string(i), \" Foo \", \" Bar \")\n               finally\n                   unlock(l)\n               end\n           end\n       end\n1 Foo  Bar 2 Foo  Bar 3 Foo  Bar","page":"Frequently Asked Questions"},{"title":"Arrays","location":"manual/faq.html#Arrays","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What are the differences between zero-dimensional arrays and scalars?","location":"manual/faq.html#What-are-the-differences-between-zero-dimensional-arrays-and-scalars?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Zero-dimensional arrays are arrays of the form Array{T,0}. They behave similar to scalars, but there are important differences. They deserve a special mention because they are a special case which makes logical sense given the generic definition of arrays, but might be a bit unintuitive at first. The following line defines a zero-dimensional array:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"julia> A = zeros()\n0-dimensional Array{Float64,0}:\n0.0","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"In this example, A is a mutable container that contains one element, which can be set by A[] = 1.0 and retrieved with A[]. All zero-dimensional arrays have the same size (size(A) == ()), and length (length(A) == 1). In particular, zero-dimensional arrays are not empty. If you find this unintuitive, here are some ideas that might help to understand Julia's definition.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Zero-dimensional arrays are the \"point\" to vector's \"line\" and matrix's \"plane\". Just as a line has no area (but still represents a set of things), a point has no length or any dimensions at all (but still represents a thing).\nWe define prod(()) to be 1, and the total number of elements in an array is the product of the size. The size of a zero-dimensional array is (), and therefore its length is 1.\nZero-dimensional arrays don't natively have any dimensions into which you index – they’re just A[]. We can apply the same \"trailing one\" rule for them as for all other array dimensionalities, so you can indeed index them as A[1], A[1,1], etc; see Omitted and extra indices.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"It is also important to understand the differences to ordinary scalars. Scalars are not mutable containers (even though they are iterable and define things like length, getindex, e.g. 1[] == 1). In particular, if x = 0.0 is defined as a scalar, it is an error to attempt to change its value via x[] = 1.0. A scalar x can be converted into a zero-dimensional array containing it via fill(x), and conversely, a zero-dimensional array a can be converted to the contained scalar via a[]. Another difference is that a scalar can participate in linear algebra operations such as 2 * rand(2,2), but the analogous operation with a zero-dimensional array fill(2) * rand(2,2) is an error.","page":"Frequently Asked Questions"},{"title":"Why are my Julia benchmarks for linear algebra operations different from other languages?","location":"manual/faq.html#Why-are-my-Julia-benchmarks-for-linear-algebra-operations-different-from-other-languages?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"You may find that simple benchmarks of linear algebra building blocks like","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"using BenchmarkTools\nA = randn(1000, 1000)\nB = randn(1000, 1000)\n@btime $A \\ $B\n@btime $A * $B","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"can be different when compared to other languages like Matlab or R.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Since operations like this are very thin wrappers over the relevant BLAS functions, the reason for the discrepancy is very likely to be","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"the BLAS library each language is using,\nthe number of concurrent threads.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Julia compiles and uses its own copy of OpenBLAS, with threads currently capped at 8 (or the number of your cores).","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Modifying OpenBLAS settings or compiling Julia with a different BLAS library, eg Intel MKL, may provide performance improvements. You can use MKL.jl, a package that makes Julia's linear algebra use Intel MKL BLAS and LAPACK instead of OpenBLAS, or search the discussion forum for suggestions on how to set this up manually. Note that Intel MKL cannot be bundled with Julia, as it is not open source.","page":"Frequently Asked Questions"},{"title":"Computing cluster","location":"manual/faq.html#Computing-cluster","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I manage precompilation caches in distributed file systems?","location":"manual/faq.html#How-do-I-manage-precompilation-caches-in-distributed-file-systems?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"When using julia in high-performance computing (HPC) facilities, invoking n julia processes simultaneously creates at most n temporary copies of precompilation cache files. If this is an issue (slow and/or small distributed file system), you may:","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Use julia with --compiled-modules=no flag to turn off precompilation.\nConfigure a private writable depot using pushfirst!(DEPOT_PATH, private_path) where private_path is a path unique to this julia process.  This can also be done by setting environment variable JULIA_DEPOT_PATH to $private_path:$HOME/.julia.\nCreate a symlink from ~/.julia/compiled to a directory in a scratch space.","page":"Frequently Asked Questions"},{"title":"Julia Releases","location":"manual/faq.html#Julia-Releases","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Do I want to use the Stable, LTS, or nightly version of Julia?","location":"manual/faq.html#Do-I-want-to-use-the-Stable,-LTS,-or-nightly-version-of-Julia?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"The Stable version of Julia is the latest released version of Julia, this is the version most people will want to run. It has the latest features, including improved performance. The Stable version of Julia is versioned according to SemVer as v1.x.y. A new minor release of Julia corresponding to a new Stable version is made approximately every 4-5 months after a few weeks of testing as a release candidate. Unlike the LTS version the a Stable version will not normally recieve bugfixes after another Stable version of Julia has been released. However, upgrading to the next Stable release will always be possible as each release of Julia v1.x will continue to run code written for earlier versions.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"You may prefer the LTS (Long Term Support) version of Julia if you are looking for a very stable code base. The current LTS version of Julia is versioned according to SemVer as v1.0.x; this branch will continue to recieve bugfixes until a new LTS branch is chosen, at which point the v1.0.x series will no longer recieved regular bug fixes and all but the most conservative users will be advised to upgrade to the new LTS version series. As a package developer, you may prefer to develop for the LTS version, to maximize the number of users who can use your package. As per SemVer, code written for v1.0 will continue to work for all future LTS and Stable versions. In general, even if targetting the LTS, one can develop and run code in the latest Stable version, to take advantage of the improved performance; so long as one avoids using new features (such as added library functions or new methods).","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"You may prefer the nightly version of Julia if you want to take advantage of the latest updates to the language, and don't mind if the version available today occasionally doesn't actually work. As the name implies, releases to the nightly version are made roughly every night (depending on build infrastructure stability). In general nightly released are fairly safe to use—your code will not catch on fire. However, they may be occasional regressions and or issues that will not be found until more thorough pre-release testing. You may wish to test against the nightly version to ensure that such regressions that affect your use case are caught before a release is made.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Finally, you may also consider building Julia from source for yourself. This option is mainly for those individuals who are comfortable at the command line, or interested in learning. If this describes you, you may also be interested in reading our guidelines for contributing.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Links to each of these download types can be found on the download page at https://julialang.org/downloads/. Note that not all versions of Julia are available for all platforms.","page":"Frequently Asked Questions"},{"title":"How can I transfer the list of installed packages after updating my version of Julia?","location":"manual/faq.html#How-can-I-transfer-the-list-of-installed-packages-after-updating-my-version-of-Julia?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"Each minor version of julia has its own default environment. As a result, upon installing a new minor version of Julia, the packages you added using the previous minor version will not be available by default. The environment for a given julia version is defined by the files Project.toml and Manifest.toml in a folder matching the version number in .julia/environments/, for instance, .julia/environments/v1.3.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"If you install a new minor version of Julia, say 1.4, and want to use in its default environment the same packages as in a previous version (e.g. 1.3), you can copy the contents of the file Project.toml from the 1.3 folder to 1.4. Then, in a session of the new Julia version, enter the \"package management mode\" by typing the key ], and run the command instantiate.","page":"Frequently Asked Questions"},{"title":"Frequently Asked Questions","location":"manual/faq.html","category":"page","text":"This operation will resolve a set of feasible packages from the copied file that are compatible with the target Julia version, and will install or update them if suitable. If you want to reproduce not only the set of packages, but also the versions you were using in the previous Julia version, you should also copy the Manifest.toml file before running the Pkg command instantiate. However, note that packages may define compatibility constraints that may be affected by changing the version of Julia, so the exact set of versions you had in 1.3 may not work for 1.4.","page":"Frequently Asked Questions"},{"title":"Metaprogramming","location":"manual/metaprogramming.html#Metaprogramming","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp, Julia represents its own code as a data structure of the language itself. Since code is represented by objects that can be created and manipulated from within the language, it is possible for a program to transform and generate its own code. This allows sophisticated code generation without extra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees. In contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation and substitution before any actual parsing or interpretation occurs. Because all data types and code in Julia are represented by Julia data structures, powerful reflection capabilities are available to explore the internals of a program and its types just like any other data.","page":"Metaprogramming"},{"title":"Program representation","location":"manual/metaprogramming.html#Program-representation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every Julia program starts life as a string:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> prog = \"1 + 1\"\n\"1 + 1\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens next?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The next step is to parse each string into an object called an expression, represented by the Julia type Expr:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = Meta.parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects contain two parts:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"a Symbol identifying the kind of expression. A symbol is an interned string identifier (more discussion below).","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.head\n:call","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the expression arguments, which may be symbols, other expressions, or literal values:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1.args\n3-element Vector{Any}:\n  :+\n 1\n 1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions may also be constructed directly in prefix notation:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The two expressions constructed above – by parsing and by direct construction – are equivalent:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 == ex2\ntrue","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The key point here is that Julia code is internally represented as a data structure that is accessible from the language itself.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The dump function provides indented and annotated display of Expr objects:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expr objects may also be nested:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex3 = Meta.parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to view expressions is with Meta.show_sexpr, which displays the S-expression form of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating the display on a nested Expr:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)","page":"Metaprogramming"},{"title":"Symbols","location":"manual/metaprogramming.html#Symbols","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The : character has two syntactic purposes in Julia. The first form creates a Symbol, an interned string used as one building-block of expressions:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> s = :foo\n:foo\n\njulia> typeof(s)\nSymbol","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The Symbol constructor takes any number of arguments and creates a new symbol by concatenating their string representations together:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :foo == Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_sym","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that to use : syntax, the symbol's name must be a valid identifier. Otherwise the Symbol(str) constructor must be used.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the context of an expression, symbols are used to indicate access to variables; when an expression is evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Sometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> :(:)\n:(:)\n\njulia> :(::)\n:(::)","page":"Metaprogramming"},{"title":"Expressions and evaluation","location":"manual/metaprogramming.html#Expressions-and-evaluation","category":"section","text":"","page":"Metaprogramming"},{"title":"Quoting","location":"manual/metaprogramming.html#Quoting","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The second syntactic purpose of the : character is to create expression objects without using the explicit Expr constructor. This is referred to as quoting. The : character, followed by paired parentheses around a single statement of Julia code, produces an Expr object based on the enclosed code. Here is example of the short form used to quote an arithmetic expression:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(to view the structure of this expression, try ex.head and ex.args, or use dump as above or Meta.@dump)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that equivalent expressions may be constructed using Meta.parse or the direct Expr form:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia>      :(a + b*c + 1)       ==\n       Meta.parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrue","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Expressions provided by the parser generally only have symbols, other expressions, and literal values as their args, whereas expressions constructed by Julia code can have arbitrary run-time values without literal forms as args. In this specific example, + and a are symbols, *(b,c) is a subexpression, and 1 is a literal 64-bit signed integer.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is a second syntactic form of quoting for multiple expressions: blocks of code enclosed in quote ... end.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote\n    #= none:2 =#\n    x = 1\n    #= none:3 =#\n    y = 2\n    #= none:4 =#\n    x + y\nend\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Interpolation","location":"manual/metaprogramming.html#man-expression-interpolation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Direct construction of Expr objects with value arguments is powerful, but Expr constructors can be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows interpolation of literals or expressions into quoted expressions. Interpolation is indicated by a prefix $.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the value of variable a is interpolated:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Interpolating into an unquoted expression is not supported and will cause a compile-time error:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> $a + b\nERROR: syntax: \"$\" expression outside quote","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The use of $ for expression interpolation is intentionally reminiscent of string interpolation and command interpolation. Expression interpolation allows convenient, readable programmatic construction of complex Julia expressions.","page":"Metaprogramming"},{"title":"Splatting interpolation","location":"manual/metaprogramming.html#Splatting-interpolation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the $ interpolation syntax allows inserting only a single expression into an enclosing expression. Occasionally, you have an array of expressions and need them all to become arguments of the surrounding expression. This can be done with the syntax $(xs...). For example, the following code generates a function call where the number of arguments is determined programmatically:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> args = [:x, :y, :z];\n\njulia> :(f(1, $(args...)))\n:(f(1, x, y, z))","page":"Metaprogramming"},{"title":"Nested quote","location":"manual/metaprogramming.html#Nested-quote","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Naturally, it is possible for quote expressions to contain other quote expressions. Understanding how interpolation works in these cases can be a bit tricky. Consider this example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = :(1 + 2);\n\njulia> e = quote quote $x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :x))\nend))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that the result contains $x, which means that x has not been evaluated yet. In other words, the $ expression \"belongs to\" the inner quote expression, and so its argument is only evaluated when the inner quote expression is:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    1 + 2\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, the outer quote expression is able to interpolate values inside the $ in the inner quote. This is done with multiple $s:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> e = quote quote $$x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :(1 + 2)))\nend))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that (1 + 2) now appears in the result instead of the symbol x. Evaluating this expression yields an interpolated 3:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(e)\nquote\n    #= none:1 =#\n    3\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The intuition behind this behavior is that x is evaluated once for each $: one $ works similarly to eval(:x), giving x's value, while two $s do the equivalent of eval(eval(:x)).","page":"Metaprogramming"},{"title":"QuoteNode","location":"manual/metaprogramming.html#man-quote-node","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The usual representation of a quote form in an AST is an Expr with head :quote:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":(1+2)\"))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Expr\n      head: Symbol call\n      args: Array{Any}((3,))\n        1: Symbol +\n        2: Int64 1\n        3: Int64 2","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As we have seen, such expressions support interpolation with $. However, in some situations it is necessary to quote code without performing interpolation. This kind of quoting does not yet have syntax, but is represented internally as an object of type QuoteNode:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> eval(Meta.quot(Expr(:$, :(1+2))))\n3\n\njulia> eval(QuoteNode(Expr(:$, :(1+2))))\n:($(Expr(:$, :(1 + 2))))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The parser yields QuoteNodes for simple quoted items like symbols:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> dump(Meta.parse(\":x\"))\nQuoteNode\n  value: Symbol x","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"QuoteNode can also be used for certain advanced metaprogramming tasks.","page":"Metaprogramming"},{"title":"Evaluating expressions","location":"manual/metaprogramming.html#Evaluating-expressions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Given an expression object, one can cause Julia to evaluate (execute) it at global scope using eval:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex1 = :(1 + 2)\n:(1 + 2)\n\njulia> eval(ex1)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: b not defined\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Every module has its own eval function that evaluates expressions in its global scope. Expressions passed to eval are not limited to returning values – they can also have side-effects that alter the state of the enclosing module's environment:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: x not defined\n\njulia> eval(ex)\n1\n\njulia> x\n1","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, the evaluation of an expression object causes a value to be assigned to the global variable x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since expressions are just Expr objects which can be constructed programmatically and then evaluated, it is possible to dynamically generate arbitrary code which can then be run using eval. Here is a simple example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of a is used to construct the expression ex which applies the + function to the value 1 and the variable b. Note the important distinction between the way a and b are used:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The value of the variable a at expression construction time is used as an immediate value in the expression. Thus, the value of a when the expression is evaluated no longer matters: the value in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol :b is used in the expression construction, so the value of the variable b at that time is irrelevant – :b is just a symbol and the variable b need not even be defined. At expression evaluation time, however, the value of the symbol :b is resolved by looking up the value of the variable b.","page":"Metaprogramming"},{"title":"Functions on Expressions","location":"manual/metaprogramming.html#Functions-on-Expressions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate Julia code within Julia itself. We have already seen one example of a function returning Expr objects: the parse function, which takes a string of Julia code and returns the corresponding Expr. A function can also take one or more Expr objects as arguments, and return another Expr. Here is a simple, motivating example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"As another example, here is a function that doubles any numeric argument, but leaves expressions alone:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42","page":"Metaprogramming"},{"title":"Macros","location":"manual/metaprogramming.html#man-macros","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros provide a method to include generated code in the final body of a program. A macro maps a tuple of arguments to a returned expression, and the resulting expression is compiled directly rather than requiring a runtime eval call. Macro arguments may include expressions, literal values, and symbols.","page":"Metaprogramming"},{"title":"Basics","location":"manual/metaprogramming.html#Basics","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is an extraordinarily simple macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique name declared in a macro NAME ... end block. In this example, the compiler will replace all instances of @sayhello with:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":":( println(\"Hello, world!\") )","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When @sayhello is entered in the REPL, the expression executes immediately, thus we only see the evaluation result:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello()\nHello, world!","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now, consider a slightly more complex macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro takes one argument: name. When @sayhello is encountered, the quoted expression is expanded to interpolate the value of the argument into the final expression:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @sayhello(\"human\")\nHello, human","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can view the quoted return expression using the function macroexpand (important note: this is an extremely useful tool for debugging macros):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> ex = macroexpand(Main, :(@sayhello(\"human\")) )\n:(Main.println(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can see that the \"human\" literal has been interpolated into the expression.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @sayhello \"human\"\n:(println(\"Hello, \", \"human\"))","page":"Metaprogramming"},{"title":"Hold up: why macros?","location":"manual/metaprogramming.html#Hold-up:-why-macros?","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand is also such a function. So, why do macros exist?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are necessary because they execute when code is parsed, therefore, macros allow the programmer to generate and include fragments of customized code before the full program is run. To illustrate the difference, consider the following example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand(Main, :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: :((1, 2, 3))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The first call to println is executed when macroexpand is called. The resulting expression contains only the second println:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(ex)\nExpr\n\njulia> ex\n:(println(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)","page":"Metaprogramming"},{"title":"Macro invocation","location":"manual/metaprogramming.html#Macro-invocation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros are invoked with the following general syntax:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name expr1 expr2 ...\n@name(expr1, expr2, ...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note the distinguishing @ before the macro name and the lack of commas between the argument expressions in the first form, and the lack of whitespace after @name in the second form. The two styles should not be mixed. For example, the following syntax is different from the examples above; it passes the tuple (expr1, expr2, ...) as one argument to the macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name (expr1, expr2, ...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An alternative way to invoke a macro over an array literal (or comprehension) is to juxtapose both without using parentheses. In this case, the array will be the only expression fed to the macro. The following syntax is equivalent (and different from @name [a b] * v):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@name[a b] * v\n@name([a b]) * v","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is important to emphasize that macros receive their arguments as expressions, literals, or symbols. One way to explore macro arguments is to call the show function within the macro body:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In addition to the given argument list, every macro is passed extra arguments named __source__ and __module__.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __source__ provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. This allows macros to include better error diagnostic information, and is commonly used by logging, string-parser macros, and docs, for example, as well as to implement the @__LINE__, @__FILE__, and @__DIR__ macros.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The location information can be accessed by referencing __source__.line and __source__.file:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro __LOCATION__(); return QuoteNode(__source__); end\n@__LOCATION__ (macro with 1 method)\n\njulia> dump(\n            @__LOCATION__(\n       ))\nLineNumberNode\n  line: Int64 2\n  file: Symbol none","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The argument __module__ provides information (in the form of a Module object) about the expansion context of the macro invocation. This allows macros to look up contextual information, such as existing bindings, or to insert the value as an extra argument to a runtime function call doing self-reflection in the current module.","page":"Metaprogramming"},{"title":"Building an advanced macro","location":"manual/metaprogramming.html#Building-an-advanced-macro","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here is a simplified definition of Julia's @assert macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can be used like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In place of the written syntax, the macro call is expanded at parse time to its returned result. This is equivalent to writing:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot, while the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire expression, thus constructed, is placed into the syntax tree where the @assert macro call occurs. Then at execution time, if the test expression evaluates to true, then nothing is returned, whereas if the test is false, an error is raised indicating the asserted expression that was false. Notice that it would not be possible to write this as a function, since only the value of the condition is available and it would be impossible to display the expression that computed it in the error message.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The actual definition of @assert in Julia Base is more complicated. It allows the user to optionally specify their own error message, instead of just printing the failed expression. Just like in functions with a variable number of arguments (Varargs Functions), this is specified with an ellipses following the last argument:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Now @assert has two modes of operation, depending upon the number of arguments it receives! If there's only one argument, the tuple of expressions captured by msgs will be empty and it will behave the same as the simpler definition above. But now if the user specifies a second argument, it is printed in the message body instead of the failing expression. You can inspect the result of a macro expansion with the aptly named @macroexpand macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @macroexpand @assert a == b\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a == b\"))\n    end)\n\njulia> @macroexpand @assert a==b \"a should equal b!\"\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a should equal b!\"))\n    end)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"There is yet another case that the actual @assert macro handles: what if, in addition to printing \"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation in the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work as expected with the above macro. Can you see why? Recall from string interpolation that an interpolated string is rewritten to a call to string. Compare:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So now instead of getting a plain string in msg_body, the macro is receiving a full expression that will need to be evaluated in order to display as expected. This can be spliced directly into the returned expression as an argument to the string call; see error.jl for the complete implementation.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @assert macro makes great use of splicing into quoted expressions to simplify the manipulation of expressions inside the macro body.","page":"Metaprogramming"},{"title":"Hygiene","location":"manual/metaprogramming.html#Hygiene","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An issue that arises in more complex macros is that of hygiene. In short, macros must ensure that the variables they introduce in their returned expressions do not accidentally clash with existing variables in the surrounding code they expand into. Conversely, the expressions that are passed into a macro as arguments are often expected to evaluate in the context of the surrounding code, interacting with and modifying the existing variables. Another concern arises from the fact that a macro may be called in a different module from where it was defined. In this case we need to ensure that all global variables are resolved to the correct module. Julia already has a major advantage over languages with textual macro expansion (like C) in that it only needs to consider the returned expression. All the other variables (such as msg in @assert above) follow the normal scoping block behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate these issues, let us consider writing a @time macro that takes an expression as its argument, records the time, evaluates the expression, records the time again, prints the difference between the before and after times, and then has the value of the expression as its final value. The macro might look like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    return quote\n        local t0 = time_ns()\n        local val = $ex\n        local t1 = time_ns()\n        println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n        val\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here, we want t0, t1, and val to be private temporary variables, and we want time_ns to refer to the time_ns function in Julia Base, not to any time_ns variable the user might have (the same applies to println). Imagine the problems that could occur if the user expression ex also contained assignments to a variable called t0, or defined its own time_ns variable. We might get errors, or mysteriously incorrect behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's macro expander solves these problems in the following way. First, variables within a macro result are classified as either local or global. A variable is considered local if it is assigned to (and not declared global), declared local, or used as a function argument name. Otherwise, it is considered global. Local variables are then renamed to be unique (using the gensym function, which generates new symbols), and global variables are resolved within the macro definition environment. Therefore both of the above concerns are handled; the macro's locals will not conflict with any user variables, and time_ns and println will refer to the Julia Base definitions.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"One problem remains however. Consider the following use of this macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"module MyModule\nimport Base.@time\n\ntime_ns() = ... # compute something\n\n@time time_ns()\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Here the user expression ex is a call to time_ns, but not the same time_ns function that the macro uses. It clearly refers to MyModule.time_ns. Therefore we must arrange for the code in ex to be resolved in the macro call environment. This is done by \"escaping\" the expression with esc:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An expression wrapped in this manner is left alone by the macro expander and simply pasted into the output verbatim. Therefore it will be resolved in the macro call environment.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce or manipulate user variables. For example, the following macro sets x to zero in the call environment:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This kind of manipulation of variables should be used judiciously, but is occasionally quite handy.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Getting the hygiene rules correct can be a formidable challenge. Before using a macro, you might want to consider whether a function closure would be sufficient. Another useful strategy is to defer as much work as possible to runtime. For example, many macros simply wrap their arguments in a QuoteNode or other similar Expr. Some examples of this include @task body which simply returns schedule(Task(() -> $body)), and @eval expr, which simply returns eval(QuoteNode(expr)).","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To demonstrate, we might rewrite the @time example above as:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro time(expr)\n    return :(timeit(() -> $(esc(expr))))\nend\nfunction timeit(f)\n    t0 = time_ns()\n    val = f()\n    t1 = time_ns()\n    println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n    return val\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, we don't do this for a good reason: wrapping the expr in a new scope block (the anonymous function) also slightly changes the meaning of the expression (the scope of any variables in it), while we want @time to be usable with minimum impact on the wrapped code.","page":"Metaprogramming"},{"title":"Macros and dispatch","location":"manual/metaprogramming.html#Macros-and-dispatch","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Macros, just like Julia functions, are generic. This means they can also have multiple method definitions, thanks to multiple dispatch:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m end\n@m (macro with 0 methods)\n\njulia> macro m(args...)\n           println(\"$(length(args)) arguments\")\n       end\n@m (macro with 1 method)\n\njulia> macro m(x,y)\n           println(\"Two arguments\")\n       end\n@m (macro with 2 methods)\n\njulia> @m \"asd\"\n1 arguments\n\njulia> @m 1 2\nTwo arguments","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However one should keep in mind, that macro dispatch is based on the types of AST that are handed to the macro, not the types that the AST evaluates to at runtime:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> macro m(::Int)\n           println(\"An Integer\")\n       end\n@m (macro with 3 methods)\n\njulia> @m 2\nAn Integer\n\njulia> x = 2\n2\n\njulia> @m x\n1 arguments","page":"Metaprogramming"},{"title":"Code Generation","location":"manual/metaprogramming.html#Code-Generation","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When a significant amount of repetitive boilerplate code is required, it is common to generate it programmatically to avoid redundancy. In most languages, this requires an extra build step, and a separate program to generate the repetitive code. In Julia, expression interpolation and eval allow such code generation to take place in the normal course of program execution. For example, consider the following custom type","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"struct MyNumber\n    x::Float64\nend\n# output\n","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for which we want to add a number of methods to. We can do this programmatically in the following loop:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(quote\n        Base.$op(a::MyNumber) = MyNumber($op(a.x))\n    end)\nend\n# output\n","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and we can now use those functions with our custom type:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = MyNumber(π)\nMyNumber(3.141592653589793)\n\njulia> sin(x)\nMyNumber(1.2246467991473532e-16)\n\njulia> cos(x)\nMyNumber(-1.0)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this manner, Julia acts as its own preprocessor, and allows code generation from inside the language. The above code could be written slightly more tersely using the : prefix quoting form:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    eval(:(Base.$op(a::MyNumber) = MyNumber($op(a.x))))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This sort of in-language code generation, however, using the eval(quote(...)) pattern, is common enough that Julia comes with a macro to abbreviate this pattern:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for op = (:sin, :cos, :tan, :log, :exp)\n    @eval Base.$op(a::MyNumber) = MyNumber($op(a.x))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The @eval macro rewrites this call to be precisely equivalent to the above longer versions. For longer blocks of generated code, the expression argument given to @eval can be a block:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"@eval begin\n    # multiple lines\nend","page":"Metaprogramming"},{"title":"Non-Standard String Literals","location":"manual/metaprogramming.html#Non-Standard-String-Literals","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Recall from Strings that string literals prefixed by an identifier are called non-standard string literals, and can have different semantics than un-prefixed string literals. For example:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"r\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128].","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Perhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead, they are custom behaviors provided by a general mechanism that anyone can use: prefixed string literals are parsed as calls to specially-named macros. For example, the regular expression macro is just the following:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro r_str(p)\n    Regex(p)\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"That's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should be passed to the @r_str macro and the result of that expansion should be placed in the syntax tree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent to placing the following object directly into the syntax tree:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Regex(\"^\\\\s*(?:#|\\$)\")","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Not only is the string literal form shorter and far more convenient, but it is also more efficient: since the regular expression is compiled and the Regex object is actually created when the code is compiled, the compilation occurs only once, rather than every time the code is executed. Consider if the regular expression occurs in a loop:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"for line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when this code is parsed, the expression is only compiled once instead of each time the loop is executed. In order to accomplish this without macros, one would have to write this loop like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"re = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Moreover, if the compiler could not determine that the regex object was constant over all loops, certain optimizations might not be possible, making this version still less efficient than the more convenient literal form above. Of course, there are still situations where the non-literal form is more convenient: if one needs to interpolate a variable into the regular expression, one must take this more verbose approach; in cases where the regular expression pattern itself is dynamic, potentially changing upon each loop iteration, a new regular expression object must be constructed on each iteration. In the vast majority of use cases, however, regular expressions are not constructed based on run-time data. In this majority of cases, the ability to write regular expressions as compile-time values is invaluable.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Like non-standard string literals, non-standard command literals exist using a prefixed variant of the command literal syntax. The command literal custom`literal` is parsed as @custom_cmd \"literal\". Julia itself does not contain any non-standard command literals, but packages can make use of this syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix, non-standard command literals behave exactly like non-standard string literals.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In the event that two modules provide non-standard string or command literals with the same name, it is possible to qualify the string or command literal with a module name. For instance, if both Foo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\" or Bar.x\"literal\" to disambiguate between the two.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's non-standard literals implemented using it, but also the command literal syntax (`echo \"Hello, $person\"`) is implemented with the following innocuous-looking macro:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Of course, a large amount of complexity is hidden in the functions used in this macro definition, but they are just functions, written entirely in Julia. You can read their source and see precisely what they do – and all they do is construct expression objects to be inserted into your program's syntax tree.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Another way to define a macro would be like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"macro foo_str(str, flag)\n    # do stuff\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"This macro can then be called with the following syntax:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"foo\"str\"flag","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The type of flag in the above mentioned syntax would be a String with contents of whatever trails after the string literal.","page":"Metaprogramming"},{"title":"Generated functions","location":"manual/metaprogramming.html#Generated-functions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"A very special macro is @generated, which allows you to define so-called generated functions. These have the capability to generate specialized code depending on the types of their arguments with more flexibility and/or less code than what can be achieved with multiple dispatch. While macros work with expressions at parse time and cannot access the types of their inputs, a generated function gets expanded at a time when the types of the arguments are known, but the function is not yet compiled.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Instead of performing some calculation or action, a generated function declaration returns a quoted expression which then forms the body for the method corresponding to the types of the arguments. When a generated function is called, the expression it returns is compiled and then run. To make this efficient, the result is usually cached. And to make this inferable, only a limited subset of the language is usable. Thus, generated functions provide a flexible way to move work from run time to compile time, at the expense of greater restrictions on allowed constructs.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"When defining generated functions, there are five main differences to ordinary functions:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"You annotate the function declaration with the @generated macro. This adds some information to the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments – not their values.\nInstead of calculating something or performing some action, you return a quoted expression which, when evaluated, does what you want.\nGenerated functions are only permitted to call functions that were defined before the definition of the generated function. (Failure to follow this may result in getting MethodErrors referring to functions from a future world-age.)\nGenerated functions must not mutate or observe any non-constant global state (including, for example, IO, locks, non-local dictionaries, or using hasmethod). This means they can only read global constants, and cannot have any side effects. In other words, they must be completely pure. Due to an implementation limitation, this also means that they currently cannot define a closure or generator.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It's easiest to illustrate this with an example. We can declare a generated function foo as","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the body returns a quoted expression, namely :(x * x), rather than just the value of x * x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"From the caller's perspective, this is identical to a regular function; in fact, you don't have to know whether you're calling a regular or generated function. Let's see how foo behaves:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, we see that in the body of the generated function, x is the type of the passed argument, and the value returned by the generated function, is the result of evaluating the quoted expression we returned from the definition, now with the value of x.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What happens if we evaluate foo again with a type that we have already used?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> foo(4)\n16","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that there is no printout of Int64. We can see that the body of the generated function was only executed once here, for the specific set of argument types, and the result was cached. After that, for this example, the expression returned from the generated function on the first invocation was re-used as the method body. However, the actual caching behavior is an implementation-defined performance optimization, so it is invalid to depend too closely on this behavior.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The number of times a generated function is generated might be only once, but it might also be more often, or appear to not happen at all. As a consequence, you should never write a generated function with side effects - when, and how often, the side effects occur is undefined. (This is true for macros too - and just like for macros, the use of eval in a generated function is a sign that you're doing something the wrong way.) However, unlike macros, the runtime system cannot correctly handle a call to eval, so it is disallowed.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"It is also important to see how @generated functions interact with method redefinition. Following the principle that a correct @generated function must not observe any mutable state or cause any mutation of global state, we see the following behavior. Observe that the generated function cannot call any method that was not defined prior to the definition of the generated function itself.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Initially f(x) has one definition","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x) = \"original definition\";","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Define other operations that use f(x):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We now add some new definitions for f(x):","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"and compare how these results differ:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Each method of a generated function has its own view of defined functions:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The example generated function foo above did not do anything a normal function foo(x) = x * x could not do (except printing the type on the first invocation, and incurring higher overhead). However, the power of a generated function lies in its ability to compute different quoted expressions depending on the types passed to it:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"(although of course this contrived example would be more easily implemented using multiple dispatch...)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Abusing this will corrupt the runtime system and cause undefined behavior:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Since the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code is undefined.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Don't copy these examples!","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"These examples are hopefully helpful to illustrate how generated functions work, both in the definition end and at the call site; however, don't copy them, for the following reasons:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"the foo function has side-effects (the call to Core.println), and it is undefined exactly when, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x and bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathological","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Note that the set of operations that should not be attempted in a generated function is unbounded, and the runtime system can currently only detect a subset of the invalid operations. There are many other operations that will simply corrupt the runtime system without notification, usually in subtle ways not obviously connected to the bad definition. Because the function generator is run during inference, it must respect all of the limitations of that code.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Some operations that should not be attempted include:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Caching of native pointers.\nInteracting with the contents or methods of Core.Compiler in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting to observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic to call malloc, even though most implementations require locks internally) but don't attempt to hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition is relaxed for incrementally-loaded precompiled modules to allow calling any function in the module.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Alright, now that we have a better understanding of how generated functions work, let's use them to build some more advanced (and valid) functionality...","page":"Metaprogramming"},{"title":"An advanced example","location":"manual/metaprogramming.html#An-advanced-example","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Julia's base library has an internal sub2ind function to calculate a linear index into an n-dimensional array, based on a set of n multilinear indices - in other words, to calculate the index i that can be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation is the following:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end\nsub2ind_loop (generic function with 1 method)\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"The same thing can be done using recursion:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Both these implementations, although different, do essentially the same thing: a runtime loop over the dimensions of the array, collecting the offset in each dimension into the final index.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"However, all the information we need for the loop is embedded in the type information of the arguments. Thus, we can utilize generated functions to move the iteration to compile-time; in compiler parlance, we use generated functions to manually unroll the loop. The body becomes almost identical, but instead of calculating the linear index, we build up an expression that calculates the index:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"What code will this generate?","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"An easy way to find out is to extract the body into another (regular) function:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end\nsub2ind_gen (generic function with 1 method)\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end\nsub2ind_gen_impl (generic function with 1 method)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"We can now execute sub2ind_gen_impl and examine the expression it returns:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"julia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"So, the method body that will be used here doesn't include a loop at all - just indexing into the two tuples, multiplication and addition/subtraction. All the looping is performed compile-time, and we avoid looping during execution entirely. Thus, we only loop once per type, in this case once per N (except in edge cases where the function is generated more than once - see disclaimer above).","page":"Metaprogramming"},{"title":"Optionally-generated functions","location":"manual/metaprogramming.html#Optionally-generated-functions","category":"section","text":"","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Generated functions can achieve high efficiency at run time, but come with a compile time cost: a new function body must be generated for every combination of concrete argument types. Typically, Julia is able to compile \"generic\" versions of functions that will work for any arguments, but with generated functions this is impossible. This means that programs making heavy use of generated functions might be impossible to statically compile.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"To solve this problem, the language provides syntax for writing normal, non-generated alternative implementations of generated functions. Applied to the sub2ind example above, it would look like this:","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n    if N != length(I)\n        throw(ArgumentError(\"Number of dimensions must match number of indices.\"))\n    end\n    if @generated\n        ex = :(I[$N] - 1)\n        for i = (N - 1):-1:1\n            ex = :(I[$i] - 1 + dims[$i] * $ex)\n        end\n        return :($ex + 1)\n    else\n        ind = I[N] - 1\n        for i = (N - 1):-1:1\n            ind = I[i] - 1 + dims[i]*ind\n        end\n        return ind + 1\n    end\nend","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Internally, this code creates two implementations of the function: a generated one where the first block in if @generated is used, and a normal one where the else block is used. Inside the then part of the if @generated block, code has the same semantics as other generated functions: argument names refer to types, and the code should return an expression. Multiple if @generated blocks may occur, in which case the generated implementation uses all of the then blocks and the alternate implementation uses all of the else blocks.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"Notice that we added an error check to the top of the function. This code will be common to both versions, and is run-time code in both versions (it will be quoted and returned as an expression from the generated version). That means that the values and types of local variables are not available at code generation time –- the code-generation code can only see the types of arguments.","page":"Metaprogramming"},{"title":"Metaprogramming","location":"manual/metaprogramming.html","category":"page","text":"In this style of definition, the code generation feature is essentially an optional optimization. The compiler will use it if convenient, but otherwise may choose to use the normal implementation instead. This style is preferred, since it allows the compiler to make more decisions and compile programs in more ways, and since normal code is more readable than code-generating code. However, which implementation is used depends on compiler implementation details, so it is essential for the two implementations to behave identically.","page":"Metaprogramming"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html#Multi-processing-and-Distributed-Computing","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"An implementation of distributed memory parallel computing is provided by module Distributed as part of the standard library shipped with Julia.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Most modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Julia's implementation of message passing is different from other environments such as MPI[1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Distributed programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Remote references come in two flavors: Future and RemoteChannel.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait on the returned Future, and you can obtain the full value of the result using fetch.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"On the other hand, RemoteChannel s are rewritable. For example, multiple processes can co-ordinate their processing by referencing the same remote Channel.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Each process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1. As a result, adding 2 or more processes is required to gain benefits from parallel processing methods like pmap. Adding a single process is beneficial if you just wish to do other things in the main process while a long computation is running on the worker.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Let's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU threads (logical cores) on the machine. Note that the -p argument implicitly loads module Distributed.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"$ ./julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.18526  1.50912\n 1.16296  1.60607","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The first argument to remotecall is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall is considered a low-level interface providing finer control. The second argument to remotecall is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Occasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.18526337335308085","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Remember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"To make things easier, the symbol :any can be passed to @spawnat, which picks where to do the operation for you:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> r = @spawnat :any rand(2,2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat :any 1 .+ fetch(r)\nFuture(3, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Array{Float64,2}:\n 1.38854  1.9098\n 1.20939  1.57158","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Note that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch might be required to move r to the process doing the addition. In this case, @spawnat is smart enough to perform the computation on the process that owns r, so the fetch will be a no-op (no work is done).","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"(It is worth noting that @spawnat is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"An important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"@async is similar to @spawnat, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indices. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. As for v0.7 and beyond, the feeder tasks are able to share state via nextidx because they all run on the same process. Even if Tasks are scheduled cooperatively, locking may still be required in some contexts, as in asynchronous I/O. This means context switches only occur at well-defined points: in this case, when remotecall_fetch is called. This is the current state of implementation and it may change for future Julia versions, as it is intended to make it possible to run up to N Tasks on M Process, aka M:N Threading. Then a lock acquiring\\releasing model for nextidx will be needed, as it is not safe to let multiple processes read-write a resource at the same time.","page":"Multi-processing and Distributed Computing"},{"title":"Code Availability and Loading Packages","location":"manual/distributed-computing.html#code-availability","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Your code must be available on any process that runs it. For example, type the following into the Julia prompt:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Array{Float64,2}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawnat :any rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))\nStacktrace:\n[...]","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Process 1 knew about the function rand2, but process 2 did not.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Most commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"module DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In order to refer to MyType across all processes, DummyModule.jl needs to be loaded on every process.  Calling include(\"DummyModule.jl\") loads it only on a single process.  To load it on every process, use the @everywhere macro (starting Julia with julia -p 2):","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @everywhere include(\"DummyModule.jl\")\nloaded\n      From worker 3:    loaded\n      From worker 2:    loaded","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As usual, this does not bring DummyModule into scope on any of the process, which requires using or import.  Moreover, when DummyModule is brought into scope on one process, it is not on any other:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> using .DummyModule\n\njulia> MyType(7)\nMyType(7)\n\njulia> fetch(@spawnat 2 MyType(7))\nERROR: On worker 2:\nUndefVarError: MyType not defined\n⋮\n\njulia> fetch(@spawnat 2 DummyModule.MyType(7))\nMyType(7)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"However, it's still possible, for instance, to send a MyType to a process which has loaded DummyModule even if it's not in scope:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> put!(RemoteChannel(2), MyType(7))\nRemoteChannel{Channel{Any}}(2, 1, 13)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A file can also be preloaded on multiple processes at startup with the -L flag, and a driver script can be used to drive the computation:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia -p <n> -L file1.jl -L file2.jl driver.jl","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Finally, if DummyModule.jl is not a standalone file but a package, then using DummyModule will load DummyModule.jl on all processes, but only bring it into scope on the process where using was called.","page":"Multi-processing and Distributed Computing"},{"title":"Starting and managing worker processes","location":"manual/distributed-computing.html#Starting-and-managing-worker-processes","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The base Julia installation has in-built support for two types of clusters:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machine-file option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers should use to connect to this worker.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Functions addprocs, rmprocs, workers, and others are available as a programmatic means of adding, removing and querying the processes in a cluster.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> using Distributed\n\njulia> addprocs(2)\n2-element Array{Int64,1}:\n 2\n 3","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Module Distributed must be explicitly loaded on the master process before invoking addprocs. It is automatically made available on the worker processes.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Note that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as global variables, new method definitions, and loaded modules) with any of the other running processes. You may use addprocs(exeflags=\"--project\") to initialize a worker with a particular environment, and then @everywhere using <modulename> or @everywhere include(\"file.jl\").","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Other types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section.","page":"Multi-processing and Distributed Computing"},{"title":"Data Movement","location":"manual/distributed-computing.html#Data-Movement","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Sending messages and moving data constitute most of the overhead in a distributed program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various distributed programming constructs.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"fetch can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawnat (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Method 1:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> A = rand(1000,1000);\n\njulia> Bref = @spawnat :any A^2;\n\n[...]\n\njulia> fetch(Bref);","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Method 2:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> Bref = @spawnat :any rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The difference seems trivial, but in fact is quite significant due to the behavior of @spawnat. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawnat and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawnat statement just for this step.","page":"Multi-processing and Distributed Computing"},{"title":"Global variables","location":"manual/distributed-computing.html#Global-variables","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Expressions executed remotely via @spawnat, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A = rand(10,10)\nremotecall_fetch(()->sum(A), 2)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In this case sum MUST be defined in the remote process. Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->sum(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"New global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Thus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"For example:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @fetchfrom 2 InteractiveUtils.varinfo()\nname           size summary\n––––––––– ––––––––– ––––––––––––––––––––––\nA         800 bytes 10×10 Array{Float64,2}\nBase                Module\nCore                Module\nMain                Module","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2.","page":"Multi-processing and Distributed Computing"},{"title":"Parallel Map and Loops","location":"manual/distributed-computing.html#Parallel-Map-and-Loops","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawnat to flip coins on two processes. First, write the following function in count_heads.jl:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"function count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @everywhere include_string(Main, $(read(\"count_heads.jl\", String)), \"count_heads.jl\")\n\njulia> a = @spawnat :any count_heads(100000000)\nFuture(2, 1, 6, nothing)\n\njulia> b = @spawnat :any count_heads(100000000)\nFuture(3, 1, 7, nothing)\n\njulia> fetch(a)+fetch(b)\n100001564","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"This example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Notice that our use of this pattern with count_heads can be generalized. We used two explicit @spawnat statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, running in distributed memory, which can be written in Julia using @distributed like this:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"nheads = @distributed (+) for i = 1:200000000\n    Int(rand(Bool))\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"This construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Note that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"For example, the following code will not work as intended:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"a = zeros(100000)\n@distributed for i = 1:100000\n    a[i] = i\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"This code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"using SharedArrays\n\na = SharedArray{Float64}(10)\n@distributed for i = 1:10\n    a[i] = i\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Using \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"a = randn(1000)\n@distributed (+) for i = 1:100000\n    f(a[rand(1:end)])\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Here each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @distributed for.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap function. For example, we could compute the singular values of several large random matrices in parallel as follows:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svdvals, M);","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Julia's pmap is designed for the case where each function call does a large amount of work. In contrast, @distributed for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap and @distributed for for the parallel computation. In case of @distributed for, the final reduction is done on the calling process.","page":"Multi-processing and Distributed Computing"},{"title":"Remote References and AbstractChannels","location":"manual/distributed-computing.html#Remote-References-and-AbstractChannels","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Remote references always refer to an implementation of an AbstractChannel.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A concrete implementation of an AbstractChannel (like Channel), is required to implement put!, take!, fetch, isready and wait. The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"RemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f is a function executed on pid and it must return an AbstractChannel.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"For example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Methods put!, take!, fetch, isready and wait on a RemoteChannel are proxied onto the backing store on the remote process.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"RemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is provided in dictchannel.jl in the Examples repository, which uses a dictionary as its remote store.","page":"Multi-processing and Distributed Computing"},{"title":"Channels and RemoteChannels","location":"manual/distributed-computing.html#Channels-and-RemoteChannels","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The channels example from above can be modified for interprocess communication, as shown below.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"We start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> @async make_jobs(n); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, where = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds on worker $where\")\n           global n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741","page":"Multi-processing and Distributed Computing"},{"title":"Remote References and Distributed Garbage Collection","location":"manual/distributed-computing.html#Remote-References-and-Distributed-Garbage-Collection","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Objects referred to by remote references can be freed only when all held references in the cluster are deleted.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified. This is implemented in an internal cluster aware serializer. Remote references are only valid in the context of a running cluster. Serializing and deserializing references to and from regular IO objects is not supported.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Since Futures are write-once and cached locally, the act of fetching a Future also updates reference tracking information on the node owning the value.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The node which owns the value frees it once all references to it are cleared.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"With Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"It is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize results in an immediate message sent to the remote node to go ahead and remove its reference to the value.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Once finalized, a reference becomes invalid and cannot be used in any further calls.","page":"Multi-processing and Distributed Computing"},{"title":"Local invocations","location":"manual/distributed-computing.html#Local-invocations","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Data is necessarily copied over to the remote node for execution. This is the case for both remotecalls and when data is stored to aRemoteChannel / Future on a different node. As expected, this results in a copy of the serialized objects on the remote node. However, when the destination node is the local node, i.e. the calling process id is the same as the remote node id, it is executed as a local call. It is usually (not always) executed in a different task - but there is no serialization/deserialization of data. Consequently, the call refers to the same object instances as passed - no copies are created. This behavior is highlighted below:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> using Distributed;\n\njulia> rc = RemoteChannel(()->Channel(3));   # RemoteChannel created on local node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i                          # Reusing `v`\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\nArray{Int64,1}[[3], [3], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 1\n\njulia> addprocs(1);\n\njulia> rc = RemoteChannel(()->Channel(3), workers()[1]);   # RemoteChannel created on remote node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\nArray{Int64,1}[[1], [2], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 3","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As can be seen, put! on a locally owned RemoteChannel with the same object v modifed between calls results in the same single object instance stored. As opposed to copies of v being created when the node owning rc is a different node.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"It is to be noted that this is generally not an issue. It is something to be factored in only if the object is both being stored locally and modifed post the call. In such cases it may be appropriate to store a deepcopy of the object.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"This is also true for remotecalls on the local node as seen in the following example:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> using Distributed; addprocs(1);\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), myid(), v);     # Executed on local node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[1], v2=[1], true\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), workers()[1], v); # Executed on remote node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[0], v2=[1], false","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As can be seen once again, a remote call onto the local node behaves just like a direct invocation. The call modifies local objects passed as arguments. In the remote invocation, it operates on a copy of the arguments.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"To repeat, in general this is not an issue. If the local node is also being used as a compute node, and the arguments used post the call, this behavior needs to be factored in and if required deep copies of arguments must be passed to the call invoked on the local node. Calls on remote nodes will always operate on copies of arguments.","page":"Multi-processing and Distributed Computing"},{"title":"Shared Arrays","location":"manual/distributed-computing.html#man-shared-arrays","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Shared Arrays use system shared memory to map the same array across many processes. While there are some similarities to a DArray, the behavior of a SharedArray is quite different. In a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk; in contrast, in a SharedArray each \"participating\" process has access to the entire array.  A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Shared Array support is available via module SharedArrays which must be explicitly loaded on all participating workers.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"SharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata. For other AbstractArray types, sdata just returns the object itself, so it's safe to use sdata on any Array-type object.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The constructor for a shared array is of the form:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"SharedArray{T,N}(dims::NTuple; init=false, pids=Int[])","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"which creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host). Note that only elements that are isbits are supported in a SharedArray.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"If an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Here's a brief example:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> using Distributed\n\njulia> addprocs(3)\n3-element Array{Int64,1}:\n 2\n 3\n 4\n\njulia> @everywhere using SharedArrays\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[localindices(S)] = repeat([myid()], length(localindices(S))))\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedArray{Int64,2}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"SharedArrays.localindices provides disjoint one-dimensional ranges of indices, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = repeat([myid()], length( indexpids(S):length(procs(S)):length(S))))\n3×4 SharedArray{Int64,2}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Since all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"@sync begin\n    for p in procs(S)\n        @async begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"would result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As a more extended and complex example, consider running the following \"kernel\" in parallel:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"q[i,j,t+1] = q[i,j,t] + u[i,j,t]","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indices assigned to this worker:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in range(0, stop=size(q,2), length=nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       end","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Next, define the kernel:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       end","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"We also define a convenience wrapper for a SharedArray implementation","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Now let's compare three different versions, one that runs in a single process:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"one that uses @distributed:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @distributed for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"and one that delegates in chunks:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   @async remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"If we create SharedArrays and time these functions, we get the following results (with julia -p 4):","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Run the functions once to JIT-compile and @time them on the second run:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece.","page":"Multi-processing and Distributed Computing"},{"title":"Shared Arrays and Distributed Garbage Collection","location":"manual/distributed-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner.","page":"Multi-processing and Distributed Computing"},{"title":"ClusterManagers","location":"manual/distributed-computing.html#ClusterManagers","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible for","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"launching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transport","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A Julia cluster has the following characteristics:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Connections between workers (using the in-built TCP/IP transport) is established in the following manner:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"addprocs is called on the master process with a ClusterManager object.\naddprocs calls the appropriate launch method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to stdout.\nThe cluster manager captures the stdout of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"While the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Julia provides two in-built cluster managers:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"LocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnames","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"LocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Thus, a minimal cluster manager would need to:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"be a subtype of the abstract ClusterManager\nimplement launch, a method responsible for launching new workers\nimplement manage, which is called at various events during a worker's lifetime (for example, sending an interrupt signal)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"addprocs(manager::FooManager) requires FooManager to implement:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"function launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"struct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The launch method takes the following arguments:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"manager::ClusterManager: the cluster manager that addprocs is called with\nparams::Dict: all the keyword arguments passed to addprocs\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launched","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The launch method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch function MUST exit as soon as all the requested workers have been launched.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Newly launched workers are connected to each other and the master process in an all-to-all manner. Specifying the command line argument --worker[=<cookie>] results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"All workers in a cluster share the same cookie as the master. When the cookie is unspecified, i.e, with the --worker option, the worker tries to read it from its standard input.  LocalManager and SSHManager both pass the cookie to newly launched workers via their  standard inputs.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"By default a worker will listen on a free port at the address returned by a call to getipaddr(). A specific address to listen on may be specified by optional argument --bind-to bind_addr[:port]. This is useful for multi-homed hosts.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"As an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"For every worker launched, the launch method must add a WorkerConfig object (with appropriate fields initialized) to launched","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"mutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Union{IO, Nothing}\n    host::Union{AbstractString, Nothing}\n    port::Union{Integer, Nothing}\n\n    # Used when launching additional workers at a host\n    count::Union{Int, Symbol, Nothing}\n    exename::Union{AbstractString, Cmd, Nothing}\n    exeflags::Union{Cmd, Nothing}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Any\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Union{Bool, Nothing}\n    bind_addr::Union{AbstractString, Nothing}\n    sshflags::Union{Cmd, Nothing}\n    max_parallel::Union{Integer, Nothing}\n\n    # Used by Local/SSH managers\n    connect_at::Any\n\n    [...]\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Most of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"If io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of CPU threads (logical cores) on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"with :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes.","page":"Multi-processing and Distributed Computing"},{"title":"Cluster Managers with Custom Transports","location":"manual/distributed-computing.html#Cluster-Managers-with-Custom-Transports","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Each Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Replacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"connect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"connect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"A BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The folder clustermanager/0mq in the Examples repository contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"When using custom transports:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Julia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) must be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch is called. The value of this field is passed in all connect callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"kill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The Examples folder clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup.","page":"Multi-processing and Distributed Computing"},{"title":"Network Requirements for LocalManager and SSHManager","location":"manual/distributed-computing.html#Network-Requirements-for-LocalManager-and-SSHManager","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on an ephemeral port number assigned by the OS.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-i <keyfile>`.\nIn an all-to-all topology (the default), all workers connect to each other via plain TCP sockets. The security policy on the cluster nodes must thus ensure free connectivity between workers for the ephemeral port range (varies by OS).\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager.\nIf you specify multiplex=true as an option to addprocs, SSH multiplexing is used to create a tunnel between the master and workers. If you have configured SSH multiplexing on your own and the connection has already been established, SSH multiplexing is used regardless of multiplex option. If multiplexing is enabled, forwarding is set by using the existing connection (-O forward option in ssh). This is beneficial if your servers require password authentication; you can avoid authentication in Julia by logging in to the server ahead of addprocs. The control socket will be located at ~/.ssh/julia-%r@%h:%p during the session unless the existing multiplexing connection is used. Note that bandwidth may be limited if you create multiple processes on a node and enable multiplexing, because in that case processes share a single multiplexing TCP connection.","page":"Multi-processing and Distributed Computing"},{"title":"Cluster Cookie","location":"manual/distributed-computing.html#man-cluster-cookie","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"cluster_cookie() returns the cookie, while cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie may be passed to the workers at startup via argument --worker=<cookie>. If argument --worker is specified without the cookie, the worker tries to read the cookie from its standard input (stdin). The stdin is closed immediately after the cookie is retrieved.\nClusterManagers can retrieve the cookie on the master by calling cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Note that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument.","page":"Multi-processing and Distributed Computing"},{"title":"Specifying Network Topology (Experimental)","location":"manual/distributed-computing.html#Specifying-Network-Topology-(Experimental)","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":":all_to_all, the default: all workers are connected to each other.\n:master_worker: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Keyword argument lazy=true|false only affects topology option :all_to_all. If true, the cluster starts off with the master connected to all workers. Specific worker-worker connections are established at the first remote invocation between two workers. This helps in reducing initial resources allocated for intra-cluster communication. Connections are setup depending on the runtime requirements of a parallel program. Default value for lazy is true.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Currently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases.","page":"Multi-processing and Distributed Computing"},{"title":"Noteworthy external packages","location":"manual/distributed-computing.html#Noteworthy-external-packages","category":"section","text":"","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Outside of Julia parallelism there are plenty of external packages that should be mentioned. For example MPI.jl is a Julia wrapper for the MPI protocol, or DistributedArrays.jl, as presented in Shared Arrays. A mention must be made of Julia's GPU programming ecosystem, which includes:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Low-level (C kernel) based operations OpenCL.jl and CUDAdrv.jl which are respectively an OpenCL interface and a CUDA wrapper.\nLow-level (Julia Kernel) interfaces like CUDAnative.jl which is a Julia native CUDA implementation.\nHigh-level vendor-specific abstractions like CuArrays.jl and CLArrays.jl\nHigh-level libraries like ArrayFire.jl and GPUArrays.jl","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In the following example we will use both DistributedArrays.jl and CuArrays.jl to distribute an array across multiple processes by first casting it through distribute() and CuArray().","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Remember when importing DistributedArrays.jl to import it across all processes using @everywhere","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"$ ./julia -p 4\n\njulia> addprocs()\n\njulia> @everywhere using DistributedArrays\n\njulia> using CuArrays\n\njulia> B = ones(10_000) ./ 2;\n\njulia> A = ones(10_000) .* π;\n\njulia> C = 2 .* A ./ B;\n\njulia> all(C .≈ 4*π)\ntrue\n\njulia> typeof(C)\nArray{Float64,1}\n\njulia> dB = distribute(B);\n\njulia> dA = distribute(A);\n\njulia> dC = 2 .* dA ./ dB;\n\njulia> all(dC .≈ 4*π)\ntrue\n\njulia> typeof(dC)\nDistributedArrays.DArray{Float64,1,Array{Float64,1}}\n\njulia> cuB = CuArray(B);\n\njulia> cuA = CuArray(A);\n\njulia> cuC = 2 .* cuA ./ cuB;\n\njulia> all(cuC .≈ 4*π);\ntrue\n\njulia> typeof(cuC)\nCuArray{Float64,1}","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Keep in mind that some Julia features are not currently supported by CUDAnative.jl[2] , especially some functions like sin will need to be replaced with CUDAnative.sin(cc: @maleadt).","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"In the following example we will use both DistributedArrays.jl and CuArrays.jl to distribute an array across multiple processes and call a generic function on it.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"function power_method(M, v)\n    for i in 1:100\n        v = M*v\n        v /= norm(v)\n    end\n\n    return v, norm(M*v) / norm(v)  # or  (M*v) ./ v\nend","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"power_method repeatedly creates a new vector and normalizes it. We have not specified any type signature in function declaration, let's see if it works with the aforementioned datatypes:","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"julia> M = [2. 1; 1 1];\n\njulia> v = rand(2)\n2-element Array{Float64,1}:\n0.40395\n0.445877\n\njulia> power_method(M,v)\n([0.850651, 0.525731], 2.618033988749895)\n\njulia> cuM = CuArray(M);\n\njulia> cuv = CuArray(v);\n\njulia> curesult = power_method(cuM, cuv);\n\njulia> typeof(curesult)\nCuArray{Float64,1}\n\njulia> dM = distribute(M);\n\njulia> dv = distribute(v);\n\njulia> dC = power_method(dM, dv);\n\njulia> typeof(dC)\nTuple{DistributedArrays.DArray{Float64,1,Array{Float64,1}},Float64}","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"To end this short exposure to external packages, we can consider MPI.jl, a Julia wrapper of the MPI protocol. As it would take too long to consider every inner function, it would be better to simply appreciate the approach used to implement the protocol.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"Consider this toy script which simply calls each subprocess, instantiate its rank and when the master process is reached, performs the ranks' sum","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"import MPI\n\nMPI.Init()\n\ncomm = MPI.COMM_WORLD\nMPI.Barrier(comm)\n\nroot = 0\nr = MPI.Comm_rank(comm)\n\nsr = MPI.Reduce(r, MPI.SUM, root, comm)\n\nif(MPI.Comm_rank(comm) == root)\n   @printf(\"sum of ranks: %s\\n\", sr)\nend\n\nMPI.Finalize()","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"mpirun -np 4 ./julia example.jl","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding rma to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see https://mpi-forum.org/docs.","page":"Multi-processing and Distributed Computing"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html","category":"page","text":"[2]: Julia GPU man pages","page":"Multi-processing and Distributed Computing"},{"title":"Memory-mapped I/O","location":"stdlib/Mmap.html#Memory-mapped-I/O","category":"section","text":"","page":"Memory-mapped I/O"},{"title":"Memory-mapped I/O","location":"stdlib/Mmap.html","category":"page","text":"Mmap.Anonymous\nMmap.mmap\nMmap.sync!","page":"Memory-mapped I/O"},{"title":"Mmap.Anonymous","location":"stdlib/Mmap.html#Mmap.Anonymous","category":"type","text":"Mmap.Anonymous(name::AbstractString=\"\", readonly::Bool=false, create::Bool=true)\n\nCreate an IO-like object for creating zeroed-out mmapped-memory that is not tied to a file for use in Mmap.mmap. Used by SharedArray for creating shared memory arrays.\n\nExamples\n\njulia> using Mmap\n\njulia> anon = Mmap.Anonymous();\n\njulia> isreadable(anon)\ntrue\n\njulia> iswritable(anon)\ntrue\n\njulia> isopen(anon)\ntrue\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Mmap.mmap","location":"stdlib/Mmap.html#Mmap.mmap","category":"function","text":"Mmap.mmap(io::Union{IOStream,AbstractString,Mmap.AnonymousMmap}[, type::Type{Array{T,N}}, dims, offset]; grow::Bool=true, shared::Bool=true)\nMmap.mmap(type::Type{Array{T,N}}, dims)\n\nCreate an Array whose values are linked to a file, using memory-mapping. This provides a convenient way of working with data too large to fit in the computer's memory.\n\nThe type is an Array{T,N} with a bits-type element of T and dimension N that determines how the bytes of the array are interpreted. Note that the file must be stored in binary format, and no format conversions are possible (this is a limitation of operating systems, not Julia).\n\ndims is a tuple or single Integer specifying the size or length of the array.\n\nThe file is passed via the stream argument, either as an open IOStream or filename string. When you initialize the stream, use \"r\" for a \"read-only\" array, and \"w+\" to create a new array used to write values to disk.\n\nIf no type argument is specified, the default is Vector{UInt8}.\n\nOptionally, you can specify an offset (in bytes) if, for example, you want to skip over a header in the file. The default value for the offset is the current stream position for an IOStream.\n\nThe grow keyword argument specifies whether the disk file should be grown to accommodate the requested size of array (if the total file size is < requested array size). Write privileges are required to grow the file.\n\nThe shared keyword argument specifies whether the resulting Array and changes made to it will be visible to other processes mapping the same file.\n\nFor example, the following code\n\n# Create a file for mmapping\n# (you could alternatively use mmap to do this step, too)\nusing Mmap\nA = rand(1:20, 5, 30)\ns = open(\"/tmp/mmap.bin\", \"w+\")\n# We'll write the dimensions of the array as the first two Ints in the file\nwrite(s, size(A,1))\nwrite(s, size(A,2))\n# Now write the data\nwrite(s, A)\nclose(s)\n\n# Test by reading it back in\ns = open(\"/tmp/mmap.bin\")   # default is read-only\nm = read(s, Int)\nn = read(s, Int)\nA2 = Mmap.mmap(s, Matrix{Int}, (m,n))\n\ncreates a m-by-n Matrix{Int}, linked to the file associated with stream s.\n\nA more portable file would need to encode the word size – 32 bit or 64 bit – and endianness information in the header. In practice, consider encoding binary data using standard formats like HDF5 (which can be used with memory-mapping).\n\n\n\n\n\nMmap.mmap(io, BitArray, [dims, offset])\n\nCreate a BitArray whose values are linked to a file, using memory-mapping; it has the same purpose, works in the same way, and has the same arguments, as mmap, but the byte representation is different.\n\nExamples\n\njulia> using Mmap\n\njulia> io = open(\"mmap.bin\", \"w+\");\n\njulia> B = Mmap.mmap(io, BitArray, (25,30000));\n\njulia> B[3, 4000] = true;\n\njulia> Mmap.sync!(B);\n\njulia> close(io);\n\njulia> io = open(\"mmap.bin\", \"r+\");\n\njulia> C = Mmap.mmap(io, BitArray, (25,30000));\n\njulia> C[3, 4000]\ntrue\n\njulia> C[2, 4000]\nfalse\n\njulia> close(io)\n\njulia> rm(\"mmap.bin\")\n\nThis creates a 25-by-30000 BitArray, linked to the file associated with stream io.\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Mmap.sync!","location":"stdlib/Mmap.html#Mmap.sync!","category":"function","text":"Mmap.sync!(array)\n\nForces synchronization between the in-memory version of a memory-mapped Array or BitArray and the on-disk version.\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Code Loading","location":"manual/code-loading.html#code-loading","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"note: Note\nThis chapter covers the technical details of package loading. To install packages, use Pkg, Julia's built-in package manager, to add packages to your active environment. To use packages already in your active environment, write import X or using X, as described in the Modules documentation.","page":"Code Loading"},{"title":"Definitions","location":"manual/code-loading.html#Definitions","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Julia has two mechanisms for loading code:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Code inclusion: e.g. include(\"source.jl\"). Inclusion allows you to split a single program across multiple source files. The expression include(\"source.jl\") causes the contents of the file source.jl to be evaluated in the global scope of the module where the include call occurs. If include(\"source.jl\") is called multiple times, source.jl is evaluated multiple times. The included path, source.jl, is interpreted relative to the file where the include call occurs. This makes it simple to relocate a subtree of source files. In the REPL, included paths are interpreted relative to the current working directory, pwd().\nPackage loading: e.g. import X or using X. The import mechanism allows you to load a package—i.e. an independent, reusable collection of Julia code, wrapped in a module—and makes the resulting module available by the name X inside of the importing module. If the same X package is imported multiple times in the same Julia session, it is only loaded the first time—on subsequent imports, the importing module gets a reference to the same module. Note though, that import X can load different packages in different contexts: X can refer to one package named X in the main project but potentially to different packages also named X in each dependency. More on this below.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Code inclusion is quite straightforward and simple: it evaluates the given source file in the context of the caller. Package loading is built on top of code inclusion and serves a different purpose. The rest of this chapter focuses on the behavior and mechanics of package loading.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"A package is a source tree with a standard layout providing functionality that can be reused by other Julia projects. A package is loaded by import X or  using X statements. These statements also make the module named X—which results from loading the package code—available within the module where the import statement occurs. The meaning of X in import X is context-dependent: which X package is loaded depends on what code the statement occurs in. Thus, handling of import X happens in two stages: first, it determines what package is defined to be X in this context; second, it determines where that particular X package is found.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"These questions are answered by searching through the project environments listed in LOAD_PATH for project files (Project.toml or JuliaProject.toml), manifest files (Manifest.toml or JuliaManifest.toml), or folders of source files.","page":"Code Loading"},{"title":"Federation of packages","location":"manual/code-loading.html#Federation-of-packages","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Most of the time, a package is uniquely identifiable simply from its name. However, sometimes a project might encounter a situation where it needs to use two different packages that share the same name. While you might be able fix this by renaming one of the packages, being forced to do so can be highly disruptive in a large, shared code base. Instead, Julia's code loading mechanism allows the same package name to refer to different packages in different components of an application.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Julia supports federated package management, which means that multiple independent parties can maintain both public and private packages and registries of packages, and that projects can depend on a mix of public and private packages from different registries. Packages from various registries are installed and managed using a common set of tools and workflows. The Pkg package manager that ships with Julia lets you install and manage your projects' dependencies. It assists in creating and manipulating project files (which describe what other projects that your project depends on), and manifest files (which snapshot exact versions of your project's complete dependency graph).","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"One consequence of federation is that there cannot be a central authority for package naming. Different entities may use the same name to refer to unrelated packages. This possibility is unavoidable since these entities do not coordinate and may not even know about each other. Because of the lack of a central naming authority, a single project may end up depending on different packages that have the same name. Julia's package loading mechanism does not require package names to be globally unique, even within the dependency graph of a single project. Instead, packages are identified by universally unique identifiers (UUIDs), which get assigned when each package is created. Usually you won't have to work directly with these somewhat cumbersome 128-bit identifiers since Pkg will take care of generating and tracking them for you. However, these UUIDs provide the definitive answer to the question of \"what package does X refer to?\"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Since the decentralized naming problem is somewhat abstract, it may help to walk through a concrete scenario to understand the issue. Suppose you're developing an application called App, which uses two packages: Pub and  Priv. Priv is a private package that you created, whereas Pub is a public package that you use but don't control. When you created Priv, there was no public package by the name Priv. Subsequently, however, an unrelated package also named Priv has been published and become popular. In fact, the Pub package has started to use it. Therefore, when you next upgrade Pub to get the latest bug fixes and features, App will end up depending on two different packages named Priv—through no action of yours other than upgrading. App has a direct dependency on your private Priv package, and an indirect dependency, through Pub, on the new public Priv package. Since these two Priv packages are different but are both required for App to continue working correctly, the expression import Priv must refer to different Priv packages depending on whether it occurs in App's code or in Pub's code. To handle this, Julia's package loading mechanism distinguishes the two Priv packages by their UUID and picks the correct one based on its context (the module that called import). How this distinction works is determined by environments, as explained in the following sections.","page":"Code Loading"},{"title":"Environments","location":"manual/code-loading.html#Environments","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"An environment determines what import X and using X mean in various code contexts and what files these statements cause to be loaded. Julia understands two kinds of environments:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"A project environment is a directory with a project file and an optional manifest file, and forms an explicit environment. The project file determines what the names and identities of the direct dependencies of a project are. The manifest file, if present, gives a complete dependency graph, including all direct and indirect dependencies, exact versions of each dependency, and sufficient information to locate and load the correct version.\nA package directory is a directory containing the source trees of a set of packages as subdirectories, and forms an implicit environment. If X is a subdirectory of a package directory and X/src/X.jl exists, then the package X is available in the package directory environment and X/src/X.jl is the source file by which it is loaded.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"These can be intermixed to create a stacked environment: an ordered set of project environments and package directories, overlaid to make a single composite environment. The precedence and visibility rules then combine to determine which packages are available and where they get loaded from. Julia's load path forms a stacked environment, for example.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"These environment each serve a different purpose:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Project environments provide reproducibility. By checking a project environment into version control—e.g. a git repository—along with the rest of the project's source code, you can reproduce the exact state of the project and all of its dependencies. The manifest file, in particular, captures the exact version of every dependency, identified by a cryptographic hash of its source tree, which makes it possible for Pkg to retrieve the correct versions and be sure that you are running the exact code that was recorded for all dependencies.\nPackage directories provide convenience when a full carefully-tracked project environment is unnecessary. They are useful when you want to put a set of packages somewhere and be able to directly use them, without needing to create a project environment for them.\nStacked environments allow for adding tools to the primary environment. You can push an environment of development tools onto the end of the stack to make them available from the REPL and scripts, but not from inside packages.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"At a high-level, each environment conceptually defines three maps: roots, graph and paths. When resolving the meaning of import X, the roots and graph maps are used to determine the identity of X, while the paths map is used to locate the source code of X. The specific roles of the three maps are:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"roots: name::Symbol ⟶ uuid::UUID\nAn environment's roots map assigns package names to UUIDs for all the top-level dependencies that the environment makes available to the main project (i.e. the ones that can be loaded in Main). When Julia encounters import X in the main project, it looks up the identity of X as roots[:X].\ngraph: context::UUID ⟶ name::Symbol ⟶ uuid::UUID\nAn environment's graph is a multilevel map which assigns, for each context UUID, a map from names to UUIDs, similar to the roots map but specific to that context. When Julia sees import X in the code of the package whose UUID is context, it looks up the identity of X as graph[context][:X]. In particular, this means that import X can refer to different packages depending on context.\npaths: uuid::UUID × name::Symbol ⟶ path::String\nThe paths map assigns to each package UUID-name pair, the location of that package's entry-point source file. After the identity of X in import X has been resolved to a UUID via roots or graph (depending on whether it is loaded from the main project or a dependency), Julia determines what file to load to acquire X by looking up paths[uuid,:X] in the environment. Including this file should define a module named X. Once this package is loaded, any subsequent import resolving to the same uuid will create a new binding to the already-loaded package module.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Each kind of environment defines these three maps differently, as detailed in the following sections.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"note: Note\nFor ease of understanding, the examples throughout this chapter show full data structures for roots, graph and paths. However, Julia's package loading code does not explicitly create these. Instead, it lazily computes only as much of each structure as it needs to load a given package.","page":"Code Loading"},{"title":"Project environments","location":"manual/code-loading.html#Project-environments","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"A project environment is determined by a directory containing a project file called Project.toml, and optionally a manifest file called Manifest.toml. These files may also be called JuliaProject.toml and JuliaManifest.toml, in which case Project.toml and Manifest.toml are ignored. This allows for coexistence with other tools that might consider files called Project.toml and Manifest.toml significant. For pure Julia projects, however, the names Project.toml and Manifest.toml are preferred.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The roots, graph and paths maps of a project environment are defined as follows:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The roots map of the environment is determined by the contents of the project file, specifically, its top-level name and uuid entries and its [deps] section (all optional). Consider the following example project file for the hypothetical application, App, as described earlier:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"name = \"App\"\nuuid = \"8f986787-14fe-4607-ba5d-fbff2944afa9\"\n\n[deps]\nPriv = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\nPub  = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"This project file implies the following roots map, if it was represented by a Julia dictionary:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"roots = Dict(\n    :App  => UUID(\"8f986787-14fe-4607-ba5d-fbff2944afa9\"),\n    :Priv => UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"),\n    :Pub  => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Given this roots map, in App's code the statement import Priv will cause Julia to look up roots[:Priv], which yields ba13f791-ae1d-465a-978b-69c3ad90f72b, the UUID of the Priv package that is to be loaded in that context. This UUID identifies which Priv package to load and use when the main application evaluates import Priv.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The dependency graph of a project environment is determined by the contents of the manifest file, if present. If there is no manifest file, graph is empty. A manifest file contains a stanza for each of a project's direct or indirect dependencies. For each dependency, the file lists the package's UUID and a source tree hash or an explicit path to the source code. Consider the following example manifest file for App:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"[[Priv]] # the private one\ndeps = [\"Pub\", \"Zebra\"]\nuuid = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\npath = \"deps/Priv\"\n\n[[Priv]] # the public one\nuuid = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\ngit-tree-sha1 = \"1bf63d3be994fe83456a03b874b409cfd59a6373\"\nversion = \"0.1.5\"\n\n[[Pub]]\nuuid = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"\ngit-tree-sha1 = \"9ebd50e2b0dd1e110e842df3b433cb5869b0dd38\"\nversion = \"2.1.4\"\n\n  [Pub.deps]\n  Priv = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\n  Zebra = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\n\n[[Zebra]]\nuuid = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\ngit-tree-sha1 = \"e808e36a5d7173974b90a15a353b564f3494092f\"\nversion = \"3.4.2\"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"This manifest file describes a possible complete dependency graph for the App project:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"There are two different packages named Priv that the application uses. It uses a private package, which is a root dependency, and a public one, which is an indirect dependency through Pub. These are differentiated by their distinct UUIDs, and they have different deps:\nThe private Priv depends on the Pub and Zebra packages.\nThe public Priv has no dependencies.\nThe application also depends on the Pub package, which in turn depends on the public Priv and the same Zebra package that the private Priv package depends on.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"This dependency graph represented as a dictionary, looks like this:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"graph = Dict(\n    # Priv – the private one:\n    UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\") => Dict(\n        :Pub   => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Priv – the public one:\n    UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\") => Dict(),\n    # Pub:\n    UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\") => Dict(\n        :Priv  => UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Zebra:\n    UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\") => Dict(),\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Given this dependency graph, when Julia sees import Priv in the Pub package—which has UUID c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1—it looks up:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"graph[UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\")][:Priv]","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"and gets 2d15fe94-a1f7-436c-a4d8-07a9a496e01c, which indicates that in the context of the Pub package, import Priv refers to the public Priv package, rather than the private one which the app depends on directly. This is how the name Priv can refer to different packages in the main project than it does in one of its package's dependencies, which allows for duplicate names in the package ecosystem.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"What happens if import Zebra is evaluated in the main App code base? Since Zebra does not appear in the project file, the import will fail even though Zebra does appear in the manifest file. Moreover, if import Zebra occurs in the public Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—then that would also fail since that Priv package has no declared dependencies in the manifest file and therefore cannot load any packages. The Zebra package can only be loaded by packages for which it appear as an explicit dependency in the manifest file: the  Pub package and one of the Priv packages.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The paths map of a project environment is extracted from the manifest file. The path of a package uuid named X is determined by these rules (in order):","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If the project file in the directory matches uuid and name X, then either:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"It has a toplevel path entry, then uuid will be mapped to that path, interpreted relative to the directory containing the project file.\nOtherwise, uuid is mapped to  src/X.jl relative to the directory containing the project file.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If the above is not the case and the project file has a corresponding manifest file and the manifest contains a stanza matching uuid then:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If it has a path entry, use that path (relative to the directory containing the manifest file).\nIf it has a git-tree-sha1 entry, compute a deterministic hash function of uuid and git-tree-sha1—call it slug—and look for a directory named packages/X/$slug in each directory in the Julia DEPOT_PATH global array. Use the first such directory that exists.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If any of these result in success, the path to the source code entry point will be either that result, the relative path from that result plus src/X.jl; otherwise, there is no path mapping for uuid. When loading X, if no source code path is found, the lookup will fail, and the user may be prompted to install the appropriate package version or to take other corrective action (e.g. declaring X as a dependency).","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"In the example manifest file above, to find the path of the first Priv package—the one with UUID ba13f791-ae1d-465a-978b-69c3ad90f72b—Julia looks for its stanza in the manifest file, sees that it has a path entry, looks at deps/Priv relative to the App project directory—let's suppose the App code lives in /home/me/projects/App—sees that /home/me/projects/App/deps/Priv exists and therefore loads Priv from there.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If, on the other hand, Julia was loading the other Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—it finds its stanza in the manifest, see that it does not have a path entry, but that it does have a git-tree-sha1 entry. It then computes the slug for this UUID/SHA-1 pair, which is HDkrT (the exact details of this computation aren't important, but it is consistent and deterministic). This means that the path to this Priv package will be packages/Priv/HDkrT/src/Priv.jl in one of the package depots. Suppose the contents of DEPOT_PATH is [\"/home/me/.julia\", \"/usr/local/julia\"], then Julia will look at the following paths to see if they exist:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"/home/me/.julia/packages/Priv/HDkrT\n/usr/local/julia/packages/Priv/HDkrT","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Julia uses the first of these that exists to try to load the public Priv package from the file packages/Priv/HDKrT/src/Priv.jl in the depot where it was found.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Here is a representation of a possible paths map for our example App project environment, as provided in the Manifest given above for the dependency graph, after searching the local file system:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"paths = Dict(\n    # Priv – the private one:\n    (UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"), :Priv) =>\n        # relative entry-point inside `App` repo:\n        \"/home/me/projects/App/deps/Priv/src/Priv.jl\",\n    # Priv – the public one:\n    (UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"), :Priv) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Priv/HDkr/src/Priv.jl\",\n    # Pub:\n    (UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"), :Pub) =>\n        # package installed in the user depot:\n        \"/home/me/.julia/packages/Pub/oKpw/src/Pub.jl\",\n    # Zebra:\n    (UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"), :Zebra) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Zebra/me9k/src/Zebra.jl\",\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"This example map includes three different kinds of package locations (the first and third are part of the default load path):","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The private Priv package is \"vendored\" inside the App repository.\nThe public Priv and Zebra packages are in the system depot, where packages installed and managed by the system administrator live. These are available to all users on the system.\nThe Pub package is in the user depot, where packages installed by the user live. These are only available to the user who installed them.","page":"Code Loading"},{"title":"Package directories","location":"manual/code-loading.html#Package-directories","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Package directories provide a simpler kind of environment without the ability to handle name collisions. In a package directory, the set of top-level packages is the set of subdirectories that \"look like\" packages. A package X exists in a package directory if the directory contains one of the following \"entry point\" files:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"X.jl\nX/src/X.jl\nX.jl/src/X.jl","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Which dependencies a package in a package directory can import depends on whether the package contains a project file:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If it has a project file, it can only import those packages which are identified in the [deps] section of the project file.\nIf it does not have a project file, it can import any top-level package—i.e. the same packages that can be loaded in Main or the REPL.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The roots map is determined by examining the contents of the package directory to generate a list of all packages that exist. Additionally, a UUID will be assigned to each entry as follows: For a given package found inside the folder X...","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If X/Project.toml exists and has a uuid entry, then uuid is that value.\nIf X/Project.toml exists and but does not have a top-level UUID entry, uuid is a dummy UUID generated by hashing the canonical (real) path to X/Project.toml.\nOtherwise (if Project.toml does not exist), then uuid is the all-zero nil UUID.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The dependency graph of a project directory is determined by the presence and contents of project files in the subdirectory of each package. The rules are:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"If a package subdirectory has no project file, then it is omitted from graph and import statements in its code are treated as top-level, the same as the main project and REPL.\nIf a package subdirectory has a project file, then the graph entry for its UUID is the [deps] map of the project file, which is considered to be empty if the section is absent.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"As an example, suppose a package directory has the following structure and content:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Aardvark/\n    src/Aardvark.jl:\n        import Bobcat\n        import Cobra\n\nBobcat/\n    Project.toml:\n        [deps]\n        Cobra = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Bobcat.jl:\n        import Cobra\n        import Dingo\n\nCobra/\n    Project.toml:\n        uuid = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        [deps]\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Cobra.jl:\n        import Dingo\n\nDingo/\n    Project.toml:\n        uuid = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Dingo.jl:\n        # no imports","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Here is a corresponding roots structure, represented as a dictionary:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"roots = Dict(\n    :Aardvark => UUID(\"00000000-0000-0000-0000-000000000000\"), # no project file, nil UUID\n    :Bobcat   => UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), # dummy UUID based on path\n    :Cobra    => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), # UUID from project file\n    :Dingo    => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), # UUID from project file\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Here is the corresponding graph structure, represented as a dictionary:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"graph = Dict(\n    # Bobcat:\n    UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\") => Dict(\n        :Cobra => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"),\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Cobra:\n    UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\") => Dict(\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Dingo:\n    UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\") => Dict(),\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"A few general rules to note:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"A package without a project file can depend on any top-level dependency, and since every package in a package directory is available at the top-level, it can import all packages in the environment.\nA package with a project file cannot depend on one without a project file since packages with project files can only load packages in graph and packages without project files do not appear in graph.\nA package with a project file but no explicit UUID can only be depended on by packages without project files since dummy UUIDs assigned to these packages are strictly internal.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Observe the following specific instances of these rules in our example:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Aardvark can import on any of Bobcat, Cobra or Dingo; it does import Bobcat and Cobra.\nBobcat can and does import both Cobra and Dingo, which both have project files with UUIDs and are declared as dependencies in Bobcat's [deps] section.\nBobcat cannot depend on Aardvark since Aardvark does not have a project file.\nCobra can and does import Dingo, which has a project file and UUID, and is declared as a dependency in Cobra's  [deps] section.\nCobra cannot depend on Aardvark or Bobcat since neither have real UUIDs.\nDingo cannot import anything because it has a project file without a [deps] section.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The paths map in a package directory is simple: it maps subdirectory names to their corresponding entry-point paths. In other words, if the path to our example project directory is /home/me/animals then the paths map could be represented by this dictionary:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"paths = Dict(\n    (UUID(\"00000000-0000-0000-0000-000000000000\"), :Aardvark) =>\n        \"/home/me/AnimalPackages/Aardvark/src/Aardvark.jl\",\n    (UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), :Bobcat) =>\n        \"/home/me/AnimalPackages/Bobcat/src/Bobcat.jl\",\n    (UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), :Cobra) =>\n        \"/home/me/AnimalPackages/Cobra/src/Cobra.jl\",\n    (UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), :Dingo) =>\n        \"/home/me/AnimalPackages/Dingo/src/Dingo.jl\",\n)","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Since all packages in a package directory environment are, by definition, subdirectories with the expected entry-point files, their paths map entries always have this form.","page":"Code Loading"},{"title":"Environment stacks","location":"manual/code-loading.html#Environment-stacks","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The third and final kind of environment is one that combines other environments by overlaying several of them, making the packages in each available in a single composite environment. These composite environments are called environment stacks. The Julia LOAD_PATH global defines an environment stack—the environment in which the Julia process operates. If you want your Julia process to have access only to the packages in one project or package directory, make it the only entry in LOAD_PATH. It is often quite useful, however, to have access to some of your favorite tools—standard libraries, profilers, debuggers, personal utilities, etc.—even if they are not dependencies of the project you're working on. By adding an environment containing these tools to the load path, you immediately have access to them in top-level code without needing to add them to your project.","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The mechanism for combining the roots, graph and paths data structures of the components of an environment stack is simple: they are merged as dictionaries, favoring earlier entries over later ones in the case of key collisions. In other words, if we have stack = [env₁, env₂, …] then we have:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"roots = reduce(merge, reverse([roots₁, roots₂, …]))\ngraph = reduce(merge, reverse([graph₁, graph₂, …]))\npaths = reduce(merge, reverse([paths₁, paths₂, …]))","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The subscripted rootsᵢ, graphᵢ and pathsᵢ variables correspond to the subscripted environments, envᵢ, contained in stack. The reverse is present because merge favors the last argument rather than first when there are collisions between keys in its argument dictionaries. There are a couple of noteworthy features of this design:","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"The primary environment—i.e. the first environment in a stack—is faithfully embedded in a stacked environment. The full dependency graph of the first environment in a stack is guaranteed to be included intact in the stacked environment including the same versions of all dependencies.\nPackages in non-primary environments can end up using incompatible versions of their dependencies even if their own environments are entirely compatible. This can happen when one of their dependencies is shadowed by a version in an earlier environment in the stack (either by graph or path, or both).","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Since the primary environment is typically the environment of a project you're working on, while environments later in the stack contain additional tools, this is the right trade-off: it's better to break your development tools but keep the project working. When such incompatibilities occur, you'll typically want to upgrade your dev tools to versions that are compatible with the main project.","page":"Code Loading"},{"title":"Conclusion","location":"manual/code-loading.html#Conclusion","category":"section","text":"","page":"Code Loading"},{"title":"Code Loading","location":"manual/code-loading.html","category":"page","text":"Federated package management and precise software reproducibility are difficult but worthy goals in a package system. In combination, these goals lead to a more complex package loading mechanism than most dynamic languages have, but it also yields scalability and reproducibility that is more commonly associated with static languages. Typically, Julia users should be able to use the built-in package manager to manage their projects without needing a precise understanding of these interactions. A call to Pkg.add(\"X\") will add to the appropriate project and manifest files, selected via Pkg.activate(\"Y\"), so that a future call to import X will load X without further thought.","page":"Code Loading"},{"title":"Artifacts","location":"stdlib/Artifacts.html#Artifacts","category":"section","text":"","page":"Artifacts"},{"title":"Artifacts","location":"stdlib/Artifacts.html","category":"page","text":"DocTestSetup = :(using Artifacts)","page":"Artifacts"},{"title":"Artifacts","location":"stdlib/Artifacts.html","category":"page","text":"Starting with Julia 1.6, the artifacts support has moved from Pkg.jl to Julia itself. Until proper documentation can be added here, you can learn more about artifacts in the Pkg.jl manual at https://julialang.github.io/Pkg.jl/v1/artifacts/.","page":"Artifacts"},{"title":"Artifacts","location":"stdlib/Artifacts.html","category":"page","text":"compat: Julia 1.6\nJulia's artifacts API requires at least Julia 1.6. In Julia versions 1.3 to 1.5, you can use Pkg.Artifacts instead.","page":"Artifacts"},{"title":"Artifacts","location":"stdlib/Artifacts.html","category":"page","text":"Artifacts.artifact_meta\nArtifacts.artifact_hash\nArtifacts.find_artifacts_toml\nArtifacts.@artifact_str","page":"Artifacts"},{"title":"Artifacts.artifact_meta","location":"stdlib/Artifacts.html#Artifacts.artifact_meta","category":"function","text":"artifact_meta(name::String, artifacts_toml::String;\n              platform::AbstractPlatform = HostPlatform(),\n              pkg_uuid::Union{Base.UUID,Nothing}=nothing)\n\nGet metadata about a given artifact (identified by name) stored within the given (Julia)Artifacts.toml file.  If the artifact is platform-specific, use platform to choose the most appropriate mapping.  If none is found, return nothing.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.artifact_hash","location":"stdlib/Artifacts.html#Artifacts.artifact_hash","category":"function","text":"artifact_hash(name::String, artifacts_toml::String;\n              platform::AbstractPlatform = HostPlatform())\n\nThin wrapper around artifact_meta() to return the hash of the specified, platform- collapsed artifact.  Returns nothing if no mapping can be found.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.find_artifacts_toml","location":"stdlib/Artifacts.html#Artifacts.find_artifacts_toml","category":"function","text":"find_artifacts_toml(path::String)\n\nGiven the path to a .jl file, (such as the one returned by __source__.file in a macro context), find the (Julia)Artifacts.toml that is contained within the containing project (if it exists), otherwise return nothing.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.@artifact_str","location":"stdlib/Artifacts.html#Artifacts.@artifact_str","category":"macro","text":"macro artifact_str(name)\n\nReturn the on-disk path to an artifact. Automatically looks the artifact up by name in the project's (Julia)Artifacts.toml file. Throws an error on if the requested artifact is not present. If run in the REPL, searches for the toml file starting in the current directory, see find_artifacts_toml() for more.\n\nIf the artifact is marked \"lazy\" and the package has using LazyArtifacts defined, the artifact will be downloaded on-demand with Pkg the first time this macro tries to compute the path. The files will then be left installed locally for later.\n\nIf name contains a forward or backward slash, all elements after the first slash will be taken to be path names indexing into the artifact, allowing for an easy one-liner to access a single file/directory within an artifact.  Example:\n\nffmpeg_path = @artifact\"FFMPEG/bin/ffmpeg\"\n\ncompat: Julia 1.3\nThis macro requires at least Julia 1.3.\n\ncompat: Julia 1.6\nSlash-indexing requires at least Julia 1.6.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Interfaces","location":"manual/interfaces.html#Interfaces","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A lot of the power and extensibility in Julia comes from a collection of informal interfaces.  By extending a few specific methods to work for a custom type, objects of that type not only receive those functionalities, but they are also able to be used in other methods that are written to generically build upon those behaviors.","page":"Interfaces"},{"title":"Iteration","location":"manual/interfaces.html#man-interface-iteration","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Required methods  Brief description\niterate(iter)  Returns either a tuple of the first item and initial state or nothing if empty\niterate(iter, state)  Returns either a tuple of the next item and next state or nothing if no items remain\nImportant optional methods Default definition Brief description\nIteratorSize(IterType) HasLength() One of HasLength(), HasShape{N}(), IsInfinite(), or SizeUnknown() as appropriate\nIteratorEltype(IterType) HasEltype() Either EltypeUnknown() or HasEltype() as appropriate\neltype(IterType) Any The type of the first entry of the tuple returned by iterate()\nlength(iter) (undefined) The number of items, if known\nsize(iter, [dim]) (undefined) The number of items in each dimension, if known","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Value returned by IteratorSize(IterType) Required Methods\nHasLength() length(iter)\nHasShape{N}() length(iter)  and size(iter, [dim])\nIsInfinite() (none)\nSizeUnknown() (none)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Value returned by IteratorEltype(IterType) Required Methods\nHasEltype() eltype(IterType)\nEltypeUnknown() (none)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Sequential iteration is implemented by the iterate function. Instead of mutating objects as they are iterated over, Julia iterators may keep track of the iteration state externally from the object. The return value from iterate is always either a tuple of a value and a state, or nothing if no elements remain. The state object will be passed back to the iterate function on the next iteration and is generally considered an implementation detail private to the iterable object.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Any object that defines this function is iterable and can be used in the many functions that rely upon iteration. It can also be used directly in a for loop since the syntax:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"for item in iter   # or  \"for item = iter\"\n    # body\nend","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"is translated into:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"next = iterate(iter)\nwhile next !== nothing\n    (item, state) = next\n    # body\n    next = iterate(iter, state)\nend","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A simple example is an iterable sequence of square numbers with a defined length:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct Squares\n           count::Int\n       end\n\njulia> Base.iterate(S::Squares, state=1) = state > S.count ? nothing : (state*state, state+1)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"With only iterate definition, the Squares type is already pretty powerful. We can iterate over all the elements:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> for item in Squares(7)\n           println(item)\n       end\n1\n4\n9\n16\n25\n36\n49","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"We can use many of the builtin methods that work with iterables, like in, or mean and std from the Statistics standard library module:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> 25 in Squares(10)\ntrue\n\njulia> using Statistics\n\njulia> mean(Squares(100))\n3383.5\n\njulia> std(Squares(100))\n3024.355854282583","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"There are a few more methods we can extend to give Julia more information about this iterable collection.  We know that the elements in a Squares sequence will always be Int. By extending the eltype method, we can give that information to Julia and help it make more specialized code in the more complicated methods. We also know the number of elements in our sequence, so we can extend length, too:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.eltype(::Type{Squares}) = Int # Note that this is defined for the type\n\njulia> Base.length(S::Squares) = S.count","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Now, when we ask Julia to collect all the elements into an array it can preallocate a Vector{Int} of the right size instead of naively push!ing each element into a Vector{Any}:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> collect(Squares(4))\n4-element Vector{Int64}:\n  1\n  4\n  9\n 16","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"While we can rely upon generic implementations, we can also extend specific methods where we know there is a simpler algorithm. For example, there's a formula to compute the sum of squares, so we can override the generic iterative version with a more performant solution:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.sum(S::Squares) = (n = S.count; return n*(n+1)*(2n+1)÷6)\n\njulia> sum(Squares(1803))\n1955361914","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This is a very common pattern throughout Julia Base: a small set of required methods define an informal interface that enable many fancier behaviors. In some cases, types will want to additionally specialize those extra behaviors when they know a more efficient algorithm can be used in their specific case.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"It is also often useful to allow iteration over a collection in reverse order by iterating over Iterators.reverse(iterator).  To actually support reverse-order iteration, however, an iterator type T needs to implement iterate for Iterators.Reverse{T}. (Given r::Iterators.Reverse{T}, the underling iterator of type T is r.itr.) In our Squares example, we would implement Iterators.Reverse{Squares} methods:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.iterate(rS::Iterators.Reverse{Squares}, state=rS.itr.count) = state < 1 ? nothing : (state*state, state-1)\n\njulia> collect(Iterators.reverse(Squares(4)))\n4-element Vector{Int64}:\n 16\n  9\n  4\n  1","page":"Interfaces"},{"title":"Indexing","location":"manual/interfaces.html#Indexing","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement Brief description\ngetindex(X, i) X[i], indexed element access\nsetindex!(X, v, i) X[i] = v, indexed assignment\nfirstindex(X) The first index, used in X[begin]\nlastindex(X) The last index, used in X[end]","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For the Squares iterable above, we can easily compute the ith element of the sequence by squaring it.  We can expose this as an indexing expression S[i]. To opt into this behavior, Squares simply needs to define getindex:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> function Base.getindex(S::Squares, i::Int)\n           1 <= i <= S.count || throw(BoundsError(S, i))\n           return i*i\n       end\n\njulia> Squares(100)[23]\n529","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Additionally, to support the syntax S[begin] and S[end], we must define firstindex and lastindex to specify the first and last valid indices, respectively:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.firstindex(S::Squares) = 1\n\njulia> Base.lastindex(S::Squares) = length(S)\n\njulia> Squares(23)[end]\n529","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For multi-dimensional begin/end indexing as in a[3, begin, 7], for example, you should define firstindex(a, dim) and lastindex(a, dim) (which default to calling first and last on axes(a, dim), respectively).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Note, though, that the above only defines getindex with one integer index. Indexing with anything other than an Int will throw a MethodError saying that there was no matching method. In order to support indexing with ranges or vectors of Ints, separate methods must be written:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> Base.getindex(S::Squares, i::Number) = S[convert(Int, i)]\n\njulia> Base.getindex(S::Squares, I) = [S[i] for i in I]\n\njulia> Squares(10)[[3,4.,5]]\n3-element Vector{Int64}:\n  9\n 16\n 25","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"While this is starting to support more of the indexing operations supported by some of the builtin types, there's still quite a number of behaviors missing. This Squares sequence is starting to look more and more like a vector as we've added behaviors to it. Instead of defining all these behaviors ourselves, we can officially define it as a subtype of an AbstractArray.","page":"Interfaces"},{"title":"Abstract Arrays","location":"manual/interfaces.html#man-interface-array","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement  Brief description\nsize(A)  Returns a tuple containing the dimensions of A\ngetindex(A, i::Int)  (if IndexLinear) Linear scalar indexing\ngetindex(A, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexing\nsetindex!(A, v, i::Int)  (if IndexLinear) Scalar indexed assignment\nsetindex!(A, v, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexed assignment\nOptional methods Default definition Brief description\nIndexStyle(::Type) IndexCartesian() Returns either IndexLinear() or IndexCartesian(). See the description below.\ngetindex(A, I...) defined in terms of scalar getindex Multidimensional and nonscalar indexing\nsetindex!(A, X, I...) defined in terms of scalar setindex! Multidimensional and nonscalar indexed assignment\niterate defined in terms of scalar getindex Iteration\nlength(A) prod(size(A)) Number of elements\nsimilar(A) similar(A, eltype(A), size(A)) Return a mutable array with the same shape and element type\nsimilar(A, ::Type{S}) similar(A, S, size(A)) Return a mutable array with the same shape and the specified element type\nsimilar(A, dims::Dims) similar(A, eltype(A), dims) Return a mutable array with the same element type and size dims\nsimilar(A, ::Type{S}, dims::Dims) Array{S}(undef, dims) Return a mutable array with the specified element type and size\nNon-traditional indices Default definition Brief description\naxes(A) map(OneTo, size(A)) Return the a tuple of AbstractUnitRange{<:Integer} of valid indices\nsimilar(A, ::Type{S}, inds) similar(A, S, Base.to_shape(inds)) Return a mutable array with the specified indices inds (see below)\nsimilar(T::Union{Type,Function}, inds) T(Base.to_shape(inds)) Return an array similar to T with the specified indices inds (see below)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If a type is defined as a subtype of AbstractArray, it inherits a very large set of rich behaviors including iteration and multidimensional indexing built on top of single-element access.  See the arrays manual page and the Julia Base section for more supported methods.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A key part in defining an AbstractArray subtype is IndexStyle. Since indexing is such an important part of an array and often occurs in hot loops, it's important to make both indexing and indexed assignment as efficient as possible.  Array data structures are typically defined in one of two ways: either it most efficiently accesses its elements using just one index (linear indexing) or it intrinsically accesses the elements with indices specified for every dimension.  These two modalities are identified by Julia as IndexLinear() and IndexCartesian().  Converting a linear index to multiple indexing subscripts is typically very expensive, so this provides a traits-based mechanism to enable efficient generic code for all array types.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This distinction determines which scalar indexing methods the type must define. IndexLinear() arrays are simple: just define getindex(A::ArrayType, i::Int).  When the array is subsequently indexed with a multidimensional set of indices, the fallback getindex(A::AbstractArray, I...)() efficiently converts the indices into one linear index and then calls the above method. IndexCartesian() arrays, on the other hand, require methods to be defined for each supported dimensionality with ndims(A) Int indices. For example, SparseMatrixCSC from the SparseArrays standard library module, only supports two dimensions, so it just defines getindex(A::SparseMatrixCSC, i::Int, j::Int). The same holds for setindex!.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Returning to the sequence of squares from above, we could instead define it as a subtype of an AbstractArray{Int, 1}:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct SquaresVector <: AbstractArray{Int, 1}\n           count::Int\n       end\n\njulia> Base.size(S::SquaresVector) = (S.count,)\n\njulia> Base.IndexStyle(::Type{<:SquaresVector}) = IndexLinear()\n\njulia> Base.getindex(S::SquaresVector, i::Int) = i*i","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Note that it's very important to specify the two parameters of the AbstractArray; the first defines the eltype, and the second defines the ndims. That supertype and those three methods are all it takes for SquaresVector to be an iterable, indexable, and completely functional array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> s = SquaresVector(4)\n4-element SquaresVector:\n  1\n  4\n  9\n 16\n\njulia> s[s .> 8]\n2-element Vector{Int64}:\n  9\n 16\n\njulia> s + s\n4-element Vector{Int64}:\n  2\n  8\n 18\n 32\n\njulia> sin.(s)\n4-element Vector{Float64}:\n  0.8414709848078965\n -0.7568024953079282\n  0.4121184852417566\n -0.2879033166650653","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"As a more complicated example, let's define our own toy N-dimensional sparse-like array type built on top of Dict:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> struct SparseArray{T,N} <: AbstractArray{T,N}\n           data::Dict{NTuple{N,Int}, T}\n           dims::NTuple{N,Int}\n       end\n\njulia> SparseArray(::Type{T}, dims::Int...) where {T} = SparseArray(T, dims);\n\njulia> SparseArray(::Type{T}, dims::NTuple{N,Int}) where {T,N} = SparseArray{T,N}(Dict{NTuple{N,Int}, T}(), dims);\n\njulia> Base.size(A::SparseArray) = A.dims\n\njulia> Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where {T} = SparseArray(T, dims)\n\njulia> Base.getindex(A::SparseArray{T,N}, I::Vararg{Int,N}) where {T,N} = get(A.data, I, zero(T))\n\njulia> Base.setindex!(A::SparseArray{T,N}, v, I::Vararg{Int,N}) where {T,N} = (A.data[I] = v)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Notice that this is an IndexCartesian array, so we must manually define getindex and setindex! at the dimensionality of the array. Unlike the SquaresVector, we are able to define setindex!, and so we can mutate the array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A = SparseArray(Float64, 3, 3)\n3×3 SparseArray{Float64, 2}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\njulia> fill!(A, 2)\n3×3 SparseArray{Float64, 2}:\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n\njulia> A[:] = 1:length(A); A\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The result of indexing an AbstractArray can itself be an array (for instance when indexing by an AbstractRange). The AbstractArray fallback methods use similar to allocate an Array of the appropriate size and element type, which is filled in using the basic indexing method described above. However, when implementing an array wrapper you often want the result to be wrapped as well:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A[1:2,:]\n2×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In this example it is accomplished by defining Base.similar{T}(A::SparseArray, ::Type{T}, dims::Dims) to create the appropriate wrapped array. (Note that while similar supports 1- and 2-argument forms, in most case you only need to specialize the 3-argument form.) For this to work it's important that SparseArray is mutable (supports setindex!). Defining similar, getindex and setindex! for SparseArray also makes it possible to copy the array:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> copy(A)\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In addition to all the iterable and indexable methods from above, these types can also interact with each other and use most of the methods defined in Julia Base for AbstractArrays:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> A[SquaresVector(3)]\n3-element SparseArray{Float64, 1}:\n 1.0\n 4.0\n 9.0\n\njulia> sum(A)\n45.0","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If you are defining an array type that allows non-traditional indexing (indices that start at something other than 1), you should specialize axes. You should also specialize similar so that the dims argument (ordinarily a Dims size-tuple) can accept AbstractUnitRange objects, perhaps range-types Ind of your own design. For more information, see Arrays with custom indices.","page":"Interfaces"},{"title":"Strided Arrays","location":"manual/interfaces.html#man-interface-strided-arrays","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement  Brief description\nstrides(A)  Return the distance in memory (in number of elements) between adjacent elements in each dimension as a tuple. If A is an AbstractArray{T,0}, this should return an empty tuple.\nBase.unsafe_convert(::Type{Ptr{T}}, A)  Return the native address of an array.\nBase.elsize(::Type{<:A})  Return the stride between consecutive elements in the array.\nOptional methods Default definition Brief description\nstride(A, i::Int) strides(A)[i] Return the distance in memory (in number of elements) between adjacent elements in dimension k.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"A strided array is a subtype of AbstractArray whose entries are stored in memory with fixed strides. Provided the element type of the array is compatible with BLAS, a strided array can utilize BLAS and LAPACK routines for more efficient linear algebra routines.  A typical example of a user-defined strided array is one that wraps a standard Array with additional structure.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Warning: do not implement these methods if the underlying storage is not actually strided, as it may lead to incorrect results or segmentation faults.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Here are some examples to demonstrate which type of arrays are strided and which are not:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"1:5   # not strided (there is no storage associated with this array.)\nVector(1:5)  # is strided with strides (1,)\nA = [1 5; 2 6; 3 7; 4 8]  # is strided with strides (1,4)\nV = view(A, 1:2, :)   # is strided with strides (1,4)\nV = view(A, 1:2:3, 1:2)   # is strided with strides (2,4)\nV = view(A, [1,2,4], :)   # is not strided, as the spacing between rows is not fixed.","page":"Interfaces"},{"title":"Customizing broadcasting","location":"manual/interfaces.html#man-interfaces-broadcasting","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Methods to implement Brief description\nBase.BroadcastStyle(::Type{SrcType}) = SrcStyle() Broadcasting behavior of SrcType\nBase.similar(bc::Broadcasted{DestStyle}, ::Type{ElType}) Allocation of output container\nOptional methods \nBase.BroadcastStyle(::Style1, ::Style2) = Style12() Precedence rules for mixing styles\nBase.axes(x) Declaration of the indices of x, as per axes(x).\nBase.broadcastable(x) Convert x to an object that has axes and supports indexing\nBypassing default machinery \nBase.copy(bc::Broadcasted{DestStyle}) Custom implementation of broadcast\nBase.copyto!(dest, bc::Broadcasted{DestStyle}) Custom implementation of broadcast!, specializing on DestStyle\nBase.copyto!(dest::DestType, bc::Broadcasted{Nothing}) Custom implementation of broadcast!, specializing on DestType\nBase.Broadcast.broadcasted(f, args...) Override the default lazy behavior within a fused expression\nBase.Broadcast.instantiate(bc::Broadcasted{DestStyle}) Override the computation of the lazy broadcast's axes","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Broadcasting is triggered by an explicit call to broadcast or broadcast!, or implicitly by \"dot\" operations like A .+ b or f.(x, y). Any object that has axes and supports indexing can participate as an argument in broadcasting, and by default the result is stored in an Array. This basic framework is extensible in three major ways:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Ensuring that all arguments support broadcast\nSelecting an appropriate output array for the given set of arguments\nSelecting an efficient implementation for the given set of arguments","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Not all types support axes and indexing, but many are convenient to allow in broadcast. The Base.broadcastable function is called on each argument to broadcast, allowing it to return something different that supports axes and indexing. By default, this is the identity function for all AbstractArrays and Numbers — they already support axes and indexing. For a handful of other types (including but not limited to types themselves, functions, special singletons like missing and nothing, and dates), Base.broadcastable returns the argument wrapped in a Ref to act as a 0-dimensional \"scalar\" for the purposes of broadcasting. Custom types can similarly specialize Base.broadcastable to define their shape, but they should follow the convention that collect(Base.broadcastable(x)) == collect(x). A notable exception is AbstractString; strings are special-cased to behave as scalars for the purposes of broadcast even though they are iterable collections of their characters (see Strings for more).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The next two steps (selecting the output array and implementation) are dependent upon determining a single answer for a given set of arguments. Broadcast must take all the varied types of its arguments and collapse them down to just one output array and one implementation. Broadcast calls this single answer a \"style.\" Every broadcastable object each has its own preferred style, and a promotion-like system is used to combine these styles into a single answer — the \"destination style\".","page":"Interfaces"},{"title":"Broadcast Styles","location":"manual/interfaces.html#Broadcast-Styles","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle is the abstract type from which all broadcast styles are derived. When used as a function it has two possible forms, unary (single-argument) and binary. The unary variant states that you intend to implement specific broadcasting behavior and/or output type, and do not wish to rely on the default fallback Broadcast.DefaultArrayStyle.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"To override these defaults, you can define a custom BroadcastStyle for your object:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct MyStyle <: Broadcast.BroadcastStyle end\nBase.BroadcastStyle(::Type{<:MyType}) = MyStyle()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In some cases it might be convenient not to have to define MyStyle, in which case you can leverage one of the general broadcast wrappers:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Type{<:MyType}) = Broadcast.Style{MyType}() can be used for arbitrary types.\nBase.BroadcastStyle(::Type{<:MyType}) = Broadcast.ArrayStyle{MyType}() is preferred if MyType is an AbstractArray.\nFor AbstractArrays that only support a certain dimensionality, create a subtype of Broadcast.AbstractArrayStyle{N} (see below).","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"When your broadcast operation involves several arguments, individual argument styles get combined to determine a single DestStyle that controls the type of the output container. For more details, see below.","page":"Interfaces"},{"title":"Selecting an appropriate output array","location":"manual/interfaces.html#Selecting-an-appropriate-output-array","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The broadcast style is computed for every broadcasting operation to allow for dispatch and specialization. The actual allocation of the result array is handled by similar, using the Broadcasted object as its first argument.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.similar(bc::Broadcasted{DestStyle}, ::Type{ElType})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The fallback definition is","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"similar(bc::Broadcasted{DefaultArrayStyle{N}}, ::Type{ElType}) where {N,ElType} =\n    similar(Array{ElType}, axes(bc))","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"However, if needed you can specialize on any or all of these arguments. The final argument bc is a lazy representation of a (potentially fused) broadcast operation, a Broadcasted object.  For these purposes, the most important fields of the wrapper are f and args, describing the function and argument list, respectively.  Note that the argument list can — and often does — include other nested Broadcasted wrappers.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For a complete example, let's say you have created a type, ArrayAndChar, that stores an array and a single character:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct ArrayAndChar{T,N} <: AbstractArray{T,N}\n    data::Array{T,N}\n    char::Char\nend\nBase.size(A::ArrayAndChar) = size(A.data)\nBase.getindex(A::ArrayAndChar{T,N}, inds::Vararg{Int,N}) where {T,N} = A.data[inds...]\nBase.setindex!(A::ArrayAndChar{T,N}, val, inds::Vararg{Int,N}) where {T,N} = A.data[inds...] = val\nBase.showarg(io::IO, A::ArrayAndChar, toplevel) = print(io, typeof(A), \" with char '\", A.char, \"'\")","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"You might want broadcasting to preserve the char \"metadata.\" First we define","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Type{<:ArrayAndChar}) = Broadcast.ArrayStyle{ArrayAndChar}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This means we must also define a corresponding similar method:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"function Base.similar(bc::Broadcast.Broadcasted{Broadcast.ArrayStyle{ArrayAndChar}}, ::Type{ElType}) where ElType\n    # Scan the inputs for the ArrayAndChar:\n    A = find_aac(bc)\n    # Use the char field of A to create the output\n    ArrayAndChar(similar(Array{ElType}, axes(bc)), A.char)\nend\n\n\"`A = find_aac(As)` returns the first ArrayAndChar among the arguments.\"\nfind_aac(bc::Base.Broadcast.Broadcasted) = find_aac(bc.args)\nfind_aac(args::Tuple) = find_aac(find_aac(args[1]), Base.tail(args))\nfind_aac(x) = x\nfind_aac(::Tuple{}) = nothing\nfind_aac(a::ArrayAndChar, rest) = a\nfind_aac(::Any, rest) = find_aac(rest)","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"From these definitions, one obtains the following behavior:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"julia> a = ArrayAndChar([1 2; 3 4], 'x')\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 1  2\n 3  4\n\njulia> a .+ 1\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 2  3\n 4  5\n\njulia> a .+ [5,10]\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n  6   7\n 13  14","page":"Interfaces"},{"title":"Extending broadcast with custom implementations","location":"manual/interfaces.html#extending-in-place-broadcast","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In general, a broadcast operation is represented by a lazy Broadcasted container that holds onto the function to be applied alongside its arguments. Those arguments may themselves be more nested Broadcasted containers, forming a large expression tree to be evaluated. A nested tree of Broadcasted containers is directly constructed by the implicit dot syntax; 5 .+ 2.*x is transiently represented by Broadcasted(+, 5, Broadcasted(*, 2, x)), for example. This is invisible to users as it is immediately realized through a call to copy, but it is this container that provides the basis for broadcast's extensibility for authors of custom types. The built-in broadcast machinery will then determine the result type and size based upon the arguments, allocate it, and then finally copy the realization of the Broadcasted object into it with a default copyto!(::AbstractArray, ::Broadcasted) method. The built-in fallback broadcast and broadcast! methods similarly construct a transient Broadcasted representation of the operation so they can follow the same codepath. This allows custom array implementations to provide their own copyto! specialization to customize and optimize broadcasting. This is again determined by the computed broadcast style. This is such an important part of the operation that it is stored as the first type parameter of the Broadcasted type, allowing for dispatch and specialization.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For some types, the machinery to \"fuse\" operations across nested levels of broadcasting is not available or could be done more efficiently incrementally. In such cases, you may need or want to evaluate x .* (x .+ 1) as if it had been written broadcast(*, x, broadcast(+, x, 1)), where the inner operation is evaluated before tackling the outer operation. This sort of eager operation is directly supported by a bit of indirection; instead of directly constructing Broadcasted objects, Julia lowers the fused expression x .* (x .+ 1) to Broadcast.broadcasted(*, x, Broadcast.broadcasted(+, x, 1)). Now, by default, broadcasted just calls the Broadcasted constructor to create the lazy representation of the fused expression tree, but you can choose to override it for a particular combination of function and arguments.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"As an example, the builtin AbstractRange objects use this machinery to optimize pieces of broadcasted expressions that can be eagerly evaluated purely in terms of the start, step, and length (or stop) instead of computing every single element. Just like all the other machinery, broadcasted also computes and exposes the combined broadcast style of its arguments, so instead of specializing on broadcasted(f, args...), you can specialize on broadcasted(::DestStyle, f, args...) for any combination of style, function, and arguments.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For example, the following definition supports the negation of ranges:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"broadcasted(::DefaultArrayStyle{1}, ::typeof(-), r::OrdinalRange) = range(-first(r), step=-step(r), length=length(r))","page":"Interfaces"},{"title":"Extending in-place broadcasting","location":"manual/interfaces.html#extending-in-place-broadcast-2","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In-place broadcasting can be supported by defining the appropriate copyto!(dest, bc::Broadcasted) method. Because you might want to specialize either on dest or the specific subtype of bc, to avoid ambiguities between packages we recommend the following convention.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If you wish to specialize on a particular style DestStyle, define a method for","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"copyto!(dest, bc::Broadcasted{DestStyle})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Optionally, with this form you can also specialize on the type of dest.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If instead you want to specialize on the destination type DestType without specializing on DestStyle, then you should define a method with the following signature:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"copyto!(dest::DestType, bc::Broadcasted{Nothing})","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"This leverages a fallback implementation of copyto! that converts the wrapper into a Broadcasted{Nothing}. Consequently, specializing on DestType has lower precedence than methods that specialize on DestStyle.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Similarly, you can completely override out-of-place broadcasting with a copy(::Broadcasted) method.","page":"Interfaces"},{"title":"Working with Broadcasted objects","location":"manual/interfaces.html#Working-with-Broadcasted-objects","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"In order to implement such a copy or copyto!, method, of course, you must work with the Broadcasted wrapper to compute each element. There are two main ways of doing so:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Broadcast.flatten recomputes the potentially nested operation into a single function and flat list of arguments. You are responsible for implementing the broadcasting shape rules yourself, but this may be helpful in limited situations.\nIterating over the CartesianIndices of the axes(::Broadcasted) and using indexing with the resulting CartesianIndex object to compute the result.","page":"Interfaces"},{"title":"Writing binary broadcasting rules","location":"manual/interfaces.html#writing-binary-broadcasting-rules","category":"section","text":"","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"The precedence rules are defined by binary BroadcastStyle calls:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Style1, ::Style2) = Style12()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"where Style12 is the BroadcastStyle you want to choose for outputs involving arguments of Style1 and Style2. For example,","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Base.BroadcastStyle(::Broadcast.Style{Tuple}, ::Broadcast.AbstractArrayStyle{0}) = Broadcast.Style{Tuple}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"indicates that Tuple \"wins\" over zero-dimensional arrays (the output container will be a tuple). It is worth noting that you do not need to (and should not) define both argument orders of this call; defining one is sufficient no matter what order the user supplies the arguments in.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"For AbstractArray types, defining a BroadcastStyle supersedes the fallback choice, Broadcast.DefaultArrayStyle. DefaultArrayStyle and the abstract supertype, AbstractArrayStyle, store the dimensionality as a type parameter to support specialized array types that have fixed dimensionality requirements.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"DefaultArrayStyle \"loses\" to any other AbstractArrayStyle that has been defined because of the following methods:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"BroadcastStyle(a::AbstractArrayStyle{Any}, ::DefaultArrayStyle) = a\nBroadcastStyle(a::AbstractArrayStyle{N}, ::DefaultArrayStyle{N}) where N = a\nBroadcastStyle(a::AbstractArrayStyle{M}, ::DefaultArrayStyle{N}) where {M,N} =\n    typeof(a)(_max(Val(M),Val(N)))","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"You do not need to write binary BroadcastStyle rules unless you want to establish precedence for two or more non-DefaultArrayStyle types.","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"If your array type does have fixed dimensionality requirements, then you should subtype AbstractArrayStyle. For example, the sparse array code has the following definitions:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"struct SparseVecStyle <: Broadcast.AbstractArrayStyle{1} end\nstruct SparseMatStyle <: Broadcast.AbstractArrayStyle{2} end\nBase.BroadcastStyle(::Type{<:SparseVector}) = SparseVecStyle()\nBase.BroadcastStyle(::Type{<:SparseMatrixCSC}) = SparseMatStyle()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"Whenever you subtype AbstractArrayStyle, you also need to define rules for combining dimensionalities, by creating a constructor for your style that takes a Val(N) argument. For example:","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"SparseVecStyle(::Val{0}) = SparseVecStyle()\nSparseVecStyle(::Val{1}) = SparseVecStyle()\nSparseVecStyle(::Val{2}) = SparseMatStyle()\nSparseVecStyle(::Val{N}) where N = Broadcast.DefaultArrayStyle{N}()","page":"Interfaces"},{"title":"Interfaces","location":"manual/interfaces.html","category":"page","text":"These rules indicate that the combination of a SparseVecStyle with 0- or 1-dimensional arrays yields another SparseVecStyle, that its combination with a 2-dimensional array yields a SparseMatStyle, and anything of higher dimensionality falls back to the dense arbitrary-dimensional framework. These rules allow broadcasting to keep the sparse representation for operations that result in one or two dimensional outputs, but produce an Array for any other dimensionality.","page":"Interfaces"},{"title":"Future","location":"stdlib/Future.html#Future","category":"section","text":"","page":"Future"},{"title":"Future","location":"stdlib/Future.html","category":"page","text":"The Future module implements future behavior of already existing functions, which will replace the current version in a future release of Julia.","page":"Future"},{"title":"Future","location":"stdlib/Future.html","category":"page","text":"Future.copy!\nFuture.randjump","page":"Future"},{"title":"Future.copy!","location":"stdlib/Future.html#Future.copy!","category":"function","text":"Future.copy!(dst, src) -> dst\n\nCopy src into dst.\n\ncompat: Julia 1.1\nThis function has moved to Base with Julia 1.1, consider using copy!(dst, src) instead. Future.copy! will be deprecated in the future.\n\n\n\n\n\n","page":"Future"},{"title":"Future.randjump","location":"stdlib/Future.html#Future.randjump","category":"function","text":"randjump(r::MersenneTwister, steps::Integer) -> MersenneTwister\n\nCreate an initialized MersenneTwister object, whose state is moved forward (without generating numbers) from r by steps steps. One such step corresponds to the generation of two Float64 numbers. For each different value of steps, a large polynomial has to be generated internally. One is already pre-computed for steps=big(10)^20.\n\n\n\n\n\n","page":"Future"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html#gdb-debugging-tips","category":"section","text":"","page":"gdb debugging tips"},{"title":"Displaying Julia variables","location":"devdocs/debuggingtips.html#Displaying-Julia-variables","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Within gdb, any jl_value_t* object obj can be displayed using","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) call jl_(obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The object will be displayed in the julia session, not in the gdb session. This is a useful way to discover the types and values of objects being manipulated by Julia's C code.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Similarly, if you're debugging some of Julia's internals (e.g., compiler.jl), you can print obj using","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"ccall(:jl_, Cvoid, (Any,), obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"This is a good way to circumvent problems that arise from the order in which julia's output streams are initialized.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia's flisp interpreter uses value_t objects; these can be displayed with call fl_print(fl_ctx, ios_stdout, obj).","page":"gdb debugging tips"},{"title":"Useful Julia variables for Inspecting","location":"devdocs/debuggingtips.html#Useful-Julia-variables-for-Inspecting","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"While the addresses of many variables, like singletons, can be useful to print for many failures, there are a number of additional variables (see julia.h for a complete list) that are even more useful.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(when in jl_apply_generic) mfunc and jl_uncompress_ast(mfunc->def, mfunc->code) :: for figuring out a bit about the call-stack\njl_lineno and jl_filename :: for figuring out what line in a test to go start debugging from (or figure out how far into a file has been parsed)\n$1 :: not really a variable, but still a useful shorthand for referring to the result of the last gdb command (such as print)\njl_options :: sometimes useful, since it lists all of the command line options that were successfully parsed\njl_uv_stderr :: because who doesn't like to be able to interact with stdio","page":"gdb debugging tips"},{"title":"Useful Julia functions for Inspecting those variables","location":"devdocs/debuggingtips.html#Useful-Julia-functions-for-Inspecting-those-variables","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"jl_gdblookup($rip) :: For looking up the current function and line. (use $eip on i686 platforms)\njlbacktrace() :: For dumping the current Julia backtrace stack to stderr. Only usable after record_backtrace() has been called.\njl_dump_llvm_value(Value*) :: For invoking Value->dump() in gdb, where it doesn't work natively. For example, f->linfo->functionObject, f->linfo->specFunctionObject, and to_function(f->linfo).\nType->dump() :: only works in lldb. Note: add something like ;1 to prevent lldb from printing its prompt over the output\njl_eval_string(\"expr\") :: for invoking side-effects to modify the current state or to lookup symbols\njl_typeof(jl_value_t*) :: for extracting the type tag of a Julia value (in gdb, call macro define jl_typeof jl_typeof first, or pick something short like ty for the first arg to define a shorthand)","page":"gdb debugging tips"},{"title":"Inserting breakpoints for inspection from gdb","location":"devdocs/debuggingtips.html#Inserting-breakpoints-for-inspection-from-gdb","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"In your gdb session, set a breakpoint in jl_breakpoint like so:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_breakpoint","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then within your Julia code, insert a call to jl_breakpoint by adding","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"ccall(:jl_breakpoint, Cvoid, (Any,), obj)","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"where obj can be any variable or tuple you want to be accessible in the breakpoint.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"It's particularly helpful to back up to the jl_apply frame, from which you can display the arguments to a function using, e.g.,","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) call jl_(args[0])","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Another useful frame is to_function(jl_method_instance_t *li, bool cstyle). The jl_method_instance_t* argument is a struct with a reference to the final AST sent into the compiler. However, the AST at this point will usually be compressed; to view the AST, call jl_uncompress_ast and then pass the result to jl_:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"#2  0x00007ffff7928bf7 in to_function (li=0x2812060, cstyle=false) at codegen.cpp:584\n584          abort();\n(gdb) p jl_(jl_uncompress_ast(li, li->ast))","page":"gdb debugging tips"},{"title":"Inserting breakpoints upon certain conditions","location":"devdocs/debuggingtips.html#Inserting-breakpoints-upon-certain-conditions","category":"section","text":"","page":"gdb debugging tips"},{"title":"Loading a particular file","location":"devdocs/debuggingtips.html#Loading-a-particular-file","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Let's say the file is sysimg.jl:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_load if strcmp(fname, \"sysimg.jl\")==0","page":"gdb debugging tips"},{"title":"Calling a particular method","location":"devdocs/debuggingtips.html#Calling-a-particular-method","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) break jl_apply_generic if strcmp((char*)(jl_symbol_name)(jl_gf_mtable(F)->name), \"method_to_break\")==0","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Since this function is used for every call, you will make everything 1000x slower if you do this.","page":"gdb debugging tips"},{"title":"Dealing with signals","location":"devdocs/debuggingtips.html#Dealing-with-signals","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia requires a few signal to function property. The profiler uses SIGUSR2 for sampling and the garbage collector uses SIGSEGV for threads synchronization. If you are debugging some code that uses the profiler or multiple threads, you may want to let the debugger ignore these signals since they can be triggered very often during normal operations. The command to do this in GDB is (replace SIGSEGV with SIGUSRS or other signals you want to ignore):","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) handle SIGSEGV noprint nostop pass","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The corresponding LLDB command is (after the process is started):","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(lldb) pro hand -p true -s false -n false SIGSEGV","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"If you are debugging a segfault with threaded code, you can set a breakpoint on jl_critical_error (sigdie_handler should also work on Linux and BSD) in order to only catch the actual segfault rather than the GC synchronization points.","page":"gdb debugging tips"},{"title":"Debugging during Julia's build process (bootstrap)","location":"devdocs/debuggingtips.html#Debugging-during-Julia's-build-process-(bootstrap)","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Errors that occur during make need special handling. Julia is built in two stages, constructing sys0 and sys.ji. To see what commands are running at the time of failure, use make VERBOSE=1.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"At the time of this writing, you can debug build errors during the sys0 phase from the base directory using:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys0 sysimg.jl","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"You might need to delete all the files in usr/lib/julia/ to get this to work.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"You can debug the sys.ji phase using:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"julia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys -J ../usr/lib/julia/sys0.ji sysimg.jl","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"By default, any errors will cause Julia to exit, even under gdb. To catch an error \"in the act\", set a breakpoint in jl_error (there are several other useful spots, for specific kinds of failures, including: jl_too_few_args, jl_too_many_args, and jl_throw).","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Once an error is caught, a useful technique is to walk up the stack and examine the function by inspecting the related call to jl_apply. To take a real-world example:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Breakpoint 1, jl_throw (e=0x7ffdf42de400) at task.c:802\n802 {\n(gdb) p jl_(e)\nErrorException(\"auto_unbox: unable to determine argument type\")\n$2 = void\n(gdb) bt 10\n#0  jl_throw (e=0x7ffdf42de400) at task.c:802\n#1  0x00007ffff65412fe in jl_error (str=0x7ffde56be000 <_j_str267> \"auto_unbox:\n   unable to determine argument type\")\n   at builtins.c:39\n#2  0x00007ffde56bd01a in julia_convert_16886 ()\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n...","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"The most recent jl_apply is at frame #3, so we can go back there and look at the AST for the function julia_convert_16886. This is the uniqued name for some method of convert. f in this frame is a jl_function_t*, so we can look at the type signature, if any, from the specTypes field:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) f 3\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n1281            return f->fptr((jl_value_t*)f, args, nargs);\n(gdb) p f->linfo->specTypes\n$4 = (jl_tupletype_t *) 0x7ffdf39b1030\n(gdb) p jl_( f->linfo->specTypes )\nTuple{Type{Float32}, Float64}           # <-- type signature for julia_convert_16886","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then, we can look at the AST for this function:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p jl_( jl_uncompress_ast(f->linfo, f->linfo->ast) )\nExpr(:lambda, Array{Any, 1}[:#s29, :x], Array{Any, 1}[Array{Any, 1}[], Array{Any, 1}[Array{Any, 1}[:#s29, :Any, 0], Array{Any, 1}[:x, :Any, 0]], Array{Any, 1}[], 0], Expr(:body,\nExpr(:line, 90, :float.jl)::Any,\nExpr(:return, Expr(:call, :box, :Float32, Expr(:call, :fptrunc, :Float32, :x)::Any)::Any)::Any)::Any)::Any","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Finally, and perhaps most usefully, we can force the function to be recompiled in order to step through the codegen process. To do this, clear the cached functionObject from the jl_lamdbda_info_t*:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p f->linfo->functionObject\n$8 = (void *) 0x1289d070\n(gdb) set f->linfo->functionObject = NULL","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then, set a breakpoint somewhere useful (e.g. emit_function, emit_expr, emit_call, etc.), and run codegen:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) p jl_compile(f)\n... # your breakpoint here","page":"gdb debugging tips"},{"title":"Debugging precompilation errors","location":"devdocs/debuggingtips.html#Debugging-precompilation-errors","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Module precompilation spawns a separate Julia process to precompile each module. Setting a breakpoint or catching failures in a precompile worker requires attaching a debugger to the worker. The easiest approach is to set the debugger watch for new process launches matching a given name. For example:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(gdb) attach -w -n julia-debug","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"or:","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"(lldb) process attach -w -n julia-debug","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Then run a script/command to start precompilation. As described earlier, use conditional breakpoints in the parent process to catch specific file-loading events and narrow the debugging window. (some operating systems may require alternative approaches, such as following each fork from the parent process)","page":"gdb debugging tips"},{"title":"Mozilla's Record and Replay Framework (rr)","location":"devdocs/debuggingtips.html#Mozilla's-Record-and-Replay-Framework-(rr)","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"Julia now works out of the box with rr, the lightweight recording and deterministic debugging framework from Mozilla. This allows you to replay the trace of an execution deterministically.  The replayed execution's address spaces, register contents, syscall data etc are exactly the same in every run.","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"A recent version of rr (3.1.0 or higher) is required.","page":"gdb debugging tips"},{"title":"Reproducing concurrency bugs with rr","location":"devdocs/debuggingtips.html#Reproducing-concurrency-bugs-with-rr","category":"section","text":"","page":"gdb debugging tips"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html","category":"page","text":"rr simulates a single-threaded machine by default. In order to debug concurrent code you can use rr record --chaos which will cause rr to simulate between one to eight cores, chosen randomly. You might therefore want to set JULIA_NUM_THREADS=8 and rerun your code under rr until you have caught your bug.","page":"gdb debugging tips"},{"title":"SIMD Support","location":"base/simd-types.html#SIMD-Support","category":"section","text":"","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"Type VecElement{T} is intended for building libraries of SIMD operations. Practical use of it requires using llvmcall. The type is defined as:","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"struct VecElement{T}\n    value::T\nend","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"It has a special compilation rule: a homogeneous tuple of VecElement{T} maps to an LLVM vector type when T is a primitive bits type.","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"At -O3, the compiler might automatically vectorize operations on such tuples. For example, the following program, when compiled with julia -O3 generates two SIMD addition instructions (addps) on x86 systems:","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"const m128 = NTuple{4,VecElement{Float32}}\n\nfunction add(a::m128, b::m128)\n    (VecElement(a[1].value+b[1].value),\n     VecElement(a[2].value+b[2].value),\n     VecElement(a[3].value+b[3].value),\n     VecElement(a[4].value+b[4].value))\nend\n\ntriple(c::m128) = add(add(c,c),c)\n\ncode_native(triple,(m128,))","page":"SIMD Support"},{"title":"SIMD Support","location":"base/simd-types.html","category":"page","text":"However, since the automatic vectorization cannot be relied upon, future use will mostly be via libraries that use llvmcall.","page":"SIMD Support"},{"title":"CRC32c","location":"stdlib/CRC32c.html#CRC32c","category":"section","text":"","page":"CRC32c"},{"title":"CRC32c","location":"stdlib/CRC32c.html","category":"page","text":"CRC32c.crc32c\nCRC32c.crc32c(::IO, ::Integer, ::UInt32)","page":"CRC32c"},{"title":"CRC32c.crc32c","location":"stdlib/CRC32c.html#CRC32c.crc32c","category":"function","text":"crc32c(data, crc::UInt32=0x00000000)\n\nCompute the CRC-32c checksum of the given data, which can be an Array{UInt8}, a contiguous subarray thereof, or a String.  Optionally, you can pass a starting crc integer to be mixed in with the checksum.  The crc parameter can be used to compute a checksum on data divided into chunks: performing crc32c(data2, crc32c(data1)) is equivalent to the checksum of [data1; data2]. (Technically, a little-endian checksum is computed.)\n\nThere is also a method crc32c(io, nb, crc) to checksum nb bytes from a stream io, or crc32c(io, crc) to checksum all the remaining bytes. Hence you can do open(crc32c, filename) to checksum an entire file, or crc32c(seekstart(buf)) to checksum an IOBuffer without calling take!.\n\nFor a String, note that the result is specific to the UTF-8 encoding (a different checksum would be obtained from a different Unicode encoding). To checksum an a::Array of some other bitstype, you can do crc32c(reinterpret(UInt8,a)), but note that the result may be endian-dependent.\n\n\n\n\n\n","page":"CRC32c"},{"title":"CRC32c.crc32c","location":"stdlib/CRC32c.html#CRC32c.crc32c-Tuple{IO, Integer, UInt32}","category":"method","text":"crc32c(io::IO, [nb::Integer,] crc::UInt32=0x00000000)\n\nRead up to nb bytes from io and return the CRC-32c checksum, optionally mixed with a starting crc integer.  If nb is not supplied, then io will be read until the end of the stream.\n\n\n\n\n\n","page":"CRC32c"},{"title":"Markdown","location":"stdlib/Markdown.html#markdown_stdlib","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This section describes Julia's markdown syntax, which is enabled by the Markdown standard library. The following Markdown elements are supported:","page":"Markdown"},{"title":"Inline elements","location":"stdlib/Markdown.html#Inline-elements","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Here \"inline\" refers to elements that can be found within blocks of text, i.e. paragraphs. These include the following elements.","page":"Markdown"},{"title":"Bold","location":"stdlib/Markdown.html#Bold","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround words with two asterisks, **, to display the enclosed text in boldface.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a **bold** word.","page":"Markdown"},{"title":"Italics","location":"stdlib/Markdown.html#Italics","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround words with one asterisk, *, to display the enclosed text in italics.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing an *italicized* word.","page":"Markdown"},{"title":"Literals","location":"stdlib/Markdown.html#Literals","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround text that should be displayed exactly as written with single backticks, ` .","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a `literal` word.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Literals should be used when writing text that refers to names of variables, functions, or other parts of a Julia program.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nTo include a backtick character within literal text use three backticks rather than one to enclose the text.A paragraph containing ``` `backtick` characters ```.By extension any odd number of backticks may be used to enclose a lesser number of backticks.","page":"Markdown"},{"title":"LaTeX","location":"stdlib/Markdown.html#\\LaTeX","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Surround text that should be displayed as mathematics using LaTeX syntax with double backticks, `` .","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing some ``\\LaTeX`` markup.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nAs with literals in the previous section, if literal backticks need to be written within double backticks use an even number greater than two. Note that if a single literal backtick needs to be included within LaTeX markup then two enclosing backticks is sufficient.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe \\ character should be escaped appropriately if the text is embedded in a Julia source code, for example, \"``\\\\LaTeX`` syntax in a docstring.\", since it is interpreted as a string literal. Alternatively, in order to avoid escaping, it is possible to use the raw string macro together with the @doc macro:@doc raw\"``\\LaTeX`` syntax in a docstring.\" functionname","page":"Markdown"},{"title":"Links","location":"stdlib/Markdown.html#Links","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Links to either external or internal targets can be written using the following syntax, where the text enclosed in square brackets, [ ], is the name of the link and the text enclosed in parentheses, ( ), is the URL.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a link to [Julia](http://www.julialang.org).","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"It's also possible to add cross-references to other documented functions/methods/variables within the Julia documentation itself. For example:","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"\"\"\"\n    tryparse(type, str; base)\n\nLike [`parse`](@ref), but returns either a value of the requested type,\nor [`nothing`](@ref) if the string does not contain a valid number.\n\"\"\"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This will create a link in the generated docs to the parse documentation (which has more information about what this function actually does), and to the nothing documentation. It's good to include cross references to mutating/non-mutating versions of a function, or to highlight a difference between two similar-seeming functions.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe above cross referencing is not a Markdown feature, and relies on Documenter.jl, which is used to build base Julia's documentation.","page":"Markdown"},{"title":"Footnote references","location":"stdlib/Markdown.html#Footnote-references","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Named and numbered footnote references can be written using the following syntax. A footnote name must be a single alphanumeric word containing no punctuation.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph containing a numbered footnote [^1] and a named one [^named].","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe text associated with a footnote can be written anywhere within the same page as the footnote reference. The syntax used to define the footnote text is discussed in the Footnotes section below.","page":"Markdown"},{"title":"Toplevel elements","location":"stdlib/Markdown.html#Toplevel-elements","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The following elements can be written either at the \"toplevel\" of a document or within another \"toplevel\" element.","page":"Markdown"},{"title":"Paragraphs","location":"stdlib/Markdown.html#Paragraphs","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A paragraph is a block of plain text, possibly containing any number of inline elements defined in the Inline elements section above, with one or more blank lines above and below it.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This is a paragraph.\n\nAnd this is *another* paragraph containing some emphasized text.\nA new line, but still part of the same paragraph.","page":"Markdown"},{"title":"Headers","location":"stdlib/Markdown.html#Headers","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A document can be split up into different sections using headers. Headers use the following syntax:","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"# Level One\n## Level Two\n### Level Three\n#### Level Four\n##### Level Five\n###### Level Six","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A header line can contain any inline syntax in the same way as a paragraph can.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"tip: Tip\nTry to avoid using too many levels of header within a single document. A heavily nested document may be indicative of a need to restructure it or split it into several pages covering separate topics.","page":"Markdown"},{"title":"Code blocks","location":"stdlib/Markdown.html#Code-blocks","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Source code can be displayed as a literal block using an indent of four spaces as shown in the following example.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This is a paragraph.\n\n    function func(x)\n        # ...\n    end\n\nAnother paragraph.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Additionally, code blocks can be enclosed using triple backticks with an optional \"language\" to specify how a block of code should be highlighted.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A code block without a \"language\":\n\n```\nfunction func(x)\n    # ...\nend\n```\n\nand another one with the \"language\" specified as `julia`:\n\n```julia\nfunction func(x)\n    # ...\nend\n```","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\n\"Fenced\" code blocks, as shown in the last example, should be preferred over indented code blocks since there is no way to specify what language an indented code block is written in.","page":"Markdown"},{"title":"Block quotes","location":"stdlib/Markdown.html#Block-quotes","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Text from external sources, such as quotations from books or websites, can be quoted using > characters prepended to each line of the quote as follows.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Here's a quote:\n\n> Julia is a high-level, high-performance dynamic programming language for\n> technical computing, with syntax that is familiar to users of other\n> technical computing environments.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Note that a single space must appear after the > character on each line. Quoted blocks may themselves contain other toplevel or inline elements.","page":"Markdown"},{"title":"Images","location":"stdlib/Markdown.html#Images","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The syntax for images is similar to the link syntax mentioned above. Prepending a ! character to a link will display an image from the specified URL rather than a link to it.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"![alternative text](link/to/image.png)","page":"Markdown"},{"title":"Lists","location":"stdlib/Markdown.html#Lists","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Unordered lists can be written by prepending each item in a list with either *, +, or -.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A list of items:\n\n  * item one\n  * item two\n  * item three","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Note the two spaces before each * and the single space after each one.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Lists can contain other nested toplevel elements such as lists, code blocks, or quoteblocks. A blank line should be left between each list item when including any toplevel elements within a list.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Another list:\n\n  * item one\n\n  * item two\n\n    ```\n    f(x) = x\n    ```\n\n  * And a sublist:\n\n      + sub-item one\n      + sub-item two","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nThe contents of each item in the list must line up with the first line of the item. In the above example the fenced code block must be indented by four spaces to align with the i in item two.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Ordered lists are written by replacing the \"bullet\" character, either *, +, or -, with a positive integer followed by either . or ).","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Two ordered lists:\n\n 1. item one\n 2. item two\n 3. item three\n\n 5) item five\n 6) item six\n 7) item seven","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"An ordered list may start from a number other than one, as in the second list of the above example, where it is numbered from five. As with unordered lists, ordered lists can contain nested toplevel elements.","page":"Markdown"},{"title":"Display equations","location":"stdlib/Markdown.html#Display-equations","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Large LaTeX equations that do not fit inline within a paragraph may be written as display equations using a fenced code block with the \"language\" math as in the example below.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"```math\nf(a) = \\frac{1}{2\\pi}\\int_{0}^{2\\pi} (\\alpha+R\\cos(\\theta))d\\theta\n```","page":"Markdown"},{"title":"Footnotes","location":"stdlib/Markdown.html#Footnotes","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"This syntax is paired with the inline syntax for Footnote references. Make sure to read that section as well.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Footnote text is defined using the following syntax, which is similar to footnote reference syntax, aside from the : character that is appended to the footnote label.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"[^1]: Numbered footnote text.\n\n[^note]:\n\n    Named footnote text containing several toplevel elements.\n\n      * item one\n      * item two\n      * item three\n\n    ```julia\n    function func(x)\n        # ...\n    end\n    ```","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nNo checks are done during parsing to make sure that all footnote references have matching footnotes.","page":"Markdown"},{"title":"Horizontal rules","location":"stdlib/Markdown.html#Horizontal-rules","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The equivalent of an <hr> HTML tag can be achieved using three hyphens (---). For example:","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Text above the line.\n\n---\n\nAnd text below the line.","page":"Markdown"},{"title":"Tables","location":"stdlib/Markdown.html#Tables","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Basic tables can be written using the syntax described below. Note that markdown tables have limited features and cannot contain nested toplevel elements unlike other elements discussed above – only inline elements are allowed. Tables must always contain a header row with column names. Cells cannot span multiple rows or columns of the table.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"| Column One | Column Two | Column Three |\n|:---------- | ---------- |:------------:|\n| Row `1`    | Column `2` |              |\n| *Row* 2    | **Row** 2  | Column ``3`` |","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"note: Note\nAs illustrated in the above example each column of | characters must be aligned vertically.A : character on either end of a column's header separator (the row containing - characters) specifies whether the row is left-aligned, right-aligned, or (when : appears on both ends) center-aligned. Providing no : characters will default to right-aligning the column.","page":"Markdown"},{"title":"Admonitions","location":"stdlib/Markdown.html#Admonitions","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Specially formatted blocks, known as admonitions, can be used to highlight particular remarks. They can be defined using the following !!! syntax:","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"!!! note\n\n    This is the content of the note.\n\n!!! warning \"Beware!\"\n\n    And this is another one.\n\n    This warning admonition has a custom title: `\"Beware!\"`.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"The first word after !!! declares the type of the admonition. There are standard admonition types that should produce special styling. Namely (in order of decreasing severity): danger, warning, info/note, and tip.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"You can also use your own admonition types, as long as the type name only contains lowercase Latin characters (a-z). For example, you could have a terminology block like this:","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"!!! terminology \"julia vs Julia\"\n\n    Strictly speaking, \"Julia\" refers to the language,\n    and \"julia\" to the standard implementation.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"However, unless the code rendering the Markdown special-cases that particular admonition type, it will get the default styling.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"A custom title for the box can be provided as a string (in double quotes) after the admonition type. If no title text is specified after the admonition type, then the type name will be used as the title (e.g. \"Note\" for the note admonition).","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Admonitions, like most other toplevel elements, can contain other toplevel elements (e.g. lists, images).","page":"Markdown"},{"title":"Markdown Syntax Extensions","location":"stdlib/Markdown.html#Markdown-Syntax-Extensions","category":"section","text":"","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"Julia's markdown supports interpolation in a very similar way to basic string literals, with the difference that it will store the object itself in the Markdown tree (as opposed to converting it to a string). When the Markdown content is rendered the usual show methods will be called, and these can be overridden as usual. This design allows the Markdown to be extended with arbitrarily complex features (such as references) without cluttering the basic syntax.","page":"Markdown"},{"title":"Markdown","location":"stdlib/Markdown.html","category":"page","text":"In principle, the Markdown parser itself can also be arbitrarily extended by packages, or an entirely custom flavour of Markdown can be used, but this should generally be unnecessary.","page":"Markdown"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Arrays","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"DocTestSetup = :(using SparseArrays, LinearAlgebra)","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Julia has support for sparse vectors and sparse matrices in the SparseArrays stdlib module. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays.","page":"Sparse Arrays"},{"title":"Compressed Sparse Column (CSC) Sparse Matrix Storage","location":"stdlib/SparseArrays.html#man-csc","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format. Julia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the type of the stored values, and Ti is the integer type for storing column pointers and row indices. The internal representation of SparseMatrixCSC is as follows:","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"struct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column j is in colptr[j]:(colptr[j+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nend","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The compressed sparse column storage makes it easy and quick to access the elements in the column of a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations such as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is because all elements of the sparse matrix that are beyond the point of insertion have to be moved one place over.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"All operations on sparse matrices are carefully implemented to exploit the CSC data structure for performance, and to avoid expensive operations.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"If you have data in CSC format from a different application or library, and wish to import it in Julia, make sure that you use 1-based indexing. The row indices in every column need to be sorted. If your SparseMatrixCSC object contains unsorted row indices, one quick way to sort them is by doing a double transpose.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"In some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These are accepted by functions in Base (but there is no guarantee that they will be preserved in mutating operations). Such explicitly stored zeros are treated as structural nonzeros by many routines. The nnz function returns the number of elements explicitly stored in the sparse data structure, including non-structural zeros. In order to count the exact number of numerical nonzeros, use count(!iszero, x), which inspects every stored element of a sparse matrix. dropzeros, and the in-place dropzeros!, can be used to remove stored zeros from the sparse matrix.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> A = sparse([1, 1, 2, 3], [1, 3, 2, 3], [0, 1, 2, 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 0  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n ⋅  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  ⋅","page":"Sparse Arrays"},{"title":"Sparse Vector Storage","location":"stdlib/SparseArrays.html#Sparse-Vector-Storage","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Sparse vectors are stored in a close analog to compressed sparse column format for sparse matrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv is the type of the stored values and Ti the integer type for the indices. The internal representation is as follows:","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"struct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nend","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"As for SparseMatrixCSC, the SparseVector type can also contain explicitly stored zeros. (See Sparse Matrix Storage.).","page":"Sparse Arrays"},{"title":"Sparse Vector and Matrix Constructors","location":"stdlib/SparseArrays.html#Sparse-Vector-and-Matrix-Constructors","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The simplest way to create a sparse array is to use a function equivalent to the zeros function that Julia provides for working with dense arrays. To produce a sparse array instead, you can use the same name with an sp prefix:","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> spzeros(3)\n3-element SparseVector{Float64, Int64} with 0 stored entries","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The sparse function is often a handy way to construct sparse arrays. For example, to construct a sparse matrix we can input a vector I of row indices, a vector J of column indices, and a vector V of stored values (this is also known as the COO (coordinate) format). sparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The equivalent sparse vector constructor is sparsevec, which takes the (row) index vector I and the vector V with the stored values and constructs a sparse vector R such that R[I[k]] = V[k].","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n⠀⠈⠀⡀⠀⠀⠀⠀⠠\n⠀⠀⠀⠀⠁⠀⠀⠀⠀\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64, Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The inverse of the sparse and sparsevec functions is findnz, which retrieves the inputs used to create the sparse array. findall(!iszero, x) returns the cartesian indices of non-zero entries in x (including stored entries equal to zero).","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findall(!iszero, S)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 4)\n CartesianIndex(4, 7)\n CartesianIndex(5, 9)\n CartesianIndex(3, 18)\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findall(!iszero, R)\n4-element Vector{Int64}:\n 1\n 3\n 4\n 5","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Another way to create a sparse array is to convert a dense array into a sparse array using the sparse function:","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> sparse(Matrix(1.0I, 5, 5))\n5×5 SparseMatrixCSC{Float64, Int64} with 5 stored entries:\n 1.0   ⋅    ⋅    ⋅    ⋅\n  ⋅   1.0   ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅   1.0   ⋅\n  ⋅    ⋅    ⋅    ⋅   1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"You can go in the other direction using the Array constructor. The issparse function can be used to query if a matrix is sparse.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"julia> issparse(spzeros(5))\ntrue","page":"Sparse Arrays"},{"title":"Sparse matrix operations","location":"stdlib/SparseArrays.html#Sparse-matrix-operations","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of, assignment into, and concatenation of sparse matrices work in the same way as dense matrices. Indexing operations, especially assignment, are expensive, when carried out one element at a time. In many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz, manipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct the sparse matrix.","page":"Sparse Arrays"},{"title":"Correspondence of dense and sparse methods","location":"stdlib/SparseArrays.html#Correspondence-of-dense-and-sparse-methods","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"The following table gives a correspondence between built-in methods on sparse matrices and their corresponding methods on dense matrix types. In general, methods that generate sparse matrices differ from their dense counterparts in that the resulting matrix follows the same sparsity pattern as a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix element has a probability d of being non-zero.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Details can be found in the Sparse Vectors and Matrices section of the standard library reference.","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"Sparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nsparse(I,n,n) Matrix(I,n,n) Creates a n-by-n identity matrix.\nsparse(A) Array(S) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(rng,m,n,d) randn(rng,m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements generated with the rng random number generator","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html#stdlib-sparse-arrays","category":"section","text":"","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"SparseArrays.AbstractSparseArray\nSparseArrays.AbstractSparseVector\nSparseArrays.AbstractSparseMatrix\nSparseArrays.SparseVector\nSparseArrays.SparseMatrixCSC\nSparseArrays.sparse\nSparseArrays.sparsevec\nSparseArrays.issparse\nSparseArrays.nnz\nSparseArrays.findnz\nSparseArrays.spzeros\nSparseArrays.spdiagm\nSparseArrays.blockdiag\nSparseArrays.sprand\nSparseArrays.sprandn\nSparseArrays.nonzeros\nSparseArrays.rowvals\nSparseArrays.nzrange\nSparseArrays.droptol!\nSparseArrays.dropzeros!\nSparseArrays.dropzeros\nSparseArrays.permute\npermute!{Tv, Ti, Tp <: Integer, Tq <: Integer}(::SparseMatrixCSC{Tv,Ti}, ::SparseMatrixCSC{Tv,Ti}, ::AbstractArray{Tp,1}, ::AbstractArray{Tq,1})","page":"Sparse Arrays"},{"title":"SparseArrays.AbstractSparseArray","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseArray","category":"type","text":"AbstractSparseArray{Tv,Ti,N}\n\nSupertype for N-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. SparseMatrixCSC, SparseVector and SuiteSparse.CHOLMOD.Sparse are subtypes of this.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.AbstractSparseVector","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseVector","category":"type","text":"AbstractSparseVector{Tv,Ti}\n\nSupertype for one-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,1}.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.AbstractSparseMatrix","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseMatrix","category":"type","text":"AbstractSparseMatrix{Tv,Ti}\n\nSupertype for two-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,2}.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.SparseVector","location":"stdlib/SparseArrays.html#SparseArrays.SparseVector","category":"type","text":"SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n\nVector type for storing sparse vectors.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.SparseMatrixCSC","location":"stdlib/SparseArrays.html#SparseArrays.SparseMatrixCSC","category":"type","text":"SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n\nMatrix type for storing sparse matrices in the Compressed Sparse Column format. The standard way of constructing SparseMatrixCSC is through the sparse function. See also spzeros, spdiagm and sprand.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.sparse","location":"stdlib/SparseArrays.html#SparseArrays.sparse","category":"function","text":"sparse(A)\n\nConvert an AbstractMatrix A into a sparse matrix.\n\nExamples\n\njulia> A = Matrix(1.0I, 3, 3)\n3×3 Matrix{Float64}:\n 1.0  0.0  0.0\n 0.0  1.0  0.0\n 0.0  0.0  1.0\n\njulia> sparse(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   1.0   ⋅\n  ⋅    ⋅   1.0\n\n\n\n\n\nsparse(I, J, V,[ m, n, combine])\n\nCreate a sparse matrix S of dimensions m x n such that S[I[k], J[k]] = V[k]. The combine function is used to combine duplicates. If m and n are not specified, they are set to maximum(I) and maximum(J) respectively. If the combine function is not supplied, combine defaults to + unless the elements of V are Booleans in which case combine defaults to |. All elements of I must satisfy 1 <= I[k] <= m, and all elements of J must satisfy 1 <= J[k] <= n. Numerical zeros in (I, J, V) are retained as structural nonzeros; to drop numerical zeros, use dropzeros!.\n\nFor additional documentation and an expert driver, see SparseArrays.sparse!.\n\nExamples\n\njulia> Is = [1; 2; 3];\n\njulia> Js = [1; 2; 3];\n\njulia> Vs = [1; 2; 3];\n\njulia> sparse(Is, Js, Vs)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.sparsevec","location":"stdlib/SparseArrays.html#SparseArrays.sparsevec","category":"function","text":"sparsevec(I, V, [m, combine])\n\nCreate a sparse vector S of length m such that S[I[k]] = V[k]. Duplicates are combined using the combine function, which defaults to + if no combine argument is provided, unless the elements of V are Booleans in which case combine defaults to |.\n\nExamples\n\njulia> II = [1, 3, 3, 5]; V = [0.1, 0.2, 0.3, 0.2];\n\njulia> sparsevec(II, V)\n5-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  0.5\n  [5]  =  0.2\n\njulia> sparsevec(II, V, 8, -)\n8-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  -0.1\n  [5]  =  0.2\n\njulia> sparsevec([1, 3, 1, 2, 2], [true, true, false, false, false])\n3-element SparseVector{Bool, Int64} with 3 stored entries:\n  [1]  =  1\n  [2]  =  0\n  [3]  =  1\n\n\n\n\n\nsparsevec(d::Dict, [m])\n\nCreate a sparse vector of length m where the nonzero indices are keys from the dictionary, and the nonzero values are the values from the dictionary.\n\nExamples\n\njulia> sparsevec(Dict(1 => 3, 2 => 2))\n2-element SparseVector{Int64, Int64} with 2 stored entries:\n  [1]  =  3\n  [2]  =  2\n\n\n\n\n\nsparsevec(A)\n\nConvert a vector A into a sparse vector of length m.\n\nExamples\n\njulia> sparsevec([1.0, 2.0, 0.0, 0.0, 3.0, 0.0])\n6-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  2.0\n  [5]  =  3.0\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.issparse","location":"stdlib/SparseArrays.html#SparseArrays.issparse","category":"function","text":"issparse(S)\n\nReturns true if S is sparse, and false otherwise.\n\nExamples\n\njulia> sv = sparsevec([1, 4], [2.3, 2.2], 10)\n10-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1 ]  =  2.3\n  [4 ]  =  2.2\n\njulia> issparse(sv)\ntrue\n\njulia> issparse(Array(sv))\nfalse\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.nnz","location":"stdlib/SparseArrays.html#SparseArrays.nnz","category":"function","text":"nnz(A)\n\nReturns the number of stored (filled) elements in a sparse array.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nnz(A)\n3\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.findnz","location":"stdlib/SparseArrays.html#SparseArrays.findnz","category":"function","text":"findnz(A::SparseMatrixCSC)\n\nReturn a tuple (I, J, V) where I and J are the row and column indices of the stored (\"structurally non-zero\") values in sparse matrix A, and V is a vector of the values.\n\nExamples\n\njulia> A = sparse([1 2 0; 0 0 3; 0 4 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 1  2  ⋅\n ⋅  ⋅  3\n ⋅  4  ⋅\n\njulia> findnz(A)\n([1, 1, 3, 2], [1, 2, 2, 3], [1, 2, 4, 3])\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.spzeros","location":"stdlib/SparseArrays.html#SparseArrays.spzeros","category":"function","text":"spzeros([type,]m[,n])\n\nCreate a sparse vector of length m or sparse matrix of size m x n. This sparse array will not contain any nonzero values. No storage will be allocated for nonzero values during construction. The type defaults to Float64 if not specified.\n\nExamples\n\njulia> spzeros(3, 3)\n3×3 SparseMatrixCSC{Float64, Int64} with 0 stored entries:\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n\njulia> spzeros(Float32, 4)\n4-element SparseVector{Float32, Int64} with 0 stored entries\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.spdiagm","location":"stdlib/SparseArrays.html#SparseArrays.spdiagm","category":"function","text":"spdiagm(kv::Pair{<:Integer,<:AbstractVector}...)\nspdiagm(m::Integer, n::Integer, kv::Pair{<:Integer,<:AbstractVector}...)\n\nConstruct a sparse diagonal matrix from Pairs of vectors and diagonals. Each vector kv.second will be placed on the kv.first diagonal.  By default, the matrix is square and its size is inferred from kv, but a non-square size m×n (padded with zeros as needed) can be specified by passing m,n as the first arguments.\n\nExamples\n\njulia> spdiagm(-1 => [1,2,3,4], 1 => [4,3,2,1])\n5×5 SparseMatrixCSC{Int64, Int64} with 8 stored entries:\n ⋅  4  ⋅  ⋅  ⋅\n 1  ⋅  3  ⋅  ⋅\n ⋅  2  ⋅  2  ⋅\n ⋅  ⋅  3  ⋅  1\n ⋅  ⋅  ⋅  4  ⋅\n\n\n\n\n\nspdiagm(v::AbstractVector)\nspdiagm(m::Integer, n::Integer, v::AbstractVector)\n\nConstruct a sparse matrix with elements of the vector as diagonal elements. By default (no given m and n), the matrix is square and its size is given by length(v), but a non-square size m×n can be specified by passing m and n as the first arguments.\n\ncompat: Julia 1.6\nThese functions require at least Julia 1.6.\n\nExamples\n\njulia> spdiagm([1,2,3])\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\njulia> spdiagm(sparse([1,0,3]))\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n 1  ⋅  ⋅\n ⋅  ⋅  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.blockdiag","location":"stdlib/SparseArrays.html#SparseArrays.blockdiag","category":"function","text":"blockdiag(A...)\n\nConcatenate matrices block-diagonally. Currently only implemented for sparse matrices.\n\nExamples\n\njulia> blockdiag(sparse(2I, 3, 3), sparse(4I, 2, 2))\n5×5 SparseMatrixCSC{Int64, Int64} with 5 stored entries:\n 2  ⋅  ⋅  ⋅  ⋅\n ⋅  2  ⋅  ⋅  ⋅\n ⋅  ⋅  2  ⋅  ⋅\n ⋅  ⋅  ⋅  4  ⋅\n ⋅  ⋅  ⋅  ⋅  4\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.sprand","location":"stdlib/SparseArrays.html#SparseArrays.sprand","category":"function","text":"sprand([rng],[type],m,[n],p::AbstractFloat,[rfn])\n\nCreate a random length m sparse vector or m by n sparse matrix, in which the probability of any element being nonzero is independently given by p (and hence the mean density of nonzeros is also exactly p). Nonzero values are sampled from the distribution specified by rfn and have the type type. The uniform distribution is used in case rfn is not specified. The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> sprand(Bool, 2, 2, 0.5)\n2×2 SparseMatrixCSC{Bool, Int64} with 1 stored entry:\n ⋅  ⋅\n ⋅  1\n\njulia> sprand(Float64, 3, 0.75)\n3-element SparseVector{Float64, Int64} with 1 stored entry:\n  [3]  =  0.298614\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.sprandn","location":"stdlib/SparseArrays.html#SparseArrays.sprandn","category":"function","text":"sprandn([rng][,Type],m[,n],p::AbstractFloat)\n\nCreate a random sparse vector of length m or sparse matrix of size m by n with the specified (independent) probability p of any entry being nonzero, where nonzero values are sampled from the normal distribution. The optional rng argument specifies a random number generator, see Random Numbers.\n\ncompat: Julia 1.1\nSpecifying the output element type Type requires at least Julia 1.1.\n\nExamples\n\njulia> sprandn(2, 2, 0.75)\n2×2 SparseMatrixCSC{Float64, Int64} with 2 stored entries:\n  ⋅   0.586617\n  ⋅   0.297336\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.nonzeros","location":"stdlib/SparseArrays.html#SparseArrays.nonzeros","category":"function","text":"nonzeros(A)\n\nReturn a vector of the structural nonzero values in sparse array A. This includes zeros that are explicitly stored in the sparse array. The returned vector points directly to the internal nonzero storage of A, and any modifications to the returned vector will mutate A as well. See rowvals and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nonzeros(A)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.rowvals","location":"stdlib/SparseArrays.html#SparseArrays.rowvals","category":"function","text":"rowvals(A::AbstractSparseMatrixCSC)\n\nReturn a vector of the row indices of A. Any modifications to the returned vector will mutate A as well. Providing access to how the row indices are stored internally can be useful in conjunction with iterating over structural nonzero values. See also nonzeros and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> rowvals(A)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.nzrange","location":"stdlib/SparseArrays.html#SparseArrays.nzrange","category":"function","text":"nzrange(A::AbstractSparseMatrixCSC, col::Integer)\n\nReturn the range of indices to the structural nonzero values of a sparse matrix column. In conjunction with nonzeros and rowvals, this allows for convenient iterating over a sparse matrix :\n\nA = sparse(I,J,V)\nrows = rowvals(A)\nvals = nonzeros(A)\nm, n = size(A)\nfor j = 1:n\n   for i in nzrange(A, j)\n      row = rows[i]\n      val = vals[i]\n      # perform sparse wizardry...\n   end\nend\n\n\n\n\n\nnzrange(x::SparseVectorUnion, col)\n\nGive the range of indices to the structural nonzero values of a sparse vector. The column index col is ignored (assumed to be 1).\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.droptol!","location":"stdlib/SparseArrays.html#SparseArrays.droptol!","category":"function","text":"droptol!(A::AbstractSparseMatrixCSC, tol)\n\nRemoves stored values from A whose absolute value is less than or equal to tol.\n\n\n\n\n\ndroptol!(x::SparseVector, tol)\n\nRemoves stored values from x whose absolute value is less than or equal to tol.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.dropzeros!","location":"stdlib/SparseArrays.html#SparseArrays.dropzeros!","category":"function","text":"dropzeros!(A::AbstractSparseMatrixCSC;)\n\nRemoves stored numerical zeros from A.\n\nFor an out-of-place version, see dropzeros. For algorithmic information, see fkeep!.\n\n\n\n\n\ndropzeros!(x::SparseVector)\n\nRemoves stored numerical zeros from x.\n\nFor an out-of-place version, see dropzeros. For algorithmic information, see fkeep!.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.dropzeros","location":"stdlib/SparseArrays.html#SparseArrays.dropzeros","category":"function","text":"dropzeros(A::AbstractSparseMatrixCSC;)\n\nGenerates a copy of A and removes stored numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparse([1, 2, 3], [1, 2, 3], [1.0, 0.0, 1.0])\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   0.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 2 stored entries:\n 1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0\n\n\n\n\n\ndropzeros(x::SparseVector)\n\nGenerates a copy of x and removes numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparsevec([1, 2, 3], [1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  0.0\n  [3]  =  1.0\n\njulia> dropzeros(A)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"SparseArrays.permute","location":"stdlib/SparseArrays.html#SparseArrays.permute","category":"function","text":"permute(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n        q::AbstractVector{<:Integer}) where {Tv,Ti}\n\nBilaterally permute A, returning PAQ (A[p,q]). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nFor expert drivers and additional information, see permute!.\n\nExamples\n\njulia> A = spdiagm(0 => [1, 2, 3, 4], 1 => [5, 6, 7])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n 1  5  ⋅  ⋅\n ⋅  2  6  ⋅\n ⋅  ⋅  3  7\n ⋅  ⋅  ⋅  4\n\njulia> permute(A, [4, 3, 2, 1], [1, 2, 3, 4])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  ⋅  4\n ⋅  ⋅  3  7\n ⋅  2  6  ⋅\n 1  5  ⋅  ⋅\n\njulia> permute(A, [1, 2, 3, 4], [4, 3, 2, 1])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  5  1\n ⋅  6  2  ⋅\n 7  3  ⋅  ⋅\n 4  ⋅  ⋅  ⋅\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"Base.permute!","location":"stdlib/SparseArrays.html#Base.permute!-Union{Tuple{Tq}, Tuple{Tp}, Tuple{Ti}, Tuple{Tv}, Tuple{SparseMatrixCSC{Tv, Ti}, SparseMatrixCSC{Tv, Ti}, AbstractVector{Tp}, AbstractVector{Tq}}} where {Tv, Ti, Tp<:Integer, Tq<:Integer}","category":"method","text":"permute!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti},\n         p::AbstractVector{<:Integer}, q::AbstractVector{<:Integer},\n         [C::AbstractSparseMatrixCSC{Tv,Ti}]) where {Tv,Ti}\n\nBilaterally permute A, storing result PAQ (A[p,q]) in X. Stores intermediate result (AQ)^T (transpose(A[:,q])) in optional argument C if present. Requires that none of X, A, and, if present, C alias each other; to store result PAQ back into A, use the following method lacking X:\n\npermute!(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n         q::AbstractVector{<:Integer}[, C::AbstractSparseMatrixCSC{Tv,Ti},\n         [workcolptr::Vector{Ti}]]) where {Tv,Ti}\n\nX's dimensions must match those of A (size(X, 1) == size(A, 1) and size(X, 2) == size(A, 2)), and X must have enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nC's dimensions must match those of transpose(A) (size(C, 1) == size(A, 2) and size(C, 2) == size(A, 1)), and C must have enough storage to accommodate all allocated entries in A (length(rowvals(C)) >= nnz(A) and length(nonzeros(C)) >= nnz(A)).\n\nFor additional (algorithmic) information, and for versions of these methods that forgo argument checking, see (unexported) parent methods unchecked_noalias_permute! and unchecked_aliasing_permute!.\n\nSee also: permute.\n\n\n\n\n\n","page":"Sparse Arrays"},{"title":"Sparse Arrays","location":"stdlib/SparseArrays.html","category":"page","text":"DocTestSetup = nothing","page":"Sparse Arrays"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"EditURL = \"https://github.com/JuliaLang/julia/blob/master/NEWS.md\"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html#Julia-v1.6-Release-Notes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"New language features","location":"NEWS.html#New-language-features","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Types written with where syntax can now be used to define constructors, e.g. (Foo{T} where T)(x) = ....\n<-- and <--> are now available as infix operators, with the same precedence and associativity as other arrow-like operators (#36666).\nCompilation and type inference can now be enabled or disabled at the module level using the experimental macro Base.Experimental.@compiler_options (#37041).\nThe library name passed to ccall or @ccall can now be an expression involving global variables and function calls. The expression will be evaluated the first time the ccall executes (#36458).\nꜛ (U+A71B), ꜜ (U+A71C) and ꜝ (U+A71D) can now also be used as operator suffixes. They can be tab-completed from \\^uparrow, \\^downarrow and \\^! in the REPL (#37542).\nStandalone \"dotted\" operators now get parsed as Expr(:., :op), which gets lowered to Base.BroadcastFunction(op). This means .op is functionally equivalent to (x...) -> (op).(x...), which can be useful for passing the broadcasted version of an operator to higher-order functions, for example map(.*, A, B) for an elementwise product of two arrays of arrays (#37583).\nThe syntax import A as B (plus import A: x as y, import A.x as y, and using A: x as y) can now be used to rename imported modules and identifiers (#1255).\nUnsigned literals (starting with 0x) which are too big to fit in a UInt128 object are now interpreted as BigInt (#23546).\nIt is now possible to use ... on the left-hand side of assignments for taking any number of items from the front of an iterable collection, while also collecting the rest, for example a, b... = [1, 2, 3]. This syntax is implemented using Base.rest, which can be overloaded to customize its behavior for different collection types (#37410).\nThe default behavior of observing @inbounds declarations is now an option via auto in --check-bounds=yes|no|auto (#41551)","page":"Julia v1.6 Release Notes"},{"title":"Language changes","location":"NEWS.html#Language-changes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"The postfix conjugate transpose operator ' now accepts Unicode modifiers as suffixes, so e.g. a'ᵀ is parsed as var\"'ᵀ\"(a), which can be defined by the user. a'ᵀ parsed as a' * ᵀ before, so this is a minor breaking change (#37247).\nMacros that return :quote expressions (e.g. via Expr(:quote, ...)) were previously able to work without escaping (esc(...)) their output when needed. This has been corrected, and now esc must be used in these macros as it is in other macros (#37540).\nThe --> operator now lowers to a :call expression, so it can be defined as a function like other operators. The dotted version .--> is now parsed as well. For backwards compatibility, --> still parses using its own expression head instead of :call.\nThe a[begin, k] syntax now calls firstindex(a, 1) rather than first(axes(a, 1)) (#35779), but the former now defaults to the latter for any a (#38742).\n⌿ (U+233F) and ¦ (U+00A6) are now infix operators with times-like and plus-like precedence, respectively. Previously they were parsed as identifier characters (#37973).","page":"Julia v1.6 Release Notes"},{"title":"Compiler/Runtime improvements","location":"NEWS.html#Compiler/Runtime-improvements","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"All platforms can now use @executable_path within jl_load_dynamic_library(). This allows executable-relative paths to be embedded within executables on all platforms, not just MacOS, which the syntax is borrowed from (#35627).\nConstant propagation now occurs through keyword arguments (#35976).\nThe precompilation cache is now created atomically (#36416). Invoking n Julia processes simultaneously may create n temporary caches.","page":"Julia v1.6 Release Notes"},{"title":"Command-line option changes","location":"NEWS.html#Command-line-option-changes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"There is no longer a concept of \"home project\": starting julia --project=dir is now exactly equivalent to starting julia and then doing pkg> activate $dir and julia --project is exactly equivalent to doing that where dir = Base.current_project(). In particular, this means that if you do pkg> activate after starting julia with the --project option (or with JULIA_PROJECT set) it will take you to the default active project, which is @v1.6 unless you have modified LOAD_PATH (#36434).","page":"Julia v1.6 Release Notes"},{"title":"Multi-threading changes","location":"NEWS.html#Multi-threading-changes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Locks now automatically inhibit finalizers from running, to avoid deadlock (#38487).\nNew function Base.Threads.foreach(f, channel::Channel) for multithreaded Channel consumption (#34543).\nThere is no longer a restriction on the number of threads (#36778).","page":"Julia v1.6 Release Notes"},{"title":"Build system changes","location":"NEWS.html#Build-system-changes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Windows Installer now has the option to 'Add Julia to Path'. To unselect this option from the commandline simply remove the tasks you do not want to be installed: e.g. ./julia-installer.exe /TASKS=\"desktopicon,startmenu,addtopath\", adds a desktop icon, a startmenu group icon, and adds Julia to system PATH.","page":"Julia v1.6 Release Notes"},{"title":"New library functions","location":"NEWS.html#New-library-functions","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"New function Base.kron! and corresponding overloads for various matrix types for performing Kronecker product in-place (#31069).\nNew function Base.readeach(io, T) for iteratively performing read(io, T) (#36150).\nIterators.map is added. It provides another syntax Iterators.map(f, iterators...) for writing (f(args...) for args in zip(iterators...)), i.e. a lazy map (#34352).\nNew function sincospi for simultaneously computing sinpi(x) and cospi(x) more efficiently (#35816).\nNew function cispi(x) for more accurately computing cis(pi * x) (#38449).\nNew function addenv for adding environment mappings into a Cmd object, returning the new Cmd object.\nNew function insorted for determining whether an element is in a sorted collection or not (#37490).\nNew function Base.rest for taking the rest of a collection, starting from a specific iteration state, in a generic way (#37410).","page":"Julia v1.6 Release Notes"},{"title":"New library features","location":"NEWS.html#New-library-features","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"The redirect_* functions now accept devnull to discard all output redirected to it, and as an empty input (#36146).\nThe redirect_* functions can now be called on IOContext objects (#36688).\nfindfirst, findnext, findlast, and findall now support AbstractVector{<:Union{Int8,UInt8}} (pattern, array) arguments (#37283).\nNew constructor NamedTuple(iterator) that constructs a named tuple from a key-value pair iterator.\nA new reinterpret(reshape, T, a::AbstractArray{S}) reinterprets a to have eltype T while potentially inserting or consuming the first dimension depending on the ratio of sizeof(T) and sizeof(S).\nNew append!(vector, collections...) and prepend!(vector, collections...) methods accept multiple collections to be appended or prepended (#36227).\nkeys(io::IO) has been added, which returns all keys of io if io is an IOContext and an empty Base.KeySet otherwise (#37753).\ncount now accepts an optional init argument to control the accumulation type (#37461).\nNew method occursin(haystack) that returns a function that checks whether its argument occurs in haystack (#38475).\nNew methods ∉(collection), ∋(item), and ∌(item) returning corresponding containment-testing functions (#38475).\nThe nextprod function now accepts tuples and other array types for its first argument (#35791).\nThe reverse(A; dims) function for multidimensional A can now reverse multiple dimensions at once by passing a tuple for dims, and defaults to reversing all dimensions; there is also a multidimensional in-place reverse!(A; dims) (#37367).\nThe function isapprox(x,y) now accepts the norm keyword argument also for numeric (i.e., non-array) arguments x and y (#35883).\nispow2(x) now supports non-Integer arguments x (#37635).\nview, @view, and @views now work on AbstractStrings, returning a SubString when appropriate (#35879).\nAll AbstractUnitRange{<:Integer}s now work with SubString, view, @view and @views on strings (#35879).\nsum, prod, maximum, and minimum now support init keyword argument (#36188, #35839).\nunique(f, itr; seen=Set{T}()) now allows you to declare the container type used for keeping track of values returned by f on elements of itr (#36280).\nfirst and last functions now accept an integer as second argument to get that many leading or trailing elements of any iterable (#34868).\nCartesianIndices now supports step different from 1. It can also be constructed from three CartesianIndexes I, S, J using I:S:J. step for CartesianIndices now returns a CartesianIndex (#37829).\nRegexMatch objects can now be probed for whether a named capture group exists within it through haskey() (#36717).\nFor consistency haskey(r::RegexMatch, i::Integer) has also been added and returns if the capture group for i exists (#37300).","page":"Julia v1.6 Release Notes"},{"title":"Standard library changes","location":"NEWS.html#Standard-library-changes","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"A new standard library TOML has been added for parsing and printing TOML files (#37034).\nA new standard library Downloads has been added, which replaces the old Base.download function with Downloads.download, providing cross-platform, multi-protocol, in-process download functionality implemented with libcurl (#37340).\nLibdl has been moved to Base.Libc.Libdl, however it is still accessible as an stdlib (#35628).\nTo download artifacts lazily, LazyArtifacts now must be explicitly listed as a dependency, to avoid needing the support machinery to be available when it is not commonly needed (#37844).\nIt is no longer possible to create a LinRange, StepRange, or StepRangeLen with a <: Integer eltype but non-integer step (#32439).\nintersect on CartesianIndices now returns CartesianIndices instead of Vector{<:CartesianIndex} (#36643).\npush!(c::Channel, v) now returns channel c. Previously, it returned the pushed value v (#34202).\nThe composition operator ∘ now returns a Base.ComposedFunction instead of an anonymous function (#37517).\nLogging (such as @warn) no longer catches exceptions in the logger itself (#36600).\n@time now reports if the time presented included any compilation time, which is shown as a percentage (#37678).\n@varinfo can now report non-exported objects within modules, look recursively into submodules, and return a sorted results table (#38042).\n@testset now supports the option verbose to show the test result summary of the children even if they all pass (#33755).\nIn LinearIndices(::Tuple) and CartesianIndices(::Tuple), integers (as opposed to ranges of integers) in the argument tuple now consistently describe 1-based ranges, e.g, CartesianIndices((3, 1:3)) is equivalent to CartesianIndices((1:3, 1:3)). This is how tuples of integers have always been documented to work, but a bug had caused erroneous behaviors with heterogeneous tuples containing both integers and ranges (#37829, #37928).","page":"Julia v1.6 Release Notes"},{"title":"Package Manager","location":"NEWS.html#Package-Manager","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"pkg> precompile is now parallelized through depth-first precompilation of dependencies. Errors will only throw for direct dependencies listed in the Project.toml.\npkg> precompile is now automatically triggered whenever Pkg changes the active manifest. Auto-precompilation will remember if a package has errored within the given environment and will not retry until it changes. Auto-precompilation can be gracefully interrupted with a ctrl-c and disabled by setting the environment variable JULIA_PKG_PRECOMPILE_AUTO=0.\nThe Pkg.BinaryPlatforms module has been moved into Base as Base.BinaryPlatforms and heavily reworked. Applications that want to be compatible with the old API should continue to import Pkg.BinaryPlatforms, however new users should use Base.BinaryPlatforms directly (#37320).\nThe Pkg.Artifacts module has been imported as a separate standard library.  It is still available as Pkg.Artifacts, however starting from Julia v1.6+, packages may import simply Artifacts without importing all of Pkg alongside (#37320).","page":"Julia v1.6 Release Notes"},{"title":"LinearAlgebra","location":"NEWS.html#LinearAlgebra","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"New method LinearAlgebra.issuccess(::CholeskyPivoted) for checking whether pivoted Cholesky factorization was successful (#36002).\nUniformScaling can now be indexed into using ranges to return dense matrices and vectors (#24359).\nNew function LinearAlgebra.BLAS.get_num_threads() for getting the number of BLAS threads (#36360).\n(+)(::UniformScaling) is now defined, making +I a valid unary operation (#36784).\nInstances of UniformScaling are no longer isequal to matrices. Previous behaviour violated the rule that isequal(x, y) implies hash(x) == hash(y).\nTransposing *Triangular matrices now returns matrices of the opposite triangular type, consistently with adjoint!(::*Triangular) and transpose!(::*Triangular). Packages containing methods with, e.g., Adjoint{<:Any,<:LowerTriangular{<:Any,<:OwnMatrixType}} should replace that by UpperTriangular{<:Any,<:Adjoint{<:Any,<:OwnMatrixType}} in the method signature (#38168).","page":"Julia v1.6 Release Notes"},{"title":"Markdown","location":"NEWS.html#Markdown","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Printf","location":"NEWS.html#Printf","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Complete overhaul of internal code to use the ryu float printing algorithms (from Julia 1.4); leads to consistent 2-5x performance improvements.\nNew Printf.tofloat function allowing custom float types to more easily integrate with Printf formatting by converting their type to Float16, Float32, Float64, or BigFloat.\nNew Printf.format\"...\" and Printf.Format(...) functions that allow creating Printf.Format objects that can be passed to Printf.format for easier dynamic printf formatting.\nPrintf.format(f::Printf.Format, args...) as a non-macro function that applies a printf format f to provided args.","page":"Julia v1.6 Release Notes"},{"title":"Random","location":"NEWS.html#Random","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"REPL","location":"NEWS.html#REPL","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"The AbstractMenu extension interface of REPL.TerminalMenus has been extensively overhauled. The new interface does not rely on global configuration variables, is more consistent in delegating printing of the navigation/selection markers, and provides improved support for dynamic menus.  These changes are compatible with the previous (deprecated) interface, so are non-breaking.\nThe new API offers several enhancements:\nMenus are configured in their constructors via keyword arguments.\nFor custom menu types, the new Config and MultiSelectConfig replace the global CONFIG Dict.\nrequest(menu; cursor=1) allows you to control the initial cursor position in the menu (defaults to first item).\nMultiSelectMenu allows you to pass a list of initially-selected items with the selected keyword argument.\nwriteLine was deprecated to writeline, and writeline methods are not expected to print the cursor indicator. The old writeLine continues to work, and any of its method extensions should print the cursor indicator as before.\nprintMenu has been deprecated to printmenu, and it both accepts a state input and returns a state output that controls the number of terminal lines erased when the menu is next refreshed. This plus related changes makes printmenu work properly when the number of menu items might change depending on user choices.\nnumoptions, returning the number of items in the menu, has been added as an alternative to implementing options.\nsuppress_output (primarily a testing option) has been added as a keyword argument to request, rather than a configuration option.\nTab completion now supports runs of consecutive sub/superscript characters, e.g. \\^(3) tab-completes to ⁽³⁾ (#38649).\nWindows REPL now supports 24-bit colors, by correctly interpreting virtual terminal escapes.","page":"Julia v1.6 Release Notes"},{"title":"SparseArrays","location":"NEWS.html#SparseArrays","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Display large sparse matrices with a Unicode \"spy\" plot of their nonzero patterns, and display small sparse matrices by an Matrix-like 2d layout of their contents (#33821).\nNew convenient spdiagm([m, n,] v::AbstractVector) methods which call spdiagm([m, n,] 0 => v), consistently with their dense diagm counterparts (#37684).","page":"Julia v1.6 Release Notes"},{"title":"Dates","location":"NEWS.html#Dates","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Quarter period is defined (#35519).\ncanonicalize can now take Period as an input (#37391).\nZero-valued FixedPeriods and OtherPeriods now compare equal, e.g., Year(0) == Day(0). The behavior of non-zero Periods is not changed (#37486).","page":"Julia v1.6 Release Notes"},{"title":"Statistics","location":"NEWS.html#Statistics","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Sockets","location":"NEWS.html#Sockets","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Distributed","location":"NEWS.html#Distributed","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Now supports invoking Windows workers via ssh (via new keyword argument shell=:wincmd in addprocs) (#30614).\nOther new keyword arguments in addprocs: ssh to specify the ssh client path, env to pass environment variables to workers, and cmdline_cookie to work around an ssh problem with Windows workers that run older (pre-ConPTY) versions of Windows, Julia or OpenSSH (#30614).","page":"Julia v1.6 Release Notes"},{"title":"UUIDs","location":"NEWS.html#UUIDs","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"Change uuid1 and uuid4 to use Random.RandomDevice() as default random number generator (#35872).\nAdded parse(::Type{UUID}, ::AbstractString) method.","page":"Julia v1.6 Release Notes"},{"title":"Mmap","location":"NEWS.html#Mmap","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"On Unix systems, the Mmap.madvise! function (along with OS-specific Mmap.MADV_* constants) has been added to give advice on handling of memory-mapped arrays (#37369).","page":"Julia v1.6 Release Notes"},{"title":"Deprecated or removed","location":"NEWS.html#Deprecated-or-removed","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Julia v1.6 Release Notes","location":"NEWS.html","category":"page","text":"The Base.download function has been deprecated (silently, by default) in favor of the new Downloads.download standard library function (#37340).\nThe Base.Grisu code has been officially removed (float printing was switched to the ryu algorithm code in 1.4). The code is available from JuliaAttic if needed.","page":"Julia v1.6 Release Notes"},{"title":"External dependencies","location":"NEWS.html#External-dependencies","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Tooling Improvements","location":"NEWS.html#Tooling-Improvements","category":"section","text":"","page":"Julia v1.6 Release Notes"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html#Integers-and-Floating-Point-Numbers","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Integers and floating-point values are the basic building blocks of arithmetic and computation. Built-in representations of such values are called numeric primitives, while representations of integers and floating-point numbers as immediate values in code are known as numeric literals. For example, 1 is an integer literal, while 1.0 is a floating-point literal; their binary in-memory representations as objects are numeric primitives.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Julia provides a broad range of primitive numeric types, and a full complement of arithmetic and bitwise operators as well as standard mathematical functions are defined over them. These map directly onto numeric types and operations that are natively supported on modern computers, thus allowing Julia to take full advantage of computational resources. Additionally, Julia provides software support for Arbitrary Precision Arithmetic, which can handle operations on numeric values that cannot be represented effectively in native hardware representations, but at the cost of relatively slower performance.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The following are Julia's primitive numeric types:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Integer types:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Type Signed? Number of bits Smallest value Largest value\nInt8 ✓ 8 -2^7 2^7 - 1\nUInt8  8 0 2^8 - 1\nInt16 ✓ 16 -2^15 2^15 - 1\nUInt16  16 0 2^16 - 1\nInt32 ✓ 32 -2^31 2^31 - 1\nUInt32  32 0 2^32 - 1\nInt64 ✓ 64 -2^63 2^63 - 1\nUInt64  64 0 2^64 - 1\nInt128 ✓ 128 -2^127 2^127 - 1\nUInt128  128 0 2^128 - 1\nBool N/A 8 false (0) true (1)","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Floating-point types:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Type Precision Number of bits\nFloat16 half 16\nFloat32 single 32\nFloat64 double 64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Additionally, full support for Complex and Rational Numbers is built on top of these primitive numeric types. All numeric types interoperate naturally without explicit casting, thanks to a flexible, user-extensible type promotion system.","page":"Integers and Floating-Point Numbers"},{"title":"Integers","location":"manual/integers-and-floating-point-numbers.html#Integers","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Literal integers are represented in the standard manner:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 1\n1\n\njulia> 1234\n1234","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The default type for an integer literal depends on whether the target system has a 32-bit architecture or a 64-bit architecture:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"# 32-bit system:\njulia> typeof(1)\nInt32\n\n# 64-bit system:\njulia> typeof(1)\nInt64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The Julia internal variable Sys.WORD_SIZE indicates whether the target system is 32-bit or 64-bit:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"# 32-bit system:\njulia> Sys.WORD_SIZE\n32\n\n# 64-bit system:\njulia> Sys.WORD_SIZE\n64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Julia also defines the types Int and UInt, which are aliases for the system's signed and unsigned native integer types respectively:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"# 32-bit system:\njulia> Int\nInt32\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> Int\nInt64\njulia> UInt\nUInt64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Larger integer literals that cannot be represented using only 32 bits but can be represented in 64 bits always create 64-bit integers, regardless of the system type:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"# 32-bit or 64-bit system:\njulia> typeof(3000000000)\nInt64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Unsigned integers are input and output using the 0x prefix and hexadecimal (base 16) digits 0-9a-f (the capitalized digits A-F also work for input). The size of the unsigned value is determined by the number of hex digits used:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = 0x1\n0x01\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x123\n0x0123\n\njulia> typeof(x)\nUInt16\n\njulia> x = 0x1234567\n0x01234567\n\njulia> typeof(x)\nUInt32\n\njulia> x = 0x123456789abcdef\n0x0123456789abcdef\n\njulia> typeof(x)\nUInt64\n\njulia> x = 0x11112222333344445555666677778888\n0x11112222333344445555666677778888\n\njulia> typeof(x)\nUInt128","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"This behavior is based on the observation that when one uses unsigned hex literals for integer values, one typically is using them to represent a fixed numeric byte sequence, rather than just an integer value.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Binary and octal literals are also supported:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = 0b10\n0x02\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0o010\n0x08\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x00000000000000001111222233334444\n0x00000000000000001111222233334444\n\njulia> typeof(x)\nUInt128","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"As for hexadecimal literals, binary and octal literals produce unsigned integer types. The size of the binary data item is the minimal needed size, if the leading digit of the literal is not 0. In the case of leading zeros, the size is determined by the minimal needed size for a literal, which has the same length but leading digit 1. That allows the user to control the size. Values which cannot be stored in UInt128 cannot be written as such literals.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Binary, octal, and hexadecimal literals may be signed by a - immediately preceding the unsigned literal. They produce an unsigned integer of the same size as the unsigned literal would do, with the two's complement of the value:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> -0x2\n0xfe\n\njulia> -0x0002\n0xfffe","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The minimum and maximum representable values of primitive numeric types such as integers are given by the typemin and typemax functions:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> (typemin(Int32), typemax(Int32))\n(-2147483648, 2147483647)\n\njulia> for T in [Int8,Int16,Int32,Int64,Int128,UInt8,UInt16,UInt32,UInt64,UInt128]\n           println(\"$(lpad(T,7)): [$(typemin(T)),$(typemax(T))]\")\n       end\n   Int8: [-128,127]\n  Int16: [-32768,32767]\n  Int32: [-2147483648,2147483647]\n  Int64: [-9223372036854775808,9223372036854775807]\n Int128: [-170141183460469231731687303715884105728,170141183460469231731687303715884105727]\n  UInt8: [0,255]\n UInt16: [0,65535]\n UInt32: [0,4294967295]\n UInt64: [0,18446744073709551615]\nUInt128: [0,340282366920938463463374607431768211455]","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The values returned by typemin and typemax are always of the given argument type. (The above expression uses several features that have yet to be introduced, including for loops, Strings, and Interpolation, but should be easy enough to understand for users with some existing programming experience.)","page":"Integers and Floating-Point Numbers"},{"title":"Overflow behavior","location":"manual/integers-and-floating-point-numbers.html#Overflow-behavior","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"In Julia, exceeding the maximum representable value of a given type results in a wraparound behavior:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = typemax(Int64)\n9223372036854775807\n\njulia> x + 1\n-9223372036854775808\n\njulia> x + 1 == typemin(Int64)\ntrue","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Thus, arithmetic with Julia integers is actually a form of modular arithmetic. This reflects the characteristics of the underlying arithmetic of integers as implemented on modern computers. In applications where overflow is possible, explicit checking for wraparound produced by overflow is essential; otherwise, the BigInt type in Arbitrary Precision Arithmetic is recommended instead.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"An example of overflow behavior and how to potentially resolve it is as follows:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 10^19\n-8446744073709551616\n\njulia> big(10)^19\n10000000000000000000","page":"Integers and Floating-Point Numbers"},{"title":"Division errors","location":"manual/integers-and-floating-point-numbers.html#Division-errors","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Integer division (the div function) has two exceptional cases: dividing by zero, and dividing the lowest negative number (typemin) by -1. Both of these cases throw a DivideError. The remainder and modulus functions (rem and mod) throw a DivideError when their second argument is zero.","page":"Integers and Floating-Point Numbers"},{"title":"Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html#Floating-Point-Numbers","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Literal floating-point numbers are represented in the standard formats, using E-notation when necessary:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 1.0\n1.0\n\njulia> 1.\n1.0\n\njulia> 0.5\n0.5\n\njulia> .5\n0.5\n\njulia> -1.23\n-1.23\n\njulia> 1e10\n1.0e10\n\njulia> 2.5e-4\n0.00025","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The above results are all Float64 values. Literal Float32 values can be entered by writing an f in place of e:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = 0.5f0\n0.5f0\n\njulia> typeof(x)\nFloat32\n\njulia> 2.5f-4\n0.00025f0","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Values can be converted to Float32 easily:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = Float32(-1.5)\n-1.5f0\n\njulia> typeof(x)\nFloat32","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Hexadecimal floating-point literals are also valid, but only as Float64 values, with p preceding the base-2 exponent:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 0x1p0\n1.0\n\njulia> 0x1.8p3\n12.0\n\njulia> x = 0x.4p-1\n0.125\n\njulia> typeof(x)\nFloat64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Half-precision floating-point numbers are also supported (Float16), but they are implemented in software and use Float32 for calculations.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> sizeof(Float16(4.))\n2\n\njulia> 2*Float16(4.)\nFloat16(8.0)","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The underscore _ can be used as digit separator:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 10_000, 0.000_000_005, 0xdead_beef, 0b1011_0010\n(10000, 5.0e-9, 0xdeadbeef, 0xb2)","page":"Integers and Floating-Point Numbers"},{"title":"Floating-point zero","location":"manual/integers-and-floating-point-numbers.html#Floating-point-zero","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Floating-point numbers have two zeros, positive zero and negative zero. They are equal to each other but have different binary representations, as can be seen using the bitstring function:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 0.0 == -0.0\ntrue\n\njulia> bitstring(0.0)\n\"0000000000000000000000000000000000000000000000000000000000000000\"\n\njulia> bitstring(-0.0)\n\"1000000000000000000000000000000000000000000000000000000000000000\"","page":"Integers and Floating-Point Numbers"},{"title":"Special floating-point values","location":"manual/integers-and-floating-point-numbers.html#Special-floating-point-values","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"There are three specified standard floating-point values that do not correspond to any point on the real number line:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Float16 Float32 Float64 Name Description\nInf16 Inf32 Inf positive infinity a value greater than all finite floating-point values\n-Inf16 -Inf32 -Inf negative infinity a value less than all finite floating-point values\nNaN16 NaN32 NaN not a number a value not == to any floating-point value (including itself)","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"For further discussion of how these non-finite floating-point values are ordered with respect to each other and other floats, see Numeric Comparisons. By the IEEE 754 standard, these floating-point values are the results of certain arithmetic operations:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 1/Inf\n0.0\n\njulia> 1/0\nInf\n\njulia> -5/0\n-Inf\n\njulia> 0.000001/0\nInf\n\njulia> 0/0\nNaN\n\njulia> 500 + Inf\nInf\n\njulia> 500 - Inf\n-Inf\n\njulia> Inf + Inf\nInf\n\njulia> Inf - Inf\nNaN\n\njulia> Inf * Inf\nInf\n\njulia> Inf / Inf\nNaN\n\njulia> 0 * Inf\nNaN","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The typemin and typemax functions also apply to floating-point types:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> (typemin(Float16),typemax(Float16))\n(-Inf16, Inf16)\n\njulia> (typemin(Float32),typemax(Float32))\n(-Inf32, Inf32)\n\njulia> (typemin(Float64),typemax(Float64))\n(-Inf, Inf)","page":"Integers and Floating-Point Numbers"},{"title":"Machine epsilon","location":"manual/integers-and-floating-point-numbers.html#Machine-epsilon","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Most real numbers cannot be represented exactly with floating-point numbers, and so for many purposes it is important to know the distance between two adjacent representable floating-point numbers, which is often known as machine epsilon.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Julia provides eps, which gives the distance between 1.0 and the next larger representable floating-point value:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> eps(Float32)\n1.1920929f-7\n\njulia> eps(Float64)\n2.220446049250313e-16\n\njulia> eps() # same as eps(Float64)\n2.220446049250313e-16","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"These values are 2.0^-23 and 2.0^-52 as Float32 and Float64 values, respectively. The eps function can also take a floating-point value as an argument, and gives the absolute difference between that value and the next representable floating point value. That is, eps(x) yields a value of the same type as x such that x + eps(x) is the next representable floating-point value larger than x:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(1000.)\n1.1368683772161603e-13\n\njulia> eps(1e-27)\n1.793662034335766e-43\n\njulia> eps(0.0)\n5.0e-324","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The distance between two adjacent representable floating-point numbers is not constant, but is smaller for smaller values and larger for larger values. In other words, the representable floating-point numbers are densest in the real number line near zero, and grow sparser exponentially as one moves farther away from zero. By definition, eps(1.0) is the same as eps(Float64) since 1.0 is a 64-bit floating-point value.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Julia also provides the nextfloat and prevfloat functions which return the next largest or smallest representable floating-point number to the argument respectively:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = 1.25f0\n1.25f0\n\njulia> nextfloat(x)\n1.2500001f0\n\njulia> prevfloat(x)\n1.2499999f0\n\njulia> bitstring(prevfloat(x))\n\"00111111100111111111111111111111\"\n\njulia> bitstring(x)\n\"00111111101000000000000000000000\"\n\njulia> bitstring(nextfloat(x))\n\"00111111101000000000000000000001\"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"This example highlights the general principle that the adjacent representable floating-point numbers also have adjacent binary integer representations.","page":"Integers and Floating-Point Numbers"},{"title":"Rounding modes","location":"manual/integers-and-floating-point-numbers.html#Rounding-modes","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"If a number doesn't have an exact floating-point representation, it must be rounded to an appropriate representable value. However, the manner in which this rounding is done can be changed if required according to the rounding modes presented in the IEEE 754 standard.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The default mode used is always RoundNearest, which rounds to the nearest representable value, with ties rounded towards the nearest value with an even least significant bit.","page":"Integers and Floating-Point Numbers"},{"title":"Background and References","location":"manual/integers-and-floating-point-numbers.html#Background-and-References","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Floating-point arithmetic entails many subtleties which can be surprising to users who are unfamiliar with the low-level implementation details. However, these subtleties are described in detail in most books on scientific computation, and also in the following references:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The definitive guide to floating point arithmetic is the IEEE 754-2008 Standard; however, it is not available for free online.\nFor a brief but lucid presentation of how floating-point numbers are represented, see John D. Cook's article on the subject as well as his introduction to some of the issues arising from how this representation differs in behavior from the idealized abstraction of real numbers.\nAlso recommended is Bruce Dawson's series of blog posts on floating-point numbers.\nFor an excellent, in-depth discussion of floating-point numbers and issues of numerical accuracy encountered when computing with them, see David Goldberg's paper What Every Computer Scientist Should Know About Floating-Point Arithmetic.\nFor even more extensive documentation of the history of, rationale for, and issues with floating-point numbers, as well as discussion of many other topics in numerical computing, see the collected writings of William Kahan, commonly known as the \"Father of Floating-Point\". Of particular interest may be An Interview with the Old Man of Floating-Point.","page":"Integers and Floating-Point Numbers"},{"title":"Arbitrary Precision Arithmetic","location":"manual/integers-and-floating-point-numbers.html#Arbitrary-Precision-Arithmetic","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"To allow computations with arbitrary-precision integers and floating point numbers, Julia wraps the GNU Multiple Precision Arithmetic Library (GMP) and the GNU MPFR Library, respectively. The BigInt and BigFloat types are available in Julia for arbitrary precision integer and floating point numbers respectively.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Constructors exist to create these types from primitive numerical types, and the string literal @big_str or parse can be used to construct them from AbstractStrings. BigInts can also be input as integer literals when they are too big for other built-in integer types. Note that as there is no unsigned arbitrary-precision integer type in Base (BigInt is sufficient in most cases), hexadecimal, octal and binary literals can be used (in addition to decimal literals).","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Once created, they participate in arithmetic with all other numeric types thanks to Julia's type promotion and conversion mechanism:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> BigInt(typemax(Int64)) + 1\n9223372036854775808\n\njulia> big\"123456789012345678901234567890\" + 1\n123456789012345678901234567891\n\njulia> parse(BigInt, \"123456789012345678901234567890\") + 1\n123456789012345678901234567891\n\njulia> string(big\"2\"^200, base=16)\n\"100000000000000000000000000000000000000000000000000\"\n\njulia> 0x100000000000000000000000000000000-1 == typemax(UInt128)\ntrue\n\njulia> 0x000000000000000000000000000000000\n0\n\njulia> typeof(ans)\nBigInt\n\njulia> big\"1.23456789012345678901\"\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> parse(BigFloat, \"1.23456789012345678901\")\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> BigFloat(2.0^66) / 3\n2.459565876494606882133333333333333333333333333333333333333333333333333333333344e+19\n\njulia> factorial(BigInt(40))\n815915283247897734345611269596115894272000000000","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"However, type promotion between the primitive types above and BigInt/BigFloat is not automatic and must be explicitly stated.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = typemin(Int64)\n-9223372036854775808\n\njulia> x = x - 1\n9223372036854775807\n\njulia> typeof(x)\nInt64\n\njulia> y = BigInt(typemin(Int64))\n-9223372036854775808\n\njulia> y = y - 1\n-9223372036854775809\n\njulia> typeof(y)\nBigInt","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The default precision (in number of bits of the significand) and rounding mode of BigFloat operations can be changed globally by calling setprecision and setrounding, and all further calculations will take these changes in account.  Alternatively, the precision or the rounding can be changed only within the execution of a particular block of code by using the same functions with a do block:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> setrounding(BigFloat, RoundUp) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.100000000000000000000000000000000000000000000000000000000000000000000000000003\n\njulia> setrounding(BigFloat, RoundDown) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.099999999999999999999999999999999999999999999999999999999999999999999999999986\n\njulia> setprecision(40) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.1000000000004","page":"Integers and Floating-Point Numbers"},{"title":"Numeric Literal Coefficients","location":"manual/integers-and-floating-point-numbers.html#man-numeric-literal-coefficients","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"To make common numeric formulae and expressions clearer, Julia allows variables to be immediately preceded by a numeric literal, implying multiplication. This makes writing polynomial expressions much cleaner:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> x = 3\n3\n\njulia> 2x^2 - 3x + 1\n10\n\njulia> 1.5x^2 - .5x + 1\n13.0","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"It also makes writing exponential functions more elegant:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 2^2x\n64","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The precedence of numeric literal coefficients is slightly lower than that of unary operators such as negation. So -2x is parsed as (-2) * x and √2x is parsed as (√2) * x. However, numeric literal coefficients parse similarly to unary operators when combined with exponentiation. For example 2^3x is parsed as 2^(3x), and 2x^3 is parsed as 2*(x^3).","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Numeric literals also work as coefficients to parenthesized expressions:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> 2(x-1)^2 - 3(x-1) + 1\n3","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"note: Note\nThe precedence of numeric literal coefficients used for implicit multiplication is higher than other binary operators such as multiplication (*), and division (/, \\, and //).  This means, for example, that 1 / 2im equals -0.5im and 6 // 2(2 + 1) equals 1 // 1.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Additionally, parenthesized expressions can be used as coefficients to variables, implying multiplication of the expression by the variable:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> (x-1)x\n6","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Neither juxtaposition of two parenthesized expressions, nor placing a variable before a parenthesized expression, however, can be used to imply multiplication:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> (x-1)(x+1)\nERROR: MethodError: objects of type Int64 are not callable\n\njulia> x(x+1)\nERROR: MethodError: objects of type Int64 are not callable","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Both expressions are interpreted as function application: any expression that is not a numeric literal, when immediately followed by a parenthetical, is interpreted as a function applied to the values in parentheses (see Functions for more about functions). Thus, in both of these cases, an error occurs since the left-hand value is not a function.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The above syntactic enhancements significantly reduce the visual noise incurred when writing common mathematical formulae. Note that no whitespace may come between a numeric literal coefficient and the identifier or parenthesized expression which it multiplies.","page":"Integers and Floating-Point Numbers"},{"title":"Syntax Conflicts","location":"manual/integers-and-floating-point-numbers.html#Syntax-Conflicts","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Juxtaposed literal coefficient syntax may conflict with some numeric literal syntaxes: hexadecimal, octal and binary integer literals and engineering notation for floating-point literals. Here are some situations where syntactic conflicts arise:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"The hexadecimal integer literal expression 0xff could be interpreted as the numeric literal 0 multiplied by the variable xff. Similar ambiguities arise with octal and binary literals like 0o777 or 0b01001010.\nThe floating-point literal expression 1e10 could be interpreted as the numeric literal 1 multiplied by the variable e10, and similarly with the equivalent E form.\nThe 32-bit floating-point literal expression 1.5f22 could be interpreted as the numeric literal 1.5 multiplied by the variable f22.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"In all cases the ambiguity is resolved in favor of interpretation as numeric literals:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Expressions starting with 0x/0o/0b are always hexadecimal/octal/binary literals.\nExpressions starting with a numeric literal followed by e or E are always floating-point literals.\nExpressions starting with a numeric literal followed by f are always 32-bit floating-point literals.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Unlike E, which is equivalent to e in numeric literals for historical reasons, F is just another letter and does not behave like f in numeric literals. Hence, expressions starting with a numeric literal followed by F are interpreted as the numerical literal multiplied by a variable, which means that, for example, 1.5F22 is equal to 1.5 * F22.","page":"Integers and Floating-Point Numbers"},{"title":"Literal zero and one","location":"manual/integers-and-floating-point-numbers.html#Literal-zero-and-one","category":"section","text":"","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Julia provides functions which return literal 0 and 1 corresponding to a specified type or the type of a given variable.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Function Description\nzero(x) Literal zero of type x or type of variable x\none(x) Literal one of type x or type of variable x","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"These functions are useful in Numeric Comparisons to avoid overhead from unnecessary type conversion.","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"Examples:","page":"Integers and Floating-Point Numbers"},{"title":"Integers and Floating-Point Numbers","location":"manual/integers-and-floating-point-numbers.html","category":"page","text":"julia> zero(Float32)\n0.0f0\n\njulia> zero(1.0)\n0.0\n\njulia> one(Int32)\n1\n\njulia> one(BigFloat)\n1.0","page":"Integers and Floating-Point Numbers"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html#Calling-C-and-Fortran-Code","category":"section","text":"","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Though most code can be written in Julia, there are many high-quality, mature libraries for numerical computing already written in C and Fortran. To allow easy use of this existing code, Julia makes it simple and efficient to call C and Fortran functions. Julia has a \"no boilerplate\" philosophy: functions can be called directly from Julia without any \"glue\" code, code generation, or compilation – even from the interactive prompt. This is accomplished just by making an appropriate call with ccall syntax, which looks like an ordinary function call.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"The code to be called must be available as a shared library. Most C and Fortran libraries ship compiled as shared libraries already, but if you are compiling the code yourself using GCC (or Clang), you will need to use the -shared and -fPIC options. The machine instructions generated by Julia's JIT are the same as a native C call would be, so the resulting overhead is the same as calling a library function from C code. [1]","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Shared libraries and functions are referenced by a tuple of the form (:function, \"library\") or (\"function\", \"library\") where function is the C-exported function name, and library refers to the shared library name.  Shared libraries available in the (platform-specific) load path will be resolved by name.  The full path to the library may also be specified.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"A function name may be used alone in place of the tuple (just :function or \"function\"). In this case the name is resolved within the current process. This form can be used to call C library functions, functions in the Julia runtime, or functions in an application linked to Julia.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"By default, Fortran compilers generate mangled names (for example, converting function names to lowercase or uppercase, often appending an underscore), and so to call a Fortran function via ccall you must pass the mangled identifier corresponding to the rule followed by your Fortran compiler.  Also, when calling a Fortran function, all inputs must be passed as pointers to allocated values on the heap or stack. This applies not only to arrays and other mutable objects which are normally heap-allocated, but also to scalar values such as integers and floats which are normally stack-allocated and commonly passed in registers when using C or Julia calling conventions.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Finally, you can use ccall to actually generate a call to the library function. The arguments to ccall are:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"A (:function, \"library\") pair (most common),\nOR\na :function name symbol or \"function\" name string (for symbols in the current process or libc),\nOR\na function pointer (for example, from dlsym).\nThe function's return type\nA tuple of input types, corresponding to the function signature\nThe actual argument values to be passed to the function, if any; each is a separate parameter.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"note: Note\nThe (:function, \"library\") pair, return type, and input types must be literal constants (i.e., they can't be variables, but see Non-constant Function Specifications below).The remaining parameters are evaluated at compile time, when the containing method is defined.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"note: Note\nSee below for how to map C types to Julia types.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"As a complete but simple example, the following calls the clock function from the standard C library on most Unix-derived systems:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"julia> t = ccall(:clock, Int32, ())\n2292761\n\njulia> t\n2292761\n\njulia> typeof(t)\nInt32","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"clock takes no arguments and returns an Int32. One common mistake is forgetting that a 1-tuple of argument types must be written with a trailing comma. For example, to call the getenv function to get a pointer to the value of an environment variable, one makes a call like this:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"julia> path = ccall(:getenv, Cstring, (Cstring,), \"SHELL\")\nCstring(@0x00007fff5fbffc45)\n\njulia> unsafe_string(path)\n\"/bin/bash\"","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Note that the argument type tuple must be written as (Cstring,), not (Cstring). This is because (Cstring) is just the expression Cstring surrounded by parentheses, rather than a 1-tuple containing Cstring:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"julia> (Cstring)\nCstring\n\njulia> (Cstring,)\n(Cstring,)","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"In practice, especially when providing reusable functionality, one generally wraps ccall uses in Julia functions that set up arguments and then check for errors in whatever manner the C or Fortran function specifies. And if an error occurs it is thrown as a normal Julia exception. This is especially important since C and Fortran APIs are notoriously inconsistent about how they indicate error conditions. For example, the getenv C library function is wrapped in the following Julia function, which is a simplified version of the actual definition from env.jl:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"function getenv(var::AbstractString)\n    val = ccall(:getenv, Cstring, (Cstring,), var)\n    if val == C_NULL\n        error(\"getenv: undefined variable: \", var)\n    end\n    return unsafe_string(val)\nend","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"The C getenv function indicates an error by returning NULL, but other standard C functions indicate errors in various different ways, including by returning -1, 0, 1 and other special values. This wrapper throws an exception clearly indicating the problem if the caller tries to get a non-existent environment variable:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"julia> getenv(\"SHELL\")\n\"/bin/bash\"\n\njulia> getenv(\"FOOBAR\")\ngetenv: undefined variable: FOOBAR","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Here is a slightly more complex example that discovers the local machine's hostname. In this example, the networking library code is assumed to be in a shared library named \"libc\". In practice, this function is usually part of the C standard library, and so the \"libc\" portion should be omitted, but we wish to show here the usage of this syntax.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"function gethostname()\n    hostname = Vector{UInt8}(undef, 256) # MAXHOSTNAMELEN\n    err = ccall((:gethostname, \"libc\"), Int32,\n                (Ptr{UInt8}, Csize_t),\n                hostname, sizeof(hostname))\n    Base.systemerror(\"gethostname\", err != 0)\n    hostname[end] = 0 # ensure null-termination\n    return GC.@preserve hostname unsafe_string(pointer(hostname))\nend","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"This example first allocates an array of bytes. It then calls the C library function gethostname to populate the array with the hostname. Finally, it takes a pointer to the hostname buffer, and converts the pointer to a Julia string, assuming that it is a NUL-terminated C string.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"It is common for C libraries to use this pattern of requiring the caller to allocate memory to be passed to the callee and populated. Allocation of memory from Julia like this is generally accomplished by creating an uninitialized array and passing a pointer to its data to the C function. This is why we don't use the Cstring type here: as the array is uninitialized, it could contain NUL bytes. Converting to a Cstring as part of the ccall checks for contained NUL bytes and could therefore throw a conversion error.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"Deferencing pointer(hostname) with unsafe_string is an unsafe operation as it requires access to the memory allocated for hostname that may have been in the meanwhile garbage collected. The macro GC.@preserve prevents this from happening and therefore accessing an invalid memory location.","page":"Calling C and Fortran Code"},{"title":"Creating C-Compatible Julia Function Pointers","location":"manual/calling-c-and-fortran-code.html#Creating-C-Compatible-Julia-Function-Pointers","category":"section","text":"","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"It is possible to pass Julia functions to native C functions that accept function pointer arguments. For example, to match C prototypes of the form:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"typedef returntype (*functiontype)(argumenttype, ...)","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"The macro @cfunction generates the C-compatible function pointer for a call to a Julia function. The arguments to @cfunction are:","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"A Julia function\nThe function's return type\nA tuple of input types, corresponding to the function signature","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"note: Note\nAs with ccall, the return type and tuple of input types must be literal constants.","page":"Calling C and Fortran Code"},{"title":"Calling C and Fortran Code","location":"manual/calling-c-and-fortran-code.html","category":"page","text":"note: Note\nCurrently, only the platform-default C calling convention is supported. This means that @cfunction-generated pointers cannot be used in calls where WINAPI expects a stdcall function on 32-bit Windows, but can be used on WIN64 (where stdcall is unified with the C calling convention).",