//  Copyright (c) 2014-2017 Hartmut Kaiser
//
//  SPDX-License-Identifier: BSL-1.0
//  Distributed under the Boost Software License, Version 1.0. (See accompanying
//  file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

// make inspect happy: hpxinspect:nominmax

/// \file parallel/algorithms/minmax.hpp

#pragma once

#include <hpx/config.hpp>
#include <hpx/algorithms/traits/segmented_iterator_traits.hpp>
#include <hpx/assert.hpp>
#include <hpx/concepts/concepts.hpp>
#include <hpx/functional/invoke.hpp>
#include <hpx/iterator_support/traits/is_iterator.hpp>
#include <hpx/parallel/util/tagged_pair.hpp>

#include <hpx/algorithms/traits/projected.hpp>
#include <hpx/executors/execution_policy.hpp>
#include <hpx/parallel/algorithms/detail/dispatch.hpp>
#include <hpx/parallel/tagspec.hpp>
#include <hpx/parallel/util/compare_projected.hpp>
#include <hpx/parallel/util/detail/algorithm_result.hpp>
#include <hpx/parallel/util/loop.hpp>
#include <hpx/parallel/util/partitioner.hpp>
#include <hpx/parallel/util/projection_identity.hpp>

#include <algorithm>
#include <cstddef>
#include <iterator>
#include <type_traits>
#include <utility>
#include <vector>

namespace hpx { namespace parallel { inline namespace v1 {
    ///////////////////////////////////////////////////////////////////////////
    // min_element
    namespace detail {
        /// \cond NOINTERNAL
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        constexpr FwdIter sequential_min_element(ExPolicy&&, FwdIter it,
            std::size_t count, F const& f, Proj const& proj)
        {
            if (count == 0 || count == 1)
                return it;

            FwdIter smallest = it;

            util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                [&f, &smallest, &proj, value = HPX_INVOKE(proj, *smallest)](
                    FwdIter const& curr) mutable -> void {
                    auto curr_value = HPX_INVOKE(proj, *curr);
                    if (HPX_INVOKE(f, curr_value, value))
                    {
                        smallest = curr;
                        value = std::move(curr_value);
                    }
                });

            return smallest;
        }

        ///////////////////////////////////////////////////////////////////////
        template <typename Iter>
        struct min_element : public detail::algorithm<min_element<Iter>, Iter>
        {
            // this has to be a member of the algorithm type as we access this
            // generically from the segmented algorithms
            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static constexpr typename std::iterator_traits<FwdIter>::value_type
            sequential_minmax_element_ind(ExPolicy&&, FwdIter it,
                std::size_t count, F const& f, Proj const& proj)
            {
                HPX_ASSERT(count != 0);

                if (count == 1)
                    return *it;

                auto smallest = *it;

                util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                    [&f, &smallest, &proj, value = HPX_INVOKE(proj, *smallest)](
                        FwdIter const& curr) mutable -> void {
                        auto curr_value = HPX_INVOKE(proj, **curr);
                        if (HPX_INVOKE(f, curr_value, value))
                        {
                            smallest = *curr;
                            value = std::move(curr_value);
                        }
                    });

                return smallest;
            }

            min_element()
              : min_element::algorithm("min_element")
            {
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static FwdIter sequential(
                ExPolicy, FwdIter first, FwdIter last, F&& f, Proj&& proj)
            {
                return std::min_element(first, last,
                    util::compare_projected<F, Proj>(
                        std::forward<F>(f), std::forward<Proj>(proj)));
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static
                typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
                parallel(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
                    Proj&& proj)
            {
                if (first == last)
                {
                    return util::detail::algorithm_result<ExPolicy,
                        FwdIter>::get(std::move(first));
                }

                auto f1 = [f, proj, policy](
                              FwdIter it, std::size_t part_count) -> FwdIter {
                    return sequential_min_element(
                        policy, it, part_count, f, proj);
                };
                auto f2 = [policy, f = std::forward<F>(f),
                              proj = std::forward<Proj>(proj)](
                              std::vector<FwdIter>&& positions) -> FwdIter {
                    return min_element::sequential_minmax_element_ind(
                        policy, positions.begin(), positions.size(), f, proj);
                };

                return util::partitioner<ExPolicy, FwdIter, FwdIter>::call(
                    std::forward<ExPolicy>(policy), first,
                    std::distance(first, last), std::move(f1),
                    hpx::unwrapping(std::move(f2)));
            }
        };

        ///////////////////////////////////////////////////////////////////////
        // non-segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
        min_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::false_type)
        {
            return detail::min_element<FwdIter>().call(
                std::forward<ExPolicy>(policy), first, last, std::forward<F>(f),
                std::forward<Proj>(proj));
        }

        ///////////////////////////////////////////////////////////////////////
        // segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
        min_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::true_type);

        /// \endcond
    }    // namespace detail

    /// Finds the smallest element in the range [first, last) using the given
    /// comparison function \a f.
    ///
    /// \note   Complexity: Exactly \a max(N-1, 0) comparisons, where
    ///                     N = std::distance(first, last).
    ///
    /// \tparam ExPolicy    The type of the execution policy to use (deduced).
    ///                     It describes the manner in which the execution
    ///                     of the algorithm may be parallelized and the manner
    ///                     in which it executes the assignments.
    /// \tparam FwdIter     The type of the source iterators used (deduced).
    ///                     This iterator type must meet the requirements of a
    ///                     forward iterator.
    /// \tparam F           The type of the function/function object to use
    ///                     (deduced). Unlike its sequential form, the parallel
    ///                     overload of \a min_element requires \a F to meet the
    ///                     requirements of \a CopyConstructible.
    /// \tparam Proj        The type of an optional projection function. This
    ///                     defaults to \a util::projection_identity
    ///
    /// \param policy       The execution policy to use for the scheduling of
    ///                     the iterations.
    /// \param first        Refers to the beginning of the sequence of elements
    ///                     the algorithm will be applied to.
    /// \param last         Refers to the end of the sequence of elements the
    ///                     algorithm will be applied to.
    /// \param f            The binary predicate which returns true if the
    ///                     the left argument is less than the right element.
    ///                     The signature
    ///                     of the predicate function should be equivalent to
    ///                     the following:
    ///                     \code
    ///                     bool pred(const Type1 &a, const Type1 &b);
    ///                     \endcode \n
    ///                     The signature does not need to have const &, but
    ///                     the function must not modify the objects passed to
    ///                     it. The type \a Type1 must be such that objects of
    ///                     type \a FwdIter can be dereferenced and then
    ///                     implicitly converted to \a Type1.
    /// \param proj         Specifies the function (or function object) which
    ///                     will be invoked for each of the elements as a
    ///                     projection operation before the actual predicate
    ///                     \a is invoked.
    ///
    /// The comparisons in the parallel \a min_element algorithm invoked with
    /// an execution policy object of type \a sequenced_policy
    /// execute in sequential order in the calling thread.
    ///
    /// The comparisons in the parallel \a min_element algorithm invoked with
    /// an execution policy object of type \a parallel_policy or
    /// \a parallel_task_policy are permitted to execute in an unordered
    /// fashion in unspecified threads, and indeterminately sequenced
    /// within each thread.
    ///
    /// \returns  The \a min_element algorithm returns a \a hpx::future<FwdIter>
    ///           if the execution policy is of type
    ///           \a sequenced_task_policy or
    ///           \a parallel_task_policy
    ///           and returns \a FwdIter otherwise.
    ///           The \a min_element algorithm returns the iterator to the
    ///           smallest element in the range [first, last). If several
    ///           elements in the range are equivalent to the smallest element,
    ///           returns the iterator to the first such element. Returns last
    ///           if the range is empty.
    ///
    template <typename ExPolicy, typename FwdIter,
        typename Proj = util::projection_identity, typename F = detail::less,
        HPX_CONCEPT_REQUIRES_(hpx::is_execution_policy<ExPolicy>::value&& hpx::
                traits::is_iterator<FwdIter>::value&& traits::is_projected<Proj,
                    FwdIter>::value&& traits::is_indirect_callable<ExPolicy, F,
                    traits::projected<Proj, FwdIter>,
                    traits::projected<Proj, FwdIter>>::value)>
    typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
    min_element(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f = F(),
        Proj&& proj = Proj())
    {
        static_assert((hpx::traits::is_forward_iterator<FwdIter>::value),
            "Requires at least forward iterator.");

        typedef hpx::traits::is_segmented_iterator<FwdIter> is_segmented;

        return detail::min_element_(std::forward<ExPolicy>(policy), first, last,
            std::forward<F>(f), std::forward<Proj>(proj), is_segmented());
    }

    ///////////////////////////////////////////////////////////////////////////
    // max_element
    namespace detail {
        /// \cond NOINTERNAL
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        constexpr FwdIter sequential_max_element(ExPolicy&&, FwdIter it,
            std::size_t count, F const& f, Proj const& proj)
        {
            if (count == 0 || count == 1)
                return it;

            FwdIter greatest = it;

            util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                [&f, &greatest, &proj, value = HPX_INVOKE(proj, *greatest)](
                    FwdIter const& curr) mutable -> void {
                    auto curr_value = HPX_INVOKE(proj, *curr);
                    if (HPX_INVOKE(f, value, curr_value))
                    {
                        greatest = curr;
                        value = std::move(curr_value);
                    }
                });

            return greatest;
        }

        ///////////////////////////////////////////////////////////////////////
        template <typename Iter>
        struct max_element : public detail::algorithm<max_element<Iter>, Iter>
        {
            // this has to be a member of the algorithm type as we access this
            // generically from the segmented algorithms
            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static constexpr typename std::iterator_traits<FwdIter>::value_type
            sequential_minmax_element_ind(ExPolicy&&, FwdIter it,
                std::size_t count, F const& f, Proj const& proj)
            {
                HPX_ASSERT(count != 0);

                if (count == 1)
                    return *it;

                auto greatest = *it;

                util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                    [&f, &greatest, &proj, value = HPX_INVOKE(proj, *greatest)](
                        FwdIter const& curr) mutable -> void {
                        auto curr_value = HPX_INVOKE(proj, **curr);
                        if (HPX_INVOKE(f, value, curr_value))
                        {
                            greatest = *curr;
                            value = std::move(curr_value);
                        }
                    });

                return greatest;
            }

            max_element()
              : max_element::algorithm("max_element")
            {
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static FwdIter sequential(
                ExPolicy, FwdIter first, FwdIter last, F&& f, Proj&& proj)
            {
                return std::max_element(first, last,
                    util::compare_projected<F, Proj>(
                        std::forward<F>(f), std::forward<Proj>(proj)));
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static
                typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
                parallel(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
                    Proj&& proj)
            {
                if (first == last)
                {
                    return util::detail::algorithm_result<ExPolicy,
                        FwdIter>::get(std::move(first));
                }

                auto f1 = [f, proj, policy](
                              FwdIter it, std::size_t part_count) -> FwdIter {
                    return sequential_max_element(
                        policy, it, part_count, f, proj);
                };
                auto f2 = [policy, f = std::forward<F>(f),
                              proj = std::forward<Proj>(proj)](
                              std::vector<FwdIter>&& positions) -> FwdIter {
                    return max_element::sequential_minmax_element_ind(
                        policy, positions.begin(), positions.size(), f, proj);
                };

                return util::partitioner<ExPolicy, FwdIter, FwdIter>::call(
                    std::forward<ExPolicy>(policy), first,
                    std::distance(first, last), std::move(f1),
                    hpx::unwrapping(std::move(f2)));
            }
        };

        ///////////////////////////////////////////////////////////////////////
        // non-segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
        max_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::false_type)
        {
            return detail::max_element<FwdIter>().call(
                std::forward<ExPolicy>(policy), first, last, std::forward<F>(f),
                std::forward<Proj>(proj));
        }

        ///////////////////////////////////////////////////////////////////////
        // segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
        max_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::true_type);

        /// \endcond
    }    // namespace detail

    /// Finds the greatest element in the range [first, last) using the given
    /// comparison function \a f.
    ///
    /// \note   Complexity: Exactly \a max(N-1, 0) comparisons, where
    ///                     N = std::distance(first, last).
    ///
    /// \tparam ExPolicy    The type of the execution policy to use (deduced).
    ///                     It describes the manner in which the execution
    ///                     of the algorithm may be parallelized and the manner
    ///                     in which it executes the assignments.
    /// \tparam FwdIter     The type of the source iterators used (deduced).
    ///                     This iterator type must meet the requirements of a
    ///                     forward iterator.
    /// \tparam F           The type of the function/function object to use
    ///                     (deduced). Unlike its sequential form, the parallel
    ///                     overload of \a max_element requires \a F to meet the
    ///                     requirements of \a CopyConstructible.
    /// \tparam Proj        The type of an optional projection function. This
    ///                     defaults to \a util::projection_identity
    ///
    /// \param policy       The execution policy to use for the scheduling of
    ///                     the iterations.
    /// \param first        Refers to the beginning of the sequence of elements
    ///                     the algorithm will be applied to.
    /// \param last         Refers to the end of the sequence of elements the
    ///                     algorithm will be applied to.
    /// \param f            The binary predicate which returns true if the
    ///                     This argument is optional and defaults to std::less.
    ///                     the left argument is less than the right element.
    ///                     The signature
    ///                     of the predicate function should be equivalent to
    ///                     the following:
    ///                     \code
    ///                     bool pred(const Type1 &a, const Type1 &b);
    ///                     \endcode \n
    ///                     The signature does not need to have const &, but
    ///                     the function must not modify the objects passed to
    ///                     it. The type \a Type1 must be such that objects of
    ///                     type \a FwdIter can be dereferenced and then
    ///                     implicitly converted to \a Type1.
    /// \param proj         Specifies the function (or function object) which
    ///                     will be invoked for each of the elements as a
    ///                     projection operation before the actual predicate
    ///                     \a is invoked.
    ///
    /// The comparisons in the parallel \a max_element algorithm invoked with
    /// an execution policy object of type \a sequenced_policy
    /// execute in sequential order in the calling thread.
    ///
    /// The comparisons in the parallel \a max_element algorithm invoked with
    /// an execution policy object of type \a parallel_policy or
    /// \a parallel_task_policy are permitted to execute in an unordered
    /// fashion in unspecified threads, and indeterminately sequenced
    /// within each thread.
    ///
    /// \returns  The \a max_element algorithm returns a \a hpx::future<FwdIter>
    ///           if the execution policy is of type
    ///           \a sequenced_task_policy or
    ///           \a parallel_task_policy
    ///           and returns \a FwdIter otherwise.
    ///           The \a max_element algorithm returns the iterator to the
    ///           smallest element in the range [first, last). If several
    ///           elements in the range are equivalent to the smallest element,
    ///           returns the iterator to the first such element. Returns last
    ///           if the range is empty.
    ///
    template <typename ExPolicy, typename FwdIter,
        typename Proj = util::projection_identity, typename F = detail::less,
        HPX_CONCEPT_REQUIRES_(hpx::is_execution_policy<ExPolicy>::value&& hpx::
                traits::is_iterator<FwdIter>::value&& traits::is_projected<Proj,
                    FwdIter>::value&& traits::is_indirect_callable<ExPolicy, F,
                    traits::projected<Proj, FwdIter>,
                    traits::projected<Proj, FwdIter>>::value)>
    typename util::detail::algorithm_result<ExPolicy, FwdIter>::type
    max_element(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f = F(),
        Proj&& proj = Proj())
    {
        static_assert((hpx::traits::is_forward_iterator<FwdIter>::value),
            "Requires at least forward iterator.");

        typedef hpx::traits::is_segmented_iterator<FwdIter> is_segmented;

        return detail::max_element_(std::forward<ExPolicy>(policy), first, last,
            std::forward<F>(f), std::forward<Proj>(proj), is_segmented());
    }

    ///////////////////////////////////////////////////////////////////////////
    // minmax_element
    namespace detail {
        /// \cond NOINTERNAL
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        std::pair<FwdIter, FwdIter> sequential_minmax_element(ExPolicy&&,
            FwdIter it, std::size_t count, F const& f, Proj const& proj)
        {
            std::pair<FwdIter, FwdIter> result(it, it);

            if (count == 0 || count == 1)
                return result;

            auto value = HPX_INVOKE(proj, *it);
            util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                [&f, &result, &proj, min_value = value, max_value = value](
                    FwdIter const& curr) mutable -> void {
                    auto curr_value = HPX_INVOKE(proj, *curr);
                    if (HPX_INVOKE(f, curr_value, min_value))
                    {
                        result.first = curr;
                        min_value = curr_value;
                    }

                    if (!HPX_INVOKE(f, curr_value, max_value))
                    {
                        result.second = curr;
                        max_value = std::move(curr_value);
                    }
                });

            return result;
        }

        template <typename Iter>
        struct minmax_element
          : public detail::algorithm<minmax_element<Iter>,
                std::pair<Iter, Iter>>
        {
            // this has to be a member of the algorithm type as we access this
            // generically from the segmented algorithms
            template <typename ExPolicy, typename PairIter, typename F,
                typename Proj>
            static typename std::iterator_traits<PairIter>::value_type
            sequential_minmax_element_ind(ExPolicy&&, PairIter it,
                std::size_t count, F const& f, Proj const& proj)
            {
                HPX_ASSERT(count != 0);

                if (count == 1)
                    return *it;

                auto result = *it;

                util::loop_n<std::decay_t<ExPolicy>>(++it, count - 1,
                    [&f, &result, &proj,
                        min_value = HPX_INVOKE(proj, *result.first),
                        max_value = HPX_INVOKE(proj, *result.second)](
                        PairIter const& curr) mutable -> void {
                        auto curr_min_value = HPX_INVOKE(proj, *curr->first);
                        if (HPX_INVOKE(f, curr_min_value, min_value))
                        {
                            result.first = curr->first;
                            min_value = std::move(curr_min_value);
                        }

                        auto curr_max_value = HPX_INVOKE(proj, *curr->second);
                        if (!HPX_INVOKE(f, curr_max_value, max_value))
                        {
                            result.second = curr->second;
                            max_value = std::move(curr_max_value);
                        }
                    });

                return result;
            }

            minmax_element()
              : minmax_element::algorithm("minmax_element")
            {
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static std::pair<FwdIter, FwdIter> sequential(
                ExPolicy, FwdIter first, FwdIter last, F&& f, Proj&& proj)
            {
                return std::minmax_element(first, last,
                    util::compare_projected<F, Proj>(
                        std::forward<F>(f), std::forward<Proj>(proj)));
            }

            template <typename ExPolicy, typename FwdIter, typename F,
                typename Proj>
            static typename util::detail::algorithm_result<ExPolicy,
                std::pair<FwdIter, FwdIter>>::type
            parallel(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
                Proj&& proj)
            {
                typedef std::pair<FwdIter, FwdIter> result_type;

                result_type result(first, first);
                if (first == last || ++first == last)
                {
                    return util::detail::algorithm_result<ExPolicy,
                        result_type>::get(std::move(result));
                }

                auto f1 =
                    [f, proj, policy](FwdIter it,
                        std::size_t part_count) -> std::pair<FwdIter, FwdIter> {
                    return sequential_minmax_element(
                        policy, it, part_count, f, proj);
                };
                auto f2 =
                    [policy, f = std::forward<F>(f),
                        proj = std::forward<Proj>(proj)](
                        std::vector<result_type>&& positions) -> result_type {
                    return minmax_element::sequential_minmax_element_ind(
                        policy, positions.begin(), positions.size(), f, proj);
                };

                return util::partitioner<ExPolicy, result_type,
                    result_type>::call(std::forward<ExPolicy>(policy),
                    result.first, std::distance(result.first, last),
                    std::move(f1), hpx::unwrapping(std::move(f2)));
            }
        };

        ///////////////////////////////////////////////////////////////////////
        // non-segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy,
            std::pair<FwdIter, FwdIter>>::type
        minmax_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::false_type)
        {
            return detail::minmax_element<FwdIter>().call(
                std::forward<ExPolicy>(policy), first, last, std::forward<F>(f),
                std::forward<Proj>(proj));
        }

        ///////////////////////////////////////////////////////////////////////
        // segmented implementation
        template <typename ExPolicy, typename FwdIter, typename F,
            typename Proj>
        inline typename util::detail::algorithm_result<ExPolicy,
            std::pair<FwdIter, FwdIter>>::type
        minmax_element_(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f,
            Proj&& proj, std::true_type);

        /// \endcond
    }    // namespace detail

    /// Finds the greatest element in the range [first, last) using the given
    /// comparison function \a f.
    ///
    /// \note   Complexity: At most \a max(floor(3/2*(N-1)), 0) applications of
    ///                     the predicate, where N = std::distance(first, last).
    ///
    /// \tparam ExPolicy    The type of the execution policy to use (deduced).
    ///                     It describes the manner in which the execution
    ///                     of the algorithm may be parallelized and the manner
    ///                     in which it executes the assignments.
    /// \tparam FwdIter     The type of the source iterators used (deduced).
    ///                     This iterator type must meet the requirements of a
    ///                     forward iterator.
    /// \tparam F           The type of the function/function object to use
    ///                     (deduced). Unlike its sequential form, the parallel
    ///                     overload of \a minmax_element requires \a F to meet the
    ///                     requirements of \a CopyConstructible.
    /// \tparam Proj        The type of an optional projection function. This
    ///                     defaults to \a util::projection_identity
    ///
    /// \param policy       The execution policy to use for the scheduling of
    ///                     the iterations.
    /// \param first        Refers to the beginning of the sequence of elements
    ///                     the algorithm will be applied to.
    /// \param last         Refers to the end of the sequence of elements the
    ///                     algorithm will be applied to.
    /// \param f            The binary predicate which returns true if the
    ///                     the left argument is less than the right element.
    ///                     This argument is optional and defaults to std::less.
    ///                     The signature
    ///                     of the predicate function should be equivalent to
    ///                     the following:
    ///                     \code
    ///                     bool pred(const Type1 &a, const Type1 &b);
    ///                     \endcode \n
    ///                     The signature does not need to have const &, but
    ///                     the function must not modify the objects passed to
    ///                     it. The type \a Type1 must be such that objects of
    ///                     type \a FwdIter can be dereferenced and then
    ///                     implicitly converted to \a Type1.
    /// \param proj         Specifies the function (or function object) which
    ///                     will be invoked for each of the elements as a
    ///                     projection operation before the actual predicate
    ///                     \a is invoked.
    ///
    /// The comparisons in the parallel \a minmax_element algorithm invoked with
    /// an execution policy object of type \a sequenced_policy
    /// execute in sequential order in the calling thread.
    ///
    /// The comparisons in the parallel \a minmax_element algorithm invoked with
    /// an execution policy object of type \a parallel_policy or
    /// \a parallel_task_policy are permitted to execute in an unordered
    /// fashion in unspecified threads, and indeterminately sequenced
    /// within each thread.
    ///
    /// \returns  The \a minmax_element algorithm returns a
    /// \a hpx::future<tagged_pair<tag::min(FwdIter), tag::max(FwdIter)> >
    ///           if the execution policy is of type
    ///           \a sequenced_task_policy or
    ///           \a parallel_task_policy
    ///           and returns \a tagged_pair<tag::min(FwdIter), tag::max(FwdIter)>
    ///           otherwise.
    ///           The \a minmax_element algorithm returns a pair consisting of
    ///           an iterator to the smallest element as the first element and
    ///           an iterator to the greatest element as the second. Returns
    ///           std::make_pair(first, first) if the range is empty. If
    ///           several elements are equivalent to the smallest element, the
    ///           iterator to the first such element is returned. If several
    ///           elements are equivalent to the largest element, the iterator
    ///           to the last such element is returned.
    ///
#if defined(HPX_MSVC)
#pragma push_macro("min")
#pragma push_macro("max")
#undef min
#undef max
#endif
    template <typename ExPolicy, typename FwdIter,
        typename Proj = util::projection_identity, typename F = detail::less,
        HPX_CONCEPT_REQUIRES_(hpx::is_execution_policy<ExPolicy>::value&& hpx::
                traits::is_iterator<FwdIter>::value&& traits::is_projected<Proj,
                    FwdIter>::value&& traits::is_indirect_callable<ExPolicy, F,
                    traits::projected<Proj, FwdIter>,
                    traits::projected<Proj, FwdIter>>::value)>
    typename util::detail::algorithm_result<ExPolicy,
        hpx::util::tagged_pair<tag::min(FwdIter), tag::max(FwdIter)>>::type
    minmax_element(ExPolicy&& policy, FwdIter first, FwdIter last, F&& f = F(),
        Proj&& proj = Proj())
    {
        static_assert((hpx::traits::is_forward_iterator<FwdIter>::value),
            "Requires at least forward iterator.");

        typedef hpx::traits::is_segmented_iterator<FwdIter> is_segmented;

        return hpx::util::make_tagged_pair<tag::min, tag::max>(
            detail::minmax_element_(std::forward<ExPolicy>(policy), first, last,
                std::forward<F>(f), std::forward<Proj>(proj), is_segmented()));
    }
#if defined(HPX_MSVC)
#pragma pop_macro("min")
#pragma pop_macro("max")
#endif
}}}    // namespace hpx::parallel::v1
