/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * GXMame
 *
 * Copyright 2002-2004 Stephane Pontier <shadow_walker@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "common.h"

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtkhseparator.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktextview.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvbox.h>

#include "properties.h"
#include "gxmame.h"
#include "gui.h"
#include "io.h"
#include "options.h"
#include "options_string.h"

#define BUFFER_SIZE 1000

static GtkWidget     *details_audit_result;
static GtkTextBuffer *details_audit_result_buffer;
static GtkWidget     *sample_check_result;
static GtkWidget     *rom_check_result;

static gboolean       game_checked = FALSE;
static gint           audit_page_no;

static GtkWidget *
options_frame_new (const char *text)
{
	PangoFontDescription *font_desc;
	GtkWidget *label;
	GtkWidget *frame = gtk_frame_new (text);

	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

	label = gtk_frame_get_label_widget (GTK_FRAME (frame));

	font_desc = pango_font_description_new ();
	pango_font_description_set_weight (font_desc,
					   PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font (label, font_desc);
	pango_font_description_free (font_desc);

	return frame;
}

static GtkWidget *
options_frame_create_child (GtkWidget *widget)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (widget), hbox);
	gtk_widget_show (hbox);

	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show (vbox);

	return vbox;
}

static void
audit_game (RomEntry *rom) 
{
	FILE       *xmame_pipe = NULL;
	gchar       line [BUFFER_SIZE];
	const       gchar *title;
	gchar      *command, *p;
	gchar      *rompath_option;
	RomStatus   status;
	gboolean    error_during_check = FALSE;
	GtkTextIter text_iter;

	if (game_checked)
		return;

	while (gtk_events_pending ()) gtk_main_iteration ();

	if (!current_exec) {
		title = _("Can't audit game");
		gtk_label_set_text (GTK_LABEL (rom_check_result), title);
		if (rom->nb_samples > 0) {
			gtk_label_set_text (GTK_LABEL (sample_check_result), title);
		} else {
			gtk_label_set_text (GTK_LABEL (sample_check_result), _("None required"));
		}
		game_checked = TRUE;
		return;
	}
		
	GXMAME_DEBUG ("single audit");
	rompath_option = create_rompath_options_string (current_exec);
	command = g_strdup_printf ("%s -verifyroms %s %s 2>/dev/null",
				   current_exec->path,
				   rompath_option,
				   rom->romname);

	g_free (rompath_option);
	GXMAME_DEBUG ("running command %s", command);
	xmame_pipe = popen (command, "r");
	if (!xmame_pipe) {
		GXMAME_DEBUG ("Could not run %s", command);
		g_free (command);
		return;
	}

	/* Loading */
	while (fgets (line, BUFFER_SIZE, xmame_pipe)) {
		/* jump the last comments */
		if (!line[0] || !line[1])	/* same as strlen <= 1 but faster */
			break;

		if (!strncmp (line, "romset", 6)) {
			for (p = line + 7; (*p && (*p != ' ')); p++);

			*p = '\0';
			p++;

			if (!strncmp (p, "correct", 7)) {
				title = _("Passed");
				status = CORRECT;
			} else if (!strncmp (p, "incorrect", 9) && error_during_check == TRUE) {
				status = INCORRECT;
				title = _("Incorrect");
			} else if (!strncmp (p, "not found", 9)) {
				status = INCORRECT;
				title = _("Not found");
			} else if (!strncmp (p, "best available", 14)) {
				status = CORRECT;
				title = _("Best available");
			} else {
				status = UNKNOWN;
				title = _("Unknown");
			}
			rom->has_roms = status;
		
			gtk_label_set_text (GTK_LABEL (rom_check_result), title);
			/* continue with the GUI */
			error_during_check = FALSE;
		} else if (!strncmp (line, "name", 4)) {
			/* do nothing */
		} else if (!strncmp (line, "---", 3)) {
			/* do nothing */
		} else {
			gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter);
			gtk_text_buffer_insert (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter, line, -1);
			error_during_check = TRUE;
		}
		while (gtk_events_pending ()) gtk_main_iteration ();
	}

	pclose (xmame_pipe);
	g_free (command);

	/* Samples now */
	if (rom->nb_samples > 0) {
		rompath_option = create_rompath_options_string (current_exec);

		command = g_strdup_printf ("%s %s -verifysamples %s 2>/dev/null",
					   current_exec->path,
					   rompath_option,
					   rom->romname);

		g_free (rompath_option);

		GXMAME_DEBUG ("running command %s", command);
		xmame_pipe = popen (command, "r");

		/* Loading */
		while (fgets (line, BUFFER_SIZE, xmame_pipe)) {
			/* jump the last comments */
			if (!line[0] || !line[1])
				break;

			if (!strncmp (line, "sampleset", 9)) {
				for (p = line + 10; (*p && (*p != ' ')); p++);
				*p = '\0';
				p++;

				if (!strncmp (p, "correct", 7)) {
					title = _("Passed");
					status = CORRECT;
				} else if (!strncmp (p, "incorrect", 9) && error_during_check == TRUE) {
					title = _("Incorrect");
					status = INCORRECT;
				} else {
					status = UNKNOWN;
					title = _("Not found");
				}

				gtk_label_set_text (GTK_LABEL (sample_check_result), title);

				rom->has_samples = status;
				/*continue with the GUI */
				error_during_check = FALSE;
			} else {
				gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter);
				gtk_text_buffer_insert (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter, line, -1);
				error_during_check = TRUE;
			}
			while (gtk_events_pending ()) gtk_main_iteration ();
		}
		pclose (xmame_pipe);
		g_free (command);
	} else {
		gtk_label_set_text (GTK_LABEL (sample_check_result), _("None required"));
	}

	game_checked = TRUE;
	update_game_in_list (rom);
}

static gboolean
audit_idle (gpointer data)
{
	audit_game ((RomEntry *)data);

	return FALSE;
}

static void
exit_properties_window (GtkWidget *window, gpointer user_data)
{
	GameOptions *opts;

	opts = (GameOptions*) g_object_get_data (G_OBJECT (window), "opts");
	
	game_options_free (opts);

	gtk_widget_destroy (window);
	/* reenable joystick, was disabled in callback.c (on_properties_activate)*/
	joy_focus_on ();
}


static void
properties_reset (GtkWidget *properties_window)
{
	gchar *filename;
	RomEntry *rom;
	GtkWidget *properties_apply_button;
	GtkWidget *properties_reset_button;
	GameOptions *opts;

	rom = (RomEntry*) g_object_get_data (G_OBJECT (properties_window), "rom");
	properties_apply_button = GTK_WIDGET (g_object_get_data (G_OBJECT (properties_window), "properties_apply_button"));
	properties_reset_button = GTK_WIDGET (g_object_get_data (G_OBJECT (properties_window), "properties_reset_button"));
	opts = (GameOptions*) g_object_get_data (G_OBJECT (properties_window), "opts");

	game_options_free (opts);

	filename = g_build_filename (g_get_home_dir (), ".gxmame", "options", rom->romname, NULL);
	unlink (filename);

	load_properties_options (rom, properties_window);

	gtk_widget_set_sensitive (properties_reset_button, FALSE);

	gtk_widget_set_sensitive (properties_apply_button, FALSE);
}

static void
properties_save (GtkWidget *properties_window)
{
	RomEntry *rom;
	GameOptions *opts;
	GtkWidget *button;

	button = GTK_WIDGET (g_object_get_data (G_OBJECT (properties_window), "properties_apply_button"));
	gtk_widget_set_sensitive (button, FALSE);

	rom = (RomEntry*) g_object_get_data (G_OBJECT (properties_window), "rom");
	opts = (GameOptions*) g_object_get_data (G_OBJECT (properties_window), "opts");

	save_properties_options (rom, opts);
	save_options (rom, opts);
}

static GtkWidget *
bold_label_new (const char *str)
{
	GtkWidget *label;
	char      *text;

	text = g_strdup_printf ("<b>%s:</b>", str);
	label = gtk_label_new (text);
	g_free (text);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_widget_show (label);

	return label;
}

static GtkWidget *
value_label_new (const char *str)
{
	GtkWidget *label;

	label = gtk_label_new (str);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_widget_show (label);

	return label;
}

static char *
get_rom_clone_name (RomEntry *rom)
{
	GList   *l;
	char    *value = NULL;
	gboolean game_found = FALSE;

	/* find the clone name if there is a clone */
	if (strcmp (rom->cloneof, "-")) {
		RomEntry *tmprom;

		for (l = g_list_first (game_list.roms), game_found = FALSE; l; l = g_list_next (l)) {

			tmprom = (RomEntry *)l->data;
			if (!strcmp (tmprom->romname, rom->cloneof)) {
				if (tmprom->the_trailer && gui_prefs.ModifyThe)
					value = g_strdup_printf ("%s, The %s - \"%s\"",
								 tmprom->gamename, tmprom->gamenameext, rom->cloneof);
				else if (tmprom->the_trailer && !gui_prefs.ModifyThe)
					value = g_strdup_printf ("The %s %s - \"%s\"",
								 tmprom->gamename, tmprom->gamenameext, rom->cloneof);
				else
					value = g_strdup_printf ("%s %s - \"%s\"",
								 tmprom->gamename, tmprom->gamenameext, rom->cloneof);
				break;
			}
		}

		if (!value)
			value = g_strdup_printf (" - \"%s\"", rom->cloneof);
	}

	return value;
}

static char *
get_rom_cpu_value (RomEntry *rom)
{
	char *value = NULL;
	char *values [NB_CPU];
	gint  i, j;

	j = 0;
	values[j] = NULL;
	for (i = 0; i < NB_CPU; i++) {
		if (strcmp (rom->cpu_info[i].name, "-")
		    && rom->cpu_info[i].name[0] != '\0') {
			values[j++] = g_strdup_printf ("%s %f MHz%s",
						       rom->cpu_info[i].name,
						       rom->cpu_info[i].clock / 1000000.0,
						       rom->cpu_info[i].sound_flag ? _(" (sound)") : " ");
		}
	}
	values[j++] = NULL;

	value = g_strjoinv ("\n", values);

	for (i = 0; i < j; i++)
		g_free (values[i]);

	return value;
}

static char *
get_rom_sound_value (RomEntry *rom)
{
	char *value = NULL;
	char *values [NB_CPU];
	gint  i, j;

	j = 0;
	values[j] = NULL;
	for (i = 0; i < NB_CPU; i++) {
		if (strcmp (rom->sound_info[i].name, "-")
		    && rom->sound_info[i].name[0]) {
			if (rom->sound_info[i].clock == 0)
				values[j++] = g_strdup_printf ("%s", rom->sound_info[i].name);
			else
				values[j++] = g_strdup_printf ("%s %f MHz",
							      rom->sound_info[i].name,
							      rom->sound_info[i].clock / 1000000.0);
		}
	}
	values[j++] = NULL;

	value = g_strjoinv ("\n", values);

	for (i = 0; i < j; i++)
		g_free (values[i]);

	return value;
}

static void
add_general_tab (GtkWidget    *properties_window,
		 GtkNotebook  *target_notebook,
		 RomEntry     *rom,
		 GtkWidget    *apply_button,
		 GtkWidget    *reset_button)
{
	GtkWidget    *general_vbox;
	GtkWidget    *general_table;
	GtkWidget    *general_label;
	GtkWidget    *image;
	GtkWidget    *hseparator;
	GtkWidget    *scrolledwindow;
	GtkWidget    *label;
	GtkWidget    *frame;
	gchar        *value;
	gchar        *title;
	gint          line_num = 0;

	general_vbox = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (general_vbox);

	general_table = gtk_table_new (10, 3, FALSE);
	gtk_widget_show (general_table);
	gtk_box_pack_start (GTK_BOX (general_vbox), general_table, TRUE, TRUE, 0);

	gtk_container_set_border_width (GTK_CONTAINER (general_table), 10);
	gtk_table_set_row_spacings (GTK_TABLE (general_table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (general_table), 5);

	label = bold_label_new (_("Year"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	label = value_label_new (rom->year);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("Manufacturer"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	label = value_label_new (rom->manu);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	hseparator = gtk_hseparator_new ();
	gtk_widget_show (hseparator);
	gtk_table_attach (GTK_TABLE (general_table), hseparator, 0, 3, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("CPU"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	title = get_rom_cpu_value (rom);
	label = value_label_new (title);
	g_free (title);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	hseparator = gtk_hseparator_new ();
	gtk_widget_show (hseparator);
	gtk_table_attach (GTK_TABLE (general_table), hseparator, 0, 3, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("Sound"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	title = get_rom_sound_value (rom);
	label = value_label_new (title);
	g_free (title);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	hseparator = gtk_hseparator_new ();
	gtk_widget_show (hseparator);
	gtk_table_attach (GTK_TABLE (general_table), hseparator, 0, 3, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("Screen"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	/* doesn't display resolution if this is a vector game */
	if (!rom->vector)
		value = g_strdup_printf ("%i \303\227 %i ", rom->screen_x, rom->screen_y);
	else
		value = g_strdup_printf (" ");
	title = g_strdup_printf ("%s (%s) %.2f Hz",
				 value,
				 rom->horizontal ? "horizontal" : "vertical",
				 rom->screen_freq);
	g_free (value);

	label = value_label_new (title);
	g_free (title);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("Colors"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	title = g_strdup_printf (_("%i colors"), rom->colors);
	label = value_label_new (title);
	g_free (title);
	gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	if (strcmp (rom->cloneof, "-")) {
		label = bold_label_new (_("Clone of"));
		gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

		title = get_rom_clone_name (rom);
		label = value_label_new (title);
		g_free (title);
		gtk_table_attach (GTK_TABLE (general_table), label, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

		line_num++;
	}

	hseparator = gtk_hseparator_new ();
	gtk_widget_show (hseparator);
	gtk_table_attach (GTK_TABLE (general_table), hseparator, 0, 3, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("ROM check"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	rom_check_result = value_label_new (_("Checking..."));
	gtk_table_attach (GTK_TABLE (general_table), rom_check_result, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	label = bold_label_new (_("Sample check"));
	gtk_table_attach (GTK_TABLE (general_table), label, 0, 1, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	sample_check_result = value_label_new (_("Checking..."));
	gtk_table_attach (GTK_TABLE (general_table), sample_check_result, 1, 2, line_num, line_num + 1, GTK_FILL, 0, 0, 0);

	line_num++;

	frame = options_frame_new (_("Details"));
	gtk_widget_show (frame);
	gtk_table_attach (GTK_TABLE (general_table), frame, 0, 2, line_num, line_num + 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	frame = options_frame_create_child (frame);

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (frame), scrolledwindow, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	details_audit_result_buffer = gtk_text_buffer_new (NULL);
	details_audit_result = gtk_text_view_new_with_buffer (details_audit_result_buffer);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (details_audit_result), FALSE);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (details_audit_result), FALSE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (details_audit_result), GTK_WRAP_WORD);

	gtk_widget_show (details_audit_result);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), details_audit_result);

	image = gxmame_get_image_from_stock ("gxmame-general-toolbar");

	general_label = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (general_label), image, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (general_label), gtk_label_new (_("General")), FALSE, FALSE, 0);
	gtk_widget_show_all (general_label);

	audit_page_no = gtk_notebook_append_page (GTK_NOTEBOOK (target_notebook), general_vbox, general_label);
}

static void
properties_response (GtkWidget *dialog,
		     gint       response_id,
		     gpointer   user_data)
{
	switch (response_id) {
	case GTK_RESPONSE_OK:
		properties_save (dialog);
		/*game_options_free (opts);*/
		exit_properties_window (dialog, NULL);
		break;
	case GTK_RESPONSE_APPLY:
		properties_save (dialog);
		break;
	case GTK_RESPONSE_REJECT:
		properties_reset (dialog);
		break;
	case GTK_RESPONSE_CANCEL:
	default:
		exit_properties_window (dialog, NULL);
		break;
	}
}

GtkWidget *
create_properties_windows (RomEntry *rom)
{
	GtkWidget *properties_windows;

	GtkWidget *title_label;

	GtkWidget *properties_apply_button;
	GtkWidget *properties_reset_button = NULL;
	GtkAccelGroup *accel_group;
	
	GtkWidget *notebook1;

	gchar *title;
	GameOptions *opts = NULL;

	game_checked = FALSE;

	accel_group = gtk_accel_group_new ();

	/* try to load the game options */
	
	if (rom) {
		opts = load_options (rom);
		title = g_strdup_printf (_("Properties for %s"), rom_entry_get_list_name (rom));
	} else {
		title = g_strdup (_("Default Properties"));
	}

	properties_windows = gtk_dialog_new ();
	gtk_dialog_set_has_separator (GTK_DIALOG (properties_windows), FALSE);
	gtk_window_set_title (GTK_WINDOW (properties_windows), title);
	g_free (title);
	gtk_window_set_transient_for (GTK_WINDOW (properties_windows), GTK_WINDOW (MainWindow));
	gtk_window_set_modal (GTK_WINDOW (properties_windows), TRUE);

	if (!opts)
		opts = &default_options;

	if (rom) {
		properties_reset_button = gtk_dialog_add_button (GTK_DIALOG (properties_windows),
								 _("Reset to Defaults"),
								 GTK_RESPONSE_REJECT);
		g_object_set_data (G_OBJECT (properties_windows), "properties_reset_button",
				   properties_reset_button);
		gtk_widget_set_sensitive (properties_reset_button, opts != &default_options);
	}

	properties_apply_button = gtk_dialog_add_button (GTK_DIALOG (properties_windows),
							 GTK_STOCK_APPLY,
							 GTK_RESPONSE_APPLY);	
	gtk_widget_set_sensitive (properties_apply_button, FALSE);
	g_object_set_data (G_OBJECT (properties_windows), "properties_apply_button",
			   properties_apply_button);
	gtk_dialog_add_button (GTK_DIALOG (properties_windows),
			       GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL);	
	gtk_dialog_add_button (GTK_DIALOG (properties_windows),
			       GTK_STOCK_OK,
			       GTK_RESPONSE_OK);	

	g_object_set_data (G_OBJECT (properties_windows), "opts", opts);

	if (rom)
		title = g_strdup_printf ("<big><b>%s</b></big>\n\"%s\"", rom->gamename, rom->romname);
	else
		title =  g_strdup_printf ("<big><b>%s</b></big>\n%s", _("Global game options"), _("Default options used by all games"));

	title_label = gtk_label_new (title);
	gtk_label_set_use_markup (GTK_LABEL (title_label), TRUE);
	g_free (title);
	
	gtk_widget_show (title_label);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (properties_windows)->vbox), title_label, TRUE, TRUE, 0);
	gtk_label_set_line_wrap (GTK_LABEL (title_label), TRUE);
	gtk_misc_set_padding (GTK_MISC (title_label), 10, 0);
	gtk_misc_set_alignment (GTK_MISC (title_label), 0, 0.5);			

	notebook1 = gtk_notebook_new ();
	gtk_widget_ref (notebook1);
	g_object_set_data_full (G_OBJECT (properties_windows), "notebook1", notebook1,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (notebook1);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (properties_windows)->vbox), notebook1, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (notebook1), 8);

	if (rom) {
		add_general_tab (properties_windows,
				 GTK_NOTEBOOK (notebook1),
				 rom,
				 properties_apply_button,
				 properties_reset_button);
		g_idle_add (audit_idle, rom);
	}

	GXMAME_DEBUG ("Adding display options");
	add_display_options_tab (properties_windows,
				 GTK_NOTEBOOK (notebook1),
				 rom,
				 opts,
				 properties_apply_button,
				 properties_reset_button);

	GXMAME_DEBUG ("Adding rendering options");
	add_rendering_options_tab (properties_windows,
				   GTK_NOTEBOOK (notebook1),
				   rom,
				   opts,
				   properties_apply_button,
				   properties_reset_button);

	GXMAME_DEBUG ("Adding sound options");
	add_sound_options_tab (properties_windows,
			       GTK_NOTEBOOK (notebook1),
			       rom,
			       opts,
			       properties_apply_button,
			       properties_reset_button);

	GXMAME_DEBUG ("Adding controller options");
	add_controller_options_tab (properties_windows,
				    GTK_NOTEBOOK (notebook1),
				    rom,
				    opts,
				    properties_apply_button,
				    properties_reset_button);

	GXMAME_DEBUG ("Adding misc options");
	add_misc_options_tab (properties_windows,
			      GTK_NOTEBOOK (notebook1),
			      rom,
			      opts,
			      properties_apply_button,
			      properties_reset_button);

	if (!rom || rom->vector) {
		add_vector_options_tab (properties_windows,
					GTK_NOTEBOOK (notebook1),
					rom,
					opts,
					properties_apply_button,
					properties_reset_button);
	}

	if (rom) {
		g_object_set_data (G_OBJECT (properties_windows), "rom", (gpointer) rom);
	}

	g_signal_connect (properties_windows, "response",
			  G_CALLBACK (properties_response),
			  NULL);

	gtk_window_add_accel_group (GTK_WINDOW (properties_windows), accel_group);

	return properties_windows;
}

