/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001-2006 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#include "nv-misc.h"
#include "os-interface.h"
#include "nv.h"
#include "nv-freebsd.h"


/*
 * Option: VideoMemoryTypeOverride
 *
 * Description:
 *
 * We normally detect memory type on TNT cards by scanning the embedded
 * BIOS. Unfortunately, we've seen some cases where a TNT card has been
 * flashed with the wrong bios. For example, an SDRAM based TNT has been
 * flashed with an SGRAM bios, and therefore claims to be an SGRAM TNT.
 * We've therefore provided an override here. Make sure to set the value
 * toe the type of memory used on your card.
 *
 * Possible Values:
 *
 * 1: SDRAM
 * 2: SGRAM
 *
 * Note that we can only do so much here. There are border cases where
 * even this fails. For example, if 2 TNT cards are in the same system,
 * one SGRAM, one SDRAM.
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static uint32_t NVreg_VideoMemoryTypeOverride = 1;

/*
 * Option: EnableVia4x
 *
 * Description:
 *
 * We've had problems with some Via chipsets in 4x mode, we need force
 * them back down to 2x mode. If you'd like to experiment with retaining
 * 4x mode, you may try setting this value to 1 If that hangs the system,
 * you're stuck with 2x mode; there's nothing we can do about it.
 *
 * Possible Values:
 *
 * 0: disable AGP 4x on Via chipsets (default)
 * 1: enable  AGP 4x on Via chipsets
 */

static uint32_t NVreg_EnableVia4x = 0;

/*
 * Option: EnableALiAGP
 *
 * Description:
 *
 * Some ALi chipsets (ALi1541, ALi1647) are known to cause severe system
 * stability problems with AGP enabled. To avoid lockups, we disable AGP
 * on systems with these chipsets by default. It appears that updating the
 * system BIOS and using recent versions of the kernel AGP Gart driver can
 * make such systems much more stable. If you own a system with one of the
 * aforementioned chipsets and had it working reasonably well previously,
 * or if you want to experiment with BIOS and AGPGART revisions, you can
 * re-enable AGP support by setting this option to 1.
 *
 * Possible Values:
 *
 *  0: disable AGP on Ali1541 and ALi1647 (default)
 *  1: enable  AGP on Ali1541 and ALi1647
 */

static uint32_t NVreg_EnableALiAGP = 0;

/*
 * Option: ReqAGPRate
 *
 * Description:
 *
 * Normally, the driver will compare speed modes of the chipset & card,
 * picking the highest common rate. This key forces a maximum limit, to
 * limit the driver to lower speeds. The driver will not attempt a speed
 * beyond what the chipset and card claim they are capable of.
 *
 * Make sure you really know what you're doing before you enable this
 * override. By default, AGP drivers will enable the fastest AGP rate
 * your card and motherboard chipset are capable of. Then, in some cases,
 * our driver will force this rate down to work around bugs in both our
 * chipsets, and motherboard chipsets. Using this variable will override
 * our bug fixes. This may be desirable in some cases, but not most.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  This option expects a bitmask (15 = 8 | 4 | 2 | 1, etc.)
 *
 *  Note that on AGP 3.x chipsets, the only supported AGP rates are
 *  AGP 8x and AGP 4x (if set in ReqAGPRate, AGP 2x and 1x are
 *  ignored by the driver).
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static uint32_t NVreg_ReqAGPRate = 15;

/*
 * Option: NvAGP
 *
 * Description:
 *
 * This options controls which AGP GART driver is used when no explicit
 * request is made to change the default (X server).
 *
 * Possible Values:
 *
 *   0 = disable AGP support
 *   1 = use NVIDIA's builtin driver (if possible)
 *   2 = use the kernel's AGPGART driver (if possible)
 *   3 = use any available driver (try 2, then 1)
 *
 * Please note that NVIDIA's internal AGP GART driver will not be used
 * if AGPGART was either statically linked into your kernel or built as
 * a kernel module and loaded before the NVIDIA kernel module.
 */

#ifdef NV_SUPPORT_OS_AGP
static uint32_t NVreg_NvAGP = 3;
#else
static uint32_t NVreg_NvAGP = 1;
#endif

/*
 * Option: EnableAGPSBA
 *
 * Description:
 *
 * For stability reasons, the driver will not Side Band Addressing even if
 * both the host chipset and the AGP card support it. You may override this
 * behaviour with the following registry key.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Side Band Addressing (default on x86, see below)
 *  1 = enable  Side Band Addressing (if supported)
 */

static uint32_t NVreg_EnableAGPSBA = 0;

/*
 * Option: EnableAGPFW
 *
 * Description:
 *
 * Similar to Side Band Addressing, Fast Writes are disabled by default. If
 * you wish to enable them on systems that support them, you can do so with
 * this registry key. Please note that this may render your system unstable
 * with many AGP chipsets.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Fast Writes (default)
 *  1 = enable  Fast Writes (if supported)
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static uint32_t NVreg_EnableAGPFW = 0;

/*
 * Option: SoftEDIDs
 *
 * Description:
 *
 * The SoftEDIDs registry key enables dynamic generation of an appropriate
 * EDID for mobile LCD displays from data stored in the video BIOS. If this
 * is turned off, then on mobile systems, a hardcoded EDID will be chosen
 * from a table, based on the value of the Mobile registry key.
 *
 * Possible Values:
 *
 *  0 = disable dynamic EDID generation
 *  1 = enable  dynamic EDID generation (default)
 */

static uint32_t NVreg_SoftEDIDs = 1;

/*
 * Option: Mobile
 *
 * Description:
 *
 * The Mobile registry key should only be needed on mobile systems if
 * SoftEDIDs is disabled (see above), in which case the mobile value
 * will be used to lookup the correct EDID for the mobile LCD.
 *
 * Possible Values:
 *
 *  ~0 = auto detect the correct value (default)
 *   1 = Dell notebooks
 *   2 = non-Compal Toshiba
 *   3 = all other notebooks
 *   4 = Compal/Toshiba
 *   5 = Gateway
 *
 * Make sure to specify the correct value for your notebook.
 */

static uint32_t NVreg_Mobile = ~0;

static uint32_t NVreg_ResmanDebugLevel = ~0;
static uint32_t NVreg_FlatPanelMode = 0;
static uint32_t NVreg_DevicesConnected = 0;
static uint32_t NVreg_RmLogonRC = 1;

/*
 * Option: DetectPrimaryVga
 *
 * Description:
 *
 * The NVIDIA kernel module used to rely on the X server to specify whether
 * a graphics device is the primary device or if needs to have its VBIOS
 * posted. This mechanism has been deprecated in favor of new logic in the
 * kernel module that can determine whether a given device is the primary or
 * a secondary, and post the VBIOS based on that logic.
 *
 * Possible values:
 *
 *  0 = rely on the X server to identify the primary device
 *  1 = rely on the new primary VGA device detection logic (default)
 */

static uint32_t NVreg_DetectPrimaryVga = 1;

/*
 * You can enable any of the registry options disabled by default by
 * editing their respective entries in the table below. The last field
 * determines if the option is considered valid - in order for the
 * changes to take effect, you need to recompile and reload the NVIDIA
 * kernel module.
 */

nv_parm_t nv_parms[] = {
    { "NVreg",  "VideoMemoryTypeOverride",  &NVreg_VideoMemoryTypeOverride,  0 },
    { "NVreg",  "EnableVia4x",              &NVreg_EnableVia4x,              1 },
    { "NVreg",  "EnableALiAGP",             &NVreg_EnableALiAGP,             1 },
    { "NVreg",  "NvAGP",                    &NVreg_NvAGP,                    1 },
    { "NVreg",  "ReqAGPRate",               &NVreg_ReqAGPRate,               0 },
    { "NVreg",  "EnableAGPSBA",             &NVreg_EnableAGPSBA,             1 },
    { "NVreg",  "EnableAGPFW",              &NVreg_EnableAGPFW,              1 },
    { "NVreg",  "SoftEDIDs",                &NVreg_SoftEDIDs,                1 },
    { "NVreg",  "Mobile",                   &NVreg_Mobile,                   1 },
    { "NVreg",  "ResmanDebugLevel",         &NVreg_ResmanDebugLevel,         1 },
    { "NVreg",  "FlatPanelMode",            &NVreg_FlatPanelMode,            1 },
    { "NVreg",  "DevicesConnected",         &NVreg_DevicesConnected,         1 },
    { "NVreg",  "RmLogonRC",                &NVreg_RmLogonRC,                1 },
    { "NVreg",  "DetectPrimaryVga",         &NVreg_DetectPrimaryVga,         1 },
    {  NULL,     NULL,                      NULL,                            0 }
};

void nvidia_update_registry(char *new_option_string)
{
    nv_parm_t *entry;
    char *option_string = NULL;
    char *ptr, *mod;
    char *token;
    char *name, *value;
    U032 i, data;

    option_string = strdup(new_option_string, M_NVIDIA);
    ptr = mod = option_string;

    while (*ptr != '\0') {
        if (!isspace(*ptr)) *mod++ = *ptr;
        ptr++;
    }
    *mod = '\0';

    ptr = option_string;

    while ((token = strsep(&ptr, ";")) != NULL) {
        if (!(name = strsep(&token, "=")) || !strlen(name))
            continue;
        if (!(value = strsep(&token, "=")) || !strlen(value))
            continue;
        if (strsep(&token, "=") != NULL)
            continue;

        data = (U032)strtoul(value, NULL, 0);

        for (i = 0; (entry = &nv_parms[i])->name != NULL; i++) {
            if (strcmp(entry->name, name) == 0) {
                *entry->data = data;
                break;
            }
        }

        rm_write_registry_dword(NULL, NULL, "NVreg", name, data);
    }

    free(option_string, M_NVIDIA);
}

RM_STATUS os_registry_init(void)
{
    nv_parm_t *entry;
    unsigned int i;

    for (i = 0; (entry = &nv_parms[i])->name != NULL; i++) {
        if (!entry->valid) continue;
        rm_write_registry_dword(NULL, NULL, entry->node, entry->name, *entry->data);
    }

    return RM_OK;
}
