/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001-2002 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#include "nv-misc.h"
#include "os-interface.h"
#include "nv.h"
#include "nv-freebsd.h"


/*
 * The NVIDIA kernel module's malloc identifier, needed for both tracking
 * and actual allocation/freeing purposes. M_NVIDIA is declared elsewhere
 * to make it known to other parts of the kernel module (nv-freebsd.h).
 */

MALLOC_DEFINE(M_NVIDIA, "nvidia", "NVIDIA memory allocations");


#define MAX_ERROR_STRING    256


RM_STATUS os_alloc_contig_pages(
    void **address,
    U032 size
)
{
    *address = contigmalloc(size, M_NVIDIA, 0, 0, ~0, PAGE_SIZE, 0);
    return *address ? RM_OK : RM_ERROR;
}

void os_free_contig_pages(
    void *address,
    U032 size
)
{
    contigfree(address, size, M_NVIDIA);
}

/*
 * The core resource manager's favorite source of memory, this routine is
 * called from different contexts, including ISRs. This means that it can
 * not be allowed to sleep when memory is low.
 */

RM_STATUS os_alloc_mem(
    void **address,
    U032 size
)
{
    /* XXX Fix me? (malloc flags) */
    *address = malloc(size, M_NVIDIA, M_NOWAIT | M_ZERO);
    return *address ? RM_OK : RM_ERROR;
}

void os_free_mem(void *address)
{
    free(address, M_NVIDIA);
}

#define NV_MSECS_PER_TICK       (1000 / hz)
#define NV_MSECS_TO_TICKS(ms)   ((ms) * hz / 1000)
#define NV_USECS_PER_TICK       (1000000 / hz)
#define NV_USECS_TO_TICKS(us)   ((us) * hz / 1000000)

RM_STATUS os_delay(U032 MilliSeconds)
{
    unsigned long MicroSeconds;
    unsigned long ticks;
    struct timeval tv_end, tv_aux;

    getmicrotime(&tv_aux);

    if (__NV_ITHREAD() && (MilliSeconds > NV_MAX_ISR_DELAY_MS))
        return RM_ERROR;

    if (__NV_ITHREAD()) {
        DELAY(MilliSeconds * 1000);
        return RM_OK;
    }

    MicroSeconds = MilliSeconds * 1000;
    tv_end.tv_usec = MicroSeconds;
    tv_end.tv_sec = 0;
    /* tv_end = tv_aux + tv_end */
    NV_TIMERADD(&tv_aux, &tv_end, &tv_end);

    ticks = NV_USECS_TO_TICKS(MicroSeconds);

    if (ticks > 0) {
        do {
            tsleep((void *)os_delay, PUSER | PCATCH, "delay", ticks);
            getmicrotime(&tv_aux);
            if (NV_TIMERCMP(&tv_aux, &tv_end, <)) {
                /* tv_aux = tv_end - tv_aux */
                NV_TIMERSUB(&tv_end, &tv_aux, &tv_aux);
                MicroSeconds = tv_aux.tv_usec + (tv_aux.tv_sec * 1000000);
            } else
                MicroSeconds = 0;
        } while ((ticks = NV_USECS_TO_TICKS(MicroSeconds)) > 0);
    }

    if (MicroSeconds > 0)
        DELAY(MicroSeconds);

    return RM_OK;
}

RM_STATUS os_delay_us(U032 MicroSeconds)
{
    if (__NV_ITHREAD() && (MicroSeconds > NV_MAX_ISR_DELAY_US))
        return RM_ERROR;
    DELAY(MicroSeconds);
    return RM_OK;
}

U032 os_get_cpu_frequency(void)
{
    /* XXX Fix me! (x86 only) */
    return ((tsc_freq + 4999) / 1000000);
}

U032 os_get_current_process(void)
{
    return curproc->p_pid;
}

RM_STATUS os_kill_process(
    U032 pid,
    U032 sig
)
{
    struct proc *p = pfind(pid);
    if (!p) {
        /*
         * No process with the specified PID was found running. We notify
         * the caller, but take no other action.
         */
        return RM_ERR_OPERATING_SYSTEM;
    }

    psignal(p, sig);

    return RM_OK;
}

RM_STATUS os_get_current_time(
    U032 *sec,
    U032 *usec
)
{
    struct timeval tv;

    getmicrotime(&tv);

    *sec  = tv.tv_sec;
    *usec = tv.tv_usec;

    return RM_OK;
}

BOOL os_is_administrator(PHWINFO pDev)
{
    return suser(CURTHREAD) ? FALSE : TRUE;
}

U008 os_io_read_byte(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inb(address);
}

void os_io_write_byte(
    PHWINFO pDev,
    U032 address,
    U008 value
)
{
    /* XXX Fix me? (bus_space access) */
    outb(address, value);
}

U016 os_io_read_word(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inw(address);
}

void os_io_write_word(
    PHWINFO pDev,
    U032 address,
    U016 value
)
{
    /* XXX Fix me? (bus_space access) */
    return outw(address, value);
}

U032 os_io_read_dword(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inl(address);
}

void os_io_write_dword(
    PHWINFO pDev,
    U032 address,
    U032 value
)
{
    /* XXX Fix me? (bus_space access) */
    outl(address, value);
}

/*
 * XXX On FreeBSD/amd64, pmap_mapdev() returns an address range from the
 * kernel's direct mapping of the first 4+GB; this direct mapping uses
 * 2MB pages and may be cached. Ideally, we'd want to modify it for as long
 * as we need uncached access to the I/O memory in question, but for now
 * we'll have to create a private, uncached mapping to be safe.
 *
 * XXX This introduces aliasing concerns, but it's better than using the
 * cached direct mapping.
 */

void* os_map_kernel_space(
    NvU64 start,
    NvU64 size,
    U032 mode
)
{
    void *vm;
    U032 cache_bits = (mode != NV_MEMORY_CACHED) ? PG_N : 0;
    vm_offset_t tva, va;

#if defined(NVCPU_X86) && !defined(PAE)
    if (start > 0xffffffff)
        return NULL;
#endif

    if (start & PAGE_MASK)
        return NULL;

    size = NV_ALIGN_UP(size, PAGE_SIZE);

    va = kmem_alloc_nofault(kernel_map, size);
    vm = (void *)va;

    if (vm != NULL) {
        for (tva = va; tva < (va + size); tva += PAGE_SIZE) {
            pt_entry_t *ptep = vtopte(tva);
            pte_store(ptep, start | PG_RW | PG_V | PG_G | cache_bits);
            start += PAGE_SIZE;
        }
        pmap_invalidate_range(kernel_pmap, va, tva);
    }

    return vm;
}

void os_unmap_kernel_space(
    void *address,
    NvU64 size
)
{
    vm_offset_t tva, va;
    
    size = NV_ALIGN_UP(size, PAGE_SIZE);

    va = (vm_offset_t) address;
    if (va != 0) {
        for (tva = va; tva < (va + size); tva += PAGE_SIZE)
            pte_clear(vtopte(tva));
        pmap_invalidate_range(kernel_pmap, va, tva);
        kmem_free(kernel_map, va, size);
    }
}

void* os_map_kernel_space_high(
    U032 pfn,
    U032 size
)
{
    U032 start;
    if (!(pfn & ~0xfffff)) {
        start = pfn << PAGE_SHIFT;
        return os_map_kernel_space(start, size, NV_MEMORY_CACHED);
    }
    return NULL;
}

void os_unmap_kernel_space_high(
    void *addr,
    U032 pfn,
    U032 size
)
{
    os_unmap_kernel_space(addr, size);
}

RM_STATUS os_set_mem_range(
    U032 start,
    U032 size,
    U032 mode
)
{
    int arg;
    struct mem_range_desc mrd;

    mrd.mr_base  = start;
    mrd.mr_len   = size;
    mrd.mr_flags = MDF_WRITECOMBINE;

    strcpy(mrd.mr_owner, "NVIDIA");
    arg = MEMRANGE_SET_UPDATE;

    if (mem_range_attr_set(&mrd, &arg))
        return RM_ERROR;

    return RM_OK;
}

RM_STATUS os_unset_mem_range(
    U032 start,
    U032 size
)
{
    int arg;
    struct mem_range_desc mrd;

    mrd.mr_base = start;
    mrd.mr_len  = size;
    arg = MEMRANGE_SET_REMOVE;

    if (mem_range_attr_set(&mrd, &arg))
        return RM_ERROR;

    return RM_OK;
}


/*
 * The current debug level is used to determine if certain debug messages
 * are printed to the system console/log files or not. It defaults to the
 * highest debug level, i.e. the lowest debug output.
 */

U032 cur_debuglevel = 0xffffffff;

void os_dbg_init(void)
{
    U032 new_debuglevel;

    if (rm_read_registry_dword(NULL, NULL, "NVreg", "ResmanDebugLevel",
            &new_debuglevel) == RM_OK) {
        if (new_debuglevel != 0xffffffff)
            cur_debuglevel = new_debuglevel;
    }
}

void os_dbg_set_level(U032 new_debuglevel)
{
    cur_debuglevel = new_debuglevel;
}

void NV_STACKWATCH_CALLBACK os_dbg_breakpoint(void)
{
#ifdef DEBUG
    /* FreeBSD 5.x kdb */
    kdb_enter("os_dbg_breakpoint()");
#endif
}

void out_string(const char *message)
{
#if defined(DEBUG) || defined(QA_BUILD)
    printf("%s", message);
#endif
}

static char nv_error_string[MAX_ERROR_STRING];

int nv_printf(
    U032 debuglevel,
    const char *format,
    ...
)
{
    char *message = nv_error_string;
    va_list arglist;
    int chars_written = 0;

    if (debuglevel >= ((cur_debuglevel >> 4) & 3)) {
        va_start(arglist, format);
        chars_written = vsprintf(message, format, arglist);
        va_end(arglist);
        printf("%s", message);
    }

    return chars_written;
}

int nv_snprintf(
    char *buf,
    unsigned int size,
    const char *fmt,
    ...
)
{
    va_list arglist;
    int chars_written;

    va_start(arglist, fmt);
    chars_written = vsnprintf(buf, size, fmt, arglist);
    va_end(arglist);

    return chars_written;
}

void nv_os_log(
    int loglevel,
    const char *fmt,
    void *ap
)
{
    int l;
    sprintf(nv_error_string, "NVRM: ");
    l = strlen(nv_error_string);
    vsnprintf(nv_error_string + l, MAX_ERROR_STRING - l, fmt, ap);
    printf("%s", nv_error_string);
}

S032 os_mem_cmp(
    const U008 *buf0,
    const U008 *buf1,
    U032 length
)
{
    return memcmp(buf0, buf1, length);
}

U008* os_mem_copy(
    U008 *dst,
    const U008 *src,
    U032 length
)
{
#if defined(NVCPU_X86_64)
    uint32_t i;
    for (i = 0; i < length; i++) dst[i] = src[i];
    return dst;
#else
    return memcpy(dst, src, length);
#endif
}

RM_STATUS os_memcpy_from_user(
    void *dst,
    const void *src,
    U032 length
)
{
    if (src < (void *) VM_MAXUSER_ADDRESS)
        return copyin(src, dst, length)  ? RM_ERR_INVALID_POINTER : RM_OK;

    return os_mem_copy(dst, src, length) ? RM_ERR_INVALID_POINTER : RM_OK;
}

RM_STATUS os_memcpy_to_user(
    void *dst,
    const void *src,
    U032 length
)
{
    if (dst < (void *) VM_MAXUSER_ADDRESS)
        return copyout(src, dst, length) ? RM_ERR_INVALID_POINTER : RM_OK;

    return os_mem_copy(dst, src, length) ? RM_ERR_INVALID_POINTER : RM_OK;
}

void* os_mem_set(
    void *b,
    U008 c,
    U032 length
)
{
    return memset(b, c, length);
}

S032 os_string_compare(
    const U008 *s1,
    const U008 *s2
)
{
    return strcmp(s1, s2);
}

U008* os_string_copy(
    U008 *dst,
    const U008 *src
)
{
    return strcpy(dst, src);
}

U032 os_string_length(const U008* s)
{
    return strlen(s);
}

RM_STATUS os_strncpy_from_user(
    U008 *dst,
    const U008 *src,
    U032 n
)
{
    return copyinstr(src, dst, n, NULL) ? RM_ERR_INVALID_POINTER : RM_OK;
}

U032 os_get_page_size(void)
{
    return PAGE_SIZE;
}

NvU64 os_get_page_mask(void)
{
    /*
     * On FreeBSD, PAGE_MASK means (PAGE_SIZE - 1); on Linux it means the
     * opposite, ~(PAGE_SIZE - 1); that is what this function is expected
     * to return.
     */
    return ~PAGE_MASK;
}

NvU64 os_get_system_memory_size(void)
{
    return ((NvU64)physmem * PAGE_SIZE) / RM_PAGE_SIZE;
}

U032 os_get_cpu_count(void)
{
    return mp_ncpus;
}

RM_STATUS os_flush_cpu_cache(void)
{
    /*
     * XXX This will do for now, but this may need to be extended to make
     * IPI calls (flushing all caches).
     */
    __asm__ __volatile__("wbinvd": : :"memory");
    return RM_OK;
}

void os_flush_cpu_write_combine_buffer(void)
{
    __asm__ __volatile__("sfence": : :"memory");
}

RM_STATUS os_raise_smp_barrier(void)
{
    return RM_OK;
}

RM_STATUS os_clear_smp_barrier(void)
{
    return RM_OK;
}

struct os_mutex {
    struct mtx mutex_mtx;
    /* XXX lockmgr broken w/ 5.2-current KSE */
    struct cv mutex_wait;
    int refcnt;
};

RM_STATUS os_alloc_sema(void **semaphore)
{
    struct os_mutex *mtx;
    RM_STATUS status;

    if ((status = os_alloc_mem((void **) &mtx, sizeof(struct os_mutex))
            != RM_OK))
        return status;

    mtx_init(&mtx->mutex_mtx, "rm.mutex_mtx", NULL, MTX_SPIN | MTX_RECURSE);
    cv_init(&mtx->mutex_wait, "rm.mutex_wait");

    mtx->refcnt = 1;
    *semaphore = (void *) mtx;

    return RM_OK;
}

RM_STATUS os_free_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_destroy(&mtx->mutex_mtx);
    /* XXX lockmgr broken w/ 5.2-current KSE */
    cv_destroy(&mtx->mutex_wait);

    os_free_mem(semaphore);

    return RM_OK;
}

RM_STATUS os_acquire_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock_spin(&mtx->mutex_mtx);
    if (mtx->refcnt > 0)
        rm_disable_interrupts(NULL);
    mtx->refcnt--;
    if (mtx->refcnt < 0)
        cv_wait(&mtx->mutex_wait, &mtx->mutex_mtx);
    mtx_unlock_spin(&mtx->mutex_mtx);

    return RM_OK;
}

BOOL os_cond_acquire_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock_spin(&mtx->mutex_mtx);
    if (mtx->refcnt < 1) {
        mtx_unlock_spin(&mtx->mutex_mtx);
        return FALSE;
    } else {
        rm_disable_interrupts(NULL);
        mtx->refcnt--;
        mtx_unlock_spin(&mtx->mutex_mtx);
    }

    return TRUE;
}

RM_STATUS os_release_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock_spin(&mtx->mutex_mtx);
    if (mtx->refcnt < 0)
        cv_signal(&mtx->mutex_wait);
    if (!mtx->refcnt)
        rm_enable_interrupts(NULL);
    mtx->refcnt++;
    mtx_unlock_spin(&mtx->mutex_mtx);

    return RM_OK;
}

BOOL os_is_acquired_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;
    return (mtx->refcnt < 1);
}

BOOL os_pat_supported(void)
{
    /*
     * FreeBSD has no native PAT support and there's no good
     * way to implement it privately as we do on Linux.
     */
    return FALSE;
}

void* NV_STACKWATCH_CALLBACK os_get_stack_start(void *stack_pointer)
{
    struct thread *td;
#if defined(NVCPU_X86_64)
    __asm __volatile__("movq %%gs:0,%0" : "=r" (td));
#elif defined(NVCPU_X86)
    __asm __volatile__("movl %%fs:0,%0" : "=r" (td));
#endif
    return (void *)cpu_getstack(td);
}

NvU64 os_get_current_pdpte(U032 address)
{
    return 0;
}

RM_STATUS os_set_mlock_capability()
{
    return (RM_ERROR);
}

S032 os_mlock_user_memory(
    void *address,
    U032 length
)
{
    return -1;
}

S032 os_munlock_user_memory(
    void *address,
    U032 length
)
{
    return -1;
}

RM_STATUS NV_API_CALL os_check_process_map_limit(
    NvU64 proc_max_map_count
)
{
    return (RM_ERROR);
}

void NV_API_CALL os_register_compatible_ioctl(U032 cmd, U032 size)
{
}

void NV_API_CALL os_unregister_compatible_ioctl(U032 cmd, U032 size)
{
}
