// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_CONTENT_RELATIONSHIP_VERIFICATION_DIGITAL_ASSET_LINKS_HANDLER_H_
#define COMPONENTS_CONTENT_RELATIONSHIP_VERIFICATION_DIGITAL_ASSET_LINKS_HANDLER_H_

#include <map>
#include <set>
#include <string>

#include "base/functional/callback.h"
#include "base/memory/weak_ptr.h"
#include "services/data_decoder/public/cpp/data_decoder.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace content {
class WebContents;
}  // namespace content

namespace network {
class SharedURLLoaderFactory;
class SimpleURLLoader;
}  // namespace network

namespace url {
class Origin;
}  // namespace url

namespace content_relationship_verification {

extern const char kDigitalAssetLinksCheckResponseKeyLinked[];

// GENERATED_JAVA_ENUM_PACKAGE: (
//   org.chromium.components.content_relationship_verification)
enum class RelationshipCheckResult {
  kSuccess = 0,
  kFailure,
  kNoConnection,
};

using RelationshipCheckResultCallback =
    base::OnceCallback<void(RelationshipCheckResult)>;

// A handler class for sending REST API requests to DigitalAssetLinks web
// end point. See
// https://developers.google.com/digital-asset-links/v1/getting-started
// for details of usage and APIs. These APIs are used to verify declared
// relationships between different asset types like web domains or Android apps.
// The lifecycle of this handler will be governed by the owner.
// The WebContents are used for logging console messages.
class DigitalAssetLinksHandler {
 public:
  // Optionally include |web_contents| for logging error messages to DevTools.
  explicit DigitalAssetLinksHandler(
      scoped_refptr<network::SharedURLLoaderFactory> factory,
      content::WebContents* web_contents = nullptr);

  DigitalAssetLinksHandler(const DigitalAssetLinksHandler&) = delete;
  DigitalAssetLinksHandler& operator=(const DigitalAssetLinksHandler&) = delete;

  ~DigitalAssetLinksHandler();

  // Checks whether the given "relationship" has been declared by the target
  // |web_domain| for the source Android app which is uniquely defined by the
  // |package| and SHA256 |fingerprint| (a string with 32 hexadecimals with :
  // between) given. Any error in the string params here will result in a bad
  // request and a nullptr response to the callback.
  //
  // Calling this multiple times on the same handler will cancel the previous
  // checks. See
  // https://developers.google.com/digital-asset-links/reference/rest/v1/assetlinks/check
  // for details.
  bool CheckDigitalAssetLinkRelationshipForAndroidApp(
      const url::Origin& web_domain,
      const std::string& relationship,
      std::vector<std::string> fingerprints,
      const std::string& package,
      RelationshipCheckResultCallback callback);

  // Checks if the asset links for |web_domain| allow pages controlled by
  // |manifest_url| to query for WebAPKs generated by |web_domain|.
  // TODO(rayankans): Link to the developer blog when published.
  bool CheckDigitalAssetLinkRelationshipForWebApk(
      const url::Origin& web_domain,
      const std::string& manifest_url,
      RelationshipCheckResultCallback callback);

 private:
  // Generic DAL verifier. Checks whether the given |relationship| has been
  // declared by the target |web_domain| using the values in |target_values|.
  // We require a match for every entry in the |target_values| map, but within
  // the entry, we require a match only for one value in the set.
  // For example, |target_values| may contain an entry with key="site" and
  // values={"https://example1.com", "https://example2.com"}. In order to
  // validate, the manifest must have an entry with key="site" with one or more
  // of the URLs as the value.
  bool CheckDigitalAssetLinkRelationship(
      const url::Origin& web_domain,
      const std::string& relationship,
      absl::optional<std::vector<std::string>> fingerprints,
      const std::map<std::string, std::set<std::string>>& target_values,
      RelationshipCheckResultCallback callback);

  void OnURLLoadComplete(
      std::string relationship,
      absl::optional<std::vector<std::string>> fingerprints,
      std::map<std::string, std::set<std::string>> target_values,
      std::unique_ptr<std::string> response_body);

  // Callback for the DataDecoder.
  void OnJSONParseResult(
      std::string relationship,
      absl::optional<std::vector<std::string>> fingerprints,
      std::map<std::string, std::set<std::string>> target_values,
      data_decoder::DataDecoder::ValueOrError result);

  scoped_refptr<network::SharedURLLoaderFactory> shared_url_loader_factory_;

  std::unique_ptr<network::SimpleURLLoader> url_loader_;

  // The per request callback for receiving a URLFetcher result. This gets
  // reset every time we get a new CheckDigitalAssetLinkRelationship call.
  RelationshipCheckResultCallback callback_;

  base::WeakPtr<content::WebContents> web_contents_;

  base::WeakPtrFactory<DigitalAssetLinksHandler> weak_ptr_factory_{this};
};

}  // namespace content_relationship_verification

#endif  // COMPONENTS_CONTENT_RELATIONSHIP_VERIFICATION_DIGITAL_ASSET_LINKS_HANDLER_H_
