// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/autofill/core/browser/webdata/payments/payments_autofill_table.h"

#include <map>
#include <memory>
#include <set>
#include <string>
#include <string_view>
#include <utility>

#include "base/command_line.h"
#include "base/files/file_util.h"
#include "base/files/scoped_temp_dir.h"
#include "base/strings/strcat.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "base/test/scoped_feature_list.h"
#include "base/time/time.h"
#include "base/uuid.h"
#include "build/build_config.h"
#include "components/autofill/core/browser/autofill_test_utils.h"
#include "components/autofill/core/browser/autofill_type.h"
#include "components/autofill/core/browser/data_model/autofill_metadata.h"
#include "components/autofill/core/browser/data_model/autofill_offer_data.h"
#include "components/autofill/core/browser/data_model/autofill_wallet_usage_data.h"
#include "components/autofill/core/browser/data_model/bank_account.h"
#include "components/autofill/core/browser/data_model/credit_card.h"
#include "components/autofill/core/browser/data_model/credit_card_cloud_token_data.h"
#include "components/autofill/core/browser/field_types.h"
#include "components/autofill/core/browser/payments/payments_customer_data.h"
#include "components/autofill/core/browser/test_autofill_clock.h"
#include "components/autofill/core/browser/webdata/autofill_change.h"
#include "components/autofill/core/common/autofill_clock.h"
#include "components/autofill/core/common/autofill_constants.h"
#include "components/autofill/core/common/autofill_features.h"
#include "components/autofill/core/common/autofill_payments_features.h"
#include "components/autofill/core/common/autofill_switches.h"
#include "components/autofill/core/common/autofill_util.h"
#include "components/os_crypt/sync/os_crypt_mocker.h"
#include "components/webdata/common/web_database.h"
#include "sql/statement.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "url/origin.h"

using base::Time;
using testing::ElementsAre;
using testing::UnorderedElementsAre;

namespace autofill {

class PaymentsAutofillTableTest : public testing::Test {
 protected:
  void SetUp() override {
    OSCryptMocker::SetUp();
    ASSERT_TRUE(temp_dir_.CreateUniqueTempDir());
    file_ = temp_dir_.GetPath().AppendASCII("TestWebDatabase");

    table_ = std::make_unique<PaymentsAutofillTable>();
    db_ = std::make_unique<WebDatabase>();
    db_->AddTable(table_.get());
    ASSERT_EQ(sql::INIT_OK, db_->Init(file_));
  }

  void TearDown() override { OSCryptMocker::TearDown(); }

  // Get date_modifed `column` of `table_name` with specific `instrument_id` or
  // `guid`.
  time_t GetDateModified(std::string_view table_name,
                         std::string_view column,
                         absl::variant<std::string, int64_t> id) {
    sql::Statement s(db_->GetSQLConnection()->GetUniqueStatement(
        base::StrCat({"SELECT ", column, " FROM ", table_name, " WHERE ",
                      absl::holds_alternative<std::string>(id)
                          ? "guid"
                          : "instrument_id",
                      " = ?"})
            .c_str()));
    if (const std::string* guid = absl::get_if<std::string>(&id)) {
      s.BindString(0, *guid);
    } else {
      s.BindInt64(0, absl::get<int64_t>(id));
    }
    EXPECT_TRUE(s.Step());
    return s.ColumnInt64(0);
  }

  base::FilePath file_;
  base::ScopedTempDir temp_dir_;
  std::unique_ptr<PaymentsAutofillTable> table_;
  std::unique_ptr<WebDatabase> db_;
};

TEST_F(PaymentsAutofillTableTest, Iban) {
  // Add a valid IBAN.
  Iban iban;
  std::string guid = base::Uuid::GenerateRandomV4().AsLowercaseString();
  iban.set_identifier(Iban::Guid(guid));
  iban.SetRawInfo(IBAN_VALUE, u"IE12 BOFI 9000 0112 3456 78");
  iban.set_nickname(u"My doctor's IBAN");

  EXPECT_TRUE(table_->AddLocalIban(iban));

  // Get the inserted IBAN.
  std::unique_ptr<Iban> db_iban = table_->GetLocalIban(iban.guid());
  ASSERT_TRUE(db_iban);
  EXPECT_EQ(guid, db_iban->guid());
  sql::Statement s_work(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid, use_count, use_date, "
      "value_encrypted, nickname FROM local_ibans WHERE guid = ?"));
  s_work.BindString(0, iban.guid());
  ASSERT_TRUE(s_work.is_valid());
  ASSERT_TRUE(s_work.Step());
  EXPECT_FALSE(s_work.Step());

  // Add another valid IBAN.
  Iban another_iban;
  std::string another_guid = base::Uuid::GenerateRandomV4().AsLowercaseString();
  another_iban.set_identifier(Iban::Guid(another_guid));
  another_iban.SetRawInfo(IBAN_VALUE, u"DE91 1000 0000 0123 4567 89");
  another_iban.set_nickname(u"My brother's IBAN");

  EXPECT_TRUE(table_->AddLocalIban(another_iban));

  db_iban = table_->GetLocalIban(another_iban.guid());
  ASSERT_TRUE(db_iban);

  EXPECT_EQ(another_guid, db_iban->guid());
  sql::Statement s_target(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid, use_count, use_date, "
      "value_encrypted, nickname FROM local_ibans WHERE guid = ?"));
  s_target.BindString(0, another_iban.guid());
  ASSERT_TRUE(s_target.is_valid());
  ASSERT_TRUE(s_target.Step());
  EXPECT_FALSE(s_target.Step());

  // Update the another_iban.
  another_iban.SetRawInfo(IBAN_VALUE, u"GB98 MIDL 0700 9312 3456 78");
  another_iban.set_nickname(u"My teacher's IBAN");
  EXPECT_TRUE(table_->UpdateLocalIban(another_iban));
  db_iban = table_->GetLocalIban(another_iban.guid());
  ASSERT_TRUE(db_iban);
  EXPECT_EQ(another_guid, db_iban->guid());
  sql::Statement s_target_updated(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid, use_count, use_date, "
      "value_encrypted, nickname FROM local_ibans WHERE guid = ?"));
  s_target_updated.BindString(0, another_iban.guid());
  ASSERT_TRUE(s_target_updated.is_valid());
  ASSERT_TRUE(s_target_updated.Step());
  EXPECT_FALSE(s_target_updated.Step());

  // Remove the 'Target' IBAN.
  EXPECT_TRUE(table_->RemoveLocalIban(another_iban.guid()));
  db_iban = table_->GetLocalIban(another_iban.guid());
  EXPECT_FALSE(db_iban);
}

// Test that masked IBANs can be added and loaded successfully.
TEST_F(PaymentsAutofillTableTest, MaskedServerIban) {
  Iban iban_0 = test::GetServerIban();
  Iban iban_1 = test::GetServerIban2();
  Iban iban_2 = test::GetServerIban3();
  std::vector<Iban> ibans = {iban_0, iban_1, iban_2};

  table_->SetServerIbansForTesting(ibans);

  std::vector<std::unique_ptr<Iban>> masked_server_ibans;
  EXPECT_TRUE(table_->GetServerIbans(masked_server_ibans));
  EXPECT_EQ(3U, masked_server_ibans.size());
  EXPECT_THAT(ibans, UnorderedElementsAre(*masked_server_ibans[0],
                                          *masked_server_ibans[1],
                                          *masked_server_ibans[2]));
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerIbansMetadata(outputs));
  ASSERT_FALSE(outputs.empty());
}

// Test that masked IBANs can be added and loaded successfully without updating
// their metadata.
TEST_F(PaymentsAutofillTableTest, MaskedServerIbanMetadataNotUpdated) {
  std::vector<Iban> ibans = {test::GetServerIban()};

  table_->SetServerIbansData(ibans);

  std::vector<std::unique_ptr<Iban>> masked_server_ibans;
  EXPECT_TRUE(table_->GetServerIbans(masked_server_ibans));
  EXPECT_EQ(1U, masked_server_ibans.size());
  EXPECT_THAT(ibans, UnorderedElementsAre(*masked_server_ibans[0]));
}

TEST_F(PaymentsAutofillTableTest, CreditCard) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  // Add a 'Work' credit card.
  CreditCard work_creditcard;
  work_creditcard.set_origin("https://www.example.com/");
  work_creditcard.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  work_creditcard.SetRawInfo(CREDIT_CARD_NUMBER, u"1234567890123456");
  work_creditcard.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"04");
  work_creditcard.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2013");
  work_creditcard.SetNickname(u"Corporate card");
  work_creditcard.set_cvc(u"123");

  Time pre_creation_time = AutofillClock::Now();
  EXPECT_TRUE(table_->AddCreditCard(work_creditcard));
  Time post_creation_time = AutofillClock::Now();

  // Get the 'Work' credit card.
  std::unique_ptr<CreditCard> db_creditcard =
      table_->GetCreditCard(work_creditcard.guid());
  ASSERT_TRUE(db_creditcard);
  EXPECT_EQ(work_creditcard, *db_creditcard);
  // Check GetCreditCard statement
  sql::Statement s_credit_card_work(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid, name_on_card, expiration_month, expiration_year, "
      "card_number_encrypted, date_modified, nickname "
      "FROM credit_cards WHERE guid=?"));
  s_credit_card_work.BindString(0, work_creditcard.guid());
  ASSERT_TRUE(s_credit_card_work.is_valid());
  ASSERT_TRUE(s_credit_card_work.Step());
  EXPECT_GE(s_credit_card_work.ColumnInt64(5), pre_creation_time.ToTimeT());
  EXPECT_LE(s_credit_card_work.ColumnInt64(5), post_creation_time.ToTimeT());
  EXPECT_FALSE(s_credit_card_work.Step());
  // Check GetLocalStoredCvc statement
  sql::Statement s_cvc_work(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT value_encrypted,  last_updated_timestamp "
      "FROM local_stored_cvc WHERE guid=?"));
  s_cvc_work.BindString(0, work_creditcard.guid());
  ASSERT_TRUE(s_cvc_work.is_valid());
  ASSERT_TRUE(s_cvc_work.Step());
  EXPECT_GE(s_cvc_work.ColumnInt64(1), pre_creation_time.ToTimeT());
  EXPECT_LE(s_cvc_work.ColumnInt64(1), post_creation_time.ToTimeT());
  EXPECT_FALSE(s_cvc_work.Step());

  // Add a 'Target' credit card.
  CreditCard target_creditcard;
  target_creditcard.set_origin(std::string());
  target_creditcard.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  target_creditcard.SetRawInfo(CREDIT_CARD_NUMBER, u"1111222233334444");
  target_creditcard.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"06");
  target_creditcard.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2012");
  target_creditcard.SetNickname(u"Grocery card");
  target_creditcard.set_cvc(u"234");

  pre_creation_time = AutofillClock::Now();
  EXPECT_TRUE(table_->AddCreditCard(target_creditcard));
  post_creation_time = AutofillClock::Now();
  db_creditcard = table_->GetCreditCard(target_creditcard.guid());
  ASSERT_TRUE(db_creditcard);
  EXPECT_EQ(target_creditcard, *db_creditcard);
  // Check GetCreditCard statement.
  sql::Statement s_credit_card_target(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT guid, name_on_card, expiration_month, expiration_year, "
          "card_number_encrypted, date_modified, nickname "
          "FROM credit_cards WHERE guid=?"));
  s_credit_card_target.BindString(0, target_creditcard.guid());
  ASSERT_TRUE(s_credit_card_target.is_valid());
  ASSERT_TRUE(s_credit_card_target.Step());
  EXPECT_GE(s_credit_card_target.ColumnInt64(5), pre_creation_time.ToTimeT());
  EXPECT_LE(s_credit_card_target.ColumnInt64(5), post_creation_time.ToTimeT());
  EXPECT_FALSE(s_credit_card_target.Step());
  // Check GetLocalStoredCvc statement
  sql::Statement s_cvc_target(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT value_encrypted,  last_updated_timestamp "
      "FROM local_stored_cvc WHERE guid=?"));
  s_cvc_target.BindString(0, target_creditcard.guid());
  ASSERT_TRUE(s_cvc_target.is_valid());
  ASSERT_TRUE(s_cvc_target.Step());
  EXPECT_GE(s_cvc_target.ColumnInt64(1), pre_creation_time.ToTimeT());
  EXPECT_LE(s_cvc_target.ColumnInt64(1), post_creation_time.ToTimeT());
  EXPECT_FALSE(s_cvc_target.Step());

  // Update the 'Target' credit card.
  target_creditcard.set_origin("Interactive Autofill dialog");
  target_creditcard.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Charles Grady");
  target_creditcard.SetNickname(u"Supermarket");
  target_creditcard.set_cvc(u"234");
  Time pre_modification_time = AutofillClock::Now();
  EXPECT_TRUE(table_->UpdateCreditCard(target_creditcard));
  Time post_modification_time = AutofillClock::Now();
  db_creditcard = table_->GetCreditCard(target_creditcard.guid());
  ASSERT_TRUE(db_creditcard);
  EXPECT_EQ(target_creditcard, *db_creditcard);
  sql::Statement s_target_updated(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid, name_on_card, expiration_month, expiration_year, "
      "card_number_encrypted, date_modified, nickname "
      "FROM credit_cards WHERE guid=?"));
  s_target_updated.BindString(0, target_creditcard.guid());
  ASSERT_TRUE(s_target_updated.is_valid());
  ASSERT_TRUE(s_target_updated.Step());
  EXPECT_GE(s_target_updated.ColumnInt64(5), pre_modification_time.ToTimeT());
  EXPECT_LE(s_target_updated.ColumnInt64(5), post_modification_time.ToTimeT());
  EXPECT_FALSE(s_target_updated.Step());

  // Remove the 'Target' credit card.
  EXPECT_TRUE(table_->RemoveCreditCard(target_creditcard.guid()));
  db_creditcard = table_->GetCreditCard(target_creditcard.guid());
  EXPECT_FALSE(db_creditcard);
}

TEST_F(PaymentsAutofillTableTest, AddCreditCardCvcWithFlagOff) {
  base::test::ScopedFeatureList features;
  features.InitAndDisableFeature(features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card = test::WithCvc(test::GetCreditCard());
  EXPECT_TRUE(table_->AddCreditCard(card));
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"", db_card->cvc());

  card.set_cvc(u"234");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"", db_card->cvc());
}

// Tests that verify ClearLocalPaymentMethodsData function working as expected.
TEST_F(PaymentsAutofillTableTest, ClearLocalPaymentMethodsData) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card = test::WithCvc(test::GetCreditCard());
  EXPECT_TRUE(table_->AddCreditCard(card));
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(card.cvc(), db_card->cvc());
  Iban iban = test::GetLocalIban();
  EXPECT_TRUE(table_->AddLocalIban(iban));

  // After calling ClearLocalPaymentMethodsData, the local_stored_cvc,
  // credit_cards, and local_ibans tables should be empty.
  table_->ClearLocalPaymentMethodsData();
  EXPECT_FALSE(table_->GetCreditCard(card.guid()));
  EXPECT_FALSE(table_->GetLocalIban(iban.guid()));
  sql::Statement s(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid FROM local_stored_cvc WHERE guid=?"));
  s.BindString(0, card.guid());
  ASSERT_TRUE(s.is_valid());
  EXPECT_FALSE(s.Step());
}

// Tests that adding credit card with cvc, get credit card with cvc and update
// credit card with only cvc change will not update credit_card table
// modification_date.
TEST_F(PaymentsAutofillTableTest, CreditCardCvc) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  const base::Time arbitrary_time = base::Time::FromSecondsSinceUnixEpoch(25);
  // Create the test clock and set the time to a specific value.
  TestAutofillClock test_clock;
  test_clock.SetNow(arbitrary_time);
  CreditCard card = test::WithCvc(test::GetCreditCard());
  EXPECT_TRUE(table_->AddCreditCard(card));

  // Get the credit card, cvc should match.
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(card.cvc(), db_card->cvc());

  // Verify last_updated_timestamp in local_stored_cvc table is set correctly.
  EXPECT_EQ(GetDateModified("local_stored_cvc", "last_updated_timestamp",
                            card.guid()),
            arbitrary_time.ToTimeT());

  // Set the current time to another value.
  const base::Time some_later_time =
      base::Time::FromSecondsSinceUnixEpoch(1000);
  test_clock.SetNow(some_later_time);

  // Update the credit card but CVC is same.
  card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Charles Grady");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  // credit_card table date_modified should be updated.
  EXPECT_EQ(GetDateModified("credit_cards", "date_modified", card.guid()),
            some_later_time.ToTimeT());
  // local_stored_cvc table timestamp should not be updated.
  EXPECT_EQ(GetDateModified("local_stored_cvc", "last_updated_timestamp",
                            card.guid()),
            arbitrary_time.ToTimeT());

  // Set the current time to another value.
  const base::Time much_later_time =
      base::Time::FromSecondsSinceUnixEpoch(5000);
  test_clock.SetNow(much_later_time);

  // Update the credit card and CVC is different.
  card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  card.set_cvc(u"234");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  db_card = table_->GetCreditCard(card.guid());
  // CVC should be updated to new CVC.
  EXPECT_EQ(u"234", db_card->cvc());
  // local_stored_cvc table timestamp should be updated.
  EXPECT_EQ(GetDateModified("local_stored_cvc", "last_updated_timestamp",
                            card.guid()),
            much_later_time.ToTimeT());

  // Remove the credit card. It should also remove cvc from local_stored_cvc
  // table.
  EXPECT_TRUE(table_->RemoveCreditCard(card.guid()));
  sql::Statement cvc_removed_statement(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT guid FROM local_stored_cvc WHERE guid=?"));
  cvc_removed_statement.BindString(0, card.guid());
  ASSERT_TRUE(cvc_removed_statement.is_valid());
  EXPECT_FALSE(cvc_removed_statement.Step());
}

// Tests that update a credit card CVC that doesn't have CVC set initially
// inserts a new CVC record.
TEST_F(PaymentsAutofillTableTest, UpdateCreditCardCvc_Add) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card = test::GetCreditCard();
  ASSERT_TRUE(card.cvc().empty());
  ASSERT_TRUE(table_->AddCreditCard(card));

  // Update the credit card CVC, we should expect success and CVC gets updated.
  card.set_cvc(u"123");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"123", db_card->cvc());
}

// Tests that updating a credit card CVC that is different from CVC set
// initially.
TEST_F(PaymentsAutofillTableTest, UpdateCreditCardCvc_Update) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card = test::GetCreditCard();
  ASSERT_TRUE(card.cvc().empty());
  ASSERT_TRUE(table_->AddCreditCard(card));

  // INSERT
  // Updating a card that doesn't have a CVC is the same as inserting a new CVC
  // record.
  card.set_cvc(u"123");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"123", db_card->cvc());

  // UPDATE
  // Update the credit card CVC.
  card.set_cvc(u"234");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"234", db_card->cvc());
}

// Tests that updating a credit card CVC with empty CVC will delete CVC
// record. This is necessary because if inserting a CVC, UPDATE is chosen over
// INSERT, it will causes a crash.
TEST_F(PaymentsAutofillTableTest, UpdateCreditCardCvc_Delete) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card = test::GetCreditCard();
  ASSERT_TRUE(card.cvc().empty());
  ASSERT_TRUE(table_->AddCreditCard(card));

  // INSERT
  // Updating a card that doesn't have a CVC is the same as inserting a new CVC
  // record.
  card.set_cvc(u"123");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  std::unique_ptr<CreditCard> db_card = table_->GetCreditCard(card.guid());
  EXPECT_EQ(u"123", db_card->cvc());

  // DELETE
  // Updating a card with empty CVC is the same as deleting the CVC record.
  card.set_cvc(u"");
  EXPECT_TRUE(table_->UpdateCreditCard(card));
  sql::Statement cvc_statement(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid FROM local_stored_cvc WHERE guid=?"));
  cvc_statement.BindString(0, card.guid());
  ASSERT_TRUE(cvc_statement.is_valid());
  EXPECT_FALSE(cvc_statement.Step());
}

TEST_F(PaymentsAutofillTableTest, LocalCvcs_ClearAll) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  CreditCard card_1 = test::WithCvc(test::GetCreditCard());
  CreditCard card_2 = test::WithCvc(test::GetCreditCard2());
  EXPECT_TRUE(table_->AddCreditCard(card_1));
  EXPECT_TRUE(table_->AddCreditCard(card_2));

  // Get the credit cards and the CVCs should match.
  std::unique_ptr<CreditCard> db_card_1 = table_->GetCreditCard(card_1.guid());
  std::unique_ptr<CreditCard> db_card_2 = table_->GetCreditCard(card_2.guid());
  EXPECT_EQ(card_1.cvc(), db_card_1->cvc());
  EXPECT_EQ(card_2.cvc(), db_card_2->cvc());

  // Clear all local CVCs from the web database.
  table_->ClearLocalCvcs();

  sql::Statement cvc_statement(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT guid FROM local_stored_cvc WHERE guid=?"));

  // Verify `card_1` CVC is deleted.
  cvc_statement.BindString(0, card_1.guid());
  ASSERT_TRUE(cvc_statement.is_valid());
  EXPECT_FALSE(cvc_statement.Step());
  cvc_statement.Reset(/*clear_bound_vars=*/true);

  // Verify `card_2` CVC is deleted.
  cvc_statement.BindString(0, card_2.guid());
  ASSERT_TRUE(cvc_statement.is_valid());
  EXPECT_FALSE(cvc_statement.Step());
}

// Tests that verify add, update and clear server cvc function working as
// expected.
TEST_F(PaymentsAutofillTableTest, ServerCvc) {
  const base::Time kArbitraryTime = base::Time::FromSecondsSinceUnixEpoch(25);
  int64_t kInstrumentId = 111111111111;
  const std::u16string kCvc = u"123";
  const ServerCvc kServerCvc{kInstrumentId, kCvc, kArbitraryTime};
  EXPECT_TRUE(table_->AddServerCvc(kServerCvc));
  // Database does not allow adding same instrument_id twice.
  EXPECT_FALSE(table_->AddServerCvc(kServerCvc));
  EXPECT_THAT(table_->GetAllServerCvcs(),
              UnorderedElementsAre(testing::Pointee(kServerCvc)));

  const base::Time kSomeLaterTime = base::Time::FromSecondsSinceUnixEpoch(1000);
  const std::u16string kNewCvc = u"234";
  const ServerCvc kNewServerCvcUnderSameInstrumentId{kInstrumentId, kNewCvc,
                                                     kSomeLaterTime};
  EXPECT_TRUE(table_->UpdateServerCvc(kNewServerCvcUnderSameInstrumentId));
  EXPECT_THAT(table_->GetAllServerCvcs(),
              UnorderedElementsAre(
                  testing::Pointee(kNewServerCvcUnderSameInstrumentId)));

  // Remove the server cvc. It should also remove cvc from server_stored_cvc
  // table.
  EXPECT_TRUE(table_->RemoveServerCvc(kInstrumentId));
  EXPECT_TRUE(table_->GetAllServerCvcs().empty());

  // Remove non-exist cvc will return false.
  EXPECT_FALSE(table_->RemoveServerCvc(kInstrumentId));

  // Clear the server_stored_cvc table.
  table_->AddServerCvc(kServerCvc);
  EXPECT_TRUE(table_->ClearServerCvcs());
  EXPECT_TRUE(table_->GetAllServerCvcs().empty());

  // Clear the server_stored_cvc table when table is empty will return false.
  EXPECT_FALSE(table_->ClearServerCvcs());
}

// Tests that verify reconcile server cvc function working as expected.
TEST_F(PaymentsAutofillTableTest, ReconcileServerCvcs) {
  const base::Time kArbitraryTime = base::Time::FromSecondsSinceUnixEpoch(25);
  // Add 2 server credit cards.
  CreditCard card1 = test::WithCvc(test::GetMaskedServerCard());
  CreditCard card2 = test::WithCvc(test::GetMaskedServerCard2());
  test::SetServerCreditCards(table_.get(), {card1, card2});

  // Add 1 server cvc that doesn't have a credit card associate with. We
  // should have 3 cvcs in server_stored_cvc table.
  EXPECT_TRUE(table_->AddServerCvc(ServerCvc{3333, u"456", kArbitraryTime}));
  EXPECT_EQ(3U, table_->GetAllServerCvcs().size());

  // After we reconcile server cvc, we should only see 2 cvcs in
  // server_stored_cvc table because obsolete cvc has been reconciled.
  EXPECT_TRUE(table_->ReconcileServerCvcs());
  EXPECT_EQ(2U, table_->GetAllServerCvcs().size());
}

TEST_F(PaymentsAutofillTableTest, AddFullServerCreditCard) {
  CreditCard credit_card;
  credit_card.set_record_type(CreditCard::RecordType::kFullServerCard);
  credit_card.set_server_id("server_id");
  credit_card.set_origin("https://www.example.com/");
  credit_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  credit_card.SetRawInfo(CREDIT_CARD_NUMBER, u"1234567890123456");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"04");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2013");

  EXPECT_TRUE(table_->AddFullServerCreditCard(credit_card));

  std::vector<std::unique_ptr<CreditCard>> outputs;
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(0, credit_card.Compare(*outputs[0]));
}

TEST_F(PaymentsAutofillTableTest, UpdateCreditCard) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  // Add a credit card to the db.
  CreditCard credit_card;
  credit_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  credit_card.SetRawInfo(CREDIT_CARD_NUMBER, u"1234567890123456");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"04");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2013");
  table_->AddCreditCard(credit_card);

  // Set a mocked value for the credit card's creation time.
  const time_t kMockCreationDate = AutofillClock::Now().ToTimeT() - 13;
  sql::Statement s_mock_creation_date(
      db_->GetSQLConnection()->GetUniqueStatement(
          "UPDATE credit_cards SET date_modified = ?"));
  ASSERT_TRUE(s_mock_creation_date.is_valid());
  s_mock_creation_date.BindInt64(0, kMockCreationDate);
  ASSERT_TRUE(s_mock_creation_date.Run());

  // Get the credit card.
  std::unique_ptr<CreditCard> db_credit_card =
      table_->GetCreditCard(credit_card.guid());
  ASSERT_TRUE(db_credit_card);
  EXPECT_EQ(credit_card, *db_credit_card);
  sql::Statement s_original(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_original.is_valid());
  ASSERT_TRUE(s_original.Step());
  EXPECT_EQ(kMockCreationDate, s_original.ColumnInt64(0));
  EXPECT_FALSE(s_original.Step());

  // Now, update the credit card and save the update to the database.
  // The modification date should change to reflect the update.
  credit_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"01");
  table_->UpdateCreditCard(credit_card);

  // Get the credit card.
  db_credit_card = table_->GetCreditCard(credit_card.guid());
  ASSERT_TRUE(db_credit_card);
  EXPECT_EQ(credit_card, *db_credit_card);
  sql::Statement s_updated(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_updated.is_valid());
  ASSERT_TRUE(s_updated.Step());
  EXPECT_LT(kMockCreationDate, s_updated.ColumnInt64(0));
  EXPECT_FALSE(s_updated.Step());

  // Set a mocked value for the credit card's modification time.
  const time_t mock_modification_date = AutofillClock::Now().ToTimeT() - 7;
  sql::Statement s_mock_modification_date(
      db_->GetSQLConnection()->GetUniqueStatement(
          "UPDATE credit_cards SET date_modified = ?"));
  ASSERT_TRUE(s_mock_modification_date.is_valid());
  s_mock_modification_date.BindInt64(0, mock_modification_date);
  ASSERT_TRUE(s_mock_modification_date.Run());

  // Finally, call into |UpdateCreditCard()| without changing the credit card.
  // The modification date should not change.
  table_->UpdateCreditCard(credit_card);

  // Get the credit card.
  db_credit_card = table_->GetCreditCard(credit_card.guid());
  ASSERT_TRUE(db_credit_card);
  EXPECT_EQ(credit_card, *db_credit_card);
  sql::Statement s_unchanged(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_unchanged.is_valid());
  ASSERT_TRUE(s_unchanged.Step());
  EXPECT_EQ(mock_modification_date, s_unchanged.ColumnInt64(0));
  EXPECT_FALSE(s_unchanged.Step());
}

TEST_F(PaymentsAutofillTableTest, UpdateCreditCardOriginOnly) {
  base::test::ScopedFeatureList features(
      features::kAutofillEnableCvcStorageAndFilling);
  // Add a credit card to the db.
  CreditCard credit_card;
  credit_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jack Torrance");
  credit_card.SetRawInfo(CREDIT_CARD_NUMBER, u"1234567890123456");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"04");
  credit_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2013");
  table_->AddCreditCard(credit_card);

  // Set a mocked value for the credit card's creation time.
  const time_t kMockCreationDate = AutofillClock::Now().ToTimeT() - 13;
  sql::Statement s_mock_creation_date(
      db_->GetSQLConnection()->GetUniqueStatement(
          "UPDATE credit_cards SET date_modified = ?"));
  ASSERT_TRUE(s_mock_creation_date.is_valid());
  s_mock_creation_date.BindInt64(0, kMockCreationDate);
  ASSERT_TRUE(s_mock_creation_date.Run());

  // Get the credit card.
  std::unique_ptr<CreditCard> db_credit_card =
      table_->GetCreditCard(credit_card.guid());
  ASSERT_TRUE(db_credit_card);
  EXPECT_EQ(credit_card, *db_credit_card);
  sql::Statement s_original(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_original.is_valid());
  ASSERT_TRUE(s_original.Step());
  EXPECT_EQ(kMockCreationDate, s_original.ColumnInt64(0));
  EXPECT_FALSE(s_original.Step());

  // Now, update just the credit card's origin and save the update to the
  // database.  The modification date should change to reflect the update.
  credit_card.set_origin("https://www.example.com/");
  table_->UpdateCreditCard(credit_card);

  // Get the credit card.
  db_credit_card = table_->GetCreditCard(credit_card.guid());
  ASSERT_TRUE(db_credit_card);
  EXPECT_EQ(credit_card, *db_credit_card);
  sql::Statement s_updated(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_updated.is_valid());
  ASSERT_TRUE(s_updated.Step());
  EXPECT_LT(kMockCreationDate, s_updated.ColumnInt64(0));
  EXPECT_FALSE(s_updated.Step());
}

TEST_F(PaymentsAutofillTableTest, RemoveAutofillDataModifiedBetween) {
  // Populate the credit_cards tables.
  ASSERT_TRUE(db_->GetSQLConnection()->Execute(
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000006', 17);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000006', '', 17);"
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000007', 27);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000007', '', 27);"
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000008', 37);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000008', '', 37);"
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000009', 47);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000009', '', 47);"
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000010', 57);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000010', '', 57);"
      "INSERT INTO credit_cards (guid, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000011', 67);"
      "INSERT INTO local_stored_cvc (guid, value_encrypted, "
      "last_updated_timestamp) "
      "VALUES('00000000-0000-0000-0000-000000000011', '', 67);"));

  // Remove all entries modified in the bounded time range [17,41).
  std::vector<std::unique_ptr<CreditCard>> credit_cards;
  table_->RemoveAutofillDataModifiedBetween(Time::FromTimeT(17),
                                            Time::FromTimeT(41), &credit_cards);

  // Three cards should have been removed.
  ASSERT_EQ(3UL, credit_cards.size());
  EXPECT_EQ("00000000-0000-0000-0000-000000000006", credit_cards[0]->guid());
  EXPECT_EQ("00000000-0000-0000-0000-000000000007", credit_cards[1]->guid());
  EXPECT_EQ("00000000-0000-0000-0000-000000000008", credit_cards[2]->guid());

  // Make sure the expected cards are still present.
  sql::Statement s_credit_cards_bounded(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT date_modified FROM credit_cards ORDER BY guid"));
  ASSERT_TRUE(s_credit_cards_bounded.is_valid());
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(47, s_credit_cards_bounded.ColumnInt64(0));
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(57, s_credit_cards_bounded.ColumnInt64(0));
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(67, s_credit_cards_bounded.ColumnInt64(0));
  EXPECT_FALSE(s_credit_cards_bounded.Step());

  // Make sure the expected card cvcs are still present.
  sql::Statement s_cvc_bounded(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT last_updated_timestamp FROM local_stored_cvc ORDER BY guid"));
  ASSERT_TRUE(s_cvc_bounded.is_valid());
  ASSERT_TRUE(s_cvc_bounded.Step());
  EXPECT_EQ(47, s_cvc_bounded.ColumnInt64(0));
  ASSERT_TRUE(s_cvc_bounded.Step());
  EXPECT_EQ(57, s_cvc_bounded.ColumnInt64(0));
  ASSERT_TRUE(s_cvc_bounded.Step());
  EXPECT_EQ(67, s_cvc_bounded.ColumnInt64(0));
  EXPECT_FALSE(s_cvc_bounded.Step());

  // Remove all entries modified on or after time 51 (unbounded range).
  table_->RemoveAutofillDataModifiedBetween(Time::FromTimeT(51), Time(),
                                            &credit_cards);

  // Two cards should have been removed.
  ASSERT_EQ(2UL, credit_cards.size());
  EXPECT_EQ("00000000-0000-0000-0000-000000000010", credit_cards[0]->guid());
  EXPECT_EQ("00000000-0000-0000-0000-000000000011", credit_cards[1]->guid());

  // Make sure the remaining card is the expected one.
  sql::Statement s_credit_cards_unbounded(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_credit_cards_unbounded.is_valid());
  ASSERT_TRUE(s_credit_cards_unbounded.Step());
  EXPECT_EQ(47, s_credit_cards_unbounded.ColumnInt64(0));
  EXPECT_FALSE(s_credit_cards_unbounded.Step());

  // Make sure the remaining card cvc is the expected one.
  sql::Statement s_cvc_unbounded(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT last_updated_timestamp FROM local_stored_cvc"));
  ASSERT_TRUE(s_cvc_unbounded.is_valid());
  ASSERT_TRUE(s_cvc_unbounded.Step());
  EXPECT_EQ(47, s_cvc_unbounded.ColumnInt64(0));
  EXPECT_FALSE(s_cvc_unbounded.Step());

  // Remove all remaining entries.
  table_->RemoveAutofillDataModifiedBetween(Time(), Time(), &credit_cards);

  // One credit card should have been deleted.
  ASSERT_EQ(1UL, credit_cards.size());
  EXPECT_EQ("00000000-0000-0000-0000-000000000009", credit_cards[0]->guid());

  // There should be no cards left.
  sql::Statement s_credit_cards_empty(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT date_modified FROM credit_cards"));
  ASSERT_TRUE(s_credit_cards_empty.is_valid());
  EXPECT_FALSE(s_credit_cards_empty.Step());

  // There should be no card cvcs left.
  sql::Statement s_cvc_empty(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT last_updated_timestamp FROM local_stored_cvc"));
  ASSERT_TRUE(s_cvc_empty.is_valid());
  EXPECT_FALSE(s_cvc_empty.Step());
}

TEST_F(PaymentsAutofillTableTest, RemoveOriginURLsModifiedBetween) {
  // Populate the credit_cards table.
  ASSERT_TRUE(db_->GetSQLConnection()->Execute(
      "INSERT INTO credit_cards (guid, origin, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000003', '', 17);"
      "INSERT INTO credit_cards (guid, origin, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000004', "
      "       'https://www.example.com/', 27);"
      "INSERT INTO credit_cards (guid, origin, date_modified) "
      "VALUES('00000000-0000-0000-0000-000000000005', 'Chrome settings', "
      "       37);"));

  // Remove all origin URLs set in the bounded time range [21,27).
  table_->RemoveOriginURLsModifiedBetween(Time::FromTimeT(21),
                                          Time::FromTimeT(27));
  sql::Statement s_credit_cards_bounded(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT date_modified, origin FROM credit_cards"));
  ASSERT_TRUE(s_credit_cards_bounded.is_valid());
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(17, s_credit_cards_bounded.ColumnInt64(0));
  EXPECT_EQ(std::string(), s_credit_cards_bounded.ColumnString(1));
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(27, s_credit_cards_bounded.ColumnInt64(0));
  EXPECT_EQ("https://www.example.com/", s_credit_cards_bounded.ColumnString(1));
  ASSERT_TRUE(s_credit_cards_bounded.Step());
  EXPECT_EQ(37, s_credit_cards_bounded.ColumnInt64(0));
  EXPECT_EQ(kSettingsOrigin, s_credit_cards_bounded.ColumnString(1));

  // Remove all origin URLS.
  table_->RemoveOriginURLsModifiedBetween(Time(), Time());
  sql::Statement s_credit_cards_all(db_->GetSQLConnection()->GetUniqueStatement(
      "SELECT date_modified, origin FROM credit_cards"));
  ASSERT_TRUE(s_credit_cards_all.is_valid());
  ASSERT_TRUE(s_credit_cards_all.Step());
  EXPECT_EQ(17, s_credit_cards_all.ColumnInt64(0));
  EXPECT_EQ(std::string(), s_credit_cards_all.ColumnString(1));
  ASSERT_TRUE(s_credit_cards_all.Step());
  EXPECT_EQ(27, s_credit_cards_all.ColumnInt64(0));
  EXPECT_EQ(std::string(), s_credit_cards_all.ColumnString(1));
  ASSERT_TRUE(s_credit_cards_all.Step());
  EXPECT_EQ(37, s_credit_cards_all.ColumnInt64(0));
  EXPECT_EQ(kSettingsOrigin, s_credit_cards_all.ColumnString(1));
}

TEST_F(PaymentsAutofillTableTest, SetGetServerCards) {
  std::vector<CreditCard> inputs;
  inputs.emplace_back(CreditCard::RecordType::kFullServerCard, "a123");
  inputs[0].SetRawInfo(CREDIT_CARD_NAME_FULL, u"Paul F. Tompkins");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  inputs[0].SetRawInfo(CREDIT_CARD_NUMBER, u"4111111111111111");
  inputs[0].set_card_issuer(CreditCard::Issuer::kGoogle);
  inputs[0].set_instrument_id(321);
  inputs[0].set_virtual_card_enrollment_state(
      CreditCard::VirtualCardEnrollmentState::kUnenrolled);
  inputs[0].set_virtual_card_enrollment_type(
      CreditCard::VirtualCardEnrollmentType::kIssuer);
  inputs[0].set_product_description(u"Fake description");
  inputs[0].set_cvc(u"000");

  inputs.emplace_back(CreditCard::RecordType::kMaskedServerCard, "b456");
  inputs[1].SetRawInfo(CREDIT_CARD_NAME_FULL, u"Rick Roman");
  inputs[1].SetRawInfo(CREDIT_CARD_EXP_MONTH, u"12");
  inputs[1].SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"1997");
  inputs[1].SetRawInfo(CREDIT_CARD_NUMBER, u"1111");
  inputs[1].SetNetworkForMaskedCard(kVisaCard);
  std::u16string nickname = u"Grocery card";
  inputs[1].SetNickname(nickname);
  inputs[1].set_card_issuer(CreditCard::Issuer::kExternalIssuer);
  inputs[1].set_issuer_id("amex");
  inputs[1].set_instrument_id(123);
  inputs[1].set_virtual_card_enrollment_state(
      CreditCard::VirtualCardEnrollmentState::kEnrolled);
  inputs[1].set_virtual_card_enrollment_type(
      CreditCard::VirtualCardEnrollmentType::kNetwork);
  inputs[1].set_card_art_url(GURL("https://www.example.com"));
  inputs[1].set_product_terms_url(GURL("https://www.example_term.com"));
  inputs[1].set_cvc(u"111");

  test::SetServerCreditCards(table_.get(), inputs);

  std::vector<std::unique_ptr<CreditCard>> outputs;
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(inputs.size(), outputs.size());

  // Ordering isn't guaranteed, so fix the ordering if it's backwards.
  if (outputs[1]->server_id() == inputs[0].server_id())
    std::swap(outputs[0], outputs[1]);

  // GUIDs for server cards are dynamically generated so will be different
  // after reading from the DB. Check they're valid, but otherwise don't count
  // them in the comparison.
  inputs[0].set_guid(std::string());
  inputs[1].set_guid(std::string());
  outputs[0]->set_guid(std::string());
  outputs[1]->set_guid(std::string());

  EXPECT_EQ(inputs[0], *outputs[0]);
  EXPECT_EQ(inputs[1], *outputs[1]);

  EXPECT_TRUE(outputs[0]->nickname().empty());
  EXPECT_EQ(nickname, outputs[1]->nickname());

  EXPECT_EQ(CreditCard::Issuer::kGoogle, outputs[0]->card_issuer());
  EXPECT_EQ(CreditCard::Issuer::kExternalIssuer, outputs[1]->card_issuer());
  EXPECT_EQ("", outputs[0]->issuer_id());
  EXPECT_EQ("amex", outputs[1]->issuer_id());

  EXPECT_EQ(321, outputs[0]->instrument_id());
  EXPECT_EQ(123, outputs[1]->instrument_id());

  EXPECT_EQ(CreditCard::VirtualCardEnrollmentState::kUnenrolled,
            outputs[0]->virtual_card_enrollment_state());
  EXPECT_EQ(CreditCard::VirtualCardEnrollmentState::kEnrolled,
            outputs[1]->virtual_card_enrollment_state());

  EXPECT_EQ(CreditCard::VirtualCardEnrollmentType::kIssuer,
            outputs[0]->virtual_card_enrollment_type());
  EXPECT_EQ(CreditCard::VirtualCardEnrollmentType::kNetwork,
            outputs[1]->virtual_card_enrollment_type());

  EXPECT_EQ(GURL(), outputs[0]->card_art_url());
  EXPECT_EQ(GURL("https://www.example.com"), outputs[1]->card_art_url());

  EXPECT_EQ(GURL(), outputs[0]->product_terms_url());
  EXPECT_EQ(GURL("https://www.example_term.com"),
            outputs[1]->product_terms_url());

  EXPECT_EQ(u"Fake description", outputs[0]->product_description());

  EXPECT_EQ(inputs[0].cvc(), outputs[0]->cvc());
  EXPECT_EQ(inputs[1].cvc(), outputs[1]->cvc());
}

TEST_F(PaymentsAutofillTableTest, SetGetRemoveServerCardMetadata) {
  // Create and set the metadata.
  AutofillMetadata input;
  input.id = "server id";
  input.use_count = 50;
  input.use_date = AutofillClock::Now();
  input.billing_address_id = "billing id";
  EXPECT_TRUE(table_->AddServerCardMetadata(input));

  // Make sure it was added correctly.
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(input, outputs[0]);

  // Remove the metadata from the table.
  EXPECT_TRUE(table_->RemoveServerCardMetadata(input.id));

  // Make sure it was removed correctly.
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  EXPECT_EQ(0U, outputs.size());
}

// Test that masked IBAN metadata can be added, retrieved and removed
// successfully.
TEST_F(PaymentsAutofillTableTest, SetGetRemoveServerIbanMetadata) {
  Iban iban = test::GetServerIban();
  // Set the metadata.
  iban.set_use_count(50);
  iban.set_use_date(AutofillClock::Now());
  EXPECT_TRUE(table_->AddOrUpdateServerIbanMetadata(iban.GetMetadata()));

  // Make sure it was added correctly.
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerIbansMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(iban.GetMetadata(), outputs[0]);

  // Remove the metadata from the table.
  EXPECT_TRUE(table_->RemoveServerIbanMetadata(outputs[0].id));

  // Make sure it was removed correctly.
  ASSERT_TRUE(table_->GetServerIbansMetadata(outputs));
  EXPECT_EQ(0u, outputs.size());
}

TEST_F(PaymentsAutofillTableTest, AddUpdateServerCardMetadata) {
  // Create and set the metadata.
  AutofillMetadata input;
  input.id = "server id";
  input.use_count = 50;
  input.use_date = AutofillClock::Now();
  input.billing_address_id = "billing id";
  ASSERT_TRUE(table_->AddServerCardMetadata(input));

  // Make sure it was added correctly.
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
  ASSERT_EQ(input, outputs[0]);

  // Update the metadata in the table.
  input.use_count = 51;
  EXPECT_TRUE(table_->UpdateServerCardMetadata(input));

  // Make sure it was updated correctly.
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(input, outputs[0]);

  // Insert a new entry using update - that should also be legal.
  input.id = "another server id";
  EXPECT_TRUE(table_->UpdateServerCardMetadata(input));
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(2U, outputs.size());
}

TEST_F(PaymentsAutofillTableTest, UpdateServerCardMetadataDoesNotChangeData) {
  std::vector<CreditCard> inputs;
  inputs.emplace_back(CreditCard::RecordType::kFullServerCard, "a123");
  inputs[0].SetRawInfo(CREDIT_CARD_NAME_FULL, u"Paul F. Tompkins");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  inputs[0].SetRawInfo(CREDIT_CARD_NUMBER, u"4111111111111111");
  test::SetServerCreditCards(table_.get(), inputs);

  std::vector<std::unique_ptr<CreditCard>> outputs;
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(inputs[0].server_id(), outputs[0]->server_id());

  // Update metadata in the profile.
  ASSERT_NE(outputs[0]->use_count(), 51u);
  outputs[0]->set_use_count(51);

  AutofillMetadata input_metadata = outputs[0]->GetMetadata();
  EXPECT_TRUE(table_->UpdateServerCardMetadata(input_metadata));

  // Make sure it was updated correctly.
  std::vector<AutofillMetadata> output_metadata;
  ASSERT_TRUE(table_->GetServerCardsMetadata(output_metadata));
  ASSERT_EQ(1U, output_metadata.size());
  EXPECT_EQ(input_metadata, output_metadata[0]);

  // Make sure nothing else got updated.
  std::vector<std::unique_ptr<CreditCard>> outputs2;
  table_->GetServerCreditCards(outputs2);
  ASSERT_EQ(1u, outputs2.size());
  EXPECT_EQ(0, outputs[0]->Compare(*outputs2[0]));
}

// Test that updating masked IBAN metadata won't affect IBAN data.
TEST_F(PaymentsAutofillTableTest, UpdateServerIbanMetadata) {
  std::vector<Iban> inputs = {test::GetServerIban()};
  table_->SetServerIbansForTesting(inputs);

  std::vector<std::unique_ptr<Iban>> outputs;
  EXPECT_TRUE(table_->GetServerIbans(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(inputs[0].instrument_id(), outputs[0]->instrument_id());

  // Update metadata in the IBAN.
  outputs[0]->set_use_count(outputs[0]->use_count() + 1);

  EXPECT_TRUE(table_->AddOrUpdateServerIbanMetadata(outputs[0]->GetMetadata()));

  // Make sure it was updated correctly.
  std::vector<AutofillMetadata> output_metadata;
  ASSERT_TRUE(table_->GetServerIbansMetadata(output_metadata));
  ASSERT_EQ(1U, output_metadata.size());
  EXPECT_EQ(outputs[0]->GetMetadata(), output_metadata[0]);

  // Make sure nothing else got updated.
  std::vector<std::unique_ptr<Iban>> outputs2;
  EXPECT_TRUE(table_->GetServerIbans(outputs2));
  ASSERT_EQ(1U, outputs2.size());
  EXPECT_EQ(0, outputs[0]->Compare(*outputs2[0]));
}

TEST_F(PaymentsAutofillTableTest, RemoveWrongServerCardMetadata) {
  // Crete and set some metadata.
  AutofillMetadata input;
  input.id = "server id";
  input.use_count = 50;
  input.use_date = AutofillClock::Now();
  input.billing_address_id = "billing id";
  table_->AddServerCardMetadata(input);

  // Make sure it was added correctly.
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ(input, outputs[0]);

  // Try removing some non-existent metadata.
  EXPECT_FALSE(table_->RemoveServerCardMetadata("a_wrong_id"));

  // Make sure the metadata was not removed.
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  ASSERT_EQ(1U, outputs.size());
}

TEST_F(PaymentsAutofillTableTest, SetServerCardsData) {
  // Set a card data.
  std::vector<CreditCard> inputs;
  inputs.emplace_back(CreditCard::RecordType::kMaskedServerCard, "card1");
  inputs[0].SetRawInfo(CREDIT_CARD_NAME_FULL, u"Rick Roman");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_MONTH, u"12");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"1997");
  inputs[0].SetRawInfo(CREDIT_CARD_NUMBER, u"1111");
  inputs[0].SetNetworkForMaskedCard(kVisaCard);
  inputs[0].SetNickname(u"Grocery card");
  inputs[0].set_card_issuer(CreditCard::Issuer::kExternalIssuer);
  inputs[0].set_issuer_id("amex");
  inputs[0].set_instrument_id(1);
  inputs[0].set_virtual_card_enrollment_state(
      CreditCard::VirtualCardEnrollmentState::kEnrolled);
  inputs[0].set_virtual_card_enrollment_type(
      CreditCard::VirtualCardEnrollmentType::kIssuer);
  inputs[0].set_card_art_url(GURL("https://www.example.com"));
  inputs[0].set_product_terms_url(GURL("https://www.example_term.com"));
  inputs[0].set_product_description(u"Fake description");

  table_->SetServerCardsData(inputs);

  // Make sure the card was added correctly.
  std::vector<std::unique_ptr<CreditCard>> outputs;
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(inputs.size(), outputs.size());

  // GUIDs for server cards are dynamically generated so will be different
  // after reading from the DB. Check they're valid, but otherwise don't count
  // them in the comparison.
  inputs[0].set_guid(std::string());
  outputs[0]->set_guid(std::string());

  EXPECT_EQ(inputs[0], *outputs[0]);

  EXPECT_EQ(CreditCard::VirtualCardEnrollmentState::kEnrolled,
            outputs[0]->virtual_card_enrollment_state());

  EXPECT_EQ(CreditCard::VirtualCardEnrollmentType::kIssuer,
            outputs[0]->virtual_card_enrollment_type());

  EXPECT_EQ(CreditCard::Issuer::kExternalIssuer, outputs[0]->card_issuer());
  EXPECT_EQ("amex", outputs[0]->issuer_id());

  EXPECT_EQ(GURL("https://www.example.com"), outputs[0]->card_art_url());
  EXPECT_EQ(GURL("https://www.example_term.com"),
            outputs[0]->product_terms_url());
  EXPECT_EQ(u"Fake description", outputs[0]->product_description());

  // Make sure no metadata was added.
  std::vector<AutofillMetadata> metadata;
  ASSERT_TRUE(table_->GetServerCardsMetadata(metadata));
  ASSERT_EQ(0U, metadata.size());

  // Set a different card.
  inputs[0] = CreditCard(CreditCard::RecordType::kMaskedServerCard, "card2");
  table_->SetServerCardsData(inputs);

  // The original one should have been replaced.
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1U, outputs.size());
  EXPECT_EQ("card2", outputs[0]->server_id());
  EXPECT_EQ(CreditCard::Issuer::kIssuerUnknown, outputs[0]->card_issuer());
  EXPECT_EQ("", outputs[0]->issuer_id());

  // Make sure no metadata was added.
  ASSERT_TRUE(table_->GetServerCardsMetadata(metadata));
  ASSERT_EQ(0U, metadata.size());
}

// Tests that adding server cards data does not delete the existing metadata.
TEST_F(PaymentsAutofillTableTest, SetServerCardsData_ExistingMetadata) {
  // Create and set some metadata.
  AutofillMetadata input;
  input.id = "server id";
  input.use_count = 50;
  input.use_date = AutofillClock::Now();
  input.billing_address_id = "billing id";
  table_->AddServerCardMetadata(input);

  // Set a card data.
  std::vector<CreditCard> inputs;
  inputs.emplace_back(CreditCard::RecordType::kMaskedServerCard, "server id");
  table_->SetServerCardsData(inputs);

  // Make sure the metadata is still intact.
  std::vector<AutofillMetadata> outputs;
  ASSERT_TRUE(table_->GetServerCardsMetadata(outputs));
  EXPECT_THAT(outputs, ElementsAre(input));
}

TEST_F(PaymentsAutofillTableTest, MaskUnmaskServerCards) {
  std::u16string masked_number(u"1111");
  std::vector<CreditCard> inputs;
  inputs.emplace_back(CreditCard::RecordType::kMaskedServerCard, "a123");
  inputs[0].SetRawInfo(CREDIT_CARD_NAME_FULL, u"Jay Johnson");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  inputs[0].SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  inputs[0].SetRawInfo(CREDIT_CARD_NUMBER, masked_number);
  inputs[0].SetNetworkForMaskedCard(kVisaCard);
  test::SetServerCreditCards(table_.get(), inputs);

  // Unmask the number. The full number should be available.
  std::u16string full_number(u"4111111111111111");
  ASSERT_TRUE(table_->UnmaskServerCreditCard(inputs[0], full_number));

  std::vector<std::unique_ptr<CreditCard>> outputs;
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(CreditCard::RecordType::kFullServerCard ==
              outputs[0]->record_type());
  EXPECT_EQ(full_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();

  // Re-mask the number, we should only get the last 4 digits out.
  ASSERT_TRUE(table_->MaskServerCreditCard(inputs[0].server_id()));
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(CreditCard::RecordType::kMaskedServerCard ==
              outputs[0]->record_type());
  EXPECT_EQ(masked_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();
}

// Calling SetServerCreditCards should replace all existing cards, but unmasked
// cards should not be re-masked.
TEST_F(PaymentsAutofillTableTest, SetServerCardModify) {
  // Add a masked card.
  CreditCard masked_card(CreditCard::RecordType::kMaskedServerCard, "a123");
  masked_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Paul F. Tompkins");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  masked_card.SetRawInfo(CREDIT_CARD_NUMBER, u"1111");
  masked_card.SetNetworkForMaskedCard(kVisaCard);

  std::vector<CreditCard> inputs;
  inputs.push_back(masked_card);
  test::SetServerCreditCards(table_.get(), inputs);

  // Now unmask it.
  std::u16string full_number = u"4111111111111111";
  table_->UnmaskServerCreditCard(masked_card, full_number);

  // The card should now be unmasked.
  std::vector<std::unique_ptr<CreditCard>> outputs;
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(outputs[0]->record_type() ==
              CreditCard::RecordType::kFullServerCard);
  EXPECT_EQ(full_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();

  // Call set again with the masked number.
  inputs[0] = masked_card;
  test::SetServerCreditCards(table_.get(), inputs);

  // The card should stay unmasked.
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(outputs[0]->record_type() ==
              CreditCard::RecordType::kFullServerCard);
  EXPECT_EQ(full_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();

  // Set inputs that do not include our old card.
  CreditCard random_card(CreditCard::RecordType::kMaskedServerCard, "b456");
  random_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Rick Roman");
  random_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"12");
  random_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"1997");
  random_card.SetRawInfo(CREDIT_CARD_NUMBER, u"2222");
  random_card.SetNetworkForMaskedCard(kVisaCard);
  inputs[0] = random_card;
  test::SetServerCreditCards(table_.get(), inputs);

  // We should have only the new card, the other one should have been deleted.
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(outputs[0]->record_type() ==
              CreditCard::RecordType::kMaskedServerCard);
  EXPECT_EQ(random_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(u"2222", outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();

  // Putting back the original card masked should make it masked (this tests
  // that the unmasked data was really deleted).
  inputs[0] = masked_card;
  test::SetServerCreditCards(table_.get(), inputs);
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_TRUE(outputs[0]->record_type() ==
              CreditCard::RecordType::kMaskedServerCard);
  EXPECT_EQ(masked_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(u"1111", outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));

  outputs.clear();
}

TEST_F(PaymentsAutofillTableTest, SetServerCardUpdateUsageStatsAndBillingAddress) {
  // Add a masked card.
  CreditCard masked_card(CreditCard::RecordType::kMaskedServerCard, "a123");
  masked_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Paul F. Tompkins");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  masked_card.SetRawInfo(CREDIT_CARD_NUMBER, u"1111");
  masked_card.set_billing_address_id("1");
  masked_card.SetNetworkForMaskedCard(kVisaCard);

  std::vector<CreditCard> inputs;
  inputs.push_back(masked_card);
  test::SetServerCreditCards(table_.get(), inputs);

  std::vector<std::unique_ptr<CreditCard>> outputs;
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(masked_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(1U, outputs[0]->use_count());
  EXPECT_NE(base::Time(), outputs[0]->use_date());
  // We don't track modification date for server cards. It should always be
  // base::Time().
  EXPECT_EQ(base::Time(), outputs[0]->modification_date());
  outputs.clear();

  // Update the usage stats; make sure they're reflected in GetServerProfiles.
  inputs.back().set_use_count(4U);
  inputs.back().set_use_date(base::Time());
  inputs.back().set_billing_address_id("2");
  table_->UpdateServerCardMetadata(inputs.back());
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(masked_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(4U, outputs[0]->use_count());
  EXPECT_EQ(base::Time(), outputs[0]->use_date());
  EXPECT_EQ(base::Time(), outputs[0]->modification_date());
  EXPECT_EQ("2", outputs[0]->billing_address_id());
  outputs.clear();

  // Setting the cards again shouldn't delete the usage stats.
  table_->SetServerCreditCards(inputs);
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(masked_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(4U, outputs[0]->use_count());
  EXPECT_EQ(base::Time(), outputs[0]->use_date());
  EXPECT_EQ(base::Time(), outputs[0]->modification_date());
  EXPECT_EQ("2", outputs[0]->billing_address_id());
  outputs.clear();

  // Set a card list where the card is missing --- this should clear metadata.
  CreditCard masked_card2(CreditCard::RecordType::kMaskedServerCard, "b456");
  inputs.back() = masked_card2;
  table_->SetServerCreditCards(inputs);

  // Back to the original card list.
  inputs.back() = masked_card;
  table_->SetServerCreditCards(inputs);
  table_->GetServerCreditCards(outputs);
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(masked_card.server_id(), outputs[0]->server_id());
  EXPECT_EQ(1U, outputs[0]->use_count());
  EXPECT_NE(base::Time(), outputs[0]->use_date());
  EXPECT_EQ(base::Time(), outputs[0]->modification_date());
  EXPECT_EQ("1", outputs[0]->billing_address_id());
  outputs.clear();
}

// Tests that deleting time ranges re-masks server credit cards that were
// unmasked in that time.
TEST_F(PaymentsAutofillTableTest, DeleteUnmaskedCard) {
  // This isn't the exact unmasked time, since the database will use the
  // current time that it is called. The code below has to be approximate.
  base::Time unmasked_time = AutofillClock::Now();

  // Add a masked card.
  std::u16string masked_number = u"1111";
  CreditCard masked_card(CreditCard::RecordType::kMaskedServerCard, "a123");
  masked_card.SetRawInfo(CREDIT_CARD_NAME_FULL, u"Paul F. Tompkins");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_MONTH, u"1");
  masked_card.SetRawInfo(CREDIT_CARD_EXP_4_DIGIT_YEAR, u"2020");
  masked_card.SetRawInfo(CREDIT_CARD_NUMBER, masked_number);
  masked_card.SetNetworkForMaskedCard(kVisaCard);

  std::vector<CreditCard> inputs;
  inputs.push_back(masked_card);
  table_->SetServerCreditCards(inputs);

  // Unmask it.
  std::u16string full_number = u"4111111111111111";
  table_->UnmaskServerCreditCard(masked_card, full_number);

  // Delete data in a range a year in the future.
  std::vector<std::unique_ptr<CreditCard>> credit_cards;
  ASSERT_TRUE(table_->RemoveAutofillDataModifiedBetween(
      unmasked_time + base::Days(365), unmasked_time + base::Days(530),
      &credit_cards));

  // This should not affect the unmasked card (should be unmasked).
  std::vector<std::unique_ptr<CreditCard>> outputs;
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(CreditCard::RecordType::kFullServerCard, outputs[0]->record_type());
  EXPECT_EQ(full_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));
  outputs.clear();

  // Delete data in the range of the last 24 hours.
  // Fudge |now| to make sure it's strictly greater than the |now| that
  // the database uses.
  base::Time now = AutofillClock::Now() + base::Seconds(1);
  ASSERT_TRUE(table_->RemoveAutofillDataModifiedBetween(now - base::Days(1),
                                                        now, &credit_cards));

  // This should re-mask.
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(CreditCard::RecordType::kMaskedServerCard,
            outputs[0]->record_type());
  EXPECT_EQ(masked_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));
  outputs.clear();

  // Unmask again, the card should be back.
  table_->UnmaskServerCreditCard(masked_card, full_number);
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(CreditCard::RecordType::kFullServerCard, outputs[0]->record_type());
  EXPECT_EQ(full_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));
  outputs.clear();

  // Delete all data.
  ASSERT_TRUE(table_->RemoveAutofillDataModifiedBetween(
      base::Time(), base::Time::Max(), &credit_cards));

  // Should be masked again.
  ASSERT_TRUE(table_->GetServerCreditCards(outputs));
  ASSERT_EQ(1u, outputs.size());
  EXPECT_EQ(CreditCard::RecordType::kMaskedServerCard,
            outputs[0]->record_type());
  EXPECT_EQ(masked_number, outputs[0]->GetRawInfo(CREDIT_CARD_NUMBER));
  outputs.clear();
}

// Test that we can get what we set.
TEST_F(PaymentsAutofillTableTest, SetGetPaymentsCustomerData) {
  PaymentsCustomerData input{/*customer_id=*/"deadbeef"};
  table_->SetPaymentsCustomerData(&input);

  std::unique_ptr<PaymentsCustomerData> output;
  ASSERT_TRUE(table_->GetPaymentsCustomerData(output));
  EXPECT_EQ(input, *output);
}

// We don't set anything in the table. Test that we don't crash.
TEST_F(PaymentsAutofillTableTest, GetPaymentsCustomerData_NoData) {
  std::unique_ptr<PaymentsCustomerData> output;
  ASSERT_TRUE(table_->GetPaymentsCustomerData(output));
  EXPECT_FALSE(output);
}

// The latest PaymentsCustomerData that was set is returned.
TEST_F(PaymentsAutofillTableTest, SetGetPaymentsCustomerData_MultipleSet) {
  PaymentsCustomerData input{/*customer_id=*/"deadbeef"};
  table_->SetPaymentsCustomerData(&input);

  PaymentsCustomerData input2{/*customer_id=*/"wallet"};
  table_->SetPaymentsCustomerData(&input2);

  PaymentsCustomerData input3{/*customer_id=*/"latest"};
  table_->SetPaymentsCustomerData(&input3);

  std::unique_ptr<PaymentsCustomerData> output;
  ASSERT_TRUE(table_->GetPaymentsCustomerData(output));
  EXPECT_EQ(input3, *output);
}

TEST_F(PaymentsAutofillTableTest, SetGetCreditCardCloudData_OneTimeSet) {
  std::vector<CreditCardCloudTokenData> inputs;
  inputs.push_back(test::GetCreditCardCloudTokenData1());
  inputs.push_back(test::GetCreditCardCloudTokenData2());
  table_->SetCreditCardCloudTokenData(inputs);

  std::vector<std::unique_ptr<CreditCardCloudTokenData>> outputs;
  ASSERT_TRUE(table_->GetCreditCardCloudTokenData(outputs));
  EXPECT_EQ(outputs.size(), inputs.size());
  EXPECT_EQ(0, outputs[0]->Compare(test::GetCreditCardCloudTokenData1()));
  EXPECT_EQ(0, outputs[1]->Compare(test::GetCreditCardCloudTokenData2()));
}

TEST_F(PaymentsAutofillTableTest, SetGetCreditCardCloudData_MultipleSet) {
  std::vector<CreditCardCloudTokenData> inputs;
  CreditCardCloudTokenData input1 = test::GetCreditCardCloudTokenData1();
  inputs.push_back(input1);
  table_->SetCreditCardCloudTokenData(inputs);

  inputs.clear();
  CreditCardCloudTokenData input2 = test::GetCreditCardCloudTokenData2();
  inputs.push_back(input2);
  table_->SetCreditCardCloudTokenData(inputs);

  std::vector<std::unique_ptr<CreditCardCloudTokenData>> outputs;
  ASSERT_TRUE(table_->GetCreditCardCloudTokenData(outputs));
  EXPECT_EQ(1u, outputs.size());
  EXPECT_EQ(0, outputs[0]->Compare(test::GetCreditCardCloudTokenData2()));
}

TEST_F(PaymentsAutofillTableTest, GetCreditCardCloudData_NoData) {
  std::vector<std::unique_ptr<CreditCardCloudTokenData>> output;
  ASSERT_TRUE(table_->GetCreditCardCloudTokenData(output));
  EXPECT_TRUE(output.empty());
}

TEST_F(PaymentsAutofillTableTest, SetAndGetCreditCardOfferData) {
  // Set Offer ID.
  int64_t offer_id_1 = 1;
  int64_t offer_id_2 = 2;
  int64_t offer_id_3 = 3;

  // Set reward amounts for card-linked offers on offer 1 and 2.
  std::string offer_reward_amount_1 = "$5";
  std::string offer_reward_amount_2 = "10%";

  // Set promo code for offer 3.
  std::string promo_code_3 = "5PCTOFFSHOES";

  // Set expiry.
  base::Time expiry_1 = base::Time::FromSecondsSinceUnixEpoch(1000);
  base::Time expiry_2 = base::Time::FromSecondsSinceUnixEpoch(2000);
  base::Time expiry_3 = base::Time::FromSecondsSinceUnixEpoch(3000);

  // Set details URL.
  GURL offer_details_url_1 = GURL("https://www.offer_1_example.com/");
  GURL offer_details_url_2 = GURL("https://www.offer_2_example.com/");
  GURL offer_details_url_3 = GURL("https://www.offer_3_example.com/");

  // Set merchant domains for offer 1.
  std::vector<GURL> merchant_origins_1;
  merchant_origins_1.emplace_back("http://www.merchant_domain_1_1.com/");
  std::vector<GURL> merchant_origins_2;
  merchant_origins_2.emplace_back("http://www.merchant_domain_1_2.com/");
  std::vector<GURL> merchant_origins_3;
  merchant_origins_3.emplace_back("http://www.merchant_domain_1_3.com/");
  // Set merchant domains for offer 2.
  merchant_origins_2.emplace_back("http://www.merchant_domain_2_1.com/");
  // Set merchant domains for offer 3.
  merchant_origins_3.emplace_back("http://www.merchant_domain_3_1.com/");
  merchant_origins_3.emplace_back("http://www.merchant_domain_3_2.com/");

  DisplayStrings display_strings_1;
  DisplayStrings display_strings_2;
  DisplayStrings display_strings_3;
  // Set display strings for all 3 offers.
  display_strings_1.value_prop_text = "$5 off your purchase";
  display_strings_2.value_prop_text = "10% off your purchase";
  display_strings_3.value_prop_text = "5% off shoes. Up to $50.";
  display_strings_1.see_details_text = "Terms apply.";
  display_strings_2.see_details_text = "Terms apply.";
  display_strings_3.see_details_text = "See details.";
  display_strings_1.usage_instructions_text =
      "Check out with this card to activate.";
  display_strings_2.usage_instructions_text =
      "Check out with this card to activate.";
  display_strings_3.usage_instructions_text =
      "Click the promo code field at checkout to autofill it.";

  std::vector<int64_t> eligible_instrument_id_1;
  std::vector<int64_t> eligible_instrument_id_2;
  std::vector<int64_t> eligible_instrument_id_3;

  // Set eligible card-linked instrument ID for offer 1.
  eligible_instrument_id_1.push_back(10);
  eligible_instrument_id_1.push_back(11);
  // Set eligible card-linked instrument ID for offer 2.
  eligible_instrument_id_2.push_back(20);
  eligible_instrument_id_2.push_back(21);
  eligible_instrument_id_2.push_back(22);

  // Create vector of offer data.
  std::vector<AutofillOfferData> autofill_offer_data;
  autofill_offer_data.push_back(AutofillOfferData::GPayCardLinkedOffer(
      offer_id_1, expiry_1, merchant_origins_1, offer_details_url_1,
      display_strings_2, eligible_instrument_id_1, offer_reward_amount_1));
  autofill_offer_data.push_back(AutofillOfferData::GPayCardLinkedOffer(
      offer_id_2, expiry_2, merchant_origins_2, offer_details_url_2,
      display_strings_2, eligible_instrument_id_2, offer_reward_amount_2));
  autofill_offer_data.push_back(AutofillOfferData::GPayPromoCodeOffer(
      offer_id_3, expiry_3, merchant_origins_3, offer_details_url_3,
      display_strings_3, promo_code_3));

  table_->SetAutofillOffers(autofill_offer_data);

  std::vector<std::unique_ptr<AutofillOfferData>> output_offer_data;

  EXPECT_TRUE(table_->GetAutofillOffers(&output_offer_data));
  EXPECT_EQ(autofill_offer_data.size(), output_offer_data.size());

  for (const auto& data : autofill_offer_data) {
    // Find output data with corresponding Offer ID.
    size_t output_index = 0;
    while (output_index < output_offer_data.size()) {
      if (data.GetOfferId() == output_offer_data[output_index]->GetOfferId()) {
        break;
      }
      output_index++;
    }

    // Expect to find matching Offer ID's.
    EXPECT_NE(output_index, output_offer_data.size());

    // All corresponding fields must be equal.
    EXPECT_EQ(data.GetOfferId(), output_offer_data[output_index]->GetOfferId());
    EXPECT_EQ(data.GetOfferRewardAmount(),
              output_offer_data[output_index]->GetOfferRewardAmount());
    EXPECT_EQ(data.GetPromoCode(),
              output_offer_data[output_index]->GetPromoCode());
    EXPECT_EQ(data.GetExpiry(), output_offer_data[output_index]->GetExpiry());
    EXPECT_EQ(data.GetOfferDetailsUrl().spec(),
              output_offer_data[output_index]->GetOfferDetailsUrl().spec());
    EXPECT_EQ(
        data.GetDisplayStrings().value_prop_text,
        output_offer_data[output_index]->GetDisplayStrings().value_prop_text);
    EXPECT_EQ(
        data.GetDisplayStrings().see_details_text,
        output_offer_data[output_index]->GetDisplayStrings().see_details_text);
    EXPECT_EQ(data.GetDisplayStrings().usage_instructions_text,
              output_offer_data[output_index]
                  ->GetDisplayStrings()
                  .usage_instructions_text);
    ASSERT_THAT(data.GetMerchantOrigins(),
                testing::UnorderedElementsAreArray(
                    output_offer_data[output_index]->GetMerchantOrigins()));
    ASSERT_THAT(
        data.GetEligibleInstrumentIds(),
        testing::UnorderedElementsAreArray(
            output_offer_data[output_index]->GetEligibleInstrumentIds()));
  }
}

TEST_F(PaymentsAutofillTableTest, SetAndGetVirtualCardUsageData) {
  // Create test data.
  VirtualCardUsageData virtual_card_usage_data_1 =
      test::GetVirtualCardUsageData1();
  VirtualCardUsageData virtual_card_usage_data_2 =
      test::GetVirtualCardUsageData2();

  // Create vector of VCN usage data.
  std::vector<VirtualCardUsageData> virtual_card_usage_data;
  virtual_card_usage_data.push_back(virtual_card_usage_data_1);
  virtual_card_usage_data.push_back(virtual_card_usage_data_2);

  table_->SetVirtualCardUsageData(virtual_card_usage_data);

  std::vector<std::unique_ptr<VirtualCardUsageData>> output_data;

  EXPECT_TRUE(table_->GetAllVirtualCardUsageData(&output_data));
  EXPECT_EQ(virtual_card_usage_data.size(), output_data.size());

  for (const auto& data : virtual_card_usage_data) {
    // Find output data with corresponding data.
    auto it = base::ranges::find(output_data, data.instrument_id(),
                                 &VirtualCardUsageData::instrument_id);

    // Expect to find a usage data match in the vector.
    EXPECT_NE(it, output_data.end());

    // All corresponding fields must be equal.
    EXPECT_EQ(data.usage_data_id(), (*it)->usage_data_id());
    EXPECT_EQ(data.instrument_id(), (*it)->instrument_id());
    EXPECT_EQ(data.virtual_card_last_four(), (*it)->virtual_card_last_four());
    EXPECT_EQ(data.merchant_origin().Serialize(),
              (*it)->merchant_origin().Serialize());
  }
}

TEST_F(PaymentsAutofillTableTest, AddUpdateRemoveVirtualCardUsageData) {
  // Add a valid VirtualCardUsageData.
  VirtualCardUsageData virtual_card_usage_data =
      test::GetVirtualCardUsageData1();
  EXPECT_TRUE(table_->AddOrUpdateVirtualCardUsageData(virtual_card_usage_data));

  // Get the inserted VirtualCardUsageData.
  std::string usage_data_id = *virtual_card_usage_data.usage_data_id();
  std::unique_ptr<VirtualCardUsageData> usage_data =
      table_->GetVirtualCardUsageData(usage_data_id);
  ASSERT_TRUE(usage_data);
  EXPECT_EQ(virtual_card_usage_data, *usage_data);

  // Update the virtual card usage data.
  VirtualCardUsageData virtual_card_usage_data_update =
      VirtualCardUsageData(virtual_card_usage_data.usage_data_id(),
                           virtual_card_usage_data.instrument_id(),
                           VirtualCardUsageData::VirtualCardLastFour(u"4444"),
                           virtual_card_usage_data.merchant_origin());
  EXPECT_TRUE(
      table_->AddOrUpdateVirtualCardUsageData(virtual_card_usage_data_update));
  usage_data = table_->GetVirtualCardUsageData(usage_data_id);
  ASSERT_TRUE(usage_data);
  EXPECT_EQ(virtual_card_usage_data_update, *usage_data);

  // Remove the virtual card usage data.
  EXPECT_TRUE(table_->RemoveVirtualCardUsageData(usage_data_id));
  usage_data = table_->GetVirtualCardUsageData(usage_data_id);
  EXPECT_FALSE(usage_data);
}

TEST_F(PaymentsAutofillTableTest, RemoveAllVirtualCardUsageData) {
  EXPECT_TRUE(table_->AddOrUpdateVirtualCardUsageData(
      test::GetVirtualCardUsageData1()));

  EXPECT_TRUE(table_->RemoveAllVirtualCardUsageData());

  std::vector<std::unique_ptr<VirtualCardUsageData>> usage_data;
  EXPECT_TRUE(table_->GetAllVirtualCardUsageData(&usage_data));
  EXPECT_TRUE(usage_data.empty());
}

TEST_F(PaymentsAutofillTableTest, AddBankAccount) {
  BankAccount bank_account_to_store_1 = test::CreatePixBankAccount(100);
  BankAccount bank_account_to_store_2 = test::CreatePixBankAccount(200);
  bank_account_to_store_2.AddPaymentRail(
      PaymentInstrument::PaymentRail::kUnknown);
  bank_account_to_store_1.AddToDatabase(table_.get());
  bank_account_to_store_2.AddToDatabase(table_.get());

  // Verify bank account 1 is correctly retrieved.
  std::unique_ptr<PaymentInstrument> payment_instrument_1 =
      table_->GetPaymentInstrument(
          bank_account_to_store_1.instrument_id(),
          PaymentInstrument::InstrumentType::kBankAccount);
  BankAccount* bank_account_from_db_1(
      static_cast<BankAccount*>(payment_instrument_1.get()));

  ASSERT_TRUE(bank_account_from_db_1);
  EXPECT_EQ(bank_account_to_store_1, *bank_account_from_db_1);
  EXPECT_TRUE(bank_account_from_db_1->IsSupported(
      PaymentInstrument::PaymentRail::kPix));

  // Verify bank account 2 is correctly retrieved.
  std::unique_ptr<PaymentInstrument> payment_instrument_2 =
      table_->GetPaymentInstrument(
          bank_account_to_store_2.instrument_id(),
          PaymentInstrument::InstrumentType::kBankAccount);
  BankAccount* bank_account_from_db_2(
      static_cast<BankAccount*>(payment_instrument_2.get()));

  ASSERT_TRUE(bank_account_from_db_2);
  EXPECT_EQ(bank_account_to_store_2, *bank_account_from_db_2);
  EXPECT_TRUE(bank_account_from_db_2->IsSupported(
      PaymentInstrument::PaymentRail::kPix));
  EXPECT_TRUE(bank_account_from_db_2->IsSupported(
      PaymentInstrument::PaymentRail::kUnknown));
}

TEST_F(PaymentsAutofillTableTest, UpdateBankAccount) {
  BankAccount bank_account_to_store = test::CreatePixBankAccount(100);
  ASSERT_TRUE(bank_account_to_store.AddToDatabase(table_.get()));

  BankAccount updated_bank_account_to_store(
      100, u"updated_nickname", GURL("http://www.updated-example.com"),
      u"updated_bank_name", u"updated_account_number_suffix",
      BankAccount::AccountType::kSalary);
  updated_bank_account_to_store.AddPaymentRail(
      PaymentInstrument::PaymentRail::kPix);
  updated_bank_account_to_store.AddPaymentRail(
      PaymentInstrument::PaymentRail::kUnknown);
  ASSERT_TRUE(updated_bank_account_to_store.UpdateInDatabase(table_.get()));

  std::unique_ptr<PaymentInstrument> payment_instrument =
      table_->GetPaymentInstrument(
          bank_account_to_store.instrument_id(),
          PaymentInstrument::InstrumentType::kBankAccount);
  BankAccount* bank_account_from_db(
      static_cast<BankAccount*>(payment_instrument.get()));

  ASSERT_TRUE(bank_account_from_db);
  EXPECT_EQ(updated_bank_account_to_store, *bank_account_from_db);
  // Verify existing payment rail is still supported.
  EXPECT_TRUE(
      bank_account_from_db->IsSupported(PaymentInstrument::PaymentRail::kPix));
  // Verify updated payment rail is supported.
  EXPECT_TRUE(bank_account_from_db->IsSupported(
      PaymentInstrument::PaymentRail::kUnknown));
}

TEST_F(PaymentsAutofillTableTest, UpdateBankAccount_RemoveSupportedRail) {
  // Add 2 supported payment rails.
  BankAccount bank_account_to_store = test::CreatePixBankAccount(100);
  bank_account_to_store.AddPaymentRail(
      PaymentInstrument::PaymentRail::kUnknown);
  ASSERT_TRUE(bank_account_to_store.AddToDatabase(table_.get()));

  BankAccount updated_bank_account_to_store(
      100, bank_account_to_store.nickname(),
      bank_account_to_store.display_icon_url(),
      bank_account_to_store.bank_name(),
      bank_account_to_store.account_number_suffix(),
      bank_account_to_store.account_type());
  // Add only 1 supported payment rail.
  updated_bank_account_to_store.AddPaymentRail(
      PaymentInstrument::PaymentRail::kUnknown);

  ASSERT_TRUE(updated_bank_account_to_store.UpdateInDatabase(table_.get()));

  std::unique_ptr<PaymentInstrument> payment_instrument =
      table_->GetPaymentInstrument(
          bank_account_to_store.instrument_id(),
          PaymentInstrument::InstrumentType::kBankAccount);
  BankAccount* bank_account_from_db(
      static_cast<BankAccount*>(payment_instrument.get()));

  ASSERT_TRUE(bank_account_from_db);
  EXPECT_EQ(updated_bank_account_to_store, *bank_account_from_db);
  // Verify existing payment rail is no longer supported.
  EXPECT_FALSE(
      bank_account_from_db->IsSupported(PaymentInstrument::PaymentRail::kPix));
  // Verify updated payment rail is supported.
  EXPECT_TRUE(bank_account_from_db->IsSupported(
      PaymentInstrument::PaymentRail::kUnknown));
}

TEST_F(PaymentsAutofillTableTest, RemoveBankAccount) {
  BankAccount bank_account_to_store = test::CreatePixBankAccount(100);
  bank_account_to_store.AddPaymentRail(
      PaymentInstrument::PaymentRail::kUnknown);
  ASSERT_TRUE(bank_account_to_store.AddToDatabase(table_.get()));

  // Remove bank account from db.
  ASSERT_TRUE(bank_account_to_store.DeleteFromDatabase(table_.get()));

  // Verify row is deleted from payment_instruments table.
  sql::Statement payment_instruments_select(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT COUNT(*) "
          "FROM payment_instruments WHERE instrument_id = ? AND "
          "instrument_type = ?"));
  payment_instruments_select.BindInt64(0,
                                       bank_account_to_store.instrument_id());
  payment_instruments_select.BindInt64(
      1, static_cast<int>(bank_account_to_store.GetInstrumentType()));

  EXPECT_TRUE(payment_instruments_select.Step());
  EXPECT_EQ(0, payment_instruments_select.ColumnInt(0));

  // Verify row is deleted from payment_instrument_supported_rails table.
  sql::Statement payment_instrument_supported_rails_select(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT COUNT(*) "
          "FROM payment_instrument_supported_rails WHERE instrument_id = ? AND "
          "instrument_type = ?"));
  payment_instrument_supported_rails_select.BindInt64(
      0, bank_account_to_store.instrument_id());
  payment_instrument_supported_rails_select.BindInt64(
      1, static_cast<int>(bank_account_to_store.GetInstrumentType()));

  EXPECT_TRUE(payment_instrument_supported_rails_select.Step());
  EXPECT_EQ(0, payment_instrument_supported_rails_select.ColumnInt(0));

  // Verify row is deleted from bank_accounts table.
  sql::Statement bank_accounts_select(
      db_->GetSQLConnection()->GetUniqueStatement(
          "SELECT COUNT(*) "
          "FROM bank_accounts WHERE instrument_id = ?"));
  bank_accounts_select.BindInt64(0, bank_account_to_store.instrument_id());
  EXPECT_TRUE(bank_accounts_select.Step());
  EXPECT_EQ(0, bank_accounts_select.ColumnInt(0));
}

TEST_F(PaymentsAutofillTableTest, GetPaymentInstrument) {
  sql::Statement payment_instruments_insert(
      db_->GetSQLConnection()->GetUniqueStatement(
          "INSERT INTO payment_instruments (instrument_id, instrument_type, "
          "nickname, display_icon_url) VALUES(100, 1, 'nickname', "
          "'http://display-icon-url.com')"));
  sql::Statement payment_instrument_supported_rails_insert(
      db_->GetSQLConnection()->GetUniqueStatement(
          "INSERT INTO payment_instrument_supported_rails (instrument_id, "
          "instrument_type, payment_rail)VALUES(100, 1, 1)"));
  sql::Statement bank_accounts_insert(
      db_->GetSQLConnection()->GetUniqueStatement(
          "INSERT INTO bank_accounts (instrument_id, bank_name, "
          "account_number_suffix, account_type) VALUES(100, 'bank_name', "
          "'account_number_suffix', 1)"));
  EXPECT_TRUE(payment_instruments_insert.Run());
  EXPECT_TRUE(payment_instrument_supported_rails_insert.Run());
  EXPECT_TRUE(bank_accounts_insert.Run());

  std::unique_ptr<PaymentInstrument> payment_instrument =
      table_->GetPaymentInstrument(
          100, PaymentInstrument::InstrumentType::kBankAccount);
  BankAccount* bank_account_from_db(
      static_cast<BankAccount*>(payment_instrument.get()));

  ASSERT_TRUE(bank_account_from_db);
  EXPECT_EQ(100, bank_account_from_db->instrument_id());
  EXPECT_EQ(u"nickname", bank_account_from_db->nickname());
  EXPECT_EQ(static_cast<BankAccount::AccountType>(1),
            bank_account_from_db->account_type());
  EXPECT_EQ(u"account_number_suffix",
            bank_account_from_db->account_number_suffix());
  EXPECT_EQ(GURL("http://display-icon-url.com"),
            bank_account_from_db->display_icon_url());
  EXPECT_EQ(u"bank_name", bank_account_from_db->bank_name());
  EXPECT_TRUE(
      bank_account_from_db->IsSupported(PaymentInstrument::PaymentRail::kPix));
}

}  // namespace autofill
