// Mersenne Twister from https://gist.github.com/banksean/300494
/*
A C-program for MT19937, with initialization improved 2002/1/26.
Coded by Takuji Nishimura and Makoto Matsumoto.
Before using, initialize the state by using init_genrand(seed)
or init_by_array(init_key, key_length).
Copyright (C) 1997 - 2002, Makoto Matsumoto and Takuji Nishimura,
All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.
3. The names of its contributors may not be used to endorse or promote
products derived from this software without specific prior written
permission.
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
Any feedback is very welcome.
http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
email: m-mat @ math.sci.hiroshima-u.ac.jp (remove space)
*/

class MersenneTwister {
    private N: number;
    private M: number;
    private MATRIX_A: number;
    private UPPER_MASK: number;
    private LOWER_MASK: number;

    private mt: number[];
    private mti: number;

    constructor(seed: number) {
        if (seed === undefined) {
            // kept random number same size as time used previously to ensure no unexpected results downstream
            seed = Math.floor(Math.random() * Math.pow(10, 13));
        }
        /* Period parameters */
        this.N = 624;
        this.M = 397;
        this.MATRIX_A = 0x9908b0df; /* constant vector a */
        this.UPPER_MASK = 0x80000000; /* most significant w-r bits */
        this.LOWER_MASK = 0x7fffffff; /* least significant r bits */

        this.mt = new Array(this.N); /* the array for the state vector */
        this.mti = this.N + 1; /* mti==N + 1 means mt[N] is not initialized */

        this.init_genrand(seed);
    }

    /* initializes mt[N] with a seed */
    private init_genrand(s: number) {
        this.mt[0] = s >>> 0;
        for (this.mti = 1; this.mti < this.N; this.mti++) {
            s = this.mt[this.mti - 1] ^ (this.mt[this.mti - 1] >>> 30);
            this.mt[this.mti] =
                ((((s & 0xffff0000) >>> 16) * 1812433253) << 16) + (s & 0x0000ffff) * 1812433253 + this.mti;
            /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
            /* In the previous versions, MSBs of the seed affect   */
            /* only MSBs of the array mt[].                        */
            /* 2002/01/09 modified by Makoto Matsumoto             */
            this.mt[this.mti] >>>= 0;
            /* for >32 bit machines */
        }
    }

    /* generates a random number on [0,0xffffffff]-interval */
    private genrand_int32() {
        let y;
        let mag01 = new Array(0x0, this.MATRIX_A);
        /* mag01[x] = x * MATRIX_A  for x=0,1 */

        if (this.mti >= this.N) {
            /* generate N words at one time */
            let kk;

            if (this.mti === this.N + 1) {
                /* if init_genrand() has not been called, */
                this.init_genrand(5489); /* a default initial seed is used */
            }
            for (kk = 0; kk < this.N - this.M; kk++) {
                y = (this.mt[kk] & this.UPPER_MASK) | (this.mt[kk + 1] & this.LOWER_MASK);
                this.mt[kk] = this.mt[kk + this.M] ^ (y >>> 1) ^ mag01[y & 0x1];
            }
            for (; kk < this.N - 1; kk++) {
                y = (this.mt[kk] & this.UPPER_MASK) | (this.mt[kk + 1] & this.LOWER_MASK);
                this.mt[kk] = this.mt[kk + (this.M - this.N)] ^ (y >>> 1) ^ mag01[y & 0x1];
            }
            y = (this.mt[this.N - 1] & this.UPPER_MASK) | (this.mt[0] & this.LOWER_MASK);
            this.mt[this.N - 1] = this.mt[this.M - 1] ^ (y >>> 1) ^ mag01[y & 0x1];

            this.mti = 0;
        }

        y = this.mt[this.mti++];

        /* Tempering */
        y ^= y >>> 11;
        y ^= (y << 7) & 0x9d2c5680;
        y ^= (y << 15) & 0xefc60000;
        y ^= y >>> 18;

        return y >>> 0;
    }

    /* generates a random number on [0,1)-real-interval */
    random() {
        return this.genrand_int32() * (1.0 / 4294967296.0);
        /* divided by 2^32 */
    }
}

// https://github.com/chancejs/chancejs
export class Chance {
    private mt: MersenneTwister;

    constructor(readonly seed: number) {
        // If no generator function was provided, use our MT
        this.mt = new MersenneTwister(this.seed);
    }

    random(): number {
        return this.mt.random();
    }

    /**
     *  Return a random integer
     *
     *  NOTE the max and min are INCLUDED in the range. So:
     *  chance.integer({min: 1, max: 3});
     *  would return either 1, 2, or 3.
     *
     *  @param {Object} [options={}] can specify a min and/or max
     *  @returns {Number} a single random integer number
     *  @throws {RangeError} min cannot be greater than max
     */
    integer(options: { min: number; max: number }): number {
        return Math.floor(this.random() * (options.max - options.min + 1) + options.min);
    }

    /**
     *  Return a random natural
     *
     *  NOTE the max and min are INCLUDED in the range. So:
     *  chance.natural({min: 1, max: 3});
     *  would return either 1, 2, or 3.
     *
     *  @param {Object} [options={}] can specify a min and/or maxm or a numerals count.
     *  @returns {Number} a single random integer number
     *  @throws {RangeError} min cannot be greater than max
     */
    natural(options: { max: number }): number {
        return this.integer({ min: 0, max: options.max });
    }

    pick<T>(arr: T[]): T {
        if (arr.length === 0) {
            throw new RangeError("Chance: Cannot pick() from an empty array");
        }
        return arr[this.natural({ max: arr.length - 1 })];
    }

    animal(): string {
        // if user does not put in any animal type, will return a random animal regardless
        const animalTypeArray = ["desert", "forest", "ocean", "zoo", "farm", "pet", "grassland"];
        return this.pick(animals[this.pick(animalTypeArray)]);
    }

    city(): string {
        return this.pick(cities);
    }
}

const animals: { [kind: string]: string[] } = {
    // list of ocean animals comes from https://owlcation.com/stem/list-of-ocean-animals
    ocean: [
        "Acantharea",
        "Anemone",
        "Angelfish King",
        "Ahi Tuna",
        "Albacore",
        "American Oyster",
        "Anchovy",
        "Armored Snail",
        "Arctic Char",
        "Atlantic Bluefin Tuna",
        "Atlantic Cod",
        "Atlantic Goliath Grouper",
        "Atlantic Trumpetfish",
        "Atlantic Wolffish",
        "Baleen Whale",
        "Banded Butterflyfish",
        "Banded Coral Shrimp",
        "Banded Sea Krait",
        "Barnacle",
        "Barndoor Skate",
        "Barracuda",
        "Basking Shark",
        "Bass",
        "Beluga Whale",
        "Bluebanded Goby",
        "Bluehead Wrasse",
        "Bluefish",
        "Bluestreak Cleaner-Wrasse",
        "Blue Marlin",
        "Blue Shark",
        "Blue Spiny Lobster",
        "Blue Tang",
        "Blue Whale",
        "Broadclub Cuttlefish",
        "Bull Shark",
        "Chambered Nautilus",
        "Chilean Basket Star",
        "Chilean Jack Mackerel",
        "Chinook Salmon",
        "Christmas Tree Worm",
        "Clam",
        "Clown Anemonefish",
        "Clown Triggerfish",
        "Cod",
        "Coelacanth",
        "Cockscomb Cup Coral",
        "Common Fangtooth",
        "Conch",
        "Cookiecutter Shark",
        "Copepod",
        "Coral",
        "Corydoras",
        "Cownose Ray",
        "Crab",
        "Crown-of-Thorns Starfish",
        "Cushion Star",
        "Cuttlefish",
        "California Sea Otters",
        "Dolphin",
        "Dolphinfish",
        "Dory",
        "Devil Fish",
        "Dugong",
        "Dumbo Octopus",
        "Dungeness Crab",
        "Eccentric Sand Dollar",
        "Edible Sea Cucumber",
        "Eel",
        "Elephant Seal",
        "Elkhorn Coral",
        "Emperor Shrimp",
        "Estuarine Crocodile",
        "Fathead Sculpin",
        "Fiddler Crab",
        "Fin Whale",
        "Flameback",
        "Flamingo Tongue Snail",
        "Flashlight Fish",
        "Flatback Turtle",
        "Flatfish",
        "Flying Fish",
        "Flounder",
        "Fluke",
        "French Angelfish",
        "Frilled Shark",
        "Fugu (also called Pufferfish)",
        "Gar",
        "Geoduck",
        "Giant Barrel Sponge",
        "Giant Caribbean Sea Anemone",
        "Giant Clam",
        "Giant Isopod",
        "Giant Kingfish",
        "Giant Oarfish",
        "Giant Pacific Octopus",
        "Giant Pyrosome",
        "Giant Sea Star",
        "Giant Squid",
        "Glowing Sucker Octopus",
        "Giant Tube Worm",
        "Goblin Shark",
        "Goosefish",
        "Great White Shark",
        "Greenland Shark",
        "Grey Atlantic Seal",
        "Grouper",
        "Grunion",
        "Guineafowl Puffer",
        "Haddock",
        "Hake",
        "Halibut",
        "Hammerhead Shark",
        "Hapuka",
        "Harbor Porpoise",
        "Harbor Seal",
        "Hatchetfish",
        "Hawaiian Monk Seal",
        "Hawksbill Turtle",
        "Hector's Dolphin",
        "Hermit Crab",
        "Herring",
        "Hoki",
        "Horn Shark",
        "Horseshoe Crab",
        "Humpback Anglerfish",
        "Humpback Whale",
        "Icefish",
        "Imperator Angelfish",
        "Irukandji Jellyfish",
        "Isopod",
        "Ivory Bush Coral",
        "Japanese Spider Crab",
        "Jellyfish",
        "John Dory",
        "Juan Fernandez Fur Seal",
        "Killer Whale",
        "Kiwa Hirsuta",
        "Krill",
        "Lagoon Triggerfish",
        "Lamprey",
        "Leafy Seadragon",
        "Leopard Seal",
        "Limpet",
        "Ling",
        "Lionfish",
        "Lions Mane Jellyfish",
        "Lobe Coral",
        "Lobster",
        "Loggerhead Turtle",
        "Longnose Sawshark",
        "Longsnout Seahorse",
        "Lophelia Coral",
        "Marrus Orthocanna",
        "Manatee",
        "Manta Ray",
        "Marlin",
        "Megamouth Shark",
        "Mexican Lookdown",
        "Mimic Octopus",
        "Moon Jelly",
        "Mollusk",
        "Monkfish",
        "Moray Eel",
        "Mullet",
        "Mussel",
        "Megaladon",
        "Napoleon Wrasse",
        "Nassau Grouper",
        "Narwhal",
        "Nautilus",
        "Needlefish",
        "Northern Seahorse",
        "North Atlantic Right Whale",
        "Northern Red Snapper",
        "Norway Lobster",
        "Nudibranch",
        "Nurse Shark",
        "Oarfish",
        "Ocean Sunfish",
        "Oceanic Whitetip Shark",
        "Octopus",
        "Olive Sea Snake",
        "Orange Roughy",
        "Ostracod",
        "Otter",
        "Oyster",
        "Pacific Angelshark",
        "Pacific Blackdragon",
        "Pacific Halibut",
        "Pacific Sardine",
        "Pacific Sea Nettle Jellyfish",
        "Pacific White Sided Dolphin",
        "Pantropical Spotted Dolphin",
        "Patagonian Toothfish",
        "Peacock Mantis Shrimp",
        "Pelagic Thresher Shark",
        "Penguin",
        "Peruvian Anchoveta",
        "Pilchard",
        "Pink Salmon",
        "Pinniped",
        "Plankton",
        "Porpoise",
        "Polar Bear",
        "Portuguese Man o' War",
        "Pycnogonid Sea Spider",
        "Quahog",
        "Queen Angelfish",
        "Queen Conch",
        "Queen Parrotfish",
        "Queensland Grouper",
        "Ragfish",
        "Ratfish",
        "Rattail Fish",
        "Ray",
        "Red Drum",
        "Red King Crab",
        "Ringed Seal",
        "Risso's Dolphin",
        "Ross Seals",
        "Sablefish",
        "Salmon",
        "Sand Dollar",
        "Sandbar Shark",
        "Sawfish",
        "Sarcastic Fringehead",
        "Scalloped Hammerhead Shark",
        "Seahorse",
        "Sea Cucumber",
        "Sea Lion",
        "Sea Urchin",
        "Seal",
        "Shark",
        "Shortfin Mako Shark",
        "Shovelnose Guitarfish",
        "Shrimp",
        "Silverside Fish",
        "Skipjack Tuna",
        "Slender Snipe Eel",
        "Smalltooth Sawfish",
        "Smelts",
        "Sockeye Salmon",
        "Southern Stingray",
        "Sponge",
        "Spotted Porcupinefish",
        "Spotted Dolphin",
        "Spotted Eagle Ray",
        "Spotted Moray",
        "Squid",
        "Squidworm",
        "Starfish",
        "Stickleback",
        "Stonefish",
        "Stoplight Loosejaw",
        "Sturgeon",
        "Swordfish",
        "Tan Bristlemouth",
        "Tasseled Wobbegong",
        "Terrible Claw Lobster",
        "Threespot Damselfish",
        "Tiger Prawn",
        "Tiger Shark",
        "Tilefish",
        "Toadfish",
        "Tropical Two-Wing Flyfish",
        "Tuna",
        "Umbrella Squid",
        "Velvet Crab",
        "Venus Flytrap Sea Anemone",
        "Vigtorniella Worm",
        "Viperfish",
        "Vampire Squid",
        "Vaquita",
        "Wahoo",
        "Walrus",
        "West Indian Manatee",
        "Whale",
        "Whale Shark",
        "Whiptail Gulper",
        "White-Beaked Dolphin",
        "White-Ring Garden Eel",
        "White Shrimp",
        "Wobbegong",
        "Wrasse",
        "Wreckfish",
        "Xiphosura",
        "Yellowtail Damselfish",
        "Yelloweye Rockfish",
        "Yellow Cup Black Coral",
        "Yellow Tube Sponge",
        "Yellowfin Tuna",
        "Zebrashark",
        "Zooplankton"
    ],
    // list of desert, grassland, and forest animals comes from http://www.skyenimals.com/
    desert: [
        "Aardwolf",
        "Addax",
        "African Wild Ass",
        "Ant",
        "Antelope",
        "Armadillo",
        "Baboon",
        "Badger",
        "Bat",
        "Bearded Dragon",
        "Beetle",
        "Bird",
        "Black-footed Cat",
        "Boa",
        "Brown Bear",
        "Bustard",
        "Butterfly",
        "Camel",
        "Caracal",
        "Caracara",
        "Caterpillar",
        "Centipede",
        "Cheetah",
        "Chipmunk",
        "Chuckwalla",
        "Climbing Mouse",
        "Coati",
        "Cobra",
        "Cotton Rat",
        "Cougar",
        "Courser",
        "Crane Fly",
        "Crow",
        "Dassie Rat",
        "Dove",
        "Dunnart",
        "Eagle",
        "Echidna",
        "Elephant",
        "Emu",
        "Falcon",
        "Fly",
        "Fox",
        "Frogmouth",
        "Gecko",
        "Geoffroy's Cat",
        "Gerbil",
        "Grasshopper",
        "Guanaco",
        "Gundi",
        "Hamster",
        "Hawk",
        "Hedgehog",
        "Hyena",
        "Hyrax",
        "Jackal",
        "Kangaroo",
        "Kangaroo Rat",
        "Kestrel",
        "Kowari",
        "Kultarr",
        "Leopard",
        "Lion",
        "Macaw",
        "Meerkat",
        "Mouse",
        "Oryx",
        "Ostrich",
        "Owl",
        "Pronghorn",
        "Python",
        "Rabbit",
        "Raccoon",
        "Rattlesnake",
        "Rhinoceros",
        "Sand Cat",
        "Spectacled Bear",
        "Spiny Mouse",
        "Starling",
        "Stick Bug",
        "Tarantula",
        "Tit",
        "Toad",
        "Tortoise",
        "Tyrant Flycatcher",
        "Viper",
        "Vulture",
        "Waxwing",
        "Xerus",
        "Zebra"
    ],
    grassland: [
        "Aardvark",
        "Aardwolf",
        "Accentor",
        "African Buffalo",
        "African Wild Dog",
        "Alpaca",
        "Anaconda",
        "Ant",
        "Anteater",
        "Antelope",
        "Armadillo",
        "Baboon",
        "Badger",
        "Bandicoot",
        "Barbet",
        "Bat",
        "Bee",
        "Bee-eater",
        "Beetle",
        "Bird",
        "Bison",
        "Black-footed Cat",
        "Black-footed Ferret",
        "Bluebird",
        "Boa",
        "Bowerbird",
        "Brown Bear",
        "Bush Dog",
        "Bushshrike",
        "Bustard",
        "Butterfly",
        "Buzzard",
        "Caracal",
        "Caracara",
        "Cardinal",
        "Caterpillar",
        "Cheetah",
        "Chipmunk",
        "Civet",
        "Climbing Mouse",
        "Clouded Leopard",
        "Coati",
        "Cobra",
        "Cockatoo",
        "Cockroach",
        "Common Genet",
        "Cotton Rat",
        "Cougar",
        "Courser",
        "Coyote",
        "Crane",
        "Crane Fly",
        "Cricket",
        "Crow",
        "Culpeo",
        "Death Adder",
        "Deer",
        "Deer Mouse",
        "Dingo",
        "Dinosaur",
        "Dove",
        "Drongo",
        "Duck",
        "Duiker",
        "Dunnart",
        "Eagle",
        "Echidna",
        "Elephant",
        "Elk",
        "Emu",
        "Falcon",
        "Finch",
        "Flea",
        "Fly",
        "Flying Frog",
        "Fox",
        "Frog",
        "Frogmouth",
        "Garter Snake",
        "Gazelle",
        "Gecko",
        "Geoffroy's Cat",
        "Gerbil",
        "Giant Tortoise",
        "Giraffe",
        "Grasshopper",
        "Grison",
        "Groundhog",
        "Grouse",
        "Guanaco",
        "Guinea Pig",
        "Hamster",
        "Harrier",
        "Hartebeest",
        "Hawk",
        "Hedgehog",
        "Helmetshrike",
        "Hippopotamus",
        "Hornbill",
        "Hyena",
        "Hyrax",
        "Impala",
        "Jackal",
        "Jaguar",
        "Jaguarundi",
        "Kangaroo",
        "Kangaroo Rat",
        "Kestrel",
        "Kultarr",
        "Ladybug",
        "Leopard",
        "Lion",
        "Macaw",
        "Meerkat",
        "Mouse",
        "Newt",
        "Oryx",
        "Ostrich",
        "Owl",
        "Pangolin",
        "Pheasant",
        "Prairie Dog",
        "Pronghorn",
        "Przewalski's Horse",
        "Python",
        "Quoll",
        "Rabbit",
        "Raven",
        "Rhinoceros",
        "Shelduck",
        "Sloth Bear",
        "Spectacled Bear",
        "Squirrel",
        "Starling",
        "Stick Bug",
        "Tamandua",
        "Tasmanian Devil",
        "Thornbill",
        "Thrush",
        "Toad",
        "Tortoise"
    ],
    forest: [
        "Agouti",
        "Anaconda",
        "Anoa",
        "Ant",
        "Anteater",
        "Antelope",
        "Armadillo",
        "Asian Black Bear",
        "Aye-aye",
        "Babirusa",
        "Baboon",
        "Badger",
        "Bandicoot",
        "Banteng",
        "Barbet",
        "Basilisk",
        "Bat",
        "Bearded Dragon",
        "Bee",
        "Bee-eater",
        "Beetle",
        "Bettong",
        "Binturong",
        "Bird-of-paradise",
        "Bongo",
        "Bowerbird",
        "Bulbul",
        "Bush Dog",
        "Bushbaby",
        "Bushshrike",
        "Butterfly",
        "Buzzard",
        "Caecilian",
        "Cardinal",
        "Cassowary",
        "Caterpillar",
        "Centipede",
        "Chameleon",
        "Chimpanzee",
        "Cicada",
        "Civet",
        "Clouded Leopard",
        "Coati",
        "Cobra",
        "Cockatoo",
        "Cockroach",
        "Colugo",
        "Cotinga",
        "Cotton Rat",
        "Cougar",
        "Crane Fly",
        "Cricket",
        "Crocodile",
        "Crow",
        "Cuckoo",
        "Cuscus",
        "Death Adder",
        "Deer",
        "Dhole",
        "Dingo",
        "Dinosaur",
        "Drongo",
        "Duck",
        "Duiker",
        "Eagle",
        "Echidna",
        "Elephant",
        "Finch",
        "Flat-headed Cat",
        "Flea",
        "Flowerpecker",
        "Fly",
        "Flying Frog",
        "Fossa",
        "Frog",
        "Frogmouth",
        "Gaur",
        "Gecko",
        "Gorilla",
        "Grison",
        "Hawaiian Honeycreeper",
        "Hawk",
        "Hedgehog",
        "Helmetshrike",
        "Hornbill",
        "Hyrax",
        "Iguana",
        "Jackal",
        "Jaguar",
        "Jaguarundi",
        "Kestrel",
        "Ladybug",
        "Lemur",
        "Leopard",
        "Lion",
        "Macaw",
        "Mandrill",
        "Margay",
        "Monkey",
        "Mouse",
        "Mouse Deer",
        "Newt",
        "Okapi",
        "Old World Flycatcher",
        "Orangutan",
        "Owl",
        "Pangolin",
        "Peafowl",
        "Pheasant",
        "Possum",
        "Python",
        "Quokka",
        "Rabbit",
        "Raccoon",
        "Red Panda",
        "Red River Hog",
        "Rhinoceros",
        "Sloth Bear",
        "Spectacled Bear",
        "Squirrel",
        "Starling",
        "Stick Bug",
        "Sun Bear",
        "Tamandua",
        "Tamarin",
        "Tapir",
        "Tarantula",
        "Thrush",
        "Tiger",
        "Tit",
        "Toad",
        "Tortoise",
        "Toucan",
        "Trogon",
        "Trumpeter",
        "Turaco",
        "Turtle",
        "Tyrant Flycatcher",
        "Viper",
        "Vulture",
        "Wallaby",
        "Warbler",
        "Wasp",
        "Waxwing",
        "Weaver",
        "Weaver-finch",
        "Whistler",
        "White-eye",
        "Whydah",
        "Woodswallow",
        "Worm",
        "Wren",
        "Xenops",
        "Yellowjacket",
        "Accentor",
        "African Buffalo",
        "American Black Bear",
        "Anole",
        "Bird",
        "Bison",
        "Boa",
        "Brown Bear",
        "Chipmunk",
        "Common Genet",
        "Copperhead",
        "Coyote",
        "Deer Mouse",
        "Dormouse",
        "Elk",
        "Emu",
        "Fisher",
        "Fox",
        "Garter Snake",
        "Giant Panda",
        "Giant Tortoise",
        "Groundhog",
        "Grouse",
        "Guanaco",
        "Himalayan Tahr",
        "Kangaroo",
        "Koala",
        "Numbat",
        "Quoll",
        "Raccoon dog",
        "Tasmanian Devil",
        "Thornbill",
        "Turkey",
        "Vole",
        "Weasel",
        "Wildcat",
        "Wolf",
        "Wombat",
        "Woodchuck",
        "Woodpecker"
    ],
    // list of farm animals comes from https://www.buzzle.com/articles/farm-animals-list.html
    farm: [
        "Alpaca",
        "Buffalo",
        "Banteng",
        "Cow",
        "Cat",
        "Chicken",
        "Carp",
        "Camel",
        "Donkey",
        "Dog",
        "Duck",
        "Emu",
        "Goat",
        "Gayal",
        "Guinea",
        "Goose",
        "Horse",
        "Honey",
        "Llama",
        "Pig",
        "Pigeon",
        "Rhea",
        "Rabbit",
        "Sheep",
        "Silkworm",
        "Turkey",
        "Yak",
        "Zebu"
    ],
    // list of pet animals comes from https://www.dogbreedinfo.com/pets/pet.htm
    pet: [
        "Bearded Dragon",
        "Birds",
        "Burro",
        "Cats",
        "Chameleons",
        "Chickens",
        "Chinchillas",
        "Chinese Water Dragon",
        "Cows",
        "Dogs",
        "Donkey",
        "Ducks",
        "Ferrets",
        "Fish",
        "Geckos",
        "Geese",
        "Gerbils",
        "Goats",
        "Guinea Fowl",
        "Guinea Pigs",
        "Hamsters",
        "Hedgehogs",
        "Horses",
        "Iguanas",
        "Llamas",
        "Lizards",
        "Mice",
        "Mule",
        "Peafowl",
        "Pigs and Hogs",
        "Pigeons",
        "Ponies",
        "Pot Bellied Pig",
        "Rabbits",
        "Rats",
        "Sheep",
        "Skinks",
        "Snakes",
        "Stick Insects",
        "Sugar Gliders",
        "Tarantula",
        "Turkeys",
        "Turtles"
    ],
    // list of zoo animals comes from https://bronxzoo.com/animals
    zoo: [
        "Aardvark",
        "African Wild Dog",
        "Aldabra Tortoise",
        "American Alligator",
        "American Bison",
        "Amur Tiger",
        "Anaconda",
        "Andean Condor",
        "Asian Elephant",
        "Baby Doll Sheep",
        "Bald Eagle",
        "Barred Owl",
        "Blue Iguana",
        "Boer Goat",
        "California Sea Lion",
        "Caribbean Flamingo",
        "Chinchilla",
        "Collared Lemur",
        "Coquerel's Sifaka",
        "Cuban Amazon Parrot",
        "Ebony Langur",
        "Fennec Fox",
        "Fossa",
        "Gelada",
        "Giant Anteater",
        "Giraffe",
        "Gorilla",
        "Grizzly Bear",
        "Henkel's Leaf-tailed Gecko",
        "Indian Gharial",
        "Indian Rhinoceros",
        "King Cobra",
        "King Vulture",
        "Komodo Dragon",
        "Linne's Two-toed Sloth",
        "Lion",
        "Little Penguin",
        "Madagascar Tree Boa",
        "Magellanic Penguin",
        "Malayan Tapir",
        "Malayan Tiger",
        "Matschies Tree Kangaroo",
        "Mini Donkey",
        "Monarch Butterfly",
        "Nile crocodile",
        "North American Porcupine",
        "Nubian Ibex",
        "Okapi",
        "Poison Dart Frog",
        "Polar Bear",
        "Pygmy Marmoset",
        "Radiated Tortoise",
        "Red Panda",
        "Red Ruffed Lemur",
        "Ring-tailed Lemur",
        "Ring-tailed Mongoose",
        "Rock Hyrax",
        "Small Clawed Asian Otter",
        "Snow Leopard",
        "Snowy Owl",
        "Southern White-faced Owl",
        "Southern White Rhinocerous",
        "Squirrel Monkey",
        "Tufted Puffin",
        "White Cheeked Gibbon",
        "White-throated Bee Eater",
        "Zebra"
    ]
};

// Source: https://en.wikipedia.org/wiki/List_of_population_centers_by_latitude
const cities: string[] = [
    "Alert",
    "Nord",
    "Eureka",
    "Ny-Ålesund",
    "Longyearbyen",
    "Qaanaaq",
    "Grise Fiord",
    "Dikson",
    "Upernavik",
    "Tiksi",
    "Belushya Guba",
    "Barrow",
    "Honningsvåg",
    "Hammerfest",
    "Deadhorse",
    "Nuorgam",
    "Vadsø",
    "Alta",
    "Utsjoki",
    "Kirkenes",
    "Tromsø",
    "Tuktoyaktuk",
    "Norilsk",
    "Murmansk",
    "Harstad",
    "Narvik",
    "Inuvik",
    "Kiruna",
    "Verkhoyansk",
    "Bodø",
    "Gällivare",
    "Kemijärvi",
    "Rovaniemi",
    "Ísafjörður",
    "Kuusamo",
    "Tornio",
    "Boden",
    "Haparanda",
    "Kemi",
    "Akureyri",
    "Luleå",
    "Pudasjärvi",
    "Piteå",
    "Oulu",
    "Fairbanks",
    "Skellefteå",
    "Anadyr",
    "Arkhangelsk",
    "Nome",
    "Nuuk (Godthåb)",
    "Reykjavík",
    "Umeå",
    "Iqaluit",
    "Stjørdal",
    "Trondheim",
    "Örnsköldsvik",
    "Östersund",
    "Vaasa",
    "Kuopio",
    "Yellowknife",
    "Sundsvall",
    "Jyväskylä",
    "Yakutsk",
    "Tórshavn",
    "Petrozavodsk",
    "Tampere",
    "Anchorage",
    "Lillehammer",
    "Lahti",
    "Qaqortoq",
    "Whitehorse",
    "Turku",
    "Bergen",
    "Vantaa",
    "Lerwick",
    "Espoo",
    "Helsinki",
    "Oslo",
    "Saint Petersburg",
    "Uppsala",
    "Fritz Creek",
    "Magadan",
    "Tallinn",
    "Stockholm",
    "Sarpsborg",
    "Örebro",
    "Vologda",
    "Norrköping",
    "Linköping",
    "Tartu",
    "Juneau",
    "Arendal",
    "Kristiansand",
    "Perm",
    "Valga",
    "Gothenburg",
    "Visby",
    "Yaroslavl",
    "Inverness",
    "Aberdeen",
    "Sitka",
    "Riga",
    "Izhevsk",
    "Yekaterinburg",
    "Fort McMurray",
    "Dundee",
    "Nizhny Novgorod",
    "Stirling",
    "Bratsk",
    "Aarhus",
    "Krasnoyarsk",
    "Edinburgh",
    "Glasgow",
    "Kazan",
    "Moscow",
    "Copenhagen",
    "Malmö",
    "Esbjerg",
    "Odense",
    "Grande Prairie",
    "Chelyabinsk",
    "Novosibirsk",
    "Derry",
    "Omsk",
    "Newcastle upon Tyne",
    "Sunderland",
    "Petropavl",
    "Flensburg",
    "Ufa",
    "Kaliningrad",
    "Vilnius",
    "Belfast",
    "Middlesbrough",
    "Gdynia",
    "Gdańsk",
    "Kiel",
    "Douglas",
    "Rostock",
    "Lancaster",
    "York",
    "Prince George",
    "Minsk",
    "Bradford",
    "Leeds",
    "Hamburg",
    "Edmonton",
    "Tolyatti",
    "Manchester",
    "Liverpool",
    "Sheffield",
    "Dublin",
    "Groningen",
    "Samara",
    "Bremen",
    "Petropavlovsk-Kamchatsky",
    "Nottingham",
    "Derby",
    "Leicester",
    "Norwich",
    "Peterborough",
    "Berlin",
    "Birmingham",
    "Gomel",
    "Coventry",
    "Poznań",
    "Haarlem",
    "Amsterdam",
    "Hannover",
    "Irkutsk",
    "Warsaw",
    "Enschede",
    "Cambridge",
    "Leiden",
    "Saskatoon",
    "Utrecht",
    "The Hague",
    "Münster",
    "Rotterdam",
    "Cork",
    "Adak",
    "Gloucester",
    "Oxford",
    "Voronezh",
    "Swansea",
    "Saratov",
    "Dortmund",
    "London",
    "Cardiff",
    "Greenwich",
    "Bristol",
    "Bath",
    "Leipzig",
    "Düsseldorf",
    "Antwerp",
    "Astana",
    "Winchester",
    "Calgary",
    "Ghent",
    "Dresden",
    "Erfurt",
    "Calais",
    "Cologne",
    "Southampton",
    "Maastricht",
    "Brussels",
    "Brighton and Hove",
    "Portsmouth",
    "Exeter",
    "Kamloops",
    "Lille",
    "Regina",
    "Kiev",
    "Charleroi",
    "Plymouth",
    "Frankfurt am Main",
    "Prague",
    "Kraków",
    "Kharkiv",
    "Winnipeg",
    "Kelowna",
    "Lviv",
    "Luxembourg",
    "Le Havre",
    "Nuremberg",
    "Rouen",
    "Vancouver",
    "Paris",
    "Stuttgart",
    "Bellingham",
    "Volgograd",
    "Strasbourg",
    "Dnipro",
    "Victoria",
    "Saguenay",
    "Brest",
    "Thunder Bay",
    "Linz",
    "Vienna",
    "Bratislava",
    "Munich",
    "Donetsk",
    "Ulaanbaatar",
    "Orléans",
    "Salzburg",
    "Seattle",
    "St. John's",
    "Basel",
    "Budapest",
    "Qiqihar",
    "Zürich",
    "Innsbruck",
    "Rostov-on-Don",
    "Nantes",
    "Vaduz",
    "Atyrau",
    "Graz",
    "Chişinău",
    "Bern",
    "Tiraspol",
    "Quebec City",
    "Bismarck",
    "Duluth",
    "Saint-Pierre",
    "Cluj-Napoca",
    "Moscow",
    "Helena",
    "Sault Ste. Marie",
    "Lausanne",
    "Sudbury",
    "Odessa",
    "North Bay",
    "Charlottetown",
    "Geneva",
    "Moncton",
    "Ljubljana",
    "Fredericton",
    "Zagreb",
    "Lyon",
    "Harbin",
    "New Glasgow",
    "Portland",
    "Montreal",
    "Milan",
    "Venice",
    "Ottawa",
    "Wakkanai",
    "Saint John",
    "Novi Sad",
    "Turin",
    "Krasnodar",
    "Minneapolis",
    "Simferopol",
    "Halifax",
    "Bordeaux",
    "Belgrade",
    "Šabac",
    "Drobeta Turnu Severin",
    "Bologna",
    "Bucharest",
    "Ravenna",
    "Genoa",
    "Pierre",
    "Craiova",
    "Augusta",
    "Montpelier",
    "Bath",
    "San Marino",
    "Changchun",
    "Sarajevo",
    "Yarmouth",
    "Ürümqi",
    "Florence",
    "Asahikawa",
    "Monaco",
    "Nice",
    "Toronto",
    "Boise",
    "Toulouse",
    "Sochi",
    "Cannes",
    "Gijón",
    "Sioux Falls",
    "Split",
    "Oviedo",
    "Nemuro",
    "San Sebastián",
    "Marseille",
    "Almaty",
    "Bilbao",
    "Concord",
    "Rochester",
    "Vladivostok",
    "Sapporo",
    "Milwaukee",
    "Sukhumi",
    "Kushiro",
    "London",
    "Follonica",
    "Obihiro",
    "Buffalo",
    "Santiago de Compostela",
    "Bishkek",
    "Vitoria-Gasteiz",
    "Sofia",
    "Pristina",
    "Andorra la Vella",
    "Nukus",
    "Podgorica",
    "Boston",
    "Detroit",
    "Windsor",
    "Vigo",
    "Tskhinvali",
    "Avezzano",
    "Skopje",
    "Colchester",
    "Rome",
    "Vatican City",
    "Chicago",
    "Providence",
    "Shenyang",
    "Chongjin",
    "Hartford",
    "Hakodate",
    "Tbilisi",
    "Valladolid",
    "Zaragoza",
    "Des Moines",
    "Braga",
    "Cleveland",
    "Barcelona",
    "Tirana",
    "Tashkent",
    "Porto",
    "Cheyenne",
    "Bari",
    "Fort Wayne",
    "Istanbul",
    "Trabzon",
    "Namangan",
    "Naples",
    "Aomori",
    "Lincoln",
    "Eureka",
    "Salt Lake City",
    "Jersey City",
    "New York City",
    "Thessaloniki",
    "Lleida",
    "Pittsburgh",
    "Baku",
    "Madrid",
    "Trenton",
    "Bursa",
    "Yerevan",
    "Boulder",
    "Columbus",
    "Ankara",
    "Philadelphia",
    "Beijing",
    "Stepanakert",
    "Forked River",
    "Eskisehir",
    "Indianapolis",
    "Denver",
    "Akita",
    "Morioka",
    "Palma",
    "Reno",
    "Flores Island",
    "Valencia",
    "Baltimore",
    "Cagliari",
    "Dover",
    "Wonsan",
    "Tianjin",
    "Cincinnati",
    "Kansas City",
    "Pyongyang",
    "Ibiza",
    "Dalian",
    "Washington",
    "Lisbon",
    "St. Louis",
    "Horta",
    "Sacramento",
    "Dushanbe",
    "Angra do Heroísmo",
    "Halkida",
    "İzmir",
    "Charleston",
    "Alicante",
    "Sendai",
    "Louisville",
    "Yamagata",
    "Messina",
    "Palermo",
    "Tabriz",
    "Shijiazhuang",
    "Trapani",
    "Athena",
    "Kaesong",
    "Ashgabat",
    "Niigata",
    "Taiyuan",
    "Konya",
    "Ponta Delgada",
    "Oakland",
    "San Francisco",
    "Fukushima",
    "Wichita",
    "Seoul",
    "San Jose",
    "Catania",
    "Incheon",
    "Sevilla",
    "Mugla",
    "Springfield",
    "Gaziantep",
    "Adana",
    "Virginia Beach",
    "Almería",
    "Tunis",
    "Mersin",
    "Algiers",
    "Málaga",
    "Toyama",
    "Jinan",
    "Nagano",
    "Xining",
    "Handan",
    "Kanazawa",
    "Utsunomiya",
    "Maebashi",
    "Mito",
    "Mosul",
    "Mashhad",
    "Arbil",
    "Las Vegas",
    "Nashville",
    "Gibraltar",
    "Tulsa",
    "Fukui",
    "Qingdao",
    "Lanzhou",
    "Knoxville",
    "Valletta",
    "Birkirkara",
    "Los Alamos",
    "Daegu",
    "Raleigh",
    "Tangier",
    "Tehran",
    "Tokyo",
    "Santa Fe",
    "Kōfu",
    "Chiba",
    "Latakia",
    "Kawasaki",
    "Tottori",
    "Oklahoma City",
    "Matsue",
    "Kirkuk",
    "Yokohama",
    "Gifu",
    "Tiaret",
    "Heraklion",
    "Melilla",
    "Charlotte",
    "Nagoya",
    "Busan",
    "Nicosia",
    "Memphis",
    "Albuquerque",
    "Ōtsu",
    "Kyoto",
    "Shizuoka",
    "Zhengzhou",
    "Little Rock",
    "Sfax",
    "Tsu",
    "Osaka",
    "Kōbe",
    "Nara",
    "Okayama",
    "Kabul",
    "Santa Barbara",
    "Hiroshima",
    "Takamatsu",
    "Xi'an",
    "Wakayama",
    "Wilmington",
    "Yamaguchi",
    "Arak",
    "Srinagar",
    "Tokushima",
    "Los Angeles",
    "Rabat",
    "Peshawar",
    "Columbia",
    "Riverside",
    "Beirut",
    "Matsuyama",
    "Atlanta",
    "Islamabad",
    "Birmingham",
    "Rawalpindi",
    "Fukuoka",
    "Kōchi",
    "Casablanca",
    "Damascus",
    "Phoenix",
    "Lubbock",
    "Baghdad",
    "Saga",
    "Ōita",
    "Tripoli",
    "Charleston",
    "Dallas",
    "Nagasaki",
    "Kumamoto",
    "San Diego",
    "Mexicali",
    "Funchal",
    "El Centro",
    "Isfahan",
    "Karbala",
    "Tijuana",
    "Shreveport",
    "Jackson",
    "Hamilton",
    "Tucson",
    "Benghazi",
    "Tel Aviv",
    "Nanjing",
    "Najaf",
    "Amman",
    "Miyazaki",
    "Ramallah",
    "Ensenada",
    "El Paso",
    "Jerusalem",
    "Ciudad Juárez",
    "Amritsar",
    "Marrakech",
    "Kandahar",
    "Kagoshima",
    "Lahore",
    "Gaza",
    "Faisalabad",
    "Port Said",
    "Alexandria",
    "Shanghai",
    "Shimla",
    "Ludhiana",
    "Chandigarh",
    "Mobile",
    "Chengdu",
    "Wuhan",
    "Basra",
    "Agadir",
    "Jacksonville",
    "Austin",
    "Hangzhou",
    "Multan",
    "Cairo",
    "New Orleans",
    "Suez",
    "Houston",
    "Lhasa",
    "Chongqing",
    "Eilat",
    "San Antonio",
    "Kuwait City",
    "Shigatse",
    "Hermosillo",
    "Chihuahua",
    "New Delhi",
    "Santa Cruz de Tenerife",
    "Orlando",
    "Las Palmas de Gran Canaria",
    "Tampa",
    "Corpus Christi",
    "Maspalomas",
    "Kathmandu",
    "Dibrugarh",
    "Thimphu",
    "Gangtok",
    "Agra",
    "El Aaiún",
    "Sabha",
    "Jaipur",
    "Lucknow",
    "Siliguri",
    "Tezpur",
    "Kanpur",
    "Dammam",
    "Naha",
    "Manama",
    "Guwahati",
    "Brownsville",
    "Miami",
    "Luxor",
    "Monterrey",
    "Patna",
    "Shillong",
    "Torreón",
    "Kulpahar",
    "Doha",
    "Dubai",
    "Hyderabad",
    "Nassau",
    "Kunming",
    "Taipei",
    "Karachi",
    "Culiacán",
    "Riyadh",
    "Key West",
    "Medina",
    "Abu Dhabi",
    "Al Ain",
    "Taichung",
    "Durango",
    "Agartala",
    "Dhaka",
    "Muscat",
    "Havana",
    "Guangzhou",
    "Ahmedabad",
    "Dongguan",
    "Tainan",
    "Cabo San Lucas",
    "Nanning",
    "Kaohsiung",
    "Kolkata (Calcutta)",
    "Shenzhen",
    "Chittagong",
    "Hong Kong",
    "Tampico",
    "San Luis Potosí",
    "Aguascalientes",
    "Jeddah",
    "Cockburn Town",
    "Mecca",
    "Honolulu",
    "Chhattisgarh",
    "Cancún",
    "Surat",
    "Nagpur",
    "León",
    "Hanoi",
    "Mérida",
    "Hai Phong",
    "Zapopan",
    "Puerto Vallarta",
    "Guadalajara",
    "Querétaro",
    "Haikou",
    "Santiago de Cuba",
    "Chiang Rai",
    "Naypyidaw",
    "Hilo",
    "George Town",
    "Veracruz",
    "Puebla",
    "Mumbai",
    "Chiang Mai",
    "Port-au-Prince",
    "Pune",
    "Santo Domingo",
    "San Juan",
    "Road Town",
    "Charlotte Amalie",
    "The Valley",
    "Nouakchott",
    "Kingston",
    "Vientiane",
    "Visakhapatnam",
    "Belize City",
    "Udon Thani",
    "Hyderabad",
    "Basseterre",
    "Belmopan",
    "St. John's",
    "Salalah",
    "Acapulco",
    "Yangon",
    "Timbuktu",
    "Huế",
    "Khon Kaen",
    "Da Nang",
    "Basse-Terre",
    "Omdurman",
    "Khartoum",
    "Sana'a",
    "Asmara",
    "Roseau",
    "Saipan",
    "Son My",
    "Nakhon Ratchasima",
    "Praia",
    "Tapachula",
    "Quetzaltenango",
    "Thiès",
    "Dakar",
    "Quezon City",
    "Guatemala City",
    "Fort-de-France",
    "Manila",
    "Makati",
    "Ayutthaya",
    "Tegucigalpa",
    "Castries",
    "Bangkok",
    "San Salvador",
    "Niamey",
    "Dededo",
    "Hagåtña",
    "Banjul",
    "Serekunda",
    "Siem Reap",
    "Brikama",
    "Kingstown",
    "Bridgetown",
    "Chennai (Madras)",
    "Bangalore",
    "Pattaya",
    "Koulikoro",
    "Bamako",
    "Chinandega",
    "Oranjestad",
    "Ouagadougou",
    "Managua",
    "Willemstad",
    "N'Djamena",
    "St. George's",
    "Kano",
    "Bissau",
    "Port Blair",
    "Djibouti",
    "Phnom Penh",
    "Bobo-Dioulasso",
    "Barranquilla",
    "Ho Chi Minh City",
    "Iloilo City",
    "Port of Spain",
    "Maracaibo",
    "Liberia",
    "Caracas",
    "Chaguanas",
    "Cartagena",
    "San Fernando",
    "Cebu City",
    "Maracay",
    "Valencia",
    "Alajuela",
    "Limón",
    "Jaffna",
    "Hargeisa",
    "Puntarenas",
    "Kochi",
    "San José",
    "Tagbilaran",
    "Conakry",
    "Tamale",
    "Surat Thani",
    "Abuja",
    "Addis Ababa",
    "Panama City",
    "Mérida",
    "Moundou",
    "Trivandrum",
    "Freetown",
    "Ciudad Bolívar",
    "Cúcuta",
    "Phuket",
    "San Cristóbal",
    "Nzérékoré",
    "Batticaloa",
    "Melekeok",
    "Weno",
    "Ibadan",
    "Koror",
    "Kandy",
    "Majuro",
    "Davao City",
    "Hat Yai",
    "Colombo",
    "Palikir",
    "Sri Jayawardenapura-Kotte",
    "Zamboanga City",
    "Yamoussoukro",
    "Georgetown",
    "Kumasi",
    "Porto-Novo",
    "Lagos",
    "Enugu",
    "Cotonou",
    "Benin City",
    "Monrovia",
    "Medellín",
    "Lomé",
    "Kota Bharu",
    "Alor Setar",
    "Kota Kinabalu",
    "Paramaribo",
    "Puerto Ayacucho",
    "Accra",
    "Banda Aceh",
    "George Town",
    "Abidjan",
    "Cayenne",
    "Sekondi-Takoradi",
    "Bandar Seri Begawan",
    "Juba",
    "Port Harcourt",
    "Ipoh",
    "Bogotá",
    "Miri",
    "Bangui",
    "Malé",
    "Douala",
    "Yaoundé",
    "Malabo",
    "Medan",
    "Santiago de Cali",
    "Kuala Lumpur",
    "Boa Vista",
    "Malacca Town",
    "Mogadishu",
    "Kuching",
    "Johor Bahru",
    "South Tarawa",
    "Singapore",
    "Pekanbaru",
    "Libreville",
    "São Tomé",
    "Kampala",
    "Entebbe",
    "Mbandaka",
    "Macapá",
    "Pontianak",
    "Quito",
    "Kismayo",
    "Yaren District",
    "Padang",
    "Balikpapan",
    "Nairobi",
    "Belém",
    "Kigali",
    "Guayaquil",
    "Mwanza",
    "São Luís",
    "Jayapura",
    "Palembang",
    "Manaus",
    "Bujumbura",
    "Sobral",
    "Ambon",
    "Fortaleza",
    "Iquitos",
    "Fernando de Noronha",
    "Mombasa",
    "Brazzaville",
    "Kinshasa",
    "Victoria",
    "Teresina",
    "Makassar",
    "Imperatriz",
    "Natal",
    "Zanzibar City",
    "Dodoma",
    "Jakarta",
    "Bogor",
    "Dar es Salaam",
    "Bandung",
    "Semarang",
    "João Pessoa",
    "Juazeiro do Norte",
    "Campina Grande",
    "Surabaya",
    "Yogyakarta",
    "Malang",
    "Recife",
    "Trujillo",
    "Caruaru",
    "Funafuti",
    "Dili",
    "Denpasar",
    "Porto Velho",
    "Luanda",
    "Garanhuns",
    "Nukulaelae",
    "Petrolina",
    "Honiara",
    "Port Moresby",
    "Maceió",
    "Rio Branco",
    "Aracaju",
    "Palmas",
    "Lubumbashi",
    "Moroni",
    "Sinop",
    "Lima",
    "Darwin",
    "Huambo",
    "Mamoudzou",
    "Ndola",
    "Salvador",
    "Ayacucho",
    "Mata-Utu",
    "Cusco",
    "Apia",
    "Lilongwe",
    "Pago Pago",
    "Ilhéus",
    "Lusaka",
    "Cuiabá",
    "Blantyre",
    "Brasília",
    "Jamestown",
    "Arequipa",
    "Labasa",
    "Rabi Island",
    "La Paz",
    "Goiânia",
    "Cairns",
    "Papeete",
    "Port Vila",
    "Santa Cruz de la Sierra",
    "Livingstone",
    "Harare",
    "Suva",
    "Arica",
    "Governador Valadares",
    "Uberlândia",
    "Antananarivo",
    "Mutare",
    "Sucre",
    "Alofi",
    "Townsville",
    "Gweru",
    "Belo Horizonte",
    "Port Louis",
    "Bulawayo",
    "Iquique",
    "Port Hedland",
    "Vitória",
    "Campo Grande",
    "São José do Rio Preto",
    "Saint-Denis",
    "Nukuʻalofa",
    "Francistown",
    "Ribeirão Preto",
    "Avarua",
    "Juiz de Fora",
    "Poços de Caldas",
    "Araraquara",
    "Nouméa",
    "Bauru",
    "Calama",
    "Windhoek",
    "Assis",
    "Campinas",
    "Rio de Janeiro",
    "São José dos Campos",
    "Rockhampton",
    "São Paulo",
    "Antofagasta",
    "Polokwane",
    "Gaborone",
    "Salta",
    "Adamstown",
    "Asunción",
    "Curitiba",
    "Nelspruit",
    "Ciudad del Este",
    "Pretoria",
    "Mafikeng",
    "Maputo",
    "Johannesburg",
    "Mbabane",
    "Lobamba",
    "Manzini",
    "San Miguel de Tucumán",
    "Hanga Roa",
    "Copiapó",
    "Brisbane",
    "Corrientes",
    "Florianópolis",
    "Gold Coast",
    "Kimberley",
    "Kingston",
    "Bloemfontein",
    "Maseru",
    "Pietermaritzburg",
    "Durban",
    "La Serena",
    "Córdoba",
    "Santa Fe",
    "Pelotas",
    "Perth",
    "San Juan",
    "Mandurah",
    "Bhisho",
    "Mendoza",
    "Newcastle",
    "Rosario",
    "Viña del Mar",
    "Valparaíso",
    "Santiago",
    "Chuí",
    "Sydney",
    "Cape Town",
    "Port Elizabeth",
    "Wollongong",
    "Buenos Aires",
    "Montevideo",
    "Adelaide",
    "Canberra",
    "Concepción",
    "Auckland",
    "Edinburgh of the Seven Seas",
    "Hamilton",
    "Melbourne",
    "Mar del Plata",
    "Geelong",
    "Bahía Blanca",
    "Temuco",
    "Neuquén",
    "Hastings",
    "Valdivia",
    "San Carlos de Bariloche",
    "Nelson",
    "Wellington",
    "Puerto Montt",
    "Launceston, Tasmania",
    "Hobart",
    "Trelew",
    "Christchurch",
    "Waitangi",
    "Puerto Aisén",
    "Coihaique",
    "Comodoro Rivadavia",
    "Dunedin",
    "Invercargill",
    "Río Gallegos",
    "Stanley",
    "Punta Arenas",
    "Grytviken",
    "Ushuaia",
    "Puerto Williams",
    "Puerto Toro",
    "Villa Las Estrellas",
    "Esperanza",
    "Rothera",
    "Concordia"
];
