//SPDX-FileCopyrightText: 2018 - 2023 UnionTech Software Technology Co., Ltd.
//
//SPDX-License-Identifier: GPL-3.0-or-later
#include "datetimemodel.h"
#include "dccfactory.h"
#include "timezoneMap/timezone_map_util.h"
#include "datetimeworker.h"
#include "zoneinfomodel.h"
#include "keyboard/keyboardmodel.h"
#include "languagelistmodel.h"
#include "langregionmodel.h"

#include <unicode/locid.h>
#include <unicode/unistr.h>

#include <QDateTime>
#include <QTimeZone>
#include <QSettings>
#include <QCoreApplication>
#include <QStringListModel>

static installer::ZoneInfoList g_totalZones;

static QString getDescription(const ZoneInfo &zoneInfo)
{
    const QDateTime localTime(QDateTime::currentDateTime());
    const double timeDelta = (zoneInfo.getUTCOffset() - localTime.offsetFromUtc()) / 3600.0;
    QString dateLiteral;
    if (localTime.time().hour() + timeDelta >= 24) {
        dateLiteral = DatetimeModel::tr("Tomorrow");
    } else if (localTime.time().hour() + timeDelta <= 0) {
        dateLiteral = DatetimeModel::tr("Yesterday");
    } else {
        dateLiteral = DatetimeModel::tr("Today");
    }

    int decimalNumber = 1;
    //小时取余,再取分钟,将15分钟的双倍只显示一位小数,其他的都显示两位小数
    switch ((zoneInfo.getUTCOffset() - localTime.offsetFromUtc()) % 3600 / 60 / 15) {
    case -1:
    case -3:
    case 1:
    case 3:
        decimalNumber = 2;
        break;
    default:
        decimalNumber = 1;
        break;
    }

    QString description;
    if (timeDelta > 0) {
        description = DatetimeModel::tr("%1 hours earlier than local").arg(QString::number(timeDelta, 'f', decimalNumber));
    } else {
        description = DatetimeModel::tr("%1 hours later than local").arg(QString::number(-timeDelta, 'f', decimalNumber));
    }

    return description;
}

static QString getUtcOffsetText(int utcOffset)
{
    QString gmData;
    int utcOff = utcOffset / 3600;
    if (utcOff >= 0) {
        gmData = QString("(UTC+%1:%2)").arg(utcOff, 2, 10, QLatin1Char('0')).arg(utcOffset % 3600 / 60, 2, 10, QLatin1Char('0'));
    } else {
        gmData = QString("(UTC%1:%2)").arg(utcOff, 3, 10, QLatin1Char('0')).arg(utcOffset % 3600 / 60, 2, 10, QLatin1Char('0'));
    }

    return gmData;
}

static QString getDisplayText(const ZoneInfo &zoneInfo)
{
    QString gmData = getUtcOffsetText(zoneInfo.getUTCOffset());
    QString cityName = zoneInfo.getZoneCity().isEmpty() ? zoneInfo.getZoneName() : zoneInfo.getZoneCity();

    return QString("%1 %2").arg(cityName).arg(gmData);
}

static QStringList timeZoneList(const installer::ZoneInfoList &zoneInfoList, QMap<QString, QString> &cache)
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    const QString locale = QLocale::system().name();
    QStringList timezoneList;
    for (const auto& info : zoneInfoList) {
        auto localzone = GetLocalTimezoneName(info.timezone, locale);

        if (!cache.contains(localzone)) {
            // "上海": "Asia/Shanghai"
            cache[localzone] = info.timezone;
        }

        timezoneList << localzone;
    }

    return timezoneList;
}

static inline QStringList getCurrencySymbol(bool positive, const QString &symbol)
{
    const QString money("1.1");
    if (positive)
        return {
            QString("%1%2").arg(symbol).arg(money),  // ￥1.1
            QString("%1%2").arg(money).arg(symbol),  // ￥1.1
            QString("%1 %2").arg(symbol).arg(money), // ￥ 1.1
            QString("%1 %2").arg(money).arg(symbol)  // 1.1 ￥
        };

    return {
        QString("-%1%2").arg(symbol).arg(money), // -￥1.1
        QString("%1-%2").arg(symbol).arg(money), // ￥-1.1
        QString("%1%2-").arg(symbol).arg(money), // ￥1.1-
        QString("-%1%2").arg(money).arg(symbol), // 1.1-￥
        QString("%1-%2").arg(money).arg(symbol), // 1.1-￥
        QString("%1%2-").arg(money).arg(symbol)  // 1.1￥-
    };
}

static inline QStringList separatorSymbol()
{
    return { QString("."), QString(","), QString("'"), DatetimeModel::tr("Space") };
}

static QStringList translateLangAndCountry(const QString &localeName)
{
    auto localeSystem = QLocale::system();
    auto systemLocale = icu::Locale(localeSystem.name().toStdString().data());
    auto IcuLocale = icu::Locale(localeName.toStdString().data());
    auto localeHex = icu::UnicodeString(localeName.toStdString().data());
    std::string displayLanguageIcu;
    IcuLocale.getDisplayLanguage(systemLocale, localeHex).toUTF8String(displayLanguageIcu);
    std::string displayCountryIcu;
    IcuLocale.getDisplayCountry(systemLocale, localeHex).toUTF8String(displayCountryIcu);

    return QStringList{ QString::fromStdString(displayLanguageIcu),
                        QString::fromStdString(displayCountryIcu) };
}

static QString translate(const QString &localeName, const QString &langRegion)
{
    QStringList langRegions = langRegion.split(":");
    if (langRegions.size() < 2) {
        return langRegion;
    }

    if (langRegions[0] == "Traditional Chinese" || langRegions[0] == "Simplified Chinese"
        || langRegions[1] == QLocale::countryToString(QLocale::HongKong)
        || langRegions[1] == QLocale::countryToString(QLocale::Macau)
        || langRegions[1] == QLocale::countryToString(QLocale::Taiwan)) {

        QString langCountry =
                QString("%1(%2)")
                        .arg(QCoreApplication::translate("dcc::datetime::Language",
                                                         langRegions.at(0).toUtf8().data()))
                        .arg(QCoreApplication::translate("dcc::datetime::Country",
                                                         langRegions.at(1).toUtf8().data()));
        return langCountry;
    }

    auto res = translateLangAndCountry(localeName);
    QString langCountry = QString("%1(%2)").arg(res.value(0)).arg(res.value(1));

    return langCountry;
}

DatetimeModel::DatetimeModel(QObject *parent)
    : QObject(parent)
    , m_ntp(true)
    , m_bUse24HourType(true)
    , m_work(new DatetimeWorker(this, this))
{
    connect(this, &DatetimeModel::ntpChanged, m_work, &DatetimeWorker::setNTP);
    connect(this, &DatetimeModel::hourTypeChanged, m_work, &DatetimeWorker::set24HourType);
    connect(this, &DatetimeModel::NTPServerChanged, m_work, &DatetimeWorker::setNtpServer);
    // 设置ntp地址失败回退到之前的地址
    connect(this, &DatetimeModel::NTPServerNotChanged, this, &DatetimeModel::setNtpServerAddress);

    connect(this, &DatetimeModel::userTimeZoneAdded, m_work, [this](const ZoneInfo &zone){
        m_work->addUserTimeZone(zone.getZoneName());
    });
    connect(this, &DatetimeModel::userTimeZoneRemoved, m_work, &DatetimeWorker::removeUserTimeZone);
    // set timezone
    connect(this, &DatetimeModel::timeZoneChanged, m_work, &DatetimeWorker::setTimezone);

    connect(this, &DatetimeModel::currencyFormatChanged, this, [this]() {
        int pIndex = property("__PositiveCurrency").toInt();
        int nIndex = property("__NegativeCurrency").toInt();

        if (pIndex < 0 || nIndex < 0)
            return;

        setCurrentFormat(PositiveCurrency, pIndex);
        setCurrentFormat(NegativeCurrency, nIndex);

        setProperty("__PositiveCurrency", -1);
        setProperty("__NegativeCurrency", -1);
    });
    connect(this, &DatetimeModel::symbolChanged, this, [this](int format, const QString &symbol) {
        if (format != DigitGroupingSymbol)
            return;
        int dIndex = property("__DigitGrouping").toInt();

        if (dIndex < 0)
            return;

        setCurrentFormat(DigitGrouping, dIndex);

        setProperty("__DigitGrouping", -1);
    });
}

void DatetimeModel::setNTP(bool ntp)
{
    if (m_ntp != ntp) {
        m_ntp = ntp;
        Q_EMIT ntpChanged(ntp);
    }
}

void DatetimeModel::set24HourFormat(bool state)
{
    if (m_bUse24HourType != state) {
        m_bUse24HourType = state;
        Q_EMIT hourTypeChanged(state);
    }
}

void DatetimeModel::setDateTime(const QDateTime &dateTime)
{
    if (m_work)
        m_work->setDatetime(dateTime);
}

QStringList DatetimeModel::zones(int x, int y, int map_width, int map_height)
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    const double kDistanceThreshold = 64.0;
    auto zonelist = GetNearestZones(g_totalZones, kDistanceThreshold, x, y, map_width, map_height);

    return timeZoneList(zonelist, m_timezoneCache);
}

QPoint DatetimeModel::zonePosition(const QString &timezone, int map_width, int map_height)
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    auto enZone = m_timezoneCache.value(timezone, timezone);

    int index = GetZoneInfoByZone(g_totalZones, enZone);
    if (index < 0)
        return QPoint();

    auto currentZone = g_totalZones.at(index);

    const int x = int(ConvertLongitudeToX(currentZone.longitude) * map_width);
    const int y = int(ConvertLatitudeToY(currentZone.latitude) * map_height);
    return QPoint(x, y);
}

QStringList DatetimeModel::zoneIdList()
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    QStringList list;
    for (const auto& info : g_totalZones) {
        list << info.timezone;
    }

    return list;
}

QString DatetimeModel::zoneDisplayName(const QString &zoneName)
{
    if (m_work) {
        auto zoneInfo = m_work->GetZoneInfo(zoneName);
        QString utcOffsetText = getUtcOffsetText(zoneInfo.getUTCOffset());
        QString cityName = zoneInfo.getZoneCity().isEmpty() ? zoneInfo.getZoneName() : zoneInfo.getZoneCity();
        return QString("%1 %2").arg(utcOffsetText).arg(cityName);
    }
    return QString();
}

QSortFilterProxyModel *DatetimeModel::zoneSearchModel()
{
    if (m_zoneSearchModel)
        return m_zoneSearchModel;

    m_zoneSearchModel = new QSortFilterProxyModel(this);

    auto sourceModel = new dccV25::ZoneInfoModel(this);
    m_zoneSearchModel->setSourceModel(sourceModel);
    m_zoneSearchModel->setFilterRole(dccV25::ZoneInfoModel::SearchTextRole);
    m_zoneSearchModel->setFilterCaseSensitivity(Qt::CaseInsensitive);

    return m_zoneSearchModel;
}

QSortFilterProxyModel *DatetimeModel::langSearchModel()
{
    if (m_langSearchModel)
        return m_langSearchModel;

    m_langSearchModel = new QSortFilterProxyModel(this);

    ensureLangModel();

    auto sourceModel = new dccV25::LanguageListModel(this);
    sourceModel->setMetaData(m_langModel->langLists());
    sourceModel->setLocalLang(m_langModel->localLang());
    connect(m_langModel, &dccV25::KeyboardModel::langChanged, sourceModel, &dccV25::LanguageListModel::setMetaData);
    connect(m_langModel, &dccV25::KeyboardModel::curLocalLangChanged, sourceModel, &dccV25::LanguageListModel::setLocalLang);

    m_langSearchModel->setSourceModel(sourceModel);
    m_langSearchModel->setFilterRole(dccV25::LanguageListModel::SearchTextRole);
    m_langSearchModel->setFilterCaseSensitivity(Qt::CaseInsensitive);

    return m_langSearchModel;
}

QSortFilterProxyModel *DatetimeModel::langRegionSearchModel()
{
    if (m_regionSearchModel)
        return m_regionSearchModel;

    m_regionSearchModel = new QSortFilterProxyModel(this);

    auto sourceModel = new dccV25::LangRegionModel(this);
    m_regionSearchModel->setSourceModel(sourceModel);
    m_regionSearchModel->setFilterRole(dccV25::ZoneInfoModel::SearchTextRole);
    m_regionSearchModel->setFilterCaseSensitivity(Qt::CaseInsensitive);

    return m_regionSearchModel;
}

QSortFilterProxyModel *DatetimeModel::regionSearchModel()
{
    if (m_countrySearchModel)
        return m_countrySearchModel;

    for (const auto &locale : m_regions) {
        auto langCountry = translateLangAndCountry(locale.name());
        // { 中国: CN }
        m_langRegionsCache[langCountry.value(1)] = locale.territoryToCode(locale.territory());
    }

    m_countrySearchModel = new QSortFilterProxyModel(this);
    QStringListModel *sourceModel = new QStringListModel(m_langRegionsCache.keys());
    m_countrySearchModel->setSourceModel(sourceModel);
    m_countrySearchModel->setFilterCaseSensitivity(Qt::CaseInsensitive);

    return m_countrySearchModel;
}

QString DatetimeModel::region()
{
    if (m_regionName.isEmpty()) {
        QString localeName;
        for (const auto &locale : m_regions) {
            if (locale.territoryToString(locale.territory()) == m_country) {
                localeName = locale.name();
                break;
            }
            if (locale.territoryToCode(locale.territory()) == m_country) {
                localeName = locale.name();
                break;
            }
        }
        auto langCountry = translateLangAndCountry(localeName);
        m_regionName = langCountry.value(1);
    }

    return m_regionName;
}

int DatetimeModel::currentRegionIndex()
{
    return m_langRegionsCache.keys().indexOf(region());
}

void DatetimeModel::setRegion(const QString &region)
{
    if (m_regionName == region)
        return;

    m_regionName = region;
    auto reg = m_langRegionsCache.value(region, region);
    m_work->setConfigValue(country_key, reg);
    Q_EMIT regionChanged(region);
    Q_EMIT currentRegionIndexChanged(currentRegionIndex());
}

QStringList DatetimeModel::languagesAndRegions()
{
    QStringList langAndRegions;
    for (auto locale : m_regions) {
        const QString &langCountry = translate(locale.name(), m_regions.key(locale));
        langAndRegions << langCountry;
    }

    return langAndRegions;
}

QString DatetimeModel::currentLanguageAndRegion()
{
    return translate(localeName(), langRegion());
}

void DatetimeModel::setCurrentLocaleAndLangRegion(const QString &localeName, const QString& langAndRegion)
{
    QStringList langRegions = langAndRegion.split(":");
    if (langRegions.size() < 2) {
        qWarning() << "invalid langAndRegion" << langAndRegion;
        return;
    }

    if (!m_work)
        return;

    QLocale locale(localeName);
    m_work->setConfigValue(languageRegion_key, langAndRegion);
    m_work->setConfigValue(localeName_key, localeName);

    RegionFormat regionFormat = RegionProxy::regionFormat(locale);
    // case FirstDayOfWeek:
    m_work->setConfigValue(firstDayOfWeek_key, regionFormat.firstDayOfWeekFormat);
    // case ShortDate:
    m_work->setConfigValue(shortDateFormat_key, regionFormat.shortDateFormat);
    // case LongDate:
    m_work->setConfigValue(longDateFormat_key, regionFormat.longDateFormat);
    // case ShortTime:
    m_work->setConfigValue(shortTimeFormat_key, regionFormat.shortTimeFormat);
    // case LongTime:
    m_work->setConfigValue(longTimeFormat_key, regionFormat.longTimeFormat);
    // case Currency:
    m_work->setConfigValue(currencyFormat_key, regionFormat.currencyFormat.toUtf8());
    // case Digit:
    m_work->setConfigValue(numberFormat_key, regionFormat.numberFormat.toUtf8());
    m_work->setDigitGrouping(regionFormat.numberFormat.toUtf8());
    // case PaperSize:
    m_work->setConfigValue(paperFormat_key, regionFormat.paperFormat.toUtf8());

}

QStringList DatetimeModel::availableFormats(int format)
{
    QLocale locale(m_localeName);
    RegionAvailableData regionFormatsAvailable = RegionProxy::allTextData(locale);
    switch (format) {
    // date time formats
    case DayAbbreviations:
        return QStringList{ locale.standaloneDayName(1, QLocale::LongFormat), locale.standaloneDayName(1, QLocale::ShortFormat) };
    case DayOfWeek: {
        QStringList days;
        for (int i = 1; i < 8; ++i)
            days << locale.standaloneDayName(i, QLocale::LongFormat);
        return days;
    }
    case LongDate:
        return regionFormatsAvailable.longDatesAvailable;
    case ShortDate:
        return regionFormatsAvailable.shortDatesAvailable;
    case LongTime:
        return regionFormatsAvailable.longTimesAvailable;
    case ShortTime:
        return regionFormatsAvailable.shortTimesAvailable;
    // currency formats
    case CurrencySymbol: {
        QStringList defaultSymbols { QString::fromLocal8Bit("¥"),
                                QString::fromLocal8Bit("$"),
                                QString::fromLocal8Bit("€") };
        const QString &current = RegionProxy::regionFormat(locale).currencyFormat;
        if (!defaultSymbols.contains(current))
            defaultSymbols.prepend(current);

        return defaultSymbols;
    }
    case PositiveCurrency: {
        return getCurrencySymbol(true, currencyFormat());
    }
    case NegativeCurrency: {
        return getCurrencySymbol(false, currencyFormat());
    }
    // number formats
    case DecimalSymbol:{
        return separatorSymbol();
    }
    case DigitGroupingSymbol: {
        return separatorSymbol();
    }
    case DigitGrouping: {
        QString dgSymbol = m_work->digitGroupingSymbol();
        if (dgSymbol == DatetimeModel::tr("Space"))
            dgSymbol = " ";
        return {
            QString("123456789"),
            QString("%2%1%3%1%4").arg(dgSymbol).arg("123").arg("456").arg("789"),    // 123,456,789
            QString("%2%1%3").arg(dgSymbol).arg("123456").arg("789"),                // 123456,789
            QString("%2%1%3%1%4%1%5").arg(dgSymbol).arg("12").arg("34").arg("56").arg("789"), // 12,34,56,789
        };
    }
    case PageSize:
        return {"A4"};
    default:
        break;
    }

    return QStringList();
}

int DatetimeModel::currentFormatIndex(int format)
{
#define INDEX_OF(format, MEMBER, isDate) { \
        const QDate CurrentDate(2024, 1, 1); \
        const QTime CurrentTime(1, 1, 1); \
        QLocale locale(m_localeName); \
        RegionAvailableData regionFormatsAvailable = RegionProxy::allTextData(locale); \
        const auto &fmt = isDate ? locale.toString(CurrentDate, format) : locale.toString(CurrentTime, format); \
        return regionFormatsAvailable.MEMBER.indexOf(fmt); \
    }

    switch (format) {
    // date time formats
    case DayAbbreviations:
        return weekdayFormat();
    case DayOfWeek: {
        return firstDayOfWeekFormat() - 1; // combo index start from 0
    }
    case LongDate: {
        INDEX_OF(longDateFormat(), longDatesAvailable, true);
    }
    case ShortDate: {
        INDEX_OF(shortDateFormat(), shortDatesAvailable, true);
    }
    case LongTime:{
        INDEX_OF(longTimeFormat(), longTimesAvailable, false);
    }
    case ShortTime: {
        INDEX_OF(shortTimeFormat(), shortTimesAvailable, false);
    }
    // currency formats
    case CurrencySymbol: {
        const QString &currencySymbol = currencyFormat();
        QStringList defaultSymbols = availableFormats(format);
        return defaultSymbols.indexOf(currencySymbol);
    }
    case PositiveCurrency: {
        auto fmt = m_work->positiveCurrencyFormat();
        auto fmts = getCurrencySymbol(true, currencyFormat());
        return fmts.indexOf(fmt);
    }
    case NegativeCurrency: {
        auto fmt = m_work->negativeCurrencyFormat();
        auto fmts = getCurrencySymbol(false, currencyFormat());
        return fmts.indexOf(fmt);
    }
    // number formats
    case DecimalSymbol: {
        const QString &current = m_work->decimalSymbol();
        QStringList defaultSymbols = separatorSymbol();
        return defaultSymbols.indexOf(current);
    }
    case DigitGroupingSymbol: {
        const QString &current = m_work->digitGroupingSymbol();
        QStringList defaultSymbols = separatorSymbol();
        return defaultSymbols.indexOf(current);
    }
    case DigitGrouping: {
        const QString &current = m_work->digitGrouping();
        QStringList defaultSymbols = availableFormats(format);
        return defaultSymbols.indexOf(current);
    }
    default:
        break;
    }

    return 0;
}

void DatetimeModel::setCurrentFormat(int format, int index)
{
    if (index < 0) {
        qWarning() << "Invalide index!";
        return;
    }

    RegionAvailableData regionFormat = RegionProxy::allFormat();
    QLocale locale(m_localeName);
    // const QString &symbol = RegionProxy::regionFormat(locale).currencyFormat;

    auto setConfig = [this](int index, const QString &key, const QStringList &availableList) {
        if (index < availableList.count()) {
            m_work->setConfigValue(key, availableList.at(index));
        } else {
            qWarning() << "Set [" << key << "] faild, invalid index" << index << availableList.count();
        }
    };

    switch (format) {
    // date time formats
    case DayAbbreviations: {
        setWeekdayFormat(index);
        break;
    }
    case DayOfWeek: {
        // dconfig
        m_work->setConfigValue(firstDayOfWeek_key, index + 1);
        // dbus
        m_work->setWeekStartDayFormat(index + 1);
        break;
    }
    case LongDate: {
        setConfig(index, longDateFormat_key, regionFormat.longDatesAvailable);
        break;
    }
    case ShortDate: {
        setConfig(index, shortDateFormat_key, regionFormat.shortDatesAvailable);
        break;
    }
    case LongTime:{
        setConfig(index, longTimeFormat_key, regionFormat.longTimesAvailable);
        break;
    }
    case ShortTime: {
        setConfig(index, shortTimeFormat_key, regionFormat.shortTimesAvailable);
        break;
    }
    // currency formats
    case CurrencySymbol: {
        QStringList defaultSymbols = availableFormats(format);
        if (index >= defaultSymbols.count())
            return;
        // get PositiveCurrency/NegativeCurrency index first
        int pIndex = currentFormatIndex(PositiveCurrency);
        int nIndex = currentFormatIndex(NegativeCurrency);
        setProperty("__PositiveCurrency", pIndex);
        setProperty("__NegativeCurrency", nIndex);

        // dconfig
        setConfig(index, currencyFormat_key, defaultSymbols);
        // dbus
        m_work->setCurrencySymbol(defaultSymbols.value(index));
    }
    break;
    case PositiveCurrency: {
        auto fmts = getCurrencySymbol(true, currencyFormat());
        if (index < fmts.count())
            m_work->setPositiveCurrencyFormat(fmts.value(index));
    }
    break;
    case NegativeCurrency: {
        auto fmts = getCurrencySymbol(false, currencyFormat());
        if (index < fmts.count())
            m_work->setNegativeCurrencyFormat(fmts.value(index));
    }
    break;
    // number formats
    case DecimalSymbol: {
        auto fmts = separatorSymbol();
        if (index < fmts.count())
            m_work->setDecimalSymbol(fmts.value(index));
    }
    break;
    case DigitGroupingSymbol: {
        int dIndex = currentFormatIndex(DigitGrouping);
        setProperty("__DigitGrouping", dIndex);

        auto fmts = separatorSymbol();
        if (index < fmts.count())
            m_work->setDigitGroupingSymbol(fmts.value(index));
    }
    break;
    case DigitGrouping: {
        QStringList fmts = availableFormats(format);
        if (index >= fmts.count())
            return

        // dconfig
        setConfig(index, numberFormat_key, fmts);
        // dbus
        m_work->setDigitGrouping(fmts.value(index));
    }
    break;
    default:
        break;
    }
}

QString DatetimeModel::currentDate()
{
    QLocale locale(m_localeName);
    QString dateFormat = longDateFormat();
    if (weekdayFormat() == 1)
        dateFormat.replace("dddd", "ddd");

    return locale.toString(QDate::currentDate(), dateFormat);
}

int DatetimeModel::currentLanguageAndRegionIndex()
{
    return m_regions.keys().indexOf(m_langCountry);
}

void DatetimeModel::addUserTimeZoneByName(const QString &zoneName)
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    if (!m_timezoneCache.contains(zoneName))
        timeZoneList(g_totalZones, m_timezoneCache);

    if (!m_timezoneCache.contains(zoneName)) {
        qWarning() << "timezone cache not contain.." << zoneName;
        return;
    }
    QString zoneId = m_timezoneCache.value(zoneName);

    if (m_userZoneIds.contains(zoneId)) {
        qWarning() << "user timezone already existed";
        return;
    }

    addUserTimeZoneById(zoneId);
}

void DatetimeModel::addUserTimeZoneById(const QString &zoneId)
{
    if (m_work)
        m_work->addUserTimeZone(zoneId);
}
void DatetimeModel::removeUserTimeZoneByName(const QString &name)
{
    // displayText list
    auto zonelist = userTimeZoneText(0);
    int index = zonelist.indexOf(name);
    if (index < 0) {
      qWarning() << name << "Not found in User TimeZones";
        return;
    }

    removeUserTimeZone(m_userTimeZones.value(index));
}

#ifndef DCC_DISABLE_TIMEZONE
QString DatetimeModel::systemTimeZoneId() const
{
    return m_systemTimeZoneId;
}

void DatetimeModel::setSystemTimeZoneId(const QString &systemTimeZoneId)
{
    if (m_systemTimeZoneId != systemTimeZoneId) {
        m_systemTimeZoneId = systemTimeZoneId;
        Q_EMIT systemTimeZoneIdChanged(systemTimeZoneId);
    }
}
#endif

QList<ZoneInfo> DatetimeModel::userTimeZones() const
{
    return m_userTimeZones;
}

void DatetimeModel::addUserTimeZone(const ZoneInfo &zone)
{
    const QString zoneName = zone.getZoneName();

    if (!m_userZoneIds.contains(zoneName)) {
        m_userZoneIds.append(zoneName);
        m_userTimeZones.append(zone);
        Q_EMIT userTimeZoneAdded(zone);
    }
}

void DatetimeModel::removeUserTimeZone(const ZoneInfo &zone)
{
    const QString zoneName = zone.getZoneName();

    if (m_userZoneIds.contains(zoneName)) {
        m_userZoneIds.removeAll(zoneName);
        m_userTimeZones.removeAll(zone);
        Q_EMIT userTimeZoneRemoved(zone);
    }
}

void DatetimeModel::setCurrentTimeZone(const ZoneInfo &currentTimeZone)
{
    if (m_currentTimeZone == currentTimeZone)
        return;

    m_currentTimeZone = currentTimeZone;

    Q_EMIT currentTimeZoneChanged(currentTimeZone);
}

void DatetimeModel::setCurrentUseTimeZone(const ZoneInfo &currentSysTimeZone)
{
    if (m_currentSystemTimeZone == currentSysTimeZone)
        return;

    m_currentSystemTimeZone = currentSysTimeZone;

    Q_EMIT currentSystemTimeZoneChanged(currentSysTimeZone);
}

void DatetimeModel::setNtpServerAddress(const QString &ntpServer)
{
    if (m_strNtpServerAddress != ntpServer) {
        m_strNtpServerAddress = ntpServer;
        Q_EMIT NTPServerChanged(ntpServer);
    }
}

void DatetimeModel::setNTPServerList(const QStringList &list)
{
    if (m_NtpServerList != list) {
        m_NtpServerList = list;
        Q_EMIT NTPServerListChanged(list);
    }
}

void DatetimeModel::setTimeZoneInfo(const QString &timeZone)
{
    if (m_timeZones != timeZone) {
        m_timeZones = timeZone;
        Q_EMIT timeZoneChanged(timeZone);
    }
}

void DatetimeModel::setCountry(const QString &country)
{
    if (m_country != country) {
        m_country = country;
        Q_EMIT countryChanged(country);
    }
}

void DatetimeModel::setLocaleName(const QString &localeName)
{
    if (m_localeName != localeName) {
        m_localeName = localeName;
        Q_EMIT localeNameChanged(localeName);
        Q_EMIT currentLanguageAndRegionChanged(currentLanguageAndRegion());
    }
}

void DatetimeModel::setLangRegion(const QString &langCountry)
{
    if (m_langCountry != langCountry) {
        m_langCountry = langCountry;
        Q_EMIT langCountryChanged(langCountry);
        Q_EMIT currentLanguageAndRegionChanged(currentLanguageAndRegion());
    }
}

void DatetimeModel::setFirstDayOfWeek(const int &firstDayOfWeekFormat)
{
    if (m_firstDayOfWeekFormat != firstDayOfWeekFormat) {
        m_firstDayOfWeekFormat = firstDayOfWeekFormat;
        Q_EMIT firstDayOfWeekFormatChanged(firstDayOfWeekFormat);
    }
}

void DatetimeModel::setShortDateFormat(const QString &shortDateFormat)
{
    if (m_shortDateFormat != shortDateFormat) {
        m_shortDateFormat = shortDateFormat;
        Q_EMIT shortDateFormatChanged(shortDateFormat);
    }
}

void DatetimeModel::setLongDateFormat(const QString &longDateFormat)
{
    if (m_longDateFormat != longDateFormat) {
        m_longDateFormat = longDateFormat;
        Q_EMIT longDateFormatChanged(longDateFormat);
    }
}

void DatetimeModel::setShortTimeFormat(const QString &shortTimeFormat)
{
    if (m_shortTimeFormat != shortTimeFormat) {
        m_shortTimeFormat = shortTimeFormat;
        Q_EMIT shortTimeFormatChanged(shortTimeFormat);
    }
}

void DatetimeModel::setLongTimeFormat(const QString &longTimeFormat)
{
    if (m_longTimeFormat != longTimeFormat) {
        m_longTimeFormat = longTimeFormat;
        Q_EMIT longTimeFormatChanged(longTimeFormat);
    }
}

void DatetimeModel::setCurrencyFormat(const QString &currencyFormat)
{
    if (m_currencyFormat != currencyFormat) {
        m_currencyFormat = currencyFormat;
        Q_EMIT currencyFormatChanged(currencyFormat);
    }
}

void DatetimeModel::setNumberFormat(const QString &numberFormat)
{
    if (m_numberFormat != numberFormat) {
        m_numberFormat = numberFormat;
        Q_EMIT numberFormatChanged(numberFormat);
    }
}

void DatetimeModel::setPaperFormat(const QString &paperFormat)
{
    if (m_paperFormat != paperFormat) {
        m_paperFormat = paperFormat;
        Q_EMIT paperFormatChanged(paperFormat);
    }
}

void DatetimeModel::setRegionFormat(const RegionFormat &regionFormat)
{
    if (m_regionFormat != regionFormat) {
        m_regionFormat = regionFormat;
    }
}

void DatetimeModel::setCountries(const QStringList &countries)
{
    if (m_countries != countries) {
        m_countries = countries;
        Q_EMIT countriesChanged(countries);
    }
}

void DatetimeModel::setRegions(const Regions &regions)
{
    if (m_regions != regions) {
        m_regions = regions;
    }
}

QStringList DatetimeModel::userTimeZoneText(int index) const
{
    QStringList userTimeZoneList;
    for (const ZoneInfo &zoneInfo : m_userTimeZones) {
        QString text;
        switch (index) {
        case 1:
            text = getDescription(zoneInfo);
            break;
        case 2:
            text = QString::number(zoneInfo.getUTCOffset());
            break;
        case 3:
            text = zoneInfo.getZoneName();
            break;
        case 4:
            text = zoneInfo.getZoneCity();
            break;
        default:
            text = getDisplayText(zoneInfo);
            break;
        }
        userTimeZoneList << text;
    }

    return userTimeZoneList;
}

QString DatetimeModel::currentTimeZoneName() const
{
    return getDescription(m_currentTimeZone);
}

QString DatetimeModel::timeZoneDispalyName() const
{
    return getDisplayText(m_currentSystemTimeZone);
}

int DatetimeModel::currentTimeZoneIndex() const
{
    using namespace installer;
    if (g_totalZones.empty())
        g_totalZones =  GetZoneInfoList();

    int index = -1;
    const QString &zoneName = m_currentSystemTimeZone.getZoneName();
    for (int i = 0; i < g_totalZones.size(); ++i) {
        const auto &zoneInfo = g_totalZones.value(i);
        if (zoneName == zoneInfo.timezone) {
            index = i;
            break;
        }
    }

    return index;
}

void DatetimeModel::ensureLangModel()
{
    if (m_langModel)
        return;

    m_langModel = new dccV25::KeyboardModel(this);
    connect(m_langModel, &dccV25::KeyboardModel::curLocalLangChanged, this, &DatetimeModel::langListChanged);
    connect(m_langModel, &dccV25::KeyboardModel::curLangChanged, this, &DatetimeModel::currentLangChanged);
}

void DatetimeModel::addLang(const QString &lang)
{
    ensureLangModel();
    m_langModel->addLang(lang);
}

void DatetimeModel::deleteLang(const QString &lang)
{
    ensureLangModel();
    m_langModel->deleteLang(lang);
}

void DatetimeModel::setCurrentLang(const QString &lang)
{
    ensureLangModel();
    m_langModel->doSetLang(lang);
}

QStringList DatetimeModel::langList()
{
    ensureLangModel();

    if (m_langModel)
        return m_langModel->localLang();

    return {};
}

QString DatetimeModel::currentLang()
{
    ensureLangModel();

    if (m_langModel)
        return m_langModel->curLang();

    return {};
}

int DatetimeModel::weekdayFormat() const
{
    return m_work ? m_work->weekdayFormat() : 0;
}

void DatetimeModel::setWeekdayFormat(int newWeekdayFormat)
{
    if (!m_work)
        return;

    m_work->setWeekdayFormat(newWeekdayFormat);
}

DCC_FACTORY_CLASS(DatetimeModel)

#include "datetimemodel.moc"
