# Copyright (c) 2001						RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
#------------------------------------------------------------------------------
# Module Header
# Filename          : NCC::CountryCode.pm
# Author            : Timur I. Bakeyev <timur@ripe.net>
# Date              : 200104
# Description       : Provides OO interface to the list of country codes
# Language Version  : Perl 5.6.0
# OSs Tested        : BSD/OS 3.1
# Command Line      : -
# Input Files       : -
# Output Files      : -
# External Programs : -
# Problems          : -
# Comments          : 
# $Id: CountryCode.pm,v 1.5 2001/09/10 15:55:01 timur Exp $
#------------------------------------------------------------------------------

package NCC::CountryCode;

use strict;
use vars qw($VERSION @ISA);

require Exporter;

@ISA = qw(Exporter);

$VERSION = '0.04';

# Location of the local is 3166 file
my $codes = '/usr/local/etc/NCC/iso3166-codes';


####################################################################
#  DESCRIPTION: Constructor
#        INPUT: None
#       OUTPUT: New instance of the class or undef
# SIDE EFFECTS: None
####################################################################

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my %param = @_;
    my $self = { };
    
    bless $self, $class;
    # Don't return object, if failed to initialiaze
    return unless($self->_init());

    return $self;
}

####################################################################
#  DESCRIPTION: Read country code list and put it into the hash
#        INPUT: None
#       OUTPUT: 1, if initialization was successful
#		0, otherwise
# SIDE EFFECTS: None
####################################################################

sub _init {
    my $self = shift;
    
    # Try to use external data
    if(open(FILE, $codes)) {
	while(<FILE>) {
	    if(m%^(.+?)\s+([A-Z]{2})\s+([A-Z]{3})\s+(\d{3})\s*$%) {
	    
		$self->{'CC'}{uc($2)} = uc($1);
	    }
	}
	close(FILE);
    }
    else {
	# Get the begining of the DATA stream
	my $filepos = tell(DATA);
	# Load list of country codes
	while(<DATA>) {
	    if(m%^(.+?)\s+([A-Z]{2})\s+([A-Z]{3})\s+(\d{3})\s*$%) {
		
		$self->{'CC'}{uc($2)} = uc($1);
	    }
	}
	# Rewind to the begining of the DATA stream
	seek(DATA, $filepos, 0);
    }
    # XXX: Hack(tm)
    # UK and EU are non-official, but widely used country codes, so
    # we insert the information about them.
    $self->{'CC'}{'UK'} = $self->{'CC'}{'GB'};
    $self->{'CC'}{'EU'} = 'EUROPE';
    # Success
    return 1;
}

####################################################################
#  DESCRIPTION: Return a reference to the hash of country codes
#		and assotiated countries
#        INPUT: None
#       OUTPUT: Reference to the hash
# SIDE EFFECTS: None
####################################################################

sub getCCs {
    my $self = shift;
    
    # return the CC hash ref
    return $self->{'CC'};
}

####################################################################
#  DESCRIPTION: Check if passed CC is valid
#        INPUT: Country code
#       OUTPUT: Valid CC or empty string, if no
# SIDE EFFECTS: None
####################################################################

sub isCC {
    my $self = shift;
    my($cc) = @_;
    
    return '' unless($cc);
    # Uppercase the possible country code
    $cc = uc($cc);
    
    my $ret = $self->{'CC'}{$cc};
    # Return country code if there is one or nothing.
    return ($ret) ? $cc : '';
}

####################################################################
#  DESCRIPTION: Return country name by country code
#        INPUT: Country code
#       OUTPUT: Country name or empty string if no
# SIDE EFFECTS: None
####################################################################

sub CC2Country {
    my $self = shift;
    my($cc) = @_;
    
    return '' unless($cc);
    # Uppercase the possible country code
    $cc = uc($cc);
    
    my $ret = $self->{'CC'}{$cc};
    # Return country name if there is one or nothing.
    return ($ret) ? $ret : '';
}

####################################################################
#  DESCRIPTION: Return country code(s) by country name
#        INPUT: Country name
#       OUTPUT: Hash of matching country codes with country names
#		or a string in scalar context
# SIDE EFFECTS: Return nothing if passed string less than 4 chars
####################################################################

sub Country2CC {
    my $self = shift;
    my($country) = @_;
    
    return '' unless($country);
    # Skip too short names
    return if(length($country) < 4);
    # Quote possible meta chars
    $country = quotemeta($country);
    
    # The resulting hash
    my %ret = ();
    
    foreach my $cc (keys(%{$self->{'CC'}})) {
	# If passed string is a part of country name - store the result
	if($self->{'CC'}{$cc} =~ /$country/i) {
	    $ret{$cc} = $self->{'CC'}{$cc};
	}
    }
    # Return the result
    if(wantarray()) {
	# As a hash(array)
	return(%ret);
    }
    else {
	# As a scalar
	my $ret = '';
	# Walk through the hash and convert it into scalar
	foreach my $cc (sort(keys(%ret))) {
	    $ret .= sprintf("%s: %s; ", $cc, $ret{$cc});
	}
	return $ret;
    }
}

1;

=head1 NAME

NCC::CountryCode - Perl extension for blah blah blah

=head1 SYNOPSIS

  use NCC::CountryCode;
  
  my $cc = new NCC::CountryCode();
  
  my $cc_hash = $cc->getCCs();
  
  print ($cc->isCC('NL')) ? "exists" : "non-existing";
  
  $cc->CC2Country('fr');
  
  $cc->Country2CC('russia');

=head1 DESCRIPTION

This module provides class and several methods to simplify mapping between
country names and country codes, as they assigned in the ISO3166.

The CC2Country() method maps country code to the corresponding country name
or empty string, if it doesn\'t exist.

The Country2CC() tries to map passed country name into country code. As the 
spelling of the name may vary, this function tries to find all possible matches
for passed name. In array context method returns hash of country codes and
names, in scalar - a string, containing all possible country codes and corresponding
country names.

The isCC() method verifies, that passed country code is a legal one and returns
it in upper case. Otherwise it returns empty string.

The getCCs() method returns the reference to the internal hash, that containes
all the country codes with corresponding them country names.

=head1 BUGS

In addition to the country codes defined by ISO3166 standart this module also
introduce 'UK' as a synonim for 'GB' and 'EU' as additional 'virtual' name for
the Europe itself.

=head1 AUTHOR

Timur Bakeyev, timur@ripe.net

=head1 SEE ALSO

perl(1).

=cut

__DATA__
Some Codes from ISO 3166 

Updated by the RIPE Network Coordination Centre.

Source: ISO 3166 Maintenance Agency 

Latest change: Thu Aug  7 17:59:51 MET DST 1997

Country                                         A 2     A 3     Number
----------------------------------------------------------------------
AFGHANISTAN                                     AF      AFG     004
ALBANIA                                         AL      ALB     008
ALGERIA                                         DZ      DZA     012
AMERICAN SAMOA                                  AS      ASM     016
ANDORRA                                         AD      AND     020
ANGOLA                                          AO      AGO     024
ANGUILLA                                        AI      AIA     660
ANTARCTICA                                      AQ      ATA     010
ANTIGUA AND BARBUDA                             AG      ATG     028
ARGENTINA                                       AR      ARG     032
ARMENIA                                         AM      ARM     051
ARUBA                                           AW      ABW     533
AUSTRALIA                                       AU      AUS     036
AUSTRIA                                         AT      AUT     040
AZERBAIJAN                                      AZ      AZE     031
BAHAMAS                                         BS      BHS     044
BAHRAIN                                         BH      BHR     048
BANGLADESH                                      BD      BGD     050
BARBADOS                                        BB      BRB     052
BELARUS                                         BY      BLR     112
BELGIUM                                         BE      BEL     056
BELIZE                                          BZ      BLZ     084
BENIN                                           BJ      BEN     204
BERMUDA                                         BM      BMU     060
BHUTAN                                          BT      BTN     064
BOLIVIA                                         BO      BOL     068
BOSNIA AND HERZEGOWINA                          BA      BIH     070
BOTSWANA                                        BW      BWA     072
BOUVET ISLAND                                   BV      BVT     074
BRAZIL                                          BR      BRA     076
BRITISH INDIAN OCEAN TERRITORY                  IO      IOT     086
BRUNEI DARUSSALAM                               BN      BRN     096
BULGARIA                                        BG      BGR     100
BURKINA FASO                                    BF      BFA     854
BURUNDI                                         BI      BDI     108
CAMBODIA                                        KH      KHM     116
CAMEROON                                        CM      CMR     120
CANADA                                          CA      CAN     124
CAPE VERDE                                      CV      CPV     132
CAYMAN ISLANDS                                  KY      CYM     136
CENTRAL AFRICAN REPUBLIC                        CF      CAF     140
CHAD                                            TD      TCD     148
CHILE                                           CL      CHL     152
CHINA                                           CN      CHN     156
CHRISTMAS ISLAND                                CX      CXR     162
COCOS (KEELING) ISLANDS                         CC      CCK     166
COLOMBIA                                        CO      COL     170
COMOROS                                         KM      COM     174
CONGO                                           CG      COG     178
CONGO, THE DEMOCRATIC REPUBLIC OF THE           CD      COD     180
COOK ISLANDS                                    CK      COK     184
COSTA RICA                                      CR      CRI     188
COTE D'IVOIRE                                   CI      CIV     384
CROATIA (local name: Hrvatska)                  HR      HRV     191
CUBA                                            CU      CUB     192
CYPRUS                                          CY      CYP     196
CZECH REPUBLIC                                  CZ      CZE     203
DENMARK                                         DK      DNK     208
DJIBOUTI                                        DJ      DJI     262
DOMINICA                                        DM      DMA     212
DOMINICAN REPUBLIC                              DO      DOM     214
EAST TIMOR                                      TP      TMP     626
ECUADOR                                         EC      ECU     218
EGYPT                                           EG      EGY     818
EL SALVADOR                                     SV      SLV     222
EQUATORIAL GUINEA                               GQ      GNQ     226
ERITREA                                         ER      ERI     232
ESTONIA                                         EE      EST     233
ETHIOPIA                                        ET      ETH     231
FALKLAND ISLANDS (MALVINAS)                     FK      FLK     238
FAROE ISLANDS                                   FO      FRO     234
FIJI                                            FJ      FJI     242
FINLAND                                         FI      FIN     246
FRANCE                                          FR      FRA     250
FRANCE, METROPOLITAN                            FX      FXX     249
FRENCH GUIANA                                   GF      GUF     254
FRENCH POLYNESIA                                PF      PYF     258
FRENCH SOUTHERN TERRITORIES                     TF      ATF     260
GABON                                           GA      GAB     266
GAMBIA                                          GM      GMB     270
GEORGIA                                         GE      GEO     268
GERMANY                                         DE      DEU     276
GHANA                                           GH      GHA     288
GIBRALTAR                                       GI      GIB     292
GREECE                                          GR      GRC     300
GREENLAND                                       GL      GRL     304
GRENADA                                         GD      GRD     308
GUADELOUPE                                      GP      GLP     312
GUAM                                            GU      GUM     316
GUATEMALA                                       GT      GTM     320
GUINEA                                          GN      GIN     324
GUINEA-BISSAU                                   GW      GNB     624
GUYANA                                          GY      GUY     328
HAITI                                           HT      HTI     332
HEARD AND MC DONALD ISLANDS                     HM      HMD     334
HOLY SEE (VATICAN CITY STATE)                   VA      VAT     336
HONDURAS                                        HN      HND     340
HONG KONG                                       HK      HKG     344
HUNGARY                                         HU      HUN     348
ICELAND                                         IS      ISL     352
INDIA                                           IN      IND     356
INDONESIA                                       ID      IDN     360
IRAN (ISLAMIC REPUBLIC OF)                      IR      IRN     364
IRAQ                                            IQ      IRQ     368
IRELAND                                         IE      IRL     372
ISRAEL                                          IL      ISR     376
ITALY                                           IT      ITA     380
JAMAICA                                         JM      JAM     388
JAPAN                                           JP      JPN     392
JORDAN                                          JO      JOR     400
KAZAKHSTAN                                      KZ      KAZ     398
KENYA                                           KE      KEN     404
KIRIBATI                                        KI      KIR     296
KOREA, DEMOCRATIC PEOPLE'S REPUBLIC OF          KP      PRK     408
KOREA, REPUBLIC OF                              KR      KOR     410
KUWAIT                                          KW      KWT     414
KYRGYZSTAN                                      KG      KGZ     417
LAO PEOPLE'S DEMOCRATIC REPUBLIC                LA      LAO     418
LATVIA                                          LV      LVA     428
LEBANON                                         LB      LBN     422
LESOTHO                                         LS      LSO     426
LIBERIA                                         LR      LBR     430
LIBYAN ARAB JAMAHIRIYA                          LY      LBY     434
LIECHTENSTEIN                                   LI      LIE     438
LITHUANIA                                       LT      LTU     440
LUXEMBOURG                                      LU      LUX     442
MACAU                                           MO      MAC     446
MACEDONIA, THE FORMER YUGOSLAV REPUBLIC OF      MK      MKD     807
MADAGASCAR                                      MG      MDG     450
MALAWI                                          MW      MWI     454
MALAYSIA                                        MY      MYS     458
MALDIVES                                        MV      MDV     462
MALI                                            ML      MLI     466
MALTA                                           MT      MLT     470
MARSHALL ISLANDS                                MH      MHL     584
MARTINIQUE                                      MQ      MTQ     474
MAURITANIA                                      MR      MRT     478
MAURITIUS                                       MU      MUS     480
MAYOTTE                                         YT      MYT     175
MEXICO                                          MX      MEX     484
MICRONESIA, FEDERATED STATES OF                 FM      FSM     583
MOLDOVA, REPUBLIC OF                            MD      MDA     498
MONACO                                          MC      MCO     492
MONGOLIA                                        MN      MNG     496
MONTSERRAT                                      MS      MSR     500
MOROCCO                                         MA      MAR     504
MOZAMBIQUE                                      MZ      MOZ     508
MYANMAR                                         MM      MMR     104
NAMIBIA                                         NA      NAM     516
NAURU                                           NR      NRU     520
NEPAL                                           NP      NPL     524
NETHERLANDS                                     NL      NLD     528
NETHERLANDS ANTILLES                            AN      ANT     530
NEW CALEDONIA                                   NC      NCL     540
NEW ZEALAND                                     NZ      NZL     554
NICARAGUA                                       NI      NIC     558
NIGER                                           NE      NER     562
NIGERIA                                         NG      NGA     566
NIUE                                            NU      NIU     570
NORFOLK ISLAND                                  NF      NFK     574
NORTHERN MARIANA ISLANDS                        MP      MNP     580
NORWAY                                          NO      NOR     578
OMAN                                            OM      OMN     512
PAKISTAN                                        PK      PAK     586
PALAU                                           PW      PLW     585
PALESTINIAN TERRITORY, OCCUPIED                 PS      PSE     275
PANAMA                                          PA      PAN     591
PAPUA NEW GUINEA                                PG      PNG     598
PARAGUAY                                        PY      PRY     600
PERU                                            PE      PER     604
PHILIPPINES                                     PH      PHL     608
PITCAIRN                                        PN      PCN     612
POLAND                                          PL      POL     616
PORTUGAL                                        PT      PRT     620
PUERTO RICO                                     PR      PRI     630
QATAR                                           QA      QAT     634
REUNION                                         RE      REU     638
ROMANIA                                         RO      ROM     642
RUSSIAN FEDERATION                              RU      RUS     643
RWANDA                                          RW      RWA     646
SAINT KITTS AND NEVIS                           KN      KNA     659
SAINT LUCIA                                     LC      LCA     662
SAINT VINCENT AND THE GRENADINES                VC      VCT     670
SAMOA                                           WS      WSM     882
SAN MARINO                                      SM      SMR     674
SAO TOME AND PRINCIPE                           ST      STP     678
SAUDI ARABIA                                    SA      SAU     682
SENEGAL                                         SN      SEN     686
SEYCHELLES                                      SC      SYC     690
SIERRA LEONE                                    SL      SLE     694
SINGAPORE                                       SG      SGP     702
SLOVAKIA (Slovak Republic)                      SK      SVK     703
SLOVENIA                                        SI      SVN     705
SOLOMON ISLANDS                                 SB      SLB     090
SOMALIA                                         SO      SOM     706
SOUTH AFRICA                                    ZA      ZAF     710
SOUTH GEORGIA AND THE SOUTH SANDWICH ISLANDS    GS      SGS     239
SPAIN                                           ES      ESP     724
SRI LANKA                                       LK      LKA     144
ST. HELENA                                      SH      SHN     654
ST. PIERRE AND MIQUELON                         PM      SPM     666
SUDAN                                           SD      SDN     736
SURINAME                                        SR      SUR     740
SVALBARD AND JAN MAYEN ISLANDS                  SJ      SJM     744
SWAZILAND                                       SZ      SWZ     748
SWEDEN                                          SE      SWE     752
SWITZERLAND                                     CH      CHE     756
SYRIAN ARAB REPUBLIC                            SY      SYR     760
TAIWAN, PROVINCE OF CHINA                       TW      TWN     158
TAJIKISTAN                                      TJ      TJK     762
TANZANIA, UNITED REPUBLIC OF                    TZ      TZA     834
THAILAND                                        TH      THA     764
TOGO                                            TG      TGO     768
TOKELAU                                         TK      TKL     772
TONGA                                           TO      TON     776
TRINIDAD AND TOBAGO                             TT      TTO     780
TUNISIA                                         TN      TUN     788
TURKEY                                          TR      TUR     792
TURKMENISTAN                                    TM      TKM     795
TURKS AND CAICOS ISLANDS                        TC      TCA     796
TUVALU                                          TV      TUV     798
UGANDA                                          UG      UGA     800
UKRAINE                                         UA      UKR     804
UNITED ARAB EMIRATES                            AE      ARE     784
UNITED KINGDOM                                  GB      GBR     826
UNITED STATES                                   US      USA     840
UNITED STATES MINOR OUTLYING ISLANDS            UM      UMI     581
URUGUAY                                         UY      URY     858
UZBEKISTAN                                      UZ      UZB     860
VANUATU                                         VU      VUT     548
VENEZUELA                                       VE      VEN     862
VIET NAM                                        VN      VNM     704
VIRGIN ISLANDS (BRITISH)                        VG      VGB     092
VIRGIN ISLANDS (U.S.)                           VI      VIR     850
WALLIS AND FUTUNA ISLANDS                       WF      WLF     876
WESTERN SAHARA                                  EH      ESH     732
YEMEN                                           YE      YEM     887
YUGOSLAVIA                                      YU      YUG     891
ZAMBIA                                          ZM      ZMB     894
ZIMBABWE                                        ZW      ZWE     716

----------------------------------------------------------------------

List of changes applied, as specified in registration newsletters:

Newsletter III-1, 1989-12-5:
  Burma deleted, Myanmar added (same numeric value, change of country name)

Newsletter III-2, 1990-07-16
  Namibia, changing information not included in this file

Newsletter III-3, 1990-08-14
  Afghanistan, changing information not included in this file

Newsletter III-4, 1990-08-14
  Ethiopia, changing information not included in this file

Newsletter III-5, 1990-08-14
  Fiji, changing information not included in this file

Newsletter III-6, 1990-08-14
  Hungary, changing information not included in this file

Newsletter III-7, 1990-08-14
  Unification of Yemen, under new numeric code

Newsletter III-8, 1990-08-14
  Romania, changing information not included in this file

Newsletter III-9, 1990-08-14
  Poland, changing information not included in this file

Newsletter III-10, 1990-08-14
  Kampuchea deleted, Cambodia added (same numeric value, change of name)

Newsletter III-11, 1990-08-14
  Benin, changing information not included in this file

Newsletter III-12, 1990-12-04
  Czechoslovakia, changing information not included in this file

Newsletter III-13, 1990-10-30
  Germany unified (DDR deleted, new name and numeric code for unified Germany)

Newsletter III-14 1991-02-10
  Mozambique, changing information not included in this file

Newsletter III-15 1991-02-10
  Bulgaria, changing information not included in this file

Newsletter III-16 1992-06-15
  ESTONIA added (EE, EST, 233)

Newsletter III-17 1992-06-15
  LATVIA added (LV, LVA, 428)

Newsletter III-18 1992-06-15
  Lithuania added

Newsletter III-19 1992-06-15
  Belarus, named and 3c changed

Newsletter III-20 1992-04-06
  Albania, changing information not included in this file

Newsletter III-21 1992-04-06
  Congo, changing information not included in this file

Newsletter III-22 1992-04-19
  Micronesia, name changed to Micronesia (Federated States of)

Newsletter III-23 1992-04-19
  Ukraine, name changed

Newsletter III-24 1993-06-18
  France, Metropolitan (FX) added (European part of France thus excluding:
                GF, GP, MQ, NC, PF, PM, RE, TF, WF, YT)

Newsletter III-25, 1991-12-18
  Pitcairn, changing information not included in this file

Newsletter III-26, 1992-06-15
  Croatia, added NR, HRV, 191

Newsletter III-27, 1992-06-15
  Armenia, added AM, ARM, 051

Newsletter III-28, 1992-08-28
  Georgia added

Newsletter III-29, 1992-06-15
  Russian Federation added

Newsletter III-30, 1992-06-15
  Turkmenistan added

Newsletter III-31, 1992-06-15
  Kazakhstan added

Newsletter III-32, 1992-06-15
  Kyrgyzstan added

Newsletter III-33, 1992-06-15
  Tajikistan added

Newsletter III-34, 1992-06-15
  Uzbekistan added

Newsletter III-35, 1992-06-15
  Azerbaijan added

Newsletter III-36 1992-06-15
  MOLDOVA, REPUBLIC OF added (MD, MDA, 498)

Newsletter III-37 1992-08-30
  USSR deleted

Newsletter III-38, 1993-06-15
  Slovenia, number code 705 assigned

Newsletter III-39 (undated)
  BOSNIA AND HERZEGOWINA added (BA, BIH, 070)

Newsletter III-40 1993-07-12
  YEMEN, REPUBLIC OF changed to YEMEN

Newsletter III-41 1993-07-28
  MAURITIUS, changes outside this document

Newletter III-42 1993-07-12
  SAINT VINCENT AND THE GRENADINES, changes outside this document

Newsletter III-43 1993-07-12
  MONGOLIA, changes outside this document

Newsletter III-44 1993-07-22
  PANAMA, changed numeric code from 590 to 591

Newsletter III-45 1993-07-28
  YUGOSLAVIA changed numeric code from 890 to 891

Newsletter III-46 1993-07-12
  NEUTRAL ZONE deleted

Newsletter III-47 1993-07-12
  NETHERLANDS ANTILLES changed numeric code from 532 to 530

Newsletter III-48, 1993-07-12
  Added MAYOTTE

Newsletter III-49, 1993-06-15
  Slovakia added

Newsletter III-50, 1993-06-15
  Czech Republic added

Newsletter III-51, 1993-06-15
  Czechoslovakia officially deleted

Newsletter III-52, 1993-07-02
  Angola, changing information not included in this file. Offical name
  change to Republic of Angola

Newsletter III-53, 1993-07-12
  Madagascar, changing information not included in this file. Official 
  name change to Republic of Madagascar

Newsletter III-54, 1993-07-23
  South Georgia and the South Sandwich Islands, previously covered by 
  Falkland Islands

Newsletter III-55, 1993-07-16
  Ethiopia, numeric code change to 231

Newsletter III-56, 1993-07-16
  Eritrea, changing information not included in this file
  Previously covered by the entry ET

Newsletter III-57, 1993-07-16
  Macedonia, the former Yugolslav Republic of,
  Previously covered by the entry YU  
  
Newsletter III-58, 1993-07-16
  Afghanistan, changing information not included in this file
  Official name change to Islamic State of Afghanistan  

Newsletter III-32, 1993-07-25 ammendment
  Kyrgyzstan, changing information not included in this file. Offical name
  change to Kyrgyz Republic

Newsletter III-59, 1994-01-26
  Andorra, changing information not included in this file. Offical name
  change to Pricipality of Andorra

Newsletter III-60, 1994-01-26
  Cambodia, changing information not included in this file. Offical name
  change to Kingdom of Cambodia

Announcement, 1994-02-02
  At this point the fourth edition of ISO 3166 appears. It can *now* be
  ordered from national standards institutions. 
  The RIPE NCC will continue tracking changes.

Newsletter IV-1, 1996-04-03
  Vatican, change name from VATICAN CITY STATE (HOLY SEE) to
  HOLY SEE (VATICAN CITY STATE)

Annual Report 1996, 1996-12-20
  No further changes have been made to 3166-IV.

  The fifth edition will consist of three parts:
  3166-1 country codes / DIS published April 96 / IS expected summer|fall 97
  "The list of country names remains basically unchanged."
  3166-2 country subdivision code / DIS published November 1996
  3166-3 Code for formerly used names of countries / DIS expected spring 97

  Quote: We are delighted to notice that the acceptance of ISO3166 among
  users of country codes is growing.  However, the use of ISO3166 e.g. in
  the Internet confronts ISO3166/MA with formerly unknown problems.  One
  among these problems is that more and more "non-country-entities" as
  e.g. organisations, interest groups or groupings of countries request
  ISO3166 code elements.  The scope of the standard does not allow for
  allocation of code elements to such "groups". 

Newsletter IV-2 1997-07-14
  ZAIRE (ZA) changed to CONGO, THE DEMOCRATIC REPUBLIC OF THE  (CD)
  change of name, alpha-2 and alpha-3, numeric remains 180

  3166-1 is now expected to be published by August/September 1997

