# ifndef _RHEOLEF_TEST_H
# define _RHEOLEF_TEST_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/space.h"
#include "rheolef/basis_on_pointset.h"
#include "rheolef/band.h"
#include "rheolef/piola.h"
#include "rheolef/vf_tag.h"
// TODO: #include "rheolef/field_expr_v2_variational_tag.h"

namespace rheolef {

namespace details {

struct grad_option_type {
  bool symmetrized;		// compute D(u): symmetrize grad(u) when u is a vector
  bool surfacic;		// compute grad_s(u): applies also the P=I-nxn projector on the surface
  bool broken;			// compute grad_h(u): applies also for discontinuous Galerkin
  bool batchelor_curl;		// for computing the stream function: Batchelor trick in axisymmetric geometries
  grad_option_type()
  : symmetrized(false), 
    surfacic(false),
    broken(false),
    batchelor_curl()
  {}
};

} // namespace details


/*Class:field_vf
NAME:  @code{test}, @code{trial} - symbolic arguments in variational expressions
DESCRIPTION:       
  @noindent
  These classes are used for test and trial functions
  involved in variational formulations.
  Variational formulations could be specified by expressions, in
  the spirit of c++ embedded languages.
  A variable of the @code{test} type represents the formal argument
  (the test-function) in the definition of a linear form, as in:
  @example
    geo omega ("circle");
    space Xh (omega, "P1");
    test v(Xh);
    field lh = integrate (omega, 2*v);
  @end example
  For a bilinear form, the test-function represents its second formal argument,
  while the first one is designed by the @code{trial} type:
  @example
    trial u(Xh);
    test  v(Xh);
    form m = integrate (omega, u*v),
         a = integrate (omega, dot(grad(u),grad(v)));
  @end example
  The @code{field_vf} class is abble to represents either @code{test} or
  @code{trial} types: it could be used to formally define a differential 
  operator, as:
  @example
    D(u) = (grad(u)+trans(grad(u)))
  @end example
End:
*/

template <class T, class M>
class test_rep {
public :
// typedefs:

    typedef typename std::size_t             size_type;
    typedef M                                memory_type;
    typedef undeterminated_basic<T>          value_type;
    typedef T                                scalar_type;
    typedef typename float_traits<T>::type   float_type;
    typedef geo_basic  <float_type,M>        geo_type;
    typedef space_basic<float_type,M>        space_type;

// allocator/deallocator:
  
    explicit test_rep (const space_type& V);
    test_rep (const test_rep<T,M>&);

// accessors:

    const space_type&  get_vf_space()  const { return _V; }
    static const space_constant::valued_type valued_hint = space_constant::last_valued;
    space_constant::valued_type valued_tag() const { return get_vf_space().valued_tag(); }
    size_type n_derivative() const { return 0; }

// mutable modifiers:

    void initialize (const  geo_basic<float_type,memory_type>& dom, const quadrature<T>& quad, bool ignore_sys_coord) const;
    void initialize (const band_basic<float_type,memory_type>& gh,  const quadrature<T>& quad, bool ignore_sys_coord) const;
    void initialize (const  space_basic<float_type,memory_type>& Xh, bool ignore_sys_coord) const;
    size_type element_initialize         (const geo_element& K) const;
    size_type element_initialize_on_side (const geo_element& K, const side_information_type& sid); //BUG with const for DG

    template<class ValueType>
    void basis_evaluate (const reference_element& hat_K, size_type q, std::vector<ValueType>& value) const
		{ _basis_evaluate (hat_K, q, value); }

    template<class ValueType>
    void grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
		{ _grad_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void div_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
		{ _div_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
		{ _curl_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void valued_check() const {
      space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
      check_macro (_V.valued_tag() == valued_tag, "unexpected "<<_V.valued()
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
    }
    template<class ValueType>
    void grad_valued_check() const {
      typedef typename space_constant::rank_down<ValueType>::type A1;
      space_constant::valued_type arg_valued_tag = space_constant::valued_tag_traits<A1>::value;
      check_macro (_V.valued_tag() == arg_valued_tag, "grad(): unexpected "<<_V.valued()
        << "-valued field while a " << space_constant::valued_name(arg_valued_tag)
        << "-valued one is expected in expression");
    }
    template<class ValueType>
    void div_valued_check() const {
      typedef typename space_constant::rank_up<ValueType>::type A1;
      space_constant::valued_type arg_valued_tag = space_constant::valued_tag_traits<A1>::value;
      check_macro (_V.valued_tag() == arg_valued_tag, "grad(): unexpected "<<_V.valued()
        << "-valued field while a " << space_constant::valued_name(arg_valued_tag)
        << "-valued one is expected in expression");
    }
    template<class ValueType>
    void curl_valued_check() const {
 	// TODO: depend on dim
    }

protected:
// internals
    void _basis_evaluate_init (const reference_element& hat_K, size_type q, size_type loc_ndof) const;
    void _basis_evaluate      (const reference_element& hat_K, size_type q, std::vector<T>& value) const;
    void _basis_evaluate      (const reference_element& hat_K, size_type q, std::vector<point_basic<T> >& value) const;
    void _basis_evaluate      (const reference_element& hat_K, size_type q, std::vector<tensor_basic<T> >& value) const;
    void _basis_evaluate      (const reference_element& hat_K, size_type q, std::vector<tensor3_basic<T> >& value) const;
    void _basis_evaluate      (const reference_element& hat_K, size_type q, std::vector<tensor4_basic<T> >& value) const;
    void _grad_basis_evaluate_init (const reference_element& hat_K, size_type q, size_type loc_ndof) const;
    void _grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const;
    void _grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const;
    void _grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const;
    void _grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor3_basic<T> >& value) const;
    void _grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor4_basic<T> >& value) const;
    void _div_basis_evaluate  (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const;
    void _div_basis_evaluate  (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const;
    void _div_basis_evaluate  (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const;
    void _curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const;
    void _curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const;
    void _curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const;
// data:
    space_type                                _V;
    details::grad_option_type		      _opt;
    mutable geo_basic<float_type,M>           _dom;
    mutable basis_on_pointset<float_type>     _basis_on_quad;
    mutable bool                              _is_on_band;
    mutable bool                              _ignore_sys_coord;
    mutable band_basic<float_type,M>          _gh;
    mutable reference_element                 _tilde_L;
    mutable std::vector<size_type>            _dis_inod_K;
    mutable std::vector<size_type>            _dis_inod_L;
    mutable basis_on_pointset<float_type>     _piola_on_quad;
    mutable bool                              _elements_on_bgd_dom;
    mutable bool                              _have_dg_on_sides;
};
// ----------------------
// smart_pointer version:
// ----------------------
template <class T, class M, class VfTag>
class test_basic : public smart_pointer<test_rep<T,M> > {
public :
// typedefs:

    typedef test_rep<T,M>                    rep;
    typedef smart_pointer<rep>               base;
    typedef typename rep::size_type          size_type;
    typedef typename rep::memory_type        memory_type;
    typedef typename rep::value_type         value_type;
    typedef typename rep::scalar_type        scalar_type;
    typedef typename rep::float_type         float_type;
    typedef typename rep::geo_type           geo_type;
    typedef typename rep::space_type         space_type;
    typedef VfTag                            vf_tag_type;
    typedef typename details::dual_vf_tag<VfTag>::type
                                             vf_dual_tag_type;
    typedef test_basic<T,M,VfTag>            self_type;
    typedef test_basic<T,M,vf_dual_tag_type> dual_self_type;

// allocator/deallocator:
  
    explicit test_basic (const space_type& V) : base(new_macro(rep(V))) {}

// accessors:

    const space_type&  get_vf_space()  const { return base::data().get_vf_space(); }
    static const space_constant::valued_type valued_hint = rep::valued_hint;
    space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }
    size_type n_derivative() const { return base::data().n_derivative(); }

// mutable modifiers:

    void initialize (const  geo_basic<float_type,memory_type>& dom, const quadrature<T>& quad, bool ignore_sys_coord) const
    			{ base::data().initialize (dom, quad, ignore_sys_coord); }
    void initialize (const band_basic<float_type,memory_type>& gh,  const quadrature<T>& quad, bool ignore_sys_coord) const
    			{ base::data().initialize (gh, quad, ignore_sys_coord); }
    void initialize (const  space_basic<float_type,memory_type>& Xh, bool ignore_sys_coord) const
    			{ base::data().initialize (Xh, ignore_sys_coord); }
    size_type element_initialize (const geo_element& K) const
    			{ return base::data().element_initialize (K); }
    size_type element_initialize_on_side (const geo_element& K, const side_information_type& sid)
    			{ return base::data().element_initialize_on_side (K, sid); }

    template<class ValueType>
    void basis_evaluate (const reference_element& hat_K, size_type q, std::vector<ValueType>& value) const
    			{ base::data().basis_evaluate (hat_K, q, value); }

    template<class ValueType>
    void grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
    			{ base::data().grad_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void div_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
    			{ base::data().div_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<ValueType>& value) const
    			{ base::data().curl_basis_evaluate (hat_K, q, opt, value); }

    template<class ValueType>
    void valued_check() const      { base::data().template valued_check<ValueType>(); }
    template<class ValueType>
    void grad_valued_check() const { base::data().template grad_valued_check<ValueType>(); }
    template<class ValueType>
    void div_valued_check() const  { base::data().template div_valued_check<ValueType>(); }
    template<class ValueType>
    void curl_valued_check() const { base::data().template curl_valued_check<ValueType>(); }
};
typedef test_basic<Float,rheo_default_memory_model,details::vf_tag_01> test;
typedef test_basic<Float,rheo_default_memory_model,details::vf_tag_10> trial;

}// namespace rheolef
# endif /* _RHEOLEF_TEST_H */
