package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`/@`!````0"(```````!``````````%"Q````
M`````````$``.``.`$``&P`:``8````$````0`````````!``````````$``
M````````$`,````````0`P````````@``````````P````0```#$`P``````
M`,0#````````Q`,````````<`````````!P``````````0`````````!````
M!````````````````````````````````````)@:````````F!H`````````
M$`````````$````%`````"``````````(``````````@````````03T`````
M``!!/0`````````0`````````0````0`````8`````````!@`````````&``
M``````#X10```````/A%`````````!`````````!````!@```,"K````````
MP+L```````#`NP```````&@$````````T`0`````````$`````````(````&
M````\*L```````#PNP```````/"[``````````(``````````@````````@`
M````````!`````0```!0`P```````%`#````````4`,```````!0````````
M`%``````````"``````````$````!````*`#````````H`,```````"@`P``
M`````"0`````````)``````````$``````````0````$````V*4```````#8
MI0```````-BE````````(``````````@``````````0`````````4^5T9`0`
M``!0`P```````%`#````````4`,```````!0`````````%``````````"```
M``````!0Y71D!````+"@````````L*````````"PH````````.P`````````
M[``````````$`````````%'E=&0&````````````````````````````````
M`````````````````````````!``````````4N5T9`0```#`JP```````,"[
M````````P+L```````!`!````````$`$`````````0`````````$````0```
M``4```!'3E4``@``P`0````#``````````*``,`$`````0`````````!``'`
M!`````D``````````@`!P`0````!``````````0````4`````P```$=.50``
M*/=5#F9/@KD(OPAE$SC,4GXIZB]L:6(V-"]L9"UL:6YU>"UX.#8M-C0N<V\N
M,@`[````0`````0````(````(!P0``5`!(8H`@`$@(8`A(*)`"))(`0(F(`0
M0;H``@H`````0````$$```!"````````````````````1`````````!%````
M1@```$<``````````````$@```!)`````````$H``````````````$L```!,
M````30````````!.````````````````````3P````````!0````````````
M````````40```%(`````````5`````````!6`````````%<`````````````
M``````````````````````!8````````````````````60````````!:````
M````````````````7````'^IG)MK"4/6"HH05R'_Y`X-.I<+P^4_+(D'<-/9
M<5@<K4OCP&L:GE"GW:-KU634?)E/PYLOQ9NV,V+;[6M_FGR[XY)\LP&+;T+Z
MUX=#1=7L9J_,9L6YG$`5C"7,H^#[2<_>1RWC1\8!X*<""F_%WQ\X\HL<^Z%L
M8`````````````````````````````````````"F`@``$@``````````````
M```````````-`@``$@````````````````````````"-`@``$@``````````
M``````````````"T````$@````````````````````````#H`0``$@``````
M``````````````````!)`@``$@`````````````````````````!````(```
M``````````````````````"X`0``$@````````````````````````!G````
M$@````````````````````````#%`0``$@````````````````````````#4
M````$@````````````````````````"&`@``$@``````````````````````
M``#8`@``$@````````````````````````#]`0``$@``````````````````
M```````H`0``$@`````````````````````````X`0``$@``````````````
M``````````"F````$@````````````````````````!Q`0``$@``````````
M``````````````!J`0``$@````````````````````````!]`P``$@``````
M```````````````````0````(`````````````````````````"^`@``$@``
M``````````````````````"K`@``$@````````````````````````#<````
M$@`````````````````````````L````(`````````````````````````#Z
M````$@`````````````````````````(`0``$@``````````````````````
M``"I`0``$@````````````````````````"P`@``$@``````````````````
M``````!8````(@````````````````````````"3````%@``````````````
M```````````^`0``$@````````````````````````!<`@``$@``````````
M``````````````!9`0``$@````````````````````````#)`@``$@``````
M``````````````````":`0``$@````````````````````````#V`0``$@``
M``````````````````````"[````$@````````````````````````"*````
M$@````````````````````````!^`0``$@````````````````````````"8
M`@``$@````````````````````````#7`0``$@``````````````````````
M``"%`0``$@`````````````````````````N`@``$@``````````````````
M``````!Y````$@````````````````````````"3`0``$@``````````````
M``````````",`0``$@````````````````````````#Y````$@``````````
M``````````````#*`0``$@`````````````````````````9`@``$@``````
M``````````````````!Z`@``$@````````````````````````"]`0``$@``
M```````````````````````O`0``$@`````````````````````````:`0``
M$@`````````````````````````!`0``$@````````````````````````#Q
M````$@````````````````````````!.`0``$@``````````````````````
M``#C````$@`````````````````````````Z`@``$@``````````````````
M``````!&````$@````````````````````````#A`@``$@``````````````
M```````````0`0``$@````````````````````````#1`0``$@``````````
M``````````````!K`P``$@`,`&!,````````(`````````"A`P``$``7``#`
M``````````````````!M`@``$0`8`&C``````````0`````````!!```$@`,
M`(!!````````20(```````"C`P``(``7``#```````````````````"Y`P``
M$@`,`%!`````````*@`````````.!```$0`7`"#`````````"`````````#!
M`P``$``8`$#```````````````````"2`P``$0`.``!@````````!```````
M``!A`P``$@`,`)!:````````@P(```````#,````(0`8`$#`````````"```
M```````;`P``$@`,`#!5````````2@(````````8!```$@`,`#!*````````
MO`````````#]`@``$@`,`'!!````````"0````````"G`P``$@`,`$`B````
M````)@````````!3````$@`,`"`@````````%0(```````#6`P``$``8`)#`
M```````````````````Q`P``$@`,`,!)````````8@````````"N`P``$@`,
M`*!8````````X0$```````#H`P``$``7`"C```````````````````!5`P``
M$@`,`!!$````````6@$```````#*````$0`8`$#`````````"`````````#-
M`P``$@`,`%!8````````1P`````````@!```$@`,`'!%````````1P0`````
M``#;`P``$@`,`'!-````````N@<````````.`P``$@`,`-!#````````,0``
M```````_`P``$@`,`(!`````````ZP````````"&`P``$@`,`(!7````````
MSP````````"B`0``$0`8`&#`````````"`````````#O`P``$@`,`(!,````
M````ZP``````````7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#
M;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`7U]L:6)C7W-T
M87)T7VUA:6X`7U]C>&%?9FEN86QI>F4`7U]R96=I<W1E<E]A=&9O<FL`7U]S
M=&%C:U]C:&M?9F%I;`!R96%D;&EN:P!03%]C=7)R96YT7V-O;G1E>'0`4&5R
M;%]S=E]S971P=@!M96UC:'(`4&5R;%]S=E]S971P=FX`7U]E;G9I<F]N`'-T
M<FYC;7``<W1R;&5N`%]?;65M;6]V95]C:&L`<F5A;&QO8P!L<W1A=#8T`&%C
M8V5S<P!O<&5N9&ER`')E861D:7(V-`!?7W-P<FEN=&9?8VAK`'5N;&EN:P!C
M;&]S961I<@!R;61I<@!B;V]T7T1Y;F%,;V%D97(`4&5R;%]N97=84P!097)L
M7VYE=UA37V9L86=S`&=E='!I9`!?7VUE;6-P>5]C:&L`<W1R8VAR`'-T<F1U
M<`!S=')T;VL`<W1R8VUP`'-T<G)C:'(`<W1D97)R`%]?=F9P<FEN=&9?8VAK
M`&5X:70`;'-E96LV-`!R96%D`&=E='5I9`!M:V1I<@!?7V5R<FYO7VQO8V%T
M:6]N`%]?9G!R:6YT9E]C:&L`;W!E;C8T`%!E<FQ?9W9?9F5T8VAP=@!097)L
M7V=E=%]S=@!097)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VUG7W-E=`!097)L
M7W-Y<U]I;FET,P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]A=&9O<FM?;&]C
M:P!03%]D;U]U;F1U;7``4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?<&%R<V4`
M<&5R;%]D97-T<G5C=`!P97)L7V9R964`4&5R;%]S>7-?=&5R;0!P97)L7V%L
M;&]C`'!E<FQ?8V]N<W1R=6-T`'!E<FQ?<G5N`&UE;6-P>0!L:6)P97)L+G-O
M`&QI8F,N<V\N-@!P87)?8W5R<F5N=%]E>&5C`'!A<E]B87-E;F%M90!84U])
M;G1E<FYA;'-?4$%27T)/3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E
M>&5C7W!R;V,`<&%R7V1I<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F
M<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%])3U]S=&1I;E]U<V5D`%]?9&%T
M85]S=&%R=`!S:&%?=7!D871E`'AS7VEN:70`7U]B<W-?<W1A<G0`<VAA7VEN
M:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P871H
M`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7V1I90!P87)?:6YI=%]E;G8`
M1TQ)0D-?04))7T147U)%3%(`1TQ)0D-?,BXQ-`!'3$E"0U\R+C,T`$=,24)#
M7S(N-`!'3$E"0U\R+C,S`$=,24)#7S(N,RXR`$=,24)#7S(N,RXT`$=,24)#
M7S(N,BXU`"]U<W(O;&EB+W!E<FPU+S4N-#(O8V]R95]P97)L+T-/4D4`````
M`0`!``$``@`#``$``0`"``0``@`"``(``0`!``(``@`!``,``@`"``$``0`"
M``(``0`%``(``P`!``(``0`!``$``0`!``(``@`!``(``@`!``(``@`!``8`
M`@`"``4``@`!``$``@`"``,``@`"``$``P`!``<`"``"``(``0`!``$``0`!
M``$``0`!``$``0`"``$``0`!``$``0`!``$``0`!``$``@`!``$``0`!``$`
M`0`"``$````!``@`\P(``!``````````0@[]````"0`M!```$````)21E@8`
M``@`/P0``!````"TD98&```'`$H$```0````%&EI#0``!@!5!```$````+.1
ME@8```4`7P0``!````!R&6D)```$`&H$```0````=!EI"0```P!V!```$```
M`'4::0D```(`@@0````````(O@````````8````!```````````````0O@``
M``````8````"```````````````8O@````````8````#```````````````@
MO@````````8````$```````````````HO@````````8````%````````````
M```PO@````````8````&```````````````XO@````````8````'````````
M``````!`O@````````8````(``````````````!(O@````````8````)````
M``````````!0O@````````8````*``````````````!8O@````````8````+
M``````````````!@O@````````8````,``````````````!HO@````````8`
M```-``````````````!PO@````````8````.``````````````!XO@``````
M``8````/``````````````"`O@````````8````0``````````````"(O@``
M``````8````1``````````````"0O@````````8````2``````````````"8
MO@````````8````3``````````````"@O@````````8````4````````````
M``"HO@````````8````5``````````````"PO@````````8````6````````
M``````"XO@````````8````7``````````````#`O@````````8````8````
M``````````#(O@````````8````9``````````````#0O@````````8````:
M``````````````#8O@````````8````;``````````````#@O@````````8`
M```<``````````````#HO@````````8````=``````````````#PO@``````
M``8````>``````````````#XO@```````!(````?````````````````OP``
M``````8````@```````````````(OP````````8````A```````````````0
MOP````````8````B```````````````8OP````````8````C````````````
M```@OP````````8````D```````````````HOP````````8````E````````
M```````POP````````8````F```````````````XOP````````8````G````
M``````````!`OP````````8````H``````````````!(OP````````8````I
M``````````````!0OP````````8````J``````````````!8OP````````8`
M```K``````````````!@OP````````8````L``````````````!HOP``````
M``8````M``````````````!POP````````8````N``````````````!XOP``
M``````8````O``````````````"`OP````````8````P``````````````"(
MOP````````8````Q``````````````"0OP````````8````R````````````
M``"8OP````````8````S``````````````"@OP````````8````T````````
M``````"HOP````````8````U``````````````"POP````````8````V````
M``````````"XOP````````8````W``````````````#`OP````````8````X
M``````````````#(OP````````8````Y``````````````#0OP````````8`
M```Z``````````````#8OP````````8````[``````````````#@OP``````
M``8````\``````````````#HOP````````8````]``````````````#POP``
M``````8````^``````````````#XOP````````8````_``````````````!`
MP`````````4```!5``````````````!@P`````````4```!<````````````
M``!HP`````````4```!"``````````````#`NP```````#\`````````",``
M```````)````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[Z2(/L"$B+!2F>``!(A<!T`O_02(/$",,``````/,/'OI52(GE0513
M2(/L,(E]W$B-?=Q(B7702(UUT$B)5<AD2(L4)2@```!(B57H2(U5R/\5?Y\`
M`$B+-="=``!(BSVAG@``2(GR9^BP/```@#WQGP````^$'@$``$B+!72>``!D
M2(L`@(@Z!````XM%W(UX!(U8`TAC_TC!YP/_%?.>``!(BTW01(M%W$B)QTB)
M!<F?``!(BP%(B0=(C04C00``2(E'"$B+!5&?``!(B4<02(T%$$$``$B)1QA!
M@_@!#XXR`0``2(/!"$&#P`.X!````&9F+@\?A```````2(LQ2&/0@\`!2(/!
M"$B)--=$.<!UZDB813'`2(GYB=I(QP3'`````$B--<$>``!(BSTZGP``_Q7<
MG```A<`/A(D```!(BSTEGP``_Q7WG0``2(UUY$B-/4<_``!!B<3H#!D``$B%
MP'1V2(L]`)\``/\5DIP``/\5;)T``$B+1>AD2"L$)2@````/A:````!(@\0P
M1(G@6T%<7</_%0^=``!(B07(G@``2(7`='=(B<?_%6*=``!(BP4[G0``9$B+
M$,:"M@@```#IM_[__TB+/9R>``#_%8Z<``#I9?___TB+/9*>``!(A?]T"NB0
M-0``Z73___](C3U</@``2(UUY.AK&```2(D%;)X``$B)QTB%P`^$4?___^O1
MN`0```#I\O[__[\!````_Q41G```_Q4SG0``9BX/'X0``````)#S#Q[Z,>U)
MB=%>2(GB2(/D\%!413'`,<E(C3W!_?___Q5[G0``]&8N#Q^$``````!(C3VQ
MG0``2(T%JIT``$@Y^'052(L%'IP``$B%P'0)_^`/'X``````PP\?@`````!(
MC3V!G0``2(TU>IT``$@I_DB)\$C![C](P?@#2`'&2-'^=!1(BP7]FP``2(7`
M=`C_X&8/'T0``,,/'X``````\P\>^H`]?IT```!U,U5(@SWZFP```$B)Y70-
M2(L]!IT``/\5Z)L``.AC____Q@55G0```5W#9BX/'X0``````,-F9BX/'X0`
M``````\?0`#S#Q[ZZ6?___\/'X``````54R-3WA(B>5!5T%63(V%0/W__T%5
M05132('LF`(``&1(BS0E*````$B)=<A(C7<XBP9(BPY(@\8(28/`$`_(B<!(
MB<I)B4#P2(G(2,'J($C!Z"A(P>DX)0#_``!("<A(B=%(P>((2,'A&('B``#_
M`(G)2`G(2`G028E`^$DY\76ON_____]F#V^%4/W__V8/;YU@_?__2(V%J/W_
M_V9(#V[K9@]OE7#]__](C56H9@]L[69F+@\?A```````#Q\`\P]O"&8/;^)F
M#V]0V$B#P!!F#^]`B&8/[\IF#^_!9@]OR&8/<_`!9@]ST1]F#^O!9@_;Q0\I
M0`AF#V_#9@]OW$@YPG6[3(L/3(M?$$B+=R!(BU\83(G(3(G*3(N%2/W__TC!
MZAM(P>`%2`G0`X5`_?__3HV$`YEY@EJ-E#"9>8):3(G82#'82"-'"$@QV(TT
M`DB+5PA(BT<(28GR2,'H`DC!XAY("<),B=A)P>H;B=%(,<A,(<A,,=A,`<!)
MB?!)P>`%30G01HTD`$R)R$V)R$G!Z`)(P>`>38GE3`G`3(N%4/W__TG![1M(
M,<)!B<)((?)/C80#F7F"6D@QRDP!PDV)X$G!X`5-">A,BZU8_?__00'02(GR
M2,'N`DC!XAY%B<!*C8PIF7F"6D@)\DV)Q4@QT(G63"'@2<'M&TPQT$@!R$R)
MP4C!X05,">E,BZU@_?__`<%,B>!)P>P"2,'@'HG)3XV4*IEY@EI,">!)B<U(
M,<))P>T;08G$3"'"2#'R3`'228G*2<'B!4T)ZDR+K6C]__]!`=),B<))P>@"
M2,'B'D6)TDJ-M"Z9>8):3`G"38G52#'02<'M&T&)T$@AR$PQX$@!\$R)UDC!
MY@5,">Y,BZUP_?__`<9(B<A(P>D"2,'@'HGV3XVD+)EY@EI("<A)B?5(,<))
MP>T;B<%,(=),,<),`>))B?1)P>0%30GL3(NM>/W__T$!U$R)TDG!Z@)(P>(>
M18GD3XV$*)EY@EI,"=)-B>5(,=!)P>T;08G22"'P2#'(3`'`38G@2<'@!4T)
MZ$R+K8#]__]!`<!(B?!(P>X"2,'@'D6)P$J-C"F9>8):2`GP38G%2#'"2<'M
M&XG&3"'B3#'22`'*3(G!2,'A!4P)Z4R+K8C]__\!T4R)XDG![`)(P>(>B<E/
MC90JF7F"6DP)XDF)S4@QT$G![1M!B=1,(<!(,?!,`=!)B<I)P>(%30GJ3(NM
MD/W__T$!PDR)P$G!Z`)(P>`>18G22HVT+IEY@EI,"<!-B=5(,<))P>T;08G`
M2"'*3#'B2`'R3(G62,'F!4P)[DR+K9C]__\!UDB)RDC!Z0)(P>(>B?9/C:0L
MF7F"6D@)RDF)]4@QT$G![1N)T4PAT$PQP$P!X$F)]$G!Y`5-">Q,BZV@_?__
M00'$3(G02,'@'DG!Z@)%B>1/C80HF7F"6DP)T$V)Y4@QPDG![1M!B<)((?)(
M,<I,`<)-B>!)P>`%30GH3(NMJ/W__T$!T$B)\DC![@)(P>(>18G`2HV,*9EY
M@EI("?)-B<5(,=!)P>T;B=9,(>!,,=!(`<A,B<%(P>$%3`GI3(NML/W__P'!
M3(G@2<'L`DC!X!Z)R4^-E"J9>8):3`G@28G-2#'"2<'M&T&)Q$PAPD@Q\DP!
MTDF)RDG!X@5-">I,BZVX_?__00'23(G"2,'B'DG!Z`)%B=)*C;0NF7F"6DP)
MPDV)U4@QT$G![1M!B=!((<A,,>!(`?!,B=9(P>8%3`GN3(NMP/W__P'&2(G(
M2,'I`DC!X!Z)]D^-I"R9>8):2`G(28GU2#'"2<'M&XG!3"'23#'"3`'B28GT
M2<'D!4T)[$:--"),B=))P>H"2,'B'DP)TDR+E<C]__](,=!!B=1((?!/C800
MF7F"6DV)\D@QR$G!ZAM,`<!-B?!)P>`%30G03(N5T/W__T$!P$B)\$C!X!Y(
MP>X"18G`2HV,$9EY@EI("?!-B<)(,<))P>H;B<9,(?),,>)(`=%,B<)(P>(%
M3`G238GR2<'N`DG!XAZ)TDT)\D@!T4B+E=C]__],,=!!B<U,(<!)C904F7F"
M6DV)[$@Q\$G![!M(`=!,B>I(P>(%3`GBB=)(`=!,B<))P>@"2,'B'D&)Q$P)
MPDV)T$6)TDDQT$0QP4R+A>#]__]*C;0&H>O9;DV)X$@!SDR)X4G!Z!M(P>$%
M3`G!B<E(`<Y,B>E(P>$>2<'M`D&)\$P#E>C]__],">E)B=6)TD@#E?#]__])
M,<U$,>A-B<5-C90"H>O9;DR)P$G![1M(P>`%3`GHB<!)`<),B>!)P>P"2,'@
M'D6)U4P)X$F)S(G)2`.-^/W__TDQQ$0QYDV)[$B-M#*AZ]EN3(GJ2<'L&TC!
MX@5,">*)TD@!UDR)PDG!Z`)(P>(>08GT3`G"28G`B<!),=!%,=!-B>).C80!
MH>O9;DR)X4G!ZAM(P>$%3`G1B<E)`<A,B>E)P>T"2,'A'D6)PDP)Z4F)U4@#
MA0#^__^)TDDQS4@#E0C^__]$,>Y-B=5(C;0PH>O9;DR)T$G![1M(P>`%3`GH
MB<!(`<9,B>!)P>P"2,'@'D&)]4P)X$F)S(G)2`.-$/[__TDQQ$4QX$V)[$Z-
MA`*AZ]EN3(GJ2<'L&TC!X@5,">*)TDD!T$R)TDG!Z@)(P>(>18G$3`G228G"
MB<!(`X48_O__23'21#'638GB2(VT,:'KV6Y,B>%)P>H;2,'A!4P)T8G)2`'.
M3(GI2<'M`DC!X1Y!B?),">E)B=6)TDDQS4@#E2#^__]%,>A-B=5.C80`H>O9
M;DR)T$G![1M(P>`%3`GHB<!)`<!,B>!)P>P"2,'@'D6)Q4P)X$F)S(G)2`.-
M*/[__TDQQ$0QYDV)[$B-M#*AZ]EN3(GJ2<'L&TC!X@5,">*)TD@!UDR)TDG!
MZ@)(P>(>08GT3`G228G"B<!(`X4P_O__23'213'038GB3HV$`:'KV6Y,B>%)
MP>H;2,'A!4P)T8G)20'(3(GI2<'M`DC!X1Y%B<),">E)B=6)TDDQS40Q[DV)
MU4B-M#"AZ]EN3(G02,'@!4G![1M(`Y4X_O__3`GHB<!(`<9,B>!)P>P"2,'@
M'D&)]4P)X$F)S(G)2`.-0/[__TDQQ$4QX$V)[$Z-A`*AZ]EN3(GJ2<'L&TC!
MX@5,">*)TDD!T$R)TDG!Z@)(P>(>18G$3`G228G"B<!(`X5(_O__23'21#'6
M38GB2(VT,:'KV6Y,B>%)P>H;2,'A!4P)T8G)2`'.3(GI2<'M`DC!X1Y!B?),
M">E)B=6)TDDQS44QZ$V)U4Z-A`"AZ]EN3(G02<'M&TC!X`5,">A(`Y50_O__
MB<!)`<!,B>!)P>P"2,'@'D6)Q4P)X$F)S(G)2`.-6/[__TDQQ$0QYDV)[$B-
MM#*AZ]EN3(GJ2<'L&TC!X@5,">*)TD@!UDR)TDG!Z@)(P>(>08GT3`G228G"
MB<!(`X5@_O__23'213'038GB3HV$`:'KV6Y,B>%)P>H;2,'A!4P)T8G)20'(
M3(GI2<'M`DC!X1Y%B<),">E)B=6)TDDQS40Q[DV)U4B-M#"AZ]EN3(G02<'M
M&TC!X`5,">B)P$@!QDR)X$C!X!Y)P>P"08GU2`.5:/[__TP)X$F)S(G)2`.-
M</[__TDQQ$4QX$Z-I`*AZ]EN3(GJ38GH2<'H&TC!X@5,"<*)TDD!U$R)TDG!
MZ@)(P>(>18G@3`G228G"B<!(`X5X_O__23'21#'638G"2(VT,:'KV6Y,B<%)
MP>H;2,'A!4P)T4V)ZDG![0))P>(>`<Y(B=&)TDT)ZHGV3#'138G618G51#'A
M28GT2(V,"*'KV6Y(B?!)P>P;2,'@!4P)X$2-)`%,B<%)P>@"N-R\&X](P>$>
M3`G!20G.08G(32'"22'V30G63(N5@/[__TD!PDP!TDV)XDP!\DV)YDG!X@5)
MP>X;30GR00'22(GR2,'N`DC!XAY%B=)("?)("=%)B=:)UDPAX4TAQDD)SDB+
MC8C^__](`<%,`>E-B=5,`?%-B=9)P>4%2<'N&TT)]4F)UD$!S4R)X4G![`)(
MP>$>18GM3`GA08G,20G.32'63"'B3`GR3(NUD/[__TD!QDT!QDV)Z$P!\DV)
M[DG!X`5)P>X;30GP00'03(G22<'J`DC!XAY%B<!,"=)("=%)B=9!B=),(>E-
M(>9)"<Y(BXV8_O__2`'!2`'Q3(G&3`'Q38G&2,'F!4G![AM,"?9)B=8!SDR)
MZ4G![0)(P>$>B?9,">E!B<U)"<Y-(<9,(>I,"?),B[6@_O__20'&30'F28GT
M3`'R28GV2<'D!4G![AM-"?1!`=1,B<))P>@"2,'B'D6)Y$P)PD@)T4F)UD&)
MT$@A\4TA[DD)SDB+C:C^__](`<%,`=%-B>),`?%-B>9)P>(%2<'N&TT)\DF)
MUD$!RDB)\4C![@)(P>$>18G22`GQB<Y)"<Y-(>9((?),"?),B[6P_O__20'&
M30'N38G53`'R38G62<'E!4G![AM-"?5!`=5,B>)(P>(>2<'L`D6)[4P)XD@)
MT4F)UD&)U$PAT4DA]DD)SDB+C;C^__](`<%,`<%-B>A,`?%-B>Y)P>`%2<'N
M&TT)\$F)UD$!R$R)T4G!Z@)(P>$>18G`3`G108G*20G.32'N3"'23`GR3(NU
MP/[__TD!QDD!]DR)QDP!\DV)QDC!Y@5)P>X;3`GV`=9,B>I)P>T"2,'B'HGV
M3`GJ2`G128G608G53"'!32'620G.2(N-R/[__T@!P4P!X4F)]$P!\4F)]DG!
MY`5)P>X;30GT28G600',3(G!2<'H`DC!X1Y%B>1,"<%!B<A)"<Y)(?9,(<),
M"?),B[70_O__20'&30'638GB3`'R38GF2<'B!4G![AM-"?)!`=)(B?)(P>X"
M2,'B'D6)TD@)\D@)T4F)UHG63"'A32'&20G.2(N-V/[__T@!P4P!Z4V)U4P!
M\4V)UDG!Y05)P>X;30GU28G600'-3(GA2<'L`DC!X1Y%B>U,">%!B<Q)"<Y-
M(=9,(>),"?),B[7@_O__20'&30'&38GH3`'R38GN2<'@!4G![AM-"?!!`=!,
MB=))P>H"2,'B'D6)P$P)TD@)T4F)UD&)TDPAZ4TAYDD)SDB+C>C^__](`<%(
M`?%,B<9,`?%-B<9(P>8%2<'N&TP)]DF)U@'.3(GI2<'M`DC!X1Z)]DP)Z4&)
MS4D)SDTAQDPAZDP)\DR+M?#^__])`<9-`>9)B?1,`?))B?9)P>0%2<'N&TT)
M]$$!U$R)PDG!Z`)(P>(>18GD3`G"2`G128G608G02"'Q32'N20G.2(N-^/[_
M_T@!P4P!T4V)XDP!\4V)YDG!X@5)P>X;30GR28G600'*2(GQ2,'N`DC!X1Y%
MB=)("?%)"<Z)SDTAYD@A\DP)\DR+M0#___])`<9-`>Y-B=5,`?)-B=9)P>4%
M2<'N&TT)]4$!U4R)XDG![`)(P>(>18GM3`GB2`G128G408G73"'122'T20G,
M2(N-"/___T@!P4P!P4V)Z$P!X4V)[$G!X`5)P>P;30G@00'(3(G12<'J`DC!
MX1Y%B<!,"=%)B=)!B<Q)"<I-(>I,(>),"=),BY40____20'"2`.%&/___TP!
MUDV)PDP!^$R+O3C___](`=9,B<))P>H;2,'B!4P)THG22`'63(GJ2<'M`DC!
MXAY!B?),">I("=%)B=5!B=9,(<%-(>5)"<U)C4P%`$R)T$V)U4G![1M(P>`%
M3`GHB<!(`<%,B<!)P>@"2,'@'D&)S4D)P+C6P6+*3#'",?)(B[4@____2`'&
M3`'F38GL2`'R3(GN2<'L&TC!Y@5,">9-B=2)]D@!\DG!Y!Y)`<=)P>H"B=9-
M">)-B<1%B<!-,=1!,<Q(BXTH____2`'!3`'Q28GV3`'A28GT2<'N&TG!Y`5-
M"?1-B>Y)P>T"2<'F'D6)Y$T)]4R+M3#___],`>%!B<Q)`<9-`?!-B=9-,>Y!
M,=9,B>)-`?!-B>9(P>(%2<'N&TP)\HG220'02(GR2,'N`DC!XAY%B<9("=9$
MB=)-B>I),?),`?I,B[U`____03'*2HT,$DR)\DV)\DC!X@5)P>H;20''3`G2
MB=)(`=%,B>))P>P"2,'B'D&)RDP)XD6)[$F)]8GV23'530'\3(N]6/___T4Q
MQ4^-!"Q-B=1-B=5)P>T;2<'D!4T)[$V)]4G![@))P>4>18GD30GN3(NM2/__
M_TT!X$6)Q$D!Q4P![DF)U8G233'U03'-3(GA3`'N38GE2,'A!4G![1M,">F)
MR4@!SDR)T4G!Z@)(P>$>08GU3`G13(N54/___TD!PDP!TDV)\DDQRD4QPDD!
MQTZ-!!),B>I-B>I)P>H;2,'B!4P)THG220'03(GB2<'L`DC!XAY%B<))"=1$
MB?))B<Z)R4TQYDP!^DR+O6C___]!,?9*C30R3(G238G620''2<'N&TC!X@5,
M"?*)TD@!UDR)ZDG![0)(P>(>08GV3`GJ3(NM8/___TD!Q4P!Z4V)Y4DQU44Q
MQ4Z-!"E,B?%-B?5)P>T;2,'A!4P)Z8G)20'(3(G12<'J`DC!X1Y%B<5)"<I$
MB>%)B=2)TDP!^4TQU$R+O7C___]!,?1*C30A3(GI38GL20''2<'L&TC!X05,
M">&)R4@!SDR)\4G![@)(P>$>08GT3`GQ3(NU</___TD!QDP!\DV)UDDQSD4Q
MQDZ-!#),B>)-B>9)P>X;2,'B!4P)\HG220'03(GJ2<'M`DC!XAY%B<9)"=5$
MB=))B<J)R4TQZDP!^DR+?8A!,?)*C3023(GR38GR2<'J&TC!X@5,"=*)TD@!
MUDR)XDG![`)(P>(>08GR3`GB3(ME@$D!Q$P!X4V)[$D!QTDQU$4QQ$Z-!"%,
MB=%-B=1)P>P;2,'A!4P)X8G)20'(3(GQ2<'N`DC!X1Y%B<1)"<Y$B>E)B=6)
MTDTQ]4P!^4R+?9A!,?5*C30I3(GA38GE2<'M&TC!X05,">F)R4@!SDR)T4G!
MZ@)(P>$>08GU3`G13(M5D$D!PDP!TDV)\DDQRD4QPDZ-!!),B>I-B>I)P>H;
M2,'B!4P)THG220'03(GB2<'L`DC!XAY%B<))"=1$B?))B<Y)`<=-,>9,`?J)
MR4R+?:A!,?9,B=9)`=9,B=)(P>X;20''2,'B!4@)\HG220'63(GJ2<'M`DC!
MXAY$B?9,">I,BVV@20'%3`'I38GE23'513'%3HT$*4B)\4F)]4G![1M(P>$%
M3`GIB<E)`<A,B=%)P>H"2,'A'D6)Q4D)RD2)X4F)U(G233'43`'Y13'T38GN
M3`'A38GL2<'N&TG!Y`5-"?1)B?9%B>1,`>%)P>8>2,'N`D&)S$P)]DR+=;!)
M`<9(`T6X3`'R38G63`'(18G223'V3`'013'&38G@3`'R38GF2<'@!4G![AM-
M"?!$`<)-B>A)P>T"2<'@'HG230G%28GP`W<@33'H00'=2(EW($0QP4F)T$R)
M;QA(`<A(B=%)P>@;`U<(2,'A!4B)5PA,"<$!R$B)!TR)X$G![`)(P>`>3`G@
M1`'82(E'$$B+1<AD2"L$)2@```!U$DB!Q)@"``!;05Q!74%>05]=P_\5[(4`
M``\?0`!52(GE055!5%-(@>P`$```2(,,)`!(@^PHNO\/``!D3(LL)2@```!,
MB6W828GU28G\2(VUT.___TB-/44F``#_%7.%``"%P'X72&/(@^@!2&/0@+P5
MT.___P!U1(7`=31(BP41A0``3(GJ3(GF9$B+./\5D80``$B+1=AD2"L$)2@`
M``!U7TB!Q"@0``!;05Q!75W#2&/(9@\?A```````2(F-R.___TB)RKXO````
M2(V]T.____\5Y(,``$B%P'2?2(L%L(0``$B+C<CO__](C970[___3(GF9$B+
M./\5S80``.N2_Q7]A```#Q]$``!52(GE05=!5D%505132(/L&$R+-;B%``!-
MA?8/A*0````/M@>$P'0$/#UU9$4QY$F+'DB%VW1`2(EUR$F)_4UCY$V)]V9F
M+@\?A```````D$R)XDR)[DB)W_\5B8,``(7`=0="@#PC/71&28M?"$F#QPA(
MA=MUV4B#Q!A(B=A;05Q!74%>05]=PP\?@`````!)B?Q!#[9$)`%)@\0!A,!T
M!#P]=>Y!*?SKA68/'T0``$B+1<A-*?=*C5PC`4G!_P-$B3CKLC';ZZX/'X``
M````53'`2(GE05=!5D%505132(/L*(`^/61(BQPE*````$B)7<A(B?L/E,!,
MC20&3(GG_Q5*@P``2(UUQ$F)Q4B)W^CK_O__2(7`#X0J`0``28G&2(G'_Q4F
M@P``3#GH#X/U````3&-UQ$G!Y@,/M@.$P`^$V@$``#P]#X32`0``2(G89F8N
M#Q^$``````!F9BX/'X0``````&8/'X0```````^V4`%(@\`!A-)T!8#Z/77O
M2"G82)A,BSU#A```2HU\*`+_%5B"``!-`?=)B0=(A<!T7DB+!2:$``!*BQ0P
M#[8#B`(\/70OA,!T*TB-0P'K%&9F+@\?A```````#Q]``(#Y/701#[8(2(/"
M`4B#P`&("H3)=>K&`CTQP`\?1```00^V#`2(3`(!2(/``83)=>](BT7(9$@K
M!"4H````#X4@`0``2(/$*%M!7$%=05Y!7UW##Q]$```QP&8/'T0``$$/MA0$
M08@4!DB#P`&$TG7OZ[X/'T0``$R+#8&#``!)@SD`#X3.````3(G(13'_9F8N
M#Q^$``````!F#Q]$``!(@\`(1(GZ08/'`4B#.`!U[XU*`TACR4C!X0.+!8>#
M``"%P'5C2(G/3(E-L,<%<H,```$```!(B4VX_Q5`@0``2(G'2(7`#X1%____
M36/W2(M-N$B+=;!)P>8#3(GR_Q6,@@``2(D%]8(``$B)QT6-1P%-8\!*QP3'
M`````.E)_O__9@\?1```3(G/2(G._Q5,@@``2(D%Q8(``$B)QTB%P`^$ZO[_
M_TUC]TG!Y@/KOC'`Z5_^__^Y$````$4Q_^E7_____Q6^@0``9@\?1```54B)
MY5-(C;50____2('LJ````&1(BQPE*````$B)7>A(B?O_%:>!``"%P'4[BY5H
M____@>(`\```@?H`0```=`B!^@"@``!U(;X"````2(G?_Q6Q@0``A<`/E,`/
MML#K"0\?@``````QP$B+5>AD2"L4)2@```!U!DB+7?C)P_\5,X$``&9F+@\?
MA```````54B)Y4%605132('LN````&1(BP0E*````$B)1=@QP$B%_W0(2(G[
M@#\`=21(BT789$@K!"4H````#X5H`0``2('$N````%M!7$%>7<,/'P#_%4*`
M``!)B<1(A<!TSDR)Y_\528$``$B%P`^$\````$R-2!.`>!,N=19!@'D!`'3=
M9F8N#Q^$```````/'T``@'@3+@^$Y@```$R)C3C___](B=__%=9_``!,BXTX
M____28G&3(G/3(F-,/____\5O'\``$&-5`8"2&/22(G72(F5./____\51'\`
M`$R+C3#___])B=A(BY4X____28G&2(G'O@(````QP$B-#=(@``#_%6F```!,
MB?=(C;5`_____Q5Y?P``3(GW@_C_=!*+A5C___\E`/```#T`0```=&__%0%_
M``!,B??_%3A_``!,B>?_%6>```!(A<`/A1[___]F9BX/'X0```````\?`$R)
MY_\5_W\``$B)W_\5SGX``.FU_O__9@\?A```````08!Y`2X/A0____]!@'D"
M``^$R/[__^G__O__#Q]$``#H6_[__^N0_Q5[?P``#Q\`\P\>^E5(C0UT8```
M2(TU%B```$B)Y5-(@^P(2(L=YGX``$B+%>=^``!D2(L[_Q6E?P``3(T%FB$`
M`&1(BSM%,<E(BUWX3(G!2(T5\10``$B-->\?``#)_R7#?@``#Q\`\P\>^DB+
M!9U^``!(BQ6>?@``2(T-!V```$B--:D?``!D2(LX_R5.?P``9@\?1```\P\>
M^E5(B>5!5%-(@>P`$```2(,,)`!(@>P`$```2(,,)`!(@^PP9$B+!"4H````
M2(E%Z#'`_Q7<?0``N@$0``"^`@```$R-#4(?``!!B<!(C0US'P``2(V]X.__
M_S'`_Q7*?@``2(V]X.___[H`$```2(VUT-____\5.7X``#'_A<!X0HU(`4B)
MPTACR4B)STB)C<C?____%4)]``!(B<=(A<!T(4ACVTB+C<C?__](C;70W___
M2(G:_Q50?0``Q@08`$B)QTB+1>AD2"L$)2@```!U#TB!Q#`@``!(B?A;05Q=
MP_\5_7T```\?1```\P\>^ND'____#Q^``````/,/'OI52(GE05=!5D%50513
M2('L`!```$B##"0`2('LR````&1,BRPE*````$R);<A)B?5)B?Q(C;4L[___
M2(T]DQX``.BF^/__OB\```!,B>=(B</_%65]``!(A<!T.$R)YDB-/7<>``#H
M4?G__TR)X$B+5<AD2"L4)2@````/A;X!``!(C6786T%<05U!7D%?7<,/'T``
M3(GO_Q4W?0``2(TU1AX``$B)Q_\51WT``$F)QTB%P'2G2(V%,.___TB)A1CO
M__]!@#\`#X2G````9F8N#Q^$```````/'T``2(7;=!!,B?Y(B=__%?]\``"%
MP'1>3(G__Q5"?```28U4!_])B<9).==R(>LP9F8N#Q^$```````/'X0`````
M`,8"`$B#Z@%).==T!8`Z+W3O3(G__Q4&?```28G&3(GG_Q7Z>P``28U$!@)(
M/?\/``!V34B--90=```Q__\5EGP``$F)QTB%P`^$\O[__T&`/P`/A6C___](
MA=MT!8`[+G060;X!````3(T]1QT``.NL#Q^``````(![`0!TM>OB2(/L"$B-
M#4D>``!-B?@QP$%4N@`0``!,C0TR'0``O@(```!(C;W`[____Q5@?```2(NU
M&.___TB-O<#O____%6Q[``!:687`#X5E____BX5([___)0#P```]`(````^%
M3____[X!````2(V]P.____\5(GP``(7`#X4U____2(VUP.___TB-/;4<``#H
MC_?__TB-O<#O____%9I[``#I+_[___\5GWL```\?@`````#S#Q[Z5;XO````
M2(GE4TB)^TB#[`C_%35[``!(C7@!2(7`2`]$^TB+7?C)_R57>P``9F8N#Q^$
M```````/'T``\P\>^E5(B>5(@>P`$```2(,,)`!(@^P@9$B+!"4H````2(E%
M^#'`2(7_#X3X````@#\`#X3O````2(F]Z.____\5:GH``$F)P$B-0`%(/0`0
M```/A_<```!(C97P[___2(NUZ.___TF)T8/X"'(.B<%(B=?!Z0/S2*5)B?DQ
MR:@$=`J+#D&)";D$````J`)T#0^W/`YF08D\"4B#P0*H`70(#[8$#D&(!`E*
MC40"_T@YPG(HZRMF9BX/'X0``````&9F+@\?A```````#Q]``,8``$B#Z`%(
M.=!T!8`X+W3O2(F5Z.___TB)U[XO````_Q4:>@``2(N5Z.___TB%P'0F2#G"
M<D%(B=?_%3AZ``!(BU7X9$@K%"4H````=37)PP\?@`````!(BT7X9$@K!"4H
M````=1W)2(T]!QL``/\E`WH```\?`,8``.NZ#Q\`,<#KO/\5_GD``&8/'T0`
M`/,/'OI52(GE4TB#[!AD2(L$)2@```!(B47H,<`/'T``2(UUY$B-/=<:``#H
MT/3__TB%P'0S2&-5Y$B+!9!Z``!(C0302(M0"$B)$$B%TG302(M0$$B#P`A(
MB1!(A=)U\.N^9@\?1```2(UUY$B-/9@:``#HB/3__TB%P'0[2&-5Y$B+!4AZ
M``!(C0302(M0"$B)$$B%TG30#Q^$``````!(BU`02(/`"$B)$$B%TG7PZ[9F
M#Q]$``!(C77D2(T]4!H``.@X]/__2(7`=#M(8U7D2(L%^'D``$B-!-!(BU`(
M2(D02(72=-`/'X0``````$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$B-=>1(
MC3T)&@``Z.CS__](A<!T.TAC5>1(BP6H>0``2(T$T$B+4`A(B1!(A=)TT`\?
MA```````2(M0$$B#P`A(B1!(A=)U\.NV9@\?1```2(UUY$B-/<`9``#HF//_
M_TB%P'0[2&-5Y$B+!5AY``!(C0302(M0"$B)$$B%TG30#Q^$``````!(BU`0
M2(/`"$B)$$B%TG7PZ[9F#Q]$``!(C77D2(T]@!D``.A(\___2(7`=#M(8U7D
M2(L%"'D``$B-!-!(BU`(2(D02(72=-`/'X0``````$B+4!!(@\`(2(D02(72
M=?#KMF8/'T0``$B-=>1(C3WE&```Z/CR__](A<!T.TAC5>1(BP6X>```2(T$
MT$B+4`A(B1!(A=)TT`\?A```````2(M0$$B#P`A(B1!(A=)U\.NV9@\?1```
M2(UUY$B-/>P8``#HJ/+__TB%P'0[2&-5Y$B+!6AX``!(C0302(M0"$B)$$B%
MTG30#Q^$``````!(BU`02(/`"$B)$$B%TG7PZ[9F#Q]$``!(C77D2(T]IA@`
M`.A8\O__2(7`=#M(8U7D2(L%&'@``$B-!-!(BU`(2(D02(72=-`/'X0`````
M`$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$B-=>1(C3U@&```Z`CR__](A<!T
M.TAC5>1(BP7(=P``2(T$T$B+4`A(B1!(A=)TT`\?A```````2(M0$$B#P`A(
MB1!(A=)U\.NV9@\?1```2(UUY$B-/:X7``#HN/'__TB%P'0[2&-5Y$B+!7AW
M``!(C0302(M0"$B)$$B%TG30#Q^$``````!(BU`02(/`"$B)$$B%TG7PZ[9F
M#Q]$``!(C77D2(T]RA<``.AH\?__2(7`=`](B<9(C3VB%P``Z"3R__](C77D
M2(T]MQ<``.A$\?__2(7`=`](B<9(C3VU%P``Z`#R__](C77D2(T]L!<``.@@
M\?__2(7`=#9(B<9(C3WY%@``Z-SQ__](BT7H9$@K!"4H````=3Y(BUWX2(TU
MGA<``$B-/0X7``#)Z;7Q__](C77D2(T]=1<``.C5\/__2(7`=,1(B<9(C3T%
M%P``Z)'Q___KL_\5L74``&8/'X0``````/,/'OI52(T]XQ8``$B)Y4B#[!!D
M2(LT)2@```!(B77X2(UU].B'\/__2(7`="H/MA"`^C`/E,"$T@^4P@G0@_`!
M#[;`2(M5^&1(*Q0E*````'4)R<,/'P`QP.OH_Q5&=0``9F8N#Q^$```````/
M'P#S#Q[Z54B)Y4B![-````!(B;58____2(F58/___TB)C6C___],B85P____
M3(F->/___X3`="`/*46`#RE-D`\I5:`/*5VP#REEP`\I;=`/*77@#RE]\&1(
MBP0E*````$B)A4C___\QP$B-11!(B?K'A3#___\(````2(F%./___TB-A5#_
M__](BSVB=0``2(V-,/___\>%-/___S````"^`@```$B)A4#_____%?]S``"_
M_P```/\55',```\?0`!52(GE055!5%-,C9PD``#__TB![``0``!(@PPD`$PY
MW'7O2(/L&+H"````,?9D3(LL)2@```!,B6W808G]2(V=T/_^__\59G0``$R-
M8/],B>)(P?H_2,'J,$F-!!0/M\!(*=!)*<0/B+`````/'T``,=),B>9$B>__
M%3)T``!(@_C_#X2Z````N@@``0!(B=Y$B>__%<=R``"#^/\/A(,```!(F$B#
M^`=V7DB-1`/X2#G8<E1(C15E=```ZQQF9BX/'X0```````\?A```````2(/H
M`4@YV'(O2(L(2#D*=>](*=A,`>!(BU789$@K%"4H````=6E(@<08``$`6T%<
M05U=PP\?0`!)@>P```$`28'\``#__P^%5/___TC'P/_____KPDB+1=AD2"L$
M)2@```!U*TB-/0`5```QP.C^_?__2(M%V&1(*P0E*````'4.2(T]UA0``#'`
MZ.']____%1-S``!F9BX/'X0``````/,/'OI52(GE_Q4:<P``B<?_%2IR``!(
MA<!T`TB+`%W#\P\>^E5(B>5!5E-(C77D2(/L(&1(BQPE*````$B)7>A(B?M(
MC3V-%```Z,'M__](A<!T!8`X`'4O2(M%Z&1(*P0E*`````^%F0```$B)WDB-
M9?!(C3U=%```6T%>7>E=[O__#Q]$``!(B=](B478_Q7+<0``2(M5V$F)QDB)
MUTB)5=#_%;=Q``!)C70&`DB)]TB)==C_%45Q``!(@^P(28G8_W702(G'2(M5
MV+X"````2(E%V$R-#3(3``!(C0TQ%```,<#_%65R``!86DB+1>AD2"L$)2@`
M``!U"4B+==CI:O____\5_7$```\?1```\P\>^E5(B>5!5T%6055!5$R-I;3^
M__]33(GF2('L2`$``$B)O:C^___S#WX%+VX``$B-/;`2``!D2(L$)2@```!(
MB47(2(T%FA(``$C'A?#^__\`````2,>%*/___P````!(QX70_O__`````&9(
M#V[(2(T%:A,``&8/;,%F2`]NT$B-!6(3```/*87@_O__9D@/;MCS#WX%RVT`
M`$B-!5`3``!F2`]NX&8/;,(/$84(____\P]^!;1M``!F#VS##Q&%&/____,/
M?@6I;0``9@]LQ`\IA<#^___H(>S__TB%P'0)@#@`#X6[!```_Q4E<0``B<?_
M%35P``!(A<`/A"P$``!,BSA-A?\/A"`$``!,B?__%3=P``!,C6P``4R)[TV)
M[O\5QF\``$4/M@=(B86@_O__2(G#183`=%4/'P!,B>I%#[;`2(G?,<!)@\<!
M2(T-IA(``+X"````_Q7><```33GN3(GJ10^V!TD/0]9(B=!,*>A(@\`"2#G0
M2`]"PDDIU4B#PP)-C6P%_D6$P'6N3(VM"/___TB-/>`1``#K$V8/'T0``$F+
M?0!)@\4(2(7_=#M,B>;H0^O__TF)QDB%P'3C@#@`=-Y(B<?H;N[__X7`=-),
MB??_%0EP``!)B<5(A<!U368/'X0``````$R-M>#^___K"P\?@`````!)@\8(
M38LN387M="A!@'T```^$"0,``$R)[^@A[O__A<!TW4R)[_\5O&\``$F)Q4B%
MP'3,3(GO_Q43;P``2(N=H/[__TF)QDB)W_\5`&\``$6-M`8$!```36/V3(GW
M_Q6,;@``4TV)Z$R-#9$0``!)B<=(C062$0``3(GRO@(```!02(T->1$``$R)
M_S'`_Q6L;P``3(G_OL`!``#_%>9O``!?05B#^/]U$_\5,&\``$2+"$&#^1$/
MA;L"``!(C84P____3(G_2(F%F/[__TB)QO\5.6\``(G#A<!T6TB+A:C^__]-
MB?A(BST"<```2(T5RQ$``+X"````2(L(,<#_%;-M``!%,>T/'X0``````$B+
M1<AD2"L$)2@````/A70$``!(C6783(GH6T%<05U!7D%?7<,/'P"+A4C___\E
M`/```#T`0```=9-$BZU,_____Q7!;@``03G%=8&+A4C___\E_P$``#W``0``
M#X5K____3(GW_Q5U;0``3(GF2(T]31```$F)Q>ASZ?__2(7`#X2*`@``2(N-
MJ/[__TB)QDB+.>AH\/__2(F%H/[__TB%P`^$:`(``$B+O:#^____%8MM``!(
MB<*#^`-^4TB+C:#^__](F$B)E9#^__](C34L$```2(U\`?S_%1%N``"%P'4M
M2(N5D/[__X/Z!`^$,`,``(U"^TB+C:#^__](F(`\`2\/A!H#```/'X``````
M3(GF2(T]%0\``.C1Z/__2(7`#X0(`@``#[8`A,`/A/T!```\,`^$]0$``$B-
M-5H/``!(C3WF#@``Z'+I__],C27O#P``_Q6];```051,B?)-B?A03(T-A0X`
M`+X"````3(GO2(T-J@\``#'`_Q6N;0``6%KK3F8N#Q^$``````#_%3IM``"#
M.!%U3/\5=VP``(/#`4B#[`A-B?A!5$R-#3P.``!,B?),B>]32(T-<`\``+X"
M````4#'`_Q5@;0``2(/$(+[``0``3(GO_Q66;0``@_C_=*E,B?__%4AL``!,
MB>Y(C3W>#0``Z,'H__],B>_H^?G__^GT_?__#Q]``$4Q[>D,_?__#Q^$````
M``!(C9W`_O__ZPL/'X``````2(/#"$B+.TB%_P^$E0```$R)YNBHY___2(7`
M=..`.`!TWDB)Q_\5?6P``$F)QTB%P'3-Z9C[__](BX6H_O__38GX2(L]9VT`
M`$B-%?`.``"^`@```$B+"#'`_Q48:P``Z6#]__\/'P!(B<=(B86H_O__Z%GY
M__](BU7(9$@K%"4H````#X7-`0``2(N]J/[__TB-9=A;05Q!74%>05]=_R4#
M;```OPT```!!O@T```!,C3W@#0``0;T-````_Q7M:@``0;A3````2(F%H/[_
M_TB)P^DH^___#Q^$``````!(BX6H_O__2(L`2(F%H/[__^F"_?__9BX/'X0`
M`````$B+O:#^__\Q]C'`_Q5G:P``08G$A<`/A._]__^)Q^@=]___2(7`#XC?
M_?__,=)(C7#V1(GG2(F%J/[___\5KFL``$B-M;K^__^Z!@```$2)Y_\526H`
M`$B+C:C^__](@_@&#X7U````@;VZ_O__`$-!0P^%D_W__V:!O;[^__](10^%
MA/W__S'22(UQSD2)Y_\56FL``$B+G9C^__^Z*````$2)YTB)WO\5\FD``$B#
M^"@/A:4```!(C055#0``O@(```!-B?A,B?+&A5C___\`2(T-#0T``$R-#>$+
M``!,B>]0,<!3_Q44:P``65[IQ/W__TB+A:C^__](BU`(2(72#X39_/__2(F5
MJ/[__TB)U_\5^6D``$B#^`,/AK_\__](BY6H_O__2(TUI@P``$B-?`+\_Q6&
M:@``2(N5J/[__X7`2`]%E:#^__](B96@_O__Z8G\____%5MJ``!(BT7(9$@K
M!"4H````=>M(C3UG#```,<#H!O7__V8/'T0``/,/'OI52(T]5PL``$B)Y4%5
M051(C7744TB#[!AD2(L<)2@```!(B5W82(L=F&D``&1,BR-!#[:$)+````!!
MB(0DL0```.CVY/__2(7`=`F`.#$/A(`!``#HX^___V1,BR-,B>>Y`P```+H!
M````2(TU'@P``/\5Q6@``$F)Q$B%P'0J2(M`$&1(BSM(C14##```2(LP_Q6]
M:```28M$)!!(BS#V1@Y`#X5;`0``9$B+.[D#````N@$```!(C378"P``_Q5X
M:```28G$2(7`=#%(C3U:"@``2(UUU.A@Y/__28M4)!!(BSI(A<`/A/\```!(
MB?Y(B<)D2(L[_Q57:```9$B+.[H!````2(TUBPL``/\5R6<``&1(BSM(C16/
M"P``2(G&_Q4M:```9$B+`TB-==1(C3WN"0``QH"Q`````.CZX___2(D%^VD`
M`$B)PTB%P'1#OL`!``!(B=__%65I``"-4`&#^@%V"8/X$0^%HP```$B+1=AD
M2"L$)2@````/A;T```!(@\086T%<05U=PV8/'T0``$B+/;%I``#HG/;__TB)
M!9UI``!(B<-(A<!UHDB+1=AD2"L$)2@````/A7X```!(C3V+"P``,<#_%8MH
M```/'P"`>`$`#X1__O__Z7'^__]F#Q^$``````!(BP59:0``2(LPZ%'B___I
M_?[__P\?0`!D2(L[_Q4>:```Z9;^__](BU789$@K%"4H````=1Z)P4B+!2%I
M``!(B=I(C3U'"P``2(LP,<#_%1QH``#_%>YG``!F#Q]$``#S#Q[Z54B)Y5-(
MC77D2(/L*&1(BQPE*````$B)7>A(B?M(C3T'"0``Z,/B__](A<!T'`^V$(32
M#Y3`@/HP#Y3""-!U"DB%VW0%@#L`=1I(BT7H9$@K!"4H````=6Q(BUWXR<,/
M'T0``$B)W^@@[/__OB\```!(B<=(B478_Q4>9P``2(7`2(UX`4@/1'W8_Q5$
M9P``N@0```!(C35;"0``2(G'_Q4O9@``A<!UH4B+1>AD2"L$)2@```!U#4B)
MWTB+7?C)Z??E____%1EG``"0\P\>^E6_@````$B)Y?\5_64``&8/;P452```
M2,=`,``````/$0!F#V\%$D@``,=`>``````/$4`09@]O!0](```/$4`@7<-F
M#Q^$``````#S#Q[Z54ACPDB)Y4%728G_059!54R-;SA!5$F)Q%-(@^PH2(EU
MR$B+=RA(BU<PC0S&2#GQ2(E/*$B#T@!(P>@=2`'02(E',$AC1WB%P`^%+0$`
M`$&#_#\/CH0```!$B>-,BW7(P>L&C4/_2,'C!HE%Q$P!\TB)1;@/'X0`````
M`/-!#V\&3(G_28/&0$$/$44`\T$/;T;000\111#S00]O1N!!#Q%%(/-!#V]&
M\$$/$44PZ.G)__]).=YUQ$B+1;A(BTW(1(MUQ$C!X`9(C40!0$'WWDB)1<A!
MP>8&1XUD-,!)8\1(@_@(<FY(BWW(28UW0$R)Z4B#YOA(BQ=(*?%)B5<X2(M4
M!_A(*<])B50%^$@!R$F)_4B#X/A(@_@(<B!(@^#X,=))BTP5`$B)#!9(@\((
M2#G"<NX/'X0``````$6)9WA(@\0H6T%<05U!7D%?7<,/'T0``*@$=6U(A<!T
MWTB+3<@/MA%!B%<XJ`)TT`^W5`'^9D&)5`7^Z\,/'P!!OD````!(BW7(28U\
M!0!!*<9%.>9%#T_T26/>2(G:_Q6C90``08M'>$0!\$&)1WB#^$!UCTR)_T@!
M7<A%*?3HW,C__^F*_O__2(M-R(L108E7.(M4`?Q!B50%_.E?____9F8N#Q^$
M```````/'T``\P\>^E5(B>5!5$F)]%-(B?M(@^P03(M&*$B+=C!,B<!(P>@#
M@^`_C5`!0<9$!#B`2&/"28U,!#B#^C@/CJ<!``"X0````"G0=`\QTHG7@\(!
MQ@0Y`#G"<O-,B>=(B77@3(E%Z.A'R/__9@_OP$R+1>A)QT0D:`````!(BW7@
M00\11"0X00\11"1(00\11"182(GR00^VP$V)PDV)P4C!ZAA(P>`(2(GQ2<'J
M$$F)TTR)PDG!Z1@/MOY(B?)(P>D02`GX00^V^@^VR4C!X`A("?A!#[;Y2,'@
M"$@)^$`/MOX/MO9!#[;32,'@"$@)^$R)YTC!X`A("?!(P>`(2`G(2,'@"$@)
MT$F)1"1PZ*#'__])BP0D3(GG2,'H&(@#28L$)$C!Z!"(0P%)BP0DB&,"28L$
M)(A#`TF+1"0(2,'H&(A#!$F+1"0(2,'H$(A#!4F+1"0(B&,&28M$)`B(0P=)
MBT0D$$C!Z!B(0PA)BT0D$$C!Z!"(0PE)BT0D$(AC"DF+1"00B$,+28M$)!A(
MP>@8B$,,28M$)!A(P>@0B$,-28M$)!B(8PY)BT0D&(A##TF+1"0@2,'H&(A#
M$$F+1"0@2,'H$(A#$4F+1"0@B&,228M$)""(0Q-(@\006T%<7?\E1&(```\?
M0`"X.````"G0@_@(<RRH!'5XA<`/A)#^___&`0"H`@^$A?[__S'29HE4`?[I
M>?[__V8/'X0``````(G"2,<!`````$C'1!'X`````$B-40A(@^+X2"G1`<B#
MX/B#^`@/@D7^__^#X/@QR8G/@\$(2,<$.@`````YP7+OZ2K^__]F+@\?A```
M````QP$`````QT0!_`````#I#?[__V8N#Q^$```````/'P#S#Q[Z2(L-W6(`
M`/\E%V$``````/,/'OI(@^P(2(/$",,`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````0`"`"]P<F]C+W-E;&8O97AE`"5S+R5S`$1Y;F%,;V%D97(Z.F)O
M;W1?1'EN84QO861E<@!);G1E<FYA;',Z.E!!4CHZ0D]/5``O<')O8R\E:2\E
M<P`N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`4$523#5,24(`4$523$Q)
M0@!015),-4]05`!015),24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$
M`%!!4E]#3$5!3@!005)?1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"
M54<`4$%27T=,3T)!3%]435!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%
M35``4$%27T=,3T)!3%]#3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE
M9`!,1%],24)205)97U!!5$@`4UE35$5-`"]T;7``5$5-4$1)4@!435``55-%
M4@!54T523D%-10`E,#)X`"5S)7,E<R5S`'!A<BT`<&%R;``N<&%R`'-H;W)T
M(')E860`)7,E<V-A8VAE+25S)7,`)7,E<W1E;7`M)74E<P`E<R5S=&5M<"TE
M=2TE=25S`!@`<&5R;``P`%!!4CHZ4&%C:V5R.CI615)324].`#$N,#8T`"UE
M`"TM`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*````````56YA8FQE('1O(&-R96%T92!C86-H
M92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A
M8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D
M8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2
M+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*
M)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S
M7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*
M;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE
M(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@
M)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@
M;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@
M)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H
M*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?
M34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D
M8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F
M:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?
M96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?
M<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S
M96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@
M>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO
M=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C
M(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA
M<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?
M1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD
M(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE
M(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P
M;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A
M9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B
M7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C
M:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H
M92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,
M15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I
M.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?
M1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B
M"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA
M;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B
M+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX
M?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@
M*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@
M*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI
M)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M
M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L
M;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@
M7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF
M+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M
M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U
M<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII
M9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W
M*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#
M86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD
M9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T
M=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@
M/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N
M("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD
M(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E
M<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA
M;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA
M;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H
M)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII
M9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA
M;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@
M*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@
M)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P
M87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N
M7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=
M(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%
M55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E
M:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=6
M6S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF
M("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@
M*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G
M02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G
M3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L
M92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G
M86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS
M:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE
M('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D
M,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD[
M"FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG
M4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[
M(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I
M<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`
M24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@
M*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O
M<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE
M<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII
M9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY
M;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I]
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I
M<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#
M.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/
M.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ
M1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H
M+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H
M9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D
M;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB
M*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T
M>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@
M/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A
M8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X]
M(")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@
M)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L
M;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B
M.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@
M>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO
M=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@
M*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T
M;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X
M="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L
M<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H
M)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@
M<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE
M(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER
M.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM
M>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS
M965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H
M+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C
M86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@
M*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[
M4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C
M86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-
M4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H
M<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R
M;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA
M;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?
M<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!
M4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S
M+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?
M<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!
M345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@
M>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@
M)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I
M"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H
M*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<
M+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D
M14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"
M04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[
M4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*
M:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP
M<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I
M9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF
M:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP
M('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP
M86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2
M.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A
M<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@
M<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE
M.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/
M4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@
M)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,`````
M`````2-%9P````")J\WO`````/[<NI@`````=E0R$`````#PX=+#````````
M`````````1L#.^P````<````<'___W`$``"0@?__"`$``)""__\@`0``T)C_
M_U`!``#`F?__>`$``)":__^D`0```)W__]0!``"0G?__^`$``$"?__\@`@``
MH)___T`"``#0G___5`(``,"@__]\`@``T*#__Y`"```@H___O`(``&"C___<
M`@``P*3__P@#```0J?__,`,``("I__]4`P``0*K__W`#``"PJ___F`,``-"K
M__^X`P``P*S__^`#``"`M/__&`0``-"V__]$!```H+?__Y@$``#PM___N`0`
M`."Y___H!```<+S__Q`%````````%``````````!>E(``7@0`1L,!PB0`0``
M%````!P```"`@/__)@````!$!Q``````+````#0```!H@?__/!8```!!#A"&
M`D<-!E>/`XX$C06,!H,'`Q86"@P'"$$+````)````&0```!XE___ZP````!!
M#A"&`D,-!E6-`XP$@P4">@H,!PA!"R@```",````0)C__\D`````00X0A@)#
M#09-CP..!(T%C`:#!P)W"@P'"$@++````+@```#DF/__:@(```!!#A"&`D4-
M!DV/`XX$C06,!H,'`S\!"@P'"$8+````(````.@````DF___A0````!!#A"&
M`D,-!D^#`P)K"@P'"$$+)`````P!``"0F___K0$```!!#A"&`D,-!DR.`XP$
M@P5\"@P'"$0+`!P````T`0``&)W__UT`````10X0A@)1#09%@P-\#`<($```
M`%0!``!8G?__*@`````````D````:`$``'2=___K`````$4.$(8"0PT&7XP#
M@P0"O0H,!PA!"P``$````)`!```\GO__"0`````````H````I`$``#B>__])
M`@```$4.$(8"0PT&7(\#C@2-!8P&@P<";PH,!PA%"QP```#0`0``7*#__S$`
M````10X0A@)(#09!@P-=#`<(*````/`!``!\H/__6@$```!%#A"&`D,-!@,8
M`0H,!PA("U`*#`<(4`L````D````'`(``+"A__]'!````$4.$(8"0PT&18,#
M`PD$"@P'"$4+````(````$0"``#8I?__8@````!%#A"&`DH-!@)%"@P'"$0+
M````&````&@"```DIO__O`````!%#A"&`D,-!@```"0```"$`@``R*;__V4!
M````00X0A@)##09BC0.,!(,%`MT*#`<(10L<````K`(``!"H__\@`````$4.
M$(8"0PT&5PP'"````"0```#,`@``$*C__^L`````10X0A@)##09+C@.#!`)+
M"@P'"$H+```T````]`(``-BH__^Z!P```$4.$(8"0PT&2(\#C@2-!8P&4H,'
M`Q(#"@P'"$0+`Z<""@P'"$8+`"@````L`P``8+#__TH"````10X0A@)*#09-
MC0.,!(,%`W4!"@P'"$<+````*````%@#``"$LO__SP````!%#A"&`D,-!DF#
M`P)1"@P'"$8+`EP*#`<(10LD````A`,``/AZ__\5`@```$4.$(8"0PT&1XP#
M@P0#:P$*#`<(00L`'````*P#````L___1P````!%#A"&`D@-!GD,!P@````L
M````S`,``#"S___A`0```$4.$(8"1@T&0H\#1XX$C05&C`9(@P<#,`$*#`<(
M1@LD````_`,``/"T__^#`@```$4.$(8"0PT&0HP#1(,$`]`!"@P'"$H+$```
M`"0$``!8M___$0``````````````!````!`````!````1TY5```````$````
M!```````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````,",```````#@(@```````$YA````````Z6`````````&80```````%]A
M`````````0````````#H`@````````$`````````\P(````````=````````
M`(X$````````#```````````(`````````T`````````-%T````````9````
M`````,"[````````&P`````````(`````````!H`````````R+L````````<
M``````````@`````````]?[_;P````#@`P````````4`````````2`X`````
M```&`````````'@%````````"@````````"Q!`````````L`````````&```
M```````5````````````````````!P````````!(%`````````@`````````
M,`8````````)`````````!@`````````'@`````````(`````````/O__V\`
M`````0``"`````#^__]O`````+@3````````____;P`````!`````````/#_
M_V\`````^A(````````D`````````'@:````````(P`````````@````````
M`"4`````````"```````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````#PNP``````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````C`````````"E!!4BYP;0H``````````/AB
M````````1T-#.B`H1TY5*2`Q-2XQ+C$@,C`R-3`T,C4``"YS:'-T<G1A8@`N
M;F]T92YG;G4N<')O<&5R='D`+FYO=&4N9VYU+F)U:6QD+6ED`"YI;G1E<G``
M+F=N=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV
M97)S:6]N7W(`+G)E;&$N9'EN`"YR96QR+F1Y;@`N:6YI=``N=&5X=``N9FEN
M:0`N<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YN;W1E+D%"22UT
M86<`+FEN:71?87)R87D`+F9I;FE?87)R87D`+F1A=&$N<F5L+G)O`"YD>6YA
M;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````````
M````````````````````````````````````````````````````````````
M````````"P````<````"`````````%`#````````4`,```````!0````````
M````````````"````````````````````!X````'`````@````````"@`P``
M`````*`#````````)`````````````````````0````````````````````Q
M`````0````(`````````Q`,```````#$`P```````!P`````````````````
M```!````````````````````.0```/;__V\"`````````.`#````````X`,`
M``````"4`0````````4`````````"````````````````````$,````+````
M`@````````!X!0```````'@%````````T`@````````&`````0````@`````
M````&`````````!+`````P````(`````````2`X```````!(#@```````+$$
M```````````````````!````````````````````4P```/___V\"````````
M`/H2````````^A(```````"\``````````4``````````@`````````"````
M`````&````#^__]O`@````````"X$P```````+@3````````D``````````&
M`````0````@```````````````````!O````!`````(`````````2!0`````
M``!(%````````#`&````````!0`````````(`````````!@`````````>0``
M`!,````"`````````'@:````````>!H````````@````````````````````
M"``````````(`````````(,````!````!@``````````(``````````@````
M````&P````````````````````0```````````````````")`````0````8`
M````````("`````````@(````````!$]```````````````````@````````
M````````````CP````$````&`````````#1=````````-%T````````-````
M````````````````!````````````````````)4````!`````@``````````
M8`````````!@````````L$```````````````````!``````````````````
M``"=`````0````(`````````L*````````"PH````````.P`````````````
M```````$````````````````````JP````$````"`````````*"A````````
MH*$````````X!```````````````````"````````````````````+4````'
M`````@````````#8I0```````-BE````````(`````````````````````0`
M``````````````````##````#@````,`````````P+L```````#`JP``````
M``@````````````````````(``````````@`````````SP````\````#````
M`````,B[````````R*L````````(````````````````````"``````````(
M`````````-L````!`````P````````#0NP```````-"K````````(```````
M``````````````@```````````````````#H````!@````,`````````\+L`
M``````#PJP`````````"````````!@`````````(`````````!``````````
M\0````$````#`````````/"]````````\*T````````0`@``````````````
M````"``````````(`````````/8````!`````P``````````P`````````"P
M````````*`````````````````````@```````````````````#\````"```
M``,`````````0,`````````HL````````%`````````````````````@````
M`````````````````0$```$````P````````````````````*+`````````;
M`````````````````````0`````````!``````````$````#````````````
M`````````````$.P````````"@$```````````````````$`````````````
%````````
