/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.connector.catalog

import java.util

import org.apache.spark.sql.catalyst.analysis.TableAlreadyExistsException
import org.apache.spark.sql.connector.expressions.Transform

class InMemoryPartitionTableCatalog extends InMemoryTableCatalog {
  import CatalogV2Implicits._

  override def createTable(
      ident: Identifier,
      columns: Array[Column],
      partitions: Array[Transform],
      properties: util.Map[String, String]): Table = {
    if (tables.containsKey(ident)) {
      throw new TableAlreadyExistsException(ident.asMultipartIdentifier)
    }

    InMemoryTableCatalog.maybeSimulateFailedTableCreation(properties)

    val finalCols = if (properties.containsKey("dropExistsDefault")) {
      columns.map { c =>
        if (c.defaultValue().getValue == null) {
          c
        } else {
          Column.create(
            c.name(),
            c.dataType(),
            c.nullable(),
            c.comment(),
            new ColumnDefaultValue(
              c.defaultValue().getSql,
              c.defaultValue().getExpression,
              null
            ),
            c.metadataInJSON()
          )
        }
      }
    } else {
      columns
    }
    val schema = CatalogV2Util.v2ColumnsToStructType(finalCols)
    val table = new InMemoryAtomicPartitionTable(
      s"$name.${ident.quoted}", schema, partitions, properties)
    tables.put(ident, table)
    namespaces.putIfAbsent(ident.namespace.toList, Map())
    table
  }

  override def createTable(ident: Identifier, tableInfo: TableInfo): Table = {
    createTable(ident, tableInfo.columns(), tableInfo.partitions(), tableInfo.properties)
  }
}
