use Test::More;
use strict;
use warnings;
our ( $sub, $globref );

BEGIN {
	use_ok('Hades::Realm::Exporter');
	$sub     = sub { };
	$globref = \*globref;
}
subtest 'new' => sub {
	plan tests => 9;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	ok( $obj = Hades::Realm::Exporter->new(),
		q{$obj = Hades::Realm::Exporter->new()}
	);
	isa_ok( $obj, 'Hades::Realm::Exporter' );
	ok( $obj = Hades::Realm::Exporter->new( {} ),
		q{$obj = Hades::Realm::Exporter->new({})}
	);
	ok( $obj = Hades::Realm::Exporter->new(),
		q{$obj = Hades::Realm::Exporter->new()}
	);
	is_deeply( $obj->export, {}, q{$obj->export} );
	ok( $obj
		    = Hades::Realm::Exporter->new(
			{ export => { 'test' => 'test' } } ),
		q{$obj = Hades::Realm::Exporter->new({ export => { 'test' => 'test' } })}
	);
	eval { $obj = Hades::Realm::Exporter->new( { export => [] } ) };
	like(
		$@,
		qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::Exporter->new({ export => [] })}
	);
	eval { $obj = Hades::Realm::Exporter->new( { export => 'algea' } ) };
	like(
		$@,
		qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::Exporter->new({ export => 'algea' })}
	);
};
subtest 'export' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'export' );
	is_deeply(
		$obj->export( { 'test' => 'test' } ),
		{ 'test' => 'test' },
		q{$obj->export({ 'test' => 'test' })}
	);
	eval { $obj->export( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->export([])} );
	eval { $obj->export('geras') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->export('geras')} );
	is_deeply( $obj->export, { 'test' => 'test' }, q{$obj->export} );
};
subtest 'default_export_hash' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'default_export_hash' );
	eval {
		$obj->default_export_hash(
			[],
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash([], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash(
			'penthos',
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash('penthos', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			[], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			'algea', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), 'algea', { 'test' => 'test' })}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			{ 'test' => 'test' }, [] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->default_export_hash( bless( {}, 'Test' ),
			{ 'test' => 'test' }, 'phobos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->default_export_hash(bless({}, 'Test'), { 'test' => 'test' }, 'phobos')}
	);
};
subtest 'build_new' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_new' );
	eval { $obj->build_new( [], { 'test' => 'test' }, 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new([], { 'test' => 'test' }, 'nosoi')}
	);
	eval { $obj->build_new( 'algea', { 'test' => 'test' }, 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new('algea', { 'test' => 'test' }, 'nosoi')}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), [], 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), [], 'nosoi')}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), 'limos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), 'limos', 'nosoi')}
	);
};
subtest 'build_exporter' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_exporter' );
	eval {
		$obj->build_exporter(
			[],
			bless( {}, 'Test' ),
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter([], bless({}, 'Test'), { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			\1,
			bless( {}, 'Test' ),
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter(\1, bless({}, 'Test'), { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'phobos', [],
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', [], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'phobos', 'gaudia',
			{ 'test' => 'test' },
			{ 'test' => 'test' }
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', 'gaudia', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter( 'phobos', bless( {}, 'Test' ),
			[], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter( 'phobos', bless( {}, 'Test' ),
			'hypnos', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', bless({}, 'Test'), 'hypnos', { 'test' => 'test' })}
	);
	eval {
		$obj->build_exporter(
			'phobos',
			bless( {}, 'Test' ),
			{ 'test' => 'test' }, []
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->build_exporter(
			'phobos',
			bless( {}, 'Test' ),
			{ 'test' => 'test' }, 'algea'
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_exporter('phobos', bless({}, 'Test'), { 'test' => 'test' }, 'algea')}
	);
};
subtest 'build_export_tags' => sub {
	plan tests => 12;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_export_tags' );
	eval {
		$obj->build_export_tags( [], 'limos', { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags([], 'limos', { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( \1, 'limos', { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags(\1, 'limos', { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', [], { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', [], { 'test' => 'test' }, undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', \1, { 'test' => 'test' },
			undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', \1, { 'test' => 'test' }, undef, ['test'])}
	);
	eval { $obj->build_export_tags( 'limos', 'limos', [], undef, ['test'] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', [], undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', 'limos', 'nosoi', undef, ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', 'nosoi', undef, ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', 'limos', { 'test' => 'test' },
			[], ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', { 'test' => 'test' }, [], ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', 'limos', { 'test' => 'test' },
			'phobos', ['test'] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', { 'test' => 'test' }, 'phobos', ['test'])}
	);
	eval {
		$obj->build_export_tags( 'limos', 'limos', { 'test' => 'test' },
			undef, {} );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', { 'test' => 'test' }, undef, {})}
	);
	eval {
		$obj->build_export_tags( 'limos', 'limos', { 'test' => 'test' },
			undef, 'penthos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_export_tags('limos', 'limos', { 'test' => 'test' }, undef, 'penthos')}
	);
};
subtest 'after_class' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'after_class' );
	eval { $obj->after_class( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class([])} );
	eval { $obj->after_class('geras') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class('geras')} );
};
subtest 'build_sub_or_accessor_attributes' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_or_accessor_attributes' );
};
subtest 'build_accessor_no_arguments' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor_no_arguments' );
};
subtest 'build_accessor_code' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor_code' );
	eval { $obj->build_accessor_code( [], 'geras', 'thanatos', 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code([], 'geras', 'thanatos', 'penthos')}
	);
	eval { $obj->build_accessor_code( \1, 'geras', 'thanatos', 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code(\1, 'geras', 'thanatos', 'penthos')}
	);
	eval { $obj->build_accessor_code( 'algea', [], 'thanatos', 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', [], 'thanatos', 'penthos')}
	);
	eval { $obj->build_accessor_code( 'algea', \1, 'thanatos', 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', \1, 'thanatos', 'penthos')}
	);
	eval { $obj->build_accessor_code( 'algea', 'geras', [], 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', 'geras', [], 'penthos')}
	);
	eval { $obj->build_accessor_code( 'algea', 'geras', \1, 'penthos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', 'geras', \1, 'penthos')}
	);
	eval { $obj->build_accessor_code( 'algea', 'geras', 'thanatos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', 'geras', 'thanatos', [])}
	);
	eval { $obj->build_accessor_code( 'algea', 'geras', 'thanatos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_code('algea', 'geras', 'thanatos', \1)}
	);
};
subtest 'build_accessor' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_accessor' );
};
subtest 'build_modify' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_modify' );
};
subtest 'build_sub_no_arguments' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_no_arguments' );
};
subtest 'build_sub_code' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub_code' );
	eval { $obj->build_sub_code( [], 'phobos', 'phobos', 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code([], 'phobos', 'phobos', 'aporia')}
	);
	eval { $obj->build_sub_code( \1, 'phobos', 'phobos', 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code(\1, 'phobos', 'phobos', 'aporia')}
	);
	eval { $obj->build_sub_code( 'phobos', [], 'phobos', 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', [], 'phobos', 'aporia')}
	);
	eval { $obj->build_sub_code( 'phobos', \1, 'phobos', 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', \1, 'phobos', 'aporia')}
	);
	eval { $obj->build_sub_code( 'phobos', 'phobos', [], 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', 'phobos', [], 'aporia')}
	);
	eval { $obj->build_sub_code( 'phobos', 'phobos', \1, 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', 'phobos', \1, 'aporia')}
	);
	eval { $obj->build_sub_code( 'phobos', 'phobos', 'phobos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', 'phobos', 'phobos', [])}
	);
	eval { $obj->build_sub_code( 'phobos', 'phobos', 'phobos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_sub_code('phobos', 'phobos', 'phobos', \1)}
	);
};
subtest 'build_sub' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_sub' );
};
subtest 'build_clearer' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_clearer' );
	eval { $obj->build_clearer( [], 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer([], 'aporia', { 'test' => 'test' })}
	);
	eval { $obj->build_clearer( 'curae', 'aporia', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer('curae', 'aporia', { 'test' => 'test' })}
	);
	eval {
		$obj->build_clearer( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_clearer( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_clearer( bless( {}, 'Test' ), 'aporia', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), 'aporia', [])}
	);
	eval { $obj->build_clearer( bless( {}, 'Test' ), 'aporia', 'algea' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_clearer(bless({}, 'Test'), 'aporia', 'algea')}
	);
};
subtest 'build_predicate' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_predicate' );
	eval { $obj->build_predicate( [], 'curae', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate([], 'curae', { 'test' => 'test' })}
	);
	eval { $obj->build_predicate( 'aporia', 'curae', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate('aporia', 'curae', { 'test' => 'test' })}
	);
	eval {
		$obj->build_predicate( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_predicate( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_predicate( bless( {}, 'Test' ), 'curae', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), 'curae', [])}
	);
	eval { $obj->build_predicate( bless( {}, 'Test' ), 'curae', 'thanatos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_predicate(bless({}, 'Test'), 'curae', 'thanatos')}
	);
};
subtest 'build_coerce' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_coerce' );
	eval { $obj->build_coerce( [], 'limos', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce([], 'limos', undef)}
	);
	eval { $obj->build_coerce( \1, 'limos', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce(\1, 'limos', undef)}
	);
	eval { $obj->build_coerce( 'geras', [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('geras', [], undef)}
	);
	eval { $obj->build_coerce( 'geras', \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('geras', \1, undef)}
	);
	eval { $obj->build_coerce( 'geras', 'limos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('geras', 'limos', [])}
	);
	eval { $obj->build_coerce( 'geras', 'limos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_coerce('geras', 'limos', \1)}
	);
};
subtest 'build_trigger' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_trigger' );
	eval { $obj->build_trigger( [], 'algea', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger([], 'algea', undef)}
	);
	eval { $obj->build_trigger( \1, 'algea', undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger(\1, 'algea', undef)}
	);
	eval { $obj->build_trigger( 'geras', [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('geras', [], undef)}
	);
	eval { $obj->build_trigger( 'geras', \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('geras', \1, undef)}
	);
	eval { $obj->build_trigger( 'geras', 'algea', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('geras', 'algea', [])}
	);
	eval { $obj->build_trigger( 'geras', 'algea', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_trigger('geras', 'algea', \1)}
	);
};
subtest 'build_tests' => sub {
	plan tests => 10;
	ok( my $obj = Hades::Realm::Exporter->new( {} ),
		q{my $obj = Hades::Realm::Exporter->new({})}
	);
	can_ok( $obj, 'build_tests' );
	eval { $obj->build_tests( [], { 'test' => 'test' }, undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests([], { 'test' => 'test' }, undef, undef)}
	);
	eval { $obj->build_tests( \1, { 'test' => 'test' }, undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests(\1, { 'test' => 'test' }, undef, undef)}
	);
	eval { $obj->build_tests( 'phobos', [], undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', [], undef, undef)}
	);
	eval { $obj->build_tests( 'phobos', 'aporia', undef, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', 'aporia', undef, undef)}
	);
	eval { $obj->build_tests( 'phobos', { 'test' => 'test' }, [], undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', { 'test' => 'test' }, [], undef)}
	);
	eval { $obj->build_tests( 'phobos', { 'test' => 'test' }, \1, undef ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', { 'test' => 'test' }, \1, undef)}
	);
	eval { $obj->build_tests( 'phobos', { 'test' => 'test' }, undef, [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', { 'test' => 'test' }, undef, [])}
	);
	eval {
		$obj->build_tests( 'phobos', { 'test' => 'test' }, undef, 'geras' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_tests('phobos', { 'test' => 'test' }, undef, 'geras')}
	);
};
done_testing();
